extern crate clap;

use clap::{Args, Parser, Subcommand};
use swaysome::SwaySome;

#[derive(Parser, Debug)]
#[clap(author, version, about = "Better multimonitor handling for sway", long_about = None)]
#[clap(propagate_version = true)]
struct Cli {
    #[clap(subcommand)]
    command: Command,
}

#[derive(Subcommand, Debug)]
enum Command {
    #[clap(about = "Initialize the workspace groups for all the outputs")]
    Init(Index),

    #[clap(about = "Move the focused container to another workspace on the same workspace group")]
    Move(Index),

    #[clap(
        about = "Move the focused container to the same workspace index on another workspace group"
    )]
    MoveToGroup(Index),

    #[clap(about = "Focus to another workspace on the same workspace group")]
    Focus(Index),

    #[clap(about = "Focus to workspace group")]
    FocusGroup(Index),

    #[clap(about = "Focus to another workspace on all the outputs")]
    FocusAllOutputs(Index),

    #[clap(about = "Move the focused container to the next output")]
    NextOutput,

    #[clap(about = "Move the focused container to the previous output")]
    PrevOutput,

    #[clap(about = "Move the focused workspace group to the next output")]
    WorkspaceGroupNextOutput,

    #[clap(about = "Move the focused workspace group to the previous output")]
    WorkspaceGroupPrevOutput,

    #[clap(about = "Move the focused container to the next group")]
    NextGroup,

    #[clap(about = "Move the focused container to the previous group")]
    PrevGroup,

    #[clap(
        about = "Rearrange already opened workspaces to the correct outputs, useful when plugging new monitors"
    )]
    RearrangeWorkspaces,
}

#[derive(Args, Debug)]
struct Index {
    #[clap(value_name = "index", help = "The workspace index to work with")]
    index: usize,
}

fn main() {
    let cli = Cli::parse();

    let swaysome = SwaySome::new();

    match &cli.command {
        Command::Init(action) => {
            swaysome.init_workspaces(action.index);
        }
        Command::Move(action) => {
            swaysome.move_container_to_workspace(action.index);
        }
        Command::MoveToGroup(action) => {
            swaysome.move_container_to_workspace_group(action.index);
        }
        Command::Focus(action) => {
            swaysome.focus_to_workspace(action.index);
        }
        Command::FocusGroup(action) => {
            swaysome.focus_to_group(action.index);
        }
        Command::FocusAllOutputs(action) => {
            swaysome.focus_all_outputs_to_workspace(action.index);
        }
        Command::NextOutput => {
            swaysome.move_container_to_next_output();
        }
        Command::PrevOutput => {
            swaysome.move_container_to_prev_output();
        }
        Command::WorkspaceGroupNextOutput => {
            swaysome.move_workspace_group_to_next_output();
        }
        Command::WorkspaceGroupPrevOutput => {
            swaysome.move_workspace_group_to_prev_output();
        }
        Command::NextGroup => {
            swaysome.focus_to_next_group();
        }
        Command::PrevGroup => {
            swaysome.focus_to_prev_group();
        }
        Command::RearrangeWorkspaces => {
            swaysome.rearrange_workspaces();
        }
    }
}
