/* textviewmode.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2011-2014 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <aguix/acontainer.h>
#include <aguix/textview.h>
#include "textviewmode.hh"
#include "worker_locale.h"
#include "wconfig.h"
#include "worker.h"
#include "fileentry.hh"
#include "nmspecialsourceext.hh"
#include "textstoragefile.h"
#include "method_callback.hh"

const char *TextViewMode::type = "TextViewMode";

#define TVM_DEFAULT_SIZE ( 16 * 1024 )

TextViewMode::TextViewMode( Lister *parent ) : ListerMode( parent )
{
    m_lastactivefe = NULL;
    m_tv = NULL;
    m_cont = NULL;
    m_info_text = nullptr;
    m_file_name_text = nullptr;
    m_file_percentage_text = nullptr;

    registerCommand( "up", new MethodCallback<TextViewMode, &TextViewMode::command_up>( this ) );
    registerCommand( "down", new MethodCallback<TextViewMode, &TextViewMode::command_down>( this ) );
    registerCommand( "first", new MethodCallback<TextViewMode, &TextViewMode::command_first>( this ) );
    registerCommand( "last", new MethodCallback<TextViewMode, &TextViewMode::command_last>( this ) );
    registerCommand( "pageup", new MethodCallback<TextViewMode, &TextViewMode::command_pageup>( this ) );
    registerCommand( "pagedown", new MethodCallback<TextViewMode, &TextViewMode::command_pagedown>( this ) );
}

TextViewMode::~TextViewMode()
{
    clearLastActiveFE();
}

void TextViewMode::messageHandler( AGMessage *msg )
{
    bool ma = false;
    switch ( msg->type ) {
        case AG_KEYPRESSED:
            if ( parentlister->isActive() == true ) {
                m_tv->handleKeyMessage( msg->key.key,
                                        msg->key.keystate );
            }
            break;
        case AG_MOUSECLICKED:
            if ( m_tv->isParent( msg->mouse.window, false ) ) {
                ma = true;
            }
            break;
        case AG_TEXTVIEW_END_REACHED:
            if ( msg->textview.textview == m_tv &&
                 m_ts->incompleteFile() ) {
                m_ts->readMore();
                m_tv->textStorageChanged();
                setFilePercentage();
            }
    }
    if ( ma == true ) parentlister->makeActive();
}

void TextViewMode::on()
{
    m_lencalc = new AFontWidth( aguix, aguix->getFont( wconfig->getFont( 4 ).c_str() ) );
    m_ts = new TextStorageFile( "", m_lencalc, TVM_DEFAULT_SIZE );

    m_cont = new AContainer( parentawindow, 1, 2 );
    m_cont->setBorderWidth( 0 );
    m_cont->setMinSpace( 0 );
    m_cont->setMaxSpace( 0 );
    parentlister->setContainer( m_cont );

    m_ac1 = (AContainer*)m_cont->add( new AContainer( parentawindow, 3, 1 ),
                                      0, 0 );
    m_ac1->setBorderWidth( 2 );
    m_ac1->setMinSpace( 5 );
    m_ac1->setMaxSpace( 5 );

    m_info_text = m_ac1->addWidget( new Text( aguix, 0, 0, catalog.getLocale( 404 ) ), 0, 0, AContainer::CO_FIX );
    m_file_name_text = (Text*)m_ac1->add( new Text( aguix, 0, 0, "" ),
                                          1, 0, AContainer::ACONT_MINH + AContainer::ACONT_MAXH );
    m_file_percentage_text = (Text*)m_ac1->add( new Text( aguix, 0, 0, "" ),
                                                2, 0, AContainer::CO_FIX );

    m_tv = new TextView( aguix, 0, 0, 50, 50, "", *m_ts );
    m_tv->setFont( wconfig->getFont( 4 ).c_str() );
    m_tv->redraw();
    m_tv->setAcceptFocus( false );

    m_tv = (TextView*)m_cont->add( m_tv, 0, 1 );
    m_tv->show();

    parentlister->setActiveMode( this );
    setName();

    parentawindow->updateCont();
}

void TextViewMode::off()
{
    delete m_tv;
    m_tv = NULL;

    delete m_info_text;
    m_info_text = nullptr;

    delete m_file_name_text;
    m_file_name_text = nullptr;

    delete m_file_percentage_text;
    m_file_percentage_text = nullptr;

    parentlister->setContainer( NULL );
    delete m_cont;
    m_cont = NULL;

    parentlister->setActiveMode( NULL );
    parentlister->setName( "" );
    clearLastActiveFE();

    m_ts = NULL;
    m_lencalc = NULL;
}

void TextViewMode::activate()
{
}

void TextViewMode::deactivate()
{
}

bool TextViewMode::isType( const char *str )
{
    if ( strcmp( str, type ) == 0 ) return true; else return false;
}

const char *TextViewMode::getType()
{
    return type;
}

const char *TextViewMode::getStaticType()
{
    return type;
}


int TextViewMode::configure()
{
    return 0;
}

void TextViewMode::cyclicfunc( cyclicfunc_mode_t mode )
{
    update( false );
}

const char* TextViewMode::getLocaleName()
{
    return getStaticLocaleName();
}

const char* TextViewMode::getStaticLocaleName()
{
    return catalog.getLocale( 939 );
}

int TextViewMode::load()
{
    return 0;
}

bool TextViewMode::save( Datei *fh )
{
    return false;
}

void TextViewMode::setName()
{
    parentlister->setName( catalog.getLocale( 939 ) );
}

void TextViewMode::relayout()
{
    if ( m_tv != NULL ) {
        off();
        on();
    }
}

void TextViewMode::update( bool force )
{
    ListerMode *lm1 = NULL;
    Lister *ol = NULL;
    const FileEntry *fe;
  
    ol = parentlister->getWorker()->getOtherLister( parentlister );
    if ( ol != NULL ) {
        lm1 = ol->getActiveMode();
    }
  
    if ( force == true )
        clearLastActiveFE();
  
    if ( lm1 != NULL ) {
        std::list< NM_specialsourceExt > files;
    
        lm1->getSelFiles( files, ListerMode::LM_GETFILES_ONLYACTIVE );
        if ( ! files.empty() ) {
            fe = files.begin()->entry();
            if ( fe != NULL ) {
                if ( fe->equals( m_lastactivefe ) == false ) {
                    clearLastActiveFE();
                    m_lastactivefe = new FileEntry( *fe );
                    showFile();
                }
            }
        } else {
            if ( ol->isActive() == true ) {
                // only clear LV when other side is active and has no active fe
                if ( m_lastactivefe != NULL ) {
                    clearLastActiveFE();
                    showFile();
                }
            }
        }
    }
}

void TextViewMode::clearLastActiveFE() {
    if ( m_lastactivefe != NULL ) {
        delete m_lastactivefe;
        m_lastactivefe = NULL;
    }
}

void TextViewMode::showFile()
{
    if ( m_lastactivefe != NULL ) {
        m_ts->setFileName( m_lastactivefe->fullname, TVM_DEFAULT_SIZE );
        m_tv->textStorageChanged();
        m_file_name_text->setText( m_lastactivefe->name );
        setFilePercentage();
    }
}

void TextViewMode::command_up()
{
    m_tv->handleKeyMessage( XK_Up, 0 );
}

void TextViewMode::command_down()
{
    m_tv->handleKeyMessage( XK_Down, 0 );
}

void TextViewMode::command_first()
{
    m_tv->handleKeyMessage( XK_Home, 0 );
}

void TextViewMode::command_last()
{
    m_tv->handleKeyMessage( XK_End, 0 );
}

void TextViewMode::command_pageup()
{
    m_tv->handleKeyMessage( XK_Prior, 0 );
}

void TextViewMode::command_pagedown()
{
    m_tv->handleKeyMessage( XK_Next, 0 );
}

void TextViewMode::setFilePercentage()
{
    if ( m_lastactivefe == NULL ) return;
    double v = 0.0;

    if ( m_lastactivefe->isLink && ! m_lastactivefe->isCorrupt ) {
        v = m_lastactivefe->dsize();
    } else {
        v = m_lastactivefe->size();
    }

    if ( v > 0.0 ) {
        v = m_ts->getCurrentSize() / v;
    }

    if ( v < 0.0 ) v = 0.0;
    else if ( v > 1.0 ) v = 1.0;

    std::string p = AGUIXUtils::formatStringToString( "%3.0f %%", v * 100.0 );

    m_file_percentage_text->setText( p.c_str() );
    m_ac1->readLimits();
    parentawindow->updateCont();
}
