/* worker.h
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2012 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef WORKER_H
#define WORKER_H

#include "wdefines.h"
#include "functionproto.h"
#include <string>
#include "generic_callback.hh"
#include "layoutsettings.hh"
#include "timedtext.hh"
#include "processhandler.hh"
#include <map>
#include <aguix/message.h>
#include "hw_volume_manager.hh"
#include <memory>
#include "event_callbacks.hh"
#include "temporaryfilecache.hh"
#include "deeppathstore.hh"
#include <set>

class ListerMode;
class Lister;
class BookmarkDBProxy;
class PopUpMenu;
class HWVolume;
class WaitThread;
class TimeoutStore;
class EventQueue;
class Requester;
class Button;
class AWindow;
class AContainer;
class WCHotkey;
class WCButton;
class WCPath;
class PartSpace;
class CopyfileProgressCallback;
class PersDeepPathStore;
class HintDB;
class PathnameWatcher;
class Slider;

class Worker
{
public:
  Worker( int argc, char **argv, int optind = -1 );
  ~Worker();
  Worker( const Worker &other );
  Worker &operator=( const Worker &other );

  void run();
  static AGUIX *getAGUIX();
  static int getMaxModeNr();
  static int getID4Mode(ListerMode*);
  static const char *getNameOfMode( int nr );
  static const char *getLocaleNameOfMode( int nr );
  static int getID4Name( const char *str );
  ListerMode *getMode4ID(int,Lister *);
  AWindow *getMainWin();
  void setStatebarText(const char*);
  static int getNrOfCommands();
  static FunctionProto *getCommand4ID(int);
  static int getID4Command(FunctionProto*);
  Lister *getOtherLister(Lister*);
  int getSide(Lister*);
  Lister *getLister(int);
  Lister *getActiveLister();
  int interpret(List*,ActionMessage *);
  AWindow *getListerWin( Lister *l );
  
  int quit(int mode);
  void checkfirststart( char **setlang );
  void setTitle(const char *add_infos);
  
  static Requester *getRequester();
  void activateShortkeyFromList();

  int PS_readSpace( const char*name );
  
  loff_t PS_getBlocksize();
  loff_t PS_getFreeSpace();
  loff_t PS_getSpace();
  
  std::string PS_getFreeSpaceH() const;
  std::string PS_getSpaceH() const;
  void PS_setLifetime( double t );
  void setWaitCursor();
  void unsetWaitCursor();
  static const std::string getDataDir();
  
  int runCommand( const char *exestr,
		  const char *tmpname,
		  const char *tmpoutput,
		  bool inbackground,
                  RefCount< GenericCallbackArg< void, int > > pea );

  static std::string getWorkerConfigDir();

  void openWorkerMenu();

  int getStatebarWidth();

  static BookmarkDBProxy &getBookmarkDBInstance();

  void registerPopUpCallback( RefCount<GenericCallbackArg<void, AGMessage*> > cb,
                              const PopUpMenu *menu );
  void unregisterPopUpCallback( RefCount<GenericCallbackArg<void, AGMessage*> > cb );

  LayoutSettings getCustomLayout() const;
  bool getUseCustomLayout() const;
  void setCustomLayout( const LayoutSettings &nv );
  void unsetCustomLayout();

    void pushStatebarText( const TimedText &text );
    void updateStatebarText( const TimedText &text );
    void removeStatebarText( const TimedText &text );

    HWVolumeManager &getVolumeManager();
    void clearNewVolumesInfo();

    void registerTimeout( loff_t timeout_ms );
    void unregisterTimeout( loff_t timeout_ms );
    void wakeupMainLoop();

    void showCommandFailed( int status );

    void setIgnoreKey( KeySym key, bool state );
    bool isIgnoredKey( KeySym key ) const;

    void shuffleButton( int dir );
    void switchToButtonBank( int nr );

    EventCallbacks &getEventHandler();
    TemporaryFileCache::TemporaryFileCacheEntry getTempCopy( const std::string &filename,
                                                             CopyfileProgressCallback *progress_callback = NULL );

    DeepPathStore &getPathStore();

    void storePathPers( const std::string &path );
    void removePathPers( const std::string &path );
    std::list< std::string > getPathsPers();

    void copyToClipboard( const std::string &str );

    void showNextHint();

    RefCount< PathnameWatcher > getPathnameWatcher();
 protected:
  void createMainWin();
  void setupMainWin();
  void closeMainWin();
  void shufflePath();
  void shuffleButton( int dir, bool nocycle );
  void about();
  bool configure();
  void showPathBank();
  void showButtonBank();
  void setPath(int);
  void setPath( WCPath *);
  int activateShortkey(AGMessage*);
  void activateButton(int,int,int);
  int saveListerState();
  int initListerState();
  void updateTime();

  static AGUIX *aguix;
  static int workerInstances;
  Lister *lister[2];
  AWindow *mainwin;
  Button *aboutb,*configureb,*clockbar;
  Button *statebar;
  Button **pathbs;
  Button **buttons;
  Slider *button_bank_slider;
  AWindow *listerwin[2];
  AContainer *maincont;
  
  int w,h;
  unsigned int curRows,curColumns;
  int pbanknr,bbanknr;
  
  enum {WORKER_NORMAL,WORKER_QUIT};
  int runningmode;
  static int argc;
  static char **argv;
  static int m_optind;
  static std::string startcwd;
  time_t lasttimep;
  
  DNDMsg *dm;

  static Requester *req_static;
  
    static int shortkeysort( void *p1, void *p2, int sortmode );
  struct shortkeylisttype {WCHotkey *h;
                           WCButton *b;
                           WCPath *p;};
  PartSpace *freesp;
  bool isRoot;
  
  KeySym lastkey;
  unsigned int lastmod;
  Time lastkey_time;
  
  typedef struct _keyhash {
    WCButton *b;
    WCHotkey *h;
    WCPath *p;
    struct _keyhash *next;
  } keyhash;
  keyhash **keyhashtable;
  int keyhashtable_size;
  void buildtable();
  void deletetable();
  unsigned int keyhashfunc( KeySym k, unsigned int m );
  void insertkey( KeySym k, unsigned int m, WCButton *b, WCHotkey *h, WCPath *p );
  int findkey( KeySym k, unsigned int m, WCButton **b, WCHotkey **h, WCPath **p );
  int findkey( KeySym k1, unsigned int m1, KeySym k2, unsigned int m2, WCButton **b, WCHotkey **h, WCPath **p );
  bool isKeyChainPrefix( KeySym k, unsigned int m );
  int waitcursorcount;
  
  int getMaxButtonsWidth();

  std::string m_statebar_text;

  std::string queryLanguage();

  void createWorkerMenu();
  PopUpMenu *m_worker_menu;
  void startMenuAction( AGMessage *msg );
  void saveWorkerState();

  int m_interpret_recursion_depth;
  static const int m_interpret_max_recursion = 1000;

  static std::auto_ptr<BookmarkDBProxy> m_bookmarkproxy;

  std::map<const PopUpMenu *, std::list< RefCount<GenericCallbackArg<void, AGMessage*> > > > m_popup_callbacks;

  void handlePopUps( AGMessage *msg );

  bool m_use_custom_layout;
  LayoutSettings m_custom_layout;

    typedef enum {
        RS_RECONFIG,
        RS_RELAYOUT,
        RS_NONE
    } restart_t;
    restart_t restart;

    std::list< TimedText > m_statebar_texts;
    bool getValidStatebarText( TimedText &return_text );
    bool getNextValidStatebarText( TimedText &return_text );
    void updateStatebar();

    TimedText m_key_message;
    std::string getNamesOfKeyChainCompletions( KeySym k, unsigned int m );

    HWVolumeManager m_volume_manager;
    void checkNewVolumes();
    std::list< HWVolume > m_known_volumes;
    TimedText m_new_volumes_infotext;

    void updateVolumeManagerConfig();
    void initVolumeManagerStuff();

    void queryNewVolumeAction( const std::list< HWVolume > &l );

    void initWaitThread();
    void shutdownWaitThread();

    RefCount< EventQueue > m_eventqueue;
    RefCount< TimeoutStore > m_timeouts;
    RefCount< WaitThread > m_waitthread;

    loff_t m_current_clock_update_ms;
    void setClockUpdate( loff_t time_ms );

    loff_t m_current_dbus_update_ms;
    void setDbusUpdate( loff_t time_ms );

    loff_t m_current_statebar_update_ms;
    void setStatebarUpdate( loff_t time_ms );

    TemporaryFileCache m_temp_file_cache;
    ProcessHandler m_process_handler;
    void waitForEvent();

    struct worker_event_store {
        worker_event_store() : check_children( false ) {}
        bool check_children;
    } m_event_store;

    int buildShortkeyLV( FieldListView *lv,
                         List *sklist );

    std::map< KeySym, int > m_ignored_keys;

    void updateSettingsFromWConfig();

    EventCallbacks m_event_handler;
    std::map< EventCallbacks::event_callback_type_t, bool > m_events_seen;
    void fireSeenEvents();

    void updateHintDB();

    DeepPathStore m_path_store;
    std::auto_ptr< PersDeepPathStore > m_pers_path_store;

    std::string m_current_hint;
    time_t m_hint_time;
    RefCount< HintDB > m_hint_db;
    RefCount< PathnameWatcher > m_pathname_watcher;
    std::set< std::string > m_changed_paths;
};

#endif

/* Local Variables: */
/* mode:c++ */
/* End: */
