import { Spec } from 'vega';

export const spec: Spec = {
  $schema: 'https://vega.github.io/schema/vega/v5.json',
  width: 900,
  height: 560,
  padding: 0,
  autosize: 'none',

  signals: [
    {
      name: 'bandwidth',
      value: 20,
      bind: { input: 'range', min: -1, max: 100, step: 1 }
    },
    {
      name: 'cellSize',
      value: 4,
      bind: { input: 'select', options: [1, 2, 4, 8, 16, 32] }
    },
    {
      name: 'levels',
      value: 10,
      bind: { input: 'select', options: [1, 5, 10, 20] }
    }
  ],

  title: {
    text: 'Density of U.S. Airports, 2008',
    offset: -20
  },

  data: [
    {
      name: 'states',
      url: 'data/us-10m.json',
      format: { type: 'topojson', feature: 'states' },
      transform: [
        {
          type: 'geopath',
          projection: 'projection'
        }
      ]
    },
    {
      name: 'airports',
      url: 'data/airports.csv',
      format: { type: 'csv', parse: 'auto' },
      transform: [
        {
          type: 'geopoint',
          projection: 'projection',
          fields: ['longitude', 'latitude']
        },
        {
          type: 'filter',
          expr: 'datum.x != null && datum.y != null'
        }
      ]
    },
    {
      name: 'contours',
      source: 'airports',
      transform: [
        {
          type: 'kde2d',
          x: 'x',
          y: 'y',
          size: [{ signal: 'width' }, { signal: 'height' }],
          bandwidth: { signal: '[bandwidth, bandwidth]' },
          cellSize: { signal: 'cellSize' }
        },
        {
          type: 'isocontour',
          field: 'grid',
          levels: { signal: 'levels' },
          as: null
        }
      ]
    }
  ],

  projections: [
    {
      name: 'projection',
      type: 'albers',
      scale: 1150,
      translate: [{ signal: 'width / 2' }, { signal: 'height / 2' }]
    }
  ],

  scales: [
    {
      name: 'color',
      type: 'linear',
      domain: { data: 'contours', field: 'value' },
      range: { scheme: 'viridis' }
    }
  ],

  marks: [
    {
      type: 'path',
      clip: true,
      from: { data: 'states' },
      encode: {
        enter: {
          fill: { value: '#dedede' },
          stroke: { value: 'white' }
        },
        update: {
          path: { field: 'path' }
        }
      }
    },
    {
      type: 'symbol',
      clip: true,
      from: { data: 'airports' },
      encode: {
        enter: {
          size: { value: 10 },
          fill: { value: 'steelblue' },
          fillOpacity: { value: 0.8 },
          stroke: { value: 'white' },
          strokeWidth: { value: 0.5 }
        },
        update: {
          x: { field: 'x' },
          y: { field: 'y' }
        }
      }
    },
    {
      type: 'shape',
      clip: true,
      from: { data: 'contours' },
      encode: {
        enter: {
          stroke: { value: '#000' },
          fill: { scale: 'color', field: 'value' },
          fillOpacity: { value: 0.4 }
        },
        update: {
          strokeWidth: { value: 0 }
        },
        hover: {
          strokeWidth: { value: 1.5 }
        }
      },
      transform: [{ type: 'geoshape', field: 'datum' }]
    }
  ]
};
