/***************************************************************************
                   userlistmodel.h  -  User List Model header
                             -------------------
    begin                : Sun Nov 18 2007
    copyright            : (C) 2007 by Edward Sheldrake
    email                : ejs1920@yahoo.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef USERLISTMODEL_H
#define USERLISTMODEL_H

#include <QAbstractItemModel>
#include <QString>
#include <QPixmap>
#include <QList>
#include <QHash>
#include <QStringList>
#include <QRegExp>

#include <dclib/dcos.h>

/**
 * @author Edward Sheldrake - inspired by dc-qt 0.2.0 alpha
 * 
 * Using QSortFilterProxyModel for sorting and filtering
 * was far too CPU intensive, apparently.
 */

class UserListItem {

public:
	/** Constructor */
	UserListItem()
	{
		shared = 0;
		index = -1;
		sorttop = false;
		highlight = false;
	}
	/** Destructor */
	~UserListItem() {};
	
	/** Icon in user list */
	QPixmap pixmap;
	/** Nick */
	QString nick;
	/** Comment/Description */
	QString comment;
	/** Tag */
	QString tag;
	/** Speed/Connection */
	QString speed;
	/** Email address */
	QString email;
	/** IP address */
	QString ip;
	/** Lock/PK */
	QString lockpk;
	/** $Supports */
	QString supports;
	/** Lower cased nick for tab completion */
	QString nick_lc;
	/** Lower cased nick with [TAG] s removed for tab completion */
	QString nick_stripped;
	/** Lower cased comment for sorting */
	QString comment_lc;
	/** Lower cased tag for sorting */
	QString tag_lc;
	/** Lower cased speed for sorting */
	QString speed_lc;
	/** Lower cased email for sorting */
	QString email_lc;
	/** Share size in bytes */
	ulonglong shared;
	/**
	 * Position in list. Unfortunately index has to be updated
	 * when people quit, assumes updating indexes is faster than
	 * using QList::indexOf when updating existing users.
	 */
	int index;
	/** Sort on top (is operator) */
	bool sorttop;
	/** Highlight nick in red (unopened message) */
	bool highlight;
};

#define COLUMN_NICK     0
#define COLUMN_COMMENT  1
#define COLUMN_TAG      2
#define COLUMN_SPEED    3
#define COLUMN_EMAIL    4
#define COLUMN_SHARE    5
#define COLUMN_IP       6
#define COLUMN_LOCKPK   7
#define COLUMN_SUPPORTS 8

class UserListModel : public QAbstractItemModel {
Q_OBJECT

public:
	/** Constructor */
	UserListModel( QObject * parent = 0 );
	/** Destructor */
	virtual ~UserListModel();
	
	/** number of rows */
	virtual int rowCount( const QModelIndex & index = QModelIndex() ) const;
	/** number of columns */
	virtual int columnCount( const QModelIndex & index = QModelIndex() ) const;
	/** gets data */
	virtual QVariant data( const QModelIndex & index, int role = Qt::DisplayRole ) const;
	/** gets headings */
	virtual QVariant headerData( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;
	/** sort list */
	virtual void sort( int column, Qt::SortOrder order = Qt::AscendingOrder );
	/** */
	virtual QModelIndex index( int row, int column, const QModelIndex & parent = QModelIndex() ) const;
	/** */
	virtual QModelIndex parent( const QModelIndex & parent ) const;
	
	/** delete all data */
	void clear();
	/** Tests if the nick is in the list */
	bool hasNick( const QString & nick ) const;
	/** Remove a user from the list */
	void removeUser( const QString & nick );
	/** Add a user to the list */
	void addUser( const QString nick,
	              const QPixmap icon = QPixmap(),
		      const QString comment = "",
		      const QString tag = "",
		      const QString speed = "",
		      const QString email = "",
		      const ulonglong share = 0,
		      const QString ip = "",
		      const QString lockpk = "",
		      const QString supports = "" );
	/** Update icon */
	void updateIcon( const QString & nick, const QPixmap icon );
	/** Update comment */
	void updateComment( const QString & nick, const QString comment );
	/** Update tag */
	void updateTag( const QString & nick, const QString tag );
	/** Update speed */
	void updateSpeed( const QString & nick, const QString speed );
	/** Update email */
	void updateEmail( const QString & nick, const QString email );
	/** Update share */
	void updateShare( const QString & nick, const ulonglong share );
	/** Update IP address */
	void updateIP( const QString & nick, const QString ip );
	/** Update Lock/PK */
	void updateLockPK( const QString & nick, const QString lockpk );
	/** Update Supports */
	void updateSupports( const QString & nick, const QString supports );
	/** Set sorttop */
	void setSortTop( const QString & nick, const bool top );
	/** Set highlight */
	void setHighlight( const QString & nick, const bool highlight );
	/** Get index for nick */
	QModelIndex indexForNick( const QString & nick ) const;
	
	/** Get sort column */
	int getSortColumn() const { return sortColumn; };
	/** Get sort order */
	Qt::SortOrder getSortOrder() const { return sortOrder; };
	
	/** Get the list of nicks that match the given string
	 *  stripTags means anything inside [] will be removed before matching
	 *  e.g. [BBB]xyzzy = xyzzy, but the nicks returned will include any tags
	 */
	QStringList matchNicksContaining( const QString & part, bool stripTags = false ) const;
	/** The same, but using startsWith not contains */
	QStringList matchNicksStartingWith( const QString & part, bool stripTags = false ) const;

private:
	/** Hash of items keyed on nick */
	QHash<QString, UserListItem*> itemHash;
	/** Sorted list of items */
	QList<UserListItem*> itemList;
	/** Column sorted by */
	int sortColumn;
	/** Sort order */
	Qt::SortOrder sortOrder;
	/** regular expression to remove tags from nicks */
	QRegExp stripper;
};

#endif // USERLISTMODEL_H
