/*
 * Copyright (C) 2013-2014 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "shellserverconfiguration.h"

#include "displayconfigurationpolicy.h"
#include "unityprotobufservice.h"
#include "initialsurfaceplacementstrategy.h"
#include "serverstatuslistener.h"
#include "sessioncreator.h"
#include "sessionlistener.h"
#include "surfaceconfigurator.h"
#include "sessionauthorizer.h"
#include "focussetter.h"
#include "logging.h"

namespace mg = mir::graphics;
namespace msh = mir::shell;
namespace ms = mir::scene;

ShellServerConfiguration::ShellServerConfiguration(int argc, char const* argv[], QObject* parent)
    : QObject(parent)
    , DefaultServerConfiguration(argc, argv)
    , m_unityService(std::make_shared<UnityProtobufService>())
{
    DLOG("ShellServerConfiguration created");
}

ShellServerConfiguration::~ShellServerConfiguration()
{
}

std::shared_ptr<mg::DisplayConfigurationPolicy>
ShellServerConfiguration::the_display_configuration_policy()
{
    return display_configuration_policy(
        [this]
        {
            bool alpha = (qgetenv("XDG_SESSION_CLASS") == "greeter");
            return std::make_shared<DisplayConfigurationPolicy>(DefaultServerConfiguration::the_display_configuration_policy(), alpha);
        });
}

std::shared_ptr<msh::PlacementStrategy>
ShellServerConfiguration::the_shell_placement_strategy()
{
    return shell_placement_strategy(
        [this]
        {
            return std::make_shared<InitialSurfacePlacementStrategy>(the_shell_display_layout());
        });
}

std::shared_ptr<msh::SessionListener>
ShellServerConfiguration::the_shell_session_listener()
{
    return shell_session_listener(
        [this]
        {
            return std::make_shared<SessionListener>();
        });
}

std::shared_ptr<ms::SurfaceConfigurator>
ShellServerConfiguration::the_surface_configurator()
{
    return surface_configurator(
        [this]()
        {
            return std::make_shared<SurfaceConfigurator>();
        });
}

std::shared_ptr<mir::frontend::SessionAuthorizer>
ShellServerConfiguration::the_session_authorizer()
{
    return session_authorizer(
    []
    {
        return std::make_shared<SessionAuthorizer>();
    });
}

std::shared_ptr<mir::frontend::SessionCreator>
ShellServerConfiguration::the_session_creator()
{
    return session_creator([this]
        {
            return std::make_shared<SessionCreator>(
                m_unityService,
                the_ipc_factory(the_frontend_shell(), the_buffer_allocator()),
                the_session_authorizer(),
                the_message_processor_report());
        });
}

std::shared_ptr<msh::FocusSetter>
ShellServerConfiguration::the_shell_focus_setter()
{
    return shell_focus_setter(
       [this]
       {
           return std::make_shared<FocusSetter>(DefaultServerConfiguration::the_shell_focus_setter(), the_input_targeter());
       });
}

std::shared_ptr<mir::ServerStatusListener>
ShellServerConfiguration::the_server_status_listener()
{
    return server_status_listener(
        []()
        {
            return std::make_shared<ServerStatusListener>();
        });
}

/************************************ Shell side ************************************/

//
// Note about the
//     if (sharedPtr.unique()) return 0;
// constructs used in the functions below.
// The rationale is that if when you do
//     the_session_authorizer()
// get a pointer that is unique means that Mir is not
// holding the pointer and thus when we return from the 
//     sessionAuthorizer()
// scope the unique pointer will be destroyed so we return 0
//

InitialSurfacePlacementStrategy *ShellServerConfiguration::placementStrategy()
{
    auto sharedPtr = the_shell_placement_strategy();
    if (sharedPtr.unique()) return 0;

    return static_cast<InitialSurfacePlacementStrategy*>(sharedPtr.get());
}

SessionAuthorizer *ShellServerConfiguration::sessionAuthorizer()
{
    auto sharedPtr = the_session_authorizer();
    if (sharedPtr.unique()) return 0;

    return static_cast<SessionAuthorizer*>(sharedPtr.get());
}

SessionListener *ShellServerConfiguration::sessionListener()
{
    auto sharedPtr = the_shell_session_listener();
    if (sharedPtr.unique()) return 0;

    return static_cast<SessionListener*>(sharedPtr.get());
}

SurfaceConfigurator *ShellServerConfiguration::surfaceConfigurator()
{
    auto sharedPtr = the_surface_configurator();
    if (sharedPtr.unique()) return 0;

    return static_cast<SurfaceConfigurator*>(sharedPtr.get());
}

FocusSetter *ShellServerConfiguration::focusSetter()
{
    auto sharedPtr = the_shell_focus_setter();
    if (sharedPtr.unique()) return 0;

    return static_cast<FocusSetter*>(sharedPtr.get());
}
