/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License version 3, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
 * SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef APPLICATIONMANAGER_H
#define APPLICATIONMANAGER_H

// local
#include "desktopfilereader.h"

// std
#include <memory>

// Qt
#include <QObject>
#include <QSharedPointer>
#include <QStringList>

// Unity API
#include <unity/shell/application/ApplicationManagerInterface.h>

// local
#include "application.h"

namespace mir {
    namespace geometry {
        class Size;
    }
    namespace shell {
        class Session;
        class Surface;
        class FocusController;
    }
}

class ShellServerConfiguration;

namespace unitymir
{
class DBusWindowStack;
class MirSurfaceManager;
class TaskController;
class ProcInfo;

class ApplicationManager : public unity::shell::application::ApplicationManagerInterface
{
    Q_OBJECT
    Q_FLAGS(ExecFlags)

public:
    class Factory
    {
    public:
        ApplicationManager* create();
    };
    // Mapping enums to Ubuntu Platform API enums.
    enum Flag {
        NoFlag = 0x0,
        ForceMainStage = 0x1,
    };
    Q_DECLARE_FLAGS(ExecFlags, Flag)

    static ApplicationManager* singleton();

    explicit ApplicationManager(const QSharedPointer<TaskController>& taskController,
                                const QSharedPointer<DesktopFileReader::Factory>& desktopFileReaderFactory,
                                const QSharedPointer<ProcInfo>& processInfo,
                                std::shared_ptr<mir::shell::FocusController> const& controller,
                                QSize const& displaySize,
                                QObject *parent = 0);
    virtual ~ApplicationManager();

    // ApplicationManagerInterface
    QString focusedApplicationId() const override;
    bool suspended() const;
    void setSuspended(bool suspended);
    Q_INVOKABLE unitymir::Application* get(int index) const override;
    Q_INVOKABLE unitymir::Application* findApplication(const QString &appId) const override;
    Q_INVOKABLE bool requestFocusApplication(const QString &appId) override;
    Q_INVOKABLE bool focusApplication(const QString &appId) override;
    Q_INVOKABLE void unfocusCurrentApplication() override;
    Q_INVOKABLE unitymir::Application* startApplication(const QString &appId, const QStringList &arguments) override;
    Q_INVOKABLE bool stopApplication(const QString &appId) override;
    Q_INVOKABLE bool updateScreenshot(const QString &appId);

    // QAbstractListModel
    int rowCount(const QModelIndex & parent = QModelIndex()) const override;
    QVariant data(const QModelIndex & index, int role = Qt::DisplayRole) const override;

    Q_INVOKABLE unitymir::Application *startApplication(const QString &appId, ExecFlags flags,
                                                        const QStringList &arguments = QStringList());
    Q_INVOKABLE void move(int from, int to);

    const QList<Application*> &list() const { return m_applications; }
    unitymir::Application* findApplicationWithPid(const qint64 pid);

    // Internal helpers
    void suspendApplication(Application *application);
    void resumeApplication(Application *application);
    int panelHeight();
    QSize displaySize() const { return m_displaySize; }

public Q_SLOTS:
    void authorizeSession(const quint64 pid, bool &authorized);
    void placeSession(mir::shell::Session const*, uint32_t &x, uint32_t &y);
    
    void onSessionStarting(std::shared_ptr<mir::shell::Session> const& session);
    void onSessionStopping(std::shared_ptr<mir::shell::Session> const& session);
    void onSessionFocused(std::shared_ptr<mir::shell::Session> const& session);
    void onSessionUnfocused();

    void onSessionCreatedSurface(mir::shell::Session const*, std::shared_ptr<mir::shell::Surface> const&);

    void onProcessStartReportReceived(const QString& appId, const bool failure);
    void onProcessStopped(const QString& appId, const bool unexpected);
    void onFocusRequested(const QString& appId);
    void onResumeRequested(const QString& appId);

Q_SIGNALS:
    void focusRequested(const QString &appId);

private Q_SLOTS:
    void screenshotUpdated();

private:
    void setFocused(Application *application);
    void add(Application *application);
    void remove(Application* application);
    Application* findApplicationWithSession(const std::shared_ptr<mir::shell::Session> &session);
    Application* findApplicationWithSession(const mir::shell::Session *session);
    Application* applicationForStage(Application::Stage stage);
    QModelIndex findIndex(Application* application);
    bool checkFocusOnRemovedApplication(Application* application);
    void shutdownApplication(Application* application);
    void stopStartingApplication(const QString &appId);

    QList<Application*> m_applications;
    Application* m_focusedApplication; // remove as Mir has API for this
    Application* m_mainStageApplication;
    Application* m_sideStageApplication;
    Application* m_msApplicationToBeFocused; // placeholder store for async focusing
    Application* m_ssApplicationToBeFocused; // placeholder store for async focusing
    QStringList m_lifecycleExceptions;
    std::shared_ptr<mir::shell::FocusController> m_focusController;
    DBusWindowStack* m_dbusWindowStack;
    QSharedPointer<TaskController> m_taskController;
    QSharedPointer<DesktopFileReader::Factory> m_desktopFileReaderFactory;
    QSharedPointer<ProcInfo> m_procInfo;
    int m_gridUnitPx;
    bool m_fenceNext;
    QString m_nextFocusedAppId;
    QSize m_displaySize;
    int m_panelHeight;
    bool m_suspended;

    friend class DBusWindowStack;
    friend class MirSurfaceManager;
};

} // namespace unitymir

Q_DECLARE_METATYPE(unitymir::ApplicationManager*)

#endif // APPLICATIONMANAGER_H
