/*
 * Copyright 2014 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <stdlib.h>

#include <QObject>
#include <QProcessEnvironment>
#include <QDebug>
#include <QVariant>
#include <QUrlQuery>
#include <QString>
#include <QDBusConnection>
#include <QTest>
#include <QSignalSpy>

#include "../ubuntupurchase.h"
#include "../purchase_service.h"
#include "fakessoservice.h"

using namespace UbuntuPurchases;

class TestUbuntuPurchase : public QObject
{
    Q_OBJECT

private slots:
    void testPurchaseUrlEnvironmentVariable();
    void testLaunchService();
    void testCredentialsInvalid();
    void testPurchaseItem();
    void testPurchaseItemDuringPurchase();
    void testPurchaseSuccessed();
    void testPurchaseFailed();
};

void TestUbuntuPurchase::testPurchaseUrlEnvironmentVariable()
{
    char* envVar = getenv("PURCHASE_URL");
    setenv("PURCHASE_URL", "success.html", 1);
    UbuntuPurchase purchase;
    QString url = purchase.getPurchaseUrl();
    QCOMPARE(url, QString("success.html"));
    setenv("PURCHASE_URL", envVar, 1);
}

void TestUbuntuPurchase::testLaunchService()
{
    UbuntuPurchase purchase;
    QVERIFY(purchase.getPurchaseService() == nullptr);
    QSignalSpy spy(&purchase, SIGNAL(serviceRegistrationFailed()));
    purchase.launchService();
    QVERIFY(purchase.getPurchaseService() != nullptr);
    QTRY_COMPARE(spy.count(), 0);
}

void TestUbuntuPurchase::testCredentialsInvalid()
{
    UbuntuPurchase purchase;
    FakeSsoService& service = purchase.getFakeService();
    service.valid = false;
    QSignalSpy spy(&service, SIGNAL(credentialsNotFound()));
    purchase.launchService();
    QTRY_COMPARE(spy.count(), 1);
}

void TestUbuntuPurchase::testPurchaseItem()
{
    UbuntuPurchase purchase;
    QSignalSpy spy(&purchase, SIGNAL(urlChanged()));
    purchase.launchService();
    PurchaseService* service = purchase.getPurchaseService();
    QVariantMap map;
    QString packageName("purchase.test");
    map["package_name"] = QVariant(packageName);
    service->purchase(map);
    QTRY_COMPARE(spy.count(), 1);
    service->deleteLater();
}

void TestUbuntuPurchase::testPurchaseItemDuringPurchase()
{
    UbuntuPurchase purchase;
    QSignalSpy spy(&purchase, SIGNAL(urlChanged()));
    purchase.launchService();
    PurchaseService* service = purchase.getPurchaseService();
    QSignalSpy spyService(service, SIGNAL(failure(const QString&)));
    QVariantMap map;
    QString packageName("purchase.test");
    map["package_name"] = QVariant(packageName);
    service->purchase(map);
    QTRY_COMPARE(spy.count(), 1);

    QString packageName2("purchase2.test2");
    map["package_name"] = QVariant(packageName2);
    service->purchase(map);
    QTRY_COMPARE(spyService.count(), 1);
    QList<QVariant> arguments = spyService.takeFirst();
    QTRY_COMPARE(arguments.at(0).toString(), packageName);
    service->deleteLater();
}

void TestUbuntuPurchase::testPurchaseSuccessed()
{
    UbuntuPurchase purchase;
    QSignalSpy spy(&purchase, SIGNAL(urlChanged()));
    purchase.launchService();
    PurchaseService* service = purchase.getPurchaseService();
    QSignalSpy spyService(service, SIGNAL(success(const QString&)));
    QVariantMap map;
    QString packageName("purchase.test");
    map["package_name"] = QVariant(packageName);
    service->purchase(map);
    purchase.purchaseSucceeded();
    QTRY_COMPARE(spyService.count(), 1);
    QList<QVariant> arguments = spyService.takeFirst();
    QTRY_COMPARE(arguments.at(0).toString(), packageName);
    service->deleteLater();
}

void TestUbuntuPurchase::testPurchaseFailed()
{
    UbuntuPurchase purchase;
    QSignalSpy spy(&purchase, SIGNAL(urlChanged()));
    purchase.launchService();
    PurchaseService* service = purchase.getPurchaseService();
    QSignalSpy spyService(service, SIGNAL(failure(const QString&)));
    QVariantMap map;
    QString packageName("purchase.test");
    map["package_name"] = QVariant(packageName);
    service->purchase(map);
    purchase.purchaseCanceled();
    QTRY_COMPARE(spyService.count(), 1);
    QList<QVariant> arguments = spyService.takeFirst();
    QTRY_COMPARE(arguments.at(0).toString(), packageName);
    service->deleteLater();
}

QTEST_MAIN(TestUbuntuPurchase)
#include "test_ubuntupurchase.moc"
