/*
	SuperCollider real time audio synthesis system
 Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 */

//Nick Collins begun 20 May 2005, corrections up till 6/6/2006; adapted for new SC FFT scheme, 25 March 2011
//trying to implement the best onset detection algo from AES118 paper, with event analysis data to be written to a buffer
//for potential NRT and RT use

//stores up to a second of audio, assumes that events are not longer than that minus a few FFT frames
//assumes 44100 SR and FFT of 1024, 512 overlap

//inputs for pitch tracker

//after collecting events, will analyse some timbral qualities (spec centroid, rolloff, temp centroid, zcr) (drum classifier), PAT (energy integration algorithm),
//loudness of event, envelope attacktime, release time etc,  pitch envelope (median pitch)
//heuristics on event selection (reject right skewed, multihit)
//need to combine with an SC3 GUI that shows the events being collected?




#include "SC_PlugIn.h"
#include "SC_fftlib.h"
#include <stdio.h>

//helpful constants
//#define PI 3.1415926535898f
//#define TWOPI 6.28318530717952646f

//WARNING- if N was changed here, the Hanning window is pre created for 1024 samples, hanning1024, you'd need to change that too, and also as refed in the files
//FFT data
#define N 1024  //FFT size
#define NOVER2 512  //FFT size
#define NOVER4 256  //FFT size
#define OVERLAP 512
#define OVERLAPINDEX 512
#define HOPSIZE 512
#define FS 44100 //assumes fixed sampling rate
#define FRAMESR 86.1328
#define FRAMEPERIOD 0.01161 //seconds


#define NUMERBBANDS 40
#define PASTERBBANDS 3
//3 usually, but time resolution improved if made 1?

//in FFT frames
#define MAXEVENTDUR 80
#define MINEVENTDUR 3
//4 or maybe 2

#define CIRCBUFSIZE 15

//7 frames is about 40 mS
//peak picker will use 3 back, 3 forward
#define DFFRAMESSTORED 7

#define MAXBLOCKSIZE 64
#define MAXBLOCKS 700

//MAXEVENTDUR+20 for safety
#define LOUDNESSSTORED 100





int eqlbandbins[43]= {1,2,3,4,5,6,7,8,9,11,13,15,17,19,22,25,28,32,36,41,46,52,58,65,73,82,92,103,116,129,144,161,180,201,225,251,280,312,348,388,433,483,513};
int eqlbandsizes[42]= {1,1,1,1,1,1,1,1,2,2,2,2,2,3,3,3,4,4,5,5,6,6,7,8,9,10,11,13,13,15,17,19,21,24,26,29,32,36,40,45,50,29};
float contours[42][11]= {{ 47.88, 59.68, 68.55, 75.48, 81.71, 87.54, 93.24, 98.84,104.44,109.94,115.31},{ 29.04, 41.78, 51.98, 60.18, 67.51, 74.54, 81.34, 87.97, 94.61,101.21,107.74},{ 20.72, 32.83, 43.44, 52.18, 60.24, 67.89, 75.34, 82.70, 89.97, 97.23,104.49},{ 15.87, 27.14, 37.84, 46.94, 55.44, 63.57, 71.51, 79.34, 87.14, 94.97,102.37},{ 12.64, 23.24, 33.91, 43.27, 52.07, 60.57, 68.87, 77.10, 85.24, 93.44,100.90},{ 10.31, 20.43, 31.03, 40.54, 49.59, 58.33, 66.89, 75.43, 83.89, 92.34,100.80},{  8.51, 18.23, 28.83, 38.41, 47.65, 56.59, 65.42, 74.16, 82.89, 91.61,100.33},{  7.14, 16.55, 27.11, 36.79, 46.16, 55.27, 64.29, 73.24, 82.15, 91.06, 99.97},{  5.52, 14.58, 25.07, 34.88, 44.40, 53.73, 62.95, 72.18, 81.31, 90.44, 99.57},{  3.98, 12.69, 23.10, 32.99, 42.69, 52.27, 61.66, 71.15, 80.54, 89.93, 99.31},{  2.99, 11.43, 21.76, 31.73, 41.49, 51.22, 60.88, 70.51, 80.11, 89.70, 99.30},{  2.35, 10.58, 20.83, 30.86, 40.68, 50.51, 60.33, 70.08, 79.83, 89.58, 99.32},{  2.05, 10.12, 20.27, 30.35, 40.22, 50.10, 59.97, 69.82, 79.67, 89.52, 99.38},{  2.00,  9.93, 20.00, 30.07, 40.00, 49.93, 59.87, 69.80, 79.73, 89.67, 99.60},{  2.19, 10.00, 20.00, 30.00, 40.00, 50.00, 59.99, 69.99, 79.98, 89.98, 99.97},{  2.71, 10.56, 20.61, 30.71, 40.76, 50.81, 60.86, 70.96, 81.01, 91.06,101.17},{  3.11, 11.05, 21.19, 31.41, 41.53, 51.64, 61.75, 71.95, 82.05, 92.15,102.33},{  2.39, 10.69, 21.14, 31.52, 41.73, 51.95, 62.11, 72.31, 82.46, 92.56,102.59},{  1.50, 10.11, 20.82, 31.32, 41.62, 51.92, 62.12, 72.32, 82.52, 92.63,102.56},{ -0.17,  8.50, 19.27, 29.77, 40.07, 50.37, 60.57, 70.77, 80.97, 91.13,101.23},{ -1.80,  6.96, 17.77, 28.29, 38.61, 48.91, 59.13, 69.33, 79.53, 89.71, 99.86},{ -3.42,  5.49, 16.36, 26.94, 37.31, 47.61, 57.88, 68.08, 78.28, 88.41, 98.39},{ -4.73,  4.38, 15.34, 25.99, 36.39, 46.71, 57.01, 67.21, 77.41, 87.51, 97.41},{ -5.73,  3.63, 14.74, 25.48, 35.88, 46.26, 56.56, 66.76, 76.96, 87.06, 96.96},{ -6.24,  3.33, 14.59, 25.39, 35.84, 46.22, 56.52, 66.72, 76.92, 87.04, 97.00},{ -6.09,  3.62, 15.03, 25.83, 36.37, 46.70, 57.00, 67.20, 77.40, 87.57, 97.68},{ -5.32,  4.44, 15.90, 26.70, 37.28, 47.60, 57.90, 68.10, 78.30, 88.52, 98.78},{ -3.49,  6.17, 17.52, 28.32, 38.85, 49.22, 59.52, 69.72, 79.92, 90.20,100.61},{ -0.81,  8.58, 19.73, 30.44, 40.90, 51.24, 61.52, 71.69, 81.87, 92.15,102.63},{  2.91, 11.82, 22.64, 33.17, 43.53, 53.73, 63.96, 74.09, 84.22, 94.45,104.89},{  6.68, 15.19, 25.71, 36.03, 46.25, 56.31, 66.45, 76.49, 86.54, 96.72,107.15},{ 10.43, 18.65, 28.94, 39.02, 49.01, 58.98, 68.93, 78.78, 88.69, 98.83,109.36},{ 13.56, 21.65, 31.78, 41.68, 51.45, 61.31, 71.07, 80.73, 90.48,100.51,111.01},{ 14.36, 22.91, 33.19, 43.09, 52.71, 62.37, 71.92, 81.38, 90.88,100.56,110.56},{ 15.06, 23.90, 34.23, 44.05, 53.48, 62.90, 72.21, 81.43, 90.65, 99.93,109.34},{ 15.36, 23.90, 33.89, 43.31, 52.40, 61.42, 70.29, 79.18, 88.00, 96.69,105.17},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70}};
double phons[11]={2,10,20,30,40,50,60,70,80,90,100};


//neural net parameters for PAT calculation, 0.0043 resolution

//old version assumes different EQLC scale factor
//double g_w1[400]= {-0.118610779474,-0.170817684721,-0.119164033256,-0.100108317910,-0.040417536101,0.020409273308,0.041689123871,0.071926208877,0.164451143421,0.139254603697,-0.041753979149,-0.093713883235,-0.116440604262,-0.096305004652,-0.128678445100,-0.086024774724,-0.069462915638,-0.036968824239,-0.007796853337,-0.134534424350,0.067885650332,0.135015346356,0.052758976553,0.042341966210,0.065868771326,-0.009011066474,0.014837120779,0.061771362393,0.041663650157,0.095907922780,-0.000458829635,0.038953549306,0.048360628921,0.058948793471,0.071337856165,0.020454814874,0.031191177071,0.045236400339,0.045882275620,0.058001662636,-0.096734290018,-0.097425187263,-0.053764032805,-0.064210900983,-0.067308382202,-0.056732809711,-0.036159598509,-0.036437475671,-0.080006298351,-0.055264721080,-0.025329518579,-0.064782020699,-0.040640754244,-0.051948060694,-0.062899273974,-0.050565552768,-0.061360948522,-0.047927235867,-0.037249965342,-0.024179710052,-0.005044054452,-0.154402190875,-0.038305879172,-0.049028133626,-0.041570648927,-0.066593333819,-0.071287774049,-0.047691447062,-0.038192354992,-0.082147139154,0.041166245100,-0.103317581905,-0.042353217150,-0.086915583510,-0.055569633496,-0.058515731544,-0.013995377097,-0.022700815927,-0.027467870259,-0.075464562294,0.081528746356,0.069692932735,0.053838822771,0.020740194222,0.014039791907,0.053875916736,0.083343839728,0.043070097383,0.070651901593,0.020017807419,0.051759486528,0.099413022196,0.056698205150,0.028256899542,0.065267665535,0.031744600176,0.060242767520,0.064234014120,0.071190971685,0.070918322664,0.127020819208,0.083168667022,0.057746279897,0.087451609622,0.068215025827,0.038213362407,0.037288785223,0.053566539916,0.034996750991,0.084059914876,0.112474489043,0.034706550728,0.066387218635,0.037421571853,0.066987776237,-0.001391974614,0.039394153946,0.042568478823,0.040948377170,0.048725747505,-0.070517114675,-0.026577594654,0.010149265558,0.136702386813,0.097378349755,-0.020973887708,-0.079637452820,-0.109204722828,-0.113585751891,-0.124848830035,-0.165800077230,-0.058836722280,-0.079985962119,0.044940136833,0.034266940715,-0.041385507565,-0.049014366709,-0.032668130789,-0.064278927025,-0.037478704443,0.072716158868,0.237924182122,0.064412241310,-0.136275196482,-0.067491072526,-0.150915276586,-0.185790814410,-0.178224271020,-0.083577680971,0.109426927037,0.679280614975,0.641085289107,-0.029472026054,0.045724934527,0.001688729506,-0.011396000031,0.207583591454,0.108705081001,-0.101242336452,-0.185909920922,0.046945521642,0.113627660898,0.040796850967,0.057426569249,0.048876443103,0.039638115761,0.074702579171,0.067586134440,0.053559590895,0.025436936532,0.041395916921,0.085124439462,0.060805388959,0.058216961434,0.084874521479,0.065389708423,0.061751661568,-0.001739631791,0.030078002548,0.025307466546,-0.096917779328,-0.057546223080,-0.004721197578,0.006792452221,-0.014566193218,-0.030985773924,-0.054628232677,-0.054394505228,-0.052954923836,-0.081074246543,-0.083091836429,-0.083599195649,-0.092022385983,-0.038783716299,-0.042534435461,-0.080434093414,-0.078825737968,-0.053534976809,-0.077151682045,-0.017580398340,-0.064376439978,-0.149698139266,0.287316773204,0.137982575307,-0.011872716575,-0.334923524535,-0.383183204294,-0.029512653257,0.230301909477,0.121106970736,-0.001646687700,-0.091753255067,-0.012309672527,-0.001879586667,0.151363795274,0.131238523169,0.127772554585,-0.111044951301,0.002158379507,0.071943843270,0.098498361638,0.026380306781,-0.010059138351,-0.168801846500,-0.067880561110,0.051992081093,0.060579343629,0.118090997119,0.117811316154,0.083172259619,0.112908988997,0.093315124683,0.030288717350,-0.021283756043,0.033371330227,-0.000483464840,0.064302075304,0.074128295379,0.038514478094,0.018018591019,0.048852390112,0.099310317274,0.040171666458,0.047708308877,0.083972696922,0.034389105251,0.040069240321,0.076759058245,0.068751881249,0.024474418097,0.044981282114,0.088854721470,0.042125222600,0.063975838910,0.074879186455,0.073919339836,0.072570204043,0.026865415850,0.031941478051,0.052561945454,0.780115079880,-0.159538221083,0.049398028876,-0.084842981317,-0.118927379018,-0.133338755086,-0.058336326390,-0.000510146104,0.062392548881,0.014010424863,-0.780427557563,-0.712758785989,-0.177041435549,-0.170572470172,-0.085042816927,-0.053874539521,-0.046551379067,-0.202087860708,-0.138587930587,-0.183218817819,-0.080912001234,-0.395892335571,-0.139534927376,0.085052849460,0.231095314428,0.198965381039,0.099552691916,0.136478444450,0.070049213068,-0.067613968697,0.774668365045,0.259014882643,-0.397985837234,-0.144000450669,-0.247856469434,-0.408715125434,-0.339523363662,-0.298642020657,-0.270291578268,-0.313594464656,0.106676025487,0.118519930918,0.061298309258,-0.013486804726,-0.010481123664,0.052346123856,0.044812910166,0.072060612223,0.043896378370,0.064348851164,0.068695326095,0.060354722203,0.071054626028,0.051117395237,0.031668547690,0.074893003076,0.035271930131,0.053597836040,0.100957345868,0.049149566211,-0.559516802728,-0.091763723058,-0.129750588792,-0.396810120107,-0.218880070182,0.402335704374,0.252571924910,0.400254853585,0.320063373726,0.197586548305,-0.336313130366,0.202048843800,-0.071674304462,-0.238016477132,-0.088366247617,-0.032383371198,-0.115038894842,-0.177971153688,0.111844958835,0.176177409394,0.068039234277,0.161874693128,-0.253701231253,0.037926939035,-0.041709919831,0.025159725891,0.216559218453,0.099499349230,-0.087346039445,-0.102549395325,-0.029408243562,-0.209114435985,0.220668097485,0.211423959917,-0.001339933196,-0.076932961570,-0.103313659628,-0.035746852114,-0.102290184476,-0.072568478632,0.098296383872,-0.021047408957,-0.077424671613,-0.160852764496,-0.104783726126,0.040854151967,0.088588605923,0.139302909261,0.160567000139,0.142224963442,0.130795950546,0.052291969161,0.025784862139,0.016969687686,0.002709508882,0.042093361373,0.037541675942,-0.007924964867,0.023769675572,0.033858363847,-0.235229124602,0.199212256840,0.339894667300,0.340100521356,0.115055392513,-0.079648080968,-0.059569439140,0.079632113951,0.128498770695,0.092590837044,-0.328565319308,-0.057400729913,-0.179942250812,-0.116612335111,-0.061804497353,-0.104433694324,-0.224486610807,-0.216826901022,-0.182113811235,-0.095373235796};
//double g_b1[20]= {-0.087109526473,0.061784635821,-0.035138042476,-0.046833752274,0.045715481997,0.039081186131,-0.046072321113,-0.001661182773,0.045956486259,-0.019434086053,0.108479802408,0.046105538362,0.025364336754,-0.019241634358,-0.173442390885,0.058962978831,0.144845572064,-0.064951962362,0.056012829345,-0.082176639289};
//double g_w2[20]= {0.558852397707,-0.327553374995,0.351196265964,0.395592659092,-0.382644890761,-0.378087277326,0.466726852297,1.262998521769,-0.363299479672,0.409136168813,0.964797438627,-0.444114574473,-0.375677602104,-1.925584103148,1.220972040059,-0.467434811122,1.234394744996,-0.746502604568,-0.471142863786,0.713632981946};
//double g_b2= -0.430787264163;

double g_w1[400]= {0.169018409381,-0.103493979969,-0.081116239353,-0.160328665532,-0.128713559310,0.014571096150,0.069552903670,-0.058197162306,0.052340592299,0.111824086091,-0.025717895092,-0.124286393433,-0.026262382825,0.072389504928,0.028372373044,0.144211708547,0.199830906453,-0.049243016495,-0.147333426272,-0.088029165875,-0.074946527033,-0.007965873624,-0.060110030215,0.008834727279,-0.156231390219,-0.085683545360,0.008492678650,-0.159949899674,-0.115555899263,-0.080099619808,-0.128693050829,-0.113778770008,0.016469358668,0.097956704279,-0.103695131325,-0.002259095299,-0.030644717178,-0.044971099919,0.020399047016,0.047702968632,-0.047467968609,0.041941534740,0.156571285773,0.057382560846,-0.135252730035,-0.034127350934,0.067499424529,-0.011145953811,-0.128525224891,-0.142349903465,0.044215889062,-0.097641325069,-0.012062376340,0.061028131803,-0.084440296037,0.118762527354,0.035625385936,-0.052761306816,0.083340338280,0.016820102886,0.118756976133,0.156481209290,0.048452337716,-0.026368537976,0.191175739737,0.090013084193,0.018696505038,0.170542377524,-0.001512989100,0.057630147505,-0.016946364911,0.065383593382,0.072932448268,0.104541875034,-0.019735604806,0.067085430718,0.000128574589,0.063464510147,0.003788134598,0.101073925412,-0.087316625343,-0.149529087030,-0.111335411030,-0.116145801886,-0.060641155003,0.005561622688,-0.091556244543,-0.132321264796,-0.076291069776,-0.003335762223,0.026246943030,-0.011468310694,-0.035126329274,-0.106070763339,-0.124952577836,-0.078404830643,-0.151453814283,0.024714830242,-0.060650252590,-0.173506441345,0.106715458765,0.148569973333,0.050213107815,-0.028839049370,0.114161644933,0.096331373589,0.104805842678,0.010486378719,0.121458838874,0.107423837956,0.044979074940,0.072597542358,0.160271775771,-0.038145666333,0.005730798943,0.034979938030,0.041970697009,0.030010293321,0.006684817099,0.144376773042,-0.196017702758,-0.206756385022,-0.252082528559,-0.219553900284,-0.130020241730,0.081204320949,0.156489208353,0.134073280292,0.098427970600,0.126081116444,-0.075290344756,-0.145776698224,-0.142540936409,-0.110823148024,-0.231334227757,-0.052702104975,-0.092946092996,-0.177032510672,-0.229982162386,-0.218378920091,0.135871375078,0.183216794416,0.146292678909,0.148135982867,-0.194096736968,-0.021703591884,0.045094859505,0.017863475027,0.006859545887,0.156610521746,-0.066816447209,-0.180997519162,-0.081371379224,-0.047287897606,-0.041289118989,-0.029090055377,-0.068382662951,-0.182740513910,-0.225168590100,-0.346868302372,0.199722873299,0.033087851452,-0.030891310093,0.078941246859,-0.072809955360,-0.002809553644,-0.036251400513,0.005217056949,-0.045240969118,-0.066073411154,-0.034754025006,0.043126633614,0.028326641466,-0.065083329424,-0.067367524160,-0.069097857445,-0.169247770758,-0.136500768908,-0.100908391450,-0.152800779914,-0.110329704724,-0.058330470336,-0.166222175735,-0.071681883178,-0.011907413141,-0.047213089912,-0.136494746120,-0.084312350326,-0.198660386406,-0.136450188336,0.048112641993,-0.068486378808,-0.039060020579,-0.042321146507,0.049584715285,-0.123861818379,-0.044999436569,-0.065258338307,-0.003089607861,-0.162330353743,-0.746319165643,-0.418601928817,0.249261951313,0.317976941227,0.113231249221,-0.102109335755,0.009560254731,0.424416959626,0.881639118216,0.218126791608,0.137498811524,-0.268876999451,-0.295536693417,0.020876071932,-0.070878958336,-0.114121717264,-0.055836654847,-0.250013900652,-0.043558404755,-0.084505118617,-0.170519547559,-0.187825898822,-0.027093287996,-0.046528918189,-0.125933308259,-0.067393378835,-0.073133385306,-0.050993869028,-0.121218246260,-0.130702551958,0.070077728670,-0.103564384986,-0.111554191177,0.020772172226,-0.089629351242,0.014795308167,0.006658415369,0.055715320230,0.055557287457,-0.026772862197,-0.148064743706,-0.109219809324,-0.114094896357,-0.164706419286,-0.011201095575,-0.081823473106,-0.027773610737,-0.158317299813,-0.017466443677,-0.074360167249,-0.029028095454,-0.082822348597,-0.053835987081,0.009531164612,-0.116909172129,-0.154863837084,0.000477107311,-0.128319136225,-0.077398671131,-0.075683931779,-0.122838090741,0.287191729180,0.322075917912,-0.000919124648,-0.508222124655,0.165617356283,-0.323602336615,-0.273125035656,-0.018244856499,0.234688806453,0.219459645093,-0.235618805258,-0.307708573685,0.043493798807,0.176163450169,0.160300414765,0.149737694876,-0.070075934249,0.283129036688,-0.223999923964,-0.350800381583,0.249310107747,0.278111348166,0.281860374750,-0.032452304628,0.016236773027,-0.135938944897,-0.388427558571,-0.052856598896,-0.399180430031,0.587077748114,-0.291051228627,-0.046334683105,0.185290163977,0.016657330063,-0.012196878767,-0.022090354212,-0.080833454323,0.232810526212,-0.044536572541,-0.108150456129,-0.132632936798,-0.008464732576,-0.200487235354,-0.100820599558,0.004327487056,-0.129005053928,-0.153708449374,-0.017812814035,-0.110236779485,-0.013456811841,0.048443699582,-0.068543597630,-0.037160314295,-0.031812127837,-0.046315497204,-0.112636465653,-0.044679313068,-0.030991376401,0.036716704582,0.256719564529,0.501006380106,0.331167689076,0.187540535390,0.175160841938,-0.337947986349,-0.267148572046,-0.218829663741,-0.357403395240,-0.289398992735,-0.068330514947,0.131567824370,-0.097808878680,-0.061238333845,-0.013953185955,-0.019377032079,0.132355191015,0.119975758562,-0.140750376330,0.277384742063,0.130818083451,-0.164515791329,-0.044154647518,0.102810680709,0.128526792060,-0.073415622302,-0.076820811313,0.125570395964,0.198979432570,0.050496692298,-0.199996204047,-0.108349766354,-0.091682596450,-0.203771101219,-0.091054801866,-0.198597061746,-0.184646296016,-0.038564660636,-0.158914390358,-0.048339828029,0.016332137015,0.035746838605,0.153410058067,0.187475531924,0.036928888576,0.008361646951,-0.041057020702,-0.069834965899,0.114811965011,0.200308397981,-0.021200132511,-0.258865315495,-0.200090874974,-0.087886783957,-0.077827623342,-0.131129181320,-0.045734007428,-0.168029416400,-0.021285246913,0.035978930619,-0.238555032392,-0.228516059755,-0.207676648520,-0.063532758411,-0.085313279937,-0.010414739883,0.128789935024,0.130544421894,0.066119199910,0.034006031221,-0.062625317611,-0.069408487173,-0.021731387590,-0.104425338377,-0.180080767010,-0.047229561036,-0.176929580809,-0.061467824194,-0.141643404616,-0.150344267825};
double g_b1[20]= {0.044124385940,-0.143118250152,-0.076742199098,0.133214778175,-0.096067776932,0.081384577779,-0.071702615342,0.004898800527,-0.025684527790,-0.084848810759,-0.123990504457,0.069901536860,-0.068830130439,0.052514931015,-0.000586394165,-0.039736511894,-0.250016420113,-0.086944761722,0.056277945659,0.015314992943};
double g_w2[20]= {0.427300965978,0.259895044355,0.326537871621,-0.381667249466,0.377026442298,-0.357101642585,0.730314328667,0.577186129977,-0.231575219511,0.408910632693,1.335027340420,0.273113722249,0.468749294988,1.197003616563,1.093811599529,0.235623167839,-0.949362337888,-0.508935656626,0.322886677517,0.427700356067};
double g_b2= -0.288662007579;


//for timbre
//double g_timbrew1[225]= {0.153229841377,-0.160448659760,-0.285870839126,0.035388279054,-0.083669004589,-0.490014185047,-0.666007163086,-0.232161776650,-0.381771805673,-0.309206912575,-0.191762246252,-0.053750821796,0.128416269000,-0.035318199720,-0.152318081184,0.072979557897,0.154285107949,-0.114668986389,0.230418257796,0.091834204761,-0.154405384992,0.058767204862,0.136301806524,0.011980890917,0.014722191889,0.039801350355,0.068319956279,0.023474728001,-0.026562825064,-0.069733297018,0.019406471050,0.028144944306,-0.025764262904,0.083016624218,0.033848199951,-0.124906244897,-0.309931200196,0.010291078655,-0.128471819459,-0.014617621052,-0.151305782215,-0.172122782648,0.006239329167,-0.020162162275,0.062733568869,-0.121892938924,0.025447221526,0.032080773593,-0.048324144512,-0.035864760406,-0.115149206994,0.292091276565,-0.125444928408,0.023841800385,-0.232352558036,0.148988457554,-0.028875761169,-0.067228464556,0.093697646586,0.252907234123,-0.024502758771,0.078651625606,-0.097621773986,0.072746782858,0.023151044077,-0.002721338609,-0.146614108597,-0.018772528444,0.332789445960,0.120572449870,-0.232687364276,0.026278128113,-0.102181166068,-0.182046325110,-0.067555056337,-0.069367675660,-0.073799392850,0.087629429920,-0.030317227462,-0.124788786896,0.107926248204,-0.203320246030,0.227753628040,0.062661647248,0.265249753154,-0.257268093471,0.054900715169,-0.234154480322,-0.046224626460,-0.048493264912,-0.112657329409,0.313733920901,-0.049263491952,-0.028397329988,0.090002857649,0.004782946878,-0.191817954437,-0.239789149925,0.011009220058,-0.141861984590,-0.026911657711,0.102450922432,-0.057159832634,0.164056278880,-0.115311080103,-0.042491516581,0.047466975059,0.292089987561,-0.019803918634,-0.064487037710,-0.175211928817,0.041933236849,-0.078310153308,-0.074446250895,0.329866571218,-0.027010837583,0.118572081586,0.020635018448,0.032094997847,0.190636389607,0.064446857979,-0.056067133829,-0.091299872515,0.066150255699,0.227737672199,0.218854948315,-0.316024123759,0.079350400935,-0.158891958855,0.180569516814,-0.177476647289,-0.064591017936,0.027859905348,0.151625420258,0.027875034991,-0.010273879886,0.031025736057,0.015985415937,0.091537421626,-0.137247926379,0.091715157178,-0.178005855722,-0.037468345900,-0.047687396581,-0.271918409143,-0.133311213150,0.024460550920,-0.107759921799,-0.068518015096,0.080811802234,-0.117412521419,0.161731313490,0.154237196567,0.030522779319,0.181305392715,-0.066034412831,-0.019872964216,0.106601584823,0.085857080333,0.124265005617,0.003276579239,-0.001158620041,0.063185155752,-0.038140043949,-0.136280881311,0.130705146640,-0.084428715389,0.096858764201,0.187637748901,-0.007424024158,0.133640643012,-0.023285172341,-0.085535070010,0.031166435611,0.057231818144,0.000346177963,-0.214168058742,-0.065152815115,0.249847329369,0.044479060618,-0.375754722499,0.289359263939,0.140579433782,-0.062263530182,0.049546470165,-0.188914551544,-0.004841090021,0.106746363812,0.268193332742,-0.162260563876,-0.090572632912,-0.020618091053,0.132854690649,-0.357279158514,-0.027712171494,0.130224886398,0.087321090886,0.026080959782,0.219649266712,0.007952256777,-0.062933682819,-0.183338030355,0.178929537389,0.199618508705,0.040420029930,0.038842473119,0.020384265681,0.140745694837,0.020759162784,0.023271411352,0.049267053348,-0.071795095999,0.070237567044,-0.015206265951,-0.084975600960,-0.062668276853,-0.147478912308,-0.043542993871,-0.116445275388,-0.074352483596,-0.027594132265,0.044685284863,-0.296620673024,0.034444946116,0.001673496378};
//double g_timbreb1[15]= {-0.281414690472,0.189517807184,-0.070458414285,-0.028092503847,0.106751066301,-0.168612316079,0.118959270974,0.019818288202,-0.018117195229,-0.044395225051,0.203267335375,0.021317982716,0.015646770819,-0.080239819232,-0.133152971939};
//double g_timbrew2[15]= {0.802257836140,-0.567661626378,0.469815369998,-1.756172895439,1.106271455210,0.907572243787,0.470849796580,1.093743977377,0.427508753118,0.382224528424,-0.696688748681,0.238962108871,-0.694909053374,-0.764508415426,0.810397108163};
//double g_timbreb2= -0.562490008054;

//better, uses 12 features
//double g_timbrew1[144]= {-0.081694669652,-0.349781031245,-0.062051941886,-0.305376054106,0.339586790311,0.204403778710,-0.080382124527,-0.471274291498,0.411204075977,-0.047579624687,0.284019891453,0.045335537439,0.112872987688,0.238863336007,0.090538122108,0.083785529541,-0.193168950265,-0.354636932676,-0.301937234489,-0.164452138112,-0.065022201476,-0.222421213160,-0.336355216186,-0.597042910352,0.004086656303,0.109766605651,0.015340512026,0.103634866556,0.319038269963,0.116688292060,-0.063737187278,0.033986871960,0.243290368237,0.303658723595,-0.266183090862,0.131126607608,-0.098825712205,0.061579846264,-0.539551446036,-0.321277141372,0.168057399341,0.235933100227,-0.125075317356,-0.086493205368,-0.221321724809,0.063993297109,0.200370377470,0.011013334647,-0.168672352946,-0.349057617706,-0.156993705266,0.519152633738,0.011201084234,0.190994447342,-0.302016235771,-0.033291605315,0.271072858870,-0.155108857233,-0.458455760216,-0.046476343457,0.200269545265,-0.138630073654,-0.048662328698,0.308686320282,-0.116190535733,-0.531320296584,0.020995206083,0.215572495503,0.155428792642,-0.020627059218,-0.295626856090,0.011782107818,-0.000728597110,-0.203371666647,0.080221347158,-0.094298850138,-0.328884214409,-0.331323185742,0.293377640099,-0.182897266156,-0.604796355486,-0.060275449189,-0.118561860683,0.046534318018,-0.225412634004,-0.123342027834,0.011805501570,0.196332600238,0.179975990083,-0.195130929891,0.216474087046,0.224361048494,-0.356770453636,-0.363181849227,-0.196748228184,-0.761944777159,-0.379932192281,-0.186238936615,0.239619551732,-0.293606921506,-0.361025452248,0.156448848838,-0.218132144871,0.039189764537,0.243799539016,-0.454482077153,0.234196648413,0.246412889450,-0.136002222502,-0.048092100039,-0.131976414589,-0.037053524914,-0.420721789598,-0.366889949769,0.228676049498,0.128368921030,0.168773744119,-0.684617560756,-0.291244347296,0.017853078226,-0.015451648691,0.159356836249,0.071515951890,-0.354000828702,0.381872712709,0.114253760370,0.033985947527,-0.251059901220,-0.391769126939,-0.252663323143,0.081357309038,-0.012741269087,-0.117185348816,0.117291805120,-0.177904162404,-0.489404544827,-0.292229610393,-0.409247552782,-0.052315953734,-0.142819513146,0.110067397599,0.204525184695,-0.162957201207,-0.258822460832};
//double g_timbreb1[12]= {-0.403853825549,-0.224341491766,0.019106746771,0.087824403732,0.288199063859,0.106881431781,-0.145634557914,0.037112402008,0.017713690642,-0.447867133368,-0.142426058275,-0.142423769283};
//double g_timbrew2[12]= {0.316719595203,0.680891495037,-0.600337367890,-0.244673395057,0.958153472009,-0.291534656858,0.793899489273,-0.835585966663,0.477177498887,0.251094348011,1.098946101538,0.534850006229};
//double g_timbreb2= 0.230863865305;


double g_timbrew1[144]= {0.806883844174,0.149607145229,0.006962610946,-0.290054045077,0.006444276618,0.163522263540,-0.077172478830,-0.255631408345,-0.891724450185,0.330892673960,-0.416537670738,0.320772547748,0.323219360927,-0.699512422080,-0.067164160629,-0.057723864715,0.318302585707,0.173712411929,0.137258195088,0.065189766102,0.087873390996,0.399473603736,0.102896255587,-0.071179245810,0.097837448556,0.211290269573,-0.100867254506,-0.169832477104,-0.098370898031,-0.100245394334,0.323601946838,0.394315272617,0.285979552328,-0.127152142510,0.258761082186,-0.206284270389,0.058365970129,-0.415581774634,0.273396736537,0.115551444236,0.297719940871,0.065720236305,0.191131054415,-0.149105416727,-0.205046429226,-0.138379887443,0.572408629018,0.202383427324,0.127268690128,-0.051093632521,-0.185789012511,0.379528412937,0.269974922955,0.138354490637,0.136330071812,0.351074769190,0.309153293148,0.390640688576,-0.230569343371,-0.141221499300,0.303029750804,0.275071000622,0.275167852771,0.034077354025,0.064344431269,-0.082355642802,-0.009528229080,0.345699213376,0.077468622320,0.061265185892,0.102438215522,-0.418334705412,0.514022426816,-0.134682376476,0.537159437068,-0.006487140764,0.161074467660,-0.123087125398,0.406343537923,0.028210272087,0.004763550350,0.142507685405,0.280170445203,-0.258797823572,-0.375921340420,0.111713797622,0.334091947988,-0.551830316123,0.328883144043,-0.006307095198,0.081396283538,0.266068145597,0.217375400867,-0.175673478895,-0.010995584653,0.236338757738,0.310805687027,0.149058202719,0.386912535913,0.234865129025,0.132947648600,-0.180435235862,0.074311023611,0.190376904055,0.265914024367,-0.002406793923,-0.025250453999,0.281590480536,0.403858183963,-0.013257724431,-0.299692980167,0.272407916273,-0.085379379722,0.005960840741,-0.871668226412,0.306314573282,-0.533843264056,-0.319759795549,-0.492015223230,-0.294952485005,-0.309455283297,0.119538432412,0.399648202619,-0.204549191132,-0.165067545682,-0.305300388355,0.180921173566,-0.294834601699,-0.019227193216,-0.214503831705,-0.093164984073,0.081676126473,0.157202106210,-0.340743434448,-0.276718826522,-0.197314463035,-0.192605016601,0.139639563935,-0.154680226754,-0.222331188957,-0.605212317926,-0.100919079533,0.108277714709,0.220418636875};
double g_timbreb1[12]= {0.133369484959,-0.162582965002,-0.194074188031,0.350325178917,-0.656657253284,0.302730818929,0.033625260397,-0.253403732418,0.134145082352,-0.225761487171,0.297927951586,0.443175801761};
double g_timbrew2[12]= {2.158089004310,0.490876756825,-1.261159090586,-1.206127420600,-0.448901380838,-0.419801143831,-0.813114123138,1.022854248292,-0.106615994724,0.878430041387,-0.309410321053,0.261644868684};
double g_timbreb2= -0.409191154001;




//float bweights[40]= { 0, 0, 0, 0, 0, 0.15432646298622, 0.1986156888837, 0.17553530220127, 0.12767367625204, 0.011026279818415, 0.13399548316955, 0.042041540062882, 0.010204505881597, 0.036055636290171, 0, 0.095077049841321, 0.0051353555556125, 0, 0, 0, 0, 0, 0.01031301905723, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};



static InterfaceTable *ft;




struct AnalyseEvents2 : Unit {

	//FFT data
	int m_bufWritePos;
	float * m_prepareFFTBuf;
	float * m_FFTBuf;

	scfft *m_scfft;

	//time positions
	long m_frame;
	long m_lastdetect;

	//loudness measure
	float m_loudbands[NUMERBBANDS][PASTERBBANDS]; //stores previous loudness bands
	int m_pasterbbandcounter;
	float m_df[DFFRAMESSTORED]; //detection function buffer (last 5)
	int m_dfcounter;

	float m_loudness[LOUDNESSSTORED];   //store loudness, sent back with triggers for determining accents (probably not that useful since not measured at peak)
	int m_loudnesscounter;

	//recording state
	int m_onsetdetected;
	int m_recording;
	int m_start;
	float m_testintensity; //end can trigger if drop below this

	//into global frame buffer
	int m_numframes;
	int m_startframe;
	int m_endframe;

	int m_startblock;
	int m_endblock;

	//into loudness frame buffer
	int m_lstartframe;
	int m_lendframe;

	//into pitch frame buffer

	//triggers- greater than 0, send trigger message with that ID
	int m_triggerid;

	//target buffer for event data
	uint32 m_bufNum, m_bufSize;
	int m_maxEvents; //, m_lastEvent; //maximum that can be stored
	float * m_eventData; //[0] position stores num of events stored
	int m_numstored;

	//different structure to buffer saving for m_circular = true
	//support for circular buffers and on-the-fly event capture
	int m_circular; //act as if a circular buffer, uses m_numstored for which to write/pick up next
	//SendTrigger on a detected event giving place to pick up

	//pause just by pausing detection Synth (RecordBuf paused at the same time)
	//int m_pause;	//pause recording, do not store any detected events, zero counts as go round, but Lang side


	uint32 m_now;

	//int m_maxblocksstored; //since FS fixed at 44100, fix this too
	float * m_maxintensity;
	int m_maxcount;
	float * m_store;
	int m_storecounter;

	//uses maxcount
	float * m_pitch;
	float * m_sortbuf;
	float m_patband1[LOUDNESSSTORED];
	float m_patband2[LOUDNESSSTORED];

	int m_featurecounter;
	double m_featurevector[20];

	double m_timbreFV[12];
	float m_zcr[LOUDNESSSTORED];
	float m_speccentroid[LOUDNESSSTORED];


	//evaluation- write beat locations to a buffer
	//	uint32 m_bufNum2, m_bufSize2;
	//	//int m_maxbeats,
	//	int m_dfsaved;
	//	float * m_dfData;
	//	//uint32 m_now, m_then;
	//	//uint32 m_lastbeattime;
	//
	//
};



extern "C"
{
	//required interface functions
	void AnalyseEvents2_next(AnalyseEvents2 *unit, int wrongNumSamples);
	void AnalyseEvents2_Ctor(AnalyseEvents2 *unit);
	void AnalyseEvents2_Dtor(AnalyseEvents2 *unit);
}



//other functions
void AnalyseEvents2_preparefft(AnalyseEvents2 *unit, float* in, int n);
void AnalyseEvents2_dofft(AnalyseEvents2 *unit);
void calculatedf(AnalyseEvents2 *unit);
void peakpickdf(AnalyseEvents2 *unit);

void storeEvent(AnalyseEvents2 *unit, int start, int stop);
int findZeroCrossing(AnalyseEvents2 *unit, int base);
int findMinimumMaxIntensity(AnalyseEvents2 *unit, int framenow);

//feature extraction
float calculatePAT(AnalyseEvents2 *unit);
float calculatePAT2(AnalyseEvents2 *unit);
float calculatePitch(AnalyseEvents2 *unit);
float calculatePATnn(AnalyseEvents2 *unit);

float calculateTimbrenn(AnalyseEvents2 *unit);
float calculateTimbre(AnalyseEvents2 *unit);
void countZeroCrossing(AnalyseEvents2 *unit);



void AnalyseEvents2_Ctor(AnalyseEvents2* unit) {
	int i,j;


	//CHECK SAMPLING RATE AND BUFFER SIZE
	 int blocksize = unit->mWorld->mFullRate.mBufLength;

	if(blocksize!=64) {
		printf("AnalyzeEvents2 complains: block size not 64, you have %d\n", blocksize);
		SETCALC(*ClearUnitOutputs);
		unit->mDone = true;
		return;
	}

	int testsr = unit->mWorld->mSampleRate; //(int)(unit->mWorld->mSampleRate+0.1);

	if(testsr!=44100) {
		printf("AnalyzeEvents2 complains: sample rate not 44100, you have %d\n", testsr);
		//return 0; //not necessarily catastrophic
	}


	////////FFT data///////////

	unit->m_prepareFFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_FFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_bufWritePos = 0;

	//N=1024
	SCWorld_Allocator alloc(ft, unit->mWorld);
	//no overlap
	unit->m_scfft = scfft_create(N, N, kHannWindow, unit->m_FFTBuf, unit->m_FFTBuf, kForward, alloc);


	////////vDSP///////////////
	//
	//	unit->m_vA.realp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float));
	//	unit->m_vA.imagp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float));
	//	unit->m_vlog2n = 10; //N is hard coded as 1024, so 10^2=1024 //log2max(N);
	//	unit->m_vsetup = create_fftsetup(unit->m_vlog2n, 0);
	//
	////////time positions//////////
	unit->m_frame=0;
	unit->m_lastdetect=-100;


	/////////df measure////////
	unit->m_dfcounter=DFFRAMESSTORED-1;
	//zero loudness store
	for(j=0;j<DFFRAMESSTORED;++j) {
		unit->m_df[j]=0.0;
	}

	//LOUDNESS
	unit->m_loudnesscounter=LOUDNESSSTORED-1;
	//zero loudness store
	for(j=0;j<LOUDNESSSTORED;++j) {
		unit->m_loudness[j]=0.0;
		unit->m_patband1[j]=0.0;
		unit->m_patband2[j]=0.0;
		unit->m_speccentroid[j]=0.0;
		unit->m_zcr[j]=0.0;
	}



	//zero previous specific loudness in Bark bands
	for(i=0;i<PASTERBBANDS; ++i)
		for(j=0;j<NUMERBBANDS;++j) {
			unit->m_loudbands[j][i]=0.0;
		}

	unit->m_pasterbbandcounter=0;

	unit->m_onsetdetected=0;
	unit->m_recording=0;

	//triggers
	unit->m_triggerid=(int)ZIN0(3);

	//printf("triggerid %d \n", unit->m_triggerid);

	//event analysis info buffer
	World *world = unit->mWorld;

	uint32 bufnum = (uint32)ZIN0(1);
	if (bufnum >= world->mNumSndBufs) bufnum = 0;
	unit->m_bufNum=bufnum;

	//printf("%d \n",bufnum);

	SndBuf *buf = world->mSndBufs + bufnum;
	unit->m_bufSize = buf->samples;

	unit->m_maxEvents= (unit->m_bufSize-2)/10; //require at least 10 floats per event, may change later

	//printf("max events %d \n",unit->m_maxEvents);

	unit->m_eventData= buf->data;

	unit->m_circular= (int)ZIN0(4);

	//printf("circular %d \n", unit->m_circular);

	//must be greater than 14
	//if(unit->m_circular) {unit->m_maxEvents=16;}

	//wrap number for next event to overwrite, sending corresponding pick up trigger to lang, uses m_numstored

	//unit->m_whichsubbuffer= 0; //place in second subbuffers
	//unit->m_numsubbuffers= unit->m_maxEvents/15; //Buffer must be at least 152 floats long

	//SendTrigger on a detected event giving subbuffer num and num in that subbuffer



	//unit->m_lastEvent= -1; //require at least 4 ints per event, may change later
	unit->m_numstored=0;

	unit->m_now=0;
	unit->m_startframe=0;
	unit->m_endframe=0;
	unit->m_lstartframe=0;
	unit->m_lendframe=0;
	unit->m_numframes=1;

	buf->data[0]=0;
	buf->data[1]=0;

	//calc_BufRate
	//	if (OUTRATE(0) == calc_FullRate) {
	//		SETCALC(LFPulse_next_a);
	//	} else {
	//		SETCALC(LFPulse_next_k);
	//	}

	//prepare stores for sample data needed for event analysis once captured
	unit->m_maxintensity=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));

	Clear(MAXBLOCKS, unit->m_maxintensity);

	unit->m_maxcount=0;
	//one second worth of samples held in memory
	unit->m_store=(float*)RTAlloc(unit->mWorld, FS * sizeof(float));

	Clear(MAXBLOCKS, unit->m_store);
	unit->m_storecounter=0;

	unit->m_pitch=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));
	unit->m_sortbuf=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));

	/////////DEBUG

	//	uint32 bufnum2 = (uint32)ZIN0(6);
	//	if (bufnum2 >= world->mNumSndBufs) bufnum2 = 0;
	//	unit->m_bufNum2=bufnum2;
	//
	//	printf("bufnum2 %d \n",bufnum2);
	//
	//	SndBuf *buf2 = world->mSndBufs + bufnum2;
	//	unit->m_bufSize2 = buf2->samples;
	//
	////	unit->m_maxbeats= (unit->m_bufSize-1); //1 float per event, plus counter
	//	unit->m_dfData= buf2->data;
	//	unit->m_dfsaved=0;
	////
	////	unit->m_now=0;
	////	unit->m_then=0;
	//
	///////////

	unit->mCalcFunc = (UnitCalcFunc)&AnalyseEvents2_next;

}



void AnalyseEvents2_Dtor(AnalyseEvents2 *unit) {

	RTFree(unit->mWorld, unit->m_prepareFFTBuf);
	RTFree(unit->mWorld, unit->m_FFTBuf);

	RTFree(unit->mWorld, unit->m_maxintensity);
	RTFree(unit->mWorld, unit->m_store);
	RTFree(unit->mWorld, unit->m_pitch);
	RTFree(unit->mWorld, unit->m_sortbuf);

	if(unit->m_scfft) {
		SCWorld_Allocator alloc(ft, unit->mWorld);
		scfft_destroy(unit->m_scfft, alloc);
	}

}


void AnalyseEvents2_next(AnalyseEvents2 *unit, int wrongNumSamples)
{
	//would normally be float,will be cast to int for Tristan's optimisation
	float *in = IN(0);

	int numSamples = unit->mWorld->mFullRate.mBufLength;

	float *output = ZOUT(0);

	//update stored samples, doesn't matter if FFT is calculated halfway through block, just need to know which sample.
	//OK to approximate looking for minima via block based maxs.
	float * store=unit->m_store;
	int storecounter= unit->m_storecounter;

	float maxintensity= 0.0;

	for(int j=0; j<numSamples; ++j) {

		float val= in[j];
		store[storecounter]= val;

		storecounter= (storecounter+1)%FS;
		val *= val;
		if(val>maxintensity) maxintensity=val;

	}

	unit->m_storecounter=storecounter;

	unit->m_maxintensity[unit->m_maxcount]= maxintensity;

	float pitch=ZIN0(5); //pitch .kr input

	unit->m_pitch[unit->m_maxcount]= pitch;

	unit->m_now+=numSamples;

	unit->m_eventData[1]=(float)unit->m_now;

	AnalyseEvents2_preparefft(unit, in, numSamples);

	//updated here so that if you find an event, maxcount is pointing to the most recently stored value
	unit->m_maxcount= (unit->m_maxcount+1)% MAXBLOCKS;

	//always output zero- no audio output as such
	float outval= 0.0;

	if(unit->m_onsetdetected) {

		//printf("onset detected %d \n",(unit->m_onsetdetected));

		//if(unit->m_triggerid) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, unit->m_loudness);

		unit->m_onsetdetected=0;
		outval=1.0;
	}

	//control rate output- no, trouble
	//ZOUT0(0)=outval;

	for (int i=0; i<numSamples; ++i) {
		*++output = outval;
	}

}


//Tristan recommends copying ints rather than floats- I say negligible compared to other algorithm costs for the moment
// TO TREAT, check, update, probably replace entirely with pre allocated buffer based scheme?
void AnalyseEvents2_preparefft(AnalyseEvents2 *unit, float* in, int n) {

	int i, index = 0, cpt = n, maxindex;

	int bufpos= unit->m_bufWritePos;

	float * preparefftbuf=unit->m_prepareFFTBuf;
	float * fftbuf= unit->m_FFTBuf;

	// Copy input samples into prepare buffer
	while ((bufpos < N) && (cpt > 0)) {
		preparefftbuf[bufpos] = in[index];
		bufpos++;
		index++;
		cpt--;
	}

	// When Buffer is full...
	if (bufpos >= N) {

		// Make a copy of prepared buffer into FFT buffer for computation
		for (i=0; i<N; i++)
			fftbuf[i] = preparefftbuf[i];

		// Save overlapping samples back into buffer- no danger since no indices overwritten
		for (i=0; i<OVERLAP; i++)
			preparefftbuf[i] = preparefftbuf[OVERLAPINDEX+i];

		unit->m_frame= unit->m_frame+1;
		AnalyseEvents2_dofft(unit);

		maxindex = n - index + OVERLAPINDEX;

		//blockSize less than N-OVERLAPINDEX so no problem
		// Copy the rest of incoming samples into prepareFFTBuffer
		for (i=OVERLAPINDEX; i<maxindex; i++) {
			preparefftbuf[i] = in[index];
			index++;
		}

		bufpos = maxindex;

	}


	unit->m_bufWritePos= bufpos;
}



//calculation function once FFT data ready
void AnalyseEvents2_dofft(AnalyseEvents2 *unit) {

	int i;

	//done before samples leave time domain...
	countZeroCrossing(unit);

	//printf("dofft \n");

	float * fftbuf= unit->m_FFTBuf;

	scfft_dofft(unit->m_scfft);

	//format is dc, nyquist, bin[1] ,real, bin[1].imag, etc

	//fftbuf[0] already bin 0
	fftbuf[0]= fftbuf[0]* fftbuf[0]; //get power

	float val1, val2;
	// Squared Absolute so get power
	for (i=2; i<N; i+=2) {
		val1 = fftbuf[i];
		val2 = fftbuf[i+1];
		//i>>1 is i/2
		fftbuf[i>>1] = (val1*val1)+(val2*val2);
	}

	//calculate loudness detection function
	calculatedf(unit);

	//use detection function
	peakpickdf(unit);


	//is an event recording that could finish?
	//must do with respect to older positions, else an immediate new event could get stuck on the other side of this!
	//actually that should be OK as long as new event is correctly started
	if(unit->m_recording)
		if(((unit->m_maxintensity[unit->m_maxcount] < unit->m_testintensity) && ((unit->m_frame-unit->m_lastdetect)>=MINEVENTDUR))  ||
		   ((unit->m_frame-unit->m_lastdetect)>MAXEVENTDUR)) {

			int stop= findMinimumMaxIntensity(unit, unit->m_maxcount);

			//can set other params of events but need to recover
			//sample in 1 second buffer
			int samplenow= unit->m_storecounter; //-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;

			int samplesold= samplenow-stop;

			if(stop>samplenow) samplesold= samplenow+FS-stop;

			int blocksold= (int)(samplesold/MAXBLOCKS);

			int framesold= (int)(samplesold/HOPSIZE);

			unit->m_endframe=unit->m_frame-framesold;
			unit->m_lendframe=(unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;

			unit->m_endblock= (unit->m_maxcount+ MAXBLOCKS- blocksold)%MAXBLOCKS;

			unit->m_numframes= unit->m_endframe-unit->m_startframe;

			//will convert locations in buffer into time locations since UGen began
			//may also process it further to find peaks, PAT etc
			storeEvent(unit, unit->m_start, stop);

		}


}



void calculatedf(AnalyseEvents2 *unit) {

	int j,k;

	float * fftbuf= unit->m_FFTBuf;

	float dfsum=0.0;

	float loudsum=0.0;

	int pastband=unit->m_pasterbbandcounter;

	for (k=0; k<NUMERBBANDS; ++k){

		int bandstart=eqlbandbins[k];
		//int bandend=eqlbandbins[k+1];
		int bandsize= eqlbandsizes[k];

		float bsum=0.0;

		for (int h=0; h<bandsize;++h) {
			bsum= bsum+fftbuf[h+bandstart];
		}

		//ADDED 16/2/06
		//correction for Altivec being 2* MATLAB FFT, powers 4*
		//bsum*=0.25;

		//store recips of bandsizes?
		bsum= bsum/bandsize;

		//into dB, avoid log of 0
		//float db= 10*log10((bsum*10000000)+0.001);
		float db= 10*log10((bsum*32382)+0.001);

		//printf("bsum %f db %f \n",bsum,db);

		//convert via contour
		if(db<contours[k][0]) db=0;
        else if (db>contours[k][10]) db=phons[10];
        else {

            float prop=0.0;

            for (j=1; j<11; ++j) {
                if(db<contours[k][j]) {
                    prop= (db-contours[k][j-1])/(contours[k][j]-contours[k][j-1]);
                    break;
				}

				if(j==10)
					prop=1.0;
            }

            db= (1.0-prop)*phons[j-1]+ prop*phons[j];
			//printf("prop %f db %f j %d\n",prop,db,j);

		}

		//float lastloud=unit->m_loudbands[k];
		float lastloud=0.0;

		for(j=0;j<PASTERBBANDS; ++j)
			lastloud+=unit->m_loudbands[k][j];

		lastloud /= PASTERBBANDS;

		float diff= sc_max(db-lastloud,0.f);

		dfsum=dfsum+diff; //(bweights[k]*diff);

		unit->m_loudbands[k][pastband]=db;

		//must sum as intensities, not dbs once corrected, pow used to be other way around
		loudsum= loudsum+((pow(10, 0.1*db)-0.001)*0.0000308813538386);

	}

	unit->m_loudnesscounter=(unit->m_loudnesscounter+1)%LOUDNESSSTORED;
	//total excitation, correct back to dB scale
	unit->m_loudness[unit->m_loudnesscounter]=10*log10((loudsum*32382)+0.001);

	//calculate PAT features= combine PAT bands 1-12 /12 and 27-40 /14
	float psum=0.0;

	for (k=0;k<11;++k)
		psum+= pow(10, 0.1*unit->m_loudbands[k][pastband]);

	//psum=unit->m_loudbands[1][pastband];

	unit->m_patband1[unit->m_loudnesscounter]=(10*log10(psum))/11;

	////DEBUG
	//	unit->m_dfData[unit->m_dfsaved+1]=unit->m_patband1[unit->m_loudnesscounter];
	//
	//	if (unit->m_dfsaved<(unit->m_bufSize2-2))  unit->m_dfsaved=unit->m_dfsaved+1;
	//
	//	//printf("dfsaved %d size %d df %f %f \n",unit->m_dfsaved, unit->m_bufSize2, dfsum*0.025, unit->m_dfData[unit->m_dfsaved]);
	//
	//	unit->m_dfData[0]=unit->m_dfsaved;
	//	////////


	psum=0.0;

	for (k=26;k<NUMERBBANDS;++k)
		psum+= pow(10, 0.1*unit->m_loudbands[k][pastband]);

	unit->m_patband2[unit->m_loudnesscounter]=(10*log10(psum))/14;


	psum=0.0;
	//spec centroid
	for (k=0;k<NUMERBBANDS;++k)
		psum+= ((unit->m_loudbands[k][pastband])*0.01)*(k+1);

	unit->m_speccentroid[unit->m_loudnesscounter]=psum*0.025; // mean over 40 /(40)



	unit->m_pasterbbandcounter=(pastband+1)%PASTERBBANDS;

	//increment first so this frame is unit->m_loudnesscounterdfcounter
	unit->m_dfcounter=(unit->m_dfcounter+1)%DFFRAMESSTORED;

	//	////DEBUG
	//	unit->m_dfData[unit->m_dfsaved+1]=dfsum*0.025;
	//
	//	if (unit->m_dfsaved<(unit->m_bufSize2-2))  unit->m_dfsaved=unit->m_dfsaved+1;
	//
	//	printf("dfsaved %d size %d df %f %f \n",unit->m_dfsaved, unit->m_bufSize2, dfsum*0.025, unit->m_dfData[unit->m_dfsaved]);
	//
	//	unit->m_dfData[0]=unit->m_dfsaved;
	//	////////

	unit->m_df[unit->m_dfcounter]=dfsum*0.025; //divide by num of bands to get a dB answer

	//printf("loudness %f %f \n",unit->loudness[unit->loudnesscounter], lsum);


}


//score rating peak picker
void peakpickdf(AnalyseEvents2 *unit) {
	int i;

	//smoothed already with df looking at average of previous values
	int dfnow= unit->m_dfcounter+DFFRAMESSTORED;

	//rate the peak in the central position

	int dfassess= ((dfnow-3)%DFFRAMESSTORED)+DFFRAMESSTORED;

	//look at three either side

	int pos;
	float val;

	float centreval=unit->m_df[dfassess%DFFRAMESSTORED];

	//must normalise
	//printf("centreval %f \n",centreval);

	float score=0.0;

	for (i=(-3); i<4; ++i) {
		pos= (dfassess+i)%DFFRAMESSTORED;

		val= centreval-(unit->m_df[pos]);

		if(val<0) val*=10; //exacerbate negative evidence

		score=score+val;
	}

	if(score<0.0) score=0.0;

	//normalise such that df max assumed at 200, 0.005, was 50, 0.02

	//score *= 0.08; //0.005;

	score= score*0.0073;

	//	////DEBUG
	//	unit->m_dfData[unit->m_dfsaved+1]=score;
	//
	//	if (unit->m_dfsaved<(unit->m_bufSize2-2))  unit->m_dfsaved=unit->m_dfsaved+1;
	//
	//	//printf("dfsaved %d size %d df %f %f \n",unit->m_dfsaved, unit->m_bufSize2, dfsum*0.025, unit->m_dfData[unit->m_dfsaved]);
	//
	//	unit->m_dfData[0]=unit->m_dfsaved;
	//	////////


	//if enough time since last detection
	if((unit->m_frame-unit->m_lastdetect)>=MINEVENTDUR) {

		//SIMPLE THRESHOLDING PEAKPICKER
		float threshold= ZIN0(2); //0.34 best in trials

		if(score>=threshold) {

			//printf("threshold %f score %f FRAME %d \n",threshold, score, unit->m_frame);

			unit->m_lastdetect=unit->m_frame;
			unit->m_onsetdetected=1;

			//run a detection search, look at min of maxintensity, then for a zero crossing

			//correct maxcount position for 3 FFT frames ago lag, which at this overlap is 16*3/2= 24
			//back by 16*4/2=32, always late then, corrected
			int maxpos= (unit->m_maxcount+ MAXBLOCKS- 32)%MAXBLOCKS;

			int start= findMinimumMaxIntensity(unit, maxpos);


			//can set other params of events but need to recover
			//sample in 1 second buffer
			int samplenow= unit->m_storecounter; //-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;

			int samplesold= samplenow-start;

			if(start>samplenow) samplesold= samplenow+FS-start;

			int framesold= (int)(samplesold/HOPSIZE);
			int blocksold= (int)(samplesold/MAXBLOCKS);

			int startframe=  unit->m_frame-framesold;
			int lstartframe= (unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;

			//printf("detected an onset %d \n", start);

			//if already in action recording, must stop previous at sample start-1
			if(unit->m_recording) {

				//use calculated positions above but less one frame so never overlap
				unit->m_endframe=startframe-1;
				unit->m_lendframe=(lstartframe+LOUDNESSSTORED-1)%LOUDNESSSTORED;

				//if(unit->m_endframe>unit->m_startframe) ALWAYS positive since always increasing
				unit->m_numframes= unit->m_endframe-unit->m_startframe;

				unit->m_endblock= (unit->m_maxcount+ MAXBLOCKS- blocksold-16)%MAXBLOCKS;

				//will convert locations in buffer into time locations since UGen began
				//may also process it further to find peaks, PAT etc
				storeEvent(unit, unit->m_start, start);

			}

			unit->m_recording=true;

			unit->m_startblock= (unit->m_maxcount+ MAXBLOCKS- blocksold)%MAXBLOCKS;

			maxpos= unit->m_startblock;

			//find min intensity in previous 16 Max blocks- no, better to find min intensity near start, not at this detection point
			float mintest= unit->m_maxintensity[maxpos];

			for (i=1; i<16; ++i) {
				int pos= (maxpos+MAXBLOCKS-i)%MAXBLOCKS;

				float val;
				val= unit->m_maxintensity[pos];

				if (val<mintest) mintest=val;
			}

			//could have a multiplier for how much drop allowed here, can calculate as dBs difference
			unit->m_testintensity= sc_min(mintest,0.001f); //was 0.1 //(unit->m_maxintensity[maxpos]*0.1)+0.001; //end can trigger if drop below this
			unit->m_start= start;

			unit->m_startframe=startframe;
			unit->m_lstartframe=lstartframe;


		}
	}
}




//danger- may put up to 22mS of silence on front, bit of code to avoid this is floating point test for minima below
int findMinimumMaxIntensity(AnalyseEvents2 *unit, int framenow) {
	int i;
	//search up to 2-0 FFT frames ago, ie look for a min within 16*2=32 max intensities, no just within this FFT frame

	float * intensity= unit->m_maxintensity;
	int minindex=0;
	float minval=intensity[framenow];

	int pos;
	float val;

	for (i=0; i<16; ++i) {
		pos= (framenow+MAXBLOCKS-i)%MAXBLOCKS;
		val= intensity[pos];
		//do this to stop silences continually beating it
		if(val<(minval-0.00001)){
			minval=val;
			minindex=i;
		}

		//condition to exit if already near zero here
		if(val<0.01) {minindex=i;


			//printf("minindex %d val %f minval %f\n",minindex, val, minval);


			break;}

	}

	//now find zero crossing at this point, searching previous 200 samples, must minus 1 since storecounter is currently pointing to next storage space
	//remove HOPSIZE*3 to make it snap to the
	int samplelocation= (unit->m_storecounter-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;

	return findZeroCrossing(unit,samplelocation);

}


//takes first ZC, also requiring that value at this point is sufficiently small
//otherwise, return the point of lowest intensity in the range
//find within 441 samples, ie at FS=44100, within 10mS
int findZeroCrossing(AnalyseEvents2 *unit, int base) {
	int i, pos, prevpos;

	float * store=unit->m_store;

	int minindex;
	float minval;

	minval= store[base];
	minindex=base;

	for (i=0; i<440; ++i) {
		pos= (base+FS-i)%FS;
		prevpos= (base+FS-i-1)%FS;

		float val= store[pos];
		float intensity= val*val;

		if ((val>=(-0.00000001)) && (store[prevpos]<0.00000001) && (intensity <0.1)) {
			return pos;
		}

		if(intensity<minval) {
			minval=intensity;
			minindex=pos;
		}

	}

	return minindex;
}



//will later need to analyse PAT and transient region too
void storeEvent(AnalyseEvents2 *unit, int start, int stop) {
	uint32 length, ago;

	int storecounter= unit->m_storecounter;

	if(stop<start) {
		length=(stop+FS-start);
	} else {
		length=stop-start;
	}

	if(storecounter<stop) {
		ago= (storecounter+FS-stop);
	} else {
		ago=storecounter-stop;
	}

	//printf("event found now %d start %d stop %d length %d ago %d\n",unit->m_numstored, start, stop, length, ago);

	float * data= unit->m_eventData;

	//data[1] is always set by the plugin to show its current time, event starts and ends are relative to this

	if((unit->m_numstored)<unit->m_maxEvents) {

		int offset;

		offset=(10*(unit->m_numstored))+2;

		uint32 now= unit->m_now;
		uint32 correctedstart, correctedend;

		if(now<(ago+length))
			correctedstart= 0;
		else
			correctedstart=now-ago-length;

		if(now<ago)
			correctedend= 0;
		else
			correctedend=now-ago;

		//need to analyse first 200mS of the event for these factors

		data[offset]=correctedstart; //in samples since UGen began
		data[offset+1]=correctedend; //in samples since UGen began
		data[offset+2]=0; //transient length (from HFC and ZC stability?)
		data[offset+3]=calculatePAT2(unit); //perceptual attack time (from some event assessment algorithm)
		data[offset+4]=calculateTimbre(unit); //event categorisation type
		data[offset+5]=0; //perceived loudness of event, weighted to first 200mS
		data[offset+6]=calculatePitch(unit); //(monophonic) f0
		data[offset+7]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc
		data[offset+8]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc
		data[offset+9]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc

		//all quantising, time analysis done in Lang where more flexible, not here
		//data[offset+5]=0; //closest 0.25 beat from input clock signal
		//data[offset+6]=0; //time deviation from closest 0.25


		data[0]=(float)(unit->m_numstored);

		//printf("send an event? start %d end %d length %d circ %d \n",correctedstart, correctedend,(correctedend-correctedstart), unit->m_circular);

		//if event length greater than 40mS, also tried 2500 samp=50mS before
		//was 1600, 512= about 11ms
		if((correctedend-correctedstart)>512) {

			if(unit->m_circular>0.5) {

				//printf("sent an event %d \n", correctedend);

				//printf("send trigger %d \n", (unit->m_triggerid));

				if(unit->m_triggerid) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, unit->m_numstored);

				unit->m_numstored=(unit->m_numstored+1)%CIRCBUFSIZE;
			} else {

				unit->m_numstored=unit->m_numstored+1;

			}

		}

	}

	// else {
	//printf("failed to send an event %d %d \n", (unit->m_numstored), unit->m_maxEvents);
	//}

	unit->m_recording=false;

}



//energy integration algorithm converted to C code

//using loudness frame values
float calculatePAT(AnalyseEvents2 *unit) {

	int lstartframe=unit->m_lstartframe;
	//int lendframe=unit->m_lendframe;
	int frames= unit->m_numframes;
	float * loudness= unit->m_loudness;

	//(unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;

	float running=0.0;

	int pat=1; //default to immediate in case no result

	int pos, base;

	base= lstartframe + LOUDNESSSTORED;

	for (int i=0; i<frames; ++i) {

		pos= (i+ base)%LOUDNESSSTORED;

		running= running + (loudness[pos]*0.001); //1/1000 is 1/10*100ie 100 dB * 10 frames on average to overload 1 ie 100mS

		if(running > 1) {
			pat=i+1; //always at least a frame, 10mS preschedule is standard
			break; //no need to calculate any further
		}

	}

	//printf("pat calc lstart %d lend %d frames %d pat %f \n", lstartframe, lendframe, frames,(pat*FRAMEPERIOD));

	return (pat*FRAMEPERIOD);

	//sc_max((pat*FRAMEPERIOD)- 0.0836,0.005); //adjustment for model's impulse reference tone = 80mS but this only gives relative PAT, not an absolute solution
	//should really spot if maxima reached quickly, ie attack envelope percussive, minimise PAT. Else slower attack, other matching criteria...

}



float calculatePAT2(AnalyseEvents2 *unit) {
	int i;
	int lstartframe=unit->m_lstartframe;
	//int lendframe=unit->m_lendframe;
	int frames= unit->m_numframes;
	float * patband1= unit->m_patband1;
	float * patband2= unit->m_patband2;
	//initialise feature vector for PAT
	int pos, base;

	base= lstartframe + LOUDNESSSTORED;

	for(i=0;i<10;++i) {

		pos= (i+ base)%LOUDNESSSTORED;

		unit->m_featurevector[i*2]=patband1[pos];
		unit->m_featurevector[i*2+1]=patband2[pos];
	}

	//zero any end frames if event shorter than 10 FFT frames
	if(frames<10) {

		for(i=frames;i<10;++i) {

			//pos= (i+ base)%LOUDNESSSTORED;

			unit->m_featurevector[i*2]=0.0;
			unit->m_featurevector[i*2+1]=0.0;
		}

	}

	float pat=calculatePATnn(unit);

	//printf("calc PAT NN %f \n",pat);

	if(pat <(-0.005)) pat=(-0.005);

	if(pat>0.1) pat=0.1;

	return pat;
}



double houtput[20];
double g_mincut= -708.3964;
double g_maxcut= 36.0437;

float calculatePATnn(AnalyseEvents2 *unit) {
	int i,j;
	double sum;
	int baseindex;
	double * inputs;

	inputs= unit->m_featurevector;

	for(i=0;i<20;++i) {
		baseindex= 20*i;
		sum=0.0;

		for (j=0;j<20;++j) {
			sum+= (g_w1[baseindex+j])*(inputs[j]);
		}

		houtput[i]= tanh(sum+g_b1[i]);
	}

	sum=g_b2;

	for (j=0;j<20;++j) {
		sum+= (g_w2[j])*(houtput[j]);
	}

	sum= sc_min(sum, g_maxcut);
	sum= sc_max(sum, g_mincut);

	double output = 1./(1 + exp(-sum));

	return output;

}



//512 window, count all positive or negative zero crossings
void countZeroCrossing(AnalyseEvents2 *unit) {
	int i;

	//used just before FFT, so time domain samples at this point
	float * store=unit->m_FFTBuf;

	int nextlcounter=(unit->m_loudnesscounter+1)%LOUDNESSSTORED;

	int count=0;

	float nextval=store[0];

	for (i=0; i<511; ++i) {

		float val= nextval;
		nextval= store[i+1];

		//if ( (  ( val>=(-0.00000001)) && (nextval<0.00000001) ) ||  ( (val<0.00000001) && (nextval>=(-0.00000001)) )  ) {
		if ( (  ( val<0.0) && (nextval>=0.0) ) ||  ( (val>0.0) && (nextval<=(0.0)) )  ) {
			count+=1;
		}


	}

	unit->m_zcr[nextlcounter]= count;
}


float calculateTimbre(AnalyseEvents2 *unit) {
	int i;
	int lstartframe=unit->m_lstartframe;
	//int lendframe=unit->m_lendframe;
	int frames= unit->m_numframes;
	float * speccentroid= unit->m_speccentroid;
	float * zcr= unit->m_zcr;

	//float * patband2= unit->m_patband2;

	//initialise feature vector for timbre
	int pos, base;

	base= lstartframe + LOUDNESSSTORED;

	for(i=0;i<6;++i) {

		pos= (i+ base)%LOUDNESSSTORED;

		unit->m_timbreFV[i]=zcr[pos];
		unit->m_timbreFV[6+i]=speccentroid[pos];
	}

	//zero any end frames if event shorter than 10 FFT frames
	if(frames<6) {

		for(i=frames;i<6;++i) {

			//pos= (i+ base)%LOUDNESSSTORED;
			unit->m_timbreFV[i]=0.0;
			unit->m_timbreFV[6+i]=8.0;
		}

	}

	float timbre=calculateTimbrenn(unit);

	//printf("calc Timbre NN %f \n",timbre);

	return timbre;
}



double houtput2[12];

float calculateTimbrenn(AnalyseEvents2 *unit) {
	int i,j;
	double sum;
	int baseindex;
	double * inputs;

	inputs= unit->m_timbreFV;

	for(i=0;i<12;++i) {
		baseindex= 12*i;
		sum=0.0;

		for (j=0;j<12;++j) {
			sum+= (g_timbrew1[baseindex+j])*(inputs[j]);
		}

		houtput2[i]= tanh(sum+g_timbreb1[i]);
	}

	sum=g_timbreb2;

	for (j=0;j<12;++j) {
		sum+= (g_timbrew2[j])*(houtput2[j]);
	}

	sum= sc_min(sum, g_maxcut);
	sum= sc_max(sum, g_mincut);

	double output = 1./(1 + exp(-sum));

	if (output < 0.165)
		output=1.0; //kick
	else if(output>=0.5)
		output=2.0; //snare
	else
		output=3.0; //hihat

	return output;

}








#include<stdlib.h>

int cmp(const void* vp, const void* vq);

int cmp(const void* vp,const void* vq) {
	const float *p= static_cast<const float*>(vp);
	const float *q= static_cast<const float*>(vq);

	float diff= *p - *q;

	return ((diff>=0.0) ? ((diff >0.0) ? -1 : 0) : +1);
}

//get best pitch average across event blocks
//find most common in rough note based histogram, then average freqs in that note bin?
//or find median, sort data then select middle index

float calculatePitch(AnalyseEvents2 *unit) {

	int startblock= unit->m_startblock;

	int endblock=unit->m_endblock;

	int numblocks = endblock-startblock;

	if(numblocks<1) numblocks= (endblock+MAXBLOCKS-startblock)%MAXBLOCKS;

	float * pitch= unit->m_pitch;
	float * sortbuf= unit->m_sortbuf;

	//copy to sort buffer
	int pos, base;

	base= startblock + MAXBLOCKS;

	for (int i=0; i<numblocks; ++i) {

		pos= (i+ base)%MAXBLOCKS;
		sortbuf[i]=pitch[pos];

	}

	//run sort
	qsort(sortbuf,numblocks,sizeof(float),cmp);

	//printf("median freq %f midinote %f blocks involved %d \n",sortbuf[numblocks/2], 12*log2(sortbuf[numblocks/2]/261.626)+60, numblocks);
	//
	//for (int i=0; i<numblocks; ++i) {
	//		printf("%f ", sortbuf[i]);
	//	}
	//
	//printf("\n\n");

	//select middleindex frequency as result
	return sortbuf[numblocks/2];

}






PluginLoad(BBCut2UGens) {

	ft = inTable;

	DefineDtorUnit(AnalyseEvents2);
}
