/*
 * $Id: XMLStreamEventWriterTest.java,v 1.2 2004-07-15 02:18:48 cniles Exp $
 * 
 * Copyright (c) 2004, Christian Niles, Unit12
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *		*   Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 * 
 *	    *	Redistributions in binary form must reproduce the above copyright
 *          notice, this list of conditions and the following disclaimer in the
 *          documentation and/or other materials provided with the distribution.
 * 
 *      *   Neither the name of Christian Niles, Unit12, nor the names of its
 *          contributors may be used to endorse or promote products derived from
 *          this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 */
package javanet.staxutils;

import java.io.StringWriter;
import java.io.Writer;

import javanet.staxutils.io.StAXStreamWriter;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

/**
 *
 * @author Christian Niles
 * @version $Revision: 1.2 $
 */
public class XMLStreamEventWriterTest extends StAXTestCase {

    public static final QName ELEMENT_NAME = new QName("element");

    public static final QName PREFIXED_ELEMENT_NAME = new QName(
            "http://www.example.com/namespace", "element", "prefix");

    /**
     * Writes a single, text-only element mapped to a undefined prefix to test
     * namespace defaulting.
     */
    public void testDefaulting() throws Exception {

        // create writers
        StringWriter writer = new StringWriter();
        XMLStreamWriter streamWriter = createXMLStreamWriter(writer);
        XMLStreamEventWriter eventWriter = new XMLStreamEventWriter(
                streamWriter);

        // add empty element events
        eventWriter.add(eventFactory.createStartElement(PREFIXED_ELEMENT_NAME,
                null, null));
        eventWriter.add(eventFactory.createCharacters("text"));
        eventWriter.add(eventFactory.createEndElement(PREFIXED_ELEMENT_NAME,
                null));

        // flush the writers
        eventWriter.flush();
        streamWriter.flush();
        writer.flush();

        String xml = writer.getBuffer().toString();
        assertEquals(
                "XML output failed (" + xml + ")",
                "<prefix:element xmlns:prefix=\"http://www.example.com/namespace\">text</prefix:element>",
                xml);

    }

    /**
     * Writes a single, text-only element to test that caching works with
     * text nodes.
     */
    public void testTextElement() throws Exception {

        // create writers
        StringWriter writer = new StringWriter();
        XMLStreamWriter streamWriter = createXMLStreamWriter(writer);
        XMLStreamEventWriter eventWriter = new XMLStreamEventWriter(
                streamWriter);

        // add empty element events
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createCharacters("text"));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));

        // flush the writers
        eventWriter.flush();
        streamWriter.flush();
        writer.flush();

        String xml = writer.getBuffer().toString();
        assertEquals("XML output failed (" + xml + ")",
                "<element>text</element>", xml);

    }

    /**
     * Writes a single, empty element to test that caching works with a small
     * dataset.
     */
    public void testSingleElement() throws Exception {

        // create writers
        StringWriter writer = new StringWriter();
        XMLStreamWriter streamWriter = createXMLStreamWriter(writer);
        XMLStreamEventWriter eventWriter = new XMLStreamEventWriter(
                streamWriter);

        // add empty element events
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));
        
        // flush the writers
        eventWriter.close();
        streamWriter.close();
        writer.close();

        String xml = writer.toString();
        assertEquals("XML output failed (" + xml + ")", "<element/>", xml);

    }

    /**
     * Tests to make sure that two adjacent StartElement and EndElement events
     * are coalesced into an empty element on the underlying XMLStreamWriter.
     */
    public void testEmptyElement() throws Exception {

        // create writers
        StringWriter writer = new StringWriter();
        XMLStreamWriter streamWriter = createXMLStreamWriter(writer);
        XMLStreamEventWriter eventWriter = new XMLStreamEventWriter(
                streamWriter);

        // add empty element events
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));

        // flush the writers
        eventWriter.flush();
        streamWriter.flush();
        writer.flush();

        String xml = writer.getBuffer().toString();
        assertEquals("XML output failed (" + xml + ")",
                "<element><element><element/></element></element>", xml);

    }

    /**
     * Tests the caching of Namespace events, making sure they are
     * properly added to the next StartElement
     */
    public void testNamespaceCaching() throws Exception {

        // create writers
        StringWriter writer = new StringWriter();
        XMLStreamWriter streamWriter = createXMLStreamWriter(writer);
        XMLStreamEventWriter eventWriter = new XMLStreamEventWriter(
                streamWriter);

        // add events
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createStartElement(ELEMENT_NAME, null,
                null));
        eventWriter.add(eventFactory.createNamespace("prefix",
                "http://example.net/namespace"));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));
        eventWriter.add(eventFactory.createEndElement(ELEMENT_NAME, null));

        // flush the writers
        eventWriter.flush();
        streamWriter.flush();
        writer.flush();

        String xml = writer.getBuffer().toString();
        assertEquals(
                "XML output failed (" + xml + ")",
                "<element><element><element xmlns:prefix=\"http://example.net/namespace\"/></element></element>",
                xml);

    }

    protected XMLStreamWriter createXMLStreamWriter(Writer writer)
            throws XMLStreamException {

        return new StAXStreamWriter(writer);

    }

}