/*
 * Print out reservation information from INVENTORY
 */
#include "basil_alps.h"

#include <sys/types.h>
#include <time.h>
#include <grp.h>
#include <pwd.h>
#include <err.h>

/* non-reentrant pretty-printing function */
static const char *_pretty_time(time_t stamp)
{
	static char time_str[256];
	static time_t today;
	struct tm *time_tm = localtime(&stamp);

	if (time_tm == NULL)
		err(1, "failed to decode timestamp %u", (unsigned)stamp);
	if (today == 0)
		time(&today);
	/* Suppress date if timestamp has the same date as today */
	strftime(time_str, sizeof(time_str),
		 ((stamp / 86400) == (today / 86400))
		 ?            "%_I:%M %P"
		 : "%a, %_I:%M %P", time_tm);
	return time_str;
}

static const char *_getgrgid(gid_t gid)
{
	struct group *grp = getgrgid(gid);
	static char gid_to_str[64];

	snprintf(gid_to_str, sizeof(gid_to_str), "%u", gid);
	return grp ? grp->gr_name : gid_to_str;
}

static const char *_getpwuid(uid_t uid)
{
	struct passwd *pwd = getpwuid(uid);
	static char uid_to_str[64];

	snprintf(uid_to_str, sizeof(uid_to_str), "%u", uid);
	return pwd ? pwd->pw_name : uid_to_str;
}

int main(int argc, char **argv)
{
	enum basil_version  version = get_basil_version();
	struct basil_inventory *inv = get_full_inventory(version);
	struct basil_rsvn *rsvn;

	if (inv == NULL)
		errx(1, "failed to get an INVENTORY");

	if (*inv->mpp_host)
		printf("%s host '%s', ", inv->is_gemini ? "XE" : "XT", inv->mpp_host);

	printf("BASIL %s, %s, avail/batch/total: %d/%d/%d\n",
		bv_names_long[version], _pretty_time(inv->timestamp),
		inv->batch_avail, inv->batch_total, inv->nodes_total);

	if (inv->f->rsvn_head)
		printf("\n"
		       " ResId     ApId From          User        Group   PEs  N  d  Memory Time\n");

	for (rsvn = inv->f->rsvn_head; rsvn; rsvn = rsvn->next) {
		struct basil_rsvn_app *app = rsvn->app_head;
		struct basil_rsvn_app_cmd *cmd;
		/*
		 * basil_rsvn_app entries are parsed in stack/LIFO order, hence
		 * the first entry, which refers to the reservation itself and
		 * has a timestamp of 0, is at the end.
		 */
		while (app && app->timestamp)
			app = app->next;

		/*
		 * Sanity checks:
		 * - reservation entry must exist
		 * - there must be exactly one cmd entry
		 * - check the reservation APID also
		 */
		assert(app != NULL && app->cmd_head != NULL);
		assert(app->cmd_head->next == NULL);
		assert(app->apid ==  basil_get_rsvn_apid(inv, rsvn->rsvn_id));

		printf("%6u %8llu batch:%-7s %-8s %8s  %4u %2u %2u  %uMB %s",
			rsvn->rsvn_id, (unsigned long long)app->apid, rsvn->batch_id,
			rsvn->user_name,
			/* account_name is 'DEFAULT' on PBS */
			_getgrgid(app->group_id),
			app->cmd_head->width,
			app->cmd_head->nppn,
			app->cmd_head->depth,
			app->cmd_head->memory,
			_pretty_time(rsvn->timestamp));

		if (version >= BV_3_1)
			printf(", mode %s, gpc %s", nam_rsvn_mode[rsvn->rsvn_mode],
				nam_gpc_mode[rsvn->gpc_mode]);
		printf("\n");

		for (app = rsvn->app_head; app; app = app->next) {
			/* FIXME: stack order */
			if (app->timestamp == 0)
				continue;

			printf("%-6s %8llu               %-8s %8s ", " ",
				(unsigned long long)app->apid,
				_getpwuid(app->user_id),
				_getgrgid(app->group_id));

			cmd = app->cmd_head;
			if (cmd->next != NULL) {
				for ( ; cmd; cmd = cmd->next) 
  					printf(" %4u %2u %2u  %uMB %s",
						cmd->width, cmd->nppn, cmd->depth,
						cmd->memory, cmd->cmd);
				printf("%s\n", _pretty_time(app->timestamp));
			} else {
				printf(" %4u %2u %2u  %uMB %s, %s\n",
					cmd->width, cmd->nppn, cmd->depth, cmd->memory,
					_pretty_time(app->timestamp),
					cmd->cmd);
			}
		}
	}
	free_inv(inv);

	return EXIT_SUCCESS;
}
