;;;; Loading shared object files

;;;; This software is part of the SBCL system. See the README file for
;;;; more information.
;;;;
;;;; This software is derived from the CMU CL system, which was
;;;; written at Carnegie Mellon University and released into the
;;;; public domain. The software is in the public domain and is
;;;; provided with absolutely no warranty. See the COPYING and CREDITS
;;;; files for more information.

(in-package "SB-ALIEN")

;;; Used to serialize modifications to *shared-objects*.
(define-load-time-global *shared-objects-lock*
  (sb-thread:make-mutex :name "shared object list lock"))

(define-unsupported-fun load-foreign
    "Unsupported as of SBCL 0.8.13. See LOAD-SHARED-OBJECT."
  "~S is unsupported as of SBCL 0.8.13. See LOAD-SHARED-OBJECT."
  (load-foreign))

(define-unsupported-fun load-1-foreign
    "Unsupported as of SBCL 0.8.13. Please use LOAD-SHARED-OBJECT."
  "~S is unsupported as of SBCL 0.8.13. Please use LOAD-SHARED-OBJECT."
  (load-1-foreign))

(defvar *runtime-dlhandle*)

(defvar *shared-objects*)

(defstruct (shared-object (:copier nil)) pathname namestring handle dont-save)
(declaim (freeze-type shared-object))

(defun load-shared-object (pathname &key dont-save)
  "Load a shared library / dynamic shared object file / similar foreign
container specified by designated PATHNAME, such as a .so on an ELF platform.

Locating the shared object follows standard rules of the platform, consult the
manual page for dlopen(3) for details. Typically paths specified by
environment variables such as LD_LIBRARY_PATH are searched if the PATHNAME has
no directory, but on some systems (eg. Mac OS X) search may happen even if
PATHNAME is absolute. (On Windows LoadLibrary is used instead of dlopen(3).)

On non-Windows platforms calling LOAD-SHARED-OBJECT again with a PATHNAME
EQUAL to the designated pathname of a previous call will replace the old
definitions; if a symbol was previously referenced through the object and
is not present in the reloaded version an error will be signalled. Reloading
may not work as expected if user or library-code has called dlopen(3) on the
same shared object or running on a system where dlclose(3) is a noop.

LOAD-SHARED-OBJECT interacts with SB-EXT:SAVE-LISP-AND-DIE:

1. If DONT-SAVE is true (default is NIL), the shared object will be dropped
when SAVE-LISP-AND-DIE is called -- otherwise shared objects are reloaded
automatically when a saved core starts up. Specifying DONT-SAVE can be useful
when the location of the shared object on startup is uncertain.

2. On most platforms references in compiled code to foreign symbols in shared
objects (such as those generated by DEFINE-ALIEN-ROUTINE) remain valid across
SAVE-LISP-AND-DIE. On those platforms where this is not supported, a WARNING
will be signalled when the core is saved -- this is orthogonal from DONT-SAVE."
  (let ((pathname (pathname pathname)))
    (sb-thread:with-mutex (*shared-objects-lock*)
      (let* ((old (find pathname *shared-objects*
                        :key #'shared-object-pathname
                        :test #'equal))
             (obj (or old (make-shared-object
                           :pathname pathname
                           :namestring (native-namestring
                                        (translate-logical-pathname pathname)
                                        :as-file t)))))
        (setf (shared-object-dont-save obj) dont-save)
        ;; FIXME: Why doesn's dlopen-or-lose on already loaded stuff work on
        ;; Windows?
        ;;
        ;; Kovalenko 2010-11-24: It would work, but it does nothing
        ;; useful on Windows: library reference count is increased
        ;; after each LoadLibrary, making it harder to unload it, and
        ;; that's all the effect. Also, equal pathnames on Windows
        ;; always designate _exactly the same library image_; Unix
        ;; tricks like deleting an open library and replacing it with
        ;; another version just don't work here.
        #-win32
        (dlopen-or-lose obj)
        #+win32
        (unless old
          (dlopen-or-lose obj))
        (setf *shared-objects* (append (remove obj *shared-objects*)
                                       (list obj)))
        (when (or old (cdr *linkage-info*))
          ;; If OLD is non-NIL, then we're passing "true" which causes all foreign
          ;; symbols to get looked up again. Otherwise we're passing "false"
          ;; which only tries to find symbols that aren't already found.
          (update-linkage-table old))))
    pathname))

(defun unload-shared-object (pathname)
  "Unloads the shared object loaded earlier using the designated PATHNAME with
LOAD-SHARED-OBJECT, to the degree supported on the platform.

Experimental."
  (let ((pathname (pathname pathname)))
    (sb-thread:with-mutex (*shared-objects-lock*)
      (let ((old (find pathname *shared-objects*
                       :key #'shared-object-pathname
                       :test #'equal)))
        (when old
          (dlclose-or-lose old)
          (setf *shared-objects* (remove old *shared-objects*))
          (update-linkage-table t)
          ;; Return T for unloaded, vs whatever update-linkage-info returns
          t)))))

(defun try-reopen-shared-object (obj)
  (declare (type shared-object obj))
  (tagbody :dlopen
     (restart-case
         (dlopen-or-lose obj)
       (continue ()
         :report "Skip this shared object and continue."
         ;; By returning NIL the shared object is dropped from the list.
         (setf (shared-object-handle obj) nil)
         (return-from try-reopen-shared-object nil))
       (retry ()
         :report "Retry loading this shared object."
         (go :dlopen))
       (change-pathname ()
         :report "Specify a different pathname to load the shared object from."
         (tagbody :query
            (format *query-io* "~&Enter pathname (evaluated):~%")
            (force-output *query-io*)
            (let ((pathname (ignore-errors (pathname (read *query-io*)))))
              (unless (pathnamep pathname)
                (format *query-io* "~&Error: invalid pathname.~%")
                (go :query))
              (setf (shared-object-pathname obj) pathname)
              (setf (shared-object-namestring obj)
                    (native-namestring (translate-logical-pathname pathname)
                                       :as-file t))))
         (go :dlopen))))
  obj)

;;; Open libraries in *SHARED-OBJECTS* and the runtime. Called during
;;; initialization.
(defun reopen-shared-objects ()
  ;; Ensure that the runtime is open
  (setf *runtime-dlhandle* (dlopen-or-lose))
  ;; Without this many symbols aren't accessible.
  #+android (load-shared-object "libc.so" :dont-save t)
  ;; Reopen stuff.
  (setf *shared-objects*
        (remove nil (mapcar #'try-reopen-shared-object *shared-objects*))))

;;; Close all dlopened libraries and clear out sap entries in
;;; *SHARED-OBJECTS*, and drop the ones with DONT-SAVE set.
(defun close-shared-objects ()
  (let (saved)
    (dolist (obj (reverse *shared-objects*))
      (dlclose-or-lose obj)
      (unless (shared-object-dont-save obj)
        (push obj saved)))
    (setf *shared-objects* saved))
  (dlclose-or-lose))
