// Copyright (C) 2007-2012  CEA/DEN, EDF R&D, OPEN CASCADE
//
// Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
// CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
// See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//

//  File   : DataIdFilter.hxx
//  Author : Eric Fayolle (EDF)
//  Module : KERNEL
//
/*   Module Filtre
 *   -------------
 *
 *   Implemente les fonctions de filtrage et conversion d'un port de DATASTREAM
 *
 *   Rappel des fonctions du Filtrage:
 *   --------------------------------
 *
 *   Dans une communication de type DATASTREAM, le destinataire indique  l'avance la liste
 *   des instances qu'il veut recevoir, c'est  dire celles qui lui sont ncessaires.
 *   Il indique pour cela la liste des 'times' et la liste des 'tags' qui
 *   caractrisent les instances dsires.
 *   Ces deux listes sont indpendantes. Toute instance dont les paramtres 'time' et
 *   'tag' figurent dans la liste des 'times' et respectivement dans la liste des
 *   'tags' est dsire par le destinataire.
 *   Par la suite, une telle instance sera accepte par le port-DATASTREAM. Les autres
 *   seront rejetes.
 *
 *   Le filtrage consiste  limiter les valeurs possibles du paramtre TIME ou TAG (un
 *   entier). La liste des valeurs possibles est dcrite sous la forme d'une liste de
 *   valeurs ou de squences arithmtiques de valeurs.
 *   Exemple: 
 *     La liste 1; 3; 30:34; 40:50:2 autorise les valeurs 1 et 3 et toutes les valeurs
 *     comprises entre 30 et 34 inclus et toutes les valeurs de la squence 40  50
 *     inclus par pas de 2, c'est  dire 40, 42, ... 50.
 *   On appelle rgle lmentaire de filtrage celle spcifiant un lment de la liste
 *   des valeurs autorises: soit une seule valeur, soit une squence de valeurs. Une
 *   squence de valeurs est spcifie par sa valeur de dpart, sa valeur de fin et
 *   son pas. Le filtrage est donc dfini par une suite de rgles de filtrage.
 *   La fonction lmentaire de configuration du filtrage sert  spcifier une rgle
 *   de filtrage.
 *
 *   Rappels des fonctions de conversion:
 *   -----------------------------------
 *
 *   La conversion est intimement lie au filtrage car seules les valeurs passant le
 *   filtre sont converties. La conversion n'est pas obligatoire. Une valeur de TIME ou TAG
 *   entrante peut ne pas tre convertie. Elle garde alors sa valeur et est garde
 *   telle quelle pour l'objet destinataire.
 *   DATASTREAM peut associer une rgle de conversion  chaque rgle lmentaire de
 *   filtrage.
 *   La conversion consiste  changer:
 *     - un valeur de TIME ou TAG en une valeur diffrente
 *     - une squence de valeurs en une autre squence de valeurs de mme taille
 *       (ex: 30:40 en 300:400:10)
 *   Mais la conversion permet aussi de transformer:
 *     - une valeur de TIME ou TAG unique en une squence de valeurs (les donnes entrantes sont
 *       alors duppliques et  chaque fois que l'objet destinataire rclame une donne
 *       de la squence, il reoit en fait une copie de la donne reue une seule fois)
 *
 *     - une squence de valeurs en une valeur unique (alors, chaque donne entrante
 *       associe  un TIME ou TAG de la squence correspond  une donne unique pour le
 *       destinataire: seule la dernire reue est la donne valide)
 *
 */

#include <vector>
#include <iostream>

// Pour l'utilisation de "vector" de la STL
// Classe filtre_elementaire
//
// Implmente une structure de donne dcrivant un filtre lmentaire
// sur le paramtre TIME ou TAG; c'est
//    - soit une valeur entire unique
//    - soit une squence arithmtique de valeurs
//
class filtre_elementaire
{
public:
    int len;    // Longueur de squence ou 1 pour une valeur unique
    int debut;  // Dbut de la squence ou valeur pour une valeur unique
    int fin;    // Fin de la squence
    int pas;    // Pas de la squence

    // Constructeur par dfaut
    filtre_elementaire() {}
    
    // Cration d'un filtre lmentaire pour une valeur unique
    filtre_elementaire(int valeur)
    {
        this->len = 1;
        this->debut = valeur;
    }

    // Cration d'un filtre lmentaire pour une squence de valeurs entires
    // Le pas par dfaut est 1
    filtre_elementaire (int _debut, int _fin, int _pas=1)
    {
        this->debut = _debut;
        this->len = (_fin - _debut) / _pas;
        if (this->len > 0)
        {
            this->fin = _debut + _pas * this->len;  // Calcule la vrai borne de fin
            this->pas = _pas;
            this->len += 1;   // Compte les bornes et non les intervalles
        }
        else  // erreur de spcification: on ne prend que la premire valeur
            this->len = 1;
    }

    // Constructeur par copie
    filtre_elementaire (filtre_elementaire &_f)
    {
      this->len = _f.len;
      this->debut = _f.debut;
      this->fin = _f.fin;
      this->pas = _f.pas;
    }
};

// Classe filtre_conversion
//
// Implmente le filtrage et la conversion du paramtre TIME ou TAG
// des donnes reues par un port DATASTREAM.
//
// Mode d'emploi:
//      1) Cration d'un objet
//      2) Configuration de cet objet par passage de paramtres
//         de filtage et de conversion
//      3) A la cration d'un port DATASTREAM, on passe au constructeur
//         deux objets 'filtre_conversion', l'un pour le TIME, l'autre pour le TAG.
//      4) A l'utilisation du port DATASTREAM, celui-ci appelle la mthode
//         "applique_filtre_conversion" pour oprer
//
class filtre_conversion
{
private:
    // Structure de donnes dcrivant une conversion lmentaire:
    // un filtre lementaire
    // et un pointeur ventuel vers les paramtres de conversion associs
    class conversion_elementaire
    {
    public :
        // Data
        filtre_elementaire filtre;
        filtre_elementaire * p_convers;

        // Constructeur
        conversion_elementaire() {}

        // Constructeur par copie d'un objet non modifie (const)
        conversion_elementaire (const conversion_elementaire& _ce)
        {
            *this = _ce;
        }
        // Remarque: le Constructeur par copie d'un objet existe par defaut mais sans le modificateur 'const'
        //           et l'emploi d'un objet comme element dans un vecteur oblige d'avoir un tel const-copy-constructor.
    };

    // Donnes de configuration de filtrage et conversion:
    //    une table de filtres lmentaires
    //    avec leurs donnes de conversion associes ventuelles
    std::vector<conversion_elementaire> config;

public:
    // Constructeur: juste une allocation mmoire initiale
    filtre_conversion() {}

    // Destructeur:
    // rclamer la mmoire utilise par tous les lments du vecteur config
    ~filtre_conversion()
    {
        std::vector<conversion_elementaire>::iterator i;
        for (i = this->config.begin(); i != this->config.end(); i ++)
        {
            delete (*i).p_convers;
        }
    }

    // Configuration partielle par ajout d'un filtre lmentaire
    bool config_elementaire (filtre_elementaire& _f)
    {
//    cout << "ajout config_elementaire 1  " << this << endl;
        conversion_elementaire conv_elem;
        
        conv_elem.filtre = _f;
        conv_elem.p_convers = NULL;

        // Ajoute cette conversion/filtrage elementaire a la liste
        this->config.push_back (conv_elem);
    
//    vector<conversion_elementaire>::iterator i;
//    cout << "liste apres ajout:" << endl;
//    for (i = this->config.begin(); i != this->config.end(); i ++)
//    {
//        cout << "config elem   " << endl;
//        cout << "filtre: len, debut, fin, pas " << (*i).filtre.len << " " << (*i).filtre.debut << " " << (*i).filtre.fin << " " << (*i).filtre.pas << endl;
//    }
        
        return true;
    }
    
    // Configuration partielle par ajout d'un filtre lmentaire
    // et sa conversion associe
    //
    // Retourne false si les param de conversion sont incompatibles avec le filtre lmentaire.
    // La configuration partielle est alors refuse.
    //
    bool config_elementaire (filtre_elementaire& _f, filtre_elementaire& _conv)
    {
//    cout << "ajout config_elementaire 2  " << this << endl;
    
        if (_f.len == 1 || _conv.len == 1 || _f.len == _conv.len)
        {
            conversion_elementaire conv_elem;
            conv_elem.filtre = _f;
            conv_elem.p_convers = new filtre_elementaire(_conv);

            // Ajoute cette conversion/filtrage elementaire a la liste
            this->config.push_back (conv_elem);
    
//    vector<conversion_elementaire>::iterator i;
//    cout << "liste apres ajout:" << endl;
//    for (i = this->config.begin(); i != this->config.end(); i ++)
//    {
//        cout << "config elem   " << endl;
//        cout << "filtre: len, debut, fin, pas " << (*i).filtre.len << " " << (*i).filtre.debut << " " << (*i).filtre.fin << " " << (*i).filtre.pas << endl;
//    }
        
            return true;
        }
        else
        {
            // Filtre et conversion incompatibles
            return false;
        }
    }

    // applique_filtre_conversion: Opration du filtre et de la conversion
    template <typename T > T applique_filtre_conversion (T valeur_initiale, std::vector<T>& liste_conversions) const;
};



// filtre_conversion::applique_filtre_conversion: Opration du filtre et de la conversion
//
// Etant donn une valeur entire (de TIME ou de TAG), cette mthode dtermine :
//   - si cette valeur passe le filtre
//   - dans le cas o une conversion existe, la liste des valeurs de conversion
//     qui correspondent  la valeur initiale
//
// Dans tous les cas, cette mthode retourne une liste de valeurs.
// Dans le cas o il n'y a pas de conversion, cette liste a une longueur 1
// et ne contient que la valeur initiale.
//
// Paramtre d'entre :         la valeur initiale (integer)
//
// Paramtre de sortie :        la liste des valeurs aprs conversion (vector<int>)
//
// Valeur de retour :           la longueur de la liste
//     si cette longueur est 0, c'est que la valeur initiale ne passe pas le filtre
//
template <typename T>
T filtre_conversion::applique_filtre_conversion (T valeur_initiale, std::vector<T>& liste_conversions) const
{
    // Part d'une liste vierge
    liste_conversions.clear();

//    cout << "config applique_filtre_conversion " << this << endl;
    
    // Balaye tous les lments de configuration
    // et cherche pour chacun d'eux si la valeur initiale est prsente parmi les valeurs filtres

    // Pour tous les lments de configuration du filtrage/conversion
    std::vector<conversion_elementaire>::const_iterator i;
    for (i = config.begin(); i != config.end(); i ++)
    {

//    cout << "config elem   " << endl;
//    cout << "filtre: len, debut, fin, pas " << (*i).filtre.len << " " << (*i).filtre.debut << " " << (*i).filtre.fin << " " << (*i).filtre.pas << endl;
    
        bool si_passe_filtre = false;

        // Si la longueur du filtre est 1
        if ((*i).filtre.len == 1) {
          // Si la valeur initiale correspond  la valeur du filtre
          if ((*i).filtre.debut == valeur_initiale)
            si_passe_filtre = true;
        } else  {
          // Si la valeur initiale est dans la squence des valeurs du filtre
          //   la valeur est comprise dans les bornes [debut,fin]
          //   et sa distance du dbut de la squence est modulo le pas
          if (  ((*i).filtre.fin - valeur_initiale >= 0) == (valeur_initiale - (*i).filtre.debut >= 0)
                &&  (valeur_initiale - (*i).filtre.debut) % (*i).filtre.pas == 0  ) {
            si_passe_filtre = true;
          }
        }

        // Si la valeur initiale passe le filtre
        if (si_passe_filtre) {
          //    cout << "config: filtre passe    " << endl;
            
          // Si il y a une conversion  effectuer
          if ((*i).p_convers != NULL) {

            // Si la longueur du filtre est 1
            if ((*i).filtre.len == 1) {

              // Si la longueur des paramtres de conversion est aussi 1
              if ((*i).p_convers->len == 1) {
                // Ajoute la valeur de conversion  la liste des valeurs aprs conversion
                liste_conversions.push_back ((*i).p_convers->debut);
              } else {
                // Ajoute la squence de conversion  la liste des valeurs aprs conversion
                for (int s = (*i).p_convers->debut; s != (*i).p_convers->fin; s += (*i).p_convers->pas) {
                  liste_conversions.push_back (s);
                }
                liste_conversions.push_back ((*i).p_convers->fin);
              }

            } else {
              // Le filtre est une squence qui est convertie en une autre squence de mme longueur
              // Choisit la valeur au rang dsir dans la squence de conversion
              int rang = (valeur_initiale - (*i).filtre.debut) / (*i).filtre.pas;

              int valeur_convertie = (*i).p_convers->debut + rang * (*i).p_convers->pas;

              // Ajoute cette valeur  la liste des valeurs aprs conversion
              liste_conversions.push_back (valeur_convertie);
            }
          } else {
            // Ajoute la valeur initiale telle-quelle  la liste des valeurs aprs conversion
            liste_conversions.push_back (valeur_initiale);
          }
        }
    }

    return liste_conversions.size();
}
