# ifndef _SKIT_ASR_H
# define _SKIT_ASR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:
NAME: @code{asr} - associative sparse row matrix 
DESCRIPTION:       
 The associative sparse row (ASR) format is the basic
 format for incremental building of sparse matrix, as:
@example
	a.entry(i,j) += 3.14;
@end example

 A declaration whithout any parametrers correspond to a matrix with null size:
@example
	asr<double> a;
@end example
 The constructor can be invocated whith row and column numbers.
@example
	asr<double> a(nrow, ncol);
@end example
 
 Conversion beetween the @code{csr} and the @code{asr} are available: *TODO*
@example
          csr<double> c = ...
          asr<double> a = asr(c);
@end example
 or 
@example
          csr<double> c2 = csr(a);
@end example
 
 The read-only access "a(i,j)" return the associated value, or
 zero if the entry is not represented:
@example
          cout << a(i,j) << endl;
@end example

 For input/output and liner algebra, please convert to the
 more efficient read-only @code{csr} format (see @ref{csr class}).

NOTE:
 The  data structure consists of an array row(nrow).
 Each list row(i), i=1..nrow contains the non-zero
 entries (j, aij) in an associative red-black tree.
 This format allows fast retrival of a value aij for a given (i,j)
 pair.
 
 The @code{asr} class supports also a row-oriented STL interface,
 and is implemented using the STL map associative container.

AUTHOR: 
     Pierre Saramito
   | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:   5 march 1997
METHODS: @asr
End:
*/
# include "rheolef/asrrep.h"
namespace rheolef { 
//<asr:
template <class T>
class asr : smart_pointer<asrrep<T> >
{
  public:
    asr (Index nrow, Index ncol) : smart_pointer<asrrep<T> >(new_macro(asrrep<T>(nrow,ncol))) {}
    Index nrow() const { return data().nrow(); }
    Index ncol() const { return data().ncol(); }
    Index nnz()  const { return data().nnz(); }

    // direct read/write access to values via (i,j)
    T  operator() (Index i, Index j) const
    {
        assert_macro (i < nrow(), "index #1 " << i << " out of range 0.." << nrow() - 1);
        assert_macro (j < ncol(), "index #2 " << j << " out of range 0.." << ncol() - 1);
        return data().operator()(i,j);
    }
    T& entry      (Index i, Index j)
    {
        assert_macro (i < nrow(), "index #1 " << i << " out of range 0.." << nrow() - 1);
        assert_macro (j < ncol(), "index #2 " << j << " out of range 0.." << ncol() - 1);
        return data().entry(i,j);
    }
    // row access via iterator
    typedef typename asrrep<T>::const_iterator const_iterator;
    const_iterator begin() const { return data().begin(); }
    const_iterator end() const { return data().end(); }
    
    // direct row access via [i]
    typedef typename asrrep<T>::row row;
    const row& operator[](Index i) const
    {
        assert_macro (i < nrow(), "index " << i << " out of range 0.." << nrow() - 1);
        return data().operator[](i);
    }
  protected:
	const asrrep<T>& data() const {
		return smart_pointer<asrrep<T> >::data();
	}
	asrrep<T>& data() {
		return smart_pointer<asrrep<T> >::data();
	}
};
//>asr:
}// namespace rheolef
# endif // _SKIT_ASR_H
