% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/periods.r
\name{period}
\alias{period}
\alias{periods}
\alias{is.period}
\alias{seconds}
\alias{minutes}
\alias{hours}
\alias{days}
\alias{weeks}
\alias{years}
\alias{milliseconds}
\alias{microseconds}
\alias{nanoseconds}
\alias{picoseconds}
\alias{months.numeric}
\title{Create or parse period objects}
\usage{
period(num = NULL, units = "second", ...)

is.period(x)

seconds(x = 1)

minutes(x = 1)

hours(x = 1)

days(x = 1)

weeks(x = 1)

years(x = 1)

milliseconds(x = 1)

microseconds(x = 1)

nanoseconds(x = 1)

picoseconds(x = 1)

\method{months}{numeric}(x, abbreviate)
}
\arguments{
\item{num}{a numeric or character vector. A character vector can specify
periods in a convenient shorthand format or ISO 8601 specification. All
unambiguous name units and abbreviations are supported, "m" stands for
months, "M" for minutes unless ISO 8601 "P" modifier is present (see
examples). Fractional units are supported but the fractional part is always
converted to seconds.}

\item{units}{a character vector that lists the type of units to be used. The
units in units are matched to the values in num according to their
order. When \code{num} is character, this argument is ignored.}

\item{...}{a list of time units to be included in the period and their
amounts. Seconds, minutes,  hours, days, weeks, months, and years are
supported. Normally only one of \code{num} or \code{...} are present. If both are
present, the periods are concatenated.}

\item{x}{Any R object for \code{is.periods} and a numeric value of the number of
units for elementary constructors. With the exception of seconds(), x must
be an integer.}

\item{abbreviate}{Ignored. For consistency with S3 generic in base namespace.}
}
\value{
a period object
}
\description{
\code{period()} creates or parses a period object with the specified values.
}
\details{
Within a Period object, time units do not have a fixed length (except for
seconds) until they are added to a date-time. The length of each time unit
will depend on the date-time to which it is added. For example, a year that
begins on 2009-01-01 will be 365 days long.  A year that begins on 2012-01-01
will be 366 days long. When math is performed with a period object, each unit
is applied separately. How the length of a period is distributed among its
units is non-trivial. For example, when leap seconds occur 1 minute is longer
than 60 seconds.

Periods track the change in the "clock time" between two date-times. They
are measured in common time related units: years, months, days, hours,
minutes, and seconds. Each unit except for seconds must be expressed in
integer values.

Besides the main constructor and parser \code{\link[=period]{period()}}, period objects can also
be created with the specialized functions \code{\link[=years]{years()}}, \code{\link[=months]{months()}}, \code{\link[=weeks]{weeks()}},
\code{\link[=days]{days()}}, \code{\link[=hours]{hours()}}, \code{\link[=minutes]{minutes()}}, and \code{\link[=seconds]{seconds()}}. These objects can be added
to and subtracted to date-times to create a user interface similar to object
oriented programming.

Note: Arithmetic with periods can result in undefined behavior when
non-existent dates are involved (such as February 29th in non-leap years).
Please see \linkS4class{Period} for more details and \code{\link{\%m+\%}} and
\code{\link[=add_with_rollback]{add_with_rollback()}} for alternative operations.
}
\examples{

### Separate period and units vectors

period(c(90, 5), c("second", "minute"))
#  "5M 90S"
period(-1, "days")
period(c(3, 1, 2, 13, 1), c("second", "minute", "hour", "day", "week"))
period(c(1, -60), c("hour", "minute"))
period(0, "second")

### Units as arguments

period(second = 90, minute = 5)
period(day = -1)
period(second = 3, minute = 1, hour = 2, day = 13, week = 1)
period(hour = 1, minute = -60)
period(second = 0)
period(c(1, -60), c("hour", "minute"), hour = c(1, 2), minute = c(3, 4))

### Lubridate style parsing

period("2M 1sec")
period("2hours 2minutes 1second")
period("2d 2H 2M 2S")
period("2days 2hours 2mins 2secs")
period("2 days, 2 hours, 2 mins, 2 secs")
# Missing numerals default to 1. Repeated units are added up.
period("day day")

### ISO 8601 parsing

period("P10M23DT23H") # M stands for months
period("10DT10M") # M stands for minutes
period("P3Y6M4DT12H30M5S") # M for both minutes and months
period("P23DT60H 20min 100 sec") # mixing ISO and lubridate style parsing

### Comparison with characters (from v1.6.0)

period("day 2 sec") > "day 1sec"

### Elementary Constructors

x <- ymd("2009-08-03")
x + days(1) + hours(6) + minutes(30)
x + days(100) - hours(8)

class(as.Date("2009-08-09") + days(1)) # retains Date class
as.Date("2009-08-09") + hours(12)
class(as.Date("2009-08-09") + hours(12))
# converts to POSIXt class to accomodate time units

years(1) - months(7)
c(1:3) * hours(1)
hours(1:3)

# sequencing
y <- ymd(090101) # "2009-01-01 CST"
y + months(0:11)

# compare DST handling to durations
boundary <- ymd_hms("2009-03-08 01:59:59", tz = "America/Chicago")
boundary + days(1) # period
boundary + ddays(1) # duration
is.period(as.Date("2009-08-03")) # FALSE
is.period(period(months = 1, days = 15)) # TRUE
}
\seealso{
\linkS4class{Period}, \code{\link[=period]{period()}}, \code{\link{\%m+\%}},
\code{\link[=add_with_rollback]{add_with_rollback()}}
}
\keyword{chron}
\keyword{classes}
