
// All rights reserved.
//
// Author: Li Liu
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#include <cassert>
#include <cstring>
#include <sys/types.h>
#include <sys/user.h>

#include "client/linux/handler/cpu_context.h"
#include "client/linux/handler/linux_thread.h"

namespace google_breakpad {

CPUContext::CPUContext(const struct user_regs_struct *user_regs,
                       const struct user_fpregs_struct *fp_regs,
                       const DebugRegs *debug_regs) :
    user_regs_(user_regs),
    fp_regs_(fp_regs),
    debug_regs_(debug_regs) {
}

uintptr_t CPUContext::GetFramePointer() const {
  assert(user_regs_);
#ifdef __i386__
  return user_regs_->ebp;
#elif defined(__x86_64__)
  return user_regs_->rbp;
#endif
}

uintptr_t CPUContext::GetStackPointer() const {
  assert(user_regs_);
#ifdef __i386__
  return user_regs_->esp;
#elif defined(__x86_64__)
  return user_regs_->rsp;
#endif
}

bool CPUContext::CopyTo(RawContext *context) const {
  assert(user_regs_ || fp_regs_ || debug_regs_);
  if (!(user_regs_ || fp_regs_ || debug_regs_))
    return false;

  if (user_regs_)
    CopyGeneralRegisters(context);
  if (fp_regs_)
    CopyFloatingPointRegisters(context);
  if (debug_regs_)
    CopyDebugRegisters(context);

  return true;
}

#ifdef __i386__
void CPUContext::CopyGeneralRegisters(RawContext *context) const {
  context->context_flags = MD_CONTEXT_X86_FULL;
  context->cs = user_regs_->xcs;
  context->ds = user_regs_->xds;
  context->es = user_regs_->xes;
  context->fs = user_regs_->xfs;
  context->gs = user_regs_->xgs;
  context->ss = user_regs_->xss;
  context->edi = user_regs_->edi;
  context->esi = user_regs_->esi;
  context->ebx = user_regs_->ebx;
  context->edx = user_regs_->edx;
  context->ecx = user_regs_->ecx;
  context->eax = user_regs_->eax;
  context->ebp = user_regs_->ebp;
  context->eip = user_regs_->eip;
  context->esp = user_regs_->esp;
  context->eflags = user_regs_->eflags;
}
#endif

#ifdef __x86_64__
void CPUContext::CopyGeneralRegisters(RawContext *context) const {
  context->context_flags = MD_CONTEXT_AMD64_CONTROL |
                           MD_CONTEXT_AMD64_INTEGER |
                           MD_CONTEXT_AMD64_SEGMENTS;
  context->cs = user_regs_->cs;
  context->ds = user_regs_->ds;
  context->es = user_regs_->es;
  context->fs = user_regs_->fs;
  context->gs = user_regs_->gs;
  context->ss = user_regs_->ss;
  context->eflags = user_regs_->eflags;
  context->rip = user_regs_->rip;
  context->rax = user_regs_->rax;
  context->rbx = user_regs_->rbx;
  context->rcx = user_regs_->rcx;
  context->rdx = user_regs_->rdx;
  context->rsp = user_regs_->rsp;
  context->rbp = user_regs_->rbp;
  context->rsi = user_regs_->rsi;
  context->rdi = user_regs_->rdi;
  context->r8 = user_regs_->r8;
  context->r9 = user_regs_->r9;
  context->r10 = user_regs_->r10;
  context->r11 = user_regs_->r11;
  context->r12 = user_regs_->r12;
  context->r13 = user_regs_->r13;
  context->r14 = user_regs_->r14;
  context->r15 = user_regs_->r15;
}
#endif

void CPUContext::CopyFloatingPointRegisters(RawContext *context) const {
#ifdef __i386__
  context->context_flags |= MD_CONTEXT_X86_FLOATING_POINT;
  context->float_save.control_word = fp_regs_->cwd;
  context->float_save.status_word = fp_regs_->swd;
  context->float_save.tag_word = fp_regs_->twd;
  context->float_save.error_offset = fp_regs_->fip;
  context->float_save.error_selector = fp_regs_->fcs;
  context->float_save.data_offset = fp_regs_->foo;
  context->float_save.data_selector = fp_regs_->fos;
  context->float_save.data_selector = fp_regs_->fos;

  memcpy(context->float_save.register_area,
         fp_regs_->st_space,
         sizeof(context->float_save.register_area));
#endif
}

void CPUContext::CopyDebugRegisters(RawContext *context) const {
#ifdef __i386__
  context->context_flags |= MD_CONTEXT_X86_DEBUG_REGISTERS;
  context->dr0 = debug_regs_->dr0;
  context->dr1 = debug_regs_->dr1;
  context->dr2 = debug_regs_->dr2;
  context->dr3 = debug_regs_->dr3;
  context->dr6 = debug_regs_->dr6;
  context->dr7 = debug_regs_->dr7;
#endif
}

}  // namespace google_breakpad
