/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment interactive test common routines.
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include <gtk/gtk.h>
#include <pgm/gtk/pgmgtk.h>

#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION >= 12
/* GtkBuilder is only in GTK+ >= 2.12 */
#  define USE_GTKBUILDER
#else
#  ifndef HAVE_LIBGLADE_2_0
#    error Cannot compile without libglade or a recent GTK+ (>=2.12).
#  endif
#  define USE_GLADE
#endif

#ifdef USE_GLADE
#include <glade/glade.h>
#endif
#include "interactive_tests.h"

#define ROOT_WINDOW "root"
#define YES_BUTTON "yes"
#define NO_BUTTON "no"
#define SCREENSHOT_IMAGE "screenshot"
#define VIEWPORT_CONTAINER "viewport_container"

#ifdef USE_GTKBUILDER
#  define PGM_INTERACTIVE_TEST_INTERFACE_FILE "interface.xml"
#else
#  define PGM_INTERACTIVE_TEST_INTERFACE_FILE "interface.glade"
#endif

static test_return_value = FALSE;
static int test_argc = 0;
static char **test_argv = NULL;

guint async_rendering_count = 0;

/* exported stuff */
PgmViewport *interactive_test_viewport = NULL;
const gchar *pgm_current_test_name = NULL;
gboolean pgm_screenshot_mode = FALSE;

void
pgm_interactive_test_asynchronous_rendering (void)
{
  async_rendering_count += 1;
}

void
pgm_interactive_test_rendering_done (void)
{
  async_rendering_count -= 1;
}

static gpointer
gtk_thread_init (gpointer data)
{
  gtk_main();
}

static void
do_success (void)
{
  test_return_value = TRUE;
  gtk_main_quit ();
}

static void
do_fail (void)
{
  test_return_value = FALSE;
  gtk_main_quit ();
}

void
yes_clicked_callback (GtkButton *button, gpointer data)
{
  do_success ();
}

void
no_clicked_callback (GtkButton *button, gpointer data)
{
  do_fail ();
}

gboolean
root_delete_callback (GtkWidget *widget, GdkEvent  *event, gpointer user_data)
{
  do_fail ();
  return FALSE;
}

void
root_show (GtkWidget *widget, gpointer   user_data)
{
  pgm_viewport_show (interactive_test_viewport);
}

static void
screenshot_callback (PgmViewport *viewport,
                     guint        width,
                     guint        height,
                     gpointer     pixels,
                     gpointer     user_data)
{
  GdkPixbuf *pixbuf = NULL;
  gchar *file_name = NULL;

  file_name = g_strdup_printf ("%s.png", pgm_current_test_name);
  if (!file_name)
    {
      GST_WARNING ("could not create file name for %s", pgm_current_test_name);
      goto tidy;
    }
  pixbuf = gdk_pixbuf_new_from_data (pixels, GDK_COLORSPACE_RGB, TRUE, 8,
                                     width, height, width*4, NULL, NULL);
  if (!pixbuf)
    {
      GST_WARNING ("could not create pixbuf for %s", pgm_current_test_name);
      goto tidy;
    }

  gdk_pixbuf_save (pixbuf, file_name, "png", NULL,
                   "compression", "9", NULL);

tidy:
  if (pixbuf)
    g_object_unref (pixbuf);
  free (pixels);
  free (file_name);

  pgm_main_quit ();
}

pgm_interactive_test_take_screenshot (gpointer data)
{
  gint width = 0, height = 0;
  guint8 *pixels = NULL;
  PgmError ret = PGM_ERROR_OK;
  PgmViewport *viewport = data;

  pgm_viewport_get_size (viewport, &width, &height);
  pixels = malloc (width * height * 4);
  if (!pixels)
    {
      GST_WARNING ("could not create pixel buffer for screenshot");
      return FALSE;
    }

  while (async_rendering_count)
    {
      pgm_main_iteration_do (FALSE);
    }

  ret = pgm_viewport_read_pixels (viewport, 0, 0, width, height, pixels);
  if (PGM_ERROR_OK != ret)
    {
      GST_WARNING ("pgm_viewport_read_pixels() returned an error");
      free (pixels);
      return FALSE;
    }
  /* the pixel buffer shall be cleaned in the signal callback */
  pixels = NULL;
  return FALSE;
}

gboolean
pgm_interactive_test_ask_user ()
{
  GtkImage *screenshot = NULL;
  GtkWidget *window = NULL;
  GtkButton *yes = NULL, *no = NULL;
  GtkContainer *viewport_container = NULL;
  PgmGtk *embedded_viewport = NULL;
  test_return_value = FALSE;
  gchar *image_file = NULL;

  if (pgm_screenshot_mode)
    {
      g_idle_add (pgm_interactive_test_take_screenshot,
                     interactive_test_viewport);
      pgm_main ();
      return TRUE;
    }

#ifdef USE_GTKBUILDER
  GtkBuilder *builder = NULL;

  builder = gtk_builder_new ();
  gtk_builder_add_from_file (builder,
                             PGM_INTERACTIVE_TEST_INTERFACE_FILE,
                             NULL);
  yes = GTK_BUTTON (gtk_builder_get_object (builder, YES_BUTTON));
  no = GTK_BUTTON (gtk_builder_get_object (builder, NO_BUTTON));
  window = GTK_WIDGET (gtk_builder_get_object (builder, ROOT_WINDOW));
  screenshot = GTK_IMAGE (gtk_builder_get_object (builder, SCREENSHOT_IMAGE));
  viewport_container =
    GTK_CONTAINER (gtk_builder_get_object (builder, VIEWPORT_CONTAINER));
#else /* glade */
  GladeXML *interface = NULL;

  interface = glade_xml_new (PGM_INTERACTIVE_TEST_INTERFACE_FILE, NULL, NULL);

  yes = GTK_BUTTON (glade_xml_get_widget (interface, YES_BUTTON));
  no = GTK_BUTTON (glade_xml_get_widget (interface, NO_BUTTON));
  window = GTK_WIDGET (glade_xml_get_widget (interface, ROOT_WINDOW));
  screenshot = GTK_IMAGE (glade_xml_get_widget (interface, SCREENSHOT_IMAGE));
  viewport_container =
    GTK_CONTAINER (glade_xml_get_widget (interface, VIEWPORT_CONTAINER));
#endif

  g_signal_connect (GTK_OBJECT (yes), "clicked",
                    G_CALLBACK (yes_clicked_callback), NULL);
  g_signal_connect (GTK_OBJECT (no), "clicked",
                    G_CALLBACK (no_clicked_callback), NULL);
  g_signal_connect (GTK_OBJECT (window), "delete-event",
                    G_CALLBACK (root_delete_callback), NULL);
  g_signal_connect (GTK_OBJECT (window), "show",
                    G_CALLBACK (root_show), NULL);

  gtk_window_set_title (GTK_WINDOW (window), pgm_current_test_name);
  image_file = g_strdup_printf ("%s.png", pgm_current_test_name);
  if (!image_file)
    {
      GST_WARNING ("could not create file name for %s", pgm_current_test_name);
      goto tidy;
    }

  if (!screenshot)
    return FALSE;
  if (image_file)
    gtk_image_set_from_file (screenshot, image_file);

  /* viewport widget */
  embedded_viewport = pgm_gtk_new ();
  pgm_gtk_set_viewport (embedded_viewport, interactive_test_viewport);
  gtk_container_add (viewport_container, GTK_WIDGET (embedded_viewport));

  gtk_widget_show_all (window);

  gtk_main();

tidy:
  gtk_widget_destroy (GTK_WIDGET (window));
#ifdef USE_GTKBUILDER
  g_object_unref (builder);
#else /* glade */
#endif
  free (image_file);
  return test_return_value;
}

void
pgm_interactive_test_start (void)
{
  fail_unless (NULL == interactive_test_viewport, "viewport is not null");
  gtk_init (&test_argc, &test_argv);
  pgm_init (&test_argc, &test_argv);
  ASSERT_PGM_SUCCESS (pgm_viewport_factory_make("opengl",
                                                &interactive_test_viewport));
  ASSERT_PGM_SUCCESS (pgm_viewport_set_size (interactive_test_viewport,
                                             640, 480));

  if (pgm_screenshot_mode)
    {
      g_signal_connect (G_OBJECT (interactive_test_viewport), "pixels-read",
                        G_CALLBACK (screenshot_callback), NULL);
      pgm_viewport_hide (interactive_test_viewport);
    }
}

void
pgm_interactive_test_end (void)
{
  gst_object_unref (interactive_test_viewport);
  interactive_test_viewport = NULL;
  pgm_deinit ();
}

static parse_args(int *argc, char ***argv)
{
  GOptionEntry entries[] =
    {
      { "screenshot-mode", 's', 0, G_OPTION_ARG_NONE, &pgm_screenshot_mode,
        "Take screenshots instead of testing", NULL },
      { NULL }
    };
  GOptionContext *context =
    g_option_context_new (" - launch interactive tests");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, argc, argv, NULL);
}

int
main (int argc, char **argv)
{
  int ret = 0, i=0;
  gst_check_init (&argc, &argv);
  parse_args (&argc, &argv);

  test_argc = argc;
  test_argv = argv;

  ret = gst_check_run_suite (pgm_text_suite (), "pgm_text_suite", __FILE__);
  ret += gst_check_run_suite (pgm_image_suite (), "pgm_image_suite", __FILE__);


  return ret;
}

