///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/objects/SceneObject.h>
#include <core/data/DataSet.h>
#include <core/data/DataSetManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/plugins/PluginManager.h>
#include <core/plugins/Plugin.h>
#include <core/rendering/RenderSettings.h>

#include "POVRayExporter.h"
#include "POVRayExportInterface.h"
#include "MeshExportInterface.h"
#include "../renderer/POVRayRenderer.h"

namespace POVRay {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(POVRayExporter, ImporterExporter)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(POVRayExportInterface, PluginClass)

/******************************************************************************
* Exports the scene to the given file.
* Return true if the file has been exported.
* Return false if the export has been aborted by the user.
* Throws an exception when the export has failed.
******************************************************************************/
bool POVRayExporter::exportToFile(const QString& filePath, DataSet* dataset, bool suppressDialogs)
{
	MsgLogger() << "Exporting scene to POV-Ray file:" << filePath << endl;

	// Open output file for writing.
	QFile file(filePath);
	if(!file.open(QIODevice::WriteOnly|QIODevice::Text))
		throw Exception(tr("Failed to open POV-Ray file for writing: %1 (%2)").arg(filePath, file.errorString()));

	return exportToPOVRay(&file, dataset, dataset->animationSettings()->time());
}

/******************************************************************************
* Writes the scene to the given IO device.
* Return true if the file has been exported.
* Return false if the export has been aborted by the user.
* Throws an exception when the export has failed.
******************************************************************************/
bool POVRayExporter::exportToPOVRay(QIODevice* device, DataSet* dataset, TimeTicks time, const CameraViewDescription* view, POVRayRenderer* renderer)
{
	OVITO_ASSERT(device->isWritable());

	CameraViewDescription cameraView;
	if(view == NULL) {

		// Export camera view of active viewport.
		Viewport* vp = VIEWPORT_MANAGER.activeViewport();
		if(!vp) throw Exception(tr("There is no active viewport."));

		// Determine aspect ratio.
		FloatType aspectRatio = vp->aspectRatio();
		if(renderer != NULL) {
			aspectRatio = renderer->renderSettings()->outputImageAspectRatio();
		}
		cameraView = vp->getViewDescription(time, aspectRatio);
	}
	else cameraView = *view;

	loadExportInterfaces();

	QTextStream stream(device);
	POVRayWriter writer(stream, dataset, time, cameraView, renderer);
	writer << "#include \"transforms.inc\"" << endl;

	writeGlobalSettings(writer);
	writeEnvironment(writer);
	writeView(writer);
	writeLights(writer);
	writeScene(writer);

	return true;
}

/******************************************************************************
* Loads all installed scene object export interfaces.
******************************************************************************/
void POVRayExporter::loadExportInterfaces()
{
	if(exportInterfaces.empty()) {
		// Load custom object export interfaces (from other plugins only).
		Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(POVRayExportInterface))) {
			// Skip interfaces that are defined in this POVRay plugin because
			// they will be added to the list later.
			if(clazz->plugin()->pluginId() == "POVRay")
				continue;
			try {
				POVRayExportInterface::SmartPtr iface = static_object_cast<POVRayExportInterface>(clazz->createInstance());
				exportInterfaces.push_back(iface);
				MsgLogger() << "Custom POV-Ray export interface found:" << clazz->name() << endl;
			}
			catch(const Exception& ex) {
				qWarning() << QString("WARNING: Failed to load POV-Ray object export interface %1 from plugin %2.").arg(clazz->name(), clazz->plugin()->pluginId());
				qWarning() << QString("Reason: %1").arg(ex.message());
			}
		}

		// Add export interfaces defined in this POV-Ray plugin now to
		// give them the lowest priority.
		exportInterfaces.push_back(new MeshExportInterface());
	}
}

/******************************************************************************
* Writes the global POV-Ray settings to the POV file.
******************************************************************************/
void POVRayExporter::writeGlobalSettings(POVRayWriter& writer)
{
	if(writer.renderer() && writer.renderer()->_enableRadiosity->getValueAtTime(writer.time())) {
		writer << "global_settings {" << endl;
		writer << "radiosity {" << endl;
		writer << "count " << writer.renderer()->_radiosityRayCount->getValueAtTime(writer.time()) << endl;
		writer << "recursion_limit " << writer.renderer()->_radiosityRecursionLimit->getValueAtTime(writer.time()) << endl;
		writer << "error_bound " << writer.renderer()->_radiosityErrorBound->getValueAtTime(writer.time()) << endl;
		writer << "}" << endl;
		writer << "}" << endl;
	}
}

/******************************************************************************
* Writes the background color etc. to the POVRay file.
******************************************************************************/
void POVRayExporter::writeEnvironment(POVRayWriter& writer)
{
	Color backgroundColor = Color(0,0,0);
	RenderSettings* renderSettings = NULL;
	if(writer.renderer() != NULL)
		renderSettings = writer.renderer()->renderSettings();
	else
		renderSettings = DATASET_MANAGER.currentSet()->renderSettings();

	if(renderSettings) {
		backgroundColor = renderSettings->backgroundColorController()->getValueAtTime(writer.time());
	}
	writer << "background { color " << backgroundColor << "}" << endl;
}

/******************************************************************************
* Writes the camera to the POVRay file.
******************************************************************************/
void POVRayExporter::writeView(POVRayWriter& writer)
{
	writer << "camera {" << endl;
	// Write projection transformation
	if(writer.view().isPerspective) {
		writer << "  perspective" << endl;

		Point3 p0 = writer.view().inverseProjectionMatrix * Point3(0,0,0);
		Point3 px = writer.view().inverseProjectionMatrix * Point3(1,0,0);
		Point3 py = writer.view().inverseProjectionMatrix * Point3(0,1,0);
		Point3 lookat = writer.view().inverseProjectionMatrix * Point3(0,0,0);
		Vector3 direction = lookat - ORIGIN;
		Vector3 right = px - p0;
		Vector3 up = Normalize(CrossProduct(right, direction));
		right = Normalize(CrossProduct(direction, up)) * (Length(up) / writer.view().aspectRatio);

		writer << "  location " << ORIGIN << endl;
		writer << "  direction " << Normalize(direction) << endl;
		writer << "  right " << right << endl;
		writer << "  up " << up << endl;
		writer << "  angle " << (atan(tan(writer.view().fieldOfView * 0.5) / writer.view().aspectRatio) * 2.0 * 180.0 / FLOATTYPE_PI) << endl;
	}
	else {
		writer << "  orthographic" << endl;

		Point3 px = writer.view().inverseProjectionMatrix * Point3(1,0,0);
		Point3 py = writer.view().inverseProjectionMatrix * Point3(0,1,0);
		Vector3 direction = writer.view().inverseProjectionMatrix * Point3(0,0,1) - ORIGIN;
		Vector3 up = (py - ORIGIN) * 2.0;
		Vector3 right = px - ORIGIN;

		right = Normalize(CrossProduct(direction, up)) * (Length(up) / writer.view().aspectRatio);

		writer << "  location " << (-(direction*2.0)) << endl;
		writer << "  direction " << direction << endl;
		writer << "  right " << right << endl;
		writer << "  up " << up << endl;
		writer << "  sky " << up << endl;
		writer << "  look_at " << (-direction) << endl;
	}
	// Write camera transformation.
	Rotation rot(writer.view().viewMatrix);
	writer << "  Axis_Rotate_Trans(" << rot.axis << ", " << (rot.angle * 180.0f / FLOATTYPE_PI) << ")" << endl;
	writer << "  translate " << writer.view().inverseViewMatrix.getTranslation() << endl;
	writer << "}" << endl;
}

/******************************************************************************
* Writes the lights to the POVRay file.
******************************************************************************/
void POVRayExporter::writeLights(POVRayWriter& writer)
{
	writer << "light_source {" << endl;
	writer << "  <0, 0, 0>" << endl;
	writer << "  color " << Color(1.5f, 1.5f, 1.5f) << endl;
	writer << "  parallel" << endl;
	writer << "  shadowless" << endl;
	writer << "  point_at " << (writer.view().inverseViewMatrix * Vector3(0,0,-1)) << endl;
	writer << "}" << endl;

	/*
	// Export light sources
	for(size_t i=0; i<MAX_WIN3D_LIGHTS; i++) {
		Window3DLight* light = writer.GetViewport()->GetLight(i);
		if(!light) continue;

		if(light->type == Window3DLight::OMNI_LGT) {
			writer << "light_source {" << endl;
			writer << "  " << light->position << endl;
			writer << "  color " << (light->color * light->intensity * 2.0) << endl;
			writer << "  pointlight" << endl;
			writer << "  shadowless" << endl;
			writer << "}" << endl;
		}
		else if(light->type == Window3DLight::DIRECT_LGT) {
			writer << "light_source {" << endl;
			writer << "  <0, 0, 0>" << endl;
			writer << "  color " << (light->color * light->intensity * 2.0) << endl;
			writer << "  parallel" << endl;
			writer << "  shadowless" << endl;
			writer << "  point_at " << light->position << endl;
			writer << "}" << endl;
		}
		else if(light->type == Window3DLight::AMBIENT_LGT) {
			writer << "global_settings {" << endl;
			writer << "  ambient_light " << (light->color * light->intensity) << endl;
			writer << "}" << endl;
		}
	}
	*/
}

/******************************************************************************
* Writes the scene geometry to the POVRay file.
******************************************************************************/
void POVRayExporter::writeScene(POVRayWriter& writer)
{
	// Export scene objects.
	TimeTicks currentTime = writer.time();
	SceneNodesIterator iter(writer.dataSet()->sceneRoot());
	for(; !iter.finished(); iter.next()) {
		ObjectNode* node = dynamic_object_cast<ObjectNode>(iter.current());
		if(!node) continue;

		PipelineFlowState state = node->evalPipeline(currentTime);
		if(!state.result()) continue;

		TimeInterval iv;
		AffineTransformation nodeTM = node->objectTransform() * node->getWorldTransform(currentTime, iv);

		Q_FOREACH(const POVRayExportInterface::SmartPtr& iface, exportInterfaces) {
			if(iface->exportSceneObject(state.result(), writer, node, nodeTM))
				break;
		}
	}
}

};

