///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>
#include <scripting/engine/ScriptEngine.h>
#include <scripting/interface/ContainerWrappers.h>

#include <atomviz/AtomViz.h>
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/AtomType.h>
#include <atomviz/atoms/datachannels/DataChannel.h>
#include <atomviz/atoms/datachannels/PositionDataChannel.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>
#include "../parser/AtomsFileParser.h"
#include "../parser/AbstractFileColumnParser.h"
#include "../parser/AtomsImportObject.h"
#include "../parser/ColumnChannelMapping.h"
#include "../parser/lammps/LAMMPSDumpParser.h"
#include "../parser/lammps/LAMMPSTextDumpParser.h"
#include "../parser/lammps/LAMMPSBinaryDumpParser.h"
#include "../parser/lammps/LAMMPSDataParser.h"
#include "../parser/xyz/XYZParser.h"
#include "../parser/imd/IMDAtomFileParser.h"
#include "../writer/AtomsFileWriter.h"
#include "../writer/AbstractFileColumnWriter.h"
#include "../writer/ChannelColumnMapping.h"
#include "../writer/MultiFileWriter.h"
#include "../writer/xyz/XYZWriter.h"
#include "../writer/lammps/LAMMPSDumpWriter.h"
#include "../writer/lammps/LAMMPSTextDumpWriter.h"
#include "../writer/lammps/LAMMPSBinaryDumpWriter.h"
#include "../writer/lammps/LAMMPSDataWriter.h"
#include "../writer/imd/IMDAtomFileWriter.h"
#include "../modifier/ambientlighting/AmbientLightingModifier.h"
#include "../modifier/coloring/AssignColorModifier.h"
#include "../modifier/coloring/AtomTypeColorModifier.h"
#include "../modifier/coloring/ColorCodingModifier.h"
#include "../modifier/deleteatoms/DeleteAtomsModifier.h"
#include "../modifier/displacement/CalcDisplacementsModifier.h"
#include "../modifier/datachannels/CreateExpressionChannelModifier.h"
#include "../modifier/selection/SelectAtomTypeModifier.h"
#include "../modifier/selection/InvertSelectionModifier.h"
#include "../modifier/wrap/WrapPeriodicImagesModifier.h"
#include "../modifier/analysis/cna/CNAModifier.h"
#include "../modifier/analysis/coordination/CoordinationNumberModifier.h"

using namespace boost::python;
using namespace AtomViz;
using namespace Scripting;

BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(ColumnChannelMapping_defineStandardColumn_overloads, defineStandardColumn, 2, 4)
BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(ColumnChannelMapping_defineColumn_overloads, defineColumn, 4, 6)
BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(ChannelColumnMapping_insertColumn_overloads, insertColumn, 3, 4)

BOOST_PYTHON_MODULE(AtomViz)
{
	// AtomType class
	class_<AtomType, bases<RefTarget>, intrusive_ptr<AtomType>, noncopyable>("AtomType", no_init)
		.add_property("Name", make_function(&AtomType::name, return_value_policy<copy_const_reference>()), &AtomType::setName)
		.add_property("ColorController", make_function(&AtomType::colorController, return_internal_reference<>()))
		.add_property("RadiusController", make_function(&AtomType::radiusController, return_internal_reference<>()))
	;

	// AtomTypeList type
	class_<QVector<AtomType*>, noncopyable>("AtomTypeList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<AtomType*>, return_internal_reference<> >())
	;

	// DataChannel types
	enum_<int>("DataChannelType")
    	.value("TypeNone", QMetaType::Void)
    	.value("TypeInteger", qMetaTypeId<int>())
    	.value("TypeFloat", qMetaTypeId<FloatType>())
    ;

	// DataChannelIdentifier enum
	enum_<DataChannel::DataChannelIdentifier>("DataChannelIdentifier")
		.value("UserDataChannel", DataChannel::UserDataChannel)
    	.value("AtomTypeChannel", DataChannel::AtomTypeChannel)
    	.value("PositionChannel", DataChannel::PositionChannel)
    	.value("SelectionChannel", DataChannel::SelectionChannel)
    	.value("ColorChannel", DataChannel::ColorChannel)
    	.value("DisplacementChannel", DataChannel::DisplacementChannel)
    	.value("PotentialEnergyChannel", DataChannel::PotentialEnergyChannel)
    	.value("KineticEnergyChannel", DataChannel::KineticEnergyChannel)
    	.value("TotalEnergyChannel", DataChannel::TotalEnergyChannel)
    	.value("VelocityChannel", DataChannel::VelocityChannel)
    	.value("RadiusChannel", DataChannel::RadiusChannel)
    	.value("ClusterChannel", DataChannel::ClusterChannel)
    	.value("CoordinationChannel", DataChannel::CoordinationChannel)
    	.value("CNATypeChannel", DataChannel::CNATypeChannel)
    	.value("AtomIndexChannel", DataChannel::AtomIndexChannel)
    	.value("StressTensorChannel", DataChannel::StressTensorChannel)
    	.value("StrainTensorChannel", DataChannel::StrainTensorChannel)
    	.value("DeformationGradientChannel", DataChannel::DeformationGradientChannel)
    	.value("ForceChannel", DataChannel::ForceChannel)
    ;

	// DataChannel class
	class_<DataChannel, bases<RefTarget>, intrusive_ptr<DataChannel>, noncopyable>("DataChannel", no_init)
		.add_property("Name", make_function(&DataChannel::name, return_value_policy<copy_const_reference>()), &DataChannel::setName)
		.add_property("Size", &DataChannel::size)
		.add_property("Id", &DataChannel::id)
		.add_property("Type", &DataChannel::type)
		.add_property("ComponentCount", &DataChannel::componentCount)
		.add_property("Visible", &DataChannel::isVisible, &DataChannel::setVisible)
		.def("GetInt", &DataChannel::getInt)
		.def("GetFloat", &DataChannel::getFloat)
		.def("GetIntComponent", &DataChannel::getIntComponent)
		.def("GetFloatComponent", &DataChannel::getFloatComponent)
		.def("GetVector3", make_function(&DataChannel::getVector3, return_value_policy<copy_const_reference>()))
		.def("GetPoint3", make_function(&DataChannel::getPoint3, return_value_policy<copy_const_reference>()))
		.def("GetTensor2", make_function(&DataChannel::getTensor2, return_value_policy<copy_const_reference>()))
		.def("GetSymmetricTensor2", make_function(&DataChannel::getSymmetricTensor2, return_value_policy<copy_const_reference>()))
		.def("GetQuaternion", make_function(&DataChannel::getQuaternion, return_value_policy<copy_const_reference>()))
		.def("SetInt", &DataChannel::setInt)
		.def("SetFloat", &DataChannel::setFloat)
		.def("SetIntComponent", &DataChannel::setIntComponent)
		.def("SetFloatComponent", &DataChannel::setFloatComponent)
		.def("SetVector3", &DataChannel::setVector3)
		.def("SetPoint3", &DataChannel::setPoint3)
		.def("SetTensor2", &DataChannel::setTensor2)
		.def("SetSymmtricTensor2", &DataChannel::setSymmetricTensor2)
		.def("SetQuaternion", &DataChannel::setQuaternion)
	;

	// PositionDataChannel class
	class_<PositionDataChannel, bases<DataChannel>, intrusive_ptr<PositionDataChannel>, noncopyable>("PositionDataChannel", no_init)
		.add_property("FlatAtomRendering", &PositionDataChannel::flatAtomRendering, &PositionDataChannel::setFlatAtomRendering)
		.add_property("HighQualityRenderingInViewports", &PositionDataChannel::highQualityRenderingInViewports, &PositionDataChannel::setHighQualityRenderingInViewports)
		.add_property("GlobalAtomRadiusScale", &PositionDataChannel::globalAtomRadiusScale, &PositionDataChannel::setGlobalAtomRadiusScale)
	;

	// AtomTypeDataChannel class
	class_<AtomTypeDataChannel, bases<DataChannel>, intrusive_ptr<AtomTypeDataChannel>, noncopyable>("AtomTypeDataChannel", no_init)
		.add_property("AtomTypes", make_function(&AtomTypeDataChannel::atomTypes, return_internal_reference<>()))
	;

	class_<DataChannelReference>("DataChannelReference", init<>())
		.add_property("Id", &DataChannelReference::id, &DataChannelReference::setId)
		.add_property("Name", make_function(&DataChannelReference::name, return_value_policy<copy_const_reference>()))
	;

	// DataChannelList type
	class_<QVector<DataChannel*>, noncopyable>("DataChannelList", no_init)
		.def(QVector_readonly_indexing_suite< QVector<DataChannel*>, return_internal_reference<> >())
	;

	// SimulationCell class
	class_<SimulationCell, bases<RefTarget>, intrusive_ptr<SimulationCell>, noncopyable>("SimulationCell", init<>())
		.def(init<const Vector3&, const Vector3&, const Vector3&>())
		.def(init<const Point3&, const Vector3&, const Vector3&, const Vector3&>())
		.def(init<const AffineTransformation&>())
		.def(init<const Box3&, bool, bool, bool>())
		.def<void (SimulationCell::*)(bool,bool,bool)>("SetPeriodicity", &SimulationCell::setPeriodicity)
		.def("SetCellShape", &SimulationCell::setCellShape)
		.def("SetBoxShape", &SimulationCell::setBoxShape)
		.add_property("CellMatrix", &SimulationCell::cellMatrix, &SimulationCell::setCellMatrix)
		.add_property("Volume", &SimulationCell::volume)
		.add_property("BoundingBox", &SimulationCell::boundingBox)
	;

	// AtomsObject class
	class_<AtomsObject, bases<SceneObject>, intrusive_ptr<AtomsObject>, noncopyable>("AtomsObject", init<>())
		.add_property("AtomsCount", &AtomsObject::atomsCount, &AtomsObject::setAtomsCount)
		.def("FindDataChannelByName", make_function(&AtomsObject::findDataChannelByName, return_internal_reference<>()))
		.def("GetStandardDataChannel", make_function(&AtomsObject::getStandardDataChannel, return_internal_reference<>()))
		.def("CreateStandardDataChannel", make_function(&AtomsObject::createStandardDataChannel, return_internal_reference<>()))
		.def("CreateCustomDataChannel", make_function(&AtomsObject::createCustomDataChannel, return_internal_reference<>()))
		.def("CopyShallowChannel", make_function(&AtomsObject::copyShallowChannel, return_internal_reference<>()))
		.def("Invalidate", &AtomsObject::invalidate)
		.add_property("SimulationCell", make_function(&AtomsObject::simulationCell, return_internal_reference<>()), &AtomsObject::setSimulationCell)
		.add_property("DataChannels", make_function(&AtomsObject::dataChannels, return_internal_reference<>()))
	;

	class_<AtomsFileParser, bases<ImporterExporter>, intrusive_ptr<AtomsFileParser>, noncopyable>("AtomsFileParser", no_init)
		.def("ImportFile", &AtomsFileParser::importFile)
		.def("PrepareInputFile", &AtomsFileParser::prepareInputFile)
		.def("LoadAtomsFile", &AtomsFileParser::loadAtomsFile)
		.add_property("InputFile", make_function(&AtomsFileParser::inputFile, return_value_policy<copy_const_reference>()), &AtomsFileParser::setInputFile)
		.add_property("NumberOfMovieFrames", &AtomsFileParser::numberOfMovieFrames)
	;

	class_<AtomsImportObject, bases<SceneObject>, intrusive_ptr<AtomsImportObject>, noncopyable>("AtomsImportObject", init<>())
		.add_property("Parser", make_function(&AtomsImportObject::parser, return_internal_reference<>()), &AtomsImportObject::setParser)
		.def("ReloadInputFile", &AtomsImportObject::reloadInputFile)
		.add_property("LoadedMovieFrame", &AtomsImportObject::loadedMovieFrame)
		.add_property("Atoms", make_function(&AtomsImportObject::atomsObject, return_internal_reference<>()))
	;

	class_<ColumnChannelMapping>("ColumnChannelMapping")
		.add_property("ColumnCount", &ColumnChannelMapping::columnCount, &ColumnChannelMapping::setColumnCount)
		.def("DefineColumn", &ColumnChannelMapping::defineColumn, ColumnChannelMapping_defineColumn_overloads())
		.def("DefineStandardColumn", &ColumnChannelMapping::defineStandardColumn, ColumnChannelMapping_defineStandardColumn_overloads())
		.def("IgnoreColumn", &ColumnChannelMapping::ignoreColumn)
		.def("GetChannelId", &ColumnChannelMapping::getChannelId)
		.def("GetChannelName", &ColumnChannelMapping::getChannelName)
		.def("GetChannelType", &ColumnChannelMapping::getChannelType)
		.def("GetVectorComponent", &ColumnChannelMapping::getVectorComponent)
		.add_property("AtomIndexRemapping", &ColumnChannelMapping::atomIndexRemappingEnabled, &ColumnChannelMapping::enableAtomIndexRemapping)
	;

	class_<AbstractFileColumnParser, bases<AtomsFileParser>, intrusive_ptr<AbstractFileColumnParser>, noncopyable>("AbstractFileColumnParser", no_init)
		.add_property("ColumnMapping", make_function(&AbstractFileColumnParser::columnMapping, return_value_policy<copy_const_reference>()), &AbstractFileColumnParser::setColumnMapping)
	;

	class_<MultiFileParser, bases<AbstractFileColumnParser>, intrusive_ptr<MultiFileParser>, noncopyable>("MultiFileParser", no_init)
		.add_property("UseWildcardFilename", &MultiFileParser::useWildcardFilename, &MultiFileParser::setUseWildcardFilename)
		.add_property("WildcardFilename", make_function(&MultiFileParser::wildcardFilename, return_value_policy<copy_const_reference>()), &MultiFileParser::setWildcardFilename)
		.add_property("MovieFileEnabled", &MultiFileParser::movieFileEnabled, &MultiFileParser::setMovieFileEnabled)
	;

	class_<LAMMPSDumpParser, bases<MultiFileParser>, intrusive_ptr<LAMMPSDumpParser>, noncopyable>("LAMMPSDumpParser", no_init)
	;

	class_<LAMMPSTextDumpParser, bases<LAMMPSDumpParser>, intrusive_ptr<LAMMPSTextDumpParser>, noncopyable>("LAMMPSTextDumpParser")
	;

	class_<LAMMPSBinaryDumpParser, bases<LAMMPSDumpParser>, intrusive_ptr<LAMMPSBinaryDumpParser>, noncopyable>("LAMMPSBinaryDumpParser")
	;

	class_<LAMMPSDataParser, bases<AtomsFileParser>, intrusive_ptr<LAMMPSDataParser>, noncopyable>("LAMMPSDataParser")
	;

	class_<XYZParser, bases<MultiFileParser>, intrusive_ptr<XYZParser>, noncopyable>("XYZParser")
	;

	class_<IMDAtomFileParser, bases<AtomsFileParser>, intrusive_ptr<IMDAtomFileParser>, noncopyable>("IMDAtomFileParser")
	;

	class_<AtomsFileWriter, bases<ImporterExporter>, intrusive_ptr<AtomsFileWriter>, noncopyable>("AtomsFileWriter", no_init)
		.def("ExportToFile", &AtomsFileWriter::exportToFile)
		.def("PrepareScene", &AtomsFileWriter::prepareScene)
		.def("ExportAtoms", &AtomsFileWriter::exportAtoms)
		.add_property("OutputFile", make_function(&AtomsFileWriter::outputFile, return_value_policy<copy_const_reference>()), &AtomsFileWriter::setOutputFile)
	;

	class_<ChannelColumnMapping>("ChannelColumnMapping")
		.add_property("ColumnCount", &ChannelColumnMapping::columnCount)
		.def("InsertColumn", &ChannelColumnMapping::insertColumn, ChannelColumnMapping_insertColumn_overloads())
		.def("RemoveColumn", &ChannelColumnMapping::removeColumn)
		.def("GetChannelId", &ChannelColumnMapping::getChannelId)
		.def("GetChannelName", &ChannelColumnMapping::getChannelName)
		.def("GetVectorComponent", &ChannelColumnMapping::getVectorComponent)
	;

	class_<AbstractFileColumnWriter, bases<AtomsFileWriter>, intrusive_ptr<AbstractFileColumnWriter>, noncopyable>("AbstractFileColumnWriter", no_init)
		.add_property("ChannelMapping", make_function(&AbstractFileColumnWriter::channelMapping, return_value_policy<copy_const_reference>()), &AbstractFileColumnWriter::setChannelMapping)
	;

	class_<MultiFileWriter, bases<AbstractFileColumnWriter>, intrusive_ptr<MultiFileWriter>, noncopyable>("MultiFileWriter", no_init)
		.add_property("UseWildcardFilename", &MultiFileWriter::useWildcardFilename, &MultiFileWriter::setUseWildcardFilename)
		.add_property("WildcardFilename", make_function(&MultiFileWriter::wildcardFilename, return_value_policy<copy_const_reference>()), &MultiFileWriter::setWildcardFilename)
		.add_property("StartFrame", &MultiFileWriter::startFrame, &MultiFileWriter::setStartFrame)
		.add_property("EndFrame", &MultiFileWriter::endFrame, &MultiFileWriter::setEndFrame)
	;

	class_<XYZWriter, bases<MultiFileWriter>, intrusive_ptr<XYZWriter>, noncopyable>("XYZWriter")
	;

	class_<LAMMPSDumpWriter, bases<MultiFileWriter>, intrusive_ptr<LAMMPSDumpWriter>, noncopyable>("LAMMPSDumpWriter", no_init)
	;

	class_<LAMMPSBinaryDumpWriter, bases<LAMMPSDumpWriter>, intrusive_ptr<LAMMPSBinaryDumpWriter>, noncopyable>("LAMMPSBinaryDumpWriter")
	;

	class_<LAMMPSTextDumpWriter, bases<LAMMPSDumpWriter>, intrusive_ptr<LAMMPSTextDumpWriter>, noncopyable>("LAMMPSTextDumpWriter")
	;

	class_<LAMMPSDataWriter, bases<MultiFileWriter>, intrusive_ptr<LAMMPSDataWriter>, noncopyable>("LAMMPSDataWriter")
	;

	class_<IMDAtomFileWriter, bases<MultiFileWriter>, intrusive_ptr<IMDAtomFileWriter>, noncopyable>("IMDAtomFileWriter")
	;

	class_<AtomsObjectModifierBase, bases<Modifier>, intrusive_ptr<AtomsObjectModifierBase>, noncopyable>("AtomsObjectModifierBase", no_init)
	;

	class_<NearestNeighborList, bases<RefTarget>, intrusive_ptr<NearestNeighborList>, noncopyable>("NearestNeighborList")
		.add_property("CutoffRadius", &NearestNeighborList::nearestNeighborCutoff, &NearestNeighborList::setNearestNeighborCutoff)
	;

	class_<AtomsObjectAnalyzerBase, bases<AtomsObjectModifierBase>, intrusive_ptr<AtomsObjectAnalyzerBase>, noncopyable>("AtomsObjectAnalyzerBase", no_init)
		.def("PerformAnalysis", (EvaluationStatus (AtomsObjectAnalyzerBase::*)(TimeTicks, bool))&AtomsObjectAnalyzerBase::performAnalysis)
		.add_property("AutoUpdateOnTimeChangeEnabled", &AtomsObjectAnalyzerBase::autoUpdateOnTimeChangeEnabled, &AtomsObjectAnalyzerBase::setAutoUpdateOnTimeChangeEnabled)
		.add_property("NearestNeighborList", make_function(&AtomsObjectAnalyzerBase::nearestNeighborList, return_internal_reference<>()), &AtomsObjectAnalyzerBase::setNearestNeighborList)
	;

	class_<AmbientLightingModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<AmbientLightingModifier>, noncopyable>("AmbientLightingModifier", init<>())
		.add_property("Intensity", &AmbientLightingModifier::intensity, &AmbientLightingModifier::setIntensity)
		.add_property("IntensityController", make_function(&AmbientLightingModifier::intensityController, return_internal_reference<>()), &AmbientLightingModifier::setIntensityController)
		.add_property("SamplingLevel", &AmbientLightingModifier::samplingLevel, &AmbientLightingModifier::setSamplingLevel)
		.add_property("BufferResolution", &AmbientLightingModifier::bufferResolution, &AmbientLightingModifier::setBufferResolution)
	;

	class_<AssignColorModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<AssignColorModifier>, noncopyable>("AssignConstantColorModifier", init<>())
		.add_property("Color", &AssignColorModifier::color, &AssignColorModifier::setColor)
		.add_property("ColorController", make_function(&AssignColorModifier::colorController, return_internal_reference<>()), &AssignColorModifier::setColorController)
	;

	class_<AtomTypeColorModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<AtomTypeColorModifier>, noncopyable>("AtomTypeColorModifier", init<>())
	;

	class_<ColorCodingGradient, bases<RefTarget>, intrusive_ptr<ColorCodingGradient>, noncopyable>("ColorCodingGradient", no_init)
	;

	class_<ColorCodingHSVGradient, bases<ColorCodingGradient>, intrusive_ptr<ColorCodingHSVGradient>, noncopyable>("ColorCodingHSVGradient", init<>())
	;

	class_<ColorCodingGrayscaleGradient, bases<ColorCodingGradient>, intrusive_ptr<ColorCodingGrayscaleGradient>, noncopyable>("ColorCodingGrayscaleGradient", init<>())
	;

	class_<ColorCodingModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<ColorCodingModifier>, noncopyable>("ColorCodingModifier", init<>())
		.add_property("StartValue", &ColorCodingModifier::startValue, &ColorCodingModifier::setStartValue)
		.add_property("StartValueController", make_function(&ColorCodingModifier::startValueController, return_internal_reference<>()), &ColorCodingModifier::setStartValueController)
		.add_property("EndValue", &ColorCodingModifier::endValue, &ColorCodingModifier::setEndValue)
		.add_property("EndValueController", make_function(&ColorCodingModifier::endValueController, return_internal_reference<>()), &ColorCodingModifier::setEndValueController)
		.add_property("SourceDataChannel", make_function(&ColorCodingModifier::sourceDataChannel, return_value_policy<copy_const_reference>()), &ColorCodingModifier::setSourceDataChannel)
		.add_property("SourceVectorComponent", &ColorCodingModifier::sourceVectorComponent, &ColorCodingModifier::setSourceVectorComponent)
		.add_property("ColorGradient", make_function(&ColorCodingModifier::colorGradient, return_internal_reference<>()), &ColorCodingModifier::setColorGradient)
	;

	class_<DeleteAtomsModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<DeleteAtomsModifier>, noncopyable>("DeleteAtomsModifier", init<>())
	;

	class_<SelectAtomTypeModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<SelectAtomTypeModifier>, noncopyable>("SelectAtomTypeModifier", init<>())
		.add_property("SourceDataChannel", make_function(&SelectAtomTypeModifier::sourceDataChannel, return_internal_reference<>()), &SelectAtomTypeModifier::setSourceDataChannel)
		.def("SetSelectedAtomType", &SelectAtomTypeModifier::setSelectedAtomType)
	;

	class_<InvertSelectionModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<InvertSelectionModifier>, noncopyable>("InvertSelectionModifier", init<>())
	;

	class_<CalcDisplacementsModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<CalcDisplacementsModifier>, noncopyable>("CalcDisplacementsModifier", init<>())
		.add_property("ReferenceConfiguration", make_function(&CalcDisplacementsModifier::referenceConfiguration, return_internal_reference<>()), &CalcDisplacementsModifier::setReferenceConfiguration)
		.add_property("DisplacementChannelPrototype", make_function(&CalcDisplacementsModifier::displacementChannelPrototype, return_internal_reference<>()))
	;

	enum_<CommonNeighborAnalysisModifier::CNAAtomType>("CNAAtomType")
    	.value("FCC", CommonNeighborAnalysisModifier::FCC)
    	.value("HCP", CommonNeighborAnalysisModifier::HCP)
    	.value("BCC", CommonNeighborAnalysisModifier::BCC)
		.value("Icosahedral", CommonNeighborAnalysisModifier::ICOSAHEDRAL)
		.value("Other", CommonNeighborAnalysisModifier::OTHER)
		.value("Diamond", CommonNeighborAnalysisModifier::DIAMOND)
		.value("HexagonalDiamond", CommonNeighborAnalysisModifier::HEX_DIAMOND)
		.value("DiamondStackingFault", CommonNeighborAnalysisModifier::DIAMOND_STACKINGFAULT)
		.value("BCCTwin", CommonNeighborAnalysisModifier::BCC_TWIN)
    ;

	class_<CommonNeighborAnalysisModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<CommonNeighborAnalysisModifier>, noncopyable>("CommonNeighborAnalysisModifier", init<>())
		.add_property("CNAAtomTypes", make_function(&CommonNeighborAnalysisModifier::cnaAtomTypes, return_internal_reference<>()))
	;

	class_<CoordinationNumberModifier, bases<AtomsObjectAnalyzerBase>, intrusive_ptr<CoordinationNumberModifier>, noncopyable>("CoordinationNumberModifier", init<>())
	;

	class_<WrapPeriodicImagesModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<WrapPeriodicImagesModifier>, noncopyable>("WrapPeriodicImagesModifier", init<>())
	;

	class_<CreateExpressionChannelModifier, bases<AtomsObjectModifierBase>, intrusive_ptr<CreateExpressionChannelModifier>, noncopyable>("CreateExpressionChannelModifier", init<>())
		.add_property("DataChannelId", &CreateExpressionChannelModifier::dataChannelId, &CreateExpressionChannelModifier::setDataChannelId)
		.def("SetExpression", &CreateExpressionChannelModifier::setExpression)
		.def("GetExpression", make_function(&CreateExpressionChannelModifier::expression, return_value_policy<copy_const_reference>()))
	;
}

OVITO_REGISTER_PLUGIN_PYTHON_INTERFACE(AtomViz)
