/*
 *  Copyright (c) 2008,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstring>

// GTLCore
#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Image.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/StdTypes.h>
#include <GTLCore/Type.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>

// GTLFragment Headers
#include <GTLFragment/LibrariesManager.h>

// OpenRijn Headers
#include <OpenRijn/Sketch.h>
#include <OpenRijn/Version.h>
#include <OpenRijn/LibrariesManager.h>
#include <OpenRijn/ImageEngine/Canvas.h>
#include <OpenRijn/ImageEngine/Palette.h>

void printVersion()
{
  std::cout << OpenRijn::LibraryShortName() << " - " << OpenRijn::LibraryName() << " - " << OpenRijn::LibraryVersionString() << std::endl;
  std::cout << OpenRijn::LibraryCopyright() << std::endl;
  std::cout << "Rijn Version : " << OpenRijn::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shivac [option] fileName.shiva" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << "  -w --width [w]          define the width of the output" << std::endl;
  std::cout << "  -h --height [h]         define the height of the output" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  GTLCore::String outputFileName = "";
  std::list< GTLCore::String > fileNames;
  int width = 800;
  int height = 600;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-w", "--width")) {
      if( ai == argc )
      {
        std::cerr << "Expected width after -w --width." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        width = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-h", "--height")) {
      if( ai == argc )
      {
        std::cerr << "Expected height after -h --height." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        height = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLFragment::LibrariesManager::instance()->addDirectory(argv[ai]);
        OpenRijn::LibrariesManager::instance()->addDirectory(argv[ai]);
      }
    } else {
      fileNames.push_back( argv[ai] );
    }
  }
  if( fileNames.size() != 2)
  {
    printHelp();
  } else {
    fileName = *( fileNames.begin() );
    outputFileName = *( --fileNames.end() );
    
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    std::string str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    
    // Compile
    OpenRijn::Sketch sketch;
    sketch.setSource( source );
    sketch.compile();
    if(not sketch.isCompiled())
    {
      std::cout << "Error: " << std::endl << sketch.compilationMessages().toString() << std::endl;
      return EXIT_FAILURE;
    }
    
    // Draw on the image
    GTLCore::Image image(width, height, GTLCore::PixelDescription(GTLCore::Type::Float32, 4, 3));
    std::memset(image.rawData(0,0),0, width * height * sizeof(float));
    OpenRijn::ImageEngine::Canvas c(&image);
    OpenRijn::ImageEngine::Palette p;
    sketch.draw(&c, &p);
    
    GTLCore::Image image2(width, height, GTLCore::PixelDescription(GTLCore::Type::UnsignedInteger16, 4, 3));
    for(int x = 0; x < width; ++x)
    {
      for(int y = 0; y < height; ++y)
      {
        for(int k = 0; k < 4; ++k)
        {
          image2.data<gtl_uint16>(x, y)[k] = 65535 * image.data<float>(x, y)[k];
        }
      }
    }
    // Save the image
    const GTLImageIO::ImageDC* encoder = GTLImageIO::ImageDCRegistry::instance()->encoder( outputFileName );
    if( not encoder )
    {
      std::cerr << "Can't find encoder for " << outputFileName << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::String errMsg;
    if( not encoder->encode( &image2, image2.boundingBox(), outputFileName, 0, &errMsg ) )
    {
      std::cerr << "Can't encode " << outputFileName << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
  }
  return EXIT_SUCCESS;
}
