/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Dynamic loading

   Dynamic linking loader.

   This API may be thread-safe or not depending on the platform.

   If it is not thread-safe on the target platform, calls to these functions in
   multi-threaded applications should be also synchronized w.r.t. other
   functions that interface to the dynamic linking loader.
 */

#ifndef _NASPRO_CORE_DL_H
#define _NASPRO_CORE_DL_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Type: nacore_dl_handle

   Dynamically loaded library handle.
 */
typedef struct _nacore_dl_handle * nacore_dl_handle;

/*
   Function: nacore_dl_open()

   Loads a dynamic library.

   It is possible to try loading the same library more than once, in which case
   it will not be deallocated at least until <nacore_dl_close()> is called as
   many times as this function succeded on it.

   It is however possible that even at that time the library is not deallocated,
   since it may still be referenced if another API interfacing to the
   dynamically linking loader was used in the same process.

   Unlike the POSIX dlopen() API, it is not guaranteed that using this function
   to load the same library more than once will lead to the same handle being
   returned.

   Parameters:

     filename		- Absolute path name.
     msg_context	- Message context or NULL.
     msg_opaque		- Extra opaque data to be passed to message reporting
			  callback or NULL.

   Returns:

     Dynamically loaded library handle or NULL if some error happened.
 */
_NACORE_DEF nacore_dl_handle
nacore_dl_open(const char *filename, nacore_msg_context msg_context,
	       void *msg_opaque);

/*
   Function: nacore_dl_sym()

   Retrieves the memory address where a symbol from a dynamically loaded library
   is loaded into memory.

   Parameters:

     handle		- Dynamically loaded library handle.
     symbol		- Symbol name.
     msg_context	- Message context or NULL.
     msg_opaque		- Extra opaque data to be passed to message reporting
			  callback or NULL.

   Returns:

     Memory address or NULL if some error happened. Since NULL may also be a
     valid memory address, if some error happend errno is set to
     <NACORE_EUNKNOWN>.
 */
_NACORE_DEF void *
nacore_dl_sym(nacore_dl_handle handle, const char *symbol,
	      nacore_msg_context msg_context, void *msg_opaque);

/*
   Function: nacore_dl_close()

   Decrements the reference count on the dynamically loaded library handle.

   Parameters:

     handle		- Dynamically loaded library handle.
     msg_context	- Message context or NULL.
     msg_opaque		- Extra opaque data to be passed to message reporting
			  callback or NULL.

   Returns:

     0 on success, non-0 if some error happened.
 */
_NACORE_DEF int
nacore_dl_close(nacore_dl_handle handle, nacore_msg_context msg_context,
		void *msg_opaque);

NACORE_END_C_DECLS

#endif
