/*
 * Copyright (c) 2018, JANET(UK)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of JANET(UK) nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Monitoring handlers for trust router TID server */

#include <jansson.h>
#include <tid_internal.h>
#include <tr_tid.h>
#include <mon_internal.h>
#include <mons_handlers.h>

/**
 * Get the count of successfully completed TID requests
 */
static MON_RC handle_show_req_count(void *cookie, json_t **response_ptr)
{
  TIDS_INSTANCE *tids = talloc_get_type_abort(cookie, TIDS_INSTANCE);
  *response_ptr = json_integer(tids->req_count);
  return (*response_ptr == NULL) ? MON_NOMEM : MON_SUCCESS;
}

/**
 * Get the count of completed TID requests that resulted in error responses
 */
static MON_RC handle_show_req_error_count(void *cookie, json_t **response_ptr)
{
  TIDS_INSTANCE *tids = talloc_get_type_abort(cookie, TIDS_INSTANCE);
  *response_ptr = json_integer(tids->req_error_count);
  return (*response_ptr == NULL) ? MON_NOMEM : MON_SUCCESS;
}

/**
 * Get the count of TID requests that could not be completed
 */
static MON_RC handle_show_error_count(void *cookie, json_t **response_ptr)
{
  TIDS_INSTANCE *tids = talloc_get_type_abort(cookie, TIDS_INSTANCE);
  *response_ptr = json_integer(tids->error_count);
  return (*response_ptr == NULL) ? MON_NOMEM : MON_SUCCESS;
}

static MON_RC handle_show_req_pending(void *cookie, json_t **response_ptr)
{
  TIDS_INSTANCE *tids = talloc_get_type_abort(cookie, TIDS_INSTANCE);
  *response_ptr = json_integer(tids->pids->len);
  return (*response_ptr == NULL) ? MON_NOMEM : MON_SUCCESS;
}

void tr_tid_register_mons_handlers(TIDS_INSTANCE *tids, MONS_INSTANCE *mons)
{
  mons_register_handler(mons,
                        MON_CMD_SHOW, OPT_TYPE_SHOW_TID_REQS_PROCESSED,
                        handle_show_req_count, tids);
  mons_register_handler(mons,
                        MON_CMD_SHOW, OPT_TYPE_SHOW_TID_REQS_FAILED,
                        handle_show_req_error_count, tids);
  mons_register_handler(mons,
                        MON_CMD_SHOW, OPT_TYPE_SHOW_TID_ERROR_COUNT,
                        handle_show_error_count, tids);
  mons_register_handler(mons,
                        MON_CMD_SHOW, OPT_TYPE_SHOW_TID_REQS_PENDING,
                        handle_show_req_pending, tids);
}
