#include "osl/move.h"
#include "osl/stl/vector.h"
#include "osl/ptypeTable.h"
#include "osl/state/simpleState.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <boost/foreach.hpp>

using namespace osl;

class MoveTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(MoveTest);
  CPPUNIT_TEST(testPass);
  CPPUNIT_TEST(testValid);
  CPPUNIT_TEST(testDeclareWin);
  CPPUNIT_TEST(testNormal);
  CPPUNIT_TEST(testSetFrom);
  CPPUNIT_TEST(testCapturePtpeO);
  CPPUNIT_TEST(testIgnoreUnpromote);
  CPPUNIT_TEST(testHasIgnoredUnpromote);
  CPPUNIT_TEST(testRotate180);
  CPPUNIT_TEST(testHash);
  CPPUNIT_TEST_SUITE_END();
public:
  void testValid() {
    const Move drop(Square(5,5), KNIGHT, WHITE);
    CPPUNIT_ASSERT(drop.isValid()); 

    const Move bug = Move::makeDirect(-58698240);
    CPPUNIT_ASSERT(! bug.isValid()); // KINGを取る55桂打
  }
  void testNormal()
  {
    const Move m76fu(Square(7,7),Square(7,6),PAWN,PTYPE_EMPTY,false,BLACK);
    CPPUNIT_ASSERT(! m76fu.isPass());
    CPPUNIT_ASSERT(m76fu.isNormal());
    CPPUNIT_ASSERT(! m76fu.isInvalid());

    const Move m34fu(Square(3,3),Square(3,4),PAWN,PTYPE_EMPTY,false,WHITE);
    CPPUNIT_ASSERT(! m34fu.isPass());
    CPPUNIT_ASSERT(m34fu.isNormal());
    CPPUNIT_ASSERT(! m34fu.isInvalid());
  }
  void testPass()
  {
    const Move pass_black = Move::PASS(BLACK);
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_black.ptype());
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_black.oldPtype());
    CPPUNIT_ASSERT_EQUAL(Square::STAND(), pass_black.from());
    CPPUNIT_ASSERT_EQUAL(Square::STAND(), pass_black.to());
    CPPUNIT_ASSERT_EQUAL(BLACK, pass_black.player());

    CPPUNIT_ASSERT(pass_black.isPass());
    CPPUNIT_ASSERT(! pass_black.isNormal());
    CPPUNIT_ASSERT(! pass_black.isInvalid());

    const Move pass_white = Move::PASS(WHITE);
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_white.ptype());
    CPPUNIT_ASSERT_EQUAL(PTYPE_EMPTY, pass_white.oldPtype());
    CPPUNIT_ASSERT_EQUAL(Square::STAND(), pass_white.from());
    CPPUNIT_ASSERT_EQUAL(Square::STAND(), pass_white.to());
    CPPUNIT_ASSERT_EQUAL(WHITE, pass_white.player());

    CPPUNIT_ASSERT(pass_white.isPass());
    CPPUNIT_ASSERT(! pass_white.isNormal());
    CPPUNIT_ASSERT(! pass_white.isInvalid());
  }
  void testDeclareWin()
  {
    CPPUNIT_ASSERT(Move::DeclareWin().isInvalid());
    CPPUNIT_ASSERT(! Move::DeclareWin().isNormal());
  }
  void testSetFrom()
  {
    const Square from(7,7);
    const Square to(7,6);
    const Ptype ptype = GOLD;
    const Player player = BLACK;
    const Ptype capture_ptype = PTYPE_EMPTY;
    const bool promote = false;
    Move m(from, to, ptype, capture_ptype, promote, player);
    Move m_copy(m);
    m=m.newFrom(Square(8,7));
    CPPUNIT_ASSERT_EQUAL(Square(8,7), m.from());
    CPPUNIT_ASSERT_EQUAL(to, m.to());
    CPPUNIT_ASSERT_EQUAL(ptype, m.ptype());
    CPPUNIT_ASSERT_EQUAL(capture_ptype, m.capturePtype());
    CPPUNIT_ASSERT_EQUAL(promote, m.isPromotion());
    CPPUNIT_ASSERT_EQUAL(player, m.player());

    m=m.newFrom(Square(7,7));
    CPPUNIT_ASSERT_EQUAL(m_copy, m);
  }
  void testCapturePtpeO()
  {
    const Move drop_b(Square(5,5), KNIGHT, BLACK);
    const Move drop_w(Square(5,5), KNIGHT, WHITE);
    CPPUNIT_ASSERT_EQUAL(PTYPEO_EMPTY, drop_w.capturePtypeOSafe());
    CPPUNIT_ASSERT_EQUAL(PTYPEO_EMPTY, drop_b.capturePtypeOSafe());
  }
  void testIgnoreUnpromote()
  {
    // pawn
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,3),PAWN,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,3),PPAWN,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),PAWN,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,4),PAWN,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,6),Square(4,7),PAWN,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,6),Square(4,7),PPAWN,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,7),PAWN,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,6),PAWN,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    // lance
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,3),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,2),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,3),PLANCE,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),LANCE,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,4),LANCE,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,6),Square(4,7),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,6),Square(4,8),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,6),Square(4,7),PLANCE,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,7),LANCE,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,6),LANCE,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    // bishop
    CPPUNIT_ASSERT(Move(Square(4,4),Square(2,2),BISHOP,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,2),Square(8,6),BISHOP,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(2,2),PBISHOP,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,2),Square(8,6),PBISHOP,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),BISHOP,BLACK).ignoreUnpromote());

    CPPUNIT_ASSERT(Move(Square(6,6),Square(8,8),BISHOP,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(6,8),Square(2,4),BISHOP,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,6),Square(8,8),PBISHOP,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,8),Square(2,4),PBISHOP,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,7),BISHOP,WHITE).ignoreUnpromote());
    // ROOK
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,2),ROOK,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,2),Square(4,6),ROOK,PTYPE_EMPTY,false,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,2),PROOK,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,2),Square(4,6),PROOK,PTYPE_EMPTY,true,BLACK).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),ROOK,BLACK).ignoreUnpromote());

    CPPUNIT_ASSERT(Move(Square(6,6),Square(6,8),ROOK,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(Move(Square(6,8),Square(6,4),ROOK,PTYPE_EMPTY,false,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,6),Square(6,8),PROOK,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,8),Square(6,4),PROOK,PTYPE_EMPTY,true,WHITE).ignoreUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,7),ROOK,WHITE).ignoreUnpromote());
  }
  void testHasIgnoredUnpromote()
  {
    // pawn
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,3),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,3),Square(4,2),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,2),Square(4,1),PPAWN,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),PAWN,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,4),PAWN,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote());

    CPPUNIT_ASSERT(Move(Square(4,6),Square(4,7),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,7),Square(4,8),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,8),Square(4,9),PPAWN,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,7),PAWN,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,6),PAWN,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote());
    // lance
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,2),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,3),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(4,1),PLANCE,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),LANCE,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,4),LANCE,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,6),Square(4,8),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,6),Square(4,7),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,6),Square(4,9),PLANCE,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,7),LANCE,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,5),Square(4,6),LANCE,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote());
    // bishop
    CPPUNIT_ASSERT(Move(Square(4,4),Square(2,2),PBISHOP,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,2),Square(8,6),PBISHOP,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,4),Square(2,2),PBISHOP,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),BISHOP,BLACK).hasIgnoredUnpromote());

    CPPUNIT_ASSERT(Move(Square(6,6),Square(8,8),PBISHOP,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(6,8),Square(2,4),PBISHOP,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,6),Square(8,8),PBISHOP,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,7),BISHOP,WHITE).hasIgnoredUnpromote());
    // ROOK
    CPPUNIT_ASSERT(Move(Square(4,4),Square(4,2),PROOK,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(4,2),Square(4,6),PROOK,PTYPE_EMPTY,true,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,2),Square(4,6),PROOK,PTYPE_EMPTY,false,BLACK).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(4,3),ROOK,BLACK).hasIgnoredUnpromote());

    CPPUNIT_ASSERT(Move(Square(6,6),Square(6,8),PROOK,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(Move(Square(6,8),Square(6,4),PROOK,PTYPE_EMPTY,true,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,8),Square(6,4),PROOK,PTYPE_EMPTY,false,WHITE).hasIgnoredUnpromote());
    CPPUNIT_ASSERT(!Move(Square(6,7),ROOK,WHITE).hasIgnoredUnpromote());
  }
  void testRotate180()
  {
    const Move m76(Square(7,7), Square(7,6), PAWN, PTYPE_EMPTY, false, BLACK);
    const Move m34(Square(3,3), Square(3,4), PAWN, PTYPE_EMPTY, false, WHITE);
    CPPUNIT_ASSERT_EQUAL(m76.rotate180(), m34);

    const Move m41(Square(4,1), SILVER, BLACK);
    const Move m69(Square(6,9), SILVER, WHITE);
    CPPUNIT_ASSERT_EQUAL(m41.rotate180(), m69);

    CPPUNIT_ASSERT_EQUAL(Move::PASS(BLACK).rotate180(), Move::PASS(WHITE));
  }
  static void registerMove(Move move, vector<char>& table)
  {
    const unsigned int hash = move.hash();
    CPPUNIT_ASSERT(! table.at(hash));
    table[hash] = true;
  }
  void testHash()
  {
    vector<char> table(16305+1);
    for (int z=0; z<2; ++z) {
      const Player player = indexToPlayer(z);
      for (int y=1; y<=9; ++y) {
	for (int x=1; x<=9; ++x) {
	  const Square to(x,y);
	  for (int p=PTYPE_PIECE_MIN; p<PTYPE_MAX; ++p) {
	    const Ptype ptype = static_cast<Ptype>(p);
	    if (isBasic(ptype) && ptype != KING
		&& Ptype_Table.canDropTo(player, ptype, to))
	      registerMove(Move(to, ptype, player), table);
	    for (int y2=1; y2<=9; ++y2) {
	      for (int x2=1; x2<=9; ++x2) {
		const Square from(x2, y2);
		Move move(from, to, ptype, PTYPE_EMPTY, false, player);
		if (SimpleState::isValidMoveByRule(move, false))
		  registerMove(move, table);
		if (canPromote(ptype)
		    && SimpleState::isValidMoveByRule(move.promote(), false))
		  registerMove(move.promote(), table);
	      }
	    }
	  }
	}
      }
    }    
  }
};

CPPUNIT_TEST_SUITE_REGISTRATION(MoveTest);

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
