/* ----- decode/653x1541, derived from supercop/crypto_decode/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *decode_653x1541_checksums[] = {
  "2173daf6ce1c01f7ed73da595704d4f9bb8bf3daefeca1b87d877bfed38357c8",
  "f70df1d328c5e9ed7779316472068293f57418ac0d14490ca9faf459cd8860d9",
} ;

static void (*crypto_decode)(void *,const unsigned char *);
#define crypto_decode_STRBYTES ntruprime_decode_653x1541_STRBYTES
#define crypto_decode_ITEMS ntruprime_decode_653x1541_ITEMS
#define crypto_decode_ITEMBYTES ntruprime_decode_653x1541_ITEMBYTES

static void *storage_decode_653x1541_x;
static unsigned char *test_decode_653x1541_x;
static void *storage_decode_653x1541_s;
static unsigned char *test_decode_653x1541_s;
static void *storage_decode_653x1541_x2;
static unsigned char *test_decode_653x1541_x2;
static void *storage_decode_653x1541_s2;
static unsigned char *test_decode_653x1541_s2;

#define precomputed_decode_653x1541_NUM 4

static const int16_t precomputed_decode_653x1541_x[precomputed_decode_653x1541_NUM][crypto_decode_ITEMS] = {
  {531,-1284,69,60,-1893,-222,-2262,2085,2127,-501,138,-1176,-879,-1134,2118,2115,-2010,1281,-1746,1539,75,1431,1425,906,507,-1887,2244,-1158,-1590,-2031,2244,207,1884,594,600,-387,-60,-351,-1857,1506,-1152,1539,528,192,2292,396,-492,-1371,471,1881,900,-1389,-1329,-933,720,1959,2172,1668,-255,1332,-2265,-1170,-1881,1149,957,-132,-135,1434,1944,-438,-2199,-1104,696,-1038,-30,-723,36,1623,-1500,-867,-2304,-705,-1248,585,-1542,-1563,258,840,1860,1836,-1935,-453,474,534,-1572,-237,-1404,0,-1221,-2070,-720,186,1038,-600,-1719,807,-1110,819,-1416,1875,-1620,-1356,741,-2172,231,1875,-1485,-1992,1581,462,963,123,1668,2097,-2016,-1773,2205,1803,-1677,2064,393,612,-714,789,-42,1323,63,-1026,-1608,534,1287,-1116,264,-1827,387,-984,-1251,-807,873,-1080,-72,819,24,-2226,75,132,-1359,1242,1533,885,45,-1098,-1155,1779,-261,1872,-912,1440,-867,-1959,648,441,444,993,-2076,-1650,954,-96,-1002,-1269,-858,438,-369,1053,-1692,270,2067,-1110,1689,1935,990,306,2193,1422,-1305,1104,1020,1920,2217,2142,2022,-1497,-642,-1248,1968,93,486,321,375,-870,1530,2109,-828,633,-696,-363,-1932,-2298,-1503,936,-2103,-636,1086,1380,-2058,576,2064,288,348,900,15,-1134,402,-1911,1599,1971,915,720,2163,-2310,-288,2142,-1263,963,-1833,-1413,729,-1329,-1860,717,1638,-978,-2139,1047,-1749,-1386,2118,-1725,1242,-1551,-447,-576,-156,1185,-78,597,-1476,-918,-1620,-1698,-1665,1830,39,-2229,1542,1362,-1677,-1644,1485,222,1746,-2289,879,-1065,-960,1719,474,360,345,642,-204,-840,54,2001,2190,1341,1074,-771,555,-1335,99,852,-1728,-1461,1551,2142,-1986,-195,-120,2208,1659,-897,-516,-2220,-1344,1017,1527,-753,108,2145,963,-1752,-213,1689,-1347,753,1761,78,1470,1872,1590,-201,-552,156,765,1578,-987,-729,-1284,519,1674,1137,1590,-1545,-411,75,-447,-1170,1662,-330,-162,-132,177,906,-1017,-1857,-2139,-1473,-492,1989,-1521,129,-1905,-2241,213,-390,237,873,900,1032,1338,-1857,-1077,1602,-2112,-1623,867,594,-1302,1428,111,1857,-1167,2253,2187,-195,-978,495,1956,1086,-552,-306,-1848,474,483,228,1203,-864,783,-1044,303,-1071,780,-1563,-1248,-804,675,774,-1791,-2193,960,1185,-1647,-1584,-1776,318,-1956,339,-432,486,1845,1278,1620,2115,1653,1257,222,1437,-2304,-999,66,-318,1065,1557,1407,2277,363,1752,708,93,456,2076,1065,-768,-1557,1569,150,810,-1932,1227,-1785,66,-948,573,1878,165,339,1278,48,1974,-6,1431,-585,780,831,-573,681,2184,-900,1203,-1152,-213,1506,-798,-2247,750,2223,543,-243,1881,174,-174,1839,-492,-828,1554,-1791,2010,792,-1269,1629,-2001,-1380,765,-1287,1095,-1248,1779,111,-1911,555,2238,-1338,159,-1740,1284,-2277,18,-1470,-1575,2043,-606,1524,1917,-969,1218,-477,-663,699,1590,1068,-2085,-1209,1431,1725,2199,2160,1428,-1677,1569,-27,-813,1233,-810,1284,-468,1689,1515,-2253,930,951,-480,-960,243,1575,-534,-1755,-1680,-1515,-2274,-531,1125,831,2226,531,-1902,-1791,1533,-960,177,-564,1953,144,1965,-930,-1422,-2235,1500,-921,-2103,-738,-51,1311,-1758,795,1620,-1134,1302,-2118,936,2172,975,-1098,-1095,744,-894,252,1386,2001,1932,-1200,1110,-1989,-1761,-1371,1701,228,-1566,-2292,-963,1590,75,1257,-2244,-1830,1953,1869,-381,1497,-2163,-1845,1029,-1338,-1485,-1380,-1095,-1053,1374,-342,-1440,915,-1488,-279,-1566,-528,1755,177,684,1707,612,-366,801,2019,465,-1497,-1638,-2076,1503,-1866,1785,1614,2232,36,-429,-1671,15,-1557,1413,-1932,1179,-1524,492,1626},
  {1446,336,-990,2184,-261,1848,-1659,-2298,-306,1446,1995,2238,-447,408,-435,876,-1248,-1953,-2046,444,861,759,102,1908,-1857,-1530,-822,213,3,-975,-969,423,-264,1596,-1491,-1392,705,-75,-342,1023,-276,2271,2259,1674,-864,1617,1725,132,213,-1371,-1242,597,-2277,-990,69,-564,816,381,-90,-561,-786,-1704,2250,2220,-27,213,2115,-576,1962,-2211,-735,1572,-399,369,-1860,-1416,-123,-1047,-1746,609,1299,2274,2037,-1719,1734,2298,1041,1524,-1005,1785,-2259,549,-2094,-2184,810,1824,1191,1473,615,-1263,1389,-138,1923,1281,15,-228,-696,534,-1644,-1521,1416,570,2235,-123,-546,-1263,-1275,-837,1236,-1407,-486,-1167,1419,-894,-696,753,774,411,891,-1524,2049,1815,1890,1917,-327,-90,1191,0,-1185,-1998,2106,-1209,-2094,-1692,-2196,-2082,1386,-1968,-2064,-762,105,-2043,-1917,-1788,870,588,948,180,1050,-69,-1752,1635,-2247,1056,-1749,-237,-318,-177,435,-2199,1830,735,315,1911,-1317,-2205,-1653,2211,1155,-1317,-1248,810,1620,-1101,-846,-1482,-942,1263,2268,-843,-438,-2310,-357,-801,-1482,-345,1215,636,-753,1620,1290,-201,-2277,-1344,-2259,-657,-1935,-693,888,-1347,-1656,-378,2034,-312,174,1050,-297,2070,-201,-996,-732,-489,1458,1263,-84,-1422,747,-1452,-2280,-1368,1830,444,294,-1296,-612,-2181,-1755,657,-2034,-285,855,-261,99,-699,345,1740,672,-2079,-42,9,711,-2004,1521,-1791,-1890,2265,-75,1149,-1857,1731,-876,-2256,-1581,-1356,-2025,-393,-2010,-906,-1092,1599,-111,1944,2238,2238,1032,759,1017,-1803,117,1752,222,669,-714,54,-453,1629,2025,-2088,-2136,-417,-615,-639,-132,-105,-210,-1842,-711,-1338,-1791,-420,1158,-1554,-882,1992,1137,180,1488,-144,396,1797,-2037,-948,1362,-1416,-654,-1932,1674,1866,-657,1473,-213,-1155,-924,552,357,-282,2211,1407,-1932,798,-300,1728,-147,-642,1890,660,414,60,-630,660,2076,1911,-1431,-966,-1098,-1020,747,486,-1701,2199,102,1320,909,-1731,-1512,-1059,1674,-168,-1926,-858,1014,-1734,831,1674,225,-1371,-645,-639,2121,-1947,1332,792,-1242,2232,-42,-2166,201,-1569,-354,603,999,327,363,273,-1026,-1059,2274,-1473,1014,1977,-1107,-690,-1737,531,-39,-936,-1605,351,-1062,471,1656,375,912,-189,2292,930,-1365,-1977,-1806,-558,-567,-351,102,2256,150,-1896,-51,1317,366,1299,2166,-18,-495,564,1134,1059,-1371,-414,1389,1776,-1296,-489,681,1869,69,-1290,-1629,-1452,-2199,648,1368,-1428,-1473,1023,-1080,-747,2190,444,738,-1554,-744,1236,-1047,1182,2193,-1320,1608,-2112,-2277,-1104,1734,72,-2010,1980,-2115,-249,864,2235,-18,654,1662,-1911,-1155,1500,-2280,1029,1554,-1791,1206,-426,105,1782,-537,-1875,-1446,-1152,-1047,693,-1644,-2148,-2244,-1128,-348,984,-966,513,1911,1302,879,1953,1806,-966,345,-1734,-309,-1971,-918,-2280,-846,-852,534,-1467,-1485,1098,-2085,-348,-453,-1827,-303,1188,2223,-1380,-1950,-927,1239,-639,144,876,795,-675,-576,-1503,768,2232,-1590,-33,84,-1335,1323,-189,2232,-840,-591,951,-1020,-2037,1842,-564,1335,1398,2139,246,-744,54,-1800,-294,-405,-255,1968,-138,612,774,-1884,2274,1431,27,-1443,-2148,336,-2181,-48,-2232,2181,1680,1644,-579,801,-1488,534,-930,-822,-1224,753,1098,-1641,-1122,-1632,-2139,369,1515,-1887,-1032,627,-1458,-1677,405,1641,-1248,-1572,1614,-48,918,1398,2286,1926,-888,-708,948,-987,2256,1782,-486,-591,-906,1428,-966,-2148,-906,651,-1251,1701,966,-192,1482,282,1647,-138,522,1260,2232,87,-1881,-150,1959,-1257,414,984,360,1701,-126,375,-231,561,-1002,-834,-1518,-1272,963,-384,1560,-813,-1293,324},
  {1596,1497,-855,1494,1320,-930,333,-624,639,1410,12,2025,-1764,885,1083,300,-201,396,-24,936,1977,-711,2307,1485,1650,228,399,-375,-1755,1056,315,897,-2241,273,1563,1977,1479,-2274,-1731,180,-996,912,900,-1464,-1242,462,1245,858,1878,-2019,1245,-1374,-114,1935,-2151,1749,276,-2232,972,-1176,-1140,-333,-6,-531,300,-606,1074,-1377,-750,933,765,1746,-1809,1227,1047,-2244,-1002,-1083,-2085,-1191,-2025,840,1992,456,771,-177,2157,759,471,1089,-1263,210,87,726,-678,222,468,-732,534,-1722,411,498,-729,-1557,-1878,1977,-1677,819,-1389,-1875,1947,-1182,-978,-2214,-1014,1305,1098,-759,1173,306,852,744,633,-216,750,-1887,1125,-18,1980,522,-2274,1791,-1710,-1563,-588,2094,-135,-501,450,-1518,1251,-1038,1659,-1677,60,2037,-1053,-2187,-1656,1431,510,-2214,2019,-153,1383,768,1500,-2136,2022,927,954,-531,279,2193,-2238,1914,1329,-687,234,1920,153,-576,963,2229,-984,369,-444,822,-1782,1863,-1323,-1056,1311,-1806,-438,477,261,153,-456,-1185,-1716,1317,-1593,-2163,-384,894,-1176,-501,-678,102,1689,-150,1626,147,1554,-342,2169,-2094,774,1071,-1245,-1161,2241,1896,-243,1350,-399,45,447,-1011,1038,801,-903,-390,345,609,1131,-486,-813,-780,-2076,2031,-1062,273,1317,1959,-558,-1293,342,276,984,-2211,1002,1722,-120,1341,2118,1044,-1605,-984,-1644,2226,-1191,573,-1341,-1674,-945,-1308,-1644,1536,-693,936,1368,-15,123,-1491,-1977,-27,99,1722,1515,69,-174,579,1494,-1524,-1737,1188,1488,1494,-717,63,-459,1539,-693,-660,-1512,-1872,-2247,2031,-1188,-1200,-2082,2157,-438,-81,-2280,-1941,-2289,-339,-690,-2115,-1215,-282,-1653,1452,1758,-1593,1152,2052,-858,-1233,180,495,675,861,-414,1209,33,0,2112,351,-2268,-1320,-978,-1137,-126,2094,1053,-1563,-951,1560,1401,1932,-279,1188,402,1428,1755,-1182,774,1482,-1053,-723,2109,1611,1392,588,-1977,-987,-315,-2133,-2259,1614,405,-2121,1845,-753,-1377,1131,-2178,528,-225,-1098,972,-162,-279,1281,864,1434,2154,-138,-1866,-810,-1830,-2058,-2178,-1806,915,-1404,798,-1356,-816,-963,-1185,-1689,2256,615,-450,2256,1314,2253,1110,-1530,1659,402,-627,1392,153,-1983,-1092,1080,858,888,54,-900,909,-630,2298,1695,-1956,-1158,2253,1749,-318,1566,-501,-1692,-765,-1173,-384,-354,-1644,1383,987,1005,2250,-1683,1503,264,2199,-1392,1206,555,-219,324,1164,-366,-1791,-687,1983,-120,-1485,1536,1800,561,1587,-57,-906,540,1224,-2046,468,66,-945,-2307,-753,1656,2124,-1809,-438,-687,-1404,1683,633,2253,-1359,-2145,129,1200,-630,-267,-1860,-66,1953,1569,-1653,882,-1731,297,828,1407,-2067,-171,-747,-1005,-1383,-2235,-417,-1791,-552,-2034,-1083,1140,-213,-1467,-45,-1698,1044,-261,-240,-1449,1545,-864,-1185,219,609,492,-1182,-1941,-2184,1644,-1956,570,3,-921,-1605,-1158,645,-1581,-615,1518,525,-1392,-1038,-141,2166,1668,1533,-825,-2085,-738,-1515,-1740,507,-1794,48,1191,-1605,-1074,-1728,885,45,-456,-1572,930,402,351,444,-588,1263,-1281,1650,1041,1830,792,1995,2238,-357,-3,123,-636,-6,-198,-585,300,-1110,1635,513,-2136,135,-192,1086,345,2022,318,363,1428,-783,-1458,1404,-618,1728,504,471,2064,2175,423,-246,-1260,-2238,2205,1155,999,-2235,408,-2019,351,-1629,2160,1728,-1197,513,-342,1701,426,-1329,-327,1686,-2289,2217,2247,-1095,-1206,2004,-741,1062,1053,-318,1239,-684,-615,-18,-228,1164,-2019,-30,1350,-924,969,-780,351,2157,867,1467,-588,2163,-1002,-2262,1059,510,1299,-1764,1458,-1143,1287,-564,-1353,-2304,1491,1995,-549,-867},
  {435,1842,480,1929,-2277,-297,-1374,-597,-1020,270,-2103,-702,1206,-1431,1401,-1896,636,-1404,1914,864,-891,-2292,-600,-849,-1212,-309,2064,1119,1467,-1710,-2199,-1377,177,1674,-1338,-2127,-1344,1416,783,2280,-1809,-1377,1287,2016,-2067,783,-1614,-1182,1644,6,-2190,750,93,1902,228,2199,-984,-855,18,-387,606,-582,2268,-741,1119,-207,2283,-633,2004,2310,-1221,-1884,-2157,663,-1590,1995,-1485,-741,-111,-87,357,1749,-1692,-453,1470,522,-1227,2124,-1914,204,1839,-69,-2052,1338,978,168,-15,1386,720,-237,204,-1197,-1533,-1587,1932,258,1473,897,-759,171,-522,186,1140,-1299,-99,1578,-1437,2217,1995,-795,2019,585,-1764,2145,21,-492,-1464,1590,-999,-1938,1128,2100,957,2310,1641,291,-1014,-924,639,1503,-2289,-2271,1893,-1101,1971,924,-1209,486,-2289,1215,1224,-102,1161,798,450,1518,-1773,-1251,153,-1368,-39,-390,1515,-1092,1221,1467,-1461,63,1515,-1656,1311,1626,-1845,-1617,-1680,-144,-876,-1767,96,-1107,600,1467,-1512,-960,-450,666,-1743,2223,-159,-1224,1698,1476,-1125,-1962,1272,-114,-768,1194,657,546,1428,2190,-2139,1971,-2010,-1659,912,-342,1227,-1878,-627,-330,1557,1551,2187,-1029,39,1929,-750,1806,795,1332,-1050,-2163,2310,1170,-1416,549,975,2247,1311,-321,-657,-1011,-2268,2058,1911,-2007,-1320,1122,-945,2133,711,-48,354,1461,-24,306,822,-420,468,-207,-633,1920,-840,-1824,2112,285,1362,1944,2022,-1308,-192,1467,1407,1779,657,-1017,-966,483,1986,-864,-1629,-1932,1326,1065,366,414,99,-732,6,-1194,159,2289,2241,1539,156,-1320,810,-1863,1479,-2229,1179,-2292,1296,2235,888,-996,1110,-1917,1593,555,-354,-2232,-762,-1899,261,-414,633,-1527,765,-801,-1224,-1962,-678,1416,2211,690,1104,-2310,1254,1563,2103,-2067,-915,1920,936,-774,561,-345,-423,510,-603,1542,-351,-1245,-174,-1686,1395,438,-1239,-2181,-1584,831,408,-1230,-804,-1080,-1860,1314,1755,-81,-66,-1755,1485,-231,2055,1881,330,1017,1389,-1689,-1722,-1590,-477,474,-807,1548,-2124,627,1668,525,-1260,2145,-33,1332,-1290,-1335,-801,-2241,720,-1659,2028,1290,297,-561,1308,2253,639,-300,-978,-564,-882,1311,1488,1752,1359,-1986,1062,678,-261,1608,1053,1905,-399,-213,1440,-2163,1230,-591,2010,891,-2166,-972,465,-1026,789,771,1671,69,1239,-1197,-891,-1884,438,1233,-1398,-1347,1572,1656,-966,-1869,1029,-672,-1767,945,2214,-603,657,-1554,-1842,-924,837,-1689,-1752,-1356,-849,-1356,-1524,1794,-1257,1026,-396,-957,1980,693,-105,441,1782,1884,264,-132,-1254,-942,-195,1782,2280,-864,-228,-618,-867,1029,-2253,645,1203,-1692,894,-1581,1857,-1899,1254,-1209,687,-939,1020,-2187,2301,-432,-1164,-948,2208,804,1491,-1350,-180,438,-1740,-237,-2268,1101,-1209,-300,1287,-2163,-702,1278,-261,381,-393,1119,1716,-1212,618,-1962,-2205,1935,1182,1923,636,12,-423,180,1743,36,1956,81,-861,615,633,624,-1524,-1950,396,-1302,1218,2052,-27,777,-1761,-168,1332,2052,1845,2091,1401,2178,27,-1977,-1251,-1626,228,-714,681,1761,-762,-1197,-1824,-2103,-1938,-981,1335,510,-2121,657,1482,1116,-210,2142,-1311,-423,-2286,1941,711,1326,-720,-369,1218,-1464,-1521,-447,-270,2304,-1914,1275,-2151,-288,1305,1098,1965,-663,-2088,1167,1980,-39,2019,558,-816,1746,897,369,-39,45,1710,-1713,546,1956,-939,1707,1578,-1329,2292,1476,-78,-1530,624,1002,612,-1395,1713,-933,-2163,420,462,381,600,690,-207,-405,-1509,-1032,2040,1569,-1062,-795,-2019,1425,630,891,1389,423,456,-1662,-1317,-888,570,-1764,1644,-267,-981,-1305,-1329,-468,-1986},
} ;

static const unsigned char precomputed_decode_653x1541_s[precomputed_decode_653x1541_NUM][crypto_decode_STRBYTES] = {
  {97,135,35,173,142,146,133,147,197,203,118,205,108,110,193,81,94,79,175,111,145,252,156,123,232,45,62,189,188,99,123,20,111,53,186,255,50,77,169,115,117,59,221,218,108,148,36,117,56,251,82,126,16,239,105,28,223,142,37,29,26,223,225,168,77,139,65,155,115,110,117,127,37,42,39,73,150,1,93,206,245,32,229,60,42,195,23,154,170,123,200,108,154,131,216,84,41,137,224,225,239,22,55,197,223,197,23,199,146,47,43,114,30,156,92,175,33,246,237,211,158,172,118,88,75,208,20,191,147,145,183,143,213,38,226,203,150,236,41,109,107,66,253,2,173,80,37,161,109,20,185,73,3,49,68,94,236,235,34,39,229,196,50,217,38,207,88,81,4,193,161,15,0,217,16,250,45,162,172,91,232,250,201,10,240,179,19,187,11,166,72,50,160,255,170,86,186,41,253,69,120,227,48,247,112,3,151,206,181,166,197,237,98,177,139,16,146,180,237,147,40,111,60,132,133,39,139,145,207,246,91,93,102,39,181,41,208,1,122,250,36,61,22,205,209,36,59,128,1,192,87,87,36,51,225,195,131,183,28,79,221,202,156,40,20,184,97,172,28,43,41,97,159,211,69,174,159,91,178,243,7,7,148,231,40,193,79,161,12,98,57,125,50,102,59,182,254,86,38,203,155,19,97,0,234,88,163,31,159,58,141,125,201,187,65,201,180,111,33,61,174,77,248,56,53,68,204,141,136,77,1,133,29,158,100,88,234,35,109,255,240,182,6,199,226,243,15,39,161,14,40,74,161,8,9,229,216,146,197,219,24,123,99,45,230,16,254,124,130,20,90,144,142,245,246,76,17,82,154,236,29,53,168,18,2,13,41,107,239,251,11,128,216,81,242,229,252,160,77,29,77,38,246,12,107,48,216,164,23,122,201,210,99,133,96,235,241,149,50,167,251,184,189,233,237,90,165,246,117,197,23,201,71,74,66,24,79,178,249,185,254,115,238,138,244,204,239,223,236,44,155,9,22,140,118,44,36,17,141,152,51,236,65,141,226,212,93,44,2,1,142,246,164,224,28,127,211,78,46,230,36,40,122,7,239,214,213,36,207,254,40,194,31,235,31,169,253,25,33,24,78,241,14,234,115,94,0,156,173,153,194,87,5,47,32,230,157,90,226,225,62,149,26,116,137,41,159,167,2,107,240,76,38,114,255,125,203,140,23,192,102,67,107,144,142,33,150,172,195,174,217,11,25,195,224,52,39,11,42,7,119,234,234,50,145,182,50,43,183,31,25,129,72,88,101,203,88,140,47,81,163,181,46,88,108,74,142,170,230,167,174,162,106,205,17,95,80,99,32,251,255,226,111,127,98,159,29,210,99,20,255,209,37,192,62,16,240,90,236,62,152,160,58,102,174,129,91,119,254,204,167,109,163,71,101,141,176,166,240,223,192,246,54,11,216,71,155,203,108,162,52,19,97,52,169,61,248,61,32,118,147,55,20,39,208,118,77,4,78,89,189,78,251,227,253,3,203,99,89,202,62,192,175,71,186,153,145,48,91,23,188,179,124,247,23,88,170,35,103,69,174,161,44,245,12,84,205,98,1,93,3,113,251,198,139,221,18,50,31,226,48,56,114,185,108,16,47,219,134,253,152,53,120,92,117,62,150,228,13,101,231,57,9,199,225,227,96,130,193,149,38,190,243,252,208,23,117,183,163,204,49,246,10,163,141,141,249,228,63,35,50,127,113,154,32,104,127,149,155,165,233,13,149,114,201,117,140,126,51,61,98,84,122,254,6,228,84,234,178,132,42,81,117,15,200,209,10,101,73,113,113,125,57,129,92,117,23,142,114,158,177,155,217,92,19,3,81,136,100,183,188,71,250,128,152,89,225,223,236,197,109,18,215,52,209,175,19,59,231,5,165,123,72,73,193,186,132,171,23,6,13,192,245,59,39},
  {30,250,189,237,16,55,31,47,181,70,28,154,171,93,180,138,24,11,122,67,121,227,115,39,102,81,163,119,151,71,238,110,102,13,53,83,238,104,18,189,139,130,68,83,92,174,26,250,48,160,245,228,149,158,1,154,40,29,240,127,209,19,11,187,73,136,253,9,153,127,235,78,162,10,102,226,233,2,116,247,105,255,58,123,76,55,230,250,74,250,178,253,76,11,131,112,94,223,197,157,107,85,212,4,111,110,170,28,35,77,233,43,174,145,44,82,254,13,10,139,118,160,211,99,13,237,94,85,110,230,56,41,218,136,9,143,136,85,162,155,177,120,16,23,141,22,49,200,105,251,112,222,179,12,67,57,154,158,96,200,64,112,173,22,186,232,173,206,250,134,49,229,250,47,38,198,246,47,34,244,20,7,106,12,34,240,246,228,24,131,122,76,209,107,61,29,62,231,62,187,165,1,22,102,16,6,222,230,142,251,192,19,107,145,107,45,26,189,121,103,136,202,199,202,228,32,97,79,100,229,198,82,172,185,72,197,79,199,105,216,201,36,170,50,236,200,0,33,3,232,28,53,208,2,102,49,67,17,144,99,245,121,128,240,176,131,148,195,119,138,124,120,117,198,227,111,110,202,220,182,226,102,22,47,73,116,211,107,103,176,2,45,155,255,100,223,95,128,15,130,246,192,198,76,115,52,97,158,90,192,141,114,35,68,56,234,155,27,172,113,8,65,20,169,180,146,148,161,81,134,50,169,31,85,139,63,92,129,197,220,114,121,199,1,162,19,237,113,142,115,129,221,194,85,243,57,236,119,50,102,63,221,123,8,134,33,189,187,48,178,226,22,63,162,3,243,12,175,112,16,43,164,69,194,68,28,29,231,191,105,159,99,43,11,134,211,30,144,112,210,84,52,166,186,204,19,9,12,146,52,249,157,95,255,12,175,123,19,54,31,255,83,97,51,110,138,86,102,45,102,97,215,227,218,42,179,181,147,1,54,81,249,38,250,101,59,234,254,8,59,53,105,130,127,27,215,189,20,67,210,23,185,245,103,178,30,1,198,89,246,69,250,132,69,48,207,6,19,75,252,109,255,30,156,200,209,192,33,112,162,100,67,49,74,251,226,192,231,211,187,118,131,224,118,236,106,78,46,206,242,226,158,241,84,13,179,105,48,80,143,130,120,176,163,181,26,36,28,212,204,229,202,73,14,140,138,161,180,21,185,246,130,2,45,237,40,142,192,48,254,201,121,8,22,249,109,208,149,254,121,15,11,29,102,190,169,148,252,51,49,60,162,238,39,78,103,185,26,112,201,82,251,212,104,196,224,138,216,236,76,21,131,82,232,187,14,24,135,9,150,106,220,98,170,188,171,229,31,4,56,233,130,9,116,249,106,215,142,11,151,242,16,217,56,146,164,177,23,73,173,24,241,156,98,17,70,65,123,168,221,65,77,39,4,72,22,131,18,63,195,233,44,98,62,42,220,203,152,25,223,73,24,74,24,161,204,75,60,98,150,121,209,56,59,202,247,102,8,216,67,208,130,141,221,121,254,172,253,132,100,171,15,109,147,229,8,27,96,227,183,6,114,215,216,97,196,178,142,99,26,47,194,197,211,131,198,15,186,177,87,92,46,50,110,24,102,247,244,194,55,88,128,214,189,174,148,113,59,187,22,236,8,186,29,219,151,245,114,1,131,221,120,14,195,2,177,241,85,207,171,197,100,157,199,39,45,68,59,35,138,222,187,96,110,159,88,125,254,129,217,127,3,82,123,235,122,103,81,97,127,144,149,3,240,103,202,136,49,33,63,15,112,73,54,198,224,25,209,104,42,219,19,123,83,132,239,244,135,234,88,140,224,26,145,21,214,158,86,243,76,25,185,109,107,78,177,195,31,114,9,55,232,7,73,113,52,128,159,109,25,243,51,145,84,194,174,76,107,211,56,59,19,187,234,66},
  {223,169,182,107,15,63,131,105,93,28,254,184,174,32,163,76,232,244,43,247,176,176,112,65,89,185,127,162,224,168,101,149,126,123,37,105,174,205,177,148,225,156,230,204,192,197,227,156,228,136,211,246,121,50,8,227,92,57,137,145,0,119,189,101,6,191,165,234,156,192,235,74,99,112,53,76,174,217,24,183,213,176,152,78,218,127,208,43,210,219,115,255,149,98,189,99,228,94,65,212,69,31,216,61,7,222,79,133,210,12,157,85,118,91,233,145,113,132,231,74,239,144,159,154,141,102,58,103,77,232,61,187,128,76,99,120,151,18,133,182,134,120,217,228,88,176,97,243,113,220,97,161,89,247,232,231,129,136,97,23,21,128,95,132,62,247,115,107,222,163,180,248,72,176,12,192,157,196,140,158,166,155,66,6,130,244,88,68,116,25,66,183,130,253,38,109,136,211,75,86,168,242,66,33,19,246,94,212,97,106,62,233,224,5,185,217,59,223,46,62,217,29,11,190,98,82,22,204,217,9,1,45,138,80,7,125,50,206,163,186,73,62,136,178,216,22,15,56,239,181,77,111,238,11,223,232,54,14,107,152,33,171,199,186,172,135,191,7,221,247,143,35,14,219,204,90,249,150,55,144,17,192,61,63,207,249,181,192,209,137,218,148,25,162,134,125,14,203,180,253,129,125,9,48,87,144,197,213,73,179,39,138,107,204,39,81,128,33,87,255,43,62,240,129,193,18,229,192,167,89,117,202,224,253,220,117,65,180,223,41,112,112,228,78,193,54,147,226,112,133,191,106,216,245,170,155,164,178,98,156,131,165,33,92,144,212,87,112,66,121,164,243,192,122,68,130,77,30,197,245,214,15,125,253,221,163,237,85,35,152,204,206,201,23,136,105,203,133,45,61,7,220,129,225,54,30,139,111,36,190,220,6,212,72,175,87,176,183,39,20,9,71,138,89,180,83,222,164,12,52,254,135,103,255,97,136,188,166,244,191,109,255,82,54,34,253,112,193,115,125,59,123,227,218,173,106,176,135,255,125,31,122,157,7,44,43,84,151,99,122,234,132,21,117,14,185,87,35,84,97,94,45,229,58,189,164,27,54,94,63,93,88,61,254,187,41,169,248,80,213,112,147,199,49,142,125,56,162,98,102,251,184,84,172,25,238,206,203,137,132,116,202,207,239,13,102,87,228,67,203,21,66,109,73,217,48,1,253,47,7,76,136,68,126,86,27,61,152,191,160,150,173,78,16,254,59,20,86,227,34,183,160,83,129,120,147,218,171,55,106,143,184,178,230,85,133,52,99,101,120,46,249,97,7,90,142,164,73,40,172,50,131,11,24,99,42,42,28,76,74,131,238,137,218,38,175,114,63,240,198,140,230,121,54,2,81,206,251,57,34,212,98,77,1,0,178,110,118,211,121,56,24,245,91,167,103,79,193,182,224,94,38,96,225,209,208,242,105,44,46,17,21,214,57,198,36,168,120,10,84,9,45,226,73,19,26,176,201,178,201,185,154,57,255,159,136,136,246,68,211,166,57,11,240,220,80,114,189,77,175,112,165,63,45,5,219,248,208,215,189,95,92,230,129,115,93,38,68,109,43,55,150,41,12,220,32,9,25,68,93,188,111,24,143,16,77,142,245,75,110,119,225,78,215,126,245,246,139,93,140,147,246,189,138,99,39,70,240,32,31,85,50,55,77,142,37,163,91,255,252,148,13,17,80,192,149,38,100,45,244,16,125,189,221,123,24,18,177,234,247,47,213,223,128,135,190,224,254,178,138,186,25,107,89,1,73,113,15,233,14,182,131,186,89,19,228,165,150,135,164,203,154,120,49,46,60,94,1,75,145,201,114,0,109,220,134,163,201,132,39,216,101,15,0,244,55,199,162,161,43,133,69,211,157,102,76,205,14,227,252,145,229,94,184,49,61,115,69,102,132,81,56,17,239,206},
  {155,59,38,95,122,189,77,135,188,42,247,189,117,5,211,56,45,117,132,233,186,217,116,64,58,20,141,169,51,141,12,45,40,230,178,49,142,247,74,86,230,229,92,71,226,254,22,106,13,113,133,184,62,206,44,148,19,49,156,120,124,183,223,126,33,48,85,20,182,170,72,88,161,163,250,250,193,132,152,87,117,233,48,142,61,87,30,236,103,247,97,212,204,64,223,222,207,246,210,117,42,24,161,2,107,21,44,50,168,212,193,153,172,199,242,170,156,126,120,162,180,169,5,2,98,79,185,20,167,122,42,89,118,72,239,2,89,181,56,105,72,40,252,168,249,118,169,73,184,3,188,14,177,177,238,212,174,98,107,114,219,84,91,58,136,209,77,124,216,183,60,101,146,248,46,206,137,122,248,25,220,116,131,19,93,163,95,173,173,106,47,22,200,107,237,26,111,100,227,216,45,210,36,215,205,119,230,249,24,172,222,159,165,117,224,159,3,230,1,201,162,36,240,29,190,23,77,35,200,162,136,72,101,100,186,158,80,153,235,26,207,212,35,59,92,40,51,170,61,67,133,164,116,48,210,100,20,198,27,244,171,141,31,238,239,94,22,254,18,169,36,16,13,54,213,178,102,67,251,35,27,144,62,184,150,181,155,181,26,40,178,219,95,229,74,237,95,79,90,158,74,31,126,240,115,164,252,98,195,52,247,33,82,36,93,235,125,178,171,156,210,141,78,115,70,69,23,160,253,186,236,188,223,57,235,77,6,47,126,208,200,2,237,54,126,68,140,76,182,147,56,57,120,53,106,151,25,133,130,27,51,80,147,217,192,207,169,143,113,166,53,56,72,126,242,104,123,110,113,142,62,69,125,187,13,45,181,116,12,223,109,52,2,161,172,130,145,31,93,94,42,196,84,207,94,255,95,150,81,52,11,152,1,57,243,3,31,151,189,174,178,255,141,69,198,201,32,26,7,151,146,178,24,196,112,97,23,3,81,54,170,167,124,162,112,231,26,21,6,145,23,217,246,109,72,247,73,42,31,227,148,150,143,49,122,172,85,188,33,172,79,167,60,142,69,75,92,152,93,234,97,20,153,144,6,223,216,231,127,248,22,68,226,231,188,31,231,58,203,136,49,219,108,39,89,148,51,145,215,38,142,247,194,232,1,217,167,83,96,180,241,54,146,127,1,52,209,210,177,99,41,249,3,29,172,114,181,52,97,188,56,138,18,141,151,55,183,252,6,64,118,215,176,195,230,251,103,199,196,160,254,157,149,47,128,131,223,224,65,60,228,84,231,217,12,223,1,99,83,203,2,169,47,164,16,16,78,22,121,1,73,111,78,126,87,128,169,235,21,31,14,181,123,128,30,137,156,196,234,231,24,99,219,156,149,77,33,90,43,150,46,139,82,110,131,141,31,93,60,54,31,35,116,58,191,218,212,232,244,138,193,121,143,221,202,244,37,5,51,246,207,92,64,116,164,0,100,77,135,94,239,148,69,253,135,185,142,247,11,235,96,181,222,12,52,205,38,254,138,37,54,52,49,169,58,5,63,108,141,92,20,132,45,61,133,111,209,92,121,241,240,68,65,7,182,171,42,174,253,47,131,58,113,13,92,108,93,249,244,100,137,78,42,229,69,222,65,52,46,255,121,52,156,99,202,130,137,106,188,156,45,228,225,199,7,40,72,218,110,61,215,31,67,118,117,98,113,91,22,193,202,171,3,15,190,33,111,89,14,246,43,187,219,6,167,222,199,143,222,26,240,105,112,250,105,47,161,21,234,45,9,127,244,237,197,113,127,22,179,54,25,164,37,74,235,145,176,200,126,221,100,186,53,110,238,68,82,166,57,165,129,77,225,50,165,37,97,71,183,123,104,143,131,204,95,169,62,158,24,116,182,79,9,69,43,202,56,202,155,7,112,154,34,151,44,37,55,36,167,170,81,72,56,152,102},
} ;

static void test_decode_653x1541_impl(long long impl)
{
  unsigned char *x = test_decode_653x1541_x;
  unsigned char *s = test_decode_653x1541_s;
  unsigned char *x2 = test_decode_653x1541_x2;
  unsigned char *s2 = test_decode_653x1541_s2;
  long long xwords = crypto_decode_ITEMS;
  long long xlen;
  long long slen = crypto_decode_STRBYTES;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_decode_653x1541_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_decode = ntruprime_dispatch_decode_653x1541(impl);
    printf("decode_653x1541 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_decode_653x1541_implementation(impl),ntruprime_dispatch_decode_653x1541_compiler(impl));
  } else {
    crypto_decode = ntruprime_decode_653x1541;
    printf("decode_653x1541 selected implementation %s compiler %s\n",ntruprime_decode_653x1541_implementation(),ntruprime_decode_653x1541_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 1024;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {
      xlen = xwords*crypto_decode_ITEMBYTES;

      output_prepare(x2,x,xlen);
      input_prepare(s2,s,slen);
      secret(s,slen);
      crypto_decode(x,s);
      public(s,slen);
      public(x,xlen);
      endianness(x,xwords,crypto_decode_ITEMBYTES);
      checksum(x,xlen);
      output_compare(x2,x,xlen,"crypto_decode");
      input_compare(s2,s,slen,"crypto_decode");

      double_canary(x2,x,xlen);
      double_canary(s2,s,slen);
      secret(s2,slen);
      crypto_decode(x2,s2);
      public(s2,slen);
      public(x2,xlen);
      endianness(x2,xwords,crypto_decode_ITEMBYTES);
      if (memcmp(x2,x,xlen) != 0) fail("failure: crypto_decode is nondeterministic\n");
    }
    checksum_expected(decode_653x1541_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_decode_653x1541_NUM;++precomp) {
    output_prepare(x2,x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
    input_prepare(s2,s,crypto_decode_STRBYTES);
    memcpy(s,precomputed_decode_653x1541_s[precomp],crypto_decode_STRBYTES);
    memcpy(s2,precomputed_decode_653x1541_s[precomp],crypto_decode_STRBYTES);
    crypto_decode(x,s);
    if (memcmp(x,precomputed_decode_653x1541_x[precomp],crypto_decode_ITEMS*crypto_decode_ITEMBYTES)) {
      fail("failure: crypto_decode fails precomputed test vectors\n");
      printf("expected x: ");
      for (long long pos = 0;pos < crypto_decode_ITEMS*crypto_decode_ITEMBYTES;++pos) printf("%02x",((unsigned char *) precomputed_decode_653x1541_x[precomp])[pos]);
      printf("\n");
      printf("received x: ");
      for (long long pos = 0;pos < crypto_decode_ITEMS*crypto_decode_ITEMBYTES;++pos) printf("%02x",x[pos]);
      printf("\n");
    }
    output_compare(x2,x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES,"crypto_decode");
    input_compare(s2,s,crypto_decode_STRBYTES,"crypto_decode");
  }
}

void test_decode_653x1541(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"decode")) return;
  if (targetp && strcmp(targetp,"653x1541")) return;
  storage_decode_653x1541_x = callocplus(crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  test_decode_653x1541_x = aligned(storage_decode_653x1541_x,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  if (crypto_decode_ITEMS*crypto_decode_ITEMBYTES > maxalloc) maxalloc = crypto_decode_ITEMS*crypto_decode_ITEMBYTES;
  storage_decode_653x1541_s = callocplus(crypto_decode_STRBYTES);
  test_decode_653x1541_s = aligned(storage_decode_653x1541_s,crypto_decode_STRBYTES);
  if (crypto_decode_STRBYTES > maxalloc) maxalloc = crypto_decode_STRBYTES;
  storage_decode_653x1541_x2 = callocplus(maxalloc);
  test_decode_653x1541_x2 = aligned(storage_decode_653x1541_x2,crypto_decode_ITEMS*crypto_decode_ITEMBYTES);
  storage_decode_653x1541_s2 = callocplus(maxalloc);
  test_decode_653x1541_s2 = aligned(storage_decode_653x1541_s2,crypto_decode_STRBYTES);

  for (long long offset = 0;offset < 1;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("decode_653x1541 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_decode_653x1541();++impl)
      forked(test_decode_653x1541_impl,impl);
    ++test_decode_653x1541_x;
    ++test_decode_653x1541_s;
    ++test_decode_653x1541_x2;
    ++test_decode_653x1541_s2;
  }
  free(storage_decode_653x1541_s2);
  free(storage_decode_653x1541_x2);
  free(storage_decode_653x1541_s);
  free(storage_decode_653x1541_x);
}
#undef crypto_decode_STRBYTES
#undef crypto_decode_ITEMS
#undef crypto_decode_ITEMBYTES

