#if !defined(__SINGLETON_HPP)
#define  __SINGLETON_HPP

/*
   CoreLinux++ 
   Copyright (C) 1999 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

namespace corelinux
{
   /**
   Ensure a class only has one instance, and provide a global point of access to it.
   This is easier said than done outside of the solution domain. While we can
   declare a protocol by which the use of this type will prevent multiple applications,
   we can't insure that the implementor won't violate said protocol.
   */

   template< class TypeImpl >
      class Singleton : public CoreLinuxObject
      {

      public:

         //
         // Constructors and destructor
         //

                        /**
                        Default constructor sets theSingleton and theType
                        after insuring that they are not already
                        instantiated
                        @exception Assertion
                        */

                        Singleton( void ) throw( Assertion )
                           :
                           CoreLinuxObject()
                        {
                           REQUIRE( theSingleton == NULLPTR );
                           REQUIRE( theType == NULLPTR );
                           theSingleton = this;
                           theType = new TypeImpl;
                        }

                        /**
                        Initializing constructor
                        */
                        Singleton( TypeImpl *aTypePtr ) throw( Assertion )
                           :
                           CoreLinuxObject()
                        {
                           ENSURE( aTypePtr != NULLPTR );
                           REQUIRE( theSingleton == NULLPTR );
                           REQUIRE( theType == NULLPTR );
                           theSingleton = this;
                           theType = aTypePtr;
                        }

                        /// Virtual destructor

         virtual        ~Singleton( void )
                        {
                           if( theSingleton == this )
                           {
                              theSingleton = NULLPTR;
                              if( theType != NULLPTR )
                              {
                                 delete theType;
                                 theType = NULLPTR;
                              }
                              else
                              {
                                 ;  // do nothing
                              }
                           }
                           else
                           {
                              ;  // do nothing
                           }
                        }
   
         //
         // Operator overload
         //

                  /**
                  Equality operator tests that theSingleton instances
                  are equal
                  @param Singleton const reference
                  @return bool true if equal
                  */

                  bool  operator==( const Singleton & aSingleton ) const
                  {
                     return ( &aSingleton == theSingleton );
                  }

         //
         // Accessor
         //

                  /**
                  Returns the instance of the TypeImpl
                  @return TypeImpl pointer
                  */

         static   TypeImpl * instance( void )
                  {
                     return theType;
                  }

      private:

            //
            // Constructor
            //
                           /**
                           Copy constructor is not allowed
                           @exception Assertion - NEVER_GET_HERE
                           */
                  
                           Singleton( const Singleton & ) throw( Assertion )
                              :
                              CoreLinuxObject()
                           {
                              NEVER_GET_HERE;
                           }

            //
            // Operator overload
            //

                  /**
                  Assignment operator is not allowed
                  @exception Assertion - NEVER_GET_HERE
                  */

                  Singleton & operator=( const Singleton & ) 
                                 throw( Assertion )
                  {
                     NEVER_GET_HERE;
                     return (*this);
                  }

      private:

                  /// The singleton instance

         static   Singleton<TypeImpl>  *theSingleton;

                  /// The type instance

         static   TypeImpl             *theType;
      };

/// Static definition for theSingleton

template< class TypeImpl > 
   Singleton<TypeImpl> *Singleton<TypeImpl>::theSingleton( NULLPTR );

/// Static definition for theType

template< class TypeImpl >
   TypeImpl *Singleton<TypeImpl>::theType( NULLPTR );
}

#endif // if !defined(__SINGLETON_HPP)

/*
   Common rcs information do not modify
   $Author: prudhomm $
   $Revision: 1.1 $
   $Date: 2000/04/23 20:43:13 $
   $Locker:  $
*/

