/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quarter@users.sourceforge.net> */

#include "include/support.h"
#include "include/environment.h"

extern gboolean enable_debug;

#define DISALLOW_RECURSIVE \
{\
	static gboolean _recursive=FALSE;\
	if (_recursive == TRUE) {\
		_recursive = FALSE;\
		return;\
	}\
	_recursive = TRUE;\
}


static gchar *correct_filename (gchar *filename)
{
	g_return_val_if_fail (filename != NULL, NULL);
	
	if (TRUE == g_str_has_suffix (filename, ".lwp"))
		return filename;
	else {	
		gchar *s = g_strdup_printf ("%s.lwp", filename);
		g_free (filename);
		return s;
	}
}

void set_title (gchar *name)
{
	g_return_if_fail (name != NULL);
	
	gsize bytes_read=0, bytes_written=0;
	
	GtkWidget *w = 
		gtk_widget_get_ancestor (LWEnvironment::getWidget(), GTK_TYPE_WINDOW);

	gchar *base_name = g_path_get_basename (name);

	gchar *name_utf = 
		g_filename_to_utf8 (base_name, -1, &bytes_read, &bytes_written, NULL);
	
	gchar *title = g_strdup_printf (_("Little Wizard <%s>"), name_utf);
	gtk_window_set_title (GTK_WINDOW (w), title);		

	g_free (base_name);
	g_free (title);
	g_free (name_utf);	
}

static void clear_title ()
{
	GtkWidget *w = 
		gtk_widget_get_ancestor (LWEnvironment::getWidget(), GTK_TYPE_WINDOW);

	gtk_window_set_title (GTK_WINDOW (w), _("Little Wizard"));		
}

static GtkFileFilter *create_filter ()
{
	GtkFileFilter *filter = gtk_file_filter_new ();
	gtk_file_filter_add_pattern (filter, "*.lwp");
	gtk_file_filter_set_name (filter, "Little Wizard Projects");
	
	return filter;
}

void on_new ()
{
	GtkWidget *question_dialog = gtk_message_dialog_new (NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
				    _("Do you want to create new project ?"));

	GtkResponseType r = (GtkResponseType) gtk_dialog_run (GTK_DIALOG (question_dialog));

	if (r == GTK_RESPONSE_YES) {
			delete LWEnvironment::getProject();
			LWEnvironment::setProject(new LWProject);
			clear_title();
	}
	
	gtk_widget_destroy (question_dialog);	
	
}

void on_open ()
{
	GtkWidget *open_dialog = gtk_file_chooser_dialog_new (_("Open"), NULL, 
			GTK_FILE_CHOOSER_ACTION_OPEN, 
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (open_dialog), TRUE);
	gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (open_dialog), create_filter());		
	GtkResponseType r = (GtkResponseType) gtk_dialog_run (GTK_DIALOG (open_dialog));

	gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (open_dialog));
	
	if (r == GTK_RESPONSE_ACCEPT) {
			LWEnvironment::getProject()->load (filename);
			set_title (filename);
	}

	g_free (filename);
	
	gtk_widget_destroy (open_dialog);	
}

void on_save ()
{
	gchar *filename = LWEnvironment::getProject()->getFileName();
	
	if (filename == NULL) {
		
		GtkWidget *save_dialog = gtk_file_chooser_dialog_new (_("Save"), NULL,
				GTK_FILE_CHOOSER_ACTION_SAVE, 
				GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
		gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (save_dialog), TRUE);
		gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (save_dialog), create_filter());		
		
		GtkResponseType r = (GtkResponseType) gtk_dialog_run (GTK_DIALOG (save_dialog));
	
		if (r != GTK_RESPONSE_ACCEPT) {
			gtk_widget_destroy (save_dialog);	
			g_free (filename);
			return;
		}

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (save_dialog));
		
		filename = correct_filename (filename);

		gtk_widget_destroy (save_dialog);	
		
	}

	LWEnvironment::getProject()->save (filename);
	set_title (filename);
	g_free (filename);
}

void on_saveas ()
{
	GtkWidget *save_dialog = gtk_file_chooser_dialog_new (_("Save as"), NULL,
			GTK_FILE_CHOOSER_ACTION_SAVE, 
			GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
			GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (save_dialog), TRUE);
	gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (save_dialog), create_filter());		
	
	GtkResponseType r = (GtkResponseType) gtk_dialog_run (GTK_DIALOG (save_dialog));
	
	if (r == GTK_RESPONSE_ACCEPT) {
			gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (save_dialog));
			filename = correct_filename (filename);
			LWEnvironment::getProject()->save (filename);
			set_title (filename);
			g_free (filename);
	}
	
	gtk_widget_destroy (save_dialog);	
}

gboolean on_delete ()
{
	GtkWidget *question_dialog = gtk_message_dialog_new (NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
				    _("Do you want to quit ?"));

	GtkResponseType r = (GtkResponseType) gtk_dialog_run (GTK_DIALOG (question_dialog));

	gtk_widget_destroy (question_dialog);	
	
	if (r == GTK_RESPONSE_YES) {
		LWEnvironment::uninit();
		gtk_main_quit();
		return FALSE;
	} else
		return TRUE;
}

void on_quit ()
{
	on_delete();
}

void on_grid (GtkWidget *w)
{
	DISALLOW_RECURSIVE

	gboolean b=FALSE;
	
	if (TRUE == GTK_IS_CHECK_MENU_ITEM (w)) {
		b = (TRUE == gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (w))) ? FALSE : TRUE;
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (w), b);
	}

	if (TRUE == GTK_IS_TOGGLE_TOOL_BUTTON (w)) {
		b = (TRUE == gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (w))) ? FALSE : TRUE;
		gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (w), b);
	}
	
	LWEnvironment::enableGrid(b);
}

void on_execute ()
{
	LWProgram *program = new LWProgram();
	program->setWorld(LWEnvironment::getProject()->getWorld());
	
	if (TRUE == program->parse(LWEnvironment::getProject()->getProgram(), enable_debug))
		program->execute();
}

void on_world (GtkWidget *w)
{
	DISALLOW_RECURSIVE
	
	if (TRUE == GTK_IS_TOGGLE_TOOL_BUTTON (w))
		gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (w), TRUE);

	if (TRUE == GTK_IS_CHECK_MENU_ITEM (w))
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (w), TRUE);
	
	LWEnvironment::setDesignMode (LW_DESIGN_WORLD);
}

void on_program (GtkWidget *w)
{
	DISALLOW_RECURSIVE
	
	if (TRUE == GTK_IS_TOGGLE_TOOL_BUTTON (w))
		gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (w), TRUE);

	if (TRUE == GTK_IS_CHECK_MENU_ITEM (w))
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (w), TRUE);
	
	LWEnvironment::setDesignMode (LW_DESIGN_PROGRAM);
}

void on_both (GtkWidget *w)
{
	DISALLOW_RECURSIVE
	
	if (TRUE == GTK_IS_TOGGLE_TOOL_BUTTON (w))
		gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (w), TRUE);

	if (TRUE == GTK_IS_CHECK_MENU_ITEM (w))
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (w), TRUE);
	
	LWEnvironment::setDesignMode (LW_DESIGN_MIXED);
}

void on_about (GtkWidget *window)
{
	GtkWidget *dialog = gtk_dialog_new_with_buttons (_("About Little Wizard"),
                                                  GTK_WINDOW(window),
                                                  GTK_DIALOG_MODAL,
                                                  GTK_STOCK_CLOSE,
                                                  GTK_RESPONSE_CLOSE,
                                                  NULL);
	gtk_window_set_resizable (GTK_WINDOW(dialog), FALSE);
	
	GtkWidget *hbox = gtk_hbox_new (0, FALSE);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), hbox);
	gtk_widget_show (hbox);
	
	GtkWidget *image = gtk_image_new_from_file (LWEnvironment::buildPixmapFilename ("logo.jpg"));	
	gtk_box_pack_start_defaults (GTK_BOX (hbox), image);
	gtk_widget_show (image);
	
	GtkTextIter iter;
	GtkTextBuffer *buffer = gtk_text_buffer_new (NULL);
	
	gtk_text_buffer_get_start_iter (buffer, &iter);

	gtk_text_buffer_create_tag (buffer, "bigandbold",
			    "weight", PANGO_WEIGHT_BOLD,
			    "size", 20 * PANGO_SCALE,
				NULL);  

	gtk_text_buffer_insert_with_tags_by_name (buffer, &iter,
					    _("Little Wizard"), -1,
					    "bigandbold", NULL);

	gtk_text_buffer_insert_with_tags_by_name (buffer, &iter,
					    " ", -1,
					    "bigandbold", NULL);

	gtk_text_buffer_insert_with_tags_by_name (buffer, &iter,
					    VERSION, -1,
					    "bigandbold", NULL);

	gtk_text_buffer_insert (buffer, &iter,
					    _("\n\nDevelopment environment for children"), -1);

	gtk_text_buffer_insert (buffer, &iter,
					    "\nCopyright (c) Marcin Kwadrans\n", -1);

	gtk_text_buffer_insert (buffer, &iter, 
						_("\nSpecial thanks for helping with graphics to:"), -1);
	
	gtk_text_buffer_insert (buffer, &iter, 
						"\nAgata Cuber"
						"\nMaciej Drozdowski", -1);

	GtkWidget *view = gtk_text_view_new_with_buffer (buffer);
 	gtk_text_view_set_editable (GTK_TEXT_VIEW(view), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(view), FALSE);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), view);
	gtk_widget_show (view);
	
	gtk_widget_show (dialog);
	
	gtk_dialog_run (GTK_DIALOG(dialog));
	gtk_widget_destroy (dialog);
	
}
