/*
 *  Copyright 1994-2022 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "oscillo.h"
#include "parameters.h"


uint32_t version = 0;
uint32_t options = BO_SFX | BO_SCHEMES;
enum LayerMode mode = LM_OVERLAY;
char desc[] = "Oscilloscope based on a sine-wave";
char dname[] = "Sin oscillo 1";


static Porteuse_t *P = NULL;

const float sin1_phi_inc = 0.0101667; /* do not ask me why --oliv3 */
const float sin1_freq_min = 2, sin1_freq_max = 10;

static float sin1_phi = 0;
static float sin1_freq;        /* initialized on create */
static float sin1_target_freq; /* initialized on create */
static float sin1_freq_inc;    /* initialized on create */
static int   sin1oscillo_connect = 1;

static double volume_scale = 1;


json_t *
get_parameters(const uint8_t fetch_all)
{
  json_t *params = json_object();

  plugin_parameters_add_double(params, BPP_VOLUME_SCALE, volume_scale, 0.1, 10, 0.1, "Volume scale");

  return params;
}


void
set_parameters(const Context_t *ctx, const json_t *in_parameters)
{
  plugin_parameter_parse_double_range(in_parameters, BPP_VOLUME_SCALE, &volume_scale);
}


json_t *
parameters(const Context_t *ctx, const json_t *in_parameters, const uint8_t fetch_all)
{
  if (NULL != in_parameters) {
    set_parameters(ctx, in_parameters);
  }

  return get_parameters(fetch_all);
}


static inline float
rnd_freq(void)
{
  /* drand48() a l'arrache --oliv3 */
  return drand48() * (sin1_freq_max - sin1_freq_min) + sin1_freq_min;
}


static void
change_params(void)
{
  if (sin1_freq_inc > 0) {
    sin1_freq += sin1_freq_inc;
    if (sin1_freq > sin1_target_freq) {
      float new_freq = rnd_freq();
      while (new_freq >= sin1_freq) {
        new_freq = rnd_freq();
      }
      sin1_freq_inc = -(drand48() / 10 + .01);
      sin1_target_freq = new_freq;
    }
  } else {
    sin1_freq += sin1_freq_inc;
    if (sin1_freq < sin1_target_freq) {
      float new_freq = rnd_freq();
      while (new_freq <= sin1_freq) {
        new_freq = rnd_freq();
      }
      sin1_freq_inc = drand48() / 10 + .01;
      sin1_target_freq = new_freq;
    }
  }
  sin1_phi += sin1_phi_inc;
}


static void
init(Context_t *ctx)
{
  uint32_t i;
  Transform_t t;

  /* BOF keep gcc happy about uninitialized variables */
  memset(&t, 0, sizeof(t));

  P->origin.x = 1;
  P->origin.y = HMAXY;

  t.v_j_factor = HMAXY / 2 * volume_scale;

  for (i = 0; i < P->size-1; i++) {
    float ya = HMAXY + t.v_j_factor * sin(sin1_freq * (float)(i) / (float)(ctx->input->size - 1) + sin1_phi);
    float yb = HMAXY + t.v_j_factor * sin(sin1_freq * (float)(i+1) / (float)(ctx->input->size - 1) + sin1_phi);

    t.v_i.x = 1.0 / (float)(ctx->input->size - 2) * MAXX;
    t.v_i.y = yb - ya;

    P->trans[i] = t;
  }
  P->trans[i] = t;

  Porteuse_init_alpha(P);
}


int8_t
create(Context_t *ctx)
{
  P = Porteuse_new(ctx->input->size, A_MONO);
  sin1_freq = sin1_freq_min;
  sin1_target_freq = rnd_freq();
  sin1_freq_inc = drand48() / 10 + .01;
  init(ctx);

  return 1;
}


void
destroy(Context_t *ctx)
{
  if (NULL != P) {
    Porteuse_delete(P);
  }
}


void
on_switch_on(Context_t *ctx)
{
  /* Initialize parameters */
  volume_scale = 1;
  /* connect = b_rand_boolean(); */
}


void
run(Context_t *ctx)
{
  Buffer8_clear(passive_buffer(ctx));
  Porteuse_draw(P, ctx, sin1oscillo_connect);
  change_params();
  init(ctx);
}
