/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/***************************************************************************
   grid-proxy-verify.c

 Sample C program that verifies a Globus GSI proxy.
 The following checks are performed:
 - certificate chain is verified , including proxy certs
 - proxy itself is verified (SUBJ/CN=proxy vs ISSUER etc)
 - proxy private key is matched against proxy public key
 - file permissions of proxy file are checked

 Build instructions:
    gcc -o grid-proxy-verify grid-proxy-verify.c \
        -I<OPENSSL-INCLUDE> -L<OPENSSL-LIB> -lssl -lcrypto
 (code is CFLAGS="-Wall -g -Wuninitialized -O" clean)

 ***************************************************************************/

#define _GNU_SOURCE

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>


#include <sys/types.h>
#include <sys/stat.h>

#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/x509_vfy.h>
#include <openssl/err.h>
#include <openssl/pem.h>

#include <openssl/rsa.h>
#include <openssl/evp.h>
#include <openssl/bio.h>
#include <openssl/des.h>
#include <openssl/rand.h>

#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include <openssl/buffer.h>
#include <openssl/objects.h>
#include <openssl/asn1.h>


#include "log.h"
#include "_verify_x509.h"
#include "verify_x509.h"



unsigned long grid_X509_knownCriticalExts(X509 *cert)
{
   int  i;
   char s[80];
   X509_EXTENSION *ex;

   for (i = 0; i < X509_get_ext_count(cert); ++i)
      {
        ex = X509_get_ext(cert, i);

        if (X509_EXTENSION_get_critical(ex) &&
                                 !X509_supported_extension(ex))
          {
            OBJ_obj2txt(s, sizeof(s), X509_EXTENSION_get_object(ex), 1);

            Log( L_DEBUG, "Critical extension found: %s", s );

            if (strcmp(s, OID_RFC_PROXY) == 0) return X509_V_OK;
            if (strcmp(s, OID_GLOBUS_PROXY_V3) == 0) return X509_V_OK;
            if (strcmp(s, OID_GLOBUS_PROXY_V2) == 0) return X509_V_OK;

            return X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION;
          }
      }

   return X509_V_OK;
}


unsigned long grid_readProxy( char *filename, STACK_OF(X509) **certstack, EVP_PKEY **pkey )
{
    char                *oper = "Reading proxy";

    STACK_OF(X509_INFO) *sk      = NULL;
    BIO                 *certbio = NULL;
    X509_INFO           *xi;
    unsigned long        err;

    Log( L_DEBUG, "--- Welcome to the grid_readProxy function ---");

    *certstack = sk_X509_new_null();
    if (*certstack == NULL) return ERR_get_error();

    if ( (certbio = BIO_new(BIO_s_file())) == NULL ) return ERR_get_error();

    Log( L_INFO, "Reading file %s", filename );
    if ( BIO_read_filename(certbio, filename) <= 0 ) return ERR_get_error();

    Log( L_DEBUG, "Reading X509_INFO records" );
    if ( !(sk=PEM_X509_INFO_read_bio(certbio, NULL, NULL, NULL)) )
    {
        err = ERR_get_error();
        Error( oper, "No X509 records found" );
        BIO_free(certbio);
        sk_X509_INFO_free(sk);
        sk_X509_free(*certstack);
        return err;
    }

    Log( L_DEBUG, "Resetting BIO" );
    if ( (err = BIO_reset( certbio )) != X509_V_OK ) return err;

    Log( L_DEBUG, "Reading Private key" );
    *pkey = PEM_read_bio_PrivateKey( certbio, NULL, grid_X509_empty_callback, NULL );

    if ( *pkey == NULL ) Log( L_WARN, "No private key found." );

    while (sk_X509_INFO_num(sk))
    {
        xi=sk_X509_INFO_shift(sk);
        if (xi->x509 != NULL)
        {
            sk_X509_push(*certstack, xi->x509);
            xi->x509=NULL;
        }
        X509_INFO_free(xi);
    }

    if (!sk_X509_num(*certstack))
    {
        err = ERR_get_error();
        Error( oper, "No certificates found" );
        BIO_free(certbio);
        sk_X509_INFO_free(sk);
        sk_X509_free(*certstack);
        return err;
    }

    BIO_free(certbio);
    sk_X509_INFO_free(sk);

    return X509_V_OK;
}



int grid_X509_check_issued_wrapper(X509_STORE_CTX *ctx,X509 *x,X509 *issuer)
/* We change the default callback to use our wrapper and discard errors
   due to GSI proxy chains (ie where users certs act as CAs) */
{
    int ret;
    ret = X509_check_issued(issuer, x);
    if (ret == X509_V_OK) return 1;

    /* Non self-signed certs without signing are ok if they passed
           the other checks inside X509_check_issued. Is this enough? */
    if ((ret == X509_V_ERR_KEYUSAGE_NO_CERTSIGN) &&
        (X509_NAME_cmp(X509_get_subject_name(issuer),
                       X509_get_subject_name(x)) != 0)) return 1;

#if OPENSSL_VERSION_NUMBER < 0x00908000L
    /* If we haven't asked for issuer errors don't set ctx */
    if (!(ctx->flags & X509_V_FLAG_CB_ISSUER_CHECK)) return 0;
#else
    if (!(ctx->param->flags & X509_V_FLAG_CB_ISSUER_CHECK)) return 0;
#endif

    ctx->error = ret;
    ctx->current_cert = x;
    ctx->current_issuer = issuer;
    return ctx->verify_cb(0, ctx);
}



/******************************************************************************
Function:   grid_verifyProxy
Description:
    Tries to verify the proxies in the certstack
******************************************************************************/
unsigned long grid_verifyProxy( STACK_OF(X509) *certstack )
{
    char    *oper = "Verifying proxy";

    int      i = 0;
    int      serialLen, j = 0;
    X509    *cert = NULL;
    time_t   now = time((time_t *)NULL);
    size_t   len = 0;             /* Lengths of issuer and cert DN */
    size_t   len2 = 0;            /* Lengths of issuer and cert DN */
    int      prevIsLimited = 0;   /* previous cert was proxy and limited */
    char    *cert_DN = NULL;      /* Pointer to current-certificate-in-certstack's DN */
    char    *issuer_DN = NULL;    /* Pointer to issuer-of-current-cert-in-certstack's DN */
    char    *proxy_part_DN = NULL;
    int      depth = sk_X509_num (certstack);
    int      amount_of_CAs = 0;
    int      is_old_style_proxy = 0;
    int      is_limited_proxy = 0;
    ASN1_INTEGER   *cert_Serial = NULL;
    ASN1_INTEGER   *issuer_Serial = NULL;
    unsigned char   serialNumberDER[127];
    unsigned char   serialStr[255];
    unsigned char  *temp;

    Log( L_DEBUG, "--- Welcome to the grid_verifyProxy function ---");

    /* And there was (current) time... */
    time(&now);

    /* How many CA certs are there in the certstack? */
    for (i = 0; i < depth; i++)
    {
        if (grid_x509IsCA(sk_X509_value(certstack, i)))
            amount_of_CAs++;
    }

    Log( L_DEBUG, "#CA's = %d , depth = %d", amount_of_CAs, depth );

    if ((amount_of_CAs + 2) > depth)
    {
        if ((depth - amount_of_CAs) > 0)
        {
            Log( L_WARN, "No proxy certificate in certificate stack to check." );
            return X509_V_OK;
        }
        else
        {
            Error( oper, "No personal certificate (neither proxy or user certificate) found in the certficiate stack." );
            return X509_V_ERR_APPLICATION_VERIFICATION;
        }
    }


    /* Changed this value to start checking the proxy and such and
       to skip the CA and the user_cert
    */
    for (i = depth - (amount_of_CAs + 2); i >= 0; i--)
    {
        /* Check for X509 certificate and point to it with 'cert' */
        if ( (cert = sk_X509_value(certstack, i)) != NULL )
        {
            cert_DN   = X509_NAME_oneline( X509_get_subject_name( cert), NULL, 0);
            issuer_DN = X509_NAME_oneline( X509_get_issuer_name( cert ), NULL, 0);
            len       = strlen( cert_DN );
            len2      = strlen( issuer_DN );

            Log( L_INFO, "Proxy to verify:" );
            Log( L_INFO, "  DN:        %s", cert_DN );
            Log( L_INFO, "  Issuer DN: %s", issuer_DN );

            if (now < grid_asn1TimeToTimeT(ASN1_STRING_data(X509_get_notBefore(cert)),0))
            {
                Error( oper, "Proxy certificate is not yet valid." );
                return X509_V_ERR_CERT_NOT_YET_VALID;
            }

            if (now > grid_asn1TimeToTimeT(ASN1_STRING_data(X509_get_notAfter(cert)),0))
            {
                Error( oper, "Proxy certificate expired." );
/* error will be caught later by x509_verify
                return X509_V_ERR_CERT_HAS_EXPIRED;
*/
            }

            /* User not allowed to sign shortened DN */
            if (len2 > len)
            {
                Error( oper, "It is not allowed to sign a shorthened DN.");
                return X509_V_ERR_APPLICATION_VERIFICATION;
            }

            /* Proxy subject must begin with issuer. */
            if (strncmp(cert_DN, issuer_DN, len2) != 0)
            {
                Error( oper, "Proxy subject must begin with the issuer.");
                return X509_V_ERR_APPLICATION_VERIFICATION;
            }

            /* Set pointer to end of base DN in cert_DN */
            proxy_part_DN = &cert_DN[len2];

            /* First attempt at support for Old and New style GSI
               proxies: /CN=anything is ok for now */
            if (strncmp(proxy_part_DN, "/CN=", 4) != 0)
            {
                Error( oper, "Could not find a /CN= structure in the DN, thus it is not a proxy.");
                return X509_V_ERR_APPLICATION_VERIFICATION;
            }

            if (strncmp(proxy_part_DN, "/CN=proxy", 9) == 0)
            {
                Log( L_INFO, "Current certificate is an old style proxy.");
                is_old_style_proxy = 1;
                is_limited_proxy = 0;
            }
            else if (strncmp(proxy_part_DN, "/CN=limited proxy", 17) == 0)
            {
                Log( L_INFO, "Current certificate is an old limited style proxy.");
                is_old_style_proxy = 1;
                is_limited_proxy = 1;
            }
            else
            {
                Log( L_INFO, "Current certificate is a GSI/RFC3820 proxy.");
            }

            if ( is_old_style_proxy )
            {
                cert_Serial = X509_get_serialNumber( cert );
                temp = serialNumberDER;
                serialLen = i2c_ASN1_INTEGER( cert_Serial, &temp );
                bzero( serialStr, sizeof( serialStr) );
                temp = serialStr;
                for (j = 0; j < serialLen; j++)
                {
                    sprintf( temp, "%02X", serialNumberDER[j] );
                    temp = temp + 2;
                }
                Log( L_DEBUG, "Serial number: %s", serialStr);

                issuer_Serial = X509_get_serialNumber( sk_X509_value(certstack, i+1));
                temp = serialNumberDER;
                serialLen = i2c_ASN1_INTEGER( issuer_Serial, &temp );
                bzero( serialStr, sizeof( serialStr) );
                temp = serialStr;
                for (j = 0; j < serialLen; j++)
                {
                    sprintf( temp, "%02X", serialNumberDER[j] );
                    temp = temp + 2;
                }
                Log( L_DEBUG, "Issuer serial number: %s", serialStr);

                if (cert_Serial && issuer_Serial)
                {
                    if ( ASN1_INTEGER_cmp( cert_Serial, issuer_Serial ) )
                    {
                        Log( L_WARN, "Serial numbers do not match." );
                    }
                }
            }

            if ( is_limited_proxy )
            {
                prevIsLimited = 1;
                if (i > 0) Log( L_WARN, "Found limited proxy.");
            }
            else
            {
                if (prevIsLimited)
                {
                    Error( oper, "Proxy chain integrity error. Previous proxy in chain was limited, but this one is a regular proxy.");
                    return X509_V_ERR_APPLICATION_VERIFICATION;
                }
            }

            if (cert_DN) free(cert_DN);
            if (issuer_DN) free(issuer_DN);
        }
    }

    return X509_V_OK;
}


/******************************************************************************
Function:   verify_certificate_type_str
Description:
            lcmaps_proxy_type_t to human readable string
******************************************************************************/
const char *
verify_certificate_type_str(lcmaps_proxy_type_t cert_type) {
    return cert_type == CA ? "CA" :
           cert_type == EEC ? "EEC" :
           cert_type == GT2_PROXY ? "GT2/old-style Proxy" :
           cert_type == GT3_PROXY ? "GT3/pre-RFC Proxy" :
           cert_type == RFC_PROXY ? "RFC3820 Proxy" :
           cert_type == GT2_LIMITED_PROXY ? "GT2/old-style Limited Proxy" :
           cert_type == GT3_LIMITED_PROXY ? "GT3/pre-RFC Limited Proxy" :
           cert_type == RFC_LIMITED_PROXY ? "RFC3820 Limited Proxy" :
           "Unknown";
}

/******************************************************************************
Function:   verify_PROXYCERTINFO_get_policy
Description:
            Get a policy from the PROXYCERTINFO structure
******************************************************************************/
PROXYPOLICY *
verify_PROXYCERTINFO_get_policy(PROXYCERTINFO *cert_info) {
    if(cert_info) {
        return cert_info->policy;
    }
    return NULL;
}

/******************************************************************************
Function:   verify_PROXYPOLICY_get_policy_language
Description:
            Get the proxy language from the proxy policy
******************************************************************************/
ASN1_OBJECT *
verify_PROXYPOLICY_get_policy_language(PROXYPOLICY *policy) {
        return policy->policy_language;
}

/******************************************************************************
Function:   verify_PROXYCERTINFO_get_path_length
Description:
            Get the proxy path length value
******************************************************************************/
long
verify_PROXYCERTINFO_get_path_length(PROXYCERTINFO *cert_info) {
    if(cert_info && cert_info->path_length) {
        return ASN1_INTEGER_get(cert_info->path_length);
    } else {
        return -1;
    }
}

/******************************************************************************
Function:   lcmaps_type_of_proxy
Description:
            This function detects the type of certificates
Parameters:
    certificate
Returns:
          NONE
          CA
          EEC
          GT2_PROXY
          RFC_PROXY
          GT2_LIMITED_PROXY
          RFC_LIMITED_PROXY
          GT3_PROXY
          GT3_LIMITED_PROXY

******************************************************************************/
lcmaps_proxy_type_t lcmaps_type_of_proxy(X509 * cert) {
    lcmaps_proxy_type_t pt = NONE;
    char * cert_subjectdn = NULL;
    char * cert_issuerdn = NULL;
    char * tail_str = NULL;
    int len_subject_dn = 0;
    int len_issuer_dn = 0;

    X509_NAME *                         subject = NULL;
    X509_NAME *                         name = NULL;
    X509_NAME_ENTRY *                   ne = NULL;
    X509_NAME_ENTRY *                   new_ne = NULL;
    X509_EXTENSION *                    pci_ext = NULL;
    ASN1_STRING *                       data = NULL;
    PROXYCERTINFO *                     pci = NULL;
    PROXYPOLICY *                       policy = NULL;
    ASN1_OBJECT *                       policy_lang = NULL;
    int                                 policy_nid;
    int                                 index = -1;
    int                                 critical;
    BASIC_CONSTRAINTS *                 x509v3_bc = NULL;

    /* Is it a CA certificate */
    if (grid_x509IsCA(cert)) {
        /* Log (L_DEBUG, "%s: Detected CA certificate", __func__); */
        pt = CA;
        goto finalize;
    }

    int  i;
    char s[80];
    X509_EXTENSION *ex;

    /* Check by OID */
    for (i = 0; i < X509_get_ext_count(cert); ++i) {
        ex = X509_get_ext(cert, i);

        if (X509_EXTENSION_get_object(ex)) {
            OBJ_obj2txt(s, sizeof(s), X509_EXTENSION_get_object(ex), 1);

            if (strcmp(s, OID_RFC_PROXY) == 0) {
                pt = RFC_PROXY;

                /* Find index of OID_RFC_PROXY */
                if((index = X509_get_ext_by_NID(cert, OBJ_txt2nid(OID_RFC_PROXY), -1)) != -1  &&
                    (pci_ext = X509_get_ext(cert,index)) && X509_EXTENSION_get_critical(pci_ext)) {
                    if((pci = X509V3_EXT_d2i(pci_ext)) == NULL) {
                        Error(__func__, "Can't convert DER encoded PROXYCERTINFO extension to internal form");
                        goto failure;
                    }

                    /* Pull a certificate policy from the extension */
                    if((policy = verify_PROXYCERTINFO_get_policy(pci)) == NULL) {
                        Error(__func__, "Can't get policy from PROXYCERTINFO extension");
                        goto failure;
                    }

                    /* Get policy language */
                    if((policy_lang = verify_PROXYPOLICY_get_policy_language(policy)) == NULL) {
                        Error(__func__, "Can't get policy language from PROXYCERTINFO extension");
                        goto failure;
                    }

                    /* Lang to NID, lang's NID holds RFC Proxy type, like limited. Impersonation is the default */
                    policy_nid = OBJ_obj2nid(policy_lang);

                    if(policy_nid == OBJ_txt2nid(IMPERSONATION_PROXY_OID)) {
                        pt = RFC_PROXY;
                    } else if(policy_nid == OBJ_txt2nid(INDEPENDENT_PROXY_OID)) {
                        pt = RFC_PROXY;
                    } else if(policy_nid == OBJ_txt2nid(LIMITED_PROXY_OID)) {
                        pt = RFC_LIMITED_PROXY;
                    } else {
                        /* RFC_RESTRICTED_PROXY */
                        pt = RFC_PROXY;
                    }

                    if(X509_get_ext_by_NID(cert, OBJ_txt2nid(OID_RFC_PROXY), index) != -1) {
                        Error(__func__, "Found more than one PCI extension");
                        goto failure;
                    }
                }
                goto finalize;
            }
            if (strcmp(s, OID_GLOBUS_PROXY_V3) == 0) {
                pt = GT3_PROXY;

                /* Find index of OID_GT3_PROXY - Don't make it search for critical extentions... VOMS doesn't set those. */
                if((index = X509_get_ext_by_NID(cert, OBJ_txt2nid(OID_GLOBUS_PROXY_V3), -1)) != -1  &&
                    (pci_ext = X509_get_ext(cert,index))) {
                    if((pci = X509V3_EXT_d2i(pci_ext)) == NULL) {
                        Error(__func__, "Can't convert DER encoded PROXYCERTINFO extension to internal form");
                        goto failure;
                    }

                    /* Pull a certificate policy from the extension */
                    if((policy = verify_PROXYCERTINFO_get_policy(pci)) == NULL) {
                        Error(__func__, "Can't get policy from PROXYCERTINFO extension");
                        goto failure;
                    }

                    /* Get policy language */
                    if((policy_lang = verify_PROXYPOLICY_get_policy_language(policy)) == NULL) {
                        Error(__func__, "Can't get policy language from PROXYCERTINFO extension");
                        goto failure;
                    }

                    /* Lang to NID, lang's NID holds RFC Proxy type, like limited. Impersonation is the default */
                    policy_nid = OBJ_obj2nid(policy_lang);

                    if(policy_nid == OBJ_txt2nid(IMPERSONATION_PROXY_OID)) {
                        pt = GT3_PROXY;
                    } else if(policy_nid == OBJ_txt2nid(INDEPENDENT_PROXY_OID)) {
                        pt = GT3_PROXY;
                    } else if(policy_nid == OBJ_txt2nid(LIMITED_PROXY_OID)) {
                        pt = GT3_LIMITED_PROXY;
                    } else {
                        /* GT3_RESTRICTED_PROXY */
                        pt = GT3_PROXY;
                    }

                    if(X509_get_ext_by_NID(cert, OBJ_txt2nid(OID_GLOBUS_PROXY_V3), index) != -1) {
                        Error(__func__, "Found more than one PCI extension");
                        goto failure;
                    }
                }

                goto finalize;
            }
            if (strcmp(s, OID_GLOBUS_PROXY_V2) == 0) {
                pt = GT3_PROXY;

                /* Check for GT2_PROXY tail */
                if (cert_subjectdn
                    && (strlen(cert_subjectdn) > strlen("/cn=proxy"))
                    && (tail_str = &cert_subjectdn[strlen(cert_subjectdn) - strlen("/cn=proxy")])
                    && (strcasecmp(tail_str, "/cn=proxy") == 0)
                   ) {
                    /* Log (L_DEBUG, "%s: Detected GT2 proxy certificate", __func__); */

                    pt = GT2_PROXY;
                    goto finalize;
                }

                /* Check for GT2_LIMITED_PROXY tail */
                if (cert_subjectdn
                    && (strlen(cert_subjectdn) > strlen("/cn=limited proxy"))
                    && (tail_str = &cert_subjectdn[strlen(cert_subjectdn) - strlen("/cn=limited proxy")])
                    && (strcasecmp(tail_str, "/cn=limited proxy") == 0)
                   ) {
                    /* Log (L_DEBUG, "%s: Detected GT2 limited proxy certificate", __func__); */

                    pt = GT2_LIMITED_PROXY;
                    goto finalize;
                }

                Error(__func__, "Detected the Globus GT2 OID in the certificate, "
                                "but seems to have a malformed Subject DN: \"%s\"", cert_subjectdn);
                goto failure;
            }
        }
    }

    /* Options left: GT2_PROXY, GT2_LIMITED_PROXY, EEC */
    /* Extract Subject DN - Needs free */
    if (!(cert_subjectdn = X509_NAME_oneline (X509_get_subject_name (cert), NULL, 0))) {
        Error (__func__, "Error in %s: Couldn't get the subject DN from the certificate.\n", __func__);
        goto failure;
    }
    if (!(cert_issuerdn = X509_NAME_oneline (X509_get_issuer_name (cert), NULL, 0))) {
        Error (__func__, "Error in %s: Couldn't get the issuer DN from the certificate.\n", __func__);
        goto failure;
    }

    /* Check length of the DNs */
    len_subject_dn = strlen(cert_subjectdn);
    len_issuer_dn  = strlen(cert_issuerdn);


    /* Lower case the Subject DN */
    /* for (j = 0; j < strlen(cert_subjectdn); j++) { cert_subjectdn[j] = tolower(cert_subjectdn[j]); } */

    /* Proxies always has a longer subject_dn then a issuer_dn and
     * the issuer_dn is a substring of the subject_dn
     */
    if (   (len_issuer_dn < len_subject_dn)
        && (strncmp(cert_subjectdn, cert_issuerdn, len_issuer_dn) == 0)
       ) {
        /* Check for GT2_PROXY tail */
        if (cert_subjectdn
            && (strlen(cert_subjectdn) > strlen("/cn=proxy"))
            && (tail_str = &cert_subjectdn[strlen(cert_subjectdn) - strlen("/cn=proxy")])
            && (strcasecmp(tail_str, "/cn=proxy") == 0)
           ) {
            /* Log (L_DEBUG, "%s: Detected GT2 proxy certificate", __func__); */
            pt = GT2_PROXY;
            goto finalize;
        }

        /* Check for GT2_LIMITED_PROXY tail */
        if (cert_subjectdn
            && (strlen(cert_subjectdn) > strlen("/cn=limited proxy"))
            && (tail_str = &cert_subjectdn[strlen(cert_subjectdn) - strlen("/cn=limited proxy")])
            && (strcasecmp(tail_str, "/cn=limited proxy") == 0)
           ) {
            /* Log (L_DEBUG, "%s: Detected GT2 limited proxy certificate", __func__); */
            pt = GT2_LIMITED_PROXY;
            goto finalize;
        }

        /* Check for RFC_PROXY, without the need for OpenSSL proxy support */
        /* Method: Check if the subject_dn is long enough, grab its tail and
         * snip of the 10 characters. Then check if the 10 characters are
         * numbers. */
        if (cert_subjectdn
            && (strlen(cert_subjectdn) > strlen("/cn=0123456789"))
            && (tail_str = strrchr(cert_subjectdn, '='))
            && (tail_str = &tail_str[1])
            && (strtol(tail_str, NULL, 10))
            && (errno != ERANGE)
           ) {
            /* Log (L_DEBUG, "%s: Detected RFC proxy certificate", __func__); */
            pt = RFC_PROXY;
            goto finalize;
        }

        /* Don't know the type of proxy, could be an RFC proxy with
         * improper/incomplete implementation in the active OpenSSL version or
         * a mistake in the client software */
        goto failure;
    }


    /* I have no idea what else it is, so I conclude that it's an EEC */
    pt = EEC;
    goto finalize;

failure:
    /* On failure, or non-distinct selections of the certificate, indicate NONE */
    pt = NONE;
finalize:
    if (cert_subjectdn)
        free(cert_subjectdn);
    if (cert_issuerdn)
        free(cert_issuerdn);

    return pt;
}


/******************************************************************************
Function:   verify_generate_proxy_expectation_error_message
Description:
            Prints the expected types of certificates in human readable form.
Parameters:
    lcmaps_proxy_type_t expected_proxy_type : bitfield of types of certificates that we expected.
    lcmaps_proxy_type_t got_type            : the (offending) certificate
Returns:
    char *message, MUST be free'd
******************************************************************************/
char *
verify_generate_proxy_expectation_error_message(lcmaps_proxy_type_t got_type, lcmaps_proxy_type_t expected_proxy_type) {
    char *msg = NULL;
    char *expected_sub_msg = NULL;
    int  len = 0;
    int  expectations_cnt = 0;
    int  expectations_len = 0;
    const char *expectation_delimiter = " or ";
    const char *template = NULL;

    if (expected_proxy_type & CA) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(CA));
        } else {                     expectations_len += strlen(verify_certificate_type_str(CA)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & EEC) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(EEC));
        } else {                     expectations_len += strlen(verify_certificate_type_str(EEC)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT2_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(GT2_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(GT2_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT3_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(GT3_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(GT3_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & RFC_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(RFC_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(RFC_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT2_LIMITED_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(GT2_LIMITED_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(GT2_LIMITED_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT3_LIMITED_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(GT3_LIMITED_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(GT3_LIMITED_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }
    if (expected_proxy_type & RFC_LIMITED_PROXY) {
        if (expectations_cnt == 0) { expectations_len += strlen(verify_certificate_type_str(RFC_LIMITED_PROXY));
        } else {                     expectations_len += strlen(verify_certificate_type_str(RFC_LIMITED_PROXY)) + strlen(expectation_delimiter); }
        expectations_cnt++;
    }

    /* Creating the buffer for the humanly readable exected list of proxy types */
    expected_sub_msg = malloc(expectations_len + 1);
    if (expected_sub_msg == NULL) {
        return NULL;
    }
    memset(expected_sub_msg, 0, expectations_len + 1);
    expectations_cnt = 0;

    if (expected_proxy_type & CA) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(CA));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(CA));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & EEC) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(EEC));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(EEC));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT2_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(GT2_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(GT2_PROXY));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT3_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(GT3_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(GT3_PROXY));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & RFC_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(RFC_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(RFC_PROXY));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT2_LIMITED_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(GT2_LIMITED_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(GT2_LIMITED_PROXY));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & GT3_LIMITED_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(GT3_LIMITED_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(GT3_LIMITED_PROXY));
        }
        expectations_cnt++;
    }
    if (expected_proxy_type & RFC_LIMITED_PROXY) {
        if (expectations_cnt == 0) { strcpy(expected_sub_msg, verify_certificate_type_str(RFC_LIMITED_PROXY));
        } else {                     strcat(expected_sub_msg, expectation_delimiter);
                                     strcat(expected_sub_msg, verify_certificate_type_str(RFC_LIMITED_PROXY));
        }
        expectations_cnt++;
    }


    template = "Certificate chain not build in the right order. Got %s certificate, but expected a(n) %s certificate.";
    len = snprintf(NULL, 0, template,
                            verify_certificate_type_str(got_type),
                            expected_sub_msg);
    msg = malloc(len + 1);
    if (msg == NULL)
        goto finalize;

    snprintf(msg, len + 1, template,
                           verify_certificate_type_str(got_type),
                           expected_sub_msg);

finalize:
    free(expected_sub_msg);
    return msg;
}


/******************************************************************************
Function:   grid_verifyPathLenConstraints
Description:
            This function will check the certificate chain on CA based (RFC5280)
            and RFC3820 Proxy based Path Length Constraints.
Parameters:
    chain of certificates
Returns:
    0       : Not ok, failure in the verification or the verification failed
    1       : Ok, verification has succeeded and positive
******************************************************************************/
int grid_verifyPathLenConstraints (STACK_OF(X509) * chain)
{
    char *oper = "grid_verifyPathLenConstraints";
    X509 *cert = NULL;
    int i, j, depth, c;
    lcmaps_proxy_type_t curr_cert_type = NONE, expe_cert_type = CA|EEC|RFC_PROXY|GT2_PROXY;
    int found_EEC = 0;
    int found_limited_proxy = 0;
    char *cert_subjectdn = NULL;
    char *error_msg = NULL;

    int ca_path_len_countdown    = -1;
    int proxy_path_len_countdown = -1;

    /* No chain, no game */
    if (!chain) {
        Error( oper, "No certificate chain detected.");
        goto failure;
    }

    /* Go through the list, from the CA(s) down through the EEC to the final delegation */
    depth = sk_X509_num (chain);
    for (i=depth-1; i >= 0; --i) {
        if ((cert = sk_X509_value(chain, i))) {
            /* Init to None, indicating not to have identified it yet */
            curr_cert_type = NONE;

            /* Extract Subject DN - Needs free */
            if (!(cert_subjectdn = X509_NAME_oneline (X509_get_subject_name (cert), NULL, 0))) {
                Error (oper, "Couldn't get the subject DN from the certificate at depth %d\n", depth);
                goto failure;
            }

            /* Log (L_DEBUG, "\tCert here is: %s\n", cert_subjectdn); */
            curr_cert_type = lcmaps_type_of_proxy(cert);
            if (curr_cert_type == NONE) {
                Error (oper, "Couldn't classify certificate at depth %d with subject DN \"%s\"\n",
                             depth, cert_subjectdn);
                goto failure;
            }

            /* Mark that we've found an EEC - When we see it again, it's a failure */
            if (curr_cert_type == EEC && found_EEC == 0) {
                found_EEC = 1;
            } else if (curr_cert_type == EEC && found_EEC == 1) {
                Error (oper, "Found another EEC classified certificate in the same chain at depth %d with subject DN \"%s\"\n",
                             depth, cert_subjectdn);
                goto failure;
            }


#if 0
                /* NOTE: This is for quick debugging only */
                error_msg = verify_generate_proxy_expectation_error_message(curr_cert_type, expe_cert_type);
                printf("%s: Build chain checker: %s. Cert at depth %d of %d with Subject DN: %s\n",
                            oper,
                            error_msg,
                            i,
                            depth,
                            cert_subjectdn);
                free(error_msg); error_msg = NULL;
#endif

            /* Expectation management */
            if (!((expe_cert_type & curr_cert_type) == curr_cert_type)) {
                /* Failed to comply with the expectations! */
#define USE_STRICT_PATH_VALIDATION
#ifdef USE_STRICT_PATH_VALIDATION
                error_msg = verify_generate_proxy_expectation_error_message(curr_cert_type, expe_cert_type);
                Error(oper, "Certificate chain not build in the right order. %s. Cert at depth %d of %d with Subject DN: %s\n",
                            error_msg,
                            i,
                            depth,
                            cert_subjectdn);
                free(error_msg); error_msg = NULL;
                goto failure;
#else
                error_msg = verify_generate_proxy_expectation_error_message(curr_cert_type, expe_cert_type);
                Log(L_INFO, "%s: Certificate chain not build in the right order. %s. Cert at depth %d of %d with Subject DN: %s\n",
                            oper,
                            error_msg,
                            i,
                            depth,
                            cert_subjectdn);
                free(error_msg); error_msg = NULL;
                goto continue_after_warning;
#endif
            }
#ifndef USE_STRICT_PATH_VALIDATION
continue_after_warning:
#endif

            if (curr_cert_type == CA) {
                /* Expected next certificate type is: CA or EEC certificate */
                expe_cert_type = CA|EEC;
                Log (L_DEBUG, "Current cert is a CA: %s\n", cert_subjectdn);

                /* Exceeded CA Path Length ? */
                if (ca_path_len_countdown == 0) {
                    Error(oper, "CA Path Length Constraint exceeded on depth %d for certificate \"%s\". No CA certifcates were expected at this stage.\n", i, cert_subjectdn);
                    goto failure;
                }

                /* Store pathlen, override when small, otherwise keep the smallest */
                if (cert->ex_pathlen != -1) {
                    /* Update when ca_path_len_countdown is the initial value
                     * or when the PathLenConstraint is smaller then the
                     * remembered ca_path_len_countdown */
                    if ((ca_path_len_countdown == -1) || (cert->ex_pathlen < ca_path_len_countdown)) {
                        ca_path_len_countdown = cert->ex_pathlen;
                    } else {
                        /* If a path length was already issuesd, lower ca_path_len_countdown */
                        if (ca_path_len_countdown != -1)
                            ca_path_len_countdown--;
                    }
                } else {
                    /* If a path length was already issuesd, lower ca_path_len_countdown */
                    if (ca_path_len_countdown != -1)
                        ca_path_len_countdown--;
                }

            } else if (curr_cert_type == EEC) {
                /* Expected next certificate type is: GT2_PROXY, GT3_PROXY, RFC_PROXY or a Limited proxy of these flavors certificate */
                expe_cert_type = GT2_PROXY|GT3_PROXY|RFC_PROXY|GT2_LIMITED_PROXY|GT3_LIMITED_PROXY|RFC_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a EEC: %s\n", cert_subjectdn);

            } else if (curr_cert_type == GT2_PROXY) {
                /* Expected next certificate type is: GT2_PROXY certificate */
                expe_cert_type = GT2_PROXY|GT2_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a GT2 Proxy: %s\n", cert_subjectdn);

            } else if (curr_cert_type == GT2_LIMITED_PROXY) {
                /* Expected next certificate type is: GT2_LIMITED_PROXY certificate */
                expe_cert_type = GT2_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a GT2 Limited Proxy: %s\n", cert_subjectdn);

            } else if (curr_cert_type == GT3_PROXY) {
                /* Expected next certificate type is: GT3_PROXY certificate */
                expe_cert_type = GT3_PROXY|GT3_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a GT3 Proxy: %s\n", cert_subjectdn);
            } else if (curr_cert_type == GT3_LIMITED_PROXY) {
                /* Expected next certificate type is: GT3_LIMITED_PROXY certificate */
                expe_cert_type = GT3_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a GT3 Limited Proxy: %s\n", cert_subjectdn);

            } else if (curr_cert_type == RFC_PROXY) {
                /* Expected next certificate type is: RFC_PROXY certificate */
                expe_cert_type = RFC_PROXY|RFC_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a RFC Proxy: %s\n", cert_subjectdn);

                /* Exceeded CA Path Length ? */
                if (proxy_path_len_countdown == 0) {
                    Error(oper, "Proxy Path Length Constraint exceeded on depth %d of %d for certificate \"%s\". No Proxy certifcates were expected at this stage.\n", i, depth, cert_subjectdn);
                    goto failure;
                }

                /* Store pathlen, override when small, otherwise keep the smallest */
                if (cert->ex_pcpathlen != -1) {
                    /* Update when proxy_path_len_countdown is the initial value
                     * or when the PathLenConstraint is smaller then the
                     * remembered proxy_path_len_countdown */

                    if ((proxy_path_len_countdown == -1) || (cert->ex_pcpathlen < proxy_path_len_countdown)) {
                        proxy_path_len_countdown = cert->ex_pcpathlen;
                        Log (L_DEBUG, "Cert here is: %s -> Setting proxy path len constraint to: %d\n", cert_subjectdn, cert->ex_pcpathlen);
                    } else {
                        /* If a path length was already issuesd, lower ca_path_len_countdown */
                        if (proxy_path_len_countdown != -1)
                            proxy_path_len_countdown--;

                        Log (L_DEBUG, "Cert here is: %s -> Countdown is at %d\n", cert_subjectdn, proxy_path_len_countdown);
                    }
                } else {
                    /* If a path length was already issued, lower ca_path_len_countdown */
                    if (proxy_path_len_countdown != -1) {
                        proxy_path_len_countdown--;
                        Log (L_DEBUG, "Cert here is: %s -> Countdown is at %d\n", cert_subjectdn, proxy_path_len_countdown);
                    }

                }
            } else if (curr_cert_type == RFC_LIMITED_PROXY) {
                /* Expected next certificate type is: RFC_LIMITED_PROXY certificate */
                expe_cert_type = RFC_LIMITED_PROXY;
                Log (L_DEBUG, "Current cert is a RFC Limited Proxy: %s\n", cert_subjectdn);

                /* Exceeded CA Path Length ? */
                if (proxy_path_len_countdown == 0) {
                    Error(oper, "Proxy Path Length Constraint exceeded on depth %d of %d for certificate \"%s\". No Proxy certifcates were expected at this stage.\n", i, depth, cert_subjectdn);
                    goto failure;
                }

                /* Store pathlen, override when small, otherwise keep the smallest */
                if (cert->ex_pcpathlen != -1) {
                    /* Update when proxy_path_len_countdown is the initial value
                     * or when the PathLenConstraint is smaller then the
                     * remembered proxy_path_len_countdown */

                    if ((proxy_path_len_countdown == -1) || (cert->ex_pcpathlen < proxy_path_len_countdown)) {
                        proxy_path_len_countdown = cert->ex_pcpathlen;
                        Log (L_DEBUG, "Cert here is: %s -> Setting proxy path len constraint to: %d\n", cert_subjectdn, cert->ex_pcpathlen);
                    } else {
                        /* If a path length was already issuesd, lower ca_path_len_countdown */
                        if (proxy_path_len_countdown != -1)
                            proxy_path_len_countdown--;

                        Log (L_DEBUG, "Cert here is: %s -> Countdown is at %d\n", cert_subjectdn, proxy_path_len_countdown);
                    }
                } else {
                    /* If a path length was already issued, lower ca_path_len_countdown */
                    if (proxy_path_len_countdown != -1) {
                        proxy_path_len_countdown--;
                        Log (L_DEBUG, "Cert here is: %s -> Countdown is at %d\n", cert_subjectdn, proxy_path_len_countdown);
                    }

                }
            }

            /* Free memory during each cycle */
            if (cert_subjectdn) {
                free(cert_subjectdn);
                cert_subjectdn = NULL;
            }
        }
    }

success:
    /* Return an OK (thumbs up) in the grid_X509_verify_callback() */
    if (cert_subjectdn) {
        free(cert_subjectdn);
        cert_subjectdn = NULL;
    }
    return X509_V_OK;

failure:
    if (cert_subjectdn) {
        free(cert_subjectdn);
        cert_subjectdn = NULL;
    }
    return X509_V_ERR_CERT_REJECTED;
}



/******************************************************************************
Function:   verify_callback
Description:
    Callback function for OpenSSL to put the errors
Parameters:
    ok, X509_STORE_CTX
Returns:
******************************************************************************/
int grid_X509_verify_callback(int ok, X509_STORE_CTX *ctx)
{
    unsigned long   errnum   = X509_STORE_CTX_get_error(ctx);
    int             errdepth = X509_STORE_CTX_get_error_depth(ctx);
    STACK_OF(X509) *certstack;

    /* When not ok... */
    if (ok != 1)
    {
        Log( L_DEBUG, "verification callback indicated \'ok = %d\' and error number: %d\n", ok, errnum);

        if (errnum == X509_V_ERR_INVALID_CA) ok=1;
        if (errnum == X509_V_ERR_UNABLE_TO_GET_CRL) ok=1;
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
        /* I don't want to do this, really, but I have yet to figure out
           how to get openssl 0.9.8 to accept old-style proxy certificates...
        */
        if (errnum == X509_V_ERR_INVALID_PURPOSE) ok=1;
#endif
        if (errnum == X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION)
        {
            errnum = grid_X509_knownCriticalExts(X509_STORE_CTX_get_current_cert(ctx));
            ctx->error = errnum;
            if (errnum == X509_V_OK) ok=1;
        }

        /* Path length exceeded for the CA (should never happen in OpenSSL - famous last words) */
        if (ctx->error == X509_V_ERR_PATH_LENGTH_EXCEEDED)
        {
            Log( L_DEBUG, "Shallow Error X509_V_ERR_PATH_LENGTH_EXCEEDED: Running alternative RFC5280 and RFC3820 compliance tests.\n");
            if ((ctx->error = grid_verifyPathLenConstraints(X509_STORE_CTX_get_chain(ctx))) != X509_V_OK)
                goto failure;
        }

        /* Path length exceeded for the Proxy! -> Override and continue */
        /* This is NOT about X509_V_ERR_PATH_LENGTH_EXCEEDED */
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
        if (ctx->error == X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED)
        {
            Log( L_DEBUG, "Shallow Error X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED: Running alternative RFC5280 and RFC3820 compliance tests.\n");
            if ((ctx->error = grid_verifyPathLenConstraints(X509_STORE_CTX_get_chain(ctx))) != X509_V_OK)
                goto failure;
        }
#endif
    }

    /*
     * We've now got the last certificate - the identity being used for
     * this connection. At this point we check the whole chain for valid
     * CAs or, failing that, GSI-proxy validity using grid_verifyProxy
     */
    if ((errdepth == 0) && (ok == 1)) {
        certstack = (STACK_OF(X509) *) X509_STORE_CTX_get_chain( ctx );

        if ((ctx->error = grid_verifyProxy(certstack)) != X509_V_OK) {
            goto failure;
        } else {
            /* I'm in the OK state - Let's see if the chain is really well formed */
            if ((ctx->error = grid_verifyPathLenConstraints(X509_STORE_CTX_get_chain(ctx))) != X509_V_OK)
                goto failure;
        }
    }

    /* Ready and probably ok */
    return ok;

failure:
    /* Indicate failure */
    ok = 0;
    Log( L_INFO, "grid_verify_callback: error code: %d, message: \"%s\"", ctx->error, X509_verify_cert_error_string (ctx->error));

    return ok;
}


/******************************************************************************
Function:   grid_verifyCert
Description:
    Validates a certificate with the CRL list and the signing CA
******************************************************************************/
unsigned long grid_verifyCert( char * CA_DIR, STACK_OF(X509) *certstack )
{
    char           *oper = "Verifying certificate chain";

    X509_STORE     *store      = NULL;
    X509_LOOKUP    *lookup     = NULL;
    X509_STORE_CTX *verify_ctx = NULL;
    X509           *cert       = NULL;
    char *          cert_DN    = NULL;
    char *          issuer_DN  = NULL;
    int             i = 0;
    int             depth = 0;

    Log( L_DEBUG, "--- Welcome to the grid_verifyCert function ---");

    /* Initials must be good */
    if ( CA_DIR == NULL )
    {
        Error( oper, "No CA certificate directory specified." );
        return X509_V_ERR_APPLICATION_VERIFICATION;
    }
    if ( certstack == NULL )
    {
        Error( oper, "Certificate stack is empty." );
        return X509_V_ERR_APPLICATION_VERIFICATION;
    }

    Log( L_INFO, "Using CA Directory: %s", CA_DIR);

    Log( L_DEBUG, "X509_STORE_new");
    if (!(store = X509_STORE_new()))
    {
       Error( oper, "Could not create a X509 STORE." );
       return ERR_get_error();
    }

    Log( L_DEBUG, "X509_STORE_set_verify_cb_func");
    X509_STORE_set_verify_cb_func (store, grid_X509_verify_callback);
#if 0
/* NOTE: the following two calls are defined in openssl ./crypto/x509/x509_d2.c
 * (at least for openssl-1.0.0f)
 * Also note that the X509_STORE_set_default_paths adds X509_LOOKUP_file (among
 * others) which is effectively the file pointed at by the env variable
 * SSL_CERT_FILE */
    /* Executing the lookups to the CA and CRL files */
    Log( L_DEBUG, "X509_STORE_load_locations");
    if (X509_STORE_load_locations (store, NULL, CA_DIR) != 1)
    {
        Error( oper, "Could not load the CA directory.");
        return ERR_get_error();
    }

    Log( L_DEBUG, "X509_STORE_set_default_paths");
    if (X509_STORE_set_default_paths(store) != 1)
    {
        Error( oper, "Could not load the system wide CA certificates.");
        return ERR_get_error();
    }
#endif

    /* Add CA_DIR, note: the next lines will effectively call dir_ctrl() from
     * ./crypto/x509/by_dir.c with type X509_FILETYPE_PEM */
    Log( L_DEBUG, "X509_STORE_add_lookup");
    if (!(lookup = X509_STORE_add_lookup (store, X509_LOOKUP_hash_dir())))
    {
        Error( oper, "Could not create X509_LOOKUP object.");
        return ERR_get_error();
    }

    Log( L_DEBUG, "X509_LOOKUP_add_dir");
    i=X509_LOOKUP_add_dir (lookup, CA_DIR, X509_FILETYPE_PEM);
    if (!i)
    {
        Error( oper, "Could not add CA_DIR.");
        return ERR_get_error();
    }

    Log( L_DEBUG, "X509_STORE_set_flags");

    store->check_issued = grid_X509_check_issued_wrapper;
#if OPENSSL_VERSION_NUMBER < 0x00908000L
    X509_STORE_set_flags( store, X509_V_FLAG_CRL_CHECK |
                                 X509_V_FLAG_CRL_CHECK_ALL );
#else
    X509_STORE_set_flags( store, X509_V_FLAG_CRL_CHECK |
                                 X509_V_FLAG_CRL_CHECK_ALL |
                                 X509_V_FLAG_ALLOW_PROXY_CERTS );
#endif

    Log( L_DEBUG, "X509_STORE_CTX_new");
    /* Creating a verification context and init it */
    if (!(verify_ctx = X509_STORE_CTX_new()))
    {
        Error( oper, "Could not create a X509 STORE CTX (context).");
        return ERR_get_error();
    }

    if (certstack)
    {
        depth = sk_X509_num (certstack);
        Log( L_DEBUG, "Depth of certstore %d", depth);
    }
    else
    {
        Error( oper, "No certificate stack detected.");
        return 1;
    }


    for (i=depth-1; i >= 0; --i)
    {
        if ((cert = sk_X509_value(certstack, i)))
        {
            cert_DN   = X509_NAME_oneline(X509_get_subject_name(cert),NULL,0);
            issuer_DN = X509_NAME_oneline(X509_get_issuer_name(cert),NULL,0);

            Log( L_DEBUG, "DN[%d]:        %s", i, cert_DN );
            Log( L_DEBUG, "Issuer DN[%d]: %s", i, issuer_DN);

            /* The X509_NAME_oneline() result without providing a buffer must be free'd */
            free(cert_DN);
            free(issuer_DN);

            if (grid_x509IsCA(cert))
            {
                Log( L_DEBUG, "This certificate is a CA certificate" );
                Log( L_DEBUG, "continue search for user certificate..." );
            }
            else
            {
                break;
            }
        }
    }

    cert = sk_X509_value( certstack, 0 );
    cert_DN   = X509_NAME_oneline( X509_get_subject_name( cert ) , NULL, 0 );
    issuer_DN = X509_NAME_oneline( X509_get_issuer_name( cert )  , NULL, 0 );

    Log( L_INFO, "Certificate to verify:" );
    Log( L_INFO, "  DN:        %s", cert_DN );
    Log( L_INFO, "  Issuer DN: %s", issuer_DN );

    /* The X509_NAME_oneline() result without providing a buffer must be free'd */
    free(cert_DN);
    free(issuer_DN);


    Log( L_DEBUG, "X509_STORE_CTX_init" );
    if ( X509_STORE_CTX_init( verify_ctx, store, cert, certstack) != 1 )
    {
        Error( oper, "Could not initialize verification context.");
        return ERR_get_error();
    }

    X509_STORE_CTX_set_purpose( verify_ctx, X509_PURPOSE_SSL_CLIENT );
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
    cert->ex_flags |= EXFLAG_PROXY;
#endif


    Log( L_DEBUG, "The certificate chain has a depth of %d. For verification the depth is extended to fit the chain and (subordinate) CAs to %d\n", depth, depth + VERIFICATION_CHAIN_DEPTH_EXTENSION);

    /* Alter verification depth to fit the certificate chain, sub-CAs and root CA */
    X509_STORE_CTX_set_depth (verify_ctx, depth + VERIFICATION_CHAIN_DEPTH_EXTENSION);


    Log( L_DEBUG, "X509_verify");

    if ( (X509_verify_cert( verify_ctx ) ) != 1 )
    {
        return verify_ctx->error;
    }
    else
    {
        Log( L_INFO, "The verification of the certicate has succeeded.");
    }

    if ( verify_ctx ) X509_STORE_CTX_free( verify_ctx );
    if ( store )      X509_STORE_free( store );

    return X509_V_OK;
}


/******************************************************************************
Function:   verifyPrivateyKey
Description:
    Tries to verify a privatekey with the first cert in the certstack
******************************************************************************/
unsigned long grid_verifyPrivateKey( STACK_OF(X509) *certstack, EVP_PKEY *pkey )
{
    X509 *cert = NULL;

    Log( L_DEBUG, "--- Welcome to the grid_verifyPrivateKey function ---");

    if ( pkey == NULL )
    {
        Log( L_WARN, "No private key available." );
        return X509_V_OK;
    }

    /* Check for X509 certificate and point to it with 'cert' */
    if ((cert = sk_X509_value(certstack, 0)) != NULL)
    {
        Log( L_DEBUG, "X509_check_private_key" );
        if ( X509_check_private_key( cert, pkey ) != 1 )
        {
            return ERR_get_error();
        }
    }

    return X509_V_OK;
}

