#include <stdio.h>
#include <stringf.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <Exception.h>
#include <popt.h>
#include <cstring>

#define VERSION "1.0"

/*
 * Print to stdout/stderr and print the signals it receives: used to test the launchtool.
 *
 * Called with -l will run in a loop
 *
 * Exits after three SIGINTs
 */

using namespace std;
using namespace stringf;

FILE* log = 0;

static void out(const string& s) throw ()
{
	fprintf(stdout, "out: %.*s\n", PFSTR(s));
	if (log)
	{
		fprintf(log, "out: %.*s\n", PFSTR(s));
		fflush(log);
	}
}

static void err(const string& s) throw ()
{
	fprintf(stderr, "err: %.*s\n", PFSTR(s));
	if (log)
	{
		fprintf(log, "err: %.*s\n", PFSTR(s));
		fflush(log);
	}
}

static void sig(int signum)
{
	static int term_count = 0;

	err("Received signal " + fmt(signum) + " (" + sys_siglist[signum] + ")");

	if (signum == SIGINT)
	{
		term_count++;
		if (term_count == 3)
			_exit(0);
	}
}

// Setup signal masks and handlers
void setup_signals()
{
	struct sigaction sa;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = 0;

	sa.sa_handler = sig;
	for (int i = 0; i < 64; i++)
		sigaction(i, &sa, 0);
}


int main(int argc, const char* argv[])
{
	// Special actions
	int op_version = 0;

	// Operating parameters
	int op_status = -1;
	int op_loop = -1;
	
	char* help =
			"Test launchtool printing something on stdout and stderr,\n"
			"logging what it prints and printing the signals it receives.\n";
	struct poptOption emptyTable[] = { POPT_TABLEEND };
	struct poptOption optionsTable[] = {
		// Special actions
		{ "version", 'V', POPT_ARG_NONE, &op_version, 0, "print version and exit", 0 },
		{ "status", 's', POPT_ARG_INT, &op_status, 0, "exit with the given status", "value" },
		{ "loop", 'l', POPT_ARG_NONE, &op_loop, 0, "never exit: repeat the prints at 1 second intervals", 0 },
		POPT_AUTOHELP
		{ NULL, 0, POPT_ARG_INCLUDE_TABLE, emptyTable, 0, help, 0 },
		POPT_TABLEEND
	};
	poptContext optCon = poptGetContext(NULL, argc, argv, optionsTable, 0);
	poptSetOtherOptionHelp(optCon, "[options]");

	set_unexpected(DefaultUnexpected);

	if (argc < 1)
	{
		poptPrintHelp(optCon, stderr, 0);
		return 1;
	}

	// Process commandline
	int res = poptGetNextOpt(optCon);
	if (res != -1)
	{
		fprintf(stderr, "%s: %s\n\n",
			poptBadOption(optCon, POPT_BADOPTION_NOALIAS),
			poptStrerror(res));
		poptPrintUsage(optCon, stderr, 0);
		return 1;
	}

	if (op_version)
	{
		printf("%s ver." VERSION "\n", argv[0]);
		return 0;
	}

	int exitStatus = op_status == -1 ? 0 : op_status;
	bool loop = op_loop == -1 ? false : op_loop;

	try {
		log = fopen("test.log", "a");
		if (!log)
			throw FileException(errno, "opening test.log");
		out("Starting");
		err("Starting");
		setup_signals();
		do {
			time_t now = time(0);
			char* t = ctime(&now);
			t[strlen(t) - 1] = 0;
			out(t);
			err(t);
			sleep(1);
		} while (loop);
	} catch (Exception& e) {
		err(string(e.type()) + ": " + e.desc());
	}
	out("Ending");
	err("Ending");
	if (log) fclose(log);
	return exitStatus;
}
