/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icrosssectionpipeline.h"


#include "icrosssectionviewsubject.h"
#include "ierror.h"
#include "iorthopolygonplanefilter.h"
#include "iorthoslicer.h"
#include "iorthotextureplanefilter.h"

#include <vtkStructuredPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"
#include "iviewsubjectpipelinetemplate.h"


//
// Main class
//
iCrossSectionPipeline::iCrossSectionPipeline(iCrossSectionViewSubject *s)  : iViewSubjectPipeline(s,1)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mSlicer = this->CreateFilter<iOrthoSlicer>();
	mPolygonConverter = this->CreateFilter<iOrthoPolygonPlaneFilter>();
	mTextureConverter = this->CreateFilter<iOrthoTexturePlaneFilter>();
	
	mSlicer->SetDir(mParent->GetDir());
	mSlicer->SetCurrentVar(mParent->GetVar());

	mPolygonConverter->SetInput(mSlicer->GetOutput());
	mTextureConverter->SetInput(mSlicer->GetOutput());
	
	this->UpdateContents(_All);
}


iCrossSectionPipeline::~iCrossSectionPipeline()
{
}


void iCrossSectionPipeline::ProduceOutput()
{
	vtkPolyData *output = this->GetOutput();
	vtkImageData *input = vtkImageData::SafeDownCast(this->GetInput());
	if(input == 0)
	{
		IERROR_FATAL("iCrossSectionPipeline is configured incorrectly.");
		return;
	}

	if(mSlicer->GetInput() != input)
	{
		mSlicer->SetInput(input);
	}
	
	switch(mParent->GetActualMethod())
	{
	case 0:
		{
			mPolygonConverter->Update();
			output->ShallowCopy(mPolygonConverter->GetOutput());
			break;
		}
	case 1: 
		{
			//
			//  Need to assign texture offsets first
			//
			this->SetTextureOffset(input);
			mTextureConverter->Update(); 
			output->ShallowCopy(mTextureConverter->GetOutput());
			break;
		}
	default: return;
	}
}


void iCrossSectionPipeline::SerialPostProcess(int index)
{
	if(index==0 && mParent->GetActualMethod()==1)
	{
		mTextureConverter->UpdateBoundaryConditions();
	}
}


void iCrossSectionPipeline::SetTextureOffset(vtkImageData *input)
{
	//
	//  Assign texture offsets
	//
	if(mParent->GetMethod() == 1)
	{
		double org[3], globalOrg[3], spa[3];
		int dims[3], globalDims[3];
		int Uidx, Vidx;
		int Axis = mParent->GetDir();
		iOrthoSlicer::GetUV(Axis,Uidx,Vidx);

		input->GetOrigin(org);
		input->GetSpacing(spa);
		input->GetDimensions(dims);

		vtkImageData *globalInput = vtkImageData::SafeDownCast(mGlobalInput);
		if(globalInput == 0)
		{
			IERROR_LOW("Incompatible global input.");
			return;
		}

		globalInput->GetOrigin(globalOrg);
		globalInput->GetDimensions(globalDims);

		if(dims[Axis] != globalDims[Axis])  
		{
			//
			//  Split along Axis
			//
			mTextureConverter->SetTexturePiece(mParent->GetTextureData(),0,0,globalOrg,globalDims);
		}
		else
		{
			int offx = round((org[Uidx]-globalOrg[Uidx])/spa[Uidx]);
			int offy = round((org[Vidx]-globalOrg[Vidx])/spa[Vidx]);
			mTextureConverter->SetTexturePiece(mParent->GetTextureData(),offx,offy,globalOrg,globalDims);
		}
	}
}


void iCrossSectionPipeline::UpdateMethod()
{
	switch(mParent->GetActualMethod())
	{
	case 0:
		{
			mPolygonConverter->Modified();
			break;
		}
	case 1: 
		{
			mTextureConverter->Modified(); 
			break;
		}
	default: return;
	}
	this->Modified();
}


void iCrossSectionPipeline::UpdateDir()
{ 
	mSlicer->SetDir(mParent->GetDir());
	mSlicer->SetPos(mParent->GetLocation());
	this->Modified();
}


void iCrossSectionPipeline::UpdateVar()
{ 
	mSlicer->SetCurrentVar(mParent->GetVar());
	this->Modified();
}


void iCrossSectionPipeline::UpdatePos()
{ 
	mSlicer->SetPos(mParent->GetLocation());
	this->Modified();
}


void iCrossSectionPipeline::UpdateInterpolateData()
{
	mPolygonConverter->SetInterpolation(mParent->GetInterpolateData());
	mSlicer->SetInterpolation(mParent->GetInterpolateData());
	this->Modified();
}


void iCrossSectionPipeline::UpdateSampleRate()
{
	mSlicer->SetSampleRate(mParent->GetSampleRate());
	this->Modified();
}


float iCrossSectionPipeline::GetContentsMemorySize() const
{
	float s = 0.0f;
	s += mSlicer->GetMemorySize();
	s += mPolygonConverter->GetMemorySize();
	s += mTextureConverter->GetMemorySize();
	return s;
}


void iCrossSectionPipeline::UpdateContents(int n, int)
{
	if(n==_All || n==_Dir) this->UpdateDir();
	if(n==_All || n==_InterpolateData) this->UpdateInterpolateData();
	if(n==_All || n==_Method) this->UpdateMethod();
	if(n==_All || n==_Pos) this->UpdatePos();
	if(n==_All || n==_SampleRate) this->UpdateSampleRate();
	if(n==_All || n==_Var) this->UpdateVar();
}

