#!/usr/bin/env python3
# GStreamer
# Copyright (C) 2025 Seungha Yang <seungha@centricular.com>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
# Boston, MA 02110-1301, USA.

import sys
import os
import argparse

start_header = """/*
 * This file is autogenerated by collect_ptx_headers.py
 */
#pragma once

"""

start_map = """
#define MAKE_BYTECODE(name) { G_STRINGIFY (name), g_##name }
static std::unordered_map<std::string, const char *>
"""

end_map = """};
#undef MAKE_BYTECODE
"""

def convert_ptx_to_header(ptx_file, header_file):
    with open(ptx_file, 'r', encoding='utf8') as ptx:
        ptx_content = ptx.read()

    with open(header_file, 'w', newline='\n', encoding='utf8') as header:
        header.write('#pragma once\n')
        header.write('// This file is autogenerated by collect_ptx_headers.py\n')
        header.write(f'static const char* g_{os.path.splitext(os.path.basename(ptx_file))[0]} = R"(\n')
        header.write(ptx_content)
        header.write(')";\n\n')


def main(args):
    parser = argparse.ArgumentParser(description='Read CUDA PTX from directory and make single header')
    parser.add_argument("--input", help="the precompiled CUDA PTX directory")
    parser.add_argument("--output", help="output header file location")
    parser.add_argument("--prefix", help="CUDA PTX header filename prefix")
    parser.add_argument("--name", help="Hash map variable name")

    args = parser.parse_args(args)

    ptx_files = [os.path.join(args.input, file) for file in os.listdir(args.input) if file.startswith(args.prefix) and file.endswith(".ptx") ]

    with open(args.output, 'w', newline='\n', encoding='utf8') as f:
        f.write(start_header)
        for ptx_file in ptx_files:
            header_file = os.path.splitext(ptx_file)[0] + '.h'
            convert_ptx_to_header(ptx_file, header_file)
            f.write("#include \"")
            f.write(os.path.basename(header_file))
            f.write("\"\n")
        f.write(start_map)
        f.write(args.name)
        f.write(" = {\n")
        for ptx_file in ptx_files:
            f.write("  MAKE_BYTECODE ({}),\n".format(os.path.splitext(os.path.basename(ptx_file))[0]))
        f.write(end_map)

if __name__ == "__main__":
    sys.exit(main(sys.argv[1:]))
