// DO NOT EDIT. Generated by code.google.com/p/rsc/cmd/bundle
// bundle -p main -x arm_ rsc.io/arm/armasm

/* decode.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package main

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"io"
	"strings"
)

// An instFormat describes the format of an instruction encoding.
// An instruction with 32-bit value x matches the format if x&mask == value
// and the condition matches.
// The condition matches if x>>28 == 0xF && value>>28==0xF
// or if x>>28 != 0xF and value>>28 == 0.
// If x matches the format, then the rest of the fields describe how to interpret x.
// The opBits describe bits that should be extracted from x and added to the opcode.
// For example opBits = 0x1234 means that the value
//	(2 bits at offset 1) followed by (4 bits at offset 3)
// should be added to op.
// Finally the args describe how to decode the instruction arguments.
// args is stored as a fixed-size array; if there are fewer than len(args) arguments,
// args[i] == 0 marks the end of the argument list.
type arm_instFormat struct {
	mask     uint32
	value    uint32
	priority int8
	op       arm_Op
	opBits   uint64
	args     arm_instArgs
}

type arm_instArgs [4]arm_instArg

var (
	arm_errMode    = fmt.Errorf("unsupported execution mode")
	arm_errShort   = fmt.Errorf("truncated instruction")
	arm_errUnknown = fmt.Errorf("unknown instruction")
)

var arm_decoderCover []bool

// Decode decodes the leading bytes in src as a single instruction.
func arm_Decode(src []byte, mode arm_Mode) (inst arm_Inst, err error) {
	if mode != arm_ModeARM {
		return arm_Inst{}, arm_errMode
	}
	if len(src) < 4 {
		return arm_Inst{}, arm_errShort
	}

	if arm_decoderCover == nil {
		arm_decoderCover = make([]bool, len(arm_instFormats))
	}

	x := binary.LittleEndian.Uint32(src)

	// The instFormat table contains both conditional and unconditional instructions.
	// Considering only the top 4 bits, the conditional instructions use mask=0, value=0,
	// while the unconditional instructions use mask=f, value=f.
	// Prepare a version of x with the condition cleared to 0 in conditional instructions
	// and then assume mask=f during matching.
	const condMask = 0xf0000000
	xNoCond := x
	if x&condMask != condMask {
		xNoCond &^= condMask
	}
	var priority int8
Search:
	for i := range arm_instFormats {
		f := &arm_instFormats[i]
		if xNoCond&(f.mask|condMask) != f.value || f.priority <= priority {
			continue
		}
		delta := uint32(0)
		deltaShift := uint(0)
		for opBits := f.opBits; opBits != 0; opBits >>= 16 {
			n := uint(opBits & 0xFF)
			off := uint((opBits >> 8) & 0xFF)
			delta |= (x >> off) & (1<<n - 1) << deltaShift
			deltaShift += n
		}
		op := f.op + arm_Op(delta)

		// Special case: BKPT encodes with condition but cannot have one.
		if op&^15 == arm_BKPT_EQ && op != arm_BKPT {
			continue Search
		}

		var args arm_Args
		for j, aop := range f.args {
			if aop == 0 {
				break
			}
			arg := arm_decodeArg(aop, x)
			if arg == nil { // cannot decode argument
				continue Search
			}
			args[j] = arg
		}

		arm_decoderCover[i] = true

		inst = arm_Inst{
			Op:   op,
			Args: args,
			Enc:  x,
			Len:  4,
		}
		priority = f.priority
		continue Search
	}
	if inst.Op != 0 {
		return inst, nil
	}
	return arm_Inst{}, arm_errUnknown
}

// An instArg describes the encoding of a single argument.
// In the names used for arguments, _p_ means +, _m_ means -,
// _pm_ means ± (usually keyed by the U bit).
// The _W suffix indicates a general addressing mode based on the P and W bits.
// The _offset and _postindex suffixes force the given addressing mode.
// The rest should be somewhat self-explanatory, at least given
// the decodeArg function.
type arm_instArg uint8

const (
	_ arm_instArg = iota
	arm_arg_APSR
	arm_arg_FPSCR
	arm_arg_Dn_half
	arm_arg_R1_0
	arm_arg_R1_12
	arm_arg_R2_0
	arm_arg_R2_12
	arm_arg_R_0
	arm_arg_R_12
	arm_arg_R_12_nzcv
	arm_arg_R_16
	arm_arg_R_16_WB
	arm_arg_R_8
	arm_arg_R_rotate
	arm_arg_R_shift_R
	arm_arg_R_shift_imm
	arm_arg_SP
	arm_arg_Sd
	arm_arg_Sd_Dd
	arm_arg_Dd_Sd
	arm_arg_Sm
	arm_arg_Sm_Dm
	arm_arg_Sn
	arm_arg_Sn_Dn
	arm_arg_const
	arm_arg_endian
	arm_arg_fbits
	arm_arg_fp_0
	arm_arg_imm24
	arm_arg_imm5
	arm_arg_imm5_32
	arm_arg_imm5_nz
	arm_arg_imm_12at8_4at0
	arm_arg_imm_4at16_12at0
	arm_arg_imm_vfp
	arm_arg_label24
	arm_arg_label24H
	arm_arg_label_m_12
	arm_arg_label_p_12
	arm_arg_label_pm_12
	arm_arg_label_pm_4_4
	arm_arg_lsb_width
	arm_arg_mem_R
	arm_arg_mem_R_pm_R_W
	arm_arg_mem_R_pm_R_postindex
	arm_arg_mem_R_pm_R_shift_imm_W
	arm_arg_mem_R_pm_R_shift_imm_offset
	arm_arg_mem_R_pm_R_shift_imm_postindex
	arm_arg_mem_R_pm_imm12_W
	arm_arg_mem_R_pm_imm12_offset
	arm_arg_mem_R_pm_imm12_postindex
	arm_arg_mem_R_pm_imm8_W
	arm_arg_mem_R_pm_imm8_postindex
	arm_arg_mem_R_pm_imm8at0_offset
	arm_arg_option
	arm_arg_registers
	arm_arg_registers1
	arm_arg_registers2
	arm_arg_satimm4
	arm_arg_satimm5
	arm_arg_satimm4m1
	arm_arg_satimm5m1
	arm_arg_widthm1
)

// decodeArg decodes the arg described by aop from the instruction bits x.
// It returns nil if x cannot be decoded according to aop.
func arm_decodeArg(aop arm_instArg, x uint32) arm_Arg {
	switch aop {
	default:
		return nil

	case arm_arg_APSR:
		return arm_APSR
	case arm_arg_FPSCR:
		return arm_FPSCR

	case arm_arg_R_0:
		return arm_Reg(x & (1<<4 - 1))
	case arm_arg_R_8:
		return arm_Reg((x >> 8) & (1<<4 - 1))
	case arm_arg_R_12:
		return arm_Reg((x >> 12) & (1<<4 - 1))
	case arm_arg_R_16:
		return arm_Reg((x >> 16) & (1<<4 - 1))

	case arm_arg_R_12_nzcv:
		r := arm_Reg((x >> 12) & (1<<4 - 1))
		if r == arm_R15 {
			return arm_APSR_nzcv
		}
		return r

	case arm_arg_R_16_WB:
		mode := arm_AddrLDM
		if (x>>21)&1 != 0 {
			mode = arm_AddrLDM_WB
		}
		return arm_Mem{Base: arm_Reg((x >> 16) & (1<<4 - 1)), Mode: mode}

	case arm_arg_R_rotate:
		Rm := arm_Reg(x & (1<<4 - 1))
		typ, count := arm_decodeShift(x)
		// ROR #0 here means ROR #0, but decodeShift rewrites to RRX #1.
		if typ == arm_RotateRightExt {
			return arm_Reg(Rm)
		}
		return arm_RegShift{Rm, typ, uint8(count)}

	case arm_arg_R_shift_R:
		Rm := arm_Reg(x & (1<<4 - 1))
		Rs := arm_Reg((x >> 8) & (1<<4 - 1))
		typ := arm_Shift((x >> 5) & (1<<2 - 1))
		return arm_RegShiftReg{Rm, typ, Rs}

	case arm_arg_R_shift_imm:
		Rm := arm_Reg(x & (1<<4 - 1))
		typ, count := arm_decodeShift(x)
		if typ == arm_ShiftLeft && count == 0 {
			return arm_Reg(Rm)
		}
		return arm_RegShift{Rm, typ, uint8(count)}

	case arm_arg_R1_0:
		return arm_Reg((x & (1<<4 - 1)))
	case arm_arg_R1_12:
		return arm_Reg(((x >> 12) & (1<<4 - 1)))
	case arm_arg_R2_0:
		return arm_Reg((x & (1<<4 - 1)) | 1)
	case arm_arg_R2_12:
		return arm_Reg(((x >> 12) & (1<<4 - 1)) | 1)

	case arm_arg_SP:
		return arm_SP

	case arm_arg_Sd_Dd:
		v := (x >> 12) & (1<<4 - 1)
		vx := (x >> 22) & 1
		sz := (x >> 8) & 1
		if sz != 0 {
			return arm_D0 + arm_Reg(vx<<4+v)
		} else {
			return arm_S0 + arm_Reg(v<<1+vx)
		}

	case arm_arg_Dd_Sd:
		return arm_decodeArg(arm_arg_Sd_Dd, x^(1<<8))

	case arm_arg_Sd:
		v := (x >> 12) & (1<<4 - 1)
		vx := (x >> 22) & 1
		return arm_S0 + arm_Reg(v<<1+vx)

	case arm_arg_Sm_Dm:
		v := (x >> 0) & (1<<4 - 1)
		vx := (x >> 5) & 1
		sz := (x >> 8) & 1
		if sz != 0 {
			return arm_D0 + arm_Reg(vx<<4+v)
		} else {
			return arm_S0 + arm_Reg(v<<1+vx)
		}

	case arm_arg_Sm:
		v := (x >> 0) & (1<<4 - 1)
		vx := (x >> 5) & 1
		return arm_S0 + arm_Reg(v<<1+vx)

	case arm_arg_Dn_half:
		v := (x >> 16) & (1<<4 - 1)
		vx := (x >> 7) & 1
		return arm_RegX{arm_D0 + arm_Reg(vx<<4+v), int((x >> 21) & 1)}

	case arm_arg_Sn_Dn:
		v := (x >> 16) & (1<<4 - 1)
		vx := (x >> 7) & 1
		sz := (x >> 8) & 1
		if sz != 0 {
			return arm_D0 + arm_Reg(vx<<4+v)
		} else {
			return arm_S0 + arm_Reg(v<<1+vx)
		}

	case arm_arg_Sn:
		v := (x >> 16) & (1<<4 - 1)
		vx := (x >> 7) & 1
		return arm_S0 + arm_Reg(v<<1+vx)

	case arm_arg_const:
		v := x & (1<<8 - 1)
		rot := (x >> 8) & (1<<4 - 1) * 2
		if rot > 0 && v&3 == 0 {
			// could rotate less
			return arm_ImmAlt{uint8(v), uint8(rot)}
		}
		if rot >= 24 && ((v<<(32-rot))&0xFF)>>(32-rot) == v {
			// could wrap around to rot==0.
			return arm_ImmAlt{uint8(v), uint8(rot)}
		}
		return arm_Imm(v>>rot | v<<(32-rot))

	case arm_arg_endian:
		return arm_Endian((x >> 9) & 1)

	case arm_arg_fbits:
		return arm_Imm((16 << ((x >> 7) & 1)) - ((x&(1<<4-1))<<1 | (x>>5)&1))

	case arm_arg_fp_0:
		return arm_Imm(0)

	case arm_arg_imm24:
		return arm_Imm(x & (1<<24 - 1))

	case arm_arg_imm5:
		return arm_Imm((x >> 7) & (1<<5 - 1))

	case arm_arg_imm5_32:
		x = (x >> 7) & (1<<5 - 1)
		if x == 0 {
			x = 32
		}
		return arm_Imm(x)

	case arm_arg_imm5_nz:
		x = (x >> 7) & (1<<5 - 1)
		if x == 0 {
			return nil
		}
		return arm_Imm(x)

	case arm_arg_imm_4at16_12at0:
		return arm_Imm((x>>16)&(1<<4-1)<<12 | x&(1<<12-1))

	case arm_arg_imm_12at8_4at0:
		return arm_Imm((x>>8)&(1<<12-1)<<4 | x&(1<<4-1))

	case arm_arg_imm_vfp:
		x = (x>>16)&(1<<4-1)<<4 | x&(1<<4-1)
		return arm_Imm(x)

	case arm_arg_label24:
		imm := (x & (1<<24 - 1)) << 2
		return arm_PCRel(int32(imm<<6) >> 6)

	case arm_arg_label24H:
		h := (x >> 24) & 1
		imm := (x&(1<<24-1))<<2 | h<<1
		return arm_PCRel(int32(imm<<6) >> 6)

	case arm_arg_label_m_12:
		d := int32(x & (1<<12 - 1))
		return arm_Mem{Base: arm_PC, Mode: arm_AddrOffset, Offset: int16(-d)}

	case arm_arg_label_p_12:
		d := int32(x & (1<<12 - 1))
		return arm_Mem{Base: arm_PC, Mode: arm_AddrOffset, Offset: int16(d)}

	case arm_arg_label_pm_12:
		d := int32(x & (1<<12 - 1))
		u := (x >> 23) & 1
		if u == 0 {
			d = -d
		}
		return arm_Mem{Base: arm_PC, Mode: arm_AddrOffset, Offset: int16(d)}

	case arm_arg_label_pm_4_4:
		d := int32((x>>8)&(1<<4-1)<<4 | x&(1<<4-1))
		u := (x >> 23) & 1
		if u == 0 {
			d = -d
		}
		return arm_PCRel(d)

	case arm_arg_lsb_width:
		lsb := (x >> 7) & (1<<5 - 1)
		msb := (x >> 16) & (1<<5 - 1)
		if msb < lsb || msb >= 32 {
			return nil
		}
		return arm_Imm(msb + 1 - lsb)

	case arm_arg_mem_R:
		Rn := arm_Reg((x >> 16) & (1<<4 - 1))
		return arm_Mem{Base: Rn, Mode: arm_AddrOffset}

	case arm_arg_mem_R_pm_R_postindex:
		// Treat [<Rn>],+/-<Rm> like [<Rn>,+/-<Rm>{,<shift>}]{!}
		// by forcing shift bits to <<0 and P=0, W=0 (postindex=true).
		return arm_decodeArg(arm_arg_mem_R_pm_R_shift_imm_W, x&^((1<<7-1)<<5|1<<24|1<<21))

	case arm_arg_mem_R_pm_R_W:
		// Treat [<Rn>,+/-<Rm>]{!} like [<Rn>,+/-<Rm>{,<shift>}]{!}
		// by forcing shift bits to <<0.
		return arm_decodeArg(arm_arg_mem_R_pm_R_shift_imm_W, x&^((1<<7-1)<<5))

	case arm_arg_mem_R_pm_R_shift_imm_offset:
		// Treat [<Rn>],+/-<Rm>{,<shift>} like [<Rn>,+/-<Rm>{,<shift>}]{!}
		// by forcing P=1, W=0 (index=false, wback=false).
		return arm_decodeArg(arm_arg_mem_R_pm_R_shift_imm_W, x&^(1<<21)|1<<24)

	case arm_arg_mem_R_pm_R_shift_imm_postindex:
		// Treat [<Rn>],+/-<Rm>{,<shift>} like [<Rn>,+/-<Rm>{,<shift>}]{!}
		// by forcing P=0, W=0 (postindex=true).
		return arm_decodeArg(arm_arg_mem_R_pm_R_shift_imm_W, x&^(1<<24|1<<21))

	case arm_arg_mem_R_pm_R_shift_imm_W:
		Rn := arm_Reg((x >> 16) & (1<<4 - 1))
		Rm := arm_Reg(x & (1<<4 - 1))
		typ, count := arm_decodeShift(x)
		u := (x >> 23) & 1
		w := (x >> 21) & 1
		p := (x >> 24) & 1
		if p == 0 && w == 1 {
			return nil
		}
		sign := int8(+1)
		if u == 0 {
			sign = -1
		}
		mode := arm_AddrMode(uint8(p<<1) | uint8(w^1))
		return arm_Mem{Base: Rn, Mode: mode, Sign: sign, Index: Rm, Shift: typ, Count: count}

	case arm_arg_mem_R_pm_imm12_offset:
		// Treat [<Rn>,#+/-<imm12>] like [<Rn>{,#+/-<imm12>}]{!}
		// by forcing P=1, W=0 (index=false, wback=false).
		return arm_decodeArg(arm_arg_mem_R_pm_imm12_W, x&^(1<<21)|1<<24)

	case arm_arg_mem_R_pm_imm12_postindex:
		// Treat [<Rn>],#+/-<imm12> like [<Rn>{,#+/-<imm12>}]{!}
		// by forcing P=0, W=0 (postindex=true).
		return arm_decodeArg(arm_arg_mem_R_pm_imm12_W, x&^(1<<24|1<<21))

	case arm_arg_mem_R_pm_imm12_W:
		Rn := arm_Reg((x >> 16) & (1<<4 - 1))
		u := (x >> 23) & 1
		w := (x >> 21) & 1
		p := (x >> 24) & 1
		if p == 0 && w == 1 {
			return nil
		}
		sign := int8(+1)
		if u == 0 {
			sign = -1
		}
		imm := int16(x & (1<<12 - 1))
		mode := arm_AddrMode(uint8(p<<1) | uint8(w^1))
		return arm_Mem{Base: Rn, Mode: mode, Offset: int16(sign) * imm}

	case arm_arg_mem_R_pm_imm8_postindex:
		// Treat [<Rn>],#+/-<imm8> like [<Rn>{,#+/-<imm8>}]{!}
		// by forcing P=0, W=0 (postindex=true).
		return arm_decodeArg(arm_arg_mem_R_pm_imm8_W, x&^(1<<24|1<<21))

	case arm_arg_mem_R_pm_imm8_W:
		Rn := arm_Reg((x >> 16) & (1<<4 - 1))
		u := (x >> 23) & 1
		w := (x >> 21) & 1
		p := (x >> 24) & 1
		if p == 0 && w == 1 {
			return nil
		}
		sign := int8(+1)
		if u == 0 {
			sign = -1
		}
		imm := int16((x>>8)&(1<<4-1)<<4 | x&(1<<4-1))
		mode := arm_AddrMode(uint8(p<<1) | uint8(w^1))
		return arm_Mem{Base: Rn, Mode: mode, Offset: int16(sign) * imm}

	case arm_arg_mem_R_pm_imm8at0_offset:
		Rn := arm_Reg((x >> 16) & (1<<4 - 1))
		u := (x >> 23) & 1
		sign := int8(+1)
		if u == 0 {
			sign = -1
		}
		imm := int16(x&(1<<8-1)) << 2
		return arm_Mem{Base: Rn, Mode: arm_AddrOffset, Offset: int16(sign) * imm}

	case arm_arg_option:
		return arm_Imm(x & (1<<4 - 1))

	case arm_arg_registers:
		return arm_RegList(x & (1<<16 - 1))

	case arm_arg_registers2:
		x &= 1<<16 - 1
		n := 0
		for i := 0; i < 16; i++ {
			if x>>uint(i)&1 != 0 {
				n++
			}
		}
		if n < 2 {
			return nil
		}
		return arm_RegList(x)

	case arm_arg_registers1:
		Rt := (x >> 12) & (1<<4 - 1)
		return arm_RegList(1 << Rt)

	case arm_arg_satimm4:
		return arm_Imm((x >> 16) & (1<<4 - 1))

	case arm_arg_satimm5:
		return arm_Imm((x >> 16) & (1<<5 - 1))

	case arm_arg_satimm4m1:
		return arm_Imm((x>>16)&(1<<4-1) + 1)

	case arm_arg_satimm5m1:
		return arm_Imm((x>>16)&(1<<5-1) + 1)

	case arm_arg_widthm1:
		return arm_Imm((x>>16)&(1<<5-1) + 1)

	}
}

// decodeShift decodes the shift-by-immediate encoded in x.
func arm_decodeShift(x uint32) (arm_Shift, uint8) {
	count := (x >> 7) & (1<<5 - 1)
	typ := arm_Shift((x >> 5) & (1<<2 - 1))
	switch typ {
	case arm_ShiftRight, arm_ShiftRightSigned:
		if count == 0 {
			count = 32
		}
	case arm_RotateRight:
		if count == 0 {
			typ = arm_RotateRightExt
			count = 1
		}
	}
	return typ, uint8(count)
}

/* gnu.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

var arm_saveDot = strings.NewReplacer(
	".F16", "_dot_F16",
	".F32", "_dot_F32",
	".F64", "_dot_F64",
	".S32", "_dot_S32",
	".U32", "_dot_U32",
	".FXS", "_dot_S",
	".FXU", "_dot_U",
	".32", "_dot_32",
)

// GNUSyntax returns the GNU assembler syntax for the instruction, as defined by GNU binutils.
// This form typically matches the syntax defined in the ARM Reference Manual.
func arm_GNUSyntax(inst arm_Inst) string {
	var buf bytes.Buffer
	op := inst.Op.String()
	op = arm_saveDot.Replace(op)
	op = strings.Replace(op, ".", "", -1)
	op = strings.Replace(op, "_dot_", ".", -1)
	op = strings.ToLower(op)
	buf.WriteString(op)
	sep := " "
	for i, arg := range inst.Args {
		if arg == nil {
			break
		}
		text := arm_gnuArg(&inst, i, arg)
		if text == "" {
			continue
		}
		buf.WriteString(sep)
		sep = ", "
		buf.WriteString(text)
	}
	return buf.String()
}

func arm_gnuArg(inst *arm_Inst, argIndex int, arg arm_Arg) string {
	switch inst.Op &^ 15 {
	case arm_LDRD_EQ, arm_LDREXD_EQ, arm_STRD_EQ:
		if argIndex == 1 {
			// second argument in consecutive pair not printed
			return ""
		}
	case arm_STREXD_EQ:
		if argIndex == 2 {
			// second argument in consecutive pair not printed
			return ""
		}
	}

	switch arg := arg.(type) {
	case arm_Imm:
		switch inst.Op &^ 15 {
		case arm_BKPT_EQ:
			return fmt.Sprintf("%#04x", uint32(arg))
		case arm_SVC_EQ:
			return fmt.Sprintf("%#08x", uint32(arg))
		}
		return fmt.Sprintf("#%d", int32(arg))

	case arm_ImmAlt:
		return fmt.Sprintf("#%d, %d", arg.Val, arg.Rot)

	case arm_Mem:
		R := arm_gnuArg(inst, -1, arg.Base)
		X := ""
		if arg.Sign != 0 {
			X = ""
			if arg.Sign < 0 {
				X = "-"
			}
			X += arm_gnuArg(inst, -1, arg.Index)
			if arg.Shift == arm_ShiftLeft && arg.Count == 0 {
				// nothing
			} else if arg.Shift == arm_RotateRightExt {
				X += ", rrx"
			} else {
				X += fmt.Sprintf(", %s #%d", strings.ToLower(arg.Shift.String()), arg.Count)
			}
		} else {
			X = fmt.Sprintf("#%d", arg.Offset)
		}

		switch arg.Mode {
		case arm_AddrOffset:
			if X == "#0" {
				return fmt.Sprintf("[%s]", R)
			}
			return fmt.Sprintf("[%s, %s]", R, X)
		case arm_AddrPreIndex:
			return fmt.Sprintf("[%s, %s]!", R, X)
		case arm_AddrPostIndex:
			return fmt.Sprintf("[%s], %s", R, X)
		case arm_AddrLDM:
			if X == "#0" {
				return R
			}
		case arm_AddrLDM_WB:
			if X == "#0" {
				return R + "!"
			}
		}
		return fmt.Sprintf("[%s Mode(%d) %s]", R, int(arg.Mode), X)

	case arm_PCRel:
		return fmt.Sprintf(".%+#x", int32(arg)+4)

	case arm_Reg:
		switch inst.Op &^ 15 {
		case arm_LDREX_EQ:
			if argIndex == 0 {
				return fmt.Sprintf("r%d", int32(arg))
			}
		}
		switch arg {
		case arm_R10:
			return "sl"
		case arm_R11:
			return "fp"
		case arm_R12:
			return "ip"
		}

	case arm_RegList:
		var buf bytes.Buffer
		fmt.Fprintf(&buf, "{")
		sep := ""
		for i := 0; i < 16; i++ {
			if arg&(1<<uint(i)) != 0 {
				fmt.Fprintf(&buf, "%s%s", sep, arm_gnuArg(inst, -1, arm_Reg(i)))
				sep = ", "
			}
		}
		fmt.Fprintf(&buf, "}")
		return buf.String()

	case arm_RegShift:
		if arg.Shift == arm_ShiftLeft && arg.Count == 0 {
			return arm_gnuArg(inst, -1, arg.Reg)
		}
		if arg.Shift == arm_RotateRightExt {
			return arm_gnuArg(inst, -1, arg.Reg) + ", rrx"
		}
		return fmt.Sprintf("%s, %s #%d", arm_gnuArg(inst, -1, arg.Reg), strings.ToLower(arg.Shift.String()), arg.Count)

	case arm_RegShiftReg:
		return fmt.Sprintf("%s, %s %s", arm_gnuArg(inst, -1, arg.Reg), strings.ToLower(arg.Shift.String()), arm_gnuArg(inst, -1, arg.RegCount))

	}
	return strings.ToLower(arg.String())
}

/* inst.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// A Mode is an instruction execution mode.
type arm_Mode int

const (
	_ arm_Mode = iota
	arm_ModeARM
	arm_ModeThumb
)

func (m arm_Mode) String() string {
	switch m {
	case arm_ModeARM:
		return "ARM"
	case arm_ModeThumb:
		return "Thumb"
	}
	return fmt.Sprintf("Mode(%d)", int(m))
}

// An Op is an ARM opcode.
type arm_Op uint16

// NOTE: The actual Op values are defined in tables.go.
// They are chosen to simplify instruction decoding and
// are not a dense packing from 0 to N, although the
// density is high, probably at least 90%.

func (op arm_Op) String() string {
	if op >= arm_Op(len(arm_opstr)) || arm_opstr[op] == "" {
		return fmt.Sprintf("Op(%d)", int(op))
	}
	return arm_opstr[op]
}

// An Inst is a single instruction.
type arm_Inst struct {
	Op   arm_Op   // Opcode mnemonic
	Enc  uint32   // Raw encoding bits.
	Len  int      // Length of encoding in bytes.
	Args arm_Args // Instruction arguments, in ARM manual order.
}

func (i arm_Inst) String() string {
	var buf bytes.Buffer
	buf.WriteString(i.Op.String())
	for j, arg := range i.Args {
		if arg == nil {
			break
		}
		if j == 0 {
			buf.WriteString(" ")
		} else {
			buf.WriteString(", ")
		}
		buf.WriteString(arg.String())
	}
	return buf.String()
}

// An Args holds the instruction arguments.
// If an instruction has fewer than 4 arguments,
// the final elements in the array are nil.
type arm_Args [4]arm_Arg

// An Arg is a single instruction argument, one of these types:
// Endian, Imm, Mem, PCRel, Reg, RegList, RegShift, RegShiftReg.
type arm_Arg interface {
	IsArg()
	String() string
}

type arm_Float32Imm float32

func (arm_Float32Imm) IsArg() {}

func (f arm_Float32Imm) String() string {
	return fmt.Sprintf("#%v", float32(f))
}

type arm_Float64Imm float32

func (arm_Float64Imm) IsArg() {}

func (f arm_Float64Imm) String() string {
	return fmt.Sprintf("#%v", float64(f))
}

// An Imm is an integer constant.
type arm_Imm uint32

func (arm_Imm) IsArg() {}

func (i arm_Imm) String() string {
	return fmt.Sprintf("#%#x", uint32(i))
}

// A ImmAlt is an alternate encoding of an integer constant.
type arm_ImmAlt struct {
	Val uint8
	Rot uint8
}

func (arm_ImmAlt) IsArg() {}

func (i arm_ImmAlt) Imm() arm_Imm {
	v := uint32(i.Val)
	r := uint(i.Rot)
	return arm_Imm(v>>r | v<<(32-r))
}

func (i arm_ImmAlt) String() string {
	return fmt.Sprintf("#%#x, %d", i.Val, i.Rot)
}

// A Label is a text (code) address.
type arm_Label uint32

func (arm_Label) IsArg() {}

func (i arm_Label) String() string {
	return fmt.Sprintf("%#x", uint32(i))
}

// A Reg is a single register.
// The zero value denotes R0, not the absence of a register.
type arm_Reg uint8

const (
	arm_R0 arm_Reg = iota
	arm_R1
	arm_R2
	arm_R3
	arm_R4
	arm_R5
	arm_R6
	arm_R7
	arm_R8
	arm_R9
	arm_R10
	arm_R11
	arm_R12
	arm_R13
	arm_R14
	arm_R15

	arm_S0
	arm_S1
	arm_S2
	arm_S3
	arm_S4
	arm_S5
	arm_S6
	arm_S7
	arm_S8
	arm_S9
	arm_S10
	arm_S11
	arm_S12
	arm_S13
	arm_S14
	arm_S15
	arm_S16
	arm_S17
	arm_S18
	arm_S19
	arm_S20
	arm_S21
	arm_S22
	arm_S23
	arm_S24
	arm_S25
	arm_S26
	arm_S27
	arm_S28
	arm_S29
	arm_S30
	arm_S31

	arm_D0
	arm_D1
	arm_D2
	arm_D3
	arm_D4
	arm_D5
	arm_D6
	arm_D7
	arm_D8
	arm_D9
	arm_D10
	arm_D11
	arm_D12
	arm_D13
	arm_D14
	arm_D15
	arm_D16
	arm_D17
	arm_D18
	arm_D19
	arm_D20
	arm_D21
	arm_D22
	arm_D23
	arm_D24
	arm_D25
	arm_D26
	arm_D27
	arm_D28
	arm_D29
	arm_D30
	arm_D31

	arm_APSR
	arm_APSR_nzcv
	arm_FPSCR

	arm_SP = arm_R13
	arm_LR = arm_R14
	arm_PC = arm_R15
)

func (arm_Reg) IsArg() {}

func (r arm_Reg) String() string {
	switch r {
	case arm_APSR:
		return "APSR"
	case arm_APSR_nzcv:
		return "APSR_nzcv"
	case arm_FPSCR:
		return "FPSCR"
	case arm_SP:
		return "SP"
	case arm_PC:
		return "PC"
	case arm_LR:
		return "LR"
	}
	if arm_R0 <= r && r <= arm_R15 {
		return fmt.Sprintf("R%d", int(r-arm_R0))
	}
	if arm_S0 <= r && r <= arm_S31 {
		return fmt.Sprintf("S%d", int(r-arm_S0))
	}
	if arm_D0 <= r && r <= arm_D31 {
		return fmt.Sprintf("D%d", int(r-arm_D0))
	}
	return fmt.Sprintf("Reg(%d)", int(r))
}

// A RegX represents a fraction of a multi-value register.
// The Index field specifies the index number,
// but the size of the fraction is not specified.
// It must be inferred from the instruction and the register type.
// For example, in a VMOV instruction, RegX{D5, 1} represents
// the top 32 bits of the 64-bit D5 register.
type arm_RegX struct {
	Reg   arm_Reg
	Index int
}

func (arm_RegX) IsArg() {}

func (r arm_RegX) String() string {
	return fmt.Sprintf("%s[%d]", r.Reg, r.Index)
}

// A RegList is a register list.
// Bits at indexes x = 0 through 15 indicate whether the corresponding Rx register is in the list.
type arm_RegList uint16

func (arm_RegList) IsArg() {}

func (r arm_RegList) String() string {
	var buf bytes.Buffer
	fmt.Fprintf(&buf, "{")
	sep := ""
	for i := 0; i < 16; i++ {
		if r&(1<<uint(i)) != 0 {
			fmt.Fprintf(&buf, "%s%s", sep, arm_Reg(i).String())
			sep = ","
		}
	}
	fmt.Fprintf(&buf, "}")
	return buf.String()
}

// An Endian is the argument to the SETEND instruction.
type arm_Endian uint8

const (
	arm_LittleEndian arm_Endian = 0
	arm_BigEndian    arm_Endian = 1
)

func (arm_Endian) IsArg() {}

func (e arm_Endian) String() string {
	if e != 0 {
		return "BE"
	}
	return "LE"
}

// A Shift describes an ARM shift operation.
type arm_Shift uint8

const (
	arm_ShiftLeft        arm_Shift = 0 // left shift
	arm_ShiftRight       arm_Shift = 1 // logical (unsigned) right shift
	arm_ShiftRightSigned arm_Shift = 2 // arithmetic (signed) right shift
	arm_RotateRight      arm_Shift = 3 // right rotate
	arm_RotateRightExt   arm_Shift = 4 // right rotate through carry (Count will always be 1)
)

var arm_shiftName = [...]string{
	"LSL", "LSR", "ASR", "ROR", "RRX",
}

func (s arm_Shift) String() string {
	if s < 5 {
		return arm_shiftName[s]
	}
	return fmt.Sprintf("Shift(%d)", int(s))
}

// A RegShift is a register shifted by a constant.
type arm_RegShift struct {
	Reg   arm_Reg
	Shift arm_Shift
	Count uint8
}

func (arm_RegShift) IsArg() {}

func (r arm_RegShift) String() string {
	return fmt.Sprintf("%s %s #%d", r.Reg, r.Shift, r.Count)
}

// A RegShiftReg is a register shifted by a register.
type arm_RegShiftReg struct {
	Reg      arm_Reg
	Shift    arm_Shift
	RegCount arm_Reg
}

func (arm_RegShiftReg) IsArg() {}

func (r arm_RegShiftReg) String() string {
	return fmt.Sprintf("%s %s %s", r.Reg, r.Shift, r.RegCount)
}

// A PCRel describes a memory address (usually a code label)
// as a distance relative to the program counter.
// TODO(rsc): Define which program counter (PC+4? PC+8? PC?).
type arm_PCRel int32

func (arm_PCRel) IsArg() {}

func (r arm_PCRel) String() string {
	return fmt.Sprintf("PC%+#x", int32(r))
}

// An AddrMode is an ARM addressing mode.
type arm_AddrMode uint8

const (
	_                 arm_AddrMode = iota
	arm_AddrPostIndex              // [R], X – use address R, set R = R + X
	arm_AddrPreIndex               // [R, X]! – use address R + X, set R = R + X
	arm_AddrOffset                 // [R, X] – use address R + X
	arm_AddrLDM                    // R – [R] but formats as R, for LDM/STM only
	arm_AddrLDM_WB                 // R! - [R], X where X is instruction-specific amount, for LDM/STM only
)

// A Mem is a memory reference made up of a base R and index expression X.
// The effective memory address is R or R+X depending on AddrMode.
// The index expression is X = Sign*(Index Shift Count) + Offset,
// but in any instruction either Sign = 0 or Offset = 0.
type arm_Mem struct {
	Base   arm_Reg
	Mode   arm_AddrMode
	Sign   int8
	Index  arm_Reg
	Shift  arm_Shift
	Count  uint8
	Offset int16
}

func (arm_Mem) IsArg() {}

func (m arm_Mem) String() string {
	R := m.Base.String()
	X := ""
	if m.Sign != 0 {
		X = "+"
		if m.Sign < 0 {
			X = "-"
		}
		X += m.Index.String()
		if m.Shift != arm_ShiftLeft || m.Count != 0 {
			X += fmt.Sprintf(", %s #%d", m.Shift, m.Count)
		}
	} else {
		X = fmt.Sprintf("#%d", m.Offset)
	}

	switch m.Mode {
	case arm_AddrOffset:
		if X == "#0" {
			return fmt.Sprintf("[%s]", R)
		}
		return fmt.Sprintf("[%s, %s]", R, X)
	case arm_AddrPreIndex:
		return fmt.Sprintf("[%s, %s]!", R, X)
	case arm_AddrPostIndex:
		return fmt.Sprintf("[%s], %s", R, X)
	case arm_AddrLDM:
		if X == "#0" {
			return R
		}
	case arm_AddrLDM_WB:
		if X == "#0" {
			return R + "!"
		}
	}
	return fmt.Sprintf("[%s Mode(%d) %s]", R, int(m.Mode), X)
}

/* plan9x.go */

// Copyright 2014 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// plan9Syntax returns the Go assembler syntax for the instruction.
// The syntax was originally defined by Plan 9.
// The pc is the program counter of the instruction, used for expanding
// PC-relative addresses into absolute ones.
// The symname function queries the symbol table for the program
// being disassembled. Given a target address it returns the name and base
// address of the symbol containing the target, if any; otherwise it returns "", 0.
// The reader r should read from the text segment using text addresses
// as offsets; it is used to display pc-relative loads as constant loads.
func arm_plan9Syntax(inst arm_Inst, pc uint64, symname func(uint64) (string, uint64), text io.ReaderAt) string {
	if symname == nil {
		symname = func(uint64) (string, uint64) { return "", 0 }
	}

	var args []string
	for _, a := range inst.Args {
		if a == nil {
			break
		}
		args = append(args, arm_plan9Arg(&inst, pc, symname, a))
	}

	op := inst.Op.String()

	switch inst.Op &^ 15 {
	case arm_LDR_EQ, arm_LDRB_EQ, arm_LDRH_EQ:
		// Check for RET
		reg, _ := inst.Args[0].(arm_Reg)
		mem, _ := inst.Args[1].(arm_Mem)
		if inst.Op&^15 == arm_LDR_EQ && reg == arm_R15 && mem.Base == arm_SP && mem.Sign == 0 && mem.Mode == arm_AddrPostIndex {
			return fmt.Sprintf("RET%s #%d", op[3:], mem.Offset)
		}

		// Check for PC-relative load.
		if mem.Base == arm_PC && mem.Sign == 0 && mem.Mode == arm_AddrOffset && text != nil {
			addr := uint32(pc) + 8 + uint32(mem.Offset)
			buf := make([]byte, 4)
			switch inst.Op &^ 15 {
			case arm_LDRB_EQ:
				if _, err := text.ReadAt(buf[:1], int64(addr)); err != nil {
					break
				}
				args[1] = fmt.Sprintf("$%#x", buf[0])

			case arm_LDRH_EQ:
				if _, err := text.ReadAt(buf[:2], int64(addr)); err != nil {
					break
				}
				args[1] = fmt.Sprintf("$%#x", binary.LittleEndian.Uint16(buf))

			case arm_LDR_EQ:
				if _, err := text.ReadAt(buf, int64(addr)); err != nil {
					break
				}
				x := binary.LittleEndian.Uint32(buf)
				if s, base := symname(uint64(x)); s != "" && uint64(x) == base {
					args[1] = fmt.Sprintf("$%s(SB)", s)
				} else {
					args[1] = fmt.Sprintf("$%#x", x)
				}
			}
		}
	}

	// Move addressing mode into opcode suffix.
	suffix := ""
	switch inst.Op &^ 15 {
	case arm_LDR_EQ, arm_LDRB_EQ, arm_LDRH_EQ, arm_STR_EQ, arm_STRB_EQ, arm_STRH_EQ:
		mem, _ := inst.Args[1].(arm_Mem)
		switch mem.Mode {
		case arm_AddrOffset, arm_AddrLDM:
			// no suffix
		case arm_AddrPreIndex, arm_AddrLDM_WB:
			suffix = ".W"
		case arm_AddrPostIndex:
			suffix = ".P"
		}
		off := ""
		if mem.Offset != 0 {
			off = fmt.Sprintf("%#x", mem.Offset)
		}
		base := fmt.Sprintf("(R%d)", int(mem.Base))
		index := ""
		if mem.Sign != 0 {
			sign := ""
			if mem.Sign < 0 {
				sign = ""
			}
			shift := ""
			if mem.Count != 0 {
				shift = fmt.Sprintf("%s%d", arm_plan9Shift[mem.Shift], mem.Count)
			}
			index = fmt.Sprintf("(%sR%d%s)", sign, int(mem.Index), shift)
		}
		args[1] = off + base + index
	}

	// Reverse args, placing dest last.
	for i, j := 0, len(args)-1; i < j; i, j = i+1, j-1 {
		args[i], args[j] = args[j], args[i]
	}

	switch inst.Op &^ 15 {
	case arm_MOV_EQ:
		op = "MOVW" + op[3:]

	case arm_LDR_EQ:
		op = "MOVW" + op[3:] + suffix
	case arm_LDRB_EQ:
		op = "MOVB" + op[4:] + suffix
	case arm_LDRH_EQ:
		op = "MOVH" + op[4:] + suffix

	case arm_STR_EQ:
		op = "MOVW" + op[3:] + suffix
		args[0], args[1] = args[1], args[0]
	case arm_STRB_EQ:
		op = "MOVB" + op[4:] + suffix
		args[0], args[1] = args[1], args[0]
	case arm_STRH_EQ:
		op = "MOVH" + op[4:] + suffix
		args[0], args[1] = args[1], args[0]
	}

	if args != nil {
		op += " " + strings.Join(args, ", ")
	}

	return op
}

// assembler syntax for the various shifts.
// @x> is a lie; the assembler uses @> 0
// instead of @x> 1, but i wanted to be clear that it
// was a different operation (rotate right extended, not rotate right).
var arm_plan9Shift = []string{"<<", ">>", "->", "@>", "@x>"}

func arm_plan9Arg(inst *arm_Inst, pc uint64, symname func(uint64) (string, uint64), arg arm_Arg) string {
	switch a := arg.(type) {
	case arm_Endian:

	case arm_Imm:
		return fmt.Sprintf("$%d", int(a))

	case arm_Mem:

	case arm_PCRel:
		addr := uint32(pc) + 8 + uint32(a)
		if s, base := symname(uint64(addr)); s != "" && uint64(addr) == base {
			return fmt.Sprintf("%s(SB)", s)
		}
		return fmt.Sprintf("%#x", addr)

	case arm_Reg:
		if a < 16 {
			return fmt.Sprintf("R%d", int(a))
		}

	case arm_RegList:
		var buf bytes.Buffer
		start := -2
		end := -2
		fmt.Fprintf(&buf, "[")
		flush := func() {
			if start >= 0 {
				if buf.Len() > 1 {
					fmt.Fprintf(&buf, ",")
				}
				if start == end {
					fmt.Fprintf(&buf, "R%d", start)
				} else {
					fmt.Fprintf(&buf, "R%d-R%d", start, end)
				}
			}
		}
		for i := 0; i < 16; i++ {
			if a&(1<<uint(i)) != 0 {
				if i == end+1 {
					end++
					continue
				}
				start = i
				end = i
			}
		}
		flush()
		fmt.Fprintf(&buf, "]")
		return buf.String()

	case arm_RegShift:
		return fmt.Sprintf("R%d%s$%d", int(a.Reg), arm_plan9Shift[a.Shift], int(a.Count))

	case arm_RegShiftReg:
		return fmt.Sprintf("R%d%sR%d", int(a.Reg), arm_plan9Shift[a.Shift], int(a.RegCount))
	}
	return strings.ToUpper(arg.String())
}

/* tables.go */

const (
	_ arm_Op = iota
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_ADC_EQ
	arm_ADC_NE
	arm_ADC_CS
	arm_ADC_CC
	arm_ADC_MI
	arm_ADC_PL
	arm_ADC_VS
	arm_ADC_VC
	arm_ADC_HI
	arm_ADC_LS
	arm_ADC_GE
	arm_ADC_LT
	arm_ADC_GT
	arm_ADC_LE
	arm_ADC
	arm_ADC_ZZ
	arm_ADC_S_EQ
	arm_ADC_S_NE
	arm_ADC_S_CS
	arm_ADC_S_CC
	arm_ADC_S_MI
	arm_ADC_S_PL
	arm_ADC_S_VS
	arm_ADC_S_VC
	arm_ADC_S_HI
	arm_ADC_S_LS
	arm_ADC_S_GE
	arm_ADC_S_LT
	arm_ADC_S_GT
	arm_ADC_S_LE
	arm_ADC_S
	arm_ADC_S_ZZ
	arm_ADD_EQ
	arm_ADD_NE
	arm_ADD_CS
	arm_ADD_CC
	arm_ADD_MI
	arm_ADD_PL
	arm_ADD_VS
	arm_ADD_VC
	arm_ADD_HI
	arm_ADD_LS
	arm_ADD_GE
	arm_ADD_LT
	arm_ADD_GT
	arm_ADD_LE
	arm_ADD
	arm_ADD_ZZ
	arm_ADD_S_EQ
	arm_ADD_S_NE
	arm_ADD_S_CS
	arm_ADD_S_CC
	arm_ADD_S_MI
	arm_ADD_S_PL
	arm_ADD_S_VS
	arm_ADD_S_VC
	arm_ADD_S_HI
	arm_ADD_S_LS
	arm_ADD_S_GE
	arm_ADD_S_LT
	arm_ADD_S_GT
	arm_ADD_S_LE
	arm_ADD_S
	arm_ADD_S_ZZ
	arm_AND_EQ
	arm_AND_NE
	arm_AND_CS
	arm_AND_CC
	arm_AND_MI
	arm_AND_PL
	arm_AND_VS
	arm_AND_VC
	arm_AND_HI
	arm_AND_LS
	arm_AND_GE
	arm_AND_LT
	arm_AND_GT
	arm_AND_LE
	arm_AND
	arm_AND_ZZ
	arm_AND_S_EQ
	arm_AND_S_NE
	arm_AND_S_CS
	arm_AND_S_CC
	arm_AND_S_MI
	arm_AND_S_PL
	arm_AND_S_VS
	arm_AND_S_VC
	arm_AND_S_HI
	arm_AND_S_LS
	arm_AND_S_GE
	arm_AND_S_LT
	arm_AND_S_GT
	arm_AND_S_LE
	arm_AND_S
	arm_AND_S_ZZ
	arm_ASR_EQ
	arm_ASR_NE
	arm_ASR_CS
	arm_ASR_CC
	arm_ASR_MI
	arm_ASR_PL
	arm_ASR_VS
	arm_ASR_VC
	arm_ASR_HI
	arm_ASR_LS
	arm_ASR_GE
	arm_ASR_LT
	arm_ASR_GT
	arm_ASR_LE
	arm_ASR
	arm_ASR_ZZ
	arm_ASR_S_EQ
	arm_ASR_S_NE
	arm_ASR_S_CS
	arm_ASR_S_CC
	arm_ASR_S_MI
	arm_ASR_S_PL
	arm_ASR_S_VS
	arm_ASR_S_VC
	arm_ASR_S_HI
	arm_ASR_S_LS
	arm_ASR_S_GE
	arm_ASR_S_LT
	arm_ASR_S_GT
	arm_ASR_S_LE
	arm_ASR_S
	arm_ASR_S_ZZ
	arm_B_EQ
	arm_B_NE
	arm_B_CS
	arm_B_CC
	arm_B_MI
	arm_B_PL
	arm_B_VS
	arm_B_VC
	arm_B_HI
	arm_B_LS
	arm_B_GE
	arm_B_LT
	arm_B_GT
	arm_B_LE
	arm_B
	arm_B_ZZ
	arm_BFC_EQ
	arm_BFC_NE
	arm_BFC_CS
	arm_BFC_CC
	arm_BFC_MI
	arm_BFC_PL
	arm_BFC_VS
	arm_BFC_VC
	arm_BFC_HI
	arm_BFC_LS
	arm_BFC_GE
	arm_BFC_LT
	arm_BFC_GT
	arm_BFC_LE
	arm_BFC
	arm_BFC_ZZ
	arm_BFI_EQ
	arm_BFI_NE
	arm_BFI_CS
	arm_BFI_CC
	arm_BFI_MI
	arm_BFI_PL
	arm_BFI_VS
	arm_BFI_VC
	arm_BFI_HI
	arm_BFI_LS
	arm_BFI_GE
	arm_BFI_LT
	arm_BFI_GT
	arm_BFI_LE
	arm_BFI
	arm_BFI_ZZ
	arm_BIC_EQ
	arm_BIC_NE
	arm_BIC_CS
	arm_BIC_CC
	arm_BIC_MI
	arm_BIC_PL
	arm_BIC_VS
	arm_BIC_VC
	arm_BIC_HI
	arm_BIC_LS
	arm_BIC_GE
	arm_BIC_LT
	arm_BIC_GT
	arm_BIC_LE
	arm_BIC
	arm_BIC_ZZ
	arm_BIC_S_EQ
	arm_BIC_S_NE
	arm_BIC_S_CS
	arm_BIC_S_CC
	arm_BIC_S_MI
	arm_BIC_S_PL
	arm_BIC_S_VS
	arm_BIC_S_VC
	arm_BIC_S_HI
	arm_BIC_S_LS
	arm_BIC_S_GE
	arm_BIC_S_LT
	arm_BIC_S_GT
	arm_BIC_S_LE
	arm_BIC_S
	arm_BIC_S_ZZ
	arm_BKPT_EQ
	arm_BKPT_NE
	arm_BKPT_CS
	arm_BKPT_CC
	arm_BKPT_MI
	arm_BKPT_PL
	arm_BKPT_VS
	arm_BKPT_VC
	arm_BKPT_HI
	arm_BKPT_LS
	arm_BKPT_GE
	arm_BKPT_LT
	arm_BKPT_GT
	arm_BKPT_LE
	arm_BKPT
	arm_BKPT_ZZ
	arm_BL_EQ
	arm_BL_NE
	arm_BL_CS
	arm_BL_CC
	arm_BL_MI
	arm_BL_PL
	arm_BL_VS
	arm_BL_VC
	arm_BL_HI
	arm_BL_LS
	arm_BL_GE
	arm_BL_LT
	arm_BL_GT
	arm_BL_LE
	arm_BL
	arm_BL_ZZ
	arm_BLX_EQ
	arm_BLX_NE
	arm_BLX_CS
	arm_BLX_CC
	arm_BLX_MI
	arm_BLX_PL
	arm_BLX_VS
	arm_BLX_VC
	arm_BLX_HI
	arm_BLX_LS
	arm_BLX_GE
	arm_BLX_LT
	arm_BLX_GT
	arm_BLX_LE
	arm_BLX
	arm_BLX_ZZ
	arm_BX_EQ
	arm_BX_NE
	arm_BX_CS
	arm_BX_CC
	arm_BX_MI
	arm_BX_PL
	arm_BX_VS
	arm_BX_VC
	arm_BX_HI
	arm_BX_LS
	arm_BX_GE
	arm_BX_LT
	arm_BX_GT
	arm_BX_LE
	arm_BX
	arm_BX_ZZ
	arm_BXJ_EQ
	arm_BXJ_NE
	arm_BXJ_CS
	arm_BXJ_CC
	arm_BXJ_MI
	arm_BXJ_PL
	arm_BXJ_VS
	arm_BXJ_VC
	arm_BXJ_HI
	arm_BXJ_LS
	arm_BXJ_GE
	arm_BXJ_LT
	arm_BXJ_GT
	arm_BXJ_LE
	arm_BXJ
	arm_BXJ_ZZ
	arm_CLREX
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_CLZ_EQ
	arm_CLZ_NE
	arm_CLZ_CS
	arm_CLZ_CC
	arm_CLZ_MI
	arm_CLZ_PL
	arm_CLZ_VS
	arm_CLZ_VC
	arm_CLZ_HI
	arm_CLZ_LS
	arm_CLZ_GE
	arm_CLZ_LT
	arm_CLZ_GT
	arm_CLZ_LE
	arm_CLZ
	arm_CLZ_ZZ
	arm_CMN_EQ
	arm_CMN_NE
	arm_CMN_CS
	arm_CMN_CC
	arm_CMN_MI
	arm_CMN_PL
	arm_CMN_VS
	arm_CMN_VC
	arm_CMN_HI
	arm_CMN_LS
	arm_CMN_GE
	arm_CMN_LT
	arm_CMN_GT
	arm_CMN_LE
	arm_CMN
	arm_CMN_ZZ
	arm_CMP_EQ
	arm_CMP_NE
	arm_CMP_CS
	arm_CMP_CC
	arm_CMP_MI
	arm_CMP_PL
	arm_CMP_VS
	arm_CMP_VC
	arm_CMP_HI
	arm_CMP_LS
	arm_CMP_GE
	arm_CMP_LT
	arm_CMP_GT
	arm_CMP_LE
	arm_CMP
	arm_CMP_ZZ
	arm_DBG_EQ
	arm_DBG_NE
	arm_DBG_CS
	arm_DBG_CC
	arm_DBG_MI
	arm_DBG_PL
	arm_DBG_VS
	arm_DBG_VC
	arm_DBG_HI
	arm_DBG_LS
	arm_DBG_GE
	arm_DBG_LT
	arm_DBG_GT
	arm_DBG_LE
	arm_DBG
	arm_DBG_ZZ
	arm_DMB
	arm_DSB
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_EOR_EQ
	arm_EOR_NE
	arm_EOR_CS
	arm_EOR_CC
	arm_EOR_MI
	arm_EOR_PL
	arm_EOR_VS
	arm_EOR_VC
	arm_EOR_HI
	arm_EOR_LS
	arm_EOR_GE
	arm_EOR_LT
	arm_EOR_GT
	arm_EOR_LE
	arm_EOR
	arm_EOR_ZZ
	arm_EOR_S_EQ
	arm_EOR_S_NE
	arm_EOR_S_CS
	arm_EOR_S_CC
	arm_EOR_S_MI
	arm_EOR_S_PL
	arm_EOR_S_VS
	arm_EOR_S_VC
	arm_EOR_S_HI
	arm_EOR_S_LS
	arm_EOR_S_GE
	arm_EOR_S_LT
	arm_EOR_S_GT
	arm_EOR_S_LE
	arm_EOR_S
	arm_EOR_S_ZZ
	arm_ISB
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_LDM_EQ
	arm_LDM_NE
	arm_LDM_CS
	arm_LDM_CC
	arm_LDM_MI
	arm_LDM_PL
	arm_LDM_VS
	arm_LDM_VC
	arm_LDM_HI
	arm_LDM_LS
	arm_LDM_GE
	arm_LDM_LT
	arm_LDM_GT
	arm_LDM_LE
	arm_LDM
	arm_LDM_ZZ
	arm_LDMDA_EQ
	arm_LDMDA_NE
	arm_LDMDA_CS
	arm_LDMDA_CC
	arm_LDMDA_MI
	arm_LDMDA_PL
	arm_LDMDA_VS
	arm_LDMDA_VC
	arm_LDMDA_HI
	arm_LDMDA_LS
	arm_LDMDA_GE
	arm_LDMDA_LT
	arm_LDMDA_GT
	arm_LDMDA_LE
	arm_LDMDA
	arm_LDMDA_ZZ
	arm_LDMDB_EQ
	arm_LDMDB_NE
	arm_LDMDB_CS
	arm_LDMDB_CC
	arm_LDMDB_MI
	arm_LDMDB_PL
	arm_LDMDB_VS
	arm_LDMDB_VC
	arm_LDMDB_HI
	arm_LDMDB_LS
	arm_LDMDB_GE
	arm_LDMDB_LT
	arm_LDMDB_GT
	arm_LDMDB_LE
	arm_LDMDB
	arm_LDMDB_ZZ
	arm_LDMIB_EQ
	arm_LDMIB_NE
	arm_LDMIB_CS
	arm_LDMIB_CC
	arm_LDMIB_MI
	arm_LDMIB_PL
	arm_LDMIB_VS
	arm_LDMIB_VC
	arm_LDMIB_HI
	arm_LDMIB_LS
	arm_LDMIB_GE
	arm_LDMIB_LT
	arm_LDMIB_GT
	arm_LDMIB_LE
	arm_LDMIB
	arm_LDMIB_ZZ
	arm_LDR_EQ
	arm_LDR_NE
	arm_LDR_CS
	arm_LDR_CC
	arm_LDR_MI
	arm_LDR_PL
	arm_LDR_VS
	arm_LDR_VC
	arm_LDR_HI
	arm_LDR_LS
	arm_LDR_GE
	arm_LDR_LT
	arm_LDR_GT
	arm_LDR_LE
	arm_LDR
	arm_LDR_ZZ
	arm_LDRB_EQ
	arm_LDRB_NE
	arm_LDRB_CS
	arm_LDRB_CC
	arm_LDRB_MI
	arm_LDRB_PL
	arm_LDRB_VS
	arm_LDRB_VC
	arm_LDRB_HI
	arm_LDRB_LS
	arm_LDRB_GE
	arm_LDRB_LT
	arm_LDRB_GT
	arm_LDRB_LE
	arm_LDRB
	arm_LDRB_ZZ
	arm_LDRBT_EQ
	arm_LDRBT_NE
	arm_LDRBT_CS
	arm_LDRBT_CC
	arm_LDRBT_MI
	arm_LDRBT_PL
	arm_LDRBT_VS
	arm_LDRBT_VC
	arm_LDRBT_HI
	arm_LDRBT_LS
	arm_LDRBT_GE
	arm_LDRBT_LT
	arm_LDRBT_GT
	arm_LDRBT_LE
	arm_LDRBT
	arm_LDRBT_ZZ
	arm_LDRD_EQ
	arm_LDRD_NE
	arm_LDRD_CS
	arm_LDRD_CC
	arm_LDRD_MI
	arm_LDRD_PL
	arm_LDRD_VS
	arm_LDRD_VC
	arm_LDRD_HI
	arm_LDRD_LS
	arm_LDRD_GE
	arm_LDRD_LT
	arm_LDRD_GT
	arm_LDRD_LE
	arm_LDRD
	arm_LDRD_ZZ
	arm_LDREX_EQ
	arm_LDREX_NE
	arm_LDREX_CS
	arm_LDREX_CC
	arm_LDREX_MI
	arm_LDREX_PL
	arm_LDREX_VS
	arm_LDREX_VC
	arm_LDREX_HI
	arm_LDREX_LS
	arm_LDREX_GE
	arm_LDREX_LT
	arm_LDREX_GT
	arm_LDREX_LE
	arm_LDREX
	arm_LDREX_ZZ
	arm_LDREXB_EQ
	arm_LDREXB_NE
	arm_LDREXB_CS
	arm_LDREXB_CC
	arm_LDREXB_MI
	arm_LDREXB_PL
	arm_LDREXB_VS
	arm_LDREXB_VC
	arm_LDREXB_HI
	arm_LDREXB_LS
	arm_LDREXB_GE
	arm_LDREXB_LT
	arm_LDREXB_GT
	arm_LDREXB_LE
	arm_LDREXB
	arm_LDREXB_ZZ
	arm_LDREXD_EQ
	arm_LDREXD_NE
	arm_LDREXD_CS
	arm_LDREXD_CC
	arm_LDREXD_MI
	arm_LDREXD_PL
	arm_LDREXD_VS
	arm_LDREXD_VC
	arm_LDREXD_HI
	arm_LDREXD_LS
	arm_LDREXD_GE
	arm_LDREXD_LT
	arm_LDREXD_GT
	arm_LDREXD_LE
	arm_LDREXD
	arm_LDREXD_ZZ
	arm_LDREXH_EQ
	arm_LDREXH_NE
	arm_LDREXH_CS
	arm_LDREXH_CC
	arm_LDREXH_MI
	arm_LDREXH_PL
	arm_LDREXH_VS
	arm_LDREXH_VC
	arm_LDREXH_HI
	arm_LDREXH_LS
	arm_LDREXH_GE
	arm_LDREXH_LT
	arm_LDREXH_GT
	arm_LDREXH_LE
	arm_LDREXH
	arm_LDREXH_ZZ
	arm_LDRH_EQ
	arm_LDRH_NE
	arm_LDRH_CS
	arm_LDRH_CC
	arm_LDRH_MI
	arm_LDRH_PL
	arm_LDRH_VS
	arm_LDRH_VC
	arm_LDRH_HI
	arm_LDRH_LS
	arm_LDRH_GE
	arm_LDRH_LT
	arm_LDRH_GT
	arm_LDRH_LE
	arm_LDRH
	arm_LDRH_ZZ
	arm_LDRHT_EQ
	arm_LDRHT_NE
	arm_LDRHT_CS
	arm_LDRHT_CC
	arm_LDRHT_MI
	arm_LDRHT_PL
	arm_LDRHT_VS
	arm_LDRHT_VC
	arm_LDRHT_HI
	arm_LDRHT_LS
	arm_LDRHT_GE
	arm_LDRHT_LT
	arm_LDRHT_GT
	arm_LDRHT_LE
	arm_LDRHT
	arm_LDRHT_ZZ
	arm_LDRSB_EQ
	arm_LDRSB_NE
	arm_LDRSB_CS
	arm_LDRSB_CC
	arm_LDRSB_MI
	arm_LDRSB_PL
	arm_LDRSB_VS
	arm_LDRSB_VC
	arm_LDRSB_HI
	arm_LDRSB_LS
	arm_LDRSB_GE
	arm_LDRSB_LT
	arm_LDRSB_GT
	arm_LDRSB_LE
	arm_LDRSB
	arm_LDRSB_ZZ
	arm_LDRSBT_EQ
	arm_LDRSBT_NE
	arm_LDRSBT_CS
	arm_LDRSBT_CC
	arm_LDRSBT_MI
	arm_LDRSBT_PL
	arm_LDRSBT_VS
	arm_LDRSBT_VC
	arm_LDRSBT_HI
	arm_LDRSBT_LS
	arm_LDRSBT_GE
	arm_LDRSBT_LT
	arm_LDRSBT_GT
	arm_LDRSBT_LE
	arm_LDRSBT
	arm_LDRSBT_ZZ
	arm_LDRSH_EQ
	arm_LDRSH_NE
	arm_LDRSH_CS
	arm_LDRSH_CC
	arm_LDRSH_MI
	arm_LDRSH_PL
	arm_LDRSH_VS
	arm_LDRSH_VC
	arm_LDRSH_HI
	arm_LDRSH_LS
	arm_LDRSH_GE
	arm_LDRSH_LT
	arm_LDRSH_GT
	arm_LDRSH_LE
	arm_LDRSH
	arm_LDRSH_ZZ
	arm_LDRSHT_EQ
	arm_LDRSHT_NE
	arm_LDRSHT_CS
	arm_LDRSHT_CC
	arm_LDRSHT_MI
	arm_LDRSHT_PL
	arm_LDRSHT_VS
	arm_LDRSHT_VC
	arm_LDRSHT_HI
	arm_LDRSHT_LS
	arm_LDRSHT_GE
	arm_LDRSHT_LT
	arm_LDRSHT_GT
	arm_LDRSHT_LE
	arm_LDRSHT
	arm_LDRSHT_ZZ
	arm_LDRT_EQ
	arm_LDRT_NE
	arm_LDRT_CS
	arm_LDRT_CC
	arm_LDRT_MI
	arm_LDRT_PL
	arm_LDRT_VS
	arm_LDRT_VC
	arm_LDRT_HI
	arm_LDRT_LS
	arm_LDRT_GE
	arm_LDRT_LT
	arm_LDRT_GT
	arm_LDRT_LE
	arm_LDRT
	arm_LDRT_ZZ
	arm_LSL_EQ
	arm_LSL_NE
	arm_LSL_CS
	arm_LSL_CC
	arm_LSL_MI
	arm_LSL_PL
	arm_LSL_VS
	arm_LSL_VC
	arm_LSL_HI
	arm_LSL_LS
	arm_LSL_GE
	arm_LSL_LT
	arm_LSL_GT
	arm_LSL_LE
	arm_LSL
	arm_LSL_ZZ
	arm_LSL_S_EQ
	arm_LSL_S_NE
	arm_LSL_S_CS
	arm_LSL_S_CC
	arm_LSL_S_MI
	arm_LSL_S_PL
	arm_LSL_S_VS
	arm_LSL_S_VC
	arm_LSL_S_HI
	arm_LSL_S_LS
	arm_LSL_S_GE
	arm_LSL_S_LT
	arm_LSL_S_GT
	arm_LSL_S_LE
	arm_LSL_S
	arm_LSL_S_ZZ
	arm_LSR_EQ
	arm_LSR_NE
	arm_LSR_CS
	arm_LSR_CC
	arm_LSR_MI
	arm_LSR_PL
	arm_LSR_VS
	arm_LSR_VC
	arm_LSR_HI
	arm_LSR_LS
	arm_LSR_GE
	arm_LSR_LT
	arm_LSR_GT
	arm_LSR_LE
	arm_LSR
	arm_LSR_ZZ
	arm_LSR_S_EQ
	arm_LSR_S_NE
	arm_LSR_S_CS
	arm_LSR_S_CC
	arm_LSR_S_MI
	arm_LSR_S_PL
	arm_LSR_S_VS
	arm_LSR_S_VC
	arm_LSR_S_HI
	arm_LSR_S_LS
	arm_LSR_S_GE
	arm_LSR_S_LT
	arm_LSR_S_GT
	arm_LSR_S_LE
	arm_LSR_S
	arm_LSR_S_ZZ
	arm_MLA_EQ
	arm_MLA_NE
	arm_MLA_CS
	arm_MLA_CC
	arm_MLA_MI
	arm_MLA_PL
	arm_MLA_VS
	arm_MLA_VC
	arm_MLA_HI
	arm_MLA_LS
	arm_MLA_GE
	arm_MLA_LT
	arm_MLA_GT
	arm_MLA_LE
	arm_MLA
	arm_MLA_ZZ
	arm_MLA_S_EQ
	arm_MLA_S_NE
	arm_MLA_S_CS
	arm_MLA_S_CC
	arm_MLA_S_MI
	arm_MLA_S_PL
	arm_MLA_S_VS
	arm_MLA_S_VC
	arm_MLA_S_HI
	arm_MLA_S_LS
	arm_MLA_S_GE
	arm_MLA_S_LT
	arm_MLA_S_GT
	arm_MLA_S_LE
	arm_MLA_S
	arm_MLA_S_ZZ
	arm_MLS_EQ
	arm_MLS_NE
	arm_MLS_CS
	arm_MLS_CC
	arm_MLS_MI
	arm_MLS_PL
	arm_MLS_VS
	arm_MLS_VC
	arm_MLS_HI
	arm_MLS_LS
	arm_MLS_GE
	arm_MLS_LT
	arm_MLS_GT
	arm_MLS_LE
	arm_MLS
	arm_MLS_ZZ
	arm_MOV_EQ
	arm_MOV_NE
	arm_MOV_CS
	arm_MOV_CC
	arm_MOV_MI
	arm_MOV_PL
	arm_MOV_VS
	arm_MOV_VC
	arm_MOV_HI
	arm_MOV_LS
	arm_MOV_GE
	arm_MOV_LT
	arm_MOV_GT
	arm_MOV_LE
	arm_MOV
	arm_MOV_ZZ
	arm_MOV_S_EQ
	arm_MOV_S_NE
	arm_MOV_S_CS
	arm_MOV_S_CC
	arm_MOV_S_MI
	arm_MOV_S_PL
	arm_MOV_S_VS
	arm_MOV_S_VC
	arm_MOV_S_HI
	arm_MOV_S_LS
	arm_MOV_S_GE
	arm_MOV_S_LT
	arm_MOV_S_GT
	arm_MOV_S_LE
	arm_MOV_S
	arm_MOV_S_ZZ
	arm_MOVT_EQ
	arm_MOVT_NE
	arm_MOVT_CS
	arm_MOVT_CC
	arm_MOVT_MI
	arm_MOVT_PL
	arm_MOVT_VS
	arm_MOVT_VC
	arm_MOVT_HI
	arm_MOVT_LS
	arm_MOVT_GE
	arm_MOVT_LT
	arm_MOVT_GT
	arm_MOVT_LE
	arm_MOVT
	arm_MOVT_ZZ
	arm_MOVW_EQ
	arm_MOVW_NE
	arm_MOVW_CS
	arm_MOVW_CC
	arm_MOVW_MI
	arm_MOVW_PL
	arm_MOVW_VS
	arm_MOVW_VC
	arm_MOVW_HI
	arm_MOVW_LS
	arm_MOVW_GE
	arm_MOVW_LT
	arm_MOVW_GT
	arm_MOVW_LE
	arm_MOVW
	arm_MOVW_ZZ
	arm_MRS_EQ
	arm_MRS_NE
	arm_MRS_CS
	arm_MRS_CC
	arm_MRS_MI
	arm_MRS_PL
	arm_MRS_VS
	arm_MRS_VC
	arm_MRS_HI
	arm_MRS_LS
	arm_MRS_GE
	arm_MRS_LT
	arm_MRS_GT
	arm_MRS_LE
	arm_MRS
	arm_MRS_ZZ
	arm_MUL_EQ
	arm_MUL_NE
	arm_MUL_CS
	arm_MUL_CC
	arm_MUL_MI
	arm_MUL_PL
	arm_MUL_VS
	arm_MUL_VC
	arm_MUL_HI
	arm_MUL_LS
	arm_MUL_GE
	arm_MUL_LT
	arm_MUL_GT
	arm_MUL_LE
	arm_MUL
	arm_MUL_ZZ
	arm_MUL_S_EQ
	arm_MUL_S_NE
	arm_MUL_S_CS
	arm_MUL_S_CC
	arm_MUL_S_MI
	arm_MUL_S_PL
	arm_MUL_S_VS
	arm_MUL_S_VC
	arm_MUL_S_HI
	arm_MUL_S_LS
	arm_MUL_S_GE
	arm_MUL_S_LT
	arm_MUL_S_GT
	arm_MUL_S_LE
	arm_MUL_S
	arm_MUL_S_ZZ
	arm_MVN_EQ
	arm_MVN_NE
	arm_MVN_CS
	arm_MVN_CC
	arm_MVN_MI
	arm_MVN_PL
	arm_MVN_VS
	arm_MVN_VC
	arm_MVN_HI
	arm_MVN_LS
	arm_MVN_GE
	arm_MVN_LT
	arm_MVN_GT
	arm_MVN_LE
	arm_MVN
	arm_MVN_ZZ
	arm_MVN_S_EQ
	arm_MVN_S_NE
	arm_MVN_S_CS
	arm_MVN_S_CC
	arm_MVN_S_MI
	arm_MVN_S_PL
	arm_MVN_S_VS
	arm_MVN_S_VC
	arm_MVN_S_HI
	arm_MVN_S_LS
	arm_MVN_S_GE
	arm_MVN_S_LT
	arm_MVN_S_GT
	arm_MVN_S_LE
	arm_MVN_S
	arm_MVN_S_ZZ
	arm_NOP_EQ
	arm_NOP_NE
	arm_NOP_CS
	arm_NOP_CC
	arm_NOP_MI
	arm_NOP_PL
	arm_NOP_VS
	arm_NOP_VC
	arm_NOP_HI
	arm_NOP_LS
	arm_NOP_GE
	arm_NOP_LT
	arm_NOP_GT
	arm_NOP_LE
	arm_NOP
	arm_NOP_ZZ
	arm_ORR_EQ
	arm_ORR_NE
	arm_ORR_CS
	arm_ORR_CC
	arm_ORR_MI
	arm_ORR_PL
	arm_ORR_VS
	arm_ORR_VC
	arm_ORR_HI
	arm_ORR_LS
	arm_ORR_GE
	arm_ORR_LT
	arm_ORR_GT
	arm_ORR_LE
	arm_ORR
	arm_ORR_ZZ
	arm_ORR_S_EQ
	arm_ORR_S_NE
	arm_ORR_S_CS
	arm_ORR_S_CC
	arm_ORR_S_MI
	arm_ORR_S_PL
	arm_ORR_S_VS
	arm_ORR_S_VC
	arm_ORR_S_HI
	arm_ORR_S_LS
	arm_ORR_S_GE
	arm_ORR_S_LT
	arm_ORR_S_GT
	arm_ORR_S_LE
	arm_ORR_S
	arm_ORR_S_ZZ
	arm_PKHBT_EQ
	arm_PKHBT_NE
	arm_PKHBT_CS
	arm_PKHBT_CC
	arm_PKHBT_MI
	arm_PKHBT_PL
	arm_PKHBT_VS
	arm_PKHBT_VC
	arm_PKHBT_HI
	arm_PKHBT_LS
	arm_PKHBT_GE
	arm_PKHBT_LT
	arm_PKHBT_GT
	arm_PKHBT_LE
	arm_PKHBT
	arm_PKHBT_ZZ
	arm_PKHTB_EQ
	arm_PKHTB_NE
	arm_PKHTB_CS
	arm_PKHTB_CC
	arm_PKHTB_MI
	arm_PKHTB_PL
	arm_PKHTB_VS
	arm_PKHTB_VC
	arm_PKHTB_HI
	arm_PKHTB_LS
	arm_PKHTB_GE
	arm_PKHTB_LT
	arm_PKHTB_GT
	arm_PKHTB_LE
	arm_PKHTB
	arm_PKHTB_ZZ
	arm_PLD_W
	arm_PLD
	arm_PLI
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_POP_EQ
	arm_POP_NE
	arm_POP_CS
	arm_POP_CC
	arm_POP_MI
	arm_POP_PL
	arm_POP_VS
	arm_POP_VC
	arm_POP_HI
	arm_POP_LS
	arm_POP_GE
	arm_POP_LT
	arm_POP_GT
	arm_POP_LE
	arm_POP
	arm_POP_ZZ
	arm_PUSH_EQ
	arm_PUSH_NE
	arm_PUSH_CS
	arm_PUSH_CC
	arm_PUSH_MI
	arm_PUSH_PL
	arm_PUSH_VS
	arm_PUSH_VC
	arm_PUSH_HI
	arm_PUSH_LS
	arm_PUSH_GE
	arm_PUSH_LT
	arm_PUSH_GT
	arm_PUSH_LE
	arm_PUSH
	arm_PUSH_ZZ
	arm_QADD_EQ
	arm_QADD_NE
	arm_QADD_CS
	arm_QADD_CC
	arm_QADD_MI
	arm_QADD_PL
	arm_QADD_VS
	arm_QADD_VC
	arm_QADD_HI
	arm_QADD_LS
	arm_QADD_GE
	arm_QADD_LT
	arm_QADD_GT
	arm_QADD_LE
	arm_QADD
	arm_QADD_ZZ
	arm_QADD16_EQ
	arm_QADD16_NE
	arm_QADD16_CS
	arm_QADD16_CC
	arm_QADD16_MI
	arm_QADD16_PL
	arm_QADD16_VS
	arm_QADD16_VC
	arm_QADD16_HI
	arm_QADD16_LS
	arm_QADD16_GE
	arm_QADD16_LT
	arm_QADD16_GT
	arm_QADD16_LE
	arm_QADD16
	arm_QADD16_ZZ
	arm_QADD8_EQ
	arm_QADD8_NE
	arm_QADD8_CS
	arm_QADD8_CC
	arm_QADD8_MI
	arm_QADD8_PL
	arm_QADD8_VS
	arm_QADD8_VC
	arm_QADD8_HI
	arm_QADD8_LS
	arm_QADD8_GE
	arm_QADD8_LT
	arm_QADD8_GT
	arm_QADD8_LE
	arm_QADD8
	arm_QADD8_ZZ
	arm_QASX_EQ
	arm_QASX_NE
	arm_QASX_CS
	arm_QASX_CC
	arm_QASX_MI
	arm_QASX_PL
	arm_QASX_VS
	arm_QASX_VC
	arm_QASX_HI
	arm_QASX_LS
	arm_QASX_GE
	arm_QASX_LT
	arm_QASX_GT
	arm_QASX_LE
	arm_QASX
	arm_QASX_ZZ
	arm_QDADD_EQ
	arm_QDADD_NE
	arm_QDADD_CS
	arm_QDADD_CC
	arm_QDADD_MI
	arm_QDADD_PL
	arm_QDADD_VS
	arm_QDADD_VC
	arm_QDADD_HI
	arm_QDADD_LS
	arm_QDADD_GE
	arm_QDADD_LT
	arm_QDADD_GT
	arm_QDADD_LE
	arm_QDADD
	arm_QDADD_ZZ
	arm_QDSUB_EQ
	arm_QDSUB_NE
	arm_QDSUB_CS
	arm_QDSUB_CC
	arm_QDSUB_MI
	arm_QDSUB_PL
	arm_QDSUB_VS
	arm_QDSUB_VC
	arm_QDSUB_HI
	arm_QDSUB_LS
	arm_QDSUB_GE
	arm_QDSUB_LT
	arm_QDSUB_GT
	arm_QDSUB_LE
	arm_QDSUB
	arm_QDSUB_ZZ
	arm_QSAX_EQ
	arm_QSAX_NE
	arm_QSAX_CS
	arm_QSAX_CC
	arm_QSAX_MI
	arm_QSAX_PL
	arm_QSAX_VS
	arm_QSAX_VC
	arm_QSAX_HI
	arm_QSAX_LS
	arm_QSAX_GE
	arm_QSAX_LT
	arm_QSAX_GT
	arm_QSAX_LE
	arm_QSAX
	arm_QSAX_ZZ
	arm_QSUB_EQ
	arm_QSUB_NE
	arm_QSUB_CS
	arm_QSUB_CC
	arm_QSUB_MI
	arm_QSUB_PL
	arm_QSUB_VS
	arm_QSUB_VC
	arm_QSUB_HI
	arm_QSUB_LS
	arm_QSUB_GE
	arm_QSUB_LT
	arm_QSUB_GT
	arm_QSUB_LE
	arm_QSUB
	arm_QSUB_ZZ
	arm_QSUB16_EQ
	arm_QSUB16_NE
	arm_QSUB16_CS
	arm_QSUB16_CC
	arm_QSUB16_MI
	arm_QSUB16_PL
	arm_QSUB16_VS
	arm_QSUB16_VC
	arm_QSUB16_HI
	arm_QSUB16_LS
	arm_QSUB16_GE
	arm_QSUB16_LT
	arm_QSUB16_GT
	arm_QSUB16_LE
	arm_QSUB16
	arm_QSUB16_ZZ
	arm_QSUB8_EQ
	arm_QSUB8_NE
	arm_QSUB8_CS
	arm_QSUB8_CC
	arm_QSUB8_MI
	arm_QSUB8_PL
	arm_QSUB8_VS
	arm_QSUB8_VC
	arm_QSUB8_HI
	arm_QSUB8_LS
	arm_QSUB8_GE
	arm_QSUB8_LT
	arm_QSUB8_GT
	arm_QSUB8_LE
	arm_QSUB8
	arm_QSUB8_ZZ
	arm_RBIT_EQ
	arm_RBIT_NE
	arm_RBIT_CS
	arm_RBIT_CC
	arm_RBIT_MI
	arm_RBIT_PL
	arm_RBIT_VS
	arm_RBIT_VC
	arm_RBIT_HI
	arm_RBIT_LS
	arm_RBIT_GE
	arm_RBIT_LT
	arm_RBIT_GT
	arm_RBIT_LE
	arm_RBIT
	arm_RBIT_ZZ
	arm_REV_EQ
	arm_REV_NE
	arm_REV_CS
	arm_REV_CC
	arm_REV_MI
	arm_REV_PL
	arm_REV_VS
	arm_REV_VC
	arm_REV_HI
	arm_REV_LS
	arm_REV_GE
	arm_REV_LT
	arm_REV_GT
	arm_REV_LE
	arm_REV
	arm_REV_ZZ
	arm_REV16_EQ
	arm_REV16_NE
	arm_REV16_CS
	arm_REV16_CC
	arm_REV16_MI
	arm_REV16_PL
	arm_REV16_VS
	arm_REV16_VC
	arm_REV16_HI
	arm_REV16_LS
	arm_REV16_GE
	arm_REV16_LT
	arm_REV16_GT
	arm_REV16_LE
	arm_REV16
	arm_REV16_ZZ
	arm_REVSH_EQ
	arm_REVSH_NE
	arm_REVSH_CS
	arm_REVSH_CC
	arm_REVSH_MI
	arm_REVSH_PL
	arm_REVSH_VS
	arm_REVSH_VC
	arm_REVSH_HI
	arm_REVSH_LS
	arm_REVSH_GE
	arm_REVSH_LT
	arm_REVSH_GT
	arm_REVSH_LE
	arm_REVSH
	arm_REVSH_ZZ
	arm_ROR_EQ
	arm_ROR_NE
	arm_ROR_CS
	arm_ROR_CC
	arm_ROR_MI
	arm_ROR_PL
	arm_ROR_VS
	arm_ROR_VC
	arm_ROR_HI
	arm_ROR_LS
	arm_ROR_GE
	arm_ROR_LT
	arm_ROR_GT
	arm_ROR_LE
	arm_ROR
	arm_ROR_ZZ
	arm_ROR_S_EQ
	arm_ROR_S_NE
	arm_ROR_S_CS
	arm_ROR_S_CC
	arm_ROR_S_MI
	arm_ROR_S_PL
	arm_ROR_S_VS
	arm_ROR_S_VC
	arm_ROR_S_HI
	arm_ROR_S_LS
	arm_ROR_S_GE
	arm_ROR_S_LT
	arm_ROR_S_GT
	arm_ROR_S_LE
	arm_ROR_S
	arm_ROR_S_ZZ
	arm_RRX_EQ
	arm_RRX_NE
	arm_RRX_CS
	arm_RRX_CC
	arm_RRX_MI
	arm_RRX_PL
	arm_RRX_VS
	arm_RRX_VC
	arm_RRX_HI
	arm_RRX_LS
	arm_RRX_GE
	arm_RRX_LT
	arm_RRX_GT
	arm_RRX_LE
	arm_RRX
	arm_RRX_ZZ
	arm_RRX_S_EQ
	arm_RRX_S_NE
	arm_RRX_S_CS
	arm_RRX_S_CC
	arm_RRX_S_MI
	arm_RRX_S_PL
	arm_RRX_S_VS
	arm_RRX_S_VC
	arm_RRX_S_HI
	arm_RRX_S_LS
	arm_RRX_S_GE
	arm_RRX_S_LT
	arm_RRX_S_GT
	arm_RRX_S_LE
	arm_RRX_S
	arm_RRX_S_ZZ
	arm_RSB_EQ
	arm_RSB_NE
	arm_RSB_CS
	arm_RSB_CC
	arm_RSB_MI
	arm_RSB_PL
	arm_RSB_VS
	arm_RSB_VC
	arm_RSB_HI
	arm_RSB_LS
	arm_RSB_GE
	arm_RSB_LT
	arm_RSB_GT
	arm_RSB_LE
	arm_RSB
	arm_RSB_ZZ
	arm_RSB_S_EQ
	arm_RSB_S_NE
	arm_RSB_S_CS
	arm_RSB_S_CC
	arm_RSB_S_MI
	arm_RSB_S_PL
	arm_RSB_S_VS
	arm_RSB_S_VC
	arm_RSB_S_HI
	arm_RSB_S_LS
	arm_RSB_S_GE
	arm_RSB_S_LT
	arm_RSB_S_GT
	arm_RSB_S_LE
	arm_RSB_S
	arm_RSB_S_ZZ
	arm_RSC_EQ
	arm_RSC_NE
	arm_RSC_CS
	arm_RSC_CC
	arm_RSC_MI
	arm_RSC_PL
	arm_RSC_VS
	arm_RSC_VC
	arm_RSC_HI
	arm_RSC_LS
	arm_RSC_GE
	arm_RSC_LT
	arm_RSC_GT
	arm_RSC_LE
	arm_RSC
	arm_RSC_ZZ
	arm_RSC_S_EQ
	arm_RSC_S_NE
	arm_RSC_S_CS
	arm_RSC_S_CC
	arm_RSC_S_MI
	arm_RSC_S_PL
	arm_RSC_S_VS
	arm_RSC_S_VC
	arm_RSC_S_HI
	arm_RSC_S_LS
	arm_RSC_S_GE
	arm_RSC_S_LT
	arm_RSC_S_GT
	arm_RSC_S_LE
	arm_RSC_S
	arm_RSC_S_ZZ
	arm_SADD16_EQ
	arm_SADD16_NE
	arm_SADD16_CS
	arm_SADD16_CC
	arm_SADD16_MI
	arm_SADD16_PL
	arm_SADD16_VS
	arm_SADD16_VC
	arm_SADD16_HI
	arm_SADD16_LS
	arm_SADD16_GE
	arm_SADD16_LT
	arm_SADD16_GT
	arm_SADD16_LE
	arm_SADD16
	arm_SADD16_ZZ
	arm_SADD8_EQ
	arm_SADD8_NE
	arm_SADD8_CS
	arm_SADD8_CC
	arm_SADD8_MI
	arm_SADD8_PL
	arm_SADD8_VS
	arm_SADD8_VC
	arm_SADD8_HI
	arm_SADD8_LS
	arm_SADD8_GE
	arm_SADD8_LT
	arm_SADD8_GT
	arm_SADD8_LE
	arm_SADD8
	arm_SADD8_ZZ
	arm_SASX_EQ
	arm_SASX_NE
	arm_SASX_CS
	arm_SASX_CC
	arm_SASX_MI
	arm_SASX_PL
	arm_SASX_VS
	arm_SASX_VC
	arm_SASX_HI
	arm_SASX_LS
	arm_SASX_GE
	arm_SASX_LT
	arm_SASX_GT
	arm_SASX_LE
	arm_SASX
	arm_SASX_ZZ
	arm_SBC_EQ
	arm_SBC_NE
	arm_SBC_CS
	arm_SBC_CC
	arm_SBC_MI
	arm_SBC_PL
	arm_SBC_VS
	arm_SBC_VC
	arm_SBC_HI
	arm_SBC_LS
	arm_SBC_GE
	arm_SBC_LT
	arm_SBC_GT
	arm_SBC_LE
	arm_SBC
	arm_SBC_ZZ
	arm_SBC_S_EQ
	arm_SBC_S_NE
	arm_SBC_S_CS
	arm_SBC_S_CC
	arm_SBC_S_MI
	arm_SBC_S_PL
	arm_SBC_S_VS
	arm_SBC_S_VC
	arm_SBC_S_HI
	arm_SBC_S_LS
	arm_SBC_S_GE
	arm_SBC_S_LT
	arm_SBC_S_GT
	arm_SBC_S_LE
	arm_SBC_S
	arm_SBC_S_ZZ
	arm_SBFX_EQ
	arm_SBFX_NE
	arm_SBFX_CS
	arm_SBFX_CC
	arm_SBFX_MI
	arm_SBFX_PL
	arm_SBFX_VS
	arm_SBFX_VC
	arm_SBFX_HI
	arm_SBFX_LS
	arm_SBFX_GE
	arm_SBFX_LT
	arm_SBFX_GT
	arm_SBFX_LE
	arm_SBFX
	arm_SBFX_ZZ
	arm_SEL_EQ
	arm_SEL_NE
	arm_SEL_CS
	arm_SEL_CC
	arm_SEL_MI
	arm_SEL_PL
	arm_SEL_VS
	arm_SEL_VC
	arm_SEL_HI
	arm_SEL_LS
	arm_SEL_GE
	arm_SEL_LT
	arm_SEL_GT
	arm_SEL_LE
	arm_SEL
	arm_SEL_ZZ
	arm_SETEND
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_SEV_EQ
	arm_SEV_NE
	arm_SEV_CS
	arm_SEV_CC
	arm_SEV_MI
	arm_SEV_PL
	arm_SEV_VS
	arm_SEV_VC
	arm_SEV_HI
	arm_SEV_LS
	arm_SEV_GE
	arm_SEV_LT
	arm_SEV_GT
	arm_SEV_LE
	arm_SEV
	arm_SEV_ZZ
	arm_SHADD16_EQ
	arm_SHADD16_NE
	arm_SHADD16_CS
	arm_SHADD16_CC
	arm_SHADD16_MI
	arm_SHADD16_PL
	arm_SHADD16_VS
	arm_SHADD16_VC
	arm_SHADD16_HI
	arm_SHADD16_LS
	arm_SHADD16_GE
	arm_SHADD16_LT
	arm_SHADD16_GT
	arm_SHADD16_LE
	arm_SHADD16
	arm_SHADD16_ZZ
	arm_SHADD8_EQ
	arm_SHADD8_NE
	arm_SHADD8_CS
	arm_SHADD8_CC
	arm_SHADD8_MI
	arm_SHADD8_PL
	arm_SHADD8_VS
	arm_SHADD8_VC
	arm_SHADD8_HI
	arm_SHADD8_LS
	arm_SHADD8_GE
	arm_SHADD8_LT
	arm_SHADD8_GT
	arm_SHADD8_LE
	arm_SHADD8
	arm_SHADD8_ZZ
	arm_SHASX_EQ
	arm_SHASX_NE
	arm_SHASX_CS
	arm_SHASX_CC
	arm_SHASX_MI
	arm_SHASX_PL
	arm_SHASX_VS
	arm_SHASX_VC
	arm_SHASX_HI
	arm_SHASX_LS
	arm_SHASX_GE
	arm_SHASX_LT
	arm_SHASX_GT
	arm_SHASX_LE
	arm_SHASX
	arm_SHASX_ZZ
	arm_SHSAX_EQ
	arm_SHSAX_NE
	arm_SHSAX_CS
	arm_SHSAX_CC
	arm_SHSAX_MI
	arm_SHSAX_PL
	arm_SHSAX_VS
	arm_SHSAX_VC
	arm_SHSAX_HI
	arm_SHSAX_LS
	arm_SHSAX_GE
	arm_SHSAX_LT
	arm_SHSAX_GT
	arm_SHSAX_LE
	arm_SHSAX
	arm_SHSAX_ZZ
	arm_SHSUB16_EQ
	arm_SHSUB16_NE
	arm_SHSUB16_CS
	arm_SHSUB16_CC
	arm_SHSUB16_MI
	arm_SHSUB16_PL
	arm_SHSUB16_VS
	arm_SHSUB16_VC
	arm_SHSUB16_HI
	arm_SHSUB16_LS
	arm_SHSUB16_GE
	arm_SHSUB16_LT
	arm_SHSUB16_GT
	arm_SHSUB16_LE
	arm_SHSUB16
	arm_SHSUB16_ZZ
	arm_SHSUB8_EQ
	arm_SHSUB8_NE
	arm_SHSUB8_CS
	arm_SHSUB8_CC
	arm_SHSUB8_MI
	arm_SHSUB8_PL
	arm_SHSUB8_VS
	arm_SHSUB8_VC
	arm_SHSUB8_HI
	arm_SHSUB8_LS
	arm_SHSUB8_GE
	arm_SHSUB8_LT
	arm_SHSUB8_GT
	arm_SHSUB8_LE
	arm_SHSUB8
	arm_SHSUB8_ZZ
	arm_SMLABB_EQ
	arm_SMLABB_NE
	arm_SMLABB_CS
	arm_SMLABB_CC
	arm_SMLABB_MI
	arm_SMLABB_PL
	arm_SMLABB_VS
	arm_SMLABB_VC
	arm_SMLABB_HI
	arm_SMLABB_LS
	arm_SMLABB_GE
	arm_SMLABB_LT
	arm_SMLABB_GT
	arm_SMLABB_LE
	arm_SMLABB
	arm_SMLABB_ZZ
	arm_SMLABT_EQ
	arm_SMLABT_NE
	arm_SMLABT_CS
	arm_SMLABT_CC
	arm_SMLABT_MI
	arm_SMLABT_PL
	arm_SMLABT_VS
	arm_SMLABT_VC
	arm_SMLABT_HI
	arm_SMLABT_LS
	arm_SMLABT_GE
	arm_SMLABT_LT
	arm_SMLABT_GT
	arm_SMLABT_LE
	arm_SMLABT
	arm_SMLABT_ZZ
	arm_SMLATB_EQ
	arm_SMLATB_NE
	arm_SMLATB_CS
	arm_SMLATB_CC
	arm_SMLATB_MI
	arm_SMLATB_PL
	arm_SMLATB_VS
	arm_SMLATB_VC
	arm_SMLATB_HI
	arm_SMLATB_LS
	arm_SMLATB_GE
	arm_SMLATB_LT
	arm_SMLATB_GT
	arm_SMLATB_LE
	arm_SMLATB
	arm_SMLATB_ZZ
	arm_SMLATT_EQ
	arm_SMLATT_NE
	arm_SMLATT_CS
	arm_SMLATT_CC
	arm_SMLATT_MI
	arm_SMLATT_PL
	arm_SMLATT_VS
	arm_SMLATT_VC
	arm_SMLATT_HI
	arm_SMLATT_LS
	arm_SMLATT_GE
	arm_SMLATT_LT
	arm_SMLATT_GT
	arm_SMLATT_LE
	arm_SMLATT
	arm_SMLATT_ZZ
	arm_SMLAD_EQ
	arm_SMLAD_NE
	arm_SMLAD_CS
	arm_SMLAD_CC
	arm_SMLAD_MI
	arm_SMLAD_PL
	arm_SMLAD_VS
	arm_SMLAD_VC
	arm_SMLAD_HI
	arm_SMLAD_LS
	arm_SMLAD_GE
	arm_SMLAD_LT
	arm_SMLAD_GT
	arm_SMLAD_LE
	arm_SMLAD
	arm_SMLAD_ZZ
	arm_SMLAD_X_EQ
	arm_SMLAD_X_NE
	arm_SMLAD_X_CS
	arm_SMLAD_X_CC
	arm_SMLAD_X_MI
	arm_SMLAD_X_PL
	arm_SMLAD_X_VS
	arm_SMLAD_X_VC
	arm_SMLAD_X_HI
	arm_SMLAD_X_LS
	arm_SMLAD_X_GE
	arm_SMLAD_X_LT
	arm_SMLAD_X_GT
	arm_SMLAD_X_LE
	arm_SMLAD_X
	arm_SMLAD_X_ZZ
	arm_SMLAL_EQ
	arm_SMLAL_NE
	arm_SMLAL_CS
	arm_SMLAL_CC
	arm_SMLAL_MI
	arm_SMLAL_PL
	arm_SMLAL_VS
	arm_SMLAL_VC
	arm_SMLAL_HI
	arm_SMLAL_LS
	arm_SMLAL_GE
	arm_SMLAL_LT
	arm_SMLAL_GT
	arm_SMLAL_LE
	arm_SMLAL
	arm_SMLAL_ZZ
	arm_SMLAL_S_EQ
	arm_SMLAL_S_NE
	arm_SMLAL_S_CS
	arm_SMLAL_S_CC
	arm_SMLAL_S_MI
	arm_SMLAL_S_PL
	arm_SMLAL_S_VS
	arm_SMLAL_S_VC
	arm_SMLAL_S_HI
	arm_SMLAL_S_LS
	arm_SMLAL_S_GE
	arm_SMLAL_S_LT
	arm_SMLAL_S_GT
	arm_SMLAL_S_LE
	arm_SMLAL_S
	arm_SMLAL_S_ZZ
	arm_SMLALBB_EQ
	arm_SMLALBB_NE
	arm_SMLALBB_CS
	arm_SMLALBB_CC
	arm_SMLALBB_MI
	arm_SMLALBB_PL
	arm_SMLALBB_VS
	arm_SMLALBB_VC
	arm_SMLALBB_HI
	arm_SMLALBB_LS
	arm_SMLALBB_GE
	arm_SMLALBB_LT
	arm_SMLALBB_GT
	arm_SMLALBB_LE
	arm_SMLALBB
	arm_SMLALBB_ZZ
	arm_SMLALBT_EQ
	arm_SMLALBT_NE
	arm_SMLALBT_CS
	arm_SMLALBT_CC
	arm_SMLALBT_MI
	arm_SMLALBT_PL
	arm_SMLALBT_VS
	arm_SMLALBT_VC
	arm_SMLALBT_HI
	arm_SMLALBT_LS
	arm_SMLALBT_GE
	arm_SMLALBT_LT
	arm_SMLALBT_GT
	arm_SMLALBT_LE
	arm_SMLALBT
	arm_SMLALBT_ZZ
	arm_SMLALTB_EQ
	arm_SMLALTB_NE
	arm_SMLALTB_CS
	arm_SMLALTB_CC
	arm_SMLALTB_MI
	arm_SMLALTB_PL
	arm_SMLALTB_VS
	arm_SMLALTB_VC
	arm_SMLALTB_HI
	arm_SMLALTB_LS
	arm_SMLALTB_GE
	arm_SMLALTB_LT
	arm_SMLALTB_GT
	arm_SMLALTB_LE
	arm_SMLALTB
	arm_SMLALTB_ZZ
	arm_SMLALTT_EQ
	arm_SMLALTT_NE
	arm_SMLALTT_CS
	arm_SMLALTT_CC
	arm_SMLALTT_MI
	arm_SMLALTT_PL
	arm_SMLALTT_VS
	arm_SMLALTT_VC
	arm_SMLALTT_HI
	arm_SMLALTT_LS
	arm_SMLALTT_GE
	arm_SMLALTT_LT
	arm_SMLALTT_GT
	arm_SMLALTT_LE
	arm_SMLALTT
	arm_SMLALTT_ZZ
	arm_SMLALD_EQ
	arm_SMLALD_NE
	arm_SMLALD_CS
	arm_SMLALD_CC
	arm_SMLALD_MI
	arm_SMLALD_PL
	arm_SMLALD_VS
	arm_SMLALD_VC
	arm_SMLALD_HI
	arm_SMLALD_LS
	arm_SMLALD_GE
	arm_SMLALD_LT
	arm_SMLALD_GT
	arm_SMLALD_LE
	arm_SMLALD
	arm_SMLALD_ZZ
	arm_SMLALD_X_EQ
	arm_SMLALD_X_NE
	arm_SMLALD_X_CS
	arm_SMLALD_X_CC
	arm_SMLALD_X_MI
	arm_SMLALD_X_PL
	arm_SMLALD_X_VS
	arm_SMLALD_X_VC
	arm_SMLALD_X_HI
	arm_SMLALD_X_LS
	arm_SMLALD_X_GE
	arm_SMLALD_X_LT
	arm_SMLALD_X_GT
	arm_SMLALD_X_LE
	arm_SMLALD_X
	arm_SMLALD_X_ZZ
	arm_SMLAWB_EQ
	arm_SMLAWB_NE
	arm_SMLAWB_CS
	arm_SMLAWB_CC
	arm_SMLAWB_MI
	arm_SMLAWB_PL
	arm_SMLAWB_VS
	arm_SMLAWB_VC
	arm_SMLAWB_HI
	arm_SMLAWB_LS
	arm_SMLAWB_GE
	arm_SMLAWB_LT
	arm_SMLAWB_GT
	arm_SMLAWB_LE
	arm_SMLAWB
	arm_SMLAWB_ZZ
	arm_SMLAWT_EQ
	arm_SMLAWT_NE
	arm_SMLAWT_CS
	arm_SMLAWT_CC
	arm_SMLAWT_MI
	arm_SMLAWT_PL
	arm_SMLAWT_VS
	arm_SMLAWT_VC
	arm_SMLAWT_HI
	arm_SMLAWT_LS
	arm_SMLAWT_GE
	arm_SMLAWT_LT
	arm_SMLAWT_GT
	arm_SMLAWT_LE
	arm_SMLAWT
	arm_SMLAWT_ZZ
	arm_SMLSD_EQ
	arm_SMLSD_NE
	arm_SMLSD_CS
	arm_SMLSD_CC
	arm_SMLSD_MI
	arm_SMLSD_PL
	arm_SMLSD_VS
	arm_SMLSD_VC
	arm_SMLSD_HI
	arm_SMLSD_LS
	arm_SMLSD_GE
	arm_SMLSD_LT
	arm_SMLSD_GT
	arm_SMLSD_LE
	arm_SMLSD
	arm_SMLSD_ZZ
	arm_SMLSD_X_EQ
	arm_SMLSD_X_NE
	arm_SMLSD_X_CS
	arm_SMLSD_X_CC
	arm_SMLSD_X_MI
	arm_SMLSD_X_PL
	arm_SMLSD_X_VS
	arm_SMLSD_X_VC
	arm_SMLSD_X_HI
	arm_SMLSD_X_LS
	arm_SMLSD_X_GE
	arm_SMLSD_X_LT
	arm_SMLSD_X_GT
	arm_SMLSD_X_LE
	arm_SMLSD_X
	arm_SMLSD_X_ZZ
	arm_SMLSLD_EQ
	arm_SMLSLD_NE
	arm_SMLSLD_CS
	arm_SMLSLD_CC
	arm_SMLSLD_MI
	arm_SMLSLD_PL
	arm_SMLSLD_VS
	arm_SMLSLD_VC
	arm_SMLSLD_HI
	arm_SMLSLD_LS
	arm_SMLSLD_GE
	arm_SMLSLD_LT
	arm_SMLSLD_GT
	arm_SMLSLD_LE
	arm_SMLSLD
	arm_SMLSLD_ZZ
	arm_SMLSLD_X_EQ
	arm_SMLSLD_X_NE
	arm_SMLSLD_X_CS
	arm_SMLSLD_X_CC
	arm_SMLSLD_X_MI
	arm_SMLSLD_X_PL
	arm_SMLSLD_X_VS
	arm_SMLSLD_X_VC
	arm_SMLSLD_X_HI
	arm_SMLSLD_X_LS
	arm_SMLSLD_X_GE
	arm_SMLSLD_X_LT
	arm_SMLSLD_X_GT
	arm_SMLSLD_X_LE
	arm_SMLSLD_X
	arm_SMLSLD_X_ZZ
	arm_SMMLA_EQ
	arm_SMMLA_NE
	arm_SMMLA_CS
	arm_SMMLA_CC
	arm_SMMLA_MI
	arm_SMMLA_PL
	arm_SMMLA_VS
	arm_SMMLA_VC
	arm_SMMLA_HI
	arm_SMMLA_LS
	arm_SMMLA_GE
	arm_SMMLA_LT
	arm_SMMLA_GT
	arm_SMMLA_LE
	arm_SMMLA
	arm_SMMLA_ZZ
	arm_SMMLA_R_EQ
	arm_SMMLA_R_NE
	arm_SMMLA_R_CS
	arm_SMMLA_R_CC
	arm_SMMLA_R_MI
	arm_SMMLA_R_PL
	arm_SMMLA_R_VS
	arm_SMMLA_R_VC
	arm_SMMLA_R_HI
	arm_SMMLA_R_LS
	arm_SMMLA_R_GE
	arm_SMMLA_R_LT
	arm_SMMLA_R_GT
	arm_SMMLA_R_LE
	arm_SMMLA_R
	arm_SMMLA_R_ZZ
	arm_SMMLS_EQ
	arm_SMMLS_NE
	arm_SMMLS_CS
	arm_SMMLS_CC
	arm_SMMLS_MI
	arm_SMMLS_PL
	arm_SMMLS_VS
	arm_SMMLS_VC
	arm_SMMLS_HI
	arm_SMMLS_LS
	arm_SMMLS_GE
	arm_SMMLS_LT
	arm_SMMLS_GT
	arm_SMMLS_LE
	arm_SMMLS
	arm_SMMLS_ZZ
	arm_SMMLS_R_EQ
	arm_SMMLS_R_NE
	arm_SMMLS_R_CS
	arm_SMMLS_R_CC
	arm_SMMLS_R_MI
	arm_SMMLS_R_PL
	arm_SMMLS_R_VS
	arm_SMMLS_R_VC
	arm_SMMLS_R_HI
	arm_SMMLS_R_LS
	arm_SMMLS_R_GE
	arm_SMMLS_R_LT
	arm_SMMLS_R_GT
	arm_SMMLS_R_LE
	arm_SMMLS_R
	arm_SMMLS_R_ZZ
	arm_SMMUL_EQ
	arm_SMMUL_NE
	arm_SMMUL_CS
	arm_SMMUL_CC
	arm_SMMUL_MI
	arm_SMMUL_PL
	arm_SMMUL_VS
	arm_SMMUL_VC
	arm_SMMUL_HI
	arm_SMMUL_LS
	arm_SMMUL_GE
	arm_SMMUL_LT
	arm_SMMUL_GT
	arm_SMMUL_LE
	arm_SMMUL
	arm_SMMUL_ZZ
	arm_SMMUL_R_EQ
	arm_SMMUL_R_NE
	arm_SMMUL_R_CS
	arm_SMMUL_R_CC
	arm_SMMUL_R_MI
	arm_SMMUL_R_PL
	arm_SMMUL_R_VS
	arm_SMMUL_R_VC
	arm_SMMUL_R_HI
	arm_SMMUL_R_LS
	arm_SMMUL_R_GE
	arm_SMMUL_R_LT
	arm_SMMUL_R_GT
	arm_SMMUL_R_LE
	arm_SMMUL_R
	arm_SMMUL_R_ZZ
	arm_SMUAD_EQ
	arm_SMUAD_NE
	arm_SMUAD_CS
	arm_SMUAD_CC
	arm_SMUAD_MI
	arm_SMUAD_PL
	arm_SMUAD_VS
	arm_SMUAD_VC
	arm_SMUAD_HI
	arm_SMUAD_LS
	arm_SMUAD_GE
	arm_SMUAD_LT
	arm_SMUAD_GT
	arm_SMUAD_LE
	arm_SMUAD
	arm_SMUAD_ZZ
	arm_SMUAD_X_EQ
	arm_SMUAD_X_NE
	arm_SMUAD_X_CS
	arm_SMUAD_X_CC
	arm_SMUAD_X_MI
	arm_SMUAD_X_PL
	arm_SMUAD_X_VS
	arm_SMUAD_X_VC
	arm_SMUAD_X_HI
	arm_SMUAD_X_LS
	arm_SMUAD_X_GE
	arm_SMUAD_X_LT
	arm_SMUAD_X_GT
	arm_SMUAD_X_LE
	arm_SMUAD_X
	arm_SMUAD_X_ZZ
	arm_SMULBB_EQ
	arm_SMULBB_NE
	arm_SMULBB_CS
	arm_SMULBB_CC
	arm_SMULBB_MI
	arm_SMULBB_PL
	arm_SMULBB_VS
	arm_SMULBB_VC
	arm_SMULBB_HI
	arm_SMULBB_LS
	arm_SMULBB_GE
	arm_SMULBB_LT
	arm_SMULBB_GT
	arm_SMULBB_LE
	arm_SMULBB
	arm_SMULBB_ZZ
	arm_SMULBT_EQ
	arm_SMULBT_NE
	arm_SMULBT_CS
	arm_SMULBT_CC
	arm_SMULBT_MI
	arm_SMULBT_PL
	arm_SMULBT_VS
	arm_SMULBT_VC
	arm_SMULBT_HI
	arm_SMULBT_LS
	arm_SMULBT_GE
	arm_SMULBT_LT
	arm_SMULBT_GT
	arm_SMULBT_LE
	arm_SMULBT
	arm_SMULBT_ZZ
	arm_SMULTB_EQ
	arm_SMULTB_NE
	arm_SMULTB_CS
	arm_SMULTB_CC
	arm_SMULTB_MI
	arm_SMULTB_PL
	arm_SMULTB_VS
	arm_SMULTB_VC
	arm_SMULTB_HI
	arm_SMULTB_LS
	arm_SMULTB_GE
	arm_SMULTB_LT
	arm_SMULTB_GT
	arm_SMULTB_LE
	arm_SMULTB
	arm_SMULTB_ZZ
	arm_SMULTT_EQ
	arm_SMULTT_NE
	arm_SMULTT_CS
	arm_SMULTT_CC
	arm_SMULTT_MI
	arm_SMULTT_PL
	arm_SMULTT_VS
	arm_SMULTT_VC
	arm_SMULTT_HI
	arm_SMULTT_LS
	arm_SMULTT_GE
	arm_SMULTT_LT
	arm_SMULTT_GT
	arm_SMULTT_LE
	arm_SMULTT
	arm_SMULTT_ZZ
	arm_SMULL_EQ
	arm_SMULL_NE
	arm_SMULL_CS
	arm_SMULL_CC
	arm_SMULL_MI
	arm_SMULL_PL
	arm_SMULL_VS
	arm_SMULL_VC
	arm_SMULL_HI
	arm_SMULL_LS
	arm_SMULL_GE
	arm_SMULL_LT
	arm_SMULL_GT
	arm_SMULL_LE
	arm_SMULL
	arm_SMULL_ZZ
	arm_SMULL_S_EQ
	arm_SMULL_S_NE
	arm_SMULL_S_CS
	arm_SMULL_S_CC
	arm_SMULL_S_MI
	arm_SMULL_S_PL
	arm_SMULL_S_VS
	arm_SMULL_S_VC
	arm_SMULL_S_HI
	arm_SMULL_S_LS
	arm_SMULL_S_GE
	arm_SMULL_S_LT
	arm_SMULL_S_GT
	arm_SMULL_S_LE
	arm_SMULL_S
	arm_SMULL_S_ZZ
	arm_SMULWB_EQ
	arm_SMULWB_NE
	arm_SMULWB_CS
	arm_SMULWB_CC
	arm_SMULWB_MI
	arm_SMULWB_PL
	arm_SMULWB_VS
	arm_SMULWB_VC
	arm_SMULWB_HI
	arm_SMULWB_LS
	arm_SMULWB_GE
	arm_SMULWB_LT
	arm_SMULWB_GT
	arm_SMULWB_LE
	arm_SMULWB
	arm_SMULWB_ZZ
	arm_SMULWT_EQ
	arm_SMULWT_NE
	arm_SMULWT_CS
	arm_SMULWT_CC
	arm_SMULWT_MI
	arm_SMULWT_PL
	arm_SMULWT_VS
	arm_SMULWT_VC
	arm_SMULWT_HI
	arm_SMULWT_LS
	arm_SMULWT_GE
	arm_SMULWT_LT
	arm_SMULWT_GT
	arm_SMULWT_LE
	arm_SMULWT
	arm_SMULWT_ZZ
	arm_SMUSD_EQ
	arm_SMUSD_NE
	arm_SMUSD_CS
	arm_SMUSD_CC
	arm_SMUSD_MI
	arm_SMUSD_PL
	arm_SMUSD_VS
	arm_SMUSD_VC
	arm_SMUSD_HI
	arm_SMUSD_LS
	arm_SMUSD_GE
	arm_SMUSD_LT
	arm_SMUSD_GT
	arm_SMUSD_LE
	arm_SMUSD
	arm_SMUSD_ZZ
	arm_SMUSD_X_EQ
	arm_SMUSD_X_NE
	arm_SMUSD_X_CS
	arm_SMUSD_X_CC
	arm_SMUSD_X_MI
	arm_SMUSD_X_PL
	arm_SMUSD_X_VS
	arm_SMUSD_X_VC
	arm_SMUSD_X_HI
	arm_SMUSD_X_LS
	arm_SMUSD_X_GE
	arm_SMUSD_X_LT
	arm_SMUSD_X_GT
	arm_SMUSD_X_LE
	arm_SMUSD_X
	arm_SMUSD_X_ZZ
	arm_SSAT_EQ
	arm_SSAT_NE
	arm_SSAT_CS
	arm_SSAT_CC
	arm_SSAT_MI
	arm_SSAT_PL
	arm_SSAT_VS
	arm_SSAT_VC
	arm_SSAT_HI
	arm_SSAT_LS
	arm_SSAT_GE
	arm_SSAT_LT
	arm_SSAT_GT
	arm_SSAT_LE
	arm_SSAT
	arm_SSAT_ZZ
	arm_SSAT16_EQ
	arm_SSAT16_NE
	arm_SSAT16_CS
	arm_SSAT16_CC
	arm_SSAT16_MI
	arm_SSAT16_PL
	arm_SSAT16_VS
	arm_SSAT16_VC
	arm_SSAT16_HI
	arm_SSAT16_LS
	arm_SSAT16_GE
	arm_SSAT16_LT
	arm_SSAT16_GT
	arm_SSAT16_LE
	arm_SSAT16
	arm_SSAT16_ZZ
	arm_SSAX_EQ
	arm_SSAX_NE
	arm_SSAX_CS
	arm_SSAX_CC
	arm_SSAX_MI
	arm_SSAX_PL
	arm_SSAX_VS
	arm_SSAX_VC
	arm_SSAX_HI
	arm_SSAX_LS
	arm_SSAX_GE
	arm_SSAX_LT
	arm_SSAX_GT
	arm_SSAX_LE
	arm_SSAX
	arm_SSAX_ZZ
	arm_SSUB16_EQ
	arm_SSUB16_NE
	arm_SSUB16_CS
	arm_SSUB16_CC
	arm_SSUB16_MI
	arm_SSUB16_PL
	arm_SSUB16_VS
	arm_SSUB16_VC
	arm_SSUB16_HI
	arm_SSUB16_LS
	arm_SSUB16_GE
	arm_SSUB16_LT
	arm_SSUB16_GT
	arm_SSUB16_LE
	arm_SSUB16
	arm_SSUB16_ZZ
	arm_SSUB8_EQ
	arm_SSUB8_NE
	arm_SSUB8_CS
	arm_SSUB8_CC
	arm_SSUB8_MI
	arm_SSUB8_PL
	arm_SSUB8_VS
	arm_SSUB8_VC
	arm_SSUB8_HI
	arm_SSUB8_LS
	arm_SSUB8_GE
	arm_SSUB8_LT
	arm_SSUB8_GT
	arm_SSUB8_LE
	arm_SSUB8
	arm_SSUB8_ZZ
	arm_STM_EQ
	arm_STM_NE
	arm_STM_CS
	arm_STM_CC
	arm_STM_MI
	arm_STM_PL
	arm_STM_VS
	arm_STM_VC
	arm_STM_HI
	arm_STM_LS
	arm_STM_GE
	arm_STM_LT
	arm_STM_GT
	arm_STM_LE
	arm_STM
	arm_STM_ZZ
	arm_STMDA_EQ
	arm_STMDA_NE
	arm_STMDA_CS
	arm_STMDA_CC
	arm_STMDA_MI
	arm_STMDA_PL
	arm_STMDA_VS
	arm_STMDA_VC
	arm_STMDA_HI
	arm_STMDA_LS
	arm_STMDA_GE
	arm_STMDA_LT
	arm_STMDA_GT
	arm_STMDA_LE
	arm_STMDA
	arm_STMDA_ZZ
	arm_STMDB_EQ
	arm_STMDB_NE
	arm_STMDB_CS
	arm_STMDB_CC
	arm_STMDB_MI
	arm_STMDB_PL
	arm_STMDB_VS
	arm_STMDB_VC
	arm_STMDB_HI
	arm_STMDB_LS
	arm_STMDB_GE
	arm_STMDB_LT
	arm_STMDB_GT
	arm_STMDB_LE
	arm_STMDB
	arm_STMDB_ZZ
	arm_STMIB_EQ
	arm_STMIB_NE
	arm_STMIB_CS
	arm_STMIB_CC
	arm_STMIB_MI
	arm_STMIB_PL
	arm_STMIB_VS
	arm_STMIB_VC
	arm_STMIB_HI
	arm_STMIB_LS
	arm_STMIB_GE
	arm_STMIB_LT
	arm_STMIB_GT
	arm_STMIB_LE
	arm_STMIB
	arm_STMIB_ZZ
	arm_STR_EQ
	arm_STR_NE
	arm_STR_CS
	arm_STR_CC
	arm_STR_MI
	arm_STR_PL
	arm_STR_VS
	arm_STR_VC
	arm_STR_HI
	arm_STR_LS
	arm_STR_GE
	arm_STR_LT
	arm_STR_GT
	arm_STR_LE
	arm_STR
	arm_STR_ZZ
	arm_STRB_EQ
	arm_STRB_NE
	arm_STRB_CS
	arm_STRB_CC
	arm_STRB_MI
	arm_STRB_PL
	arm_STRB_VS
	arm_STRB_VC
	arm_STRB_HI
	arm_STRB_LS
	arm_STRB_GE
	arm_STRB_LT
	arm_STRB_GT
	arm_STRB_LE
	arm_STRB
	arm_STRB_ZZ
	arm_STRBT_EQ
	arm_STRBT_NE
	arm_STRBT_CS
	arm_STRBT_CC
	arm_STRBT_MI
	arm_STRBT_PL
	arm_STRBT_VS
	arm_STRBT_VC
	arm_STRBT_HI
	arm_STRBT_LS
	arm_STRBT_GE
	arm_STRBT_LT
	arm_STRBT_GT
	arm_STRBT_LE
	arm_STRBT
	arm_STRBT_ZZ
	arm_STRD_EQ
	arm_STRD_NE
	arm_STRD_CS
	arm_STRD_CC
	arm_STRD_MI
	arm_STRD_PL
	arm_STRD_VS
	arm_STRD_VC
	arm_STRD_HI
	arm_STRD_LS
	arm_STRD_GE
	arm_STRD_LT
	arm_STRD_GT
	arm_STRD_LE
	arm_STRD
	arm_STRD_ZZ
	arm_STREX_EQ
	arm_STREX_NE
	arm_STREX_CS
	arm_STREX_CC
	arm_STREX_MI
	arm_STREX_PL
	arm_STREX_VS
	arm_STREX_VC
	arm_STREX_HI
	arm_STREX_LS
	arm_STREX_GE
	arm_STREX_LT
	arm_STREX_GT
	arm_STREX_LE
	arm_STREX
	arm_STREX_ZZ
	arm_STREXB_EQ
	arm_STREXB_NE
	arm_STREXB_CS
	arm_STREXB_CC
	arm_STREXB_MI
	arm_STREXB_PL
	arm_STREXB_VS
	arm_STREXB_VC
	arm_STREXB_HI
	arm_STREXB_LS
	arm_STREXB_GE
	arm_STREXB_LT
	arm_STREXB_GT
	arm_STREXB_LE
	arm_STREXB
	arm_STREXB_ZZ
	arm_STREXD_EQ
	arm_STREXD_NE
	arm_STREXD_CS
	arm_STREXD_CC
	arm_STREXD_MI
	arm_STREXD_PL
	arm_STREXD_VS
	arm_STREXD_VC
	arm_STREXD_HI
	arm_STREXD_LS
	arm_STREXD_GE
	arm_STREXD_LT
	arm_STREXD_GT
	arm_STREXD_LE
	arm_STREXD
	arm_STREXD_ZZ
	arm_STREXH_EQ
	arm_STREXH_NE
	arm_STREXH_CS
	arm_STREXH_CC
	arm_STREXH_MI
	arm_STREXH_PL
	arm_STREXH_VS
	arm_STREXH_VC
	arm_STREXH_HI
	arm_STREXH_LS
	arm_STREXH_GE
	arm_STREXH_LT
	arm_STREXH_GT
	arm_STREXH_LE
	arm_STREXH
	arm_STREXH_ZZ
	arm_STRH_EQ
	arm_STRH_NE
	arm_STRH_CS
	arm_STRH_CC
	arm_STRH_MI
	arm_STRH_PL
	arm_STRH_VS
	arm_STRH_VC
	arm_STRH_HI
	arm_STRH_LS
	arm_STRH_GE
	arm_STRH_LT
	arm_STRH_GT
	arm_STRH_LE
	arm_STRH
	arm_STRH_ZZ
	arm_STRHT_EQ
	arm_STRHT_NE
	arm_STRHT_CS
	arm_STRHT_CC
	arm_STRHT_MI
	arm_STRHT_PL
	arm_STRHT_VS
	arm_STRHT_VC
	arm_STRHT_HI
	arm_STRHT_LS
	arm_STRHT_GE
	arm_STRHT_LT
	arm_STRHT_GT
	arm_STRHT_LE
	arm_STRHT
	arm_STRHT_ZZ
	arm_STRT_EQ
	arm_STRT_NE
	arm_STRT_CS
	arm_STRT_CC
	arm_STRT_MI
	arm_STRT_PL
	arm_STRT_VS
	arm_STRT_VC
	arm_STRT_HI
	arm_STRT_LS
	arm_STRT_GE
	arm_STRT_LT
	arm_STRT_GT
	arm_STRT_LE
	arm_STRT
	arm_STRT_ZZ
	arm_SUB_EQ
	arm_SUB_NE
	arm_SUB_CS
	arm_SUB_CC
	arm_SUB_MI
	arm_SUB_PL
	arm_SUB_VS
	arm_SUB_VC
	arm_SUB_HI
	arm_SUB_LS
	arm_SUB_GE
	arm_SUB_LT
	arm_SUB_GT
	arm_SUB_LE
	arm_SUB
	arm_SUB_ZZ
	arm_SUB_S_EQ
	arm_SUB_S_NE
	arm_SUB_S_CS
	arm_SUB_S_CC
	arm_SUB_S_MI
	arm_SUB_S_PL
	arm_SUB_S_VS
	arm_SUB_S_VC
	arm_SUB_S_HI
	arm_SUB_S_LS
	arm_SUB_S_GE
	arm_SUB_S_LT
	arm_SUB_S_GT
	arm_SUB_S_LE
	arm_SUB_S
	arm_SUB_S_ZZ
	arm_SVC_EQ
	arm_SVC_NE
	arm_SVC_CS
	arm_SVC_CC
	arm_SVC_MI
	arm_SVC_PL
	arm_SVC_VS
	arm_SVC_VC
	arm_SVC_HI
	arm_SVC_LS
	arm_SVC_GE
	arm_SVC_LT
	arm_SVC_GT
	arm_SVC_LE
	arm_SVC
	arm_SVC_ZZ
	arm_SWP_EQ
	arm_SWP_NE
	arm_SWP_CS
	arm_SWP_CC
	arm_SWP_MI
	arm_SWP_PL
	arm_SWP_VS
	arm_SWP_VC
	arm_SWP_HI
	arm_SWP_LS
	arm_SWP_GE
	arm_SWP_LT
	arm_SWP_GT
	arm_SWP_LE
	arm_SWP
	arm_SWP_ZZ
	arm_SWP_B_EQ
	arm_SWP_B_NE
	arm_SWP_B_CS
	arm_SWP_B_CC
	arm_SWP_B_MI
	arm_SWP_B_PL
	arm_SWP_B_VS
	arm_SWP_B_VC
	arm_SWP_B_HI
	arm_SWP_B_LS
	arm_SWP_B_GE
	arm_SWP_B_LT
	arm_SWP_B_GT
	arm_SWP_B_LE
	arm_SWP_B
	arm_SWP_B_ZZ
	arm_SXTAB_EQ
	arm_SXTAB_NE
	arm_SXTAB_CS
	arm_SXTAB_CC
	arm_SXTAB_MI
	arm_SXTAB_PL
	arm_SXTAB_VS
	arm_SXTAB_VC
	arm_SXTAB_HI
	arm_SXTAB_LS
	arm_SXTAB_GE
	arm_SXTAB_LT
	arm_SXTAB_GT
	arm_SXTAB_LE
	arm_SXTAB
	arm_SXTAB_ZZ
	arm_SXTAB16_EQ
	arm_SXTAB16_NE
	arm_SXTAB16_CS
	arm_SXTAB16_CC
	arm_SXTAB16_MI
	arm_SXTAB16_PL
	arm_SXTAB16_VS
	arm_SXTAB16_VC
	arm_SXTAB16_HI
	arm_SXTAB16_LS
	arm_SXTAB16_GE
	arm_SXTAB16_LT
	arm_SXTAB16_GT
	arm_SXTAB16_LE
	arm_SXTAB16
	arm_SXTAB16_ZZ
	arm_SXTAH_EQ
	arm_SXTAH_NE
	arm_SXTAH_CS
	arm_SXTAH_CC
	arm_SXTAH_MI
	arm_SXTAH_PL
	arm_SXTAH_VS
	arm_SXTAH_VC
	arm_SXTAH_HI
	arm_SXTAH_LS
	arm_SXTAH_GE
	arm_SXTAH_LT
	arm_SXTAH_GT
	arm_SXTAH_LE
	arm_SXTAH
	arm_SXTAH_ZZ
	arm_SXTB_EQ
	arm_SXTB_NE
	arm_SXTB_CS
	arm_SXTB_CC
	arm_SXTB_MI
	arm_SXTB_PL
	arm_SXTB_VS
	arm_SXTB_VC
	arm_SXTB_HI
	arm_SXTB_LS
	arm_SXTB_GE
	arm_SXTB_LT
	arm_SXTB_GT
	arm_SXTB_LE
	arm_SXTB
	arm_SXTB_ZZ
	arm_SXTB16_EQ
	arm_SXTB16_NE
	arm_SXTB16_CS
	arm_SXTB16_CC
	arm_SXTB16_MI
	arm_SXTB16_PL
	arm_SXTB16_VS
	arm_SXTB16_VC
	arm_SXTB16_HI
	arm_SXTB16_LS
	arm_SXTB16_GE
	arm_SXTB16_LT
	arm_SXTB16_GT
	arm_SXTB16_LE
	arm_SXTB16
	arm_SXTB16_ZZ
	arm_SXTH_EQ
	arm_SXTH_NE
	arm_SXTH_CS
	arm_SXTH_CC
	arm_SXTH_MI
	arm_SXTH_PL
	arm_SXTH_VS
	arm_SXTH_VC
	arm_SXTH_HI
	arm_SXTH_LS
	arm_SXTH_GE
	arm_SXTH_LT
	arm_SXTH_GT
	arm_SXTH_LE
	arm_SXTH
	arm_SXTH_ZZ
	arm_TEQ_EQ
	arm_TEQ_NE
	arm_TEQ_CS
	arm_TEQ_CC
	arm_TEQ_MI
	arm_TEQ_PL
	arm_TEQ_VS
	arm_TEQ_VC
	arm_TEQ_HI
	arm_TEQ_LS
	arm_TEQ_GE
	arm_TEQ_LT
	arm_TEQ_GT
	arm_TEQ_LE
	arm_TEQ
	arm_TEQ_ZZ
	arm_TST_EQ
	arm_TST_NE
	arm_TST_CS
	arm_TST_CC
	arm_TST_MI
	arm_TST_PL
	arm_TST_VS
	arm_TST_VC
	arm_TST_HI
	arm_TST_LS
	arm_TST_GE
	arm_TST_LT
	arm_TST_GT
	arm_TST_LE
	arm_TST
	arm_TST_ZZ
	arm_UADD16_EQ
	arm_UADD16_NE
	arm_UADD16_CS
	arm_UADD16_CC
	arm_UADD16_MI
	arm_UADD16_PL
	arm_UADD16_VS
	arm_UADD16_VC
	arm_UADD16_HI
	arm_UADD16_LS
	arm_UADD16_GE
	arm_UADD16_LT
	arm_UADD16_GT
	arm_UADD16_LE
	arm_UADD16
	arm_UADD16_ZZ
	arm_UADD8_EQ
	arm_UADD8_NE
	arm_UADD8_CS
	arm_UADD8_CC
	arm_UADD8_MI
	arm_UADD8_PL
	arm_UADD8_VS
	arm_UADD8_VC
	arm_UADD8_HI
	arm_UADD8_LS
	arm_UADD8_GE
	arm_UADD8_LT
	arm_UADD8_GT
	arm_UADD8_LE
	arm_UADD8
	arm_UADD8_ZZ
	arm_UASX_EQ
	arm_UASX_NE
	arm_UASX_CS
	arm_UASX_CC
	arm_UASX_MI
	arm_UASX_PL
	arm_UASX_VS
	arm_UASX_VC
	arm_UASX_HI
	arm_UASX_LS
	arm_UASX_GE
	arm_UASX_LT
	arm_UASX_GT
	arm_UASX_LE
	arm_UASX
	arm_UASX_ZZ
	arm_UBFX_EQ
	arm_UBFX_NE
	arm_UBFX_CS
	arm_UBFX_CC
	arm_UBFX_MI
	arm_UBFX_PL
	arm_UBFX_VS
	arm_UBFX_VC
	arm_UBFX_HI
	arm_UBFX_LS
	arm_UBFX_GE
	arm_UBFX_LT
	arm_UBFX_GT
	arm_UBFX_LE
	arm_UBFX
	arm_UBFX_ZZ
	arm_UHADD16_EQ
	arm_UHADD16_NE
	arm_UHADD16_CS
	arm_UHADD16_CC
	arm_UHADD16_MI
	arm_UHADD16_PL
	arm_UHADD16_VS
	arm_UHADD16_VC
	arm_UHADD16_HI
	arm_UHADD16_LS
	arm_UHADD16_GE
	arm_UHADD16_LT
	arm_UHADD16_GT
	arm_UHADD16_LE
	arm_UHADD16
	arm_UHADD16_ZZ
	arm_UHADD8_EQ
	arm_UHADD8_NE
	arm_UHADD8_CS
	arm_UHADD8_CC
	arm_UHADD8_MI
	arm_UHADD8_PL
	arm_UHADD8_VS
	arm_UHADD8_VC
	arm_UHADD8_HI
	arm_UHADD8_LS
	arm_UHADD8_GE
	arm_UHADD8_LT
	arm_UHADD8_GT
	arm_UHADD8_LE
	arm_UHADD8
	arm_UHADD8_ZZ
	arm_UHASX_EQ
	arm_UHASX_NE
	arm_UHASX_CS
	arm_UHASX_CC
	arm_UHASX_MI
	arm_UHASX_PL
	arm_UHASX_VS
	arm_UHASX_VC
	arm_UHASX_HI
	arm_UHASX_LS
	arm_UHASX_GE
	arm_UHASX_LT
	arm_UHASX_GT
	arm_UHASX_LE
	arm_UHASX
	arm_UHASX_ZZ
	arm_UHSAX_EQ
	arm_UHSAX_NE
	arm_UHSAX_CS
	arm_UHSAX_CC
	arm_UHSAX_MI
	arm_UHSAX_PL
	arm_UHSAX_VS
	arm_UHSAX_VC
	arm_UHSAX_HI
	arm_UHSAX_LS
	arm_UHSAX_GE
	arm_UHSAX_LT
	arm_UHSAX_GT
	arm_UHSAX_LE
	arm_UHSAX
	arm_UHSAX_ZZ
	arm_UHSUB16_EQ
	arm_UHSUB16_NE
	arm_UHSUB16_CS
	arm_UHSUB16_CC
	arm_UHSUB16_MI
	arm_UHSUB16_PL
	arm_UHSUB16_VS
	arm_UHSUB16_VC
	arm_UHSUB16_HI
	arm_UHSUB16_LS
	arm_UHSUB16_GE
	arm_UHSUB16_LT
	arm_UHSUB16_GT
	arm_UHSUB16_LE
	arm_UHSUB16
	arm_UHSUB16_ZZ
	arm_UHSUB8_EQ
	arm_UHSUB8_NE
	arm_UHSUB8_CS
	arm_UHSUB8_CC
	arm_UHSUB8_MI
	arm_UHSUB8_PL
	arm_UHSUB8_VS
	arm_UHSUB8_VC
	arm_UHSUB8_HI
	arm_UHSUB8_LS
	arm_UHSUB8_GE
	arm_UHSUB8_LT
	arm_UHSUB8_GT
	arm_UHSUB8_LE
	arm_UHSUB8
	arm_UHSUB8_ZZ
	arm_UMAAL_EQ
	arm_UMAAL_NE
	arm_UMAAL_CS
	arm_UMAAL_CC
	arm_UMAAL_MI
	arm_UMAAL_PL
	arm_UMAAL_VS
	arm_UMAAL_VC
	arm_UMAAL_HI
	arm_UMAAL_LS
	arm_UMAAL_GE
	arm_UMAAL_LT
	arm_UMAAL_GT
	arm_UMAAL_LE
	arm_UMAAL
	arm_UMAAL_ZZ
	arm_UMLAL_EQ
	arm_UMLAL_NE
	arm_UMLAL_CS
	arm_UMLAL_CC
	arm_UMLAL_MI
	arm_UMLAL_PL
	arm_UMLAL_VS
	arm_UMLAL_VC
	arm_UMLAL_HI
	arm_UMLAL_LS
	arm_UMLAL_GE
	arm_UMLAL_LT
	arm_UMLAL_GT
	arm_UMLAL_LE
	arm_UMLAL
	arm_UMLAL_ZZ
	arm_UMLAL_S_EQ
	arm_UMLAL_S_NE
	arm_UMLAL_S_CS
	arm_UMLAL_S_CC
	arm_UMLAL_S_MI
	arm_UMLAL_S_PL
	arm_UMLAL_S_VS
	arm_UMLAL_S_VC
	arm_UMLAL_S_HI
	arm_UMLAL_S_LS
	arm_UMLAL_S_GE
	arm_UMLAL_S_LT
	arm_UMLAL_S_GT
	arm_UMLAL_S_LE
	arm_UMLAL_S
	arm_UMLAL_S_ZZ
	arm_UMULL_EQ
	arm_UMULL_NE
	arm_UMULL_CS
	arm_UMULL_CC
	arm_UMULL_MI
	arm_UMULL_PL
	arm_UMULL_VS
	arm_UMULL_VC
	arm_UMULL_HI
	arm_UMULL_LS
	arm_UMULL_GE
	arm_UMULL_LT
	arm_UMULL_GT
	arm_UMULL_LE
	arm_UMULL
	arm_UMULL_ZZ
	arm_UMULL_S_EQ
	arm_UMULL_S_NE
	arm_UMULL_S_CS
	arm_UMULL_S_CC
	arm_UMULL_S_MI
	arm_UMULL_S_PL
	arm_UMULL_S_VS
	arm_UMULL_S_VC
	arm_UMULL_S_HI
	arm_UMULL_S_LS
	arm_UMULL_S_GE
	arm_UMULL_S_LT
	arm_UMULL_S_GT
	arm_UMULL_S_LE
	arm_UMULL_S
	arm_UMULL_S_ZZ
	arm_UNDEF
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	_
	arm_UQADD16_EQ
	arm_UQADD16_NE
	arm_UQADD16_CS
	arm_UQADD16_CC
	arm_UQADD16_MI
	arm_UQADD16_PL
	arm_UQADD16_VS
	arm_UQADD16_VC
	arm_UQADD16_HI
	arm_UQADD16_LS
	arm_UQADD16_GE
	arm_UQADD16_LT
	arm_UQADD16_GT
	arm_UQADD16_LE
	arm_UQADD16
	arm_UQADD16_ZZ
	arm_UQADD8_EQ
	arm_UQADD8_NE
	arm_UQADD8_CS
	arm_UQADD8_CC
	arm_UQADD8_MI
	arm_UQADD8_PL
	arm_UQADD8_VS
	arm_UQADD8_VC
	arm_UQADD8_HI
	arm_UQADD8_LS
	arm_UQADD8_GE
	arm_UQADD8_LT
	arm_UQADD8_GT
	arm_UQADD8_LE
	arm_UQADD8
	arm_UQADD8_ZZ
	arm_UQASX_EQ
	arm_UQASX_NE
	arm_UQASX_CS
	arm_UQASX_CC
	arm_UQASX_MI
	arm_UQASX_PL
	arm_UQASX_VS
	arm_UQASX_VC
	arm_UQASX_HI
	arm_UQASX_LS
	arm_UQASX_GE
	arm_UQASX_LT
	arm_UQASX_GT
	arm_UQASX_LE
	arm_UQASX
	arm_UQASX_ZZ
	arm_UQSAX_EQ
	arm_UQSAX_NE
	arm_UQSAX_CS
	arm_UQSAX_CC
	arm_UQSAX_MI
	arm_UQSAX_PL
	arm_UQSAX_VS
	arm_UQSAX_VC
	arm_UQSAX_HI
	arm_UQSAX_LS
	arm_UQSAX_GE
	arm_UQSAX_LT
	arm_UQSAX_GT
	arm_UQSAX_LE
	arm_UQSAX
	arm_UQSAX_ZZ
	arm_UQSUB16_EQ
	arm_UQSUB16_NE
	arm_UQSUB16_CS
	arm_UQSUB16_CC
	arm_UQSUB16_MI
	arm_UQSUB16_PL
	arm_UQSUB16_VS
	arm_UQSUB16_VC
	arm_UQSUB16_HI
	arm_UQSUB16_LS
	arm_UQSUB16_GE
	arm_UQSUB16_LT
	arm_UQSUB16_GT
	arm_UQSUB16_LE
	arm_UQSUB16
	arm_UQSUB16_ZZ
	arm_UQSUB8_EQ
	arm_UQSUB8_NE
	arm_UQSUB8_CS
	arm_UQSUB8_CC
	arm_UQSUB8_MI
	arm_UQSUB8_PL
	arm_UQSUB8_VS
	arm_UQSUB8_VC
	arm_UQSUB8_HI
	arm_UQSUB8_LS
	arm_UQSUB8_GE
	arm_UQSUB8_LT
	arm_UQSUB8_GT
	arm_UQSUB8_LE
	arm_UQSUB8
	arm_UQSUB8_ZZ
	arm_USAD8_EQ
	arm_USAD8_NE
	arm_USAD8_CS
	arm_USAD8_CC
	arm_USAD8_MI
	arm_USAD8_PL
	arm_USAD8_VS
	arm_USAD8_VC
	arm_USAD8_HI
	arm_USAD8_LS
	arm_USAD8_GE
	arm_USAD8_LT
	arm_USAD8_GT
	arm_USAD8_LE
	arm_USAD8
	arm_USAD8_ZZ
	arm_USADA8_EQ
	arm_USADA8_NE
	arm_USADA8_CS
	arm_USADA8_CC
	arm_USADA8_MI
	arm_USADA8_PL
	arm_USADA8_VS
	arm_USADA8_VC
	arm_USADA8_HI
	arm_USADA8_LS
	arm_USADA8_GE
	arm_USADA8_LT
	arm_USADA8_GT
	arm_USADA8_LE
	arm_USADA8
	arm_USADA8_ZZ
	arm_USAT_EQ
	arm_USAT_NE
	arm_USAT_CS
	arm_USAT_CC
	arm_USAT_MI
	arm_USAT_PL
	arm_USAT_VS
	arm_USAT_VC
	arm_USAT_HI
	arm_USAT_LS
	arm_USAT_GE
	arm_USAT_LT
	arm_USAT_GT
	arm_USAT_LE
	arm_USAT
	arm_USAT_ZZ
	arm_USAT16_EQ
	arm_USAT16_NE
	arm_USAT16_CS
	arm_USAT16_CC
	arm_USAT16_MI
	arm_USAT16_PL
	arm_USAT16_VS
	arm_USAT16_VC
	arm_USAT16_HI
	arm_USAT16_LS
	arm_USAT16_GE
	arm_USAT16_LT
	arm_USAT16_GT
	arm_USAT16_LE
	arm_USAT16
	arm_USAT16_ZZ
	arm_USAX_EQ
	arm_USAX_NE
	arm_USAX_CS
	arm_USAX_CC
	arm_USAX_MI
	arm_USAX_PL
	arm_USAX_VS
	arm_USAX_VC
	arm_USAX_HI
	arm_USAX_LS
	arm_USAX_GE
	arm_USAX_LT
	arm_USAX_GT
	arm_USAX_LE
	arm_USAX
	arm_USAX_ZZ
	arm_USUB16_EQ
	arm_USUB16_NE
	arm_USUB16_CS
	arm_USUB16_CC
	arm_USUB16_MI
	arm_USUB16_PL
	arm_USUB16_VS
	arm_USUB16_VC
	arm_USUB16_HI
	arm_USUB16_LS
	arm_USUB16_GE
	arm_USUB16_LT
	arm_USUB16_GT
	arm_USUB16_LE
	arm_USUB16
	arm_USUB16_ZZ
	arm_USUB8_EQ
	arm_USUB8_NE
	arm_USUB8_CS
	arm_USUB8_CC
	arm_USUB8_MI
	arm_USUB8_PL
	arm_USUB8_VS
	arm_USUB8_VC
	arm_USUB8_HI
	arm_USUB8_LS
	arm_USUB8_GE
	arm_USUB8_LT
	arm_USUB8_GT
	arm_USUB8_LE
	arm_USUB8
	arm_USUB8_ZZ
	arm_UXTAB_EQ
	arm_UXTAB_NE
	arm_UXTAB_CS
	arm_UXTAB_CC
	arm_UXTAB_MI
	arm_UXTAB_PL
	arm_UXTAB_VS
	arm_UXTAB_VC
	arm_UXTAB_HI
	arm_UXTAB_LS
	arm_UXTAB_GE
	arm_UXTAB_LT
	arm_UXTAB_GT
	arm_UXTAB_LE
	arm_UXTAB
	arm_UXTAB_ZZ
	arm_UXTAB16_EQ
	arm_UXTAB16_NE
	arm_UXTAB16_CS
	arm_UXTAB16_CC
	arm_UXTAB16_MI
	arm_UXTAB16_PL
	arm_UXTAB16_VS
	arm_UXTAB16_VC
	arm_UXTAB16_HI
	arm_UXTAB16_LS
	arm_UXTAB16_GE
	arm_UXTAB16_LT
	arm_UXTAB16_GT
	arm_UXTAB16_LE
	arm_UXTAB16
	arm_UXTAB16_ZZ
	arm_UXTAH_EQ
	arm_UXTAH_NE
	arm_UXTAH_CS
	arm_UXTAH_CC
	arm_UXTAH_MI
	arm_UXTAH_PL
	arm_UXTAH_VS
	arm_UXTAH_VC
	arm_UXTAH_HI
	arm_UXTAH_LS
	arm_UXTAH_GE
	arm_UXTAH_LT
	arm_UXTAH_GT
	arm_UXTAH_LE
	arm_UXTAH
	arm_UXTAH_ZZ
	arm_UXTB_EQ
	arm_UXTB_NE
	arm_UXTB_CS
	arm_UXTB_CC
	arm_UXTB_MI
	arm_UXTB_PL
	arm_UXTB_VS
	arm_UXTB_VC
	arm_UXTB_HI
	arm_UXTB_LS
	arm_UXTB_GE
	arm_UXTB_LT
	arm_UXTB_GT
	arm_UXTB_LE
	arm_UXTB
	arm_UXTB_ZZ
	arm_UXTB16_EQ
	arm_UXTB16_NE
	arm_UXTB16_CS
	arm_UXTB16_CC
	arm_UXTB16_MI
	arm_UXTB16_PL
	arm_UXTB16_VS
	arm_UXTB16_VC
	arm_UXTB16_HI
	arm_UXTB16_LS
	arm_UXTB16_GE
	arm_UXTB16_LT
	arm_UXTB16_GT
	arm_UXTB16_LE
	arm_UXTB16
	arm_UXTB16_ZZ
	arm_UXTH_EQ
	arm_UXTH_NE
	arm_UXTH_CS
	arm_UXTH_CC
	arm_UXTH_MI
	arm_UXTH_PL
	arm_UXTH_VS
	arm_UXTH_VC
	arm_UXTH_HI
	arm_UXTH_LS
	arm_UXTH_GE
	arm_UXTH_LT
	arm_UXTH_GT
	arm_UXTH_LE
	arm_UXTH
	arm_UXTH_ZZ
	arm_VABS_EQ_F32
	arm_VABS_NE_F32
	arm_VABS_CS_F32
	arm_VABS_CC_F32
	arm_VABS_MI_F32
	arm_VABS_PL_F32
	arm_VABS_VS_F32
	arm_VABS_VC_F32
	arm_VABS_HI_F32
	arm_VABS_LS_F32
	arm_VABS_GE_F32
	arm_VABS_LT_F32
	arm_VABS_GT_F32
	arm_VABS_LE_F32
	arm_VABS_F32
	arm_VABS_ZZ_F32
	arm_VABS_EQ_F64
	arm_VABS_NE_F64
	arm_VABS_CS_F64
	arm_VABS_CC_F64
	arm_VABS_MI_F64
	arm_VABS_PL_F64
	arm_VABS_VS_F64
	arm_VABS_VC_F64
	arm_VABS_HI_F64
	arm_VABS_LS_F64
	arm_VABS_GE_F64
	arm_VABS_LT_F64
	arm_VABS_GT_F64
	arm_VABS_LE_F64
	arm_VABS_F64
	arm_VABS_ZZ_F64
	arm_VADD_EQ_F32
	arm_VADD_NE_F32
	arm_VADD_CS_F32
	arm_VADD_CC_F32
	arm_VADD_MI_F32
	arm_VADD_PL_F32
	arm_VADD_VS_F32
	arm_VADD_VC_F32
	arm_VADD_HI_F32
	arm_VADD_LS_F32
	arm_VADD_GE_F32
	arm_VADD_LT_F32
	arm_VADD_GT_F32
	arm_VADD_LE_F32
	arm_VADD_F32
	arm_VADD_ZZ_F32
	arm_VADD_EQ_F64
	arm_VADD_NE_F64
	arm_VADD_CS_F64
	arm_VADD_CC_F64
	arm_VADD_MI_F64
	arm_VADD_PL_F64
	arm_VADD_VS_F64
	arm_VADD_VC_F64
	arm_VADD_HI_F64
	arm_VADD_LS_F64
	arm_VADD_GE_F64
	arm_VADD_LT_F64
	arm_VADD_GT_F64
	arm_VADD_LE_F64
	arm_VADD_F64
	arm_VADD_ZZ_F64
	arm_VCMP_EQ_F32
	arm_VCMP_NE_F32
	arm_VCMP_CS_F32
	arm_VCMP_CC_F32
	arm_VCMP_MI_F32
	arm_VCMP_PL_F32
	arm_VCMP_VS_F32
	arm_VCMP_VC_F32
	arm_VCMP_HI_F32
	arm_VCMP_LS_F32
	arm_VCMP_GE_F32
	arm_VCMP_LT_F32
	arm_VCMP_GT_F32
	arm_VCMP_LE_F32
	arm_VCMP_F32
	arm_VCMP_ZZ_F32
	arm_VCMP_EQ_F64
	arm_VCMP_NE_F64
	arm_VCMP_CS_F64
	arm_VCMP_CC_F64
	arm_VCMP_MI_F64
	arm_VCMP_PL_F64
	arm_VCMP_VS_F64
	arm_VCMP_VC_F64
	arm_VCMP_HI_F64
	arm_VCMP_LS_F64
	arm_VCMP_GE_F64
	arm_VCMP_LT_F64
	arm_VCMP_GT_F64
	arm_VCMP_LE_F64
	arm_VCMP_F64
	arm_VCMP_ZZ_F64
	arm_VCMP_E_EQ_F32
	arm_VCMP_E_NE_F32
	arm_VCMP_E_CS_F32
	arm_VCMP_E_CC_F32
	arm_VCMP_E_MI_F32
	arm_VCMP_E_PL_F32
	arm_VCMP_E_VS_F32
	arm_VCMP_E_VC_F32
	arm_VCMP_E_HI_F32
	arm_VCMP_E_LS_F32
	arm_VCMP_E_GE_F32
	arm_VCMP_E_LT_F32
	arm_VCMP_E_GT_F32
	arm_VCMP_E_LE_F32
	arm_VCMP_E_F32
	arm_VCMP_E_ZZ_F32
	arm_VCMP_E_EQ_F64
	arm_VCMP_E_NE_F64
	arm_VCMP_E_CS_F64
	arm_VCMP_E_CC_F64
	arm_VCMP_E_MI_F64
	arm_VCMP_E_PL_F64
	arm_VCMP_E_VS_F64
	arm_VCMP_E_VC_F64
	arm_VCMP_E_HI_F64
	arm_VCMP_E_LS_F64
	arm_VCMP_E_GE_F64
	arm_VCMP_E_LT_F64
	arm_VCMP_E_GT_F64
	arm_VCMP_E_LE_F64
	arm_VCMP_E_F64
	arm_VCMP_E_ZZ_F64
	arm_VCVT_EQ_F32_FXS16
	arm_VCVT_NE_F32_FXS16
	arm_VCVT_CS_F32_FXS16
	arm_VCVT_CC_F32_FXS16
	arm_VCVT_MI_F32_FXS16
	arm_VCVT_PL_F32_FXS16
	arm_VCVT_VS_F32_FXS16
	arm_VCVT_VC_F32_FXS16
	arm_VCVT_HI_F32_FXS16
	arm_VCVT_LS_F32_FXS16
	arm_VCVT_GE_F32_FXS16
	arm_VCVT_LT_F32_FXS16
	arm_VCVT_GT_F32_FXS16
	arm_VCVT_LE_F32_FXS16
	arm_VCVT_F32_FXS16
	arm_VCVT_ZZ_F32_FXS16
	arm_VCVT_EQ_F32_FXS32
	arm_VCVT_NE_F32_FXS32
	arm_VCVT_CS_F32_FXS32
	arm_VCVT_CC_F32_FXS32
	arm_VCVT_MI_F32_FXS32
	arm_VCVT_PL_F32_FXS32
	arm_VCVT_VS_F32_FXS32
	arm_VCVT_VC_F32_FXS32
	arm_VCVT_HI_F32_FXS32
	arm_VCVT_LS_F32_FXS32
	arm_VCVT_GE_F32_FXS32
	arm_VCVT_LT_F32_FXS32
	arm_VCVT_GT_F32_FXS32
	arm_VCVT_LE_F32_FXS32
	arm_VCVT_F32_FXS32
	arm_VCVT_ZZ_F32_FXS32
	arm_VCVT_EQ_F32_FXU16
	arm_VCVT_NE_F32_FXU16
	arm_VCVT_CS_F32_FXU16
	arm_VCVT_CC_F32_FXU16
	arm_VCVT_MI_F32_FXU16
	arm_VCVT_PL_F32_FXU16
	arm_VCVT_VS_F32_FXU16
	arm_VCVT_VC_F32_FXU16
	arm_VCVT_HI_F32_FXU16
	arm_VCVT_LS_F32_FXU16
	arm_VCVT_GE_F32_FXU16
	arm_VCVT_LT_F32_FXU16
	arm_VCVT_GT_F32_FXU16
	arm_VCVT_LE_F32_FXU16
	arm_VCVT_F32_FXU16
	arm_VCVT_ZZ_F32_FXU16
	arm_VCVT_EQ_F32_FXU32
	arm_VCVT_NE_F32_FXU32
	arm_VCVT_CS_F32_FXU32
	arm_VCVT_CC_F32_FXU32
	arm_VCVT_MI_F32_FXU32
	arm_VCVT_PL_F32_FXU32
	arm_VCVT_VS_F32_FXU32
	arm_VCVT_VC_F32_FXU32
	arm_VCVT_HI_F32_FXU32
	arm_VCVT_LS_F32_FXU32
	arm_VCVT_GE_F32_FXU32
	arm_VCVT_LT_F32_FXU32
	arm_VCVT_GT_F32_FXU32
	arm_VCVT_LE_F32_FXU32
	arm_VCVT_F32_FXU32
	arm_VCVT_ZZ_F32_FXU32
	arm_VCVT_EQ_F64_FXS16
	arm_VCVT_NE_F64_FXS16
	arm_VCVT_CS_F64_FXS16
	arm_VCVT_CC_F64_FXS16
	arm_VCVT_MI_F64_FXS16
	arm_VCVT_PL_F64_FXS16
	arm_VCVT_VS_F64_FXS16
	arm_VCVT_VC_F64_FXS16
	arm_VCVT_HI_F64_FXS16
	arm_VCVT_LS_F64_FXS16
	arm_VCVT_GE_F64_FXS16
	arm_VCVT_LT_F64_FXS16
	arm_VCVT_GT_F64_FXS16
	arm_VCVT_LE_F64_FXS16
	arm_VCVT_F64_FXS16
	arm_VCVT_ZZ_F64_FXS16
	arm_VCVT_EQ_F64_FXS32
	arm_VCVT_NE_F64_FXS32
	arm_VCVT_CS_F64_FXS32
	arm_VCVT_CC_F64_FXS32
	arm_VCVT_MI_F64_FXS32
	arm_VCVT_PL_F64_FXS32
	arm_VCVT_VS_F64_FXS32
	arm_VCVT_VC_F64_FXS32
	arm_VCVT_HI_F64_FXS32
	arm_VCVT_LS_F64_FXS32
	arm_VCVT_GE_F64_FXS32
	arm_VCVT_LT_F64_FXS32
	arm_VCVT_GT_F64_FXS32
	arm_VCVT_LE_F64_FXS32
	arm_VCVT_F64_FXS32
	arm_VCVT_ZZ_F64_FXS32
	arm_VCVT_EQ_F64_FXU16
	arm_VCVT_NE_F64_FXU16
	arm_VCVT_CS_F64_FXU16
	arm_VCVT_CC_F64_FXU16
	arm_VCVT_MI_F64_FXU16
	arm_VCVT_PL_F64_FXU16
	arm_VCVT_VS_F64_FXU16
	arm_VCVT_VC_F64_FXU16
	arm_VCVT_HI_F64_FXU16
	arm_VCVT_LS_F64_FXU16
	arm_VCVT_GE_F64_FXU16
	arm_VCVT_LT_F64_FXU16
	arm_VCVT_GT_F64_FXU16
	arm_VCVT_LE_F64_FXU16
	arm_VCVT_F64_FXU16
	arm_VCVT_ZZ_F64_FXU16
	arm_VCVT_EQ_F64_FXU32
	arm_VCVT_NE_F64_FXU32
	arm_VCVT_CS_F64_FXU32
	arm_VCVT_CC_F64_FXU32
	arm_VCVT_MI_F64_FXU32
	arm_VCVT_PL_F64_FXU32
	arm_VCVT_VS_F64_FXU32
	arm_VCVT_VC_F64_FXU32
	arm_VCVT_HI_F64_FXU32
	arm_VCVT_LS_F64_FXU32
	arm_VCVT_GE_F64_FXU32
	arm_VCVT_LT_F64_FXU32
	arm_VCVT_GT_F64_FXU32
	arm_VCVT_LE_F64_FXU32
	arm_VCVT_F64_FXU32
	arm_VCVT_ZZ_F64_FXU32
	arm_VCVT_EQ_F32_U32
	arm_VCVT_NE_F32_U32
	arm_VCVT_CS_F32_U32
	arm_VCVT_CC_F32_U32
	arm_VCVT_MI_F32_U32
	arm_VCVT_PL_F32_U32
	arm_VCVT_VS_F32_U32
	arm_VCVT_VC_F32_U32
	arm_VCVT_HI_F32_U32
	arm_VCVT_LS_F32_U32
	arm_VCVT_GE_F32_U32
	arm_VCVT_LT_F32_U32
	arm_VCVT_GT_F32_U32
	arm_VCVT_LE_F32_U32
	arm_VCVT_F32_U32
	arm_VCVT_ZZ_F32_U32
	arm_VCVT_EQ_F32_S32
	arm_VCVT_NE_F32_S32
	arm_VCVT_CS_F32_S32
	arm_VCVT_CC_F32_S32
	arm_VCVT_MI_F32_S32
	arm_VCVT_PL_F32_S32
	arm_VCVT_VS_F32_S32
	arm_VCVT_VC_F32_S32
	arm_VCVT_HI_F32_S32
	arm_VCVT_LS_F32_S32
	arm_VCVT_GE_F32_S32
	arm_VCVT_LT_F32_S32
	arm_VCVT_GT_F32_S32
	arm_VCVT_LE_F32_S32
	arm_VCVT_F32_S32
	arm_VCVT_ZZ_F32_S32
	arm_VCVT_EQ_F64_U32
	arm_VCVT_NE_F64_U32
	arm_VCVT_CS_F64_U32
	arm_VCVT_CC_F64_U32
	arm_VCVT_MI_F64_U32
	arm_VCVT_PL_F64_U32
	arm_VCVT_VS_F64_U32
	arm_VCVT_VC_F64_U32
	arm_VCVT_HI_F64_U32
	arm_VCVT_LS_F64_U32
	arm_VCVT_GE_F64_U32
	arm_VCVT_LT_F64_U32
	arm_VCVT_GT_F64_U32
	arm_VCVT_LE_F64_U32
	arm_VCVT_F64_U32
	arm_VCVT_ZZ_F64_U32
	arm_VCVT_EQ_F64_S32
	arm_VCVT_NE_F64_S32
	arm_VCVT_CS_F64_S32
	arm_VCVT_CC_F64_S32
	arm_VCVT_MI_F64_S32
	arm_VCVT_PL_F64_S32
	arm_VCVT_VS_F64_S32
	arm_VCVT_VC_F64_S32
	arm_VCVT_HI_F64_S32
	arm_VCVT_LS_F64_S32
	arm_VCVT_GE_F64_S32
	arm_VCVT_LT_F64_S32
	arm_VCVT_GT_F64_S32
	arm_VCVT_LE_F64_S32
	arm_VCVT_F64_S32
	arm_VCVT_ZZ_F64_S32
	arm_VCVT_EQ_F64_F32
	arm_VCVT_NE_F64_F32
	arm_VCVT_CS_F64_F32
	arm_VCVT_CC_F64_F32
	arm_VCVT_MI_F64_F32
	arm_VCVT_PL_F64_F32
	arm_VCVT_VS_F64_F32
	arm_VCVT_VC_F64_F32
	arm_VCVT_HI_F64_F32
	arm_VCVT_LS_F64_F32
	arm_VCVT_GE_F64_F32
	arm_VCVT_LT_F64_F32
	arm_VCVT_GT_F64_F32
	arm_VCVT_LE_F64_F32
	arm_VCVT_F64_F32
	arm_VCVT_ZZ_F64_F32
	arm_VCVT_EQ_F32_F64
	arm_VCVT_NE_F32_F64
	arm_VCVT_CS_F32_F64
	arm_VCVT_CC_F32_F64
	arm_VCVT_MI_F32_F64
	arm_VCVT_PL_F32_F64
	arm_VCVT_VS_F32_F64
	arm_VCVT_VC_F32_F64
	arm_VCVT_HI_F32_F64
	arm_VCVT_LS_F32_F64
	arm_VCVT_GE_F32_F64
	arm_VCVT_LT_F32_F64
	arm_VCVT_GT_F32_F64
	arm_VCVT_LE_F32_F64
	arm_VCVT_F32_F64
	arm_VCVT_ZZ_F32_F64
	arm_VCVT_EQ_FXS16_F32
	arm_VCVT_NE_FXS16_F32
	arm_VCVT_CS_FXS16_F32
	arm_VCVT_CC_FXS16_F32
	arm_VCVT_MI_FXS16_F32
	arm_VCVT_PL_FXS16_F32
	arm_VCVT_VS_FXS16_F32
	arm_VCVT_VC_FXS16_F32
	arm_VCVT_HI_FXS16_F32
	arm_VCVT_LS_FXS16_F32
	arm_VCVT_GE_FXS16_F32
	arm_VCVT_LT_FXS16_F32
	arm_VCVT_GT_FXS16_F32
	arm_VCVT_LE_FXS16_F32
	arm_VCVT_FXS16_F32
	arm_VCVT_ZZ_FXS16_F32
	arm_VCVT_EQ_FXS16_F64
	arm_VCVT_NE_FXS16_F64
	arm_VCVT_CS_FXS16_F64
	arm_VCVT_CC_FXS16_F64
	arm_VCVT_MI_FXS16_F64
	arm_VCVT_PL_FXS16_F64
	arm_VCVT_VS_FXS16_F64
	arm_VCVT_VC_FXS16_F64
	arm_VCVT_HI_FXS16_F64
	arm_VCVT_LS_FXS16_F64
	arm_VCVT_GE_FXS16_F64
	arm_VCVT_LT_FXS16_F64
	arm_VCVT_GT_FXS16_F64
	arm_VCVT_LE_FXS16_F64
	arm_VCVT_FXS16_F64
	arm_VCVT_ZZ_FXS16_F64
	arm_VCVT_EQ_FXS32_F32
	arm_VCVT_NE_FXS32_F32
	arm_VCVT_CS_FXS32_F32
	arm_VCVT_CC_FXS32_F32
	arm_VCVT_MI_FXS32_F32
	arm_VCVT_PL_FXS32_F32
	arm_VCVT_VS_FXS32_F32
	arm_VCVT_VC_FXS32_F32
	arm_VCVT_HI_FXS32_F32
	arm_VCVT_LS_FXS32_F32
	arm_VCVT_GE_FXS32_F32
	arm_VCVT_LT_FXS32_F32
	arm_VCVT_GT_FXS32_F32
	arm_VCVT_LE_FXS32_F32
	arm_VCVT_FXS32_F32
	arm_VCVT_ZZ_FXS32_F32
	arm_VCVT_EQ_FXS32_F64
	arm_VCVT_NE_FXS32_F64
	arm_VCVT_CS_FXS32_F64
	arm_VCVT_CC_FXS32_F64
	arm_VCVT_MI_FXS32_F64
	arm_VCVT_PL_FXS32_F64
	arm_VCVT_VS_FXS32_F64
	arm_VCVT_VC_FXS32_F64
	arm_VCVT_HI_FXS32_F64
	arm_VCVT_LS_FXS32_F64
	arm_VCVT_GE_FXS32_F64
	arm_VCVT_LT_FXS32_F64
	arm_VCVT_GT_FXS32_F64
	arm_VCVT_LE_FXS32_F64
	arm_VCVT_FXS32_F64
	arm_VCVT_ZZ_FXS32_F64
	arm_VCVT_EQ_FXU16_F32
	arm_VCVT_NE_FXU16_F32
	arm_VCVT_CS_FXU16_F32
	arm_VCVT_CC_FXU16_F32
	arm_VCVT_MI_FXU16_F32
	arm_VCVT_PL_FXU16_F32
	arm_VCVT_VS_FXU16_F32
	arm_VCVT_VC_FXU16_F32
	arm_VCVT_HI_FXU16_F32
	arm_VCVT_LS_FXU16_F32
	arm_VCVT_GE_FXU16_F32
	arm_VCVT_LT_FXU16_F32
	arm_VCVT_GT_FXU16_F32
	arm_VCVT_LE_FXU16_F32
	arm_VCVT_FXU16_F32
	arm_VCVT_ZZ_FXU16_F32
	arm_VCVT_EQ_FXU16_F64
	arm_VCVT_NE_FXU16_F64
	arm_VCVT_CS_FXU16_F64
	arm_VCVT_CC_FXU16_F64
	arm_VCVT_MI_FXU16_F64
	arm_VCVT_PL_FXU16_F64
	arm_VCVT_VS_FXU16_F64
	arm_VCVT_VC_FXU16_F64
	arm_VCVT_HI_FXU16_F64
	arm_VCVT_LS_FXU16_F64
	arm_VCVT_GE_FXU16_F64
	arm_VCVT_LT_FXU16_F64
	arm_VCVT_GT_FXU16_F64
	arm_VCVT_LE_FXU16_F64
	arm_VCVT_FXU16_F64
	arm_VCVT_ZZ_FXU16_F64
	arm_VCVT_EQ_FXU32_F32
	arm_VCVT_NE_FXU32_F32
	arm_VCVT_CS_FXU32_F32
	arm_VCVT_CC_FXU32_F32
	arm_VCVT_MI_FXU32_F32
	arm_VCVT_PL_FXU32_F32
	arm_VCVT_VS_FXU32_F32
	arm_VCVT_VC_FXU32_F32
	arm_VCVT_HI_FXU32_F32
	arm_VCVT_LS_FXU32_F32
	arm_VCVT_GE_FXU32_F32
	arm_VCVT_LT_FXU32_F32
	arm_VCVT_GT_FXU32_F32
	arm_VCVT_LE_FXU32_F32
	arm_VCVT_FXU32_F32
	arm_VCVT_ZZ_FXU32_F32
	arm_VCVT_EQ_FXU32_F64
	arm_VCVT_NE_FXU32_F64
	arm_VCVT_CS_FXU32_F64
	arm_VCVT_CC_FXU32_F64
	arm_VCVT_MI_FXU32_F64
	arm_VCVT_PL_FXU32_F64
	arm_VCVT_VS_FXU32_F64
	arm_VCVT_VC_FXU32_F64
	arm_VCVT_HI_FXU32_F64
	arm_VCVT_LS_FXU32_F64
	arm_VCVT_GE_FXU32_F64
	arm_VCVT_LT_FXU32_F64
	arm_VCVT_GT_FXU32_F64
	arm_VCVT_LE_FXU32_F64
	arm_VCVT_FXU32_F64
	arm_VCVT_ZZ_FXU32_F64
	arm_VCVTB_EQ_F32_F16
	arm_VCVTB_NE_F32_F16
	arm_VCVTB_CS_F32_F16
	arm_VCVTB_CC_F32_F16
	arm_VCVTB_MI_F32_F16
	arm_VCVTB_PL_F32_F16
	arm_VCVTB_VS_F32_F16
	arm_VCVTB_VC_F32_F16
	arm_VCVTB_HI_F32_F16
	arm_VCVTB_LS_F32_F16
	arm_VCVTB_GE_F32_F16
	arm_VCVTB_LT_F32_F16
	arm_VCVTB_GT_F32_F16
	arm_VCVTB_LE_F32_F16
	arm_VCVTB_F32_F16
	arm_VCVTB_ZZ_F32_F16
	arm_VCVTB_EQ_F16_F32
	arm_VCVTB_NE_F16_F32
	arm_VCVTB_CS_F16_F32
	arm_VCVTB_CC_F16_F32
	arm_VCVTB_MI_F16_F32
	arm_VCVTB_PL_F16_F32
	arm_VCVTB_VS_F16_F32
	arm_VCVTB_VC_F16_F32
	arm_VCVTB_HI_F16_F32
	arm_VCVTB_LS_F16_F32
	arm_VCVTB_GE_F16_F32
	arm_VCVTB_LT_F16_F32
	arm_VCVTB_GT_F16_F32
	arm_VCVTB_LE_F16_F32
	arm_VCVTB_F16_F32
	arm_VCVTB_ZZ_F16_F32
	arm_VCVTT_EQ_F32_F16
	arm_VCVTT_NE_F32_F16
	arm_VCVTT_CS_F32_F16
	arm_VCVTT_CC_F32_F16
	arm_VCVTT_MI_F32_F16
	arm_VCVTT_PL_F32_F16
	arm_VCVTT_VS_F32_F16
	arm_VCVTT_VC_F32_F16
	arm_VCVTT_HI_F32_F16
	arm_VCVTT_LS_F32_F16
	arm_VCVTT_GE_F32_F16
	arm_VCVTT_LT_F32_F16
	arm_VCVTT_GT_F32_F16
	arm_VCVTT_LE_F32_F16
	arm_VCVTT_F32_F16
	arm_VCVTT_ZZ_F32_F16
	arm_VCVTT_EQ_F16_F32
	arm_VCVTT_NE_F16_F32
	arm_VCVTT_CS_F16_F32
	arm_VCVTT_CC_F16_F32
	arm_VCVTT_MI_F16_F32
	arm_VCVTT_PL_F16_F32
	arm_VCVTT_VS_F16_F32
	arm_VCVTT_VC_F16_F32
	arm_VCVTT_HI_F16_F32
	arm_VCVTT_LS_F16_F32
	arm_VCVTT_GE_F16_F32
	arm_VCVTT_LT_F16_F32
	arm_VCVTT_GT_F16_F32
	arm_VCVTT_LE_F16_F32
	arm_VCVTT_F16_F32
	arm_VCVTT_ZZ_F16_F32
	arm_VCVTR_EQ_U32_F32
	arm_VCVTR_NE_U32_F32
	arm_VCVTR_CS_U32_F32
	arm_VCVTR_CC_U32_F32
	arm_VCVTR_MI_U32_F32
	arm_VCVTR_PL_U32_F32
	arm_VCVTR_VS_U32_F32
	arm_VCVTR_VC_U32_F32
	arm_VCVTR_HI_U32_F32
	arm_VCVTR_LS_U32_F32
	arm_VCVTR_GE_U32_F32
	arm_VCVTR_LT_U32_F32
	arm_VCVTR_GT_U32_F32
	arm_VCVTR_LE_U32_F32
	arm_VCVTR_U32_F32
	arm_VCVTR_ZZ_U32_F32
	arm_VCVTR_EQ_U32_F64
	arm_VCVTR_NE_U32_F64
	arm_VCVTR_CS_U32_F64
	arm_VCVTR_CC_U32_F64
	arm_VCVTR_MI_U32_F64
	arm_VCVTR_PL_U32_F64
	arm_VCVTR_VS_U32_F64
	arm_VCVTR_VC_U32_F64
	arm_VCVTR_HI_U32_F64
	arm_VCVTR_LS_U32_F64
	arm_VCVTR_GE_U32_F64
	arm_VCVTR_LT_U32_F64
	arm_VCVTR_GT_U32_F64
	arm_VCVTR_LE_U32_F64
	arm_VCVTR_U32_F64
	arm_VCVTR_ZZ_U32_F64
	arm_VCVTR_EQ_S32_F32
	arm_VCVTR_NE_S32_F32
	arm_VCVTR_CS_S32_F32
	arm_VCVTR_CC_S32_F32
	arm_VCVTR_MI_S32_F32
	arm_VCVTR_PL_S32_F32
	arm_VCVTR_VS_S32_F32
	arm_VCVTR_VC_S32_F32
	arm_VCVTR_HI_S32_F32
	arm_VCVTR_LS_S32_F32
	arm_VCVTR_GE_S32_F32
	arm_VCVTR_LT_S32_F32
	arm_VCVTR_GT_S32_F32
	arm_VCVTR_LE_S32_F32
	arm_VCVTR_S32_F32
	arm_VCVTR_ZZ_S32_F32
	arm_VCVTR_EQ_S32_F64
	arm_VCVTR_NE_S32_F64
	arm_VCVTR_CS_S32_F64
	arm_VCVTR_CC_S32_F64
	arm_VCVTR_MI_S32_F64
	arm_VCVTR_PL_S32_F64
	arm_VCVTR_VS_S32_F64
	arm_VCVTR_VC_S32_F64
	arm_VCVTR_HI_S32_F64
	arm_VCVTR_LS_S32_F64
	arm_VCVTR_GE_S32_F64
	arm_VCVTR_LT_S32_F64
	arm_VCVTR_GT_S32_F64
	arm_VCVTR_LE_S32_F64
	arm_VCVTR_S32_F64
	arm_VCVTR_ZZ_S32_F64
	arm_VCVT_EQ_U32_F32
	arm_VCVT_NE_U32_F32
	arm_VCVT_CS_U32_F32
	arm_VCVT_CC_U32_F32
	arm_VCVT_MI_U32_F32
	arm_VCVT_PL_U32_F32
	arm_VCVT_VS_U32_F32
	arm_VCVT_VC_U32_F32
	arm_VCVT_HI_U32_F32
	arm_VCVT_LS_U32_F32
	arm_VCVT_GE_U32_F32
	arm_VCVT_LT_U32_F32
	arm_VCVT_GT_U32_F32
	arm_VCVT_LE_U32_F32
	arm_VCVT_U32_F32
	arm_VCVT_ZZ_U32_F32
	arm_VCVT_EQ_U32_F64
	arm_VCVT_NE_U32_F64
	arm_VCVT_CS_U32_F64
	arm_VCVT_CC_U32_F64
	arm_VCVT_MI_U32_F64
	arm_VCVT_PL_U32_F64
	arm_VCVT_VS_U32_F64
	arm_VCVT_VC_U32_F64
	arm_VCVT_HI_U32_F64
	arm_VCVT_LS_U32_F64
	arm_VCVT_GE_U32_F64
	arm_VCVT_LT_U32_F64
	arm_VCVT_GT_U32_F64
	arm_VCVT_LE_U32_F64
	arm_VCVT_U32_F64
	arm_VCVT_ZZ_U32_F64
	arm_VCVT_EQ_S32_F32
	arm_VCVT_NE_S32_F32
	arm_VCVT_CS_S32_F32
	arm_VCVT_CC_S32_F32
	arm_VCVT_MI_S32_F32
	arm_VCVT_PL_S32_F32
	arm_VCVT_VS_S32_F32
	arm_VCVT_VC_S32_F32
	arm_VCVT_HI_S32_F32
	arm_VCVT_LS_S32_F32
	arm_VCVT_GE_S32_F32
	arm_VCVT_LT_S32_F32
	arm_VCVT_GT_S32_F32
	arm_VCVT_LE_S32_F32
	arm_VCVT_S32_F32
	arm_VCVT_ZZ_S32_F32
	arm_VCVT_EQ_S32_F64
	arm_VCVT_NE_S32_F64
	arm_VCVT_CS_S32_F64
	arm_VCVT_CC_S32_F64
	arm_VCVT_MI_S32_F64
	arm_VCVT_PL_S32_F64
	arm_VCVT_VS_S32_F64
	arm_VCVT_VC_S32_F64
	arm_VCVT_HI_S32_F64
	arm_VCVT_LS_S32_F64
	arm_VCVT_GE_S32_F64
	arm_VCVT_LT_S32_F64
	arm_VCVT_GT_S32_F64
	arm_VCVT_LE_S32_F64
	arm_VCVT_S32_F64
	arm_VCVT_ZZ_S32_F64
	arm_VDIV_EQ_F32
	arm_VDIV_NE_F32
	arm_VDIV_CS_F32
	arm_VDIV_CC_F32
	arm_VDIV_MI_F32
	arm_VDIV_PL_F32
	arm_VDIV_VS_F32
	arm_VDIV_VC_F32
	arm_VDIV_HI_F32
	arm_VDIV_LS_F32
	arm_VDIV_GE_F32
	arm_VDIV_LT_F32
	arm_VDIV_GT_F32
	arm_VDIV_LE_F32
	arm_VDIV_F32
	arm_VDIV_ZZ_F32
	arm_VDIV_EQ_F64
	arm_VDIV_NE_F64
	arm_VDIV_CS_F64
	arm_VDIV_CC_F64
	arm_VDIV_MI_F64
	arm_VDIV_PL_F64
	arm_VDIV_VS_F64
	arm_VDIV_VC_F64
	arm_VDIV_HI_F64
	arm_VDIV_LS_F64
	arm_VDIV_GE_F64
	arm_VDIV_LT_F64
	arm_VDIV_GT_F64
	arm_VDIV_LE_F64
	arm_VDIV_F64
	arm_VDIV_ZZ_F64
	arm_VLDR_EQ
	arm_VLDR_NE
	arm_VLDR_CS
	arm_VLDR_CC
	arm_VLDR_MI
	arm_VLDR_PL
	arm_VLDR_VS
	arm_VLDR_VC
	arm_VLDR_HI
	arm_VLDR_LS
	arm_VLDR_GE
	arm_VLDR_LT
	arm_VLDR_GT
	arm_VLDR_LE
	arm_VLDR
	arm_VLDR_ZZ
	arm_VMLA_EQ_F32
	arm_VMLA_NE_F32
	arm_VMLA_CS_F32
	arm_VMLA_CC_F32
	arm_VMLA_MI_F32
	arm_VMLA_PL_F32
	arm_VMLA_VS_F32
	arm_VMLA_VC_F32
	arm_VMLA_HI_F32
	arm_VMLA_LS_F32
	arm_VMLA_GE_F32
	arm_VMLA_LT_F32
	arm_VMLA_GT_F32
	arm_VMLA_LE_F32
	arm_VMLA_F32
	arm_VMLA_ZZ_F32
	arm_VMLA_EQ_F64
	arm_VMLA_NE_F64
	arm_VMLA_CS_F64
	arm_VMLA_CC_F64
	arm_VMLA_MI_F64
	arm_VMLA_PL_F64
	arm_VMLA_VS_F64
	arm_VMLA_VC_F64
	arm_VMLA_HI_F64
	arm_VMLA_LS_F64
	arm_VMLA_GE_F64
	arm_VMLA_LT_F64
	arm_VMLA_GT_F64
	arm_VMLA_LE_F64
	arm_VMLA_F64
	arm_VMLA_ZZ_F64
	arm_VMLS_EQ_F32
	arm_VMLS_NE_F32
	arm_VMLS_CS_F32
	arm_VMLS_CC_F32
	arm_VMLS_MI_F32
	arm_VMLS_PL_F32
	arm_VMLS_VS_F32
	arm_VMLS_VC_F32
	arm_VMLS_HI_F32
	arm_VMLS_LS_F32
	arm_VMLS_GE_F32
	arm_VMLS_LT_F32
	arm_VMLS_GT_F32
	arm_VMLS_LE_F32
	arm_VMLS_F32
	arm_VMLS_ZZ_F32
	arm_VMLS_EQ_F64
	arm_VMLS_NE_F64
	arm_VMLS_CS_F64
	arm_VMLS_CC_F64
	arm_VMLS_MI_F64
	arm_VMLS_PL_F64
	arm_VMLS_VS_F64
	arm_VMLS_VC_F64
	arm_VMLS_HI_F64
	arm_VMLS_LS_F64
	arm_VMLS_GE_F64
	arm_VMLS_LT_F64
	arm_VMLS_GT_F64
	arm_VMLS_LE_F64
	arm_VMLS_F64
	arm_VMLS_ZZ_F64
	arm_VMOV_EQ
	arm_VMOV_NE
	arm_VMOV_CS
	arm_VMOV_CC
	arm_VMOV_MI
	arm_VMOV_PL
	arm_VMOV_VS
	arm_VMOV_VC
	arm_VMOV_HI
	arm_VMOV_LS
	arm_VMOV_GE
	arm_VMOV_LT
	arm_VMOV_GT
	arm_VMOV_LE
	arm_VMOV
	arm_VMOV_ZZ
	arm_VMOV_EQ_32
	arm_VMOV_NE_32
	arm_VMOV_CS_32
	arm_VMOV_CC_32
	arm_VMOV_MI_32
	arm_VMOV_PL_32
	arm_VMOV_VS_32
	arm_VMOV_VC_32
	arm_VMOV_HI_32
	arm_VMOV_LS_32
	arm_VMOV_GE_32
	arm_VMOV_LT_32
	arm_VMOV_GT_32
	arm_VMOV_LE_32
	arm_VMOV_32
	arm_VMOV_ZZ_32
	arm_VMOV_EQ_F32
	arm_VMOV_NE_F32
	arm_VMOV_CS_F32
	arm_VMOV_CC_F32
	arm_VMOV_MI_F32
	arm_VMOV_PL_F32
	arm_VMOV_VS_F32
	arm_VMOV_VC_F32
	arm_VMOV_HI_F32
	arm_VMOV_LS_F32
	arm_VMOV_GE_F32
	arm_VMOV_LT_F32
	arm_VMOV_GT_F32
	arm_VMOV_LE_F32
	arm_VMOV_F32
	arm_VMOV_ZZ_F32
	arm_VMOV_EQ_F64
	arm_VMOV_NE_F64
	arm_VMOV_CS_F64
	arm_VMOV_CC_F64
	arm_VMOV_MI_F64
	arm_VMOV_PL_F64
	arm_VMOV_VS_F64
	arm_VMOV_VC_F64
	arm_VMOV_HI_F64
	arm_VMOV_LS_F64
	arm_VMOV_GE_F64
	arm_VMOV_LT_F64
	arm_VMOV_GT_F64
	arm_VMOV_LE_F64
	arm_VMOV_F64
	arm_VMOV_ZZ_F64
	arm_VMRS_EQ
	arm_VMRS_NE
	arm_VMRS_CS
	arm_VMRS_CC
	arm_VMRS_MI
	arm_VMRS_PL
	arm_VMRS_VS
	arm_VMRS_VC
	arm_VMRS_HI
	arm_VMRS_LS
	arm_VMRS_GE
	arm_VMRS_LT
	arm_VMRS_GT
	arm_VMRS_LE
	arm_VMRS
	arm_VMRS_ZZ
	arm_VMSR_EQ
	arm_VMSR_NE
	arm_VMSR_CS
	arm_VMSR_CC
	arm_VMSR_MI
	arm_VMSR_PL
	arm_VMSR_VS
	arm_VMSR_VC
	arm_VMSR_HI
	arm_VMSR_LS
	arm_VMSR_GE
	arm_VMSR_LT
	arm_VMSR_GT
	arm_VMSR_LE
	arm_VMSR
	arm_VMSR_ZZ
	arm_VMUL_EQ_F32
	arm_VMUL_NE_F32
	arm_VMUL_CS_F32
	arm_VMUL_CC_F32
	arm_VMUL_MI_F32
	arm_VMUL_PL_F32
	arm_VMUL_VS_F32
	arm_VMUL_VC_F32
	arm_VMUL_HI_F32
	arm_VMUL_LS_F32
	arm_VMUL_GE_F32
	arm_VMUL_LT_F32
	arm_VMUL_GT_F32
	arm_VMUL_LE_F32
	arm_VMUL_F32
	arm_VMUL_ZZ_F32
	arm_VMUL_EQ_F64
	arm_VMUL_NE_F64
	arm_VMUL_CS_F64
	arm_VMUL_CC_F64
	arm_VMUL_MI_F64
	arm_VMUL_PL_F64
	arm_VMUL_VS_F64
	arm_VMUL_VC_F64
	arm_VMUL_HI_F64
	arm_VMUL_LS_F64
	arm_VMUL_GE_F64
	arm_VMUL_LT_F64
	arm_VMUL_GT_F64
	arm_VMUL_LE_F64
	arm_VMUL_F64
	arm_VMUL_ZZ_F64
	arm_VNEG_EQ_F32
	arm_VNEG_NE_F32
	arm_VNEG_CS_F32
	arm_VNEG_CC_F32
	arm_VNEG_MI_F32
	arm_VNEG_PL_F32
	arm_VNEG_VS_F32
	arm_VNEG_VC_F32
	arm_VNEG_HI_F32
	arm_VNEG_LS_F32
	arm_VNEG_GE_F32
	arm_VNEG_LT_F32
	arm_VNEG_GT_F32
	arm_VNEG_LE_F32
	arm_VNEG_F32
	arm_VNEG_ZZ_F32
	arm_VNEG_EQ_F64
	arm_VNEG_NE_F64
	arm_VNEG_CS_F64
	arm_VNEG_CC_F64
	arm_VNEG_MI_F64
	arm_VNEG_PL_F64
	arm_VNEG_VS_F64
	arm_VNEG_VC_F64
	arm_VNEG_HI_F64
	arm_VNEG_LS_F64
	arm_VNEG_GE_F64
	arm_VNEG_LT_F64
	arm_VNEG_GT_F64
	arm_VNEG_LE_F64
	arm_VNEG_F64
	arm_VNEG_ZZ_F64
	arm_VNMLS_EQ_F32
	arm_VNMLS_NE_F32
	arm_VNMLS_CS_F32
	arm_VNMLS_CC_F32
	arm_VNMLS_MI_F32
	arm_VNMLS_PL_F32
	arm_VNMLS_VS_F32
	arm_VNMLS_VC_F32
	arm_VNMLS_HI_F32
	arm_VNMLS_LS_F32
	arm_VNMLS_GE_F32
	arm_VNMLS_LT_F32
	arm_VNMLS_GT_F32
	arm_VNMLS_LE_F32
	arm_VNMLS_F32
	arm_VNMLS_ZZ_F32
	arm_VNMLS_EQ_F64
	arm_VNMLS_NE_F64
	arm_VNMLS_CS_F64
	arm_VNMLS_CC_F64
	arm_VNMLS_MI_F64
	arm_VNMLS_PL_F64
	arm_VNMLS_VS_F64
	arm_VNMLS_VC_F64
	arm_VNMLS_HI_F64
	arm_VNMLS_LS_F64
	arm_VNMLS_GE_F64
	arm_VNMLS_LT_F64
	arm_VNMLS_GT_F64
	arm_VNMLS_LE_F64
	arm_VNMLS_F64
	arm_VNMLS_ZZ_F64
	arm_VNMLA_EQ_F32
	arm_VNMLA_NE_F32
	arm_VNMLA_CS_F32
	arm_VNMLA_CC_F32
	arm_VNMLA_MI_F32
	arm_VNMLA_PL_F32
	arm_VNMLA_VS_F32
	arm_VNMLA_VC_F32
	arm_VNMLA_HI_F32
	arm_VNMLA_LS_F32
	arm_VNMLA_GE_F32
	arm_VNMLA_LT_F32
	arm_VNMLA_GT_F32
	arm_VNMLA_LE_F32
	arm_VNMLA_F32
	arm_VNMLA_ZZ_F32
	arm_VNMLA_EQ_F64
	arm_VNMLA_NE_F64
	arm_VNMLA_CS_F64
	arm_VNMLA_CC_F64
	arm_VNMLA_MI_F64
	arm_VNMLA_PL_F64
	arm_VNMLA_VS_F64
	arm_VNMLA_VC_F64
	arm_VNMLA_HI_F64
	arm_VNMLA_LS_F64
	arm_VNMLA_GE_F64
	arm_VNMLA_LT_F64
	arm_VNMLA_GT_F64
	arm_VNMLA_LE_F64
	arm_VNMLA_F64
	arm_VNMLA_ZZ_F64
	arm_VNMUL_EQ_F32
	arm_VNMUL_NE_F32
	arm_VNMUL_CS_F32
	arm_VNMUL_CC_F32
	arm_VNMUL_MI_F32
	arm_VNMUL_PL_F32
	arm_VNMUL_VS_F32
	arm_VNMUL_VC_F32
	arm_VNMUL_HI_F32
	arm_VNMUL_LS_F32
	arm_VNMUL_GE_F32
	arm_VNMUL_LT_F32
	arm_VNMUL_GT_F32
	arm_VNMUL_LE_F32
	arm_VNMUL_F32
	arm_VNMUL_ZZ_F32
	arm_VNMUL_EQ_F64
	arm_VNMUL_NE_F64
	arm_VNMUL_CS_F64
	arm_VNMUL_CC_F64
	arm_VNMUL_MI_F64
	arm_VNMUL_PL_F64
	arm_VNMUL_VS_F64
	arm_VNMUL_VC_F64
	arm_VNMUL_HI_F64
	arm_VNMUL_LS_F64
	arm_VNMUL_GE_F64
	arm_VNMUL_LT_F64
	arm_VNMUL_GT_F64
	arm_VNMUL_LE_F64
	arm_VNMUL_F64
	arm_VNMUL_ZZ_F64
	arm_VSQRT_EQ_F32
	arm_VSQRT_NE_F32
	arm_VSQRT_CS_F32
	arm_VSQRT_CC_F32
	arm_VSQRT_MI_F32
	arm_VSQRT_PL_F32
	arm_VSQRT_VS_F32
	arm_VSQRT_VC_F32
	arm_VSQRT_HI_F32
	arm_VSQRT_LS_F32
	arm_VSQRT_GE_F32
	arm_VSQRT_LT_F32
	arm_VSQRT_GT_F32
	arm_VSQRT_LE_F32
	arm_VSQRT_F32
	arm_VSQRT_ZZ_F32
	arm_VSQRT_EQ_F64
	arm_VSQRT_NE_F64
	arm_VSQRT_CS_F64
	arm_VSQRT_CC_F64
	arm_VSQRT_MI_F64
	arm_VSQRT_PL_F64
	arm_VSQRT_VS_F64
	arm_VSQRT_VC_F64
	arm_VSQRT_HI_F64
	arm_VSQRT_LS_F64
	arm_VSQRT_GE_F64
	arm_VSQRT_LT_F64
	arm_VSQRT_GT_F64
	arm_VSQRT_LE_F64
	arm_VSQRT_F64
	arm_VSQRT_ZZ_F64
	arm_VSTR_EQ
	arm_VSTR_NE
	arm_VSTR_CS
	arm_VSTR_CC
	arm_VSTR_MI
	arm_VSTR_PL
	arm_VSTR_VS
	arm_VSTR_VC
	arm_VSTR_HI
	arm_VSTR_LS
	arm_VSTR_GE
	arm_VSTR_LT
	arm_VSTR_GT
	arm_VSTR_LE
	arm_VSTR
	arm_VSTR_ZZ
	arm_VSUB_EQ_F32
	arm_VSUB_NE_F32
	arm_VSUB_CS_F32
	arm_VSUB_CC_F32
	arm_VSUB_MI_F32
	arm_VSUB_PL_F32
	arm_VSUB_VS_F32
	arm_VSUB_VC_F32
	arm_VSUB_HI_F32
	arm_VSUB_LS_F32
	arm_VSUB_GE_F32
	arm_VSUB_LT_F32
	arm_VSUB_GT_F32
	arm_VSUB_LE_F32
	arm_VSUB_F32
	arm_VSUB_ZZ_F32
	arm_VSUB_EQ_F64
	arm_VSUB_NE_F64
	arm_VSUB_CS_F64
	arm_VSUB_CC_F64
	arm_VSUB_MI_F64
	arm_VSUB_PL_F64
	arm_VSUB_VS_F64
	arm_VSUB_VC_F64
	arm_VSUB_HI_F64
	arm_VSUB_LS_F64
	arm_VSUB_GE_F64
	arm_VSUB_LT_F64
	arm_VSUB_GT_F64
	arm_VSUB_LE_F64
	arm_VSUB_F64
	arm_VSUB_ZZ_F64
	arm_WFE_EQ
	arm_WFE_NE
	arm_WFE_CS
	arm_WFE_CC
	arm_WFE_MI
	arm_WFE_PL
	arm_WFE_VS
	arm_WFE_VC
	arm_WFE_HI
	arm_WFE_LS
	arm_WFE_GE
	arm_WFE_LT
	arm_WFE_GT
	arm_WFE_LE
	arm_WFE
	arm_WFE_ZZ
	arm_WFI_EQ
	arm_WFI_NE
	arm_WFI_CS
	arm_WFI_CC
	arm_WFI_MI
	arm_WFI_PL
	arm_WFI_VS
	arm_WFI_VC
	arm_WFI_HI
	arm_WFI_LS
	arm_WFI_GE
	arm_WFI_LT
	arm_WFI_GT
	arm_WFI_LE
	arm_WFI
	arm_WFI_ZZ
	arm_YIELD_EQ
	arm_YIELD_NE
	arm_YIELD_CS
	arm_YIELD_CC
	arm_YIELD_MI
	arm_YIELD_PL
	arm_YIELD_VS
	arm_YIELD_VC
	arm_YIELD_HI
	arm_YIELD_LS
	arm_YIELD_GE
	arm_YIELD_LT
	arm_YIELD_GT
	arm_YIELD_LE
	arm_YIELD
	arm_YIELD_ZZ
)

var arm_opstr = [...]string{
	arm_ADC_EQ:            "ADC.EQ",
	arm_ADC_NE:            "ADC.NE",
	arm_ADC_CS:            "ADC.CS",
	arm_ADC_CC:            "ADC.CC",
	arm_ADC_MI:            "ADC.MI",
	arm_ADC_PL:            "ADC.PL",
	arm_ADC_VS:            "ADC.VS",
	arm_ADC_VC:            "ADC.VC",
	arm_ADC_HI:            "ADC.HI",
	arm_ADC_LS:            "ADC.LS",
	arm_ADC_GE:            "ADC.GE",
	arm_ADC_LT:            "ADC.LT",
	arm_ADC_GT:            "ADC.GT",
	arm_ADC_LE:            "ADC.LE",
	arm_ADC:               "ADC",
	arm_ADC_ZZ:            "ADC.ZZ",
	arm_ADC_S_EQ:          "ADC.S.EQ",
	arm_ADC_S_NE:          "ADC.S.NE",
	arm_ADC_S_CS:          "ADC.S.CS",
	arm_ADC_S_CC:          "ADC.S.CC",
	arm_ADC_S_MI:          "ADC.S.MI",
	arm_ADC_S_PL:          "ADC.S.PL",
	arm_ADC_S_VS:          "ADC.S.VS",
	arm_ADC_S_VC:          "ADC.S.VC",
	arm_ADC_S_HI:          "ADC.S.HI",
	arm_ADC_S_LS:          "ADC.S.LS",
	arm_ADC_S_GE:          "ADC.S.GE",
	arm_ADC_S_LT:          "ADC.S.LT",
	arm_ADC_S_GT:          "ADC.S.GT",
	arm_ADC_S_LE:          "ADC.S.LE",
	arm_ADC_S:             "ADC.S",
	arm_ADC_S_ZZ:          "ADC.S.ZZ",
	arm_ADD_EQ:            "ADD.EQ",
	arm_ADD_NE:            "ADD.NE",
	arm_ADD_CS:            "ADD.CS",
	arm_ADD_CC:            "ADD.CC",
	arm_ADD_MI:            "ADD.MI",
	arm_ADD_PL:            "ADD.PL",
	arm_ADD_VS:            "ADD.VS",
	arm_ADD_VC:            "ADD.VC",
	arm_ADD_HI:            "ADD.HI",
	arm_ADD_LS:            "ADD.LS",
	arm_ADD_GE:            "ADD.GE",
	arm_ADD_LT:            "ADD.LT",
	arm_ADD_GT:            "ADD.GT",
	arm_ADD_LE:            "ADD.LE",
	arm_ADD:               "ADD",
	arm_ADD_ZZ:            "ADD.ZZ",
	arm_ADD_S_EQ:          "ADD.S.EQ",
	arm_ADD_S_NE:          "ADD.S.NE",
	arm_ADD_S_CS:          "ADD.S.CS",
	arm_ADD_S_CC:          "ADD.S.CC",
	arm_ADD_S_MI:          "ADD.S.MI",
	arm_ADD_S_PL:          "ADD.S.PL",
	arm_ADD_S_VS:          "ADD.S.VS",
	arm_ADD_S_VC:          "ADD.S.VC",
	arm_ADD_S_HI:          "ADD.S.HI",
	arm_ADD_S_LS:          "ADD.S.LS",
	arm_ADD_S_GE:          "ADD.S.GE",
	arm_ADD_S_LT:          "ADD.S.LT",
	arm_ADD_S_GT:          "ADD.S.GT",
	arm_ADD_S_LE:          "ADD.S.LE",
	arm_ADD_S:             "ADD.S",
	arm_ADD_S_ZZ:          "ADD.S.ZZ",
	arm_AND_EQ:            "AND.EQ",
	arm_AND_NE:            "AND.NE",
	arm_AND_CS:            "AND.CS",
	arm_AND_CC:            "AND.CC",
	arm_AND_MI:            "AND.MI",
	arm_AND_PL:            "AND.PL",
	arm_AND_VS:            "AND.VS",
	arm_AND_VC:            "AND.VC",
	arm_AND_HI:            "AND.HI",
	arm_AND_LS:            "AND.LS",
	arm_AND_GE:            "AND.GE",
	arm_AND_LT:            "AND.LT",
	arm_AND_GT:            "AND.GT",
	arm_AND_LE:            "AND.LE",
	arm_AND:               "AND",
	arm_AND_ZZ:            "AND.ZZ",
	arm_AND_S_EQ:          "AND.S.EQ",
	arm_AND_S_NE:          "AND.S.NE",
	arm_AND_S_CS:          "AND.S.CS",
	arm_AND_S_CC:          "AND.S.CC",
	arm_AND_S_MI:          "AND.S.MI",
	arm_AND_S_PL:          "AND.S.PL",
	arm_AND_S_VS:          "AND.S.VS",
	arm_AND_S_VC:          "AND.S.VC",
	arm_AND_S_HI:          "AND.S.HI",
	arm_AND_S_LS:          "AND.S.LS",
	arm_AND_S_GE:          "AND.S.GE",
	arm_AND_S_LT:          "AND.S.LT",
	arm_AND_S_GT:          "AND.S.GT",
	arm_AND_S_LE:          "AND.S.LE",
	arm_AND_S:             "AND.S",
	arm_AND_S_ZZ:          "AND.S.ZZ",
	arm_ASR_EQ:            "ASR.EQ",
	arm_ASR_NE:            "ASR.NE",
	arm_ASR_CS:            "ASR.CS",
	arm_ASR_CC:            "ASR.CC",
	arm_ASR_MI:            "ASR.MI",
	arm_ASR_PL:            "ASR.PL",
	arm_ASR_VS:            "ASR.VS",
	arm_ASR_VC:            "ASR.VC",
	arm_ASR_HI:            "ASR.HI",
	arm_ASR_LS:            "ASR.LS",
	arm_ASR_GE:            "ASR.GE",
	arm_ASR_LT:            "ASR.LT",
	arm_ASR_GT:            "ASR.GT",
	arm_ASR_LE:            "ASR.LE",
	arm_ASR:               "ASR",
	arm_ASR_ZZ:            "ASR.ZZ",
	arm_ASR_S_EQ:          "ASR.S.EQ",
	arm_ASR_S_NE:          "ASR.S.NE",
	arm_ASR_S_CS:          "ASR.S.CS",
	arm_ASR_S_CC:          "ASR.S.CC",
	arm_ASR_S_MI:          "ASR.S.MI",
	arm_ASR_S_PL:          "ASR.S.PL",
	arm_ASR_S_VS:          "ASR.S.VS",
	arm_ASR_S_VC:          "ASR.S.VC",
	arm_ASR_S_HI:          "ASR.S.HI",
	arm_ASR_S_LS:          "ASR.S.LS",
	arm_ASR_S_GE:          "ASR.S.GE",
	arm_ASR_S_LT:          "ASR.S.LT",
	arm_ASR_S_GT:          "ASR.S.GT",
	arm_ASR_S_LE:          "ASR.S.LE",
	arm_ASR_S:             "ASR.S",
	arm_ASR_S_ZZ:          "ASR.S.ZZ",
	arm_B_EQ:              "B.EQ",
	arm_B_NE:              "B.NE",
	arm_B_CS:              "B.CS",
	arm_B_CC:              "B.CC",
	arm_B_MI:              "B.MI",
	arm_B_PL:              "B.PL",
	arm_B_VS:              "B.VS",
	arm_B_VC:              "B.VC",
	arm_B_HI:              "B.HI",
	arm_B_LS:              "B.LS",
	arm_B_GE:              "B.GE",
	arm_B_LT:              "B.LT",
	arm_B_GT:              "B.GT",
	arm_B_LE:              "B.LE",
	arm_B:                 "B",
	arm_B_ZZ:              "B.ZZ",
	arm_BFC_EQ:            "BFC.EQ",
	arm_BFC_NE:            "BFC.NE",
	arm_BFC_CS:            "BFC.CS",
	arm_BFC_CC:            "BFC.CC",
	arm_BFC_MI:            "BFC.MI",
	arm_BFC_PL:            "BFC.PL",
	arm_BFC_VS:            "BFC.VS",
	arm_BFC_VC:            "BFC.VC",
	arm_BFC_HI:            "BFC.HI",
	arm_BFC_LS:            "BFC.LS",
	arm_BFC_GE:            "BFC.GE",
	arm_BFC_LT:            "BFC.LT",
	arm_BFC_GT:            "BFC.GT",
	arm_BFC_LE:            "BFC.LE",
	arm_BFC:               "BFC",
	arm_BFC_ZZ:            "BFC.ZZ",
	arm_BFI_EQ:            "BFI.EQ",
	arm_BFI_NE:            "BFI.NE",
	arm_BFI_CS:            "BFI.CS",
	arm_BFI_CC:            "BFI.CC",
	arm_BFI_MI:            "BFI.MI",
	arm_BFI_PL:            "BFI.PL",
	arm_BFI_VS:            "BFI.VS",
	arm_BFI_VC:            "BFI.VC",
	arm_BFI_HI:            "BFI.HI",
	arm_BFI_LS:            "BFI.LS",
	arm_BFI_GE:            "BFI.GE",
	arm_BFI_LT:            "BFI.LT",
	arm_BFI_GT:            "BFI.GT",
	arm_BFI_LE:            "BFI.LE",
	arm_BFI:               "BFI",
	arm_BFI_ZZ:            "BFI.ZZ",
	arm_BIC_EQ:            "BIC.EQ",
	arm_BIC_NE:            "BIC.NE",
	arm_BIC_CS:            "BIC.CS",
	arm_BIC_CC:            "BIC.CC",
	arm_BIC_MI:            "BIC.MI",
	arm_BIC_PL:            "BIC.PL",
	arm_BIC_VS:            "BIC.VS",
	arm_BIC_VC:            "BIC.VC",
	arm_BIC_HI:            "BIC.HI",
	arm_BIC_LS:            "BIC.LS",
	arm_BIC_GE:            "BIC.GE",
	arm_BIC_LT:            "BIC.LT",
	arm_BIC_GT:            "BIC.GT",
	arm_BIC_LE:            "BIC.LE",
	arm_BIC:               "BIC",
	arm_BIC_ZZ:            "BIC.ZZ",
	arm_BIC_S_EQ:          "BIC.S.EQ",
	arm_BIC_S_NE:          "BIC.S.NE",
	arm_BIC_S_CS:          "BIC.S.CS",
	arm_BIC_S_CC:          "BIC.S.CC",
	arm_BIC_S_MI:          "BIC.S.MI",
	arm_BIC_S_PL:          "BIC.S.PL",
	arm_BIC_S_VS:          "BIC.S.VS",
	arm_BIC_S_VC:          "BIC.S.VC",
	arm_BIC_S_HI:          "BIC.S.HI",
	arm_BIC_S_LS:          "BIC.S.LS",
	arm_BIC_S_GE:          "BIC.S.GE",
	arm_BIC_S_LT:          "BIC.S.LT",
	arm_BIC_S_GT:          "BIC.S.GT",
	arm_BIC_S_LE:          "BIC.S.LE",
	arm_BIC_S:             "BIC.S",
	arm_BIC_S_ZZ:          "BIC.S.ZZ",
	arm_BKPT_EQ:           "BKPT.EQ",
	arm_BKPT_NE:           "BKPT.NE",
	arm_BKPT_CS:           "BKPT.CS",
	arm_BKPT_CC:           "BKPT.CC",
	arm_BKPT_MI:           "BKPT.MI",
	arm_BKPT_PL:           "BKPT.PL",
	arm_BKPT_VS:           "BKPT.VS",
	arm_BKPT_VC:           "BKPT.VC",
	arm_BKPT_HI:           "BKPT.HI",
	arm_BKPT_LS:           "BKPT.LS",
	arm_BKPT_GE:           "BKPT.GE",
	arm_BKPT_LT:           "BKPT.LT",
	arm_BKPT_GT:           "BKPT.GT",
	arm_BKPT_LE:           "BKPT.LE",
	arm_BKPT:              "BKPT",
	arm_BKPT_ZZ:           "BKPT.ZZ",
	arm_BL_EQ:             "BL.EQ",
	arm_BL_NE:             "BL.NE",
	arm_BL_CS:             "BL.CS",
	arm_BL_CC:             "BL.CC",
	arm_BL_MI:             "BL.MI",
	arm_BL_PL:             "BL.PL",
	arm_BL_VS:             "BL.VS",
	arm_BL_VC:             "BL.VC",
	arm_BL_HI:             "BL.HI",
	arm_BL_LS:             "BL.LS",
	arm_BL_GE:             "BL.GE",
	arm_BL_LT:             "BL.LT",
	arm_BL_GT:             "BL.GT",
	arm_BL_LE:             "BL.LE",
	arm_BL:                "BL",
	arm_BL_ZZ:             "BL.ZZ",
	arm_BLX_EQ:            "BLX.EQ",
	arm_BLX_NE:            "BLX.NE",
	arm_BLX_CS:            "BLX.CS",
	arm_BLX_CC:            "BLX.CC",
	arm_BLX_MI:            "BLX.MI",
	arm_BLX_PL:            "BLX.PL",
	arm_BLX_VS:            "BLX.VS",
	arm_BLX_VC:            "BLX.VC",
	arm_BLX_HI:            "BLX.HI",
	arm_BLX_LS:            "BLX.LS",
	arm_BLX_GE:            "BLX.GE",
	arm_BLX_LT:            "BLX.LT",
	arm_BLX_GT:            "BLX.GT",
	arm_BLX_LE:            "BLX.LE",
	arm_BLX:               "BLX",
	arm_BLX_ZZ:            "BLX.ZZ",
	arm_BX_EQ:             "BX.EQ",
	arm_BX_NE:             "BX.NE",
	arm_BX_CS:             "BX.CS",
	arm_BX_CC:             "BX.CC",
	arm_BX_MI:             "BX.MI",
	arm_BX_PL:             "BX.PL",
	arm_BX_VS:             "BX.VS",
	arm_BX_VC:             "BX.VC",
	arm_BX_HI:             "BX.HI",
	arm_BX_LS:             "BX.LS",
	arm_BX_GE:             "BX.GE",
	arm_BX_LT:             "BX.LT",
	arm_BX_GT:             "BX.GT",
	arm_BX_LE:             "BX.LE",
	arm_BX:                "BX",
	arm_BX_ZZ:             "BX.ZZ",
	arm_BXJ_EQ:            "BXJ.EQ",
	arm_BXJ_NE:            "BXJ.NE",
	arm_BXJ_CS:            "BXJ.CS",
	arm_BXJ_CC:            "BXJ.CC",
	arm_BXJ_MI:            "BXJ.MI",
	arm_BXJ_PL:            "BXJ.PL",
	arm_BXJ_VS:            "BXJ.VS",
	arm_BXJ_VC:            "BXJ.VC",
	arm_BXJ_HI:            "BXJ.HI",
	arm_BXJ_LS:            "BXJ.LS",
	arm_BXJ_GE:            "BXJ.GE",
	arm_BXJ_LT:            "BXJ.LT",
	arm_BXJ_GT:            "BXJ.GT",
	arm_BXJ_LE:            "BXJ.LE",
	arm_BXJ:               "BXJ",
	arm_BXJ_ZZ:            "BXJ.ZZ",
	arm_CLREX:             "CLREX",
	arm_CLZ_EQ:            "CLZ.EQ",
	arm_CLZ_NE:            "CLZ.NE",
	arm_CLZ_CS:            "CLZ.CS",
	arm_CLZ_CC:            "CLZ.CC",
	arm_CLZ_MI:            "CLZ.MI",
	arm_CLZ_PL:            "CLZ.PL",
	arm_CLZ_VS:            "CLZ.VS",
	arm_CLZ_VC:            "CLZ.VC",
	arm_CLZ_HI:            "CLZ.HI",
	arm_CLZ_LS:            "CLZ.LS",
	arm_CLZ_GE:            "CLZ.GE",
	arm_CLZ_LT:            "CLZ.LT",
	arm_CLZ_GT:            "CLZ.GT",
	arm_CLZ_LE:            "CLZ.LE",
	arm_CLZ:               "CLZ",
	arm_CLZ_ZZ:            "CLZ.ZZ",
	arm_CMN_EQ:            "CMN.EQ",
	arm_CMN_NE:            "CMN.NE",
	arm_CMN_CS:            "CMN.CS",
	arm_CMN_CC:            "CMN.CC",
	arm_CMN_MI:            "CMN.MI",
	arm_CMN_PL:            "CMN.PL",
	arm_CMN_VS:            "CMN.VS",
	arm_CMN_VC:            "CMN.VC",
	arm_CMN_HI:            "CMN.HI",
	arm_CMN_LS:            "CMN.LS",
	arm_CMN_GE:            "CMN.GE",
	arm_CMN_LT:            "CMN.LT",
	arm_CMN_GT:            "CMN.GT",
	arm_CMN_LE:            "CMN.LE",
	arm_CMN:               "CMN",
	arm_CMN_ZZ:            "CMN.ZZ",
	arm_CMP_EQ:            "CMP.EQ",
	arm_CMP_NE:            "CMP.NE",
	arm_CMP_CS:            "CMP.CS",
	arm_CMP_CC:            "CMP.CC",
	arm_CMP_MI:            "CMP.MI",
	arm_CMP_PL:            "CMP.PL",
	arm_CMP_VS:            "CMP.VS",
	arm_CMP_VC:            "CMP.VC",
	arm_CMP_HI:            "CMP.HI",
	arm_CMP_LS:            "CMP.LS",
	arm_CMP_GE:            "CMP.GE",
	arm_CMP_LT:            "CMP.LT",
	arm_CMP_GT:            "CMP.GT",
	arm_CMP_LE:            "CMP.LE",
	arm_CMP:               "CMP",
	arm_CMP_ZZ:            "CMP.ZZ",
	arm_DBG_EQ:            "DBG.EQ",
	arm_DBG_NE:            "DBG.NE",
	arm_DBG_CS:            "DBG.CS",
	arm_DBG_CC:            "DBG.CC",
	arm_DBG_MI:            "DBG.MI",
	arm_DBG_PL:            "DBG.PL",
	arm_DBG_VS:            "DBG.VS",
	arm_DBG_VC:            "DBG.VC",
	arm_DBG_HI:            "DBG.HI",
	arm_DBG_LS:            "DBG.LS",
	arm_DBG_GE:            "DBG.GE",
	arm_DBG_LT:            "DBG.LT",
	arm_DBG_GT:            "DBG.GT",
	arm_DBG_LE:            "DBG.LE",
	arm_DBG:               "DBG",
	arm_DBG_ZZ:            "DBG.ZZ",
	arm_DMB:               "DMB",
	arm_DSB:               "DSB",
	arm_EOR_EQ:            "EOR.EQ",
	arm_EOR_NE:            "EOR.NE",
	arm_EOR_CS:            "EOR.CS",
	arm_EOR_CC:            "EOR.CC",
	arm_EOR_MI:            "EOR.MI",
	arm_EOR_PL:            "EOR.PL",
	arm_EOR_VS:            "EOR.VS",
	arm_EOR_VC:            "EOR.VC",
	arm_EOR_HI:            "EOR.HI",
	arm_EOR_LS:            "EOR.LS",
	arm_EOR_GE:            "EOR.GE",
	arm_EOR_LT:            "EOR.LT",
	arm_EOR_GT:            "EOR.GT",
	arm_EOR_LE:            "EOR.LE",
	arm_EOR:               "EOR",
	arm_EOR_ZZ:            "EOR.ZZ",
	arm_EOR_S_EQ:          "EOR.S.EQ",
	arm_EOR_S_NE:          "EOR.S.NE",
	arm_EOR_S_CS:          "EOR.S.CS",
	arm_EOR_S_CC:          "EOR.S.CC",
	arm_EOR_S_MI:          "EOR.S.MI",
	arm_EOR_S_PL:          "EOR.S.PL",
	arm_EOR_S_VS:          "EOR.S.VS",
	arm_EOR_S_VC:          "EOR.S.VC",
	arm_EOR_S_HI:          "EOR.S.HI",
	arm_EOR_S_LS:          "EOR.S.LS",
	arm_EOR_S_GE:          "EOR.S.GE",
	arm_EOR_S_LT:          "EOR.S.LT",
	arm_EOR_S_GT:          "EOR.S.GT",
	arm_EOR_S_LE:          "EOR.S.LE",
	arm_EOR_S:             "EOR.S",
	arm_EOR_S_ZZ:          "EOR.S.ZZ",
	arm_ISB:               "ISB",
	arm_LDM_EQ:            "LDM.EQ",
	arm_LDM_NE:            "LDM.NE",
	arm_LDM_CS:            "LDM.CS",
	arm_LDM_CC:            "LDM.CC",
	arm_LDM_MI:            "LDM.MI",
	arm_LDM_PL:            "LDM.PL",
	arm_LDM_VS:            "LDM.VS",
	arm_LDM_VC:            "LDM.VC",
	arm_LDM_HI:            "LDM.HI",
	arm_LDM_LS:            "LDM.LS",
	arm_LDM_GE:            "LDM.GE",
	arm_LDM_LT:            "LDM.LT",
	arm_LDM_GT:            "LDM.GT",
	arm_LDM_LE:            "LDM.LE",
	arm_LDM:               "LDM",
	arm_LDM_ZZ:            "LDM.ZZ",
	arm_LDMDA_EQ:          "LDMDA.EQ",
	arm_LDMDA_NE:          "LDMDA.NE",
	arm_LDMDA_CS:          "LDMDA.CS",
	arm_LDMDA_CC:          "LDMDA.CC",
	arm_LDMDA_MI:          "LDMDA.MI",
	arm_LDMDA_PL:          "LDMDA.PL",
	arm_LDMDA_VS:          "LDMDA.VS",
	arm_LDMDA_VC:          "LDMDA.VC",
	arm_LDMDA_HI:          "LDMDA.HI",
	arm_LDMDA_LS:          "LDMDA.LS",
	arm_LDMDA_GE:          "LDMDA.GE",
	arm_LDMDA_LT:          "LDMDA.LT",
	arm_LDMDA_GT:          "LDMDA.GT",
	arm_LDMDA_LE:          "LDMDA.LE",
	arm_LDMDA:             "LDMDA",
	arm_LDMDA_ZZ:          "LDMDA.ZZ",
	arm_LDMDB_EQ:          "LDMDB.EQ",
	arm_LDMDB_NE:          "LDMDB.NE",
	arm_LDMDB_CS:          "LDMDB.CS",
	arm_LDMDB_CC:          "LDMDB.CC",
	arm_LDMDB_MI:          "LDMDB.MI",
	arm_LDMDB_PL:          "LDMDB.PL",
	arm_LDMDB_VS:          "LDMDB.VS",
	arm_LDMDB_VC:          "LDMDB.VC",
	arm_LDMDB_HI:          "LDMDB.HI",
	arm_LDMDB_LS:          "LDMDB.LS",
	arm_LDMDB_GE:          "LDMDB.GE",
	arm_LDMDB_LT:          "LDMDB.LT",
	arm_LDMDB_GT:          "LDMDB.GT",
	arm_LDMDB_LE:          "LDMDB.LE",
	arm_LDMDB:             "LDMDB",
	arm_LDMDB_ZZ:          "LDMDB.ZZ",
	arm_LDMIB_EQ:          "LDMIB.EQ",
	arm_LDMIB_NE:          "LDMIB.NE",
	arm_LDMIB_CS:          "LDMIB.CS",
	arm_LDMIB_CC:          "LDMIB.CC",
	arm_LDMIB_MI:          "LDMIB.MI",
	arm_LDMIB_PL:          "LDMIB.PL",
	arm_LDMIB_VS:          "LDMIB.VS",
	arm_LDMIB_VC:          "LDMIB.VC",
	arm_LDMIB_HI:          "LDMIB.HI",
	arm_LDMIB_LS:          "LDMIB.LS",
	arm_LDMIB_GE:          "LDMIB.GE",
	arm_LDMIB_LT:          "LDMIB.LT",
	arm_LDMIB_GT:          "LDMIB.GT",
	arm_LDMIB_LE:          "LDMIB.LE",
	arm_LDMIB:             "LDMIB",
	arm_LDMIB_ZZ:          "LDMIB.ZZ",
	arm_LDR_EQ:            "LDR.EQ",
	arm_LDR_NE:            "LDR.NE",
	arm_LDR_CS:            "LDR.CS",
	arm_LDR_CC:            "LDR.CC",
	arm_LDR_MI:            "LDR.MI",
	arm_LDR_PL:            "LDR.PL",
	arm_LDR_VS:            "LDR.VS",
	arm_LDR_VC:            "LDR.VC",
	arm_LDR_HI:            "LDR.HI",
	arm_LDR_LS:            "LDR.LS",
	arm_LDR_GE:            "LDR.GE",
	arm_LDR_LT:            "LDR.LT",
	arm_LDR_GT:            "LDR.GT",
	arm_LDR_LE:            "LDR.LE",
	arm_LDR:               "LDR",
	arm_LDR_ZZ:            "LDR.ZZ",
	arm_LDRB_EQ:           "LDRB.EQ",
	arm_LDRB_NE:           "LDRB.NE",
	arm_LDRB_CS:           "LDRB.CS",
	arm_LDRB_CC:           "LDRB.CC",
	arm_LDRB_MI:           "LDRB.MI",
	arm_LDRB_PL:           "LDRB.PL",
	arm_LDRB_VS:           "LDRB.VS",
	arm_LDRB_VC:           "LDRB.VC",
	arm_LDRB_HI:           "LDRB.HI",
	arm_LDRB_LS:           "LDRB.LS",
	arm_LDRB_GE:           "LDRB.GE",
	arm_LDRB_LT:           "LDRB.LT",
	arm_LDRB_GT:           "LDRB.GT",
	arm_LDRB_LE:           "LDRB.LE",
	arm_LDRB:              "LDRB",
	arm_LDRB_ZZ:           "LDRB.ZZ",
	arm_LDRBT_EQ:          "LDRBT.EQ",
	arm_LDRBT_NE:          "LDRBT.NE",
	arm_LDRBT_CS:          "LDRBT.CS",
	arm_LDRBT_CC:          "LDRBT.CC",
	arm_LDRBT_MI:          "LDRBT.MI",
	arm_LDRBT_PL:          "LDRBT.PL",
	arm_LDRBT_VS:          "LDRBT.VS",
	arm_LDRBT_VC:          "LDRBT.VC",
	arm_LDRBT_HI:          "LDRBT.HI",
	arm_LDRBT_LS:          "LDRBT.LS",
	arm_LDRBT_GE:          "LDRBT.GE",
	arm_LDRBT_LT:          "LDRBT.LT",
	arm_LDRBT_GT:          "LDRBT.GT",
	arm_LDRBT_LE:          "LDRBT.LE",
	arm_LDRBT:             "LDRBT",
	arm_LDRBT_ZZ:          "LDRBT.ZZ",
	arm_LDRD_EQ:           "LDRD.EQ",
	arm_LDRD_NE:           "LDRD.NE",
	arm_LDRD_CS:           "LDRD.CS",
	arm_LDRD_CC:           "LDRD.CC",
	arm_LDRD_MI:           "LDRD.MI",
	arm_LDRD_PL:           "LDRD.PL",
	arm_LDRD_VS:           "LDRD.VS",
	arm_LDRD_VC:           "LDRD.VC",
	arm_LDRD_HI:           "LDRD.HI",
	arm_LDRD_LS:           "LDRD.LS",
	arm_LDRD_GE:           "LDRD.GE",
	arm_LDRD_LT:           "LDRD.LT",
	arm_LDRD_GT:           "LDRD.GT",
	arm_LDRD_LE:           "LDRD.LE",
	arm_LDRD:              "LDRD",
	arm_LDRD_ZZ:           "LDRD.ZZ",
	arm_LDREX_EQ:          "LDREX.EQ",
	arm_LDREX_NE:          "LDREX.NE",
	arm_LDREX_CS:          "LDREX.CS",
	arm_LDREX_CC:          "LDREX.CC",
	arm_LDREX_MI:          "LDREX.MI",
	arm_LDREX_PL:          "LDREX.PL",
	arm_LDREX_VS:          "LDREX.VS",
	arm_LDREX_VC:          "LDREX.VC",
	arm_LDREX_HI:          "LDREX.HI",
	arm_LDREX_LS:          "LDREX.LS",
	arm_LDREX_GE:          "LDREX.GE",
	arm_LDREX_LT:          "LDREX.LT",
	arm_LDREX_GT:          "LDREX.GT",
	arm_LDREX_LE:          "LDREX.LE",
	arm_LDREX:             "LDREX",
	arm_LDREX_ZZ:          "LDREX.ZZ",
	arm_LDREXB_EQ:         "LDREXB.EQ",
	arm_LDREXB_NE:         "LDREXB.NE",
	arm_LDREXB_CS:         "LDREXB.CS",
	arm_LDREXB_CC:         "LDREXB.CC",
	arm_LDREXB_MI:         "LDREXB.MI",
	arm_LDREXB_PL:         "LDREXB.PL",
	arm_LDREXB_VS:         "LDREXB.VS",
	arm_LDREXB_VC:         "LDREXB.VC",
	arm_LDREXB_HI:         "LDREXB.HI",
	arm_LDREXB_LS:         "LDREXB.LS",
	arm_LDREXB_GE:         "LDREXB.GE",
	arm_LDREXB_LT:         "LDREXB.LT",
	arm_LDREXB_GT:         "LDREXB.GT",
	arm_LDREXB_LE:         "LDREXB.LE",
	arm_LDREXB:            "LDREXB",
	arm_LDREXB_ZZ:         "LDREXB.ZZ",
	arm_LDREXD_EQ:         "LDREXD.EQ",
	arm_LDREXD_NE:         "LDREXD.NE",
	arm_LDREXD_CS:         "LDREXD.CS",
	arm_LDREXD_CC:         "LDREXD.CC",
	arm_LDREXD_MI:         "LDREXD.MI",
	arm_LDREXD_PL:         "LDREXD.PL",
	arm_LDREXD_VS:         "LDREXD.VS",
	arm_LDREXD_VC:         "LDREXD.VC",
	arm_LDREXD_HI:         "LDREXD.HI",
	arm_LDREXD_LS:         "LDREXD.LS",
	arm_LDREXD_GE:         "LDREXD.GE",
	arm_LDREXD_LT:         "LDREXD.LT",
	arm_LDREXD_GT:         "LDREXD.GT",
	arm_LDREXD_LE:         "LDREXD.LE",
	arm_LDREXD:            "LDREXD",
	arm_LDREXD_ZZ:         "LDREXD.ZZ",
	arm_LDREXH_EQ:         "LDREXH.EQ",
	arm_LDREXH_NE:         "LDREXH.NE",
	arm_LDREXH_CS:         "LDREXH.CS",
	arm_LDREXH_CC:         "LDREXH.CC",
	arm_LDREXH_MI:         "LDREXH.MI",
	arm_LDREXH_PL:         "LDREXH.PL",
	arm_LDREXH_VS:         "LDREXH.VS",
	arm_LDREXH_VC:         "LDREXH.VC",
	arm_LDREXH_HI:         "LDREXH.HI",
	arm_LDREXH_LS:         "LDREXH.LS",
	arm_LDREXH_GE:         "LDREXH.GE",
	arm_LDREXH_LT:         "LDREXH.LT",
	arm_LDREXH_GT:         "LDREXH.GT",
	arm_LDREXH_LE:         "LDREXH.LE",
	arm_LDREXH:            "LDREXH",
	arm_LDREXH_ZZ:         "LDREXH.ZZ",
	arm_LDRH_EQ:           "LDRH.EQ",
	arm_LDRH_NE:           "LDRH.NE",
	arm_LDRH_CS:           "LDRH.CS",
	arm_LDRH_CC:           "LDRH.CC",
	arm_LDRH_MI:           "LDRH.MI",
	arm_LDRH_PL:           "LDRH.PL",
	arm_LDRH_VS:           "LDRH.VS",
	arm_LDRH_VC:           "LDRH.VC",
	arm_LDRH_HI:           "LDRH.HI",
	arm_LDRH_LS:           "LDRH.LS",
	arm_LDRH_GE:           "LDRH.GE",
	arm_LDRH_LT:           "LDRH.LT",
	arm_LDRH_GT:           "LDRH.GT",
	arm_LDRH_LE:           "LDRH.LE",
	arm_LDRH:              "LDRH",
	arm_LDRH_ZZ:           "LDRH.ZZ",
	arm_LDRHT_EQ:          "LDRHT.EQ",
	arm_LDRHT_NE:          "LDRHT.NE",
	arm_LDRHT_CS:          "LDRHT.CS",
	arm_LDRHT_CC:          "LDRHT.CC",
	arm_LDRHT_MI:          "LDRHT.MI",
	arm_LDRHT_PL:          "LDRHT.PL",
	arm_LDRHT_VS:          "LDRHT.VS",
	arm_LDRHT_VC:          "LDRHT.VC",
	arm_LDRHT_HI:          "LDRHT.HI",
	arm_LDRHT_LS:          "LDRHT.LS",
	arm_LDRHT_GE:          "LDRHT.GE",
	arm_LDRHT_LT:          "LDRHT.LT",
	arm_LDRHT_GT:          "LDRHT.GT",
	arm_LDRHT_LE:          "LDRHT.LE",
	arm_LDRHT:             "LDRHT",
	arm_LDRHT_ZZ:          "LDRHT.ZZ",
	arm_LDRSB_EQ:          "LDRSB.EQ",
	arm_LDRSB_NE:          "LDRSB.NE",
	arm_LDRSB_CS:          "LDRSB.CS",
	arm_LDRSB_CC:          "LDRSB.CC",
	arm_LDRSB_MI:          "LDRSB.MI",
	arm_LDRSB_PL:          "LDRSB.PL",
	arm_LDRSB_VS:          "LDRSB.VS",
	arm_LDRSB_VC:          "LDRSB.VC",
	arm_LDRSB_HI:          "LDRSB.HI",
	arm_LDRSB_LS:          "LDRSB.LS",
	arm_LDRSB_GE:          "LDRSB.GE",
	arm_LDRSB_LT:          "LDRSB.LT",
	arm_LDRSB_GT:          "LDRSB.GT",
	arm_LDRSB_LE:          "LDRSB.LE",
	arm_LDRSB:             "LDRSB",
	arm_LDRSB_ZZ:          "LDRSB.ZZ",
	arm_LDRSBT_EQ:         "LDRSBT.EQ",
	arm_LDRSBT_NE:         "LDRSBT.NE",
	arm_LDRSBT_CS:         "LDRSBT.CS",
	arm_LDRSBT_CC:         "LDRSBT.CC",
	arm_LDRSBT_MI:         "LDRSBT.MI",
	arm_LDRSBT_PL:         "LDRSBT.PL",
	arm_LDRSBT_VS:         "LDRSBT.VS",
	arm_LDRSBT_VC:         "LDRSBT.VC",
	arm_LDRSBT_HI:         "LDRSBT.HI",
	arm_LDRSBT_LS:         "LDRSBT.LS",
	arm_LDRSBT_GE:         "LDRSBT.GE",
	arm_LDRSBT_LT:         "LDRSBT.LT",
	arm_LDRSBT_GT:         "LDRSBT.GT",
	arm_LDRSBT_LE:         "LDRSBT.LE",
	arm_LDRSBT:            "LDRSBT",
	arm_LDRSBT_ZZ:         "LDRSBT.ZZ",
	arm_LDRSH_EQ:          "LDRSH.EQ",
	arm_LDRSH_NE:          "LDRSH.NE",
	arm_LDRSH_CS:          "LDRSH.CS",
	arm_LDRSH_CC:          "LDRSH.CC",
	arm_LDRSH_MI:          "LDRSH.MI",
	arm_LDRSH_PL:          "LDRSH.PL",
	arm_LDRSH_VS:          "LDRSH.VS",
	arm_LDRSH_VC:          "LDRSH.VC",
	arm_LDRSH_HI:          "LDRSH.HI",
	arm_LDRSH_LS:          "LDRSH.LS",
	arm_LDRSH_GE:          "LDRSH.GE",
	arm_LDRSH_LT:          "LDRSH.LT",
	arm_LDRSH_GT:          "LDRSH.GT",
	arm_LDRSH_LE:          "LDRSH.LE",
	arm_LDRSH:             "LDRSH",
	arm_LDRSH_ZZ:          "LDRSH.ZZ",
	arm_LDRSHT_EQ:         "LDRSHT.EQ",
	arm_LDRSHT_NE:         "LDRSHT.NE",
	arm_LDRSHT_CS:         "LDRSHT.CS",
	arm_LDRSHT_CC:         "LDRSHT.CC",
	arm_LDRSHT_MI:         "LDRSHT.MI",
	arm_LDRSHT_PL:         "LDRSHT.PL",
	arm_LDRSHT_VS:         "LDRSHT.VS",
	arm_LDRSHT_VC:         "LDRSHT.VC",
	arm_LDRSHT_HI:         "LDRSHT.HI",
	arm_LDRSHT_LS:         "LDRSHT.LS",
	arm_LDRSHT_GE:         "LDRSHT.GE",
	arm_LDRSHT_LT:         "LDRSHT.LT",
	arm_LDRSHT_GT:         "LDRSHT.GT",
	arm_LDRSHT_LE:         "LDRSHT.LE",
	arm_LDRSHT:            "LDRSHT",
	arm_LDRSHT_ZZ:         "LDRSHT.ZZ",
	arm_LDRT_EQ:           "LDRT.EQ",
	arm_LDRT_NE:           "LDRT.NE",
	arm_LDRT_CS:           "LDRT.CS",
	arm_LDRT_CC:           "LDRT.CC",
	arm_LDRT_MI:           "LDRT.MI",
	arm_LDRT_PL:           "LDRT.PL",
	arm_LDRT_VS:           "LDRT.VS",
	arm_LDRT_VC:           "LDRT.VC",
	arm_LDRT_HI:           "LDRT.HI",
	arm_LDRT_LS:           "LDRT.LS",
	arm_LDRT_GE:           "LDRT.GE",
	arm_LDRT_LT:           "LDRT.LT",
	arm_LDRT_GT:           "LDRT.GT",
	arm_LDRT_LE:           "LDRT.LE",
	arm_LDRT:              "LDRT",
	arm_LDRT_ZZ:           "LDRT.ZZ",
	arm_LSL_EQ:            "LSL.EQ",
	arm_LSL_NE:            "LSL.NE",
	arm_LSL_CS:            "LSL.CS",
	arm_LSL_CC:            "LSL.CC",
	arm_LSL_MI:            "LSL.MI",
	arm_LSL_PL:            "LSL.PL",
	arm_LSL_VS:            "LSL.VS",
	arm_LSL_VC:            "LSL.VC",
	arm_LSL_HI:            "LSL.HI",
	arm_LSL_LS:            "LSL.LS",
	arm_LSL_GE:            "LSL.GE",
	arm_LSL_LT:            "LSL.LT",
	arm_LSL_GT:            "LSL.GT",
	arm_LSL_LE:            "LSL.LE",
	arm_LSL:               "LSL",
	arm_LSL_ZZ:            "LSL.ZZ",
	arm_LSL_S_EQ:          "LSL.S.EQ",
	arm_LSL_S_NE:          "LSL.S.NE",
	arm_LSL_S_CS:          "LSL.S.CS",
	arm_LSL_S_CC:          "LSL.S.CC",
	arm_LSL_S_MI:          "LSL.S.MI",
	arm_LSL_S_PL:          "LSL.S.PL",
	arm_LSL_S_VS:          "LSL.S.VS",
	arm_LSL_S_VC:          "LSL.S.VC",
	arm_LSL_S_HI:          "LSL.S.HI",
	arm_LSL_S_LS:          "LSL.S.LS",
	arm_LSL_S_GE:          "LSL.S.GE",
	arm_LSL_S_LT:          "LSL.S.LT",
	arm_LSL_S_GT:          "LSL.S.GT",
	arm_LSL_S_LE:          "LSL.S.LE",
	arm_LSL_S:             "LSL.S",
	arm_LSL_S_ZZ:          "LSL.S.ZZ",
	arm_LSR_EQ:            "LSR.EQ",
	arm_LSR_NE:            "LSR.NE",
	arm_LSR_CS:            "LSR.CS",
	arm_LSR_CC:            "LSR.CC",
	arm_LSR_MI:            "LSR.MI",
	arm_LSR_PL:            "LSR.PL",
	arm_LSR_VS:            "LSR.VS",
	arm_LSR_VC:            "LSR.VC",
	arm_LSR_HI:            "LSR.HI",
	arm_LSR_LS:            "LSR.LS",
	arm_LSR_GE:            "LSR.GE",
	arm_LSR_LT:            "LSR.LT",
	arm_LSR_GT:            "LSR.GT",
	arm_LSR_LE:            "LSR.LE",
	arm_LSR:               "LSR",
	arm_LSR_ZZ:            "LSR.ZZ",
	arm_LSR_S_EQ:          "LSR.S.EQ",
	arm_LSR_S_NE:          "LSR.S.NE",
	arm_LSR_S_CS:          "LSR.S.CS",
	arm_LSR_S_CC:          "LSR.S.CC",
	arm_LSR_S_MI:          "LSR.S.MI",
	arm_LSR_S_PL:          "LSR.S.PL",
	arm_LSR_S_VS:          "LSR.S.VS",
	arm_LSR_S_VC:          "LSR.S.VC",
	arm_LSR_S_HI:          "LSR.S.HI",
	arm_LSR_S_LS:          "LSR.S.LS",
	arm_LSR_S_GE:          "LSR.S.GE",
	arm_LSR_S_LT:          "LSR.S.LT",
	arm_LSR_S_GT:          "LSR.S.GT",
	arm_LSR_S_LE:          "LSR.S.LE",
	arm_LSR_S:             "LSR.S",
	arm_LSR_S_ZZ:          "LSR.S.ZZ",
	arm_MLA_EQ:            "MLA.EQ",
	arm_MLA_NE:            "MLA.NE",
	arm_MLA_CS:            "MLA.CS",
	arm_MLA_CC:            "MLA.CC",
	arm_MLA_MI:            "MLA.MI",
	arm_MLA_PL:            "MLA.PL",
	arm_MLA_VS:            "MLA.VS",
	arm_MLA_VC:            "MLA.VC",
	arm_MLA_HI:            "MLA.HI",
	arm_MLA_LS:            "MLA.LS",
	arm_MLA_GE:            "MLA.GE",
	arm_MLA_LT:            "MLA.LT",
	arm_MLA_GT:            "MLA.GT",
	arm_MLA_LE:            "MLA.LE",
	arm_MLA:               "MLA",
	arm_MLA_ZZ:            "MLA.ZZ",
	arm_MLA_S_EQ:          "MLA.S.EQ",
	arm_MLA_S_NE:          "MLA.S.NE",
	arm_MLA_S_CS:          "MLA.S.CS",
	arm_MLA_S_CC:          "MLA.S.CC",
	arm_MLA_S_MI:          "MLA.S.MI",
	arm_MLA_S_PL:          "MLA.S.PL",
	arm_MLA_S_VS:          "MLA.S.VS",
	arm_MLA_S_VC:          "MLA.S.VC",
	arm_MLA_S_HI:          "MLA.S.HI",
	arm_MLA_S_LS:          "MLA.S.LS",
	arm_MLA_S_GE:          "MLA.S.GE",
	arm_MLA_S_LT:          "MLA.S.LT",
	arm_MLA_S_GT:          "MLA.S.GT",
	arm_MLA_S_LE:          "MLA.S.LE",
	arm_MLA_S:             "MLA.S",
	arm_MLA_S_ZZ:          "MLA.S.ZZ",
	arm_MLS_EQ:            "MLS.EQ",
	arm_MLS_NE:            "MLS.NE",
	arm_MLS_CS:            "MLS.CS",
	arm_MLS_CC:            "MLS.CC",
	arm_MLS_MI:            "MLS.MI",
	arm_MLS_PL:            "MLS.PL",
	arm_MLS_VS:            "MLS.VS",
	arm_MLS_VC:            "MLS.VC",
	arm_MLS_HI:            "MLS.HI",
	arm_MLS_LS:            "MLS.LS",
	arm_MLS_GE:            "MLS.GE",
	arm_MLS_LT:            "MLS.LT",
	arm_MLS_GT:            "MLS.GT",
	arm_MLS_LE:            "MLS.LE",
	arm_MLS:               "MLS",
	arm_MLS_ZZ:            "MLS.ZZ",
	arm_MOV_EQ:            "MOV.EQ",
	arm_MOV_NE:            "MOV.NE",
	arm_MOV_CS:            "MOV.CS",
	arm_MOV_CC:            "MOV.CC",
	arm_MOV_MI:            "MOV.MI",
	arm_MOV_PL:            "MOV.PL",
	arm_MOV_VS:            "MOV.VS",
	arm_MOV_VC:            "MOV.VC",
	arm_MOV_HI:            "MOV.HI",
	arm_MOV_LS:            "MOV.LS",
	arm_MOV_GE:            "MOV.GE",
	arm_MOV_LT:            "MOV.LT",
	arm_MOV_GT:            "MOV.GT",
	arm_MOV_LE:            "MOV.LE",
	arm_MOV:               "MOV",
	arm_MOV_ZZ:            "MOV.ZZ",
	arm_MOV_S_EQ:          "MOV.S.EQ",
	arm_MOV_S_NE:          "MOV.S.NE",
	arm_MOV_S_CS:          "MOV.S.CS",
	arm_MOV_S_CC:          "MOV.S.CC",
	arm_MOV_S_MI:          "MOV.S.MI",
	arm_MOV_S_PL:          "MOV.S.PL",
	arm_MOV_S_VS:          "MOV.S.VS",
	arm_MOV_S_VC:          "MOV.S.VC",
	arm_MOV_S_HI:          "MOV.S.HI",
	arm_MOV_S_LS:          "MOV.S.LS",
	arm_MOV_S_GE:          "MOV.S.GE",
	arm_MOV_S_LT:          "MOV.S.LT",
	arm_MOV_S_GT:          "MOV.S.GT",
	arm_MOV_S_LE:          "MOV.S.LE",
	arm_MOV_S:             "MOV.S",
	arm_MOV_S_ZZ:          "MOV.S.ZZ",
	arm_MOVT_EQ:           "MOVT.EQ",
	arm_MOVT_NE:           "MOVT.NE",
	arm_MOVT_CS:           "MOVT.CS",
	arm_MOVT_CC:           "MOVT.CC",
	arm_MOVT_MI:           "MOVT.MI",
	arm_MOVT_PL:           "MOVT.PL",
	arm_MOVT_VS:           "MOVT.VS",
	arm_MOVT_VC:           "MOVT.VC",
	arm_MOVT_HI:           "MOVT.HI",
	arm_MOVT_LS:           "MOVT.LS",
	arm_MOVT_GE:           "MOVT.GE",
	arm_MOVT_LT:           "MOVT.LT",
	arm_MOVT_GT:           "MOVT.GT",
	arm_MOVT_LE:           "MOVT.LE",
	arm_MOVT:              "MOVT",
	arm_MOVT_ZZ:           "MOVT.ZZ",
	arm_MOVW_EQ:           "MOVW.EQ",
	arm_MOVW_NE:           "MOVW.NE",
	arm_MOVW_CS:           "MOVW.CS",
	arm_MOVW_CC:           "MOVW.CC",
	arm_MOVW_MI:           "MOVW.MI",
	arm_MOVW_PL:           "MOVW.PL",
	arm_MOVW_VS:           "MOVW.VS",
	arm_MOVW_VC:           "MOVW.VC",
	arm_MOVW_HI:           "MOVW.HI",
	arm_MOVW_LS:           "MOVW.LS",
	arm_MOVW_GE:           "MOVW.GE",
	arm_MOVW_LT:           "MOVW.LT",
	arm_MOVW_GT:           "MOVW.GT",
	arm_MOVW_LE:           "MOVW.LE",
	arm_MOVW:              "MOVW",
	arm_MOVW_ZZ:           "MOVW.ZZ",
	arm_MRS_EQ:            "MRS.EQ",
	arm_MRS_NE:            "MRS.NE",
	arm_MRS_CS:            "MRS.CS",
	arm_MRS_CC:            "MRS.CC",
	arm_MRS_MI:            "MRS.MI",
	arm_MRS_PL:            "MRS.PL",
	arm_MRS_VS:            "MRS.VS",
	arm_MRS_VC:            "MRS.VC",
	arm_MRS_HI:            "MRS.HI",
	arm_MRS_LS:            "MRS.LS",
	arm_MRS_GE:            "MRS.GE",
	arm_MRS_LT:            "MRS.LT",
	arm_MRS_GT:            "MRS.GT",
	arm_MRS_LE:            "MRS.LE",
	arm_MRS:               "MRS",
	arm_MRS_ZZ:            "MRS.ZZ",
	arm_MUL_EQ:            "MUL.EQ",
	arm_MUL_NE:            "MUL.NE",
	arm_MUL_CS:            "MUL.CS",
	arm_MUL_CC:            "MUL.CC",
	arm_MUL_MI:            "MUL.MI",
	arm_MUL_PL:            "MUL.PL",
	arm_MUL_VS:            "MUL.VS",
	arm_MUL_VC:            "MUL.VC",
	arm_MUL_HI:            "MUL.HI",
	arm_MUL_LS:            "MUL.LS",
	arm_MUL_GE:            "MUL.GE",
	arm_MUL_LT:            "MUL.LT",
	arm_MUL_GT:            "MUL.GT",
	arm_MUL_LE:            "MUL.LE",
	arm_MUL:               "MUL",
	arm_MUL_ZZ:            "MUL.ZZ",
	arm_MUL_S_EQ:          "MUL.S.EQ",
	arm_MUL_S_NE:          "MUL.S.NE",
	arm_MUL_S_CS:          "MUL.S.CS",
	arm_MUL_S_CC:          "MUL.S.CC",
	arm_MUL_S_MI:          "MUL.S.MI",
	arm_MUL_S_PL:          "MUL.S.PL",
	arm_MUL_S_VS:          "MUL.S.VS",
	arm_MUL_S_VC:          "MUL.S.VC",
	arm_MUL_S_HI:          "MUL.S.HI",
	arm_MUL_S_LS:          "MUL.S.LS",
	arm_MUL_S_GE:          "MUL.S.GE",
	arm_MUL_S_LT:          "MUL.S.LT",
	arm_MUL_S_GT:          "MUL.S.GT",
	arm_MUL_S_LE:          "MUL.S.LE",
	arm_MUL_S:             "MUL.S",
	arm_MUL_S_ZZ:          "MUL.S.ZZ",
	arm_MVN_EQ:            "MVN.EQ",
	arm_MVN_NE:            "MVN.NE",
	arm_MVN_CS:            "MVN.CS",
	arm_MVN_CC:            "MVN.CC",
	arm_MVN_MI:            "MVN.MI",
	arm_MVN_PL:            "MVN.PL",
	arm_MVN_VS:            "MVN.VS",
	arm_MVN_VC:            "MVN.VC",
	arm_MVN_HI:            "MVN.HI",
	arm_MVN_LS:            "MVN.LS",
	arm_MVN_GE:            "MVN.GE",
	arm_MVN_LT:            "MVN.LT",
	arm_MVN_GT:            "MVN.GT",
	arm_MVN_LE:            "MVN.LE",
	arm_MVN:               "MVN",
	arm_MVN_ZZ:            "MVN.ZZ",
	arm_MVN_S_EQ:          "MVN.S.EQ",
	arm_MVN_S_NE:          "MVN.S.NE",
	arm_MVN_S_CS:          "MVN.S.CS",
	arm_MVN_S_CC:          "MVN.S.CC",
	arm_MVN_S_MI:          "MVN.S.MI",
	arm_MVN_S_PL:          "MVN.S.PL",
	arm_MVN_S_VS:          "MVN.S.VS",
	arm_MVN_S_VC:          "MVN.S.VC",
	arm_MVN_S_HI:          "MVN.S.HI",
	arm_MVN_S_LS:          "MVN.S.LS",
	arm_MVN_S_GE:          "MVN.S.GE",
	arm_MVN_S_LT:          "MVN.S.LT",
	arm_MVN_S_GT:          "MVN.S.GT",
	arm_MVN_S_LE:          "MVN.S.LE",
	arm_MVN_S:             "MVN.S",
	arm_MVN_S_ZZ:          "MVN.S.ZZ",
	arm_NOP_EQ:            "NOP.EQ",
	arm_NOP_NE:            "NOP.NE",
	arm_NOP_CS:            "NOP.CS",
	arm_NOP_CC:            "NOP.CC",
	arm_NOP_MI:            "NOP.MI",
	arm_NOP_PL:            "NOP.PL",
	arm_NOP_VS:            "NOP.VS",
	arm_NOP_VC:            "NOP.VC",
	arm_NOP_HI:            "NOP.HI",
	arm_NOP_LS:            "NOP.LS",
	arm_NOP_GE:            "NOP.GE",
	arm_NOP_LT:            "NOP.LT",
	arm_NOP_GT:            "NOP.GT",
	arm_NOP_LE:            "NOP.LE",
	arm_NOP:               "NOP",
	arm_NOP_ZZ:            "NOP.ZZ",
	arm_ORR_EQ:            "ORR.EQ",
	arm_ORR_NE:            "ORR.NE",
	arm_ORR_CS:            "ORR.CS",
	arm_ORR_CC:            "ORR.CC",
	arm_ORR_MI:            "ORR.MI",
	arm_ORR_PL:            "ORR.PL",
	arm_ORR_VS:            "ORR.VS",
	arm_ORR_VC:            "ORR.VC",
	arm_ORR_HI:            "ORR.HI",
	arm_ORR_LS:            "ORR.LS",
	arm_ORR_GE:            "ORR.GE",
	arm_ORR_LT:            "ORR.LT",
	arm_ORR_GT:            "ORR.GT",
	arm_ORR_LE:            "ORR.LE",
	arm_ORR:               "ORR",
	arm_ORR_ZZ:            "ORR.ZZ",
	arm_ORR_S_EQ:          "ORR.S.EQ",
	arm_ORR_S_NE:          "ORR.S.NE",
	arm_ORR_S_CS:          "ORR.S.CS",
	arm_ORR_S_CC:          "ORR.S.CC",
	arm_ORR_S_MI:          "ORR.S.MI",
	arm_ORR_S_PL:          "ORR.S.PL",
	arm_ORR_S_VS:          "ORR.S.VS",
	arm_ORR_S_VC:          "ORR.S.VC",
	arm_ORR_S_HI:          "ORR.S.HI",
	arm_ORR_S_LS:          "ORR.S.LS",
	arm_ORR_S_GE:          "ORR.S.GE",
	arm_ORR_S_LT:          "ORR.S.LT",
	arm_ORR_S_GT:          "ORR.S.GT",
	arm_ORR_S_LE:          "ORR.S.LE",
	arm_ORR_S:             "ORR.S",
	arm_ORR_S_ZZ:          "ORR.S.ZZ",
	arm_PKHBT_EQ:          "PKHBT.EQ",
	arm_PKHBT_NE:          "PKHBT.NE",
	arm_PKHBT_CS:          "PKHBT.CS",
	arm_PKHBT_CC:          "PKHBT.CC",
	arm_PKHBT_MI:          "PKHBT.MI",
	arm_PKHBT_PL:          "PKHBT.PL",
	arm_PKHBT_VS:          "PKHBT.VS",
	arm_PKHBT_VC:          "PKHBT.VC",
	arm_PKHBT_HI:          "PKHBT.HI",
	arm_PKHBT_LS:          "PKHBT.LS",
	arm_PKHBT_GE:          "PKHBT.GE",
	arm_PKHBT_LT:          "PKHBT.LT",
	arm_PKHBT_GT:          "PKHBT.GT",
	arm_PKHBT_LE:          "PKHBT.LE",
	arm_PKHBT:             "PKHBT",
	arm_PKHBT_ZZ:          "PKHBT.ZZ",
	arm_PKHTB_EQ:          "PKHTB.EQ",
	arm_PKHTB_NE:          "PKHTB.NE",
	arm_PKHTB_CS:          "PKHTB.CS",
	arm_PKHTB_CC:          "PKHTB.CC",
	arm_PKHTB_MI:          "PKHTB.MI",
	arm_PKHTB_PL:          "PKHTB.PL",
	arm_PKHTB_VS:          "PKHTB.VS",
	arm_PKHTB_VC:          "PKHTB.VC",
	arm_PKHTB_HI:          "PKHTB.HI",
	arm_PKHTB_LS:          "PKHTB.LS",
	arm_PKHTB_GE:          "PKHTB.GE",
	arm_PKHTB_LT:          "PKHTB.LT",
	arm_PKHTB_GT:          "PKHTB.GT",
	arm_PKHTB_LE:          "PKHTB.LE",
	arm_PKHTB:             "PKHTB",
	arm_PKHTB_ZZ:          "PKHTB.ZZ",
	arm_PLD_W:             "PLD.W",
	arm_PLD:               "PLD",
	arm_PLI:               "PLI",
	arm_POP_EQ:            "POP.EQ",
	arm_POP_NE:            "POP.NE",
	arm_POP_CS:            "POP.CS",
	arm_POP_CC:            "POP.CC",
	arm_POP_MI:            "POP.MI",
	arm_POP_PL:            "POP.PL",
	arm_POP_VS:            "POP.VS",
	arm_POP_VC:            "POP.VC",
	arm_POP_HI:            "POP.HI",
	arm_POP_LS:            "POP.LS",
	arm_POP_GE:            "POP.GE",
	arm_POP_LT:            "POP.LT",
	arm_POP_GT:            "POP.GT",
	arm_POP_LE:            "POP.LE",
	arm_POP:               "POP",
	arm_POP_ZZ:            "POP.ZZ",
	arm_PUSH_EQ:           "PUSH.EQ",
	arm_PUSH_NE:           "PUSH.NE",
	arm_PUSH_CS:           "PUSH.CS",
	arm_PUSH_CC:           "PUSH.CC",
	arm_PUSH_MI:           "PUSH.MI",
	arm_PUSH_PL:           "PUSH.PL",
	arm_PUSH_VS:           "PUSH.VS",
	arm_PUSH_VC:           "PUSH.VC",
	arm_PUSH_HI:           "PUSH.HI",
	arm_PUSH_LS:           "PUSH.LS",
	arm_PUSH_GE:           "PUSH.GE",
	arm_PUSH_LT:           "PUSH.LT",
	arm_PUSH_GT:           "PUSH.GT",
	arm_PUSH_LE:           "PUSH.LE",
	arm_PUSH:              "PUSH",
	arm_PUSH_ZZ:           "PUSH.ZZ",
	arm_QADD_EQ:           "QADD.EQ",
	arm_QADD_NE:           "QADD.NE",
	arm_QADD_CS:           "QADD.CS",
	arm_QADD_CC:           "QADD.CC",
	arm_QADD_MI:           "QADD.MI",
	arm_QADD_PL:           "QADD.PL",
	arm_QADD_VS:           "QADD.VS",
	arm_QADD_VC:           "QADD.VC",
	arm_QADD_HI:           "QADD.HI",
	arm_QADD_LS:           "QADD.LS",
	arm_QADD_GE:           "QADD.GE",
	arm_QADD_LT:           "QADD.LT",
	arm_QADD_GT:           "QADD.GT",
	arm_QADD_LE:           "QADD.LE",
	arm_QADD:              "QADD",
	arm_QADD_ZZ:           "QADD.ZZ",
	arm_QADD16_EQ:         "QADD16.EQ",
	arm_QADD16_NE:         "QADD16.NE",
	arm_QADD16_CS:         "QADD16.CS",
	arm_QADD16_CC:         "QADD16.CC",
	arm_QADD16_MI:         "QADD16.MI",
	arm_QADD16_PL:         "QADD16.PL",
	arm_QADD16_VS:         "QADD16.VS",
	arm_QADD16_VC:         "QADD16.VC",
	arm_QADD16_HI:         "QADD16.HI",
	arm_QADD16_LS:         "QADD16.LS",
	arm_QADD16_GE:         "QADD16.GE",
	arm_QADD16_LT:         "QADD16.LT",
	arm_QADD16_GT:         "QADD16.GT",
	arm_QADD16_LE:         "QADD16.LE",
	arm_QADD16:            "QADD16",
	arm_QADD16_ZZ:         "QADD16.ZZ",
	arm_QADD8_EQ:          "QADD8.EQ",
	arm_QADD8_NE:          "QADD8.NE",
	arm_QADD8_CS:          "QADD8.CS",
	arm_QADD8_CC:          "QADD8.CC",
	arm_QADD8_MI:          "QADD8.MI",
	arm_QADD8_PL:          "QADD8.PL",
	arm_QADD8_VS:          "QADD8.VS",
	arm_QADD8_VC:          "QADD8.VC",
	arm_QADD8_HI:          "QADD8.HI",
	arm_QADD8_LS:          "QADD8.LS",
	arm_QADD8_GE:          "QADD8.GE",
	arm_QADD8_LT:          "QADD8.LT",
	arm_QADD8_GT:          "QADD8.GT",
	arm_QADD8_LE:          "QADD8.LE",
	arm_QADD8:             "QADD8",
	arm_QADD8_ZZ:          "QADD8.ZZ",
	arm_QASX_EQ:           "QASX.EQ",
	arm_QASX_NE:           "QASX.NE",
	arm_QASX_CS:           "QASX.CS",
	arm_QASX_CC:           "QASX.CC",
	arm_QASX_MI:           "QASX.MI",
	arm_QASX_PL:           "QASX.PL",
	arm_QASX_VS:           "QASX.VS",
	arm_QASX_VC:           "QASX.VC",
	arm_QASX_HI:           "QASX.HI",
	arm_QASX_LS:           "QASX.LS",
	arm_QASX_GE:           "QASX.GE",
	arm_QASX_LT:           "QASX.LT",
	arm_QASX_GT:           "QASX.GT",
	arm_QASX_LE:           "QASX.LE",
	arm_QASX:              "QASX",
	arm_QASX_ZZ:           "QASX.ZZ",
	arm_QDADD_EQ:          "QDADD.EQ",
	arm_QDADD_NE:          "QDADD.NE",
	arm_QDADD_CS:          "QDADD.CS",
	arm_QDADD_CC:          "QDADD.CC",
	arm_QDADD_MI:          "QDADD.MI",
	arm_QDADD_PL:          "QDADD.PL",
	arm_QDADD_VS:          "QDADD.VS",
	arm_QDADD_VC:          "QDADD.VC",
	arm_QDADD_HI:          "QDADD.HI",
	arm_QDADD_LS:          "QDADD.LS",
	arm_QDADD_GE:          "QDADD.GE",
	arm_QDADD_LT:          "QDADD.LT",
	arm_QDADD_GT:          "QDADD.GT",
	arm_QDADD_LE:          "QDADD.LE",
	arm_QDADD:             "QDADD",
	arm_QDADD_ZZ:          "QDADD.ZZ",
	arm_QDSUB_EQ:          "QDSUB.EQ",
	arm_QDSUB_NE:          "QDSUB.NE",
	arm_QDSUB_CS:          "QDSUB.CS",
	arm_QDSUB_CC:          "QDSUB.CC",
	arm_QDSUB_MI:          "QDSUB.MI",
	arm_QDSUB_PL:          "QDSUB.PL",
	arm_QDSUB_VS:          "QDSUB.VS",
	arm_QDSUB_VC:          "QDSUB.VC",
	arm_QDSUB_HI:          "QDSUB.HI",
	arm_QDSUB_LS:          "QDSUB.LS",
	arm_QDSUB_GE:          "QDSUB.GE",
	arm_QDSUB_LT:          "QDSUB.LT",
	arm_QDSUB_GT:          "QDSUB.GT",
	arm_QDSUB_LE:          "QDSUB.LE",
	arm_QDSUB:             "QDSUB",
	arm_QDSUB_ZZ:          "QDSUB.ZZ",
	arm_QSAX_EQ:           "QSAX.EQ",
	arm_QSAX_NE:           "QSAX.NE",
	arm_QSAX_CS:           "QSAX.CS",
	arm_QSAX_CC:           "QSAX.CC",
	arm_QSAX_MI:           "QSAX.MI",
	arm_QSAX_PL:           "QSAX.PL",
	arm_QSAX_VS:           "QSAX.VS",
	arm_QSAX_VC:           "QSAX.VC",
	arm_QSAX_HI:           "QSAX.HI",
	arm_QSAX_LS:           "QSAX.LS",
	arm_QSAX_GE:           "QSAX.GE",
	arm_QSAX_LT:           "QSAX.LT",
	arm_QSAX_GT:           "QSAX.GT",
	arm_QSAX_LE:           "QSAX.LE",
	arm_QSAX:              "QSAX",
	arm_QSAX_ZZ:           "QSAX.ZZ",
	arm_QSUB_EQ:           "QSUB.EQ",
	arm_QSUB_NE:           "QSUB.NE",
	arm_QSUB_CS:           "QSUB.CS",
	arm_QSUB_CC:           "QSUB.CC",
	arm_QSUB_MI:           "QSUB.MI",
	arm_QSUB_PL:           "QSUB.PL",
	arm_QSUB_VS:           "QSUB.VS",
	arm_QSUB_VC:           "QSUB.VC",
	arm_QSUB_HI:           "QSUB.HI",
	arm_QSUB_LS:           "QSUB.LS",
	arm_QSUB_GE:           "QSUB.GE",
	arm_QSUB_LT:           "QSUB.LT",
	arm_QSUB_GT:           "QSUB.GT",
	arm_QSUB_LE:           "QSUB.LE",
	arm_QSUB:              "QSUB",
	arm_QSUB_ZZ:           "QSUB.ZZ",
	arm_QSUB16_EQ:         "QSUB16.EQ",
	arm_QSUB16_NE:         "QSUB16.NE",
	arm_QSUB16_CS:         "QSUB16.CS",
	arm_QSUB16_CC:         "QSUB16.CC",
	arm_QSUB16_MI:         "QSUB16.MI",
	arm_QSUB16_PL:         "QSUB16.PL",
	arm_QSUB16_VS:         "QSUB16.VS",
	arm_QSUB16_VC:         "QSUB16.VC",
	arm_QSUB16_HI:         "QSUB16.HI",
	arm_QSUB16_LS:         "QSUB16.LS",
	arm_QSUB16_GE:         "QSUB16.GE",
	arm_QSUB16_LT:         "QSUB16.LT",
	arm_QSUB16_GT:         "QSUB16.GT",
	arm_QSUB16_LE:         "QSUB16.LE",
	arm_QSUB16:            "QSUB16",
	arm_QSUB16_ZZ:         "QSUB16.ZZ",
	arm_QSUB8_EQ:          "QSUB8.EQ",
	arm_QSUB8_NE:          "QSUB8.NE",
	arm_QSUB8_CS:          "QSUB8.CS",
	arm_QSUB8_CC:          "QSUB8.CC",
	arm_QSUB8_MI:          "QSUB8.MI",
	arm_QSUB8_PL:          "QSUB8.PL",
	arm_QSUB8_VS:          "QSUB8.VS",
	arm_QSUB8_VC:          "QSUB8.VC",
	arm_QSUB8_HI:          "QSUB8.HI",
	arm_QSUB8_LS:          "QSUB8.LS",
	arm_QSUB8_GE:          "QSUB8.GE",
	arm_QSUB8_LT:          "QSUB8.LT",
	arm_QSUB8_GT:          "QSUB8.GT",
	arm_QSUB8_LE:          "QSUB8.LE",
	arm_QSUB8:             "QSUB8",
	arm_QSUB8_ZZ:          "QSUB8.ZZ",
	arm_RBIT_EQ:           "RBIT.EQ",
	arm_RBIT_NE:           "RBIT.NE",
	arm_RBIT_CS:           "RBIT.CS",
	arm_RBIT_CC:           "RBIT.CC",
	arm_RBIT_MI:           "RBIT.MI",
	arm_RBIT_PL:           "RBIT.PL",
	arm_RBIT_VS:           "RBIT.VS",
	arm_RBIT_VC:           "RBIT.VC",
	arm_RBIT_HI:           "RBIT.HI",
	arm_RBIT_LS:           "RBIT.LS",
	arm_RBIT_GE:           "RBIT.GE",
	arm_RBIT_LT:           "RBIT.LT",
	arm_RBIT_GT:           "RBIT.GT",
	arm_RBIT_LE:           "RBIT.LE",
	arm_RBIT:              "RBIT",
	arm_RBIT_ZZ:           "RBIT.ZZ",
	arm_REV_EQ:            "REV.EQ",
	arm_REV_NE:            "REV.NE",
	arm_REV_CS:            "REV.CS",
	arm_REV_CC:            "REV.CC",
	arm_REV_MI:            "REV.MI",
	arm_REV_PL:            "REV.PL",
	arm_REV_VS:            "REV.VS",
	arm_REV_VC:            "REV.VC",
	arm_REV_HI:            "REV.HI",
	arm_REV_LS:            "REV.LS",
	arm_REV_GE:            "REV.GE",
	arm_REV_LT:            "REV.LT",
	arm_REV_GT:            "REV.GT",
	arm_REV_LE:            "REV.LE",
	arm_REV:               "REV",
	arm_REV_ZZ:            "REV.ZZ",
	arm_REV16_EQ:          "REV16.EQ",
	arm_REV16_NE:          "REV16.NE",
	arm_REV16_CS:          "REV16.CS",
	arm_REV16_CC:          "REV16.CC",
	arm_REV16_MI:          "REV16.MI",
	arm_REV16_PL:          "REV16.PL",
	arm_REV16_VS:          "REV16.VS",
	arm_REV16_VC:          "REV16.VC",
	arm_REV16_HI:          "REV16.HI",
	arm_REV16_LS:          "REV16.LS",
	arm_REV16_GE:          "REV16.GE",
	arm_REV16_LT:          "REV16.LT",
	arm_REV16_GT:          "REV16.GT",
	arm_REV16_LE:          "REV16.LE",
	arm_REV16:             "REV16",
	arm_REV16_ZZ:          "REV16.ZZ",
	arm_REVSH_EQ:          "REVSH.EQ",
	arm_REVSH_NE:          "REVSH.NE",
	arm_REVSH_CS:          "REVSH.CS",
	arm_REVSH_CC:          "REVSH.CC",
	arm_REVSH_MI:          "REVSH.MI",
	arm_REVSH_PL:          "REVSH.PL",
	arm_REVSH_VS:          "REVSH.VS",
	arm_REVSH_VC:          "REVSH.VC",
	arm_REVSH_HI:          "REVSH.HI",
	arm_REVSH_LS:          "REVSH.LS",
	arm_REVSH_GE:          "REVSH.GE",
	arm_REVSH_LT:          "REVSH.LT",
	arm_REVSH_GT:          "REVSH.GT",
	arm_REVSH_LE:          "REVSH.LE",
	arm_REVSH:             "REVSH",
	arm_REVSH_ZZ:          "REVSH.ZZ",
	arm_ROR_EQ:            "ROR.EQ",
	arm_ROR_NE:            "ROR.NE",
	arm_ROR_CS:            "ROR.CS",
	arm_ROR_CC:            "ROR.CC",
	arm_ROR_MI:            "ROR.MI",
	arm_ROR_PL:            "ROR.PL",
	arm_ROR_VS:            "ROR.VS",
	arm_ROR_VC:            "ROR.VC",
	arm_ROR_HI:            "ROR.HI",
	arm_ROR_LS:            "ROR.LS",
	arm_ROR_GE:            "ROR.GE",
	arm_ROR_LT:            "ROR.LT",
	arm_ROR_GT:            "ROR.GT",
	arm_ROR_LE:            "ROR.LE",
	arm_ROR:               "ROR",
	arm_ROR_ZZ:            "ROR.ZZ",
	arm_ROR_S_EQ:          "ROR.S.EQ",
	arm_ROR_S_NE:          "ROR.S.NE",
	arm_ROR_S_CS:          "ROR.S.CS",
	arm_ROR_S_CC:          "ROR.S.CC",
	arm_ROR_S_MI:          "ROR.S.MI",
	arm_ROR_S_PL:          "ROR.S.PL",
	arm_ROR_S_VS:          "ROR.S.VS",
	arm_ROR_S_VC:          "ROR.S.VC",
	arm_ROR_S_HI:          "ROR.S.HI",
	arm_ROR_S_LS:          "ROR.S.LS",
	arm_ROR_S_GE:          "ROR.S.GE",
	arm_ROR_S_LT:          "ROR.S.LT",
	arm_ROR_S_GT:          "ROR.S.GT",
	arm_ROR_S_LE:          "ROR.S.LE",
	arm_ROR_S:             "ROR.S",
	arm_ROR_S_ZZ:          "ROR.S.ZZ",
	arm_RRX_EQ:            "RRX.EQ",
	arm_RRX_NE:            "RRX.NE",
	arm_RRX_CS:            "RRX.CS",
	arm_RRX_CC:            "RRX.CC",
	arm_RRX_MI:            "RRX.MI",
	arm_RRX_PL:            "RRX.PL",
	arm_RRX_VS:            "RRX.VS",
	arm_RRX_VC:            "RRX.VC",
	arm_RRX_HI:            "RRX.HI",
	arm_RRX_LS:            "RRX.LS",
	arm_RRX_GE:            "RRX.GE",
	arm_RRX_LT:            "RRX.LT",
	arm_RRX_GT:            "RRX.GT",
	arm_RRX_LE:            "RRX.LE",
	arm_RRX:               "RRX",
	arm_RRX_ZZ:            "RRX.ZZ",
	arm_RRX_S_EQ:          "RRX.S.EQ",
	arm_RRX_S_NE:          "RRX.S.NE",
	arm_RRX_S_CS:          "RRX.S.CS",
	arm_RRX_S_CC:          "RRX.S.CC",
	arm_RRX_S_MI:          "RRX.S.MI",
	arm_RRX_S_PL:          "RRX.S.PL",
	arm_RRX_S_VS:          "RRX.S.VS",
	arm_RRX_S_VC:          "RRX.S.VC",
	arm_RRX_S_HI:          "RRX.S.HI",
	arm_RRX_S_LS:          "RRX.S.LS",
	arm_RRX_S_GE:          "RRX.S.GE",
	arm_RRX_S_LT:          "RRX.S.LT",
	arm_RRX_S_GT:          "RRX.S.GT",
	arm_RRX_S_LE:          "RRX.S.LE",
	arm_RRX_S:             "RRX.S",
	arm_RRX_S_ZZ:          "RRX.S.ZZ",
	arm_RSB_EQ:            "RSB.EQ",
	arm_RSB_NE:            "RSB.NE",
	arm_RSB_CS:            "RSB.CS",
	arm_RSB_CC:            "RSB.CC",
	arm_RSB_MI:            "RSB.MI",
	arm_RSB_PL:            "RSB.PL",
	arm_RSB_VS:            "RSB.VS",
	arm_RSB_VC:            "RSB.VC",
	arm_RSB_HI:            "RSB.HI",
	arm_RSB_LS:            "RSB.LS",
	arm_RSB_GE:            "RSB.GE",
	arm_RSB_LT:            "RSB.LT",
	arm_RSB_GT:            "RSB.GT",
	arm_RSB_LE:            "RSB.LE",
	arm_RSB:               "RSB",
	arm_RSB_ZZ:            "RSB.ZZ",
	arm_RSB_S_EQ:          "RSB.S.EQ",
	arm_RSB_S_NE:          "RSB.S.NE",
	arm_RSB_S_CS:          "RSB.S.CS",
	arm_RSB_S_CC:          "RSB.S.CC",
	arm_RSB_S_MI:          "RSB.S.MI",
	arm_RSB_S_PL:          "RSB.S.PL",
	arm_RSB_S_VS:          "RSB.S.VS",
	arm_RSB_S_VC:          "RSB.S.VC",
	arm_RSB_S_HI:          "RSB.S.HI",
	arm_RSB_S_LS:          "RSB.S.LS",
	arm_RSB_S_GE:          "RSB.S.GE",
	arm_RSB_S_LT:          "RSB.S.LT",
	arm_RSB_S_GT:          "RSB.S.GT",
	arm_RSB_S_LE:          "RSB.S.LE",
	arm_RSB_S:             "RSB.S",
	arm_RSB_S_ZZ:          "RSB.S.ZZ",
	arm_RSC_EQ:            "RSC.EQ",
	arm_RSC_NE:            "RSC.NE",
	arm_RSC_CS:            "RSC.CS",
	arm_RSC_CC:            "RSC.CC",
	arm_RSC_MI:            "RSC.MI",
	arm_RSC_PL:            "RSC.PL",
	arm_RSC_VS:            "RSC.VS",
	arm_RSC_VC:            "RSC.VC",
	arm_RSC_HI:            "RSC.HI",
	arm_RSC_LS:            "RSC.LS",
	arm_RSC_GE:            "RSC.GE",
	arm_RSC_LT:            "RSC.LT",
	arm_RSC_GT:            "RSC.GT",
	arm_RSC_LE:            "RSC.LE",
	arm_RSC:               "RSC",
	arm_RSC_ZZ:            "RSC.ZZ",
	arm_RSC_S_EQ:          "RSC.S.EQ",
	arm_RSC_S_NE:          "RSC.S.NE",
	arm_RSC_S_CS:          "RSC.S.CS",
	arm_RSC_S_CC:          "RSC.S.CC",
	arm_RSC_S_MI:          "RSC.S.MI",
	arm_RSC_S_PL:          "RSC.S.PL",
	arm_RSC_S_VS:          "RSC.S.VS",
	arm_RSC_S_VC:          "RSC.S.VC",
	arm_RSC_S_HI:          "RSC.S.HI",
	arm_RSC_S_LS:          "RSC.S.LS",
	arm_RSC_S_GE:          "RSC.S.GE",
	arm_RSC_S_LT:          "RSC.S.LT",
	arm_RSC_S_GT:          "RSC.S.GT",
	arm_RSC_S_LE:          "RSC.S.LE",
	arm_RSC_S:             "RSC.S",
	arm_RSC_S_ZZ:          "RSC.S.ZZ",
	arm_SADD16_EQ:         "SADD16.EQ",
	arm_SADD16_NE:         "SADD16.NE",
	arm_SADD16_CS:         "SADD16.CS",
	arm_SADD16_CC:         "SADD16.CC",
	arm_SADD16_MI:         "SADD16.MI",
	arm_SADD16_PL:         "SADD16.PL",
	arm_SADD16_VS:         "SADD16.VS",
	arm_SADD16_VC:         "SADD16.VC",
	arm_SADD16_HI:         "SADD16.HI",
	arm_SADD16_LS:         "SADD16.LS",
	arm_SADD16_GE:         "SADD16.GE",
	arm_SADD16_LT:         "SADD16.LT",
	arm_SADD16_GT:         "SADD16.GT",
	arm_SADD16_LE:         "SADD16.LE",
	arm_SADD16:            "SADD16",
	arm_SADD16_ZZ:         "SADD16.ZZ",
	arm_SADD8_EQ:          "SADD8.EQ",
	arm_SADD8_NE:          "SADD8.NE",
	arm_SADD8_CS:          "SADD8.CS",
	arm_SADD8_CC:          "SADD8.CC",
	arm_SADD8_MI:          "SADD8.MI",
	arm_SADD8_PL:          "SADD8.PL",
	arm_SADD8_VS:          "SADD8.VS",
	arm_SADD8_VC:          "SADD8.VC",
	arm_SADD8_HI:          "SADD8.HI",
	arm_SADD8_LS:          "SADD8.LS",
	arm_SADD8_GE:          "SADD8.GE",
	arm_SADD8_LT:          "SADD8.LT",
	arm_SADD8_GT:          "SADD8.GT",
	arm_SADD8_LE:          "SADD8.LE",
	arm_SADD8:             "SADD8",
	arm_SADD8_ZZ:          "SADD8.ZZ",
	arm_SASX_EQ:           "SASX.EQ",
	arm_SASX_NE:           "SASX.NE",
	arm_SASX_CS:           "SASX.CS",
	arm_SASX_CC:           "SASX.CC",
	arm_SASX_MI:           "SASX.MI",
	arm_SASX_PL:           "SASX.PL",
	arm_SASX_VS:           "SASX.VS",
	arm_SASX_VC:           "SASX.VC",
	arm_SASX_HI:           "SASX.HI",
	arm_SASX_LS:           "SASX.LS",
	arm_SASX_GE:           "SASX.GE",
	arm_SASX_LT:           "SASX.LT",
	arm_SASX_GT:           "SASX.GT",
	arm_SASX_LE:           "SASX.LE",
	arm_SASX:              "SASX",
	arm_SASX_ZZ:           "SASX.ZZ",
	arm_SBC_EQ:            "SBC.EQ",
	arm_SBC_NE:            "SBC.NE",
	arm_SBC_CS:            "SBC.CS",
	arm_SBC_CC:            "SBC.CC",
	arm_SBC_MI:            "SBC.MI",
	arm_SBC_PL:            "SBC.PL",
	arm_SBC_VS:            "SBC.VS",
	arm_SBC_VC:            "SBC.VC",
	arm_SBC_HI:            "SBC.HI",
	arm_SBC_LS:            "SBC.LS",
	arm_SBC_GE:            "SBC.GE",
	arm_SBC_LT:            "SBC.LT",
	arm_SBC_GT:            "SBC.GT",
	arm_SBC_LE:            "SBC.LE",
	arm_SBC:               "SBC",
	arm_SBC_ZZ:            "SBC.ZZ",
	arm_SBC_S_EQ:          "SBC.S.EQ",
	arm_SBC_S_NE:          "SBC.S.NE",
	arm_SBC_S_CS:          "SBC.S.CS",
	arm_SBC_S_CC:          "SBC.S.CC",
	arm_SBC_S_MI:          "SBC.S.MI",
	arm_SBC_S_PL:          "SBC.S.PL",
	arm_SBC_S_VS:          "SBC.S.VS",
	arm_SBC_S_VC:          "SBC.S.VC",
	arm_SBC_S_HI:          "SBC.S.HI",
	arm_SBC_S_LS:          "SBC.S.LS",
	arm_SBC_S_GE:          "SBC.S.GE",
	arm_SBC_S_LT:          "SBC.S.LT",
	arm_SBC_S_GT:          "SBC.S.GT",
	arm_SBC_S_LE:          "SBC.S.LE",
	arm_SBC_S:             "SBC.S",
	arm_SBC_S_ZZ:          "SBC.S.ZZ",
	arm_SBFX_EQ:           "SBFX.EQ",
	arm_SBFX_NE:           "SBFX.NE",
	arm_SBFX_CS:           "SBFX.CS",
	arm_SBFX_CC:           "SBFX.CC",
	arm_SBFX_MI:           "SBFX.MI",
	arm_SBFX_PL:           "SBFX.PL",
	arm_SBFX_VS:           "SBFX.VS",
	arm_SBFX_VC:           "SBFX.VC",
	arm_SBFX_HI:           "SBFX.HI",
	arm_SBFX_LS:           "SBFX.LS",
	arm_SBFX_GE:           "SBFX.GE",
	arm_SBFX_LT:           "SBFX.LT",
	arm_SBFX_GT:           "SBFX.GT",
	arm_SBFX_LE:           "SBFX.LE",
	arm_SBFX:              "SBFX",
	arm_SBFX_ZZ:           "SBFX.ZZ",
	arm_SEL_EQ:            "SEL.EQ",
	arm_SEL_NE:            "SEL.NE",
	arm_SEL_CS:            "SEL.CS",
	arm_SEL_CC:            "SEL.CC",
	arm_SEL_MI:            "SEL.MI",
	arm_SEL_PL:            "SEL.PL",
	arm_SEL_VS:            "SEL.VS",
	arm_SEL_VC:            "SEL.VC",
	arm_SEL_HI:            "SEL.HI",
	arm_SEL_LS:            "SEL.LS",
	arm_SEL_GE:            "SEL.GE",
	arm_SEL_LT:            "SEL.LT",
	arm_SEL_GT:            "SEL.GT",
	arm_SEL_LE:            "SEL.LE",
	arm_SEL:               "SEL",
	arm_SEL_ZZ:            "SEL.ZZ",
	arm_SETEND:            "SETEND",
	arm_SEV_EQ:            "SEV.EQ",
	arm_SEV_NE:            "SEV.NE",
	arm_SEV_CS:            "SEV.CS",
	arm_SEV_CC:            "SEV.CC",
	arm_SEV_MI:            "SEV.MI",
	arm_SEV_PL:            "SEV.PL",
	arm_SEV_VS:            "SEV.VS",
	arm_SEV_VC:            "SEV.VC",
	arm_SEV_HI:            "SEV.HI",
	arm_SEV_LS:            "SEV.LS",
	arm_SEV_GE:            "SEV.GE",
	arm_SEV_LT:            "SEV.LT",
	arm_SEV_GT:            "SEV.GT",
	arm_SEV_LE:            "SEV.LE",
	arm_SEV:               "SEV",
	arm_SEV_ZZ:            "SEV.ZZ",
	arm_SHADD16_EQ:        "SHADD16.EQ",
	arm_SHADD16_NE:        "SHADD16.NE",
	arm_SHADD16_CS:        "SHADD16.CS",
	arm_SHADD16_CC:        "SHADD16.CC",
	arm_SHADD16_MI:        "SHADD16.MI",
	arm_SHADD16_PL:        "SHADD16.PL",
	arm_SHADD16_VS:        "SHADD16.VS",
	arm_SHADD16_VC:        "SHADD16.VC",
	arm_SHADD16_HI:        "SHADD16.HI",
	arm_SHADD16_LS:        "SHADD16.LS",
	arm_SHADD16_GE:        "SHADD16.GE",
	arm_SHADD16_LT:        "SHADD16.LT",
	arm_SHADD16_GT:        "SHADD16.GT",
	arm_SHADD16_LE:        "SHADD16.LE",
	arm_SHADD16:           "SHADD16",
	arm_SHADD16_ZZ:        "SHADD16.ZZ",
	arm_SHADD8_EQ:         "SHADD8.EQ",
	arm_SHADD8_NE:         "SHADD8.NE",
	arm_SHADD8_CS:         "SHADD8.CS",
	arm_SHADD8_CC:         "SHADD8.CC",
	arm_SHADD8_MI:         "SHADD8.MI",
	arm_SHADD8_PL:         "SHADD8.PL",
	arm_SHADD8_VS:         "SHADD8.VS",
	arm_SHADD8_VC:         "SHADD8.VC",
	arm_SHADD8_HI:         "SHADD8.HI",
	arm_SHADD8_LS:         "SHADD8.LS",
	arm_SHADD8_GE:         "SHADD8.GE",
	arm_SHADD8_LT:         "SHADD8.LT",
	arm_SHADD8_GT:         "SHADD8.GT",
	arm_SHADD8_LE:         "SHADD8.LE",
	arm_SHADD8:            "SHADD8",
	arm_SHADD8_ZZ:         "SHADD8.ZZ",
	arm_SHASX_EQ:          "SHASX.EQ",
	arm_SHASX_NE:          "SHASX.NE",
	arm_SHASX_CS:          "SHASX.CS",
	arm_SHASX_CC:          "SHASX.CC",
	arm_SHASX_MI:          "SHASX.MI",
	arm_SHASX_PL:          "SHASX.PL",
	arm_SHASX_VS:          "SHASX.VS",
	arm_SHASX_VC:          "SHASX.VC",
	arm_SHASX_HI:          "SHASX.HI",
	arm_SHASX_LS:          "SHASX.LS",
	arm_SHASX_GE:          "SHASX.GE",
	arm_SHASX_LT:          "SHASX.LT",
	arm_SHASX_GT:          "SHASX.GT",
	arm_SHASX_LE:          "SHASX.LE",
	arm_SHASX:             "SHASX",
	arm_SHASX_ZZ:          "SHASX.ZZ",
	arm_SHSAX_EQ:          "SHSAX.EQ",
	arm_SHSAX_NE:          "SHSAX.NE",
	arm_SHSAX_CS:          "SHSAX.CS",
	arm_SHSAX_CC:          "SHSAX.CC",
	arm_SHSAX_MI:          "SHSAX.MI",
	arm_SHSAX_PL:          "SHSAX.PL",
	arm_SHSAX_VS:          "SHSAX.VS",
	arm_SHSAX_VC:          "SHSAX.VC",
	arm_SHSAX_HI:          "SHSAX.HI",
	arm_SHSAX_LS:          "SHSAX.LS",
	arm_SHSAX_GE:          "SHSAX.GE",
	arm_SHSAX_LT:          "SHSAX.LT",
	arm_SHSAX_GT:          "SHSAX.GT",
	arm_SHSAX_LE:          "SHSAX.LE",
	arm_SHSAX:             "SHSAX",
	arm_SHSAX_ZZ:          "SHSAX.ZZ",
	arm_SHSUB16_EQ:        "SHSUB16.EQ",
	arm_SHSUB16_NE:        "SHSUB16.NE",
	arm_SHSUB16_CS:        "SHSUB16.CS",
	arm_SHSUB16_CC:        "SHSUB16.CC",
	arm_SHSUB16_MI:        "SHSUB16.MI",
	arm_SHSUB16_PL:        "SHSUB16.PL",
	arm_SHSUB16_VS:        "SHSUB16.VS",
	arm_SHSUB16_VC:        "SHSUB16.VC",
	arm_SHSUB16_HI:        "SHSUB16.HI",
	arm_SHSUB16_LS:        "SHSUB16.LS",
	arm_SHSUB16_GE:        "SHSUB16.GE",
	arm_SHSUB16_LT:        "SHSUB16.LT",
	arm_SHSUB16_GT:        "SHSUB16.GT",
	arm_SHSUB16_LE:        "SHSUB16.LE",
	arm_SHSUB16:           "SHSUB16",
	arm_SHSUB16_ZZ:        "SHSUB16.ZZ",
	arm_SHSUB8_EQ:         "SHSUB8.EQ",
	arm_SHSUB8_NE:         "SHSUB8.NE",
	arm_SHSUB8_CS:         "SHSUB8.CS",
	arm_SHSUB8_CC:         "SHSUB8.CC",
	arm_SHSUB8_MI:         "SHSUB8.MI",
	arm_SHSUB8_PL:         "SHSUB8.PL",
	arm_SHSUB8_VS:         "SHSUB8.VS",
	arm_SHSUB8_VC:         "SHSUB8.VC",
	arm_SHSUB8_HI:         "SHSUB8.HI",
	arm_SHSUB8_LS:         "SHSUB8.LS",
	arm_SHSUB8_GE:         "SHSUB8.GE",
	arm_SHSUB8_LT:         "SHSUB8.LT",
	arm_SHSUB8_GT:         "SHSUB8.GT",
	arm_SHSUB8_LE:         "SHSUB8.LE",
	arm_SHSUB8:            "SHSUB8",
	arm_SHSUB8_ZZ:         "SHSUB8.ZZ",
	arm_SMLABB_EQ:         "SMLABB.EQ",
	arm_SMLABB_NE:         "SMLABB.NE",
	arm_SMLABB_CS:         "SMLABB.CS",
	arm_SMLABB_CC:         "SMLABB.CC",
	arm_SMLABB_MI:         "SMLABB.MI",
	arm_SMLABB_PL:         "SMLABB.PL",
	arm_SMLABB_VS:         "SMLABB.VS",
	arm_SMLABB_VC:         "SMLABB.VC",
	arm_SMLABB_HI:         "SMLABB.HI",
	arm_SMLABB_LS:         "SMLABB.LS",
	arm_SMLABB_GE:         "SMLABB.GE",
	arm_SMLABB_LT:         "SMLABB.LT",
	arm_SMLABB_GT:         "SMLABB.GT",
	arm_SMLABB_LE:         "SMLABB.LE",
	arm_SMLABB:            "SMLABB",
	arm_SMLABB_ZZ:         "SMLABB.ZZ",
	arm_SMLABT_EQ:         "SMLABT.EQ",
	arm_SMLABT_NE:         "SMLABT.NE",
	arm_SMLABT_CS:         "SMLABT.CS",
	arm_SMLABT_CC:         "SMLABT.CC",
	arm_SMLABT_MI:         "SMLABT.MI",
	arm_SMLABT_PL:         "SMLABT.PL",
	arm_SMLABT_VS:         "SMLABT.VS",
	arm_SMLABT_VC:         "SMLABT.VC",
	arm_SMLABT_HI:         "SMLABT.HI",
	arm_SMLABT_LS:         "SMLABT.LS",
	arm_SMLABT_GE:         "SMLABT.GE",
	arm_SMLABT_LT:         "SMLABT.LT",
	arm_SMLABT_GT:         "SMLABT.GT",
	arm_SMLABT_LE:         "SMLABT.LE",
	arm_SMLABT:            "SMLABT",
	arm_SMLABT_ZZ:         "SMLABT.ZZ",
	arm_SMLATB_EQ:         "SMLATB.EQ",
	arm_SMLATB_NE:         "SMLATB.NE",
	arm_SMLATB_CS:         "SMLATB.CS",
	arm_SMLATB_CC:         "SMLATB.CC",
	arm_SMLATB_MI:         "SMLATB.MI",
	arm_SMLATB_PL:         "SMLATB.PL",
	arm_SMLATB_VS:         "SMLATB.VS",
	arm_SMLATB_VC:         "SMLATB.VC",
	arm_SMLATB_HI:         "SMLATB.HI",
	arm_SMLATB_LS:         "SMLATB.LS",
	arm_SMLATB_GE:         "SMLATB.GE",
	arm_SMLATB_LT:         "SMLATB.LT",
	arm_SMLATB_GT:         "SMLATB.GT",
	arm_SMLATB_LE:         "SMLATB.LE",
	arm_SMLATB:            "SMLATB",
	arm_SMLATB_ZZ:         "SMLATB.ZZ",
	arm_SMLATT_EQ:         "SMLATT.EQ",
	arm_SMLATT_NE:         "SMLATT.NE",
	arm_SMLATT_CS:         "SMLATT.CS",
	arm_SMLATT_CC:         "SMLATT.CC",
	arm_SMLATT_MI:         "SMLATT.MI",
	arm_SMLATT_PL:         "SMLATT.PL",
	arm_SMLATT_VS:         "SMLATT.VS",
	arm_SMLATT_VC:         "SMLATT.VC",
	arm_SMLATT_HI:         "SMLATT.HI",
	arm_SMLATT_LS:         "SMLATT.LS",
	arm_SMLATT_GE:         "SMLATT.GE",
	arm_SMLATT_LT:         "SMLATT.LT",
	arm_SMLATT_GT:         "SMLATT.GT",
	arm_SMLATT_LE:         "SMLATT.LE",
	arm_SMLATT:            "SMLATT",
	arm_SMLATT_ZZ:         "SMLATT.ZZ",
	arm_SMLAD_EQ:          "SMLAD.EQ",
	arm_SMLAD_NE:          "SMLAD.NE",
	arm_SMLAD_CS:          "SMLAD.CS",
	arm_SMLAD_CC:          "SMLAD.CC",
	arm_SMLAD_MI:          "SMLAD.MI",
	arm_SMLAD_PL:          "SMLAD.PL",
	arm_SMLAD_VS:          "SMLAD.VS",
	arm_SMLAD_VC:          "SMLAD.VC",
	arm_SMLAD_HI:          "SMLAD.HI",
	arm_SMLAD_LS:          "SMLAD.LS",
	arm_SMLAD_GE:          "SMLAD.GE",
	arm_SMLAD_LT:          "SMLAD.LT",
	arm_SMLAD_GT:          "SMLAD.GT",
	arm_SMLAD_LE:          "SMLAD.LE",
	arm_SMLAD:             "SMLAD",
	arm_SMLAD_ZZ:          "SMLAD.ZZ",
	arm_SMLAD_X_EQ:        "SMLAD.X.EQ",
	arm_SMLAD_X_NE:        "SMLAD.X.NE",
	arm_SMLAD_X_CS:        "SMLAD.X.CS",
	arm_SMLAD_X_CC:        "SMLAD.X.CC",
	arm_SMLAD_X_MI:        "SMLAD.X.MI",
	arm_SMLAD_X_PL:        "SMLAD.X.PL",
	arm_SMLAD_X_VS:        "SMLAD.X.VS",
	arm_SMLAD_X_VC:        "SMLAD.X.VC",
	arm_SMLAD_X_HI:        "SMLAD.X.HI",
	arm_SMLAD_X_LS:        "SMLAD.X.LS",
	arm_SMLAD_X_GE:        "SMLAD.X.GE",
	arm_SMLAD_X_LT:        "SMLAD.X.LT",
	arm_SMLAD_X_GT:        "SMLAD.X.GT",
	arm_SMLAD_X_LE:        "SMLAD.X.LE",
	arm_SMLAD_X:           "SMLAD.X",
	arm_SMLAD_X_ZZ:        "SMLAD.X.ZZ",
	arm_SMLAL_EQ:          "SMLAL.EQ",
	arm_SMLAL_NE:          "SMLAL.NE",
	arm_SMLAL_CS:          "SMLAL.CS",
	arm_SMLAL_CC:          "SMLAL.CC",
	arm_SMLAL_MI:          "SMLAL.MI",
	arm_SMLAL_PL:          "SMLAL.PL",
	arm_SMLAL_VS:          "SMLAL.VS",
	arm_SMLAL_VC:          "SMLAL.VC",
	arm_SMLAL_HI:          "SMLAL.HI",
	arm_SMLAL_LS:          "SMLAL.LS",
	arm_SMLAL_GE:          "SMLAL.GE",
	arm_SMLAL_LT:          "SMLAL.LT",
	arm_SMLAL_GT:          "SMLAL.GT",
	arm_SMLAL_LE:          "SMLAL.LE",
	arm_SMLAL:             "SMLAL",
	arm_SMLAL_ZZ:          "SMLAL.ZZ",
	arm_SMLAL_S_EQ:        "SMLAL.S.EQ",
	arm_SMLAL_S_NE:        "SMLAL.S.NE",
	arm_SMLAL_S_CS:        "SMLAL.S.CS",
	arm_SMLAL_S_CC:        "SMLAL.S.CC",
	arm_SMLAL_S_MI:        "SMLAL.S.MI",
	arm_SMLAL_S_PL:        "SMLAL.S.PL",
	arm_SMLAL_S_VS:        "SMLAL.S.VS",
	arm_SMLAL_S_VC:        "SMLAL.S.VC",
	arm_SMLAL_S_HI:        "SMLAL.S.HI",
	arm_SMLAL_S_LS:        "SMLAL.S.LS",
	arm_SMLAL_S_GE:        "SMLAL.S.GE",
	arm_SMLAL_S_LT:        "SMLAL.S.LT",
	arm_SMLAL_S_GT:        "SMLAL.S.GT",
	arm_SMLAL_S_LE:        "SMLAL.S.LE",
	arm_SMLAL_S:           "SMLAL.S",
	arm_SMLAL_S_ZZ:        "SMLAL.S.ZZ",
	arm_SMLALBB_EQ:        "SMLALBB.EQ",
	arm_SMLALBB_NE:        "SMLALBB.NE",
	arm_SMLALBB_CS:        "SMLALBB.CS",
	arm_SMLALBB_CC:        "SMLALBB.CC",
	arm_SMLALBB_MI:        "SMLALBB.MI",
	arm_SMLALBB_PL:        "SMLALBB.PL",
	arm_SMLALBB_VS:        "SMLALBB.VS",
	arm_SMLALBB_VC:        "SMLALBB.VC",
	arm_SMLALBB_HI:        "SMLALBB.HI",
	arm_SMLALBB_LS:        "SMLALBB.LS",
	arm_SMLALBB_GE:        "SMLALBB.GE",
	arm_SMLALBB_LT:        "SMLALBB.LT",
	arm_SMLALBB_GT:        "SMLALBB.GT",
	arm_SMLALBB_LE:        "SMLALBB.LE",
	arm_SMLALBB:           "SMLALBB",
	arm_SMLALBB_ZZ:        "SMLALBB.ZZ",
	arm_SMLALBT_EQ:        "SMLALBT.EQ",
	arm_SMLALBT_NE:        "SMLALBT.NE",
	arm_SMLALBT_CS:        "SMLALBT.CS",
	arm_SMLALBT_CC:        "SMLALBT.CC",
	arm_SMLALBT_MI:        "SMLALBT.MI",
	arm_SMLALBT_PL:        "SMLALBT.PL",
	arm_SMLALBT_VS:        "SMLALBT.VS",
	arm_SMLALBT_VC:        "SMLALBT.VC",
	arm_SMLALBT_HI:        "SMLALBT.HI",
	arm_SMLALBT_LS:        "SMLALBT.LS",
	arm_SMLALBT_GE:        "SMLALBT.GE",
	arm_SMLALBT_LT:        "SMLALBT.LT",
	arm_SMLALBT_GT:        "SMLALBT.GT",
	arm_SMLALBT_LE:        "SMLALBT.LE",
	arm_SMLALBT:           "SMLALBT",
	arm_SMLALBT_ZZ:        "SMLALBT.ZZ",
	arm_SMLALTB_EQ:        "SMLALTB.EQ",
	arm_SMLALTB_NE:        "SMLALTB.NE",
	arm_SMLALTB_CS:        "SMLALTB.CS",
	arm_SMLALTB_CC:        "SMLALTB.CC",
	arm_SMLALTB_MI:        "SMLALTB.MI",
	arm_SMLALTB_PL:        "SMLALTB.PL",
	arm_SMLALTB_VS:        "SMLALTB.VS",
	arm_SMLALTB_VC:        "SMLALTB.VC",
	arm_SMLALTB_HI:        "SMLALTB.HI",
	arm_SMLALTB_LS:        "SMLALTB.LS",
	arm_SMLALTB_GE:        "SMLALTB.GE",
	arm_SMLALTB_LT:        "SMLALTB.LT",
	arm_SMLALTB_GT:        "SMLALTB.GT",
	arm_SMLALTB_LE:        "SMLALTB.LE",
	arm_SMLALTB:           "SMLALTB",
	arm_SMLALTB_ZZ:        "SMLALTB.ZZ",
	arm_SMLALTT_EQ:        "SMLALTT.EQ",
	arm_SMLALTT_NE:        "SMLALTT.NE",
	arm_SMLALTT_CS:        "SMLALTT.CS",
	arm_SMLALTT_CC:        "SMLALTT.CC",
	arm_SMLALTT_MI:        "SMLALTT.MI",
	arm_SMLALTT_PL:        "SMLALTT.PL",
	arm_SMLALTT_VS:        "SMLALTT.VS",
	arm_SMLALTT_VC:        "SMLALTT.VC",
	arm_SMLALTT_HI:        "SMLALTT.HI",
	arm_SMLALTT_LS:        "SMLALTT.LS",
	arm_SMLALTT_GE:        "SMLALTT.GE",
	arm_SMLALTT_LT:        "SMLALTT.LT",
	arm_SMLALTT_GT:        "SMLALTT.GT",
	arm_SMLALTT_LE:        "SMLALTT.LE",
	arm_SMLALTT:           "SMLALTT",
	arm_SMLALTT_ZZ:        "SMLALTT.ZZ",
	arm_SMLALD_EQ:         "SMLALD.EQ",
	arm_SMLALD_NE:         "SMLALD.NE",
	arm_SMLALD_CS:         "SMLALD.CS",
	arm_SMLALD_CC:         "SMLALD.CC",
	arm_SMLALD_MI:         "SMLALD.MI",
	arm_SMLALD_PL:         "SMLALD.PL",
	arm_SMLALD_VS:         "SMLALD.VS",
	arm_SMLALD_VC:         "SMLALD.VC",
	arm_SMLALD_HI:         "SMLALD.HI",
	arm_SMLALD_LS:         "SMLALD.LS",
	arm_SMLALD_GE:         "SMLALD.GE",
	arm_SMLALD_LT:         "SMLALD.LT",
	arm_SMLALD_GT:         "SMLALD.GT",
	arm_SMLALD_LE:         "SMLALD.LE",
	arm_SMLALD:            "SMLALD",
	arm_SMLALD_ZZ:         "SMLALD.ZZ",
	arm_SMLALD_X_EQ:       "SMLALD.X.EQ",
	arm_SMLALD_X_NE:       "SMLALD.X.NE",
	arm_SMLALD_X_CS:       "SMLALD.X.CS",
	arm_SMLALD_X_CC:       "SMLALD.X.CC",
	arm_SMLALD_X_MI:       "SMLALD.X.MI",
	arm_SMLALD_X_PL:       "SMLALD.X.PL",
	arm_SMLALD_X_VS:       "SMLALD.X.VS",
	arm_SMLALD_X_VC:       "SMLALD.X.VC",
	arm_SMLALD_X_HI:       "SMLALD.X.HI",
	arm_SMLALD_X_LS:       "SMLALD.X.LS",
	arm_SMLALD_X_GE:       "SMLALD.X.GE",
	arm_SMLALD_X_LT:       "SMLALD.X.LT",
	arm_SMLALD_X_GT:       "SMLALD.X.GT",
	arm_SMLALD_X_LE:       "SMLALD.X.LE",
	arm_SMLALD_X:          "SMLALD.X",
	arm_SMLALD_X_ZZ:       "SMLALD.X.ZZ",
	arm_SMLAWB_EQ:         "SMLAWB.EQ",
	arm_SMLAWB_NE:         "SMLAWB.NE",
	arm_SMLAWB_CS:         "SMLAWB.CS",
	arm_SMLAWB_CC:         "SMLAWB.CC",
	arm_SMLAWB_MI:         "SMLAWB.MI",
	arm_SMLAWB_PL:         "SMLAWB.PL",
	arm_SMLAWB_VS:         "SMLAWB.VS",
	arm_SMLAWB_VC:         "SMLAWB.VC",
	arm_SMLAWB_HI:         "SMLAWB.HI",
	arm_SMLAWB_LS:         "SMLAWB.LS",
	arm_SMLAWB_GE:         "SMLAWB.GE",
	arm_SMLAWB_LT:         "SMLAWB.LT",
	arm_SMLAWB_GT:         "SMLAWB.GT",
	arm_SMLAWB_LE:         "SMLAWB.LE",
	arm_SMLAWB:            "SMLAWB",
	arm_SMLAWB_ZZ:         "SMLAWB.ZZ",
	arm_SMLAWT_EQ:         "SMLAWT.EQ",
	arm_SMLAWT_NE:         "SMLAWT.NE",
	arm_SMLAWT_CS:         "SMLAWT.CS",
	arm_SMLAWT_CC:         "SMLAWT.CC",
	arm_SMLAWT_MI:         "SMLAWT.MI",
	arm_SMLAWT_PL:         "SMLAWT.PL",
	arm_SMLAWT_VS:         "SMLAWT.VS",
	arm_SMLAWT_VC:         "SMLAWT.VC",
	arm_SMLAWT_HI:         "SMLAWT.HI",
	arm_SMLAWT_LS:         "SMLAWT.LS",
	arm_SMLAWT_GE:         "SMLAWT.GE",
	arm_SMLAWT_LT:         "SMLAWT.LT",
	arm_SMLAWT_GT:         "SMLAWT.GT",
	arm_SMLAWT_LE:         "SMLAWT.LE",
	arm_SMLAWT:            "SMLAWT",
	arm_SMLAWT_ZZ:         "SMLAWT.ZZ",
	arm_SMLSD_EQ:          "SMLSD.EQ",
	arm_SMLSD_NE:          "SMLSD.NE",
	arm_SMLSD_CS:          "SMLSD.CS",
	arm_SMLSD_CC:          "SMLSD.CC",
	arm_SMLSD_MI:          "SMLSD.MI",
	arm_SMLSD_PL:          "SMLSD.PL",
	arm_SMLSD_VS:          "SMLSD.VS",
	arm_SMLSD_VC:          "SMLSD.VC",
	arm_SMLSD_HI:          "SMLSD.HI",
	arm_SMLSD_LS:          "SMLSD.LS",
	arm_SMLSD_GE:          "SMLSD.GE",
	arm_SMLSD_LT:          "SMLSD.LT",
	arm_SMLSD_GT:          "SMLSD.GT",
	arm_SMLSD_LE:          "SMLSD.LE",
	arm_SMLSD:             "SMLSD",
	arm_SMLSD_ZZ:          "SMLSD.ZZ",
	arm_SMLSD_X_EQ:        "SMLSD.X.EQ",
	arm_SMLSD_X_NE:        "SMLSD.X.NE",
	arm_SMLSD_X_CS:        "SMLSD.X.CS",
	arm_SMLSD_X_CC:        "SMLSD.X.CC",
	arm_SMLSD_X_MI:        "SMLSD.X.MI",
	arm_SMLSD_X_PL:        "SMLSD.X.PL",
	arm_SMLSD_X_VS:        "SMLSD.X.VS",
	arm_SMLSD_X_VC:        "SMLSD.X.VC",
	arm_SMLSD_X_HI:        "SMLSD.X.HI",
	arm_SMLSD_X_LS:        "SMLSD.X.LS",
	arm_SMLSD_X_GE:        "SMLSD.X.GE",
	arm_SMLSD_X_LT:        "SMLSD.X.LT",
	arm_SMLSD_X_GT:        "SMLSD.X.GT",
	arm_SMLSD_X_LE:        "SMLSD.X.LE",
	arm_SMLSD_X:           "SMLSD.X",
	arm_SMLSD_X_ZZ:        "SMLSD.X.ZZ",
	arm_SMLSLD_EQ:         "SMLSLD.EQ",
	arm_SMLSLD_NE:         "SMLSLD.NE",
	arm_SMLSLD_CS:         "SMLSLD.CS",
	arm_SMLSLD_CC:         "SMLSLD.CC",
	arm_SMLSLD_MI:         "SMLSLD.MI",
	arm_SMLSLD_PL:         "SMLSLD.PL",
	arm_SMLSLD_VS:         "SMLSLD.VS",
	arm_SMLSLD_VC:         "SMLSLD.VC",
	arm_SMLSLD_HI:         "SMLSLD.HI",
	arm_SMLSLD_LS:         "SMLSLD.LS",
	arm_SMLSLD_GE:         "SMLSLD.GE",
	arm_SMLSLD_LT:         "SMLSLD.LT",
	arm_SMLSLD_GT:         "SMLSLD.GT",
	arm_SMLSLD_LE:         "SMLSLD.LE",
	arm_SMLSLD:            "SMLSLD",
	arm_SMLSLD_ZZ:         "SMLSLD.ZZ",
	arm_SMLSLD_X_EQ:       "SMLSLD.X.EQ",
	arm_SMLSLD_X_NE:       "SMLSLD.X.NE",
	arm_SMLSLD_X_CS:       "SMLSLD.X.CS",
	arm_SMLSLD_X_CC:       "SMLSLD.X.CC",
	arm_SMLSLD_X_MI:       "SMLSLD.X.MI",
	arm_SMLSLD_X_PL:       "SMLSLD.X.PL",
	arm_SMLSLD_X_VS:       "SMLSLD.X.VS",
	arm_SMLSLD_X_VC:       "SMLSLD.X.VC",
	arm_SMLSLD_X_HI:       "SMLSLD.X.HI",
	arm_SMLSLD_X_LS:       "SMLSLD.X.LS",
	arm_SMLSLD_X_GE:       "SMLSLD.X.GE",
	arm_SMLSLD_X_LT:       "SMLSLD.X.LT",
	arm_SMLSLD_X_GT:       "SMLSLD.X.GT",
	arm_SMLSLD_X_LE:       "SMLSLD.X.LE",
	arm_SMLSLD_X:          "SMLSLD.X",
	arm_SMLSLD_X_ZZ:       "SMLSLD.X.ZZ",
	arm_SMMLA_EQ:          "SMMLA.EQ",
	arm_SMMLA_NE:          "SMMLA.NE",
	arm_SMMLA_CS:          "SMMLA.CS",
	arm_SMMLA_CC:          "SMMLA.CC",
	arm_SMMLA_MI:          "SMMLA.MI",
	arm_SMMLA_PL:          "SMMLA.PL",
	arm_SMMLA_VS:          "SMMLA.VS",
	arm_SMMLA_VC:          "SMMLA.VC",
	arm_SMMLA_HI:          "SMMLA.HI",
	arm_SMMLA_LS:          "SMMLA.LS",
	arm_SMMLA_GE:          "SMMLA.GE",
	arm_SMMLA_LT:          "SMMLA.LT",
	arm_SMMLA_GT:          "SMMLA.GT",
	arm_SMMLA_LE:          "SMMLA.LE",
	arm_SMMLA:             "SMMLA",
	arm_SMMLA_ZZ:          "SMMLA.ZZ",
	arm_SMMLA_R_EQ:        "SMMLA.R.EQ",
	arm_SMMLA_R_NE:        "SMMLA.R.NE",
	arm_SMMLA_R_CS:        "SMMLA.R.CS",
	arm_SMMLA_R_CC:        "SMMLA.R.CC",
	arm_SMMLA_R_MI:        "SMMLA.R.MI",
	arm_SMMLA_R_PL:        "SMMLA.R.PL",
	arm_SMMLA_R_VS:        "SMMLA.R.VS",
	arm_SMMLA_R_VC:        "SMMLA.R.VC",
	arm_SMMLA_R_HI:        "SMMLA.R.HI",
	arm_SMMLA_R_LS:        "SMMLA.R.LS",
	arm_SMMLA_R_GE:        "SMMLA.R.GE",
	arm_SMMLA_R_LT:        "SMMLA.R.LT",
	arm_SMMLA_R_GT:        "SMMLA.R.GT",
	arm_SMMLA_R_LE:        "SMMLA.R.LE",
	arm_SMMLA_R:           "SMMLA.R",
	arm_SMMLA_R_ZZ:        "SMMLA.R.ZZ",
	arm_SMMLS_EQ:          "SMMLS.EQ",
	arm_SMMLS_NE:          "SMMLS.NE",
	arm_SMMLS_CS:          "SMMLS.CS",
	arm_SMMLS_CC:          "SMMLS.CC",
	arm_SMMLS_MI:          "SMMLS.MI",
	arm_SMMLS_PL:          "SMMLS.PL",
	arm_SMMLS_VS:          "SMMLS.VS",
	arm_SMMLS_VC:          "SMMLS.VC",
	arm_SMMLS_HI:          "SMMLS.HI",
	arm_SMMLS_LS:          "SMMLS.LS",
	arm_SMMLS_GE:          "SMMLS.GE",
	arm_SMMLS_LT:          "SMMLS.LT",
	arm_SMMLS_GT:          "SMMLS.GT",
	arm_SMMLS_LE:          "SMMLS.LE",
	arm_SMMLS:             "SMMLS",
	arm_SMMLS_ZZ:          "SMMLS.ZZ",
	arm_SMMLS_R_EQ:        "SMMLS.R.EQ",
	arm_SMMLS_R_NE:        "SMMLS.R.NE",
	arm_SMMLS_R_CS:        "SMMLS.R.CS",
	arm_SMMLS_R_CC:        "SMMLS.R.CC",
	arm_SMMLS_R_MI:        "SMMLS.R.MI",
	arm_SMMLS_R_PL:        "SMMLS.R.PL",
	arm_SMMLS_R_VS:        "SMMLS.R.VS",
	arm_SMMLS_R_VC:        "SMMLS.R.VC",
	arm_SMMLS_R_HI:        "SMMLS.R.HI",
	arm_SMMLS_R_LS:        "SMMLS.R.LS",
	arm_SMMLS_R_GE:        "SMMLS.R.GE",
	arm_SMMLS_R_LT:        "SMMLS.R.LT",
	arm_SMMLS_R_GT:        "SMMLS.R.GT",
	arm_SMMLS_R_LE:        "SMMLS.R.LE",
	arm_SMMLS_R:           "SMMLS.R",
	arm_SMMLS_R_ZZ:        "SMMLS.R.ZZ",
	arm_SMMUL_EQ:          "SMMUL.EQ",
	arm_SMMUL_NE:          "SMMUL.NE",
	arm_SMMUL_CS:          "SMMUL.CS",
	arm_SMMUL_CC:          "SMMUL.CC",
	arm_SMMUL_MI:          "SMMUL.MI",
	arm_SMMUL_PL:          "SMMUL.PL",
	arm_SMMUL_VS:          "SMMUL.VS",
	arm_SMMUL_VC:          "SMMUL.VC",
	arm_SMMUL_HI:          "SMMUL.HI",
	arm_SMMUL_LS:          "SMMUL.LS",
	arm_SMMUL_GE:          "SMMUL.GE",
	arm_SMMUL_LT:          "SMMUL.LT",
	arm_SMMUL_GT:          "SMMUL.GT",
	arm_SMMUL_LE:          "SMMUL.LE",
	arm_SMMUL:             "SMMUL",
	arm_SMMUL_ZZ:          "SMMUL.ZZ",
	arm_SMMUL_R_EQ:        "SMMUL.R.EQ",
	arm_SMMUL_R_NE:        "SMMUL.R.NE",
	arm_SMMUL_R_CS:        "SMMUL.R.CS",
	arm_SMMUL_R_CC:        "SMMUL.R.CC",
	arm_SMMUL_R_MI:        "SMMUL.R.MI",
	arm_SMMUL_R_PL:        "SMMUL.R.PL",
	arm_SMMUL_R_VS:        "SMMUL.R.VS",
	arm_SMMUL_R_VC:        "SMMUL.R.VC",
	arm_SMMUL_R_HI:        "SMMUL.R.HI",
	arm_SMMUL_R_LS:        "SMMUL.R.LS",
	arm_SMMUL_R_GE:        "SMMUL.R.GE",
	arm_SMMUL_R_LT:        "SMMUL.R.LT",
	arm_SMMUL_R_GT:        "SMMUL.R.GT",
	arm_SMMUL_R_LE:        "SMMUL.R.LE",
	arm_SMMUL_R:           "SMMUL.R",
	arm_SMMUL_R_ZZ:        "SMMUL.R.ZZ",
	arm_SMUAD_EQ:          "SMUAD.EQ",
	arm_SMUAD_NE:          "SMUAD.NE",
	arm_SMUAD_CS:          "SMUAD.CS",
	arm_SMUAD_CC:          "SMUAD.CC",
	arm_SMUAD_MI:          "SMUAD.MI",
	arm_SMUAD_PL:          "SMUAD.PL",
	arm_SMUAD_VS:          "SMUAD.VS",
	arm_SMUAD_VC:          "SMUAD.VC",
	arm_SMUAD_HI:          "SMUAD.HI",
	arm_SMUAD_LS:          "SMUAD.LS",
	arm_SMUAD_GE:          "SMUAD.GE",
	arm_SMUAD_LT:          "SMUAD.LT",
	arm_SMUAD_GT:          "SMUAD.GT",
	arm_SMUAD_LE:          "SMUAD.LE",
	arm_SMUAD:             "SMUAD",
	arm_SMUAD_ZZ:          "SMUAD.ZZ",
	arm_SMUAD_X_EQ:        "SMUAD.X.EQ",
	arm_SMUAD_X_NE:        "SMUAD.X.NE",
	arm_SMUAD_X_CS:        "SMUAD.X.CS",
	arm_SMUAD_X_CC:        "SMUAD.X.CC",
	arm_SMUAD_X_MI:        "SMUAD.X.MI",
	arm_SMUAD_X_PL:        "SMUAD.X.PL",
	arm_SMUAD_X_VS:        "SMUAD.X.VS",
	arm_SMUAD_X_VC:        "SMUAD.X.VC",
	arm_SMUAD_X_HI:        "SMUAD.X.HI",
	arm_SMUAD_X_LS:        "SMUAD.X.LS",
	arm_SMUAD_X_GE:        "SMUAD.X.GE",
	arm_SMUAD_X_LT:        "SMUAD.X.LT",
	arm_SMUAD_X_GT:        "SMUAD.X.GT",
	arm_SMUAD_X_LE:        "SMUAD.X.LE",
	arm_SMUAD_X:           "SMUAD.X",
	arm_SMUAD_X_ZZ:        "SMUAD.X.ZZ",
	arm_SMULBB_EQ:         "SMULBB.EQ",
	arm_SMULBB_NE:         "SMULBB.NE",
	arm_SMULBB_CS:         "SMULBB.CS",
	arm_SMULBB_CC:         "SMULBB.CC",
	arm_SMULBB_MI:         "SMULBB.MI",
	arm_SMULBB_PL:         "SMULBB.PL",
	arm_SMULBB_VS:         "SMULBB.VS",
	arm_SMULBB_VC:         "SMULBB.VC",
	arm_SMULBB_HI:         "SMULBB.HI",
	arm_SMULBB_LS:         "SMULBB.LS",
	arm_SMULBB_GE:         "SMULBB.GE",
	arm_SMULBB_LT:         "SMULBB.LT",
	arm_SMULBB_GT:         "SMULBB.GT",
	arm_SMULBB_LE:         "SMULBB.LE",
	arm_SMULBB:            "SMULBB",
	arm_SMULBB_ZZ:         "SMULBB.ZZ",
	arm_SMULBT_EQ:         "SMULBT.EQ",
	arm_SMULBT_NE:         "SMULBT.NE",
	arm_SMULBT_CS:         "SMULBT.CS",
	arm_SMULBT_CC:         "SMULBT.CC",
	arm_SMULBT_MI:         "SMULBT.MI",
	arm_SMULBT_PL:         "SMULBT.PL",
	arm_SMULBT_VS:         "SMULBT.VS",
	arm_SMULBT_VC:         "SMULBT.VC",
	arm_SMULBT_HI:         "SMULBT.HI",
	arm_SMULBT_LS:         "SMULBT.LS",
	arm_SMULBT_GE:         "SMULBT.GE",
	arm_SMULBT_LT:         "SMULBT.LT",
	arm_SMULBT_GT:         "SMULBT.GT",
	arm_SMULBT_LE:         "SMULBT.LE",
	arm_SMULBT:            "SMULBT",
	arm_SMULBT_ZZ:         "SMULBT.ZZ",
	arm_SMULTB_EQ:         "SMULTB.EQ",
	arm_SMULTB_NE:         "SMULTB.NE",
	arm_SMULTB_CS:         "SMULTB.CS",
	arm_SMULTB_CC:         "SMULTB.CC",
	arm_SMULTB_MI:         "SMULTB.MI",
	arm_SMULTB_PL:         "SMULTB.PL",
	arm_SMULTB_VS:         "SMULTB.VS",
	arm_SMULTB_VC:         "SMULTB.VC",
	arm_SMULTB_HI:         "SMULTB.HI",
	arm_SMULTB_LS:         "SMULTB.LS",
	arm_SMULTB_GE:         "SMULTB.GE",
	arm_SMULTB_LT:         "SMULTB.LT",
	arm_SMULTB_GT:         "SMULTB.GT",
	arm_SMULTB_LE:         "SMULTB.LE",
	arm_SMULTB:            "SMULTB",
	arm_SMULTB_ZZ:         "SMULTB.ZZ",
	arm_SMULTT_EQ:         "SMULTT.EQ",
	arm_SMULTT_NE:         "SMULTT.NE",
	arm_SMULTT_CS:         "SMULTT.CS",
	arm_SMULTT_CC:         "SMULTT.CC",
	arm_SMULTT_MI:         "SMULTT.MI",
	arm_SMULTT_PL:         "SMULTT.PL",
	arm_SMULTT_VS:         "SMULTT.VS",
	arm_SMULTT_VC:         "SMULTT.VC",
	arm_SMULTT_HI:         "SMULTT.HI",
	arm_SMULTT_LS:         "SMULTT.LS",
	arm_SMULTT_GE:         "SMULTT.GE",
	arm_SMULTT_LT:         "SMULTT.LT",
	arm_SMULTT_GT:         "SMULTT.GT",
	arm_SMULTT_LE:         "SMULTT.LE",
	arm_SMULTT:            "SMULTT",
	arm_SMULTT_ZZ:         "SMULTT.ZZ",
	arm_SMULL_EQ:          "SMULL.EQ",
	arm_SMULL_NE:          "SMULL.NE",
	arm_SMULL_CS:          "SMULL.CS",
	arm_SMULL_CC:          "SMULL.CC",
	arm_SMULL_MI:          "SMULL.MI",
	arm_SMULL_PL:          "SMULL.PL",
	arm_SMULL_VS:          "SMULL.VS",
	arm_SMULL_VC:          "SMULL.VC",
	arm_SMULL_HI:          "SMULL.HI",
	arm_SMULL_LS:          "SMULL.LS",
	arm_SMULL_GE:          "SMULL.GE",
	arm_SMULL_LT:          "SMULL.LT",
	arm_SMULL_GT:          "SMULL.GT",
	arm_SMULL_LE:          "SMULL.LE",
	arm_SMULL:             "SMULL",
	arm_SMULL_ZZ:          "SMULL.ZZ",
	arm_SMULL_S_EQ:        "SMULL.S.EQ",
	arm_SMULL_S_NE:        "SMULL.S.NE",
	arm_SMULL_S_CS:        "SMULL.S.CS",
	arm_SMULL_S_CC:        "SMULL.S.CC",
	arm_SMULL_S_MI:        "SMULL.S.MI",
	arm_SMULL_S_PL:        "SMULL.S.PL",
	arm_SMULL_S_VS:        "SMULL.S.VS",
	arm_SMULL_S_VC:        "SMULL.S.VC",
	arm_SMULL_S_HI:        "SMULL.S.HI",
	arm_SMULL_S_LS:        "SMULL.S.LS",
	arm_SMULL_S_GE:        "SMULL.S.GE",
	arm_SMULL_S_LT:        "SMULL.S.LT",
	arm_SMULL_S_GT:        "SMULL.S.GT",
	arm_SMULL_S_LE:        "SMULL.S.LE",
	arm_SMULL_S:           "SMULL.S",
	arm_SMULL_S_ZZ:        "SMULL.S.ZZ",
	arm_SMULWB_EQ:         "SMULWB.EQ",
	arm_SMULWB_NE:         "SMULWB.NE",
	arm_SMULWB_CS:         "SMULWB.CS",
	arm_SMULWB_CC:         "SMULWB.CC",
	arm_SMULWB_MI:         "SMULWB.MI",
	arm_SMULWB_PL:         "SMULWB.PL",
	arm_SMULWB_VS:         "SMULWB.VS",
	arm_SMULWB_VC:         "SMULWB.VC",
	arm_SMULWB_HI:         "SMULWB.HI",
	arm_SMULWB_LS:         "SMULWB.LS",
	arm_SMULWB_GE:         "SMULWB.GE",
	arm_SMULWB_LT:         "SMULWB.LT",
	arm_SMULWB_GT:         "SMULWB.GT",
	arm_SMULWB_LE:         "SMULWB.LE",
	arm_SMULWB:            "SMULWB",
	arm_SMULWB_ZZ:         "SMULWB.ZZ",
	arm_SMULWT_EQ:         "SMULWT.EQ",
	arm_SMULWT_NE:         "SMULWT.NE",
	arm_SMULWT_CS:         "SMULWT.CS",
	arm_SMULWT_CC:         "SMULWT.CC",
	arm_SMULWT_MI:         "SMULWT.MI",
	arm_SMULWT_PL:         "SMULWT.PL",
	arm_SMULWT_VS:         "SMULWT.VS",
	arm_SMULWT_VC:         "SMULWT.VC",
	arm_SMULWT_HI:         "SMULWT.HI",
	arm_SMULWT_LS:         "SMULWT.LS",
	arm_SMULWT_GE:         "SMULWT.GE",
	arm_SMULWT_LT:         "SMULWT.LT",
	arm_SMULWT_GT:         "SMULWT.GT",
	arm_SMULWT_LE:         "SMULWT.LE",
	arm_SMULWT:            "SMULWT",
	arm_SMULWT_ZZ:         "SMULWT.ZZ",
	arm_SMUSD_EQ:          "SMUSD.EQ",
	arm_SMUSD_NE:          "SMUSD.NE",
	arm_SMUSD_CS:          "SMUSD.CS",
	arm_SMUSD_CC:          "SMUSD.CC",
	arm_SMUSD_MI:          "SMUSD.MI",
	arm_SMUSD_PL:          "SMUSD.PL",
	arm_SMUSD_VS:          "SMUSD.VS",
	arm_SMUSD_VC:          "SMUSD.VC",
	arm_SMUSD_HI:          "SMUSD.HI",
	arm_SMUSD_LS:          "SMUSD.LS",
	arm_SMUSD_GE:          "SMUSD.GE",
	arm_SMUSD_LT:          "SMUSD.LT",
	arm_SMUSD_GT:          "SMUSD.GT",
	arm_SMUSD_LE:          "SMUSD.LE",
	arm_SMUSD:             "SMUSD",
	arm_SMUSD_ZZ:          "SMUSD.ZZ",
	arm_SMUSD_X_EQ:        "SMUSD.X.EQ",
	arm_SMUSD_X_NE:        "SMUSD.X.NE",
	arm_SMUSD_X_CS:        "SMUSD.X.CS",
	arm_SMUSD_X_CC:        "SMUSD.X.CC",
	arm_SMUSD_X_MI:        "SMUSD.X.MI",
	arm_SMUSD_X_PL:        "SMUSD.X.PL",
	arm_SMUSD_X_VS:        "SMUSD.X.VS",
	arm_SMUSD_X_VC:        "SMUSD.X.VC",
	arm_SMUSD_X_HI:        "SMUSD.X.HI",
	arm_SMUSD_X_LS:        "SMUSD.X.LS",
	arm_SMUSD_X_GE:        "SMUSD.X.GE",
	arm_SMUSD_X_LT:        "SMUSD.X.LT",
	arm_SMUSD_X_GT:        "SMUSD.X.GT",
	arm_SMUSD_X_LE:        "SMUSD.X.LE",
	arm_SMUSD_X:           "SMUSD.X",
	arm_SMUSD_X_ZZ:        "SMUSD.X.ZZ",
	arm_SSAT_EQ:           "SSAT.EQ",
	arm_SSAT_NE:           "SSAT.NE",
	arm_SSAT_CS:           "SSAT.CS",
	arm_SSAT_CC:           "SSAT.CC",
	arm_SSAT_MI:           "SSAT.MI",
	arm_SSAT_PL:           "SSAT.PL",
	arm_SSAT_VS:           "SSAT.VS",
	arm_SSAT_VC:           "SSAT.VC",
	arm_SSAT_HI:           "SSAT.HI",
	arm_SSAT_LS:           "SSAT.LS",
	arm_SSAT_GE:           "SSAT.GE",
	arm_SSAT_LT:           "SSAT.LT",
	arm_SSAT_GT:           "SSAT.GT",
	arm_SSAT_LE:           "SSAT.LE",
	arm_SSAT:              "SSAT",
	arm_SSAT_ZZ:           "SSAT.ZZ",
	arm_SSAT16_EQ:         "SSAT16.EQ",
	arm_SSAT16_NE:         "SSAT16.NE",
	arm_SSAT16_CS:         "SSAT16.CS",
	arm_SSAT16_CC:         "SSAT16.CC",
	arm_SSAT16_MI:         "SSAT16.MI",
	arm_SSAT16_PL:         "SSAT16.PL",
	arm_SSAT16_VS:         "SSAT16.VS",
	arm_SSAT16_VC:         "SSAT16.VC",
	arm_SSAT16_HI:         "SSAT16.HI",
	arm_SSAT16_LS:         "SSAT16.LS",
	arm_SSAT16_GE:         "SSAT16.GE",
	arm_SSAT16_LT:         "SSAT16.LT",
	arm_SSAT16_GT:         "SSAT16.GT",
	arm_SSAT16_LE:         "SSAT16.LE",
	arm_SSAT16:            "SSAT16",
	arm_SSAT16_ZZ:         "SSAT16.ZZ",
	arm_SSAX_EQ:           "SSAX.EQ",
	arm_SSAX_NE:           "SSAX.NE",
	arm_SSAX_CS:           "SSAX.CS",
	arm_SSAX_CC:           "SSAX.CC",
	arm_SSAX_MI:           "SSAX.MI",
	arm_SSAX_PL:           "SSAX.PL",
	arm_SSAX_VS:           "SSAX.VS",
	arm_SSAX_VC:           "SSAX.VC",
	arm_SSAX_HI:           "SSAX.HI",
	arm_SSAX_LS:           "SSAX.LS",
	arm_SSAX_GE:           "SSAX.GE",
	arm_SSAX_LT:           "SSAX.LT",
	arm_SSAX_GT:           "SSAX.GT",
	arm_SSAX_LE:           "SSAX.LE",
	arm_SSAX:              "SSAX",
	arm_SSAX_ZZ:           "SSAX.ZZ",
	arm_SSUB16_EQ:         "SSUB16.EQ",
	arm_SSUB16_NE:         "SSUB16.NE",
	arm_SSUB16_CS:         "SSUB16.CS",
	arm_SSUB16_CC:         "SSUB16.CC",
	arm_SSUB16_MI:         "SSUB16.MI",
	arm_SSUB16_PL:         "SSUB16.PL",
	arm_SSUB16_VS:         "SSUB16.VS",
	arm_SSUB16_VC:         "SSUB16.VC",
	arm_SSUB16_HI:         "SSUB16.HI",
	arm_SSUB16_LS:         "SSUB16.LS",
	arm_SSUB16_GE:         "SSUB16.GE",
	arm_SSUB16_LT:         "SSUB16.LT",
	arm_SSUB16_GT:         "SSUB16.GT",
	arm_SSUB16_LE:         "SSUB16.LE",
	arm_SSUB16:            "SSUB16",
	arm_SSUB16_ZZ:         "SSUB16.ZZ",
	arm_SSUB8_EQ:          "SSUB8.EQ",
	arm_SSUB8_NE:          "SSUB8.NE",
	arm_SSUB8_CS:          "SSUB8.CS",
	arm_SSUB8_CC:          "SSUB8.CC",
	arm_SSUB8_MI:          "SSUB8.MI",
	arm_SSUB8_PL:          "SSUB8.PL",
	arm_SSUB8_VS:          "SSUB8.VS",
	arm_SSUB8_VC:          "SSUB8.VC",
	arm_SSUB8_HI:          "SSUB8.HI",
	arm_SSUB8_LS:          "SSUB8.LS",
	arm_SSUB8_GE:          "SSUB8.GE",
	arm_SSUB8_LT:          "SSUB8.LT",
	arm_SSUB8_GT:          "SSUB8.GT",
	arm_SSUB8_LE:          "SSUB8.LE",
	arm_SSUB8:             "SSUB8",
	arm_SSUB8_ZZ:          "SSUB8.ZZ",
	arm_STM_EQ:            "STM.EQ",
	arm_STM_NE:            "STM.NE",
	arm_STM_CS:            "STM.CS",
	arm_STM_CC:            "STM.CC",
	arm_STM_MI:            "STM.MI",
	arm_STM_PL:            "STM.PL",
	arm_STM_VS:            "STM.VS",
	arm_STM_VC:            "STM.VC",
	arm_STM_HI:            "STM.HI",
	arm_STM_LS:            "STM.LS",
	arm_STM_GE:            "STM.GE",
	arm_STM_LT:            "STM.LT",
	arm_STM_GT:            "STM.GT",
	arm_STM_LE:            "STM.LE",
	arm_STM:               "STM",
	arm_STM_ZZ:            "STM.ZZ",
	arm_STMDA_EQ:          "STMDA.EQ",
	arm_STMDA_NE:          "STMDA.NE",
	arm_STMDA_CS:          "STMDA.CS",
	arm_STMDA_CC:          "STMDA.CC",
	arm_STMDA_MI:          "STMDA.MI",
	arm_STMDA_PL:          "STMDA.PL",
	arm_STMDA_VS:          "STMDA.VS",
	arm_STMDA_VC:          "STMDA.VC",
	arm_STMDA_HI:          "STMDA.HI",
	arm_STMDA_LS:          "STMDA.LS",
	arm_STMDA_GE:          "STMDA.GE",
	arm_STMDA_LT:          "STMDA.LT",
	arm_STMDA_GT:          "STMDA.GT",
	arm_STMDA_LE:          "STMDA.LE",
	arm_STMDA:             "STMDA",
	arm_STMDA_ZZ:          "STMDA.ZZ",
	arm_STMDB_EQ:          "STMDB.EQ",
	arm_STMDB_NE:          "STMDB.NE",
	arm_STMDB_CS:          "STMDB.CS",
	arm_STMDB_CC:          "STMDB.CC",
	arm_STMDB_MI:          "STMDB.MI",
	arm_STMDB_PL:          "STMDB.PL",
	arm_STMDB_VS:          "STMDB.VS",
	arm_STMDB_VC:          "STMDB.VC",
	arm_STMDB_HI:          "STMDB.HI",
	arm_STMDB_LS:          "STMDB.LS",
	arm_STMDB_GE:          "STMDB.GE",
	arm_STMDB_LT:          "STMDB.LT",
	arm_STMDB_GT:          "STMDB.GT",
	arm_STMDB_LE:          "STMDB.LE",
	arm_STMDB:             "STMDB",
	arm_STMDB_ZZ:          "STMDB.ZZ",
	arm_STMIB_EQ:          "STMIB.EQ",
	arm_STMIB_NE:          "STMIB.NE",
	arm_STMIB_CS:          "STMIB.CS",
	arm_STMIB_CC:          "STMIB.CC",
	arm_STMIB_MI:          "STMIB.MI",
	arm_STMIB_PL:          "STMIB.PL",
	arm_STMIB_VS:          "STMIB.VS",
	arm_STMIB_VC:          "STMIB.VC",
	arm_STMIB_HI:          "STMIB.HI",
	arm_STMIB_LS:          "STMIB.LS",
	arm_STMIB_GE:          "STMIB.GE",
	arm_STMIB_LT:          "STMIB.LT",
	arm_STMIB_GT:          "STMIB.GT",
	arm_STMIB_LE:          "STMIB.LE",
	arm_STMIB:             "STMIB",
	arm_STMIB_ZZ:          "STMIB.ZZ",
	arm_STR_EQ:            "STR.EQ",
	arm_STR_NE:            "STR.NE",
	arm_STR_CS:            "STR.CS",
	arm_STR_CC:            "STR.CC",
	arm_STR_MI:            "STR.MI",
	arm_STR_PL:            "STR.PL",
	arm_STR_VS:            "STR.VS",
	arm_STR_VC:            "STR.VC",
	arm_STR_HI:            "STR.HI",
	arm_STR_LS:            "STR.LS",
	arm_STR_GE:            "STR.GE",
	arm_STR_LT:            "STR.LT",
	arm_STR_GT:            "STR.GT",
	arm_STR_LE:            "STR.LE",
	arm_STR:               "STR",
	arm_STR_ZZ:            "STR.ZZ",
	arm_STRB_EQ:           "STRB.EQ",
	arm_STRB_NE:           "STRB.NE",
	arm_STRB_CS:           "STRB.CS",
	arm_STRB_CC:           "STRB.CC",
	arm_STRB_MI:           "STRB.MI",
	arm_STRB_PL:           "STRB.PL",
	arm_STRB_VS:           "STRB.VS",
	arm_STRB_VC:           "STRB.VC",
	arm_STRB_HI:           "STRB.HI",
	arm_STRB_LS:           "STRB.LS",
	arm_STRB_GE:           "STRB.GE",
	arm_STRB_LT:           "STRB.LT",
	arm_STRB_GT:           "STRB.GT",
	arm_STRB_LE:           "STRB.LE",
	arm_STRB:              "STRB",
	arm_STRB_ZZ:           "STRB.ZZ",
	arm_STRBT_EQ:          "STRBT.EQ",
	arm_STRBT_NE:          "STRBT.NE",
	arm_STRBT_CS:          "STRBT.CS",
	arm_STRBT_CC:          "STRBT.CC",
	arm_STRBT_MI:          "STRBT.MI",
	arm_STRBT_PL:          "STRBT.PL",
	arm_STRBT_VS:          "STRBT.VS",
	arm_STRBT_VC:          "STRBT.VC",
	arm_STRBT_HI:          "STRBT.HI",
	arm_STRBT_LS:          "STRBT.LS",
	arm_STRBT_GE:          "STRBT.GE",
	arm_STRBT_LT:          "STRBT.LT",
	arm_STRBT_GT:          "STRBT.GT",
	arm_STRBT_LE:          "STRBT.LE",
	arm_STRBT:             "STRBT",
	arm_STRBT_ZZ:          "STRBT.ZZ",
	arm_STRD_EQ:           "STRD.EQ",
	arm_STRD_NE:           "STRD.NE",
	arm_STRD_CS:           "STRD.CS",
	arm_STRD_CC:           "STRD.CC",
	arm_STRD_MI:           "STRD.MI",
	arm_STRD_PL:           "STRD.PL",
	arm_STRD_VS:           "STRD.VS",
	arm_STRD_VC:           "STRD.VC",
	arm_STRD_HI:           "STRD.HI",
	arm_STRD_LS:           "STRD.LS",
	arm_STRD_GE:           "STRD.GE",
	arm_STRD_LT:           "STRD.LT",
	arm_STRD_GT:           "STRD.GT",
	arm_STRD_LE:           "STRD.LE",
	arm_STRD:              "STRD",
	arm_STRD_ZZ:           "STRD.ZZ",
	arm_STREX_EQ:          "STREX.EQ",
	arm_STREX_NE:          "STREX.NE",
	arm_STREX_CS:          "STREX.CS",
	arm_STREX_CC:          "STREX.CC",
	arm_STREX_MI:          "STREX.MI",
	arm_STREX_PL:          "STREX.PL",
	arm_STREX_VS:          "STREX.VS",
	arm_STREX_VC:          "STREX.VC",
	arm_STREX_HI:          "STREX.HI",
	arm_STREX_LS:          "STREX.LS",
	arm_STREX_GE:          "STREX.GE",
	arm_STREX_LT:          "STREX.LT",
	arm_STREX_GT:          "STREX.GT",
	arm_STREX_LE:          "STREX.LE",
	arm_STREX:             "STREX",
	arm_STREX_ZZ:          "STREX.ZZ",
	arm_STREXB_EQ:         "STREXB.EQ",
	arm_STREXB_NE:         "STREXB.NE",
	arm_STREXB_CS:         "STREXB.CS",
	arm_STREXB_CC:         "STREXB.CC",
	arm_STREXB_MI:         "STREXB.MI",
	arm_STREXB_PL:         "STREXB.PL",
	arm_STREXB_VS:         "STREXB.VS",
	arm_STREXB_VC:         "STREXB.VC",
	arm_STREXB_HI:         "STREXB.HI",
	arm_STREXB_LS:         "STREXB.LS",
	arm_STREXB_GE:         "STREXB.GE",
	arm_STREXB_LT:         "STREXB.LT",
	arm_STREXB_GT:         "STREXB.GT",
	arm_STREXB_LE:         "STREXB.LE",
	arm_STREXB:            "STREXB",
	arm_STREXB_ZZ:         "STREXB.ZZ",
	arm_STREXD_EQ:         "STREXD.EQ",
	arm_STREXD_NE:         "STREXD.NE",
	arm_STREXD_CS:         "STREXD.CS",
	arm_STREXD_CC:         "STREXD.CC",
	arm_STREXD_MI:         "STREXD.MI",
	arm_STREXD_PL:         "STREXD.PL",
	arm_STREXD_VS:         "STREXD.VS",
	arm_STREXD_VC:         "STREXD.VC",
	arm_STREXD_HI:         "STREXD.HI",
	arm_STREXD_LS:         "STREXD.LS",
	arm_STREXD_GE:         "STREXD.GE",
	arm_STREXD_LT:         "STREXD.LT",
	arm_STREXD_GT:         "STREXD.GT",
	arm_STREXD_LE:         "STREXD.LE",
	arm_STREXD:            "STREXD",
	arm_STREXD_ZZ:         "STREXD.ZZ",
	arm_STREXH_EQ:         "STREXH.EQ",
	arm_STREXH_NE:         "STREXH.NE",
	arm_STREXH_CS:         "STREXH.CS",
	arm_STREXH_CC:         "STREXH.CC",
	arm_STREXH_MI:         "STREXH.MI",
	arm_STREXH_PL:         "STREXH.PL",
	arm_STREXH_VS:         "STREXH.VS",
	arm_STREXH_VC:         "STREXH.VC",
	arm_STREXH_HI:         "STREXH.HI",
	arm_STREXH_LS:         "STREXH.LS",
	arm_STREXH_GE:         "STREXH.GE",
	arm_STREXH_LT:         "STREXH.LT",
	arm_STREXH_GT:         "STREXH.GT",
	arm_STREXH_LE:         "STREXH.LE",
	arm_STREXH:            "STREXH",
	arm_STREXH_ZZ:         "STREXH.ZZ",
	arm_STRH_EQ:           "STRH.EQ",
	arm_STRH_NE:           "STRH.NE",
	arm_STRH_CS:           "STRH.CS",
	arm_STRH_CC:           "STRH.CC",
	arm_STRH_MI:           "STRH.MI",
	arm_STRH_PL:           "STRH.PL",
	arm_STRH_VS:           "STRH.VS",
	arm_STRH_VC:           "STRH.VC",
	arm_STRH_HI:           "STRH.HI",
	arm_STRH_LS:           "STRH.LS",
	arm_STRH_GE:           "STRH.GE",
	arm_STRH_LT:           "STRH.LT",
	arm_STRH_GT:           "STRH.GT",
	arm_STRH_LE:           "STRH.LE",
	arm_STRH:              "STRH",
	arm_STRH_ZZ:           "STRH.ZZ",
	arm_STRHT_EQ:          "STRHT.EQ",
	arm_STRHT_NE:          "STRHT.NE",
	arm_STRHT_CS:          "STRHT.CS",
	arm_STRHT_CC:          "STRHT.CC",
	arm_STRHT_MI:          "STRHT.MI",
	arm_STRHT_PL:          "STRHT.PL",
	arm_STRHT_VS:          "STRHT.VS",
	arm_STRHT_VC:          "STRHT.VC",
	arm_STRHT_HI:          "STRHT.HI",
	arm_STRHT_LS:          "STRHT.LS",
	arm_STRHT_GE:          "STRHT.GE",
	arm_STRHT_LT:          "STRHT.LT",
	arm_STRHT_GT:          "STRHT.GT",
	arm_STRHT_LE:          "STRHT.LE",
	arm_STRHT:             "STRHT",
	arm_STRHT_ZZ:          "STRHT.ZZ",
	arm_STRT_EQ:           "STRT.EQ",
	arm_STRT_NE:           "STRT.NE",
	arm_STRT_CS:           "STRT.CS",
	arm_STRT_CC:           "STRT.CC",
	arm_STRT_MI:           "STRT.MI",
	arm_STRT_PL:           "STRT.PL",
	arm_STRT_VS:           "STRT.VS",
	arm_STRT_VC:           "STRT.VC",
	arm_STRT_HI:           "STRT.HI",
	arm_STRT_LS:           "STRT.LS",
	arm_STRT_GE:           "STRT.GE",
	arm_STRT_LT:           "STRT.LT",
	arm_STRT_GT:           "STRT.GT",
	arm_STRT_LE:           "STRT.LE",
	arm_STRT:              "STRT",
	arm_STRT_ZZ:           "STRT.ZZ",
	arm_SUB_EQ:            "SUB.EQ",
	arm_SUB_NE:            "SUB.NE",
	arm_SUB_CS:            "SUB.CS",
	arm_SUB_CC:            "SUB.CC",
	arm_SUB_MI:            "SUB.MI",
	arm_SUB_PL:            "SUB.PL",
	arm_SUB_VS:            "SUB.VS",
	arm_SUB_VC:            "SUB.VC",
	arm_SUB_HI:            "SUB.HI",
	arm_SUB_LS:            "SUB.LS",
	arm_SUB_GE:            "SUB.GE",
	arm_SUB_LT:            "SUB.LT",
	arm_SUB_GT:            "SUB.GT",
	arm_SUB_LE:            "SUB.LE",
	arm_SUB:               "SUB",
	arm_SUB_ZZ:            "SUB.ZZ",
	arm_SUB_S_EQ:          "SUB.S.EQ",
	arm_SUB_S_NE:          "SUB.S.NE",
	arm_SUB_S_CS:          "SUB.S.CS",
	arm_SUB_S_CC:          "SUB.S.CC",
	arm_SUB_S_MI:          "SUB.S.MI",
	arm_SUB_S_PL:          "SUB.S.PL",
	arm_SUB_S_VS:          "SUB.S.VS",
	arm_SUB_S_VC:          "SUB.S.VC",
	arm_SUB_S_HI:          "SUB.S.HI",
	arm_SUB_S_LS:          "SUB.S.LS",
	arm_SUB_S_GE:          "SUB.S.GE",
	arm_SUB_S_LT:          "SUB.S.LT",
	arm_SUB_S_GT:          "SUB.S.GT",
	arm_SUB_S_LE:          "SUB.S.LE",
	arm_SUB_S:             "SUB.S",
	arm_SUB_S_ZZ:          "SUB.S.ZZ",
	arm_SVC_EQ:            "SVC.EQ",
	arm_SVC_NE:            "SVC.NE",
	arm_SVC_CS:            "SVC.CS",
	arm_SVC_CC:            "SVC.CC",
	arm_SVC_MI:            "SVC.MI",
	arm_SVC_PL:            "SVC.PL",
	arm_SVC_VS:            "SVC.VS",
	arm_SVC_VC:            "SVC.VC",
	arm_SVC_HI:            "SVC.HI",
	arm_SVC_LS:            "SVC.LS",
	arm_SVC_GE:            "SVC.GE",
	arm_SVC_LT:            "SVC.LT",
	arm_SVC_GT:            "SVC.GT",
	arm_SVC_LE:            "SVC.LE",
	arm_SVC:               "SVC",
	arm_SVC_ZZ:            "SVC.ZZ",
	arm_SWP_EQ:            "SWP.EQ",
	arm_SWP_NE:            "SWP.NE",
	arm_SWP_CS:            "SWP.CS",
	arm_SWP_CC:            "SWP.CC",
	arm_SWP_MI:            "SWP.MI",
	arm_SWP_PL:            "SWP.PL",
	arm_SWP_VS:            "SWP.VS",
	arm_SWP_VC:            "SWP.VC",
	arm_SWP_HI:            "SWP.HI",
	arm_SWP_LS:            "SWP.LS",
	arm_SWP_GE:            "SWP.GE",
	arm_SWP_LT:            "SWP.LT",
	arm_SWP_GT:            "SWP.GT",
	arm_SWP_LE:            "SWP.LE",
	arm_SWP:               "SWP",
	arm_SWP_ZZ:            "SWP.ZZ",
	arm_SWP_B_EQ:          "SWP.B.EQ",
	arm_SWP_B_NE:          "SWP.B.NE",
	arm_SWP_B_CS:          "SWP.B.CS",
	arm_SWP_B_CC:          "SWP.B.CC",
	arm_SWP_B_MI:          "SWP.B.MI",
	arm_SWP_B_PL:          "SWP.B.PL",
	arm_SWP_B_VS:          "SWP.B.VS",
	arm_SWP_B_VC:          "SWP.B.VC",
	arm_SWP_B_HI:          "SWP.B.HI",
	arm_SWP_B_LS:          "SWP.B.LS",
	arm_SWP_B_GE:          "SWP.B.GE",
	arm_SWP_B_LT:          "SWP.B.LT",
	arm_SWP_B_GT:          "SWP.B.GT",
	arm_SWP_B_LE:          "SWP.B.LE",
	arm_SWP_B:             "SWP.B",
	arm_SWP_B_ZZ:          "SWP.B.ZZ",
	arm_SXTAB_EQ:          "SXTAB.EQ",
	arm_SXTAB_NE:          "SXTAB.NE",
	arm_SXTAB_CS:          "SXTAB.CS",
	arm_SXTAB_CC:          "SXTAB.CC",
	arm_SXTAB_MI:          "SXTAB.MI",
	arm_SXTAB_PL:          "SXTAB.PL",
	arm_SXTAB_VS:          "SXTAB.VS",
	arm_SXTAB_VC:          "SXTAB.VC",
	arm_SXTAB_HI:          "SXTAB.HI",
	arm_SXTAB_LS:          "SXTAB.LS",
	arm_SXTAB_GE:          "SXTAB.GE",
	arm_SXTAB_LT:          "SXTAB.LT",
	arm_SXTAB_GT:          "SXTAB.GT",
	arm_SXTAB_LE:          "SXTAB.LE",
	arm_SXTAB:             "SXTAB",
	arm_SXTAB_ZZ:          "SXTAB.ZZ",
	arm_SXTAB16_EQ:        "SXTAB16.EQ",
	arm_SXTAB16_NE:        "SXTAB16.NE",
	arm_SXTAB16_CS:        "SXTAB16.CS",
	arm_SXTAB16_CC:        "SXTAB16.CC",
	arm_SXTAB16_MI:        "SXTAB16.MI",
	arm_SXTAB16_PL:        "SXTAB16.PL",
	arm_SXTAB16_VS:        "SXTAB16.VS",
	arm_SXTAB16_VC:        "SXTAB16.VC",
	arm_SXTAB16_HI:        "SXTAB16.HI",
	arm_SXTAB16_LS:        "SXTAB16.LS",
	arm_SXTAB16_GE:        "SXTAB16.GE",
	arm_SXTAB16_LT:        "SXTAB16.LT",
	arm_SXTAB16_GT:        "SXTAB16.GT",
	arm_SXTAB16_LE:        "SXTAB16.LE",
	arm_SXTAB16:           "SXTAB16",
	arm_SXTAB16_ZZ:        "SXTAB16.ZZ",
	arm_SXTAH_EQ:          "SXTAH.EQ",
	arm_SXTAH_NE:          "SXTAH.NE",
	arm_SXTAH_CS:          "SXTAH.CS",
	arm_SXTAH_CC:          "SXTAH.CC",
	arm_SXTAH_MI:          "SXTAH.MI",
	arm_SXTAH_PL:          "SXTAH.PL",
	arm_SXTAH_VS:          "SXTAH.VS",
	arm_SXTAH_VC:          "SXTAH.VC",
	arm_SXTAH_HI:          "SXTAH.HI",
	arm_SXTAH_LS:          "SXTAH.LS",
	arm_SXTAH_GE:          "SXTAH.GE",
	arm_SXTAH_LT:          "SXTAH.LT",
	arm_SXTAH_GT:          "SXTAH.GT",
	arm_SXTAH_LE:          "SXTAH.LE",
	arm_SXTAH:             "SXTAH",
	arm_SXTAH_ZZ:          "SXTAH.ZZ",
	arm_SXTB_EQ:           "SXTB.EQ",
	arm_SXTB_NE:           "SXTB.NE",
	arm_SXTB_CS:           "SXTB.CS",
	arm_SXTB_CC:           "SXTB.CC",
	arm_SXTB_MI:           "SXTB.MI",
	arm_SXTB_PL:           "SXTB.PL",
	arm_SXTB_VS:           "SXTB.VS",
	arm_SXTB_VC:           "SXTB.VC",
	arm_SXTB_HI:           "SXTB.HI",
	arm_SXTB_LS:           "SXTB.LS",
	arm_SXTB_GE:           "SXTB.GE",
	arm_SXTB_LT:           "SXTB.LT",
	arm_SXTB_GT:           "SXTB.GT",
	arm_SXTB_LE:           "SXTB.LE",
	arm_SXTB:              "SXTB",
	arm_SXTB_ZZ:           "SXTB.ZZ",
	arm_SXTB16_EQ:         "SXTB16.EQ",
	arm_SXTB16_NE:         "SXTB16.NE",
	arm_SXTB16_CS:         "SXTB16.CS",
	arm_SXTB16_CC:         "SXTB16.CC",
	arm_SXTB16_MI:         "SXTB16.MI",
	arm_SXTB16_PL:         "SXTB16.PL",
	arm_SXTB16_VS:         "SXTB16.VS",
	arm_SXTB16_VC:         "SXTB16.VC",
	arm_SXTB16_HI:         "SXTB16.HI",
	arm_SXTB16_LS:         "SXTB16.LS",
	arm_SXTB16_GE:         "SXTB16.GE",
	arm_SXTB16_LT:         "SXTB16.LT",
	arm_SXTB16_GT:         "SXTB16.GT",
	arm_SXTB16_LE:         "SXTB16.LE",
	arm_SXTB16:            "SXTB16",
	arm_SXTB16_ZZ:         "SXTB16.ZZ",
	arm_SXTH_EQ:           "SXTH.EQ",
	arm_SXTH_NE:           "SXTH.NE",
	arm_SXTH_CS:           "SXTH.CS",
	arm_SXTH_CC:           "SXTH.CC",
	arm_SXTH_MI:           "SXTH.MI",
	arm_SXTH_PL:           "SXTH.PL",
	arm_SXTH_VS:           "SXTH.VS",
	arm_SXTH_VC:           "SXTH.VC",
	arm_SXTH_HI:           "SXTH.HI",
	arm_SXTH_LS:           "SXTH.LS",
	arm_SXTH_GE:           "SXTH.GE",
	arm_SXTH_LT:           "SXTH.LT",
	arm_SXTH_GT:           "SXTH.GT",
	arm_SXTH_LE:           "SXTH.LE",
	arm_SXTH:              "SXTH",
	arm_SXTH_ZZ:           "SXTH.ZZ",
	arm_TEQ_EQ:            "TEQ.EQ",
	arm_TEQ_NE:            "TEQ.NE",
	arm_TEQ_CS:            "TEQ.CS",
	arm_TEQ_CC:            "TEQ.CC",
	arm_TEQ_MI:            "TEQ.MI",
	arm_TEQ_PL:            "TEQ.PL",
	arm_TEQ_VS:            "TEQ.VS",
	arm_TEQ_VC:            "TEQ.VC",
	arm_TEQ_HI:            "TEQ.HI",
	arm_TEQ_LS:            "TEQ.LS",
	arm_TEQ_GE:            "TEQ.GE",
	arm_TEQ_LT:            "TEQ.LT",
	arm_TEQ_GT:            "TEQ.GT",
	arm_TEQ_LE:            "TEQ.LE",
	arm_TEQ:               "TEQ",
	arm_TEQ_ZZ:            "TEQ.ZZ",
	arm_TST_EQ:            "TST.EQ",
	arm_TST_NE:            "TST.NE",
	arm_TST_CS:            "TST.CS",
	arm_TST_CC:            "TST.CC",
	arm_TST_MI:            "TST.MI",
	arm_TST_PL:            "TST.PL",
	arm_TST_VS:            "TST.VS",
	arm_TST_VC:            "TST.VC",
	arm_TST_HI:            "TST.HI",
	arm_TST_LS:            "TST.LS",
	arm_TST_GE:            "TST.GE",
	arm_TST_LT:            "TST.LT",
	arm_TST_GT:            "TST.GT",
	arm_TST_LE:            "TST.LE",
	arm_TST:               "TST",
	arm_TST_ZZ:            "TST.ZZ",
	arm_UADD16_EQ:         "UADD16.EQ",
	arm_UADD16_NE:         "UADD16.NE",
	arm_UADD16_CS:         "UADD16.CS",
	arm_UADD16_CC:         "UADD16.CC",
	arm_UADD16_MI:         "UADD16.MI",
	arm_UADD16_PL:         "UADD16.PL",
	arm_UADD16_VS:         "UADD16.VS",
	arm_UADD16_VC:         "UADD16.VC",
	arm_UADD16_HI:         "UADD16.HI",
	arm_UADD16_LS:         "UADD16.LS",
	arm_UADD16_GE:         "UADD16.GE",
	arm_UADD16_LT:         "UADD16.LT",
	arm_UADD16_GT:         "UADD16.GT",
	arm_UADD16_LE:         "UADD16.LE",
	arm_UADD16:            "UADD16",
	arm_UADD16_ZZ:         "UADD16.ZZ",
	arm_UADD8_EQ:          "UADD8.EQ",
	arm_UADD8_NE:          "UADD8.NE",
	arm_UADD8_CS:          "UADD8.CS",
	arm_UADD8_CC:          "UADD8.CC",
	arm_UADD8_MI:          "UADD8.MI",
	arm_UADD8_PL:          "UADD8.PL",
	arm_UADD8_VS:          "UADD8.VS",
	arm_UADD8_VC:          "UADD8.VC",
	arm_UADD8_HI:          "UADD8.HI",
	arm_UADD8_LS:          "UADD8.LS",
	arm_UADD8_GE:          "UADD8.GE",
	arm_UADD8_LT:          "UADD8.LT",
	arm_UADD8_GT:          "UADD8.GT",
	arm_UADD8_LE:          "UADD8.LE",
	arm_UADD8:             "UADD8",
	arm_UADD8_ZZ:          "UADD8.ZZ",
	arm_UASX_EQ:           "UASX.EQ",
	arm_UASX_NE:           "UASX.NE",
	arm_UASX_CS:           "UASX.CS",
	arm_UASX_CC:           "UASX.CC",
	arm_UASX_MI:           "UASX.MI",
	arm_UASX_PL:           "UASX.PL",
	arm_UASX_VS:           "UASX.VS",
	arm_UASX_VC:           "UASX.VC",
	arm_UASX_HI:           "UASX.HI",
	arm_UASX_LS:           "UASX.LS",
	arm_UASX_GE:           "UASX.GE",
	arm_UASX_LT:           "UASX.LT",
	arm_UASX_GT:           "UASX.GT",
	arm_UASX_LE:           "UASX.LE",
	arm_UASX:              "UASX",
	arm_UASX_ZZ:           "UASX.ZZ",
	arm_UBFX_EQ:           "UBFX.EQ",
	arm_UBFX_NE:           "UBFX.NE",
	arm_UBFX_CS:           "UBFX.CS",
	arm_UBFX_CC:           "UBFX.CC",
	arm_UBFX_MI:           "UBFX.MI",
	arm_UBFX_PL:           "UBFX.PL",
	arm_UBFX_VS:           "UBFX.VS",
	arm_UBFX_VC:           "UBFX.VC",
	arm_UBFX_HI:           "UBFX.HI",
	arm_UBFX_LS:           "UBFX.LS",
	arm_UBFX_GE:           "UBFX.GE",
	arm_UBFX_LT:           "UBFX.LT",
	arm_UBFX_GT:           "UBFX.GT",
	arm_UBFX_LE:           "UBFX.LE",
	arm_UBFX:              "UBFX",
	arm_UBFX_ZZ:           "UBFX.ZZ",
	arm_UHADD16_EQ:        "UHADD16.EQ",
	arm_UHADD16_NE:        "UHADD16.NE",
	arm_UHADD16_CS:        "UHADD16.CS",
	arm_UHADD16_CC:        "UHADD16.CC",
	arm_UHADD16_MI:        "UHADD16.MI",
	arm_UHADD16_PL:        "UHADD16.PL",
	arm_UHADD16_VS:        "UHADD16.VS",
	arm_UHADD16_VC:        "UHADD16.VC",
	arm_UHADD16_HI:        "UHADD16.HI",
	arm_UHADD16_LS:        "UHADD16.LS",
	arm_UHADD16_GE:        "UHADD16.GE",
	arm_UHADD16_LT:        "UHADD16.LT",
	arm_UHADD16_GT:        "UHADD16.GT",
	arm_UHADD16_LE:        "UHADD16.LE",
	arm_UHADD16:           "UHADD16",
	arm_UHADD16_ZZ:        "UHADD16.ZZ",
	arm_UHADD8_EQ:         "UHADD8.EQ",
	arm_UHADD8_NE:         "UHADD8.NE",
	arm_UHADD8_CS:         "UHADD8.CS",
	arm_UHADD8_CC:         "UHADD8.CC",
	arm_UHADD8_MI:         "UHADD8.MI",
	arm_UHADD8_PL:         "UHADD8.PL",
	arm_UHADD8_VS:         "UHADD8.VS",
	arm_UHADD8_VC:         "UHADD8.VC",
	arm_UHADD8_HI:         "UHADD8.HI",
	arm_UHADD8_LS:         "UHADD8.LS",
	arm_UHADD8_GE:         "UHADD8.GE",
	arm_UHADD8_LT:         "UHADD8.LT",
	arm_UHADD8_GT:         "UHADD8.GT",
	arm_UHADD8_LE:         "UHADD8.LE",
	arm_UHADD8:            "UHADD8",
	arm_UHADD8_ZZ:         "UHADD8.ZZ",
	arm_UHASX_EQ:          "UHASX.EQ",
	arm_UHASX_NE:          "UHASX.NE",
	arm_UHASX_CS:          "UHASX.CS",
	arm_UHASX_CC:          "UHASX.CC",
	arm_UHASX_MI:          "UHASX.MI",
	arm_UHASX_PL:          "UHASX.PL",
	arm_UHASX_VS:          "UHASX.VS",
	arm_UHASX_VC:          "UHASX.VC",
	arm_UHASX_HI:          "UHASX.HI",
	arm_UHASX_LS:          "UHASX.LS",
	arm_UHASX_GE:          "UHASX.GE",
	arm_UHASX_LT:          "UHASX.LT",
	arm_UHASX_GT:          "UHASX.GT",
	arm_UHASX_LE:          "UHASX.LE",
	arm_UHASX:             "UHASX",
	arm_UHASX_ZZ:          "UHASX.ZZ",
	arm_UHSAX_EQ:          "UHSAX.EQ",
	arm_UHSAX_NE:          "UHSAX.NE",
	arm_UHSAX_CS:          "UHSAX.CS",
	arm_UHSAX_CC:          "UHSAX.CC",
	arm_UHSAX_MI:          "UHSAX.MI",
	arm_UHSAX_PL:          "UHSAX.PL",
	arm_UHSAX_VS:          "UHSAX.VS",
	arm_UHSAX_VC:          "UHSAX.VC",
	arm_UHSAX_HI:          "UHSAX.HI",
	arm_UHSAX_LS:          "UHSAX.LS",
	arm_UHSAX_GE:          "UHSAX.GE",
	arm_UHSAX_LT:          "UHSAX.LT",
	arm_UHSAX_GT:          "UHSAX.GT",
	arm_UHSAX_LE:          "UHSAX.LE",
	arm_UHSAX:             "UHSAX",
	arm_UHSAX_ZZ:          "UHSAX.ZZ",
	arm_UHSUB16_EQ:        "UHSUB16.EQ",
	arm_UHSUB16_NE:        "UHSUB16.NE",
	arm_UHSUB16_CS:        "UHSUB16.CS",
	arm_UHSUB16_CC:        "UHSUB16.CC",
	arm_UHSUB16_MI:        "UHSUB16.MI",
	arm_UHSUB16_PL:        "UHSUB16.PL",
	arm_UHSUB16_VS:        "UHSUB16.VS",
	arm_UHSUB16_VC:        "UHSUB16.VC",
	arm_UHSUB16_HI:        "UHSUB16.HI",
	arm_UHSUB16_LS:        "UHSUB16.LS",
	arm_UHSUB16_GE:        "UHSUB16.GE",
	arm_UHSUB16_LT:        "UHSUB16.LT",
	arm_UHSUB16_GT:        "UHSUB16.GT",
	arm_UHSUB16_LE:        "UHSUB16.LE",
	arm_UHSUB16:           "UHSUB16",
	arm_UHSUB16_ZZ:        "UHSUB16.ZZ",
	arm_UHSUB8_EQ:         "UHSUB8.EQ",
	arm_UHSUB8_NE:         "UHSUB8.NE",
	arm_UHSUB8_CS:         "UHSUB8.CS",
	arm_UHSUB8_CC:         "UHSUB8.CC",
	arm_UHSUB8_MI:         "UHSUB8.MI",
	arm_UHSUB8_PL:         "UHSUB8.PL",
	arm_UHSUB8_VS:         "UHSUB8.VS",
	arm_UHSUB8_VC:         "UHSUB8.VC",
	arm_UHSUB8_HI:         "UHSUB8.HI",
	arm_UHSUB8_LS:         "UHSUB8.LS",
	arm_UHSUB8_GE:         "UHSUB8.GE",
	arm_UHSUB8_LT:         "UHSUB8.LT",
	arm_UHSUB8_GT:         "UHSUB8.GT",
	arm_UHSUB8_LE:         "UHSUB8.LE",
	arm_UHSUB8:            "UHSUB8",
	arm_UHSUB8_ZZ:         "UHSUB8.ZZ",
	arm_UMAAL_EQ:          "UMAAL.EQ",
	arm_UMAAL_NE:          "UMAAL.NE",
	arm_UMAAL_CS:          "UMAAL.CS",
	arm_UMAAL_CC:          "UMAAL.CC",
	arm_UMAAL_MI:          "UMAAL.MI",
	arm_UMAAL_PL:          "UMAAL.PL",
	arm_UMAAL_VS:          "UMAAL.VS",
	arm_UMAAL_VC:          "UMAAL.VC",
	arm_UMAAL_HI:          "UMAAL.HI",
	arm_UMAAL_LS:          "UMAAL.LS",
	arm_UMAAL_GE:          "UMAAL.GE",
	arm_UMAAL_LT:          "UMAAL.LT",
	arm_UMAAL_GT:          "UMAAL.GT",
	arm_UMAAL_LE:          "UMAAL.LE",
	arm_UMAAL:             "UMAAL",
	arm_UMAAL_ZZ:          "UMAAL.ZZ",
	arm_UMLAL_EQ:          "UMLAL.EQ",
	arm_UMLAL_NE:          "UMLAL.NE",
	arm_UMLAL_CS:          "UMLAL.CS",
	arm_UMLAL_CC:          "UMLAL.CC",
	arm_UMLAL_MI:          "UMLAL.MI",
	arm_UMLAL_PL:          "UMLAL.PL",
	arm_UMLAL_VS:          "UMLAL.VS",
	arm_UMLAL_VC:          "UMLAL.VC",
	arm_UMLAL_HI:          "UMLAL.HI",
	arm_UMLAL_LS:          "UMLAL.LS",
	arm_UMLAL_GE:          "UMLAL.GE",
	arm_UMLAL_LT:          "UMLAL.LT",
	arm_UMLAL_GT:          "UMLAL.GT",
	arm_UMLAL_LE:          "UMLAL.LE",
	arm_UMLAL:             "UMLAL",
	arm_UMLAL_ZZ:          "UMLAL.ZZ",
	arm_UMLAL_S_EQ:        "UMLAL.S.EQ",
	arm_UMLAL_S_NE:        "UMLAL.S.NE",
	arm_UMLAL_S_CS:        "UMLAL.S.CS",
	arm_UMLAL_S_CC:        "UMLAL.S.CC",
	arm_UMLAL_S_MI:        "UMLAL.S.MI",
	arm_UMLAL_S_PL:        "UMLAL.S.PL",
	arm_UMLAL_S_VS:        "UMLAL.S.VS",
	arm_UMLAL_S_VC:        "UMLAL.S.VC",
	arm_UMLAL_S_HI:        "UMLAL.S.HI",
	arm_UMLAL_S_LS:        "UMLAL.S.LS",
	arm_UMLAL_S_GE:        "UMLAL.S.GE",
	arm_UMLAL_S_LT:        "UMLAL.S.LT",
	arm_UMLAL_S_GT:        "UMLAL.S.GT",
	arm_UMLAL_S_LE:        "UMLAL.S.LE",
	arm_UMLAL_S:           "UMLAL.S",
	arm_UMLAL_S_ZZ:        "UMLAL.S.ZZ",
	arm_UMULL_EQ:          "UMULL.EQ",
	arm_UMULL_NE:          "UMULL.NE",
	arm_UMULL_CS:          "UMULL.CS",
	arm_UMULL_CC:          "UMULL.CC",
	arm_UMULL_MI:          "UMULL.MI",
	arm_UMULL_PL:          "UMULL.PL",
	arm_UMULL_VS:          "UMULL.VS",
	arm_UMULL_VC:          "UMULL.VC",
	arm_UMULL_HI:          "UMULL.HI",
	arm_UMULL_LS:          "UMULL.LS",
	arm_UMULL_GE:          "UMULL.GE",
	arm_UMULL_LT:          "UMULL.LT",
	arm_UMULL_GT:          "UMULL.GT",
	arm_UMULL_LE:          "UMULL.LE",
	arm_UMULL:             "UMULL",
	arm_UMULL_ZZ:          "UMULL.ZZ",
	arm_UMULL_S_EQ:        "UMULL.S.EQ",
	arm_UMULL_S_NE:        "UMULL.S.NE",
	arm_UMULL_S_CS:        "UMULL.S.CS",
	arm_UMULL_S_CC:        "UMULL.S.CC",
	arm_UMULL_S_MI:        "UMULL.S.MI",
	arm_UMULL_S_PL:        "UMULL.S.PL",
	arm_UMULL_S_VS:        "UMULL.S.VS",
	arm_UMULL_S_VC:        "UMULL.S.VC",
	arm_UMULL_S_HI:        "UMULL.S.HI",
	arm_UMULL_S_LS:        "UMULL.S.LS",
	arm_UMULL_S_GE:        "UMULL.S.GE",
	arm_UMULL_S_LT:        "UMULL.S.LT",
	arm_UMULL_S_GT:        "UMULL.S.GT",
	arm_UMULL_S_LE:        "UMULL.S.LE",
	arm_UMULL_S:           "UMULL.S",
	arm_UMULL_S_ZZ:        "UMULL.S.ZZ",
	arm_UNDEF:             "UNDEF",
	arm_UQADD16_EQ:        "UQADD16.EQ",
	arm_UQADD16_NE:        "UQADD16.NE",
	arm_UQADD16_CS:        "UQADD16.CS",
	arm_UQADD16_CC:        "UQADD16.CC",
	arm_UQADD16_MI:        "UQADD16.MI",
	arm_UQADD16_PL:        "UQADD16.PL",
	arm_UQADD16_VS:        "UQADD16.VS",
	arm_UQADD16_VC:        "UQADD16.VC",
	arm_UQADD16_HI:        "UQADD16.HI",
	arm_UQADD16_LS:        "UQADD16.LS",
	arm_UQADD16_GE:        "UQADD16.GE",
	arm_UQADD16_LT:        "UQADD16.LT",
	arm_UQADD16_GT:        "UQADD16.GT",
	arm_UQADD16_LE:        "UQADD16.LE",
	arm_UQADD16:           "UQADD16",
	arm_UQADD16_ZZ:        "UQADD16.ZZ",
	arm_UQADD8_EQ:         "UQADD8.EQ",
	arm_UQADD8_NE:         "UQADD8.NE",
	arm_UQADD8_CS:         "UQADD8.CS",
	arm_UQADD8_CC:         "UQADD8.CC",
	arm_UQADD8_MI:         "UQADD8.MI",
	arm_UQADD8_PL:         "UQADD8.PL",
	arm_UQADD8_VS:         "UQADD8.VS",
	arm_UQADD8_VC:         "UQADD8.VC",
	arm_UQADD8_HI:         "UQADD8.HI",
	arm_UQADD8_LS:         "UQADD8.LS",
	arm_UQADD8_GE:         "UQADD8.GE",
	arm_UQADD8_LT:         "UQADD8.LT",
	arm_UQADD8_GT:         "UQADD8.GT",
	arm_UQADD8_LE:         "UQADD8.LE",
	arm_UQADD8:            "UQADD8",
	arm_UQADD8_ZZ:         "UQADD8.ZZ",
	arm_UQASX_EQ:          "UQASX.EQ",
	arm_UQASX_NE:          "UQASX.NE",
	arm_UQASX_CS:          "UQASX.CS",
	arm_UQASX_CC:          "UQASX.CC",
	arm_UQASX_MI:          "UQASX.MI",
	arm_UQASX_PL:          "UQASX.PL",
	arm_UQASX_VS:          "UQASX.VS",
	arm_UQASX_VC:          "UQASX.VC",
	arm_UQASX_HI:          "UQASX.HI",
	arm_UQASX_LS:          "UQASX.LS",
	arm_UQASX_GE:          "UQASX.GE",
	arm_UQASX_LT:          "UQASX.LT",
	arm_UQASX_GT:          "UQASX.GT",
	arm_UQASX_LE:          "UQASX.LE",
	arm_UQASX:             "UQASX",
	arm_UQASX_ZZ:          "UQASX.ZZ",
	arm_UQSAX_EQ:          "UQSAX.EQ",
	arm_UQSAX_NE:          "UQSAX.NE",
	arm_UQSAX_CS:          "UQSAX.CS",
	arm_UQSAX_CC:          "UQSAX.CC",
	arm_UQSAX_MI:          "UQSAX.MI",
	arm_UQSAX_PL:          "UQSAX.PL",
	arm_UQSAX_VS:          "UQSAX.VS",
	arm_UQSAX_VC:          "UQSAX.VC",
	arm_UQSAX_HI:          "UQSAX.HI",
	arm_UQSAX_LS:          "UQSAX.LS",
	arm_UQSAX_GE:          "UQSAX.GE",
	arm_UQSAX_LT:          "UQSAX.LT",
	arm_UQSAX_GT:          "UQSAX.GT",
	arm_UQSAX_LE:          "UQSAX.LE",
	arm_UQSAX:             "UQSAX",
	arm_UQSAX_ZZ:          "UQSAX.ZZ",
	arm_UQSUB16_EQ:        "UQSUB16.EQ",
	arm_UQSUB16_NE:        "UQSUB16.NE",
	arm_UQSUB16_CS:        "UQSUB16.CS",
	arm_UQSUB16_CC:        "UQSUB16.CC",
	arm_UQSUB16_MI:        "UQSUB16.MI",
	arm_UQSUB16_PL:        "UQSUB16.PL",
	arm_UQSUB16_VS:        "UQSUB16.VS",
	arm_UQSUB16_VC:        "UQSUB16.VC",
	arm_UQSUB16_HI:        "UQSUB16.HI",
	arm_UQSUB16_LS:        "UQSUB16.LS",
	arm_UQSUB16_GE:        "UQSUB16.GE",
	arm_UQSUB16_LT:        "UQSUB16.LT",
	arm_UQSUB16_GT:        "UQSUB16.GT",
	arm_UQSUB16_LE:        "UQSUB16.LE",
	arm_UQSUB16:           "UQSUB16",
	arm_UQSUB16_ZZ:        "UQSUB16.ZZ",
	arm_UQSUB8_EQ:         "UQSUB8.EQ",
	arm_UQSUB8_NE:         "UQSUB8.NE",
	arm_UQSUB8_CS:         "UQSUB8.CS",
	arm_UQSUB8_CC:         "UQSUB8.CC",
	arm_UQSUB8_MI:         "UQSUB8.MI",
	arm_UQSUB8_PL:         "UQSUB8.PL",
	arm_UQSUB8_VS:         "UQSUB8.VS",
	arm_UQSUB8_VC:         "UQSUB8.VC",
	arm_UQSUB8_HI:         "UQSUB8.HI",
	arm_UQSUB8_LS:         "UQSUB8.LS",
	arm_UQSUB8_GE:         "UQSUB8.GE",
	arm_UQSUB8_LT:         "UQSUB8.LT",
	arm_UQSUB8_GT:         "UQSUB8.GT",
	arm_UQSUB8_LE:         "UQSUB8.LE",
	arm_UQSUB8:            "UQSUB8",
	arm_UQSUB8_ZZ:         "UQSUB8.ZZ",
	arm_USAD8_EQ:          "USAD8.EQ",
	arm_USAD8_NE:          "USAD8.NE",
	arm_USAD8_CS:          "USAD8.CS",
	arm_USAD8_CC:          "USAD8.CC",
	arm_USAD8_MI:          "USAD8.MI",
	arm_USAD8_PL:          "USAD8.PL",
	arm_USAD8_VS:          "USAD8.VS",
	arm_USAD8_VC:          "USAD8.VC",
	arm_USAD8_HI:          "USAD8.HI",
	arm_USAD8_LS:          "USAD8.LS",
	arm_USAD8_GE:          "USAD8.GE",
	arm_USAD8_LT:          "USAD8.LT",
	arm_USAD8_GT:          "USAD8.GT",
	arm_USAD8_LE:          "USAD8.LE",
	arm_USAD8:             "USAD8",
	arm_USAD8_ZZ:          "USAD8.ZZ",
	arm_USADA8_EQ:         "USADA8.EQ",
	arm_USADA8_NE:         "USADA8.NE",
	arm_USADA8_CS:         "USADA8.CS",
	arm_USADA8_CC:         "USADA8.CC",
	arm_USADA8_MI:         "USADA8.MI",
	arm_USADA8_PL:         "USADA8.PL",
	arm_USADA8_VS:         "USADA8.VS",
	arm_USADA8_VC:         "USADA8.VC",
	arm_USADA8_HI:         "USADA8.HI",
	arm_USADA8_LS:         "USADA8.LS",
	arm_USADA8_GE:         "USADA8.GE",
	arm_USADA8_LT:         "USADA8.LT",
	arm_USADA8_GT:         "USADA8.GT",
	arm_USADA8_LE:         "USADA8.LE",
	arm_USADA8:            "USADA8",
	arm_USADA8_ZZ:         "USADA8.ZZ",
	arm_USAT_EQ:           "USAT.EQ",
	arm_USAT_NE:           "USAT.NE",
	arm_USAT_CS:           "USAT.CS",
	arm_USAT_CC:           "USAT.CC",
	arm_USAT_MI:           "USAT.MI",
	arm_USAT_PL:           "USAT.PL",
	arm_USAT_VS:           "USAT.VS",
	arm_USAT_VC:           "USAT.VC",
	arm_USAT_HI:           "USAT.HI",
	arm_USAT_LS:           "USAT.LS",
	arm_USAT_GE:           "USAT.GE",
	arm_USAT_LT:           "USAT.LT",
	arm_USAT_GT:           "USAT.GT",
	arm_USAT_LE:           "USAT.LE",
	arm_USAT:              "USAT",
	arm_USAT_ZZ:           "USAT.ZZ",
	arm_USAT16_EQ:         "USAT16.EQ",
	arm_USAT16_NE:         "USAT16.NE",
	arm_USAT16_CS:         "USAT16.CS",
	arm_USAT16_CC:         "USAT16.CC",
	arm_USAT16_MI:         "USAT16.MI",
	arm_USAT16_PL:         "USAT16.PL",
	arm_USAT16_VS:         "USAT16.VS",
	arm_USAT16_VC:         "USAT16.VC",
	arm_USAT16_HI:         "USAT16.HI",
	arm_USAT16_LS:         "USAT16.LS",
	arm_USAT16_GE:         "USAT16.GE",
	arm_USAT16_LT:         "USAT16.LT",
	arm_USAT16_GT:         "USAT16.GT",
	arm_USAT16_LE:         "USAT16.LE",
	arm_USAT16:            "USAT16",
	arm_USAT16_ZZ:         "USAT16.ZZ",
	arm_USAX_EQ:           "USAX.EQ",
	arm_USAX_NE:           "USAX.NE",
	arm_USAX_CS:           "USAX.CS",
	arm_USAX_CC:           "USAX.CC",
	arm_USAX_MI:           "USAX.MI",
	arm_USAX_PL:           "USAX.PL",
	arm_USAX_VS:           "USAX.VS",
	arm_USAX_VC:           "USAX.VC",
	arm_USAX_HI:           "USAX.HI",
	arm_USAX_LS:           "USAX.LS",
	arm_USAX_GE:           "USAX.GE",
	arm_USAX_LT:           "USAX.LT",
	arm_USAX_GT:           "USAX.GT",
	arm_USAX_LE:           "USAX.LE",
	arm_USAX:              "USAX",
	arm_USAX_ZZ:           "USAX.ZZ",
	arm_USUB16_EQ:         "USUB16.EQ",
	arm_USUB16_NE:         "USUB16.NE",
	arm_USUB16_CS:         "USUB16.CS",
	arm_USUB16_CC:         "USUB16.CC",
	arm_USUB16_MI:         "USUB16.MI",
	arm_USUB16_PL:         "USUB16.PL",
	arm_USUB16_VS:         "USUB16.VS",
	arm_USUB16_VC:         "USUB16.VC",
	arm_USUB16_HI:         "USUB16.HI",
	arm_USUB16_LS:         "USUB16.LS",
	arm_USUB16_GE:         "USUB16.GE",
	arm_USUB16_LT:         "USUB16.LT",
	arm_USUB16_GT:         "USUB16.GT",
	arm_USUB16_LE:         "USUB16.LE",
	arm_USUB16:            "USUB16",
	arm_USUB16_ZZ:         "USUB16.ZZ",
	arm_USUB8_EQ:          "USUB8.EQ",
	arm_USUB8_NE:          "USUB8.NE",
	arm_USUB8_CS:          "USUB8.CS",
	arm_USUB8_CC:          "USUB8.CC",
	arm_USUB8_MI:          "USUB8.MI",
	arm_USUB8_PL:          "USUB8.PL",
	arm_USUB8_VS:          "USUB8.VS",
	arm_USUB8_VC:          "USUB8.VC",
	arm_USUB8_HI:          "USUB8.HI",
	arm_USUB8_LS:          "USUB8.LS",
	arm_USUB8_GE:          "USUB8.GE",
	arm_USUB8_LT:          "USUB8.LT",
	arm_USUB8_GT:          "USUB8.GT",
	arm_USUB8_LE:          "USUB8.LE",
	arm_USUB8:             "USUB8",
	arm_USUB8_ZZ:          "USUB8.ZZ",
	arm_UXTAB_EQ:          "UXTAB.EQ",
	arm_UXTAB_NE:          "UXTAB.NE",
	arm_UXTAB_CS:          "UXTAB.CS",
	arm_UXTAB_CC:          "UXTAB.CC",
	arm_UXTAB_MI:          "UXTAB.MI",
	arm_UXTAB_PL:          "UXTAB.PL",
	arm_UXTAB_VS:          "UXTAB.VS",
	arm_UXTAB_VC:          "UXTAB.VC",
	arm_UXTAB_HI:          "UXTAB.HI",
	arm_UXTAB_LS:          "UXTAB.LS",
	arm_UXTAB_GE:          "UXTAB.GE",
	arm_UXTAB_LT:          "UXTAB.LT",
	arm_UXTAB_GT:          "UXTAB.GT",
	arm_UXTAB_LE:          "UXTAB.LE",
	arm_UXTAB:             "UXTAB",
	arm_UXTAB_ZZ:          "UXTAB.ZZ",
	arm_UXTAB16_EQ:        "UXTAB16.EQ",
	arm_UXTAB16_NE:        "UXTAB16.NE",
	arm_UXTAB16_CS:        "UXTAB16.CS",
	arm_UXTAB16_CC:        "UXTAB16.CC",
	arm_UXTAB16_MI:        "UXTAB16.MI",
	arm_UXTAB16_PL:        "UXTAB16.PL",
	arm_UXTAB16_VS:        "UXTAB16.VS",
	arm_UXTAB16_VC:        "UXTAB16.VC",
	arm_UXTAB16_HI:        "UXTAB16.HI",
	arm_UXTAB16_LS:        "UXTAB16.LS",
	arm_UXTAB16_GE:        "UXTAB16.GE",
	arm_UXTAB16_LT:        "UXTAB16.LT",
	arm_UXTAB16_GT:        "UXTAB16.GT",
	arm_UXTAB16_LE:        "UXTAB16.LE",
	arm_UXTAB16:           "UXTAB16",
	arm_UXTAB16_ZZ:        "UXTAB16.ZZ",
	arm_UXTAH_EQ:          "UXTAH.EQ",
	arm_UXTAH_NE:          "UXTAH.NE",
	arm_UXTAH_CS:          "UXTAH.CS",
	arm_UXTAH_CC:          "UXTAH.CC",
	arm_UXTAH_MI:          "UXTAH.MI",
	arm_UXTAH_PL:          "UXTAH.PL",
	arm_UXTAH_VS:          "UXTAH.VS",
	arm_UXTAH_VC:          "UXTAH.VC",
	arm_UXTAH_HI:          "UXTAH.HI",
	arm_UXTAH_LS:          "UXTAH.LS",
	arm_UXTAH_GE:          "UXTAH.GE",
	arm_UXTAH_LT:          "UXTAH.LT",
	arm_UXTAH_GT:          "UXTAH.GT",
	arm_UXTAH_LE:          "UXTAH.LE",
	arm_UXTAH:             "UXTAH",
	arm_UXTAH_ZZ:          "UXTAH.ZZ",
	arm_UXTB_EQ:           "UXTB.EQ",
	arm_UXTB_NE:           "UXTB.NE",
	arm_UXTB_CS:           "UXTB.CS",
	arm_UXTB_CC:           "UXTB.CC",
	arm_UXTB_MI:           "UXTB.MI",
	arm_UXTB_PL:           "UXTB.PL",
	arm_UXTB_VS:           "UXTB.VS",
	arm_UXTB_VC:           "UXTB.VC",
	arm_UXTB_HI:           "UXTB.HI",
	arm_UXTB_LS:           "UXTB.LS",
	arm_UXTB_GE:           "UXTB.GE",
	arm_UXTB_LT:           "UXTB.LT",
	arm_UXTB_GT:           "UXTB.GT",
	arm_UXTB_LE:           "UXTB.LE",
	arm_UXTB:              "UXTB",
	arm_UXTB_ZZ:           "UXTB.ZZ",
	arm_UXTB16_EQ:         "UXTB16.EQ",
	arm_UXTB16_NE:         "UXTB16.NE",
	arm_UXTB16_CS:         "UXTB16.CS",
	arm_UXTB16_CC:         "UXTB16.CC",
	arm_UXTB16_MI:         "UXTB16.MI",
	arm_UXTB16_PL:         "UXTB16.PL",
	arm_UXTB16_VS:         "UXTB16.VS",
	arm_UXTB16_VC:         "UXTB16.VC",
	arm_UXTB16_HI:         "UXTB16.HI",
	arm_UXTB16_LS:         "UXTB16.LS",
	arm_UXTB16_GE:         "UXTB16.GE",
	arm_UXTB16_LT:         "UXTB16.LT",
	arm_UXTB16_GT:         "UXTB16.GT",
	arm_UXTB16_LE:         "UXTB16.LE",
	arm_UXTB16:            "UXTB16",
	arm_UXTB16_ZZ:         "UXTB16.ZZ",
	arm_UXTH_EQ:           "UXTH.EQ",
	arm_UXTH_NE:           "UXTH.NE",
	arm_UXTH_CS:           "UXTH.CS",
	arm_UXTH_CC:           "UXTH.CC",
	arm_UXTH_MI:           "UXTH.MI",
	arm_UXTH_PL:           "UXTH.PL",
	arm_UXTH_VS:           "UXTH.VS",
	arm_UXTH_VC:           "UXTH.VC",
	arm_UXTH_HI:           "UXTH.HI",
	arm_UXTH_LS:           "UXTH.LS",
	arm_UXTH_GE:           "UXTH.GE",
	arm_UXTH_LT:           "UXTH.LT",
	arm_UXTH_GT:           "UXTH.GT",
	arm_UXTH_LE:           "UXTH.LE",
	arm_UXTH:              "UXTH",
	arm_UXTH_ZZ:           "UXTH.ZZ",
	arm_VABS_EQ_F32:       "VABS.EQ.F32",
	arm_VABS_NE_F32:       "VABS.NE.F32",
	arm_VABS_CS_F32:       "VABS.CS.F32",
	arm_VABS_CC_F32:       "VABS.CC.F32",
	arm_VABS_MI_F32:       "VABS.MI.F32",
	arm_VABS_PL_F32:       "VABS.PL.F32",
	arm_VABS_VS_F32:       "VABS.VS.F32",
	arm_VABS_VC_F32:       "VABS.VC.F32",
	arm_VABS_HI_F32:       "VABS.HI.F32",
	arm_VABS_LS_F32:       "VABS.LS.F32",
	arm_VABS_GE_F32:       "VABS.GE.F32",
	arm_VABS_LT_F32:       "VABS.LT.F32",
	arm_VABS_GT_F32:       "VABS.GT.F32",
	arm_VABS_LE_F32:       "VABS.LE.F32",
	arm_VABS_F32:          "VABS.F32",
	arm_VABS_ZZ_F32:       "VABS.ZZ.F32",
	arm_VABS_EQ_F64:       "VABS.EQ.F64",
	arm_VABS_NE_F64:       "VABS.NE.F64",
	arm_VABS_CS_F64:       "VABS.CS.F64",
	arm_VABS_CC_F64:       "VABS.CC.F64",
	arm_VABS_MI_F64:       "VABS.MI.F64",
	arm_VABS_PL_F64:       "VABS.PL.F64",
	arm_VABS_VS_F64:       "VABS.VS.F64",
	arm_VABS_VC_F64:       "VABS.VC.F64",
	arm_VABS_HI_F64:       "VABS.HI.F64",
	arm_VABS_LS_F64:       "VABS.LS.F64",
	arm_VABS_GE_F64:       "VABS.GE.F64",
	arm_VABS_LT_F64:       "VABS.LT.F64",
	arm_VABS_GT_F64:       "VABS.GT.F64",
	arm_VABS_LE_F64:       "VABS.LE.F64",
	arm_VABS_F64:          "VABS.F64",
	arm_VABS_ZZ_F64:       "VABS.ZZ.F64",
	arm_VADD_EQ_F32:       "VADD.EQ.F32",
	arm_VADD_NE_F32:       "VADD.NE.F32",
	arm_VADD_CS_F32:       "VADD.CS.F32",
	arm_VADD_CC_F32:       "VADD.CC.F32",
	arm_VADD_MI_F32:       "VADD.MI.F32",
	arm_VADD_PL_F32:       "VADD.PL.F32",
	arm_VADD_VS_F32:       "VADD.VS.F32",
	arm_VADD_VC_F32:       "VADD.VC.F32",
	arm_VADD_HI_F32:       "VADD.HI.F32",
	arm_VADD_LS_F32:       "VADD.LS.F32",
	arm_VADD_GE_F32:       "VADD.GE.F32",
	arm_VADD_LT_F32:       "VADD.LT.F32",
	arm_VADD_GT_F32:       "VADD.GT.F32",
	arm_VADD_LE_F32:       "VADD.LE.F32",
	arm_VADD_F32:          "VADD.F32",
	arm_VADD_ZZ_F32:       "VADD.ZZ.F32",
	arm_VADD_EQ_F64:       "VADD.EQ.F64",
	arm_VADD_NE_F64:       "VADD.NE.F64",
	arm_VADD_CS_F64:       "VADD.CS.F64",
	arm_VADD_CC_F64:       "VADD.CC.F64",
	arm_VADD_MI_F64:       "VADD.MI.F64",
	arm_VADD_PL_F64:       "VADD.PL.F64",
	arm_VADD_VS_F64:       "VADD.VS.F64",
	arm_VADD_VC_F64:       "VADD.VC.F64",
	arm_VADD_HI_F64:       "VADD.HI.F64",
	arm_VADD_LS_F64:       "VADD.LS.F64",
	arm_VADD_GE_F64:       "VADD.GE.F64",
	arm_VADD_LT_F64:       "VADD.LT.F64",
	arm_VADD_GT_F64:       "VADD.GT.F64",
	arm_VADD_LE_F64:       "VADD.LE.F64",
	arm_VADD_F64:          "VADD.F64",
	arm_VADD_ZZ_F64:       "VADD.ZZ.F64",
	arm_VCMP_EQ_F32:       "VCMP.EQ.F32",
	arm_VCMP_NE_F32:       "VCMP.NE.F32",
	arm_VCMP_CS_F32:       "VCMP.CS.F32",
	arm_VCMP_CC_F32:       "VCMP.CC.F32",
	arm_VCMP_MI_F32:       "VCMP.MI.F32",
	arm_VCMP_PL_F32:       "VCMP.PL.F32",
	arm_VCMP_VS_F32:       "VCMP.VS.F32",
	arm_VCMP_VC_F32:       "VCMP.VC.F32",
	arm_VCMP_HI_F32:       "VCMP.HI.F32",
	arm_VCMP_LS_F32:       "VCMP.LS.F32",
	arm_VCMP_GE_F32:       "VCMP.GE.F32",
	arm_VCMP_LT_F32:       "VCMP.LT.F32",
	arm_VCMP_GT_F32:       "VCMP.GT.F32",
	arm_VCMP_LE_F32:       "VCMP.LE.F32",
	arm_VCMP_F32:          "VCMP.F32",
	arm_VCMP_ZZ_F32:       "VCMP.ZZ.F32",
	arm_VCMP_EQ_F64:       "VCMP.EQ.F64",
	arm_VCMP_NE_F64:       "VCMP.NE.F64",
	arm_VCMP_CS_F64:       "VCMP.CS.F64",
	arm_VCMP_CC_F64:       "VCMP.CC.F64",
	arm_VCMP_MI_F64:       "VCMP.MI.F64",
	arm_VCMP_PL_F64:       "VCMP.PL.F64",
	arm_VCMP_VS_F64:       "VCMP.VS.F64",
	arm_VCMP_VC_F64:       "VCMP.VC.F64",
	arm_VCMP_HI_F64:       "VCMP.HI.F64",
	arm_VCMP_LS_F64:       "VCMP.LS.F64",
	arm_VCMP_GE_F64:       "VCMP.GE.F64",
	arm_VCMP_LT_F64:       "VCMP.LT.F64",
	arm_VCMP_GT_F64:       "VCMP.GT.F64",
	arm_VCMP_LE_F64:       "VCMP.LE.F64",
	arm_VCMP_F64:          "VCMP.F64",
	arm_VCMP_ZZ_F64:       "VCMP.ZZ.F64",
	arm_VCMP_E_EQ_F32:     "VCMP.E.EQ.F32",
	arm_VCMP_E_NE_F32:     "VCMP.E.NE.F32",
	arm_VCMP_E_CS_F32:     "VCMP.E.CS.F32",
	arm_VCMP_E_CC_F32:     "VCMP.E.CC.F32",
	arm_VCMP_E_MI_F32:     "VCMP.E.MI.F32",
	arm_VCMP_E_PL_F32:     "VCMP.E.PL.F32",
	arm_VCMP_E_VS_F32:     "VCMP.E.VS.F32",
	arm_VCMP_E_VC_F32:     "VCMP.E.VC.F32",
	arm_VCMP_E_HI_F32:     "VCMP.E.HI.F32",
	arm_VCMP_E_LS_F32:     "VCMP.E.LS.F32",
	arm_VCMP_E_GE_F32:     "VCMP.E.GE.F32",
	arm_VCMP_E_LT_F32:     "VCMP.E.LT.F32",
	arm_VCMP_E_GT_F32:     "VCMP.E.GT.F32",
	arm_VCMP_E_LE_F32:     "VCMP.E.LE.F32",
	arm_VCMP_E_F32:        "VCMP.E.F32",
	arm_VCMP_E_ZZ_F32:     "VCMP.E.ZZ.F32",
	arm_VCMP_E_EQ_F64:     "VCMP.E.EQ.F64",
	arm_VCMP_E_NE_F64:     "VCMP.E.NE.F64",
	arm_VCMP_E_CS_F64:     "VCMP.E.CS.F64",
	arm_VCMP_E_CC_F64:     "VCMP.E.CC.F64",
	arm_VCMP_E_MI_F64:     "VCMP.E.MI.F64",
	arm_VCMP_E_PL_F64:     "VCMP.E.PL.F64",
	arm_VCMP_E_VS_F64:     "VCMP.E.VS.F64",
	arm_VCMP_E_VC_F64:     "VCMP.E.VC.F64",
	arm_VCMP_E_HI_F64:     "VCMP.E.HI.F64",
	arm_VCMP_E_LS_F64:     "VCMP.E.LS.F64",
	arm_VCMP_E_GE_F64:     "VCMP.E.GE.F64",
	arm_VCMP_E_LT_F64:     "VCMP.E.LT.F64",
	arm_VCMP_E_GT_F64:     "VCMP.E.GT.F64",
	arm_VCMP_E_LE_F64:     "VCMP.E.LE.F64",
	arm_VCMP_E_F64:        "VCMP.E.F64",
	arm_VCMP_E_ZZ_F64:     "VCMP.E.ZZ.F64",
	arm_VCVT_EQ_F32_FXS16: "VCVT.EQ.F32.FXS16",
	arm_VCVT_NE_F32_FXS16: "VCVT.NE.F32.FXS16",
	arm_VCVT_CS_F32_FXS16: "VCVT.CS.F32.FXS16",
	arm_VCVT_CC_F32_FXS16: "VCVT.CC.F32.FXS16",
	arm_VCVT_MI_F32_FXS16: "VCVT.MI.F32.FXS16",
	arm_VCVT_PL_F32_FXS16: "VCVT.PL.F32.FXS16",
	arm_VCVT_VS_F32_FXS16: "VCVT.VS.F32.FXS16",
	arm_VCVT_VC_F32_FXS16: "VCVT.VC.F32.FXS16",
	arm_VCVT_HI_F32_FXS16: "VCVT.HI.F32.FXS16",
	arm_VCVT_LS_F32_FXS16: "VCVT.LS.F32.FXS16",
	arm_VCVT_GE_F32_FXS16: "VCVT.GE.F32.FXS16",
	arm_VCVT_LT_F32_FXS16: "VCVT.LT.F32.FXS16",
	arm_VCVT_GT_F32_FXS16: "VCVT.GT.F32.FXS16",
	arm_VCVT_LE_F32_FXS16: "VCVT.LE.F32.FXS16",
	arm_VCVT_F32_FXS16:    "VCVT.F32.FXS16",
	arm_VCVT_ZZ_F32_FXS16: "VCVT.ZZ.F32.FXS16",
	arm_VCVT_EQ_F32_FXS32: "VCVT.EQ.F32.FXS32",
	arm_VCVT_NE_F32_FXS32: "VCVT.NE.F32.FXS32",
	arm_VCVT_CS_F32_FXS32: "VCVT.CS.F32.FXS32",
	arm_VCVT_CC_F32_FXS32: "VCVT.CC.F32.FXS32",
	arm_VCVT_MI_F32_FXS32: "VCVT.MI.F32.FXS32",
	arm_VCVT_PL_F32_FXS32: "VCVT.PL.F32.FXS32",
	arm_VCVT_VS_F32_FXS32: "VCVT.VS.F32.FXS32",
	arm_VCVT_VC_F32_FXS32: "VCVT.VC.F32.FXS32",
	arm_VCVT_HI_F32_FXS32: "VCVT.HI.F32.FXS32",
	arm_VCVT_LS_F32_FXS32: "VCVT.LS.F32.FXS32",
	arm_VCVT_GE_F32_FXS32: "VCVT.GE.F32.FXS32",
	arm_VCVT_LT_F32_FXS32: "VCVT.LT.F32.FXS32",
	arm_VCVT_GT_F32_FXS32: "VCVT.GT.F32.FXS32",
	arm_VCVT_LE_F32_FXS32: "VCVT.LE.F32.FXS32",
	arm_VCVT_F32_FXS32:    "VCVT.F32.FXS32",
	arm_VCVT_ZZ_F32_FXS32: "VCVT.ZZ.F32.FXS32",
	arm_VCVT_EQ_F32_FXU16: "VCVT.EQ.F32.FXU16",
	arm_VCVT_NE_F32_FXU16: "VCVT.NE.F32.FXU16",
	arm_VCVT_CS_F32_FXU16: "VCVT.CS.F32.FXU16",
	arm_VCVT_CC_F32_FXU16: "VCVT.CC.F32.FXU16",
	arm_VCVT_MI_F32_FXU16: "VCVT.MI.F32.FXU16",
	arm_VCVT_PL_F32_FXU16: "VCVT.PL.F32.FXU16",
	arm_VCVT_VS_F32_FXU16: "VCVT.VS.F32.FXU16",
	arm_VCVT_VC_F32_FXU16: "VCVT.VC.F32.FXU16",
	arm_VCVT_HI_F32_FXU16: "VCVT.HI.F32.FXU16",
	arm_VCVT_LS_F32_FXU16: "VCVT.LS.F32.FXU16",
	arm_VCVT_GE_F32_FXU16: "VCVT.GE.F32.FXU16",
	arm_VCVT_LT_F32_FXU16: "VCVT.LT.F32.FXU16",
	arm_VCVT_GT_F32_FXU16: "VCVT.GT.F32.FXU16",
	arm_VCVT_LE_F32_FXU16: "VCVT.LE.F32.FXU16",
	arm_VCVT_F32_FXU16:    "VCVT.F32.FXU16",
	arm_VCVT_ZZ_F32_FXU16: "VCVT.ZZ.F32.FXU16",
	arm_VCVT_EQ_F32_FXU32: "VCVT.EQ.F32.FXU32",
	arm_VCVT_NE_F32_FXU32: "VCVT.NE.F32.FXU32",
	arm_VCVT_CS_F32_FXU32: "VCVT.CS.F32.FXU32",
	arm_VCVT_CC_F32_FXU32: "VCVT.CC.F32.FXU32",
	arm_VCVT_MI_F32_FXU32: "VCVT.MI.F32.FXU32",
	arm_VCVT_PL_F32_FXU32: "VCVT.PL.F32.FXU32",
	arm_VCVT_VS_F32_FXU32: "VCVT.VS.F32.FXU32",
	arm_VCVT_VC_F32_FXU32: "VCVT.VC.F32.FXU32",
	arm_VCVT_HI_F32_FXU32: "VCVT.HI.F32.FXU32",
	arm_VCVT_LS_F32_FXU32: "VCVT.LS.F32.FXU32",
	arm_VCVT_GE_F32_FXU32: "VCVT.GE.F32.FXU32",
	arm_VCVT_LT_F32_FXU32: "VCVT.LT.F32.FXU32",
	arm_VCVT_GT_F32_FXU32: "VCVT.GT.F32.FXU32",
	arm_VCVT_LE_F32_FXU32: "VCVT.LE.F32.FXU32",
	arm_VCVT_F32_FXU32:    "VCVT.F32.FXU32",
	arm_VCVT_ZZ_F32_FXU32: "VCVT.ZZ.F32.FXU32",
	arm_VCVT_EQ_F64_FXS16: "VCVT.EQ.F64.FXS16",
	arm_VCVT_NE_F64_FXS16: "VCVT.NE.F64.FXS16",
	arm_VCVT_CS_F64_FXS16: "VCVT.CS.F64.FXS16",
	arm_VCVT_CC_F64_FXS16: "VCVT.CC.F64.FXS16",
	arm_VCVT_MI_F64_FXS16: "VCVT.MI.F64.FXS16",
	arm_VCVT_PL_F64_FXS16: "VCVT.PL.F64.FXS16",
	arm_VCVT_VS_F64_FXS16: "VCVT.VS.F64.FXS16",
	arm_VCVT_VC_F64_FXS16: "VCVT.VC.F64.FXS16",
	arm_VCVT_HI_F64_FXS16: "VCVT.HI.F64.FXS16",
	arm_VCVT_LS_F64_FXS16: "VCVT.LS.F64.FXS16",
	arm_VCVT_GE_F64_FXS16: "VCVT.GE.F64.FXS16",
	arm_VCVT_LT_F64_FXS16: "VCVT.LT.F64.FXS16",
	arm_VCVT_GT_F64_FXS16: "VCVT.GT.F64.FXS16",
	arm_VCVT_LE_F64_FXS16: "VCVT.LE.F64.FXS16",
	arm_VCVT_F64_FXS16:    "VCVT.F64.FXS16",
	arm_VCVT_ZZ_F64_FXS16: "VCVT.ZZ.F64.FXS16",
	arm_VCVT_EQ_F64_FXS32: "VCVT.EQ.F64.FXS32",
	arm_VCVT_NE_F64_FXS32: "VCVT.NE.F64.FXS32",
	arm_VCVT_CS_F64_FXS32: "VCVT.CS.F64.FXS32",
	arm_VCVT_CC_F64_FXS32: "VCVT.CC.F64.FXS32",
	arm_VCVT_MI_F64_FXS32: "VCVT.MI.F64.FXS32",
	arm_VCVT_PL_F64_FXS32: "VCVT.PL.F64.FXS32",
	arm_VCVT_VS_F64_FXS32: "VCVT.VS.F64.FXS32",
	arm_VCVT_VC_F64_FXS32: "VCVT.VC.F64.FXS32",
	arm_VCVT_HI_F64_FXS32: "VCVT.HI.F64.FXS32",
	arm_VCVT_LS_F64_FXS32: "VCVT.LS.F64.FXS32",
	arm_VCVT_GE_F64_FXS32: "VCVT.GE.F64.FXS32",
	arm_VCVT_LT_F64_FXS32: "VCVT.LT.F64.FXS32",
	arm_VCVT_GT_F64_FXS32: "VCVT.GT.F64.FXS32",
	arm_VCVT_LE_F64_FXS32: "VCVT.LE.F64.FXS32",
	arm_VCVT_F64_FXS32:    "VCVT.F64.FXS32",
	arm_VCVT_ZZ_F64_FXS32: "VCVT.ZZ.F64.FXS32",
	arm_VCVT_EQ_F64_FXU16: "VCVT.EQ.F64.FXU16",
	arm_VCVT_NE_F64_FXU16: "VCVT.NE.F64.FXU16",
	arm_VCVT_CS_F64_FXU16: "VCVT.CS.F64.FXU16",
	arm_VCVT_CC_F64_FXU16: "VCVT.CC.F64.FXU16",
	arm_VCVT_MI_F64_FXU16: "VCVT.MI.F64.FXU16",
	arm_VCVT_PL_F64_FXU16: "VCVT.PL.F64.FXU16",
	arm_VCVT_VS_F64_FXU16: "VCVT.VS.F64.FXU16",
	arm_VCVT_VC_F64_FXU16: "VCVT.VC.F64.FXU16",
	arm_VCVT_HI_F64_FXU16: "VCVT.HI.F64.FXU16",
	arm_VCVT_LS_F64_FXU16: "VCVT.LS.F64.FXU16",
	arm_VCVT_GE_F64_FXU16: "VCVT.GE.F64.FXU16",
	arm_VCVT_LT_F64_FXU16: "VCVT.LT.F64.FXU16",
	arm_VCVT_GT_F64_FXU16: "VCVT.GT.F64.FXU16",
	arm_VCVT_LE_F64_FXU16: "VCVT.LE.F64.FXU16",
	arm_VCVT_F64_FXU16:    "VCVT.F64.FXU16",
	arm_VCVT_ZZ_F64_FXU16: "VCVT.ZZ.F64.FXU16",
	arm_VCVT_EQ_F64_FXU32: "VCVT.EQ.F64.FXU32",
	arm_VCVT_NE_F64_FXU32: "VCVT.NE.F64.FXU32",
	arm_VCVT_CS_F64_FXU32: "VCVT.CS.F64.FXU32",
	arm_VCVT_CC_F64_FXU32: "VCVT.CC.F64.FXU32",
	arm_VCVT_MI_F64_FXU32: "VCVT.MI.F64.FXU32",
	arm_VCVT_PL_F64_FXU32: "VCVT.PL.F64.FXU32",
	arm_VCVT_VS_F64_FXU32: "VCVT.VS.F64.FXU32",
	arm_VCVT_VC_F64_FXU32: "VCVT.VC.F64.FXU32",
	arm_VCVT_HI_F64_FXU32: "VCVT.HI.F64.FXU32",
	arm_VCVT_LS_F64_FXU32: "VCVT.LS.F64.FXU32",
	arm_VCVT_GE_F64_FXU32: "VCVT.GE.F64.FXU32",
	arm_VCVT_LT_F64_FXU32: "VCVT.LT.F64.FXU32",
	arm_VCVT_GT_F64_FXU32: "VCVT.GT.F64.FXU32",
	arm_VCVT_LE_F64_FXU32: "VCVT.LE.F64.FXU32",
	arm_VCVT_F64_FXU32:    "VCVT.F64.FXU32",
	arm_VCVT_ZZ_F64_FXU32: "VCVT.ZZ.F64.FXU32",
	arm_VCVT_EQ_F32_U32:   "VCVT.EQ.F32.U32",
	arm_VCVT_NE_F32_U32:   "VCVT.NE.F32.U32",
	arm_VCVT_CS_F32_U32:   "VCVT.CS.F32.U32",
	arm_VCVT_CC_F32_U32:   "VCVT.CC.F32.U32",
	arm_VCVT_MI_F32_U32:   "VCVT.MI.F32.U32",
	arm_VCVT_PL_F32_U32:   "VCVT.PL.F32.U32",
	arm_VCVT_VS_F32_U32:   "VCVT.VS.F32.U32",
	arm_VCVT_VC_F32_U32:   "VCVT.VC.F32.U32",
	arm_VCVT_HI_F32_U32:   "VCVT.HI.F32.U32",
	arm_VCVT_LS_F32_U32:   "VCVT.LS.F32.U32",
	arm_VCVT_GE_F32_U32:   "VCVT.GE.F32.U32",
	arm_VCVT_LT_F32_U32:   "VCVT.LT.F32.U32",
	arm_VCVT_GT_F32_U32:   "VCVT.GT.F32.U32",
	arm_VCVT_LE_F32_U32:   "VCVT.LE.F32.U32",
	arm_VCVT_F32_U32:      "VCVT.F32.U32",
	arm_VCVT_ZZ_F32_U32:   "VCVT.ZZ.F32.U32",
	arm_VCVT_EQ_F32_S32:   "VCVT.EQ.F32.S32",
	arm_VCVT_NE_F32_S32:   "VCVT.NE.F32.S32",
	arm_VCVT_CS_F32_S32:   "VCVT.CS.F32.S32",
	arm_VCVT_CC_F32_S32:   "VCVT.CC.F32.S32",
	arm_VCVT_MI_F32_S32:   "VCVT.MI.F32.S32",
	arm_VCVT_PL_F32_S32:   "VCVT.PL.F32.S32",
	arm_VCVT_VS_F32_S32:   "VCVT.VS.F32.S32",
	arm_VCVT_VC_F32_S32:   "VCVT.VC.F32.S32",
	arm_VCVT_HI_F32_S32:   "VCVT.HI.F32.S32",
	arm_VCVT_LS_F32_S32:   "VCVT.LS.F32.S32",
	arm_VCVT_GE_F32_S32:   "VCVT.GE.F32.S32",
	arm_VCVT_LT_F32_S32:   "VCVT.LT.F32.S32",
	arm_VCVT_GT_F32_S32:   "VCVT.GT.F32.S32",
	arm_VCVT_LE_F32_S32:   "VCVT.LE.F32.S32",
	arm_VCVT_F32_S32:      "VCVT.F32.S32",
	arm_VCVT_ZZ_F32_S32:   "VCVT.ZZ.F32.S32",
	arm_VCVT_EQ_F64_U32:   "VCVT.EQ.F64.U32",
	arm_VCVT_NE_F64_U32:   "VCVT.NE.F64.U32",
	arm_VCVT_CS_F64_U32:   "VCVT.CS.F64.U32",
	arm_VCVT_CC_F64_U32:   "VCVT.CC.F64.U32",
	arm_VCVT_MI_F64_U32:   "VCVT.MI.F64.U32",
	arm_VCVT_PL_F64_U32:   "VCVT.PL.F64.U32",
	arm_VCVT_VS_F64_U32:   "VCVT.VS.F64.U32",
	arm_VCVT_VC_F64_U32:   "VCVT.VC.F64.U32",
	arm_VCVT_HI_F64_U32:   "VCVT.HI.F64.U32",
	arm_VCVT_LS_F64_U32:   "VCVT.LS.F64.U32",
	arm_VCVT_GE_F64_U32:   "VCVT.GE.F64.U32",
	arm_VCVT_LT_F64_U32:   "VCVT.LT.F64.U32",
	arm_VCVT_GT_F64_U32:   "VCVT.GT.F64.U32",
	arm_VCVT_LE_F64_U32:   "VCVT.LE.F64.U32",
	arm_VCVT_F64_U32:      "VCVT.F64.U32",
	arm_VCVT_ZZ_F64_U32:   "VCVT.ZZ.F64.U32",
	arm_VCVT_EQ_F64_S32:   "VCVT.EQ.F64.S32",
	arm_VCVT_NE_F64_S32:   "VCVT.NE.F64.S32",
	arm_VCVT_CS_F64_S32:   "VCVT.CS.F64.S32",
	arm_VCVT_CC_F64_S32:   "VCVT.CC.F64.S32",
	arm_VCVT_MI_F64_S32:   "VCVT.MI.F64.S32",
	arm_VCVT_PL_F64_S32:   "VCVT.PL.F64.S32",
	arm_VCVT_VS_F64_S32:   "VCVT.VS.F64.S32",
	arm_VCVT_VC_F64_S32:   "VCVT.VC.F64.S32",
	arm_VCVT_HI_F64_S32:   "VCVT.HI.F64.S32",
	arm_VCVT_LS_F64_S32:   "VCVT.LS.F64.S32",
	arm_VCVT_GE_F64_S32:   "VCVT.GE.F64.S32",
	arm_VCVT_LT_F64_S32:   "VCVT.LT.F64.S32",
	arm_VCVT_GT_F64_S32:   "VCVT.GT.F64.S32",
	arm_VCVT_LE_F64_S32:   "VCVT.LE.F64.S32",
	arm_VCVT_F64_S32:      "VCVT.F64.S32",
	arm_VCVT_ZZ_F64_S32:   "VCVT.ZZ.F64.S32",
	arm_VCVT_EQ_F64_F32:   "VCVT.EQ.F64.F32",
	arm_VCVT_NE_F64_F32:   "VCVT.NE.F64.F32",
	arm_VCVT_CS_F64_F32:   "VCVT.CS.F64.F32",
	arm_VCVT_CC_F64_F32:   "VCVT.CC.F64.F32",
	arm_VCVT_MI_F64_F32:   "VCVT.MI.F64.F32",
	arm_VCVT_PL_F64_F32:   "VCVT.PL.F64.F32",
	arm_VCVT_VS_F64_F32:   "VCVT.VS.F64.F32",
	arm_VCVT_VC_F64_F32:   "VCVT.VC.F64.F32",
	arm_VCVT_HI_F64_F32:   "VCVT.HI.F64.F32",
	arm_VCVT_LS_F64_F32:   "VCVT.LS.F64.F32",
	arm_VCVT_GE_F64_F32:   "VCVT.GE.F64.F32",
	arm_VCVT_LT_F64_F32:   "VCVT.LT.F64.F32",
	arm_VCVT_GT_F64_F32:   "VCVT.GT.F64.F32",
	arm_VCVT_LE_F64_F32:   "VCVT.LE.F64.F32",
	arm_VCVT_F64_F32:      "VCVT.F64.F32",
	arm_VCVT_ZZ_F64_F32:   "VCVT.ZZ.F64.F32",
	arm_VCVT_EQ_F32_F64:   "VCVT.EQ.F32.F64",
	arm_VCVT_NE_F32_F64:   "VCVT.NE.F32.F64",
	arm_VCVT_CS_F32_F64:   "VCVT.CS.F32.F64",
	arm_VCVT_CC_F32_F64:   "VCVT.CC.F32.F64",
	arm_VCVT_MI_F32_F64:   "VCVT.MI.F32.F64",
	arm_VCVT_PL_F32_F64:   "VCVT.PL.F32.F64",
	arm_VCVT_VS_F32_F64:   "VCVT.VS.F32.F64",
	arm_VCVT_VC_F32_F64:   "VCVT.VC.F32.F64",
	arm_VCVT_HI_F32_F64:   "VCVT.HI.F32.F64",
	arm_VCVT_LS_F32_F64:   "VCVT.LS.F32.F64",
	arm_VCVT_GE_F32_F64:   "VCVT.GE.F32.F64",
	arm_VCVT_LT_F32_F64:   "VCVT.LT.F32.F64",
	arm_VCVT_GT_F32_F64:   "VCVT.GT.F32.F64",
	arm_VCVT_LE_F32_F64:   "VCVT.LE.F32.F64",
	arm_VCVT_F32_F64:      "VCVT.F32.F64",
	arm_VCVT_ZZ_F32_F64:   "VCVT.ZZ.F32.F64",
	arm_VCVT_EQ_FXS16_F32: "VCVT.EQ.FXS16.F32",
	arm_VCVT_NE_FXS16_F32: "VCVT.NE.FXS16.F32",
	arm_VCVT_CS_FXS16_F32: "VCVT.CS.FXS16.F32",
	arm_VCVT_CC_FXS16_F32: "VCVT.CC.FXS16.F32",
	arm_VCVT_MI_FXS16_F32: "VCVT.MI.FXS16.F32",
	arm_VCVT_PL_FXS16_F32: "VCVT.PL.FXS16.F32",
	arm_VCVT_VS_FXS16_F32: "VCVT.VS.FXS16.F32",
	arm_VCVT_VC_FXS16_F32: "VCVT.VC.FXS16.F32",
	arm_VCVT_HI_FXS16_F32: "VCVT.HI.FXS16.F32",
	arm_VCVT_LS_FXS16_F32: "VCVT.LS.FXS16.F32",
	arm_VCVT_GE_FXS16_F32: "VCVT.GE.FXS16.F32",
	arm_VCVT_LT_FXS16_F32: "VCVT.LT.FXS16.F32",
	arm_VCVT_GT_FXS16_F32: "VCVT.GT.FXS16.F32",
	arm_VCVT_LE_FXS16_F32: "VCVT.LE.FXS16.F32",
	arm_VCVT_FXS16_F32:    "VCVT.FXS16.F32",
	arm_VCVT_ZZ_FXS16_F32: "VCVT.ZZ.FXS16.F32",
	arm_VCVT_EQ_FXS16_F64: "VCVT.EQ.FXS16.F64",
	arm_VCVT_NE_FXS16_F64: "VCVT.NE.FXS16.F64",
	arm_VCVT_CS_FXS16_F64: "VCVT.CS.FXS16.F64",
	arm_VCVT_CC_FXS16_F64: "VCVT.CC.FXS16.F64",
	arm_VCVT_MI_FXS16_F64: "VCVT.MI.FXS16.F64",
	arm_VCVT_PL_FXS16_F64: "VCVT.PL.FXS16.F64",
	arm_VCVT_VS_FXS16_F64: "VCVT.VS.FXS16.F64",
	arm_VCVT_VC_FXS16_F64: "VCVT.VC.FXS16.F64",
	arm_VCVT_HI_FXS16_F64: "VCVT.HI.FXS16.F64",
	arm_VCVT_LS_FXS16_F64: "VCVT.LS.FXS16.F64",
	arm_VCVT_GE_FXS16_F64: "VCVT.GE.FXS16.F64",
	arm_VCVT_LT_FXS16_F64: "VCVT.LT.FXS16.F64",
	arm_VCVT_GT_FXS16_F64: "VCVT.GT.FXS16.F64",
	arm_VCVT_LE_FXS16_F64: "VCVT.LE.FXS16.F64",
	arm_VCVT_FXS16_F64:    "VCVT.FXS16.F64",
	arm_VCVT_ZZ_FXS16_F64: "VCVT.ZZ.FXS16.F64",
	arm_VCVT_EQ_FXS32_F32: "VCVT.EQ.FXS32.F32",
	arm_VCVT_NE_FXS32_F32: "VCVT.NE.FXS32.F32",
	arm_VCVT_CS_FXS32_F32: "VCVT.CS.FXS32.F32",
	arm_VCVT_CC_FXS32_F32: "VCVT.CC.FXS32.F32",
	arm_VCVT_MI_FXS32_F32: "VCVT.MI.FXS32.F32",
	arm_VCVT_PL_FXS32_F32: "VCVT.PL.FXS32.F32",
	arm_VCVT_VS_FXS32_F32: "VCVT.VS.FXS32.F32",
	arm_VCVT_VC_FXS32_F32: "VCVT.VC.FXS32.F32",
	arm_VCVT_HI_FXS32_F32: "VCVT.HI.FXS32.F32",
	arm_VCVT_LS_FXS32_F32: "VCVT.LS.FXS32.F32",
	arm_VCVT_GE_FXS32_F32: "VCVT.GE.FXS32.F32",
	arm_VCVT_LT_FXS32_F32: "VCVT.LT.FXS32.F32",
	arm_VCVT_GT_FXS32_F32: "VCVT.GT.FXS32.F32",
	arm_VCVT_LE_FXS32_F32: "VCVT.LE.FXS32.F32",
	arm_VCVT_FXS32_F32:    "VCVT.FXS32.F32",
	arm_VCVT_ZZ_FXS32_F32: "VCVT.ZZ.FXS32.F32",
	arm_VCVT_EQ_FXS32_F64: "VCVT.EQ.FXS32.F64",
	arm_VCVT_NE_FXS32_F64: "VCVT.NE.FXS32.F64",
	arm_VCVT_CS_FXS32_F64: "VCVT.CS.FXS32.F64",
	arm_VCVT_CC_FXS32_F64: "VCVT.CC.FXS32.F64",
	arm_VCVT_MI_FXS32_F64: "VCVT.MI.FXS32.F64",
	arm_VCVT_PL_FXS32_F64: "VCVT.PL.FXS32.F64",
	arm_VCVT_VS_FXS32_F64: "VCVT.VS.FXS32.F64",
	arm_VCVT_VC_FXS32_F64: "VCVT.VC.FXS32.F64",
	arm_VCVT_HI_FXS32_F64: "VCVT.HI.FXS32.F64",
	arm_VCVT_LS_FXS32_F64: "VCVT.LS.FXS32.F64",
	arm_VCVT_GE_FXS32_F64: "VCVT.GE.FXS32.F64",
	arm_VCVT_LT_FXS32_F64: "VCVT.LT.FXS32.F64",
	arm_VCVT_GT_FXS32_F64: "VCVT.GT.FXS32.F64",
	arm_VCVT_LE_FXS32_F64: "VCVT.LE.FXS32.F64",
	arm_VCVT_FXS32_F64:    "VCVT.FXS32.F64",
	arm_VCVT_ZZ_FXS32_F64: "VCVT.ZZ.FXS32.F64",
	arm_VCVT_EQ_FXU16_F32: "VCVT.EQ.FXU16.F32",
	arm_VCVT_NE_FXU16_F32: "VCVT.NE.FXU16.F32",
	arm_VCVT_CS_FXU16_F32: "VCVT.CS.FXU16.F32",
	arm_VCVT_CC_FXU16_F32: "VCVT.CC.FXU16.F32",
	arm_VCVT_MI_FXU16_F32: "VCVT.MI.FXU16.F32",
	arm_VCVT_PL_FXU16_F32: "VCVT.PL.FXU16.F32",
	arm_VCVT_VS_FXU16_F32: "VCVT.VS.FXU16.F32",
	arm_VCVT_VC_FXU16_F32: "VCVT.VC.FXU16.F32",
	arm_VCVT_HI_FXU16_F32: "VCVT.HI.FXU16.F32",
	arm_VCVT_LS_FXU16_F32: "VCVT.LS.FXU16.F32",
	arm_VCVT_GE_FXU16_F32: "VCVT.GE.FXU16.F32",
	arm_VCVT_LT_FXU16_F32: "VCVT.LT.FXU16.F32",
	arm_VCVT_GT_FXU16_F32: "VCVT.GT.FXU16.F32",
	arm_VCVT_LE_FXU16_F32: "VCVT.LE.FXU16.F32",
	arm_VCVT_FXU16_F32:    "VCVT.FXU16.F32",
	arm_VCVT_ZZ_FXU16_F32: "VCVT.ZZ.FXU16.F32",
	arm_VCVT_EQ_FXU16_F64: "VCVT.EQ.FXU16.F64",
	arm_VCVT_NE_FXU16_F64: "VCVT.NE.FXU16.F64",
	arm_VCVT_CS_FXU16_F64: "VCVT.CS.FXU16.F64",
	arm_VCVT_CC_FXU16_F64: "VCVT.CC.FXU16.F64",
	arm_VCVT_MI_FXU16_F64: "VCVT.MI.FXU16.F64",
	arm_VCVT_PL_FXU16_F64: "VCVT.PL.FXU16.F64",
	arm_VCVT_VS_FXU16_F64: "VCVT.VS.FXU16.F64",
	arm_VCVT_VC_FXU16_F64: "VCVT.VC.FXU16.F64",
	arm_VCVT_HI_FXU16_F64: "VCVT.HI.FXU16.F64",
	arm_VCVT_LS_FXU16_F64: "VCVT.LS.FXU16.F64",
	arm_VCVT_GE_FXU16_F64: "VCVT.GE.FXU16.F64",
	arm_VCVT_LT_FXU16_F64: "VCVT.LT.FXU16.F64",
	arm_VCVT_GT_FXU16_F64: "VCVT.GT.FXU16.F64",
	arm_VCVT_LE_FXU16_F64: "VCVT.LE.FXU16.F64",
	arm_VCVT_FXU16_F64:    "VCVT.FXU16.F64",
	arm_VCVT_ZZ_FXU16_F64: "VCVT.ZZ.FXU16.F64",
	arm_VCVT_EQ_FXU32_F32: "VCVT.EQ.FXU32.F32",
	arm_VCVT_NE_FXU32_F32: "VCVT.NE.FXU32.F32",
	arm_VCVT_CS_FXU32_F32: "VCVT.CS.FXU32.F32",
	arm_VCVT_CC_FXU32_F32: "VCVT.CC.FXU32.F32",
	arm_VCVT_MI_FXU32_F32: "VCVT.MI.FXU32.F32",
	arm_VCVT_PL_FXU32_F32: "VCVT.PL.FXU32.F32",
	arm_VCVT_VS_FXU32_F32: "VCVT.VS.FXU32.F32",
	arm_VCVT_VC_FXU32_F32: "VCVT.VC.FXU32.F32",
	arm_VCVT_HI_FXU32_F32: "VCVT.HI.FXU32.F32",
	arm_VCVT_LS_FXU32_F32: "VCVT.LS.FXU32.F32",
	arm_VCVT_GE_FXU32_F32: "VCVT.GE.FXU32.F32",
	arm_VCVT_LT_FXU32_F32: "VCVT.LT.FXU32.F32",
	arm_VCVT_GT_FXU32_F32: "VCVT.GT.FXU32.F32",
	arm_VCVT_LE_FXU32_F32: "VCVT.LE.FXU32.F32",
	arm_VCVT_FXU32_F32:    "VCVT.FXU32.F32",
	arm_VCVT_ZZ_FXU32_F32: "VCVT.ZZ.FXU32.F32",
	arm_VCVT_EQ_FXU32_F64: "VCVT.EQ.FXU32.F64",
	arm_VCVT_NE_FXU32_F64: "VCVT.NE.FXU32.F64",
	arm_VCVT_CS_FXU32_F64: "VCVT.CS.FXU32.F64",
	arm_VCVT_CC_FXU32_F64: "VCVT.CC.FXU32.F64",
	arm_VCVT_MI_FXU32_F64: "VCVT.MI.FXU32.F64",
	arm_VCVT_PL_FXU32_F64: "VCVT.PL.FXU32.F64",
	arm_VCVT_VS_FXU32_F64: "VCVT.VS.FXU32.F64",
	arm_VCVT_VC_FXU32_F64: "VCVT.VC.FXU32.F64",
	arm_VCVT_HI_FXU32_F64: "VCVT.HI.FXU32.F64",
	arm_VCVT_LS_FXU32_F64: "VCVT.LS.FXU32.F64",
	arm_VCVT_GE_FXU32_F64: "VCVT.GE.FXU32.F64",
	arm_VCVT_LT_FXU32_F64: "VCVT.LT.FXU32.F64",
	arm_VCVT_GT_FXU32_F64: "VCVT.GT.FXU32.F64",
	arm_VCVT_LE_FXU32_F64: "VCVT.LE.FXU32.F64",
	arm_VCVT_FXU32_F64:    "VCVT.FXU32.F64",
	arm_VCVT_ZZ_FXU32_F64: "VCVT.ZZ.FXU32.F64",
	arm_VCVTB_EQ_F32_F16:  "VCVTB.EQ.F32.F16",
	arm_VCVTB_NE_F32_F16:  "VCVTB.NE.F32.F16",
	arm_VCVTB_CS_F32_F16:  "VCVTB.CS.F32.F16",
	arm_VCVTB_CC_F32_F16:  "VCVTB.CC.F32.F16",
	arm_VCVTB_MI_F32_F16:  "VCVTB.MI.F32.F16",
	arm_VCVTB_PL_F32_F16:  "VCVTB.PL.F32.F16",
	arm_VCVTB_VS_F32_F16:  "VCVTB.VS.F32.F16",
	arm_VCVTB_VC_F32_F16:  "VCVTB.VC.F32.F16",
	arm_VCVTB_HI_F32_F16:  "VCVTB.HI.F32.F16",
	arm_VCVTB_LS_F32_F16:  "VCVTB.LS.F32.F16",
	arm_VCVTB_GE_F32_F16:  "VCVTB.GE.F32.F16",
	arm_VCVTB_LT_F32_F16:  "VCVTB.LT.F32.F16",
	arm_VCVTB_GT_F32_F16:  "VCVTB.GT.F32.F16",
	arm_VCVTB_LE_F32_F16:  "VCVTB.LE.F32.F16",
	arm_VCVTB_F32_F16:     "VCVTB.F32.F16",
	arm_VCVTB_ZZ_F32_F16:  "VCVTB.ZZ.F32.F16",
	arm_VCVTB_EQ_F16_F32:  "VCVTB.EQ.F16.F32",
	arm_VCVTB_NE_F16_F32:  "VCVTB.NE.F16.F32",
	arm_VCVTB_CS_F16_F32:  "VCVTB.CS.F16.F32",
	arm_VCVTB_CC_F16_F32:  "VCVTB.CC.F16.F32",
	arm_VCVTB_MI_F16_F32:  "VCVTB.MI.F16.F32",
	arm_VCVTB_PL_F16_F32:  "VCVTB.PL.F16.F32",
	arm_VCVTB_VS_F16_F32:  "VCVTB.VS.F16.F32",
	arm_VCVTB_VC_F16_F32:  "VCVTB.VC.F16.F32",
	arm_VCVTB_HI_F16_F32:  "VCVTB.HI.F16.F32",
	arm_VCVTB_LS_F16_F32:  "VCVTB.LS.F16.F32",
	arm_VCVTB_GE_F16_F32:  "VCVTB.GE.F16.F32",
	arm_VCVTB_LT_F16_F32:  "VCVTB.LT.F16.F32",
	arm_VCVTB_GT_F16_F32:  "VCVTB.GT.F16.F32",
	arm_VCVTB_LE_F16_F32:  "VCVTB.LE.F16.F32",
	arm_VCVTB_F16_F32:     "VCVTB.F16.F32",
	arm_VCVTB_ZZ_F16_F32:  "VCVTB.ZZ.F16.F32",
	arm_VCVTT_EQ_F32_F16:  "VCVTT.EQ.F32.F16",
	arm_VCVTT_NE_F32_F16:  "VCVTT.NE.F32.F16",
	arm_VCVTT_CS_F32_F16:  "VCVTT.CS.F32.F16",
	arm_VCVTT_CC_F32_F16:  "VCVTT.CC.F32.F16",
	arm_VCVTT_MI_F32_F16:  "VCVTT.MI.F32.F16",
	arm_VCVTT_PL_F32_F16:  "VCVTT.PL.F32.F16",
	arm_VCVTT_VS_F32_F16:  "VCVTT.VS.F32.F16",
	arm_VCVTT_VC_F32_F16:  "VCVTT.VC.F32.F16",
	arm_VCVTT_HI_F32_F16:  "VCVTT.HI.F32.F16",
	arm_VCVTT_LS_F32_F16:  "VCVTT.LS.F32.F16",
	arm_VCVTT_GE_F32_F16:  "VCVTT.GE.F32.F16",
	arm_VCVTT_LT_F32_F16:  "VCVTT.LT.F32.F16",
	arm_VCVTT_GT_F32_F16:  "VCVTT.GT.F32.F16",
	arm_VCVTT_LE_F32_F16:  "VCVTT.LE.F32.F16",
	arm_VCVTT_F32_F16:     "VCVTT.F32.F16",
	arm_VCVTT_ZZ_F32_F16:  "VCVTT.ZZ.F32.F16",
	arm_VCVTT_EQ_F16_F32:  "VCVTT.EQ.F16.F32",
	arm_VCVTT_NE_F16_F32:  "VCVTT.NE.F16.F32",
	arm_VCVTT_CS_F16_F32:  "VCVTT.CS.F16.F32",
	arm_VCVTT_CC_F16_F32:  "VCVTT.CC.F16.F32",
	arm_VCVTT_MI_F16_F32:  "VCVTT.MI.F16.F32",
	arm_VCVTT_PL_F16_F32:  "VCVTT.PL.F16.F32",
	arm_VCVTT_VS_F16_F32:  "VCVTT.VS.F16.F32",
	arm_VCVTT_VC_F16_F32:  "VCVTT.VC.F16.F32",
	arm_VCVTT_HI_F16_F32:  "VCVTT.HI.F16.F32",
	arm_VCVTT_LS_F16_F32:  "VCVTT.LS.F16.F32",
	arm_VCVTT_GE_F16_F32:  "VCVTT.GE.F16.F32",
	arm_VCVTT_LT_F16_F32:  "VCVTT.LT.F16.F32",
	arm_VCVTT_GT_F16_F32:  "VCVTT.GT.F16.F32",
	arm_VCVTT_LE_F16_F32:  "VCVTT.LE.F16.F32",
	arm_VCVTT_F16_F32:     "VCVTT.F16.F32",
	arm_VCVTT_ZZ_F16_F32:  "VCVTT.ZZ.F16.F32",
	arm_VCVTR_EQ_U32_F32:  "VCVTR.EQ.U32.F32",
	arm_VCVTR_NE_U32_F32:  "VCVTR.NE.U32.F32",
	arm_VCVTR_CS_U32_F32:  "VCVTR.CS.U32.F32",
	arm_VCVTR_CC_U32_F32:  "VCVTR.CC.U32.F32",
	arm_VCVTR_MI_U32_F32:  "VCVTR.MI.U32.F32",
	arm_VCVTR_PL_U32_F32:  "VCVTR.PL.U32.F32",
	arm_VCVTR_VS_U32_F32:  "VCVTR.VS.U32.F32",
	arm_VCVTR_VC_U32_F32:  "VCVTR.VC.U32.F32",
	arm_VCVTR_HI_U32_F32:  "VCVTR.HI.U32.F32",
	arm_VCVTR_LS_U32_F32:  "VCVTR.LS.U32.F32",
	arm_VCVTR_GE_U32_F32:  "VCVTR.GE.U32.F32",
	arm_VCVTR_LT_U32_F32:  "VCVTR.LT.U32.F32",
	arm_VCVTR_GT_U32_F32:  "VCVTR.GT.U32.F32",
	arm_VCVTR_LE_U32_F32:  "VCVTR.LE.U32.F32",
	arm_VCVTR_U32_F32:     "VCVTR.U32.F32",
	arm_VCVTR_ZZ_U32_F32:  "VCVTR.ZZ.U32.F32",
	arm_VCVTR_EQ_U32_F64:  "VCVTR.EQ.U32.F64",
	arm_VCVTR_NE_U32_F64:  "VCVTR.NE.U32.F64",
	arm_VCVTR_CS_U32_F64:  "VCVTR.CS.U32.F64",
	arm_VCVTR_CC_U32_F64:  "VCVTR.CC.U32.F64",
	arm_VCVTR_MI_U32_F64:  "VCVTR.MI.U32.F64",
	arm_VCVTR_PL_U32_F64:  "VCVTR.PL.U32.F64",
	arm_VCVTR_VS_U32_F64:  "VCVTR.VS.U32.F64",
	arm_VCVTR_VC_U32_F64:  "VCVTR.VC.U32.F64",
	arm_VCVTR_HI_U32_F64:  "VCVTR.HI.U32.F64",
	arm_VCVTR_LS_U32_F64:  "VCVTR.LS.U32.F64",
	arm_VCVTR_GE_U32_F64:  "VCVTR.GE.U32.F64",
	arm_VCVTR_LT_U32_F64:  "VCVTR.LT.U32.F64",
	arm_VCVTR_GT_U32_F64:  "VCVTR.GT.U32.F64",
	arm_VCVTR_LE_U32_F64:  "VCVTR.LE.U32.F64",
	arm_VCVTR_U32_F64:     "VCVTR.U32.F64",
	arm_VCVTR_ZZ_U32_F64:  "VCVTR.ZZ.U32.F64",
	arm_VCVTR_EQ_S32_F32:  "VCVTR.EQ.S32.F32",
	arm_VCVTR_NE_S32_F32:  "VCVTR.NE.S32.F32",
	arm_VCVTR_CS_S32_F32:  "VCVTR.CS.S32.F32",
	arm_VCVTR_CC_S32_F32:  "VCVTR.CC.S32.F32",
	arm_VCVTR_MI_S32_F32:  "VCVTR.MI.S32.F32",
	arm_VCVTR_PL_S32_F32:  "VCVTR.PL.S32.F32",
	arm_VCVTR_VS_S32_F32:  "VCVTR.VS.S32.F32",
	arm_VCVTR_VC_S32_F32:  "VCVTR.VC.S32.F32",
	arm_VCVTR_HI_S32_F32:  "VCVTR.HI.S32.F32",
	arm_VCVTR_LS_S32_F32:  "VCVTR.LS.S32.F32",
	arm_VCVTR_GE_S32_F32:  "VCVTR.GE.S32.F32",
	arm_VCVTR_LT_S32_F32:  "VCVTR.LT.S32.F32",
	arm_VCVTR_GT_S32_F32:  "VCVTR.GT.S32.F32",
	arm_VCVTR_LE_S32_F32:  "VCVTR.LE.S32.F32",
	arm_VCVTR_S32_F32:     "VCVTR.S32.F32",
	arm_VCVTR_ZZ_S32_F32:  "VCVTR.ZZ.S32.F32",
	arm_VCVTR_EQ_S32_F64:  "VCVTR.EQ.S32.F64",
	arm_VCVTR_NE_S32_F64:  "VCVTR.NE.S32.F64",
	arm_VCVTR_CS_S32_F64:  "VCVTR.CS.S32.F64",
	arm_VCVTR_CC_S32_F64:  "VCVTR.CC.S32.F64",
	arm_VCVTR_MI_S32_F64:  "VCVTR.MI.S32.F64",
	arm_VCVTR_PL_S32_F64:  "VCVTR.PL.S32.F64",
	arm_VCVTR_VS_S32_F64:  "VCVTR.VS.S32.F64",
	arm_VCVTR_VC_S32_F64:  "VCVTR.VC.S32.F64",
	arm_VCVTR_HI_S32_F64:  "VCVTR.HI.S32.F64",
	arm_VCVTR_LS_S32_F64:  "VCVTR.LS.S32.F64",
	arm_VCVTR_GE_S32_F64:  "VCVTR.GE.S32.F64",
	arm_VCVTR_LT_S32_F64:  "VCVTR.LT.S32.F64",
	arm_VCVTR_GT_S32_F64:  "VCVTR.GT.S32.F64",
	arm_VCVTR_LE_S32_F64:  "VCVTR.LE.S32.F64",
	arm_VCVTR_S32_F64:     "VCVTR.S32.F64",
	arm_VCVTR_ZZ_S32_F64:  "VCVTR.ZZ.S32.F64",
	arm_VCVT_EQ_U32_F32:   "VCVT.EQ.U32.F32",
	arm_VCVT_NE_U32_F32:   "VCVT.NE.U32.F32",
	arm_VCVT_CS_U32_F32:   "VCVT.CS.U32.F32",
	arm_VCVT_CC_U32_F32:   "VCVT.CC.U32.F32",
	arm_VCVT_MI_U32_F32:   "VCVT.MI.U32.F32",
	arm_VCVT_PL_U32_F32:   "VCVT.PL.U32.F32",
	arm_VCVT_VS_U32_F32:   "VCVT.VS.U32.F32",
	arm_VCVT_VC_U32_F32:   "VCVT.VC.U32.F32",
	arm_VCVT_HI_U32_F32:   "VCVT.HI.U32.F32",
	arm_VCVT_LS_U32_F32:   "VCVT.LS.U32.F32",
	arm_VCVT_GE_U32_F32:   "VCVT.GE.U32.F32",
	arm_VCVT_LT_U32_F32:   "VCVT.LT.U32.F32",
	arm_VCVT_GT_U32_F32:   "VCVT.GT.U32.F32",
	arm_VCVT_LE_U32_F32:   "VCVT.LE.U32.F32",
	arm_VCVT_U32_F32:      "VCVT.U32.F32",
	arm_VCVT_ZZ_U32_F32:   "VCVT.ZZ.U32.F32",
	arm_VCVT_EQ_U32_F64:   "VCVT.EQ.U32.F64",
	arm_VCVT_NE_U32_F64:   "VCVT.NE.U32.F64",
	arm_VCVT_CS_U32_F64:   "VCVT.CS.U32.F64",
	arm_VCVT_CC_U32_F64:   "VCVT.CC.U32.F64",
	arm_VCVT_MI_U32_F64:   "VCVT.MI.U32.F64",
	arm_VCVT_PL_U32_F64:   "VCVT.PL.U32.F64",
	arm_VCVT_VS_U32_F64:   "VCVT.VS.U32.F64",
	arm_VCVT_VC_U32_F64:   "VCVT.VC.U32.F64",
	arm_VCVT_HI_U32_F64:   "VCVT.HI.U32.F64",
	arm_VCVT_LS_U32_F64:   "VCVT.LS.U32.F64",
	arm_VCVT_GE_U32_F64:   "VCVT.GE.U32.F64",
	arm_VCVT_LT_U32_F64:   "VCVT.LT.U32.F64",
	arm_VCVT_GT_U32_F64:   "VCVT.GT.U32.F64",
	arm_VCVT_LE_U32_F64:   "VCVT.LE.U32.F64",
	arm_VCVT_U32_F64:      "VCVT.U32.F64",
	arm_VCVT_ZZ_U32_F64:   "VCVT.ZZ.U32.F64",
	arm_VCVT_EQ_S32_F32:   "VCVT.EQ.S32.F32",
	arm_VCVT_NE_S32_F32:   "VCVT.NE.S32.F32",
	arm_VCVT_CS_S32_F32:   "VCVT.CS.S32.F32",
	arm_VCVT_CC_S32_F32:   "VCVT.CC.S32.F32",
	arm_VCVT_MI_S32_F32:   "VCVT.MI.S32.F32",
	arm_VCVT_PL_S32_F32:   "VCVT.PL.S32.F32",
	arm_VCVT_VS_S32_F32:   "VCVT.VS.S32.F32",
	arm_VCVT_VC_S32_F32:   "VCVT.VC.S32.F32",
	arm_VCVT_HI_S32_F32:   "VCVT.HI.S32.F32",
	arm_VCVT_LS_S32_F32:   "VCVT.LS.S32.F32",
	arm_VCVT_GE_S32_F32:   "VCVT.GE.S32.F32",
	arm_VCVT_LT_S32_F32:   "VCVT.LT.S32.F32",
	arm_VCVT_GT_S32_F32:   "VCVT.GT.S32.F32",
	arm_VCVT_LE_S32_F32:   "VCVT.LE.S32.F32",
	arm_VCVT_S32_F32:      "VCVT.S32.F32",
	arm_VCVT_ZZ_S32_F32:   "VCVT.ZZ.S32.F32",
	arm_VCVT_EQ_S32_F64:   "VCVT.EQ.S32.F64",
	arm_VCVT_NE_S32_F64:   "VCVT.NE.S32.F64",
	arm_VCVT_CS_S32_F64:   "VCVT.CS.S32.F64",
	arm_VCVT_CC_S32_F64:   "VCVT.CC.S32.F64",
	arm_VCVT_MI_S32_F64:   "VCVT.MI.S32.F64",
	arm_VCVT_PL_S32_F64:   "VCVT.PL.S32.F64",
	arm_VCVT_VS_S32_F64:   "VCVT.VS.S32.F64",
	arm_VCVT_VC_S32_F64:   "VCVT.VC.S32.F64",
	arm_VCVT_HI_S32_F64:   "VCVT.HI.S32.F64",
	arm_VCVT_LS_S32_F64:   "VCVT.LS.S32.F64",
	arm_VCVT_GE_S32_F64:   "VCVT.GE.S32.F64",
	arm_VCVT_LT_S32_F64:   "VCVT.LT.S32.F64",
	arm_VCVT_GT_S32_F64:   "VCVT.GT.S32.F64",
	arm_VCVT_LE_S32_F64:   "VCVT.LE.S32.F64",
	arm_VCVT_S32_F64:      "VCVT.S32.F64",
	arm_VCVT_ZZ_S32_F64:   "VCVT.ZZ.S32.F64",
	arm_VDIV_EQ_F32:       "VDIV.EQ.F32",
	arm_VDIV_NE_F32:       "VDIV.NE.F32",
	arm_VDIV_CS_F32:       "VDIV.CS.F32",
	arm_VDIV_CC_F32:       "VDIV.CC.F32",
	arm_VDIV_MI_F32:       "VDIV.MI.F32",
	arm_VDIV_PL_F32:       "VDIV.PL.F32",
	arm_VDIV_VS_F32:       "VDIV.VS.F32",
	arm_VDIV_VC_F32:       "VDIV.VC.F32",
	arm_VDIV_HI_F32:       "VDIV.HI.F32",
	arm_VDIV_LS_F32:       "VDIV.LS.F32",
	arm_VDIV_GE_F32:       "VDIV.GE.F32",
	arm_VDIV_LT_F32:       "VDIV.LT.F32",
	arm_VDIV_GT_F32:       "VDIV.GT.F32",
	arm_VDIV_LE_F32:       "VDIV.LE.F32",
	arm_VDIV_F32:          "VDIV.F32",
	arm_VDIV_ZZ_F32:       "VDIV.ZZ.F32",
	arm_VDIV_EQ_F64:       "VDIV.EQ.F64",
	arm_VDIV_NE_F64:       "VDIV.NE.F64",
	arm_VDIV_CS_F64:       "VDIV.CS.F64",
	arm_VDIV_CC_F64:       "VDIV.CC.F64",
	arm_VDIV_MI_F64:       "VDIV.MI.F64",
	arm_VDIV_PL_F64:       "VDIV.PL.F64",
	arm_VDIV_VS_F64:       "VDIV.VS.F64",
	arm_VDIV_VC_F64:       "VDIV.VC.F64",
	arm_VDIV_HI_F64:       "VDIV.HI.F64",
	arm_VDIV_LS_F64:       "VDIV.LS.F64",
	arm_VDIV_GE_F64:       "VDIV.GE.F64",
	arm_VDIV_LT_F64:       "VDIV.LT.F64",
	arm_VDIV_GT_F64:       "VDIV.GT.F64",
	arm_VDIV_LE_F64:       "VDIV.LE.F64",
	arm_VDIV_F64:          "VDIV.F64",
	arm_VDIV_ZZ_F64:       "VDIV.ZZ.F64",
	arm_VLDR_EQ:           "VLDR.EQ",
	arm_VLDR_NE:           "VLDR.NE",
	arm_VLDR_CS:           "VLDR.CS",
	arm_VLDR_CC:           "VLDR.CC",
	arm_VLDR_MI:           "VLDR.MI",
	arm_VLDR_PL:           "VLDR.PL",
	arm_VLDR_VS:           "VLDR.VS",
	arm_VLDR_VC:           "VLDR.VC",
	arm_VLDR_HI:           "VLDR.HI",
	arm_VLDR_LS:           "VLDR.LS",
	arm_VLDR_GE:           "VLDR.GE",
	arm_VLDR_LT:           "VLDR.LT",
	arm_VLDR_GT:           "VLDR.GT",
	arm_VLDR_LE:           "VLDR.LE",
	arm_VLDR:              "VLDR",
	arm_VLDR_ZZ:           "VLDR.ZZ",
	arm_VMLA_EQ_F32:       "VMLA.EQ.F32",
	arm_VMLA_NE_F32:       "VMLA.NE.F32",
	arm_VMLA_CS_F32:       "VMLA.CS.F32",
	arm_VMLA_CC_F32:       "VMLA.CC.F32",
	arm_VMLA_MI_F32:       "VMLA.MI.F32",
	arm_VMLA_PL_F32:       "VMLA.PL.F32",
	arm_VMLA_VS_F32:       "VMLA.VS.F32",
	arm_VMLA_VC_F32:       "VMLA.VC.F32",
	arm_VMLA_HI_F32:       "VMLA.HI.F32",
	arm_VMLA_LS_F32:       "VMLA.LS.F32",
	arm_VMLA_GE_F32:       "VMLA.GE.F32",
	arm_VMLA_LT_F32:       "VMLA.LT.F32",
	arm_VMLA_GT_F32:       "VMLA.GT.F32",
	arm_VMLA_LE_F32:       "VMLA.LE.F32",
	arm_VMLA_F32:          "VMLA.F32",
	arm_VMLA_ZZ_F32:       "VMLA.ZZ.F32",
	arm_VMLA_EQ_F64:       "VMLA.EQ.F64",
	arm_VMLA_NE_F64:       "VMLA.NE.F64",
	arm_VMLA_CS_F64:       "VMLA.CS.F64",
	arm_VMLA_CC_F64:       "VMLA.CC.F64",
	arm_VMLA_MI_F64:       "VMLA.MI.F64",
	arm_VMLA_PL_F64:       "VMLA.PL.F64",
	arm_VMLA_VS_F64:       "VMLA.VS.F64",
	arm_VMLA_VC_F64:       "VMLA.VC.F64",
	arm_VMLA_HI_F64:       "VMLA.HI.F64",
	arm_VMLA_LS_F64:       "VMLA.LS.F64",
	arm_VMLA_GE_F64:       "VMLA.GE.F64",
	arm_VMLA_LT_F64:       "VMLA.LT.F64",
	arm_VMLA_GT_F64:       "VMLA.GT.F64",
	arm_VMLA_LE_F64:       "VMLA.LE.F64",
	arm_VMLA_F64:          "VMLA.F64",
	arm_VMLA_ZZ_F64:       "VMLA.ZZ.F64",
	arm_VMLS_EQ_F32:       "VMLS.EQ.F32",
	arm_VMLS_NE_F32:       "VMLS.NE.F32",
	arm_VMLS_CS_F32:       "VMLS.CS.F32",
	arm_VMLS_CC_F32:       "VMLS.CC.F32",
	arm_VMLS_MI_F32:       "VMLS.MI.F32",
	arm_VMLS_PL_F32:       "VMLS.PL.F32",
	arm_VMLS_VS_F32:       "VMLS.VS.F32",
	arm_VMLS_VC_F32:       "VMLS.VC.F32",
	arm_VMLS_HI_F32:       "VMLS.HI.F32",
	arm_VMLS_LS_F32:       "VMLS.LS.F32",
	arm_VMLS_GE_F32:       "VMLS.GE.F32",
	arm_VMLS_LT_F32:       "VMLS.LT.F32",
	arm_VMLS_GT_F32:       "VMLS.GT.F32",
	arm_VMLS_LE_F32:       "VMLS.LE.F32",
	arm_VMLS_F32:          "VMLS.F32",
	arm_VMLS_ZZ_F32:       "VMLS.ZZ.F32",
	arm_VMLS_EQ_F64:       "VMLS.EQ.F64",
	arm_VMLS_NE_F64:       "VMLS.NE.F64",
	arm_VMLS_CS_F64:       "VMLS.CS.F64",
	arm_VMLS_CC_F64:       "VMLS.CC.F64",
	arm_VMLS_MI_F64:       "VMLS.MI.F64",
	arm_VMLS_PL_F64:       "VMLS.PL.F64",
	arm_VMLS_VS_F64:       "VMLS.VS.F64",
	arm_VMLS_VC_F64:       "VMLS.VC.F64",
	arm_VMLS_HI_F64:       "VMLS.HI.F64",
	arm_VMLS_LS_F64:       "VMLS.LS.F64",
	arm_VMLS_GE_F64:       "VMLS.GE.F64",
	arm_VMLS_LT_F64:       "VMLS.LT.F64",
	arm_VMLS_GT_F64:       "VMLS.GT.F64",
	arm_VMLS_LE_F64:       "VMLS.LE.F64",
	arm_VMLS_F64:          "VMLS.F64",
	arm_VMLS_ZZ_F64:       "VMLS.ZZ.F64",
	arm_VMOV_EQ:           "VMOV.EQ",
	arm_VMOV_NE:           "VMOV.NE",
	arm_VMOV_CS:           "VMOV.CS",
	arm_VMOV_CC:           "VMOV.CC",
	arm_VMOV_MI:           "VMOV.MI",
	arm_VMOV_PL:           "VMOV.PL",
	arm_VMOV_VS:           "VMOV.VS",
	arm_VMOV_VC:           "VMOV.VC",
	arm_VMOV_HI:           "VMOV.HI",
	arm_VMOV_LS:           "VMOV.LS",
	arm_VMOV_GE:           "VMOV.GE",
	arm_VMOV_LT:           "VMOV.LT",
	arm_VMOV_GT:           "VMOV.GT",
	arm_VMOV_LE:           "VMOV.LE",
	arm_VMOV:              "VMOV",
	arm_VMOV_ZZ:           "VMOV.ZZ",
	arm_VMOV_EQ_32:        "VMOV.EQ.32",
	arm_VMOV_NE_32:        "VMOV.NE.32",
	arm_VMOV_CS_32:        "VMOV.CS.32",
	arm_VMOV_CC_32:        "VMOV.CC.32",
	arm_VMOV_MI_32:        "VMOV.MI.32",
	arm_VMOV_PL_32:        "VMOV.PL.32",
	arm_VMOV_VS_32:        "VMOV.VS.32",
	arm_VMOV_VC_32:        "VMOV.VC.32",
	arm_VMOV_HI_32:        "VMOV.HI.32",
	arm_VMOV_LS_32:        "VMOV.LS.32",
	arm_VMOV_GE_32:        "VMOV.GE.32",
	arm_VMOV_LT_32:        "VMOV.LT.32",
	arm_VMOV_GT_32:        "VMOV.GT.32",
	arm_VMOV_LE_32:        "VMOV.LE.32",
	arm_VMOV_32:           "VMOV.32",
	arm_VMOV_ZZ_32:        "VMOV.ZZ.32",
	arm_VMOV_EQ_F32:       "VMOV.EQ.F32",
	arm_VMOV_NE_F32:       "VMOV.NE.F32",
	arm_VMOV_CS_F32:       "VMOV.CS.F32",
	arm_VMOV_CC_F32:       "VMOV.CC.F32",
	arm_VMOV_MI_F32:       "VMOV.MI.F32",
	arm_VMOV_PL_F32:       "VMOV.PL.F32",
	arm_VMOV_VS_F32:       "VMOV.VS.F32",
	arm_VMOV_VC_F32:       "VMOV.VC.F32",
	arm_VMOV_HI_F32:       "VMOV.HI.F32",
	arm_VMOV_LS_F32:       "VMOV.LS.F32",
	arm_VMOV_GE_F32:       "VMOV.GE.F32",
	arm_VMOV_LT_F32:       "VMOV.LT.F32",
	arm_VMOV_GT_F32:       "VMOV.GT.F32",
	arm_VMOV_LE_F32:       "VMOV.LE.F32",
	arm_VMOV_F32:          "VMOV.F32",
	arm_VMOV_ZZ_F32:       "VMOV.ZZ.F32",
	arm_VMOV_EQ_F64:       "VMOV.EQ.F64",
	arm_VMOV_NE_F64:       "VMOV.NE.F64",
	arm_VMOV_CS_F64:       "VMOV.CS.F64",
	arm_VMOV_CC_F64:       "VMOV.CC.F64",
	arm_VMOV_MI_F64:       "VMOV.MI.F64",
	arm_VMOV_PL_F64:       "VMOV.PL.F64",
	arm_VMOV_VS_F64:       "VMOV.VS.F64",
	arm_VMOV_VC_F64:       "VMOV.VC.F64",
	arm_VMOV_HI_F64:       "VMOV.HI.F64",
	arm_VMOV_LS_F64:       "VMOV.LS.F64",
	arm_VMOV_GE_F64:       "VMOV.GE.F64",
	arm_VMOV_LT_F64:       "VMOV.LT.F64",
	arm_VMOV_GT_F64:       "VMOV.GT.F64",
	arm_VMOV_LE_F64:       "VMOV.LE.F64",
	arm_VMOV_F64:          "VMOV.F64",
	arm_VMOV_ZZ_F64:       "VMOV.ZZ.F64",
	arm_VMRS_EQ:           "VMRS.EQ",
	arm_VMRS_NE:           "VMRS.NE",
	arm_VMRS_CS:           "VMRS.CS",
	arm_VMRS_CC:           "VMRS.CC",
	arm_VMRS_MI:           "VMRS.MI",
	arm_VMRS_PL:           "VMRS.PL",
	arm_VMRS_VS:           "VMRS.VS",
	arm_VMRS_VC:           "VMRS.VC",
	arm_VMRS_HI:           "VMRS.HI",
	arm_VMRS_LS:           "VMRS.LS",
	arm_VMRS_GE:           "VMRS.GE",
	arm_VMRS_LT:           "VMRS.LT",
	arm_VMRS_GT:           "VMRS.GT",
	arm_VMRS_LE:           "VMRS.LE",
	arm_VMRS:              "VMRS",
	arm_VMRS_ZZ:           "VMRS.ZZ",
	arm_VMSR_EQ:           "VMSR.EQ",
	arm_VMSR_NE:           "VMSR.NE",
	arm_VMSR_CS:           "VMSR.CS",
	arm_VMSR_CC:           "VMSR.CC",
	arm_VMSR_MI:           "VMSR.MI",
	arm_VMSR_PL:           "VMSR.PL",
	arm_VMSR_VS:           "VMSR.VS",
	arm_VMSR_VC:           "VMSR.VC",
	arm_VMSR_HI:           "VMSR.HI",
	arm_VMSR_LS:           "VMSR.LS",
	arm_VMSR_GE:           "VMSR.GE",
	arm_VMSR_LT:           "VMSR.LT",
	arm_VMSR_GT:           "VMSR.GT",
	arm_VMSR_LE:           "VMSR.LE",
	arm_VMSR:              "VMSR",
	arm_VMSR_ZZ:           "VMSR.ZZ",
	arm_VMUL_EQ_F32:       "VMUL.EQ.F32",
	arm_VMUL_NE_F32:       "VMUL.NE.F32",
	arm_VMUL_CS_F32:       "VMUL.CS.F32",
	arm_VMUL_CC_F32:       "VMUL.CC.F32",
	arm_VMUL_MI_F32:       "VMUL.MI.F32",
	arm_VMUL_PL_F32:       "VMUL.PL.F32",
	arm_VMUL_VS_F32:       "VMUL.VS.F32",
	arm_VMUL_VC_F32:       "VMUL.VC.F32",
	arm_VMUL_HI_F32:       "VMUL.HI.F32",
	arm_VMUL_LS_F32:       "VMUL.LS.F32",
	arm_VMUL_GE_F32:       "VMUL.GE.F32",
	arm_VMUL_LT_F32:       "VMUL.LT.F32",
	arm_VMUL_GT_F32:       "VMUL.GT.F32",
	arm_VMUL_LE_F32:       "VMUL.LE.F32",
	arm_VMUL_F32:          "VMUL.F32",
	arm_VMUL_ZZ_F32:       "VMUL.ZZ.F32",
	arm_VMUL_EQ_F64:       "VMUL.EQ.F64",
	arm_VMUL_NE_F64:       "VMUL.NE.F64",
	arm_VMUL_CS_F64:       "VMUL.CS.F64",
	arm_VMUL_CC_F64:       "VMUL.CC.F64",
	arm_VMUL_MI_F64:       "VMUL.MI.F64",
	arm_VMUL_PL_F64:       "VMUL.PL.F64",
	arm_VMUL_VS_F64:       "VMUL.VS.F64",
	arm_VMUL_VC_F64:       "VMUL.VC.F64",
	arm_VMUL_HI_F64:       "VMUL.HI.F64",
	arm_VMUL_LS_F64:       "VMUL.LS.F64",
	arm_VMUL_GE_F64:       "VMUL.GE.F64",
	arm_VMUL_LT_F64:       "VMUL.LT.F64",
	arm_VMUL_GT_F64:       "VMUL.GT.F64",
	arm_VMUL_LE_F64:       "VMUL.LE.F64",
	arm_VMUL_F64:          "VMUL.F64",
	arm_VMUL_ZZ_F64:       "VMUL.ZZ.F64",
	arm_VNEG_EQ_F32:       "VNEG.EQ.F32",
	arm_VNEG_NE_F32:       "VNEG.NE.F32",
	arm_VNEG_CS_F32:       "VNEG.CS.F32",
	arm_VNEG_CC_F32:       "VNEG.CC.F32",
	arm_VNEG_MI_F32:       "VNEG.MI.F32",
	arm_VNEG_PL_F32:       "VNEG.PL.F32",
	arm_VNEG_VS_F32:       "VNEG.VS.F32",
	arm_VNEG_VC_F32:       "VNEG.VC.F32",
	arm_VNEG_HI_F32:       "VNEG.HI.F32",
	arm_VNEG_LS_F32:       "VNEG.LS.F32",
	arm_VNEG_GE_F32:       "VNEG.GE.F32",
	arm_VNEG_LT_F32:       "VNEG.LT.F32",
	arm_VNEG_GT_F32:       "VNEG.GT.F32",
	arm_VNEG_LE_F32:       "VNEG.LE.F32",
	arm_VNEG_F32:          "VNEG.F32",
	arm_VNEG_ZZ_F32:       "VNEG.ZZ.F32",
	arm_VNEG_EQ_F64:       "VNEG.EQ.F64",
	arm_VNEG_NE_F64:       "VNEG.NE.F64",
	arm_VNEG_CS_F64:       "VNEG.CS.F64",
	arm_VNEG_CC_F64:       "VNEG.CC.F64",
	arm_VNEG_MI_F64:       "VNEG.MI.F64",
	arm_VNEG_PL_F64:       "VNEG.PL.F64",
	arm_VNEG_VS_F64:       "VNEG.VS.F64",
	arm_VNEG_VC_F64:       "VNEG.VC.F64",
	arm_VNEG_HI_F64:       "VNEG.HI.F64",
	arm_VNEG_LS_F64:       "VNEG.LS.F64",
	arm_VNEG_GE_F64:       "VNEG.GE.F64",
	arm_VNEG_LT_F64:       "VNEG.LT.F64",
	arm_VNEG_GT_F64:       "VNEG.GT.F64",
	arm_VNEG_LE_F64:       "VNEG.LE.F64",
	arm_VNEG_F64:          "VNEG.F64",
	arm_VNEG_ZZ_F64:       "VNEG.ZZ.F64",
	arm_VNMLS_EQ_F32:      "VNMLS.EQ.F32",
	arm_VNMLS_NE_F32:      "VNMLS.NE.F32",
	arm_VNMLS_CS_F32:      "VNMLS.CS.F32",
	arm_VNMLS_CC_F32:      "VNMLS.CC.F32",
	arm_VNMLS_MI_F32:      "VNMLS.MI.F32",
	arm_VNMLS_PL_F32:      "VNMLS.PL.F32",
	arm_VNMLS_VS_F32:      "VNMLS.VS.F32",
	arm_VNMLS_VC_F32:      "VNMLS.VC.F32",
	arm_VNMLS_HI_F32:      "VNMLS.HI.F32",
	arm_VNMLS_LS_F32:      "VNMLS.LS.F32",
	arm_VNMLS_GE_F32:      "VNMLS.GE.F32",
	arm_VNMLS_LT_F32:      "VNMLS.LT.F32",
	arm_VNMLS_GT_F32:      "VNMLS.GT.F32",
	arm_VNMLS_LE_F32:      "VNMLS.LE.F32",
	arm_VNMLS_F32:         "VNMLS.F32",
	arm_VNMLS_ZZ_F32:      "VNMLS.ZZ.F32",
	arm_VNMLS_EQ_F64:      "VNMLS.EQ.F64",
	arm_VNMLS_NE_F64:      "VNMLS.NE.F64",
	arm_VNMLS_CS_F64:      "VNMLS.CS.F64",
	arm_VNMLS_CC_F64:      "VNMLS.CC.F64",
	arm_VNMLS_MI_F64:      "VNMLS.MI.F64",
	arm_VNMLS_PL_F64:      "VNMLS.PL.F64",
	arm_VNMLS_VS_F64:      "VNMLS.VS.F64",
	arm_VNMLS_VC_F64:      "VNMLS.VC.F64",
	arm_VNMLS_HI_F64:      "VNMLS.HI.F64",
	arm_VNMLS_LS_F64:      "VNMLS.LS.F64",
	arm_VNMLS_GE_F64:      "VNMLS.GE.F64",
	arm_VNMLS_LT_F64:      "VNMLS.LT.F64",
	arm_VNMLS_GT_F64:      "VNMLS.GT.F64",
	arm_VNMLS_LE_F64:      "VNMLS.LE.F64",
	arm_VNMLS_F64:         "VNMLS.F64",
	arm_VNMLS_ZZ_F64:      "VNMLS.ZZ.F64",
	arm_VNMLA_EQ_F32:      "VNMLA.EQ.F32",
	arm_VNMLA_NE_F32:      "VNMLA.NE.F32",
	arm_VNMLA_CS_F32:      "VNMLA.CS.F32",
	arm_VNMLA_CC_F32:      "VNMLA.CC.F32",
	arm_VNMLA_MI_F32:      "VNMLA.MI.F32",
	arm_VNMLA_PL_F32:      "VNMLA.PL.F32",
	arm_VNMLA_VS_F32:      "VNMLA.VS.F32",
	arm_VNMLA_VC_F32:      "VNMLA.VC.F32",
	arm_VNMLA_HI_F32:      "VNMLA.HI.F32",
	arm_VNMLA_LS_F32:      "VNMLA.LS.F32",
	arm_VNMLA_GE_F32:      "VNMLA.GE.F32",
	arm_VNMLA_LT_F32:      "VNMLA.LT.F32",
	arm_VNMLA_GT_F32:      "VNMLA.GT.F32",
	arm_VNMLA_LE_F32:      "VNMLA.LE.F32",
	arm_VNMLA_F32:         "VNMLA.F32",
	arm_VNMLA_ZZ_F32:      "VNMLA.ZZ.F32",
	arm_VNMLA_EQ_F64:      "VNMLA.EQ.F64",
	arm_VNMLA_NE_F64:      "VNMLA.NE.F64",
	arm_VNMLA_CS_F64:      "VNMLA.CS.F64",
	arm_VNMLA_CC_F64:      "VNMLA.CC.F64",
	arm_VNMLA_MI_F64:      "VNMLA.MI.F64",
	arm_VNMLA_PL_F64:      "VNMLA.PL.F64",
	arm_VNMLA_VS_F64:      "VNMLA.VS.F64",
	arm_VNMLA_VC_F64:      "VNMLA.VC.F64",
	arm_VNMLA_HI_F64:      "VNMLA.HI.F64",
	arm_VNMLA_LS_F64:      "VNMLA.LS.F64",
	arm_VNMLA_GE_F64:      "VNMLA.GE.F64",
	arm_VNMLA_LT_F64:      "VNMLA.LT.F64",
	arm_VNMLA_GT_F64:      "VNMLA.GT.F64",
	arm_VNMLA_LE_F64:      "VNMLA.LE.F64",
	arm_VNMLA_F64:         "VNMLA.F64",
	arm_VNMLA_ZZ_F64:      "VNMLA.ZZ.F64",
	arm_VNMUL_EQ_F32:      "VNMUL.EQ.F32",
	arm_VNMUL_NE_F32:      "VNMUL.NE.F32",
	arm_VNMUL_CS_F32:      "VNMUL.CS.F32",
	arm_VNMUL_CC_F32:      "VNMUL.CC.F32",
	arm_VNMUL_MI_F32:      "VNMUL.MI.F32",
	arm_VNMUL_PL_F32:      "VNMUL.PL.F32",
	arm_VNMUL_VS_F32:      "VNMUL.VS.F32",
	arm_VNMUL_VC_F32:      "VNMUL.VC.F32",
	arm_VNMUL_HI_F32:      "VNMUL.HI.F32",
	arm_VNMUL_LS_F32:      "VNMUL.LS.F32",
	arm_VNMUL_GE_F32:      "VNMUL.GE.F32",
	arm_VNMUL_LT_F32:      "VNMUL.LT.F32",
	arm_VNMUL_GT_F32:      "VNMUL.GT.F32",
	arm_VNMUL_LE_F32:      "VNMUL.LE.F32",
	arm_VNMUL_F32:         "VNMUL.F32",
	arm_VNMUL_ZZ_F32:      "VNMUL.ZZ.F32",
	arm_VNMUL_EQ_F64:      "VNMUL.EQ.F64",
	arm_VNMUL_NE_F64:      "VNMUL.NE.F64",
	arm_VNMUL_CS_F64:      "VNMUL.CS.F64",
	arm_VNMUL_CC_F64:      "VNMUL.CC.F64",
	arm_VNMUL_MI_F64:      "VNMUL.MI.F64",
	arm_VNMUL_PL_F64:      "VNMUL.PL.F64",
	arm_VNMUL_VS_F64:      "VNMUL.VS.F64",
	arm_VNMUL_VC_F64:      "VNMUL.VC.F64",
	arm_VNMUL_HI_F64:      "VNMUL.HI.F64",
	arm_VNMUL_LS_F64:      "VNMUL.LS.F64",
	arm_VNMUL_GE_F64:      "VNMUL.GE.F64",
	arm_VNMUL_LT_F64:      "VNMUL.LT.F64",
	arm_VNMUL_GT_F64:      "VNMUL.GT.F64",
	arm_VNMUL_LE_F64:      "VNMUL.LE.F64",
	arm_VNMUL_F64:         "VNMUL.F64",
	arm_VNMUL_ZZ_F64:      "VNMUL.ZZ.F64",
	arm_VSQRT_EQ_F32:      "VSQRT.EQ.F32",
	arm_VSQRT_NE_F32:      "VSQRT.NE.F32",
	arm_VSQRT_CS_F32:      "VSQRT.CS.F32",
	arm_VSQRT_CC_F32:      "VSQRT.CC.F32",
	arm_VSQRT_MI_F32:      "VSQRT.MI.F32",
	arm_VSQRT_PL_F32:      "VSQRT.PL.F32",
	arm_VSQRT_VS_F32:      "VSQRT.VS.F32",
	arm_VSQRT_VC_F32:      "VSQRT.VC.F32",
	arm_VSQRT_HI_F32:      "VSQRT.HI.F32",
	arm_VSQRT_LS_F32:      "VSQRT.LS.F32",
	arm_VSQRT_GE_F32:      "VSQRT.GE.F32",
	arm_VSQRT_LT_F32:      "VSQRT.LT.F32",
	arm_VSQRT_GT_F32:      "VSQRT.GT.F32",
	arm_VSQRT_LE_F32:      "VSQRT.LE.F32",
	arm_VSQRT_F32:         "VSQRT.F32",
	arm_VSQRT_ZZ_F32:      "VSQRT.ZZ.F32",
	arm_VSQRT_EQ_F64:      "VSQRT.EQ.F64",
	arm_VSQRT_NE_F64:      "VSQRT.NE.F64",
	arm_VSQRT_CS_F64:      "VSQRT.CS.F64",
	arm_VSQRT_CC_F64:      "VSQRT.CC.F64",
	arm_VSQRT_MI_F64:      "VSQRT.MI.F64",
	arm_VSQRT_PL_F64:      "VSQRT.PL.F64",
	arm_VSQRT_VS_F64:      "VSQRT.VS.F64",
	arm_VSQRT_VC_F64:      "VSQRT.VC.F64",
	arm_VSQRT_HI_F64:      "VSQRT.HI.F64",
	arm_VSQRT_LS_F64:      "VSQRT.LS.F64",
	arm_VSQRT_GE_F64:      "VSQRT.GE.F64",
	arm_VSQRT_LT_F64:      "VSQRT.LT.F64",
	arm_VSQRT_GT_F64:      "VSQRT.GT.F64",
	arm_VSQRT_LE_F64:      "VSQRT.LE.F64",
	arm_VSQRT_F64:         "VSQRT.F64",
	arm_VSQRT_ZZ_F64:      "VSQRT.ZZ.F64",
	arm_VSTR_EQ:           "VSTR.EQ",
	arm_VSTR_NE:           "VSTR.NE",
	arm_VSTR_CS:           "VSTR.CS",
	arm_VSTR_CC:           "VSTR.CC",
	arm_VSTR_MI:           "VSTR.MI",
	arm_VSTR_PL:           "VSTR.PL",
	arm_VSTR_VS:           "VSTR.VS",
	arm_VSTR_VC:           "VSTR.VC",
	arm_VSTR_HI:           "VSTR.HI",
	arm_VSTR_LS:           "VSTR.LS",
	arm_VSTR_GE:           "VSTR.GE",
	arm_VSTR_LT:           "VSTR.LT",
	arm_VSTR_GT:           "VSTR.GT",
	arm_VSTR_LE:           "VSTR.LE",
	arm_VSTR:              "VSTR",
	arm_VSTR_ZZ:           "VSTR.ZZ",
	arm_VSUB_EQ_F32:       "VSUB.EQ.F32",
	arm_VSUB_NE_F32:       "VSUB.NE.F32",
	arm_VSUB_CS_F32:       "VSUB.CS.F32",
	arm_VSUB_CC_F32:       "VSUB.CC.F32",
	arm_VSUB_MI_F32:       "VSUB.MI.F32",
	arm_VSUB_PL_F32:       "VSUB.PL.F32",
	arm_VSUB_VS_F32:       "VSUB.VS.F32",
	arm_VSUB_VC_F32:       "VSUB.VC.F32",
	arm_VSUB_HI_F32:       "VSUB.HI.F32",
	arm_VSUB_LS_F32:       "VSUB.LS.F32",
	arm_VSUB_GE_F32:       "VSUB.GE.F32",
	arm_VSUB_LT_F32:       "VSUB.LT.F32",
	arm_VSUB_GT_F32:       "VSUB.GT.F32",
	arm_VSUB_LE_F32:       "VSUB.LE.F32",
	arm_VSUB_F32:          "VSUB.F32",
	arm_VSUB_ZZ_F32:       "VSUB.ZZ.F32",
	arm_VSUB_EQ_F64:       "VSUB.EQ.F64",
	arm_VSUB_NE_F64:       "VSUB.NE.F64",
	arm_VSUB_CS_F64:       "VSUB.CS.F64",
	arm_VSUB_CC_F64:       "VSUB.CC.F64",
	arm_VSUB_MI_F64:       "VSUB.MI.F64",
	arm_VSUB_PL_F64:       "VSUB.PL.F64",
	arm_VSUB_VS_F64:       "VSUB.VS.F64",
	arm_VSUB_VC_F64:       "VSUB.VC.F64",
	arm_VSUB_HI_F64:       "VSUB.HI.F64",
	arm_VSUB_LS_F64:       "VSUB.LS.F64",
	arm_VSUB_GE_F64:       "VSUB.GE.F64",
	arm_VSUB_LT_F64:       "VSUB.LT.F64",
	arm_VSUB_GT_F64:       "VSUB.GT.F64",
	arm_VSUB_LE_F64:       "VSUB.LE.F64",
	arm_VSUB_F64:          "VSUB.F64",
	arm_VSUB_ZZ_F64:       "VSUB.ZZ.F64",
	arm_WFE_EQ:            "WFE.EQ",
	arm_WFE_NE:            "WFE.NE",
	arm_WFE_CS:            "WFE.CS",
	arm_WFE_CC:            "WFE.CC",
	arm_WFE_MI:            "WFE.MI",
	arm_WFE_PL:            "WFE.PL",
	arm_WFE_VS:            "WFE.VS",
	arm_WFE_VC:            "WFE.VC",
	arm_WFE_HI:            "WFE.HI",
	arm_WFE_LS:            "WFE.LS",
	arm_WFE_GE:            "WFE.GE",
	arm_WFE_LT:            "WFE.LT",
	arm_WFE_GT:            "WFE.GT",
	arm_WFE_LE:            "WFE.LE",
	arm_WFE:               "WFE",
	arm_WFE_ZZ:            "WFE.ZZ",
	arm_WFI_EQ:            "WFI.EQ",
	arm_WFI_NE:            "WFI.NE",
	arm_WFI_CS:            "WFI.CS",
	arm_WFI_CC:            "WFI.CC",
	arm_WFI_MI:            "WFI.MI",
	arm_WFI_PL:            "WFI.PL",
	arm_WFI_VS:            "WFI.VS",
	arm_WFI_VC:            "WFI.VC",
	arm_WFI_HI:            "WFI.HI",
	arm_WFI_LS:            "WFI.LS",
	arm_WFI_GE:            "WFI.GE",
	arm_WFI_LT:            "WFI.LT",
	arm_WFI_GT:            "WFI.GT",
	arm_WFI_LE:            "WFI.LE",
	arm_WFI:               "WFI",
	arm_WFI_ZZ:            "WFI.ZZ",
	arm_YIELD_EQ:          "YIELD.EQ",
	arm_YIELD_NE:          "YIELD.NE",
	arm_YIELD_CS:          "YIELD.CS",
	arm_YIELD_CC:          "YIELD.CC",
	arm_YIELD_MI:          "YIELD.MI",
	arm_YIELD_PL:          "YIELD.PL",
	arm_YIELD_VS:          "YIELD.VS",
	arm_YIELD_VC:          "YIELD.VC",
	arm_YIELD_HI:          "YIELD.HI",
	arm_YIELD_LS:          "YIELD.LS",
	arm_YIELD_GE:          "YIELD.GE",
	arm_YIELD_LT:          "YIELD.LT",
	arm_YIELD_GT:          "YIELD.GT",
	arm_YIELD_LE:          "YIELD.LE",
	arm_YIELD:             "YIELD",
	arm_YIELD_ZZ:          "YIELD.ZZ",
}

var arm_instFormats = [...]arm_instFormat{
	{0x0fe00000, 0x02a00000, 2, arm_ADC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // ADC{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|1|0|1|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00a00010, 4, arm_ADC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // ADC{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|1|0|1|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00a00000, 2, arm_ADC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // ADC{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|1|0|1|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fe00000, 0x02800000, 2, arm_ADD_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // ADD{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|1|0|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00800010, 4, arm_ADD_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // ADD{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|1|0|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00800000, 2, arm_ADD_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // ADD{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|1|0|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fef0000, 0x028d0000, 2, arm_ADD_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_SP, arm_arg_const}},                        // ADD{S}<c> <Rd>,SP,#<const> cond:4|0|0|1|0|1|0|0|S|1|1|0|1|Rd:4|imm12:12
	{0x0fef0010, 0x008d0000, 2, arm_ADD_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_SP, arm_arg_R_shift_imm}},                  // ADD{S}<c> <Rd>,SP,<Rm>{,<shift>} cond:4|0|0|0|0|1|0|0|S|1|1|0|1|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fe00000, 0x02000000, 2, arm_AND_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // AND{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|0|0|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00000010, 4, arm_AND_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // AND{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|0|0|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00000000, 2, arm_AND_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // AND{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|0|0|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fef0070, 0x01a00040, 4, arm_ASR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5_32}},                     // ASR{S}<c> <Rd>,<Rm>,#<imm5_32> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|imm5:5|1|0|0|Rm:4
	{0x0fef00f0, 0x01a00050, 4, arm_ASR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_8}},                         // ASR{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|Rm:4|0|1|0|1|Rn:4
	{0x0f000000, 0x0a000000, 4, arm_B_EQ, 0x1c04, arm_instArgs{arm_arg_label24}},                                                      // B<c> <label24> cond:4|1|0|1|0|imm24:24
	{0x0fe0007f, 0x07c0001f, 4, arm_BFC_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_imm5, arm_arg_lsb_width}},                      // BFC<c> <Rd>,#<lsb>,#<width> cond:4|0|1|1|1|1|1|0|msb:5|Rd:4|lsb:5|0|0|1|1|1|1|1
	{0x0fe00070, 0x07c00010, 2, arm_BFI_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5, arm_arg_lsb_width}},         // BFI<c> <Rd>,<Rn>,#<lsb>,#<width> cond:4|0|1|1|1|1|1|0|msb:5|Rd:4|lsb:5|0|0|1|Rn:4
	{0x0fe00000, 0x03c00000, 2, arm_BIC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // BIC{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|1|1|1|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x01c00010, 4, arm_BIC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // BIC{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|1|1|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x01c00000, 2, arm_BIC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // BIC{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|1|1|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0ff000f0, 0x01200070, 4, arm_BKPT_EQ, 0x1c04, arm_instArgs{arm_arg_imm_12at8_4at0}},                                            // BKPT<c> #<imm12+4> cond:4|0|0|0|1|0|0|1|0|imm12:12|0|1|1|1|imm4:4
	{0x0f000000, 0x0b000000, 4, arm_BL_EQ, 0x1c04, arm_instArgs{arm_arg_label24}},                                                     // BL<c> <label24> cond:4|1|0|1|1|imm24:24
	{0xfe000000, 0xfa000000, 4, arm_BLX, 0x0, arm_instArgs{arm_arg_label24H}},                                                         // BLX <label24H> 1|1|1|1|1|0|1|H|imm24:24
	{0x0ffffff0, 0x012fff30, 4, arm_BLX_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                        // BLX<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x012fff30, 3, arm_BLX_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                        // BLX<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ffffff0, 0x012fff10, 4, arm_BX_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                         // BX<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x012fff10, 3, arm_BX_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                         // BX<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ffffff0, 0x012fff20, 4, arm_BXJ_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                        // BXJ<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|1|0|Rm:4
	{0x0ff000f0, 0x012fff20, 3, arm_BXJ_EQ, 0x1c04, arm_instArgs{arm_arg_R_0}},                                                        // BXJ<c> <Rm> cond:4|0|0|0|1|0|0|1|0|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|0|0|1|0|Rm:4
	{0xffffffff, 0xf57ff01f, 4, arm_CLREX, 0x0, arm_instArgs{}},                                                                       // CLREX 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|1|(1)|(1)|(1)|(1)
	{0xfff000f0, 0xf57ff01f, 3, arm_CLREX, 0x0, arm_instArgs{}},                                                                       // CLREX 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|1|(1)|(1)|(1)|(1)
	{0x0fff0ff0, 0x016f0f10, 4, arm_CLZ_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                          // CLZ<c> <Rd>,<Rm> cond:4|0|0|0|1|0|1|1|0|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x016f0f10, 3, arm_CLZ_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                          // CLZ<c> <Rd>,<Rm> cond:4|0|0|0|1|0|1|1|0|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff0f000, 0x03700000, 4, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // CMN<c> <Rn>,#<const> cond:4|0|0|1|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff00000, 0x03700000, 3, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // CMN<c> <Rn>,#<const> cond:4|0|0|1|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff0f090, 0x01700010, 4, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // CMN<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff00090, 0x01700010, 3, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // CMN<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff0f010, 0x01700000, 4, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // CMN<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff00010, 0x01700000, 3, arm_CMN_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // CMN<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|1|1|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff0f000, 0x03500000, 4, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // CMP<c> <Rn>,#<const> cond:4|0|0|1|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff00000, 0x03500000, 3, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // CMP<c> <Rn>,#<const> cond:4|0|0|1|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff0f090, 0x01500010, 4, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // CMP<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff00090, 0x01500010, 3, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // CMP<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff0f010, 0x01500000, 4, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // CMP<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff00010, 0x01500000, 3, arm_CMP_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // CMP<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|1|0|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ffffff0, 0x0320f0f0, 4, arm_DBG_EQ, 0x1c04, arm_instArgs{arm_arg_option}},                                                     // DBG<c> #<option> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|1|1|1|1|option:4
	{0x0fff00f0, 0x0320f0f0, 3, arm_DBG_EQ, 0x1c04, arm_instArgs{arm_arg_option}},                                                     // DBG<c> #<option> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|1|1|1|1|option:4
	{0xfffffff0, 0xf57ff050, 4, arm_DMB, 0x0, arm_instArgs{arm_arg_option}},                                                           // DMB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|0|1|option:4
	{0xfff000f0, 0xf57ff050, 3, arm_DMB, 0x0, arm_instArgs{arm_arg_option}},                                                           // DMB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|0|1|option:4
	{0xfffffff0, 0xf57ff040, 4, arm_DSB, 0x0, arm_instArgs{arm_arg_option}},                                                           // DSB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|0|0|option:4
	{0xfff000f0, 0xf57ff040, 3, arm_DSB, 0x0, arm_instArgs{arm_arg_option}},                                                           // DSB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|0|0|option:4
	{0x0fe00000, 0x02200000, 2, arm_EOR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // EOR{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|0|0|1|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00200010, 4, arm_EOR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // EOR{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|0|0|1|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00200000, 2, arm_EOR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // EOR{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|0|0|1|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0xfffffff0, 0xf57ff060, 4, arm_ISB, 0x0, arm_instArgs{arm_arg_option}},                                                           // ISB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|1|0|option:4
	{0xfff000f0, 0xf57ff060, 3, arm_ISB, 0x0, arm_instArgs{arm_arg_option}},                                                           // ISB #<option> 1|1|1|1|0|1|0|1|0|1|1|1|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|1|1|0|option:4
	{0x0fd00000, 0x08900000, 2, arm_LDM_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                                 // LDM<c> <Rn>{!},<registers> cond:4|1|0|0|0|1|0|W|1|Rn:4|register_list:16
	{0x0fd00000, 0x08100000, 4, arm_LDMDA_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // LDMDA<c> <Rn>{!},<registers> cond:4|1|0|0|0|0|0|W|1|Rn:4|register_list:16
	{0x0fd00000, 0x09100000, 4, arm_LDMDB_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // LDMDB<c> <Rn>{!},<registers> cond:4|1|0|0|1|0|0|W|1|Rn:4|register_list:16
	{0x0fd00000, 0x09900000, 4, arm_LDMIB_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // LDMIB<c> <Rn>{!},<registers> cond:4|1|0|0|1|1|0|W|1|Rn:4|register_list:16
	{0x0f7f0000, 0x051f0000, 4, arm_LDR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_label_pm_12}},                                  // LDR<c> <Rt>,<label+/-12> cond:4|0|1|0|(1)|U|0|(0)|1|1|1|1|1|Rt:4|imm12:12
	{0x0e5f0000, 0x051f0000, 3, arm_LDR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_label_pm_12}},                                  // LDR<c> <Rt>,<label+/-12> cond:4|0|1|0|(1)|U|0|(0)|1|1|1|1|1|Rt:4|imm12:12
	{0x0e500010, 0x06100000, 2, arm_LDR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_W}},                       // LDR<c> <Rt>,[<Rn>,+/-<Rm>{, <shift>}]{!} cond:4|0|1|1|P|U|0|W|1|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500000, 0x04100000, 2, arm_LDR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_W}},                             // LDR<c> <Rt>,[<Rn>{,#+/-<imm12>}]{!} cond:4|0|1|0|P|U|0|W|1|Rn:4|Rt:4|imm12:12
	{0x0f7f0000, 0x055f0000, 4, arm_LDRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_label_pm_12}},                                 // LDRB<c> <Rt>,<label+/-12> cond:4|0|1|0|(1)|U|1|(0)|1|1|1|1|1|Rt:4|imm12:12
	{0x0e5f0000, 0x055f0000, 3, arm_LDRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_label_pm_12}},                                 // LDRB<c> <Rt>,<label+/-12> cond:4|0|1|0|(1)|U|1|(0)|1|1|1|1|1|Rt:4|imm12:12
	{0x0e500010, 0x06500000, 2, arm_LDRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_W}},                      // LDRB<c> <Rt>,[<Rn>,+/-<Rm>{, <shift>}]{!} cond:4|0|1|1|P|U|1|W|1|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500000, 0x04500000, 2, arm_LDRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_W}},                            // LDRB<c> <Rt>,[<Rn>{,#+/-<imm12>}]{!} cond:4|0|1|0|P|U|1|W|1|Rn:4|Rt:4|imm12:12
	{0x0f700000, 0x04700000, 4, arm_LDRBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_postindex}},                   // LDRBT<c> <Rt>,[<Rn>],#+/-<imm12> cond:4|0|1|0|0|U|1|1|1|Rn:4|Rt:4|imm12:12
	{0x0f700010, 0x06700000, 4, arm_LDRBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_postindex}},             // LDRBT<c> <Rt>,[<Rn>],+/-<Rm>{, <shift>} cond:4|0|1|1|0|U|1|1|1|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500ff0, 0x000000d0, 4, arm_LDRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_R_W}},                // LDRD<c> <Rt1>,<Rt2>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|0|Rn:4|Rt:4|(0)|(0)|(0)|(0)|1|1|0|1|Rm:4
	{0x0e5000f0, 0x000000d0, 3, arm_LDRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_R_W}},                // LDRD<c> <Rt1>,<Rt2>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|0|Rn:4|Rt:4|(0)|(0)|(0)|(0)|1|1|0|1|Rm:4
	{0x0e5000f0, 0x004000d0, 2, arm_LDRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_imm8_W}},             // LDRD<c> <Rt1>,<Rt2>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|0|Rn:4|Rt:4|imm4H:4|1|1|0|1|imm4L:4
	{0x0ff00fff, 0x01900f9f, 4, arm_LDREX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                      // LDREX<c> <Rt>,[<Rn>] cond:4|0|0|0|1|1|0|0|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff000f0, 0x01900f9f, 3, arm_LDREX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                      // LDREX<c> <Rt>,[<Rn>] cond:4|0|0|0|1|1|0|0|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff00fff, 0x01d00f9f, 4, arm_LDREXB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                     // LDREXB<c> <Rt>, [<Rn>] cond:4|0|0|0|1|1|1|0|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff000f0, 0x01d00f9f, 3, arm_LDREXB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                     // LDREXB<c> <Rt>, [<Rn>] cond:4|0|0|0|1|1|1|0|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff00fff, 0x01b00f9f, 4, arm_LDREXD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R}},                     // LDREXD<c> <Rt1>,<Rt2>,[<Rn>] cond:4|0|0|0|1|1|0|1|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff000f0, 0x01b00f9f, 3, arm_LDREXD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R}},                     // LDREXD<c> <Rt1>,<Rt2>,[<Rn>] cond:4|0|0|0|1|1|0|1|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff00fff, 0x01f00f9f, 4, arm_LDREXH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                     // LDREXH<c> <Rt>, [<Rn>] cond:4|0|0|0|1|1|1|1|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0ff000f0, 0x01f00f9f, 3, arm_LDREXH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R}},                                     // LDREXH<c> <Rt>, [<Rn>] cond:4|0|0|0|1|1|1|1|1|Rn:4|Rt:4|(1)|(1)|(1)|(1)|1|0|0|1|(1)|(1)|(1)|(1)
	{0x0e500ff0, 0x001000b0, 2, arm_LDRH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_W}},                                // LDRH<c> <Rt>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|1|Rn:4|Rt:4|0|0|0|0|1|0|1|1|Rm:4
	{0x0e5000f0, 0x005000b0, 2, arm_LDRH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_W}},                             // LDRH<c> <Rt>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|1|Rn:4|Rt:4|imm4H:4|1|0|1|1|imm4L:4
	{0x0f7000f0, 0x007000b0, 4, arm_LDRHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_postindex}},                    // LDRHT<c> <Rt>, [<Rn>] {,#+/-<imm8>} cond:4|0|0|0|0|U|1|1|1|Rn:4|Rt:4|imm4H:4|1|0|1|1|imm4L:4
	{0x0f700ff0, 0x003000b0, 4, arm_LDRHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_postindex}},                       // LDRHT<c> <Rt>, [<Rn>], +/-<Rm> cond:4|0|0|0|0|U|0|1|1|Rn:4|Rt:4|0|0|0|0|1|0|1|1|Rm:4
	{0x0e500ff0, 0x001000d0, 2, arm_LDRSB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_W}},                               // LDRSB<c> <Rt>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|1|Rn:4|Rt:4|0|0|0|0|1|1|0|1|Rm:4
	{0x0e5000f0, 0x005000d0, 2, arm_LDRSB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_W}},                            // LDRSB<c> <Rt>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|1|Rn:4|Rt:4|imm4H:4|1|1|0|1|imm4L:4
	{0x0f7000f0, 0x007000d0, 4, arm_LDRSBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_postindex}},                   // LDRSBT<c> <Rt>, [<Rn>] {,#+/-<imm8>} cond:4|0|0|0|0|U|1|1|1|Rn:4|Rt:4|imm4H:4|1|1|0|1|imm4L:4
	{0x0f700ff0, 0x003000d0, 4, arm_LDRSBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_postindex}},                      // LDRSBT<c> <Rt>, [<Rn>], +/-<Rm> cond:4|0|0|0|0|U|0|1|1|Rn:4|Rt:4|0|0|0|0|1|1|0|1|Rm:4
	{0x0e500ff0, 0x001000f0, 2, arm_LDRSH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_W}},                               // LDRSH<c> <Rt>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|1|Rn:4|Rt:4|0|0|0|0|1|1|1|1|Rm:4
	{0x0e5000f0, 0x005000f0, 2, arm_LDRSH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_W}},                            // LDRSH<c> <Rt>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|1|Rn:4|Rt:4|imm4H:4|1|1|1|1|imm4L:4
	{0x0f7000f0, 0x007000f0, 4, arm_LDRSHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_postindex}},                   // LDRSHT<c> <Rt>, [<Rn>] {,#+/-<imm8>} cond:4|0|0|0|0|U|1|1|1|Rn:4|Rt:4|imm4H:4|1|1|1|1|imm4L:4
	{0x0f700ff0, 0x003000f0, 4, arm_LDRSHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_postindex}},                      // LDRSHT<c> <Rt>, [<Rn>], +/-<Rm> cond:4|0|0|0|0|U|0|1|1|Rn:4|Rt:4|0|0|0|0|1|1|1|1|Rm:4
	{0x0f700000, 0x04300000, 4, arm_LDRT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_postindex}},                    // LDRT<c> <Rt>, [<Rn>] {,#+/-<imm12>} cond:4|0|1|0|0|U|0|1|1|Rn:4|Rt:4|imm12:12
	{0x0f700010, 0x06300000, 4, arm_LDRT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_postindex}},              // LDRT<c> <Rt>,[<Rn>],+/-<Rm>{, <shift>} cond:4|0|1|1|0|U|0|1|1|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0fef0070, 0x01a00000, 2, arm_LSL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5_nz}},                     // LSL{S}<c> <Rd>,<Rm>,#<imm5_nz> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|imm5:5|0|0|0|Rm:4
	{0x0fef00f0, 0x01a00010, 4, arm_LSL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_8}},                         // LSL{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|Rm:4|0|0|0|1|Rn:4
	{0x0fef0070, 0x01a00020, 4, arm_LSR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5_32}},                     // LSR{S}<c> <Rd>,<Rm>,#<imm5_32> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|imm5:5|0|1|0|Rm:4
	{0x0fef00f0, 0x01a00030, 4, arm_LSR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_8}},                         // LSR{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|Rm:4|0|0|1|1|Rn:4
	{0x0fe000f0, 0x00200090, 4, arm_MLA_EQ, 0x14011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},           // MLA{S}<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|0|0|0|0|0|1|S|Rd:4|Ra:4|Rm:4|1|0|0|1|Rn:4
	{0x0ff000f0, 0x00600090, 4, arm_MLS_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},               // MLS<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|0|0|0|0|1|1|0|Rd:4|Ra:4|Rm:4|1|0|0|1|Rn:4
	{0x0ff00000, 0x03400000, 4, arm_MOVT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_imm_4at16_12at0}},                             // MOVT<c> <Rd>,#<imm12+4> cond:4|0|0|1|1|0|1|0|0|imm4:4|Rd:4|imm12:12
	{0x0ff00000, 0x03000000, 4, arm_MOVW_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_imm_4at16_12at0}},                             // MOVW<c> <Rd>,#<imm12+4> cond:4|0|0|1|1|0|0|0|0|imm4:4|Rd:4|imm12:12
	{0x0fef0000, 0x03a00000, 2, arm_MOV_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_const}},                                    // MOV{S}<c> <Rd>,#<const> cond:4|0|0|1|1|1|0|1|S|0|0|0|0|Rd:4|imm12:12
	{0x0fef0ff0, 0x01a00000, 2, arm_MOV_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                      // MOV{S}<c> <Rd>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|0|0|0|0|0|0|0|0|Rm:4
	{0x0fff0fff, 0x010f0000, 4, arm_MRS_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_APSR}},                                         // MRS<c> <Rd>,APSR cond:4|0|0|0|1|0|0|0|0|(1)|(1)|(1)|(1)|Rd:4|(0)|(0)|(0)|(0)|0|0|0|0|(0)|(0)|(0)|(0)
	{0x0ff000f0, 0x010f0000, 3, arm_MRS_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_APSR}},                                         // MRS<c> <Rd>,APSR cond:4|0|0|0|1|0|0|0|0|(1)|(1)|(1)|(1)|Rd:4|(0)|(0)|(0)|(0)|0|0|0|0|(0)|(0)|(0)|(0)
	{0x0fe0f0f0, 0x00000090, 4, arm_MUL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                         // MUL{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|0|0|0|0|S|Rd:4|(0)|(0)|(0)|(0)|Rm:4|1|0|0|1|Rn:4
	{0x0fe000f0, 0x00000090, 3, arm_MUL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                         // MUL{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|0|0|0|0|S|Rd:4|(0)|(0)|(0)|(0)|Rm:4|1|0|0|1|Rn:4
	{0x0fef0000, 0x03e00000, 2, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_const}},                                    // MVN{S}<c> <Rd>,#<const> cond:4|0|0|1|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|imm12:12
	{0x0fe00000, 0x03e00000, 1, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_const}},                                    // MVN{S}<c> <Rd>,#<const> cond:4|0|0|1|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|imm12:12
	{0x0fef0090, 0x01e00010, 4, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_shift_R}},                                // MVN{S}<c> <Rd>,<Rm>,<type> <Rs> cond:4|0|0|0|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00090, 0x01e00010, 3, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_shift_R}},                                // MVN{S}<c> <Rd>,<Rm>,<type> <Rs> cond:4|0|0|0|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fef0010, 0x01e00000, 2, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_shift_imm}},                              // MVN{S}<c> <Rd>,<Rm>{,<shift>} cond:4|0|0|0|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fe00010, 0x01e00000, 1, arm_MVN_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_shift_imm}},                              // MVN{S}<c> <Rd>,<Rm>{,<shift>} cond:4|0|0|0|1|1|1|1|S|(0)|(0)|(0)|(0)|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fffffff, 0x0320f000, 4, arm_NOP_EQ, 0x1c04, arm_instArgs{}},                                                                   // NOP<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|0|0
	{0x0fff00ff, 0x0320f000, 3, arm_NOP_EQ, 0x1c04, arm_instArgs{}},                                                                   // NOP<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|0|0
	{0x0fe00000, 0x03800000, 2, arm_ORR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // ORR{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|1|1|0|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x01800010, 4, arm_ORR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // ORR{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|1|0|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x01800000, 2, arm_ORR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // ORR{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|1|0|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0ff00030, 0x06800010, 4, arm_PKHBT_EQ, 0x6011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},               // PKH<BT,TB><c> <Rd>,<Rn>,<Rm>{,LSL #<imm5>} cond:4|0|1|1|0|1|0|0|0|Rn:4|Rd:4|imm5:5|tb|0|1|Rm:4
	{0xff7ff000, 0xf55ff000, 4, arm_PLD, 0x0, arm_instArgs{arm_arg_label_pm_12}},                                                      // PLD <label+/-12> 1|1|1|1|0|1|0|1|U|(1)|0|1|1|1|1|1|(1)|(1)|(1)|(1)|imm12:12
	{0xff3f0000, 0xf55ff000, 3, arm_PLD, 0x0, arm_instArgs{arm_arg_label_pm_12}},                                                      // PLD <label+/-12> 1|1|1|1|0|1|0|1|U|(1)|0|1|1|1|1|1|(1)|(1)|(1)|(1)|imm12:12
	{0xff30f000, 0xf510f000, 2, arm_PLD_W, 0x1601, arm_instArgs{arm_arg_mem_R_pm_imm12_offset}},                                       // PLD{W} [<Rn>,#+/-<imm12>] 1|1|1|1|0|1|0|1|U|R|0|1|Rn:4|(1)|(1)|(1)|(1)|imm12:12
	{0xff300000, 0xf510f000, 1, arm_PLD_W, 0x1601, arm_instArgs{arm_arg_mem_R_pm_imm12_offset}},                                       // PLD{W} [<Rn>,#+/-<imm12>] 1|1|1|1|0|1|0|1|U|R|0|1|Rn:4|(1)|(1)|(1)|(1)|imm12:12
	{0xff30f010, 0xf710f000, 4, arm_PLD_W, 0x1601, arm_instArgs{arm_arg_mem_R_pm_R_shift_imm_offset}},                                 // PLD{W} [<Rn>,+/-<Rm>{, <shift>}] 1|1|1|1|0|1|1|1|U|R|0|1|Rn:4|(1)|(1)|(1)|(1)|imm5:5|type:2|0|Rm:4
	{0xff300010, 0xf710f000, 3, arm_PLD_W, 0x1601, arm_instArgs{arm_arg_mem_R_pm_R_shift_imm_offset}},                                 // PLD{W} [<Rn>,+/-<Rm>{, <shift>}] 1|1|1|1|0|1|1|1|U|R|0|1|Rn:4|(1)|(1)|(1)|(1)|imm5:5|type:2|0|Rm:4
	{0xff70f000, 0xf450f000, 4, arm_PLI, 0x0, arm_instArgs{arm_arg_mem_R_pm_imm12_offset}},                                            // PLI [<Rn>,#+/-<imm12>] 1|1|1|1|0|1|0|0|U|1|0|1|Rn:4|(1)|(1)|(1)|(1)|imm12:12
	{0xff700000, 0xf450f000, 3, arm_PLI, 0x0, arm_instArgs{arm_arg_mem_R_pm_imm12_offset}},                                            // PLI [<Rn>,#+/-<imm12>] 1|1|1|1|0|1|0|0|U|1|0|1|Rn:4|(1)|(1)|(1)|(1)|imm12:12
	{0xff70f010, 0xf650f000, 4, arm_PLI, 0x0, arm_instArgs{arm_arg_mem_R_pm_R_shift_imm_offset}},                                      // PLI [<Rn>,+/-<Rm>{, <shift>}] 1|1|1|1|0|1|1|0|U|1|0|1|Rn:4|(1)|(1)|(1)|(1)|imm5:5|type:2|0|Rm:4
	{0xff700010, 0xf650f000, 3, arm_PLI, 0x0, arm_instArgs{arm_arg_mem_R_pm_R_shift_imm_offset}},                                      // PLI [<Rn>,+/-<Rm>{, <shift>}] 1|1|1|1|0|1|1|0|U|1|0|1|Rn:4|(1)|(1)|(1)|(1)|imm5:5|type:2|0|Rm:4
	{0x0fff0000, 0x08bd0000, 4, arm_POP_EQ, 0x1c04, arm_instArgs{arm_arg_registers2}},                                                 // POP<c> <registers2> cond:4|1|0|0|0|1|0|1|1|1|1|0|1|register_list:16
	{0x0fff0fff, 0x049d0004, 4, arm_POP_EQ, 0x1c04, arm_instArgs{arm_arg_registers1}},                                                 // POP<c> <registers1> cond:4|0|1|0|0|1|0|0|1|1|1|0|1|Rt:4|0|0|0|0|0|0|0|0|0|1|0|0
	{0x0fff0000, 0x092d0000, 4, arm_PUSH_EQ, 0x1c04, arm_instArgs{arm_arg_registers2}},                                                // PUSH<c> <registers2> cond:4|1|0|0|1|0|0|1|0|1|1|0|1|register_list:16
	{0x0fff0fff, 0x052d0004, 4, arm_PUSH_EQ, 0x1c04, arm_instArgs{arm_arg_registers1}},                                                // PUSH<c> <registers1> cond:4|0|1|0|1|0|0|1|0|1|1|0|1|Rt:4|0|0|0|0|0|0|0|0|0|1|0|0
	{0x0ff00ff0, 0x06200f10, 4, arm_QADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // QADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06200f10, 3, arm_QADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // QADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06200f90, 4, arm_QADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // QADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06200f90, 3, arm_QADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // QADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x01000050, 4, arm_QADD_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                           // QADD<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|0|0|0|Rn:4|Rd:4|(0)|(0)|(0)|(0)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x01000050, 3, arm_QADD_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                           // QADD<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|0|0|0|Rn:4|Rd:4|(0)|(0)|(0)|(0)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06200f30, 4, arm_QASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // QASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06200f30, 3, arm_QASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // QASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff00ff0, 0x01400050, 4, arm_QDADD_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                          // QDADD<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|1|0|0|Rn:4|Rd:4|(0)|(0)|(0)|(0)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x01400050, 3, arm_QDADD_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                          // QDADD<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|1|0|0|Rn:4|Rd:4|(0)|(0)|(0)|(0)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x01600050, 4, arm_QDSUB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                          // QDSUB<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|1|1|0|Rn:4|Rd:4|0|0|0|0|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06200f50, 4, arm_QSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // QSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06200f50, 3, arm_QSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // QSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06200f70, 4, arm_QSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // QSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06200f70, 3, arm_QSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // QSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06200ff0, 4, arm_QSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // QSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06200ff0, 3, arm_QSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // QSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff00ff0, 0x01200050, 4, arm_QSUB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_16}},                           // QSUB<c> <Rd>,<Rm>,<Rn> cond:4|0|0|0|1|0|0|1|0|Rn:4|Rd:4|0|0|0|0|0|1|0|1|Rm:4
	{0x0fff0ff0, 0x06ff0f30, 4, arm_RBIT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                         // RBIT<c> <Rd>,<Rm> cond:4|0|1|1|0|1|1|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06ff0f30, 3, arm_RBIT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                         // RBIT<c> <Rd>,<Rm> cond:4|0|1|1|0|1|1|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0fff0ff0, 0x06bf0fb0, 4, arm_REV16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                        // REV16<c> <Rd>,<Rm> cond:4|0|1|1|0|1|0|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0x0ff000f0, 0x06bf0fb0, 3, arm_REV16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                        // REV16<c> <Rd>,<Rm> cond:4|0|1|1|0|1|0|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0x0fff0ff0, 0x06bf0f30, 4, arm_REV_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                          // REV<c> <Rd>,<Rm> cond:4|0|1|1|0|1|0|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06bf0f30, 3, arm_REV_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                          // REV<c> <Rd>,<Rm> cond:4|0|1|1|0|1|0|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0fff0ff0, 0x06ff0fb0, 4, arm_REVSH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                        // REVSH<c> <Rd>,<Rm> cond:4|0|1|1|0|1|1|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0x0ff000f0, 0x06ff0fb0, 3, arm_REVSH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                        // REVSH<c> <Rd>,<Rm> cond:4|0|1|1|0|1|1|1|1|(1)|(1)|(1)|(1)|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0x0fef0070, 0x01a00060, 2, arm_ROR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5}},                        // ROR{S}<c> <Rd>,<Rm>,#<imm5> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|imm5:5|1|1|0|Rm:4
	{0x0fef00f0, 0x01a00070, 4, arm_ROR_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_R_8}},                         // ROR{S}<c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|Rm:4|0|1|1|1|Rn:4
	{0x0fef0ff0, 0x01a00060, 4, arm_RRX_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0}},                                      // RRX{S}<c> <Rd>,<Rm> cond:4|0|0|0|1|1|0|1|S|0|0|0|0|Rd:4|0|0|0|0|0|1|1|0|Rm:4
	{0x0fe00000, 0x02600000, 2, arm_RSB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // RSB{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|0|1|1|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00600010, 4, arm_RSB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // RSB{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|0|1|1|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00600000, 2, arm_RSB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // RSB{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|0|1|1|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fe00000, 0x02e00000, 2, arm_RSC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // RSC{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|1|1|1|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00e00010, 4, arm_RSC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // RSC{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|1|1|1|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00e00000, 2, arm_RSC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // RSC{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|1|1|1|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0ff00ff0, 0x06100f10, 4, arm_SADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06100f10, 3, arm_SADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06100f90, 4, arm_SADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06100f90, 3, arm_SADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x06100f30, 4, arm_SASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // SASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06100f30, 3, arm_SASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // SASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0fe00000, 0x02c00000, 2, arm_SBC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // SBC{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|1|1|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00c00010, 4, arm_SBC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // SBC{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|1|1|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00c00000, 2, arm_SBC_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // SBC{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|1|1|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fe00070, 0x07a00050, 4, arm_SBFX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5, arm_arg_widthm1}},          // SBFX<c> <Rd>,<Rn>,#<lsb>,#<widthm1> cond:4|0|1|1|1|1|0|1|widthm1:5|Rd:4|lsb:5|1|0|1|Rn:4
	{0x0ff00ff0, 0x06800fb0, 4, arm_SEL_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                            // SEL<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|1|0|0|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0x0ff000f0, 0x06800fb0, 3, arm_SEL_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                            // SEL<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|1|0|0|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|1|1|Rm:4
	{0xfffffdff, 0xf1010000, 4, arm_SETEND, 0x0, arm_instArgs{arm_arg_endian}},                                                        // SETEND <endian_specifier> 1|1|1|1|0|0|0|1|0|0|0|0|0|0|0|1|0|0|0|0|0|0|E|(0)|(0)|(0)|(0)|(0)|(0)|(0)|(0)|(0)
	{0xfffffc00, 0xf1010000, 3, arm_SETEND, 0x0, arm_instArgs{arm_arg_endian}},                                                        // SETEND <endian_specifier> 1|1|1|1|0|0|0|1|0|0|0|0|0|0|0|1|0|0|0|0|0|0|E|(0)|(0)|(0)|(0)|(0)|(0)|(0)|(0)|(0)
	{0x0fffffff, 0x0320f004, 4, arm_SEV_EQ, 0x1c04, arm_instArgs{}},                                                                   // SEV<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|1|0|0
	{0x0fff00ff, 0x0320f004, 3, arm_SEV_EQ, 0x1c04, arm_instArgs{}},                                                                   // SEV<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|1|0|0
	{0x0ff00ff0, 0x06300f10, 4, arm_SHADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // SHADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06300f10, 3, arm_SHADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // SHADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06300f90, 4, arm_SHADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SHADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06300f90, 3, arm_SHADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SHADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x06300f30, 4, arm_SHASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SHASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06300f30, 3, arm_SHASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SHASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff00ff0, 0x06300f50, 4, arm_SHSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SHSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06300f50, 3, arm_SHSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SHSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06300f70, 4, arm_SHSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // SHSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06300f70, 3, arm_SHSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // SHSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06300ff0, 4, arm_SHSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SHSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06300ff0, 3, arm_SHSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SHSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff00090, 0x01000080, 4, arm_SMLABB_EQ, 0x50106011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},     // SMLA<x><y><c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|0|0|1|0|0|0|0|Rd:4|Ra:4|Rm:4|1|M|N|0|Rn:4
	{0x0ff000d0, 0x07000010, 2, arm_SMLAD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},          // SMLAD{X}<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|1|1|1|0|0|0|0|Rd:4|Ra:4|Rm:4|0|0|M|1|Rn:4
	{0x0ff00090, 0x01400080, 4, arm_SMLALBB_EQ, 0x50106011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},    // SMLAL<x><y><c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|1|0|1|0|0|RdHi:4|RdLo:4|Rm:4|1|M|N|0|Rn:4
	{0x0ff000d0, 0x07400010, 4, arm_SMLALD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // SMLALD{X}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|1|1|1|0|1|0|0|RdHi:4|RdLo:4|Rm:4|0|0|M|1|Rn:4
	{0x0fe000f0, 0x00e00090, 4, arm_SMLAL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // SMLAL{S}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|0|1|1|1|S|RdHi:4|RdLo:4|Rm:4|1|0|0|1|Rn:4
	{0x0ff000b0, 0x01200080, 4, arm_SMLAWB_EQ, 0x6011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},         // SMLAW<y><c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|0|0|1|0|0|1|0|Rd:4|Ra:4|Rm:4|1|M|0|0|Rn:4
	{0x0ff000d0, 0x07000050, 2, arm_SMLSD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},          // SMLSD{X}<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|1|1|1|0|0|0|0|Rd:4|Ra:4|Rm:4|0|1|M|1|Rn:4
	{0x0ff000d0, 0x07400050, 4, arm_SMLSLD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // SMLSLD{X}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|1|1|1|0|1|0|0|RdHi:4|RdLo:4|Rm:4|0|1|M|1|Rn:4
	{0x0ff000d0, 0x07500010, 2, arm_SMMLA_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},          // SMMLA{R}<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|1|1|1|0|1|0|1|Rd:4|Ra:4|Rm:4|0|0|R|1|Rn:4
	{0x0ff000d0, 0x075000d0, 4, arm_SMMLS_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},          // SMMLS{R}<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|1|1|1|0|1|0|1|Rd:4|Ra:4|Rm:4|1|1|R|1|Rn:4
	{0x0ff0f0d0, 0x0750f010, 4, arm_SMMUL_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                        // SMMUL{R}<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|1|0|1|0|1|Rd:4|1|1|1|1|Rm:4|0|0|R|1|Rn:4
	{0x0ff0f0d0, 0x0700f010, 4, arm_SMUAD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                        // SMUAD{X}<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|1|0|0|0|0|Rd:4|1|1|1|1|Rm:4|0|0|M|1|Rn:4
	{0x0ff0f090, 0x01600080, 4, arm_SMULBB_EQ, 0x50106011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                   // SMUL<x><y><c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|0|1|1|0|Rd:4|0|0|0|0|Rm:4|1|M|N|0|Rn:4
	{0x0fe000f0, 0x00c00090, 4, arm_SMULL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // SMULL{S}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|0|1|1|0|S|RdHi:4|RdLo:4|Rm:4|1|0|0|1|Rn:4
	{0x0ff0f0b0, 0x012000a0, 4, arm_SMULWB_EQ, 0x6011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                       // SMULW<y><c> <Rd>,<Rn>,<Rm> cond:4|0|0|0|1|0|0|1|0|Rd:4|0|0|0|0|Rm:4|1|M|1|0|Rn:4
	{0x0ff0f0d0, 0x0700f050, 4, arm_SMUSD_EQ, 0x5011c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                        // SMUSD{X}<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|1|0|0|0|0|Rd:4|1|1|1|1|Rm:4|0|1|M|1|Rn:4
	{0x0ff00ff0, 0x06a00f30, 4, arm_SSAT16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm4m1, arm_arg_R_0}},                    // SSAT16<c> <Rd>,#<sat_imm4m1>,<Rn> cond:4|0|1|1|0|1|0|1|0|sat_imm:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rn:4
	{0x0ff000f0, 0x06a00f30, 3, arm_SSAT16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm4m1, arm_arg_R_0}},                    // SSAT16<c> <Rd>,#<sat_imm4m1>,<Rn> cond:4|0|1|1|0|1|0|1|0|sat_imm:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rn:4
	{0x0fe00030, 0x06a00010, 4, arm_SSAT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm5m1, arm_arg_R_shift_imm}},              // SSAT<c> <Rd>,#<sat_imm5m1>,<Rn>{,<shift>} cond:4|0|1|1|0|1|0|1|sat_imm:5|Rd:4|imm5:5|sh|0|1|Rn:4
	{0x0ff00ff0, 0x06100f50, 4, arm_SSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // SSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06100f50, 3, arm_SSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // SSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06100f70, 4, arm_SSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06100f70, 3, arm_SSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // SSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06100ff0, 4, arm_SSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06100ff0, 3, arm_SSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // SSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|0|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0fd00000, 0x08800000, 4, arm_STM_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                                 // STM<c> <Rn>{!},<registers> cond:4|1|0|0|0|1|0|W|0|Rn:4|register_list:16
	{0x0fd00000, 0x08000000, 4, arm_STMDA_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // STMDA<c> <Rn>{!},<registers> cond:4|1|0|0|0|0|0|W|0|Rn:4|register_list:16
	{0x0fd00000, 0x09000000, 2, arm_STMDB_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // STMDB<c> <Rn>{!},<registers> cond:4|1|0|0|1|0|0|W|0|Rn:4|register_list:16
	{0x0fd00000, 0x09800000, 4, arm_STMIB_EQ, 0x1c04, arm_instArgs{arm_arg_R_16_WB, arm_arg_registers}},                               // STMIB<c> <Rn>{!},<registers> cond:4|1|0|0|1|1|0|W|0|Rn:4|register_list:16
	{0x0e500018, 0x06000000, 2, arm_STR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_W}},                       // STR<c> <Rt>,[<Rn>,+/-<Rm>{, <shift>}]{!} cond:4|0|1|1|P|U|0|W|0|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500000, 0x04000000, 2, arm_STR_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_W}},                             // STR<c> <Rt>,[<Rn>{,#+/-<imm12>}]{!} cond:4|0|1|0|P|U|0|W|0|Rn:4|Rt:4|imm12:12
	{0x0e500010, 0x06400000, 2, arm_STRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_W}},                      // STRB<c> <Rt>,[<Rn>,+/-<Rm>{, <shift>}]{!} cond:4|0|1|1|P|U|1|W|0|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500000, 0x04400000, 2, arm_STRB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_W}},                            // STRB<c> <Rt>,[<Rn>{,#+/-<imm12>}]{!} cond:4|0|1|0|P|U|1|W|0|Rn:4|Rt:4|imm12:12
	{0x0f700000, 0x04600000, 4, arm_STRBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_postindex}},                   // STRBT<c> <Rt>,[<Rn>],#+/-<imm12> cond:4|0|1|0|0|U|1|1|0|Rn:4|Rt:4|imm12:12
	{0x0f700010, 0x06600000, 4, arm_STRBT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_postindex}},             // STRBT<c> <Rt>,[<Rn>],+/-<Rm>{, <shift>} cond:4|0|1|1|0|U|1|1|0|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0e500ff0, 0x000000f0, 4, arm_STRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_R_W}},                // STRD<c> <Rt1>,<Rt2>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|0|Rn:4|Rt:4|(0)|(0)|(0)|(0)|1|1|1|1|Rm:4
	{0x0e5000f0, 0x000000f0, 3, arm_STRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_R_W}},                // STRD<c> <Rt1>,<Rt2>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|0|Rn:4|Rt:4|(0)|(0)|(0)|(0)|1|1|1|1|Rm:4
	{0x0e5000f0, 0x004000f0, 4, arm_STRD_EQ, 0x1c04, arm_instArgs{arm_arg_R1_12, arm_arg_R2_12, arm_arg_mem_R_pm_imm8_W}},             // STRD<c> <Rt1>,<Rt2>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|0|Rn:4|Rt:4|imm4H:4|1|1|1|1|imm4L:4
	{0x0ff00ff0, 0x01800f90, 4, arm_STREX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_mem_R}},                         // STREX<c> <Rd>,<Rt>,[<Rn>] cond:4|0|0|0|1|1|0|0|0|Rn:4|Rd:4|1|1|1|1|1|0|0|1|Rt:4
	{0x0ff00ff0, 0x01c00f90, 4, arm_STREXB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_mem_R}},                        // STREXB<c> <Rd>,<Rt>,[<Rn>] cond:4|0|0|0|1|1|1|0|0|Rn:4|Rd:4|1|1|1|1|1|0|0|1|Rt:4
	{0x0ff00ff0, 0x01a00f90, 4, arm_STREXD_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R1_0, arm_arg_R2_0, arm_arg_mem_R}},         // STREXD<c> <Rd>,<Rt1>,<Rt2>,[<Rn>] cond:4|0|0|0|1|1|0|1|0|Rn:4|Rd:4|1|1|1|1|1|0|0|1|Rt:4
	{0x0ff00ff0, 0x01e00f90, 4, arm_STREXH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_mem_R}},                        // STREXH<c> <Rd>,<Rt>,[<Rn>] cond:4|0|0|0|1|1|1|1|0|Rn:4|Rd:4|1|1|1|1|1|0|0|1|Rt:4
	{0x0e500ff0, 0x000000b0, 2, arm_STRH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_W}},                                // STRH<c> <Rt>,[<Rn>,+/-<Rm>]{!} cond:4|0|0|0|P|U|0|W|0|Rn:4|Rt:4|0|0|0|0|1|0|1|1|Rm:4
	{0x0e5000f0, 0x004000b0, 2, arm_STRH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_W}},                             // STRH<c> <Rt>,[<Rn>{,#+/-<imm8>}]{!} cond:4|0|0|0|P|U|1|W|0|Rn:4|Rt:4|imm4H:4|1|0|1|1|imm4L:4
	{0x0f7000f0, 0x006000b0, 4, arm_STRHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm8_postindex}},                    // STRHT<c> <Rt>, [<Rn>] {,#+/-<imm8>} cond:4|0|0|0|0|U|1|1|0|Rn:4|Rt:4|imm4H:4|1|0|1|1|imm4L:4
	{0x0f700ff0, 0x002000b0, 4, arm_STRHT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_postindex}},                       // STRHT<c> <Rt>, [<Rn>], +/-<Rm> cond:4|0|0|0|0|U|0|1|0|Rn:4|Rt:4|0|0|0|0|1|0|1|1|Rm:4
	{0x0f700000, 0x04200000, 4, arm_STRT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_imm12_postindex}},                    // STRT<c> <Rt>, [<Rn>] {,#+/-<imm12>} cond:4|0|1|0|0|U|0|1|0|Rn:4|Rt:4|imm12:12
	{0x0f700010, 0x06200000, 4, arm_STRT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_mem_R_pm_R_shift_imm_postindex}},              // STRT<c> <Rt>,[<Rn>],+/-<Rm>{, <shift>} cond:4|0|1|1|0|U|0|1|0|Rn:4|Rt:4|imm5:5|type:2|0|Rm:4
	{0x0fe00000, 0x02400000, 2, arm_SUB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_const}},                      // SUB{S}<c> <Rd>,<Rn>,#<const> cond:4|0|0|1|0|0|1|0|S|Rn:4|Rd:4|imm12:12
	{0x0fe00090, 0x00400010, 4, arm_SUB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_R}},                  // SUB{S}<c> <Rd>,<Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|0|0|1|0|S|Rn:4|Rd:4|Rs:4|0|type:2|1|Rm:4
	{0x0fe00010, 0x00400000, 2, arm_SUB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_shift_imm}},                // SUB{S}<c> <Rd>,<Rn>,<Rm>{,<shift>} cond:4|0|0|0|0|0|1|0|S|Rn:4|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0fef0000, 0x024d0000, 2, arm_SUB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_SP, arm_arg_const}},                        // SUB{S}<c> <Rd>,SP,#<const> cond:4|0|0|1|0|0|1|0|S|1|1|0|1|Rd:4|imm12:12
	{0x0fef0010, 0x004d0000, 2, arm_SUB_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_SP, arm_arg_R_shift_imm}},                  // SUB{S}<c> <Rd>,SP,<Rm>{,<shift>} cond:4|0|0|0|0|0|1|0|S|1|1|0|1|Rd:4|imm5:5|type:2|0|Rm:4
	{0x0f000000, 0x0f000000, 4, arm_SVC_EQ, 0x1c04, arm_instArgs{arm_arg_imm24}},                                                      // SVC<c> #<imm24> cond:4|1|1|1|1|imm24:24
	{0x0fb00ff0, 0x01000090, 4, arm_SWP_EQ, 0x16011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_mem_R}},                       // SWP{B}<c> <Rt>,<Rm>,[<Rn>] cond:4|0|0|0|1|0|B|0|0|Rn:4|Rt:4|0|0|0|0|1|0|0|1|Rm:4
	{0x0ff003f0, 0x06800070, 2, arm_SXTAB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                   // SXTAB16<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|0|0|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0ff003f0, 0x06a00070, 2, arm_SXTAB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                     // SXTAB<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|1|0|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0ff003f0, 0x06b00070, 2, arm_SXTAH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                     // SXTAH<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|1|1|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x068f0070, 4, arm_SXTB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                  // SXTB16<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|0|0|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x06af0070, 4, arm_SXTB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                    // SXTB<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|1|0|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x06bf0070, 4, arm_SXTH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                    // SXTH<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|0|1|1|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0ff0f000, 0x03300000, 4, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // TEQ<c> <Rn>,#<const> cond:4|0|0|1|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff00000, 0x03300000, 3, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // TEQ<c> <Rn>,#<const> cond:4|0|0|1|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff0f090, 0x01300010, 4, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // TEQ<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff00090, 0x01300010, 3, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // TEQ<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff0f010, 0x01300000, 4, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // TEQ<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff00010, 0x01300000, 3, arm_TEQ_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // TEQ<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|0|1|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff0f000, 0x03100000, 4, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // TST<c> <Rn>,#<const> cond:4|0|0|1|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff00000, 0x03100000, 3, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_const}},                                        // TST<c> <Rn>,#<const> cond:4|0|0|1|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|imm12:12
	{0x0ff0f090, 0x01100010, 4, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // TST<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff00090, 0x01100010, 3, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_R}},                                    // TST<c> <Rn>,<Rm>,<type> <Rs> cond:4|0|0|0|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|Rs:4|0|type:2|1|Rm:4
	{0x0ff0f010, 0x01100000, 4, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // TST<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff00010, 0x01100000, 3, arm_TST_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_shift_imm}},                                  // TST<c> <Rn>,<Rm>{,<shift>} cond:4|0|0|0|1|0|0|0|1|Rn:4|(0)|(0)|(0)|(0)|imm5:5|type:2|0|Rm:4
	{0x0ff00ff0, 0x06500f10, 4, arm_UADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06500f10, 3, arm_UADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06500f90, 4, arm_UADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06500f90, 3, arm_UADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x06500f30, 4, arm_UASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // UASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06500f30, 3, arm_UASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // UASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0fe00070, 0x07e00050, 4, arm_UBFX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_0, arm_arg_imm5, arm_arg_widthm1}},          // UBFX<c> <Rd>,<Rn>,#<lsb>,#<widthm1> cond:4|0|1|1|1|1|1|1|widthm1:5|Rd:4|lsb:5|1|0|1|Rn:4
	{0x0ff00ff0, 0x06700f10, 4, arm_UHADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UHADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06700f10, 3, arm_UHADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UHADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06700f90, 4, arm_UHADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UHADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06700f90, 3, arm_UHADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UHADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x06700f30, 4, arm_UHASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UHASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06700f30, 3, arm_UHASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UHASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff00ff0, 0x06700f50, 4, arm_UHSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UHSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06700f50, 3, arm_UHSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UHSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06700f70, 4, arm_UHSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UHSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06700f70, 3, arm_UHSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UHSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06700ff0, 4, arm_UHSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UHSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06700ff0, 3, arm_UHSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UHSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x00400090, 4, arm_UMAAL_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},             // UMAAL<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|0|0|1|0|0|RdHi:4|RdLo:4|Rm:4|1|0|0|1|Rn:4
	{0x0fe000f0, 0x00a00090, 4, arm_UMLAL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // UMLAL{S}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|0|1|0|1|S|RdHi:4|RdLo:4|Rm:4|1|0|0|1|Rn:4
	{0x0fe000f0, 0x00800090, 4, arm_UMULL_EQ, 0x14011c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},         // UMULL{S}<c> <RdLo>,<RdHi>,<Rn>,<Rm> cond:4|0|0|0|0|1|0|0|S|RdHi:4|RdLo:4|Rm:4|1|0|0|1|Rn:4
	{0x0ff00ff0, 0x06600f10, 4, arm_UQADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UQADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff000f0, 0x06600f10, 3, arm_UQADD16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UQADD16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|0|1|Rm:4
	{0x0ff00ff0, 0x06600f90, 4, arm_UQADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UQADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff000f0, 0x06600f90, 3, arm_UQADD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UQADD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|0|0|1|Rm:4
	{0x0ff00ff0, 0x06600f30, 4, arm_UQASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UQASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff000f0, 0x06600f30, 3, arm_UQASX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UQASX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rm:4
	{0x0ff00ff0, 0x06600f50, 4, arm_UQSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UQSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06600f50, 3, arm_UQSAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // UQSAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06600f70, 4, arm_UQSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UQSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06600f70, 3, arm_UQSUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                        // UQSUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06600ff0, 4, arm_UQSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UQSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06600ff0, 3, arm_UQSUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // UQSUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|1|0|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff0f0f0, 0x0780f010, 4, arm_USAD8_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8}},                           // USAD8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|1|1|0|0|0|Rd:4|1|1|1|1|Rm:4|0|0|0|1|Rn:4
	{0x0ff000f0, 0x07800010, 2, arm_USADA8_EQ, 0x1c04, arm_instArgs{arm_arg_R_16, arm_arg_R_0, arm_arg_R_8, arm_arg_R_12}},            // USADA8<c> <Rd>,<Rn>,<Rm>,<Ra> cond:4|0|1|1|1|1|0|0|0|Rd:4|Ra:4|Rm:4|0|0|0|1|Rn:4
	{0x0ff00ff0, 0x06e00f30, 4, arm_USAT16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm4, arm_arg_R_0}},                      // USAT16<c> <Rd>,#<sat_imm4>,<Rn> cond:4|0|1|1|0|1|1|1|0|sat_imm:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rn:4
	{0x0ff000f0, 0x06e00f30, 3, arm_USAT16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm4, arm_arg_R_0}},                      // USAT16<c> <Rd>,#<sat_imm4>,<Rn> cond:4|0|1|1|0|1|1|1|0|sat_imm:4|Rd:4|(1)|(1)|(1)|(1)|0|0|1|1|Rn:4
	{0x0fe00030, 0x06e00010, 4, arm_USAT_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_satimm5, arm_arg_R_shift_imm}},                // USAT<c> <Rd>,#<sat_imm5>,<Rn>{,<shift>} cond:4|0|1|1|0|1|1|1|sat_imm:5|Rd:4|imm5:5|sh|0|1|Rn:4
	{0x0ff00ff0, 0x06500f50, 4, arm_USAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // USAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff000f0, 0x06500f50, 3, arm_USAX_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                           // USAX<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|0|1|Rm:4
	{0x0ff00ff0, 0x06500f70, 4, arm_USUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // USUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff000f0, 0x06500f70, 3, arm_USUB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                         // USUB16<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|0|1|1|1|Rm:4
	{0x0ff00ff0, 0x06500ff0, 4, arm_USUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // USUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff000f0, 0x06500ff0, 3, arm_USUB8_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_0}},                          // USUB8<c> <Rd>,<Rn>,<Rm> cond:4|0|1|1|0|0|1|0|1|Rn:4|Rd:4|(1)|(1)|(1)|(1)|1|1|1|1|Rm:4
	{0x0ff003f0, 0x06c00070, 2, arm_UXTAB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                   // UXTAB16<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|0|0|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0ff003f0, 0x06e00070, 2, arm_UXTAB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                     // UXTAB<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|1|0|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0ff003f0, 0x06f00070, 2, arm_UXTAH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_16, arm_arg_R_rotate}},                     // UXTAH<c> <Rd>,<Rn>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|1|1|Rn:4|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x06cf0070, 4, arm_UXTB16_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                  // UXTB16<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|0|0|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x06ef0070, 4, arm_UXTB_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                    // UXTB<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|1|0|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fff03f0, 0x06ff0070, 4, arm_UXTH_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_R_rotate}},                                    // UXTH<c> <Rd>,<Rm>{,<rotation>} cond:4|0|1|1|0|1|1|1|1|1|1|1|1|Rd:4|rotate:2|0|0|0|1|1|1|Rm:4
	{0x0fb00e10, 0x0e000a00, 4, arm_VMLA_EQ_F32, 0x60108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},            // V<MLA,MLS><c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|0|0|Vn:4|Vd:4|1|0|1|sz|N|op|M|0|Vm:4
	{0x0fbf0ed0, 0x0eb00ac0, 4, arm_VABS_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm_Dm}},                               // VABS<c>.F<32,64> <Sd,Dd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|0|0|0|Vd:4|1|0|1|sz|1|1|M|0|Vm:4
	{0x0fb00e50, 0x0e300a00, 4, arm_VADD_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},                // VADD<c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|1|1|Vn:4|Vd:4|1|0|1|sz|N|0|M|0|Vm:4
	{0x0fbf0e7f, 0x0eb50a40, 4, arm_VCMP_EQ_F32, 0x70108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_fp_0}},                            // VCMP{E}<c>.F<32,64> <Sd,Dd>, #0.0 cond:4|1|1|1|0|1|D|1|1|0|1|0|1|Vd:4|1|0|1|sz|E|1|0|0|(0)|(0)|(0)|(0)
	{0x0fbf0e70, 0x0eb50a40, 3, arm_VCMP_EQ_F32, 0x70108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_fp_0}},                            // VCMP{E}<c>.F<32,64> <Sd,Dd>, #0.0 cond:4|1|1|1|0|1|D|1|1|0|1|0|1|Vd:4|1|0|1|sz|E|1|0|0|(0)|(0)|(0)|(0)
	{0x0fbf0e50, 0x0eb40a40, 4, arm_VCMP_EQ_F32, 0x70108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm_Dm}},                           // VCMP{E}<c>.F<32,64> <Sd,Dd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|1|0|0|Vd:4|1|0|1|sz|E|1|M|0|Vm:4
	{0x0fbe0e50, 0x0eba0a40, 4, arm_VCVT_EQ_F32_FXS16, 0x801100107011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sd_Dd, arm_arg_fbits}},  // VCVT<c>.F<32,64>.FX<S,U><16,32> <Sd,Dd>, <Sd,Dd>, #<fbits> cond:4|1|1|1|0|1|D|1|1|1|0|1|U|Vd:4|1|0|1|sz|sx|1|i|0|imm4:4
	{0x0fbe0e50, 0x0ebe0a40, 4, arm_VCVT_EQ_FXS16_F32, 0x1001070108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sd_Dd, arm_arg_fbits}}, // VCVT<c>.FX<S,U><16,32>.F<32,64> <Sd,Dd>, <Sd,Dd>, #<fbits> cond:4|1|1|1|0|1|D|1|1|1|1|1|U|Vd:4|1|0|1|sz|sx|1|i|0|imm4:4
	{0x0fbf0ed0, 0x0eb70ac0, 4, arm_VCVT_EQ_F64_F32, 0x8011c04, arm_instArgs{arm_arg_Dd_Sd, arm_arg_Sm_Dm}},                           // VCVT<c>.<F64.F32,F32.F64> <Dd,Sd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|1|1|1|Vd:4|1|0|1|sz|1|1|M|0|Vm:4
	{0x0fbe0f50, 0x0eb20a40, 4, arm_VCVTB_EQ_F32_F16, 0x70110011c04, arm_instArgs{arm_arg_Sd, arm_arg_Sm}},                            // VCVT<B,T><c>.<F32.F16,F16.F32> <Sd>, <Sm> cond:4|1|1|1|0|1|D|1|1|0|0|1|op|Vd:4|1|0|1|0|T|1|M|0|Vm:4
	{0x0fbf0e50, 0x0eb80a40, 4, arm_VCVT_EQ_F32_U32, 0x80107011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm}},                          // VCVT<c>.F<32,64>.<U,S>32 <Sd,Dd>, <Sm> cond:4|1|1|1|0|1|D|1|1|1|0|0|0|Vd:4|1|0|1|sz|op|1|M|0|Vm:4
	{0x0fbe0e50, 0x0ebc0a40, 4, arm_VCVTR_EQ_U32_F32, 0x701100108011c04, arm_instArgs{arm_arg_Sd, arm_arg_Sm_Dm}},                     // VCVT<R,><c>.<U,S>32.F<32,64> <Sd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|1|1|0|signed|Vd:4|1|0|1|sz|op|1|M|0|Vm:4
	{0x0fb00e50, 0x0e800a00, 4, arm_VDIV_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},                // VDIV<c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|1|D|0|0|Vn:4|Vd:4|1|0|1|sz|N|0|M|0|Vm:4
	{0x0f300e00, 0x0d100a00, 4, arm_VLDR_EQ, 0x1c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_mem_R_pm_imm8at0_offset}},                    // VLDR<c> <Sd,Dd>, [<Rn>{,#+/-<imm8>}] cond:4|1|1|0|1|U|D|0|1|Rn:4|Vd:4|1|0|1|sz|imm8:8
	{0x0ff00f7f, 0x0e000a10, 4, arm_VMOV_EQ, 0x1c04, arm_instArgs{arm_arg_Sn, arm_arg_R_12}},                                          // VMOV<c> <Sn>, <Rt> cond:4|1|1|1|0|0|0|0|0|Vn:4|Rt:4|1|0|1|0|N|0|0|1|0|0|0|0
	{0x0ff00f7f, 0x0e100a10, 4, arm_VMOV_EQ, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_Sn}},                                          // VMOV<c> <Rt>, <Sn> cond:4|1|1|1|0|0|0|0|1|Vn:4|Rt:4|1|0|1|0|N|0|0|1|0|0|0|0
	{0x0fd00f7f, 0x0e100b10, 4, arm_VMOV_EQ_32, 0x1c04, arm_instArgs{arm_arg_R_12, arm_arg_Dn_half}},                                  // VMOV<c>.32 <Rt>, <Dn[x]> cond:4|1|1|1|0|0|0|opc1|1|Vn:4|Rt:4|1|0|1|1|N|0|0|1|0|0|0|0
	{0x0fd00f7f, 0x0e000b10, 4, arm_VMOV_EQ_32, 0x1c04, arm_instArgs{arm_arg_Dn_half, arm_arg_R_12}},                                  // VMOV<c>.32 <Dd[x]>, <Rt> cond:4|1|1|1|0|0|0|opc1|0|Vd:4|Rt:4|1|0|1|1|D|0|0|1|0|0|0|0
	{0x0fb00ef0, 0x0eb00a00, 4, arm_VMOV_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_imm_vfp}},                             // VMOV<c>.F<32,64> <Sd,Dd>, #<imm_vfp> cond:4|1|1|1|0|1|D|1|1|imm4H:4|Vd:4|1|0|1|sz|0|0|0|0|imm4L:4
	{0x0fbf0ed0, 0x0eb00a40, 4, arm_VMOV_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm_Dm}},                               // VMOV<c>.F<32,64> <Sd,Dd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|0|0|0|Vd:4|1|0|1|sz|0|1|M|0|Vm:4
	{0x0fff0fff, 0x0ef10a10, 4, arm_VMRS_EQ, 0x1c04, arm_instArgs{arm_arg_R_12_nzcv, arm_arg_FPSCR}},                                  // VMRS<c> <Rt_nzcv>, FPSCR cond:4|1|1|1|0|1|1|1|1|0|0|0|1|Rt:4|1|0|1|0|0|0|0|1|0|0|0|0
	{0x0fff0fff, 0x0ee10a10, 4, arm_VMSR_EQ, 0x1c04, arm_instArgs{arm_arg_FPSCR, arm_arg_R_12}},                                       // VMSR<c> FPSCR, <Rt> cond:4|1|1|1|0|1|1|1|0|0|0|0|1|Rt:4|1|0|1|0|0|0|0|1|0|0|0|0
	{0x0fb00e50, 0x0e200a00, 4, arm_VMUL_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},                // VMUL<c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|1|0|Vn:4|Vd:4|1|0|1|sz|N|0|M|0|Vm:4
	{0x0fbf0ed0, 0x0eb10a40, 4, arm_VNEG_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm_Dm}},                               // VNEG<c>.F<32,64> <Sd,Dd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|0|0|1|Vd:4|1|0|1|sz|0|1|M|0|Vm:4
	{0x0fb00e10, 0x0e100a00, 4, arm_VNMLS_EQ_F32, 0x60108011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},           // VN<MLS,MLA><c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|0|1|Vn:4|Vd:4|1|0|1|sz|N|op|M|0|Vm:4
	{0x0fb00e50, 0x0e200a40, 4, arm_VNMUL_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},               // VNMUL<c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|1|0|Vn:4|Vd:4|1|0|1|sz|N|1|M|0|Vm:4
	{0x0fbf0ed0, 0x0eb10ac0, 4, arm_VSQRT_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sm_Dm}},                              // VSQRT<c>.F<32,64> <Sd,Dd>, <Sm,Dm> cond:4|1|1|1|0|1|D|1|1|0|0|0|1|Vd:4|1|0|1|sz|1|1|M|0|Vm:4
	{0x0f300e00, 0x0d000a00, 4, arm_VSTR_EQ, 0x1c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_mem_R_pm_imm8at0_offset}},                    // VSTR<c> <Sd,Dd>, [<Rn>{,#+/-<imm8>}] cond:4|1|1|0|1|U|D|0|0|Rn:4|Vd:4|1|0|1|sz|imm8:8
	{0x0fb00e50, 0x0e300a40, 4, arm_VSUB_EQ_F32, 0x8011c04, arm_instArgs{arm_arg_Sd_Dd, arm_arg_Sn_Dn, arm_arg_Sm_Dm}},                // VSUB<c>.F<32,64> <Sd,Dd>, <Sn,Dn>, <Sm,Dm> cond:4|1|1|1|0|0|D|1|1|Vn:4|Vd:4|1|0|1|sz|N|1|M|0|Vm:4
	{0x0fffffff, 0x0320f002, 4, arm_WFE_EQ, 0x1c04, arm_instArgs{}},                                                                   // WFE<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|1|0
	{0x0fff00ff, 0x0320f002, 3, arm_WFE_EQ, 0x1c04, arm_instArgs{}},                                                                   // WFE<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|1|0
	{0x0fffffff, 0x0320f003, 4, arm_WFI_EQ, 0x1c04, arm_instArgs{}},                                                                   // WFI<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|1|1
	{0x0fff00ff, 0x0320f003, 3, arm_WFI_EQ, 0x1c04, arm_instArgs{}},                                                                   // WFI<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|1|1
	{0x0fffffff, 0x0320f001, 4, arm_YIELD_EQ, 0x1c04, arm_instArgs{}},                                                                 // YIELD<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|0|1
	{0x0fff00ff, 0x0320f001, 3, arm_YIELD_EQ, 0x1c04, arm_instArgs{}},                                                                 // YIELD<c> cond:4|0|0|1|1|0|0|1|0|0|0|0|0|(1)|(1)|(1)|(1)|(0)|(0)|(0)|(0)|0|0|0|0|0|0|0|1
	{0xffffffff, 0xf7fabcfd, 4, arm_UNDEF, 0x0, arm_instArgs{}},                                                                       // UNDEF 1|1|1|1|0|1|1|1|1|1|1|1|1|0|1|0|1|0|1|1|1|1|0|0|1|1|1|1|1|1|0|1
}
