#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <libgnome/libgnome.h>

#include "gm-world-info-dialog.h"
#include "gm-world.h"
#include "gm-support.h"
#include "gm-pixbuf.h"
#include "gm-debug.h"

#define GM_WORLD_INFO_DIALOG_XML PACKAGE_DATA_DIR "/" PACKAGE "/ui/gm-world-info.glade"

gboolean on_gm_world_info_dialog_url_button_release(GtkWidget *button, 
		GdkEventButton *event, gchar *link);

/* Private */
void
gm_world_info_dialog_set_label(GtkLabel *label, gchar *text) {
	if (text) {
		gtk_label_set_markup(label, text);
	} else {
		gtk_label_set_markup(label, _("<i>unspecified</i>"));
	}
}

void
gm_world_info_dialog_event_box_cursor(GtkWidget *widget) {
	GdkCursor *cursor = gdk_cursor_new(GDK_HAND2);
	gdk_window_set_cursor(widget->window, cursor);
	gdk_cursor_destroy(cursor);
}

/* Public */

GtkDialog *
gm_world_info_dialog_new(GmWorldInfo const *world_info) {
	GladeXML *xml;
	GtkWidget *dlg, *widget;
	gchar *tmp;

	xml = glade_xml_new(GM_WORLD_INFO_DIALOG_XML, "gm_world_info_dialog", NULL);

	if (xml == NULL) {
		gm_debug_msg(DEBUG_ALWAYS, "Couldn't find glade file %s!", 
				GM_WORLD_INFO_DIALOG_XML);
		return NULL;
	}

	dlg = glade_xml_get_widget(xml, "gm_world_info_dialog");

	if (world_info->name) {
		tmp = g_strconcat(world_info->name, " - ", _("World info"), NULL);
		gtk_window_set_title(GTK_WINDOW(dlg), tmp);
		g_free(tmp);
	}

	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_world_name")), world_info->name);
	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_name")), world_info->admin);
	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_location")), world_info->location);
	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_system")), world_info->system);
	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_characterset")), world_info->charset);
	gm_world_info_dialog_set_label(GTK_LABEL(glade_xml_get_widget(xml, 
			"label_language")), world_info->language);
  
	if (world_info->homepage) {
		tmp = g_strconcat("<u><span color=\"#0000ff\">", world_info->homepage, 
				"</span></u>", NULL);
		glade_xml_signal_connect_data(xml, 
				"on_label_homepage_button_release",
				G_CALLBACK(on_gm_world_info_dialog_url_button_release), NULL);
		widget = glade_xml_get_widget(xml, "label_homepage");
		gtk_label_set_markup(GTK_LABEL(widget), tmp);
		g_free(tmp);
		
		gm_world_info_dialog_event_box_cursor(glade_xml_get_widget(xml, 
				"event_box_homepage"));
	} else {
		gtk_label_set_markup(GTK_LABEL(glade_xml_get_widget(xml, 
				"label_homepage")), _("<i>unspecified</i>"));
	}
  
	if (world_info->contact) {
		tmp = g_strconcat("<u><span color=\"#0000ff\">", world_info->contact, 
				"</span></u>", NULL);
		glade_xml_signal_connect_data(xml, 
				"on_label_email_button_release", 
				G_CALLBACK(on_gm_world_info_dialog_url_button_release), "mailto:");
		widget = glade_xml_get_widget(xml, "label_email");
		gtk_label_set_markup(GTK_LABEL(widget), tmp);
		g_free(tmp);
		
		gm_world_info_dialog_event_box_cursor(glade_xml_get_widget(xml, 
				"event_box_email"));
	} else {
		gtk_label_set_markup(GTK_LABEL(glade_xml_get_widget(xml, 
				"label_email")), _("<i>unspecified</i>"));
	}
  
	if (world_info->logo) {
		gtk_image_set_from_pixbuf(GTK_IMAGE(glade_xml_get_widget(xml, 
				"image_logo")), gm_pixbuf_get(world_info->logo));
	}
  
	glade_xml_signal_connect(xml, "on_gm_world_info_dialog_delete", 
			G_CALLBACK(gtk_widget_destroy));
	glade_xml_signal_connect_data(xml, "on_button_close_clicked", 
			G_CALLBACK(gm_widget_destroy_data), dlg);

	gtk_widget_show_all(dlg);

	g_object_unref(xml);
	return GTK_DIALOG(dlg);
}

/* Callbacks */

gboolean
on_gm_world_info_dialog_url_button_release(GtkWidget *label, 
		GdkEventButton *event, gchar *prefix) {
	GError *err = NULL;
	gchar *tmp, *link;

	if (prefix != NULL) {
		link = g_strconcat(prefix, gtk_label_get_text(GTK_LABEL(label)), NULL);
	} else {
		link = g_strdup(gtk_label_get_text(GTK_LABEL(label)));
	}

	if (!gnome_url_show(link, &err)) {
		tmp = g_strdup_printf(_("Could not open link: %s"), err->message);
		gm_error_dialog(tmp, NULL);
		g_free(tmp);
		g_error_free(err);
	}

	g_free(link);  
	return FALSE;
}
