/*
 * Copyright (C) 2019 Peter Ross
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * VP4 video decoder
 */

#ifndef AVCODEC_VP4DATA_H
#define AVCODEC_VP4DATA_H

#include <stdint.h>

static const uint8_t vp4_generic_dequant[64] = {
    16, 17, 18, 20, 22, 24, 26, 28,
    17, 18, 20, 22, 24, 26, 28, 32,
    18, 20, 22, 24, 26, 28, 32, 36,
    20, 22, 24, 26, 28, 32, 36, 40,
    22, 24, 26, 28, 32, 36, 40, 44,
    24, 26, 28, 32, 36, 40, 44, 48,
    26, 28, 32, 36, 40, 44, 48, 52,
    28, 32, 36, 40, 44, 48, 52, 56
};

static const uint8_t vp4_y_dc_scale_factor[64] = {
    180, 180, 180, 180, 180, 180, 175, 170,
    165, 160, 157, 155, 152, 150, 147, 145,
    142, 140, 137, 135, 132, 130, 127, 125,
    122, 120, 117, 115, 112, 110, 107, 105,
    102, 100,  97,  95,  92,  90,  87,  85,
     82,  80,  77,  75,  72,  70,  67,  65,
     62,  60,  57,  55,  52,  50,  47,  45,
     42,  40,  37,  35,  32,  30,  27,  25
};

static const uint8_t vp4_uv_dc_scale_factor[64] = {
    150, 150, 150, 150, 150, 150, 150, 150,
    150, 150, 150, 150, 150, 150, 147, 145,
    142, 140, 137, 135, 132, 130, 127, 125,
    122, 120, 117, 115, 112, 110, 107, 105,
    102, 100,  97,  95,  92,  90,  87,  85,
     82,  80,  77,  75,  72,  70,  67,  65,
     62,  60,  57,  55,  52,  50,  47,  45,
     42,  40,  37,  35,  32,  30,  27,  25
};

static const uint16_t vp4_ac_scale_factor[64] = {
    500, 475, 450, 430, 410, 390, 370, 350,
    330, 315, 300, 285, 270, 260, 250, 240,
    230, 220, 210, 200, 190, 185, 180, 170,
    160, 150, 143, 135, 128, 120, 113, 106,
    100,  94,  90,  85,  80,  75,  70,  66,
     62,  57,  52,  49,  45,  41,  38,  35,
     33,  30,  27,  24,  22,  20,  18,  16,
     14,  12,  10,   9,   7,   6,   4,   1
};

static const uint8_t vp4_filter_limit_values[64] = {
    30, 25, 20, 20, 15, 15, 14, 14,
    13, 13, 12, 12, 11, 11, 10, 10,
     9,  9,  8,  8,  7,  7,  7,  7,
     6,  6,  6,  6,  5,  5,  5,  5,
     4,  4,  4,  4,  3,  3,  3,  3,
     2,  2,  2,  2,  2,  2,  2,  2,
     2,  2,  2,  2,  2,  2,  2,  2,
     1,  1,  1,  1,  1,  1,  1,  1
};

static const uint8_t vp4_block_pattern_table_selector[14] = {
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1
};

static const uint8_t vp4_block_pattern_vlc[2][14][2] = {
    {
        { 0x0, 3 }, { 0xf, 4 }, { 0x9, 4 }, { 0x2, 3 },
        { 0xd, 4 }, { 0xe, 5 }, { 0xb, 4 }, { 0x1, 3 },
        { 0xf, 5 }, { 0x8, 4 }, { 0x6, 4 }, { 0xe, 4 },
        { 0xc, 4 }, { 0xa, 4 }
    },
    {
        { 0x7, 4 }, { 0xa, 4 }, { 0x9, 4 }, { 0xc, 4 },
        { 0x8, 4 }, { 0xd, 5 }, { 0x0, 3 }, { 0xe, 4 },
        { 0xc, 5 }, { 0xd, 4 }, { 0x1, 3 }, { 0xb, 4 },
        { 0xf, 4 }, { 0x2, 3 }
    }
};

static const uint8_t vp4_mv_table_selector[32] = {
    0, 1, 2, 2, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5,
    6, 6, 6, 6, 6, 6, 6, 6,
};

static const uint16_t vp4_mv_vlc[2][7][63][2] = {
    { /* x-axis tables */
        {
            {   0x6A,  7 }, {  0x11A,  9 }, {  0x18E,  9 }, {  0x237, 10 },
            {   0x4A,  8 }, {  0x236, 10 }, {   0x7A,  8 }, {   0xD6,  9 },
            {   0x7E,  8 }, {  0x1FD, 10 }, {   0x8C,  8 }, {   0xD7,  9 },
            {   0x87,  8 }, {  0x183,  9 }, {   0x3C,  7 }, {   0x61,  7 },
            {   0x47,  7 }, {   0x69,  8 }, {   0x40,  7 }, {   0x48,  8 },
            {   0x49,  7 }, {   0x86,  8 }, {   0x13,  6 }, {   0xD2,  8 },
            {   0x1C,  6 }, {   0x42,  7 }, {   0x25,  6 }, {   0x1B,  6 },
            {   0x13,  5 }, {    0x5,  4 }, {    0x0,  2 }, {    0x7,  3 },
            {    0x5,  3 }, {   0x1B,  5 }, {   0x19,  5 }, {   0x19,  6 },
            {    0x8,  5 }, {   0x45,  7 }, {   0x1D,  6 }, {   0xC6,  8 },
            {   0x68,  7 }, {   0x90,  8 }, {   0x41,  7 }, {   0x4B,  8 },
            {   0x31,  7 }, {  0x18F,  9 }, {   0x62,  7 }, {   0x3E,  7 },
            {   0x44,  7 }, {   0x68,  8 }, {   0x30,  7 }, {  0x182,  9 },
            {   0xC0,  8 }, {  0x1A7,  9 }, {   0x91,  8 }, {   0x92,  9 },
            {   0x7B,  8 }, {   0xFF,  9 }, {  0x1A6,  9 }, {  0x1FC, 10 },
            {   0x6A,  8 }, {   0x93,  9 }, {   0x6B,  7 }
        },
        {
            {   0x39,  7 }, {  0x259, 10 }, {   0x1B,  8 }, {  0x1D1, 10 },
            {  0x137,  9 }, {  0x1D0, 10 }, {   0x1A,  8 }, {  0x1B5, 10 },
            {   0x1D,  8 }, {  0x4BC, 11 }, {   0x6C,  8 }, {   0x38,  9 },
            {   0x71,  8 }, {   0x2D,  9 }, {   0x7D,  8 }, {   0x75,  8 },
            {   0x19,  7 }, {   0xE9,  9 }, {   0x37,  7 }, {   0x15,  8 },
            {   0x1E,  7 }, {   0xDB,  9 }, {   0x4C,  7 }, {   0x70,  8 },
            {    0xD,  6 }, {    0xC,  7 }, {   0x27,  6 }, {    0x4,  6 },
            {    0x2,  4 }, {    0x0,  4 }, {    0x5,  3 }, {    0x7,  3 },
            {    0x6,  3 }, {    0x2,  3 }, {    0x8,  4 }, {   0x24,  6 },
            {    0xC,  5 }, {   0x3B,  7 }, {   0x1E,  6 }, {   0x9A,  8 },
            {    0xE,  6 }, {   0x69,  8 }, {   0x4A,  7 }, {  0x12D,  9 },
            {   0x35,  7 }, {   0xF9,  9 }, {   0x18,  7 }, {   0x7F,  8 },
            {    0xF,  7 }, {   0xF8,  9 }, {   0x7E,  8 }, {  0x25F, 10 },
            {   0x68,  8 }, {   0x2C,  9 }, {   0x14,  8 }, {  0x258, 10 },
            {  0x136,  9 }, {  0x4BD, 11 }, {  0x12E,  9 }, {  0x1B4, 10 },
            {   0x17,  8 }, {   0x39,  9 }, {   0x1F,  7 }
        },
        {
            {   0x29,  7 }, {  0x3CB, 10 }, {  0x1F5,  9 }, {  0x263, 10 },
            {  0x1F4,  9 }, {  0x3DA, 10 }, {   0x50,  8 }, {  0x260, 10 },
            {  0x1EC,  9 }, {  0x3D3, 10 }, {  0x109,  9 }, {  0x3D2, 10 },
            {   0x51,  8 }, {  0x792, 11 }, {   0xF3,  8 }, {   0x9A,  8 },
            {   0xF7,  8 }, {  0x132,  9 }, {   0xC1,  8 }, {  0x1E8,  9 },
            {   0x2A,  7 }, {   0x85,  8 }, {   0x61,  7 }, {  0x1F7,  9 },
            {   0x78,  7 }, {   0xC7,  8 }, {   0x23,  6 }, {   0x7C,  7 },
            {   0x12,  5 }, {    0xB,  5 }, {    0xE,  4 }, {    0xD,  4 },
            {    0x0,  2 }, {    0x5,  3 }, {    0x3,  3 }, {    0x4,  4 },
            {   0x19,  5 }, {   0x20,  6 }, {   0x3F,  6 }, {   0x43,  7 },
            {   0x62,  7 }, {   0x9F,  8 }, {   0x4E,  7 }, {  0x181,  9 },
            {   0x2B,  7 }, {  0x137,  9 }, {   0xF5,  8 }, {   0x89,  8 },
            {   0xC6,  8 }, {  0x262, 10 }, {   0x88,  8 }, {  0x3C8, 10 },
            {  0x1F6,  9 }, {  0x3CA, 10 }, {   0x9E,  8 }, {  0x261, 10 },
            {  0x136,  9 }, {  0x108,  9 }, {  0x133,  9 }, {  0x793, 11 },
            {  0x180,  9 }, {  0x3DB, 10 }, {   0x45,  7 }
        },
        {
            {    0x1,  6 }, {  0x1C7,  9 }, {   0x67,  8 }, {   0xB5,  9 },
            {   0x66,  8 }, {  0x139,  9 }, {   0x99,  8 }, {   0xB4,  9 },
            {   0xC3,  8 }, {  0x130,  9 }, {    0x0,  7 }, {  0x131,  9 },
            {   0x9E,  8 }, {   0xB7,  9 }, {   0x2C,  7 }, {    0x1,  7 },
            {   0x28,  7 }, {  0x138,  9 }, {   0x4B,  7 }, {   0x31,  8 },
            {   0x60,  7 }, {   0x91,  8 }, {    0x3,  6 }, {   0x9D,  8 },
            {   0x17,  6 }, {   0x4D,  7 }, {   0x31,  6 }, {   0x70,  7 },
            {    0x7,  5 }, {   0x3A,  6 }, {    0x7,  4 }, {    0x2,  4 },
            {    0xB,  4 }, {    0x1,  4 }, {    0xF,  4 }, {    0x8,  4 },
            {    0xD,  4 }, {    0x4,  4 }, {    0xA,  4 }, {    0xD,  5 },
            {   0x19,  5 }, {    0x2,  6 }, {   0x3B,  6 }, {   0x4A,  7 },
            {   0x15,  6 }, {   0xC2,  8 }, {   0x18,  6 }, {   0x32,  7 },
            {   0x72,  7 }, {  0x1C6,  9 }, {   0x29,  7 }, {  0x1C5,  9 },
            {   0x49,  7 }, {  0x121,  9 }, {   0x1B,  7 }, {   0x30,  8 },
            {   0x1A,  7 }, {  0x1C4,  9 }, {   0x9F,  8 }, {   0xB6,  9 },
            {   0x19,  7 }, {  0x120,  9 }, {   0x73,  7 }
        },
        {
            {   0x23,  6 }, {  0x1C8,  9 }, {   0x43,  8 }, {  0x110,  9 },
            {    0xC,  7 }, {  0x153,  9 }, {   0x22,  7 }, {  0x111,  9 },
            {    0xF,  7 }, {   0x42,  8 }, {   0x23,  7 }, {  0x1C9,  9 },
            {   0x2A,  7 }, {   0x1B,  8 }, {   0x73,  7 }, {   0x45,  7 },
            {   0x6E,  7 }, {   0x89,  8 }, {   0x6C,  7 }, {   0x1A,  8 },
            {   0x6F,  7 }, {   0xB6,  8 }, {    0xB,  6 }, {   0xE5,  8 },
            {   0x25,  6 }, {   0x20,  7 }, {   0x29,  6 }, {   0x4D,  7 },
            {    0x2,  5 }, {   0x14,  6 }, {   0x1A,  5 }, {   0x17,  5 },
            {   0x1E,  5 }, {   0x27,  6 }, {   0x18,  5 }, {   0x28,  6 },
            {   0x1F,  5 }, {    0x0,  5 }, {    0x6,  4 }, {   0x10,  5 },
            {    0x7,  4 }, {    0xB,  5 }, {    0x3,  4 }, {    0x4,  5 },
            {   0x1D,  5 }, {   0x2C,  6 }, {   0x19,  5 }, {   0x2B,  6 },
            {    0x9,  5 }, {   0x55,  7 }, {   0x38,  6 }, {    0xE,  7 },
            {   0x24,  6 }, {   0xA8,  8 }, {    0xA,  6 }, {   0x99,  8 },
            {   0x5A,  7 }, {   0x98,  8 }, {   0x6D,  7 }, {  0x152,  9 },
            {   0x2B,  7 }, {   0xB7,  8 }, {    0x1,  5 }
        },
        {
            {   0x3D,  6 }, {   0xB1,  8 }, {   0xDD,  8 }, {  0x1F6,  9 },
            {   0xC5,  8 }, {  0x188,  9 }, {   0x37,  7 }, {   0x3F,  8 },
            {   0x1E,  7 }, {  0x189,  9 }, {    0xF,  7 }, {   0x3E,  8 },
            {   0x6A,  7 }, {  0x1F7,  9 }, {   0x61,  7 }, {   0x79,  7 },
            {   0x18,  6 }, {   0xB0,  8 }, {    0xE,  6 }, {   0xB3,  8 },
            {    0xC,  6 }, {   0xDF,  8 }, {    0x6,  6 }, {   0xDC,  8 },
            {   0x19,  6 }, {   0xDE,  8 }, {   0x27,  6 }, {    0xE,  7 },
            {   0x1A,  6 }, {   0x63,  7 }, {    0xF,  5 }, {    0xE,  5 },
            {   0x14,  5 }, {   0x7C,  7 }, {   0x36,  6 }, {   0x6B,  7 },
            {   0x3F,  6 }, {   0x60,  7 }, {    0x8,  5 }, {   0x74,  7 },
            {    0x9,  5 }, {   0x78,  7 }, {   0x12,  5 }, {    0xD,  6 },
            {   0x15,  5 }, {   0x2D,  6 }, {    0x2,  4 }, {   0x1C,  5 },
            {    0x5,  4 }, {   0x3B,  6 }, {    0x0,  4 }, {   0x34,  6 },
            {   0x19,  5 }, {   0x26,  6 }, {   0x10,  5 }, {   0x75,  7 },
            {    0x2,  5 }, {   0x36,  7 }, {   0x23,  6 }, {   0xB2,  8 },
            {   0x22,  6 }, {   0xFA,  8 }, {   0x17,  5 }
        },
        {
            {   0x15,  5 }, {   0xDD,  8 }, {   0x3E,  7 }, {  0x16E,  9 },
            {   0x4C,  7 }, {   0x12,  8 }, {   0x5D,  7 }, {   0xB6,  8 },
            {   0x6F,  7 }, {  0x1F1,  9 }, {   0x69,  7 }, {  0x1F0,  9 },
            {   0x1D,  7 }, {  0x16F,  9 }, {    0x2,  6 }, {   0x6B,  7 },
            {    0xC,  6 }, {   0xDC,  8 }, {   0x68,  7 }, {   0x9B,  8 },
            {   0x7D,  7 }, {   0x9A,  8 }, {    0xD,  6 }, {   0x13,  8 },
            {    0x8,  6 }, {   0xF9,  8 }, {   0x2C,  6 }, {   0x12,  7 },
            {   0x33,  6 }, {   0x4F,  7 }, {    0xD,  5 }, {    0x5,  5 },
            {   0x12,  5 }, {   0x3F,  7 }, {   0x32,  6 }, {   0x13,  7 },
            {   0x3B,  6 }, {    0x5,  6 }, {   0x2F,  6 }, {   0x5A,  7 },
            {   0x3F,  6 }, {   0x1C,  7 }, {   0x3A,  6 }, {    0x8,  7 },
            {   0x36,  6 }, {   0x5C,  7 }, {   0x10,  5 }, {    0x0,  5 },
            {    0xC,  5 }, {   0x4E,  7 }, {    0x3,  5 }, {   0x6A,  7 },
            {    0xE,  5 }, {    0x3,  6 }, {   0x14,  5 }, {   0x1E,  6 },
            {   0x1C,  5 }, {    0xF,  6 }, {   0x18,  5 }, {   0x23,  6 },
            {   0x1E,  5 }, {   0x22,  6 }, {    0x2,  3 }
        }
    },
    { /* y-axis tables */
        {
            {   0x52,  7 }, {  0x14C,  9 }, {  0x1FA,  9 }, {  0x124,  9 },
            {   0x82,  8 }, {  0x29E, 10 }, {   0x8E,  8 }, {  0x24B, 10 },
            {   0x9C,  8 }, {  0x3F7, 10 }, {   0x86,  8 }, {  0x114,  9 },
            {   0x83,  8 }, {  0x3A5, 10 }, {   0xFA,  8 }, {   0x4F,  7 },
            {   0xFB,  8 }, {  0x13B,  9 }, {   0xFC,  8 }, {  0x172,  9 },
            {   0x44,  7 }, {  0x173,  9 }, {   0x51,  7 }, {   0x87,  8 },
            {   0x5F,  7 }, {   0xBA,  8 }, {   0x26,  6 }, {   0x5E,  7 },
            {   0x16,  5 }, {   0x15,  5 }, {    0x6,  3 }, {    0x1,  2 },
            {    0x0,  2 }, {   0x1C,  5 }, {   0x1E,  5 }, {   0x75,  7 },
            {   0x3B,  6 }, {   0xFF,  8 }, {   0x25,  6 }, {   0xBB,  8 },
            {   0x7C,  7 }, {   0x8B,  8 }, {   0x48,  7 }, {  0x171,  9 },
            {   0x42,  7 }, {  0x14E,  9 }, {   0x46,  7 }, {   0xFE,  8 },
            {   0x40,  7 }, {  0x13A,  9 }, {   0x93,  8 }, {  0x115,  9 },
            {   0x8F,  8 }, {  0x3F6, 10 }, {  0x170,  9 }, {  0x29F, 10 },
            {  0x1D1,  9 }, {  0x24A, 10 }, {  0x1D3,  9 }, {  0x3A4, 10 },
            {  0x1D0,  9 }, {  0x14D,  9 }, {   0x50,  7 }
        },
        {
            {   0xDE,  8 }, {  0x223, 10 }, {  0x136,  9 }, {  0x7C5, 11 },
            {  0x12F,  9 }, {  0x4A1, 11 }, {  0x3D7, 10 }, {  0x7AC, 11 },
            {  0x133,  9 }, {  0x7C4, 11 }, {  0x1B8,  9 }, {  0x222, 10 },
            {   0x96,  8 }, {  0x251, 10 }, {   0x95,  8 }, {  0x1F0,  9 },
            {   0xDA,  8 }, {  0x110,  9 }, {   0x9A,  8 }, {  0x360, 10 },
            {   0xDD,  8 }, {  0x12E,  9 }, {   0x48,  7 }, {   0x92,  8 },
            {   0x78,  7 }, {   0x98,  8 }, {   0x27,  6 }, {   0x45,  7 },
            {   0x1A,  5 }, {   0x10,  5 }, {    0x5,  3 }, {    0x0,  2 },
            {    0x1,  2 }, {    0xE,  4 }, {    0xC,  4 }, {   0x23,  6 },
            {   0x3F,  6 }, {   0xF4,  8 }, {   0x7D,  7 }, {   0x89,  8 },
            {   0x7B,  7 }, {  0x1BE,  9 }, {   0xF9,  8 }, {  0x3E3, 10 },
            {   0xF3,  8 }, {  0x127,  9 }, {   0xDB,  8 }, {  0x1EA,  9 },
            {   0xD9,  8 }, {  0x6E7, 11 }, {  0x1BF,  9 }, {  0x4A0, 11 },
            {  0x1B1,  9 }, {  0x6E6, 11 }, {  0x137,  9 }, {  0x7AD, 11 },
            {  0x126,  9 }, {  0x6C2, 11 }, {  0x132,  9 }, {  0x6C3, 11 },
            {  0x129,  9 }, {  0x372, 10 }, {   0xF2,  8 }
        },
        {
            {   0x16,  7 }, {   0x9C,  9 }, {  0x13C,  9 }, {   0x9E, 10 },
            {  0x12B,  9 }, {   0xBA, 10 }, {  0x181,  9 }, {  0x317, 10 },
            {   0x84,  8 }, {   0x4E,  9 }, {   0x26,  8 }, {  0x316, 10 },
            {  0x180,  9 }, {   0x5C,  9 }, {   0xC1,  8 }, {   0x2F,  8 },
            {   0x10,  7 }, {   0x45,  9 }, {   0x12,  7 }, {  0x189,  9 },
            {   0x24,  7 }, {  0x13D,  9 }, {   0x66,  7 }, {   0x23,  8 },
            {   0x67,  7 }, {   0xC6,  8 }, {   0x24,  6 }, {   0x4B,  7 },
            {   0x11,  5 }, {   0x32,  6 }, {    0xD,  4 }, {    0x0,  3 },
            {    0x7,  3 }, {    0x5,  3 }, {    0x3,  3 }, {    0x3,  4 },
            {    0x5,  4 }, {   0x20,  6 }, {    0x8,  5 }, {   0x25,  7 },
            {   0x26,  6 }, {   0x4F,  8 }, {   0x61,  7 }, {   0x2B,  8 },
            {   0x4E,  7 }, {  0x18A,  9 }, {   0x43,  7 }, {   0x9F,  8 },
            {   0x14,  7 }, {  0x254, 10 }, {   0x94,  8 }, {  0x310, 10 },
            {   0x85,  8 }, {  0x311, 10 }, {   0x2A,  8 }, {   0xBB, 10 },
            {  0x18F,  9 }, {  0x255, 10 }, {   0x9D,  9 }, {   0x9F, 10 },
            {  0x18E,  9 }, {   0x44,  9 }, {   0x26,  7 }
        },
        {
            {   0x61,  7 }, {  0x12A,  9 }, {    0xD,  8 }, {  0x3BD, 10 },
            {   0x89,  8 }, {  0x109,  9 }, {  0x18E,  9 }, {  0x210, 10 },
            {  0x1D3,  9 }, {  0x211, 10 }, {   0x88,  8 }, {   0x19,  9 },
            {   0x85,  8 }, {   0x18,  9 }, {   0xE8,  8 }, {   0xCE,  8 },
            {   0x40,  7 }, {  0x119,  9 }, {   0x45,  7 }, {  0x1D2,  9 },
            {   0x4B,  7 }, {  0x1DD,  9 }, {   0x62,  7 }, {   0x94,  8 },
            {   0x75,  7 }, {    0xC,  7 }, {   0x27,  6 }, {    0xD,  7 },
            {    0x2,  5 }, {   0x26,  6 }, {    0x6,  4 }, {   0x1E,  5 },
            {    0xD,  4 }, {   0x1F,  5 }, {    0x1,  3 }, {    0xA,  4 },
            {    0x2,  3 }, {    0x7,  4 }, {    0xB,  4 }, {    0x0,  5 },
            {   0x1C,  5 }, {   0x76,  7 }, {   0x32,  6 }, {    0x7,  7 },
            {   0x24,  6 }, {   0xC0,  8 }, {    0x7,  6 }, {   0x41,  7 },
            {    0x2,  6 }, {  0x18F,  9 }, {   0x47,  7 }, {  0x1DC,  9 },
            {   0x43,  7 }, {  0x12B,  9 }, {   0xCF,  8 }, {  0x118,  9 },
            {   0xC6,  8 }, {  0x3BC, 10 }, {   0x8D,  8 }, {  0x3BF, 10 },
            {   0xC1,  8 }, {  0x3BE, 10 }, {   0x66,  7 }
        },
        {
            {    0x7,  6 }, {  0x14D,  9 }, {   0xA0,  8 }, {   0x9E,  9 },
            {   0xCF,  8 }, {  0x39C, 10 }, {   0xA1,  8 }, {  0x39D, 10 },
            {   0xAB,  8 }, {  0x1C5,  9 }, {   0x26,  7 }, {  0x14C,  9 },
            {   0x25,  7 }, {  0x19C,  9 }, {   0x3F,  7 }, {   0xE1,  8 },
            {   0x66,  7 }, {  0x1CF,  9 }, {   0x3E,  7 }, {  0x1C4,  9 },
            {   0x72,  7 }, {   0x4E,  8 }, {    0x6,  6 }, {   0xAA,  8 },
            {   0x1C,  6 }, {   0xE6,  8 }, {   0x32,  6 }, {   0x51,  7 },
            {   0x3B,  6 }, {    0x5,  6 }, {   0x1F,  5 }, {   0x18,  5 },
            {    0x2,  4 }, {   0x3A,  6 }, {    0x0,  4 }, {   0x36,  6 },
            {    0x5,  4 }, {    0x8,  5 }, {    0x8,  4 }, {   0x16,  5 },
            {    0x9,  4 }, {    0xD,  5 }, {    0x3,  4 }, {   0x2F,  6 },
            {   0x1E,  5 }, {   0x2E,  6 }, {   0x1A,  5 }, {   0x2B,  6 },
            {    0xC,  5 }, {   0x24,  7 }, {   0x1E,  6 }, {   0xE0,  8 },
            {    0x4,  6 }, {   0xA7,  8 }, {   0x54,  7 }, {  0x1C7,  9 },
            {   0x52,  7 }, {  0x19D,  9 }, {   0x3A,  7 }, {   0x9F,  9 },
            {   0x3B,  7 }, {  0x1C6,  9 }, {   0x37,  6 }
        },
        {
            {   0x2A,  6 }, {   0x39,  8 }, {   0x25,  7 }, {  0x115,  9 },
            {   0x24,  7 }, {  0x1FA,  9 }, {   0x2F,  7 }, {  0x114,  9 },
            {   0x75,  7 }, {   0x38,  8 }, {   0xFC,  8 }, {   0x36,  8 },
            {   0x1E,  7 }, {  0x1FB,  9 }, {   0x7F,  7 }, {   0x68,  7 },
            {   0x16,  6 }, {   0x37,  8 }, {   0x1F,  7 }, {   0x5C,  8 },
            {   0x13,  6 }, {   0x8B,  8 }, {    0x1,  6 }, {   0xFB,  8 },
            {   0x21,  6 }, {   0x44,  7 }, {   0x2B,  6 }, {   0x6B,  7 },
            {   0x3B,  6 }, {    0xC,  6 }, {   0x1C,  5 }, {   0x19,  5 },
            {    0x1,  4 }, {   0x20,  6 }, {   0x16,  5 }, {   0x7C,  7 },
            {    0xC,  5 }, {   0x74,  7 }, {    0xA,  5 }, {   0x1C,  6 },
            {   0x12,  5 }, {   0x69,  7 }, {    0xF,  5 }, {   0x6A,  7 },
            {   0x14,  5 }, {   0x11,  6 }, {   0x1E,  5 }, {   0x17,  5 },
            {    0x2,  4 }, {   0x31,  6 }, {   0x1B,  5 }, {   0x30,  6 },
            {    0xD,  5 }, {    0x0,  6 }, {    0x1,  5 }, {   0x1D,  7 },
            {   0x23,  6 }, {   0x1A,  7 }, {   0x1D,  6 }, {   0x5D,  8 },
            {   0x10,  6 }, {   0xFA,  8 }, {   0x13,  5 }
        },
        {
            {   0x12,  5 }, {   0x26,  7 }, {   0x41,  7 }, {   0x22,  8 },
            {   0x1A,  7 }, {   0xA9,  8 }, {   0x4C,  7 }, {  0x1B2,  9 },
            {   0x5C,  7 }, {   0xA8,  8 }, {   0x58,  7 }, {  0x1B3,  9 },
            {   0x40,  7 }, {   0x79,  8 }, {    0xC,  6 }, {   0x55,  7 },
            {   0x1F,  6 }, {   0xD8,  8 }, {   0x76,  7 }, {   0x23,  8 },
            {   0x5F,  7 }, {   0x78,  8 }, {    0xB,  6 }, {   0x1B,  7 },
            {   0x2D,  6 }, {   0x10,  7 }, {   0x37,  6 }, {   0x6D,  7 },
            {   0x32,  6 }, {    0xA,  6 }, {   0x1A,  5 }, {   0x1E,  5 },
            {   0x1F,  5 }, {   0x2B,  6 }, {    0xD,  5 }, {   0x77,  7 },
            {   0x31,  6 }, {   0x5D,  7 }, {   0x38,  6 }, {   0x27,  7 },
            {    0xC,  5 }, {   0xE9,  8 }, {   0x33,  6 }, {   0x5E,  7 },
            {   0x30,  6 }, {   0x4D,  7 }, {    0xA,  5 }, {   0x21,  6 },
            {    0x7,  5 }, {   0x3D,  7 }, {   0x39,  6 }, {   0xE8,  8 },
            {    0xB,  5 }, {   0x59,  7 }, {   0x14,  5 }, {   0x27,  6 },
            {   0x11,  5 }, {   0x75,  7 }, {    0xE,  5 }, {    0x9,  6 },
            {    0x8,  5 }, {   0x12,  6 }, {    0x0,  3 }
        }
    }
};

static const uint16_t vp4_dc_bias[16][32][2] = {
    { /* DC bias table 0 */
        {    0xC,  5 }, {   0x70,  7 }, {  0x1CA,  9 }, {  0x1CB,  9 },
        {  0x391, 10 }, { 0x1C9B, 13 }, { 0x3935, 14 }, {   0x71,  7 },
        { 0x3934, 14 }, {    0xB,  4 }, {    0xF,  4 }, {   0x19,  5 },
        {    0x2,  4 }, {    0x9,  4 }, {    0x3,  4 }, {   0x1D,  5 },
        {   0x18,  5 }, {    0x7,  4 }, {    0xD,  4 }, {    0x2,  3 },
        {    0x0,  3 }, {    0xA,  4 }, {    0x8,  4 }, {   0x1A,  6 },
        {   0x73,  7 }, {   0x6F,  8 }, {  0xE4C, 12 }, {  0x727, 11 },
        {  0x392, 10 }, {  0x390, 10 }, {   0x36,  7 }, {   0x6E,  8 }
    },
    { /* DC bias table 1 */
        {   0x11,  5 }, {   0x7A,  7 }, {   0x83,  8 }, {   0x40,  7 },
        {  0x105,  9 }, {  0x413, 11 }, {  0x410, 11 }, {   0x7B,  7 },
        {  0x822, 12 }, {    0xE,  4 }, {    0x2,  3 }, {    0x2,  4 },
        {    0x6,  4 }, {    0xA,  4 }, {    0x7,  4 }, {   0x1F,  5 },
        {   0x17,  5 }, {    0x9,  4 }, {    0xD,  4 }, {    0x0,  3 },
        {    0xC,  4 }, {    0x3,  4 }, {   0x3C,  6 }, {   0x2C,  6 },
        {   0x21,  6 }, {  0x169,  9 }, {  0x412, 11 }, {  0x2D0, 10 },
        {  0x2D1, 10 }, {  0x823, 12 }, {   0x5B,  7 }, {   0xB5,  8 }
    },
    { /* DC bias table 2 */
        {   0x17,  5 }, {   0x10,  6 }, {   0xB6,  8 }, {   0x22,  7 },
        {  0x16A,  9 }, {  0x2D0, 10 }, {  0xB48, 12 }, {   0x77,  7 },
        { 0x1692, 13 }, {    0x0,  3 }, {    0x3,  3 }, {    0x3,  4 },
        {    0x9,  4 }, {    0xC,  4 }, {    0x5,  4 }, {    0x2,  4 },
        {   0x1C,  5 }, {    0x8,  4 }, {    0xD,  4 }, {    0xF,  4 },
        {    0xA,  4 }, {    0x9,  5 }, {   0x23,  7 }, {   0x3A,  6 },
        {   0x2C,  6 }, {  0x16B,  9 }, {  0x5A5, 11 }, {  0x2D3, 10 },
        {  0x2D1, 10 }, { 0x1693, 13 }, {   0x76,  7 }, {   0xB7,  8 }
    },
    { /* DC bias table 3 */
        {   0x1E,  5 }, {   0x13,  6 }, {   0xFB,  8 }, {   0x7C,  7 },
        {   0x46,  8 }, {  0x7D6, 11 }, {  0xFA9, 12 }, {   0x12,  6 },
        { 0x1F50, 13 }, {    0x1,  3 }, {    0x4,  3 }, {    0x5,  4 },
        {    0xA,  4 }, {    0xE,  4 }, {    0x7,  4 }, {    0x0,  4 },
        {   0x17,  5 }, {    0x6,  4 }, {    0xD,  4 }, {    0xC,  4 },
        {    0x1,  4 }, {   0x2C,  6 }, {   0x8F,  9 }, {   0x3F,  6 },
        {   0x2D,  6 }, {  0x1F4,  9 }, {  0x7D5, 11 }, {   0x8E,  9 },
        {  0x7D7, 11 }, { 0x1F51, 13 }, {   0x10,  6 }, {   0x22,  7 }
    },
    { /* DC bias table 4 */
        {    0x1,  4 }, {   0x2B,  6 }, {   0x12,  7 }, {   0x55,  7 },
        {   0x27,  8 }, {  0x3B0, 10 }, {  0x762, 11 }, {   0x77,  7 },
        {  0x261, 12 }, {    0x2,  3 }, {    0x6,  3 }, {    0x7,  4 },
        {    0xB,  4 }, {    0xF,  4 }, {    0x8,  4 }, {    0x0,  4 },
        {   0x1C,  5 }, {    0x3,  4 }, {    0x9,  4 }, {    0x6,  4 },
        {   0x14,  5 }, {   0x54,  7 }, {  0x131, 11 }, {    0x5,  5 },
        {   0x3A,  6 }, {  0x1D9,  9 }, {   0x99, 10 }, {   0x4D,  9 },
        {  0x763, 11 }, {  0x260, 12 }, {    0x8,  6 }, {   0xED,  8 }
    },
    { /* DC bias table 5 */
        {    0x4,  4 }, {   0x33,  6 }, {   0x60,  7 }, {   0x65,  7 },
        {   0xC2,  8 }, {  0x30D, 10 }, {  0x619, 11 }, {   0x64,  7 },
        { 0x1862, 13 }, {    0x4,  3 }, {    0x7,  3 }, {    0xA,  4 },
        {    0xB,  4 }, {    0xD,  4 }, {    0x6,  4 }, {    0x0,  4 },
        {    0xF,  5 }, {    0x3,  4 }, {    0x5,  4 }, {    0x2,  4 },
        {    0x2,  5 }, {   0x77,  8 }, {  0xC30, 12 }, {    0x3,  5 },
        {   0x31,  6 }, {  0x187,  9 }, {  0x1D9, 10 }, {   0xED,  9 },
        {  0x1D8, 10 }, { 0x1863, 13 }, {   0x1C,  6 }, {   0x3A,  7 }
    },
    { /* DC bias table 6 */
        {    0x8,  4 }, {    0xA,  5 }, {   0x6A,  7 }, {   0x16,  6 },
        {   0x1E,  7 }, {  0x34E, 10 }, {  0x69F, 11 }, {   0x68,  7 },
        {  0xD28, 12 }, {    0x5,  3 }, {    0x7,  3 }, {    0x7,  4 },
        {    0xC,  4 }, {    0x0,  3 }, {    0x6,  4 }, {   0x1B,  5 },
        {   0x12,  5 }, {    0x2,  4 }, {    0x4,  4 }, {   0x13,  5 },
        {    0xE,  6 }, {  0x34B, 10 }, { 0x1A53, 13 }, {    0x6,  5 },
        {   0x17,  6 }, {  0x1A6,  9 }, {  0x69E, 11 }, {  0x1A4,  9 },
        {  0x695, 11 }, { 0x1A52, 13 }, {   0x6B,  7 }, {   0x1F,  7 }
    },
    { /* DC bias table 7 */
        {    0xE,  4 }, {    0xF,  5 }, {   0x17,  6 }, {   0x25,  6 },
        {   0x9F,  8 }, {  0x138,  9 }, {  0x24B, 10 }, {   0x93,  8 },
        {  0x92A, 12 }, {    0x5,  3 }, {    0x0,  2 }, {    0x8,  4 },
        {    0xD,  4 }, {    0xF,  4 }, {    0x6,  4 }, {    0x4,  4 },
        {    0xE,  5 }, {   0x19,  5 }, {   0x18,  5 }, {    0xA,  5 },
        {   0x9E,  8 }, {  0x494, 11 }, { 0x1256, 13 }, {   0x26,  6 },
        {   0x16,  6 }, {  0x124,  9 }, {  0x4E5, 11 }, {  0x273, 10 },
        {  0x4E4, 11 }, { 0x1257, 13 }, {   0x48,  7 }, {   0x9D,  8 }
    },
    { /* DC bias table 8 */
        {    0x4,  4 }, {   0x2C,  6 }, {   0x50,  7 }, {   0x1E,  7 },
        {   0x71,  9 }, {   0xE1, 10 }, {   0xE0, 10 }, {   0x1D,  7 },
        {    0x6,  6 }, {    0x7,  3 }, {    0x6,  3 }, {    0x7,  4 },
        {    0x5,  4 }, {    0x6,  4 }, {   0x15,  5 }, {    0x0,  5 },
        {   0x29,  6 }, {    0x2,  5 }, {    0x6,  5 }, {    0x1,  5 },
        {   0x23,  6 }, {   0x1F,  7 }, {   0x39,  8 }, {    0x9,  4 },
        {    0x2,  4 }, {   0x10,  5 }, {    0x7,  6 }, {   0x2D,  6 },
        {   0x2F,  6 }, {   0x2E,  6 }, {   0x22,  6 }, {   0x51,  7 }
    },
    { /* DC bias table 9 */
        {    0x8,  4 }, {   0x2F,  6 }, {   0x51,  7 }, {   0x50,  7 },
        {  0x2ED, 10 }, {  0x5D9, 11 }, {  0x5D8, 11 }, {   0xBA,  8 },
        {   0x5C,  7 }, {    0x7,  3 }, {    0x6,  3 }, {    0x9,  4 },
        {    0x6,  4 }, {    0x7,  4 }, {   0x16,  5 }, {    0x5,  5 },
        {   0x2B,  6 }, {    0x6,  5 }, {    0xA,  5 }, {    0x1,  5 },
        {    0xF,  6 }, {   0x1D,  7 }, {  0x177,  9 }, {    0x4,  4 },
        {    0x1,  4 }, {    0x4,  5 }, {    0x1,  6 }, {   0x2A,  6 },
        {    0xB,  5 }, {   0x29,  6 }, {    0x0,  6 }, {   0x1C,  7 }
    },
    { /* DC bias table 10 */
        {    0xA,  4 }, {   0x3C,  6 }, {   0x74,  7 }, {   0x4E,  7 },
        {  0x26D, 10 }, {  0x4D9, 11 }, {  0x4D8, 11 }, {   0x9A,  8 },
        {   0x4C,  7 }, {    0x0,  2 }, {    0x6,  3 }, {    0x8,  4 },
        {    0x7,  4 }, {    0x6,  4 }, {   0x16,  5 }, {    0x8,  5 },
        {   0x2E,  6 }, {    0xA,  5 }, {    0xB,  5 }, {   0x3D,  6 },
        {   0x24,  6 }, {   0xEB,  8 }, {  0x137,  9 }, {   0x1F,  5 },
        {   0x1C,  5 }, {   0x3B,  6 }, {   0x12,  6 }, {   0x25,  6 },
        {   0x2F,  6 }, {   0x13,  6 }, {   0x4F,  7 }, {   0xEA,  8 }
    },
    { /* DC bias table 11 */
        {    0xA,  4 }, {    0xA,  5 }, {    0x3,  6 }, {   0x16,  6 },
        {    0x9,  8 }, {   0x21, 10 }, {   0x20, 10 }, {   0xB3,  8 },
        {   0x58,  7 }, {    0x7,  3 }, {    0x6,  3 }, {    0x7,  4 },
        {    0x6,  4 }, {    0x4,  4 }, {   0x13,  5 }, {    0x2,  5 },
        {   0x25,  6 }, {    0x0,  5 }, {    0x3,  5 }, {   0x2D,  6 },
        {   0x5D,  7 }, {   0xB2,  8 }, {   0x11,  9 }, {    0x8,  4 },
        {    0x2,  4 }, {    0x6,  5 }, {   0x17,  6 }, {   0x2F,  6 },
        {    0x7,  5 }, {   0x24,  6 }, {   0x5C,  7 }, {    0x5,  7 }
    },
    { /* DC bias table 12 */
        {    0xB,  4 }, {   0x13,  5 }, {   0x1F,  6 }, {   0x31,  6 },
        {   0x21,  7 }, {  0x295, 10 }, {  0x528, 11 }, {   0xA4,  8 },
        {   0x3C,  7 }, {    0x0,  2 }, {    0x7,  3 }, {    0x6,  4 },
        {    0x5,  4 }, {   0x1B,  5 }, {   0x12,  5 }, {   0x32,  6 },
        {   0x1D,  6 }, {   0x2B,  6 }, {   0x30,  6 }, {   0x1C,  6 },
        {   0x3D,  7 }, {  0x14B,  9 }, {  0x529, 11 }, {    0x8,  4 },
        {   0x1A,  5 }, {   0x33,  6 }, {   0x11,  6 }, {   0x2A,  6 },
        {    0x9,  5 }, {   0x28,  6 }, {   0x53,  7 }, {   0x20,  7 }
    },
    { /* DC bias table 13 */
        {    0xE,  4 }, {   0x15,  5 }, {   0x29,  6 }, {   0x3F,  6 },
        {   0x4D,  7 }, {  0x2F1, 10 }, {  0x5E0, 11 }, {   0x92,  8 },
        {   0x48,  7 }, {    0x0,  2 }, {    0x6,  3 }, {    0x6,  4 },
        {    0x5,  4 }, {    0x4,  4 }, {    0xF,  5 }, {   0x2E,  6 },
        {   0x1D,  6 }, {   0x28,  6 }, {   0x27,  6 }, {   0x5F,  7 },
        {   0xBD,  8 }, {  0x179,  9 }, {  0x5E1, 11 }, {    0x8,  4 },
        {   0x1E,  5 }, {   0x2D,  6 }, {   0x1C,  6 }, {   0x2C,  6 },
        {   0x3E,  6 }, {   0x25,  6 }, {   0x4C,  7 }, {   0x93,  8 }
    },
    { /* DC bias table 14 */
        {    0xC,  4 }, {   0x17,  5 }, {   0x35,  6 }, {   0x13,  5 },
        {   0x21,  6 }, {   0xAD,  8 }, {  0x6F1, 11 }, {  0x1BD,  9 },
        {   0xD9,  8 }, {    0x0,  2 }, {    0x7,  3 }, {    0x7,  4 },
        {    0x6,  4 }, {    0x4,  4 }, {   0x11,  5 }, {   0x2A,  6 },
        {   0x6E,  7 }, {   0x25,  6 }, {   0x24,  6 }, {   0x57,  7 },
        {   0xD8,  8 }, {  0x379, 10 }, {  0x6F0, 11 }, {    0x5,  4 },
        {   0x16,  5 }, {   0x29,  6 }, {   0x6D,  7 }, {   0x28,  6 },
        {   0x34,  6 }, {   0x20,  6 }, {   0xDF,  8 }, {   0xAC,  8 }
    },
    { /* DC bias table 15 */
        {    0x0,  3 }, {   0x1A,  5 }, {    0x6,  5 }, {   0x19,  5 },
        {   0x30,  6 }, {   0x5A,  7 }, {  0x18A,  9 }, {  0x2DD, 10 },
        {  0x18B,  9 }, {    0x1,  2 }, {    0x7,  3 }, {    0xA,  4 },
        {    0x9,  4 }, {    0x2,  4 }, {   0x10,  5 }, {   0x2E,  6 },
        {   0x6E,  7 }, {   0x2C,  6 }, {    0xE,  6 }, {   0x5E,  7 },
        {   0xC4,  8 }, {  0x5B9, 11 }, {  0x5B8, 11 }, {   0x11,  5 },
        {   0x36,  6 }, {   0x5F,  7 }, {   0x1E,  7 }, {   0x63,  7 },
        {   0x6F,  7 }, {   0x1F,  7 }, {   0xB6,  8 }, {  0x16F,  9 }
    }
};

static const uint16_t vp4_ac_bias_0[16][32][2] = {
    { /* AC bias group 1, table 0 */
        {    0x6,  5 }, {   0x1E,  7 }, {  0x1CC,  9 }, {  0x1CE,  9 },
        {  0x734, 11 }, { 0x1CD5, 13 }, { 0x1CD4, 13 }, {   0x18,  5 },
        {  0xE6B, 12 }, {    0x0,  3 }, {    0xF,  4 }, {    0x6,  4 },
        {    0x7,  4 }, {    0xD,  4 }, {    0x8,  4 }, {    0x2,  4 },
        {   0x19,  5 }, {    0x5,  4 }, {    0xB,  4 }, {    0xA,  4 },
        {   0x1D,  5 }, {   0x27,  6 }, {  0x1CF,  9 }, {    0x4,  4 },
        {   0x38,  6 }, {    0xE,  6 }, {   0x4C,  7 }, {   0x1F,  7 },
        {   0x4D,  7 }, {  0x39B, 10 }, {   0x12,  5 }, {   0x72,  7 }
    },
    { /* AC bias group 1, table 1 */
        {    0x9,  5 }, {   0x4B,  7 }, {   0x90,  8 }, {   0x91,  8 },
        {  0x745, 11 }, { 0x1D11, 13 }, { 0x1D10, 13 }, {   0x19,  5 },
        {  0xE89, 12 }, {    0x0,  3 }, {    0xF,  4 }, {    0x8,  4 },
        {    0x7,  4 }, {    0xD,  4 }, {    0xB,  4 }, {    0x2,  4 },
        {   0x1C,  5 }, {    0x3,  4 }, {    0xA,  4 }, {    0x5,  4 },
        {   0x18,  5 }, {   0x10,  6 }, {  0x1D0,  9 }, {    0x6,  4 },
        {   0x3B,  6 }, {   0x11,  6 }, {   0x4A,  7 }, {   0x49,  7 },
        {   0xE9,  8 }, {  0x3A3, 10 }, {   0x13,  5 }, {   0x75,  7 }
    },
    { /* AC bias group 1, table 2 */
        {   0x19,  5 }, {   0x74,  7 }, {   0x1D,  8 }, {   0xEA,  8 },
        {   0x73, 10 }, {  0x1CA, 12 }, {  0x396, 13 }, {   0x1C,  5 },
        {   0xE4, 11 }, {    0x2,  3 }, {    0x1,  3 }, {    0x7,  4 },
        {    0x8,  4 }, {    0xD,  4 }, {    0x9,  4 }, {   0x1F,  5 },
        {   0x18,  5 }, {    0x0,  4 }, {    0x6,  4 }, {   0x1E,  5 },
        {   0x3B,  6 }, {   0xEB,  8 }, {  0x397, 13 }, {    0xA,  4 },
        {    0x2,  5 }, {   0x2C,  6 }, {   0x5B,  7 }, {   0x5A,  7 },
        {    0xF,  7 }, {   0x38,  9 }, {   0x17,  5 }, {    0x6,  6 }
    },
    { /* AC bias group 1, table 3 */
        {   0x1E,  5 }, {   0x6F,  7 }, {   0xAE,  8 }, {   0xAF,  8 },
        {  0x187, 10 }, {  0x61B, 12 }, {  0xC35, 13 }, {   0x1A,  5 },
        {  0x30C, 11 }, {    0x2,  3 }, {    0x1,  3 }, {    0x7,  4 },
        {    0x8,  4 }, {    0xE,  4 }, {    0x9,  4 }, {   0x1F,  5 },
        {   0x14,  5 }, {    0x0,  4 }, {    0x1,  4 }, {   0x19,  5 },
        {   0x2A,  6 }, {   0x60,  8 }, {  0xC34, 13 }, {    0xB,  4 },
        {    0xD,  5 }, {   0x36,  6 }, {   0x6E,  7 }, {   0x56,  7 },
        {   0x31,  7 }, {   0xC2,  9 }, {   0x18,  5 }, {   0x19,  6 }
    },
    { /* AC bias group 1, table 4 */
        {    0x1,  4 }, {   0x2C,  6 }, {    0x5,  7 }, {   0x15,  7 },
        {    0x8,  8 }, {   0x97, 12 }, {  0x12D, 13 }, {   0x17,  5 },
        {   0x4A, 11 }, {    0x3,  3 }, {    0x2,  3 }, {    0x9,  4 },
        {    0xA,  4 }, {    0xE,  4 }, {    0x8,  4 }, {   0x1F,  5 },
        {    0x7,  5 }, {   0x1E,  5 }, {   0x1B,  5 }, {    0x4,  5 },
        {   0x5A,  7 }, {   0x24, 10 }, {  0x12C, 13 }, {    0xC,  4 },
        {    0x6,  5 }, {    0x0,  5 }, {    0x3,  6 }, {   0x5B,  7 },
        {   0x14,  7 }, {   0x13,  9 }, {   0x1A,  5 }, {    0xB,  6 }
    },
    { /* AC bias group 1, table 5 */
        {    0x4,  4 }, {    0x0,  5 }, {   0x17,  7 }, {   0x63,  7 },
        {  0x18B,  9 }, {  0x310, 10 }, {  0xC44, 12 }, {   0x19,  5 },
        {  0x623, 11 }, {    0x4,  3 }, {    0x3,  3 }, {    0xA,  4 },
        {    0xB,  4 }, {    0xD,  4 }, {    0x3,  4 }, {   0x1C,  5 },
        {    0x3,  5 }, {    0xA,  5 }, {    0x4,  5 }, {    0x3,  6 },
        {  0x18A,  9 }, { 0x188B, 13 }, { 0x188A, 13 }, {    0xF,  4 },
        {    0xB,  5 }, {    0x2,  5 }, {    0xA,  6 }, {    0x2,  6 },
        {   0x16,  7 }, {  0x189,  9 }, {   0x1D,  5 }, {   0x30,  6 }
    },
    { /* AC bias group 1, table 6 */
        {    0xD,  4 }, {    0x3,  5 }, {   0x77,  7 }, {    0xD,  6 },
        {   0x82,  8 }, {  0x20D, 10 }, {  0x830, 12 }, {   0x19,  5 },
        {  0x419, 11 }, {    0x3,  3 }, {    0x2,  3 }, {    0xA,  4 },
        {    0x9,  4 }, {    0xB,  4 }, {    0x2,  4 }, {   0x11,  5 },
        {   0x39,  6 }, {    0x2,  5 }, {   0x21,  6 }, {   0x40,  7 },
        { 0x1063, 13 }, { 0x20C5, 14 }, { 0x20C4, 14 }, {    0xF,  4 },
        {   0x18,  5 }, {    0x7,  5 }, {   0x38,  6 }, {    0xC,  6 },
        {   0x76,  7 }, {  0x107,  9 }, {    0x0,  4 }, {   0x3A,  6 }
    },
    { /* AC bias group 1, table 7 */
        {    0xF,  4 }, {   0x1C,  5 }, {   0x36,  6 }, {    0x8,  5 },
        {   0x61,  7 }, {   0x91,  8 }, {  0x243, 10 }, {    0x9,  5 },
        {  0x120,  9 }, {    0x5,  3 }, {    0x3,  3 }, {    0x8,  4 },
        {    0x5,  4 }, {    0x1,  4 }, {   0x13,  5 }, {   0x31,  6 },
        {   0x76,  7 }, {   0x60,  7 }, {   0x93,  8 }, {  0x909, 12 },
        {  0x908, 12 }, {  0x90B, 12 }, {  0x90A, 12 }, {    0x1,  3 },
        {   0x1A,  5 }, {   0x19,  5 }, {   0x3A,  6 }, {   0x25,  6 },
        {   0x77,  7 }, {   0x92,  8 }, {    0x0,  4 }, {   0x37,  6 }
    },
    { /* AC bias group 1, table 8 */
        {   0x1F,  5 }, {   0x79,  7 }, {   0xF1,  8 }, {   0xF0,  8 },
        {  0x11B, 10 }, {  0x469, 12 }, {  0x468, 12 }, {   0x3B,  6 },
        {   0x22,  7 }, {    0x5,  3 }, {    0x4,  3 }, {    0x7,  4 },
        {    0x5,  4 }, {    0x6,  4 }, {   0x1C,  5 }, {    0x1,  5 },
        {   0x35,  6 }, {   0x3D,  6 }, {   0x3A,  6 }, {   0x10,  6 },
        {   0x47,  8 }, {   0x8C,  9 }, {  0x235, 11 }, {    0x1,  3 },
        {    0x1,  4 }, {   0x19,  5 }, {    0x0,  5 }, {   0x30,  6 },
        {    0x9,  5 }, {   0x31,  6 }, {   0x1B,  5 }, {   0x34,  6 }
    },
    { /* AC bias group 1, table 9 */
        {    0x3,  4 }, {   0x1B,  6 }, {   0xF3,  8 }, {   0xFD,  8 },
        {  0x3C9, 10 }, {  0xF20, 12 }, { 0x1E42, 13 }, {   0x3D,  6 },
        {   0xFC,  8 }, {    0x6,  3 }, {    0x4,  3 }, {    0x2,  4 },
        {    0x0,  4 }, {    0x1,  4 }, {   0x17,  5 }, {   0x3E,  6 },
        {   0x1A,  6 }, {   0x39,  6 }, {   0x2B,  6 }, {   0x78,  7 },
        {  0x1E5,  9 }, {  0x791, 11 }, { 0x1E43, 13 }, {    0x2,  3 },
        {    0x7,  4 }, {   0x1D,  5 }, {    0xC,  5 }, {   0x38,  6 },
        {   0x14,  5 }, {   0x7F,  7 }, {   0x16,  5 }, {   0x2A,  6 }
    },
    { /* AC bias group 1, table 10 */
        {    0x7,  4 }, {   0x39,  6 }, {   0x51,  7 }, {   0x78,  7 },
        {  0x3CB, 10 }, {  0xF29, 12 }, { 0x1E51, 13 }, {   0x3D,  6 },
        {   0xF3,  8 }, {    0x6,  3 }, {    0x4,  3 }, {    0x2,  4 },
        {    0x0,  4 }, {    0x1,  4 }, {   0x17,  5 }, {   0x3E,  6 },
        {   0x7F,  7 }, {   0x2B,  6 }, {   0x7E,  7 }, {   0x50,  7 },
        {  0x1E4,  9 }, {  0x795, 11 }, { 0x1E50, 13 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x1D,  5 }, {    0x6,  5 }, {   0x38,  6 },
        {    0x7,  5 }, {   0x29,  6 }, {   0x16,  5 }, {   0x2A,  6 }
    },
    { /* AC bias group 1, table 11 */
        {    0x8,  4 }, {   0x3B,  6 }, {   0x1D,  7 }, {   0x72,  7 },
        {  0x1CC,  9 }, {  0x734, 11 }, { 0x1CD5, 13 }, {   0x3A,  6 },
        {   0x1C,  7 }, {    0x6,  3 }, {    0x5,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {    0x0,  4 }, {   0x12,  5 }, {   0x3E,  6 },
        {   0x7F,  7 }, {   0x1E,  6 }, {   0x7E,  7 }, {   0xE7,  8 },
        {  0x39B, 10 }, {  0xE6B, 12 }, { 0x1CD4, 13 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x1E,  5 }, {    0xE,  5 }, {   0x38,  6 },
        {    0x6,  5 }, {    0xF,  6 }, {   0x13,  5 }, {   0x1F,  6 }
    },
    { /* AC bias group 1, table 12 */
        {    0xD,  4 }, {   0x3F,  6 }, {   0x73,  7 }, {    0xC,  6 },
        {   0xE4,  8 }, {  0x72B, 11 }, {  0xE54, 12 }, {   0x3A,  6 },
        {   0x1A,  7 }, {    0x5,  3 }, {    0x4,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {    0x0,  4 }, {    0x7,  5 }, {   0x38,  6 },
        {   0x76,  7 }, {   0x77,  7 }, {   0x1B,  7 }, {  0x1CB,  9 },
        {  0x394, 10 }, { 0x1CAB, 13 }, { 0x1CAA, 13 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x1E,  5 }, {    0xE,  5 }, {   0x3E,  6 },
        {   0x19,  5 }, {   0x1F,  6 }, {   0x18,  5 }, {   0x1E,  6 }
    },
    { /* AC bias group 1, table 13 */
        {    0xE,  4 }, {    0x7,  5 }, {    0xC,  6 }, {   0x1C,  6 },
        {   0xBD,  8 }, {  0x2F3, 10 }, {  0xBC9, 12 }, {   0x1F,  6 },
        {   0xBF,  8 }, {    0x6,  3 }, {    0x4,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {   0x1E,  5 }, {    0x1,  5 }, {    0xD,  6 },
        {   0x3A,  7 }, {   0x3B,  7 }, {   0xBE,  8 }, {  0x178,  9 },
        {  0x5E5, 11 }, { 0x1791, 13 }, { 0x1790, 13 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x1F,  5 }, {   0x16,  5 }, {    0x0,  5 },
        {   0x15,  5 }, {   0x2E,  6 }, {   0x14,  5 }, {   0x1E,  6 }
    },
    { /* AC bias group 1, table 14 */
        {    0x0,  3 }, {   0x1B,  5 }, {   0x31,  6 }, {   0x3A,  6 },
        {   0x60,  7 }, {   0x6F,  9 }, {  0x1B9, 11 }, {    0xE,  6 },
        {   0x1A,  7 }, {    0x5,  3 }, {    0x3,  3 }, {    0x2,  4 },
        {   0x1F,  5 }, {   0x1A,  5 }, {   0x39,  6 }, {    0xC,  6 },
        {   0xC3,  8 }, {   0xC2,  8 }, {   0x36,  8 }, {   0xDD, 10 },
        {  0x370, 12 }, {  0x6E3, 13 }, {  0x6E2, 13 }, {    0x2,  3 },
        {    0x8,  4 }, {   0x1E,  5 }, {   0x19,  5 }, {   0x3B,  6 },
        {   0x12,  5 }, {    0xF,  6 }, {   0x13,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 1, table 15 */
        {    0x2,  3 }, {    0x0,  4 }, {    0x3,  5 }, {   0x1C,  5 },
        {   0x32,  6 }, {   0x1C,  7 }, {  0x199,  9 }, {    0x4,  6 },
        {   0xCD,  8 }, {    0x4,  3 }, {    0x3,  3 }, {   0x1B,  5 },
        {   0x1A,  5 }, {   0x3D,  6 }, {   0x67,  7 }, {   0x3B,  8 },
        {  0x198,  9 }, {   0x75,  9 }, {   0xE9, 10 }, {  0x3A1, 12 },
        {  0x3A0, 12 }, {  0x3A3, 12 }, {  0x3A2, 12 }, {    0x5,  3 },
        {    0x2,  4 }, {   0x1F,  5 }, {   0x1D,  5 }, {   0x3C,  6 },
        {   0x18,  5 }, {    0xF,  6 }, {    0x6,  5 }, {    0x5,  6 }
    }
};

static const uint16_t vp4_ac_bias_1[16][32][2] = {
    { /* AC bias group 2, table 0 */
        {    0x4,  5 }, {   0xF5,  8 }, {  0x182,  9 }, {  0x60F, 11 },
        { 0x1839, 13 }, { 0x1838, 13 }, { 0x183B, 13 }, {   0x13,  5 },
        {   0xC0,  8 }, {    0x3,  3 }, {    0x2,  3 }, {    0xB,  4 },
        {    0xA,  4 }, {    0xE,  4 }, {    0x8,  4 }, {    0x1,  4 },
        {   0x12,  5 }, {   0x1F,  5 }, {    0x0,  4 }, {    0x6,  5 },
        {   0x7B,  7 }, {  0x306, 10 }, { 0x183A, 13 }, {    0xD,  4 },
        {    0x7,  5 }, {   0x31,  6 }, {    0xA,  6 }, {   0x61,  7 },
        {   0x3C,  6 }, {   0xF4,  8 }, {   0x19,  5 }, {    0xB,  6 }
    },
    { /* AC bias group 2, table 1 */
        {    0xA,  5 }, {   0x1A,  7 }, {  0x1D8,  9 }, {  0x3B3, 10 },
        {  0xECA, 12 }, { 0x1D96, 13 }, { 0x3B2F, 14 }, {   0x14,  5 },
        {   0x36,  8 }, {    0x4,  3 }, {    0x3,  3 }, {    0xC,  4 },
        {    0xB,  4 }, {    0x0,  3 }, {    0x4,  4 }, {   0x1C,  5 },
        {    0x5,  5 }, {   0x15,  5 }, {    0x7,  5 }, {   0x17,  6 },
        {   0x37,  8 }, {  0x764, 11 }, { 0x3B2E, 14 }, {    0xF,  4 },
        {   0x1A,  5 }, {   0x3A,  6 }, {    0xC,  6 }, {   0x77,  7 },
        {    0x4,  5 }, {   0xED,  8 }, {   0x1B,  5 }, {   0x16,  6 }
    },
    { /* AC bias group 2, table 2 */
        {   0x1A,  5 }, {   0x2D,  7 }, {   0x58,  8 }, {  0x1F4,  9 },
        {  0x7D4, 11 }, { 0x1F55, 13 }, { 0x1F54, 13 }, {   0x14,  5 },
        {   0x59,  8 }, {    0x4,  3 }, {    0x3,  3 }, {    0xB,  4 },
        {    0xC,  4 }, {    0xE,  4 }, {    0x4,  4 }, {   0x15,  5 },
        {    0x5,  5 }, {    0x7,  5 }, {    0x4,  5 }, {   0x7C,  7 },
        {  0x3EB, 10 }, { 0x1F57, 13 }, { 0x1F56, 13 }, {    0x0,  3 },
        {   0x1B,  5 }, {   0x3F,  6 }, {    0xD,  6 }, {    0xC,  6 },
        {    0xA,  5 }, {   0xFB,  8 }, {   0x1E,  5 }, {   0x17,  6 }
    },
    { /* AC bias group 2, table 3 */
        {    0x0,  4 }, {   0x75,  7 }, {   0x4A,  8 }, {   0x97,  9 },
        {  0x25B, 11 }, {  0x969, 13 }, {  0x968, 13 }, {    0xB,  5 },
        {   0xE8,  8 }, {    0x5,  3 }, {    0x4,  3 }, {    0x7,  4 },
        {    0xC,  4 }, {    0xD,  4 }, {    0x1,  4 }, {    0xA,  5 },
        {   0x39,  6 }, {   0x3B,  6 }, {   0x18,  6 }, {   0xE9,  8 },
        {  0x12C, 10 }, {  0x96B, 13 }, {  0x96A, 13 }, {    0x1,  3 },
        {   0x1F,  5 }, {    0x8,  5 }, {   0x19,  6 }, {   0x13,  6 },
        {    0xD,  5 }, {   0x24,  7 }, {   0x1E,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 2, table 4 */
        {    0x4,  4 }, {   0x14,  6 }, {   0x6E,  8 }, {   0x57,  8 },
        {  0x159, 10 }, {  0x562, 12 }, {  0xAC7, 13 }, {    0xB,  5 },
        {   0x6F,  8 }, {    0x6,  3 }, {    0x5,  3 }, {    0x8,  4 },
        {    0x9,  4 }, {    0x7,  4 }, {   0x1E,  5 }, {    0x2,  5 },
        {    0x7,  6 }, {    0x6,  6 }, {   0x2A,  7 }, {   0xAD,  9 },
        {  0xAC6, 13 }, {  0x561, 12 }, {  0x560, 12 }, {    0x1,  3 },
        {   0x1F,  5 }, {    0xC,  5 }, {   0x39,  6 }, {   0x1A,  6 },
        {    0x0,  4 }, {   0x36,  7 }, {   0x1D,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 2, table 5 */
        {    0x7,  4 }, {   0x1B,  6 }, {    0xE,  7 }, {    0xD,  7 },
        {  0x3E1, 10 }, { 0x1F06, 13 }, { 0x3E0F, 14 }, {    0x2,  5 },
        {   0xF9,  8 }, {    0x5,  3 }, {    0x6,  3 }, {    0x8,  4 },
        {    0x9,  4 }, {    0x4,  4 }, {    0xC,  5 }, {   0x1A,  6 },
        {    0xF,  7 }, {    0xC,  7 }, {  0x1F1,  9 }, {  0x7C0, 11 },
        { 0x3E0E, 14 }, { 0x1F05, 13 }, { 0x1F04, 13 }, {    0x1,  3 },
        {    0x0,  4 }, {   0x1C,  5 }, {   0x3F,  6 }, {   0x3D,  6 },
        {    0x5,  4 }, {   0x7D,  7 }, {   0x1D,  5 }, {   0x3C,  6 }
    },
    { /* AC bias group 2, table 6 */
        {    0xF,  4 }, {    0xA,  5 }, {   0x71,  7 }, {    0x6,  6 },
        {  0x1C2,  9 }, {  0x702, 11 }, { 0x1C0E, 13 }, {    0x2,  5 },
        {    0xE,  7 }, {    0x5,  3 }, {    0x4,  3 }, {    0x6,  4 },
        {    0x7,  4 }, {   0x1D,  5 }, {   0x17,  6 }, {    0xF,  7 },
        {  0x1C3,  9 }, {  0x1C1,  9 }, {  0x380, 10 }, { 0x381F, 14 },
        { 0x381E, 14 }, { 0x1C0D, 13 }, { 0x1C0C, 13 }, {    0x1,  3 },
        {    0x4,  4 }, {   0x18,  5 }, {    0x1,  5 }, {    0x0,  5 },
        {    0xD,  4 }, {   0x16,  6 }, {   0x19,  5 }, {   0x39,  6 }
    },
    { /* AC bias group 2, table 7 */
        {    0x2,  3 }, {   0x1E,  5 }, {   0x3B,  6 }, {    0xD,  5 },
        {   0x61,  7 }, {  0x1FA,  9 }, { 0x1FB5, 13 }, {   0x31,  6 },
        {   0xFC,  8 }, {    0x4,  3 }, {    0x5,  3 }, {    0x1,  4 },
        {    0x7,  4 }, {   0x3A,  6 }, {   0x60,  7 }, {  0x3F7, 10 },
        {  0x7EC, 11 }, { 0x1FB7, 13 }, { 0x3F6C, 14 }, { 0x7EDB, 15 },
        { 0x7EDA, 15 }, { 0x3F69, 14 }, { 0x3F68, 14 }, {    0x1,  3 },
        {    0x0,  4 }, {   0x19,  5 }, {   0x3E,  6 }, {   0x39,  6 },
        {    0xD,  4 }, {   0x38,  6 }, {    0xC,  5 }, {   0x7F,  7 }
    },
    { /* AC bias group 2, table 8 */
        {   0x1E,  5 }, {   0x70,  7 }, {  0x127,  9 }, {  0x126,  9 },
        {  0x492, 11 }, { 0x124D, 13 }, { 0x124C, 13 }, {    0x1,  5 },
        {   0x7F,  7 }, {    0x6,  3 }, {    0x5,  3 }, {    0x5,  4 },
        {    0x4,  4 }, {    0x1,  4 }, {    0x7,  5 }, {   0x25,  6 },
        {   0x71,  7 }, {   0x7E,  7 }, {   0x48,  7 }, {  0x125,  9 },
        {  0x248, 10 }, { 0x124F, 13 }, { 0x124E, 13 }, {    0x3,  3 },
        {    0x8,  4 }, {   0x1D,  5 }, {    0x6,  5 }, {   0x3E,  6 },
        {    0x2,  4 }, {    0x0,  5 }, {   0x13,  5 }, {   0x39,  6 }
    },
    { /* AC bias group 2, table 9 */
        {    0x1,  4 }, {    0x1,  6 }, {   0xE7,  8 }, {   0x91,  8 },
        {  0x240, 10 }, { 0x120D, 13 }, { 0x120C, 13 }, {   0x3C,  6 },
        {    0x0,  6 }, {    0x6,  3 }, {    0x5,  3 }, {    0x5,  4 },
        {    0x4,  4 }, {   0x1F,  5 }, {    0x4,  5 }, {   0x25,  6 },
        {   0x72,  7 }, {   0x49,  7 }, {   0xE6,  8 }, {  0x121,  9 },
        {  0x482, 11 }, { 0x120F, 13 }, { 0x120E, 13 }, {    0x3,  3 },
        {    0x8,  4 }, {   0x1D,  5 }, {    0x5,  5 }, {   0x3D,  6 },
        {    0x3,  4 }, {    0x1,  5 }, {   0x13,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 2, table 10 */
        {    0x4,  4 }, {    0xF,  6 }, {   0xF4,  8 }, {   0x5B,  8 },
        {  0x2D3, 11 }, {  0xB4A, 13 }, { 0x1697, 14 }, {   0x3C,  6 },
        {    0xE,  6 }, {    0x6,  3 }, {    0x5,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {   0x1D,  5 }, {    0x0,  5 }, {   0x7B,  7 },
        {   0x2C,  7 }, {   0xF5,  8 }, {   0xB5,  9 }, {  0x168, 10 },
        { 0x1696, 14 }, {  0xB49, 13 }, {  0xB48, 13 }, {    0x3,  3 },
        {    0x9,  4 }, {   0x1F,  5 }, {    0xA,  5 }, {    0x1,  5 },
        {    0x8,  4 }, {    0x6,  5 }, {   0x1C,  5 }, {   0x17,  6 }
    },
    { /* AC bias group 2, table 11 */
        {    0x8,  4 }, {   0x39,  6 }, {   0x1A,  7 }, {    0x3,  7 },
        {   0xDB, 10 }, {  0x6D6, 13 }, {  0xDAF, 14 }, {   0x3C,  6 },
        {    0xC,  6 }, {    0x6,  3 }, {    0x5,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {   0x1D,  5 }, {   0x3D,  6 }, {    0x0,  6 },
        {    0x2,  7 }, {   0x37,  8 }, {   0x6C,  9 }, {  0x1B4, 11 },
        {  0xDAE, 14 }, {  0x6D5, 13 }, {  0x6D4, 13 }, {    0x2,  3 },
        {    0x7,  4 }, {   0x1F,  5 }, {    0x7,  5 }, {    0x1,  5 },
        {    0x9,  4 }, {    0xD,  5 }, {    0xC,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 2, table 12 */
        {    0xF,  4 }, {    0x4,  5 }, {   0x2F,  7 }, {   0x2E,  7 },
        {   0x54,  9 }, {  0x555, 13 }, {  0x554, 13 }, {   0x16,  6 },
        {    0xE,  6 }, {    0x6,  3 }, {    0x5,  3 }, {    0x1,  4 },
        {    0x0,  4 }, {    0x9,  5 }, {    0xB,  6 }, {   0x14,  7 },
        {   0x57,  9 }, {   0x56,  9 }, {   0xAB, 10 }, {  0x557, 13 },
        {  0x556, 13 }, {  0x2A9, 12 }, {  0x2A8, 12 }, {    0x3,  3 },
        {    0x8,  4 }, {   0x13,  5 }, {    0xA,  5 }, {    0x8,  5 },
        {    0xE,  4 }, {   0x12,  5 }, {    0x6,  5 }, {    0xF,  6 }
    },
    { /* AC bias group 2, table 13 */
        {    0x1,  3 }, {    0xE,  5 }, {    0x6,  6 }, {    0x4,  6 },
        {   0xDA,  9 }, {  0xDBE, 13 }, { 0x1B7E, 14 }, {    0x7,  6 },
        {   0x1A,  6 }, {    0x5,  3 }, {    0x4,  3 }, {   0x1C,  5 },
        {   0x1B,  5 }, {   0x3A,  6 }, {   0x37,  7 }, {   0x6C,  8 },
        {  0x1B6, 10 }, {  0x36E, 11 }, {  0xDBD, 13 }, { 0x36FF, 15 },
        { 0x36FE, 15 }, { 0x1B79, 14 }, { 0x1B78, 14 }, {    0x2,  3 },
        {    0xC,  4 }, {    0x0,  4 }, {    0xF,  5 }, {    0xC,  5 },
        {    0xF,  4 }, {   0x1A,  5 }, {   0x3B,  6 }, {    0x5,  6 }
    },
    { /* AC bias group 2, table 14 */
        {    0x5,  3 }, {   0x1E,  5 }, {   0x3A,  6 }, {   0x3E,  6 },
        {   0xFC,  8 }, {  0xFD7, 12 }, { 0x3F55, 14 }, {   0x77,  7 },
        {   0x30,  6 }, {    0x3,  3 }, {    0x4,  3 }, {   0x1A,  5 },
        {   0x19,  5 }, {   0x7F,  7 }, {  0x1FB,  9 }, {  0x3F4, 10 },
        {  0xFD6, 12 }, { 0x1FA9, 13 }, { 0x3F54, 14 }, { 0x3F57, 14 },
        { 0x3F56, 14 }, { 0x3F51, 14 }, { 0x3F50, 14 }, {    0x1,  3 },
        {    0x4,  4 }, {   0x1C,  5 }, {    0xB,  5 }, {    0xA,  5 },
        {    0x0,  3 }, {   0x1B,  5 }, {   0x31,  6 }, {   0x76,  7 }
    },
    { /* AC bias group 2, table 15 */
        {    0x5,  3 }, {    0xC,  4 }, {   0x1B,  5 }, {    0x8,  4 },
        {   0x38,  6 }, {   0x15,  8 }, {   0xA3, 11 }, {   0xE6,  8 },
        {    0x4,  6 }, {    0x1,  3 }, {    0x2,  3 }, {   0x12,  5 },
        {    0x3,  5 }, {    0xB,  7 }, {   0x29,  9 }, {   0xA0, 11 },
        {  0x142, 12 }, {  0x287, 13 }, {  0x286, 13 }, {  0x289, 13 },
        {  0x288, 13 }, {  0x28B, 13 }, {  0x28A, 13 }, {    0xF,  4 },
        {   0x1D,  5 }, {   0x13,  5 }, {    0x1,  5 }, {    0x0,  5 },
        {    0x3,  3 }, {   0x1A,  5 }, {   0x72,  7 }, {   0xE7,  8 }
    }
};

static const uint16_t vp4_ac_bias_2[16][32][2] = {
    { /* AC bias group 3, table 0 */
        {    0x9,  5 }, {   0x15,  7 }, {   0x28,  8 }, {   0x52,  9 },
        {  0x29A, 12 }, {  0x537, 13 }, {  0x536, 13 }, {    0xA,  5 },
        {   0x54,  7 }, {    0x4,  3 }, {    0x3,  3 }, {    0xC,  4 },
        {    0xB,  4 }, {    0xD,  4 }, {    0x3,  4 }, {   0x14,  5 },
        {   0x3A,  6 }, {    0x4,  5 }, {   0x38,  6 }, {   0x55,  7 },
        {   0xA7, 10 }, {  0x299, 12 }, {  0x298, 12 }, {    0x0,  3 },
        {   0x1E,  5 }, {    0x8,  5 }, {   0x2B,  6 }, {    0xB,  6 },
        {    0xB,  5 }, {   0x3B,  6 }, {   0x1F,  5 }, {   0x39,  6 }
    },
    { /* AC bias group 3, table 1 */
        {   0x1D,  5 }, {   0x2F,  7 }, {    0x2,  8 }, {    0x7,  9 },
        {   0x19, 11 }, {   0x35, 12 }, {   0x34, 12 }, {    0x9,  5 },
        {   0x2E,  7 }, {    0x6,  3 }, {    0x5,  3 }, {    0x9,  4 },
        {    0x8,  4 }, {    0x7,  4 }, {   0x1F,  5 }, {    0x8,  5 },
        {   0x18,  6 }, {   0x19,  6 }, {    0x1,  6 }, {    0x0,  7 },
        {   0x18, 11 }, {   0x37, 12 }, {   0x36, 12 }, {    0x1,  3 },
        {    0x1,  4 }, {    0xA,  5 }, {   0x39,  6 }, {   0x16,  6 },
        {    0xD,  5 }, {    0x1,  5 }, {   0x1E,  5 }, {   0x38,  6 }
    },
    { /* AC bias group 3, table 2 */
        {    0x1,  4 }, {   0x71,  7 }, {   0xE0,  8 }, {  0x1C3,  9 },
        {  0x708, 11 }, { 0x1C26, 13 }, { 0x384F, 14 }, {    0x1,  5 },
        {   0x31,  7 }, {    0x6,  3 }, {    0x5,  3 }, {    0x9,  4 },
        {    0x8,  4 }, {    0x5,  4 }, {    0xF,  5 }, {   0x39,  6 },
        {   0x77,  7 }, {   0x76,  7 }, {   0x30,  7 }, {  0x385, 10 },
        { 0x384E, 14 }, { 0x1C25, 13 }, { 0x1C24, 13 }, {    0x1,  3 },
        {    0x4,  4 }, {    0xD,  5 }, {    0x0,  5 }, {   0x19,  6 },
        {   0x1F,  5 }, {    0xE,  5 }, {   0x1E,  5 }, {   0x3A,  6 }
    },
    { /* AC bias group 3, table 3 */
        {    0x6,  4 }, {    0xC,  6 }, {   0xD6,  8 }, {   0x7B,  8 },
        {  0x1E8, 10 }, {  0x7A4, 12 }, {  0xF4B, 13 }, {   0x36,  6 },
        {   0x6A,  7 }, {    0x7,  3 }, {    0x5,  3 }, {    0x8,  4 },
        {    0x9,  4 }, {    0x1,  4 }, {    0x7,  5 }, {    0xD,  6 },
        {   0x3C,  7 }, {   0xD7,  8 }, {   0xF5,  9 }, {  0x7A7, 12 },
        {  0xF4A, 13 }, {  0xF4D, 13 }, {  0xF4C, 13 }, {    0x2,  3 },
        {    0x2,  4 }, {    0xE,  5 }, {   0x37,  6 }, {   0x34,  6 },
        {    0x0,  4 }, {   0x19,  5 }, {   0x18,  5 }, {   0x1F,  6 }
    },
    { /* AC bias group 3, table 4 */
        {    0xA,  4 }, {   0x27,  6 }, {   0xBF,  8 }, {   0xBE,  8 },
        {  0x224, 10 }, { 0x225D, 14 }, { 0x225C, 14 }, {   0x26,  6 },
        {   0x5E,  7 }, {    0x7,  3 }, {    0x6,  3 }, {    0x6,  4 },
        {    0x7,  4 }, {   0x16,  5 }, {   0x2E,  6 }, {   0x45,  7 },
        {   0x88,  8 }, {  0x113,  9 }, {  0x44A, 11 }, { 0x225F, 14 },
        { 0x225E, 14 }, { 0x112D, 13 }, { 0x112C, 13 }, {    0x2,  3 },
        {    0x2,  4 }, {   0x12,  5 }, {    0x3,  5 }, {    0x2,  5 },
        {    0x3,  4 }, {    0x0,  4 }, {   0x10,  5 }, {   0x23,  6 }
    },
    { /* AC bias group 3, table 5 */
        {    0xF,  4 }, {    0x6,  5 }, {   0x75,  7 }, {   0x74,  7 },
        {    0xA,  9 }, {   0xBF, 13 }, {   0xB9, 13 }, {   0x22,  6 },
        {    0x3,  7 }, {    0x5,  3 }, {    0x6,  3 }, {    0x1,  4 },
        {    0x2,  4 }, {    0x7,  5 }, {    0x0,  6 }, {    0x4,  8 },
        {   0x16, 10 }, {   0x5E, 12 }, {   0xB8, 13 }, {   0xBB, 13 },
        {   0xBA, 13 }, {  0x17D, 14 }, {  0x17C, 14 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x1C,  5 }, {   0x10,  5 }, {   0x3B,  6 },
        {    0x9,  4 }, {    0x7,  4 }, {    0x1,  5 }, {   0x23,  6 }
    },
    { /* AC bias group 3, table 6 */
        {    0x1,  3 }, {   0x1C,  5 }, {   0x36,  6 }, {   0x3B,  6 },
        {   0xEA,  8 }, {  0x75B, 11 }, { 0x1D65, 13 }, {   0x19,  6 },
        {   0x74,  7 }, {    0x4,  3 }, {    0x5,  3 }, {    0x0,  4 },
        {    0x1,  4 }, {   0x37,  6 }, {  0x1D7,  9 }, {  0x75A, 11 },
        { 0x1D64, 13 }, { 0x1D67, 13 }, { 0x1D66, 13 }, { 0x1D61, 13 },
        { 0x1D60, 13 }, { 0x1D63, 13 }, { 0x1D62, 13 }, {    0x2,  3 },
        {   0x1F,  5 }, {   0x1A,  5 }, {    0xD,  5 }, {   0x3D,  6 },
        {    0xC,  4 }, {    0x7,  4 }, {   0x3C,  6 }, {   0x18,  6 }
    },
    { /* AC bias group 3, table 7 */
        {    0x2,  3 }, {    0x1,  4 }, {   0x14,  5 }, {    0x0,  4 },
        {   0x2F,  6 }, {   0xBB,  8 }, {  0x2E4, 10 }, {   0x7D,  7 },
        {   0xBA,  8 }, {    0x3,  3 }, {    0x4,  3 }, {   0x16,  5 },
        {   0x1A,  5 }, {   0xB8,  8 }, { 0x172E, 13 }, { 0x2E5F, 14 },
        { 0x2E5E, 14 }, { 0x1729, 13 }, { 0x1728, 13 }, { 0x172B, 13 },
        { 0x172A, 13 }, { 0x172D, 13 }, { 0x172C, 13 }, {    0x1,  3 },
        {   0x1E,  5 }, {   0x15,  5 }, {   0x1B,  5 }, {   0x3F,  6 },
        {    0xC,  4 }, {    0xE,  4 }, {   0x7C,  7 }, {  0x173,  9 }
    },
    { /* AC bias group 3, table 8 */
        {    0x3,  4 }, {   0x7B,  7 }, {   0x58,  8 }, {  0x1EA,  9 },
        { 0x1EB1, 13 }, { 0x1EB0, 13 }, { 0x1EB3, 13 }, {   0x13,  6 },
        {   0x12,  6 }, {    0x5,  3 }, {    0x6,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {   0x13,  5 }, {   0x3C,  6 }, {   0x2D,  7 },
        {   0xF4,  8 }, {   0x59,  8 }, {  0x3D7, 10 }, {  0xF5B, 12 },
        { 0x1EB2, 13 }, { 0x1EB5, 13 }, { 0x1EB4, 13 }, {    0x3,  3 },
        {    0xE,  4 }, {   0x1F,  5 }, {   0x12,  5 }, {    0x8,  5 },
        {    0x8,  4 }, {    0x0,  4 }, {    0xA,  5 }, {   0x17,  6 }
    },
    { /* AC bias group 3, table 9 */
        {    0x8,  4 }, {   0x3C,  6 }, {   0xF5,  8 }, {   0xF4,  8 },
        { 0x1EF7, 13 }, { 0x3DE9, 14 }, { 0x3DE8, 14 }, {   0x1C,  6 },
        {    0xD,  6 }, {    0x5,  3 }, {    0x6,  3 }, {    0x1,  4 },
        {    0x0,  4 }, {    0x7,  5 }, {    0xC,  6 }, {   0xF6,  8 },
        {  0x1EE,  9 }, {  0x3DF, 10 }, {  0x7BC, 11 }, { 0x3DEB, 14 },
        { 0x3DEA, 14 }, { 0x3DED, 14 }, { 0x3DEC, 14 }, {    0x2,  3 },
        {    0x9,  4 }, {   0x1F,  5 }, {    0xF,  5 }, {    0x5,  5 },
        {    0xE,  4 }, {    0x6,  4 }, {    0x4,  5 }, {   0x1D,  6 }
    },
    { /* AC bias group 3, table 10 */
        {    0x9,  4 }, {   0x39,  6 }, {   0x19,  7 }, {   0x18,  7 },
        {  0x706, 11 }, { 0x383D, 14 }, { 0x383C, 14 }, {    0xD,  6 },
        {    0xF,  6 }, {    0x5,  3 }, {    0x6,  3 }, {    0x0,  4 },
        {   0x1D,  5 }, {    0x3,  5 }, {   0x71,  7 }, {   0xE1,  8 },
        {  0x1C0,  9 }, {  0x382, 10 }, { 0x1C1D, 13 }, { 0x383F, 14 },
        { 0x383E, 14 }, { 0x3839, 14 }, { 0x3838, 14 }, {    0x2,  3 },
        {    0x8,  4 }, {    0x2,  4 }, {    0xD,  5 }, {    0xC,  5 },
        {    0xF,  4 }, {    0x7,  4 }, {    0x2,  5 }, {    0xE,  6 }
    },
    { /* AC bias group 3, table 11 */
        {    0x0,  3 }, {    0x6,  5 }, {   0x35,  7 }, {   0x34,  7 },
        {  0x777, 11 }, { 0x1DD4, 13 }, { 0x3BAB, 14 }, {    0xE,  6 },
        {    0xF,  6 }, {    0x5,  3 }, {    0x4,  3 }, {   0x1C,  5 },
        {   0x19,  5 }, {   0x3A,  6 }, {   0xEF,  8 }, {  0x1DC,  9 },
        {  0x776, 11 }, {  0x774, 11 }, { 0x3BAA, 14 }, { 0x3BAD, 14 },
        { 0x3BAC, 14 }, { 0x3BAF, 14 }, { 0x3BAE, 14 }, {    0x2,  3 },
        {    0x7,  4 }, {    0x2,  4 }, {   0x18,  5 }, {    0xC,  5 },
        {    0xF,  4 }, {    0xD,  4 }, {   0x1B,  6 }, {   0x76,  7 }
    },
    { /* AC bias group 3, table 12 */
        {    0x2,  3 }, {   0x11,  5 }, {    0x6,  6 }, {   0x4F,  7 },
        {  0x130,  9 }, { 0x1319, 13 }, { 0x1318, 13 }, {   0x4E,  7 },
        {    0x7,  6 }, {    0x6,  3 }, {    0x5,  3 }, {   0x10,  5 },
        {    0xD,  5 }, {    0x5,  6 }, {   0x99,  8 }, {  0x262, 10 },
        {  0x98E, 12 }, { 0x131B, 13 }, { 0x131A, 13 }, { 0x263D, 14 },
        { 0x263C, 14 }, { 0x263F, 14 }, { 0x263E, 14 }, {    0x1,  3 },
        {    0x7,  4 }, {    0x0,  4 }, {   0x12,  5 }, {    0xC,  5 },
        {    0xE,  4 }, {    0xF,  4 }, {    0x4,  6 }, {   0x4D,  7 }
    },
    { /* AC bias group 3, table 13 */
        {    0x3,  3 }, {    0x0,  4 }, {    0x2,  5 }, {   0x37,  6 },
        {  0x1B7,  9 }, {  0xDB5, 12 }, { 0x36DD, 14 }, {   0x6C,  7 },
        {   0x16,  6 }, {    0x5,  3 }, {    0x4,  3 }, {    0x3,  5 },
        {    0xA,  5 }, {   0x2E,  7 }, {  0x36C, 10 }, {  0xDB4, 12 },
        { 0x36DC, 14 }, { 0x36DF, 14 }, { 0x36DE, 14 }, { 0x36D9, 14 },
        { 0x36D8, 14 }, { 0x36DB, 14 }, { 0x36DA, 14 }, {    0xE,  4 },
        {    0x4,  4 }, {   0x1A,  5 }, {   0x19,  5 }, {   0x18,  5 },
        {    0xF,  4 }, {    0x1,  3 }, {   0x2F,  7 }, {   0xDA,  8 }
    },
    { /* AC bias group 3, table 14 */
        {    0x6,  3 }, {    0x6,  4 }, {    0xF,  5 }, {    0x0,  4 },
        {   0x75,  7 }, {  0x3B8, 10 }, { 0x1DCA, 13 }, {   0x74,  7 },
        {   0x76,  7 }, {    0x4,  3 }, {    0x5,  3 }, {    0x3,  5 },
        {    0x2,  5 }, {  0x1DE,  9 }, {  0xEE6, 12 }, { 0x3B97, 14 },
        { 0x3B96, 14 }, { 0x3B9D, 14 }, { 0x3B9C, 14 }, { 0x3B9F, 14 },
        { 0x3B9E, 14 }, { 0x1DC9, 13 }, { 0x1DC8, 13 }, {    0x5,  4 },
        {   0x1C,  5 }, {    0x9,  5 }, {    0xE,  5 }, {    0x8,  5 },
        {    0xF,  4 }, {    0x1,  3 }, {  0x1DF,  9 }, {  0x1DD,  9 }
    },
    { /* AC bias group 3, table 15 */
        {    0x4,  3 }, {    0xB,  4 }, {   0x1D,  5 }, {    0xC,  4 },
        {   0x14,  5 }, {   0xE0,  8 }, { 0x3875, 14 }, {  0x386, 10 },
        {  0x1C2,  9 }, {    0x0,  2 }, {    0x1,  2 }, {   0x71,  7 },
        {   0x72,  7 }, { 0x1C3F, 13 }, { 0x3874, 14 }, { 0x3877, 14 },
        { 0x3876, 14 }, { 0x3871, 14 }, { 0x3870, 14 }, { 0x3873, 14 },
        { 0x3872, 14 }, { 0x3879, 14 }, { 0x3878, 14 }, {   0x3C,  6 },
        {   0x73,  7 }, {   0x2A,  6 }, {   0x3D,  6 }, {   0x2B,  6 },
        {   0x1F,  5 }, {    0xD,  4 }, { 0x1C3E, 13 }, { 0x1C3D, 13 }
    }
};

static const uint16_t vp4_ac_bias_3[16][32][2] = {
    { /* AC bias group 4, table 0 */
        {    0x7,  4 }, {    0xF,  6 }, {   0xBB,  8 }, {   0xBA,  8 },
        {  0x5CF, 11 }, { 0x173A, 13 }, { 0x2E77, 14 }, {   0x29,  6 },
        {  0x172,  9 }, {    0x7,  3 }, {    0x6,  3 }, {    0x9,  4 },
        {    0x8,  4 }, {    0x1,  4 }, {    0x5,  5 }, {    0xD,  6 },
        {   0x1D,  7 }, {   0x1C,  7 }, {   0xB8,  8 }, {  0x2E6, 10 },
        { 0x2E76, 14 }, { 0x1739, 13 }, { 0x1738, 13 }, {    0x2,  3 },
        {    0x6,  4 }, {   0x16,  5 }, {    0x4,  5 }, {   0x28,  6 },
        {   0x15,  5 }, {    0xC,  6 }, {    0x0,  4 }, {   0x2F,  6 }
    },
    { /* AC bias group 4, table 1 */
        {    0xB,  4 }, {    0x2,  5 }, {   0x54,  7 }, {   0x2F,  7 },
        {  0x2AC, 10 }, { 0x156B, 13 }, { 0x1568, 13 }, {   0x16,  6 },
        {  0x154,  9 }, {    0x7,  3 }, {    0x6,  3 }, {    0x4,  4 },
        {    0x3,  4 }, {   0x13,  5 }, {   0x28,  6 }, {   0x2E,  7 },
        {  0x157,  9 }, {  0x155,  9 }, {  0x55B, 11 }, { 0x2AD3, 14 },
        { 0x2AD2, 14 }, { 0x2AD5, 14 }, { 0x2AD4, 14 }, {    0x3,  3 },
        {    0x8,  4 }, {    0x0,  4 }, {    0xA,  5 }, {    0x3,  5 },
        {    0x2,  4 }, {   0x2B,  6 }, {   0x12,  5 }, {   0x29,  6 }
    },
    { /* AC bias group 4, table 2 */
        {    0xF,  4 }, {    0x7,  5 }, {    0x1,  6 }, {    0x0,  6 },
        {  0x1C4,  9 }, {  0x703, 11 }, {  0xE02, 12 }, {   0x11,  6 },
        {   0xE1,  8 }, {    0x5,  3 }, {    0x6,  3 }, {    0x2,  4 },
        {    0x1,  4 }, {    0x9,  5 }, {   0x10,  6 }, {   0xE3,  8 },
        {  0x1C5,  9 }, {  0x1C1,  9 }, {  0x702, 11 }, { 0x1C07, 13 },
        { 0x1C06, 13 }, {  0xE01, 12 }, {  0xE00, 12 }, {    0x4,  3 },
        {    0x7,  4 }, {   0x1D,  5 }, {    0xD,  5 }, {    0x1,  5 },
        {    0x5,  4 }, {    0x6,  5 }, {    0xC,  5 }, {   0x39,  6 }
    },
    { /* AC bias group 4, table 3 */
        {    0x1,  3 }, {   0x1C,  5 }, {   0x11,  6 }, {   0x13,  6 },
        {   0x42,  8 }, {  0x207, 11 }, {  0x815, 13 }, {   0x75,  7 },
        {   0x41,  8 }, {    0x5,  3 }, {    0x6,  3 }, {    0x0,  4 },
        {   0x1F,  5 }, {   0x3B,  6 }, {   0x74,  7 }, {   0x43,  8 },
        {   0x80,  9 }, {  0x206, 11 }, {  0x814, 13 }, {  0x817, 13 },
        {  0x816, 13 }, {  0x409, 12 }, {  0x408, 12 }, {    0x3,  3 },
        {    0x9,  4 }, {   0x1E,  5 }, {   0x11,  5 }, {    0x3,  5 },
        {    0x5,  4 }, {   0x10,  5 }, {    0x2,  5 }, {   0x12,  6 }
    },
    { /* AC bias group 4, table 4 */
        {    0x1,  3 }, {   0x1F,  5 }, {   0x27,  6 }, {    0x1,  5 },
        {   0x4B,  8 }, {  0x123, 10 }, {  0x915, 13 }, {    0x0,  6 },
        {   0x49,  8 }, {    0x5,  3 }, {    0x6,  3 }, {   0x1D,  5 },
        {   0x1C,  5 }, {   0x13,  6 }, {   0x4A,  8 }, {   0x90,  9 },
        {  0x914, 13 }, {  0x917, 13 }, {  0x916, 13 }, {  0x911, 13 },
        {  0x910, 13 }, {  0x913, 13 }, {  0x912, 13 }, {    0x3,  3 },
        {    0x5,  4 }, {    0x1,  4 }, {   0x12,  5 }, {    0x8,  5 },
        {    0x8,  4 }, {   0x1E,  5 }, {   0x26,  6 }, {    0x1,  6 }
    },
    { /* AC bias group 4, table 5 */
        {    0x3,  3 }, {    0x1,  4 }, {   0x3F,  6 }, {    0xB,  5 },
        {   0x4E,  7 }, {  0x132,  9 }, {  0x99A, 12 }, {   0x4F,  7 },
        {   0x98,  8 }, {    0x6,  3 }, {    0x5,  3 }, {   0x1D,  5 },
        {   0x1C,  5 }, {   0x7C,  7 }, {  0x267, 10 }, { 0x1331, 13 },
        { 0x1330, 13 }, { 0x1333, 13 }, { 0x1332, 13 }, { 0x266D, 14 },
        { 0x266C, 14 }, { 0x266F, 14 }, { 0x266E, 14 }, {    0x1,  3 },
        {    0x4,  4 }, {   0x1E,  5 }, {   0x12,  5 }, {    0xA,  5 },
        {    0x8,  4 }, {    0x0,  4 }, {   0x7D,  7 }, {   0x4D,  7 }
    },
    { /* AC bias group 4, table 6 */
        {    0x2,  3 }, {    0x7,  4 }, {   0x15,  5 }, {    0x3,  4 },
        {    0x4,  5 }, {   0xA7,  8 }, {  0x536, 11 }, {   0x28,  6 },
        {  0x29A, 10 }, {    0x6,  3 }, {    0x4,  3 }, {   0x1C,  5 },
        {   0x17,  5 }, {   0xA4,  8 }, { 0x29BE, 14 }, { 0x537F, 15 },
        { 0x537E, 15 }, { 0x29B9, 14 }, { 0x29B8, 14 }, { 0x29BB, 14 },
        { 0x29BA, 14 }, { 0x29BD, 14 }, { 0x29BC, 14 }, {    0xF,  4 },
        {    0x0,  4 }, {    0x5,  5 }, {   0x16,  5 }, {   0x1D,  5 },
        {    0x6,  4 }, {    0x1,  4 }, {   0xA5,  8 }, {  0x14C,  9 }
    },
    { /* AC bias group 4, table 7 */
        {    0x4,  3 }, {    0x7,  4 }, {   0x1A,  5 }, {    0xC,  4 },
        {    0x6,  4 }, {   0x29,  6 }, {  0x1BD,  9 }, { 0x1BE3, 13 },
        { 0x1BE0, 13 }, {    0x0,  2 }, {    0x7,  3 }, {   0x6E,  7 },
        {  0x1BC,  9 }, { 0x37C3, 14 }, { 0x37C2, 14 }, { 0x37CD, 14 },
        { 0x37CC, 14 }, { 0x37CF, 14 }, { 0x37CE, 14 }, { 0x37C9, 14 },
        { 0x37C8, 14 }, { 0x37CB, 14 }, { 0x37CA, 14 }, {   0x15,  5 },
        {  0x1BF,  9 }, {  0x37D, 10 }, {   0x36,  6 }, {    0x2,  3 },
        {    0xB,  4 }, {   0x28,  6 }, { 0x37C5, 14 }, { 0x37C4, 14 }
    },
    { /* AC bias group 4, table 8 */
        {    0x1,  3 }, {    0x9,  5 }, {    0x3,  6 }, {    0x2,  6 },
        {  0x11F, 10 }, {  0x8E9, 13 }, {  0x8E8, 13 }, {   0x2D,  7 },
        {   0x22,  7 }, {    0x6,  3 }, {    0x7,  3 }, {   0x10,  5 },
        {   0x11,  5 }, {   0x17,  6 }, {   0x2C,  7 }, {   0x46,  8 },
        {  0x11E, 10 }, {  0x11C, 10 }, {  0x477, 12 }, {  0x8EB, 13 },
        {  0x8EA, 13 }, {  0x8ED, 13 }, {  0x8EC, 13 }, {    0x3,  3 },
        {    0xB,  4 }, {    0x1,  4 }, {   0x14,  5 }, {    0xA,  5 },
        {    0x9,  4 }, {   0x15,  5 }, {    0x0,  5 }, {   0x10,  6 }
    },
    { /* AC bias group 4, table 9 */
        {    0x1,  3 }, {   0x1D,  5 }, {   0x22,  6 }, {   0x13,  6 },
        {  0x11E,  9 }, {  0x8FC, 12 }, { 0x23F5, 14 }, {   0x23,  7 },
        {   0x22,  7 }, {    0x5,  3 }, {    0x6,  3 }, {   0x10,  5 },
        {    0xB,  5 }, {   0x10,  6 }, {   0x8E,  8 }, {  0x23E, 10 },
        {  0x8FF, 12 }, { 0x11FD, 13 }, { 0x23F4, 14 }, { 0x23F7, 14 },
        { 0x23F6, 14 }, { 0x23F9, 14 }, { 0x23F8, 14 }, {    0x3,  3 },
        {    0x9,  4 }, {    0x0,  4 }, {   0x1C,  5 }, {    0xA,  5 },
        {    0xF,  4 }, {    0x1,  4 }, {   0x12,  6 }, {   0x46,  7 }
    },
    { /* AC bias group 4, table 10 */
        {    0x3,  3 }, {   0x1F,  5 }, {   0x3C,  6 }, {   0x3D,  6 },
        {   0x86,  8 }, {  0x877, 12 }, { 0x10E8, 13 }, {   0x41,  7 },
        {   0x40,  7 }, {    0x5,  3 }, {    0x6,  3 }, {    0x7,  5 },
        {    0x6,  5 }, {    0x4,  6 }, {  0x10F,  9 }, {  0x21C, 10 },
        {  0x875, 12 }, { 0x21D3, 14 }, { 0x21D2, 14 }, { 0x21D9, 14 },
        { 0x21D8, 14 }, { 0x21DB, 14 }, { 0x21DA, 14 }, {    0x2,  3 },
        {    0x9,  4 }, {    0x0,  4 }, {   0x11,  5 }, {    0x3,  5 },
        {    0xE,  4 }, {    0x2,  4 }, {    0x5,  6 }, {   0x42,  7 }
    },
    { /* AC bias group 4, table 11 */
        {    0x4,  3 }, {    0x1,  4 }, {   0x3D,  6 }, {    0x9,  5 },
        {   0xF3,  8 }, {  0x793, 11 }, { 0x1E45, 13 }, {    0x0,  7 },
        {    0x2,  7 }, {    0x5,  3 }, {    0x6,  3 }, {    0x8,  5 },
        {    0x1,  5 }, {    0x3,  7 }, {  0x1E5,  9 }, {  0x792, 11 },
        { 0x1E44, 13 }, { 0x1E47, 13 }, { 0x1E46, 13 }, { 0x1E41, 13 },
        { 0x1E40, 13 }, { 0x1E43, 13 }, { 0x1E42, 13 }, {    0x1,  3 },
        {    0x6,  4 }, {   0x1F,  5 }, {    0xF,  5 }, {    0xE,  5 },
        {    0xE,  4 }, {    0x5,  4 }, {   0x78,  7 }, {    0x1,  7 }
    },
    { /* AC bias group 4, table 12 */
        {    0x4,  3 }, {    0x5,  4 }, {    0xE,  5 }, {   0x17,  5 },
        {   0x3E,  7 }, {   0xF0,  9 }, {  0xF1E, 13 }, {   0x7A,  8 },
        {   0x7F,  8 }, {    0x6,  3 }, {    0x7,  3 }, {    0x5,  5 },
        {    0x4,  5 }, {   0x7B,  8 }, {  0x1E2, 10 }, { 0x1E3F, 14 },
        { 0x1E3E, 14 }, {  0xF19, 13 }, {  0xF18, 13 }, {  0xF1B, 13 },
        {  0xF1A, 13 }, {  0xF1D, 13 }, {  0xF1C, 13 }, {    0x0,  3 },
        {    0x3,  4 }, {   0x16,  5 }, {    0x9,  5 }, {    0x8,  5 },
        {    0xA,  4 }, {    0x6,  4 }, {   0x7E,  8 }, {   0x79,  8 }
    },
    { /* AC bias group 4, table 13 */
        {    0x5,  3 }, {    0xC,  4 }, {   0x1A,  5 }, {    0x4,  4 },
        {   0x1A,  6 }, {   0xDE,  9 }, {  0xDF4, 13 }, {   0xDD,  9 },
        {   0x6D,  8 }, {    0x0,  2 }, {    0x7,  3 }, {   0x25,  6 },
        {   0x24,  6 }, {   0xDC,  9 }, {  0xDF7, 13 }, { 0x1BEB, 14 },
        { 0x1BEA, 14 }, {  0xDF1, 13 }, {  0xDF0, 13 }, {  0xDF3, 13 },
        {  0xDF2, 13 }, { 0x1BED, 14 }, { 0x1BEC, 14 }, {    0x8,  4 },
        {   0x13,  5 }, {    0xC,  5 }, {   0x37,  6 }, {   0x36,  6 },
        {    0x5,  4 }, {    0x7,  4 }, {   0x6C,  8 }, {  0x1BF, 10 }
    },
    { /* AC bias group 4, table 14 */
        {    0x5,  3 }, {    0xD,  4 }, {   0x1F,  5 }, {    0xC,  4 },
        {   0x3B,  6 }, {   0x40,  7 }, {  0x41A, 11 }, {  0x104,  9 },
        {  0x107,  9 }, {    0x1,  2 }, {    0x0,  2 }, {   0x24,  6 },
        {   0x21,  6 }, {  0x20B, 10 }, { 0x106E, 13 }, { 0x20DF, 14 },
        { 0x20DE, 14 }, { 0x1055, 13 }, { 0x1054, 13 }, { 0x1057, 13 },
        { 0x1056, 13 }, { 0x106D, 13 }, { 0x106C, 13 }, {   0x11,  5 },
        {   0x3A,  6 }, {   0x25,  6 }, {   0x38,  6 }, {   0x39,  6 },
        {   0x13,  5 }, {   0x1E,  5 }, {  0x20C, 10 }, {  0x414, 11 }
    },
    { /* AC bias group 4, table 15 */
        {    0x0,  2 }, {    0x7,  4 }, {    0xD,  5 }, {    0x5,  4 },
        {    0x9,  5 }, {   0x22,  7 }, {  0xCD1, 13 }, {  0xCD0, 13 },
        {  0xCD3, 13 }, {    0x3,  2 }, {    0x2,  2 }, {   0x8D,  9 },
        {   0xCC,  9 }, {  0x66B, 12 }, {  0xCD2, 13 }, { 0x19B5, 14 },
        { 0x19B4, 14 }, { 0x19B7, 14 }, { 0x19B6, 14 }, { 0x19B1, 14 },
        { 0x19B0, 14 }, { 0x19B3, 14 }, { 0x19B2, 14 }, {   0x47,  8 },
        {   0x8C,  9 }, {  0x337, 11 }, {   0x67,  8 }, {   0x18,  6 },
        {   0x10,  6 }, {   0x32,  7 }, {  0xCD5, 13 }, {  0xCD4, 13 }
    }
};

#endif /* AVCODEC_VP4DATA_H */
