/* $Id$
 *
 * Resolve ranges of positional and named aggregates.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __RESOLVE_AGGREGATES_HPP_INCLUDED
#define __RESOLVE_AGGREGATES_HPP_INCLUDED

#include "frontend/visitor/NullVisitor.hpp"
#include "frontend/ast/Types.hpp"
#include "frontend/misc/RangeSet.hpp"

namespace ast {

//! Resolve ranges of one positional and named aggregate.
/** This visitor will resolve an array aggregate and set the range
 *  member of an ElementAssociation to the result.
 *  In case of erroneous specifications, errors will get reported.
 *
 *  This class will only handle one single Aggregate.
 *  Dependencies:
 *  - TypeResolution
 *  - ConstantPropagation
 */
class ResolveAggregates : public NullVisitor {
public:
	//! c'tor
	/** @param arrayRange range of the target type.
	 */
	ResolveAggregates(const DiscreteRange &arrayRange);

	//! d'tor
	virtual ~ResolveAggregates();

	/** Visit an Aggregate node.
	 *  @param node Aggregate node that get's visited.
	 */
	virtual void visit(Aggregate &node);

private:
	/** visit a ElementAssociation
         *  @param node node that get's visited.
         */
	virtual void visit(ElementAssociation &node);

	/** Visit an Others node.
	 *  @param node Others node that get's visited.
	 */
	virtual void visit(Others &node);

	/** Visit a ConstInteger node.
	 *  @param node ConstInteger node that get's visited.
	 */
	virtual void visit(ConstInteger &node);

	/** Visit a DiscreteRange
	 *  @param node DiscreteRange node that get's visited.
	 */
	virtual void visit(DiscreteRange &node);

	//! Process a generic AstNode.
        /** Failmatch method. Must not be called.
         *
         *  @param node AstNode
         */
	virtual void process(AstNode &node);

	//! others node present in assocation list?
	bool haveOthers;
	//! named association list?
	bool named;
	//! positional association list?
	bool positional;
	//! was an error for a wrong others node reported already?
	bool othersErrorReported;
	//! current range for an ElementAssociation (or NULL)
	RangeSet *currentRange;

	//! produce sequences of numbers.
	class Sequencer {
	public:
		//! c'tor
		/** @param byRange range to construct the sequence from.
		 */
		Sequencer(const DiscreteRange &byRange);

		/** get the next number in the Sequence. Update the sequence
		 *  appropriately as well.
		 *  @return logical next number in the Sequence.
		 */
		universal_integer getNext(void);

		/** get the RangeSet containing elements which haven't been
		 *  taken yet.
		 *  @return new RangeSet containing all elements that haven't
		 *          been taken yet. Might be empty.
		 */
		RangeSet *getRemainder(void) const;

		//! underlying RangeSet 
		RangeSet rs;

	private:
		//! direction of the sequence.
		enum DiscreteRange::Direction direction;
	};

	//! sequencer for this array aggregate.
	Sequencer sequencer;
};

}; /* namespace ast */

#endif /* __RESOLVE_AGGREGATES_HPP_INCLUDED */
