@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-job-get: Download a Job Output/Input/Error File
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use MIME::Base64;

my %opt = ( input => 0, error => 0, encrypt => 0, binary => 0, bufsize => 524288, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  input
	  error
	  encrypt
	  binary
	  bufsize=i
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $jobId  = $ARGV[0];
my $target = $ARGV[1];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No job-id specified!\n" )                 if not($jobId);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No target specified!\n" )                 if not($target);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Incompatible options!(-input,-error)\n" ) if ( $opt{input} && $opt{error} );

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Get the Data Channel
my $dataChannel = $controlChannel;
if ( not $opt{encrypt} ) {
	$dataChannel = $ecaccess->getDataChannel();    # Plain Text rather than SSL (faster)
}

# Get the file handle (output, input or error)
$handle = $controlChannel->getJobOutputHandle( $token, $jobId )->result if not ( $opt{input} || $opt{error} );
$handle = $controlChannel->getJobInputHandle( $token,  $jobId )->result if ( $opt{input} );
$handle = $controlChannel->getJobErrorHandle( $token,  $jobId )->result if ( $opt{error} );

# Open the Target File
open FILE, ">", $target or die "Error creating file: " . $target . "\n";

# Donwload the content
if ( not $opt{binary} ) {

	# In Text mode
	while ( length( $data = $dataChannel->readStringHandle( $handle, $opt{bufsize} )->result ) > 0 ) {
		print FILE $data;
	}
}
else {

	# In Binary mode
	binmode FILE;
	while ( length( $data = decode_base64( $dataChannel->readBytesHandle( $handle, $opt{bufsize} )->result ) ) > 0 ) {
		print FILE $data;
	}
}

# close the file handles
$controlChannel->closeHandle($handle);
close FILE;

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-job-get - Download a Job Output/Input/Error File

=head1 SYNOPSIS

B<ecaccess-job-get -version|-help|-manual>

B<ecaccess-job-get [-debug] [-input|-error] [-encrypt] [-binary] [-bufsize> I<length>B<]> I<job-id> I<local-target-file>

=head1 DESCRIPTION

Allow downloading the Job Output/Input/Error Files with identifier I<job-id>.

The file is downloaded localy in the I<local-target-file>.

=head1 ARGUMENTS

=over 8

=item I<job-id>

The identifier of the ECaccess Job to retrieve.

=item I<local-target-file>

The name of the Local Target File.

=back

=head1 OPTIONS

=over 8

=item B<-input>

By default the Job Output File is downloaded. Using this option allow downloading
the Job Input File instead.

=item B<-error>

By default the Job Output File is downloaded. Using this option allow downloading
the Job Error File instead.

=item B<-encrypt>

By default files are downloaded through the plain text channel (http). Using this
option will force the download to occurs through the SSL secure channel (https).

=item B<-binary>

By default files are downloaded as text files. This option will download files
as binary files (decode_base64 required). Please note that text files can also
be downloaded in binary mode but text mode is faster. You should use this
option if your job output is containing non-text characters.

=item B<-bufsize> I<length>

Specify the I<length> of the buffer (in bytes) which is used to download the file.
The larger the buffer the smaller the number of http/s requests. By default a
buffer of 524288 bytes (512KB) is used.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-job-get> I<124356> I<./ecaccess-job-124356.output>

Download the output of the ECaccess Job I<124356> in the local I<ecaccess-job-124356.output> file.

=head1 SEE ALSO

B<ecaccess-job-delete>, B<ecaccess-job-list>, B<ecaccess-job-restart>, B<ecaccess-job-submit>
and B<ecaccess>.

=cut

__END__
:endofperl
