# Updating to a new Major Version & TFM

At the end of each release cycle, we update our Major Version (branding version of build assets) & TFM (Target Framework Moniker) in `main`, in preparation for the next major release. This doc describes the process of doing those updates, which can include many subtle gotchas in the aspnetcore repo.

In the event that we do a minor release (e.g. 3.1), the guidance in this document still applies (but with all instances of `MajorVersion` replaced by `MinorVersion`).

## Updating Major Version

Typically, we will update the Major Version before updating the TFM. This is because updating Major Version only requires a logical agreement that `main` is no longer the place for work on the previous major release, while updating the TFM requires waiting for dotnet/runtime to update their TFM so that we can ingest that change. For an example, [this](https://github.com/dotnet/aspnetcore/pull/35402) is the PR where we updated our branding from 6.0.0 to 7.0.0 (note that branding does *not* have to happen in a dependency update PR - this particular branding exercise needed to include some reaction to runtime changes).

### Required changes

* In [eng/Versions.props](/eng/Versions.props):
  1. Increment `AspNetCoreMajorVersion` by 1.
  2. Change `PreReleaseVersionIteration` to `1`.
  3. Change `PreReleaseVersionLabel` to `alpha`.
  4. Change `PreReleaseBrandingLabel` to `Alpha $(PreReleaseVersionIteration)`.
* In [src/Framework/test/TestData.cs](/src/Framework/test/TestData.cs), update `ListedTargetingPackAssemblies` by incrementing the AssemblyVersion of all aspnetcore assemblies by 1 major version. Once dotnet/runtime updates their AssemblyVersions, we also need to update those in this file. They typically make that change at the same time as their TFM update, but we change our AssemblyVersions as soon as we update branding.
* Add entries to [NuGet.config](/NuGet.config) for the new Major Version's feed. This just means copying the current feeds (e.g. `dotnet7` and `dotnet7-transport`) and adding entries for the new feeds (`dotnet8` and `dotnet8-transport`). Make an effort to remove old feeds here at the same time.
* In [src/ProjectTemplates/Shared/TemplatePackageInstaller.cs](/src/ProjectTemplates/Shared/TemplatePackageInstaller.cs), add entries to `_templatePackages ` for `Microsoft.DotNet.Web.ProjectTemplates` and `Microsoft.DotNet.Web.Spa.ProjectTemplates` matching the new version.
* In [src/Framework/AspNetCoreAnalyzers/test/Verifiers/CSharpRouteHandlerCodeFixVerifier.cs], update the references to `ReferenceAssemblies.Net.Netx0` with the latest version.

### Validation

* CI must be green.
* Assets produced by the build (packages, installers) should have the new branding version (e.g. if you have just updated `MajorVersion` to `8`, packages should be branded `8.0.0-alpha.1.{BuildNumber}`).
* Assemblies produced by the build should have the new `AssemblyVersion` (e.g. if you have just updated `MajorVersion` to `8`, Assemblies should have `AssemblyVersion` `8.0.0.0`).

## Updating TFM

Once dotnet/runtime has updated their TFM, we update ours in the dependency update PR ingesting that change. We won't be able to ingest new dotnet/runtime dependencies in `main` until this is done. For an example, [this](https://github.com/dotnet/aspnetcore/pull/36328) is the PR where we updated our TFM to `net7.0`. This step can be tricky - we have workarounds in [eng/tools/GenerateFiles/Directory.Build.targets.in](/eng/tools/GenerateFiles/Directory.Build.targets.in) to make the build work before we get an SDK containing runtime references with the new TFM. We copy the `KnownFrameworkReference`, `KnownRuntimePack`, and `KnownAppHostPack` from the previous TFM, give them the incoming runtime dependency versions, and give them the new TFM (these TFMs no-op most of the time - they only apply during this period when we're using an SDK that doesn't know about the new TFM). These workarounds allow us to build against the new TFM before we get an SDK with a reference to it, but there are often problems that arise in this area. The best way to debug build errors related to FrameworkReferences it to get a binlog of a failing project (`dotnet build /bl`) and look at the inputs to the task that failed. Confirm that the `Known___` items look as expected (there is an entry with the current TFM & the current dotnet/runtime dependency version), and look at the source code of the task in [dotnet/sdk](https://github.com/dotnet/sdk) for hints.

### Required changes

* In [eng/Versions.props](/eng/Versions.props), increment `DefaultNetCoreTargetFramework` by 1.
* Do a global repo search for the current version string, and update almost everything by 1 (e.g. find `net7`, replace with `net8`). See the PR linked above for examples - this shouldn't be done blindly, but on a case-by-case basis. Most things should be updated, and most choices should be obvious.
  * Exceptions to this are [eng/tools/RepoTasks/RepoTasks.csproj](/eng/tools/RepoTasks/RepoTasks.csproj), [eng/tools/RepoTasks/RepoTasks.tasks](/eng/tools/RepoTasks/RepoTasks.tasks), and [eng/tools/HelixTestRunner/HelixTestRunner.csproj](/eng/tools/HelixTestRunner/HelixTestRunner.csproj). These build without the workarounds from [eng/tools/GenerateFiles/Directory.Build.targets.in](/eng/tools/GenerateFiles/Directory.Build.targets.in), and need to be kept at the previous TFM until we get an SDK containing a runtime with the new TFM. Generally this means we have to hard-code the previous TFM for these files, rather than using `DefaultNetCoreTargetFramework`.
* Add a reference to the new `SiteExtensions` package for the previous Major Version.
  1. Add references to [src/SiteExtensions/LoggingAggregate/src/Microsoft.AspNetCore.AzureAppServices.SiteExtension/Microsoft.AspNetCore.AzureAppServices.SiteExtension.csproj](/src/SiteExtensions/LoggingAggregate/src/Microsoft.AspNetCore.AzureAppServices.SiteExtension/Microsoft.AspNetCore.AzureAppServices.SiteExtension.csproj) to `Microsoft.AspNetCore.AzureAppServices.SiteExtension.{PreviousMajorVersion}.0.x64` and `Microsoft.AspNetCore.AzureAppServices.SiteExtension.{PreviousMajorVersion}.0.x86`.
  2. Add entries in [eng/Versions.props](/eng/Versions.props) similar to [these](https://github.com/dotnet/aspnetcore/blob/216c92b78bce31d5e81a70b589707ec2ae5ab21a/eng/Versions.props#L224-L226) - the version should be from the latest released build of .Net.
  3. Add entries in [eng/Dependencies.props](/eng/Dependencies.props) similar to [these](https://github.com/dotnet/aspnetcore/blob/a47c0a58d7002b9a530c67532366b9db96d73cc6/eng/Dependencies.props#L119-L120).
* Update AssemblyVersions for dotnet/runtime assemblies in [src/Framework/test/TestData.cs](/src/Framework/test/TestData.cs).
* Update dotnet/spa-templates
  1. Create a `release/{n}.0` branch in dotnet/spa-template based off of `main`, where `n` is the MajorVersion we are updating from (not the one we are updating to).
  2. Create a PR like [this one](https://github.com/dotnet/aspnetcore/pull/36932) updating the current release branch in `aspnetcore` to reference the new release branch you just created in dotnet/spa-templates.
  3. Create a PR like [this one](https://github.com/dotnet/spa-templates/pull/21) updating the branding & TFM in the `main` branch of dotnet/spa-templates.
    * Do not merge this until the PR from the previous step is merged.
* Update template precedence
  1. Create & merge a PR like [this one](https://github.com/dotnet/spa-templates/pull/39) in dotnet/spa-templates updating `precedence` and `identity` elements in all template.json files.
  2. Create a PR like [this one](https://github.com/dotnet/aspnetcore/pull/39783) in dotnet/aspnetcore that updates the spa-templates submodule, and updates the `precedence`, `identity`, and (if it exists) `thirdPartyNotices` elements in all template.json files.
  3. Make sure the new aka.ms link you're referencing in `thirdPartyNotices` exists.

### Validation

* CI must be green.
* Packages produced by the build should be placing assemblies in a folder named after the new TFM.

## Ingesting an SDK with the new TFM

Typically we update the SDK we use in `main` every Monday. Once we have one that contains `Microsoft.Netcore.App` entries with the new TFM, we can update [eng/tools/RepoTasks/RepoTasks.csproj](/eng/tools/RepoTasks/RepoTasks.csproj), [eng/tools/RepoTasks/RepoTasks.tasks](/eng/tools/RepoTasks/RepoTasks.tasks), and [eng/tools/HelixTestRunner/HelixTestRunner.csproj](/eng/helix/tools/HelixTestRunner.csproj) to use `DefaultNetCoreTargetFramework` again rather than hard-coding the previous TFM.
