/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_config.h"
#include "cdw_mkisofsrc.h"
#include "cdw_fs.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_form.h"
#include "cdw_logging.h"


/* the value must be larger than any of CDW_ISO9660_*_LEN values
   defined in optical_file_systems/cdw_iso9660.h */
#define BUFFER_SIZE 200


static int   cdw_mkisofsrc_read(cdw_form_t *cdw_form, void *dummy);
static char *cdw_mkisofsrc_options_top_message(int n_cols);
static void  cdw_mkisofs_get_option_from_line(char *line, const char *option_name, cdw_form_t *cdw_form, int fi, size_t len);

static cdw_form_text_maker_t text_makers[] = {
	cdw_mkisofsrc_options_top_message,
};




enum {
	f_mkisofsrc_top_message_l,
	f_mkisofsrc_top_message_b,
	f_mkisofsrc_abstract_l,
	f_mkisofsrc_abstract_i,
	f_mkisofsrc_volume_set_id_l,
	f_mkisofsrc_volume_set_id_i,
	f_mkisofsrc_copyright_l,
	f_mkisofsrc_copyright_i,
	f_mkisofsrc_publisher_l,
	f_mkisofsrc_publisher_i,
	f_mkisofsrc_preparer_l,
	f_mkisofsrc_preparer_i,
	f_mkisofsrc_system_id_l,
	f_mkisofsrc_system_id_i };


static FIELD *page_mkisofsrc_fields[CDW_MKISOFS_ISO_META_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */
static cdw_iso9660_t *local_iso = (cdw_iso9660_t *) NULL;



/**
   \brief Create fields that are shown on "Meta information" page in options

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_ERROR if one of fields was created incorrectly
*/
cdw_rv_t cdw_mkisofsrc_options_form(cdw_form_t *cdw_form, void *cdw_iso, int first_col, __attribute__((unused)) int second_col, int width_wide, __attribute__((unused)) int width_narrow)
{
	cdw_iso9660_t *iso = (cdw_iso9660_t *) cdw_iso;
	local_iso = iso;
	cdw_form->fields = page_mkisofsrc_fields;

	cdw_form_descr_t descr[] = {
		/* type                       begin_y   begin_x     n_cols           n_lines  field enum                     data1                               data2 */

		{ CDW_WIDGET_ID_TEXT,               1,  0,          width_wide + 3,        6,  f_mkisofsrc_top_message_l,    text_makers,                            0 },
		/* 2TRANS: this is button label; a verb: read data from config file */
		{ CDW_WIDGET_ID_BUTTON,             8,  1,          1,                     1,  f_mkisofsrc_top_message_b,    _("Read"),              CDW_COLORS_DIALOG },

		/* 2TRANS: this is a label next to input field, in which user can enter abstract of ISO9660 volume; don't localize "ABST" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,      9,  first_col,  width_wide,            1,  f_mkisofsrc_abstract_l,       _("Abstract (ABST)"),                   0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   10,  first_col,  CDW_ISO9660_ABST_LEN,  1,  f_mkisofsrc_abstract_i,       iso->abstract,       CDW_ISO9660_ABST_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter ID of ISO9660 volume set; don't localize "VOLS" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     11,  first_col,  width_wide,            1,  f_mkisofsrc_volume_set_id_l,  _("Volume set ID (VOLS)"),              0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   12,  first_col,  width_wide,            1,  f_mkisofsrc_volume_set_id_i,  iso->volume_set_id,  CDW_ISO9660_VOLS_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter copyright of ISO9660 volume; don't localize "COPY" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     13,  first_col,  width_wide,            1,  f_mkisofsrc_copyright_l,      _("Copyright (COPY)"),                  0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   14,  first_col,  CDW_ISO9660_COPY_LEN,  1,  f_mkisofsrc_copyright_i,      iso->copyright,      CDW_ISO9660_COPY_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter information about publisher of ISO9660 volume; don't localize "PUBL" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     15,  first_col,  width_wide,            1,  f_mkisofsrc_publisher_l,      _("Publisher (PUBL)"),                  0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   16,  first_col,  width_wide,            1,  f_mkisofsrc_publisher_i,      iso->publisher,      CDW_ISO9660_PUBL_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter information about preparer of ISO9660 volume; don't localize "PREP" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     17,  first_col,  width_wide,            1,  f_mkisofsrc_preparer_l,       _("Preparer (PREP)"),                   0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   18,  first_col,  width_wide,            1,  f_mkisofsrc_preparer_i,       iso->preparer,       CDW_ISO9660_PREP_LEN },

		/* 2TRANS: this is a label next to input field, in which user can enter information about system on which ISO9660 volume has been created; don't localize "SYSI" */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     19,  first_col,  width_wide,            1,  f_mkisofsrc_system_id_l,      _("System ID (SYSI)"),                  0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   20,  first_col,  CDW_ISO9660_SYSI_LEN,  1,  f_mkisofsrc_system_id_i,      iso->system_id,      CDW_ISO9660_SYSI_LEN },

		/* guard */
		{ -1,                               0,  0,          0,                     0,  0,                            (void *) NULL,                          0 }};

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_widget_callback(cdw_form, f_mkisofsrc_top_message_b, cdw_mkisofsrc_read);

		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_abstract_i, f_mkisofsrc_abstract_l);
		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_volume_set_id_i, f_mkisofsrc_volume_set_id_l);
		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_copyright_i, f_mkisofsrc_copyright_l);
		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_publisher_i, f_mkisofsrc_publisher_l);
		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_preparer_i, f_mkisofsrc_preparer_l);
		cdw_form_bind_input_and_label(cdw_form, f_mkisofsrc_system_id_i, f_mkisofsrc_system_id_l);

		return CDW_OK;
	}
}





cdw_rv_t cdw_mkisofsrc_options_validate(cdw_iso9660_t *iso, int *fi)
{
	cdw_rv_t crv = cdw_string_security_parser(iso->abstract, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_abstract_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->volume_set_id, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_volume_set_id_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->copyright, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_copyright_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->publisher, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_publisher_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->preparer, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_preparer_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(iso->system_id, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofsrc_system_id_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_mkisofsrc_options_save(cdw_form_t *cdw_form, cdw_iso9660_t *iso)
{
	char *s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_abstract_i));
	strncpy(iso->abstract, s, CDW_ISO9660_ABST_LEN);
	        iso->abstract[CDW_ISO9660_ABST_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_volume_set_id_i));
	strncpy(iso->volume_set_id, s, CDW_ISO9660_VOLS_LEN);
	        iso->volume_set_id[CDW_ISO9660_VOLS_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_copyright_i));
	strncpy(iso->copyright, s, CDW_ISO9660_COPY_LEN);
	        iso->copyright[CDW_ISO9660_COPY_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_publisher_i));
	strncpy(iso->publisher, s, CDW_ISO9660_PUBL_LEN);
	        iso->publisher[CDW_ISO9660_PUBL_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_preparer_i));
	strncpy(iso->preparer, s, CDW_ISO9660_PREP_LEN);
	        iso->preparer[CDW_ISO9660_PREP_LEN] = '\0';

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofsrc_system_id_i));
	strncpy(iso->system_id, s, CDW_ISO9660_SYSI_LEN);
	        iso->system_id[CDW_ISO9660_SYSI_LEN] = '\0';

	return CDW_OK;
}





char *cdw_mkisofsrc_options_top_message(int n_cols)
{
	/* 2TRANS: this is message printed in configuration window;
	   don't localize "$HOME/.mkisofsrc" */
	return cdw_string_wrap(_("These values won't be saved after closing cdw. Leave the fields empty (values from mkisofs config file will be used) or enter custom values. You can populate the fields with values from $HOME/.mkisofsrc using the button below."), (size_t) n_cols, CDW_ALIGN_CENTER);
}



static const char *file_name = ".mkisofsrc";
static FILE *file = (FILE *) NULL;





int cdw_mkisofsrc_read(cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	const char *home_dir = cdw_fs_get_home_dir_fullpath();
	char *path = cdw_string_concat(home_dir, file_name, (char *) NULL);
	file = fopen(path, "r");
	free(path);
	path = (char *) NULL;
	if (!file) {
		/* 2TRANS: this is message printed to log file;
		   '%s' is configuration file name */
		cdw_logging_write("WARNING: can't open %s file in home directory\n", file_name);
		cdw_vdm ("WARNING: can't open mkisofsrc file\n");
		return -1;
	}

	char *line = (char *) NULL;
	while (1) {
		if (line) {
			free(line);
			line = (char *) NULL;
		}

		line = my_readline_10k(file);
		if (!line) {
			break;
		}

		cdw_mkisofs_get_option_from_line(line, "ABST", cdw_form, f_mkisofsrc_abstract_i,      CDW_ISO9660_ABST_LEN);
		cdw_mkisofs_get_option_from_line(line, "COPY", cdw_form, f_mkisofsrc_copyright_i,     CDW_ISO9660_COPY_LEN);
		cdw_mkisofs_get_option_from_line(line, "PUBL", cdw_form, f_mkisofsrc_publisher_i,     CDW_ISO9660_PUBL_LEN);
		cdw_mkisofs_get_option_from_line(line, "PREP", cdw_form, f_mkisofsrc_preparer_i,      CDW_ISO9660_PREP_LEN);
		cdw_mkisofs_get_option_from_line(line, "SYSI", cdw_form, f_mkisofsrc_system_id_i,     CDW_ISO9660_SYSI_LEN);
		cdw_mkisofs_get_option_from_line(line, "VOLS", cdw_form, f_mkisofsrc_volume_set_id_i, CDW_ISO9660_VOLS_LEN);

	}
	fclose(file);
	file = (FILE *) NULL;

	return 0;
}





void cdw_mkisofs_get_option_from_line(char *line, const char *option_name, cdw_form_t *cdw_form, int fi, size_t len)
{
	if (!strstr(line, option_name)) {
		return;
	}

	cdw_option_t option;
	if (!cdw_config_split_options_line(&option, line)) {
		return; /* empty or invalid line, or comment */
	}

	cdw_rv_t v = cdw_string_security_parser(option.value, (char *) NULL);
	if (v != CDW_OK) {
		cdw_vdm ("WARNING: option value \"%s=%s\" from mkisofsrc file rejected as insecure\n",
			 option.name, option.value);
		cdw_config_option_free(&option);
		return;
	}

	char buffer[BUFFER_SIZE + 1];
	strncpy(buffer, option.value, len);
	buffer[len] = '\0';
	set_field_buffer(cdw_form->fields[fi], 0, buffer);

	cdw_config_option_free(&option);

	return;
}
