//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/SampleItem.cpp
//! @brief     Implements class SampleItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/SampleItem.h"
#include "GUI/Model/Sample/LayerItem.h"
#include "GUI/Support/XML/Backup.h"
#include "GUI/Support/XML/UtilXML.h"
#include <QXmlStreamWriter>

namespace {
namespace Tag {

const QString Name("Name");
const QString Description("Description");
const QString CrossCorrelationLength("CrossCorrelationLength");
const QString MaterialModel("MaterialModel");
const QString Layer("Layer");
const QString ExternalField("ExternalField");
const QString ExpandInfoGroupbox("ExpandInfoGroupbox");

} // namespace Tag
} // namespace

SampleItem::SampleItem()
{
    m_name = "Sample";
    m_crossCorrelationLength.init(
        "Cross-correlation length", "Cross correlation length of roughnesses between interfaces",
        0.0, Unit::nanometer, 5 /* decimals */, RealLimits::nonnegative(), "cross");
    m_externalField.init("External field", "External field (A/m)", "A/m", "extField");
}

SampleItem::~SampleItem() = default;

void SampleItem::initFrom(const SampleItem* other)
{
    GUI::Util::copyContents(other, this);
}

QVector<ItemWithMaterial*> SampleItem::itemsWithMaterial() const
{
    QVector<ItemWithMaterial*> result;
    for (auto* layer : m_layers)
        result.append(layer->itemsWithMaterial());
    return result;
}

void SampleItem::addStandardMaterials()
{
    // add only non-existing materials
    QString name = materialMap.key(DefaultMaterials::Default);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 1e-3, 1e-5);

    name = materialMap.key(DefaultMaterials::Vacuum);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 0.0, 0.0);

    name = materialMap.key(DefaultMaterials::Particle);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 6e-4, 2e-8);

    name = materialMap.key(DefaultMaterials::Core);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 2e-4, 1e-8);

    name = materialMap.key(DefaultMaterials::Substrate);
    if (!m_materials.materialItemFromName(name))
        m_materials.addRefractiveMaterialItem(name, 6e-6, 2e-8);
}

QVector<LayerItem*> SampleItem::layerItems() const
{
    return QVector<LayerItem*>(m_layers.begin(), m_layers.end());
}

LayerItem* SampleItem::createLayerItemAt(int index)
{
    if (index < 0)
        index = m_layers.size();

    auto* layer = new LayerItem(&m_materials);
    m_layers.insert_at(index, layer);
    updateTopBottom();
    return layer;
}

void SampleItem::updateTopBottom()
{
    for (LayerItem* l : m_layers) {
        l->setIsTopLayer(l == m_layers.front());
        l->setIsBottomLayer(l == m_layers.back());
    }
}

void SampleItem::removeLayer(LayerItem* layer)
{
    m_layers.delete_element(layer);
    updateTopBottom();
}

void SampleItem::moveLayer(LayerItem* layer, LayerItem* aboveThisLayer)
{
    if (layer == aboveThisLayer)
        return;

    int currentIndex = layerItems().indexOf(layer);
    m_layers.release_at(currentIndex);

    int destIndex = m_layers.size();
    if (aboveThisLayer != nullptr)
        destIndex = layerItems().indexOf(aboveThisLayer);
    m_layers.insert_at(destIndex, layer);

    updateTopBottom();
}

void SampleItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // name
    w->writeStartElement(Tag::Name);
    XML::writeAttribute(w, XML::Attrib::value, m_name);
    w->writeEndElement();

    // description
    w->writeStartElement(Tag::Description);
    XML::writeAttribute(w, XML::Attrib::value, m_description);
    w->writeEndElement();

    // cross-correlation length
    w->writeStartElement(Tag::CrossCorrelationLength);
    m_crossCorrelationLength.writeTo(w);
    w->writeEndElement();

    // exernal field
    w->writeStartElement(Tag::ExternalField);
    m_externalField.writeTo(w);
    w->writeEndElement();

    // materials
    w->writeStartElement(Tag::MaterialModel);
    m_materials.writeTo(w);
    w->writeEndElement();

    // layers
    for (const auto* layer : m_layers) {
        w->writeStartElement(Tag::Layer);
        layer->writeTo(w);
        w->writeEndElement();
    }

    // info groupbox: is expanded?
    w->writeStartElement(Tag::ExpandInfoGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandInfo);
    w->writeEndElement();
}

void SampleItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    m_layers.clear();

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // sample name
        if (tag == Tag::Name) {
            XML::readAttribute(r, XML::Attrib::value, &m_name);
            XML::gotoEndElementOfTag(r, tag);

            // description
        } else if (tag == Tag::Description) {
            XML::readAttribute(r, XML::Attrib::value, &m_description);
            XML::gotoEndElementOfTag(r, tag);

            // cross-correlation length
        } else if (tag == Tag::CrossCorrelationLength) {
            m_crossCorrelationLength.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // exernal field
        } else if (tag == Tag::ExternalField) {
            m_externalField.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // materials
        } else if (tag == Tag::MaterialModel) {
            m_materials.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // layers
        } else if (tag == Tag::Layer) {
            createLayerItemAt()->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // info groupbox: is expanded?
        } else if (tag == Tag::ExpandInfoGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandInfo);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
