//
// SimilarityTests.cs
//
// Author:
//   Gabriel Burt <gburt@novell.com>
//
// Copyright (C) 2010 Novell, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if ENABLE_TESTS

using System;
using System.Linq;

using Mono.Addins;
using NUnit.Framework;

using Mirage;
using Banshee.Mirage;
using System.Globalization;

namespace Banshee.Mirage.Tests
{
    [TestFixture]
    public class SimilarityTests
    {
        [TestFixtureSetUp]
        public void SetUp ()
        {
            AddinManager.Initialize ();
            AddinManager.Registry.Update (null);
        }

        [TestFixtureTearDown]
        public void TearDown ()
        {
            AddinManager.Shutdown ();
        }

        [Test]
        public void TestEmptyContext ()
        {
            var context = new SimilarityContext ();
            Assert.IsTrue (context.IsEmpty);
            Assert.AreEqual (float.MaxValue, context.Distance (null).Average ());
        }

        [Test]
        public void TestShuffledDistances ()
        {
            var distance_s1_s2 = 27.21912f;
            var context = new SimilarityContext ();
            context.AddSeeds (new Seed [] { GetSeed (s1, SimilarityContext.ShuffledWeight) });
            Assert.IsFalse (context.IsEmpty);
            Assert.Less (Math.Abs (distance_s1_s2 - context.Distance (Scms.FromBytes (Bytes (s2))).Average ()), 0.1);

            // Add the same seed w/ the same weight; make sure the average is the same
            context.AddSeeds (new Seed [] { GetSeed (s1, SimilarityContext.ShuffledWeight) });
            Assert.Less (Math.Abs (distance_s1_s2 - context.Distance (Scms.FromBytes (Bytes (s2))).Average ()), 0.1);
        }

        [Test]
        [Ignore("Fails with bad value: 54.438224792480469d")]
        public void TestDiscardedDistance ()
        {
            // Test with different seed weight
            var distance_s1_s2_discarded = 81.65736f;
            var context = new SimilarityContext ();
            context.AddSeeds (new Seed [] { GetSeed (s1, SimilarityContext.DiscardedWeight) });
            Assert.IsFalse (context.IsEmpty);
            Assert.Less (Math.Abs (distance_s1_s2_discarded - context.Distance (Scms.FromBytes (Bytes (s2))).Average ()), 0.1);
        }

        [Test]
        public void TestSameDistance ()
        {
            var context = new SimilarityContext ();
            context.AddSeeds (new Seed [] { GetSeed (s1, 1.0f) });
            Assert.AreEqual (0f, context.Distance (Scms.FromBytes (Bytes (s1))).Average ());
        }

        private Seed GetSeed (string hex, float weight)
        {
            return new Seed () {
                Scms = Scms.FromBytes (Bytes (hex)),
                Weight = weight
            };
        }

        private byte [] Bytes (string hex)
        {
            var bytes = new byte [hex.Length / 2];
            for (int i = 0; i < hex.Length; i += 2) {
                bytes[i/2] = Byte.Parse (hex.Substring (i, 2), NumberStyles.HexNumber);
            }
            return bytes;
        }

        const string s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
        const string s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
    }
}

#endif
