/**
 * Keeps track of zones of a rectangular area that are occupied and provides a
 * method to find out the available area closest to a given target point.
 * The rectangular area is initially empty. The client must mark the occupied
 * area by calling the appropriate methods of this module.
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2017/09/29 20:32:38 $
 */

#ifndef RATNEST_H
#define RATNEST_H

//#include "ps.h"

#ifdef ratnest_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * A rectangular region. This module enforces this area be non empty and point
 * 1 be the bottom-left point, 2 the top-right point, so x1 < x2 and y1 < y2.
 */
typedef struct {
	double x1, y1, x2, y2;
} ratnest_Rect;

/**
 * Returns the rectangle as a string.
 * @param r
 * @return Pointer to statically allocated internal string.
 */
EXTERN char *ratnest_RectToString(ratnest_Rect *r);

/**
 * State of this module, created with ratnest_new().
 */
typedef struct ratnest_Type ratnest_Type;

/**
 * Creates a new empty rectangular area.
 * @param bounds
 * @return Empty rectangular area to monitor. Can be released with memory_dispose().
 */
EXTERN ratnest_Type * ratnest_new(ratnest_Rect *bounds);

/**
 * Marks as occupied a segment.
 * @param this
 * @param x1
 * @param y1
 * @param x2
 * @param y2
 */
EXTERN void ratnest_markSegment(ratnest_Type *this, double x1, double y1, double x2, double y2);

/**
 * Marks as occupied a given rectangular area.
 * @param this
 * @param r
 */
EXTERN void ratnest_markRect(ratnest_Type *this, ratnest_Rect *r);

/**
 * Search a suitable free area that may contain the given rectangle.
 * @param this
 * @param r Location of the rectangle as proposed by the client. The bottom-left
 * corner is assumed to be the target point.
 * @param found If the search succeeds, here returns the found available
 * free area, the same size of the proposed one but possibly in another
 * location within the boundaries of the monitored area.
 * @return 
 */
EXTERN int  ratnest_place(ratnest_Type *this, ratnest_Rect *r, ratnest_Rect *found);

//EXTERN void ratnest_debug(ratnest_Type *this, ps_Type *ps);

#undef EXTERN
#endif
