; Java Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "JAVA")

(include-book "java-primitive-array-model")
(include-book "java-types")

(include-book "kestrel/std/system/arity-plus" :dir :system)
(include-book "kestrel/std/system/function-namep" :dir :system)
(include-book "kestrel/std/system/number-of-results-plus" :dir :system)
(include-book "kestrel/std/system/table-alist-plus" :dir :system)
(include-book "std/lists/index-of" :dir :system)
(include-book "std/typed-lists/cons-listp" :dir :system)
(include-book "std/util/defval" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ atj-types
  :parents (atj-implementation)
  :short "Types used by ATJ for code generation."
  :long
  (xdoc::topstring
   (xdoc::p
    "In order to make the generated Java code more efficient and idiomatic,
     ATJ uses types that correspond to both ACL2 predicates and Java types.
     These ATJ types are used only when
     @(':deep') is @('nil') and @(':guards') is @('t').")
   (xdoc::p
    "For example, consider a unary ACL2 function
     whose guard is or implies @(tsee stringp),
     and the corresponding Java method generated by ATJ.
     Since under the assumption of guard satisfaction
     this method will always be called
     with an @('Acl2Value') that is an @('Acl2String'),
     the method can use @('Acl2String') instead of @('Acl2Value')
     as the type of the argument.
     Furthermore, suppose that, under the guard,
     the ACL2 function always returns @(tsee integerp).
     Then the Java method can use @('Acl2Integer') instead of @('Acl2Value')
     as the return type.
     In other words,
     narrower types than the one for all ACL2 values (i.e. @('Acl2Value'))
     can be used for the argument and result of this Java method.
     This narrowing is also used to generate methods
     that operate on Java primitive values and primitive arrays.")
   (xdoc::p
    "In general, establishing the narrower input and output types
     for a Java method generated from an ACL2 function
     may involve arbitrarily hard theorem proving:
     (i) proving that the guard implies that the inputs of the ACL2 function
     satisfy the ACL2 predicates corresponding to the input types, and
     (ii) proving that the guard implies that the outputs of the ACL2 function
     satisfy the ACL2 predicates corresponding to the output types;
     the number of outputs of an ACL2 function is greater than 1
     if the function returns an @(tsee mv) value;
     otherwise the number of outputs is 1.
     Since we do not want ATJ to attempt any theorem proving,
     we provide a macro @(tsee atj-main-function-type)
     to perform those theorem proving tasks under user control
     and to record the input and output types of ACL2 functions in a table,
     and we have ATJ look up types in this table.
     Note that these types are different from
     ACL2's built-in types used for typeset reasoning,
     ACL2's tau system types,
     and our ACL2 model of Java types.")
   (xdoc::p
    "With a table of the types of the involved ACL2 functions at hand
     (the table being constructed via calls of @(tsee atj-main-function-type)),
     ATJ performs a type analysis of the ACL2 terms in function bodies
     before translating them to Java;
     this analysis is part of ATJ's pre-translation steps.
     Generally speaking,
     ATJ compares the type inferred for an actual argument of a function
     (this type is inferred by analyzing terms recursively)
     with the type of the corresponding formal argument of the function
     (this type is retrieved from the table of function types):
     if they differ, ATJ inserts code to convert from the former to the latter,
     unless the former is a subtype of the latter in Java.
     The conversion may be a type cast,
     e.g. to convert from @('Acl2Value') to @('Acl2String');
     the cast is guaranteed to succeed,
     assuming that the ACL2 guards are verified.
     The conversion may also be a change in representation in other cases.")
   (xdoc::p
    "The ATJ type information stored in the table
     determines/specifies the input and output types of the Java methods
     generated for the corresponding ACL2 functions.
     In general, there may be different possible choices of types
     for certain ACL2 functions:
     different choices will lead to different Java code.
     For instance,
     if a function's guard implies that an argument satisfies @(tsee integerp),
     that function's argument can be assigned
     a type corresponding to @('Acl2Integer'),
     or a type corresponding to @('Acl2Rational').
     The types of these Java methods are part of the ``API''
     that the generated Java code provides to external Java code.")
   (xdoc::p
    "In some cases, ACL2 functions return outputs of narrower types
     when given inputs of narrower types.
     Prime examples are the arithmetic operations
     @(tsee binary-+), @(tsee binary-*), and @(tsee unary--).
     All of their input and output types are
     the type corresponding to @(tsee acl2-numberp),
     based on their guards:
     this can be recorded via @(tsee atj-main-function-type).
     Based on these types, the corresponding Java methods
     will take and return @('Acl2Number') values.
     Now, consider a unary function @('f') that takes integers
     (i.e. it has a recorded input type corresponding to @(tsee integerp)),
     and a term @('(f (binary-+ <i> <j>))'),
     where @('<i>') and @('<j>') are integer-valued terms.
     When this term is translated to Java,
     a cast (from @('Acl2Number')) to @('Acl2Integer') will be inserted
     around the call of the method corresponding to @(tsee binary-+),
     in order to fit the @('Acl2Integer') type of
     the argument of the method corresponding to @('f').")
   (xdoc::p
    "However, due to well-known closure properties,
     @(tsee binary-+), like @(tsee binary-+) and @(tsee unary--),
     maps @(tsee rationalp) inputs to @(tsee rationalp) outputs,
     and @(tsee integerp) inputs to @(tsee integerp) outputs.
     This means that we could generate three overloaded methods
     for each such ACL2 function:
     one with @('Acl2Number') argument and result types (as above),
     one with @('Acl2Rational') argument and result types, and
     one with @('Acl2Integer') argument and result types.
     This will make the cast in the example above unnecessary:
     since the Java expressions that translate @('<i>') and @('<j>')
     statically have type @('Acl2Integer'),
     the Java compiler will pick the most specific overloaded method,
     which returns @('Acl2Integer').")
   (xdoc::p
    "This is not limited to primitive arithmetic operations.
     Any ACL2 function may have the property of
     returning outputs of narrower types when given inputs of narrower types.
     Even if the output types are not narrower,
     the internal computations may be more efficient on narrower inputs,
     e.g. the cast in the example above can be avoided
     when that call of @('f') is part of some function @('g')
     that may not even return numbers (e.g. it may return booleans).")
   (xdoc::p
    "Thus, we provide another macro, @(tsee atj-other-function-type),
     to record additional input and output types for ACL2 functions.
     ATJ makes use of these additional types
     to generate multiple overloaded methods for single ACL2 functions.
     In general, via these two macros, each ACL2 function may have
     more than one input/output type associated with it
     (where an input/output type is a full function type,
     consisting of zero or more input types and one or more output types):
     (i) a primary (`main') input/output type,
     provable from the guards as described above; and
     (ii) zero or more secondary (`other') input/output types.
     The secondary input types are narrower than the primary ones,
     but do not have to be provable from the guard;
     what must be proved,
     via a theorem generated by @(tsee atj-other-function-type),
     is that the guard and the input types imply the output type.")
   (xdoc::p
    "The above is just an overview of the use of types by ATJ.
     More details are in the documentation of their implementation
     and of the code generation functions that use them."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deftagsum atj-atype
  :short "Fixtype of the ATJ types that denote built-in ACL2 types."
  :long
  (xdoc::topstring-p
   "See @(tsee atj-type).")
  (:integer ())
  (:rational ())
  (:number ())
  (:character ())
  (:string ())
  (:symbol ())
  (:boolean ())
  (:cons ())
  (:value ())
  :pred atj-atypep)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deftagsum atj-type
  :short "Fixtype of all the ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "These are used for code generation, as explained in @(see atj-types).")
   (xdoc::p
    "Currently ATJ uses types for:")
   (xdoc::ul
    (xdoc::li
     "ACL2 integers, rationals, numbers,
      characters, strings, symbols, booleans
      @(tsee cons) pairs, and all values,
      whose fixtype is @(tsee atj-atype).
      With the exception of the type of booleans,
      these all correspond to the AIJ public class types for ACL2 values.")
    (xdoc::li
     "Java primitive values and arrays."))
   (xdoc::p
    "More types may be added in the future.")
   (xdoc::p
    "Each ATJ type denotes
     (i) an ACL2 predicate (see @(tsee atj-type-to-pred)) and
     (ii) a Java type (see @(tsee atj-type-to-jitype)).
     It is not the case that
     just the @(':acl2') types denote ACL2 types
     and just the @(':jprim') and @(':jprimarr') types denote Java types:
     each type denotes both an ACL2 and a Java type.
     The distinction is just that
     the @(':acl2') types denote built-in ACL2 types,
     which are therefore independent from Java
     (even though they have a Java representation, in AIJ),
     while the @(':jprim') and @(':jprimarr') types are Java-specific."))
  (:acl2 ((get atj-atype)))
  (:jprim ((get primitive-type)))
  (:jprimarr ((comp primitive-type)))
  :pred atj-typep
  ///

  (defruled atj-type-equiv-alt-def
    (equal (atj-type-equiv x y)
           (atj-type-case
            x
            :acl2 (atj-type-case
                   y
                   :acl2 (atj-atype-equiv x.get y.get)
                   :jprim nil
                   :jprimarr nil)
            :jprim (atj-type-case
                    y
                    :acl2 nil
                    :jprim (primitive-type-equiv x.get y.get)
                    :jprimarr nil)
            :jprimarr (atj-type-case
                       y
                       :acl2 nil
                       :jprim nil
                       :jprimarr (primitive-type-equiv x.comp y.comp))))
    :enable (atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp
             atj-type-kind-possibilities)
    :rule-classes :definition))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-irrelevant ()
  :returns (type atj-typep)
  :short "An irrelevant ATJ type,
          usable as dummy return value with hard errors."
  (with-guard-checking :none (ec-call (atj-type-fix :irrelevant)))
  ///
  (in-theory (disable (:e atj-type-irrelevant))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-keyword ((type atj-typep))
  :returns (kwd keywordp)
  :short "Map each ATJ type to a distinct keyword."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is useful, for instance, to print ATJ types in a more readable form
     that hides the internal representation of their fixtype.
     We also these keywords to refer to the types
     in the developer documentation.")
   (xdoc::p
    "Also see @(tsee atj-type-from-keyword)."))
  (atj-type-case type
                 :acl2 (atj-atype-case type.get
                                       :integer :ainteger
                                       :rational :arational
                                       :number :anumber
                                       :character :acharacter
                                       :string :astring
                                       :symbol :asymbol
                                       :boolean :aboolean
                                       :cons :acons
                                       :value :avalue)
                 :jprim (primitive-type-case type.get
                                             :boolean :jboolean
                                             :char :jchar
                                             :byte :jbyte
                                             :short :jshort
                                             :int :jint
                                             :long :jlong
                                             :float :jfloat
                                             :double :jdouble)
                 :jprimarr (primitive-type-case type.comp
                                                :boolean :jboolean[]
                                                :char :jchar[]
                                                :byte :jbyte[]
                                                :short :jshort[]
                                                :int :jint[]
                                                :long :jlong[]
                                                :float :jfloat[]
                                                :double :jdouble[]))
  :hooks (:fix)
  ///

  (defrule atj-type-to-keyword-injective
    (equal (equal (atj-type-to-keyword type1)
                  (atj-type-to-keyword type2))
           (atj-type-equiv type1 type2))
    :enable (atj-type-fix
             primitive-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-from-keyword ((kwd keywordp))
  :returns (type atj-typep)
  :short "Map keywords back to ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the inverse of @(tsee atj-type-to-keyword).")
   (xdoc::p
    "An error occurs if the keyword does not correspond to any ATJ type."))
  (case kwd
    (:ainteger (atj-type-acl2 (atj-atype-integer)))
    (:arational (atj-type-acl2 (atj-atype-rational)))
    (:anumber (atj-type-acl2 (atj-atype-number)))
    (:acharacter (atj-type-acl2 (atj-atype-character)))
    (:astring (atj-type-acl2 (atj-atype-string)))
    (:asymbol (atj-type-acl2 (atj-atype-symbol)))
    (:aboolean (atj-type-acl2 (atj-atype-boolean)))
    (:acons (atj-type-acl2 (atj-atype-cons)))
    (:avalue (atj-type-acl2 (atj-atype-value)))
    (:jboolean (atj-type-jprim (primitive-type-boolean)))
    (:jchar (atj-type-jprim (primitive-type-char)))
    (:jbyte (atj-type-jprim (primitive-type-byte)))
    (:jshort (atj-type-jprim (primitive-type-short)))
    (:jint (atj-type-jprim (primitive-type-int)))
    (:jlong (atj-type-jprim (primitive-type-long)))
    (:jfloat (atj-type-jprim (primitive-type-float)))
    (:jdouble (atj-type-jprim (primitive-type-double)))
    (:jboolean[] (atj-type-jprimarr (primitive-type-boolean)))
    (:jchar[] (atj-type-jprimarr (primitive-type-char)))
    (:jbyte[] (atj-type-jprimarr (primitive-type-byte)))
    (:jshort[] (atj-type-jprimarr (primitive-type-short)))
    (:jint[] (atj-type-jprimarr (primitive-type-int)))
    (:jlong[] (atj-type-jprimarr (primitive-type-long)))
    (:jfloat[] (atj-type-jprimarr (primitive-type-float)))
    (:jdouble[] (atj-type-jprimarr (primitive-type-double)))
    (otherwise (prog2$ (raise
                        "The keyword ~x0 does not correspond to any ATJ type."
                        kwd)
                       (atj-type-irrelevant))))
  ///

  (defrule atj-type-from-keyword-of-atj-type-to-keyword
    (equal (atj-type-from-keyword (atj-type-to-keyword type))
           (atj-type-fix type))
    :enable (atj-type-to-keyword
             atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-type-list
  :short "Fixtype of lists of ATJ types."
  :elt-type atj-type
  :true-listp t
  :elementp-of-nil nil
  :pred atj-type-listp
  ///

  (defruled atj-type-list-equiv-alt-def
    (equal (atj-type-list-equiv x y)
           (cond ((endp x) (endp y))
                 ((endp y) (endp x))
                 (t (and (atj-type-equiv (car x) (car y))
                         (atj-type-list-equiv (cdr x) (cdr y))))))
    :enable atj-type-list-fix
    :rule-classes :definition))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-keyword-list ((types atj-type-listp))
  :returns (kwds keyword-listp)
  :short "Lift @(tsee atj-type-to-keyword) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-keyword (car types))
                 (atj-type-list-to-keyword-list (cdr types)))))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-from-keyword-list ((kwds keyword-listp))
  :returns (types atj-type-listp)
  :short "Lift @(tsee atj-type-from-keyword) to lists."
  (cond ((endp kwds) nil)
        (t (cons (atj-type-from-keyword (car kwds))
                 (atj-type-list-from-keyword-list (cdr kwds))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-type-list-list
  :short "Fixtype of lists of lists of ATJ types."
  :elt-type atj-type-list
  :true-listp t
  :elementp-of-nil t
  :pred atj-type-list-listp
  ///

  (defrule atj-type-list-listp-of-remove-duplicates-equal
    (implies (atj-type-list-listp x)
             (atj-type-list-listp (remove-duplicates-equal x)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defalist atj-symbol-type-alist
  :short "Fixtype of alists from symbols to ATJ types."
  :key-type symbol
  :val-type atj-type
  :true-listp t
  :keyp-of-nil t
  :valp-of-nil nil
  :pred atj-symbol-type-alistp
  ///

  (defrule atj-typep-of-cdr-of-assoc-equal-when-atj-symbol-type-alistp
    (implies (atj-symbol-type-alistp alist)
             (iff (atj-typep (cdr (assoc-equal key alist)))
                  (assoc-equal key alist))))

  (defrule atj-symbol-type-alistp-of-pairlis$
    (implies (and (symbol-listp keys)
                  (atj-type-listp vals)
                  (equal (len keys) (len vals)))
             (atj-symbol-type-alistp (pairlis$ keys vals)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-type
  atj-type
  :short "Fixtype of ATJ types and @('nil')."
  :pred atj-maybe-typep)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-maybe-type-list
  :short "Fixtype of lists of ATJ types and @('nil')."
  :elt-type atj-maybe-type
  :true-listp t
  :elementp-of-nil t
  :pred atj-maybe-type-listp
  ///

  (defruled atj-maybe-type-list-equiv-alt-def
    (equal (atj-maybe-type-list-equiv x y)
           (cond ((endp x) (endp y))
                 ((endp y) (endp x))
                 (t (and (atj-maybe-type-equiv (car x) (car y))
                         (atj-maybe-type-list-equiv (cdr x) (cdr y))))))
    :enable atj-maybe-type-list-fix
    :rule-classes :definition)

  (defrule atj-maybe-type-listp-when-atj-type-listp
    (implies (atj-type-listp x)
             (atj-maybe-type-listp x))
    :enable atj-maybe-type-listp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-pred ((type atj-typep))
  :returns (pred pseudo-termfnp)
  :short "ACL2 predicate denoted by an ATJ type."
  :long
  (xdoc::topstring
   (xdoc::p
    "The predicate recognizes the values of the type.")
   (xdoc::p
    "The predicates for the @(':acl2') types are straightforward.
     The predicates for the @(':jprim') types
     are the recognizers of the corresponding
     Java primitive values in our Java language formalization.
     The predicates for the @(':jprimarr') types
     are the recognizers of the corresponding
     Java primitive arrays in our ATJ's model of primitive arrays.
     Also see "
    (xdoc::seetopic "atj-java-primitives" "here")
    " and "
    (xdoc::seetopic "atj-java-primitive-arrays" "here")
    "."))
  (atj-type-case type
                 :acl2 (atj-atype-case type.get
                                       :integer 'integerp
                                       :rational 'rationalp
                                       :number 'acl2-numberp
                                       :character 'characterp
                                       :string 'stringp
                                       :symbol 'symbolp
                                       :boolean 'booleanp
                                       :cons 'consp
                                       :value '(lambda (_) 't))
                 :jprim (primitive-type-case type.get
                                             :boolean 'boolean-valuep
                                             :char 'char-valuep
                                             :byte 'byte-valuep
                                             :short 'short-valuep
                                             :int 'int-valuep
                                             :long 'long-valuep
                                             :float 'float-valuep
                                             :double 'double-valuep)
                 :jprimarr (primitive-type-case type.comp
                                                :boolean 'boolean-arrayp
                                                :char 'char-arrayp
                                                :byte 'byte-arrayp
                                                :short 'short-arrayp
                                                :int 'int-arrayp
                                                :long 'long-arrayp
                                                :float 'float-arrayp
                                                :double 'double-arrayp))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-atype-<= ((sub atj-atypep) (sup atj-atypep))
  :returns (yes/no booleanp)
  :short "Partial order over the ATJ types that denote built-in ACL2 types."
  :long
  (xdoc::topstring-p
   "See @(tsee atj-type-<=).")
  (and
   (atj-atype-case
    sub
    :integer (member-eq (atj-atype-kind sup) '(:integer :rational :number
                                               :value))
    :rational (member-eq (atj-atype-kind sup) '(:rational :number :value))
    :number (member-eq (atj-atype-kind sup) '(:number :value))
    :character (member-eq (atj-atype-kind sup) '(:character :value))
    :string (member-eq (atj-atype-kind sup) '(:string :value))
    :symbol (member-eq (atj-atype-kind sup) '(:symbol :value))
    :boolean (member-eq (atj-atype-kind sup) '(:boolean :symbol :value))
    :cons (member-eq (atj-atype-kind sup) '(:cons :value))
    :value (atj-atype-case sup :value))
   t)
  :hooks (:fix)
  ///

  (defrule atj-atype-<=-reflexive
    (implies (atj-atype-equiv x y)
             (atj-atype-<= x y)))

  (defrule atj-atype-<=-antisymmetric
    (implies (and (atj-atype-<= x y)
                  (atj-atype-<= y x))
             (atj-atype-equiv x y)))

  (defrule atj-atype-<=-transitive
    (implies (and (atj-atype-<= x y)
                  (atj-atype-<= y z))
             (atj-atype-<= x z)))

  (defrule atj-atype-<=-of-value-left
    (implies (atj-atype-case x :value)
             (equal (atj-atype-<= x y)
                    (atj-atype-case y :value))))

  (defrule atj-atype-<=-of-cons-left
    (implies (atj-atype-case x :cons)
             (equal (atj-atype-<= x y)
                    (or (atj-atype-case y :cons)
                        (atj-atype-case y :value))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-<= ((sub atj-typep) (sup atj-typep))
  :returns (yes/no booleanp)
  :short "Partial order over all the ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "The ATJ types form a partial order.
     The ordering on the @(':acl2') types
     is based on the inclusion of the ACL2 predicates they denote;
     this denotation is defined by @(tsee atj-type-to-pred).
     Each of the @(':jprim') and @(':jprimarr') types
     is incomparable with all the types except itself.")
   (xdoc::p
    "This definition of partial order is motivated by
     the automatic conversions that we want to allow,
     in the generated Java code,
     between (the Java representations of) the ACL2 values denoted by the types.
     While we want to allow conversions between the @(':acl2') types,
     we keep the @(':jprim') and @(':jprimarr') types
     all separate from each other and from the @(':acl2') types.")
   (xdoc::p
    "More precisely, a type is less than another type if and only if
     in the generated Java code,
     values of the Java type denoted by the smaller ATJ type
     can be automatically converted (possibly via a no-op, but not necessarily)
     to values of the Java type denoted by the larger ATJ type.
     This happens, for instance, when a value of the smaller type
     is passed to a function whose formal argument has the larger type:
     if the two types were not related like that,
     ATJ's type analysis would disallow the function call.")
   (xdoc::p
    "Furthermore, a type may be less than another type only if
     the ACL2 predicate denoted by the first ATJ type
     is a subset of the ACL2 predicate denoted by the second ATJ type;
     this is necessary, but not sufficient condition.
     The reason for this necessary condition is that, in ATJ's type analysis,
     the types of the branches of an @(tsee if) are merged
     via the least upper bound operation for the partial order:
     therefore, it must be the case that ACL2 values of a smaller type
     are also ACL2 values of a larger type.")
   (xdoc::p
    "For the @(':acl2') types, we want to automatically convert Java values
     exactly according to the hierarchy of the corresponding ACL2 values,
     e.g. we want to automatically convert from @(':asymbol') to @(':avalue').
     Note that the necessary condition explained above is also satisfied.")
   (xdoc::p
    "On the other hand, we do not want any automatic conversions
     among the @(':jprim') or @(':jprimarr') types,
     or between them and the @(':acl2') types.
     The purpose of the @(':j...') types is
     to generate very specific and idiomatic Java code,
     and so we do not want to inadvertently let
     Java primitive values and arrays be turned into
     (Java representations of) the ACL2 values.
     This shows why the necessary condition described above
     is not also a sufficient one:
     of course the ACL2 values that model Java primitive values and arrays
     are in the ACL2 predicate denoted by the type @(':avalue'),
     but despite that any @(':j...') is not a subtype of @(':avalue').")
   (xdoc::p
    "To validate this definition of partial order,
     we prove that the relation is indeed a partial order,
     i.e. reflexive, anti-symmetric, and transitive.
     We also prove that @(tsee atj-type-to-pred) is monotonic,
     i.e. that for each subtype/supertype pair
     each value satisfying the subtype's predicate
     also satisfies the supertype's predicate;
     we generate a theorem for each such pair,
     because the predicate inclusion relation is at the meta level.
     The motonocity theorem validates that the partial order
     satisfies the necessary condition described above.")
   (xdoc::p
    "While @(tsee atj-type-to-pred) is order-presering (i.e. monotonic),
     it is not order-reflecting (and thus not an order embedding):
     if @('(atj-type-to-pred x)') is included in @('(atj-type-to-pred y)'),
     @('(atj-type-<= x y)') does not necessarily hold.
     The counterexample to being order-reflective consists of
     @('x') being a @(':jprim') or @(':jprimarr') type and
     @('y') being the @(':acl2') type of all ACL2 values.
     In other words, as explained above,
     the necessary condition described above is not sufficient."))
  (atj-type-case
   sub
   :acl2 (atj-type-case sup
                        :acl2 (atj-atype-<= sub.get sup.get)
                        :jprim nil
                        :jprimarr nil)
   :jprim (atj-type-case sup
                         :acl2 nil
                         :jprim (primitive-type-equiv sub.get sup.get)
                         :jprimarr nil)
   :jprimarr (atj-type-case sup
                            :acl2 nil
                            :jprim nil
                            :jprimarr (primitive-type-equiv sub.comp sup.comp)))
  :hooks (:fix)
  ///

  (defrule atj-type-<=-reflexive
    (implies (atj-type-equiv x y)
             (atj-type-<= x y)))

  (defrule atj-type-<=-antisymmetric
    (implies (and (atj-type-<= x y)
                  (atj-type-<= y x))
             (atj-type-equiv x y))
    :disable (atj-type-equiv atj-atype-equiv primitive-type-equiv)
    :enable (atj-type-equiv-alt-def
             atj-atype-<=-antisymmetric))

  (defrule atj-type-<=-transitive
    (implies (and (atj-type-<= x y)
                  (atj-type-<= y z))
             (atj-type-<= x z))
    :enable atj-atype-<=-transitive)
  ;; rewrite rule ATJ-ATYPE-<=-OF-VALUE-LEFT applies
  ;; rewrite rule ATJ-ATYPE-<=-OF-CONS-LEFT applies

  ;; monotonicity theorem for (SUB, SUP) if SUB <= SUP, otherwise NIL:
  (define atj-type-to-pred-gen-mono-thm ((sub atj-typep) (sup atj-typep))
    (if (atj-type-<= sub sup)
        `((defthm ,(packn (list 'atj-type-to-pred-thm-
                                (atj-type-to-keyword sub)
                                '-
                                (atj-type-to-keyword sup)))
            (implies (,(atj-type-to-pred sub) val)
                     (,(atj-type-to-pred sup) val))
            :rule-classes nil))
      nil)
    :hooks (:fix))

  ;; monotonicity theorems for all (SUB, SUP) with SUP' in SUPS:
  (define atj-type-to-pred-gen-mono-thms-1 ((sub atj-typep)
                                            (sups atj-type-listp))
    (cond ((endp sups) nil)
          (t (append (atj-type-to-pred-gen-mono-thm sub (car sups))
                     (atj-type-to-pred-gen-mono-thms-1 sub (cdr sups)))))
    :hooks (:fix))

  ;; monotonicity theorems for all (SUB, SUP) with SUB in SUBS and SUP in SUPS:
  (define atj-type-to-pred-gen-mono-thms-2 ((subs atj-type-listp)
                                            (sups atj-type-listp))
    (cond ((endp subs) nil)
          (t (append (atj-type-to-pred-gen-mono-thms-1 (car subs) sups)
                     (atj-type-to-pred-gen-mono-thms-2 (cdr subs) sups))))
    :hooks (:fix))

  ;; monotonicity theorems for all pairs of types:
  (define atj-type-to-pred-gen-mono-thms ()
    (b* ((types (list (atj-type-acl2 (atj-atype-integer))
                      (atj-type-acl2 (atj-atype-rational))
                      (atj-type-acl2 (atj-atype-number))
                      (atj-type-acl2 (atj-atype-character))
                      (atj-type-acl2 (atj-atype-string))
                      (atj-type-acl2 (atj-atype-symbol))
                      (atj-type-acl2 (atj-atype-boolean))
                      (atj-type-acl2 (atj-atype-cons))
                      (atj-type-acl2 (atj-atype-value))
                      (atj-type-jprim (primitive-type-boolean))
                      (atj-type-jprim (primitive-type-char))
                      (atj-type-jprim (primitive-type-byte))
                      (atj-type-jprim (primitive-type-short))
                      (atj-type-jprim (primitive-type-int))
                      (atj-type-jprim (primitive-type-long))
                      (atj-type-jprim (primitive-type-float))
                      (atj-type-jprim (primitive-type-double))
                      (atj-type-jprimarr (primitive-type-boolean))
                      (atj-type-jprimarr (primitive-type-char))
                      (atj-type-jprimarr (primitive-type-byte))
                      (atj-type-jprimarr (primitive-type-short))
                      (atj-type-jprimarr (primitive-type-int))
                      (atj-type-jprimarr (primitive-type-long))
                      (atj-type-jprimarr (primitive-type-float))
                      (atj-type-jprimarr (primitive-type-double)))))
      `(encapsulate
         ()
         (set-ignore-ok t)
         ,@(atj-type-to-pred-gen-mono-thms-2 types types))))

  ;; macro to generate the monotonicity theorems:
  (defmacro atj-type-to-pred-mono ()
    `(make-event (atj-type-to-pred-gen-mono-thms)))

  ;; generate the monotonicity theorems:
  (atj-type-to-pred-mono))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-< ((sub atj-typep) (sup atj-typep))
  :returns (yes/no booleanp)
  :short "Irreflexive kernel (i.e. strict version) of @(tsee atj-type-<=)."
  :long
  (xdoc::topstring
   "Since @(tsee atj-type-<=) fixes its arguments,
    we use (in)equality modulo fixing here,
    so that this function fixes its arguments too.")
  (and (atj-type-<= sub sup)
       (not (atj-type-equiv sub sup)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-<= ((sub atj-type-listp) (sup atj-type-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-type-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-type-<= (car sub) (car sup))
                (atj-type-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-type-list-<=-reflexive
    (implies (atj-type-list-equiv x y)
             (atj-type-list-<= x y))
    :enable atj-type-list-fix)

  (defrule atj-type-list-<=-antisymmetric
    (implies (and (atj-type-list-<= x y)
                  (atj-type-list-<= y x))
             (atj-type-list-equiv x y))
    :disable atj-type-list-equiv
    :enable atj-type-list-equiv-alt-def)

  (defrule atj-type-list-<=-transitive
    (implies (and (atj-type-list-<= x y)
                  (atj-type-list-<= y z))
             (atj-type-list-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-< ((sub atj-type-listp) (sup atj-type-listp))
  :returns (yes/no booleanp)
  :short "Irreflexive kernel (i.e. strict version)
          of @(tsee atj-type-list-<=)."
  :long
  (xdoc::topstring
   "Since @(tsee atj-type-list-<=) fixes its arguments,
    we use (in)equality modulo fixing here,
    so that this function fixes its arguments too.")
  (and (atj-type-list-<= sub sup)
       (not (atj-type-list-equiv sub sup)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-top-<= ((sub atj-maybe-typep) (sup atj-maybe-typep))
  :returns (yes/no booleanp)
  :short "Extension of @(tsee atj-type-<=) to include @('nil') as top."
  :long
  (xdoc::topstring
   (xdoc::p
    "For certain purposes, we want to calculate
     the least upper bound of two ATJ types w.r.t. @(tsee atj-type-<=);
     see @(tsee atj-type-top-join).
     However, the ATJ types with this partial order
     do not quite form a join semilattice,
     because there are no upper bounds, for instance,
     for two different @(':jprim') types.")
   (xdoc::p
    "Thus, we extend the partial order
     to the set of ATJ types plus @('nil'),
     where @('nil') is above every type.")
   (xdoc::p
    "We show that this extended relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (or (null sup)
      (and (not (null sub))
           (atj-type-<= sub sup)))
  :hooks (:fix)
  ///

  (defrule atj-type-top-<=-reflexive
    (implies (atj-maybe-type-equiv x y)
             (atj-type-top-<= x y)))

  (defrule atj-type-top-<=-antisymmetric
    (implies (and (atj-type-top-<= x y)
                  (atj-type-top-<= y x))
             (atj-maybe-type-equiv x y))
    :enable (atj-maybe-type-fix
             atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp
             atj-type-<=
             atj-atype-<=))

  (defrule atj-type-top-<=-transitive
    (implies (and (atj-type-top-<= x y)
                  (atj-type-top-<= y z))
             (atj-type-top-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-top-join ((x atj-maybe-typep) (y atj-maybe-typep))
  :returns (lub atj-maybe-typep)
  :short "Least upper bound of two ATJ types or @('nil')s,
          with respect to the partial order @(tsee atj-type-top-<=)."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in @(tsee atj-type-top-<=),
     the addition of @('nil') as top element of the partial order
     results in a join semilattice.")
   (xdoc::p
    "We define this operation by five cases:
     the first two are obvious,
     while the remaining three are motivated by the fact that
     @(':acons'), @(':avalue'), and @('nil') are the only elements
     each of which has more than one elements that are strictly smaller.")
   (xdoc::p
    "To validate this definition of least upper bound,
     we prove that the this operation indeed returns an upper bound
     that is less than or equal to any other upper bound,
     i.e. that it returns the least upper bound.")
   (xdoc::p
    "The commutativity, idempotence, and associativity of the join operation
     follows from these and the partial order properties,
     according to lattice theory.
     So we do not prove these properties explicitly here."))
  (cond ((atj-type-top-<= x y) (atj-maybe-type-fix y))
        ((atj-type-top-<= y x) (atj-maybe-type-fix x))
        ((and (atj-type-top-<= x (atj-type-acl2 (atj-atype-cons)))
              (atj-type-top-<= y (atj-type-acl2 (atj-atype-cons))))
         (atj-type-acl2 (atj-atype-cons)))
        ((and (atj-type-top-<= x (atj-type-acl2 (atj-atype-value)))
              (atj-type-top-<= y (atj-type-acl2 (atj-atype-value))))
         (atj-type-acl2 (atj-atype-value)))
        (t nil))
  :hooks (:fix)
  ///

  (defrule atj-type-top-join-upper-bound-left
    (atj-type-top-<= x (atj-type-top-join x y))
    :enable atj-type-top-<=)

  (defrule atj-type-top-join-upper-bound-right
    (atj-type-top-<= y (atj-type-top-join x y))
    :enable atj-type-top-<=)

  (defrule atj-type-top-join-least
    (implies (and (atj-type-top-<= x z)
                  (atj-type-top-<= y z))
             (atj-type-top-<= (atj-type-top-join x y) z))
    :enable (atj-type-top-<= atj-type-<= atj-atype-<=)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-join ((x atj-typep) (y atj-typep))
  :returns (lub atj-maybe-typep)
  :short "Least upper bound of two ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "We have defined @(tsee atj-type-top-join)
     in order to exhibit and prove the semilattice structure,
     but we always want to use ATJ types as arguments, never @('nil').
     So we introduce this function,
     which operates on types but may return @('nil'),
     which can be also interpreted as saying that
     the two ATJ types have no (least) upper bound w.r.t @(tsee atj-type-<=)."))
  (atj-type-top-join x y))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-top-list-<= ((sub atj-maybe-type-listp)
                              (sup atj-maybe-type-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-type-top-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-type-top-<= (car sub) (car sup))
                (atj-type-top-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-type-top-list-<=-reflexive
    (implies (atj-maybe-type-list-equiv x y)
             (atj-type-top-list-<= x y))
    :enable atj-maybe-type-list-fix)

  (defrule atj-type-top-list-<=-antisymmetric
    (implies (and (atj-type-top-list-<= x y)
                  (atj-type-top-list-<= y x))
             (atj-maybe-type-list-equiv x y))
    :disable atj-maybe-type-list-equiv
    :enable atj-maybe-type-list-equiv-alt-def)

  (defrule atj-type-top-list-<=-transitive
    (implies (and (atj-type-top-list-<= x y)
                  (atj-type-top-list-<= y z))
             (atj-type-top-list-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-top-list-join ((x atj-maybe-type-listp)
                                (y atj-maybe-type-listp))
  :returns (lub atj-maybe-type-listp)
  :short "Lift @(tsee atj-type-top-join) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return (what remains of) the longer list.")
   (xdoc::p
    "We show that this indeed returns the least upper bound
     of the order relation lifted to lists."))
  (cond ((endp x) (atj-maybe-type-list-fix y))
        ((endp y) (atj-maybe-type-list-fix x))
        (t (cons (atj-type-top-join (car x) (car y))
                 (atj-type-top-list-join (cdr x) (cdr y)))))
  :hooks (:fix)
  ///

  (defrule atj-type-top-list-join-upper-bound-left
    (atj-type-top-list-<= x (atj-type-top-list-join x y))
    :enable atj-type-top-list-<=)

  (defrule atj-type-top-list-join-upper-bound-right
    (atj-type-top-list-<= y (atj-type-top-list-join x y))
    :enable atj-type-top-list-<=)

  (defrule atj-type-top-list-join-least
    (implies (and (atj-type-top-list-<= x z)
                  (atj-type-top-list-<= y z))
             (atj-type-top-list-<= (atj-type-top-list-join x y) z))
    :enable atj-type-top-list-<=)

  (defret consp-of-atj-type-top-list-join
    (equal (consp lub)
           (or (consp x)
               (consp y)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-join ((x atj-type-listp) (y atj-type-listp))
  :guard (= (len x) (len y))
  :returns (lub atj-maybe-type-listp)
  :short "Lift @(tsee atj-type-join) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return (what remains of) the longer list.")
   (xdoc::p
    "We show that this agrees with @(tsee atj-type-top-list-join)
     over lists of ATJ types.
     Note that @(tsee atj-type-top-list-join) has been defined
     just to show the semilattice properties,
     but we always want to use, as arguments,
     lists of ATJ types without @('nil')s of the same length
     (so we add a length equality requirement to the guard)."))
  (cond ((endp x) (atj-type-list-fix y))
        ((endp y) (atj-type-list-fix x))
        (t (cons (atj-type-join (car x) (car y))
                 (atj-type-list-join (cdr x) (cdr y)))))
  ///

  (defruled atj-type-list-join-alt-def
    (implies (and (atj-type-listp x)
                  (atj-type-listp y))
             (equal (atj-type-list-join x y)
                    (atj-type-top-list-join x y)))
    :enable (atj-type-top-list-join
             atj-type-join))

  (defret consp-of-atj-type-list-join
    (equal (consp lub)
           (or (consp x)
               (consp y)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-bottom-<= ((sub atj-maybe-typep) (sup atj-maybe-typep))
  :returns (yes/no booleanp)
  :short "Extension of @(tsee atj-type-<=) to include @('nil') as bottom."
  :long
  (xdoc::topstring
   (xdoc::p
    "For certain purposes, we want to calculate
     the greatest lower bound of two ATJ types w.r.t. @(tsee atj-type-<=);
     see @(tsee atj-type-bottom-meet).
     However, the ATJ types with this partial order
     do not quite form a meet semilattice,
     because there are no lower bounds, for instance,
     for two different @(':jprim') types.")
   (xdoc::p
    "Thus, we extend the partial order
     to the set of ATJ types plus @('nil'),
     where @('nil') is below every type.")
   (xdoc::p
    "We show that this extended relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (or (null sub)
      (and (not (null sup))
           (atj-type-<= sub sup)))
  :hooks (:fix)
  ///

  (defrule atj-type-bottom-<=-reflexive
    (implies (atj-maybe-type-equiv x y)
             (atj-type-bottom-<= x y)))

  (defrule atj-type-bottom-<=-antisymmetric
    (implies (and (atj-type-bottom-<= x y)
                  (atj-type-bottom-<= y x))
             (atj-maybe-type-equiv x y))
    :enable (atj-maybe-type-fix
             atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp
             atj-type-<=
             atj-atype-<=))

  (defrule atj-type-bottom-<=-transitive
    (implies (and (atj-type-bottom-<= x y)
                  (atj-type-bottom-<= y z))
             (atj-type-bottom-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-bottom-meet ((x atj-maybe-typep) (y atj-maybe-typep))
  :returns (glb atj-maybe-typep)
  :short "Greatest lower bound of two ATJ types or @('nil')s,
          with respect to the partial order @(tsee atj-type-bottom-<=)."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in @(tsee atj-type-bottom-<=),
     the addition of @('nil') as bottom element of the partial order
     results in a meet semilattice.")
   (xdoc::p
    "We define this operation by three cases:
     the first two are obvious,
     and the third one suffices because no element of the semilattice
     has more than one elements that are strictly larger.")
   (xdoc::p
    "To validate this definition of greatest lower bound,
     we prove that the this operation indeed returns a lower bound
     that is greater than or equal to any other lower bound,
     i.e. that it returns the greatest lower bound.")
   (xdoc::p
    "The commutativity, idempotence, and associativity of the join operation
     follows from these and the partial order properties,
     according to lattice theory.
     So we do not prove these properties explicitly here."))
  (cond ((atj-type-bottom-<= x y) (atj-maybe-type-fix x))
        ((atj-type-bottom-<= y x) (atj-maybe-type-fix y))
        (t nil))
  :hooks (:fix)
  ///

  (defrule atj-type-bottom-meet-lower-bound-left
    (atj-type-bottom-<= (atj-type-bottom-meet x y) x)
    :enable atj-type-bottom-<=)

  (defrule atj-type-bottom-meet-lower-bound-right
    (atj-type-bottom-<= (atj-type-bottom-meet x y) y)
    :enable atj-type-bottom-<=)

  (defrule atj-type-bottom-meet-greatest
    (implies (and (atj-type-bottom-<= z x)
                  (atj-type-bottom-<= z y))
             (atj-type-bottom-<= z (atj-type-bottom-meet x y)))
    :enable (atj-type-bottom-<= atj-type-<= atj-atype-<=)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-meet ((x atj-typep) (y atj-typep))
  :returns (glb atj-maybe-typep)
  :short "Greatest lower bound of two ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "We have defined @(tsee atj-type-bottom-meet)
     in order to exhibit and prove the semilattice structure,
     but we always want to use ATJ types as arguments, never @('nil').
     So we introduce this function,
     which operates on types but may return @('nil'),
     which can be also interpreted as saying that
     the two ATJ types have no (greatest) lower bound
     w.r.t @(tsee atj-type-<=)."))
  (atj-type-bottom-meet x y))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-bottom-list-<= ((sub atj-maybe-type-listp)
                                 (sup atj-maybe-type-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-type-bottom-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-type-bottom-<= (car sub) (car sup))
                (atj-type-bottom-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-type-bottom-list-<=-reflexive
    (implies (atj-maybe-type-list-equiv x y)
             (atj-type-bottom-list-<= x y))
    :enable atj-maybe-type-list-fix)

  (defrule atj-type-bottom-list-<=-antisymmetric
    (implies (and (atj-type-bottom-list-<= x y)
                  (atj-type-bottom-list-<= y x))
             (atj-maybe-type-list-equiv x y))
    :disable atj-maybe-type-list-equiv
    :enable atj-maybe-type-list-equiv-alt-def)

  (defrule atj-type-bottom-list-<=-transitive
    (implies (and (atj-type-bottom-list-<= x y)
                  (atj-type-bottom-list-<= y z))
             (atj-type-bottom-list-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-bottom-list-meet ((x atj-maybe-type-listp)
                                   (y atj-maybe-type-listp))
  :returns (glb atj-maybe-type-listp)
  :short "Lift @(tsee atj-type-bottom-meet) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return @('nil').")
   (xdoc::p
    "We show that this indeed returns the greatest lower bound
     of the order relation lifted to lists."))
  (cond ((endp x) nil)
        ((endp y) nil)
        (t (cons (atj-type-bottom-meet (car x) (car y))
                 (atj-type-bottom-list-meet (cdr x) (cdr y)))))
  :hooks (:fix)
  ///

  (defrule atj-type-bottom-list-meet-lower-bound-left
    (atj-type-bottom-list-<= (atj-type-bottom-list-meet x y) x)
    :enable atj-type-bottom-list-<=)

  (defrule atj-type-bottom-list-meet-lower-bound-right
    (atj-type-bottom-list-<= (atj-type-bottom-list-meet x y) y)
    :enable atj-type-bottom-list-<=)

  (defrule atj-type-bottom-list-meet-greatest
    (implies (and (atj-type-bottom-list-<= z x)
                  (atj-type-bottom-list-<= z y))
             (atj-type-bottom-list-<= z (atj-type-bottom-list-meet x y)))
    :enable atj-type-bottom-list-<=))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-meet ((x atj-type-listp) (y atj-type-listp))
  :guard (= (len x) (len y))
  :returns (glb atj-maybe-type-listp)
  :short "Lift @(tsee atj-type-meet) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return @('nil').")
   (xdoc::p
    "We show that this agrees with @(tsee atj-type-bottom-list-meet)
     over lists of ATJ types.
     Note that @(tsee atj-type-bottom-list-meet) has been defined
     just to show the semilattice properties,
     but we always want to use, as arguments,
     lists of ATJ types without @('nil')s of the same length
     (so we add a length equality requirement to the guard)."))
  (cond ((endp x) nil)
        ((endp y) nil)
        (t (cons (atj-type-meet (car x) (car y))
                 (atj-type-list-meet (cdr x) (cdr y)))))
  ///

  (defruled atj-type-list-meet-alt-def
    (implies (and (atj-type-listp x)
                  (atj-type-listp y))
             (equal (atj-type-list-meet x y)
                    (atj-type-bottom-list-meet x y)))
    :enable (atj-type-bottom-list-meet
             atj-type-meet)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc atj-type-semilattices
  :short "The two semilattices consisting of ATJ types and @('nil')."
  :long
  (xdoc::topstring
   (xdoc::p
    "Above we have introduced and verified two separate semilattices.
     Both semilattices consist of the values in @(tsee atj-maybe-typep),
     but in the join semilattice @('nil') is the top,
     while in the meet semilattice @('nil') is the bottom.")
   (xdoc::p
    "These two semilattices are not the semilattice halves of a lattice:
     @(tsee atj-maybe-typep) does not form a lattice,
     because @('nil') cannot be simultaneously top and bottom,
     and in fact the two partial orders,
     @(tsee atj-type-top-<=) and @(tsee atj-type-bottom-<=),
     differ even though they are defined on the same set.")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-jitype ((type atj-typep))
  :returns (jtype atj-jitypep)
  :short "Java input type denoted by an ATJ type."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @(':acl2') types except
     @(':aboolean'), @(':acharacter'), and @(':astring')
     denote the corresponding AIJ class types.
     The @(':aboolean') type denotes
     the Java primitive type @('boolean').
     The @(':acharacter') type denotes
     the Java primitive type @('char').
     The @(':astring') type denotes
     the Java reference type @('java.lang.String').
     The @(':jprim') types denote
     the corresponding Java primitive types.
     The @(':jprimarr') types denote
     the corresponding Java primitive array types.")
   (xdoc::p
    "The mapping of @(':aboolean'), @(':acharacter'), and @(':astring')
     to the Java @('boolean'), @('char'), and @('String') types
     means that we represent
     ACL2 booleans as Java booleans,
     ACL2 characters as Java characters, and
     ACL2 strings as Java strings.
     This only happens in the shallow embedding approach;
     the deep embedding approach does not use ATJ types.
     Also, @(':aboolean'), @(':acharacter'), and @(':astring')
     are used only if @(':guards') is @('t');
     otherwise, only the type @(':avalue') is used.
     In other words, we represent
     ACL2 booleans/characters/strings as Java booleans/characters/strings
     only when @(':guards') is @('t').
     Even though Java @('char') values (which consist of 16 bits)
     are not isomorphic to ACL2 characters (which consist of 8 bits),
     when @(':guards') is @('t') the satisfaction of all guards is assumed;
     thus, if external code calls the generated Java code
     with values that satisfy the guards,
     and in particular with @('char') values below 256,
     the generate code should manipulate only @('char') values below 256,
     which are isomorphic to Java characters.
     The same consideration applies to ACL2 strings vs. Java strings;
     only Java strings with characters below 256
     should be passed to ATJ-generated code,
     which will only manipulate strings satisfying that property."))
  (atj-type-case
   type
   :acl2 (atj-atype-case type.get
                         :integer *aij-type-int*
                         :rational *aij-type-rational*
                         :number *aij-type-number*
                         :character (jtype-char)
                         :string (jtype-class "String")
                         :symbol *aij-type-symbol*
                         :boolean (jtype-boolean)
                         :cons *aij-type-cons*
                         :value *aij-type-value*)
   :jprim (primitive-type-case type.get
                               :boolean (jtype-boolean)
                               :char (jtype-char)
                               :byte (jtype-byte)
                               :short (jtype-short)
                               :int (jtype-int)
                               :long (jtype-long)
                               :float (jtype-float)
                               :double (jtype-double))
   :jprimarr (primitive-type-case type.comp
                                  :boolean (jtype-array (jtype-boolean))
                                  :char (jtype-array (jtype-char))
                                  :byte (jtype-array (jtype-byte))
                                  :short (jtype-array (jtype-short))
                                  :int (jtype-array (jtype-int))
                                  :long (jtype-array (jtype-long))
                                  :float (jtype-array (jtype-float))
                                  :double (jtype-array (jtype-double))))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-jitype-list ((types atj-type-listp))
  :returns (jtypes atj-jitype-listp)
  :short "Lift @(tsee atj-type-to-jitype) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-jitype (car types))
                 (atj-type-list-to-jitype-list (cdr types)))))
  :hooks (:fix)
  ///

  (defret len-of-atj-type-list-to-jitype-list
    (equal (len jtypes)
           (len types))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-of-value (val)
  :returns (type atj-typep)
  :short "ATJ type of an ACL2 value."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the type that ATJ assigns to a quoted constant
     with the given value.
     Note that a constant like @('2') does not get type @(':jint').
     Instead, ATJ assigns @(':jint') to a term like @('(int-value 2)')
     (when @(':deep') is @('nil') and @(':guards') is @('t'));
     see the code generation functions."))
  (cond ((integerp val) (atj-type-acl2 (atj-atype-integer)))
        ((rationalp val) (atj-type-acl2 (atj-atype-rational)))
        ((acl2-numberp val) (atj-type-acl2 (atj-atype-number)))
        ((characterp val) (atj-type-acl2 (atj-atype-character)))
        ((stringp val) (atj-type-acl2 (atj-atype-string)))
        ((booleanp val) (atj-type-acl2 (atj-atype-boolean)))
        ((symbolp val) (atj-type-acl2 (atj-atype-symbol)))
        ((consp val) (atj-type-acl2 (atj-atype-cons)))
        (t (prog2$ (raise "Internal errror: ~
                           the value ~x0 is not a number, ~
                           a character, a string, a symbol, or a CONS."
                          val)
                   (atj-type-irrelevant)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod atj-function-type
  :short "Fixtype of ATJ function types."
  :long
  (xdoc::topstring
   (xdoc::p
    "An ATJ function type consists of
     (zero or more) types for the arguments (i.e. inputs)
     and (one or more) types for the results (i.e. outputs).
     This is like an arrow type in higher-order languages.")
   (xdoc::p
    "We also augment the output types with array names.
     These are represented via a list of symbols,
     whose length must match the length of the output type list
     (this length constraint is not explicitly captured in this fixtype,
     but it is an expected invariant).
     The @('nil') symbol may be used in any position of the list,
     meaning that there is no array name for the corresponding output type.
     A non-@('nil') symbol may be used only in a position
     whose corresponding output type is a @(':jprimarr') type.
     In this case the symbol must match a formal parameter of the function
     that has the same array type as input type.
     The non-@('nil') symbols must be all distinct.")
   (xdoc::p
    "The purpose of these array names is to support
     the analysis of single-threaded use of Java primitive arrays
     described at @(see atj-pre-translation-array-analysis).
     The idea is that if a function takes an array as input
     (i.e. that input type is a @(':jprimarr') type)
     and if the function may modify that array (directly or indirectly),
     then the possibly modified array must be returned as a result:
     an explicit non-@('nil') array name assigned to a result
     specifies which result that is, and simplifies the analysis.
     If instead a function does not modify an input array,
     no result with the same name as the input needs to exist.
     Results of non-array types use @('nil') as array (non-)name.
     If a function creates an array (directly or indirectly) and returns it,
     then @('nil') is used for that result,
     i.e. the array has no name because it does not modify any input array
     (and thus there is no input name to match);
     it represents a newly created array."))
  ((inputs atj-type-list)
   (outputs atj-type-list)
   (arrays symbol-list))
  :layout :list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-function-type
  atj-function-type-p
  :short "Fixtype of ATJ function types and @('nil')."
  :pred atj-maybe-function-type-p)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-function-type-list
  :short "Fixtype of lists of ATJ function types."
  :elt-type atj-function-type
  :true-listp t
  :elementp-of-nil nil
  :pred atj-function-type-listp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-list->inputs ((fn-types atj-function-type-listp))
  :returns (in-typess atj-type-list-listp)
  :short "Lift @(tsee atj-function-type->inputs) to lists."
  (cond ((endp fn-types) nil)
        (t (cons (atj-function-type->inputs (car fn-types))
                 (atj-function-type-list->inputs (cdr fn-types))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-list->outputs ((fn-types atj-function-type-listp))
  :returns (out-typess atj-type-list-listp)
  :short "Lift @(tsee atj-function-type->outputs) to lists."
  (cond ((endp fn-types) nil)
        (t (cons (atj-function-type->outputs (car fn-types))
                 (atj-function-type-list->outputs (cdr fn-types))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod atj-function-type-info
  :short "Fixtype of ATJ function type information."
  :long
  (xdoc::topstring
   (xdoc::p
    "In general, each ACL2 function has, associated with it,
     a primary (`main') function type
     and zero or more secondary (`other') function types,
     as mentioned in @(see atj-types)."))
  ((main atj-function-type)
   (others atj-function-type-list))
  :layout :list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-info->outputs ((info atj-function-type-info-p))
  :returns (out-typess atj-type-list-listp)
  :short "Return the list of all the output type lists
          in a function's type information."
  (cons
   (atj-function-type->outputs (atj-function-type-info->main info))
   (atj-function-type-list->outputs (atj-function-type-info->others info))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-function-type-info
  atj-function-type-info
  :short "Fixtype of ATJ function type information and @('nil').")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-number-of-results ((fn symbolp) (wrld plist-worldp))
  :returns (numres posp)
  :short "Number of results returned by a function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is similar to @(tsee number-of-results+),
     but that function has a guard disallowing the function symbol
     to be a member of the built-in constant @('*stobjs-out-invalid*'),
     i.e. to be @(tsee if) or @(tsee return-last).
     For ATJ's purpose, we totalize @(tsee number-of-results+)
     by having it return 1 on these two functions.
     The actual result is irrelevant,
     because @(tsee return-last) is removed
     by one of ATJ's pre-translation steps,
     and @(tsee if) is treated specially by ATJ."))
  (cond ((member-eq fn *stobjs-out-invalid*) 1)
        (t (number-of-results+ fn wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defval *atj-function-type-info-table-name*
  :short "Name of the table that associates ATJ types to ACL2 functions."
  'atj-function-type-info-table)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection atj-function-type-info-table
  :short "Table that associates ATJ types to ACL2 functions."
  :long
  (xdoc::topstring
   (xdoc::p
    "This table is populated by successful calls of
     the @(tsee atj-main-function-type)
     and @(tsee atj-other-function-type) macros."))
  (make-event
   `(table ,*atj-function-type-info-table-name* nil nil
      :guard (and (symbolp acl2::key)
                  (atj-function-type-info-p acl2::val)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-get-function-type-info-from-table ((fn symbolp) (wrld plist-worldp))
  :returns (fn-info? atj-maybe-function-type-info-p)
  :short "Retrieve the ATJ function type information
          of the specified function from the table."
  :long
  (xdoc::topstring-p
   "If the table has no entry for the function, @('nil') is returned.")
  (b* ((table (table-alist+ *atj-function-type-info-table-name* wrld))
       (pair (assoc-eq fn table))
       ((when pair)
        (b* ((fn-info (cdr pair)))
          (if (atj-function-type-info-p fn-info)
              fn-info
            (raise "Internal error: ~
                    malformed function information ~x0 for function ~x1."
                   fn-info fn)))))
    nil)
  :prepwork ((local (include-book "std/alists/assoc" :dir :system)))
  ///

  (defrule atj-function-type-info-p-of-atj-get-function-type-info-from-table
    (iff (atj-function-type-info-p
          (atj-get-function-type-info-from-table fn wrld))
         (atj-get-function-type-info-from-table fn wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-info-default ((fn symbolp) (wrld plist-worldp))
  :returns (fn-info atj-function-type-info-p)
  :short "Default ATJ function type information for a function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is used when a function has no entry in the table.
     It consists of a primary function type of all @(':avalue') types,
     no secondary function types,
     and all @('nil') array names (since there are no array types).")
   (xdoc::p
    "To calculate the output types,
     we need the number of results returned by @('fn')."))
  (b* ((number-of-inputs (arity+ fn wrld))
       (number-of-outputs (atj-number-of-results fn wrld)))
    (make-atj-function-type-info
     :main (make-atj-function-type
            :inputs (repeat number-of-inputs
                            (atj-type-acl2 (atj-atype-value)))
            :outputs (repeat number-of-outputs
                             (atj-type-acl2 (atj-atype-value)))
            :arrays (repeat number-of-outputs nil))
     :others nil))
  :prepwork ((local
              (include-book "std/typed-lists/symbol-listp" :dir :system))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-get-function-type-info ((fn symbolp)
                                    (guards$ booleanp)
                                    (wrld plist-worldp))
  :returns (fn-info atj-function-type-info-p)
  :short "Obtain the ATJ function type information of the specified function."
  :long
  (xdoc::topstring
   (xdoc::p
    "If the @(':guards') input is @('t'),
     we retrieve the type information from the table
     via @(tsee atj-get-function-type-info-from-table).
     If the @(':guards') input is @('nil'),
     we return the defult function type information,
     because in this case types are effectively ignored."))
  (if guards$
      (b* ((fn-info? (atj-get-function-type-info-from-table fn wrld)))
        (or fn-info? (atj-function-type-info-default fn wrld)))
    (atj-function-type-info-default fn wrld)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-of-min-input-types
  ((in-types atj-type-listp)
   (fn-types atj-function-type-listp))
  :returns (fn-type? atj-maybe-function-type-p)
  :short "Function type with the minimum input types."
  :long
  (xdoc::topstring
   (xdoc::p
    "When this function is called,
     @('in-types') are the types inferred for
     the actual arguments of a function call,
     and @('fn-types') are all the function types (primary and secondary)
     of the called function.
     The goal here is to see if the argument types match any function type,
     in the sense that the input types of the function type
     are greater than or equal to the types of the actual arguments.
     If no such function type is found, we return @('nil').
     If instead some exist, we select the one with the  minimum input types,
     which always exists because of the closure property
     enforced by @(tsee atj-other-function-type),
     and we return that function type.
     In other words, given the types of the actual arguments,
     the output types of the returned function type (if any)
     gives us the inferred types of the results of the function call,
     and also (through the mapping from ATJ types to Java types)
     the result type of the overloaded method
     that will be resolved at compile time."))
  (atj-function-type-of-min-input-types-aux in-types
                                            (atj-function-type-list-fix
                                             fn-types)
                                            nil
                                            nil)

  :prepwork
  ((define atj-function-type-of-min-input-types-aux
     ((in-types atj-type-listp)
      (fn-types atj-function-type-listp)
      (current-min-in-types atj-type-listp)
      (current-fn-type? atj-maybe-function-type-p))
     :returns (fn-type? atj-maybe-function-type-p
                        :hyp (and (atj-function-type-listp fn-types)
                                  (atj-maybe-function-type-p current-fn-type?)))
     (b* (((when (endp fn-types)) current-fn-type?)
          (fn-type (car fn-types))
          (fn-in-types (atj-function-type->inputs fn-type))
          ((mv current-min-in-types current-fn-type?)
           (if (and (atj-type-list-<= in-types fn-in-types)
                    (or (null current-fn-type?) ; i.e. none found yet
                        (atj-type-list-< fn-in-types current-min-in-types)))
               (mv fn-in-types fn-type)
             (mv current-min-in-types current-fn-type?))))
       (atj-function-type-of-min-input-types-aux in-types
                                                 (cdr fn-types)
                                                 current-min-in-types
                                                 current-fn-type?)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-type-list ((type atj-typep))
  :returns (types atj-type-listp)
  :short "Turn a single ATJ type into a singleton list of it."
  :long
  (xdoc::topstring-p
   "This is just @(tsee list),
    but we introduce an explicit function for greater clarity.")
  (list (atj-type-fix type))
  :hooks (:fix)
  ///

  (more-returns
   (types consp :rule-classes :type-prescription)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-type-list-list ((types atj-type-listp))
  :returns (typess atj-type-list-listp)
  :short "Lift @(tsee atj-type-to-type-list) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-type-list (car types))
                 (atj-type-list-to-type-list-list (cdr types)))))
  :hooks (:fix)
  ///

  (more-returns
   (typess cons-listp))

  (defret len-of-atj-type-list-to-type-list-list
    (equal (len typess)
           (len types))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-type ((types atj-type-listp))
  :guard (consp types)
  :returns (type atj-typep)
  :short "Ensure that a non-empty list of types is a singleton,
          and return its only element."
  :long
  (xdoc::topstring
   (xdoc::p
    "In some cases, a non-empty list of types is expected to be a singleton.
     For instance, the type list may be the output types of a function
     that is known to return a single result.
     This utility can be used in these cases,
     to check the expectation for robustness,
     and to retrieve the single type from the singleton list."))
  (if (= (len types) 1)
      (atj-type-fix (car types))
    (prog2$
     (raise "Internal error: ~x0 is not a singleton list of types." types)
     (atj-type-irrelevant)))
  :hooks (:fix))
