/*###############################################################################
# Linux Management Providers (LMP), OpenDRIM_RecordLogPackage provider package
# Copyright (C) 2007 Ilsoo Byun, ETRI <widepis@etri.re.kr, widepis@empal.com>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
###############################################################################*/

#include "cmpiOpenDRIM_LogEntry.h"
#include "OpenDRIM_LogEntryAccess.h"
#include "../OpenDRIM_RecordLog/OpenDRIM_RecordLogAccess.h"

static const CMPIBroker * _broker;

#ifdef CMPI_VER_100
#define OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderSetInstance OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderModifyInstance
#endif

int OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_init(const CMPIBroker* broker);

int OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_finalize();

/* ---------------------------------------------------------------------------*/
/*                      Instance Provider Interface                           */
/* ---------------------------------------------------------------------------*/

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderCleanup(CMPIInstanceMI * mi, const CMPIContext* ctx, CMPIBoolean terminate) {
	_E_;
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_finalize();
	CMPIStatus rc;
	CMSetStatus(&rc, (CMPIrc) errorCode);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderEnumInstanceNames(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	vector<OpenDRIM_LogEntry> instances;
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_retrieve(_broker, ctx, instances, NULL, errorMessage, "ein");
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	for (size_t i=0; i < instances.size(); i++) {
		CMPIObjectPath* op = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIObjectPath(_broker, instances[i]);
		CMReturnObjectPath(rslt, op);
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderEnumInstances(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	vector<OpenDRIM_LogEntry> instances;
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_retrieve(_broker, ctx, instances, properties, errorMessage, "ei");
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	for (size_t i=0; i < instances.size(); i++) {
		CMPIInstance* ci = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIInstance(_broker, instances[i]);
		CMReturnInstance(rslt, ci);
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderGetInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_LogEntry instance;
	OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCPP(_broker, cop, instance);
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_getInstance(_broker, ctx, instance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMPIInstance* ci = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIInstance(_broker, instance);
	CMReturnInstance(rslt, ci);
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderSetInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const CMPIInstance* ci, const char** properties) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_LogEntry newInstance, oldInstance;
	OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCPP(_broker, ci, newInstance);
	OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCPP(_broker, cop, oldInstance);
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_getInstance(_broker, ctx, oldInstance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_setInstance(_broker, ctx, newInstance, oldInstance, properties, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderCreateInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop, const CMPIInstance* ci) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_LogEntry instance;
	OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCPP(_broker, ci, instance);
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != NOT_FOUND) {
		if (errorCode == OK)
			errorCode = ALREADY_EXISTS;
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_createInstance(_broker, ctx, instance, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMPIObjectPath* _cop = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIObjectPath(_broker, instance);
	CMReturnObjectPath(rslt, _cop);
	CMReturnDone(rslt);
	_L_;
	return rc;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderDeleteInstance(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* cop) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	OpenDRIM_LogEntry instance;
	OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCPP(_broker, cop, instance);
	string errorMessage;
	
	int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_getInstance(_broker, ctx, instance, NULL, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_deleteInstance(_broker, ctx, instance, errorMessage);
	if (errorCode != OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
		CMSetStatusWithChars(_broker, &rc, (CMPIrc) errorCode, errorMessage.c_str());
		return rc;
	}
	CMReturnDone(rslt);
	_L_;
	return rc;
}

static int checkLogFile(const string& log_file, const string& pattern, const CMPIResult* rslt, string& errorMessage) {
	_E_;
	char conf_file[100];
	if (tmpnam(conf_file) == NULL) {
		errorMessage = "Failed to generate a temp file name.";
		return FAILED;
	}
	
	ofstream os(conf_file);
	os << "watchfor " << pattern << endl;
	os << "\techo" << endl;
	os.close();
	
	string stdout, stderr;
	string cmd = "swatch -c "+(string) conf_file+" -f "+log_file;
	CF_assert(CF_runCommand(cmd, stdout, stderr, errorMessage));	
	_DEBUG("## Command: "+cmd);
	_DEBUG("###########\n"+stdout);
	_DEBUG("###########");
	
//	if (unlink(conf_file) != 0) {
//		errorMessage = "Failed to delete the temp file: "+(string) conf_file;
//		return FAILED;
//	}
	
	_DEBUG("stdout:\n"+stdout);
	
	time_t timep;  
	time(&timep);
	struct tm *tp = localtime(&timep);
	
	string line;
	istringstream iss(stdout);
	while (getline(iss, line)) {
		//Remove an unkown string. It occurs only when swatch is executed.
		if (CF_startsWith(line, "[0m")) line.erase(0, 4);		
		if (CF_trimText(line).size() == 0 || CF_startsWith(line, "*** swatch")) continue;
		
		_DEBUG(line);		
		OpenDRIM_LogEntry instance;
		CF_assert(OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_populate(instance, log_file, line, tp, "ei", errorMessage));
		CMPIInstance* cmpi_instance = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIInstance(_broker, instance);
		CMReturnInstance(rslt, cmpi_instance);
	}
	CMReturnDone(rslt);
	_L_;
	return OK;
}

static string unquote(const string& str) {
	string target(str);
	if (target[target.size()-1] == '\"')
		target.erase(target.size()-1, 1);
	if (target[0] == '\"')
		target.erase(0, 1);
	return target;
}

// added by Frederic Desmons (2007/11/14)
// split a query by space and = and <= and =>
static int splitQuery(vector<string>& query_elements, const string& myQuery, string& errorMessage) {
	// split by space but keep the quoted strings in one piece
	bool quoted = false;
	query_elements.clear();
	string query_element;
	for (string::size_type i = 0; i < myQuery.size(); i++) {
		
		// entering / exiting quotes
		if (myQuery[i] == '\"' && (i==0 || myQuery[i-1] != '\\')) {
			if (quoted)
				quoted = false;
			else
				quoted = true;
			query_element += myQuery[i];
			continue;
		}
		
		// we are in quotes, skip
		if (quoted) {
			query_element += myQuery[i];
			continue;
		}
		
		// look for query operators
		if (i < myQuery.size()-1 && ( \
				(myQuery[i] == '>' && myQuery[i+1] == '=') || \
				(myQuery[i] == '<' && myQuery[i+1] == '=') || \
				(myQuery[i] == '='))) {
			// push the current element
			if (query_element.size() > 0)
				query_elements.push_back(query_element);
			// pop the last element of the pile
			if (query_elements.size() > 0) {
				query_element = query_elements[query_elements.size()-1];
				query_elements.pop_back();
			}
			if (myQuery[i] == '>' && myQuery[i+1] == '=') {
				// >=
				i+=2;
				query_element += ">=";
			} else if (myQuery[i] == '<' && myQuery[i+1] == '=') {
				// <=
				i+=2;
				query_element += "<=";
			} else if (myQuery[i] == '=') {
				// =
				i++;
				query_element += "=";
			}
			// remove all the spaces after the operator
			while (i < myQuery.size() && myQuery[i] == ' ')
				i++;
			i--;
			continue;
		}
		
		// we found a space
		if (myQuery[i] == ' ') {
			// at the beginning or we don't have anything to push, skip
			if (query_element.size() == 0)
				continue;
			else {
				// we have something to push
				query_elements.push_back(query_element);
				query_element.clear();
				continue;
			}
		}
		query_element += myQuery[i];
	}
	
	// invalid quotation
	if (quoted) {
		errorMessage = "Wrong format: unbalanced";
		return FAILED;
	}
	
	if (query_element.size() != 0)
		query_elements.push_back(query_element);
	
	return OK;
}

CMPIStatus OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProviderExecQuery(CMPIInstanceMI* mi, const CMPIContext* ctx, const CMPIResult* rslt, const CMPIObjectPath* ref, const char* lang, const char* query) {
	_E_;
	CMPIStatus rc = {CMPI_RC_OK, NULL};
	string errorMessage;
	int errorCode;	
	vector<OpenDRIM_RecordLog> record_log_instances;
	bool where_start = false;
	string log_file, pattern;
	bool has_datetime_lower_limit = false;
	bool has_datetime_upper_limit = false;
	string datetime_lower_limit = "19700101000000.000000+000";
	string datetime_upper_limit = "20370101000000.000000+000";
	vector<string> tokens;
	vector<string> parsed_where;
	//	CMPISelectExp* selectExp;
	//	CMPISelectCond* cond;
	//	CMPIString* str;
	//	CMPICount cond_count;
	
	//OpenPegasus 2.7 doesn't support CMPI_MB_QueryNormalizer
	//So the query statement is processed manually
	//selectExp = CMNewSelectExp(_broker, query, lang, NULL, &rc);
	//if (selectExp == NULL || rc.rc != CMPI_RC_OK) goto EXIT;
	
	//Parse the where clause
	//CF_splitTextBySpace(tokens, query);
	// added by Frederic Desmons (2007/11/14)
	splitQuery(tokens, query, errorMessage);
	if (tokens.size()==0)
		return rc;
	
	for (vector<string>::size_type i = 0; i < tokens.size(); i++) {
		
		// skip all the tokens before where
		if (CF_strCmpNoCase(tokens[i], "where")) {
			where_start = true;
			continue;
		}
		
		if (!where_start) continue;
		
		// key=value or key>=value or key<=value
		if (tokens[i].find("=")) {
			
			string::size_type idx;
			
			if ((idx = tokens[i].find("<=")) != string::npos) {
				parsed_where.push_back(tokens[i].substr(0, idx));
				parsed_where.push_back("<=");
				parsed_where.push_back(tokens[i].substr(idx+2));
			} else if ((idx = tokens[i].find(">=")) != string::npos) {
				parsed_where.push_back(tokens[i].substr(0, idx));
				parsed_where.push_back(">=");
				parsed_where.push_back(tokens[i].substr(idx+2));
			} else if ((idx = tokens[i].find("=")) != string::npos) {
				parsed_where.push_back(tokens[i].substr(0, idx));
				parsed_where.push_back("=");
				parsed_where.push_back(tokens[i].substr(idx+1));
			}
			
		}
		
	}
	// end of added
	
	/*for (vector<string>::iterator c = tokens.begin();c != tokens.end(); ++c) {
		string token = (*c);
		if (strcasecmp(token.c_str(), "where") == 0) {
			where_start = true;
			continue;
		}
		if (!where_start) continue;
		
		string::size_type idx;
		
		if ((token != "=") && ((idx = token.find_first_of("=")) != string::npos)) {			
			if (idx == token.size()-1) {
				DEBUG;
				//Key= Value
				parsed_where.push_back(token.substr(0, idx));
				parsed_where.push_back("=");
				++c;
				parsed_where.push_back(*c);
			} else if (idx == 0) {
				DEBUG;
				//Key =Value
				parsed_where.push_back(*(c-1));
				parsed_where.push_back("=");
				parsed_where.push_back(token.substr(idx+1));
			} else {
				DEBUG;
				//Key=Value
				parsed_where.push_back(token.substr(0, idx));
				parsed_where.push_back("=");
				parsed_where.push_back(token.substr(idx+1));
			}
		} else if (*(c+1) == "=") {
			DEBUG;
			//Key = Value
			parsed_where.push_back(token);
			parsed_where.push_back("=");
			parsed_where.push_back(*(c+2));			
			c += 2;
		}
		
		if (parsed_where.size() >= 3 && parsed_where.size()%3 == 0) {
			_DEBUG("***** parsed");
			_DEBUG(parsed_where[parsed_where.size()-3]);
			_DEBUG(parsed_where[parsed_where.size()-2]);
			_DEBUG(parsed_where[parsed_where.size()-1]);
		}
	}*/
	
	// extract values of LogName, RecordData, CreationTimestamp fields.
	// modified by Frederic Desmons (2007/11/15)
	for (vector<string>::size_type i = 0; i < parsed_where.size(); ++i) {
		if (strcasecmp(parsed_where[i].c_str(), "LogName") == 0
				&& parsed_where[i+1] == "=") {
			log_file = parsed_where[i+2];
			log_file = unquote(log_file);
			i += 2;
		} else if (strcasecmp(parsed_where[i].c_str(), "RecordData") == 0
				&& parsed_where[i+1] == "=") {
			pattern = parsed_where[i+2];
			pattern = unquote(pattern);
			i += 2;
		} else if (strcasecmp(parsed_where[i].c_str(), "CreationTimestamp") == 0
				&& parsed_where[i+1] == ">=") {
			string datetime = unquote(parsed_where[i+2]);
			int sign;
			errorCode = CF_datetimeCmp(sign, datetime, datetime_lower_limit, errorMessage);
			if (errorCode != OK) {
				CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, errorMessage.c_str());
				goto EXIT;
			}
			has_datetime_lower_limit = true;
			// datetime is higher than current lower limit
			if (sign > 0)
				datetime_lower_limit = datetime;
			i += 2;
		} else if (strcasecmp(parsed_where[i].c_str(), "CreationTimestamp") == 0
				&& parsed_where[i+1] == "<=") {
			string datetime = unquote(parsed_where[i+2]);
			int sign;
			errorCode = CF_datetimeCmp(sign, datetime, datetime_upper_limit, errorMessage);
			if (errorCode != OK) {
				CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, errorMessage.c_str());
				goto EXIT;
			}
			has_datetime_upper_limit = true;
			// datetime is lower than current upper limit
			if (sign < 0)
				datetime_upper_limit = datetime;
			i += 2;
		}
	}
	_DEBUG("LOG file: "+log_file);
	_DEBUG("Pattern: "+pattern);
	// has not pattern nor date
	if (pattern.size() == 0 && !has_datetime_upper_limit && !has_datetime_upper_limit) {
		CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, "Doesn't support queries without the RecordData property OR CreationTimestamp property");
		goto EXIT;
	}
	
	if (pattern.size() != 0 && (has_datetime_upper_limit || has_datetime_upper_limit)) {
		CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, "Doesn't support queries with the RecordData property AND CreationTimestamp property");
		goto EXIT;
	}
	
	if (log_file.size() > 0) {		//If a log file is assigned, only process it
		// modified by Frederic Desmons (2007/11/15)
		// file may not exist tho the entry is syslog.conf exists...
		OpenDRIM_RecordLog log;
		log.setInstanceID(log_file);
		errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_getInstance(_broker, ctx, log, NULL, errorMessage);
		if (errorCode!=OK) {
			CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, "No corresponding log file");
			goto EXIT;
		}	
		/*if (!CF_isExist(log_file)) {
			CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, "No corresponding log file");
			goto EXIT;			
		}*/
		if (pattern.size() != 0) {
			// by pattern
			checkLogFile(log_file, pattern, rslt, errorMessage);
		} else {
			// added by Frederic Desmons (2007/11/15)
			// by CreationTimestamp
			vector<OpenDRIM_LogEntry> log_entries;
			errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_retrieveInDatetimeBoundaries(log_file, datetime_lower_limit, datetime_upper_limit, log_entries, errorMessage);
			if (errorCode != OK) {
				CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, errorMessage.c_str());
				goto EXIT;
			}
			for (vector<OpenDRIM_LogEntry>::size_type i=0; i < log_entries.size(); i++) {
				CMPIInstance* ci = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIInstance(_broker, log_entries[i]);
				CMReturnInstance(rslt, ci);
			}
		}
	} else {								//If a log file is not assigned, process all log files.
		DEBUG;
		// First get the instances of RecordLog	
		errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_RecordLog_retrieve(_broker, ctx, record_log_instances, NULL, errorMessage, "ein");
		if (errorCode!=OK) {
			//errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ": Record_Log_OpenDRIM_RecordLog_retrieve() failed.";
			CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, "Record_Log_OpenDRIM_RecordLog_retrieve() failed.");
			goto EXIT;
		}		
		// for each RecordLog instance, gather the instances of LogEntry by pattern
		for (size_t i=0; i<record_log_instances.size(); i++) {
			string _LogInstanceID;
			record_log_instances[i].getInstanceID(_LogInstanceID);
			// or datetime
			if (pattern.size() != 0) {
				// by pattern
				checkLogFile(_LogInstanceID, pattern, rslt, errorMessage);
			}
			else {
				// added by Frederic Desmons (2007/11/15)
				// by CreationTimestamp
				vector<OpenDRIM_LogEntry> log_entries;
				errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_retrieveInDatetimeBoundaries(_LogInstanceID, datetime_lower_limit, datetime_upper_limit, log_entries, errorMessage);
				if (errorCode != OK) {
					CMSetStatusWithChars(_broker, &rc, CMPI_RC_ERR_FAILED, errorMessage.c_str());
					goto EXIT;
				}
				for (vector<OpenDRIM_LogEntry>::size_type i=0; i < log_entries.size(); i++) {
					CMPIInstance* ci = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_toCMPIInstance(_broker, log_entries[i]);
					CMReturnInstance(rslt, ci);
				}
			}
		}
	}
EXIT:
	CMReturnDone(rslt);
	if (rc.rc != CMPI_RC_OK) {
		errorMessage = (string) OpenDRIM_LogEntry_classnames[0] + ":" + (string) CMGetCharPtr(rc.msg);
		CMSetStatusWithChars(_broker, &rc, rc.rc, errorMessage.c_str());
	}	
	_L_;
	return rc;	
}

/* ---------------------------------------------------------------------------*/
/*                   End of Instance Provider Interface                       */
/* ---------------------------------------------------------------------------*/

/* ---------------------------------------------------------------------------*/
/*                              Provider Factory                              */
/* ---------------------------------------------------------------------------*/

int OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_init(const CMPIBroker* broker) {
	_E_;
	_broker = broker;
	static bool initialized = false;
	if (!initialized) {
		string errorMessage;
		int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_load(broker, errorMessage);
		if (errorCode != OK) {
			errorMessage += "OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_init FAILED: " + (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
			string cmd = "/bin/echo \"" + errorMessage + "\" >> cmpi_prov_debug.txt";
			system(cmd.c_str());
			return -1;
		}
		initialized = true;
	}
	_L_;
	return OK;
}

int OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_finalize() {
	_E_;
	static bool finalized = false;
	if (!finalized) {
		string errorMessage;
		int errorCode = OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_unload(errorMessage);
		if (errorCode != OK) {
			errorMessage += "OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_unload FAILED: " + (string) OpenDRIM_LogEntry_classnames[0] + ": " + errorMessage;
			string cmd = "/bin/echo \"" + errorMessage + "\" >> cmpi_prov_debug.txt";
			system(cmd.c_str());
			return errorCode;
		}
		finalized = true;
	}
	_L_;
	return OK;
}

#define OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_INIT if (OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_init(_broker) < 0) return NULL

CMInstanceMIStub(OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProvider, OpenDRIM_RecordLogPackage_OpenDRIM_LogEntryProvider, _broker, OpenDRIM_RecordLogPackage_OpenDRIM_LogEntry_INIT);

/* ---------------------------------------------------------------------------*/
/*                          End of Provider Factory                           */
/* ---------------------------------------------------------------------------*/
