/* BFD back-end for HP PA-RISC ELF files.
   Copyright 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1999, 2000, 2001,
   2002, 2003, 2004, 2005 Free Software Foundation, Inc.

   Original code by
	Center for Software Science
	Department of Computer Science
	University of Utah
   Largely rewritten by Alan Modra <alan@linuxcare.com.au>

   This file is part of BFD, the Binary File Descriptor library.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "bfd.h"
#include "sysdep.h"
#include "libbfd.h"
#include "elf-bfd.h"
#include "elf/hppa.h"
#include "libhppa.h"
#include "elf32-hppa.h"
#define ARCH_SIZE		32
#include "elf32-hppa.h"
#include "elf-hppa.h"

/* In order to gain some understanding of code in this file without
   knowing all the intricate details of the linker, note the
   following:

   Functions named elf32_hppa_* are called by external routines, other
   functions are only called locally.  elf32_hppa_* functions appear
   in this file more or less in the order in which they are called
   from external routines.  eg. elf32_hppa_check_relocs is called
   early in the link process, elf32_hppa_finish_dynamic_sections is
   one of the last functions.  */

/* We use two hash tables to hold information for linking PA ELF objects.

   The first is the elf32_hppa_link_hash_table which is derived
   from the standard ELF linker hash table.  We use this as a place to
   attach other hash tables and static information.

   The second is the stub hash table which is derived from the
   base BFD hash table.  The stub hash table holds the information
   necessary to build the linker stubs during a link.

   There are a number of different stubs generated by the linker.

   Long branch stub:
   :		ldil LR'X,%r1
   :		be,n RR'X(%sr4,%r1)

   PIC long branch stub:
   :		b,l .+8,%r1
   :		addil LR'X - ($PIC_pcrel$0 - 4),%r1
   :		be,n RR'X - ($PIC_pcrel$0 - 8)(%sr4,%r1)

   Import stub to call shared library routine from normal object file
   (single sub-space version)
   :		addil LR'lt_ptr+ltoff,%dp	; get procedure entry point
   :		ldw RR'lt_ptr+ltoff(%r1),%r21
   :		bv %r0(%r21)
   :		ldw RR'lt_ptr+ltoff+4(%r1),%r19	; get new dlt value.

   Import stub to call shared library routine from shared library
   (single sub-space version)
   :		addil LR'ltoff,%r19		; get procedure entry point
   :		ldw RR'ltoff(%r1),%r21
   :		bv %r0(%r21)
   :		ldw RR'ltoff+4(%r1),%r19	; get new dlt value.

   Import stub to call shared library routine from normal object file
   (multiple sub-space support)
   :		addil LR'lt_ptr+ltoff,%dp	; get procedure entry point
   :		ldw RR'lt_ptr+ltoff(%r1),%r21
   :		ldw RR'lt_ptr+ltoff+4(%r1),%r19	; get new dlt value.
   :		ldsid (%r21),%r1
   :		mtsp %r1,%sr0
   :		be 0(%sr0,%r21)			; branch to target
   :		stw %rp,-24(%sp)		; save rp

   Import stub to call shared library routine from shared library
   (multiple sub-space support)
   :		addil LR'ltoff,%r19		; get procedure entry point
   :		ldw RR'ltoff(%r1),%r21
   :		ldw RR'ltoff+4(%r1),%r19	; get new dlt value.
   :		ldsid (%r21),%r1
   :		mtsp %r1,%sr0
   :		be 0(%sr0,%r21)			; branch to target
   :		stw %rp,-24(%sp)		; save rp

   Export stub to return from shared lib routine (multiple sub-space support)
   One of these is created for each exported procedure in a shared
   library (and stored in the shared lib).  Shared lib routines are
   called via the first instruction in the export stub so that we can
   do an inter-space return.  Not required for single sub-space.
   :		bl,n X,%rp			; trap the return
   :		nop
   :		ldw -24(%sp),%rp		; restore the original rp
   :		ldsid (%rp),%r1
   :		mtsp %r1,%sr0
   :		be,n 0(%sr0,%rp)		; inter-space return.  */

#define PLT_ENTRY_SIZE 8
#define GOT_ENTRY_SIZE 4
#define ELF_DYNAMIC_INTERPRETER "/lib/ld.so.1"

static const bfd_byte plt_stub[] =
{
  0x0e, 0x80, 0x10, 0x96,  /* 1: ldw	0(%r20),%r22		*/
  0xea, 0xc0, 0xc0, 0x00,  /*    bv	%r0(%r22)		*/
  0x0e, 0x88, 0x10, 0x95,  /*    ldw	4(%r20),%r21		*/
#define PLT_STUB_ENTRY (3*4)
  0xea, 0x9f, 0x1f, 0xdd,  /*    b,l	1b,%r20			*/
  0xd6, 0x80, 0x1c, 0x1e,  /*    depi	0,31,2,%r20		*/
  0x00, 0xc0, 0xff, 0xee,  /* 9: .word	fixup_func		*/
  0xde, 0xad, 0xbe, 0xef   /*    .word	fixup_ltp		*/
};

/* Section name for stubs is the associated section name plus this
   string.  */
#define STUB_SUFFIX ".stub"

/* We don't need to copy certain PC- or GP-relative dynamic relocs
   into a shared object's dynamic section.  All the relocs of the
   limited class we are interested in, are absolute.  */
#ifndef RELATIVE_DYNRELOCS
#define RELATIVE_DYNRELOCS 0
#define IS_ABSOLUTE_RELOC(r_type) 1
#endif

/* If ELIMINATE_COPY_RELOCS is non-zero, the linker will try to avoid
   copying dynamic variables from a shared lib into an app's dynbss
   section, and instead use a dynamic relocation to point into the
   shared lib.  */
#define ELIMINATE_COPY_RELOCS 1

enum elf32_hppa_stub_type {
  hppa_stub_long_branch,
  hppa_stub_long_branch_shared,
  hppa_stub_import,
  hppa_stub_import_shared,
  hppa_stub_export,
  hppa_stub_none
};

struct elf32_hppa_stub_hash_entry {

  /* Base hash table entry structure.  */
  struct bfd_hash_entry root;

  /* The stub section.  */
  asection *stub_sec;

  /* Offset within stub_sec of the beginning of this stub.  */
  bfd_vma stub_offset;

  /* Given the symbol's value and its section we can determine its final
     value when building the stubs (so the stub knows where to jump.  */
  bfd_vma target_value;
  asection *target_section;

  enum elf32_hppa_stub_type stub_type;

  /* The symbol table entry, if any, that this was derived from.  */
  struct elf32_hppa_link_hash_entry *h;

  /* Where this stub is being called from, or, in the case of combined
     stub sections, the first input section in the group.  */
  asection *id_sec;
};

struct elf32_hppa_link_hash_entry {

  struct elf_link_hash_entry elf;

  /* A pointer to the most recently used stub hash entry against this
     symbol.  */
  struct elf32_hppa_stub_hash_entry *stub_cache;

  /* Used to count relocations for delayed sizing of relocation
     sections.  */
  struct elf32_hppa_dyn_reloc_entry {

    /* Next relocation in the chain.  */
    struct elf32_hppa_dyn_reloc_entry *next;

    /* The input section of the reloc.  */
    asection *sec;

    /* Number of relocs copied in this section.  */
    bfd_size_type count;

#if RELATIVE_DYNRELOCS
  /* Number of relative relocs copied for the input section.  */
    bfd_size_type relative_count;
#endif
  } *dyn_relocs;

  /* Set if this symbol is used by a plabel reloc.  */
  unsigned int plabel:1;
};

struct elf32_hppa_link_hash_table {

  /* The main hash table.  */
  struct elf_link_hash_table elf;

  /* The stub hash table.  */
  struct bfd_hash_table stub_hash_table;

  /* Linker stub bfd.  */
  bfd *stub_bfd;

  /* Linker call-backs.  */
  asection * (*add_stub_section) (const char *, asection *);
  void (*layout_sections_again) (void);

  /* Array to keep track of which stub sections have been created, and
     information on stub grouping.  */
  struct map_stub {
    /* This is the section to which stubs in the group will be
       attached.  */
    asection *link_sec;
    /* The stub section.  */
    asection *stub_sec;
  } *stub_group;

  /* Assorted information used by elf32_hppa_size_stubs.  */
  unsigned int bfd_count;
  int top_index;
  asection **input_list;
  Elf_Internal_Sym **all_local_syms;

  /* Short-cuts to get to dynamic linker sections.  */
  asection *sgot;
  asection *srelgot;
  asection *splt;
  asection *srelplt;
  asection *sdynbss;
  asection *srelbss;

  /* Used during a final link to store the base of the text and data
     segments so that we can perform SEGREL relocations.  */
  bfd_vma text_segment_base;
  bfd_vma data_segment_base;

  /* Whether we support multiple sub-spaces for shared libs.  */
  unsigned int multi_subspace:1;

  /* Flags set when various size branches are detected.  Used to
     select suitable defaults for the stub group size.  */
  unsigned int has_12bit_branch:1;
  unsigned int has_17bit_branch:1;
  unsigned int has_22bit_branch:1;

  /* Set if we need a .plt stub to support lazy dynamic linking.  */
  unsigned int need_plt_stub:1;

  /* Small local sym to section mapping cache.  */
  struct sym_sec_cache sym_sec;
};

/* Various hash macros and functions.  */
#define hppa_link_hash_table(p) \
  ((struct elf32_hppa_link_hash_table *) ((p)->hash))

#define hppa_stub_hash_lookup(table, string, create, copy) \
  ((struct elf32_hppa_stub_hash_entry *) \
   bfd_hash_lookup ((table), (string), (create), (copy)))

/* Assorted hash table functions.  */

/* Initialize an entry in the stub hash table.  */

static struct bfd_hash_entry *
stub_hash_newfunc (struct bfd_hash_entry *entry,
		   struct bfd_hash_table *table,
		   const char *string)
{
  /* Allocate the structure if it has not already been allocated by a
     subclass.  */
  if (entry == NULL)
    {
      entry = bfd_hash_allocate (table,
				 sizeof (struct elf32_hppa_stub_hash_entry));
      if (entry == NULL)
	return entry;
    }

  /* Call the allocation method of the superclass.  */
  entry = bfd_hash_newfunc (entry, table, string);
  if (entry != NULL)
    {
      struct elf32_hppa_stub_hash_entry *eh;

      /* Initialize the local fields.  */
      eh = (struct elf32_hppa_stub_hash_entry *) entry;
      eh->stub_sec = NULL;
      eh->stub_offset = 0;
      eh->target_value = 0;
      eh->target_section = NULL;
      eh->stub_type = hppa_stub_long_branch;
      eh->h = NULL;
      eh->id_sec = NULL;
    }

  return entry;
}

/* Initialize an entry in the link hash table.  */

static struct bfd_hash_entry *
hppa_link_hash_newfunc (struct bfd_hash_entry *entry,
			struct bfd_hash_table *table,
			const char *string)
{
  /* Allocate the structure if it has not already been allocated by a
     subclass.  */
  if (entry == NULL)
    {
      entry = bfd_hash_allocate (table,
				 sizeof (struct elf32_hppa_link_hash_entry));
      if (entry == NULL)
	return entry;
    }

  /* Call the allocation method of the superclass.  */
  entry = _bfd_elf_link_hash_newfunc (entry, table, string);
  if (entry != NULL)
    {
      struct elf32_hppa_link_hash_entry *eh;

      /* Initialize the local fields.  */
      eh = (struct elf32_hppa_link_hash_entry *) entry;
      eh->stub_cache = NULL;
      eh->dyn_relocs = NULL;
      eh->plabel = 0;
    }

  return entry;
}

/* Create the derived linker hash table.  The PA ELF port uses the derived
   hash table to keep information specific to the PA ELF linker (without
   using static variables).  */

static struct bfd_link_hash_table *
elf32_hppa_link_hash_table_create (bfd *abfd)
{
  struct elf32_hppa_link_hash_table *ret;
  bfd_size_type amt = sizeof (*ret);

  ret = bfd_malloc (amt);
  if (ret == NULL)
    return NULL;

  if (!_bfd_elf_link_hash_table_init (&ret->elf, abfd, hppa_link_hash_newfunc))
    {
      free (ret);
      return NULL;
    }

  /* Init the stub hash table too.  */
  if (!bfd_hash_table_init (&ret->stub_hash_table, stub_hash_newfunc))
    return NULL;

  ret->stub_bfd = NULL;
  ret->add_stub_section = NULL;
  ret->layout_sections_again = NULL;
  ret->stub_group = NULL;
  ret->sgot = NULL;
  ret->srelgot = NULL;
  ret->splt = NULL;
  ret->srelplt = NULL;
  ret->sdynbss = NULL;
  ret->srelbss = NULL;
  ret->text_segment_base = (bfd_vma) -1;
  ret->data_segment_base = (bfd_vma) -1;
  ret->multi_subspace = 0;
  ret->has_12bit_branch = 0;
  ret->has_17bit_branch = 0;
  ret->has_22bit_branch = 0;
  ret->need_plt_stub = 0;
  ret->sym_sec.abfd = NULL;

  return &ret->elf.root;
}

/* Free the derived linker hash table.  */

static void
elf32_hppa_link_hash_table_free (struct bfd_link_hash_table *hash)
{
  struct elf32_hppa_link_hash_table *ret
    = (struct elf32_hppa_link_hash_table *) hash;

  bfd_hash_table_free (&ret->stub_hash_table);
  _bfd_generic_link_hash_table_free (hash);
}

/* Build a name for an entry in the stub hash table.  */

static char *
hppa_stub_name (const asection *input_section,
		const asection *sym_sec,
		const struct elf32_hppa_link_hash_entry *hash,
		const Elf_Internal_Rela *rel)
{
  char *stub_name;
  bfd_size_type len;

  if (hash)
    {
      len = 8 + 1 + strlen (hash->elf.root.root.string) + 1 + 8 + 1;
      stub_name = bfd_malloc (len);
      if (stub_name != NULL)
	{
	  sprintf (stub_name, "%08x_%s+%x",
		   input_section->id & 0xffffffff,
		   hash->elf.root.root.string,
		   (int) rel->r_addend & 0xffffffff);
	}
    }
  else
    {
      len = 8 + 1 + 8 + 1 + 8 + 1 + 8 + 1;
      stub_name = bfd_malloc (len);
      if (stub_name != NULL)
	{
	  sprintf (stub_name, "%08x_%x:%x+%x",
		   input_section->id & 0xffffffff,
		   sym_sec->id & 0xffffffff,
		   (int) ELF32_R_SYM (rel->r_info) & 0xffffffff,
		   (int) rel->r_addend & 0xffffffff);
	}
    }
  return stub_name;
}

/* Look up an entry in the stub hash.  Stub entries are cached because
   creating the stub name takes a bit of time.  */

static struct elf32_hppa_stub_hash_entry *
hppa_get_stub_entry (const asection *input_section,
		     const asection *sym_sec,
		     struct elf32_hppa_link_hash_entry *hash,
		     const Elf_Internal_Rela *rel,
		     struct elf32_hppa_link_hash_table *htab)
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  const asection *id_sec;

  /* If this input section is part of a group of sections sharing one
     stub section, then use the id of the first section in the group.
     Stub names need to include a section id, as there may well be
     more than one stub used to reach say, printf, and we need to
     distinguish between them.  */
  id_sec = htab->stub_group[input_section->id].link_sec;

  if (hash != NULL && hash->stub_cache != NULL
      && hash->stub_cache->h == hash
      && hash->stub_cache->id_sec == id_sec)
    {
      stub_entry = hash->stub_cache;
    }
  else
    {
      char *stub_name;

      stub_name = hppa_stub_name (id_sec, sym_sec, hash, rel);
      if (stub_name == NULL)
	return NULL;

      stub_entry = hppa_stub_hash_lookup (&htab->stub_hash_table,
					  stub_name, FALSE, FALSE);
      if (hash != NULL)
	hash->stub_cache = stub_entry;

      free (stub_name);
    }

  return stub_entry;
}

/* Add a new stub entry to the stub hash.  Not all fields of the new
   stub entry are initialised.  */

static struct elf32_hppa_stub_hash_entry *
hppa_add_stub (const char *stub_name,
	       asection *section,
	       struct elf32_hppa_link_hash_table *htab)
{
  asection *link_sec;
  asection *stub_sec;
  struct elf32_hppa_stub_hash_entry *stub_entry;

  link_sec = htab->stub_group[section->id].link_sec;
  stub_sec = htab->stub_group[section->id].stub_sec;
  if (stub_sec == NULL)
    {
      stub_sec = htab->stub_group[link_sec->id].stub_sec;
      if (stub_sec == NULL)
	{
	  size_t namelen;
	  bfd_size_type len;
	  char *s_name;

	  namelen = strlen (link_sec->name);
	  len = namelen + sizeof (STUB_SUFFIX);
	  s_name = bfd_alloc (htab->stub_bfd, len);
	  if (s_name == NULL)
	    return NULL;

	  memcpy (s_name, link_sec->name, namelen);
	  memcpy (s_name + namelen, STUB_SUFFIX, sizeof (STUB_SUFFIX));
	  stub_sec = (*htab->add_stub_section) (s_name, link_sec);
	  if (stub_sec == NULL)
	    return NULL;
	  htab->stub_group[link_sec->id].stub_sec = stub_sec;
	}
      htab->stub_group[section->id].stub_sec = stub_sec;
    }

  /* Enter this entry into the linker stub hash table.  */
  stub_entry = hppa_stub_hash_lookup (&htab->stub_hash_table, stub_name,
				      TRUE, FALSE);
  if (stub_entry == NULL)
    {
      (*_bfd_error_handler) (_("%B: cannot create stub entry %s"),
			     section->owner,
			     stub_name);
      return NULL;
    }

  stub_entry->stub_sec = stub_sec;
  stub_entry->stub_offset = 0;
  stub_entry->id_sec = link_sec;
  return stub_entry;
}

/* Determine the type of stub needed, if any, for a call.  */

static enum elf32_hppa_stub_type
hppa_type_of_stub (asection *input_sec,
		   const Elf_Internal_Rela *rel,
		   struct elf32_hppa_link_hash_entry *hash,
		   bfd_vma destination,
		   struct bfd_link_info *info)
{
  bfd_vma location;
  bfd_vma branch_offset;
  bfd_vma max_branch_offset;
  unsigned int r_type;

  if (hash != NULL
      && hash->elf.plt.offset != (bfd_vma) -1
      && hash->elf.dynindx != -1
      && !hash->plabel
      && (info->shared
	  || !hash->elf.def_regular
	  || hash->elf.root.type == bfd_link_hash_defweak))
    {
      /* We need an import stub.  Decide between hppa_stub_import
	 and hppa_stub_import_shared later.  */
      return hppa_stub_import;
    }

  /* Determine where the call point is.  */
  location = (input_sec->output_offset
	      + input_sec->output_section->vma
	      + rel->r_offset);

  branch_offset = destination - location - 8;
  r_type = ELF32_R_TYPE (rel->r_info);

  /* Determine if a long branch stub is needed.  parisc branch offsets
     are relative to the second instruction past the branch, ie. +8
     bytes on from the branch instruction location.  The offset is
     signed and counts in units of 4 bytes.  */
  if (r_type == (unsigned int) R_PARISC_PCREL17F)
    {
      max_branch_offset = (1 << (17-1)) << 2;
    }
  else if (r_type == (unsigned int) R_PARISC_PCREL12F)
    {
      max_branch_offset = (1 << (12-1)) << 2;
    }
  else /* R_PARISC_PCREL22F.  */
    {
      max_branch_offset = (1 << (22-1)) << 2;
    }

  if (branch_offset + max_branch_offset >= 2*max_branch_offset)
    return hppa_stub_long_branch;

  return hppa_stub_none;
}

/* Build one linker stub as defined by the stub hash table entry GEN_ENTRY.
   IN_ARG contains the link info pointer.  */

#define LDIL_R1		0x20200000	/* ldil  LR'XXX,%r1		*/
#define BE_SR4_R1	0xe0202002	/* be,n  RR'XXX(%sr4,%r1)	*/

#define BL_R1		0xe8200000	/* b,l   .+8,%r1		*/
#define ADDIL_R1	0x28200000	/* addil LR'XXX,%r1,%r1		*/
#define DEPI_R1		0xd4201c1e	/* depi  0,31,2,%r1		*/

#define ADDIL_DP	0x2b600000	/* addil LR'XXX,%dp,%r1		*/
#define LDW_R1_R21	0x48350000	/* ldw   RR'XXX(%sr0,%r1),%r21	*/
#define BV_R0_R21	0xeaa0c000	/* bv    %r0(%r21)		*/
#define LDW_R1_R19	0x48330000	/* ldw   RR'XXX(%sr0,%r1),%r19	*/

#define ADDIL_R19	0x2a600000	/* addil LR'XXX,%r19,%r1	*/
#define LDW_R1_DP	0x483b0000	/* ldw   RR'XXX(%sr0,%r1),%dp	*/

#define LDSID_R21_R1	0x02a010a1	/* ldsid (%sr0,%r21),%r1	*/
#define MTSP_R1		0x00011820	/* mtsp  %r1,%sr0		*/
#define BE_SR0_R21	0xe2a00000	/* be    0(%sr0,%r21)		*/
#define STW_RP		0x6bc23fd1	/* stw   %rp,-24(%sr0,%sp)	*/

#define BL22_RP		0xe800a002	/* b,l,n XXX,%rp		*/
#define BL_RP		0xe8400002	/* b,l,n XXX,%rp		*/
#define NOP		0x08000240	/* nop				*/
#define LDW_RP		0x4bc23fd1	/* ldw   -24(%sr0,%sp),%rp	*/
#define LDSID_RP_R1	0x004010a1	/* ldsid (%sr0,%rp),%r1		*/
#define BE_SR0_RP	0xe0400002	/* be,n  0(%sr0,%rp)		*/

#ifndef R19_STUBS
#define R19_STUBS 1
#endif

#if R19_STUBS
#define LDW_R1_DLT	LDW_R1_R19
#else
#define LDW_R1_DLT	LDW_R1_DP
#endif

static bfd_boolean
hppa_build_one_stub (struct bfd_hash_entry *gen_entry, void *in_arg)
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  struct bfd_link_info *info;
  struct elf32_hppa_link_hash_table *htab;
  asection *stub_sec;
  bfd *stub_bfd;
  bfd_byte *loc;
  bfd_vma sym_value;
  bfd_vma insn;
  bfd_vma off;
  int val;
  int size;

  /* Massage our args to the form they really have.  */
  stub_entry = (struct elf32_hppa_stub_hash_entry *) gen_entry;
  info = in_arg;

  htab = hppa_link_hash_table (info);
  stub_sec = stub_entry->stub_sec;

  /* Make a note of the offset within the stubs for this entry.  */
  stub_entry->stub_offset = stub_sec->size;
  loc = stub_sec->contents + stub_entry->stub_offset;

  stub_bfd = stub_sec->owner;

  switch (stub_entry->stub_type)
    {
    case hppa_stub_long_branch:
      /* Create the long branch.  A long branch is formed with "ldil"
	 loading the upper bits of the target address into a register,
	 then branching with "be" which adds in the lower bits.
	 The "be" has its delay slot nullified.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      val = hppa_field_adjust (sym_value, 0, e_lrsel);
      insn = hppa_rebuild_insn ((int) LDIL_R1, val, 21);
      bfd_put_32 (stub_bfd, insn, loc);

      val = hppa_field_adjust (sym_value, 0, e_rrsel) >> 2;
      insn = hppa_rebuild_insn ((int) BE_SR4_R1, val, 17);
      bfd_put_32 (stub_bfd, insn, loc + 4);

      size = 8;
      break;

    case hppa_stub_long_branch_shared:
      /* Branches are relative.  This is where we are going to.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      /* And this is where we are coming from, more or less.  */
      sym_value -= (stub_entry->stub_offset
		    + stub_sec->output_offset
		    + stub_sec->output_section->vma);

      bfd_put_32 (stub_bfd, (bfd_vma) BL_R1, loc);
      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_lrsel);
      insn = hppa_rebuild_insn ((int) ADDIL_R1, val, 21);
      bfd_put_32 (stub_bfd, insn, loc + 4);

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_rrsel) >> 2;
      insn = hppa_rebuild_insn ((int) BE_SR4_R1, val, 17);
      bfd_put_32 (stub_bfd, insn, loc + 8);
      size = 12;
      break;

    case hppa_stub_import:
    case hppa_stub_import_shared:
      off = stub_entry->h->elf.plt.offset;
      if (off >= (bfd_vma) -2)
	abort ();

      off &= ~ (bfd_vma) 1;
      sym_value = (off
		   + htab->splt->output_offset
		   + htab->splt->output_section->vma
		   - elf_gp (htab->splt->output_section->owner));

      insn = ADDIL_DP;
#if R19_STUBS
      if (stub_entry->stub_type == hppa_stub_import_shared)
	insn = ADDIL_R19;
#endif
      val = hppa_field_adjust (sym_value, 0, e_lrsel),
      insn = hppa_rebuild_insn ((int) insn, val, 21);
      bfd_put_32 (stub_bfd, insn, loc);

      /* It is critical to use lrsel/rrsel here because we are using
	 two different offsets (+0 and +4) from sym_value.  If we use
	 lsel/rsel then with unfortunate sym_values we will round
	 sym_value+4 up to the next 2k block leading to a mis-match
	 between the lsel and rsel value.  */
      val = hppa_field_adjust (sym_value, 0, e_rrsel);
      insn = hppa_rebuild_insn ((int) LDW_R1_R21, val, 14);
      bfd_put_32 (stub_bfd, insn, loc + 4);

      if (htab->multi_subspace)
	{
	  val = hppa_field_adjust (sym_value, (bfd_signed_vma) 4, e_rrsel);
	  insn = hppa_rebuild_insn ((int) LDW_R1_DLT, val, 14);
	  bfd_put_32 (stub_bfd, insn, loc + 8);

	  bfd_put_32 (stub_bfd, (bfd_vma) LDSID_R21_R1, loc + 12);
	  bfd_put_32 (stub_bfd, (bfd_vma) MTSP_R1,      loc + 16);
	  bfd_put_32 (stub_bfd, (bfd_vma) BE_SR0_R21,   loc + 20);
	  bfd_put_32 (stub_bfd, (bfd_vma) STW_RP,       loc + 24);

	  size = 28;
	}
      else
	{
	  bfd_put_32 (stub_bfd, (bfd_vma) BV_R0_R21, loc + 8);
	  val = hppa_field_adjust (sym_value, (bfd_signed_vma) 4, e_rrsel);
	  insn = hppa_rebuild_insn ((int) LDW_R1_DLT, val, 14);
	  bfd_put_32 (stub_bfd, insn, loc + 12);

	  size = 16;
	}

      break;

    case hppa_stub_export:
      /* Branches are relative.  This is where we are going to.  */
      sym_value = (stub_entry->target_value
		   + stub_entry->target_section->output_offset
		   + stub_entry->target_section->output_section->vma);

      /* And this is where we are coming from.  */
      sym_value -= (stub_entry->stub_offset
		    + stub_sec->output_offset
		    + stub_sec->output_section->vma);

      if (sym_value - 8 + (1 << (17 + 1)) >= (1 << (17 + 2))
	  && (!htab->has_22bit_branch
	      || sym_value - 8 + (1 << (22 + 1)) >= (1 << (22 + 2))))
	{
	  (*_bfd_error_handler)
	    (_("%B(%A+0x%lx): cannot reach %s, recompile with -ffunction-sections"),
	     stub_entry->target_section->owner,
	     stub_sec,
	     (long) stub_entry->stub_offset,
	     stub_entry->root.string);
	  bfd_set_error (bfd_error_bad_value);
	  return FALSE;
	}

      val = hppa_field_adjust (sym_value, (bfd_signed_vma) -8, e_fsel) >> 2;
      if (!htab->has_22bit_branch)
	insn = hppa_rebuild_insn ((int) BL_RP, val, 17);
      else
	insn = hppa_rebuild_insn ((int) BL22_RP, val, 22);
      bfd_put_32 (stub_bfd, insn, loc);

      bfd_put_32 (stub_bfd, (bfd_vma) NOP,         loc + 4);
      bfd_put_32 (stub_bfd, (bfd_vma) LDW_RP,      loc + 8);
      bfd_put_32 (stub_bfd, (bfd_vma) LDSID_RP_R1, loc + 12);
      bfd_put_32 (stub_bfd, (bfd_vma) MTSP_R1,     loc + 16);
      bfd_put_32 (stub_bfd, (bfd_vma) BE_SR0_RP,   loc + 20);

      /* Point the function symbol at the stub.  */
      stub_entry->h->elf.root.u.def.section = stub_sec;
      stub_entry->h->elf.root.u.def.value = stub_sec->size;

      size = 24;
      break;

    default:
      BFD_FAIL ();
      return FALSE;
    }

  stub_sec->size += size;
  return TRUE;
}

#undef LDIL_R1
#undef BE_SR4_R1
#undef BL_R1
#undef ADDIL_R1
#undef DEPI_R1
#undef LDW_R1_R21
#undef LDW_R1_DLT
#undef LDW_R1_R19
#undef ADDIL_R19
#undef LDW_R1_DP
#undef LDSID_R21_R1
#undef MTSP_R1
#undef BE_SR0_R21
#undef STW_RP
#undef BV_R0_R21
#undef BL_RP
#undef NOP
#undef LDW_RP
#undef LDSID_RP_R1
#undef BE_SR0_RP

/* As above, but don't actually build the stub.  Just bump offset so
   we know stub section sizes.  */

static bfd_boolean
hppa_size_one_stub (struct bfd_hash_entry *gen_entry, void *in_arg)
{
  struct elf32_hppa_stub_hash_entry *stub_entry;
  struct elf32_hppa_link_hash_table *htab;
  int size;

  /* Massage our args to the form they really have.  */
  stub_entry = (struct elf32_hppa_stub_hash_entry *) gen_entry;
  htab = in_arg;

  if (stub_entry->stub_type == hppa_stub_long_branch)
    size = 8;
  else if (stub_entry->stub_type == hppa_stub_long_branch_shared)
    size = 12;
  else if (stub_entry->stub_type == hppa_stub_export)
    size = 24;
  else /* hppa_stub_import or hppa_stub_import_shared.  */
    {
      if (htab->multi_subspace)
	size = 28;
      else
	size = 16;
    }

  stub_entry->stub_sec->size += size;
  return TRUE;
}

/* Return nonzero if ABFD represents an HPPA ELF32 file.
   Additionally we set the default architecture and machine.  */

static bfd_boolean
elf32_hppa_object_p (bfd *abfd)
{
  Elf_Internal_Ehdr * i_ehdrp;
  unsigned int flags;

  i_ehdrp = elf_elfheader (abfd);
  if (strcmp (bfd_get_target (abfd), "elf32-hppa-linux") == 0)
    {
      /* GCC on hppa-linux produces binaries with OSABI=Linux,
	 but the kernel produces corefiles with OSABI=SysV.  */
      if (i_ehdrp->e_ident[EI_OSABI] != ELFOSABI_LINUX &&
	  i_ehdrp->e_ident[EI_OSABI] != ELFOSABI_NONE) /* aka SYSV */
	return FALSE;
    }
  else if (strcmp (bfd_get_target (abfd), "elf32-hppa-netbsd") == 0)
    {
      /* GCC on hppa-netbsd produces binaries with OSABI=NetBSD,
	 but the kernel produces corefiles with OSABI=SysV.  */
      if (i_ehdrp->e_ident[EI_OSABI] != ELFOSABI_NETBSD &&
	  i_ehdrp->e_ident[EI_OSABI] != ELFOSABI_NONE) /* aka SYSV */
	return FALSE;
    }
  else
    {
      if (i_ehdrp->e_ident[EI_OSABI] != ELFOSABI_HPUX)
	return FALSE;
    }

  flags = i_ehdrp->e_flags;
  switch (flags & (EF_PARISC_ARCH | EF_PARISC_WIDE))
    {
    case EFA_PARISC_1_0:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 10);
    case EFA_PARISC_1_1:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 11);
    case EFA_PARISC_2_0:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 20);
    case EFA_PARISC_2_0 | EF_PARISC_WIDE:
      return bfd_default_set_arch_mach (abfd, bfd_arch_hppa, 25);
    }
  return TRUE;
}

/* Create the .plt and .got sections, and set up our hash table
   short-cuts to various dynamic sections.  */

static bfd_boolean
elf32_hppa_create_dynamic_sections (bfd *abfd, struct bfd_link_info *info)
{
  struct elf32_hppa_link_hash_table *htab;
  struct elf_link_hash_entry *h;

  /* Don't try to create the .plt and .got twice.  */
  htab = hppa_link_hash_table (info);
  if (htab->splt != NULL)
    return TRUE;

  /* Call the generic code to do most of the work.  */
  if (! _bfd_elf_create_dynamic_sections (abfd, info))
    return FALSE;

  htab->splt = bfd_get_section_by_name (abfd, ".plt");
  htab->srelplt = bfd_get_section_by_name (abfd, ".rela.plt");

  htab->sgot = bfd_get_section_by_name (abfd, ".got");
  htab->srelgot = bfd_make_section (abfd, ".rela.got");
  if (htab->srelgot == NULL
      || ! bfd_set_section_flags (abfd, htab->srelgot,
				  (SEC_ALLOC
				   | SEC_LOAD
				   | SEC_HAS_CONTENTS
				   | SEC_IN_MEMORY
				   | SEC_LINKER_CREATED
				   | SEC_READONLY))
      || ! bfd_set_section_alignment (abfd, htab->srelgot, 2))
    return FALSE;

  htab->sdynbss = bfd_get_section_by_name (abfd, ".dynbss");
  htab->srelbss = bfd_get_section_by_name (abfd, ".rela.bss");

  /* hppa-linux needs _GLOBAL_OFFSET_TABLE_ to be visible from the main
     application, because __canonicalize_funcptr_for_compare needs it.  */
  h = elf_hash_table (info)->hgot;
  h->forced_local = 0;
  h->other = STV_DEFAULT;
  return bfd_elf_link_record_dynamic_symbol (info, h);
}

/* Copy the extra info we tack onto an elf_link_hash_entry.  */

static void
elf32_hppa_copy_indirect_symbol (const struct elf_backend_data *bed,
				 struct elf_link_hash_entry *dir,
				 struct elf_link_hash_entry *ind)
{
  struct elf32_hppa_link_hash_entry *edir, *eind;

  edir = (struct elf32_hppa_link_hash_entry *) dir;
  eind = (struct elf32_hppa_link_hash_entry *) ind;

  if (eind->dyn_relocs != NULL)
    {
      if (edir->dyn_relocs != NULL)
	{
	  struct elf32_hppa_dyn_reloc_entry **pp;
	  struct elf32_hppa_dyn_reloc_entry *p;

	  if (ind->root.type == bfd_link_hash_indirect)
	    abort ();

	  /* Add reloc counts against the weak sym to the strong sym
	     list.  Merge any entries against the same section.  */
	  for (pp = &eind->dyn_relocs; (p = *pp) != NULL; )
	    {
	      struct elf32_hppa_dyn_reloc_entry *q;

	      for (q = edir->dyn_relocs; q != NULL; q = q->next)
		if (q->sec == p->sec)
		  {
#if RELATIVE_DYNRELOCS
		    q->relative_count += p->relative_count;
#endif
		    q->count += p->count;
		    *pp = p->next;
		    break;
		  }
	      if (q == NULL)
		pp = &p->next;
	    }
	  *pp = edir->dyn_relocs;
	}

      edir->dyn_relocs = eind->dyn_relocs;
      eind->dyn_relocs = NULL;
    }

  if (ELIMINATE_COPY_RELOCS
      && ind->root.type != bfd_link_hash_indirect
      && dir->dynamic_adjusted)
    {
      /* If called to transfer flags for a weakdef during processing
	 of elf_adjust_dynamic_symbol, don't copy non_got_ref.
	 We clear it ourselves for ELIMINATE_COPY_RELOCS.  */
      dir->ref_dynamic |= ind->ref_dynamic;
      dir->ref_regular |= ind->ref_regular;
      dir->ref_regular_nonweak |= ind->ref_regular_nonweak;
      dir->needs_plt |= ind->needs_plt;
    }
  else
    _bfd_elf_link_hash_copy_indirect (bed, dir, ind);
}

/* Look through the relocs for a section during the first phase, and
   calculate needed space in the global offset table, procedure linkage
   table, and dynamic reloc sections.  At this point we haven't
   necessarily read all the input files.  */

static bfd_boolean
elf32_hppa_check_relocs (bfd *abfd,
			 struct bfd_link_info *info,
			 asection *sec,
			 const Elf_Internal_Rela *relocs)
{
  Elf_Internal_Shdr *symtab_hdr;
  struct elf_link_hash_entry **sym_hashes;
  const Elf_Internal_Rela *rel;
  const Elf_Internal_Rela *rel_end;
  struct elf32_hppa_link_hash_table *htab;
  asection *sreloc;
  asection *stubreloc;

  if (info->relocatable)
    return TRUE;

  htab = hppa_link_hash_table (info);
  symtab_hdr = &elf_tdata (abfd)->symtab_hdr;
  sym_hashes = elf_sym_hashes (abfd);
  sreloc = NULL;
  stubreloc = NULL;

  rel_end = relocs + sec->reloc_count;
  for (rel = relocs; rel < rel_end; rel++)
    {
      enum {
	NEED_GOT = 1,
	NEED_PLT = 2,
	NEED_DYNREL = 4,
	PLT_PLABEL = 8
      };

      unsigned int r_symndx, r_type;
      struct elf32_hppa_link_hash_entry *h;
      int need_entry;

      r_symndx = ELF32_R_SYM (rel->r_info);

      if (r_symndx < symtab_hdr->sh_info)
	h = NULL;
      else
	h = ((struct elf32_hppa_link_hash_entry *)
	     sym_hashes[r_symndx - symtab_hdr->sh_info]);

      r_type = ELF32_R_TYPE (rel->r_info);

      switch (r_type)
	{
	case R_PARISC_DLTIND14F:
	case R_PARISC_DLTIND14R:
	case R_PARISC_DLTIND21L:
	  /* This symbol requires a global offset table entry.  */
	  need_entry = NEED_GOT;
	  break;

	case R_PARISC_PLABEL14R: /* "Official" procedure labels.  */
	case R_PARISC_PLABEL21L:
	case R_PARISC_PLABEL32:
	  /* If the addend is non-zero, we break badly.  */
	  if (rel->r_addend != 0)
	    abort ();

	  /* If we are creating a shared library, then we need to
	     create a PLT entry for all PLABELs, because PLABELs with
	     local symbols may be passed via a pointer to another
	     object.  Additionally, output a dynamic relocation
	     pointing to the PLT entry.
	     For executables, the original 32-bit ABI allowed two
	     different styles of PLABELs (function pointers):  For
	     global functions, the PLABEL word points into the .plt
	     two bytes past a (function address, gp) pair, and for
	     local functions the PLABEL points directly at the
	     function.  The magic +2 for the first type allows us to
	     differentiate between the two.  As you can imagine, this
	     is a real pain when it comes to generating code to call
	     functions indirectly or to compare function pointers.
	     We avoid the mess by always pointing a PLABEL into the
	     .plt, even for local functions.  */
	  need_entry = PLT_PLABEL | NEED_PLT | NEED_DYNREL;
	  break;

	case R_PARISC_PCREL12F:
	  htab->has_12bit_branch = 1;
	  goto branch_common;

	case R_PARISC_PCREL17C:
	case R_PARISC_PCREL17F:
	  htab->has_17bit_branch = 1;
	  goto branch_common;

	case R_PARISC_PCREL22F:
	  htab->has_22bit_branch = 1;
	branch_common:
	  /* Function calls might need to go through the .plt, and
	     might require long branch stubs.  */
	  if (h == NULL)
	    {
	      /* We know local syms won't need a .plt entry, and if
		 they need a long branch stub we can't guarantee that
		 we can reach the stub.  So just flag an error later
		 if we're doing a shared link and find we need a long
		 branch stub.  */
	      continue;
	    }
	  else
	    {
	      /* Global symbols will need a .plt entry if they remain
		 global, and in most cases won't need a long branch
		 stub.  Unfortunately, we have to cater for the case
		 where a symbol is forced local by versioning, or due
		 to symbolic linking, and we lose the .plt entry.  */
	      need_entry = NEED_PLT;
	      if (h->elf.type == STT_PARISC_MILLI)
		need_entry = 0;
	    }
	  break;

	case R_PARISC_SEGBASE:  /* Used to set segment base.  */
	case R_PARISC_SEGREL32: /* Relative reloc, used for unwind.  */
	case R_PARISC_PCREL14F: /* PC relative load/store.  */
	case R_PARISC_PCREL14R:
	case R_PARISC_PCREL17R: /* External branches.  */
	case R_PARISC_PCREL21L: /* As above, and for load/store too.  */
	case R_PARISC_PCREL32:
	  /* We don't need to propagate the relocation if linking a
	     shared object since these are section relative.  */
	  continue;

	case R_PARISC_DPREL14F: /* Used for gp rel data load/store.  */
	case R_PARISC_DPREL14R:
	case R_PARISC_DPREL21L:
	  if (info->shared)
	    {
	      (*_bfd_error_handler)
		(_("%B: relocation %s can not be used when making a shared object; recompile with -fPIC"),
		 abfd,
		 elf_hppa_howto_table[r_type].name);
	      bfd_set_error (bfd_error_bad_value);
	      return FALSE;
	    }
	  /* Fall through.  */

	case R_PARISC_DIR17F: /* Used for external branches.  */
	case R_PARISC_DIR17R:
	case R_PARISC_DIR14F: /* Used for load/store from absolute locn.  */
	case R_PARISC_DIR14R:
	case R_PARISC_DIR21L: /* As above, and for ext branches too.  */
	case R_PARISC_DIR32: /* .word relocs.  */
	  /* We may want to output a dynamic relocation later.  */
	  need_entry = NEED_DYNREL;
	  break;

	  /* This relocation describes the C++ object vtable hierarchy.
	     Reconstruct it for later use during GC.  */
	case R_PARISC_GNU_VTINHERIT:
	  if (!bfd_elf_gc_record_vtinherit (abfd, sec, &h->elf, rel->r_offset))
	    return FALSE;
	  continue;

	  /* This relocation describes which C++ vtable entries are actually
	     used.  Record for later use during GC.  */
	case R_PARISC_GNU_VTENTRY:
	  if (!bfd_elf_gc_record_vtentry (abfd, sec, &h->elf, rel->r_addend))
	    return FALSE;
	  continue;

	default:
	  continue;
	}

      /* Now carry out our orders.  */
      if (need_entry & NEED_GOT)
	{
	  /* Allocate space for a GOT entry, as well as a dynamic
	     relocation for this entry.  */
	  if (htab->sgot == NULL)
	    {
	      if (htab->elf.dynobj == NULL)
		htab->elf.dynobj = abfd;
	      if (!elf32_hppa_create_dynamic_sections (htab->elf.dynobj, info))
		return FALSE;
	    }

	  if (h != NULL)
	    {
	      h->elf.got.refcount += 1;
	    }
	  else
	    {
	      bfd_signed_vma *local_got_refcounts;

	      /* This is a global offset table entry for a local symbol.  */
	      local_got_refcounts = elf_local_got_refcounts (abfd);
	      if (local_got_refcounts == NULL)
		{
		  bfd_size_type size;

		  /* Allocate space for local got offsets and local
		     plt offsets.  Done this way to save polluting
		     elf_obj_tdata with another target specific
		     pointer.  */
		  size = symtab_hdr->sh_info;
		  size *= 2 * sizeof (bfd_signed_vma);
		  local_got_refcounts = bfd_zalloc (abfd, size);
		  if (local_got_refcounts == NULL)
		    return FALSE;
		  elf_local_got_refcounts (abfd) = local_got_refcounts;
		}
	      local_got_refcounts[r_symndx] += 1;
	    }
	}

      if (need_entry & NEED_PLT)
	{
	  /* If we are creating a shared library, and this is a reloc
	     against a weak symbol or a global symbol in a dynamic
	     object, then we will be creating an import stub and a
	     .plt entry for the symbol.  Similarly, on a normal link
	     to symbols defined in a dynamic object we'll need the
	     import stub and a .plt entry.  We don't know yet whether
	     the symbol is defined or not, so make an entry anyway and
	     clean up later in adjust_dynamic_symbol.  */
	  if ((sec->flags & SEC_ALLOC) != 0)
	    {
	      if (h != NULL)
		{
		  h->elf.needs_plt = 1;
		  h->elf.plt.refcount += 1;

		  /* If this .plt entry is for a plabel, mark it so
		     that adjust_dynamic_symbol will keep the entry
		     even if it appears to be local.  */
		  if (need_entry & PLT_PLABEL)
		    h->plabel = 1;
		}
	      else if (need_entry & PLT_PLABEL)
		{
		  bfd_signed_vma *local_got_refcounts;
		  bfd_signed_vma *local_plt_refcounts;

		  local_got_refcounts = elf_local_got_refcounts (abfd);
		  if (local_got_refcounts == NULL)
		    {
		      bfd_size_type size;

		      /* Allocate space for local got offsets and local
			 plt offsets.  */
		      size = symtab_hdr->sh_info;
		      size *= 2 * sizeof (bfd_signed_vma);
		      local_got_refcounts = bfd_zalloc (abfd, size);
		      if (local_got_refcounts == NULL)
			return FALSE;
		      elf_local_got_refcounts (abfd) = local_got_refcounts;
		    }
		  local_plt_refcounts = (local_got_refcounts
					 + symtab_hdr->sh_info);
		  local_plt_refcounts[r_symndx] += 1;
		}
	    }
	}

      if (need_entry & NEED_DYNREL)
	{
	  /* Flag this symbol as having a non-got, non-plt reference
	     so that we generate copy relocs if it turns out to be
	     dynamic.  */
	  if (h != NULL && !info->shared)
	    h->elf.non_got_ref = 1;

	  /* If we are creating a shared library then we need to copy
	     the reloc into the shared library.  However, if we are
	     linking with -Bsymbolic, we need only copy absolute
	     relocs or relocs against symbols that are not defined in
	     an object we are including in the link.  PC- or DP- or
	     DLT-relative relocs against any local sym or global sym
	     with DEF_REGULAR set, can be discarded.  At this point we
	     have not seen all the input files, so it is possible that
	     DEF_REGULAR is not set now but will be set later (it is
	     never cleared).  We account for that possibility below by
	     storing information in the dyn_relocs field of the
	     hash table entry.

	     A similar situation to the -Bsymbolic case occurs when
	     creating shared libraries and symbol visibility changes
	     render the symbol local.

	     As it turns out, all the relocs we will be creating here
	     are absolute, so we cannot remove them on -Bsymbolic
	     links or visibility changes anyway.  A STUB_REL reloc
	     is absolute too, as in that case it is the reloc in the
	     stub we will be creating, rather than copying the PCREL
	     reloc in the branch.

	     If on the other hand, we are creating an executable, we
	     may need to keep relocations for symbols satisfied by a
	     dynamic library if we manage to avoid copy relocs for the
	     symbol.  */
	  if ((info->shared
	       && (sec->flags & SEC_ALLOC) != 0
	       && (IS_ABSOLUTE_RELOC (r_type)
		   || (h != NULL
		       && (!info->symbolic
			   || h->elf.root.type == bfd_link_hash_defweak
			   || !h->elf.def_regular))))
	      || (ELIMINATE_COPY_RELOCS
		  && !info->shared
		  && (sec->flags & SEC_ALLOC) != 0
		  && h != NULL
		  && (h->elf.root.type == bfd_link_hash_defweak
		      || !h->elf.def_regular)))
	    {
	      struct elf32_hppa_dyn_reloc_entry *p;
	      struct elf32_hppa_dyn_reloc_entry **head;

	      /* Create a reloc section in dynobj and make room for
		 this reloc.  */
	      if (sreloc == NULL)
		{
		  char *name;
		  bfd *dynobj;

		  name = (bfd_elf_string_from_elf_section
			  (abfd,
			   elf_elfheader (abfd)->e_shstrndx,
			   elf_section_data (sec)->rel_hdr.sh_name));
		  if (name == NULL)
		    {
		      (*_bfd_error_handler)
			(_("Could not find relocation section for %s"),
			 sec->name);
		      bfd_set_error (bfd_error_bad_value);
		      return FALSE;
		    }

		  if (htab->elf.dynobj == NULL)
		    htab->elf.dynobj = abfd;

		  dynobj = htab->elf.dynobj;
		  sreloc = bfd_get_section_by_name (dynobj, name);
		  if (sreloc == NULL)
		    {
		      flagword flags;

		      sreloc = bfd_make_section (dynobj, name);
		      flags = (SEC_HAS_CONTENTS | SEC_READONLY
			       | SEC_IN_MEMORY | SEC_LINKER_CREATED);
		      if ((sec->flags & SEC_ALLOC) != 0)
			flags |= SEC_ALLOC | SEC_LOAD;
		      if (sreloc == NULL
			  || !bfd_set_section_flags (dynobj, sreloc, flags)
			  || !bfd_set_section_alignment (dynobj, sreloc, 2))
			return FALSE;
		    }

		  elf_section_data (sec)->sreloc = sreloc;
		}

	      /* If this is a global symbol, we count the number of
		 relocations we need for this symbol.  */
	      if (h != NULL)
		{
		  head = &h->dyn_relocs;
		}
	      else
		{
		  /* Track dynamic relocs needed for local syms too.
		     We really need local syms available to do this
		     easily.  Oh well.  */

		  asection *s;
		  s = bfd_section_from_r_symndx (abfd, &htab->sym_sec,
						 sec, r_symndx);
		  if (s == NULL)
		    return FALSE;

		  head = ((struct elf32_hppa_dyn_reloc_entry **)
			  &elf_section_data (s)->local_dynrel);
		}

	      p = *head;
	      if (p == NULL || p->sec != sec)
		{
		  p = bfd_alloc (htab->elf.dynobj, sizeof *p);
		  if (p == NULL)
		    return FALSE;
		  p->next = *head;
		  *head = p;
		  p->sec = sec;
		  p->count = 0;
#if RELATIVE_DYNRELOCS
		  p->relative_count = 0;
#endif
		}

	      p->count += 1;
#if RELATIVE_DYNRELOCS
	      if (!IS_ABSOLUTE_RELOC (rtype))
		p->relative_count += 1;
#endif
	    }
	}
    }

  return TRUE;
}

/* Return the section that should be marked against garbage collection
   for a given relocation.  */

static asection *
elf32_hppa_gc_mark_hook (asection *sec,
			 struct bfd_link_info *info ATTRIBUTE_UNUSED,
			 Elf_Internal_Rela *rel,
			 struct elf_link_hash_entry *h,
			 Elf_Internal_Sym *sym)
{
  if (h != NULL)
    {
      switch ((unsigned int) ELF32_R_TYPE (rel->r_info))
	{
	case R_PARISC_GNU_VTINHERIT:
	case R_PARISC_GNU_VTENTRY:
	  break;

	default:
	  switch (h->root.type)
	    {
	    case bfd_link_hash_defined:
	    case bfd_link_hash_defweak:
	      return h->root.u.def.section;

	    case bfd_link_hash_common:
	      return h->root.u.c.p->section;

	    default:
	      break;
	    }
	}
    }
  else
    return bfd_section_from_elf_index (sec->owner, sym->st_shndx);

  return NULL;
}

/* Update the got and plt entry reference counts for the section being
   removed.  */

static bfd_boolean
elf32_hppa_gc_sweep_hook (bfd *abfd,
			  struct bfd_link_info *info ATTRIBUTE_UNUSED,
			  asection *sec,
			  const Elf_Internal_Rela *relocs)
{
  Elf_Internal_Shdr *symtab_hdr;
  struct elf_link_hash_entry **sym_hashes;
  bfd_signed_vma *local_got_refcounts;
  bfd_signed_vma *local_plt_refcounts;
  const Elf_Internal_Rela *rel, *relend;

  elf_section_data (sec)->local_dynrel = NULL;

  symtab_hdr = &elf_tdata (abfd)->symtab_hdr;
  sym_hashes = elf_sym_hashes (abfd);
  local_got_refcounts = elf_local_got_refcounts (abfd);
  local_plt_refcounts = local_got_refcounts;
  if (local_plt_refcounts != NULL)
    local_plt_refcounts += symtab_hdr->sh_info;

  relend = relocs + sec->reloc_count;
  for (rel = relocs; rel < relend; rel++)
    {
      unsigned long r_symndx;
      unsigned int r_type;
      struct elf_link_hash_entry *h = NULL;

      r_symndx = ELF32_R_SYM (rel->r_info);
      if (r_symndx >= symtab_hdr->sh_info)
	{
	  struct elf32_hppa_link_hash_entry *eh;
	  struct elf32_hppa_dyn_reloc_entry **pp;
	  struct elf32_hppa_dyn_reloc_entry *p;

	  h = sym_hashes[r_symndx - symtab_hdr->sh_info];
	  while (h->root.type == bfd_link_hash_indirect
		 || h->root.type == bfd_link_hash_warning)
	    h = (struct elf_link_hash_entry *) h->root.u.i.link;
	  eh = (struct elf32_hppa_link_hash_entry *) h;

	  for (pp = &eh->dyn_relocs; (p = *pp) != NULL; pp = &p->next)
	    if (p->sec == sec)
	      {
		/* Everything must go for SEC.  */
		*pp = p->next;
		break;
	      }
	}

      r_type = ELF32_R_TYPE (rel->r_info);
      switch (r_type)
	{
	case R_PARISC_DLTIND14F:
	case R_PARISC_DLTIND14R:
	case R_PARISC_DLTIND21L:
	  if (h != NULL)
	    {
	      if (h->got.refcount > 0)
		h->got.refcount -= 1;
	    }
	  else if (local_got_refcounts != NULL)
	    {
	      if (local_got_refcounts[r_symndx] > 0)
		local_got_refcounts[r_symndx] -= 1;
	    }
	  break;

	case R_PARISC_PCREL12F:
	case R_PARISC_PCREL17C:
	case R_PARISC_PCREL17F:
	case R_PARISC_PCREL22F:
	  if (h != NULL)
	    {
	      if (h->plt.refcount > 0)
		h->plt.refcount -= 1;
	    }
	  break;

	case R_PARISC_PLABEL14R:
	case R_PARISC_PLABEL21L:
	case R_PARISC_PLABEL32:
	  if (h != NULL)
	    {
	      if (h->plt.refcount > 0)
		h->plt.refcount -= 1;
	    }
	  else if (local_plt_refcounts != NULL)
	    {
	      if (local_plt_refcounts[r_symndx] > 0)
		local_plt_refcounts[r_symndx] -= 1;
	    }
	  break;

	default:
	  break;
	}
    }

  return TRUE;
}

/* Support for core dump NOTE sections.  */

static bfd_boolean
elf32_hppa_grok_prstatus (bfd *abfd, Elf_Internal_Note *note)
{
  int offset;
  size_t size;

  switch (note->descsz)
    {
      default:
	return FALSE;

      case 396:		/* Linux/hppa */
	/* pr_cursig */
	elf_tdata (abfd)->core_signal = bfd_get_16 (abfd, note->descdata + 12);

	/* pr_pid */
	elf_tdata (abfd)->core_pid = bfd_get_32 (abfd, note->descdata + 24);

	/* pr_reg */
	offset = 72;
	size = 320;

	break;
    }

  /* Make a ".reg/999" section.  */
  return _bfd_elfcore_make_pseudosection (abfd, ".reg",
					  size, note->descpos + offset);
}

static bfd_boolean
elf32_hppa_grok_psinfo (bfd *abfd, Elf_Internal_Note *note)
{
  switch (note->descsz)
    {
      default:
	return FALSE;

      case 124:		/* Linux/hppa elf_prpsinfo.  */
	elf_tdata (abfd)->core_program
	  = _bfd_elfcore_strndup (abfd, note->descdata + 28, 16);
	elf_tdata (abfd)->core_command
	  = _bfd_elfcore_strndup (abfd, note->descdata + 44, 80);
    }

  /* Note that for some reason, a spurious space is tacked
     onto the end of the args in some (at least one anyway)
     implementations, so strip it off if it exists.  */
  {
    char *command = elf_tdata (abfd)->core_command;
    int n = strlen (command);

    if (0 < n && command[n - 1] == ' ')
      command[n - 1] = '\0';
  }

  return TRUE;
}

/* Our own version of hide_symbol, so that we can keep plt entries for
   plabels.  */

static void
elf32_hppa_hide_symbol (struct bfd_link_info *info,
			struct elf_link_hash_entry *h,
			bfd_boolean force_local)
{
  if (force_local)
    {
      h->forced_local = 1;
      if (h->dynindx != -1)
	{
	  h->dynindx = -1;
	  _bfd_elf_strtab_delref (elf_hash_table (info)->dynstr,
				  h->dynstr_index);
	}
    }

  if (! ((struct elf32_hppa_link_hash_entry *) h)->plabel)
    {
      h->needs_plt = 0;
      h->plt = elf_hash_table (info)->init_refcount;
    }
}

/* Adjust a symbol defined by a dynamic object and referenced by a
   regular object.  The current definition is in some section of the
   dynamic object, but we're not including those sections.  We have to
   change the definition to something the rest of the link can
   understand.  */

static bfd_boolean
elf32_hppa_adjust_dynamic_symbol (struct bfd_link_info *info,
				  struct elf_link_hash_entry *h)
{
  struct elf32_hppa_link_hash_table *htab;
  asection *s;
  unsigned int power_of_two;

  /* If this is a function, put it in the procedure linkage table.  We
     will fill in the contents of the procedure linkage table later.  */
  if (h->type == STT_FUNC
      || h->needs_plt)
    {
      if (h->plt.refcount <= 0
	  || (h->def_regular
	      && h->root.type != bfd_link_hash_defweak
	      && ! ((struct elf32_hppa_link_hash_entry *) h)->plabel
	      && (!info->shared || info->symbolic)))
	{
	  /* The .plt entry is not needed when:
	     a) Garbage collection has removed all references to the
	     symbol, or
	     b) We know for certain the symbol is defined in this
	     object, and it's not a weak definition, nor is the symbol
	     used by a plabel relocation.  Either this object is the
	     application or we are doing a shared symbolic link.  */

	  h->plt.offset = (bfd_vma) -1;
	  h->needs_plt = 0;
	}

      return TRUE;
    }
  else
    h->plt.offset = (bfd_vma) -1;

  /* If this is a weak symbol, and there is a real definition, the
     processor independent code will have arranged for us to see the
     real definition first, and we can just use the same value.  */
  if (h->u.weakdef != NULL)
    {
      if (h->u.weakdef->root.type != bfd_link_hash_defined
	  && h->u.weakdef->root.type != bfd_link_hash_defweak)
	abort ();
      h->root.u.def.section = h->u.weakdef->root.u.def.section;
      h->root.u.def.value = h->u.weakdef->root.u.def.value;
      if (ELIMINATE_COPY_RELOCS)
	h->non_got_ref = h->u.weakdef->non_got_ref;
      return TRUE;
    }

  /* This is a reference to a symbol defined by a dynamic object which
     is not a function.  */

  /* If we are creating a shared library, we must presume that the
     only references to the symbol are via the global offset table.
     For such cases we need not do anything here; the relocations will
     be handled correctly by relocate_section.  */
  if (info->shared)
    return TRUE;

  /* If there are no references to this symbol that do not use the
     GOT, we don't need to generate a copy reloc.  */
  if (!h->non_got_ref)
    return TRUE;

  if (ELIMINATE_COPY_RELOCS)
    {
      struct elf32_hppa_link_hash_entry *eh;
      struct elf32_hppa_dyn_reloc_entry *p;

      eh = (struct elf32_hppa_link_hash_entry *) h;
      for (p = eh->dyn_relocs; p != NULL; p = p->next)
	{
	  s = p->sec->output_section;
	  if (s != NULL && (s->flags & SEC_READONLY) != 0)
	    break;
	}

      /* If we didn't find any dynamic relocs in read-only sections, then
	 we'll be keeping the dynamic relocs and avoiding the copy reloc.  */
      if (p == NULL)
	{
	  h->non_got_ref = 0;
	  return TRUE;
	}
    }

  /* We must allocate the symbol in our .dynbss section, which will
     become part of the .bss section of the executable.  There will be
     an entry for this symbol in the .dynsym section.  The dynamic
     object will contain position independent code, so all references
     from the dynamic object to this symbol will go through the global
     offset table.  The dynamic linker will use the .dynsym entry to
     determine the address it must put in the global offset table, so
     both the dynamic object and the regular object will refer to the
     same memory location for the variable.  */

  htab = hppa_link_hash_table (info);

  /* We must generate a COPY reloc to tell the dynamic linker to
     copy the initial value out of the dynamic object and into the
     runtime process image.  */
  if ((h->root.u.def.section->flags & SEC_ALLOC) != 0)
    {
      htab->srelbss->size += sizeof (Elf32_External_Rela);
      h->needs_copy = 1;
    }

  /* We need to figure out the alignment required for this symbol.  I
     have no idea how other ELF linkers handle this.  */

  power_of_two = bfd_log2 (h->size);
  if (power_of_two > 3)
    power_of_two = 3;

  /* Apply the required alignment.  */
  s = htab->sdynbss;
  s->size = BFD_ALIGN (s->size, (bfd_size_type) (1 << power_of_two));
  if (power_of_two > bfd_get_section_alignment (htab->elf.dynobj, s))
    {
      if (! bfd_set_section_alignment (htab->elf.dynobj, s, power_of_two))
	return FALSE;
    }

  /* Define the symbol as being at this point in the section.  */
  h->root.u.def.section = s;
  h->root.u.def.value = s->size;

  /* Increment the section size to make room for the symbol.  */
  s->size += h->size;

  return TRUE;
}

/* Allocate space in the .plt for entries that won't have relocations.
   ie. plabel entries.  */

static bfd_boolean
allocate_plt_static (struct elf_link_hash_entry *h, void *inf)
{
  struct bfd_link_info *info;
  struct elf32_hppa_link_hash_table *htab;
  asection *s;

  if (h->root.type == bfd_link_hash_indirect)
    return TRUE;

  if (h->root.type == bfd_link_hash_warning)
    h = (struct elf_link_hash_entry *) h->root.u.i.link;

  info = inf;
  htab = hppa_link_hash_table (info);
  if (htab->elf.dynamic_sections_created
      && h->plt.refcount > 0)
    {
      /* Make sure this symbol is output as a dynamic symbol.
	 Undefined weak syms won't yet be marked as dynamic.  */
      if (h->dynindx == -1
	  && !h->forced_local
	  && h->type != STT_PARISC_MILLI)
	{
	  if (! bfd_elf_link_record_dynamic_symbol (info, h))
	    return FALSE;
	}

      if (WILL_CALL_FINISH_DYNAMIC_SYMBOL (1, info->shared, h))
	{
	  /* Allocate these later.  From this point on, h->plabel
	     means that the plt entry is only used by a plabel.
	     We'll be using a normal plt entry for this symbol, so
	     clear the plabel indicator.  */
	  ((struct elf32_hppa_link_hash_entry *) h)->plabel = 0;
	}
      else if (((struct elf32_hppa_link_hash_entry *) h)->plabel)
	{
	  /* Make an entry in the .plt section for plabel references
	     that won't have a .plt entry for other reasons.  */
	  s = htab->splt;
	  h->plt.offset = s->size;
	  s->size += PLT_ENTRY_SIZE;
	}
      else
	{
	  /* No .plt entry needed.  */
	  h->plt.offset = (bfd_vma) -1;
	  h->needs_plt = 0;
	}
    }
  else
    {
      h->plt.offset = (bfd_vma) -1;
      h->needs_plt = 0;
    }

  return TRUE;
}

/* Allocate space in .plt, .got and associated reloc sections for
   global syms.  */

static bfd_boolean
allocate_dynrelocs (struct elf_link_hash_entry *h, void *inf)
{
  struct bfd_link_info *info;
  struct elf32_hppa_link_hash_table *htab;
  asection *s;
  struct elf32_hppa_link_hash_entry *eh;
  struct elf32_hppa_dyn_reloc_entry *p;

  if (h->root.type == bfd_link_hash_indirect)
    return TRUE;

  if (h->root.type == bfd_link_hash_warning)
    h = (struct elf_link_hash_entry *) h->root.u.i.link;

  info = inf;
  htab = hppa_link_hash_table (info);
  if (htab->elf.dynamic_sections_created
      && h->plt.offset != (bfd_vma) -1
      && !((struct elf32_hppa_link_hash_entry *) h)->plabel)
    {
      /* Make an entry in the .plt section.  */
      s = htab->splt;
      h->plt.offset = s->size;
      s->size += PLT_ENTRY_SIZE;

      /* We also need to make an entry in the .rela.plt section.  */
      htab->srelplt->size += sizeof (Elf32_External_Rela);
      htab->need_plt_stub = 1;
    }

  if (h->got.refcount > 0)
    {
      /* Make sure this symbol is output as a dynamic symbol.
	 Undefined weak syms won't yet be marked as dynamic.  */
      if (h->dynindx == -1
	  && !h->forced_local
	  && h->type != STT_PARISC_MILLI)
	{
	  if (! bfd_elf_link_record_dynamic_symbol (info, h))
	    return FALSE;
	}

      s = htab->sgot;
      h->got.offset = s->size;
      s->size += GOT_ENTRY_SIZE;
      if (htab->elf.dynamic_sections_created
	  && (info->shared
	      || (h->dynindx != -1
		  && !h->forced_local)))
	{
	  htab->srelgot->size += sizeof (Elf32_External_Rela);
	}
    }
  else
    h->got.offset = (bfd_vma) -1;

  eh = (struct elf32_hppa_link_hash_entry *) h;
  if (eh->dyn_relocs == NULL)
    return TRUE;

  /* If this is a -Bsymbolic shared link, then we need to discard all
     space allocated for dynamic pc-relative relocs against symbols
     defined in a regular object.  For the normal shared case, discard
     space for relocs that have become local due to symbol visibility
     changes.  */
  if (info->shared)
    {
#if RELATIVE_DYNRELOCS
      if (SYMBOL_CALLS_LOCAL (info, h))
	{
	  struct elf32_hppa_dyn_reloc_entry **pp;

	  for (pp = &eh->dyn_relocs; (p = *pp) != NULL; )
	    {
	      p->count -= p->relative_count;
	      p->relative_count = 0;
	      if (p->count == 0)
		*pp = p->next;
	      else
		pp = &p->next;
	    }
	}
#endif

      /* Also discard relocs on undefined weak syms with non-default
	 visibility.  */
      if (ELF_ST_VISIBILITY (h->other) != STV_DEFAULT
	  && h->root.type == bfd_link_hash_undefweak)
	eh->dyn_relocs = NULL;
    }
  else
    {
      /* For the non-shared case, discard space for relocs against
	 symbols which turn out to need copy relocs or are not
	 dynamic.  */
      if (!h->non_got_ref
	  && ((ELIMINATE_COPY_RELOCS
	       && h->def_dynamic
	       && !h->def_regular)
	      || (htab->elf.dynamic_sections_created
		  && (h->root.type == bfd_link_hash_undefweak
		      || h->root.type == bfd_link_hash_undefined))))
	{
	  /* Make sure this symbol is output as a dynamic symbol.
	     Undefined weak syms won't yet be marked as dynamic.  */
	  if (h->dynindx == -1
	      && !h->forced_local
	      && h->type != STT_PARISC_MILLI)
	    {
	      if (! bfd_elf_link_record_dynamic_symbol (info, h))
		return FALSE;
	    }

	  /* If that succeeded, we know we'll be keeping all the
	     relocs.  */
	  if (h->dynindx != -1)
	    goto keep;
	}

      eh->dyn_relocs = NULL;
      return TRUE;

    keep: ;
    }

  /* Finally, allocate space.  */
  for (p = eh->dyn_relocs; p != NULL; p = p->next)
    {
      asection *sreloc = elf_section_data (p->sec)->sreloc;
      sreloc->size += p->count * sizeof (Elf32_External_Rela);
    }

  return TRUE;
}

/* This function is called via elf_link_hash_traverse to force
   millicode symbols local so they do not end up as globals in the
   dynamic symbol table.  We ought to be able to do this in
   adjust_dynamic_symbol, but our adjust_dynamic_symbol is not called
   for all dynamic symbols.  Arguably, this is a bug in
   elf_adjust_dynamic_symbol.  */

static bfd_boolean
clobber_millicode_symbols (struct elf_link_hash_entry *h,
			   struct bfd_link_info *info)
{
  if (h->root.type == bfd_link_hash_warning)
    h = (struct elf_link_hash_entry *) h->root.u.i.link;

  if (h->type == STT_PARISC_MILLI
      && !h->forced_local)
    {
      elf32_hppa_hide_symbol (info, h, TRUE);
    }
  return TRUE;
}

/* Find any dynamic relocs that apply to read-only sections.  */

static bfd_boolean
readonly_dynrelocs (struct elf_link_hash_entry *h, void *inf)
{
  struct elf32_hppa_link_hash_entry *eh;
  struct elf32_hppa_dyn_reloc_entry *p;

  if (h->root.type == bfd_link_hash_warning)
    h = (struct elf_link_hash_entry *) h->root.u.i.link;

  eh = (struct elf32_hppa_link_hash_entry *) h;
  for (p = eh->dyn_relocs; p != NULL; p = p->next)
    {
      asection *s = p->sec->output_section;

      if (s != NULL && (s->flags & SEC_READONLY) != 0)
	{
	  struct bfd_link_info *info = inf;

	  info->flags |= DF_TEXTREL;

	  /* Not an error, just cut short the traversal.  */
	  return FALSE;
	}
    }
  return TRUE;
}

/* Set the sizes of the dynamic sections.  */

static bfd_boolean
elf32_hppa_size_dynamic_sections (bfd *output_bfd ATTRIBUTE_UNUSED,
				  struct bfd_link_info *info)
{
  struct elf32_hppa_link_hash_table *htab;
  bfd *dynobj;
  bfd *ibfd;
  asection *s;
  bfd_boolean relocs;

  htab = hppa_link_hash_table (info);
  dynobj = htab->elf.dynobj;
  if (dynobj == NULL)
    abort ();

  if (htab->elf.dynamic_sections_created)
    {
      /* Set the contents of the .interp section to the interpreter.  */
      if (info->executable)
	{
	  s = bfd_get_section_by_name (dynobj, ".interp");
	  if (s == NULL)
	    abort ();
	  s->size = sizeof ELF_DYNAMIC_INTERPRETER;
	  s->contents = (unsigned char *) ELF_DYNAMIC_INTERPRETER;
	}

      /* Force millicode symbols local.  */
      elf_link_hash_traverse (&htab->elf,
			      clobber_millicode_symbols,
			      info);
    }

  /* Set up .got and .plt offsets for local syms, and space for local
     dynamic relocs.  */
  for (ibfd = info->input_bfds; ibfd != NULL; ibfd = ibfd->link_next)
    {
      bfd_signed_vma *local_got;
      bfd_signed_vma *end_local_got;
      bfd_signed_vma *local_plt;
      bfd_signed_vma *end_local_plt;
      bfd_size_type locsymcount;
      Elf_Internal_Shdr *symtab_hdr;
      asection *srel;

      if (bfd_get_flavour (ibfd) != bfd_target_elf_flavour)
	continue;

      for (s = ibfd->sections; s != NULL; s = s->next)
	{
	  struct elf32_hppa_dyn_reloc_entry *p;

	  for (p = ((struct elf32_hppa_dyn_reloc_entry *)
		    elf_section_data (s)->local_dynrel);
	       p != NULL;
	       p = p->next)
	    {
	      if (!bfd_is_abs_section (p->sec)
		  && bfd_is_abs_section (p->sec->output_section))
		{
		  /* Input section has been discarded, either because
		     it is a copy of a linkonce section or due to
		     linker script /DISCARD/, so we'll be discarding
		     the relocs too.  */
		}
	      else if (p->count != 0)
		{
		  srel = elf_section_data (p->sec)->sreloc;
		  srel->size += p->count * sizeof (Elf32_External_Rela);
		  if ((p->sec->output_section->flags & SEC_READONLY) != 0)
		    info->flags |= DF_TEXTREL;
		}
	    }
	}

      local_got = elf_local_got_refcounts (ibfd);
      if (!local_got)
	continue;

      symtab_hdr = &elf_tdata (ibfd)->symtab_hdr;
      locsymcount = symtab_hdr->sh_info;
      end_local_got = local_got + locsymcount;
      s = htab->sgot;
      srel = htab->srelgot;
      for (; local_got < end_local_got; ++local_got)
	{
	  if (*local_got > 0)
	    {
	      *local_got = s->size;
	      s->size += GOT_ENTRY_SIZE;
	      if (info->shared)
		srel->size += sizeof (Elf32_External_Rela);
	    }
	  else
	    *local_got = (bfd_vma) -1;
	}

      local_plt = end_local_got;
      end_local_plt = local_plt + locsymcount;
      if (! htab->elf.dynamic_sections_created)
	{
	  /* Won't be used, but be safe.  */
	  for (; local_plt < end_local_plt; ++local_plt)
	    *local_plt = (bfd_vma) -1;
	}
      else
	{
	  s = htab->splt;
	  srel = htab->srelplt;
	  for (; local_plt < end_local_plt; ++local_plt)
	    {
	      if (*local_plt > 0)
		{
		  *local_plt = s->size;
		  s->size += PLT_ENTRY_SIZE;
		  if (info->shared)
		    srel->size += sizeof (Elf32_External_Rela);
		}
	      else
		*local_plt = (bfd_vma) -1;
	    }
	}
    }

  /* Do all the .plt entries without relocs first.  The dynamic linker
     uses the last .plt reloc to find the end of the .plt (and hence
     the start of the .got) for lazy linking.  */
  elf_link_hash_traverse (&htab->elf, allocate_plt_static, info);

  /* Allocate global sym .plt and .got entries, and space for global
     sym dynamic relocs.  */
  elf_link_hash_traverse (&htab->elf, allocate_dynrelocs, info);

  /* The check_relocs and adjust_dynamic_symbol entry points have
     determined the sizes of the various dynamic sections.  Allocate
     memory for them.  */
  relocs = FALSE;
  for (s = dynobj->sections; s != NULL; s = s->next)
    {
      if ((s->flags & SEC_LINKER_CREATED) == 0)
	continue;

      if (s == htab->splt)
	{
	  if (htab->need_plt_stub)
	    {
	      /* Make space for the plt stub at the end of the .plt
		 section.  We want this stub right at the end, up
		 against the .got section.  */
	      int gotalign = bfd_section_alignment (dynobj, htab->sgot);
	      int pltalign = bfd_section_alignment (dynobj, s);
	      bfd_size_type mask;

	      if (gotalign > pltalign)
		bfd_set_section_alignment (dynobj, s, gotalign);
	      mask = ((bfd_size_type) 1 << gotalign) - 1;
	      s->size = (s->size + sizeof (plt_stub) + mask) & ~mask;
	    }
	}
      else if (s == htab->sgot)
	;
      else if (strncmp (bfd_get_section_name (dynobj, s), ".rela", 5) == 0)
	{
	  if (s->size != 0)
	    {
	      /* Remember whether there are any reloc sections other
		 than .rela.plt.  */
	      if (s != htab->srelplt)
		relocs = TRUE;

	      /* We use the reloc_count field as a counter if we need
		 to copy relocs into the output file.  */
	      s->reloc_count = 0;
	    }
	}
      else
	{
	  /* It's not one of our sections, so don't allocate space.  */
	  continue;
	}

      if (s->size == 0)
	{
	  /* If we don't need this section, strip it from the
	     output file.  This is mostly to handle .rela.bss and
	     .rela.plt.  We must create both sections in
	     create_dynamic_sections, because they must be created
	     before the linker maps input sections to output
	     sections.  The linker does that before
	     adjust_dynamic_symbol is called, and it is that
	     function which decides whether anything needs to go
	     into these sections.  */
	  _bfd_strip_section_from_output (info, s);
	  continue;
	}

      /* Allocate memory for the section contents.  Zero it, because
	 we may not fill in all the reloc sections.  */
      s->contents = bfd_zalloc (dynobj, s->size);
      if (s->contents == NULL && s->size != 0)
	return FALSE;
    }

  if (htab->elf.dynamic_sections_created)
    {
      /* Like IA-64 and HPPA64, always create a DT_PLTGOT.  It
	 actually has nothing to do with the PLT, it is how we
	 communicate the LTP value of a load module to the dynamic
	 linker.  */
#define add_dynamic_entry(TAG, VAL) \
  _bfd_elf_add_dynamic_entry (info, TAG, VAL)

      if (!add_dynamic_entry (DT_PLTGOT, 0))
	return FALSE;

      /* Add some entries to the .dynamic section.  We fill in the
	 values later, in elf32_hppa_finish_dynamic_sections, but we
	 must add the entries now so that we get the correct size for
	 the .dynamic section.  The DT_DEBUG entry is filled in by the
	 dynamic linker and used by the debugger.  */
      if (!info->shared)
	{
	  if (!add_dynamic_entry (DT_DEBUG, 0))
	    return FALSE;
	}

      if (htab->srelplt->size != 0)
	{
	  if (!add_dynamic_entry (DT_PLTRELSZ, 0)
	      || !add_dynamic_entry (DT_PLTREL, DT_RELA)
	      || !add_dynamic_entry (DT_JMPREL, 0))
	    return FALSE;
	}

      if (relocs)
	{
	  if (!add_dynamic_entry (DT_RELA, 0)
	      || !add_dynamic_entry (DT_RELASZ, 0)
	      || !add_dynamic_entry (DT_RELAENT, sizeof (Elf32_External_Rela)))
	    return FALSE;

	  /* If any dynamic relocs apply to a read-only section,
	     then we need a DT_TEXTREL entry.  */
	  if ((info->flags & DF_TEXTREL) == 0)
	    elf_link_hash_traverse (&htab->elf, readonly_dynrelocs, info);

	  if ((info->flags & DF_TEXTREL) != 0)
	    {
	      if (!add_dynamic_entry (DT_TEXTREL, 0))
		return FALSE;
	    }
	}
    }
#undef add_dynamic_entry

  return TRUE;
}

/* External entry points for sizing and building linker stubs.  */

/* Set up various things so that we can make a list of input sections
   for each output section included in the link.  Returns -1 on error,
   0 when no stubs will be needed, and 1 on success.  */

int
elf32_hppa_setup_section_lists (bfd *output_bfd, struct bfd_link_info *info)
{
  bfd *input_bfd;
  unsigned int bfd_count;
  int top_id, top_index;
  asection *section;
  asection **input_list, **list;
  bfd_size_type amt;
  struct elf32_hppa_link_hash_table *htab = hppa_link_hash_table (info);

  /* Count the number of input BFDs and find the top input section id.  */
  for (input_bfd = info->input_bfds, bfd_count = 0, top_id = 0;
       input_bfd != NULL;
       input_bfd = input_bfd->link_next)
    {
      bfd_count += 1;
      for (section = input_bfd->sections;
	   section != NULL;
	   section = section->next)
	{
	  if (top_id < section->id)
	    top_id = section->id;
	}
    }
  htab->bfd_count = bfd_count;

  amt = sizeof (struct map_stub) * (top_id + 1);
  htab->stub_group = bfd_zmalloc (amt);
  if (htab->stub_group == NULL)
    return -1;

  /* We can't use output_bfd->section_count here to find the top output
     section index as some sections may have been removed, and
     _bfd_strip_section_from_output doesn't renumber the indices.  */
  for (section = output_bfd->sections, top_index = 0;
       section != NULL;
       section = section->next)
    {
      if (top_index < section->index)
	top_index = section->index;
    }

  htab->top_index = top_index;
  amt = sizeof (asection *) * (top_index + 1);
  input_list = bfd_malloc (amt);
  htab->input_list = input_list;
  if (input_list == NULL)
    return -1;

  /* For sections we aren't interested in, mark their entries with a
     value we can check later.  */
  list = input_list + top_index;
  do
    *list = bfd_abs_section_ptr;
  while (list-- != input_list);

  for (section = output_bfd->sections;
       section != NULL;
       section = section->next)
    {
      if ((section->flags & SEC_CODE) != 0)
	input_list[section->index] = NULL;
    }

  return 1;
}

/* The linker repeatedly calls this function for each input section,
   in the order that input sections are linked into output sections.
   Build lists of input sections to determine groupings between which
   we may insert linker stubs.  */

void
elf32_hppa_next_input_section (struct bfd_link_info *info, asection *isec)
{
  struct elf32_hppa_link_hash_table *htab = hppa_link_hash_table (info);

  if (isec->output_section->index <= htab->top_index)
    {
      asection **list = htab->input_list + isec->output_section->index;
      if (*list != bfd_abs_section_ptr)
	{
	  /* Steal the link_sec pointer for our list.  */
#define PREV_SEC(sec) (htab->stub_group[(sec)->id].link_sec)
	  /* This happens to make the list in reverse order,
	     which is what we want.  */
	  PREV_SEC (isec) = *list;
	  *list = isec;
	}
    }
}

/* See whether we can group stub sections together.  Grouping stub
   sections may result in fewer stubs.  More importantly, we need to
   put all .init* and .fini* stubs at the beginning of the .init or
   .fini output sections respectively, because glibc splits the
   _init and _fini functions into multiple parts.  Putting a stub in
   the middle of a function is not a good idea.  */

static void
group_sections (struct elf32_hppa_link_hash_table *htab,
		bfd_size_type stub_group_size,
		bfd_boolean stubs_always_before_branch)
{
  asection **list = htab->input_list + htab->top_index;
  do
    {
      asection *tail = *list;
      if (tail == bfd_abs_section_ptr)
	continue;
      while (tail != NULL)
	{
	  asection *curr;
	  asection *prev;
	  bfd_size_type total;
	  bfd_boolean big_sec;

	  curr = tail;
	  total = tail->size;
	  big_sec = total >= stub_group_size;

	  while ((prev = PREV_SEC (curr)) != NULL
		 && ((total += curr->output_offset - prev->output_offset)
		     < stub_group_size))
	    curr = prev;

	  /* OK, the size from the start of CURR to the end is less
	     than 240000 bytes and thus can be handled by one stub
	     section.  (or the tail section is itself larger than
	     240000 bytes, in which case we may be toast.)
	     We should really be keeping track of the total size of
	     stubs added here, as stubs contribute to the final output
	     section size.  That's a little tricky, and this way will
	     only break if stubs added total more than 22144 bytes, or
	     2768 long branch stubs.  It seems unlikely for more than
	     2768 different functions to be called, especially from
	     code only 240000 bytes long.  This limit used to be
	     250000, but c++ code tends to generate lots of little
	     functions, and sometimes violated the assumption.  */
	  do
	    {
	      prev = PREV_SEC (tail);
	      /* Set up this stub group.  */
	      htab->stub_group[tail->id].link_sec = curr;
	    }
	  while (tail != curr && (tail = prev) != NULL);

	  /* But wait, there's more!  Input sections up to 240000
	     bytes before the stub section can be handled by it too.
	     Don't do this if we have a really large section after the
	     stubs, as adding more stubs increases the chance that
	     branches may not reach into the stub section.  */
	  if (!stubs_always_before_branch && !big_sec)
	    {
	      total = 0;
	      while (prev != NULL
		     && ((total += tail->output_offset - prev->output_offset)
			 < stub_group_size))
		{
		  tail = prev;
		  prev = PREV_SEC (tail);
		  htab->stub_group[tail->id].link_sec = curr;
		}
	    }
	  tail = prev;
	}
    }
  while (list-- != htab->input_list);
  free (htab->input_list);
#undef PREV_SEC
}

/* Read in all local syms for all input bfds, and create hash entries
   for export stubs if we are building a multi-subspace shared lib.
   Returns -1 on error, 1 if export stubs created, 0 otherwise.  */

static int
get_local_syms (bfd *output_bfd, bfd *input_bfd, struct bfd_link_info *info)
{
  unsigned int bfd_indx;
  Elf_Internal_Sym *local_syms, **all_local_syms;
  int stub_changed = 0;
  struct elf32_hppa_link_hash_table *htab = hppa_link_hash_table (info);

  /* We want to read in symbol extension records only once.  To do this
     we need to read in the local symbols in parallel and save them for
     later use; so hold pointers to the local symbols in an array.  */
  bfd_size_type amt = sizeof (Elf_Internal_Sym *) * htab->bfd_count;
  all_local_syms = bfd_zmalloc (amt);
  htab->all_local_syms = all_local_syms;
  if (all_local_syms == NULL)
    return -1;

  /* Walk over all the input BFDs, swapping in local symbols.
     If we are creating a shared library, create hash entries for the
     export stubs.  */
  for (bfd_indx = 0;
       input_bfd != NULL;
       input_bfd = input_bfd->link_next, bfd_indx++)
    {
      Elf_Internal_Shdr *symtab_hdr;

      /* We'll need the symbol table in a second.  */
      symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;
      if (symtab_hdr->sh_info == 0)
	continue;

      /* We need an array of the local symbols attached to the input bfd.  */
      local_syms = (Elf_Internal_Sym *) symtab_hdr->contents;
      if (local_syms == NULL)
	{
	  local_syms = bfd_elf_get_elf_syms (input_bfd, symtab_hdr,
					     symtab_hdr->sh_info, 0,
					     NULL, NULL, NULL);
	  /* Cache them for elf_link_input_bfd.  */
	  symtab_hdr->contents = (unsigned char *) local_syms;
	}
      if (local_syms == NULL)
	return -1;

      all_local_syms[bfd_indx] = local_syms;

      if (info->shared && htab->multi_subspace)
	{
	  struct elf_link_hash_entry **sym_hashes;
	  struct elf_link_hash_entry **end_hashes;
	  unsigned int symcount;

	  symcount = (symtab_hdr->sh_size / sizeof (Elf32_External_Sym)
		      - symtab_hdr->sh_info);
	  sym_hashes = elf_sym_hashes (input_bfd);
	  end_hashes = sym_hashes + symcount;

	  /* Look through the global syms for functions;  We need to
	     build export stubs for all globally visible functions.  */
	  for (; sym_hashes < end_hashes; sym_hashes++)
	    {
	      struct elf32_hppa_link_hash_entry *hash;

	      hash = (struct elf32_hppa_link_hash_entry *) *sym_hashes;

	      while (hash->elf.root.type == bfd_link_hash_indirect
		     || hash->elf.root.type == bfd_link_hash_warning)
		hash = ((struct elf32_hppa_link_hash_entry *)
			hash->elf.root.u.i.link);

	      /* At this point in the link, undefined syms have been
		 resolved, so we need to check that the symbol was
		 defined in this BFD.  */
	      if ((hash->elf.root.type == bfd_link_hash_defined
		   || hash->elf.root.type == bfd_link_hash_defweak)
		  && hash->elf.type == STT_FUNC
		  && hash->elf.root.u.def.section->output_section != NULL
		  && (hash->elf.root.u.def.section->output_section->owner
		      == output_bfd)
		  && hash->elf.root.u.def.section->owner == input_bfd
		  && hash->elf.def_regular
		  && !hash->elf.forced_local
		  && ELF_ST_VISIBILITY (hash->elf.other) == STV_DEFAULT)
		{
		  asection *sec;
		  const char *stub_name;
		  struct elf32_hppa_stub_hash_entry *stub_entry;

		  sec = hash->elf.root.u.def.section;
		  stub_name = hash->elf.root.root.string;
		  stub_entry = hppa_stub_hash_lookup (&htab->stub_hash_table,
						      stub_name,
						      FALSE, FALSE);
		  if (stub_entry == NULL)
		    {
		      stub_entry = hppa_add_stub (stub_name, sec, htab);
		      if (!stub_entry)
			return -1;

		      stub_entry->target_value = hash->elf.root.u.def.value;
		      stub_entry->target_section = hash->elf.root.u.def.section;
		      stub_entry->stub_type = hppa_stub_export;
		      stub_entry->h = hash;
		      stub_changed = 1;
		    }
		  else
		    {
		      (*_bfd_error_handler) (_("%B: duplicate export stub %s"),
					     input_bfd,
					     stub_name);
		    }
		}
	    }
	}
    }

  return stub_changed;
}

/* Determine and set the size of the stub section for a final link.

   The basic idea here is to examine all the relocations looking for
   PC-relative calls to a target that is unreachable with a "bl"
   instruction.  */

bfd_boolean
elf32_hppa_size_stubs
  (bfd *output_bfd, bfd *stub_bfd, struct bfd_link_info *info,
   bfd_boolean multi_subspace, bfd_signed_vma group_size,
   asection * (*add_stub_section) (const char *, asection *),
   void (*layout_sections_again) (void))
{
  bfd_size_type stub_group_size;
  bfd_boolean stubs_always_before_branch;
  bfd_boolean stub_changed;
  struct elf32_hppa_link_hash_table *htab = hppa_link_hash_table (info);

  /* Stash our params away.  */
  htab->stub_bfd = stub_bfd;
  htab->multi_subspace = multi_subspace;
  htab->add_stub_section = add_stub_section;
  htab->layout_sections_again = layout_sections_again;
  stubs_always_before_branch = group_size < 0;
  if (group_size < 0)
    stub_group_size = -group_size;
  else
    stub_group_size = group_size;
  if (stub_group_size == 1)
    {
      /* Default values.  */
      if (stubs_always_before_branch)
	{
	  stub_group_size = 7680000;
	  if (htab->has_17bit_branch || htab->multi_subspace)
	    stub_group_size = 240000;
	  if (htab->has_12bit_branch)
	    stub_group_size = 7500;
	}
      else
	{
	  stub_group_size = 6971392;
	  if (htab->has_17bit_branch || htab->multi_subspace)
	    stub_group_size = 217856;
	  if (htab->has_12bit_branch)
	    stub_group_size = 6808;
	}
    }

  group_sections (htab, stub_group_size, stubs_always_before_branch);

  switch (get_local_syms (output_bfd, info->input_bfds, info))
    {
    default:
      if (htab->all_local_syms)
	goto error_ret_free_local;
      return FALSE;

    case 0:
      stub_changed = FALSE;
      break;

    case 1:
      stub_changed = TRUE;
      break;
    }

  while (1)
    {
      bfd *input_bfd;
      unsigned int bfd_indx;
      asection *stub_sec;

      for (input_bfd = info->input_bfds, bfd_indx = 0;
	   input_bfd != NULL;
	   input_bfd = input_bfd->link_next, bfd_indx++)
	{
	  Elf_Internal_Shdr *symtab_hdr;
	  asection *section;
	  Elf_Internal_Sym *local_syms;

	  /* We'll need the symbol table in a second.  */
	  symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;
	  if (symtab_hdr->sh_info == 0)
	    continue;

	  local_syms = htab->all_local_syms[bfd_indx];

	  /* Walk over each section attached to the input bfd.  */
	  for (section = input_bfd->sections;
	       section != NULL;
	       section = section->next)
	    {
	      Elf_Internal_Rela *internal_relocs, *irelaend, *irela;

	      /* If there aren't any relocs, then there's nothing more
		 to do.  */
	      if ((section->flags & SEC_RELOC) == 0
		  || section->reloc_count == 0)
		continue;

	      /* If this section is a link-once section that will be
		 discarded, then don't create any stubs.  */
	      if (section->output_section == NULL
		  || section->output_section->owner != output_bfd)
		continue;

	      /* Get the relocs.  */
	      internal_relocs
		= _bfd_elf_link_read_relocs (input_bfd, section, NULL, NULL,
					     info->keep_memory);
	      if (internal_relocs == NULL)
		goto error_ret_free_local;

	      /* Now examine each relocation.  */
	      irela = internal_relocs;
	      irelaend = irela + section->reloc_count;
	      for (; irela < irelaend; irela++)
		{
		  unsigned int r_type, r_indx;
		  enum elf32_hppa_stub_type stub_type;
		  struct elf32_hppa_stub_hash_entry *stub_entry;
		  asection *sym_sec;
		  bfd_vma sym_value;
		  bfd_vma destination;
		  struct elf32_hppa_link_hash_entry *hash;
		  char *stub_name;
		  const asection *id_sec;

		  r_type = ELF32_R_TYPE (irela->r_info);
		  r_indx = ELF32_R_SYM (irela->r_info);

		  if (r_type >= (unsigned int) R_PARISC_UNIMPLEMENTED)
		    {
		      bfd_set_error (bfd_error_bad_value);
		    error_ret_free_internal:
		      if (elf_section_data (section)->relocs == NULL)
			free (internal_relocs);
		      goto error_ret_free_local;
		    }

		  /* Only look for stubs on call instructions.  */
		  if (r_type != (unsigned int) R_PARISC_PCREL12F
		      && r_type != (unsigned int) R_PARISC_PCREL17F
		      && r_type != (unsigned int) R_PARISC_PCREL22F)
		    continue;

		  /* Now determine the call target, its name, value,
		     section.  */
		  sym_sec = NULL;
		  sym_value = 0;
		  destination = 0;
		  hash = NULL;
		  if (r_indx < symtab_hdr->sh_info)
		    {
		      /* It's a local symbol.  */
		      Elf_Internal_Sym *sym;
		      Elf_Internal_Shdr *hdr;

		      sym = local_syms + r_indx;
		      hdr = elf_elfsections (input_bfd)[sym->st_shndx];
		      sym_sec = hdr->bfd_section;
		      if (ELF_ST_TYPE (sym->st_info) != STT_SECTION)
			sym_value = sym->st_value;
		      destination = (sym_value + irela->r_addend
				     + sym_sec->output_offset
				     + sym_sec->output_section->vma);
		    }
		  else
		    {
		      /* It's an external symbol.  */
		      int e_indx;

		      e_indx = r_indx - symtab_hdr->sh_info;
		      hash = ((struct elf32_hppa_link_hash_entry *)
			      elf_sym_hashes (input_bfd)[e_indx]);

		      while (hash->elf.root.type == bfd_link_hash_indirect
			     || hash->elf.root.type == bfd_link_hash_warning)
			hash = ((struct elf32_hppa_link_hash_entry *)
				hash->elf.root.u.i.link);

		      if (hash->elf.root.type == bfd_link_hash_defined
			  || hash->elf.root.type == bfd_link_hash_defweak)
			{
			  sym_sec = hash->elf.root.u.def.section;
			  sym_value = hash->elf.root.u.def.value;
			  if (sym_sec->output_section != NULL)
			    destination = (sym_value + irela->r_addend
					   + sym_sec->output_offset
					   + sym_sec->output_section->vma);
			}
		      else if (hash->elf.root.type == bfd_link_hash_undefweak)
			{
			  if (! info->shared)
			    continue;
			}
		      else if (hash->elf.root.type == bfd_link_hash_undefined)
			{
			  if (! (info->unresolved_syms_in_objects == RM_IGNORE
				 && (ELF_ST_VISIBILITY (hash->elf.other)
				     == STV_DEFAULT)
				 && hash->elf.type != STT_PARISC_MILLI))
			    continue;
			}
		      else
			{
			  bfd_set_error (bfd_error_bad_value);
			  goto error_ret_free_internal;
			}
		    }

		  /* Determine what (if any) linker stub is needed.  */
		  stub_type = hppa_type_of_stub (section, irela, hash,
						 destination, info);
		  if (stub_type == hppa_stub_none)
		    continue;

		  /* Support for grouping stub sections.  */
		  id_sec = htab->stub_group[section->id].link_sec;

		  /* Get the name of this stub.  */
		  stub_name = hppa_stub_name (id_sec, sym_sec, hash, irela);
		  if (!stub_name)
		    goto error_ret_free_internal;

		  stub_entry = hppa_stub_hash_lookup (&htab->stub_hash_table,
						      stub_name,
						      FALSE, FALSE);
		  if (stub_entry != NULL)
		    {
		      /* The proper stub has already been created.  */
		      free (stub_name);
		      continue;
		    }

		  stub_entry = hppa_add_stub (stub_name, section, htab);
		  if (stub_entry == NULL)
		    {
		      free (stub_name);
		      goto error_ret_free_internal;
		    }

		  stub_entry->target_value = sym_value;
		  stub_entry->target_section = sym_sec;
		  stub_entry->stub_type = stub_type;
		  if (info->shared)
		    {
		      if (stub_type == hppa_stub_import)
			stub_entry->stub_type = hppa_stub_import_shared;
		      else if (stub_type == hppa_stub_long_branch)
			stub_entry->stub_type = hppa_stub_long_branch_shared;
		    }
		  stub_entry->h = hash;
		  stub_changed = TRUE;
		}

	      /* We're done with the internal relocs, free them.  */
	      if (elf_section_data (section)->relocs == NULL)
		free (internal_relocs);
	    }
	}

      if (!stub_changed)
	break;

      /* OK, we've added some stubs.  Find out the new size of the
	 stub sections.  */
      for (stub_sec = htab->stub_bfd->sections;
	   stub_sec != NULL;
	   stub_sec = stub_sec->next)
	stub_sec->size = 0;

      bfd_hash_traverse (&htab->stub_hash_table, hppa_size_one_stub, htab);

      /* Ask the linker to do its stuff.  */
      (*htab->layout_sections_again) ();
      stub_changed = FALSE;
    }

  free (htab->all_local_syms);
  return TRUE;

 error_ret_free_local:
  free (htab->all_local_syms);
  return FALSE;
}

/* For a final link, this function is called after we have sized the
   stubs to provide a value for __gp.  */

bfd_boolean
elf32_hppa_set_gp (bfd *abfd, struct bfd_link_info *info)
{
  struct bfd_link_hash_entry *h;
  asection *sec = NULL;
  bfd_vma gp_val = 0;
  struct elf32_hppa_link_hash_table *htab;

  htab = hppa_link_hash_table (info);
  h = bfd_link_hash_lookup (&htab->elf.root, "$global$", FALSE, FALSE, FALSE);

  if (h != NULL
      && (h->type == bfd_link_hash_defined
	  || h->type == bfd_link_hash_defweak))
    {
      gp_val = h->u.def.value;
      sec = h->u.def.section;
    }
  else
    {
      asection *splt = bfd_get_section_by_name (abfd, ".plt");
      asection *sgot = bfd_get_section_by_name (abfd, ".got");

      /* Choose to point our LTP at, in this order, one of .plt, .got,
	 or .data, if these sections exist.  In the case of choosing
	 .plt try to make the LTP ideal for addressing anywhere in the
	 .plt or .got with a 14 bit signed offset.  Typically, the end
	 of the .plt is the start of the .got, so choose .plt + 0x2000
	 if either the .plt or .got is larger than 0x2000.  If both
	 the .plt and .got are smaller than 0x2000, choose the end of
	 the .plt section.  */
      sec = strcmp (bfd_get_target (abfd), "elf32-hppa-netbsd") == 0
	  ? NULL : splt;
      if (sec != NULL)
	{
	  gp_val = sec->size;
	  if (gp_val > 0x2000 || (sgot && sgot->size > 0x2000))
	    {
	      gp_val = 0x2000;
	    }
	}
      else
	{
	  sec = sgot;
	  if (sec != NULL)
	    {
	      if (strcmp (bfd_get_target (abfd), "elf32-hppa-netbsd") != 0)
		{
	          /* We know we don't have a .plt.  If .got is large,
		     offset our LTP.  */
	          if (sec->size > 0x2000)
		    gp_val = 0x2000;
		}
	    }
	  else
	    {
	      /* No .plt or .got.  Who cares what the LTP is?  */
	      sec = bfd_get_section_by_name (abfd, ".data");
	    }
	}

      if (h != NULL)
	{
	  h->type = bfd_link_hash_defined;
	  h->u.def.value = gp_val;
	  if (sec != NULL)
	    h->u.def.section = sec;
	  else
	    h->u.def.section = bfd_abs_section_ptr;
	}
    }

  if (sec != NULL && sec->output_section != NULL)
    gp_val += sec->output_section->vma + sec->output_offset;

  elf_gp (abfd) = gp_val;
  return TRUE;
}

/* Build all the stubs associated with the current output file.  The
   stubs are kept in a hash table attached to the main linker hash
   table.  We also set up the .plt entries for statically linked PIC
   functions here.  This function is called via hppaelf_finish in the
   linker.  */

bfd_boolean
elf32_hppa_build_stubs (struct bfd_link_info *info)
{
  asection *stub_sec;
  struct bfd_hash_table *table;
  struct elf32_hppa_link_hash_table *htab;

  htab = hppa_link_hash_table (info);

  for (stub_sec = htab->stub_bfd->sections;
       stub_sec != NULL;
       stub_sec = stub_sec->next)
    {
      bfd_size_type size;

      /* Allocate memory to hold the linker stubs.  */
      size = stub_sec->size;
      stub_sec->contents = bfd_zalloc (htab->stub_bfd, size);
      if (stub_sec->contents == NULL && size != 0)
	return FALSE;
      stub_sec->size = 0;
    }

  /* Build the stubs as directed by the stub hash table.  */
  table = &htab->stub_hash_table;
  bfd_hash_traverse (table, hppa_build_one_stub, info);

  return TRUE;
}

/* Perform a final link.  */

static bfd_boolean
elf32_hppa_final_link (bfd *abfd, struct bfd_link_info *info)
{
  /* Invoke the regular ELF linker to do all the work.  */
  if (!bfd_elf_final_link (abfd, info))
    return FALSE;

  /* If we're producing a final executable, sort the contents of the
     unwind section.  */
  return elf_hppa_sort_unwind (abfd);
}

/* Record the lowest address for the data and text segments.  */

static void
hppa_record_segment_addr (bfd *abfd ATTRIBUTE_UNUSED,
			  asection *section,
			  void *data)
{
  struct elf32_hppa_link_hash_table *htab;

  htab = (struct elf32_hppa_link_hash_table *) data;

  if ((section->flags & (SEC_ALLOC | SEC_LOAD)) == (SEC_ALLOC | SEC_LOAD))
    {
      bfd_vma value = section->vma - section->filepos;

      if ((section->flags & SEC_READONLY) != 0)
	{
	  if (value < htab->text_segment_base)
	    htab->text_segment_base = value;
	}
      else
	{
	  if (value < htab->data_segment_base)
	    htab->data_segment_base = value;
	}
    }
}

/* Perform a relocation as part of a final link.  */

static bfd_reloc_status_type
final_link_relocate (asection *input_section,
		     bfd_byte *contents,
		     const Elf_Internal_Rela *rel,
		     bfd_vma value,
		     struct elf32_hppa_link_hash_table *htab,
		     asection *sym_sec,
		     struct elf32_hppa_link_hash_entry *h,
		     struct bfd_link_info *info)
{
  int insn;
  unsigned int r_type = ELF32_R_TYPE (rel->r_info);
  unsigned int orig_r_type = r_type;
  reloc_howto_type *howto = elf_hppa_howto_table + r_type;
  int r_format = howto->bitsize;
  enum hppa_reloc_field_selector_type_alt r_field;
  bfd *input_bfd = input_section->owner;
  bfd_vma offset = rel->r_offset;
  bfd_vma max_branch_offset = 0;
  bfd_byte *hit_data = contents + offset;
  bfd_signed_vma addend = rel->r_addend;
  bfd_vma location;
  struct elf32_hppa_stub_hash_entry *stub_entry = NULL;
  int val;

  if (r_type == R_PARISC_NONE)
    return bfd_reloc_ok;

  insn = bfd_get_32 (input_bfd, hit_data);

  /* Find out where we are and where we're going.  */
  location = (offset +
	      input_section->output_offset +
	      input_section->output_section->vma);

  /* If we are not building a shared library, convert DLTIND relocs to
     DPREL relocs.  */
  if (!info->shared)
    {
      switch (r_type)
	{
	  case R_PARISC_DLTIND21L:
	    r_type = R_PARISC_DPREL21L;
	    break;

	  case R_PARISC_DLTIND14R:
	    r_type = R_PARISC_DPREL14R;
	    break;

	  case R_PARISC_DLTIND14F:
	    r_type = R_PARISC_DPREL14F;
	    break;
	}
    }

  switch (r_type)
    {
    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL22F:
      /* If this call should go via the plt, find the import stub in
	 the stub hash.  */
      if (sym_sec == NULL
	  || sym_sec->output_section == NULL
	  || (h != NULL
	      && h->elf.plt.offset != (bfd_vma) -1
	      && h->elf.dynindx != -1
	      && !h->plabel
	      && (info->shared
		  || !h->elf.def_regular
		  || h->elf.root.type == bfd_link_hash_defweak)))
	{
	  stub_entry = hppa_get_stub_entry (input_section, sym_sec,
					    h, rel, htab);
	  if (stub_entry != NULL)
	    {
	      value = (stub_entry->stub_offset
		       + stub_entry->stub_sec->output_offset
		       + stub_entry->stub_sec->output_section->vma);
	      addend = 0;
	    }
	  else if (sym_sec == NULL && h != NULL
		   && h->elf.root.type == bfd_link_hash_undefweak)
	    {
	      /* It's OK if undefined weak.  Calls to undefined weak
		 symbols behave as if the "called" function
		 immediately returns.  We can thus call to a weak
		 function without first checking whether the function
		 is defined.  */
	      value = location;
	      addend = 8;
	    }
	  else
	    return bfd_reloc_undefined;
	}
      /* Fall thru.  */

    case R_PARISC_PCREL21L:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL17R:
    case R_PARISC_PCREL14R:
    case R_PARISC_PCREL14F:
    case R_PARISC_PCREL32:
      /* Make it a pc relative offset.  */
      value -= location;
      addend -= 8;
      break;

    case R_PARISC_DPREL21L:
    case R_PARISC_DPREL14R:
    case R_PARISC_DPREL14F:
      /* Convert instructions that use the linkage table pointer (r19) to
	 instructions that use the global data pointer (dp).  This is the
	 most efficient way of using PIC code in an incomplete executable,
	 but the user must follow the standard runtime conventions for
	 accessing data for this to work.  */
      if (orig_r_type == R_PARISC_DLTIND21L)
	{
	  /* Convert addil instructions if the original reloc was a
	     DLTIND21L.  GCC sometimes uses a register other than r19 for
	     the operation, so we must convert any addil instruction
	     that uses this relocation.  */
	  if ((insn & 0xfc000000) == ((int) OP_ADDIL << 26))
	    insn = ADDIL_DP;
	  else
	    /* We must have a ldil instruction.  It's too hard to find
	       and convert the associated add instruction, so issue an
	       error.  */
	    (*_bfd_error_handler)
	      (_("%B(%A+0x%lx): %s fixup for insn 0x%x is not supported in a non-shared link"),
	       input_bfd,
	       input_section,
	       (long) rel->r_offset,
	       howto->name,
	       insn);
	}
      else if (orig_r_type == R_PARISC_DLTIND14F)
	{
	  /* This must be a format 1 load/store.  Change the base
	     register to dp.  */
	  insn = (insn & 0xfc1ffff) | (27 << 21);
	}

    /* For all the DP relative relocations, we need to examine the symbol's
       section.  If it has no section or if it's a code section, then
       "data pointer relative" makes no sense.  In that case we don't
       adjust the "value", and for 21 bit addil instructions, we change the
       source addend register from %dp to %r0.  This situation commonly
       arises for undefined weak symbols and when a variable's "constness"
       is declared differently from the way the variable is defined.  For
       instance: "extern int foo" with foo defined as "const int foo".  */
      if (sym_sec == NULL || (sym_sec->flags & SEC_CODE) != 0)
	{
	  if ((insn & ((0x3f << 26) | (0x1f << 21)))
	      == (((int) OP_ADDIL << 26) | (27 << 21)))
	    {
	      insn &= ~ (0x1f << 21);
	    }
	  /* Now try to make things easy for the dynamic linker.  */

	  break;
	}
      /* Fall thru.  */

    case R_PARISC_DLTIND21L:
    case R_PARISC_DLTIND14R:
    case R_PARISC_DLTIND14F:
      value -= elf_gp (input_section->output_section->owner);
      break;

    case R_PARISC_SEGREL32:
      if ((sym_sec->flags & SEC_CODE) != 0)
	value -= htab->text_segment_base;
      else
	value -= htab->data_segment_base;
      break;

    default:
      break;
    }

  switch (r_type)
    {
    case R_PARISC_DIR32:
    case R_PARISC_DIR14F:
    case R_PARISC_DIR17F:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL14F:
    case R_PARISC_PCREL32:
    case R_PARISC_DPREL14F:
    case R_PARISC_PLABEL32:
    case R_PARISC_DLTIND14F:
    case R_PARISC_SEGBASE:
    case R_PARISC_SEGREL32:
      r_field = e_fsel;
      break;

    case R_PARISC_DLTIND21L:
    case R_PARISC_PCREL21L:
    case R_PARISC_PLABEL21L:
      r_field = e_lsel;
      break;

    case R_PARISC_DIR21L:
    case R_PARISC_DPREL21L:
      r_field = e_lrsel;
      break;

    case R_PARISC_PCREL17R:
    case R_PARISC_PCREL14R:
    case R_PARISC_PLABEL14R:
    case R_PARISC_DLTIND14R:
      r_field = e_rsel;
      break;

    case R_PARISC_DIR17R:
    case R_PARISC_DIR14R:
    case R_PARISC_DPREL14R:
      r_field = e_rrsel;
      break;

    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL22F:
      r_field = e_fsel;

      if (r_type == (unsigned int) R_PARISC_PCREL17F)
	{
	  max_branch_offset = (1 << (17-1)) << 2;
	}
      else if (r_type == (unsigned int) R_PARISC_PCREL12F)
	{
	  max_branch_offset = (1 << (12-1)) << 2;
	}
      else
	{
	  max_branch_offset = (1 << (22-1)) << 2;
	}

      /* sym_sec is NULL on undefined weak syms or when shared on
	 undefined syms.  We've already checked for a stub for the
	 shared undefined case.  */
      if (sym_sec == NULL)
	break;

      /* If the branch is out of reach, then redirect the
	 call to the local stub for this function.  */
      if (value + addend + max_branch_offset >= 2*max_branch_offset)
	{
	  stub_entry = hppa_get_stub_entry (input_section, sym_sec,
					    h, rel, htab);
	  if (stub_entry == NULL)
	    return bfd_reloc_undefined;

	  /* Munge up the value and addend so that we call the stub
	     rather than the procedure directly.  */
	  value = (stub_entry->stub_offset
		   + stub_entry->stub_sec->output_offset
		   + stub_entry->stub_sec->output_section->vma
		   - location);
	  addend = -8;
	}
      break;

    /* Something we don't know how to handle.  */
    default:
      return bfd_reloc_notsupported;
    }

  /* Make sure we can reach the stub.  */
  if (max_branch_offset != 0
      && value + addend + max_branch_offset >= 2*max_branch_offset)
    {
      (*_bfd_error_handler)
	(_("%B(%A+0x%lx): cannot reach %s, recompile with -ffunction-sections"),
	 input_bfd,
	 input_section,
	 (long) rel->r_offset,
	 stub_entry->root.string);
      bfd_set_error (bfd_error_bad_value);
      return bfd_reloc_notsupported;
    }

  val = hppa_field_adjust (value, addend, r_field);

  switch (r_type)
    {
    case R_PARISC_PCREL12F:
    case R_PARISC_PCREL17C:
    case R_PARISC_PCREL17F:
    case R_PARISC_PCREL17R:
    case R_PARISC_PCREL22F:
    case R_PARISC_DIR17F:
    case R_PARISC_DIR17R:
      /* This is a branch.  Divide the offset by four.
	 Note that we need to decide whether it's a branch or
	 otherwise by inspecting the reloc.  Inspecting insn won't
	 work as insn might be from a .word directive.  */
      val >>= 2;
      break;

    default:
      break;
    }

  insn = hppa_rebuild_insn (insn, val, r_format);

  /* Update the instruction word.  */
  bfd_put_32 (input_bfd, (bfd_vma) insn, hit_data);
  return bfd_reloc_ok;
}

/* Relocate an HPPA ELF section.  */

static bfd_boolean
elf32_hppa_relocate_section (bfd *output_bfd,
			     struct bfd_link_info *info,
			     bfd *input_bfd,
			     asection *input_section,
			     bfd_byte *contents,
			     Elf_Internal_Rela *relocs,
			     Elf_Internal_Sym *local_syms,
			     asection **local_sections)
{
  bfd_vma *local_got_offsets;
  struct elf32_hppa_link_hash_table *htab;
  Elf_Internal_Shdr *symtab_hdr;
  Elf_Internal_Rela *rel;
  Elf_Internal_Rela *relend;

  if (info->relocatable)
    return TRUE;

  symtab_hdr = &elf_tdata (input_bfd)->symtab_hdr;

  htab = hppa_link_hash_table (info);
  local_got_offsets = elf_local_got_offsets (input_bfd);

  rel = relocs;
  relend = relocs + input_section->reloc_count;
  for (; rel < relend; rel++)
    {
      unsigned int r_type;
      reloc_howto_type *howto;
      unsigned int r_symndx;
      struct elf32_hppa_link_hash_entry *h;
      Elf_Internal_Sym *sym;
      asection *sym_sec;
      bfd_vma relocation;
      bfd_reloc_status_type r;
      const char *sym_name;
      bfd_boolean plabel;
      bfd_boolean warned_undef;

      r_type = ELF32_R_TYPE (rel->r_info);
      if (r_type >= (unsigned int) R_PARISC_UNIMPLEMENTED)
	{
	  bfd_set_error (bfd_error_bad_value);
	  return FALSE;
	}
      if (r_type == (unsigned int) R_PARISC_GNU_VTENTRY
	  || r_type == (unsigned int) R_PARISC_GNU_VTINHERIT)
	continue;

      /* This is a final link.  */
      r_symndx = ELF32_R_SYM (rel->r_info);
      h = NULL;
      sym = NULL;
      sym_sec = NULL;
      warned_undef = FALSE;
      if (r_symndx < symtab_hdr->sh_info)
	{
	  /* This is a local symbol, h defaults to NULL.  */
	  sym = local_syms + r_symndx;
	  sym_sec = local_sections[r_symndx];
	  relocation = _bfd_elf_rela_local_sym (output_bfd, sym, &sym_sec, rel);
	}
      else
	{
	  struct elf_link_hash_entry *hh;
	  bfd_boolean unresolved_reloc;
	  struct elf_link_hash_entry **sym_hashes = elf_sym_hashes (input_bfd);

	  RELOC_FOR_GLOBAL_SYMBOL (info, input_bfd, input_section, rel,
				   r_symndx, symtab_hdr, sym_hashes,
				   hh, sym_sec, relocation,
				   unresolved_reloc, warned_undef);

	  if (relocation == 0
	      && hh->root.type != bfd_link_hash_defined
	      && hh->root.type != bfd_link_hash_defweak
	      && hh->root.type != bfd_link_hash_undefweak)
	    {
	      if (info->unresolved_syms_in_objects == RM_IGNORE
		  && ELF_ST_VISIBILITY (hh->other) == STV_DEFAULT
		  && hh->type == STT_PARISC_MILLI)
		{
		  if (! info->callbacks->undefined_symbol
		      (info, hh->root.root.string, input_bfd,
		       input_section, rel->r_offset, FALSE))
		    return FALSE;
		  warned_undef = TRUE;
		}
	    }
	  h = (struct elf32_hppa_link_hash_entry *) hh;
	}

      /* Do any required modifications to the relocation value, and
	 determine what types of dynamic info we need to output, if
	 any.  */
      plabel = 0;
      switch (r_type)
	{
	case R_PARISC_DLTIND14F:
	case R_PARISC_DLTIND14R:
	case R_PARISC_DLTIND21L:
	  {
	    bfd_vma off;
	    bfd_boolean do_got = 0;

	    /* Relocation is to the entry for this symbol in the
	       global offset table.  */
	    if (h != NULL)
	      {
		bfd_boolean dyn;

		off = h->elf.got.offset;
		dyn = htab->elf.dynamic_sections_created;
		if (! WILL_CALL_FINISH_DYNAMIC_SYMBOL (dyn, info->shared,
						       &h->elf))
		  {
		    /* If we aren't going to call finish_dynamic_symbol,
		       then we need to handle initialisation of the .got
		       entry and create needed relocs here.  Since the
		       offset must always be a multiple of 4, we use the
		       least significant bit to record whether we have
		       initialised it already.  */
		    if ((off & 1) != 0)
		      off &= ~1;
		    else
		      {
			h->elf.got.offset |= 1;
			do_got = 1;
		      }
		  }
	      }
	    else
	      {
		/* Local symbol case.  */
		if (local_got_offsets == NULL)
		  abort ();

		off = local_got_offsets[r_symndx];

		/* The offset must always be a multiple of 4.  We use
		   the least significant bit to record whether we have
		   already generated the necessary reloc.  */
		if ((off & 1) != 0)
		  off &= ~1;
		else
		  {
		    local_got_offsets[r_symndx] |= 1;
		    do_got = 1;
		  }
	      }

	    if (do_got)
	      {
		if (info->shared)
		  {
		    /* Output a dynamic relocation for this GOT entry.
		       In this case it is relative to the base of the
		       object because the symbol index is zero.  */
		    Elf_Internal_Rela outrel;
		    bfd_byte *loc;
		    asection *s = htab->srelgot;

		    outrel.r_offset = (off
				       + htab->sgot->output_offset
				       + htab->sgot->output_section->vma);
		    outrel.r_info = ELF32_R_INFO (0, R_PARISC_DIR32);
		    outrel.r_addend = relocation;
		    loc = s->contents;
		    loc += s->reloc_count++ * sizeof (Elf32_External_Rela);
		    bfd_elf32_swap_reloca_out (output_bfd, &outrel, loc);
		  }
		else
		  bfd_put_32 (output_bfd, relocation,
			      htab->sgot->contents + off);
	      }

	    if (off >= (bfd_vma) -2)
	      abort ();

	    /* Add the base of the GOT to the relocation value.  */
	    relocation = (off
			  + htab->sgot->output_offset
			  + htab->sgot->output_section->vma);
	  }
	  break;

	case R_PARISC_SEGREL32:
	  /* If this is the first SEGREL relocation, then initialize
	     the segment base values.  */
	  if (htab->text_segment_base == (bfd_vma) -1)
	    bfd_map_over_sections (output_bfd, hppa_record_segment_addr, htab);
	  break;

	case R_PARISC_PLABEL14R:
	case R_PARISC_PLABEL21L:
	case R_PARISC_PLABEL32:
	  if (htab->elf.dynamic_sections_created)
	    {
	      bfd_vma off;
	      bfd_boolean do_plt = 0;

	      /* If we have a global symbol with a PLT slot, then
		 redirect this relocation to it.  */
	      if (h != NULL)
		{
		  off = h->elf.plt.offset;
		  if (! WILL_CALL_FINISH_DYNAMIC_SYMBOL (1, info->shared,
							 &h->elf))
		    {
		      /* In a non-shared link, adjust_dynamic_symbols
			 isn't called for symbols forced local.  We
			 need to write out the plt entry here.  */
		      if ((off & 1) != 0)
			off &= ~1;
		      else
			{
			  h->elf.plt.offset |= 1;
			  do_plt = 1;
			}
		    }
		}
	      else
		{
		  bfd_vma *local_plt_offsets;

		  if (local_got_offsets == NULL)
		    abort ();

		  local_plt_offsets = local_got_offsets + symtab_hdr->sh_info;
		  off = local_plt_offsets[r_symndx];

		  /* As for the local .got entry case, we use the last
		     bit to record whether we've already initialised
		     this local .plt entry.  */
		  if ((off & 1) != 0)
		    off &= ~1;
		  else
		    {
		      local_plt_offsets[r_symndx] |= 1;
		      do_plt = 1;
		    }
		}

	      if (do_plt)
		{
		  if (info->shared)
		    {
		      /* Output a dynamic IPLT relocation for this
			 PLT entry.  */
		      Elf_Internal_Rela outrel;
		      bfd_byte *loc;
		      asection *s = htab->srelplt;

		      outrel.r_offset = (off
					 + htab->splt->output_offset
					 + htab->splt->output_section->vma);
		      outrel.r_info = ELF32_R_INFO (0, R_PARISC_IPLT);
		      outrel.r_addend = relocation;
		      loc = s->contents;
		      loc += s->reloc_count++ * sizeof (Elf32_External_Rela);
		      bfd_elf32_swap_reloca_out (output_bfd, &outrel, loc);
		    }
		  else
		    {
		      bfd_put_32 (output_bfd,
				  relocation,
				  htab->splt->contents + off);
		      bfd_put_32 (output_bfd,
				  elf_gp (htab->splt->output_section->owner),
				  htab->splt->contents + off + 4);
		    }
		}

	      if (off >= (bfd_vma) -2)
		abort ();

	      /* PLABELs contain function pointers.  Relocation is to
		 the entry for the function in the .plt.  The magic +2
		 offset signals to $$dyncall that the function pointer
		 is in the .plt and thus has a gp pointer too.
		 Exception:  Undefined PLABELs should have a value of
		 zero.  */
	      if (h == NULL
		  || (h->elf.root.type != bfd_link_hash_undefweak
		      && h->elf.root.type != bfd_link_hash_undefined))
		{
		  relocation = (off
				+ htab->splt->output_offset
				+ htab->splt->output_section->vma
				+ 2);
		}
	      plabel = 1;
	    }
	  /* Fall through and possibly emit a dynamic relocation.  */

	case R_PARISC_DIR17F:
	case R_PARISC_DIR17R:
	case R_PARISC_DIR14F:
	case R_PARISC_DIR14R:
	case R_PARISC_DIR21L:
	case R_PARISC_DPREL14F:
	case R_PARISC_DPREL14R:
	case R_PARISC_DPREL21L:
	case R_PARISC_DIR32:
	  /* r_symndx will be zero only for relocs against symbols
	     from removed linkonce sections, or sections discarded by
	     a linker script.  */
	  if (r_symndx == 0
	      || (input_section->flags & SEC_ALLOC) == 0)
	    break;

	  /* The reloc types handled here and this conditional
	     expression must match the code in ..check_relocs and
	     allocate_dynrelocs.  ie. We need exactly the same condition
	     as in ..check_relocs, with some extra conditions (dynindx
	     test in this case) to cater for relocs removed by
	     allocate_dynrelocs.  If you squint, the non-shared test
	     here does indeed match the one in ..check_relocs, the
	     difference being that here we test DEF_DYNAMIC as well as
	     !DEF_REGULAR.  All common syms end up with !DEF_REGULAR,
	     which is why we can't use just that test here.
	     Conversely, DEF_DYNAMIC can't be used in check_relocs as
	     there all files have not been loaded.  */
	  if ((info->shared
	       && (h == NULL
		   || ELF_ST_VISIBILITY (h->elf.other) == STV_DEFAULT
		   || h->elf.root.type != bfd_link_hash_undefweak)
	       && (IS_ABSOLUTE_RELOC (r_type)
		   || !SYMBOL_CALLS_LOCAL (info, &h->elf)))
	      || (!info->shared
		  && h != NULL
		  && h->elf.dynindx != -1
		  && !h->elf.non_got_ref
		  && ((ELIMINATE_COPY_RELOCS
		       && h->elf.def_dynamic
		       && !h->elf.def_regular)
		      || h->elf.root.type == bfd_link_hash_undefweak
		      || h->elf.root.type == bfd_link_hash_undefined)))
	    {
	      Elf_Internal_Rela outrel;
	      bfd_boolean skip;
	      asection *sreloc;
	      bfd_byte *loc;

	      /* When generating a shared object, these relocations
		 are copied into the output file to be resolved at run
		 time.  */

	      outrel.r_addend = rel->r_addend;
	      outrel.r_offset =
		_bfd_elf_section_offset (output_bfd, info, input_section,
					 rel->r_offset);
	      skip = (outrel.r_offset == (bfd_vma) -1
		      || outrel.r_offset == (bfd_vma) -2);
	      outrel.r_offset += (input_section->output_offset
				  + input_section->output_section->vma);

	      if (skip)
		{
		  memset (&outrel, 0, sizeof (outrel));
		}
	      else if (h != NULL
		       && h->elf.dynindx != -1
		       && (plabel
			   || !IS_ABSOLUTE_RELOC (r_type)
			   || !info->shared
			   || !info->symbolic
			   || !h->elf.def_regular))
		{
		  outrel.r_info = ELF32_R_INFO (h->elf.dynindx, r_type);
		}
	      else /* It's a local symbol, or one marked to become local.  */
		{
		  int indx = 0;

		  /* Add the absolute offset of the symbol.  */
		  outrel.r_addend += relocation;

		  /* Global plabels need to be processed by the
		     dynamic linker so that functions have at most one
		     fptr.  For this reason, we need to differentiate
		     between global and local plabels, which we do by
		     providing the function symbol for a global plabel
		     reloc, and no symbol for local plabels.  */
		  if (! plabel
		      && sym_sec != NULL
		      && sym_sec->output_section != NULL
		      && ! bfd_is_abs_section (sym_sec))
		    {
		      /* Skip this relocation if the output section has
			 been discarded.  */
		      if (bfd_is_abs_section (sym_sec->output_section))
			break;

		      indx = elf_section_data (sym_sec->output_section)->dynindx;
		      /* We are turning this relocation into one
			 against a section symbol, so subtract out the
			 output section's address but not the offset
			 of the input section in the output section.  */
		      outrel.r_addend -= sym_sec->output_section->vma;
		    }

		  outrel.r_info = ELF32_R_INFO (indx, r_type);
		}
	      sreloc = elf_section_data (input_section)->sreloc;
	      if (sreloc == NULL)
		abort ();

	      loc = sreloc->contents;
	      loc += sreloc->reloc_count++ * sizeof (Elf32_External_Rela);
	      bfd_elf32_swap_reloca_out (output_bfd, &outrel, loc);
	    }
	  break;

	default:
	  break;
	}

      r = final_link_relocate (input_section, contents, rel, relocation,
			       htab, sym_sec, h, info);

      if (r == bfd_reloc_ok)
	continue;

      if (h != NULL)
	sym_name = h->elf.root.root.string;
      else
	{
	  sym_name = bfd_elf_string_from_elf_section (input_bfd,
						      symtab_hdr->sh_link,
						      sym->st_name);
	  if (sym_name == NULL)
	    return FALSE;
	  if (*sym_name == '\0')
	    sym_name = bfd_section_name (input_bfd, sym_sec);
	}

      howto = elf_hppa_howto_table + r_type;

      if (r == bfd_reloc_undefined || r == bfd_reloc_notsupported)
	{
	  if (r == bfd_reloc_notsupported || !warned_undef)
	    {
	      (*_bfd_error_handler)
		(_("%B(%A+0x%lx): cannot handle %s for %s"),
		 input_bfd,
		 input_section,
		 (long) rel->r_offset,
		 howto->name,
		 sym_name);
	      bfd_set_error (bfd_error_bad_value);
	      return FALSE;
	    }
	}
      else
	{
	  if (!((*info->callbacks->reloc_overflow)
		(info, (h ? &h->elf.root : NULL), sym_name, howto->name,
		 (bfd_vma) 0, input_bfd, input_section, rel->r_offset)))
	    return FALSE;
	}
    }

  return TRUE;
}

/* Finish up dynamic symbol handling.  We set the contents of various
   dynamic sections here.  */

static bfd_boolean
elf32_hppa_finish_dynamic_symbol (bfd *output_bfd,
				  struct bfd_link_info *info,
				  struct elf_link_hash_entry *h,
				  Elf_Internal_Sym *sym)
{
  struct elf32_hppa_link_hash_table *htab;
  Elf_Internal_Rela rel;
  bfd_byte *loc;

  htab = hppa_link_hash_table (info);

  if (h->plt.offset != (bfd_vma) -1)
    {
      bfd_vma value;

      if (h->plt.offset & 1)
	abort ();

      /* This symbol has an entry in the procedure linkage table.  Set
	 it up.

	 The format of a plt entry is
	 <funcaddr>
	 <__gp>
      */
      value = 0;
      if (h->root.type == bfd_link_hash_defined
	  || h->root.type == bfd_link_hash_defweak)
	{
	  value = h->root.u.def.value;
	  if (h->root.u.def.section->output_section != NULL)
	    value += (h->root.u.def.section->output_offset
		      + h->root.u.def.section->output_section->vma);
	}

      /* Create a dynamic IPLT relocation for this entry.  */
      rel.r_offset = (h->plt.offset
		      + htab->splt->output_offset
		      + htab->splt->output_section->vma);
      if (h->dynindx != -1)
	{
	  rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_IPLT);
	  rel.r_addend = 0;
	}
      else
	{
	  /* This symbol has been marked to become local, and is
	     used by a plabel so must be kept in the .plt.  */
	  rel.r_info = ELF32_R_INFO (0, R_PARISC_IPLT);
	  rel.r_addend = value;
	}

      loc = htab->srelplt->contents;
      loc += htab->srelplt->reloc_count++ * sizeof (Elf32_External_Rela);
      bfd_elf32_swap_reloca_out (htab->splt->output_section->owner, &rel, loc);

      if (!h->def_regular)
	{
	  /* Mark the symbol as undefined, rather than as defined in
	     the .plt section.  Leave the value alone.  */
	  sym->st_shndx = SHN_UNDEF;
	}
    }

  if (h->got.offset != (bfd_vma) -1)
    {
      /* This symbol has an entry in the global offset table.  Set it
	 up.  */

      rel.r_offset = ((h->got.offset &~ (bfd_vma) 1)
		      + htab->sgot->output_offset
		      + htab->sgot->output_section->vma);

      /* If this is a -Bsymbolic link and the symbol is defined
	 locally or was forced to be local because of a version file,
	 we just want to emit a RELATIVE reloc.  The entry in the
	 global offset table will already have been initialized in the
	 relocate_section function.  */
      if (info->shared
	  && (info->symbolic || h->dynindx == -1)
	  && h->def_regular)
	{
	  rel.r_info = ELF32_R_INFO (0, R_PARISC_DIR32);
	  rel.r_addend = (h->root.u.def.value
			  + h->root.u.def.section->output_offset
			  + h->root.u.def.section->output_section->vma);
	}
      else
	{
	  if ((h->got.offset & 1) != 0)
	    abort ();
	  bfd_put_32 (output_bfd, 0, htab->sgot->contents + h->got.offset);
	  rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_DIR32);
	  rel.r_addend = 0;
	}

      loc = htab->srelgot->contents;
      loc += htab->srelgot->reloc_count++ * sizeof (Elf32_External_Rela);
      bfd_elf32_swap_reloca_out (output_bfd, &rel, loc);
    }

  if (h->needs_copy)
    {
      asection *s;

      /* This symbol needs a copy reloc.  Set it up.  */

      if (! (h->dynindx != -1
	     && (h->root.type == bfd_link_hash_defined
		 || h->root.type == bfd_link_hash_defweak)))
	abort ();

      s = htab->srelbss;

      rel.r_offset = (h->root.u.def.value
		      + h->root.u.def.section->output_offset
		      + h->root.u.def.section->output_section->vma);
      rel.r_addend = 0;
      rel.r_info = ELF32_R_INFO (h->dynindx, R_PARISC_COPY);
      loc = s->contents + s->reloc_count++ * sizeof (Elf32_External_Rela);
      bfd_elf32_swap_reloca_out (output_bfd, &rel, loc);
    }

  /* Mark _DYNAMIC and _GLOBAL_OFFSET_TABLE_ as absolute.  */
  if (h->root.root.string[0] == '_'
      && (strcmp (h->root.root.string, "_DYNAMIC") == 0
	  || strcmp (h->root.root.string, "_GLOBAL_OFFSET_TABLE_") == 0))
    {
      sym->st_shndx = SHN_ABS;
    }

  return TRUE;
}

/* Used to decide how to sort relocs in an optimal manner for the
   dynamic linker, before writing them out.  */

static enum elf_reloc_type_class
elf32_hppa_reloc_type_class (const Elf_Internal_Rela *rela)
{
  if (ELF32_R_SYM (rela->r_info) == 0)
    return reloc_class_relative;

  switch ((int) ELF32_R_TYPE (rela->r_info))
    {
    case R_PARISC_IPLT:
      return reloc_class_plt;
    case R_PARISC_COPY:
      return reloc_class_copy;
    default:
      return reloc_class_normal;
    }
}

/* Finish up the dynamic sections.  */

static bfd_boolean
elf32_hppa_finish_dynamic_sections (bfd *output_bfd,
				    struct bfd_link_info *info)
{
  bfd *dynobj;
  struct elf32_hppa_link_hash_table *htab;
  asection *sdyn;

  htab = hppa_link_hash_table (info);
  dynobj = htab->elf.dynobj;

  sdyn = bfd_get_section_by_name (dynobj, ".dynamic");

  if (htab->elf.dynamic_sections_created)
    {
      Elf32_External_Dyn *dyncon, *dynconend;

      if (sdyn == NULL)
	abort ();

      dyncon = (Elf32_External_Dyn *) sdyn->contents;
      dynconend = (Elf32_External_Dyn *) (sdyn->contents + sdyn->size);
      for (; dyncon < dynconend; dyncon++)
	{
	  Elf_Internal_Dyn dyn;
	  asection *s;

	  bfd_elf32_swap_dyn_in (dynobj, dyncon, &dyn);

	  switch (dyn.d_tag)
	    {
	    default:
	      continue;

	    case DT_PLTGOT:
	      /* Use PLTGOT to set the GOT register.  */
	      dyn.d_un.d_ptr = elf_gp (output_bfd);
	      break;

	    case DT_JMPREL:
	      s = htab->srelplt;
	      dyn.d_un.d_ptr = s->output_section->vma + s->output_offset;
	      break;

	    case DT_PLTRELSZ:
	      s = htab->srelplt;
	      dyn.d_un.d_val = s->size;
	      break;

	    case DT_RELASZ:
	      /* Don't count procedure linkage table relocs in the
		 overall reloc count.  */
	      s = htab->srelplt;
	      if (s == NULL)
		continue;
	      dyn.d_un.d_val -= s->size;
	      break;

	    case DT_RELA:
	      /* We may not be using the standard ELF linker script.
		 If .rela.plt is the first .rela section, we adjust
		 DT_RELA to not include it.  */
	      s = htab->srelplt;
	      if (s == NULL)
		continue;
	      if (dyn.d_un.d_ptr != s->output_section->vma + s->output_offset)
		continue;
	      dyn.d_un.d_ptr += s->size;
	      break;
	    }

	  bfd_elf32_swap_dyn_out (output_bfd, &dyn, dyncon);
	}
    }

  if (htab->sgot != NULL && htab->sgot->size != 0)
    {
      /* Fill in the first entry in the global offset table.
	 We use it to point to our dynamic section, if we have one.  */
      bfd_put_32 (output_bfd,
		  sdyn ? sdyn->output_section->vma + sdyn->output_offset : 0,
		  htab->sgot->contents);

      /* The second entry is reserved for use by the dynamic linker.  */
      memset (htab->sgot->contents + GOT_ENTRY_SIZE, 0, GOT_ENTRY_SIZE);

      /* Set .got entry size.  */
      elf_section_data (htab->sgot->output_section)
	->this_hdr.sh_entsize = GOT_ENTRY_SIZE;
    }

  if (htab->splt != NULL && htab->splt->size != 0)
    {
      /* Set plt entry size.  */
      elf_section_data (htab->splt->output_section)
	->this_hdr.sh_entsize = PLT_ENTRY_SIZE;

      if (htab->need_plt_stub)
	{
	  /* Set up the .plt stub.  */
	  memcpy (htab->splt->contents
		  + htab->splt->size - sizeof (plt_stub),
		  plt_stub, sizeof (plt_stub));

	  if ((htab->splt->output_offset
	       + htab->splt->output_section->vma
	       + htab->splt->size)
	      != (htab->sgot->output_offset
		  + htab->sgot->output_section->vma))
	    {
	      (*_bfd_error_handler)
		(_(".got section not immediately after .plt section"));
	      return FALSE;
	    }
	}
    }

  return TRUE;
}

/* Tweak the OSABI field of the elf header.  */

static void
elf32_hppa_post_process_headers (bfd *abfd,
				 struct bfd_link_info *info ATTRIBUTE_UNUSED)
{
  Elf_Internal_Ehdr * i_ehdrp;

  i_ehdrp = elf_elfheader (abfd);

  if (strcmp (bfd_get_target (abfd), "elf32-hppa-linux") == 0)
    {
      i_ehdrp->e_ident[EI_OSABI] = ELFOSABI_LINUX;
    }
  else if (strcmp (bfd_get_target (abfd), "elf32-hppa-netbsd") == 0)
    {
      i_ehdrp->e_ident[EI_OSABI] = ELFOSABI_NETBSD;
    }
  else
    {
      i_ehdrp->e_ident[EI_OSABI] = ELFOSABI_HPUX;
    }
}

/* Called when writing out an object file to decide the type of a
   symbol.  */
static int
elf32_hppa_elf_get_symbol_type (Elf_Internal_Sym *elf_sym, int type)
{
  if (ELF_ST_TYPE (elf_sym->st_info) == STT_PARISC_MILLI)
    return STT_PARISC_MILLI;
  else
    return type;
}

/* Misc BFD support code.  */
#define bfd_elf32_bfd_is_local_label_name    elf_hppa_is_local_label_name
#define bfd_elf32_bfd_reloc_type_lookup	     elf_hppa_reloc_type_lookup
#define elf_info_to_howto		     elf_hppa_info_to_howto
#define elf_info_to_howto_rel		     elf_hppa_info_to_howto_rel

/* Stuff for the BFD linker.  */
#define bfd_elf32_bfd_final_link	     elf32_hppa_final_link
#define bfd_elf32_bfd_link_hash_table_create elf32_hppa_link_hash_table_create
#define bfd_elf32_bfd_link_hash_table_free   elf32_hppa_link_hash_table_free
#define elf_backend_adjust_dynamic_symbol    elf32_hppa_adjust_dynamic_symbol
#define elf_backend_copy_indirect_symbol     elf32_hppa_copy_indirect_symbol
#define elf_backend_check_relocs	     elf32_hppa_check_relocs
#define elf_backend_create_dynamic_sections  elf32_hppa_create_dynamic_sections
#define elf_backend_fake_sections	     elf_hppa_fake_sections
#define elf_backend_relocate_section	     elf32_hppa_relocate_section
#define elf_backend_hide_symbol		     elf32_hppa_hide_symbol
#define elf_backend_finish_dynamic_symbol    elf32_hppa_finish_dynamic_symbol
#define elf_backend_finish_dynamic_sections  elf32_hppa_finish_dynamic_sections
#define elf_backend_size_dynamic_sections    elf32_hppa_size_dynamic_sections
#define elf_backend_gc_mark_hook	     elf32_hppa_gc_mark_hook
#define elf_backend_gc_sweep_hook	     elf32_hppa_gc_sweep_hook
#define elf_backend_grok_prstatus	     elf32_hppa_grok_prstatus
#define elf_backend_grok_psinfo		     elf32_hppa_grok_psinfo
#define elf_backend_object_p		     elf32_hppa_object_p
#define elf_backend_final_write_processing   elf_hppa_final_write_processing
#define elf_backend_post_process_headers     elf32_hppa_post_process_headers
#define elf_backend_get_symbol_type	     elf32_hppa_elf_get_symbol_type
#define elf_backend_reloc_type_class	     elf32_hppa_reloc_type_class

#define elf_backend_can_gc_sections	     1
#define elf_backend_can_refcount	     1
#define elf_backend_plt_alignment	     2
#define elf_backend_want_got_plt	     0
#define elf_backend_plt_readonly	     0
#define elf_backend_want_plt_sym	     0
#define elf_backend_got_header_size	     8
#define elf_backend_rela_normal		     1

#define TARGET_BIG_SYM		bfd_elf32_hppa_vec
#define TARGET_BIG_NAME		"elf32-hppa"
#define ELF_ARCH		bfd_arch_hppa
#define ELF_MACHINE_CODE	EM_PARISC
#define ELF_MAXPAGESIZE		0x1000

#include "elf32-target.h"

#undef TARGET_BIG_SYM
#define TARGET_BIG_SYM			bfd_elf32_hppa_linux_vec
#undef TARGET_BIG_NAME
#define TARGET_BIG_NAME			"elf32-hppa-linux"

#define INCLUDED_TARGET_FILE 1
#include "elf32-target.h"

#undef TARGET_BIG_SYM
#define TARGET_BIG_SYM			bfd_elf32_hppa_nbsd_vec
#undef TARGET_BIG_NAME
#define TARGET_BIG_NAME			"elf32-hppa-netbsd"

#include "elf32-target.h"
