// |reftest| skip-if(!this.hasOwnProperty('FinalizationRegistry')) async -- FinalizationRegistry is not enabled unconditionally
// Copyright (C) 2019 Leo Balter. All rights reserved.
// This code is governed by the BSD license found in the LICENSE file.

/*---
esid: sec-finalization-registry-target
description: >
  cleanupCallback has only one optional chance to be called for a GC that cleans up
  a registered target.
info: |
  FinalizationRegistry ( cleanupCallback )

  FinalizationRegistry.prototype.cleanupSome ( [ callback ] )

  ...
  4. If callback is not undefined and IsCallable(callback) is false, throw a TypeError exception.
  5. Perform ? CleanupFinalizationRegistry(finalizationRegistry, callback).
  6. Return undefined.

  Execution

  At any time, if an object obj is not live, an ECMAScript implementation may perform the following steps atomically:

  1. For each WeakRef ref such that ref.[[Target]] is obj,
    a. Set ref.[[Target]] to empty.
  2. For each FinalizationRegistry finalizationRegistry such that finalizationRegistry.[[Cells]] contains cell, such that cell.[[Target]] is obj,
    a. Set cell.[[Target]] to empty.
    b. Optionally, perform ! HostCleanupFinalizationRegistry(finalizationRegistry).
features: [cleanupSome, FinalizationRegistry.prototype.cleanupSome, FinalizationRegistry, async-functions, host-gc-required]
flags: [async, non-deterministic]
includes: [async-gc.js, compareArray.js]
---*/

let cleanupCallback = 0;
let holdings = [];
function cb(holding) {
  holdings.push(holding);
}

let finalizationRegistry = new FinalizationRegistry(function() {
  cleanupCallback += 1;
});

function emptyCells() {
  let target = {};
  finalizationRegistry.register(target, 'a');

  let prom = asyncGC(target);
  target = null;

  return prom;
}

emptyCells().then(async function() {
  await Promise.resolve(1);

  finalizationRegistry.cleanupSome(cb);

  // cleanupSome will be invoked if there are empty cells left. If the
  // cleanupCallback already ran, then cb won't be called.
  let expectedCalled = cleanupCallback === 1 ? 0 : 1;
  // This asserts the registered object was emptied in the previous GC.
  assert.sameValue(holdings.length, expectedCalled, 'cleanupSome callback for the first time');

  // At this point, we can't assert if cleanupCallback was called, because it's
  // optional. Although, we can finally assert it's not gonna be called anymore
  // for the other executions of the Garbage Collector.
  // The chance of having it called only happens right after the
  // cell.[[Target]] is set to empty.
  assert(cleanupCallback >= 0, 'cleanupCallback might be 0');
  assert(cleanupCallback <= 1, 'cleanupCallback might be 1');

  // Restoring the cleanupCallback variable to 0 will help us asserting the
  // finalizationRegistry callback is not called again.
  cleanupCallback = 0;

  await $262.gc();
  await Promise.resolve(2); // tick

  finalizationRegistry.cleanupSome(cb);

  assert.sameValue(holdings.length, expectedCalled, 'cleanupSome callback is not called anymore, no empty cells');
  assert.sameValue(cleanupCallback, 0, 'cleanupCallback is not called again #1');

  await $262.gc();
  await Promise.resolve(3); // tick

  finalizationRegistry.cleanupSome(cb);

  assert.sameValue(holdings.length, expectedCalled, 'cleanupSome callback is not called again #2');
  assert.sameValue(cleanupCallback, 0, 'cleanupCallback is not called again #2');

  if (holdings.length) {
    assert.compareArray(holdings, ['a']);
  }

  await $262.gc();
}).then($DONE, resolveAsyncGC);
