/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <cassert>
#include <string>
#include <vector>

#include <libwpd-stream/libwpd-stream.h>

#include "libebook_utils.h"
#include "EBOOKStreamSlice.h"
#include "PDXParser.h"

using boost::scoped_ptr;

namespace libebook
{

namespace
{

struct HeaderData
{
  HeaderData();

  std::string m_name;
  unsigned m_version;
  unsigned m_appInfoID;
  unsigned m_sortInfoID;
  unsigned m_type;
  unsigned m_creator;
  unsigned m_nextRecordListID;
  unsigned m_numberOfRecords;
  std::vector<unsigned> m_recordOffsets;
};

}

namespace
{

HeaderData::HeaderData()
  : m_name()
  , m_version(0)
  , m_appInfoID(0)
  , m_sortInfoID(0)
  , m_type(0)
  , m_creator(0)
  , m_nextRecordListID(0)
  , m_numberOfRecords(0)
  , m_recordOffsets()
{
}

}

struct PDXParserImpl
{
  PDXParserImpl(WPXInputStream *input, WPXDocumentInterface *document);

  HeaderData m_header;
  WPXInputStream *m_input;
  WPXDocumentInterface *m_document;

private:
// disable copying
  PDXParserImpl(const PDXParserImpl &other);
  PDXParserImpl &operator=(const PDXParserImpl &other);
};

PDXParserImpl::PDXParserImpl(WPXInputStream *const input, WPXDocumentInterface *const document)
  : m_header()
  , m_input(input)
  , m_document(document)
{
}

PDXParser::PDXParser(WPXInputStream *const input, WPXDocumentInterface *const document)
  : m_impl(new PDXParserImpl(input, document))
{
}

PDXParser::~PDXParser()
{
}

bool PDXParser::isSupported()
{
  readHeader();
  return (m_impl->m_header.m_numberOfRecords > 0)
         && (m_impl->m_header.m_recordOffsets.size() == m_impl->m_header.m_numberOfRecords)
         && isFormatSupported(m_impl->m_header.m_type, m_impl->m_header.m_creator);
}

bool PDXParser::parse()
{
  readHeader();

  if (m_impl->m_header.m_appInfoID)
  {
    // TODO: implement me
  }
  if (m_impl->m_header.m_sortInfoID)
  {
    // TODO: implement me
  }

  {
    boost::scoped_ptr<WPXInputStream> input(getRecordStream(0));
    readIndexRecord(input.get());
  }

  readDataRecords();

  return true;
}

WPXDocumentInterface *PDXParser::getDocument() const
{
  return m_impl->m_document;
}

const char *PDXParser::getName() const
{
  return m_impl->m_header.m_name.c_str();
}

WPXInputStream *PDXParser::getAppInfoRecord() const
{
  // TODO: implement me
  return 0;
}

WPXInputStream *PDXParser::getIndexRecord() const
{
  return getRecordStream(0);
}

unsigned PDXParser::getDataRecordCount() const
{
  return m_impl->m_header.m_numberOfRecords - 1;
}

WPXInputStream *PDXParser::getDataRecord(unsigned n) const
{
  return getRecordStream(n + 1);
}

WPXInputStream *PDXParser::getDataRecords() const
{
  const unsigned begin = m_impl->m_header.m_recordOffsets[1];
  m_impl->m_input->seek(0, WPX_SEEK_END);
  const unsigned end = static_cast<unsigned>(m_impl->m_input->tell());

  return new EBOOKStreamSlice(m_impl->m_input, begin, end);
}

WPXInputStream *PDXParser::getDataRecords(unsigned first, unsigned last) const
{
  if (first >= last)
    return 0;
  if ((m_impl->m_header.m_numberOfRecords - 1) < last)
    return 0;

  const unsigned begin = m_impl->m_header.m_recordOffsets[first + 1];
  unsigned end = 0;
  if ((m_impl->m_header.m_numberOfRecords - 1) == last) // ends with the last record
  {
    m_impl->m_input->seek(0, WPX_SEEK_END);
    end = static_cast<unsigned>(m_impl->m_input->tell());
  }
  else
    end = m_impl->m_header.m_recordOffsets[last + 1];

  return new EBOOKStreamSlice(m_impl->m_input, begin, end);
}

void PDXParser::readDataRecords()
{
  for (unsigned i = 1; i != m_impl->m_header.m_numberOfRecords; ++i)
  {
    scoped_ptr<WPXInputStream> record(getRecordStream(i));
    readDataRecord(record.get(), m_impl->m_header.m_numberOfRecords - 1 == i);
  }
}

void PDXParser::readHeader()
{
  m_impl->m_input->seek(0, WPX_SEEK_SET);

  char name[32];
  unsigned nameLen = 0;
  while (nameLen != 32)
  {
    unsigned char c = readU8(m_impl->m_input);
    name[nameLen++] = c;
    if (0 == c)
      break;
  }
  // We read the whole name, but there was no terminating 0. Either a
  // broken file or not a PDB file at all...
  if (nameLen == 32)
    --nameLen;
  name[nameLen] = 0;
  m_impl->m_header.m_name.assign(name, nameLen);

  // skip the unused bytes of name and ignore 2 bytes of attributes
  m_impl->m_input->seek(34, WPX_SEEK_SET);
  m_impl->m_header.m_version = readU16(m_impl->m_input, true);
  // ignore dates, modification number and IDs; 6*4 = 24 bytes
  skip(m_impl->m_input, 24);
  m_impl->m_header.m_type = readU32(m_impl->m_input, true);
  m_impl->m_header.m_creator = readU32(m_impl->m_input, true);
  skip(m_impl->m_input, 4); // skip uniqueIDseed
  m_impl->m_header.m_nextRecordListID = readU32(m_impl->m_input, true);
  assert(m_impl->m_input->tell() == 76);
  m_impl->m_header.m_numberOfRecords = readU16(m_impl->m_input, true);

  // read records
  for (unsigned i = 0; i != m_impl->m_header.m_numberOfRecords; ++i)
  {
    m_impl->m_header.m_recordOffsets.push_back(readU32(m_impl->m_input, true));
    skip(m_impl->m_input, 4); // skip the uninteresting remainder
  }
}

WPXInputStream *PDXParser::getRecordStream(const unsigned n) const
{
  if (n >= m_impl->m_header.m_numberOfRecords)
    return 0;

  const unsigned begin = m_impl->m_header.m_recordOffsets[n];
  unsigned end = 0;
  if ((m_impl->m_header.m_numberOfRecords - 1) == n) // the last record
  {
    m_impl->m_input->seek(0, WPX_SEEK_END);
    end = static_cast<unsigned>(m_impl->m_input->tell());
  }
  else
    end = m_impl->m_header.m_recordOffsets[n + 1];

  return new EBOOKStreamSlice(m_impl->m_input, begin, end);
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
