// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

//----------------------------------------------------------
// MethodContext.cpp - Primary structure to store all the EE-JIT details required to replay creation of a method
//                     CompileResult contains the stuff generated by a compilation
//----------------------------------------------------------

#include "standardpch.h"
#include "methodcontext.h"
#include "compileresult.h"
#include "lightweightmap.h"
#include "callutils.h"
#include "spmirecordhelper.h"
#include "spmidumphelper.h"
#include "spmiutil.h"

#define sparseMC // Support filling in details where guesses are okay and will still generate good code. (i.e. helper
                 // function addresses)

// static variable initialization
Hash MethodContext::m_hash;

MethodContext::MethodContext()
{
    methodSize = 0;

#define LWM(map, key, value) map = nullptr;
#include "lwmlist.h"

    cr    = new CompileResult();
    index = -1;
    isReadyToRunCompilation = ReadyToRunCompilation::Uninitialized;
}

MethodContext::~MethodContext()
{
    Destroy();
}

void MethodContext::Destroy()
{
#define LWM(map, key, value)                                                                                           \
    if (map != nullptr)                                                                                                \
        delete map;
#include "lwmlist.h"

    delete cr;
    FreeTempAllocations();
}

#define sparseAddLen(target)                                                                                           \
    if (target != nullptr)                                                                                             \
    {                                                                                                                  \
        if (target->GetCount() != 0)                                                                                   \
            totalLen += target->CalculateArraySize() + 6; /* packet canary from lightweightmap + packet marker */      \
    }

#define sparseWriteFile(target)                                                                                        \
    if (target != nullptr)                                                                                             \
    {                                                                                                                  \
        if (target->GetCount() != 0)                                                                                   \
        {                                                                                                              \
            buff2[buffIndex++] = (unsigned char)Packet_##target;                                                       \
            unsigned int loc   = target->DumpToArray(&buff2[buffIndex + 4]);                                           \
            memcpy(&buff2[buffIndex], &loc, sizeof(unsigned int));                                                     \
            buffIndex += 4 + loc;                                                                                      \
            buff2[buffIndex++] = 0x42;                                                                                 \
        }                                                                                                              \
    }

#define sparseWriteFileCR(target)                                                                                      \
    if (cr != nullptr)                                                                                                 \
    {                                                                                                                  \
        if (cr->target != nullptr)                                                                                     \
        {                                                                                                              \
            if (cr->target->GetCount() != 0)                                                                           \
            {                                                                                                          \
                buff2[buffIndex++] = (unsigned char)PacketCR_##target;                                                 \
                unsigned int loc   = cr->target->DumpToArray(&buff2[buffIndex + 4]);                                   \
                memcpy(&buff2[buffIndex], &loc, sizeof(unsigned int));                                                 \
                buffIndex += 4 + loc;                                                                                  \
                buff2[buffIndex++] = 0x42;                                                                             \
            }                                                                                                          \
        }                                                                                                              \
    }

#define sparseReadFile(target, key, value)                                                                             \
    case Packet_##target:                                                                                              \
    {                                                                                                                  \
        target = new LightWeightMap<key, value>();                                                                     \
        target->ReadFromArray(&buff2[buffIndex], localsize);                                                           \
        break;                                                                                                         \
    }

#define sparseReadFileCR(target, key, value)                                                                           \
    case PacketCR_##target:                                                                                            \
    {                                                                                                                  \
        cr->target = new LightWeightMap<key, value>();                                                                 \
        cr->target->ReadFromArray(&buff2[buffIndex], localsize);                                                       \
        break;                                                                                                         \
    }

#define sparseReadFileDense(target, value)                                                                             \
    case Packet_##target:                                                                                              \
    {                                                                                                                  \
        target = new DenseLightWeightMap<value>();                                                                     \
        target->ReadFromArray(&buff2[buffIndex], localsize);                                                           \
        break;                                                                                                         \
    }

#define sparseReadFileCRDense(target, value)                                                                           \
    case PacketCR_##target:                                                                                            \
    {                                                                                                                  \
        cr->target = new DenseLightWeightMap<value>();                                                                 \
        cr->target->ReadFromArray(&buff2[buffIndex], localsize);                                                       \
        break;                                                                                                         \
    }

unsigned int MethodContext::calculateFileSize()
{
    // Calculate file size
    unsigned int totalLen = 0;

#define LWM(map, key, value) sparseAddLen(map)
#include "lwmlist.h"

    // Compile Result members
    if (cr != nullptr)
    {
#define LWM(map, key, value) sparseAddLen(cr->map);
#include "crlwmlist.h"
    }

    return totalLen;
}

unsigned int MethodContext::calculateRawFileSize()
{
    return 2 /* leading magic cookie 'm', 'c' */ + 4 /* 4-byte data length */ + calculateFileSize() +
           2 /* end canary '4', '2' */;
}

unsigned int MethodContext::saveToFile(HANDLE hFile)
{
    unsigned int totalLen = calculateFileSize();
    unsigned int totalFileSize =
        2 /* leading magic cookie 'm', 'c' */ + 4 /* 4-byte data length */ + totalLen + 2 /* end canary '4', '2' */;

    DWORD          bytesWritten = 0;
    unsigned int   buffIndex    = 0;
    unsigned char* buff2        = new unsigned char[totalFileSize];
    buff2[buffIndex++]          = 'm';
    buff2[buffIndex++]          = 'c';
    memcpy(&buff2[buffIndex], &totalLen, sizeof(unsigned int));
    buffIndex += 4;

#define LWM(map, key, value) sparseWriteFile(map)
#include "lwmlist.h"

// Compile Result members
#define LWM(map, key, value) sparseWriteFileCR(map);
#include "crlwmlist.h"

    // Write the end canary
    buff2[buffIndex++] = '4';
    buff2[buffIndex++] = '2';

    Assert(buffIndex == totalFileSize);

    WriteFile(hFile, buff2, totalFileSize, &bytesWritten, NULL);
    delete[] buff2;
    return bytesWritten;
}

// This code can't exist in a function with C++ objects needing destruction. Returns true on success
// (and sets *ppmc with new MethodContext), false on failure.
//
// static
bool MethodContext::Initialize(int mcIndex, unsigned char* buff, DWORD size, /* OUT */ MethodContext** ppmc)
{
    MethodContext* mc = new MethodContext();
    mc->index         = mcIndex;
    *ppmc             = mc;
    return mc->Initialize(mcIndex, buff, size);
}

// static
bool MethodContext::Initialize(int mcIndex, HANDLE hFile, /* OUT */ MethodContext** ppmc)
{
    MethodContext* mc = new MethodContext();
    mc->index         = mcIndex;
    *ppmc             = mc;
    return mc->Initialize(mcIndex, hFile);
}

bool MethodContext::Initialize(int mcIndex, unsigned char* buff, DWORD size)
{
    bool result = true;

    struct Param
    {
        unsigned char* buff;
        DWORD          size;
        MethodContext* pThis;
    } param;
    param.buff  = buff;
    param.size  = size;
    param.pThis = this;

    PAL_TRY(Param*, pParam, &param)
    {
        pParam->pThis->MethodInitHelper(pParam->buff, pParam->size);
    }
    PAL_EXCEPT_FILTER(FilterSuperPMIExceptions_CatchMC)
    {
        LogError("Method %d is of low integrity.", mcIndex);
        result = false;
    }
    PAL_ENDTRY

    return result;
}

bool MethodContext::Initialize(int mcIndex, HANDLE hFile)
{
    bool result = true;

    struct Param
    {
        HANDLE         hFile;
        MethodContext* pThis;
    } param;
    param.hFile = hFile;
    param.pThis = this;

    PAL_TRY(Param*, pParam, &param)
    {
        pParam->pThis->MethodInitHelperFile(pParam->hFile);
    }
    PAL_EXCEPT_FILTER(FilterSuperPMIExceptions_CatchMC)
    {
        LogError("Method %d is of low integrity.", mcIndex);
        result = false;
    }
    PAL_ENDTRY

    return result;
}

void MethodContext::MethodInitHelperFile(HANDLE hFile)
{
    DWORD        bytesRead;
    char         buff[512];
    unsigned int totalLen = 0;

    AssertCode(ReadFile(hFile, buff, 2 + sizeof(unsigned int), &bytesRead, NULL) == TRUE,
               EXCEPTIONCODE_MC); // Read Magic number and totalLen
    AssertCodeMsg((buff[0] == 'm') && (buff[1] == 'c'), EXCEPTIONCODE_MC, "Didn't find magic number");
    memcpy(&totalLen, &buff[2], sizeof(unsigned int));
    unsigned char* buff2 = new unsigned char[totalLen + 2]; // total + End Canary
    AssertCode(ReadFile(hFile, buff2, totalLen + 2, &bytesRead, NULL) == TRUE, EXCEPTIONCODE_MC);
    AssertCodeMsg((buff2[totalLen] == '4') && (buff2[totalLen + 1] == '2'), EXCEPTIONCODE_MC, "Didn't find end canary");
    MethodInitHelper(buff2, totalLen);
}

void MethodContext::MethodInitHelper(unsigned char* buff2, unsigned int totalLen)
{
    unsigned int   buffIndex = 0;
    unsigned int   localsize = 0;
    unsigned char  canary    = 0xff;
    unsigned char* buff3     = nullptr;

    FreeTempAllocations();

    while (buffIndex < totalLen)
    {
        mcPackets packetType = (mcPackets)buff2[buffIndex++];
        memcpy(&localsize, &buff2[buffIndex], sizeof(unsigned int));
        buffIndex += sizeof(unsigned int);

        switch (packetType)
        {
#define LWM(map, key, value) sparseReadFile(map, key, value)
#define DENSELWM(map, value) sparseReadFileDense(map, value)
#include "lwmlist.h"

#define LWM(map, key, value) sparseReadFileCR(map, key, value)
#define DENSELWM(map, value) sparseReadFileCRDense(map, value)
#include "crlwmlist.h"

            default:
                LogException(EXCEPTIONCODE_MC, "Read ran into unknown packet type %u. Are you using a newer recorder?",
                             packetType);
                // break;
        }
        buffIndex += localsize;
        canary = buff2[buffIndex++];
        AssertCodeMsg(canary == 0x42, EXCEPTIONCODE_MC, "Didn't find trailing canary for map");
    }
    AssertCodeMsg((buff2[buffIndex++] == '4') && (buff2[buffIndex++] == '2'), EXCEPTIONCODE_MC,
                  "Didn't find trailing canary for map");
    delete[] buff2;
}

#define dumpStat(target)                                                                                               \
    if (target != nullptr)                                                                                             \
    {                                                                                                                  \
        if (target->GetCount() > 0)                                                                                    \
        {                                                                                                              \
            int t = sprintf_s(buff, len, "%u", target->GetCount());                                                    \
            buff += t;                                                                                                 \
            len -= t;                                                                                                  \
        }                                                                                                              \
    }                                                                                                                  \
    {                                                                                                                  \
        *buff++ = ',';                                                                                                 \
        len--;                                                                                                         \
    }                                                                                                                  \
    if (target != nullptr)                                                                                             \
    {                                                                                                                  \
        int t = sprintf_s(buff, len, "%u", target->CalculateArraySize());                                              \
        buff += t;                                                                                                     \
        len -= t;                                                                                                      \
    }                                                                                                                  \
    {                                                                                                                  \
        *buff++ = ',';                                                                                                 \
        len--;                                                                                                         \
    }

// Dump statistics about each LightWeightMap to the buffer: count of elements, and total size in bytes of map.
int MethodContext::dumpStatToBuffer(char* buff, int len)
{
    char* obuff = buff;
// assumption of enough buffer.. :-|

#define LWM(map, key, value) dumpStat(map)
#include "lwmlist.h"

// Compile Result members
#define LWM(map, key, value) dumpStat(cr->map);
#include "crlwmlist.h"

    return (int)(buff - obuff);
}
int MethodContext::dumpStatTitleToBuffer(char* buff, int len)
{
    const char* title =

#define LWM(map, key, value) #map "," #map " SZ,"
#include "lwmlist.h"

#define LWM(map, key, value) "CR_" #map ",CR_" #map " SZ,"
#include "crlwmlist.h"

        ;

    int titleLen = (int)strlen(title);
    if ((titleLen + 1) > len)
    {
        LogError("titleLen is larger than given len");
        return 0;
    }
    strcpy_s(buff, len, title);
    return titleLen;
}

#define softMapEqual(a)                                                                                                \
    if (a != nullptr)                                                                                                  \
    {                                                                                                                  \
        if (other->a == nullptr)                                                                                       \
            return false;                                                                                              \
        if (a->GetCount() != other->a->GetCount())                                                                     \
            return false;                                                                                              \
    }                                                                                                                  \
    else if (other->a != nullptr)                                                                                      \
        return false;

bool MethodContext::Equal(MethodContext* other)
{
    // returns true if equal.  Note this is permissive, that is to say that we may reason that too many things are
    // equal. Adding more detailed checks would cause us to reason more things as unique;

    // Compare MethodInfo's first.
    CORINFO_METHOD_INFO otherInfo;
    unsigned            otherFlags = 0;
    other->repCompileMethod(&otherInfo, &otherFlags);

    CORINFO_METHOD_INFO ourInfo;
    unsigned            ourFlags = 0;
    repCompileMethod(&ourInfo, &ourFlags);

    if (otherInfo.ILCodeSize != ourInfo.ILCodeSize)
        return false;
    if (otherInfo.args.numArgs != ourInfo.args.numArgs)
        return false;
    if (otherInfo.args.retType != ourInfo.args.retType)
        return false;
    if (otherInfo.locals.numArgs != ourInfo.locals.numArgs)
        return false;
    if (otherInfo.EHcount != ourInfo.EHcount)
        return false;
    if (otherInfo.options != ourInfo.options)
        return false;
    for (unsigned int j = 0; j < otherInfo.ILCodeSize; j++)
        if (otherInfo.ILCode[j] != ourInfo.ILCode[j])
            return false;
    if (otherInfo.maxStack != ourInfo.maxStack)
        return false;
    if (otherInfo.regionKind != ourInfo.regionKind)
        return false;
    if (otherInfo.args.callConv != ourInfo.args.callConv)
        return false;
    if (otherInfo.args.cbSig != ourInfo.args.cbSig)
        return false;
    if (otherInfo.args.flags != ourInfo.args.flags)
        return false;
    if (otherInfo.locals.cbSig != ourInfo.locals.cbSig)
        return false;
    if (otherFlags != ourFlags)
        return false;

// Now compare the other maps to "estimate" equality.

#define LWM(map, key, value) softMapEqual(map)
#include "lwmlist.h"

// Compile Result members
#define LWM(map, key, value) softMapEqual(cr->map)
#include "crlwmlist.h"

    // Base case is we "match"
    return true;
}

//------------------------------------------------------------------------------
// MethodContext::recGlobalContext
//    This method copies any relevant global (i.e. per-JIT-instance) data from
//    the given method context. Currently this is limited to configuration
//    values, but may grow to encompass other information in the future (e.g.
//    any information that is exposed by the ICorJitHost interface and is
//    therefore accessible outside the context of a call to
//    `ICJI::compileMethod`).
//
//    This method is intended to be called as part of initializing a method
//    during collection.
void MethodContext::recGlobalContext(const MethodContext& other)
{
    Assert(GetIntConfigValue == nullptr);
    Assert(GetStringConfigValue == nullptr);

    if (other.GetIntConfigValue != nullptr)
    {
        GetIntConfigValue = new LightWeightMap<Agnostic_ConfigIntInfo, DWORD>(*other.GetIntConfigValue);
    }

    if (other.GetStringConfigValue != nullptr)
    {
        GetStringConfigValue = new LightWeightMap<DWORD, DWORD>(*other.GetStringConfigValue);
    }
}

// dumpToConsole: Display the method context numbered `mcNumber` to the console. If `simple` is true,
// dump without function name information. This is useful to debug problems with the creation of that
// information, which requires looking at the dumped info.
void MethodContext::dumpToConsole(int mcNumber, bool simple)
{
    printf("*****************************************");
    if (mcNumber != -1)
    {
        printf(" method context #%d", mcNumber);
    }

    if (!simple)
    {
        // Dump method name, etc., to output.
        char bufferIdentityInfo[METHOD_IDENTITY_INFO_SIZE];
        int cbLen = dumpMethodIdentityInfoToBuffer(bufferIdentityInfo, METHOD_IDENTITY_INFO_SIZE);
        if (cbLen >= 0)
        {
            printf(" %s", bufferIdentityInfo);
        }
    }

    printf("\n");

#define LWM(map, key, value) dumpLWM(this, map)
#define DENSELWM(map, value) dumpLWMDense(this, map)
#include "lwmlist.h"

// Compile Result members
#define LWM(map, key, value) dumpLWM(this->cr, map)
#define DENSELWM(map, value) dumpLWMDense(this->cr, map)
#include "crlwmlist.h"
}

const char* toString(InfoAccessType iat)
{
    switch (iat)
    {
    case IAT_VALUE:
        return "VALUE";
    case IAT_PVALUE:
        return "PVALUE";
    case IAT_PPVALUE:
        return "PPVALUE";
    case IAT_RELPVALUE:
        return "RELPVALUE";
    default:
        return "UNKNOWN";
    }
};

const char* toString(CORINFO_CALL_KIND cick)
{
    switch (cick)
    {
    case CORINFO_CALL:
        return "CALL";
    case CORINFO_CALL_CODE_POINTER:
        return "CALL_CODE_POINTER";
    case CORINFO_VIRTUALCALL_STUB:
        return "VIRTUALCALL_STUB";
    case CORINFO_VIRTUALCALL_LDVIRTFTN:
        return "VIRTUALCALL_LDVIRTFTN";
    case CORINFO_VIRTUALCALL_VTABLE:
        return "VIRTUALCALL_VTABLE";
    default:
        return "UNKNOWN";
    }
};

const char* toString(CorInfoType cit)
{
    switch (cit)
    {
        case CORINFO_TYPE_UNDEF:
            return "undef";
        case CORINFO_TYPE_VOID:
            return "void";
        case CORINFO_TYPE_BOOL:
            return "bool";
        case CORINFO_TYPE_CHAR:
            return "char";
        case CORINFO_TYPE_BYTE:
            return "byte";
        case CORINFO_TYPE_UBYTE:
            return "ubyte";
        case CORINFO_TYPE_SHORT:
            return "short";
        case CORINFO_TYPE_USHORT:
            return "ushort";
        case CORINFO_TYPE_INT:
            return "int";
        case CORINFO_TYPE_UINT:
            return "uint";
        case CORINFO_TYPE_LONG:
            return "long";
        case CORINFO_TYPE_ULONG:
            return "ulong";
        case CORINFO_TYPE_NATIVEINT:
            return "nativeint";
        case CORINFO_TYPE_NATIVEUINT:
            return "nativeuint";
        case CORINFO_TYPE_FLOAT:
            return "float";
        case CORINFO_TYPE_DOUBLE:
            return "double";
        case CORINFO_TYPE_STRING:
            return "string";
        case CORINFO_TYPE_PTR:
            return "ptr";
        case CORINFO_TYPE_BYREF:
            return "byref";
        case CORINFO_TYPE_VALUECLASS:
            return "valueclass";
        case CORINFO_TYPE_CLASS:
            return "class";
        case CORINFO_TYPE_REFANY:
            return "refany";
        case CORINFO_TYPE_VAR:
            return "var";
        default:
            return "UNKNOWN";
    }
}

const char* toString(CorInfoTypeWithMod cit)
{
    // Need to cast `cit` to numeric type to avoid clang compiler warnings
    // "case value not in enumerated type 'CorInfoTypeWithMod'".
    switch ((unsigned)cit)
    {
        case (unsigned)(CORINFO_TYPE_BYREF | CORINFO_TYPE_MOD_PINNED):
            return "pinned byref";
        case (unsigned)(CORINFO_TYPE_CLASS | CORINFO_TYPE_MOD_PINNED):
            return "pinned class";
        default:
            return toString((CorInfoType)(cit & CORINFO_TYPE_MASK));
    }
}

unsigned int toCorInfoSize(CorInfoType cit)
{
    switch (cit)
    {
        case CORINFO_TYPE_BOOL:
        case CORINFO_TYPE_BYTE:
        case CORINFO_TYPE_UBYTE:
            return 1;

        case CORINFO_TYPE_CHAR:
        case CORINFO_TYPE_SHORT:
        case CORINFO_TYPE_USHORT:
            return 2;

        case CORINFO_TYPE_FLOAT:
        case CORINFO_TYPE_INT:
        case CORINFO_TYPE_UINT:
            return 4;

        case CORINFO_TYPE_DOUBLE:
        case CORINFO_TYPE_LONG:
        case CORINFO_TYPE_ULONG:
            return 8;

        case CORINFO_TYPE_NATIVEINT:
        case CORINFO_TYPE_NATIVEUINT:
        case CORINFO_TYPE_PTR:
        case CORINFO_TYPE_BYREF:
        case CORINFO_TYPE_CLASS:
            return (int)SpmiTargetPointerSize();

        case CORINFO_TYPE_STRING:
        case CORINFO_TYPE_VALUECLASS:
        case CORINFO_TYPE_REFANY:
        case CORINFO_TYPE_UNDEF:
        case CORINFO_TYPE_VOID:
        default:
            __debugbreak();
            return 0;
    }
    return -1;
}

void MethodContext::recCompileMethod(CORINFO_METHOD_INFO* info, unsigned flags)
{
    if (CompileMethod == nullptr)
        CompileMethod = new LightWeightMap<DWORD, Agnostic_CompileMethod>();

    Agnostic_CompileMethod value;

    value.info.ftn           = CastHandle(info->ftn);
    value.info.scope         = CastHandle(info->scope);
    value.info.ILCode_offset = (DWORD)CompileMethod->AddBuffer(info->ILCode, info->ILCodeSize);
    value.info.ILCodeSize    = (DWORD)info->ILCodeSize;
    value.info.maxStack      = (DWORD)info->maxStack;
    value.info.EHcount       = (DWORD)info->EHcount;
    value.info.options       = (DWORD)info->options;
    value.info.regionKind    = (DWORD)info->regionKind;

    value.info.args   = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(info->args, CompileMethod, SigInstHandleMap);
    value.info.locals = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(info->locals, CompileMethod, SigInstHandleMap);

    value.flags = (DWORD)flags;

    CompileMethod->Add(0, value);
    DEBUG_REC(dmpCompileMethod(0, value));
}
void MethodContext::dmpCompileMethod(DWORD key, const Agnostic_CompileMethod& value)
{
    printf("CompileMethod key %u, value ftn-%016llX scp-%016llX ilo-%u ils-%u ms-%u ehc-%u opt-%u rk-%u args-%s locals-%s flg-%08X",
           key, value.info.ftn, value.info.scope, value.info.ILCode_offset, value.info.ILCodeSize, value.info.maxStack,
           value.info.EHcount, value.info.options, value.info.regionKind,
           SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.info.args, CompileMethod, SigInstHandleMap).c_str(),
           SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.info.locals, CompileMethod, SigInstHandleMap).c_str(),
           value.flags);
}
void MethodContext::repCompileMethod(CORINFO_METHOD_INFO* info, unsigned* flags)
{
    AssertMapAndKeyExistNoMessage(CompileMethod, 0);

    Agnostic_CompileMethod value;

    value = CompileMethod->Get((DWORD)0); // The only item in this set is a single group of inputs to CompileMethod
    DEBUG_REP(dmpCompileMethod(0, value));

    info->ftn        = (CORINFO_METHOD_HANDLE)value.info.ftn;
    info->scope      = (CORINFO_MODULE_HANDLE)value.info.scope;
    info->ILCode     = CompileMethod->GetBuffer(value.info.ILCode_offset);
    info->ILCodeSize = (unsigned)value.info.ILCodeSize;
    methodSize       = info->ILCodeSize;
    info->maxStack   = (unsigned)value.info.maxStack;
    info->EHcount    = (unsigned)value.info.EHcount;
    info->options    = (CorInfoOptions)value.info.options;
    info->regionKind = (CorInfoRegionKind)value.info.regionKind;

    info->args   = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.info.args, CompileMethod, SigInstHandleMap);
    info->locals = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.info.locals, CompileMethod, SigInstHandleMap);

    *flags             = (unsigned)value.flags;
}

void MethodContext::recGetMethodClass(CORINFO_METHOD_HANDLE methodHandle, CORINFO_CLASS_HANDLE classHandle)
{
    if (GetMethodClass == nullptr)
        GetMethodClass = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(methodHandle);
    DWORDLONG value = CastHandle(classHandle);
    GetMethodClass->Add(key, value);
    DEBUG_REC(dmpGetMethodClass(key, value));
}
void MethodContext::dmpGetMethodClass(DWORDLONG key, DWORDLONG value)
{
    printf("GetMethodClass key %016llX, value %016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetMethodClass(CORINFO_METHOD_HANDLE methodHandle)
{
    DWORDLONG key = CastHandle(methodHandle);
    AssertMapAndKeyExist(GetMethodClass, key, ": key %016llX", key);
    DWORDLONG value = GetMethodClass->Get(key);
    DEBUG_REP(dmpGetMethodClass(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetMethodModule(CORINFO_METHOD_HANDLE methodHandle, CORINFO_MODULE_HANDLE moduleHandle)
{
    if (GetMethodModule == nullptr)
        GetMethodModule = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(methodHandle);
    DWORDLONG value = CastHandle(moduleHandle);
    GetMethodModule->Add(key, value);
    DEBUG_REC(dmpGetMethodModule(key, value));
}
void MethodContext::dmpGetMethodModule(DWORDLONG key, DWORDLONG value)
{
    printf("GetMethodModule key %016llX, value %016llX", key, value);
}
CORINFO_MODULE_HANDLE MethodContext::repGetMethodModule(CORINFO_METHOD_HANDLE methodHandle)
{
    DWORDLONG key = CastHandle(methodHandle);
    AssertMapAndKeyExist(GetMethodModule, key, ": key %016llX", key);
    DWORDLONG value = GetMethodModule->Get(key);
    DEBUG_REP(dmpGetMethodModule(key, value));
    CORINFO_MODULE_HANDLE result = (CORINFO_MODULE_HANDLE)value;
    return result;
}

void MethodContext::recGetClassAttribs(CORINFO_CLASS_HANDLE classHandle, DWORD attribs)
{
    if (GetClassAttribs == nullptr)
        GetClassAttribs = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(classHandle);
    GetClassAttribs->Add(key, attribs);
    DEBUG_REC(dmpGetClassAttribs(key, attribs));
}
void MethodContext::dmpGetClassAttribs(DWORDLONG key, DWORD value)
{
    printf("GetClassAttribs key %016llX, value %08X (%s)", key, value, SpmiDumpHelper::DumpCorInfoFlag((CorInfoFlag)value).c_str());
}
DWORD MethodContext::repGetClassAttribs(CORINFO_CLASS_HANDLE classHandle)
{
    DWORDLONG key = CastHandle(classHandle);
    AssertMapAndKeyExist(GetClassAttribs, key, ": key %016llX", key);
    DWORD value = GetClassAttribs->Get(key);
    DEBUG_REP(dmpGetClassAttribs(key, value));
    return value;
}

void MethodContext::recIsJitIntrinsic(CORINFO_METHOD_HANDLE ftn, bool result)
{
    if (IsJitIntrinsic == nullptr)
        IsJitIntrinsic = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(ftn);
    DWORD value = result ? 1 : 0;
    IsJitIntrinsic->Add(key, value);
    DEBUG_REC(dmpIsJitIntrinsic(key, value));
}
void MethodContext::dmpIsJitIntrinsic(DWORDLONG key, DWORD value)
{
    printf("IsJitIntrinsic key ftn-%016llX, value res-%u", key, value);
}
bool MethodContext::repIsJitIntrinsic(CORINFO_METHOD_HANDLE ftn)
{
    DWORDLONG key = CastHandle(ftn);
    AssertMapAndKeyExist(IsJitIntrinsic, key, ": key %016llX", key);
    DWORD value = IsJitIntrinsic->Get(key);
    DEBUG_REP(dmpIsJitIntrinsic(key, value));
    return value != 0;
}

void MethodContext::recGetMethodAttribs(CORINFO_METHOD_HANDLE methodHandle, DWORD attribs)
{
    if (GetMethodAttribs == nullptr)
        GetMethodAttribs = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(methodHandle);
    GetMethodAttribs->Add(key, attribs);
    DEBUG_REC(dmpGetMethodAttribs(key, attribs));
}
void MethodContext::dmpGetMethodAttribs(DWORDLONG key, DWORD value)
{
    printf("GetMethodAttribs key %016llX, value %08X (%s)", key, value, SpmiDumpHelper::DumpCorInfoFlag((CorInfoFlag)value).c_str());
}
DWORD MethodContext::repGetMethodAttribs(CORINFO_METHOD_HANDLE methodHandle)
{
    DWORDLONG key = CastHandle(methodHandle);
    AssertMapAndKeyExist(GetMethodAttribs, key, ": key %016llX", key);

    DWORD value = GetMethodAttribs->Get(key);
    DEBUG_REP(dmpGetMethodAttribs(key, value));

    if (cr->repSetMethodAttribs(methodHandle) == CORINFO_FLG_BAD_INLINEE)
        value ^= CORINFO_FLG_DONT_INLINE;
    return value;
}

void MethodContext::recGetClassModule(CORINFO_CLASS_HANDLE cls, CORINFO_MODULE_HANDLE mod)
{
    if (GetClassModule == nullptr)
        GetClassModule = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(mod);
    GetClassModule->Add(key, value);
    DEBUG_REC(dmpGetClassModule(key, value));
}
void MethodContext::dmpGetClassModule(DWORDLONG key, DWORDLONG value)
{
    printf("GetClassModule cls-%016llX, mod-%016llX", key, value);
}
CORINFO_MODULE_HANDLE MethodContext::repGetClassModule(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassModule, key, ": key %016llX", key);
    DWORDLONG value = GetClassModule->Get(key);
    DEBUG_REP(dmpGetClassModule(key, value));
    CORINFO_MODULE_HANDLE result = (CORINFO_MODULE_HANDLE)value;
    return result;
}

void MethodContext::recGetModuleAssembly(CORINFO_MODULE_HANDLE mod, CORINFO_ASSEMBLY_HANDLE assem)
{
    if (GetModuleAssembly == nullptr)
        GetModuleAssembly = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(mod);
    DWORDLONG value = CastHandle(assem);
    GetModuleAssembly->Add(key, value);
    DEBUG_REC(dmpGetModuleAssembly(key, value));
}
void MethodContext::dmpGetModuleAssembly(DWORDLONG key, DWORDLONG value)
{
    printf("GetModuleAssembly mod-%016llX, assem-%016llX", key, value);
}
CORINFO_ASSEMBLY_HANDLE MethodContext::repGetModuleAssembly(CORINFO_MODULE_HANDLE mod)
{
    DWORDLONG key = CastHandle(mod);
    AssertMapAndKeyExist(GetModuleAssembly, key, ": key %016llX", key);
    DWORDLONG value = GetModuleAssembly->Get(key);
    DEBUG_REP(dmpGetModuleAssembly(key, value));
    CORINFO_ASSEMBLY_HANDLE result = (CORINFO_ASSEMBLY_HANDLE)value;
    return result;
}

void MethodContext::recGetAssemblyName(CORINFO_ASSEMBLY_HANDLE assem, const char* assemblyName)
{
    if (GetAssemblyName == nullptr)
        GetAssemblyName = new LightWeightMap<DWORDLONG, DWORD>();

    DWORD value;
    if (assemblyName != nullptr)
    {
        value = GetAssemblyName->AddBuffer((const unsigned char*)assemblyName, (DWORD)strlen(assemblyName) + 1);
    }
    else
    {
        value = (DWORD)-1;
    }

    DWORDLONG key = CastHandle(assem);
    GetAssemblyName->Add(key, value);
    DEBUG_REC(dmpGetAssemblyName(key, value));
}
void MethodContext::dmpGetAssemblyName(DWORDLONG key, DWORD value)
{
    const char* assemblyName = (const char*)GetAssemblyName->GetBuffer(value);
    printf("GetAssemblyName assem-%016llX, value-%u '%s'", key, value, assemblyName);
    GetAssemblyName->Unlock();
}
const char* MethodContext::repGetAssemblyName(CORINFO_ASSEMBLY_HANDLE assem)
{
    DWORDLONG key = CastHandle(assem);
    const char* result = "hackishAssemblyName";
    DWORD value = (DWORD)-1;
    int itemIndex = -1;
    if (GetAssemblyName != nullptr)
    {
        itemIndex = GetAssemblyName->GetIndex(key);
    }
    if (itemIndex >= 0)
    {
        value = GetAssemblyName->Get(key);
        result = (const char*)GetAssemblyName->GetBuffer(value);
    }
    DEBUG_REP(dmpGetAssemblyName(key, value));
    return result;
}

// Note - the jit will call freearray on the array we give back....
void MethodContext::recGetVars(CORINFO_METHOD_HANDLE      ftn,
                               ULONG32*                   cVars,
                               ICorDebugInfo::ILVarInfo** vars_in,
                               bool*                      extendOthers)
{
    if (GetVars == nullptr)
        GetVars = new LightWeightMap<DWORDLONG, Agnostic_GetVars>();

    Agnostic_GetVars value;
    value.cVars = (DWORD)*cVars;
    value.vars_offset =
        (DWORD)GetVars->AddBuffer((unsigned char*)*vars_in, sizeof(ICorDebugInfo::ILVarInfo) * (*cVars));
    value.extendOthers = (DWORD)*extendOthers;

    DWORDLONG key = CastHandle(ftn);
    GetVars->Add(key, value);
    DEBUG_REC(dmpGetVars(key, value));
}
void MethodContext::dmpGetVars(DWORDLONG key, const Agnostic_GetVars& value)
{
    ICorDebugInfo::ILVarInfo* vars = (ICorDebugInfo::ILVarInfo*)GetVars->GetBuffer(value.vars_offset);
    printf("GetVars key ftn-%016llX, value cVars-%u extendOthers-%u (", key, value.cVars, value.extendOthers);
    for (unsigned int i = 0; i < value.cVars; i++)
        printf("(%u %u %u %u)", i, vars[i].startOffset, vars[i].endOffset, vars[i].varNumber);
    printf(")");
    GetVars->Unlock();
}
void MethodContext::repGetVars(CORINFO_METHOD_HANDLE      ftn,
                               ULONG32*                   cVars,
                               ICorDebugInfo::ILVarInfo** vars_in,
                               bool*                      extendOthers)
{
    if (GetVars == nullptr)
    {
        *cVars = 0;
        return;
    }

    DWORDLONG key = CastHandle(ftn);
    Agnostic_GetVars value = GetVars->Get(key);
    DEBUG_REP(dmpGetVars(key, value));

    *cVars = (ULONG32)value.cVars;
    if (*cVars > 0)
        *vars_in  = (ICorDebugInfo::ILVarInfo*)GetVars->GetBuffer(value.vars_offset);
    *extendOthers = value.extendOthers != 0;
}

// Note - the jit will call freearray on the array we give back....
void MethodContext::recGetBoundaries(CORINFO_METHOD_HANDLE         ftn,
                                     unsigned int*                 cILOffsets,
                                     uint32_t**                    pILOffsets,
                                     ICorDebugInfo::BoundaryTypes* implicitBoundaries)
{
    if (GetBoundaries == nullptr)
        GetBoundaries = new LightWeightMap<DWORDLONG, Agnostic_GetBoundaries>();

    Agnostic_GetBoundaries value;

    value.cILOffsets = (DWORD)*cILOffsets;
    value.pILOffset_offset =
        (DWORD)GetBoundaries->AddBuffer((unsigned char*)*pILOffsets, sizeof(DWORD) * (*cILOffsets));
    value.implicitBoundaries = *implicitBoundaries;

    DWORDLONG key = CastHandle(ftn);
    GetBoundaries->Add(key, value);
    DEBUG_REC(dmpGetBoundaries(key, value));
}
void MethodContext::dmpGetBoundaries(DWORDLONG key, const Agnostic_GetBoundaries& value)
{
    printf("GetBoundaries key ftn-%016llX, value cnt-%u imp-%u{", key, value.cILOffsets, value.implicitBoundaries);
    DWORD* bnd = (DWORD*)GetBoundaries->GetBuffer(value.pILOffset_offset);
    for (unsigned int i = 0; i < value.cILOffsets; i++)
    {
        printf("%u", bnd[i]);
        if (i < (value.cILOffsets + 1))
            printf(",");
    }
    GetBoundaries->Unlock();
    printf("}");
}
void MethodContext::repGetBoundaries(CORINFO_METHOD_HANDLE         ftn,
                                     unsigned int*                 cILOffsets,
                                     uint32_t**                    pILOffsets,
                                     ICorDebugInfo::BoundaryTypes* implicitBoundaries)
{
    DWORDLONG key = CastHandle(ftn);
    AssertMapAndKeyExist(GetBoundaries, key, ": key %016llX", key);

    Agnostic_GetBoundaries value = GetBoundaries->Get(key);
    DEBUG_REP(dmpGetBoundaries(key, value));

    *cILOffsets = (unsigned int)value.cILOffsets;
    if (*cILOffsets > 0)
        *pILOffsets    = (uint32_t*)GetBoundaries->GetBuffer(value.pILOffset_offset);
    *implicitBoundaries = (ICorDebugInfo::BoundaryTypes)value.implicitBoundaries;
}

void MethodContext::recInitClass(CORINFO_FIELD_HANDLE   field,
                                 CORINFO_METHOD_HANDLE  method,
                                 CORINFO_CONTEXT_HANDLE context,
                                 CorInfoInitClassResult result)
{
    if (InitClass == nullptr)
        InitClass = new LightWeightMap<Agnostic_InitClass, DWORD>();

    Agnostic_InitClass key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.field   = CastHandle(field);
    key.method  = CastHandle(method);
    key.context = CastHandle(context);

    DWORD value = (DWORD)result;
    InitClass->Add(key, value);
    DEBUG_REC(dmpInitClass(key, value));
}
void MethodContext::dmpInitClass(const Agnostic_InitClass& key, DWORD value)
{
    printf("InitClass key fld-%016llX meth-%016llX con-%016llX, value res-%u", key.field, key.method,
           key.context, value);
}
CorInfoInitClassResult MethodContext::repInitClass(CORINFO_FIELD_HANDLE   field,
                                                   CORINFO_METHOD_HANDLE  method,
                                                   CORINFO_CONTEXT_HANDLE context)
{
    Agnostic_InitClass key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.field   = CastHandle(field);
    key.method  = CastHandle(method);
    key.context = CastHandle(context);

    if ((InitClass == nullptr) || (InitClass->GetIndex(key) == -1))
    {
        // We could try additional inlines with stress modes, just reject them.
        return CORINFO_INITCLASS_DONT_INLINE;
    }

    DWORD value = InitClass->Get(key);
    DEBUG_REP(dmpInitClass(key, value));
    CorInfoInitClassResult result = (CorInfoInitClassResult)value;
    return result;
}

void MethodContext::recGetMethodName(CORINFO_METHOD_HANDLE ftn, char* methodname, const char** moduleName)
{
    if (GetMethodName == nullptr)
        GetMethodName = new LightWeightMap<DLD, DD>();
    DD  value;
    DLD key;
    key.A = CastHandle(ftn);
    key.B = (moduleName != nullptr);

    if (methodname != nullptr)
        value.A = GetMethodName->AddBuffer((unsigned char*)methodname, (DWORD)strlen(methodname) + 1);
    else
        value.A = (DWORD)-1;

    if ((moduleName != nullptr) && (*moduleName != nullptr))
        value.B = GetMethodName->AddBuffer((unsigned char*)*moduleName, (DWORD)strlen(*moduleName) + 1);
    else
        value.B = (DWORD)-1;

    GetMethodName->Add(key, value);
    DEBUG_REC(dmpGetMethodName(key, value));
}
void MethodContext::dmpGetMethodName(DLD key, DD value)
{
    unsigned char* methodName = (unsigned char*)GetMethodName->GetBuffer(value.A);
    unsigned char* moduleName = (unsigned char*)GetMethodName->GetBuffer(value.B);
    printf("GetMethodName key - ftn-%016llX modNonNull-%u, value meth-'%s', mod-'%s'", key.A, key.B, methodName,
           moduleName);
    GetMethodName->Unlock();
}

const char* MethodContext::repGetMethodName(CORINFO_METHOD_HANDLE ftn, const char** moduleName)
{
    const char* result = "hackishMethodName";
    DD          value;
    DLD         key;
    key.A = CastHandle(ftn);
    key.B = (moduleName != nullptr);

    int itemIndex = -1;
    if (GetMethodName != nullptr)
        itemIndex = GetMethodName->GetIndex(key);
    if (itemIndex < 0)
    {
        if (moduleName != nullptr)
            *moduleName = "hackishModuleName";
    }
    else
    {
        value = GetMethodName->Get(key);
        DEBUG_REP(dmpGetMethodName(key, value));

        if (moduleName != nullptr)
            *moduleName = (const char*)GetMethodName->GetBuffer(value.B);
        result          = (const char*)GetMethodName->GetBuffer(value.A);
    }
    return result;
}

void MethodContext::recGetMethodNameFromMetadata(CORINFO_METHOD_HANDLE ftn,
                                                 char*                 methodName,
                                                 const char**          className,
                                                 const char**          namespaceName,
                                                 const char**          enclosingClassName)
{
    if (GetMethodNameFromMetadata == nullptr)
        GetMethodNameFromMetadata = new LightWeightMap<Agnostic_CORINFO_METHODNAME_TOKENin, Agnostic_CORINFO_METHODNAME_TOKENout>();
    Agnostic_CORINFO_METHODNAME_TOKENout  value;
    Agnostic_CORINFO_METHODNAME_TOKENin key;
    key.ftn = CastHandle(ftn);
    key.className = (className != nullptr);
    key.namespaceName = (namespaceName != nullptr);
    key.enclosingClassName = (enclosingClassName != nullptr);

    if (methodName != nullptr)
        value.methodName = GetMethodNameFromMetadata->AddBuffer((unsigned char*)methodName, (DWORD)strlen(methodName) + 1);
    else
        value.methodName = (DWORD)-1;

    if ((className != nullptr) && (*className != nullptr))
        value.className = GetMethodNameFromMetadata->AddBuffer((unsigned char*)*className, (DWORD)strlen(*className) + 1);
    else
        value.className = (DWORD)-1;

    if ((namespaceName != nullptr) && (*namespaceName != nullptr))
        value.namespaceName =
            GetMethodNameFromMetadata->AddBuffer((unsigned char*)*namespaceName, (DWORD)strlen(*namespaceName) + 1);
    else
        value.namespaceName = (DWORD)-1;

    if ((enclosingClassName != nullptr) && (*enclosingClassName != nullptr))
        value.enclosingClassName =
            GetMethodNameFromMetadata->AddBuffer((unsigned char*)*enclosingClassName, (DWORD)strlen(*enclosingClassName) + 1);
    else
        value.enclosingClassName = (DWORD)-1;

    GetMethodNameFromMetadata->Add(key, value);
    DEBUG_REC(dmpGetMethodNameFromMetadata(key, value));
}

void MethodContext::dmpGetMethodNameFromMetadata(Agnostic_CORINFO_METHODNAME_TOKENin key, Agnostic_CORINFO_METHODNAME_TOKENout value)
{
    unsigned char* methodName    = (unsigned char*)GetMethodNameFromMetadata->GetBuffer(value.methodName);
    unsigned char* className     = (unsigned char*)GetMethodNameFromMetadata->GetBuffer(value.className);
    unsigned char* namespaceName = (unsigned char*)GetMethodNameFromMetadata->GetBuffer(value.namespaceName);
    unsigned char* enclosingClassName = (unsigned char*)GetMethodNameFromMetadata->GetBuffer(value.enclosingClassName);
    printf("GetMethodNameFromMetadata key - ftn-%016llX classNonNull-%u namespaceNonNull-%u enclosingClassNonNull-%u, value meth-'%s', "
           "class-'%s', namespace-'%s' enclosingClass-'%s'",
           key.ftn, key.className, key.namespaceName, key.enclosingClassName, methodName, className, namespaceName, enclosingClassName);
    GetMethodNameFromMetadata->Unlock();
}

const char* MethodContext::repGetMethodNameFromMetadata(CORINFO_METHOD_HANDLE ftn,
                                                        const char**          moduleName,
                                                        const char**          namespaceName,
                                                        const char**          enclosingClassName)
{
    const char* result = nullptr;
    Agnostic_CORINFO_METHODNAME_TOKENout value;
    Agnostic_CORINFO_METHODNAME_TOKENin key;
    key.ftn = CastHandle(ftn);
    key.className = (moduleName != nullptr);
    key.namespaceName = (namespaceName != nullptr);
    key.enclosingClassName = (enclosingClassName != nullptr);

    int itemIndex = -1;
    if (GetMethodNameFromMetadata != nullptr)
        itemIndex = GetMethodNameFromMetadata->GetIndex(key);
    if (itemIndex < 0)
    {
        if (moduleName != nullptr)
        {
            *moduleName = nullptr;
        }
    }
    else
    {
        value  = GetMethodNameFromMetadata->Get(key);
        DEBUG_REP(dmpGetMethodNameFromMetadata(key, value));

        result = (const char*)GetMethodNameFromMetadata->GetBuffer(value.methodName);

        if (moduleName != nullptr)
        {
            *moduleName = (const char*)GetMethodNameFromMetadata->GetBuffer(value.className);
        }

        if (namespaceName != nullptr)
        {
            *namespaceName = (const char*)GetMethodNameFromMetadata->GetBuffer(value.namespaceName);
        }

        if (enclosingClassName != nullptr)
        {
            *enclosingClassName = (const char*)GetMethodNameFromMetadata->GetBuffer(value.enclosingClassName);
        }
    }
    return result;
}

void MethodContext::recGetJitFlags(CORJIT_FLAGS* jitFlags, DWORD sizeInBytes, DWORD result)
{
    if (GetJitFlags == nullptr)
        GetJitFlags = new LightWeightMap<DWORD, DD>();

    DD value;
    value.A = (DWORD)GetJitFlags->AddBuffer((unsigned char*)jitFlags, sizeInBytes);
    value.B = result;

    // NOTE: getJitFlags() is expected to be idempotent per method, so the mapping key is always
    //       zero.
    GetJitFlags->Add(0, value);
    DEBUG_REC(dmpGetJitFlags(0, value));
    InitReadyToRunFlag(jitFlags);
}
void MethodContext::dmpGetJitFlags(DWORD key, DD value)
{
    CORJIT_FLAGS* jitflags = (CORJIT_FLAGS*)GetJitFlags->GetBuffer(value.A);
    printf("GetJitFlags key %u sizeInBytes-%u jitFlags-%016llX instructionSetFlags-%016llX", key, value.B, jitflags->GetFlagsRaw(), jitflags->GetInstructionSetFlagsRaw());
    GetJitFlags->Unlock();
}
DWORD MethodContext::repGetJitFlags(CORJIT_FLAGS* jitFlags, DWORD sizeInBytes)
{
    AssertMapAndKeyExistNoMessage(GetJitFlags, 0);

    DD value = GetJitFlags->Get(0);
    DEBUG_REP(dmpGetJitFlags(0, value));

    CORJIT_FLAGS* resultFlags = (CORJIT_FLAGS*)GetJitFlags->GetBuffer(value.A);
    memcpy(jitFlags, resultFlags, value.B);
    InitReadyToRunFlag(resultFlags);
    return value.B;
}

void MethodContext::recGetJitTimeLogFilename(LPCWSTR tempFileName)
{
    if (GetJitTimeLogFilename == nullptr)
        GetJitTimeLogFilename = new LightWeightMap<DWORD, DWORD>();

    DWORD name_index = -1;
    if (tempFileName != nullptr)
    {
        name_index = GetJitTimeLogFilename->AddBuffer((unsigned char*)tempFileName, (DWORD)wcslen(tempFileName) + 2);
    }
    GetJitTimeLogFilename->Add(0, name_index);
    DEBUG_REC(dmpGetJitTimeLogFilename(0, name_index));
}
void MethodContext::dmpGetJitTimeLogFilename(DWORD key, DWORD value)
{
    unsigned char* fileName = nullptr;
    if (value != 0)
        fileName = (unsigned char*)GetJitTimeLogFilename->GetBuffer(value);
    printf("GetJitTimeLogFilename key %u, value '%s'", key, fileName);
    GetJitTimeLogFilename->Unlock();
}
LPCWSTR MethodContext::repGetJitTimeLogFilename()
{
    AssertMapAndKeyExistNoMessage(GetJitTimeLogFilename, 0);

    DWORD offset = GetJitTimeLogFilename->Get(0);
    DEBUG_REP(dmpGetJitTimeLogFilename(0, offset));

    LPCWSTR value = nullptr;
    if (offset != 0)
        value = (LPCWSTR)GetJitTimeLogFilename->GetBuffer(offset);
    return value;
}

void MethodContext::recCanInline(CORINFO_METHOD_HANDLE callerHnd,
                                 CORINFO_METHOD_HANDLE calleeHnd,
                                 uint32_t*                pRestrictions,
                                 CorInfoInline         response,
                                 DWORD                 exceptionCode)
{
    if (CanInline == nullptr)
        CanInline = new LightWeightMap<DLDL, Agnostic_CanInline>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(callerHnd);
    key.B = CastHandle(calleeHnd);

    Agnostic_CanInline value;
    if (pRestrictions != nullptr)
        value.Restrictions = (DWORD)*pRestrictions;
    else
        value.Restrictions = (DWORD)0;
    value.result           = (DWORD)response;
    value.exceptionCode    = (DWORD)exceptionCode;

    CanInline->Add(key, value);
    DEBUG_REC(dmpCanInline(key, value));
}
void MethodContext::dmpCanInline(DLDL key, const Agnostic_CanInline& value)
{
    printf("CanInline key - callerHnd-%016llX calleeHnd-%016llX, value pRestrictions-%u result-%u exceptionCode-%08X",
           key.A, key.B, value.Restrictions, value.result, value.exceptionCode);
}
CorInfoInline MethodContext::repCanInline(CORINFO_METHOD_HANDLE callerHnd,
                                          CORINFO_METHOD_HANDLE calleeHnd,
                                          uint32_t*             pRestrictions,
                                          DWORD*                exceptionCode)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(callerHnd);
    key.B = CastHandle(calleeHnd);

    if ((CanInline == nullptr) || (CanInline->GetIndex(key) == -1))
    {
#ifdef sparseMC
        LogDebug("Sparse - repCanInline saying INLINE_FAIL");
        return INLINE_FAIL; // if we have no info, its pretty safe to say we can't inline it.
#else
        LogException(EXCEPTIONCODE_MC, "Didn't find %016llx, %016llx.  probably a missing exception in canInline",
                     key.A, key.B);
#endif
    }

    Agnostic_CanInline value = CanInline->Get(key);
    DEBUG_REP(dmpCanInline(key, value));

    *exceptionCode = value.exceptionCode;

    if (pRestrictions != nullptr)
        *pRestrictions     = (DWORD)value.Restrictions;
    CorInfoInline response = (CorInfoInline)value.result;
    return response;
}

void MethodContext::recResolveToken(CORINFO_RESOLVED_TOKEN* pResolvedToken, DWORD exceptionCode)
{
    if (ResolveToken == nullptr)
        ResolveToken = new LightWeightMap<Agnostic_CORINFO_RESOLVED_TOKENin, ResolveTokenValue>();

    Agnostic_CORINFO_RESOLVED_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key = SpmiRecordsHelper::CreateAgnostic_CORINFO_RESOLVED_TOKENin(pResolvedToken);

    ResolveTokenValue value;
    if (exceptionCode != ERROR_SUCCESS)
    {
        // The output token memory might be corrupt or uninitialized, so just zero it out.
        // (Set indexes to -1, indicating no buffer).
        ZeroMemory(&value.tokenOut, sizeof(value.tokenOut));
        value.tokenOut.pTypeSpec_Index   = (DWORD)-1;
        value.tokenOut.pMethodSpec_Index = (DWORD)-1;
    }
    else
    {
        value.tokenOut = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKENout(pResolvedToken, ResolveToken);
    }
    value.exceptionCode = exceptionCode;

    ResolveToken->Add(key, value);
    DEBUG_REC(dmpResolveToken(key, value));
}
void MethodContext::dmpResolveToken(const Agnostic_CORINFO_RESOLVED_TOKENin& key, const ResolveTokenValue& value)
{
    printf("ResolveToken key: %s, value: %s excp-%08X",
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKENin(key).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKENout(value.tokenOut).c_str(),
        value.exceptionCode);
}
void MethodContext::repResolveToken(CORINFO_RESOLVED_TOKEN* pResolvedToken, DWORD* exceptionCode)
{
    AssertMapExists(ResolveToken, ": key %x", pResolvedToken->token);

    Agnostic_CORINFO_RESOLVED_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key = SpmiRecordsHelper::CreateAgnostic_CORINFO_RESOLVED_TOKENin(pResolvedToken);

    AssertKeyExists(ResolveToken, key, ": token %x", pResolvedToken->token);

    ResolveTokenValue value = ResolveToken->Get(key);
    DEBUG_REP(dmpResolveToken(key, value));

    SpmiRecordsHelper::Restore_CORINFO_RESOLVED_TOKENout(pResolvedToken, value.tokenOut, ResolveToken);
    *exceptionCode = (DWORD)value.exceptionCode;
}

void MethodContext::recTryResolveToken(CORINFO_RESOLVED_TOKEN* pResolvedToken, bool success)
{
    if (TryResolveToken == nullptr)
        TryResolveToken = new LightWeightMap<Agnostic_CORINFO_RESOLVED_TOKENin, TryResolveTokenValue>();

    Agnostic_CORINFO_RESOLVED_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key = SpmiRecordsHelper::CreateAgnostic_CORINFO_RESOLVED_TOKENin(pResolvedToken);

    TryResolveTokenValue value;

    value.tokenOut = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKENout(pResolvedToken, ResolveToken);
    value.success  = success ? 0 : 1;

    TryResolveToken->Add(key, value);
    DEBUG_REC(dmpTryResolveToken(key, value));
}
void MethodContext::dmpTryResolveToken(const Agnostic_CORINFO_RESOLVED_TOKENin& key, const TryResolveTokenValue& value)
{
    printf("TryResolveToken key: %s\n", SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKENin(key).c_str());
    printf(", value: %s failed-%u", SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKENout(value.tokenOut).c_str(),
           value.success);
}
bool MethodContext::repTryResolveToken(CORINFO_RESOLVED_TOKEN* pResolvedToken)
{
    AssertMapExists(TryResolveToken, ": key %x", pResolvedToken->token);

    Agnostic_CORINFO_RESOLVED_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key = SpmiRecordsHelper::CreateAgnostic_CORINFO_RESOLVED_TOKENin(pResolvedToken);

    AssertKeyExists(TryResolveToken, key, ": token %x", pResolvedToken->token);

    TryResolveTokenValue value = TryResolveToken->Get(key);
    DEBUG_REP(dmpTryResolveToken(key, value));

    SpmiRecordsHelper::Restore_CORINFO_RESOLVED_TOKENout(pResolvedToken, value.tokenOut, ResolveToken);
    return (DWORD)value.success == 0; // recTryResolveToken encodes success as 0
}

void MethodContext::recGetCallInfo(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                   CORINFO_RESOLVED_TOKEN* pConstrainedResolvedToken,
                                   CORINFO_METHOD_HANDLE   callerHandle,
                                   CORINFO_CALLINFO_FLAGS  flags,
                                   CORINFO_CALL_INFO*      pResult,
                                   DWORD                   exceptionCode)
{
    if (GetCallInfo == nullptr)
        GetCallInfo = new LightWeightMap<Agnostic_GetCallInfo, Agnostic_CORINFO_CALL_INFO>();

    Agnostic_GetCallInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetCallInfo);

    if (pConstrainedResolvedToken != nullptr)
    {
        key.ConstrainedResolvedToken =
            SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pConstrainedResolvedToken, GetCallInfo);
    }

    key.callerHandle = CastHandle(callerHandle);
    key.flags        = (DWORD)flags;

    Agnostic_CORINFO_CALL_INFO value;
    ZeroMemory(&value, sizeof(Agnostic_CORINFO_CALL_INFO)); // init verSig with 0.

    if (exceptionCode == 0)
    {
        value.hMethod     = CastHandle(pResult->hMethod);
        value.methodFlags = (DWORD)pResult->methodFlags;
        value.classFlags  = (DWORD)pResult->classFlags;
        value.sig         = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(pResult->sig, GetCallInfo, SigInstHandleMap);
        if (flags & CORINFO_CALLINFO_VERIFICATION)
        {
            value.verMethodFlags = (DWORD)pResult->verMethodFlags;
            value.verSig         = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(pResult->verSig, GetCallInfo, SigInstHandleMap);
        }

        value.accessAllowed                   = (DWORD)pResult->accessAllowed;
        value.callsiteCalloutHelper.helperNum = (DWORD)pResult->callsiteCalloutHelper.helperNum;
        value.callsiteCalloutHelper.numArgs   = (DWORD)pResult->callsiteCalloutHelper.numArgs;
        for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
        {
            value.callsiteCalloutHelper.args[i].constant = (DWORDLONG)pResult->callsiteCalloutHelper.args[i].constant;
            value.callsiteCalloutHelper.args[i].argType  = (DWORD)pResult->callsiteCalloutHelper.args[i].argType;
        }
        value.thisTransform = (DWORD)pResult->thisTransform;

        value.kind                           = (DWORD)pResult->kind;
        value.nullInstanceCheck              = (DWORD)pResult->nullInstanceCheck;
        value.contextHandle                  = CastHandle(pResult->contextHandle);
        value.exactContextNeedsRuntimeLookup = (DWORD)pResult->exactContextNeedsRuntimeLookup;

        value.stubLookup = SpmiRecordsHelper::StoreAgnostic_CORINFO_LOOKUP(&pResult->stubLookup);

        value.instParamLookup.accessType = (DWORD)pResult->instParamLookup.accessType;
        value.instParamLookup.handle     = CastHandle(pResult->instParamLookup.handle);
        value.wrapperDelegateInvoke      = (DWORD)pResult->wrapperDelegateInvoke;
    }

    value.exceptionCode = (DWORD)exceptionCode;

    GetCallInfo->Add(key, value);
    DEBUG_REC(dmpGetCallInfo(key, value));
}

void MethodContext::dmpGetCallInfo(const Agnostic_GetCallInfo& key, const Agnostic_CORINFO_CALL_INFO& value)
{
    printf("GetCallInfo key rt{%s} crt{%s} ch-%016llX flg-%08X"
        ", value mth-%016llX, mf-%08X (%s) cf-%08X (%s)"
        " sig-%s"
        " vmf-%08X (%s)"
        " vsig-%s"
        " aa-%u"
        " cch{hn-%u na-%u (TODO: dump callsiteCalloutHelper.args)}"
        " tt-%u"
        " k-%u (%s)"
        " nic-%u (%s)"
        " ch-%016llX"
        " ecnrl-%u (%s)"
        " stubLookup{%s}"
        " ipl{at-%08X (%s) hnd-%016llX}"
        " wdi-%u (%s)"
        " excp-%08X",
        // input
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ResolvedToken).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ConstrainedResolvedToken).c_str(),
        key.callerHandle,
        key.flags,
        // output
        value.hMethod,
        value.methodFlags,
        SpmiDumpHelper::DumpCorInfoFlag((CorInfoFlag)value.methodFlags).c_str(),
        value.classFlags,
        SpmiDumpHelper::DumpCorInfoFlag((CorInfoFlag)value.classFlags).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.sig, GetCallInfo, SigInstHandleMap).c_str(),
        value.verMethodFlags,
        SpmiDumpHelper::DumpCorInfoFlag((CorInfoFlag)value.verMethodFlags).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.verSig, GetCallInfo, SigInstHandleMap).c_str(),
        value.accessAllowed,
        value.callsiteCalloutHelper.helperNum,
        value.callsiteCalloutHelper.numArgs,
        value.thisTransform,
        value.kind,
        toString((CORINFO_CALL_KIND)value.kind),
        value.nullInstanceCheck,
        (bool)value.nullInstanceCheck ? "true" : "false",
        value.contextHandle,
        value.exactContextNeedsRuntimeLookup,
        (bool)value.exactContextNeedsRuntimeLookup ? "true" : "false",
        SpmiDumpHelper::DumpAgnostic_CORINFO_LOOKUP(value.stubLookup).c_str(),
        value.instParamLookup.accessType,
        toString((InfoAccessType)value.instParamLookup.accessType),
        value.instParamLookup.handle,
        value.wrapperDelegateInvoke,
        (bool)value.wrapperDelegateInvoke ? "true" : "false",
        value.exceptionCode);
}
void MethodContext::repGetCallInfo(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                   CORINFO_RESOLVED_TOKEN* pConstrainedResolvedToken,
                                   CORINFO_METHOD_HANDLE   callerHandle,
                                   CORINFO_CALLINFO_FLAGS  flags,
                                   CORINFO_CALL_INFO*      pResult,
                                   DWORD*                  exceptionCode)
{
    AssertMapExistsNoMessage(GetCallInfo);

    Agnostic_GetCallInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetCallInfo);
    if (pConstrainedResolvedToken != nullptr)
    {
        key.ConstrainedResolvedToken =
            SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pConstrainedResolvedToken, GetCallInfo);
    }
    key.callerHandle = CastHandle(callerHandle);
    key.flags        = (DWORD)flags;

    AssertKeyExists(GetCallInfo, key, ": key %08x, %016llx",
                    key.ResolvedToken.inValue.token, key.ResolvedToken.outValue.hClass);

    Agnostic_CORINFO_CALL_INFO value = GetCallInfo->Get(key);

    if (value.exceptionCode != 0)
    {
        // If there was an exception, the stored data is ignored.
        ZeroMemory(pResult, sizeof(*pResult));
    }
    else
    {
        pResult->hMethod = (CORINFO_METHOD_HANDLE)value.hMethod;
        pResult->methodFlags = (unsigned)value.methodFlags;
        pResult->classFlags = (unsigned)value.classFlags;
        pResult->sig = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.sig, GetCallInfo, SigInstHandleMap);
        if (flags & CORINFO_CALLINFO_VERIFICATION)
        {
            pResult->verMethodFlags = (unsigned)value.verMethodFlags;
            pResult->verSig = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.verSig, GetCallInfo, SigInstHandleMap);
        }
        pResult->accessAllowed = (CorInfoIsAccessAllowedResult)value.accessAllowed;
        pResult->callsiteCalloutHelper.helperNum = (CorInfoHelpFunc)value.callsiteCalloutHelper.helperNum;
        pResult->callsiteCalloutHelper.numArgs = (unsigned)value.callsiteCalloutHelper.numArgs;
        for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
        {
            pResult->callsiteCalloutHelper.args[i].constant = (size_t)value.callsiteCalloutHelper.args[i].constant;
            pResult->callsiteCalloutHelper.args[i].argType =
                (CorInfoAccessAllowedHelperArgType)value.callsiteCalloutHelper.args[i].argType;
        }
        pResult->thisTransform = (CORINFO_THIS_TRANSFORM)value.thisTransform;
        pResult->kind = (CORINFO_CALL_KIND)value.kind;
        pResult->nullInstanceCheck = (bool)value.nullInstanceCheck;
        pResult->contextHandle = (CORINFO_CONTEXT_HANDLE)value.contextHandle;
        pResult->exactContextNeedsRuntimeLookup = (bool)value.exactContextNeedsRuntimeLookup;
        pResult->stubLookup.lookupKind.needsRuntimeLookup = value.stubLookup.lookupKind.needsRuntimeLookup != 0;
        pResult->stubLookup.lookupKind.runtimeLookupKind =
            (CORINFO_RUNTIME_LOOKUP_KIND)value.stubLookup.lookupKind.runtimeLookupKind;
        if (pResult->stubLookup.lookupKind.needsRuntimeLookup)
        {
            pResult->stubLookup.runtimeLookup = SpmiRecordsHelper::RestoreCORINFO_RUNTIME_LOOKUP(value.stubLookup.runtimeLookup);
        }
        else
        {
            pResult->stubLookup.constLookup.accessType = (InfoAccessType)value.stubLookup.constLookup.accessType;
            pResult->stubLookup.constLookup.handle = (CORINFO_GENERIC_HANDLE)value.stubLookup.constLookup.handle;
        }
        if (pResult->kind == CORINFO_VIRTUALCALL_STUB)
        {
            cr->CallTargetTypes->Add(CastPointer(pResult->codePointerLookup.constLookup.addr),
                (DWORD)CORINFO_VIRTUALCALL_STUB);
        }
        pResult->instParamLookup.accessType = (InfoAccessType)value.instParamLookup.accessType;
        pResult->instParamLookup.handle = (CORINFO_GENERIC_HANDLE)value.instParamLookup.handle;
        pResult->wrapperDelegateInvoke = (bool)value.wrapperDelegateInvoke;
    }

    *exceptionCode = (DWORD)value.exceptionCode;

    DEBUG_REP(dmpGetCallInfo(key, value));
}

//
// Variant of repGetCallInfo that only requires a method handle, i.e. it performs a reverse lookup to find the
// resolved token info that, along with the given method handle, was passed into getCallInfo.
//
// Arguments:
//    methodHandle - The method handle to find call info for.
//    pResult      - [out] The call info for the given method.
//
// Notes:
//    If this fails to find a recorded call to getCallInfo with the given method handle, this will throw an
//    exception.
//
void MethodContext::repGetCallInfoFromMethodHandle(CORINFO_METHOD_HANDLE methodHandle, CORINFO_CALL_INFO* pResult)
{
    if (GetCallInfo != nullptr)
    {
        for (unsigned int i = 0; i < GetCallInfo->GetCount(); i++)
        {
            Agnostic_GetCallInfo       key = GetCallInfo->GetKey(i);
            Agnostic_CORINFO_CALL_INFO val = GetCallInfo->GetItem(i);

            if ((CORINFO_METHOD_HANDLE)val.hMethod == methodHandle)
            {
                CORINFO_RESOLVED_TOKEN resolvedToken;
                DWORD                  exceptionCode;

                resolvedToken.tokenContext = (CORINFO_CONTEXT_HANDLE)key.ResolvedToken.inValue.tokenContext;
                resolvedToken.tokenScope   = (CORINFO_MODULE_HANDLE)key.ResolvedToken.inValue.tokenScope;
                resolvedToken.token        = (mdToken)key.ResolvedToken.inValue.token;
                resolvedToken.tokenType    = (CorInfoTokenKind)key.ResolvedToken.inValue.tokenType;

                repResolveToken(&resolvedToken, &exceptionCode);

                // If the original call to getCallInfo passed in a null constrainedResolvedToken pointer,
                // then we won't be able to replay it. In that case, we'll need to pass a null pointer into
                // repGetCallInfo for constrainedResolvedToken, instead of just passing the address of our
                // local (but meaningless) constrainedResolvedToken struct.
                CORINFO_RESOLVED_TOKEN  constrainedResolvedToken;
                CORINFO_RESOLVED_TOKEN* pConstrainedResolvedToken = nullptr;

                if (key.ConstrainedResolvedToken.inValue.tokenContext != 0 &&
                    key.ConstrainedResolvedToken.inValue.tokenScope != 0)
                {
                    constrainedResolvedToken.tokenContext =
                        (CORINFO_CONTEXT_HANDLE)key.ConstrainedResolvedToken.inValue.tokenContext;
                    constrainedResolvedToken.tokenScope =
                        (CORINFO_MODULE_HANDLE)key.ConstrainedResolvedToken.inValue.tokenScope;
                    constrainedResolvedToken.token = (mdToken)key.ConstrainedResolvedToken.inValue.token;
                    constrainedResolvedToken.tokenType =
                        (CorInfoTokenKind)key.ConstrainedResolvedToken.inValue.tokenType;
                    pConstrainedResolvedToken = &constrainedResolvedToken;

                    repResolveToken(pConstrainedResolvedToken, &exceptionCode);
                }

                repGetCallInfo(&resolvedToken, pConstrainedResolvedToken, (CORINFO_METHOD_HANDLE)key.callerHandle,
                               (CORINFO_CALLINFO_FLAGS)key.flags, pResult, &exceptionCode);
                return;
            }
        }
    }

    // If we reached here, we didn't find a key associated with the given method handle
    LogException(EXCEPTIONCODE_MC, "Didn't find key %016llX.", methodHandle);
}

void MethodContext::recGetIntrinsicID(CORINFO_METHOD_HANDLE method, bool* pMustExpand, CorInfoIntrinsics result)
{
    if (GetIntrinsicID == nullptr)
        GetIntrinsicID = new LightWeightMap<DWORDLONG, DD>();

    DD value;
    value.A = (pMustExpand != nullptr) ? (DWORD)(*pMustExpand ? 1 : 0) : (DWORD)0;
    value.B = (DWORD)result;

    DWORDLONG key = CastHandle(method);
    GetIntrinsicID->Add(key, value);
    DEBUG_REC(dmpGetIntrinsicID(key, value));
}
void MethodContext::dmpGetIntrinsicID(DWORDLONG key, DD value)
{
    printf("GetIntrinsicID key mth-%016llX, mustExpand-%u, value intr-%u", key, value.A, value.B);
}
CorInfoIntrinsics MethodContext::repGetIntrinsicID(CORINFO_METHOD_HANDLE method, bool* pMustExpand)
{
    DWORDLONG key = CastHandle(method);
    AssertMapAndKeyExist(GetIntrinsicID, key, ": key %016llX", key);

    DD value = GetIntrinsicID->Get(key);
    DEBUG_REP(dmpGetIntrinsicID(key, value));

    if (pMustExpand != nullptr)
    {
        *pMustExpand = (value.A == 0) ? false : true;
    }
    CorInfoIntrinsics result = (CorInfoIntrinsics)value.B;
    return result;
}

void MethodContext::recIsIntrinsicType(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (IsIntrinsicType == nullptr)
        IsIntrinsicType = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    IsIntrinsicType->Add(key, value);
    DEBUG_REC(dmpIsIntrinsicType(key, value));
}
void MethodContext::dmpIsIntrinsicType(DWORDLONG key, DWORD value)
{
    printf("IsIntrinsicType key mth-%016llX, value intr-%u", key, value);
}
bool MethodContext::repIsIntrinsicType(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(IsIntrinsicType, key, ": key %016llX", key);
    DWORD value = IsIntrinsicType->Get(key);
    DEBUG_REP(dmpIsIntrinsicType(key, value));
    return value != 0;
}

void MethodContext::recAsCorInfoType(CORINFO_CLASS_HANDLE cls, CorInfoType result)
{
    if (AsCorInfoType == nullptr)
        AsCorInfoType = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    AsCorInfoType->Add(key, value);
    DEBUG_REC(dmpAsCorInfoType(key, value));
}
void MethodContext::dmpAsCorInfoType(DWORDLONG key, DWORD value)
{
    printf("AsCorInfoType key cls-%016llX, value cit-%u(%s)", key, value, toString((CorInfoType)value));
}
CorInfoType MethodContext::repAsCorInfoType(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(AsCorInfoType, key, ": key %016llX", key);
    DWORD value = AsCorInfoType->Get(key);
    DEBUG_REP(dmpAsCorInfoType(key, value));
    CorInfoType result = (CorInfoType)value;
    return result;
}

void MethodContext::recIsValueClass(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (IsValueClass == nullptr)
        IsValueClass = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    IsValueClass->Add(key, value);
    DEBUG_REC(dmpIsValueClass(key, value));
}
void MethodContext::dmpIsValueClass(DWORDLONG key, DWORD value)
{
    printf("IsValueClass key cls-%016llX, value res-%u", key, value);
}
bool MethodContext::repIsValueClass(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(IsValueClass, key, ": key %016llX", key);
    DWORD value = IsValueClass->Get(key);
    DEBUG_REP(dmpIsValueClass(key, value));
    return value != 0;
}

void MethodContext::recIsStructRequiringStackAllocRetBuf(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (IsStructRequiringStackAllocRetBuf == nullptr)
        IsStructRequiringStackAllocRetBuf = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    IsStructRequiringStackAllocRetBuf->Add(key, value);
    DEBUG_REC(dmpIsStructRequiringStackAllocRetBuf(key, value));
}
void MethodContext::dmpIsStructRequiringStackAllocRetBuf(DWORDLONG key, DWORD value)
{
    printf("IsStructRequiringStackAllocRetBuf key cls-%016llX, value res-%u", key, value);
}
bool MethodContext::repIsStructRequiringStackAllocRetBuf(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(IsStructRequiringStackAllocRetBuf, key, ": key %016llX", key);
    DWORD value = IsStructRequiringStackAllocRetBuf->Get(key);
    DEBUG_REP(dmpIsStructRequiringStackAllocRetBuf(key, value));
    return value != 0;
}

void MethodContext::recGetClassSize(CORINFO_CLASS_HANDLE cls, unsigned result)
{
    if (GetClassSize == nullptr)
        GetClassSize = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetClassSize->Add(key, value);
    DEBUG_REC(dmpGetClassSize(key, value));
}
void MethodContext::dmpGetClassSize(DWORDLONG key, DWORD val)
{
    printf("GetClassSize key %016llX, value %u", key, val);
}
unsigned MethodContext::repGetClassSize(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassSize, key, ": key %016llX", key);
    DWORD value = GetClassSize->Get(key);
    DEBUG_REP(dmpGetClassSize(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recGetHeapClassSize(CORINFO_CLASS_HANDLE cls, unsigned result)
{
    if (GetHeapClassSize == nullptr)
        GetHeapClassSize = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetHeapClassSize->Add(key, value);
    DEBUG_REC(dmpGetHeapClassSize(key, value));
}
void MethodContext::dmpGetHeapClassSize(DWORDLONG key, DWORD val)
{
    printf("GetHeapClassSize key %016llX, value %u", key, val);
}
unsigned MethodContext::repGetHeapClassSize(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetHeapClassSize, key, ": key %016llX", key);
    DWORD value = GetHeapClassSize->Get(key);
    DEBUG_REP(dmpGetHeapClassSize(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recCanAllocateOnStack(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (CanAllocateOnStack == nullptr)
        CanAllocateOnStack = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    CanAllocateOnStack->Add(key, value);
    DEBUG_REC(dmpCanAllocateOnStack(key, value));
}
void MethodContext::dmpCanAllocateOnStack(DWORDLONG key, DWORD val)
{
    printf("CanAllocateOnStack key %016llX, value %u", key, val);
}
bool MethodContext::repCanAllocateOnStack(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(CanAllocateOnStack, key, ": key %016llX", key);
    DWORD value = CanAllocateOnStack->Get(key);
    DEBUG_REP(dmpCanAllocateOnStack(key, value));
    return value != 0;
}

void MethodContext::recGetClassNumInstanceFields(CORINFO_CLASS_HANDLE cls, unsigned result)
{
    if (GetClassNumInstanceFields == nullptr)
        GetClassNumInstanceFields = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetClassNumInstanceFields->Add(key, value);
    DEBUG_REC(dmpGetClassNumInstanceFields(key, value));
}
void MethodContext::dmpGetClassNumInstanceFields(DWORDLONG key, DWORD value)
{
    printf("GetClassNumInstanceFields key cls-%016llX, value res-%u", key, value);
}
unsigned MethodContext::repGetClassNumInstanceFields(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassNumInstanceFields, key, ": key %016llX", key);
    DWORD value = GetClassNumInstanceFields->Get(key);
    DEBUG_REP(dmpGetClassNumInstanceFields(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recGetNewArrHelper(CORINFO_CLASS_HANDLE arrayCls, CorInfoHelpFunc result)
{
    if (GetNewArrHelper == nullptr)
        GetNewArrHelper = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(arrayCls);
    DWORD value = (DWORD)result;
    GetNewArrHelper->Add(key, value);
    DEBUG_REC(dmpGetNewArrHelper(key, value));
}
void MethodContext::dmpGetNewArrHelper(DWORDLONG key, DWORD value)
{
    printf("GetNewArrHelper key cls-%016llX, value res-%u", key, value);
}
CorInfoHelpFunc MethodContext::repGetNewArrHelper(CORINFO_CLASS_HANDLE arrayCls)
{
    DWORDLONG key = CastHandle(arrayCls);
    AssertMapAndKeyExist(GetNewArrHelper, key, ": key %016llX", key);
    DWORD value = GetNewArrHelper->Get(key);
    DEBUG_REP(dmpGetNewArrHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recGetSharedCCtorHelper(CORINFO_CLASS_HANDLE clsHnd, CorInfoHelpFunc result)
{
    if (GetSharedCCtorHelper == nullptr)
        GetSharedCCtorHelper = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(clsHnd);
    DWORD value = (DWORD)result;
    GetSharedCCtorHelper->Add(key, value);
    DEBUG_REC(dmpGetSharedCCtorHelper(key, value));
}
void MethodContext::dmpGetSharedCCtorHelper(DWORDLONG key, DWORD value)
{
    printf("GetSharedCCtorHelper key cls-%016llX, value res-%u", key, value);
}
CorInfoHelpFunc MethodContext::repGetSharedCCtorHelper(CORINFO_CLASS_HANDLE clsHnd)
{
    DWORDLONG key = CastHandle(clsHnd);
    AssertMapAndKeyExist(GetSharedCCtorHelper, key, ": key %016llX", key);
    DWORD value = GetSharedCCtorHelper->Get(key);
    DEBUG_REP(dmpGetSharedCCtorHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recGetTypeForBox(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result)
{
    if (GetTypeForBox == nullptr)
        GetTypeForBox = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(result);
    GetTypeForBox->Add(key, value);
    DEBUG_REC(dmpGetTypeForBox(key, value));
}
void MethodContext::dmpGetTypeForBox(DWORDLONG key, DWORDLONG value)
{
    printf("GetTypeForBox key cls-%016llX, value res-%016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetTypeForBox(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetTypeForBox, key, ": key %016llX", key);
    DWORDLONG value = GetTypeForBox->Get(key);
    DEBUG_REP(dmpGetTypeForBox(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetBoxHelper(CORINFO_CLASS_HANDLE cls, CorInfoHelpFunc result)
{
    if (GetBoxHelper == nullptr)
        GetBoxHelper = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetBoxHelper->Add(key, value);
    DEBUG_REC(dmpGetBoxHelper(key, value));
}
void MethodContext::dmpGetBoxHelper(DWORDLONG key, DWORD value)
{
    printf("GetBoxHelper key cls-%016llX, value res-%u", key, value);
}
CorInfoHelpFunc MethodContext::repGetBoxHelper(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetBoxHelper, key, ": key %016llX", key);
    DWORD value = GetBoxHelper->Get(key);
    DEBUG_REP(dmpGetBoxHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recGetBuiltinClass(CorInfoClassId classId, CORINFO_CLASS_HANDLE result)
{
    if (GetBuiltinClass == nullptr)
        GetBuiltinClass = new LightWeightMap<DWORD, DWORDLONG>();

    DWORD key = (DWORD)classId;
    DWORDLONG value = CastHandle(result);
    GetBuiltinClass->Add(key, value);
    DEBUG_REC(dmpGetBuiltinClass(key, value));
}
void MethodContext::dmpGetBuiltinClass(DWORD key, DWORDLONG value)
{
    printf("GetBuiltinClass key cls-%08X, value cls-%016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetBuiltinClass(CorInfoClassId classId)
{
    DWORD key = (DWORD)classId;
    AssertMapAndKeyExist(GetBuiltinClass, key, ": key %08X", key);
    DWORDLONG value = GetBuiltinClass->Get(key);
    DEBUG_REP(dmpGetBuiltinClass(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetTypeForPrimitiveValueClass(CORINFO_CLASS_HANDLE cls, CorInfoType result)
{
    if (GetTypeForPrimitiveValueClass == nullptr)
        GetTypeForPrimitiveValueClass = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetTypeForPrimitiveValueClass->Add(key, value);
    DEBUG_REC(dmpGetTypeForPrimitiveValueClass(key, value));
}
void MethodContext::dmpGetTypeForPrimitiveValueClass(DWORDLONG key, DWORD value)
{
    printf("GetTypeForPrimitiveValueClass key cls-%016llX, value cit-%u(%s)", key, value, toString((CorInfoType)value));
}
CorInfoType MethodContext::repGetTypeForPrimitiveValueClass(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetTypeForPrimitiveValueClass, key, ": key %016llX", key);
    DWORD value = GetTypeForPrimitiveValueClass->Get(key);
    DEBUG_REP(dmpGetTypeForPrimitiveValueClass(key, value));
    CorInfoType result = (CorInfoType)value;
    return result;
}

void MethodContext::recGetTypeForPrimitiveNumericClass(CORINFO_CLASS_HANDLE cls, CorInfoType result)
{
    if (GetTypeForPrimitiveNumericClass == nullptr)
        GetTypeForPrimitiveNumericClass = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetTypeForPrimitiveNumericClass->Add(key, value);
    DEBUG_REC(dmpGetTypeForPrimitiveNumericClass(key, value));
}
void MethodContext::dmpGetTypeForPrimitiveNumericClass(DWORDLONG key, DWORD value)
{
    printf("GetTypeForPrimitiveNumericClass key cls-%016llX, value cit-%u(%s)", key, value,
           toString((CorInfoType)value));
}
CorInfoType MethodContext::repGetTypeForPrimitiveNumericClass(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetTypeForPrimitiveNumericClass, key, ": key %016llX", key);
    DWORD value = GetTypeForPrimitiveNumericClass->Get(key);
    DEBUG_REP(dmpGetTypeForPrimitiveNumericClass(key, value));
    CorInfoType result = (CorInfoType)value;
    return result;
}

void MethodContext::recGetParentType(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result)
{
    if (GetParentType == nullptr)
        GetParentType = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(result);
    GetParentType->Add(key, value);
    DEBUG_REC(dmpGetParentType(key, value));
}
void MethodContext::dmpGetParentType(DWORDLONG key, DWORDLONG value)
{
    printf("GetParentType key cls-%016llX, value cls-%016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetParentType(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetParentType, key, ": key %016llX", key);
    DWORDLONG value = GetParentType->Get(key);
    DEBUG_REP(dmpGetParentType(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recIsSDArray(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (IsSDArray == nullptr)
        IsSDArray = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    IsSDArray->Add(key, value);
    DEBUG_REC(dmpIsSDArray(key, value));
}
void MethodContext::dmpIsSDArray(DWORDLONG key, DWORD value)
{
    printf("IsSDArray key cls-%016llX, value res-%u", key, value);
}
bool MethodContext::repIsSDArray(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(IsSDArray, key, ": key %016llX", key);
    DWORD value = IsSDArray->Get(key);
    DEBUG_REP(dmpIsSDArray(key, value));
    return value != 0;
}

void MethodContext::recGetFieldClass(CORINFO_FIELD_HANDLE field, CORINFO_CLASS_HANDLE result)
{
    if (GetFieldClass == nullptr)
        GetFieldClass = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(field);
    DWORDLONG value = CastHandle(result);
    GetFieldClass->Add(key, value);
    DEBUG_REC(dmpGetFieldClass(key, value));
}
void MethodContext::dmpGetFieldClass(DWORDLONG key, DWORDLONG value)
{
    printf("GetFieldClass key %016llX, value %016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetFieldClass(CORINFO_FIELD_HANDLE field)
{
    DWORDLONG key = CastHandle(field);
    AssertMapAndKeyExist(GetFieldClass, key, ": key %016llX", key);
    DWORDLONG value = GetFieldClass->Get(key);
    DEBUG_REP(dmpGetFieldClass(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetFieldOffset(CORINFO_FIELD_HANDLE field, unsigned result)
{
    if (GetFieldOffset == nullptr)
        GetFieldOffset = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(field);
    DWORD value = (DWORD)result;
    GetFieldOffset->Add(key, value);
    DEBUG_REC(dmpGetFieldOffset(key, value));
}
void MethodContext::dmpGetFieldOffset(DWORDLONG key, DWORD value)
{
    printf("GetFieldOffset key FLD-%016llX, value %08X", key, value);
}
unsigned MethodContext::repGetFieldOffset(CORINFO_FIELD_HANDLE field)
{
    DWORDLONG key = CastHandle(field);
    AssertMapAndKeyExist(GetFieldOffset, key, ": key %016llX", key);
    DWORD value = GetFieldOffset->Get(key);
    DEBUG_REP(dmpGetFieldOffset(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recGetLazyStringLiteralHelper(CORINFO_MODULE_HANDLE handle, CorInfoHelpFunc result)
{
    if (GetLazyStringLiteralHelper == nullptr)
        GetLazyStringLiteralHelper = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(handle);
    DWORD value = (DWORD)result;
    GetLazyStringLiteralHelper->Add(key, value);
    DEBUG_REC(dmpGetLazyStringLiteralHelper(key, value));
}
void MethodContext::dmpGetLazyStringLiteralHelper(DWORDLONG key, DWORD value)
{
    printf("GetLazyStringLiteralHelper key mod-%016llX, value res-%u", key, value);
}
CorInfoHelpFunc MethodContext::repGetLazyStringLiteralHelper(CORINFO_MODULE_HANDLE handle)
{
    DWORDLONG key = CastHandle(handle);
    AssertMapAndKeyExist(GetLazyStringLiteralHelper, key, ": key %016llX", key);
    DWORD value = GetLazyStringLiteralHelper->Get(key);
    DEBUG_REP(dmpGetLazyStringLiteralHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recGetUnBoxHelper(CORINFO_CLASS_HANDLE cls, CorInfoHelpFunc result)
{
    if (GetUnBoxHelper == nullptr)
        GetUnBoxHelper = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetUnBoxHelper->Add(key, value);
    DEBUG_REC(dmpGetUnBoxHelper(key, value));
}
void MethodContext::dmpGetUnBoxHelper(DWORDLONG key, DWORD value)
{
    printf("GetUnBoxHelper key cls-%016llX, value res-%u", key, value);
}
CorInfoHelpFunc MethodContext::repGetUnBoxHelper(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetUnBoxHelper, key, ": key %016llX", key);
    DWORD value = GetUnBoxHelper->Get(key);
    DEBUG_REP(dmpGetUnBoxHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recGetReadyToRunHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                           CORINFO_LOOKUP_KIND*    pGenericLookupKind,
                                           CorInfoHelpFunc         id,
                                           CORINFO_CONST_LOOKUP*   pLookup,
                                           bool                    result)
{
    if (GetReadyToRunHelper == nullptr)
        GetReadyToRunHelper = new LightWeightMap<GetReadyToRunHelper_TOKENin, GetReadyToRunHelper_TOKENout>();

    GetReadyToRunHelper_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetReadyToRunHelper);
    key.GenericLookupKind = SpmiRecordsHelper::CreateAgnostic_CORINFO_LOOKUP_KIND(pGenericLookupKind);
    key.id                = (DWORD)id;
    GetReadyToRunHelper_TOKENout value;
    value.Lookup = SpmiRecordsHelper::StoreAgnostic_CORINFO_CONST_LOOKUP(pLookup);
    value.result = result;

    GetReadyToRunHelper->Add(key, value);
    DEBUG_REC(dmpGetReadyToRunHelper(key, value));
}

void MethodContext::dmpGetReadyToRunHelper(GetReadyToRunHelper_TOKENin key, GetReadyToRunHelper_TOKENout value)
{
    printf("GetReadyToRunHelper key: tk{%s} kind{%s} id-%u",
           SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ResolvedToken).c_str(),
           SpmiDumpHelper::DumpAgnostic_CORINFO_LOOKUP_KIND(key.GenericLookupKind).c_str(), key.id);
    printf(", value: lk{ %s } %u", SpmiDumpHelper::DumpAgnostic_CORINFO_CONST_LOOKUP(value.Lookup).c_str(),
           value.result);
}

bool MethodContext::repGetReadyToRunHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                           CORINFO_LOOKUP_KIND*    pGenericLookupKind,
                                           CorInfoHelpFunc         id,
                                           CORINFO_CONST_LOOKUP*   pLookup)
{
    AssertMapExistsNoMessage(GetReadyToRunHelper);

    GetReadyToRunHelper_TOKENin key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken     = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetReadyToRunHelper);
    key.GenericLookupKind = SpmiRecordsHelper::CreateAgnostic_CORINFO_LOOKUP_KIND(pGenericLookupKind);
    key.id                = (DWORD)id;

    AssertKeyExistsNoMessage(GetReadyToRunHelper, key);

    GetReadyToRunHelper_TOKENout value = GetReadyToRunHelper->Get(key);
    DEBUG_REP(dmpGetReadyToRunHelper(key, value));

    *pLookup = SpmiRecordsHelper::RestoreCORINFO_CONST_LOOKUP(value.Lookup);
    return value.result;
}

void MethodContext::recGetReadyToRunDelegateCtorHelper(CORINFO_RESOLVED_TOKEN* pTargetMethod,
                                                       CORINFO_CLASS_HANDLE    delegateType,
                                                       CORINFO_LOOKUP*         pLookup)
{
    if (GetReadyToRunDelegateCtorHelper == nullptr)
        GetReadyToRunDelegateCtorHelper =
            new LightWeightMap<GetReadyToRunDelegateCtorHelper_TOKENIn, Agnostic_CORINFO_LOOKUP>();

    GetReadyToRunDelegateCtorHelper_TOKENIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.TargetMethod =
        SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pTargetMethod, GetReadyToRunDelegateCtorHelper);
    key.delegateType              = CastHandle(delegateType);
    Agnostic_CORINFO_LOOKUP value = SpmiRecordsHelper::StoreAgnostic_CORINFO_LOOKUP(pLookup);
    GetReadyToRunDelegateCtorHelper->Add(key, value);
    DEBUG_REC(dmpGetReadyToRunDelegateCtorHelper(key, value));
}

void MethodContext::dmpGetReadyToRunDelegateCtorHelper(GetReadyToRunDelegateCtorHelper_TOKENIn key,
                                                       Agnostic_CORINFO_LOOKUP                 value)
{
    printf("GetReadyToRunDelegateCtorHelper key: method tk{%s} type-%016llX",
           SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.TargetMethod).c_str(), key.delegateType);
    printf(", value: %s", SpmiDumpHelper::DumpAgnostic_CORINFO_LOOKUP(value).c_str());
}

void MethodContext::repGetReadyToRunDelegateCtorHelper(CORINFO_RESOLVED_TOKEN* pTargetMethod,
                                                       CORINFO_CLASS_HANDLE    delegateType,
                                                       CORINFO_LOOKUP*         pLookup)
{
    AssertMapExistsNoMessage(GetReadyToRunDelegateCtorHelper);

    GetReadyToRunDelegateCtorHelper_TOKENIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.TargetMethod =
        SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pTargetMethod, GetReadyToRunDelegateCtorHelper);
    key.delegateType = CastHandle(delegateType);

    AssertKeyExistsNoMessage(GetReadyToRunDelegateCtorHelper, key);

    Agnostic_CORINFO_LOOKUP value = GetReadyToRunDelegateCtorHelper->Get(key);
    DEBUG_REP(dmpGetReadyToRunDelegateCtorHelper(key, value));

    *pLookup = SpmiRecordsHelper::RestoreCORINFO_LOOKUP(value);
}

void MethodContext::recGetHelperFtn(CorInfoHelpFunc ftnNum, void** ppIndirection, void* result)
{
    if (GetHelperFtn == nullptr)
        GetHelperFtn = new LightWeightMap<DWORD, DLDL>();

    DWORD key = (DWORD)ftnNum;

    DLDL value;
    value.A = CastPointer(*ppIndirection);
    value.B = CastPointer(result);

    if (GetHelperFtn->GetIndex(key) != -1)
    {
        DLDL oldValue = GetHelperFtn->Get(key);

        AssertCodeMsg(oldValue.A == value.A && oldValue.B == oldValue.B, EXCEPTIONCODE_MC,
                      "collision! old: %016llX %016llX, new: %016llX %016llX \n", oldValue.A, oldValue.B, value.A,
                      value.B);
    }

    GetHelperFtn->Add(key, value);
    DEBUG_REC(dmpGetHelperFtn(key, value));
}
void MethodContext::dmpGetHelperFtn(DWORD key, DLDL value)
{
    printf("GetHelperFtn key ftn-%u, value ppi-%016llX res-%016llX", key, value.A, value.B);
}
void* MethodContext::repGetHelperFtn(CorInfoHelpFunc ftnNum, void** ppIndirection)
{
    DWORD key = (DWORD)ftnNum;

    if ((GetHelperFtn == nullptr) || (GetHelperFtn->GetIndex(key) == -1))
    {
#ifdef sparseMC
        LogDebug("Sparse - repGetHelperFtn returning nullptr and 0XCAFE0003");
        *ppIndirection = nullptr;
        return (void*)(size_t)0xCAFE0003;
#else
        LogException(EXCEPTIONCODE_MC, "Encountered an empty LWM while looking for %08X", (DWORD)ftnNum);
#endif
    }

    DLDL value = GetHelperFtn->Get(key);
    DEBUG_REP(dmpGetHelperFtn(key, value));

    *ppIndirection = (void*)value.A;
    return (void*)value.B;
}

//
// Finds the identifier (i.e. the CorInfoHelpFunc enum) of a helper function, based on the address where it
// is located in memory.
//
// Arguments:
//    functionAddress - The starting address of the helper function in memory.
//    pResult         - [out] Pointer to write out the identifier of the helper function located at the given
//                      address.
//
// Return Value:
//    True if there is a helper function associated with the given target address; false otherwise.
//
// Assumptions:
//    Only the lower 32 bits of the method address are necessary to identify the method.
//
// Notes:
//    - See notes for fndGetFunctionEntryPoint for a more in-depth discussion of why we only match on the
//      lower 32 bits of the target address.
//    - This might not work correctly with method contexts recorded via NGen compilation.
//
bool MethodContext::fndGetHelperFtn(void* functionAddress, CorInfoHelpFunc* pResult)
{
    if (GetHelperFtn != nullptr)
    {
        for (unsigned int i = 0; i < GetHelperFtn->GetCount(); i++)
        {
            DWORD key = GetHelperFtn->GetKey(i);
            DLDL  val = GetHelperFtn->GetItem(i);

            // TODO-Cleanup: this only compares the function addresses, and doesn't account for
            // ppIndirection, which will break if the helper is a dynamic helper function.
            if (val.B == CastPointer(functionAddress))
            {
                *pResult = (CorInfoHelpFunc)key;
                return true;
            }
        }
    }

    LogDebug("fndGetHelperFtn - didn't find value %p", functionAddress);
    return false;
}

void MethodContext::recGetJustMyCodeHandle(CORINFO_METHOD_HANDLE         method,
                                           CORINFO_JUST_MY_CODE_HANDLE** ppIndirection,
                                           CORINFO_JUST_MY_CODE_HANDLE   result)
{
    if (GetJustMyCodeHandle == nullptr)
        GetJustMyCodeHandle = new LightWeightMap<DWORDLONG, DLDL>();

    DLDL value;
    value.A = CastPointer(*ppIndirection);
    value.B = CastHandle(result);

    DWORDLONG key = CastHandle(method);
    GetJustMyCodeHandle->Add(key, value);
    DEBUG_REC(dmpGetJustMyCodeHandle(key, value));
}
void MethodContext::dmpGetJustMyCodeHandle(DWORDLONG key, DLDL value)
{
    printf("GetJustMyCodeHandle key ftn-%016llX, value pp-%016llX, res-%016llX", key, value.A, value.B);
}
CORINFO_JUST_MY_CODE_HANDLE MethodContext::repGetJustMyCodeHandle(CORINFO_METHOD_HANDLE         method,
                                                                  CORINFO_JUST_MY_CODE_HANDLE** ppIndirection)
{
    DWORDLONG key = CastHandle(method);
    AssertMapAndKeyExist(GetJustMyCodeHandle, key, ": key %016llX", key);

    DLDL value = GetJustMyCodeHandle->Get(key);
    DEBUG_REP(dmpGetJustMyCodeHandle(key, value));

    *ppIndirection                     = (CORINFO_JUST_MY_CODE_HANDLE*)value.A;
    CORINFO_JUST_MY_CODE_HANDLE result = (CORINFO_JUST_MY_CODE_HANDLE)value.B;
    return result;
}

void MethodContext::recGetFunctionEntryPoint(CORINFO_METHOD_HANDLE ftn,
                                             CORINFO_CONST_LOOKUP* pResult,
                                             CORINFO_ACCESS_FLAGS  accessFlags)
{
    if (GetFunctionEntryPoint == nullptr)
        GetFunctionEntryPoint = new LightWeightMap<DLD, DLD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A   = CastHandle(ftn);
    key.B   = (DWORD)accessFlags;

    DLD value;
    value.A = CastPointer(pResult->addr); // First union member
    value.B = (DWORD)pResult->accessType;

    GetFunctionEntryPoint->Add(key, value);
    DEBUG_REC(dmpGetFunctionEntryPoint(key, value));
}
void MethodContext::dmpGetFunctionEntryPoint(DLD key, DLD value)
{
    printf("GetFunctionEntryPoint key ftn-%016llX af-%08X, value add-%016llX at-%u", key.A, key.B, value.A, value.B);
}
void MethodContext::repGetFunctionEntryPoint(CORINFO_METHOD_HANDLE ftn,
                                             CORINFO_CONST_LOOKUP* pResult,
                                             CORINFO_ACCESS_FLAGS  accessFlags)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(ftn);
    key.B = (DWORD)accessFlags;

    if (GetFunctionEntryPoint == nullptr)
    {
#ifdef sparseMC
        LogDebug("Sparse - repGetFunctionEntryPoint fabricated result for request.");
        pResult->accessType = (InfoAccessType)IAT_PVALUE;
        pResult->addr       = (void*)(CastHandle(ftn) + 0x1c);
        return;
#else
        LogException(EXCEPTIONCODE_MC, "Didn't find %016llX, %8x", CastHandle(ftn), accessFlags);
#endif
    }
    if (GetFunctionEntryPoint->GetIndex(key) == -1)
    {
#ifdef sparseMC
        key.B ^= (DWORD)CORINFO_ACCESS_NONNULL;
        if (GetFunctionEntryPoint->GetIndex(key) != -1)
        {
            LogDebug("Sparse - repGetFunctionEntryPoint found result with inverted CORINFO_ACCESS_NONNULL");
        }
        else
        {
            LogDebug("Sparse - repGetFunctionEntryPoint fabricated result for request.");
            pResult->accessType = (InfoAccessType)IAT_PVALUE;
            pResult->addr       = (void*)(CastHandle(ftn) + 0x1c);
            return;
        }
#else
        LogException(EXCEPTIONCODE_MC, "Didn't find %016llX, %8x", CastHandle(ftn), accessFlags);
#endif
    }

    DLD value = GetFunctionEntryPoint->Get(key);
    DEBUG_REP(dmpGetFunctionEntryPoint(key, value));

    pResult->accessType = (InfoAccessType)value.B;
    pResult->addr       = (void*)value.A;
}

//
// Finds the method handle associated with a method, based on the address where its generated code is located
// in memory.
//
// Arguments:
//    methodAddress - The starting address of the generated code for the method.
//    pResult       - [out] Pointer to a method handle to write into. If this successfully finds a method
//                    handle associated with the given target address, it will be written to here.
//
// Return Value:
//    True if there is a helper function associated with the given target address; false otherwise.
//
// Assumptions:
//    - The given method address does not point to a jump stub.
//    - The given method is not a generic method.
//    - Only the lower 32 bits of the method address are necessary to identify the method.
//
// Notes:
//    On 64-bit platforms, this only checks if the lower 32 bits of the method address match a recorded
//    function entry point because, on AMD64, this only supports reverse lookups for near calls, which
//    encode their target address as a 32-bit PC-relative displacement.
//
//    Practically speaking, there are two concrete reasons why we ignore the upper 64 bits. First, on
//    AMD64, when the JIT emits a near call, it records the displacement it encodes into the call with
//    the EE as a 32-bit relative "relocation". Consequently, this leads to the second reason why we
//    ignore the upper 64 bits: when SuperPMI is replaying method compilation, it patches up addresses
//    based on the "relocations" the JIT had previously recorded with the EE. Since these relocations
//    are only 32-bit deltas, what you'll usually end up seeing is that, after SuperPMI has applied
//    these fixups, the lower 32 bits of method addresses will match, but the upper 32 bits will differ.
//
bool MethodContext::fndGetFunctionEntryPoint(DLD value, CORINFO_METHOD_HANDLE* pResult)
{
    if (GetFunctionEntryPoint != nullptr)
    {
        for (unsigned int i = 0; i < GetFunctionEntryPoint->GetCount(); i++)
        {
            DLD key = GetFunctionEntryPoint->GetKey(i);
            DLD val = GetFunctionEntryPoint->GetItem(i);

            // TODO-Cleanup: we should be more conscious of the rest of the information in CORINFO_CONST_LOOKUP
            if ((DWORD)val.A == (DWORD)value.A)
            {
                *pResult = (CORINFO_METHOD_HANDLE)key.A;
                return true;
            }
        }
    }

    LogDebug("fndGetFunctionEntryPoint - didn't find value %016llX", value.A);
    return false;
}

void MethodContext::recConstructStringLiteral(CORINFO_MODULE_HANDLE moduleHandle,
                                              mdToken               metaTok,
                                              void*                 pValue,
                                              InfoAccessType        result)
{
    if (ConstructStringLiteral == nullptr)
        ConstructStringLiteral = new LightWeightMap<DLD, DLD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(moduleHandle);
    key.B = (DWORD)metaTok;

    DLD value;
    value.A = CastPointer(pValue);
    value.B = (DWORD)result;

    ConstructStringLiteral->Add(key, value);
    DEBUG_REC(dmpConstructStringLiteral(key, value));
}
void MethodContext::dmpConstructStringLiteral(DLD key, DLD value)
{
    printf("ConstructStringLiteral key mod-%016llX tok-%08X, value pp-%016llX iat-%u", key.A, key.B, value.A, value.B);
}
InfoAccessType MethodContext::repConstructStringLiteral(CORINFO_MODULE_HANDLE moduleHandle, mdToken metaTok, void** ppValue)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(moduleHandle);
    key.B = (DWORD)metaTok;

    AssertMapAndKeyExist(ConstructStringLiteral, key, ": key %016llX", CastHandle(moduleHandle));

    DLD value = ConstructStringLiteral->Get(key);
    DEBUG_REP(dmpConstructStringLiteral(key, value));

    *ppValue = (void*)value.A;
    return (InfoAccessType)value.B;
}

void MethodContext::recConvertPInvokeCalliToCall(CORINFO_RESOLVED_TOKEN* pResolvedToken, bool fMustConvert, bool result)
{
    if (ConvertPInvokeCalliToCall == nullptr)
        ConvertPInvokeCalliToCall = new LightWeightMap<DLD, DWORDLONG>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(pResolvedToken->tokenScope);
    key.B = (DWORD)pResolvedToken->token;

    DWORDLONG value = CastHandle(result ? pResolvedToken->hMethod : 0);

    ConvertPInvokeCalliToCall->Add(key, value);
    DEBUG_REC(dmpConvertPInvokeCalliToCall(key, value));
}
void MethodContext::dmpConvertPInvokeCalliToCall(DLD key, DWORDLONG value)
{
    printf("ConvertPInvokeCalliToCall key mod-%016llX tok-%08X, value %016llX", key.A, key.B, value);
}
bool MethodContext::repConvertPInvokeCalliToCall(CORINFO_RESOLVED_TOKEN* pResolvedToken, bool fMustConvert)
{
    AssertMapExistsNoMessage(ConvertPInvokeCalliToCall);

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(pResolvedToken->tokenScope);
    key.B = (DWORD)pResolvedToken->token;

    AssertKeyExistsNoMessage(ConvertPInvokeCalliToCall, key);
    DWORDLONG value = ConvertPInvokeCalliToCall->Get(key);
    DEBUG_REP(dmpConvertPInvokeCalliToCall(key, value));

    pResolvedToken->hMethod = (CORINFO_METHOD_HANDLE)value;
    return value != 0;
}

void MethodContext::recEmptyStringLiteral(void** pValue, InfoAccessType result)
{
    if (EmptyStringLiteral == nullptr)
        EmptyStringLiteral = new DenseLightWeightMap<DLD>();
    DLD temp2;
    temp2.A = CastPointer(*pValue);
    temp2.B = (DWORD)result;

    EmptyStringLiteral->Append(temp2);
}
void MethodContext::dmpEmptyStringLiteral(DWORD key, DLD value)
{
    printf("EmptyStringLiteral key %u, value pVal-%016llX res-%u", key, value.A, value.B);
}
InfoAccessType MethodContext::repEmptyStringLiteral(void** ppValue)
{
    AssertMapAndKeyExistNoMessage(EmptyStringLiteral, 0);

    // TODO-Cleanup: sketchy if someone calls this twice
    DLD temp2;
    temp2    = EmptyStringLiteral->Get(0);
    *ppValue = (void*)temp2.A;
    return (InfoAccessType)temp2.B;
}

void MethodContext::recGetArgType(CORINFO_SIG_INFO*       sig,
                                  CORINFO_ARG_LIST_HANDLE args,
                                  CORINFO_CLASS_HANDLE*   vcTypeRet,
                                  CorInfoTypeWithMod      result,
                                  DWORD                   exceptionCode)
{
    if (GetArgType == nullptr)
        GetArgType = new LightWeightMap<Agnostic_GetArgType_Key, Agnostic_GetArgType_Value>();

    Agnostic_GetArgType_Key key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding

    // Only setting values for CORINFO_SIG_INFO things the EE seems to pay attention to... this is necessary since some of the values
    // are unset and fail our precise comparisons ...
    // TODO: verify that the above comment is still true (that some of the fields of incoming argument `sig` contain garbage), or
    // can we store the whole thing and use StoreAgnostic_CORINFO_SIG_INFO()?

    key.flags           = (DWORD)sig->flags;
    key.numArgs         = (DWORD)sig->numArgs;

    SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INST_HandleArray(sig->sigInst.classInstCount, sig->sigInst.classInst, SigInstHandleMap, &key.sigInst_classInstCount, &key.sigInst_classInst_Index);
    SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INST_HandleArray(sig->sigInst.methInstCount, sig->sigInst.methInst, SigInstHandleMap, &key.sigInst_methInstCount, &key.sigInst_methInst_Index);

    key.methodSignature = CastPointer(sig->methodSignature);
    key.scope           = CastHandle(sig->scope);
    key.args            = CastHandle(args);

    Agnostic_GetArgType_Value value;
    value.vcTypeRet     = CastHandle(*vcTypeRet);
    value.result        = (DWORD)result;
    value.exceptionCode = (DWORD)exceptionCode;

    GetArgType->Add(key, value);
    DEBUG_REC(dmpGetArgType(key, value));
}
void MethodContext::dmpGetArgType(const Agnostic_GetArgType_Key& key, const Agnostic_GetArgType_Value& value)
{
    printf("GetArgType key flg-%08X na-%u %s %s msig-%016llX scp-%016llX arg-%016llX",
        key.flags, key.numArgs,
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INST_Element("", "cc", "ci", key.sigInst_classInstCount, key.sigInst_classInst_Index, SigInstHandleMap).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INST_Element("", "mc", "mi", key.sigInst_methInstCount, key.sigInst_methInst_Index, SigInstHandleMap).c_str(),
        key.methodSignature, key.scope, key.args);
    printf(", value result(cit)-%u(%s) vcType-%016llX excp-%08X", value.result, toString((CorInfoTypeWithMod)value.result), value.vcTypeRet, value.exceptionCode);
}
CorInfoTypeWithMod MethodContext::repGetArgType(CORINFO_SIG_INFO*       sig,
                                                CORINFO_ARG_LIST_HANDLE args,
                                                CORINFO_CLASS_HANDLE*   vcTypeRet,
                                                DWORD*                  exceptionCode)
{
    AssertMapExists(GetArgType, ": key %016llX %016llX", CastHandle(sig->scope), CastHandle(args));

    Agnostic_GetArgType_Key key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.flags                  = (DWORD)sig->flags;
    key.numArgs                = (DWORD)sig->numArgs;
    key.sigInst_classInstCount = (DWORD)sig->sigInst.classInstCount;
    key.sigInst_methInstCount  = (DWORD)sig->sigInst.methInstCount;
    key.methodSignature        = CastPointer(sig->methodSignature);
    key.scope                  = CastHandle(sig->scope);
    key.args                   = CastHandle(args);

    key.sigInst_classInst_Index = SpmiRecordsHelper::ContainsHandleMap(sig->sigInst.classInstCount, sig->sigInst.classInst, SigInstHandleMap);
    key.sigInst_methInst_Index  = SpmiRecordsHelper::ContainsHandleMap(sig->sigInst.methInstCount, sig->sigInst.methInst, SigInstHandleMap);

    AssertKeyExists(GetArgType, key, ": key %016llX %016llX", key.scope, key.args);

    Agnostic_GetArgType_Value value = GetArgType->Get(key);
    DEBUG_REP(dmpGetArgType(key, value));

    *vcTypeRet              = (CORINFO_CLASS_HANDLE)value.vcTypeRet;
    CorInfoTypeWithMod temp = (CorInfoTypeWithMod)value.result;
    *exceptionCode          = (DWORD)value.exceptionCode;
    return temp;
}

void MethodContext::recGetArgNext(CORINFO_ARG_LIST_HANDLE args, CORINFO_ARG_LIST_HANDLE result)
{
    if (GetArgNext == nullptr)
        GetArgNext = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(args);
    DWORDLONG value = CastHandle(result);
    GetArgNext->Add(key, value);
    DEBUG_REC(dmpGetArgNext(key, value));
}
void MethodContext::dmpGetArgNext(DWORDLONG key, DWORDLONG value)
{
    printf("GetArgNext key %016llX, value %016llX", key, value);
}
CORINFO_ARG_LIST_HANDLE MethodContext::repGetArgNext(CORINFO_ARG_LIST_HANDLE args)
{
    DWORDLONG key = CastHandle(args);
    AssertMapAndKeyExist(GetArgNext, key, ": key %016llX", key);
    DWORDLONG value = GetArgNext->Get(key);
    DEBUG_REP(dmpGetArgNext(key, value));
    CORINFO_ARG_LIST_HANDLE result = (CORINFO_ARG_LIST_HANDLE)value;
    return result;
}
void MethodContext::recGetMethodSig(CORINFO_METHOD_HANDLE ftn, CORINFO_SIG_INFO* sig, CORINFO_CLASS_HANDLE memberParent)
{
    if (GetMethodSig == nullptr)
        GetMethodSig = new LightWeightMap<DLDL, Agnostic_CORINFO_SIG_INFO>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(ftn);
    key.B = CastHandle(memberParent);

    Agnostic_CORINFO_SIG_INFO value = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(*sig, GetMethodSig, SigInstHandleMap);

    GetMethodSig->Add(key, value);
    DEBUG_REC(dmpGetMethodSig(key, value));
}
void MethodContext::dmpGetMethodSig(DLDL key, const Agnostic_CORINFO_SIG_INFO& value)
{
    printf("GetMethodSig key ftn-%016llX prt-%016llX, value-%s", key.A, key.B,
           SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value, GetMethodSig, SigInstHandleMap).c_str());
}
void MethodContext::repGetMethodSig(CORINFO_METHOD_HANDLE ftn, CORINFO_SIG_INFO* sig, CORINFO_CLASS_HANDLE memberParent)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(ftn);
    key.B = CastHandle(memberParent);

    AssertMapAndKeyExist(GetMethodSig, key, ": key ftn-%016llX prt-%016llX", key.A, key.B);

    Agnostic_CORINFO_SIG_INFO value = GetMethodSig->Get(key);
    DEBUG_REP(dmpGetMethodSig(key, value));

    *sig = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value, GetMethodSig, SigInstHandleMap);
}

void MethodContext::recGetArgClass(CORINFO_SIG_INFO*       sig,
                                   CORINFO_ARG_LIST_HANDLE args,
                                   CORINFO_CLASS_HANDLE    result,
                                   DWORD                   exceptionCode)
{
    if (GetArgClass == nullptr)
        GetArgClass = new LightWeightMap<Agnostic_GetArgClass_Key, Agnostic_GetArgClass_Value>();

    Agnostic_GetArgClass_Key key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding

    // Only setting values for CORINFO_SIG_INFO things the EE seems to pay attention to... this is necessary since some of the values
    // are unset and fail our precise comparisions...

    SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INST_HandleArray(sig->sigInst.classInstCount, sig->sigInst.classInst, SigInstHandleMap, &key.sigInst_classInstCount, &key.sigInst_classInst_Index);
    SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INST_HandleArray(sig->sigInst.methInstCount, sig->sigInst.methInst, SigInstHandleMap, &key.sigInst_methInstCount, &key.sigInst_methInst_Index);

    key.methodSignature = CastPointer(sig->methodSignature);
    key.scope           = CastHandle(sig->scope);
    key.args            = CastHandle(args);

    Agnostic_GetArgClass_Value value;
    value.result        = CastHandle(result);
    value.exceptionCode = exceptionCode;

    GetArgClass->Add(key, value);
    DEBUG_REC(dmpGetArgClass(key, value));
}
void MethodContext::dmpGetArgClass(const Agnostic_GetArgClass_Key& key, const Agnostic_GetArgClass_Value& value)
{
    printf("GetArgClass key %s %s msig-%016llX scp-%016llX args-%016llX",
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INST_Element("", "cc", "ci", key.sigInst_classInstCount, key.sigInst_classInst_Index, SigInstHandleMap).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INST_Element("", "mc", "mi", key.sigInst_methInstCount, key.sigInst_methInst_Index, SigInstHandleMap).c_str(),
        key.methodSignature, key.scope, key.args);
    printf(", value %016llX excp-%08X", value.result, value.exceptionCode);
}
CORINFO_CLASS_HANDLE MethodContext::repGetArgClass(CORINFO_SIG_INFO*       sig,
                                                   CORINFO_ARG_LIST_HANDLE args,
                                                   DWORD*                  exceptionCode)
{
    Agnostic_GetArgClass_Key key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.sigInst_classInstCount = (DWORD)sig->sigInst.classInstCount;
    key.sigInst_methInstCount  = (DWORD)sig->sigInst.methInstCount;
    key.methodSignature        = CastPointer(sig->methodSignature);
    key.scope                  = CastHandle(sig->scope);
    key.args                   = CastHandle(args);
    key.sigInst_classInst_Index = SpmiRecordsHelper::ContainsHandleMap(sig->sigInst.classInstCount, sig->sigInst.classInst, SigInstHandleMap);
    key.sigInst_methInst_Index  = SpmiRecordsHelper::ContainsHandleMap(sig->sigInst.methInstCount, sig->sigInst.methInst, SigInstHandleMap);

    AssertMapAndKeyExist(GetArgClass, key, ": key %016llX %016llX", key.scope, key.args);

    Agnostic_GetArgClass_Value value = GetArgClass->Get(key);
    DEBUG_REP(dmpGetArgClass(key, value));

    *exceptionCode = value.exceptionCode;
    return (CORINFO_CLASS_HANDLE)value.result;
}

void MethodContext::recGetHFAType(CORINFO_CLASS_HANDLE clsHnd, CorInfoHFAElemType result)
{
    if (GetHFAType == nullptr)
        GetHFAType = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(clsHnd);
    DWORD value = (DWORD)result;
    GetHFAType->Add(key, value);
    DEBUG_REC(dmpGetHFAType(key, value));
}
void MethodContext::dmpGetHFAType(DWORDLONG key, DWORD value)
{
    printf("GetHFAType key %016llX, value %u ", key, value);
}
CorInfoHFAElemType MethodContext::repGetHFAType(CORINFO_CLASS_HANDLE clsHnd)
{
    DWORDLONG key = CastHandle(clsHnd);
    AssertMapAndKeyExist(GetHFAType, key, ": key %016llX", key);
    DWORD value = GetHFAType->Get(key);
    DEBUG_REP(dmpGetHFAType(key, value));
    return (CorInfoHFAElemType)value;
}

void MethodContext::recGetMethodInfo(CORINFO_METHOD_HANDLE ftn,
                                     CORINFO_METHOD_INFO*  info,
                                     bool                  result,
                                     DWORD                 exceptionCode)
{
    if (GetMethodInfo == nullptr)
        GetMethodInfo = new LightWeightMap<DWORDLONG, Agnostic_GetMethodInfo>();

    Agnostic_GetMethodInfo value;
    ZeroMemory(&value, sizeof(value));

    if (result)
    {
        value.info.ftn           = CastHandle(info->ftn);
        value.info.scope         = CastHandle(info->scope);
        value.info.ILCode_offset = (DWORD)GetMethodInfo->AddBuffer(info->ILCode, info->ILCodeSize);
        value.info.ILCodeSize    = (DWORD)info->ILCodeSize;
        value.info.maxStack      = (DWORD)info->maxStack;
        value.info.EHcount       = (DWORD)info->EHcount;
        value.info.options       = (DWORD)info->options;
        value.info.regionKind    = (DWORD)info->regionKind;

        value.info.args   = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(info->args, GetMethodInfo, SigInstHandleMap);
        value.info.locals = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(info->locals, GetMethodInfo, SigInstHandleMap);
    }
    value.result        = result;
    value.exceptionCode = (DWORD)exceptionCode;

    DWORDLONG key = CastHandle(ftn);
    GetMethodInfo->Add(key, value);
    DEBUG_REC(dmpGetMethodInfo(key, value));
}
void MethodContext::dmpGetMethodInfo(DWORDLONG key, const Agnostic_GetMethodInfo& value)
{
    if (value.result)
    {
        printf("GetMethodInfo key ftn-%016llX, value res-%u ftn-%016llX scp-%016llX ilo-%u ils-%u ms-%u ehc-%u opt-%08X rk-%u args-%s locals-%s excp-%08X",
            key, value.result, value.info.ftn, value.info.scope, value.info.ILCode_offset, value.info.ILCodeSize,
            value.info.maxStack, value.info.EHcount, value.info.options, value.info.regionKind,
            SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.info.args, GetMethodInfo, SigInstHandleMap).c_str(),
            SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value.info.locals, GetMethodInfo, SigInstHandleMap).c_str(),
            value.exceptionCode);
    }
    else
    {
        printf("GetMethodInfo key ftn-%016llX, value res-%u excp-%08X",
            key, value.result, value.exceptionCode);
    }
}
bool MethodContext::repGetMethodInfo(CORINFO_METHOD_HANDLE ftn, CORINFO_METHOD_INFO* info, DWORD* exceptionCode)
{
    DWORDLONG key = CastHandle(ftn);
    AssertMapAndKeyExist(GetMethodInfo, key, ": key %016llX", key);

    Agnostic_GetMethodInfo value = GetMethodInfo->Get(key);
    DEBUG_REP(dmpGetMethodInfo(key, value));

    if (value.result)
    {
        info->ftn        = (CORINFO_METHOD_HANDLE)value.info.ftn;
        info->scope      = (CORINFO_MODULE_HANDLE)value.info.scope;
        info->ILCode     = GetMethodInfo->GetBuffer(value.info.ILCode_offset);
        info->ILCodeSize = (unsigned)value.info.ILCodeSize;
        info->maxStack   = (unsigned)value.info.maxStack;
        info->EHcount    = (unsigned)value.info.EHcount;
        info->options    = (CorInfoOptions)value.info.options;
        info->regionKind = (CorInfoRegionKind)value.info.regionKind;

        info->args   = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.info.args, GetMethodInfo, SigInstHandleMap);
        info->locals = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value.info.locals, GetMethodInfo, SigInstHandleMap);
    }
    bool result    = value.result;
    *exceptionCode = (DWORD)value.exceptionCode;
    return result;
}

void MethodContext::recGetNewHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                    CORINFO_METHOD_HANDLE   callerHandle,
                                    bool*                   pHasSideEffects,
                                    CorInfoHelpFunc         result)
{
    if (GetNewHelper == nullptr)
        GetNewHelper = new LightWeightMap<Agnostic_GetNewHelper, DD>();

    Agnostic_GetNewHelper key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hClass       = CastHandle(pResolvedToken->hClass);
    key.callerHandle = CastHandle(callerHandle);

    DD value;
    value.A = (pHasSideEffects != nullptr) ? (DWORD)(*pHasSideEffects ? 1 : 0) : (DWORD)0;
    value.B = (DWORD)result;

    GetNewHelper->Add(key, value);
    DEBUG_REC(dmpGetNewHelper(key, value));
}
void MethodContext::dmpGetNewHelper(const Agnostic_GetNewHelper& key, DD value)
{
    printf("GetNewHelper key cls-%016llX chan-%016llX, hasSideEffects-%u, value res-%u", key.hClass, key.callerHandle, value.A, value.B);
}
CorInfoHelpFunc MethodContext::repGetNewHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                               CORINFO_METHOD_HANDLE   callerHandle,
                                               bool*                   pHasSideEffects)
{
    Agnostic_GetNewHelper key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hClass       = CastHandle(pResolvedToken->hClass);
    key.callerHandle = CastHandle(callerHandle);

    AssertMapAndKeyExist(GetNewHelper, key, ": key %016llX %016llX", key.hClass, key.callerHandle);

    DD value = GetNewHelper->Get(key);
    DEBUG_REP(dmpGetNewHelper(key, value));

    if (pHasSideEffects != nullptr)
    {
        *pHasSideEffects = (value.A == 0) ? false : true;
    }
    CorInfoHelpFunc result = (CorInfoHelpFunc)value.B;
    return result;
}

void MethodContext::recEmbedGenericHandle(CORINFO_RESOLVED_TOKEN*       pResolvedToken,
                                          bool                          fEmbedParent,
                                          CORINFO_GENERICHANDLE_RESULT* pResult)
{
    if (EmbedGenericHandle == nullptr)
        EmbedGenericHandle = new LightWeightMap<Agnostic_EmbedGenericHandle, Agnostic_CORINFO_GENERICHANDLE_RESULT>();

    Agnostic_EmbedGenericHandle key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, EmbedGenericHandle);
    key.fEmbedParent  = (DWORD)fEmbedParent;

    Agnostic_CORINFO_GENERICHANDLE_RESULT value;
    value.lookup            = SpmiRecordsHelper::StoreAgnostic_CORINFO_LOOKUP(&pResult->lookup);
    value.compileTimeHandle = CastHandle(pResult->compileTimeHandle);
    value.handleType        = (DWORD)pResult->handleType;

    EmbedGenericHandle->Add(key, value);
    DEBUG_REC(dmpEmbedGenericHandle(key, value));
}
void MethodContext::dmpEmbedGenericHandle(const Agnostic_EmbedGenericHandle&           key,
                                          const Agnostic_CORINFO_GENERICHANDLE_RESULT& value)
{
    printf("EmbedGenericHandle key rt{%s} emb-%u, value %s cth-%016llX ht-%u",
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ResolvedToken).c_str(),
        key.fEmbedParent,
        SpmiDumpHelper::DumpAgnostic_CORINFO_LOOKUP(value.lookup).c_str(),
        value.compileTimeHandle,
        value.handleType);
}
void MethodContext::repEmbedGenericHandle(CORINFO_RESOLVED_TOKEN*       pResolvedToken,
                                          bool                          fEmbedParent,
                                          CORINFO_GENERICHANDLE_RESULT* pResult)
{
    AssertMapExistsNoMessage(EmbedGenericHandle);

    Agnostic_EmbedGenericHandle key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, EmbedGenericHandle);
    key.fEmbedParent  = (DWORD)fEmbedParent;

    AssertKeyExistsNoMessage(EmbedGenericHandle, key);

    Agnostic_CORINFO_GENERICHANDLE_RESULT value = EmbedGenericHandle->Get(key);
    DEBUG_REP(dmpEmbedGenericHandle(key, value));

    pResult->lookup            = SpmiRecordsHelper::RestoreCORINFO_LOOKUP(value.lookup);
    pResult->compileTimeHandle = (CORINFO_GENERIC_HANDLE)value.compileTimeHandle;
    pResult->handleType        = (CorInfoGenericHandleType)value.handleType;
}

void MethodContext::recGetEHinfo(CORINFO_METHOD_HANDLE ftn, unsigned EHnumber, CORINFO_EH_CLAUSE* clause)
{
    if (GetEHinfo == nullptr)
        GetEHinfo = new LightWeightMap<DLD, Agnostic_CORINFO_EH_CLAUSE>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(ftn);
    key.B = (DWORD)EHnumber;

    Agnostic_CORINFO_EH_CLAUSE value;
    value.Flags         = (DWORD)clause->Flags;
    value.TryOffset     = (DWORD)clause->TryOffset;
    value.TryLength     = (DWORD)clause->TryLength;
    value.HandlerOffset = (DWORD)clause->HandlerOffset;
    value.HandlerLength = (DWORD)clause->HandlerLength;
    value.ClassToken    = (DWORD)clause->ClassToken;

    GetEHinfo->Add(key, value);
    DEBUG_REC(dmpGetEHinfo(key, value));
}
void MethodContext::dmpGetEHinfo(DLD key, const Agnostic_CORINFO_EH_CLAUSE& value)
{
    printf("GetEHinfo key ftn-%016llX ehn-%u, value flg-%u to-%u tl-%u ho-%u hl-%u ct-%u", key.A, key.B, value.Flags,
           value.TryOffset, value.TryLength, value.HandlerOffset, value.HandlerLength, value.ClassToken);
}
void MethodContext::repGetEHinfo(CORINFO_METHOD_HANDLE ftn, unsigned EHnumber, CORINFO_EH_CLAUSE* clause)
{
    AssertMapExists(GetEHinfo, ": key %016llX", CastHandle(ftn));

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(ftn);
    key.B = (DWORD)EHnumber;

    AssertKeyExists(GetEHinfo, key, ": key %016llX", CastHandle(ftn));

    Agnostic_CORINFO_EH_CLAUSE value = GetEHinfo->Get(key);
    DEBUG_REP(dmpGetEHinfo(key, value));

    clause->Flags         = (CORINFO_EH_CLAUSE_FLAGS)value.Flags;
    clause->TryOffset     = (DWORD)value.TryOffset;
    clause->TryLength     = (DWORD)value.TryLength;
    clause->HandlerOffset = (DWORD)value.HandlerOffset;
    clause->HandlerLength = (DWORD)value.HandlerLength;
    clause->ClassToken    = (DWORD)value.ClassToken;
}

void MethodContext::recGetMethodVTableOffset(CORINFO_METHOD_HANDLE method,
                                             unsigned*             offsetOfIndirection,
                                             unsigned*             offsetAfterIndirection,
                                             bool*                 isRelative)
{
    if (GetMethodVTableOffset == nullptr)
        GetMethodVTableOffset = new LightWeightMap<DWORDLONG, DDD>();

    DDD value;
    value.A = (DWORD)*offsetOfIndirection;
    value.B = (DWORD)*offsetAfterIndirection;
    value.C = *isRelative ? 1 : 0;

    DWORDLONG key = CastHandle(method);
    GetMethodVTableOffset->Add(key, value);
    DEBUG_REC(dmpGetMethodVTableOffset(key, value));
}
void MethodContext::dmpGetMethodVTableOffset(DWORDLONG key, DDD value)
{
    printf("GetMethodVTableOffset key ftn-%016llX, value offi-%u, offa-%u. offr-%d", key, value.A, value.B, value.C);
}
void MethodContext::repGetMethodVTableOffset(CORINFO_METHOD_HANDLE method,
                                             unsigned*             offsetOfIndirection,
                                             unsigned*             offsetAfterIndirection,
                                             bool*                 isRelative)
{
    DWORDLONG key = CastHandle(method);
    AssertMapAndKeyExist(GetMethodVTableOffset, key, ": key %016llX", key);

    DDD value = GetMethodVTableOffset->Get(key);
    DEBUG_REP(dmpGetMethodVTableOffset(key, value));

    *offsetOfIndirection    = (unsigned)value.A;
    *offsetAfterIndirection = (unsigned)value.B;
    *isRelative             = (value.C != 0);
}

void MethodContext::recResolveVirtualMethod(CORINFO_DEVIRTUALIZATION_INFO * info, bool returnValue)
{
    if (ResolveVirtualMethod == nullptr)
    {
        ResolveVirtualMethod = new LightWeightMap<Agnostic_ResolveVirtualMethodKey, Agnostic_ResolveVirtualMethodResult>();
    }

    Agnostic_ResolveVirtualMethodKey key;
    ZeroMemory(&key, sizeof(key)); // Zero token including any struct padding
    key.virtualMethod  = CastHandle(info->virtualMethod);
    key.objClass       = CastHandle(info->objClass);
    key.context        = CastHandle(info->context);

    key.pResolvedTokenVirtualMethodNonNull = info->pResolvedTokenVirtualMethod != NULL ? 1 : 0;
    if (key.pResolvedTokenVirtualMethodNonNull)
        key.pResolvedTokenVirtualMethod = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(info->pResolvedTokenVirtualMethod, ResolveToken);

    Agnostic_ResolveVirtualMethodResult result;
    result.returnValue                = returnValue;
    result.devirtualizedMethod        = CastHandle(info->devirtualizedMethod);
    result.requiresInstMethodTableArg = info->requiresInstMethodTableArg;
    result.exactContext               = CastHandle(info->exactContext);
    result.detail                     = (DWORD) info->detail;

    if (returnValue)
    {
        result.resolvedTokenDevirtualizedMethod        = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(&info->resolvedTokenDevirtualizedMethod, ResolveToken);
        result.resolvedTokenDevirtualizedUnboxedMethod = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(&info->resolvedTokenDevirtualizedUnboxedMethod, ResolveToken);
    }
    else
    {
        ZeroMemory(&result.resolvedTokenDevirtualizedMethod, sizeof(result.resolvedTokenDevirtualizedMethod));
        ZeroMemory(&result.resolvedTokenDevirtualizedUnboxedMethod, sizeof(result.resolvedTokenDevirtualizedUnboxedMethod));
    }

    ResolveVirtualMethod->Add(key, result);
    DEBUG_REC(dmpResolveVirtualMethod(key, result));
}

void MethodContext::dmpResolveVirtualMethod(const Agnostic_ResolveVirtualMethodKey& key, const Agnostic_ResolveVirtualMethodResult& result)
{
    printf("ResolveVirtualMethod key virtMethod-%016llX, objClass-%016llX, context-%016llX pResolvedTokenVirtualMethodNonNull-%08X pResolvedTokenVirtualMethod{%s}",
        key.virtualMethod,
        key.objClass,
        key.context,
        key.pResolvedTokenVirtualMethodNonNull,
        key.pResolvedTokenVirtualMethodNonNull ? SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.pResolvedTokenVirtualMethod).c_str() : "???");
    printf(", value returnValue-%s, devirtMethod-%016llX, requiresInstArg-%s, exactContext-%016llX, detail-%d, tokDvMeth{%s}, tokDvUnboxMeth{%s}",
        result.returnValue ? "true" : "false",
        result.devirtualizedMethod,
        result.requiresInstMethodTableArg ? "true" : "false",
        result.exactContext,
        result.detail,
        result.returnValue ? SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(result.resolvedTokenDevirtualizedMethod).c_str() : "???",
        result.returnValue ? SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(result.resolvedTokenDevirtualizedUnboxedMethod).c_str() : "???");
}

bool MethodContext::repResolveVirtualMethod(CORINFO_DEVIRTUALIZATION_INFO * info)
{
    Agnostic_ResolveVirtualMethodKey key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.virtualMethod  = CastHandle(info->virtualMethod);
    key.objClass       = CastHandle(info->objClass);
    key.context        = CastHandle(info->context);

    key.pResolvedTokenVirtualMethodNonNull = info->pResolvedTokenVirtualMethod != NULL ? 1 : 0;
    if (key.pResolvedTokenVirtualMethodNonNull)
        key.pResolvedTokenVirtualMethod = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(info->pResolvedTokenVirtualMethod, ResolveToken);

    AssertMapAndKeyExist(ResolveVirtualMethod, key, ": %016llX-%016llX-%016llX-%08X", key.virtualMethod, key.objClass, key.context, key.pResolvedTokenVirtualMethodNonNull);

    Agnostic_ResolveVirtualMethodResult result = ResolveVirtualMethod->Get(key);
    DEBUG_REP(dmpResolveVirtualMethod(key, result));

    info->devirtualizedMethod = (CORINFO_METHOD_HANDLE) result.devirtualizedMethod;
    info->requiresInstMethodTableArg = result.requiresInstMethodTableArg;
    info->exactContext = (CORINFO_CONTEXT_HANDLE) result.exactContext;
    info->detail = (CORINFO_DEVIRTUALIZATION_DETAIL) result.detail;
    if (result.returnValue)
    {
        info->resolvedTokenDevirtualizedMethod = SpmiRecordsHelper::Restore_CORINFO_RESOLVED_TOKEN(&result.resolvedTokenDevirtualizedMethod, ResolveToken);
        info->resolvedTokenDevirtualizedUnboxedMethod = SpmiRecordsHelper::Restore_CORINFO_RESOLVED_TOKEN(&result.resolvedTokenDevirtualizedUnboxedMethod, ResolveToken);
    }
    return result.returnValue;
}

void MethodContext::recGetUnboxedEntry(CORINFO_METHOD_HANDLE ftn,
                                       bool*                 requiresInstMethodTableArg,
                                       CORINFO_METHOD_HANDLE result)
{
    if (GetUnboxedEntry == nullptr)
    {
        GetUnboxedEntry = new LightWeightMap<DWORDLONG, DLD>();
    }

    DWORDLONG key = CastHandle(ftn);
    DLD       value;
    value.A = CastHandle(result);
    if (requiresInstMethodTableArg != nullptr)
    {
        value.B = (DWORD)*requiresInstMethodTableArg ? 1 : 0;
    }
    else
    {
        value.B = 0;
    }
    GetUnboxedEntry->Add(key, value);
    DEBUG_REC(dmpGetUnboxedEntry(key, value));
}

void MethodContext::dmpGetUnboxedEntry(DWORDLONG key, DLD value)
{
    printf("GetUnboxedEntry ftn-%016llX, result-%016llX, requires-inst-%u", key, value.A, value.B);
}

CORINFO_METHOD_HANDLE MethodContext::repGetUnboxedEntry(CORINFO_METHOD_HANDLE ftn, bool* requiresInstMethodTableArg)
{
    DWORDLONG key = CastHandle(ftn);

    AssertMapAndKeyExist(GetUnboxedEntry, key, ": key %016llX", key);

    DLD value = GetUnboxedEntry->Get(key);
    DEBUG_REP(dmpGetUnboxedEntry(key, value));

    if (requiresInstMethodTableArg != nullptr)
    {
        *requiresInstMethodTableArg = (value.B == 1);
    }
    return (CORINFO_METHOD_HANDLE)(value.A);
}

void MethodContext::recGetDefaultComparerClass(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result)
{
    if (GetDefaultComparerClass == nullptr)
        GetDefaultComparerClass = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(result);
    GetDefaultComparerClass->Add(key, value);
    DEBUG_REC(dmpGetDefaultComparerClass(key, value));
}
void MethodContext::dmpGetDefaultComparerClass(DWORDLONG key, DWORDLONG value)
{
    printf("GetDefaultComparerClass key cls-%016llX, value cls-%016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetDefaultComparerClass(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetDefaultComparerClass, key, ": key %016llX", key);
    DWORDLONG value = GetDefaultComparerClass->Get(key);
    DEBUG_REP(dmpGetDefaultComparerClass(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetDefaultEqualityComparerClass(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result)
{
    if (GetDefaultEqualityComparerClass == nullptr)
        GetDefaultEqualityComparerClass = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(result);
    GetDefaultEqualityComparerClass->Add(key, value);
    DEBUG_REC(dmpGetDefaultEqualityComparerClass(key, value));
}
void MethodContext::dmpGetDefaultEqualityComparerClass(DWORDLONG key, DWORDLONG value)
{
    printf("GetDefaultEqualityComparerClass key cls-%016llX, value cls-%016llX", key, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetDefaultEqualityComparerClass(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetDefaultEqualityComparerClass, key, ": key %016llX", key);
    DWORDLONG value = GetDefaultEqualityComparerClass->Get(key);
    DEBUG_REP(dmpGetDefaultEqualityComparerClass(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetTokenTypeAsHandle(CORINFO_RESOLVED_TOKEN* pResolvedToken, CORINFO_CLASS_HANDLE result)
{
    if (GetTokenTypeAsHandle == nullptr)
        GetTokenTypeAsHandle = new LightWeightMap<GetTokenTypeAsHandleValue, DWORDLONG>();

    GetTokenTypeAsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hMethod = CastHandle(pResolvedToken->hMethod);
    key.hField  = CastHandle(pResolvedToken->hField);

    DWORDLONG value = CastHandle(result);
    GetTokenTypeAsHandle->Add(key, value);
    DEBUG_REC(dmpGetTokenTypeAsHandle(key, value));
}
void MethodContext::dmpGetTokenTypeAsHandle(const GetTokenTypeAsHandleValue& key, DWORDLONG value)
{
    printf("GetTokenTypeAsHandle key ftn-%016llX fld-%016llX, value cls-%016llX", key.hMethod, key.hField, value);
}
CORINFO_CLASS_HANDLE MethodContext::repGetTokenTypeAsHandle(CORINFO_RESOLVED_TOKEN* pResolvedToken)
{
    GetTokenTypeAsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hMethod = CastHandle(pResolvedToken->hMethod);
    key.hField  = CastHandle(pResolvedToken->hField);

    AssertMapAndKeyExistNoMessage(GetTokenTypeAsHandle, key);

    DWORDLONG value = GetTokenTypeAsHandle->Get(key);
    DEBUG_REP(dmpGetTokenTypeAsHandle(key, value));
    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value;
    return result;
}

void MethodContext::recGetFieldInfo(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                    CORINFO_METHOD_HANDLE   callerHandle,
                                    CORINFO_ACCESS_FLAGS    flags,
                                    CORINFO_FIELD_INFO*     pResult)
{
    if (GetFieldInfo == nullptr)
        GetFieldInfo = new LightWeightMap<Agnostic_GetFieldInfo, Agnostic_CORINFO_FIELD_INFO>();

    Agnostic_GetFieldInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetFieldInfo);
    key.callerHandle  = CastHandle(callerHandle);
    key.flags         = (DWORD)flags;

    Agnostic_CORINFO_FIELD_INFO value;
    value.fieldAccessor                 = (DWORD)pResult->fieldAccessor;
    value.fieldFlags                    = (DWORD)pResult->fieldFlags;
    value.helper                        = (DWORD)pResult->helper;
    value.offset                        = (DWORD)pResult->offset;
    value.fieldType                     = (DWORD)pResult->fieldType;
    value.structType                    = CastHandle(pResult->structType);
    value.accessAllowed                 = (DWORD)pResult->accessAllowed;
    value.accessCalloutHelper.helperNum = (DWORD)pResult->accessCalloutHelper.helperNum;
    value.accessCalloutHelper.numArgs   = (DWORD)pResult->accessCalloutHelper.numArgs;
    value.fieldLookup                   = SpmiRecordsHelper::StoreAgnostic_CORINFO_CONST_LOOKUP(&pResult->fieldLookup);
    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        value.accessCalloutHelper.args[i].constant = (DWORDLONG)pResult->accessCalloutHelper.args[i].constant;
        value.accessCalloutHelper.args[i].argType  = (DWORD)pResult->accessCalloutHelper.args[i].argType;
    }
    GetFieldInfo->Add(key, value);
    DEBUG_REC(dmpGetFieldInfo(key, value));
}
void MethodContext::dmpGetFieldInfo(const Agnostic_GetFieldInfo& key, const Agnostic_CORINFO_FIELD_INFO& value)
{
    printf("GetFieldInfo key ch-%016llX flg-%08X rt{%s}", key.callerHandle, key.flags,
           SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ResolvedToken).c_str());

    printf(", value fa-%u fflg-%08X hlp-%u off-%u fT-%u(%s) sT-%016llX aa-%u hnum-%u na-%u {", value.fieldAccessor,
           value.fieldFlags, value.helper, value.offset, value.fieldType, toString((CorInfoType)value.fieldType),
           value.structType, value.accessAllowed, value.accessCalloutHelper.helperNum,
           value.accessCalloutHelper.numArgs);

    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        switch ((CorInfoAccessAllowedHelperArgType)value.accessCalloutHelper.args[i].argType)
        {
            default:
                printf("{%u: illegal}", i);
                break;
            case CORINFO_HELPER_ARG_TYPE_Field:
                printf("{%u: fld-%016llX}", i, value.accessCalloutHelper.args[i].constant);
                break;
            case CORINFO_HELPER_ARG_TYPE_Method:
                printf("{%u: mth-%016llX}", i, value.accessCalloutHelper.args[i].constant);
                break;
            case CORINFO_HELPER_ARG_TYPE_Class:
                printf("{%u: cls-%016llX}", i, value.accessCalloutHelper.args[i].constant);
                break;
            case CORINFO_HELPER_ARG_TYPE_Module:
                printf("{%u: mod-%016llX}", i, value.accessCalloutHelper.args[i].constant);
                break;
            case CORINFO_HELPER_ARG_TYPE_Const:
                printf("{%u: const-%016llX}", i, value.accessCalloutHelper.args[i].constant);
                break;
        }
    }
    printf(" fl %s}", SpmiDumpHelper::DumpAgnostic_CORINFO_CONST_LOOKUP(value.fieldLookup).c_str());
}
void MethodContext::repGetFieldInfo(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                    CORINFO_METHOD_HANDLE   callerHandle,
                                    CORINFO_ACCESS_FLAGS    flags,
                                    CORINFO_FIELD_INFO*     pResult)
{
    AssertMapExists(GetFieldInfo, ": key %x", pResolvedToken->token);

    Agnostic_GetFieldInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, GetFieldInfo);
    key.callerHandle  = CastHandle(callerHandle);
    key.flags         = (DWORD)flags;

    DWORD origFlag = key.flags;

    if (GetFieldInfo->GetIndex(key) == -1)
    {
#ifdef sparseMC
        key.flags = origFlag ^ ((DWORD)CORINFO_ACCESS_THIS);
        if (GetFieldInfo->GetIndex(key) != -1)
        {
            LogDebug(
                "Sparse - repGetFieldInfo found value with inverted CORINFO_ACCESS_THIS");
        }
        else
        {
            key.flags = origFlag ^ (DWORD)CORINFO_ACCESS_INLINECHECK;
            if (GetFieldInfo->GetIndex(key) != -1)
            {
                LogDebug("Sparse - repGetFieldInfo found value with inverted CORINFO_ACCESS_INLINECHECK");
            }
            else
            {
                LogException(EXCEPTIONCODE_MC, "repGetFieldInfo: didn't find %x", pResolvedToken->token);
            }
        }
#else
        LogException(EXCEPTIONCODE_MC, "repGetFieldInfo: didn't find %x", pResolvedToken->token);
#endif
    }

    Agnostic_CORINFO_FIELD_INFO value = GetFieldInfo->Get(key);
    DEBUG_REP(dmpGetFieldInfo(key, value));

    pResult->fieldAccessor                 = (CORINFO_FIELD_ACCESSOR)value.fieldAccessor;
    pResult->fieldFlags                    = (unsigned)value.fieldFlags;
    pResult->helper                        = (CorInfoHelpFunc)value.helper;
    pResult->offset                        = (DWORD)value.offset;
    pResult->fieldType                     = (CorInfoType)value.fieldType;
    pResult->structType                    = (CORINFO_CLASS_HANDLE)value.structType;
    pResult->accessAllowed                 = (CorInfoIsAccessAllowedResult)value.accessAllowed;
    pResult->accessCalloutHelper.helperNum = (CorInfoHelpFunc)value.accessCalloutHelper.helperNum;
    pResult->accessCalloutHelper.numArgs   = (unsigned)value.accessCalloutHelper.numArgs;
    pResult->fieldLookup                   = SpmiRecordsHelper::RestoreCORINFO_CONST_LOOKUP(value.fieldLookup);
    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        pResult->accessCalloutHelper.args[i].constant = (size_t)value.accessCalloutHelper.args[i].constant;
        pResult->accessCalloutHelper.args[i].argType =
            (CorInfoAccessAllowedHelperArgType)value.accessCalloutHelper.args[i].argType;
    }
}

void MethodContext::recEmbedMethodHandle(CORINFO_METHOD_HANDLE handle,
                                         void**                ppIndirection,
                                         CORINFO_METHOD_HANDLE result)
{
    if (EmbedMethodHandle == nullptr)
        EmbedMethodHandle = new LightWeightMap<DWORDLONG, DLDL>();

    DLDL value;
    if (ppIndirection == nullptr)
        value.A = 0;
    else
        value.A = CastPointer(*ppIndirection);
    value.B     = CastHandle(result);

    DWORDLONG key = CastHandle(handle);
    EmbedMethodHandle->Add(key, value);
    DEBUG_REC(dmpEmbedMethodHandle(key, value));
}
void MethodContext::dmpEmbedMethodHandle(DWORDLONG key, DLDL value)
{
    printf("EmbedMethodHandle key ftn-%016llX, value pp-%016llX res-%016llX", key, value.A, value.B);
}
CORINFO_METHOD_HANDLE MethodContext::repEmbedMethodHandle(CORINFO_METHOD_HANDLE handle, void** ppIndirection)
{
    DWORDLONG key = CastHandle(handle);
    AssertMapAndKeyExist(EmbedMethodHandle, key, ": key %016llX", key);

    DLDL value = EmbedMethodHandle->Get(key);
    DEBUG_REP(dmpEmbedMethodHandle(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_METHOD_HANDLE)value.B;
}

void MethodContext::recGetFieldAddress(CORINFO_FIELD_HANDLE field, void** ppIndirection, void* result, CorInfoType cit)
{
    if (GetFieldAddress == nullptr)
        GetFieldAddress = new LightWeightMap<DWORDLONG, Agnostic_GetFieldAddress>();

    Agnostic_GetFieldAddress value;
    if (ppIndirection == nullptr)
        value.ppIndirection = 0;
    else
        value.ppIndirection = CastPointer(*ppIndirection);
    value.fieldAddress      = CastPointer(result);

    value.fieldValue = (DWORD)-1;

    AssertCodeMsg(isReadyToRunCompilation != ReadyToRunCompilation::Uninitialized, EXCEPTIONCODE_MC,
                  "ReadyToRun flag should be initialized");

    // Make an attempt at stashing a copy of the value, Jit can try to access
    // a static readonly field value.
    if (isReadyToRunCompilation == ReadyToRunCompilation::NotReadyToRun &&
        result > (void*)0xffff)
    {
        DWORDLONG scratch = 0x4242424242424242;
        switch (cit)
        {
            case CORINFO_TYPE_BOOL:
            case CORINFO_TYPE_BYTE:
            case CORINFO_TYPE_UBYTE:
                value.fieldValue =
                    (DWORD)GetFieldAddress->AddBuffer((unsigned char*)result, sizeof(BYTE),
                                                      true); // important to not merge two fields into one address
                break;
            case CORINFO_TYPE_CHAR:
            case CORINFO_TYPE_SHORT:
            case CORINFO_TYPE_USHORT:
                value.fieldValue =
                    (DWORD)GetFieldAddress->AddBuffer((unsigned char*)result, sizeof(WORD),
                                                      true); // important to not merge two fields into one address
                break;
            case CORINFO_TYPE_INT:
            case CORINFO_TYPE_UINT:
            case CORINFO_TYPE_FLOAT:
                value.fieldValue =
                    (DWORD)GetFieldAddress->AddBuffer((unsigned char*)result, sizeof(DWORD),
                                                      true); // important to not merge two fields into one address
                break;
            case CORINFO_TYPE_LONG:
            case CORINFO_TYPE_ULONG:
            case CORINFO_TYPE_DOUBLE:
                value.fieldValue =
                    (DWORD)GetFieldAddress->AddBuffer((unsigned char*)result, sizeof(DWORDLONG),
                                                      true); // important to not merge two fields into one address
                break;
            case CORINFO_TYPE_NATIVEINT:
            case CORINFO_TYPE_NATIVEUINT:
            case CORINFO_TYPE_PTR:
                value.fieldValue =
                    (DWORD)GetFieldAddress->AddBuffer((unsigned char*)result, sizeof(size_t),
                                                      true); // important to not merge two fields into one address
                GetFieldAddress->AddBuffer((unsigned char*)&scratch, sizeof(DWORD)); // Padding out the data so we
                                                                                     // can read it back "safetly"
                                                                                     // on x64
                break;
            default:
                break;
        }
    }

    DWORDLONG key = CastHandle(field);
    GetFieldAddress->Add(key, value);
    DEBUG_REC(dmpGetFieldAddress(key, value));
}
void MethodContext::dmpGetFieldAddress(DWORDLONG key, const Agnostic_GetFieldAddress& value)
{
    printf("GetFieldAddress key fld-%016llX, value ppi-%016llX addr-%016llX val-%u", key, value.ppIndirection,
           value.fieldAddress, value.fieldValue);
}
void* MethodContext::repGetFieldAddress(CORINFO_FIELD_HANDLE field, void** ppIndirection)
{
    DWORDLONG key = CastHandle(field);
    AssertMapAndKeyExist(GetFieldAddress, key, ": key %016llX", key);

    Agnostic_GetFieldAddress value = GetFieldAddress->Get(key);
    DEBUG_REP(dmpGetFieldAddress(key, value));

    AssertCodeMsg(isReadyToRunCompilation != ReadyToRunCompilation::Uninitialized,
        EXCEPTIONCODE_MC, "isReadyToRunCompilation should be initialized");

    if (ppIndirection != nullptr)
    {
        *ppIndirection = (void*)value.ppIndirection;
    }
    void* temp;

    if (value.fieldValue != (DWORD)-1)
    {
        temp = (void*)GetFieldAddress->GetBuffer(value.fieldValue);
        cr->recAddressMap((void*)value.fieldAddress, temp, toCorInfoSize(repGetFieldType(field, nullptr, nullptr)));
    }
    else
    {
        temp = (void*)value.fieldAddress;
    }

    return temp;
}

void MethodContext::recGetStaticFieldCurrentClass(CORINFO_FIELD_HANDLE field,
                                                  bool                 isSpeculative,
                                                  CORINFO_CLASS_HANDLE result)
{
    if (GetStaticFieldCurrentClass == nullptr)
        GetStaticFieldCurrentClass = new LightWeightMap<DWORDLONG, Agnostic_GetStaticFieldCurrentClass>();

    Agnostic_GetStaticFieldCurrentClass value;

    value.classHandle   = CastHandle(result);
    value.isSpeculative = isSpeculative;

    DWORDLONG key = CastHandle(field);
    GetStaticFieldCurrentClass->Add(key, value);
    DEBUG_REC(dmpGetStaticFieldCurrentClass(key, value));
}
void MethodContext::dmpGetStaticFieldCurrentClass(DWORDLONG key, const Agnostic_GetStaticFieldCurrentClass& value)
{
    printf("GetStaticFieldCurrentClass key fld-%016llX, value clsHnd-%016llX isSpeculative-%u", key, value.classHandle,
           value.isSpeculative);
}
CORINFO_CLASS_HANDLE MethodContext::repGetStaticFieldCurrentClass(CORINFO_FIELD_HANDLE field, bool* pIsSpeculative)
{
    DWORDLONG key = CastHandle(field);
    AssertMapAndKeyExist(GetStaticFieldCurrentClass, key, ": key %016llX", key);

    Agnostic_GetStaticFieldCurrentClass value = GetStaticFieldCurrentClass->Get(key);
    DEBUG_REP(dmpGetStaticFieldCurrentClass(key, value));

    if (pIsSpeculative != nullptr)
    {
        *pIsSpeculative = value.isSpeculative;
    }

    CORINFO_CLASS_HANDLE result = (CORINFO_CLASS_HANDLE)value.classHandle;
    return result;
}

void MethodContext::recGetClassGClayout(CORINFO_CLASS_HANDLE cls, BYTE* gcPtrs, unsigned len, unsigned result)
{
    if (GetClassGClayout == nullptr)
        GetClassGClayout = new LightWeightMap<DWORDLONG, Agnostic_GetClassGClayout>();

    Agnostic_GetClassGClayout value;

    value.gcPtrs_Index = (DWORD)GetClassGClayout->AddBuffer((unsigned char*)gcPtrs, len * sizeof(BYTE));
    value.len          = (DWORD)len;
    value.valCount     = (DWORD)result;

    DWORDLONG key = CastHandle(cls);
    GetClassGClayout->Add(key, value);
    DEBUG_REC(dmpGetClassGClayout(key, value));
}
void MethodContext::dmpGetClassGClayout(DWORDLONG key, const Agnostic_GetClassGClayout& value)
{
    printf("GetClassGCLayout key %016llX, value len %u cnt %u {", key, value.len, value.valCount);
    if (value.gcPtrs_Index != (DWORD)-1)
    {
        BYTE* ptr = (BYTE*)GetClassGClayout->GetBuffer(value.gcPtrs_Index);
        for (unsigned int i = 0; i < value.len; i++)
        {
            printf("0x%02x", ptr[i]);
            if (i + 1 < value.len)
                printf(",");
        }
        GetClassGClayout->Unlock();
    }
    printf("}");
}
unsigned MethodContext::repGetClassGClayout(CORINFO_CLASS_HANDLE cls, BYTE* gcPtrs)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassGClayout, key, ": key %016llX", key);

    Agnostic_GetClassGClayout value = GetClassGClayout->Get(key);
    DEBUG_REP(dmpGetClassGClayout(key, value));

    unsigned int len   = (unsigned int)value.len;
    unsigned int index = (unsigned int)value.gcPtrs_Index;

    if (index != (unsigned int)-1)
    {
        BYTE* ptr = (BYTE*)GetClassGClayout->GetBuffer(index);
        for (unsigned int i = 0; i < len; i++)
            gcPtrs[i]       = ptr[i];
    }
    return (unsigned)value.valCount;
}

void MethodContext::recGetClassAlignmentRequirement(CORINFO_CLASS_HANDLE cls, bool fDoubleAlignHint, unsigned result)
{
    if (GetClassAlignmentRequirement == nullptr)
        GetClassAlignmentRequirement = new LightWeightMap<DLD, DWORD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls);
    key.B = (DWORD)fDoubleAlignHint;

    DWORD value = (DWORD)result;
    GetClassAlignmentRequirement->Add(key, value);
    DEBUG_REC(dmpGetClassAlignmentRequirement(key, value));
}
void MethodContext::dmpGetClassAlignmentRequirement(DLD key, DWORD value)
{
    printf("GetClassAlignmentRequirement key %016llX %u, value %u", key.A, key.B, value);
}
unsigned MethodContext::repGetClassAlignmentRequirement(CORINFO_CLASS_HANDLE cls, bool fDoubleAlignHint)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls);
    key.B = (DWORD)fDoubleAlignHint;

    AssertMapAndKeyExist(GetClassAlignmentRequirement, key, ": key %016llX", key.A);

    DWORD value = GetClassAlignmentRequirement->Get(key);
    DEBUG_REP(dmpGetClassAlignmentRequirement(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recCanAccessClass(CORINFO_RESOLVED_TOKEN*      pResolvedToken,
                                      CORINFO_METHOD_HANDLE        callerHandle,
                                      CORINFO_HELPER_DESC*         pAccessHelper,
                                      CorInfoIsAccessAllowedResult result)
{
    if (CanAccessClass == nullptr)
        CanAccessClass = new LightWeightMap<Agnostic_CanAccessClassIn, Agnostic_CanAccessClassOut>();

    Agnostic_CanAccessClassIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, CanAccessClass);
    key.callerHandle  = CastHandle(callerHandle);

    Agnostic_CanAccessClassOut value;
    value.AccessHelper.helperNum = (DWORD)pAccessHelper->helperNum;
    value.AccessHelper.numArgs   = (DWORD)pAccessHelper->numArgs;
    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        value.AccessHelper.args[i].constant = (DWORDLONG)pAccessHelper->args[i].constant;
        value.AccessHelper.args[i].argType  = (DWORD)pAccessHelper->args[i].argType;
    }
    value.result = (DWORD)result;

    CanAccessClass->Add(key, value);
    DEBUG_REC(dmpCanAccessClass(key, value));
}
void MethodContext::dmpCanAccessClass(const Agnostic_CanAccessClassIn& key, const Agnostic_CanAccessClassOut& value)
{
    printf("CanAccessClass key rt{%s}, callerHandle %016llX\n",
           SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.ResolvedToken).c_str(), key.callerHandle);
    printf(", value hnum-%u na-%u {", value.AccessHelper.helperNum, value.AccessHelper.numArgs);
    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        printf("{%016llX %u}", value.AccessHelper.args[i].constant, value.AccessHelper.args[i].argType);
    }
    printf("} res-%u", value.result);
}
CorInfoIsAccessAllowedResult MethodContext::repCanAccessClass(CORINFO_RESOLVED_TOKEN* pResolvedToken,
                                                              CORINFO_METHOD_HANDLE   callerHandle,
                                                              CORINFO_HELPER_DESC*    pAccessHelper)
{
    AssertMapExists(CanAccessClass, ": key %016llX", CastHandle(pResolvedToken->hClass));

    Agnostic_CanAccessClassIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.ResolvedToken = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(pResolvedToken, CanAccessClass);
    key.callerHandle  = CastHandle(callerHandle);

    AssertKeyExists(CanAccessClass, key, ": key %016llX", CastHandle(pResolvedToken->hClass));

    Agnostic_CanAccessClassOut value = CanAccessClass->Get(key);
    DEBUG_REP(dmpCanAccessClass(key, value));

    pAccessHelper->helperNum = (CorInfoHelpFunc)value.AccessHelper.helperNum;
    pAccessHelper->numArgs   = (unsigned)value.AccessHelper.numArgs;
    for (int i = 0; i < CORINFO_ACCESS_ALLOWED_MAX_ARGS; i++)
    {
        pAccessHelper->args[i].constant = (size_t)value.AccessHelper.args[i].constant;
        pAccessHelper->args[i].argType  = (CorInfoAccessAllowedHelperArgType)value.AccessHelper.args[i].argType;
    }
    CorInfoIsAccessAllowedResult temp = (CorInfoIsAccessAllowedResult)value.result;
    return temp;
}

void MethodContext::recGetCastingHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken, bool fThrowing, CorInfoHelpFunc result)
{
    if (GetCastingHelper == nullptr)
        GetCastingHelper = new LightWeightMap<Agnostic_GetCastingHelper, DWORD>();

    Agnostic_GetCastingHelper key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hClass    = CastHandle(pResolvedToken->hClass);
    key.fThrowing = (DWORD)fThrowing;

    DWORD value = (DWORD)result;
    GetCastingHelper->Add(key, value);
    DEBUG_REC(dmpGetCastingHelper(key, value));
}
void MethodContext::dmpGetCastingHelper(const Agnostic_GetCastingHelper& key, DWORD value)
{
    printf("GetCastingHelper key cls-%016llX, thw-%u, value res-%u", key.hClass, key.fThrowing, value);
}
CorInfoHelpFunc MethodContext::repGetCastingHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken, bool fThrowing)
{
    Agnostic_GetCastingHelper key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hClass    = CastHandle(pResolvedToken->hClass);
    key.fThrowing = (DWORD)fThrowing;

    AssertMapAndKeyExist(GetCastingHelper, key, ": key %016llX", key.hClass);

    DWORD value = GetCastingHelper->Get(key);
    DEBUG_REP(dmpGetCastingHelper(key, value));
    CorInfoHelpFunc result = (CorInfoHelpFunc)value;
    return result;
}

void MethodContext::recEmbedModuleHandle(CORINFO_MODULE_HANDLE handle,
                                         void**                ppIndirection,
                                         CORINFO_MODULE_HANDLE result)
{
    if (EmbedModuleHandle == nullptr)
        EmbedModuleHandle = new LightWeightMap<DWORDLONG, DLDL>();

    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastHandle(result);

    DWORDLONG key = CastHandle(handle);
    EmbedModuleHandle->Add(key, value);
    DEBUG_REC(dmpEmbedModuleHandle(key, value));
}
void MethodContext::dmpEmbedModuleHandle(DWORDLONG key, DLDL value)
{
    printf("EmbedModuleHandle key mod-%016llX, value pp-%016llX res-%016llX", key, value.A, value.B);
}
CORINFO_MODULE_HANDLE MethodContext::repEmbedModuleHandle(CORINFO_MODULE_HANDLE handle, void** ppIndirection)
{
    DWORDLONG key = CastHandle(handle);
    AssertMapAndKeyExist(EmbedModuleHandle, key, ": key %016llX", key);

    DLDL value = EmbedModuleHandle->Get(key);
    DEBUG_REP(dmpEmbedModuleHandle(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_MODULE_HANDLE)value.B;
}

void MethodContext::recEmbedClassHandle(CORINFO_CLASS_HANDLE handle, void** ppIndirection, CORINFO_CLASS_HANDLE result)
{
    if (EmbedClassHandle == nullptr)
        EmbedClassHandle = new LightWeightMap<DWORDLONG, DLDL>();

    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastHandle(result);

    DWORDLONG key = CastHandle(handle);
    EmbedClassHandle->Add(key, value);
    DEBUG_REC(dmpEmbedClassHandle(key, value));
}
void MethodContext::dmpEmbedClassHandle(DWORDLONG key, DLDL value)
{
    printf("EmbedClassHandle key cls-%016llX, value pp-%016llX res-%016llX", key, value.A, value.B);
}
CORINFO_CLASS_HANDLE MethodContext::repEmbedClassHandle(CORINFO_CLASS_HANDLE handle, void** ppIndirection)
{
    DWORDLONG key = CastHandle(handle);
    AssertMapAndKeyExist(EmbedClassHandle, key, ": key %016llX", key);

    DLDL value = EmbedClassHandle->Get(key);
    DEBUG_REP(dmpEmbedClassHandle(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_CLASS_HANDLE)value.B;
}

void MethodContext::recPInvokeMarshalingRequired(CORINFO_METHOD_HANDLE method,
                                                 CORINFO_SIG_INFO*     callSiteSig,
                                                 bool                  result)
{
    if (PInvokeMarshalingRequired == nullptr)
        PInvokeMarshalingRequired = new LightWeightMap<MethodOrSigInfoValue, DWORD>();

    MethodOrSigInfoValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.method     = CastHandle(method);
    key.pSig_Index = (DWORD)PInvokeMarshalingRequired->AddBuffer((unsigned char*)callSiteSig->pSig, callSiteSig->cbSig);
    key.cbSig      = (DWORD)callSiteSig->cbSig;
    key.scope      = CastHandle(callSiteSig->scope);

    DWORD value = result ? 1 : 0;
    PInvokeMarshalingRequired->Add(key, value);
    DEBUG_REC(dmpPInvokeMarshalingRequired(key, value));
}
void MethodContext::dmpPInvokeMarshalingRequired(const MethodOrSigInfoValue& key, DWORD value)
{
    printf("PInvokeMarshalingRequired key mth-%016llX scp-%016llX sig-%s, value res-%u",
        key.method, key.scope,
        SpmiDumpHelper::DumpPSig(key.pSig_Index, key.cbSig, PInvokeMarshalingRequired).c_str(),
        value);
}
// Note the jit interface implementation seems to only care about scope and pSig from callSiteSig
bool MethodContext::repPInvokeMarshalingRequired(CORINFO_METHOD_HANDLE method, CORINFO_SIG_INFO* callSiteSig)
{
    if (PInvokeMarshalingRequired == nullptr) // so when we replay Checked on Release, we throw from lwm
        return true;                          // TODO-Cleanup: hackish...

    MethodOrSigInfoValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.method     = CastHandle(method);
    key.pSig_Index = (DWORD)PInvokeMarshalingRequired->Contains((unsigned char*)callSiteSig->pSig, callSiteSig->cbSig);
    key.cbSig      = (DWORD)callSiteSig->cbSig;
    key.scope      = CastHandle(callSiteSig->scope);

    AssertKeyExistsNoMessage(PInvokeMarshalingRequired, key);

    DWORD value = PInvokeMarshalingRequired->Get(key);
    DEBUG_REP(dmpPInvokeMarshalingRequired(key, value));
    return value != 0;
}

void MethodContext::recGetUnmanagedCallConv(CORINFO_METHOD_HANDLE    method,
                                            CORINFO_SIG_INFO*        callSiteSig,
                                            CorInfoCallConvExtension result,
                                            bool                     suppressGCTransitionResult)
{
    if (GetUnmanagedCallConv == nullptr)
        GetUnmanagedCallConv = new LightWeightMap<MethodOrSigInfoValue, DD>();

    MethodOrSigInfoValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.method = CastHandle(method);
    if (callSiteSig != nullptr)
    {
        key.pSig_Index = (DWORD)GetUnmanagedCallConv->AddBuffer((unsigned char*)callSiteSig->pSig, callSiteSig->cbSig);
        key.cbSig = (DWORD)callSiteSig->cbSig;
        key.scope = CastHandle(callSiteSig->scope);
    }
    else
    {
        key.pSig_Index = 0;
        key.cbSig = 0;
        key.scope = 0;
    }

    DD value;
    value.A = (DWORD)result;
    value.B = (DWORD)suppressGCTransitionResult;

    GetUnmanagedCallConv->Add(key, value);
    DEBUG_REC(dmpGetUnmanagedCallConv(key, value));
}
void MethodContext::dmpGetUnmanagedCallConv(const MethodOrSigInfoValue& key, DD value)
{
    printf("GetUnmanagedCallConv key mth-%016llX scp-%016llX sig-%s, value res-%u,%u",
        key.method, key.scope,
        SpmiDumpHelper::DumpPSig(key.pSig_Index, key.cbSig, GetUnmanagedCallConv).c_str(),
        value.A, value.B);
}
// Note the jit interface implementation seems to only care about scope and pSig from callSiteSig
CorInfoCallConvExtension MethodContext::repGetUnmanagedCallConv(CORINFO_METHOD_HANDLE method, CORINFO_SIG_INFO* callSiteSig, bool* pSuppressGCTransition)
{
    if (GetUnmanagedCallConv == nullptr)
    {
#ifdef sparseMC
        LogDebug("Sparse - repGetUnmanagedCallConv returning CorInfoCallConvExtension::Managed");
        return CorInfoCallConvExtension::Managed;
#else
        LogException(EXCEPTIONCODE_MC, "Found a null GetUnmanagedCallConv.  Probably missing a fatTrigger for %016llX.",
                     CastHandle(method));
#endif
    }

    MethodOrSigInfoValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.method = CastHandle(method);
    if (callSiteSig != nullptr)
    {
        key.pSig_Index = (DWORD)GetUnmanagedCallConv->Contains((unsigned char*)callSiteSig->pSig, callSiteSig->cbSig);
        key.cbSig = (DWORD)callSiteSig->cbSig;
        key.scope = CastHandle(callSiteSig->scope);
    }
    else
    {
        key.pSig_Index = 0;
        key.cbSig = 0;
        key.scope = 0;
    }

    AssertKeyExistsNoMessage(GetUnmanagedCallConv, key);

    DD value = GetUnmanagedCallConv->Get(key);
    DEBUG_REP(dmpGetUnmanagedCallConv(key, value));

    *pSuppressGCTransition = value.B != 0;
    return (CorInfoCallConvExtension)value.A;
}

void MethodContext::recFindSig(CORINFO_MODULE_HANDLE  moduleHandle,
                               unsigned               sigTOK,
                               CORINFO_CONTEXT_HANDLE context,
                               CORINFO_SIG_INFO*      sig)
{
    if (FindSig == nullptr)
        FindSig = new LightWeightMap<Agnostic_FindSig, Agnostic_CORINFO_SIG_INFO>();

    Agnostic_FindSig key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.module  = CastHandle(moduleHandle);
    key.sigTOK  = (DWORD)sigTOK;
    key.context = CastHandle(context);

    Agnostic_CORINFO_SIG_INFO value = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(*sig, FindSig, SigInstHandleMap);

    FindSig->Add(key, value);
    DEBUG_REC(dmpFindSig(key, value));
}
void MethodContext::dmpFindSig(const Agnostic_FindSig& key, const Agnostic_CORINFO_SIG_INFO& value)
{
    printf("FindSig key module-%016llX sigTOK-%08X context-%016llX", key.module, key.sigTOK, key.context);
    printf(", value-%s",
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value, FindSig, SigInstHandleMap).c_str());
}
void MethodContext::repFindSig(CORINFO_MODULE_HANDLE  moduleHandle,
                               unsigned               sigTOK,
                               CORINFO_CONTEXT_HANDLE context,
                               CORINFO_SIG_INFO*      sig)
{
    Agnostic_FindSig key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.module  = CastHandle(moduleHandle);
    key.sigTOK  = (DWORD)sigTOK;
    key.context = CastHandle(context);

    AssertMapAndKeyExistNoMessage(FindSig, key);

    Agnostic_CORINFO_SIG_INFO value = FindSig->Get(key);
    DEBUG_REP(dmpFindSig(key, value));

    *sig = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value, FindSig, SigInstHandleMap);
}

void MethodContext::recGetEEInfo(CORINFO_EE_INFO* pEEInfoOut)
{
    if (GetEEInfo == nullptr)
        GetEEInfo = new LightWeightMap<DWORD, Agnostic_CORINFO_EE_INFO>();

    Agnostic_CORINFO_EE_INFO value;

    value.inlinedCallFrameInfo.size                  = (DWORD)pEEInfoOut->inlinedCallFrameInfo.size;
    value.inlinedCallFrameInfo.offsetOfGSCookie      = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfGSCookie;
    value.inlinedCallFrameInfo.offsetOfFrameVptr     = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameVptr;
    value.inlinedCallFrameInfo.offsetOfFrameLink     = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameLink;
    value.inlinedCallFrameInfo.offsetOfCallSiteSP    = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfCallSiteSP;
    value.inlinedCallFrameInfo.offsetOfCalleeSavedFP = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfCalleeSavedFP;
    value.inlinedCallFrameInfo.offsetOfCallTarget    = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfCallTarget;
    value.inlinedCallFrameInfo.offsetOfReturnAddress = (DWORD)pEEInfoOut->inlinedCallFrameInfo.offsetOfReturnAddress;
    value.offsetOfThreadFrame                        = (DWORD)pEEInfoOut->offsetOfThreadFrame;
    value.offsetOfGCState                            = (DWORD)pEEInfoOut->offsetOfGCState;
    value.offsetOfDelegateInstance                   = (DWORD)pEEInfoOut->offsetOfDelegateInstance;
    value.offsetOfDelegateFirstTarget                = (DWORD)pEEInfoOut->offsetOfDelegateFirstTarget;
    value.offsetOfWrapperDelegateIndirectCell        = (DWORD)pEEInfoOut->offsetOfWrapperDelegateIndirectCell;
    value.sizeOfReversePInvokeFrame                  = (DWORD)pEEInfoOut->sizeOfReversePInvokeFrame;
    value.osPageSize                                 = (DWORD)pEEInfoOut->osPageSize;
    value.maxUncheckedOffsetForNullObject            = (DWORD)pEEInfoOut->maxUncheckedOffsetForNullObject;
    value.targetAbi                                  = (DWORD)pEEInfoOut->targetAbi;
    value.osType                                     = (DWORD)pEEInfoOut->osType;

    GetEEInfo->Add(0, value);
    DEBUG_REC(dmpGetEEInfo(0, value));
}
void MethodContext::dmpGetEEInfo(DWORD key, const Agnostic_CORINFO_EE_INFO& value)
{
    printf("GetEEInfo key %u, value icfi{sz-%u ogs-%u ofv-%u ofl-%u ocsp-%u ocsfp-%u oct-%u ora-%u} "
           "otf-%u ogcs-%u odi-%u odft-%u osdic-%u srpf-%u osps-%u muono-%u tabi-%u osType-%u",
           key, value.inlinedCallFrameInfo.size, value.inlinedCallFrameInfo.offsetOfGSCookie,
           value.inlinedCallFrameInfo.offsetOfFrameVptr, value.inlinedCallFrameInfo.offsetOfFrameLink,
           value.inlinedCallFrameInfo.offsetOfCallSiteSP, value.inlinedCallFrameInfo.offsetOfCalleeSavedFP,
           value.inlinedCallFrameInfo.offsetOfCallTarget, value.inlinedCallFrameInfo.offsetOfReturnAddress,
           value.offsetOfThreadFrame, value.offsetOfGCState, value.offsetOfDelegateInstance,
           value.offsetOfDelegateFirstTarget, value.offsetOfWrapperDelegateIndirectCell,
           value.sizeOfReversePInvokeFrame, value.osPageSize, value.maxUncheckedOffsetForNullObject, value.targetAbi,
           value.osType);
}
void MethodContext::repGetEEInfo(CORINFO_EE_INFO* pEEInfoOut)
{
    Agnostic_CORINFO_EE_INFO value;

    int index = -1;
    if (GetEEInfo != nullptr)
        index = GetEEInfo->GetIndex((DWORD)0);
    if (index >= 0)
    {
        value = GetEEInfo->Get(0);
        DEBUG_REP(dmpGetEEInfo(0, value));

        pEEInfoOut->inlinedCallFrameInfo.size               = (unsigned)value.inlinedCallFrameInfo.size;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfGSCookie   = (unsigned)value.inlinedCallFrameInfo.offsetOfGSCookie;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameVptr  = (unsigned)value.inlinedCallFrameInfo.offsetOfFrameVptr;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameLink  = (unsigned)value.inlinedCallFrameInfo.offsetOfFrameLink;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCallSiteSP = (unsigned)value.inlinedCallFrameInfo.offsetOfCallSiteSP;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCalleeSavedFP =
            (unsigned)value.inlinedCallFrameInfo.offsetOfCalleeSavedFP;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCallTarget = (unsigned)value.inlinedCallFrameInfo.offsetOfCallTarget;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfReturnAddress =
            (unsigned)value.inlinedCallFrameInfo.offsetOfReturnAddress;
        pEEInfoOut->offsetOfThreadFrame                = (unsigned)value.offsetOfThreadFrame;
        pEEInfoOut->offsetOfGCState                    = (unsigned)value.offsetOfGCState;
        pEEInfoOut->offsetOfDelegateInstance           = (unsigned)value.offsetOfDelegateInstance;
        pEEInfoOut->offsetOfDelegateFirstTarget        = (unsigned)value.offsetOfDelegateFirstTarget;
        pEEInfoOut->offsetOfWrapperDelegateIndirectCell= (unsigned)value.offsetOfWrapperDelegateIndirectCell;
        pEEInfoOut->sizeOfReversePInvokeFrame          = (unsigned)value.sizeOfReversePInvokeFrame;
        pEEInfoOut->osPageSize                         = (size_t)value.osPageSize;
        pEEInfoOut->maxUncheckedOffsetForNullObject    = (size_t)value.maxUncheckedOffsetForNullObject;
        pEEInfoOut->targetAbi                          = (CORINFO_RUNTIME_ABI)value.targetAbi;
        pEEInfoOut->osType                             = (CORINFO_OS)value.osType;
    }
    else
    {
        pEEInfoOut->inlinedCallFrameInfo.size                  = (unsigned)0x40;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfGSCookie      = (unsigned)0;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameVptr     = (unsigned)0x8;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfFrameLink     = (unsigned)0x10;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCallSiteSP    = (unsigned)0x28;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCalleeSavedFP = (unsigned)0x38;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfCallTarget    = (unsigned)0x18;
        pEEInfoOut->inlinedCallFrameInfo.offsetOfReturnAddress = (unsigned)0x30;
        pEEInfoOut->offsetOfThreadFrame                        = (unsigned)0x10;
        pEEInfoOut->offsetOfGCState                            = (unsigned)0xc;
        pEEInfoOut->offsetOfDelegateInstance                   = (unsigned)0x8;
        pEEInfoOut->offsetOfDelegateFirstTarget                = (unsigned)0x18;
        pEEInfoOut->offsetOfWrapperDelegateIndirectCell        = (unsigned)0x40;
        pEEInfoOut->sizeOfReversePInvokeFrame                  = (unsigned)0x8;
        pEEInfoOut->osPageSize                                 = (size_t)0x1000;
        pEEInfoOut->maxUncheckedOffsetForNullObject            = (size_t)((32 * 1024) - 1);
        pEEInfoOut->targetAbi                                  = CORINFO_DESKTOP_ABI;
#ifdef TARGET_UNIX
        pEEInfoOut->osType                                     = CORINFO_UNIX;
#else
        pEEInfoOut->osType                                     = CORINFO_WINNT;
#endif
    }
}

void MethodContext::recGetGSCookie(GSCookie* pCookieVal, GSCookie** ppCookieVal)
{
    if (GetGSCookie == nullptr)
        GetGSCookie = new LightWeightMap<DWORD, DLDL>();

    DLDL value;

    if (pCookieVal != nullptr)
        value.A = (DWORDLONG)*pCookieVal;
    else
        value.A = 0;

    if (ppCookieVal != nullptr)
        value.B = CastPointer(*ppCookieVal);
    else
        value.B = 0;

    GetGSCookie->Add(0, value);
    DEBUG_REC(dmpGetGSCookie(0, value));
}
void MethodContext::dmpGetGSCookie(DWORD key, DLDL value)
{
    printf("GetGSCookie key 0, value pCookieVal-%016llX ppCookieVal-%016llX", value.A, value.B);
}
void MethodContext::repGetGSCookie(GSCookie* pCookieVal, GSCookie** ppCookieVal)
{
    if (GetGSCookie == nullptr)
    {
        // fake the result because for the codegen it is just a constant.
        if (pCookieVal != nullptr)
        {
            *pCookieVal = (GSCookie)0x06000000;
        }
        if (ppCookieVal != nullptr)
        {
            *ppCookieVal = (GSCookie*)0x06000001;
        }
        return;
    }

    AssertMapAndKeyExistNoMessage(GetGSCookie, 0);

    DLDL value = GetGSCookie->Get(0);
    DEBUG_REP(dmpGetGSCookie(0, value));

    if (pCookieVal != nullptr)
        *pCookieVal = (GSCookie)value.A;
    if (ppCookieVal != nullptr)
        *ppCookieVal = (GSCookie*)value.B;
}

void MethodContext::recGetOSRInfo(PatchpointInfo* patchpointInfo, unsigned* ilOffset)
{
    if (GetOSRInfo == nullptr)
    {
        GetOSRInfo = new LightWeightMap<DWORD, Agnostic_GetOSRInfo>();
    }

    Agnostic_GetOSRInfo value;

    value.index = (DWORD)GetOSRInfo->AddBuffer((const unsigned char*) patchpointInfo, patchpointInfo->PatchpointInfoSize());
    value.ilOffset = *ilOffset;

    // use 0 for key
    DWORD key = 0;
    GetOSRInfo->Add(key, value);
    DEBUG_REC(dmpGetOSRInfo(key, value));
}

void MethodContext::dmpGetOSRInfo(DWORD key, const Agnostic_GetOSRInfo& value)
{
    // todo - dump patchpoint info?
    printf("GetOSRInfo key %u, value patchpointInfo-%u {...} iloffset-%u\n",
        key, value.index, value.ilOffset);
}

PatchpointInfo* MethodContext::repGetOSRInfo(unsigned* ilOffset)
{
    DWORD key = 0;

    AssertMapAndKeyExistNoMessage(GetOSRInfo, key);

    Agnostic_GetOSRInfo value = GetOSRInfo->Get(key);
    DEBUG_REP(dmpGetOSRInfo(key, value));

    *ilOffset = value.ilOffset;
    return (PatchpointInfo*)GetOSRInfo->GetBuffer(value.index);
}

void MethodContext::recGetClassModuleIdForStatics(CORINFO_CLASS_HANDLE   cls,
                                                  CORINFO_MODULE_HANDLE* pModule,
                                                  void**                 ppIndirection,
                                                  size_t                 result)
{
    if (GetClassModuleIdForStatics == nullptr)
        GetClassModuleIdForStatics = new LightWeightMap<DWORDLONG, Agnostic_GetClassModuleIdForStatics>();

    Agnostic_GetClassModuleIdForStatics value;

    if (pModule != nullptr)
        value.Module = CastHandle(*pModule);
    else
        value.Module = 0;
    if (ppIndirection != nullptr)
        value.pIndirection = CastPointer(*ppIndirection);
    else
        value.pIndirection = 0;
    value.result           = (DWORDLONG)result;

    DWORDLONG key = CastHandle(cls);
    GetClassModuleIdForStatics->Add(key, value);
    DEBUG_REC(dmpGetClassModuleIdForStatics(key, value));
}
void MethodContext::dmpGetClassModuleIdForStatics(DWORDLONG key, const Agnostic_GetClassModuleIdForStatics& value)
{
    printf("GetClassModuleIdForStatics key cls-%016llX, value mod-%016llX pp-%016llX res-%016llX", key, value.Module,
           value.pIndirection, value.result);
}
size_t MethodContext::repGetClassModuleIdForStatics(CORINFO_CLASS_HANDLE   cls,
                                                    CORINFO_MODULE_HANDLE* pModule,
                                                    void**                 ppIndirection)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassModuleIdForStatics, key, ": key %016llX", key);

    Agnostic_GetClassModuleIdForStatics value = GetClassModuleIdForStatics->Get(key);
	DEBUG_REP(dmpGetClassModuleIdForStatics(key, value));

    if (pModule != nullptr)
        *pModule = (CORINFO_MODULE_HANDLE)value.Module;
    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.pIndirection;
    return (size_t)value.result;
}

void MethodContext::recGetThreadTLSIndex(void** ppIndirection, DWORD result)
{
    if (GetThreadTLSIndex == nullptr)
        GetThreadTLSIndex = new LightWeightMap<DWORD, DLD>();

    DLD value;

    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = (DWORD)result;

    GetThreadTLSIndex->Add(0, value);
}
void MethodContext::dmpGetThreadTLSIndex(DWORD key, DLD value)
{
    printf("GetThreadTLSIndex key 0, value ppIndirection-%016llX result-%08X", value.A, value.B);
}
DWORD MethodContext::repGetThreadTLSIndex(void** ppIndirection)
{
    AssertMapAndKeyExistNoMessage(GetThreadTLSIndex, 0);

    DLD value = GetThreadTLSIndex->Get(0);
	DEBUG_REP(dmpGetThreadTLSIndex(0, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (DWORD)value.B;
}

void MethodContext::recGetInlinedCallFrameVptr(void** ppIndirection, const void* result)
{
    if (GetInlinedCallFrameVptr == nullptr)
        GetInlinedCallFrameVptr = new LightWeightMap<DWORD, DLDL>();

    DLDL value;

    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastPointer(result);

    GetInlinedCallFrameVptr->Add(0, value);
}
void MethodContext::dmpGetInlinedCallFrameVptr(DWORD key, DLDL value)
{
    printf("GetInlinedCallFrameVptr key 0, value ppIndirection-%016llX result-%016llX", value.A, value.B);
}
const void* MethodContext::repGetInlinedCallFrameVptr(void** ppIndirection)
{
    AssertMapAndKeyExistNoMessage(GetInlinedCallFrameVptr, 0);

    DLDL value = GetInlinedCallFrameVptr->Get(0);
	DEBUG_REP(dmpGetInlinedCallFrameVptr(0, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (const void*)value.B;
}

void MethodContext::recGetAddrOfCaptureThreadGlobal(void** ppIndirection, int32_t* result)
{
    if (GetAddrOfCaptureThreadGlobal == nullptr)
        GetAddrOfCaptureThreadGlobal = new LightWeightMap<DWORD, DLDL>();

    DLDL value;

    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastPointer(result);

    GetAddrOfCaptureThreadGlobal->Add(0, value);
    DEBUG_REC(dmpGetAddrOfCaptureThreadGlobal(0, value));
}
void MethodContext::dmpGetAddrOfCaptureThreadGlobal(DWORD key, DLDL value)
{
    printf("GetAddrOfCaptureThreadGlobal key %u, value ppi-%016llX res-%016llX", key, value.A, value.B);
}
int32_t* MethodContext::repGetAddrOfCaptureThreadGlobal(void** ppIndirection)
{
    if ((GetAddrOfCaptureThreadGlobal == nullptr) || (GetAddrOfCaptureThreadGlobal->GetIndex((DWORD)0) == -1))
    {
#ifdef sparseMC
        LogDebug("Sparse - repGetAddrOfCaptureThreadGlobal returning nullptr and 0xCAFE0001");
        if (ppIndirection != nullptr)
            *ppIndirection = nullptr;
        return (int32_t*)(size_t)0xCAFE0001;
#else
        LogException(EXCEPTIONCODE_MC, "Didn't find anything for GetAddrOfCaptureThreadGlobal", "");
#endif
    }

    DLDL value = GetAddrOfCaptureThreadGlobal->Get(0);
    DEBUG_REP(dmpGetAddrOfCaptureThreadGlobal(0, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (int32_t*)value.B;
}

void MethodContext::recGetClassDomainID(CORINFO_CLASS_HANDLE cls, void** ppIndirection, unsigned result)
{
    if (GetClassDomainID == nullptr)
        GetClassDomainID = new LightWeightMap<DWORDLONG, DLD>();

    DLD value;

    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = (DWORD)result;

    DWORDLONG key = CastHandle(cls);
    GetClassDomainID->Add(key, value);
    DEBUG_REC(dmpGetClassDomainID(key, value));
}
void MethodContext::dmpGetClassDomainID(DWORDLONG key, DLD value)
{
    printf("GetClassDomainID key cls-%016llX, value pp-%016llX res-%u", key, value.A, value.B);
}
unsigned MethodContext::repGetClassDomainID(CORINFO_CLASS_HANDLE cls, void** ppIndirection)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetClassDomainID, key, ": key %016llX", key);

    DLD value = GetClassDomainID->Get(key);
    DEBUG_REP(dmpGetClassDomainID(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (unsigned)value.B;
}

void MethodContext::recGetLocationOfThisType(CORINFO_METHOD_HANDLE context, CORINFO_LOOKUP_KIND* result)
{
    if (GetLocationOfThisType == nullptr)
        GetLocationOfThisType = new LightWeightMap<DWORDLONG, Agnostic_CORINFO_LOOKUP_KIND>();

    Agnostic_CORINFO_LOOKUP_KIND value = SpmiRecordsHelper::CreateAgnostic_CORINFO_LOOKUP_KIND(result);

    DWORDLONG key = CastHandle(context);
    GetLocationOfThisType->Add(key, value);
    DEBUG_REC(dmpGetLocationOfThisType(key, value));
}
void MethodContext::dmpGetLocationOfThisType(DWORDLONG key, const Agnostic_CORINFO_LOOKUP_KIND& value)
{
    printf("GetLocationOfThisType key ftn-%016llX, value %s", key,
           SpmiDumpHelper::DumpAgnostic_CORINFO_LOOKUP_KIND(value).c_str());
}
void MethodContext::repGetLocationOfThisType(CORINFO_METHOD_HANDLE context, CORINFO_LOOKUP_KIND* pLookupKind)
{
    DWORDLONG key = CastHandle(context);
    AssertMapAndKeyExist(GetLocationOfThisType, key, ": key %016llX", key);
    Agnostic_CORINFO_LOOKUP_KIND value = GetLocationOfThisType->Get(key);
	DEBUG_REP(dmpGetLocationOfThisType(key, value));
    *pLookupKind = SpmiRecordsHelper::RestoreCORINFO_LOOKUP_KIND(value);
}

void MethodContext::recGetDelegateCtor(CORINFO_METHOD_HANDLE methHnd,
                                       CORINFO_CLASS_HANDLE  clsHnd,
                                       CORINFO_METHOD_HANDLE targetMethodHnd,
                                       DelegateCtorArgs*     pCtorData,
                                       CORINFO_METHOD_HANDLE result)
{
    if (GetDelegateCtor == nullptr)
        GetDelegateCtor = new LightWeightMap<Agnostic_GetDelegateCtorIn, Agnostic_GetDelegateCtorOut>();

    Agnostic_GetDelegateCtorIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.methHnd         = CastHandle(methHnd);
    key.clsHnd          = CastHandle(clsHnd);
    key.targetMethodHnd = CastHandle(targetMethodHnd);

    Agnostic_GetDelegateCtorOut value;
    value.CtorData.pMethod = CastPointer(pCtorData->pMethod);
    value.CtorData.pArg3   = CastPointer(pCtorData->pArg3);
    value.CtorData.pArg4   = CastPointer(pCtorData->pArg4);
    value.CtorData.pArg5   = CastPointer(pCtorData->pArg5);
    value.result           = CastHandle(result);

    GetDelegateCtor->Add(key, value);
    DEBUG_REC(dmpGetDelegateCtor(key, value));
}
void MethodContext::dmpGetDelegateCtor(const Agnostic_GetDelegateCtorIn& key, const Agnostic_GetDelegateCtorOut& value)
{
    printf("GetDelegateCtor key ftn-%016llX cls-%016llX tftn-%016llX, value pm-%016llX a3-%016llX a4-%016llX "
           "a5-%016llX res-%016llX",
           key.methHnd, key.clsHnd, key.targetMethodHnd, value.CtorData.pMethod, value.CtorData.pArg3,
           value.CtorData.pArg4, value.CtorData.pArg5, value.result);
}
CORINFO_METHOD_HANDLE MethodContext::repGetDelegateCtor(CORINFO_METHOD_HANDLE methHnd,
                                                        CORINFO_CLASS_HANDLE  clsHnd,
                                                        CORINFO_METHOD_HANDLE targetMethodHnd,
                                                        DelegateCtorArgs*     pCtorData)
{
    Agnostic_GetDelegateCtorIn key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.methHnd         = CastHandle(methHnd);
    key.clsHnd          = CastHandle(clsHnd);
    key.targetMethodHnd = CastHandle(targetMethodHnd);

    AssertMapAndKeyExist(GetDelegateCtor, key, ": key %016llX", key.methHnd);

    Agnostic_GetDelegateCtorOut value = GetDelegateCtor->Get(key);
    DEBUG_REP(dmpGetDelegateCtor(key, value));

    pCtorData->pMethod = (void*)value.CtorData.pMethod;
    pCtorData->pArg3   = (void*)value.CtorData.pArg3;
    pCtorData->pArg4   = (void*)value.CtorData.pArg4;
    pCtorData->pArg5   = (void*)value.CtorData.pArg5;
    return (CORINFO_METHOD_HANDLE)value.result;
}

void MethodContext::recGetFunctionFixedEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP* pResult)
{
    if (GetFunctionFixedEntryPoint == nullptr)
        GetFunctionFixedEntryPoint = new LightWeightMap<DWORDLONG, Agnostic_CORINFO_CONST_LOOKUP>();

    Agnostic_CORINFO_CONST_LOOKUP value = SpmiRecordsHelper::StoreAgnostic_CORINFO_CONST_LOOKUP(pResult);

    DWORDLONG key = CastHandle(ftn);
    GetFunctionFixedEntryPoint->Add(key, value);
    DEBUG_REC(dmpGetFunctionFixedEntryPoint(key, value));
}
void MethodContext::dmpGetFunctionFixedEntryPoint(DWORDLONG key, const Agnostic_CORINFO_CONST_LOOKUP& value)
{
    printf("GetFunctionFixedEntryPoint key ftn-%016llX, value %s", key,
           SpmiDumpHelper::DumpAgnostic_CORINFO_CONST_LOOKUP(value).c_str());
}
void MethodContext::repGetFunctionFixedEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP* pResult)
{
    DWORDLONG key = CastHandle(ftn);
    AssertMapAndKeyExist(GetFunctionFixedEntryPoint, key, ": key %016llX", key);
    Agnostic_CORINFO_CONST_LOOKUP value = GetFunctionFixedEntryPoint->Get(key);
    DEBUG_REP(dmpGetFunctionFixedEntryPoint(key, value));
    *pResult = SpmiRecordsHelper::RestoreCORINFO_CONST_LOOKUP(value);
}

void MethodContext::recGetFieldInClass(CORINFO_CLASS_HANDLE clsHnd, INT num, CORINFO_FIELD_HANDLE result)
{
    if (GetFieldInClass == nullptr)
        GetFieldInClass = new LightWeightMap<DLD, DWORDLONG>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(clsHnd);
    key.B = (DWORD)num;

    DWORDLONG value = CastHandle(result);
    GetFieldInClass->Add(key, value);
    DEBUG_REC(dmpGetFieldInClass(key, value));
}
void MethodContext::dmpGetFieldInClass(DLD key, DWORDLONG value)
{
    printf("GetFieldInClass key cls-%016llX ind-%u, value %016llX", key.A, key.B, value);
}
CORINFO_FIELD_HANDLE MethodContext::repGetFieldInClass(CORINFO_CLASS_HANDLE clsHnd, INT num)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(clsHnd);
    key.B = (DWORD)num;

    AssertMapAndKeyExist(GetFieldInClass, key, ": key %016llX", key.A);

    DWORDLONG value = GetFieldInClass->Get(key);
    DEBUG_REP(dmpGetFieldInClass(key, value));
    CORINFO_FIELD_HANDLE result = (CORINFO_FIELD_HANDLE)value;
    return result;
}

void MethodContext::recGetFieldType(CORINFO_FIELD_HANDLE  field,
                                    CORINFO_CLASS_HANDLE* structType,
                                    CORINFO_CLASS_HANDLE  memberParent,
                                    CorInfoType           result)
{
    if (GetFieldType == nullptr)
        GetFieldType = new LightWeightMap<DLDL, DLD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(field);
    key.B = CastHandle(memberParent);

    DLD value;
    value.B = (DWORD)result;
    if (structType == nullptr)
    {
        value.A = 0;
    }
    else
    {
        value.A = CastHandle(*structType);

        // If we had a previous call with null 'structType', we will not have captured the
        // class handle (we use only 'field' and 'memberParent' as keys).
        // Update the value in that case.
        unsigned index = GetFieldType->GetIndex(key);
        if ((index != (unsigned)-1) && (GetFieldType->GetItem(index).A == 0))
        {
            GetFieldType->Update(index, value);
            DEBUG_REC(dmpGetFieldType(key, value));
            return;
        }
    }
    GetFieldType->Add(key, value);
    DEBUG_REC(dmpGetFieldType(key, value));
}
void MethodContext::dmpGetFieldType(DLDL key, DLD value)
{
    printf("GetFieldType key fld-%016llX cls-%016llX, value ch-%016llX cit-%u(%s)", key.A, key.B, value.A, value.B,
           toString((CorInfoType)value.B));
}
CorInfoType MethodContext::repGetFieldType(CORINFO_FIELD_HANDLE  field,
                                           CORINFO_CLASS_HANDLE* structType,
                                           CORINFO_CLASS_HANDLE  memberParent)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(field);
    key.B = CastHandle(memberParent);

    AssertMapAndKeyExist(GetFieldType, key, ": key %016llX", key.A);

    DLD value = GetFieldType->Get(key);
    DEBUG_REP(dmpGetFieldType(key, value));

    if (structType != nullptr)
        *structType = (CORINFO_CLASS_HANDLE)value.A;
    return (CorInfoType)value.B;
}

void MethodContext::recGetFieldName(CORINFO_FIELD_HANDLE ftn, const char** moduleName, const char* result)
{
    if (GetFieldName == nullptr)
        GetFieldName = new LightWeightMap<DWORDLONG, DD>();

    DD value;

    if (result != nullptr)
        value.A = GetFieldName->AddBuffer((unsigned char*)result, (DWORD)strlen(result) + 1);
    else
        value.A = (DWORD)-1;

    if ((moduleName != nullptr) && (*moduleName != nullptr)) // protect strlen
        value.B = (DWORD)GetFieldName->AddBuffer((unsigned char*)*moduleName, (DWORD)strlen(*moduleName) + 1);
    else
        value.B = (DWORD)-1;

    DWORDLONG key = CastHandle(ftn);
    GetFieldName->Add(key, value);
    DEBUG_REC(dmpGetFieldName(key, value));
}
void MethodContext::dmpGetFieldName(DWORDLONG key, DD value)
{
    unsigned char* fieldName  = (unsigned char*)GetFieldName->GetBuffer(value.A);
    unsigned char* moduleName = (unsigned char*)GetFieldName->GetBuffer(value.B);
    printf("GetFieldName key - ftn-%016llX, value fld-'%s', mod-'%s'", key, fieldName, moduleName);
    GetFieldName->Unlock();
}
const char* MethodContext::repGetFieldName(CORINFO_FIELD_HANDLE ftn, const char** moduleName)
{
    if (GetFieldName == nullptr)
    {
        if (moduleName != nullptr)
            *moduleName = "hackishModuleName";
        return "hackishFieldName";
    }

    DWORDLONG key = CastHandle(ftn);
    DD value = GetFieldName->Get(key);
    DEBUG_REP(dmpGetFieldName(key, value));

    if (moduleName != nullptr)
        *moduleName = (const char*)GetFieldName->GetBuffer(value.B);
    return (const char*)GetFieldName->GetBuffer(value.A);
}

void MethodContext::recCanInlineTypeCheck(CORINFO_CLASS_HANDLE         cls,
                                          CorInfoInlineTypeCheckSource source,
                                          CorInfoInlineTypeCheck       result)
{
    if (CanInlineTypeCheck == nullptr)
        CanInlineTypeCheck = new LightWeightMap<DLD, DWORD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls);
    key.B = (DWORD)source;

    DWORD value = (DWORD)result;
    CanInlineTypeCheck->Add(key, value);
    DEBUG_REC(dmpCanInlineTypeCheck(key, value));
}
void MethodContext::dmpCanInlineTypeCheck(DLD key, DWORD value)
{
    printf("CanInlineTypeCheck key cls-%016llX src-%08X, value res-%u", key.A, key.B, value);
}
CorInfoInlineTypeCheck MethodContext::repCanInlineTypeCheck(CORINFO_CLASS_HANDLE         cls,
                                                            CorInfoInlineTypeCheckSource source)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls);
    key.B = (DWORD)source;

    AssertMapAndKeyExist(CanInlineTypeCheck, key, ": key %016llX", key.A);

    DWORD value = CanInlineTypeCheck->Get(key);
    DEBUG_REP(dmpCanInlineTypeCheck(key, value));
    CorInfoInlineTypeCheck result = (CorInfoInlineTypeCheck)value;
    return result;
}

void MethodContext::recSatisfiesMethodConstraints(CORINFO_CLASS_HANDLE  parent,
                                                  CORINFO_METHOD_HANDLE method,
                                                  bool                  result)
{
    if (SatisfiesMethodConstraints == nullptr)
        SatisfiesMethodConstraints = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(parent);
    key.B = CastHandle(method);

    DWORD value = result ? 1 : 0;
    SatisfiesMethodConstraints->Add(key, value);
    DEBUG_REC(dmpSatisfiesMethodConstraints(key, value));
}
void MethodContext::dmpSatisfiesMethodConstraints(DLDL key, DWORD value)
{
    printf("SatisfiesMethodConstraints key cls-%016llX ftn-%016llX, value res-%u", key.A, key.B, value);
}
bool MethodContext::repSatisfiesMethodConstraints(CORINFO_CLASS_HANDLE parent, CORINFO_METHOD_HANDLE method)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(parent);
    key.B = CastHandle(method);

    AssertMapAndKeyExistNoMessage(SatisfiesMethodConstraints, key);

    DWORD value = SatisfiesMethodConstraints->Get(key);
    DEBUG_REP(dmpSatisfiesMethodConstraints(key, value));
    return value != 0;
}

void MethodContext::recIsValidStringRef(CORINFO_MODULE_HANDLE module, unsigned metaTOK, bool result)
{
    if (IsValidStringRef == nullptr)
        IsValidStringRef = new LightWeightMap<DLD, DWORD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    DWORD value = result ? 1 : 0;
    IsValidStringRef->Add(key, value);
    DEBUG_REC(dmpIsValidStringRef(key, value));
}
void MethodContext::dmpIsValidStringRef(DLD key, DWORD value)
{
    printf("IsValidStringRef key mod-%016llX tok-%08X, value res-%u", key.A, key.B, value);
}
bool MethodContext::repIsValidStringRef(CORINFO_MODULE_HANDLE module, unsigned metaTOK)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    AssertMapAndKeyExistNoMessage(IsValidStringRef, key);

    DWORD value = IsValidStringRef->Get(key);
    DEBUG_REP(dmpIsValidStringRef(key, value));
    return value != 0;
}


void MethodContext::recGetStringLiteral(CORINFO_MODULE_HANDLE module, unsigned metaTOK, int length, const char16_t* result)
{
    if (GetStringLiteral == nullptr)
        GetStringLiteral = new LightWeightMap<DLD, DD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    DWORD strBuf = (DWORD)-1;
    if (result != nullptr)
        strBuf = (DWORD)GetStringLiteral->AddBuffer((unsigned char*)result, (unsigned int)((wcslen((LPCWSTR)result) * 2) + 2));

    DD value;
    value.A = (DWORD)length;
    value.B = (DWORD)strBuf;

    GetStringLiteral->Add(key, value);
    DEBUG_REC(dmpGetStringLiteral(key, value));
}

void MethodContext::dmpGetStringLiteral(DLD key, DD value)
{
    printf("GetStringLiteral key mod-%016llX tok-%08X, result-%s, len-%u", key.A, key.B,
        GetStringLiteral->GetBuffer(value.B), value.A);
    GetStringLiteral->Unlock();
}

const char16_t* MethodContext::repGetStringLiteral(CORINFO_MODULE_HANDLE module, unsigned metaTOK, int* length)
{
    if (GetStringLiteral == nullptr)
    {
        *length = -1;
        return nullptr;
    }

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    int itemIndex = GetStringLiteral->GetIndex(key);
    if (itemIndex < 0)
    {
        *length = -1;
        return nullptr;
    }
    else
    {
        DD value = GetStringLiteral->Get(key);
        DEBUG_REP(dmpGetStringLiteral(key, value));

        *length = (int)value.A;
        return (const char16_t*)GetStringLiteral->GetBuffer(value.B);
    }
}

void MethodContext::recGetHelperName(CorInfoHelpFunc funcNum, const char* result)
{
    if (GetHelperName == nullptr)
        GetHelperName = new LightWeightMap<DWORD, DWORD>();

    DWORD value = (DWORD)-1;
    if (result != nullptr)
        value = (DWORD)GetHelperName->AddBuffer((unsigned char*)result, (DWORD)strlen(result) + 1);

    DWORD key = (DWORD)funcNum;
    GetHelperName->Add(key, value);
    DEBUG_REC(dmpGetHelperName(key, value));
}
void MethodContext::dmpGetHelperName(DWORD key, DWORD value)
{
    printf("GetHelperName key ftn-%u, value '%s'", key, (const char*)GetHelperName->GetBuffer(value));
    GetHelperName->Unlock();
}
const char* MethodContext::repGetHelperName(CorInfoHelpFunc funcNum)
{
    if (GetHelperName == nullptr)
        return "Yickish helper name";

    DWORD key = (DWORD)funcNum;

    int itemIndex = GetHelperName->GetIndex(key);
    if (itemIndex < 0)
    {
        return "hackishHelperName";
    }
    else
    {
        DWORD value = GetHelperName->Get(key);
        DEBUG_REP(dmpGetHelperName(key, value));
        unsigned int buffIndex = (unsigned int)value;
        return (const char*)GetHelperName->GetBuffer(buffIndex);
    }
}

void MethodContext::recCanCast(CORINFO_CLASS_HANDLE child, CORINFO_CLASS_HANDLE parent, bool result)
{
    if (CanCast == nullptr)
        CanCast = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(child);
    key.B = CastHandle(parent);

    DWORD value = result ? 1 : 0;
    CanCast->Add(key, value);
    DEBUG_REC(dmpCanCast(key, value));
}
void MethodContext::dmpCanCast(DLDL key, DWORD value)
{
    printf("CanCast key chd-%016llX par-%016llX, value res-%u", key.A, key.B, value);
}
bool MethodContext::repCanCast(CORINFO_CLASS_HANDLE child, CORINFO_CLASS_HANDLE parent)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(child);
    key.B = CastHandle(parent);

    AssertMapAndKeyExist(CanCast, key, ": key %016llX %016llX", key.A, key.B);

    DWORD value = CanCast->Get(key);
    DEBUG_REP(dmpCanCast(key, value));
    return value != 0;
}

void MethodContext::recGetChildType(CORINFO_CLASS_HANDLE clsHnd, CORINFO_CLASS_HANDLE* clsRet, CorInfoType result)
{
    if (GetChildType == nullptr)
        GetChildType = new LightWeightMap<DWORDLONG, DLD>();

    DLD value;
    value.A = CastHandle(*clsRet);
    value.B = (DWORD)result;

    DWORDLONG key = CastHandle(clsHnd);
    GetChildType->Add(key, value);
    DEBUG_REC(dmpGetChildType(key, value));
}
void MethodContext::dmpGetChildType(DWORDLONG key, DLD value)
{
    printf("GetChildType key cls-%016llX, value clsr-%016llX cit-%u(%s)", key, value.A, value.B,
           toString((CorInfoType)value.B));
}
CorInfoType MethodContext::repGetChildType(CORINFO_CLASS_HANDLE clsHnd, CORINFO_CLASS_HANDLE* clsRet)
{
    DWORDLONG key = CastHandle(clsHnd);
    AssertMapAndKeyExist(GetChildType, key, ": key %016llX", key);

    DLD value = GetChildType->Get(key);
    DEBUG_REP(dmpGetChildType(key, value));

    *clsRet = (CORINFO_CLASS_HANDLE)value.A;
    return (CorInfoType)value.B;
}

void MethodContext::recGetArrayInitializationData(CORINFO_FIELD_HANDLE field, DWORD size, void* result)
{
    if (GetArrayInitializationData == nullptr)
        GetArrayInitializationData = new LightWeightMap<DLD, DWORDLONG>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(field);
    key.B = (DWORD)size;

    DWORDLONG value = CastPointer(result);
    GetArrayInitializationData->Add(key, value);
    DEBUG_REC(dmpGetArrayInitializationData(key, value));
}
void MethodContext::dmpGetArrayInitializationData(DLD key, DWORDLONG value)
{
    printf("GetArrayInitializationData key field-%016llX size-%08X, value result-%016llX", key.A, key.B, value);
}
void* MethodContext::repGetArrayInitializationData(CORINFO_FIELD_HANDLE field, DWORD size)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(field);
    key.B = (DWORD)size;

    AssertMapAndKeyExistNoMessage(GetArrayInitializationData, key);

    DWORDLONG value = GetArrayInitializationData->Get(key);
    DEBUG_REP(dmpGetArrayInitializationData(key, value));
    void* result = (void*)value;
    return result;
}

void MethodContext::recFilterException(struct _EXCEPTION_POINTERS* pExceptionPointers, int result)
{
    if (FilterException == nullptr)
        FilterException = new LightWeightMap<DWORD, DWORD>();

    DWORD key = (DWORD)pExceptionPointers->ExceptionRecord->ExceptionCode;
    DWORD value = (DWORD)result;
    FilterException->Add(key, value);
    DEBUG_REC(dmpFilterException(key, value));
}
void MethodContext::dmpFilterException(DWORD key, DWORD value)
{
    printf("FilterException key %u, value %u", key, value);
}
int MethodContext::repFilterException(struct _EXCEPTION_POINTERS* pExceptionPointers)
{
    if (FilterException == nullptr)
        return EXCEPTION_CONTINUE_SEARCH;
    if (FilterException->GetIndex((DWORD)pExceptionPointers->ExceptionRecord->ExceptionCode) < 0)
        return EXCEPTION_CONTINUE_SEARCH;
    else
    {
        DWORD key = (DWORD)pExceptionPointers->ExceptionRecord->ExceptionCode;
        DWORD value = FilterException->Get(key);
        DEBUG_REP(dmpFilterException(key, value));
        int result = (int)value;
        return result;
    }
}

void MethodContext::recGetAddressOfPInvokeTarget(CORINFO_METHOD_HANDLE method, CORINFO_CONST_LOOKUP* pLookup)
{
    if (GetAddressOfPInvokeTarget == nullptr)
        GetAddressOfPInvokeTarget = new LightWeightMap<DWORDLONG, DLD>();

    DLD value;
    value.A = CastPointer(pLookup->addr);
    value.B = (DWORD)pLookup->accessType;

    DWORDLONG key = CastHandle(method);
    GetAddressOfPInvokeTarget->Add(key, value);
    DEBUG_REC(dmpGetAddressOfPInvokeTarget(key, value));
}
void MethodContext::dmpGetAddressOfPInvokeTarget(DWORDLONG key, DLD value)
{
    printf("GetAddressOfPInvokeTarget key ftn-%016llX, value addr-%016llX at-%u", key, value.A, value.B);
}
void MethodContext::repGetAddressOfPInvokeTarget(CORINFO_METHOD_HANDLE method, CORINFO_CONST_LOOKUP* pLookup)
{
    DWORDLONG key = CastHandle(method);
    AssertMapAndKeyExist(GetAddressOfPInvokeTarget, key, ": key %016llX", key);

    DLD value = GetAddressOfPInvokeTarget->Get(key);
    DEBUG_REP(dmpGetAddressOfPInvokeTarget(key, value));

    pLookup->addr       = (void*)value.A;
    pLookup->accessType = (InfoAccessType)value.B;
}

void MethodContext::recSatisfiesClassConstraints(CORINFO_CLASS_HANDLE cls, bool result)
{
    if (SatisfiesClassConstraints == nullptr)
        SatisfiesClassConstraints = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = result ? 1 : 0;
    SatisfiesClassConstraints->Add(key, value);
    DEBUG_REC(dmpSatisfiesClassConstraints(key, value));
}
void MethodContext::dmpSatisfiesClassConstraints(DWORDLONG key, DWORD value)
{
    printf("SatisfiesClassConstraints key cls-%016llX, value res-%u", key, value);
}
bool MethodContext::repSatisfiesClassConstraints(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(SatisfiesClassConstraints, key, ": key %016llX", key);
    DWORD value = SatisfiesClassConstraints->Get(key);
    DEBUG_REP(dmpSatisfiesClassConstraints(key, value));
    return value != 0;
}

void MethodContext::recGetMethodHash(CORINFO_METHOD_HANDLE ftn, unsigned result)
{
    if (GetMethodHash == nullptr)
        GetMethodHash = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(ftn);
    DWORD value = (DWORD)result;
    GetMethodHash->Add(key, value);
    DEBUG_REC(dmpGetMethodHash(key, value));
}
void MethodContext::dmpGetMethodHash(DWORDLONG key, DWORD value)
{
    printf("GetMethodHash key %016llX, value %u", key, value);
}
unsigned MethodContext::repGetMethodHash(CORINFO_METHOD_HANDLE ftn)
{
    DWORDLONG key = CastHandle(ftn);
    unsigned result = 0x43;
    if (GetMethodHash != nullptr)
        if (GetMethodHash->GetIndex(key) >= 0)
            result = GetMethodHash->Get(key);
    DEBUG_REP(dmpGetMethodHash(key, (DWORD)result));
    return result;
}

void MethodContext::recCanTailCall(CORINFO_METHOD_HANDLE callerHnd,
                                   CORINFO_METHOD_HANDLE declaredCalleeHnd,
                                   CORINFO_METHOD_HANDLE exactCalleeHnd,
                                   bool                  fIsTailPrefix,
                                   bool                  result)
{
    if (CanTailCall == nullptr)
        CanTailCall = new LightWeightMap<Agnostic_CanTailCall, DWORD>();

    Agnostic_CanTailCall key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.callerHnd         = CastHandle(callerHnd);
    key.declaredCalleeHnd = CastHandle(declaredCalleeHnd);
    key.exactCalleeHnd    = CastHandle(exactCalleeHnd);
    key.fIsTailPrefix     = (DWORD)fIsTailPrefix;

    DWORD value = result ? 1 : 0;
    CanTailCall->Add(key, value);
    DEBUG_REC(dmpCanTailCall(key, value));
}
void MethodContext::dmpCanTailCall(const Agnostic_CanTailCall& key, DWORD value)
{
    printf("CanTailCall key clr-%016llX dcle-%016llX ecle-%016llX pfx-%u, value res-%u", key.callerHnd,
           key.declaredCalleeHnd, key.exactCalleeHnd, key.fIsTailPrefix, value);
}
bool MethodContext::repCanTailCall(CORINFO_METHOD_HANDLE callerHnd,
                                   CORINFO_METHOD_HANDLE declaredCalleeHnd,
                                   CORINFO_METHOD_HANDLE exactCalleeHnd,
                                   bool                  fIsTailPrefix)
{
    Agnostic_CanTailCall key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.callerHnd         = CastHandle(callerHnd);
    key.declaredCalleeHnd = CastHandle(declaredCalleeHnd);
    key.exactCalleeHnd    = CastHandle(exactCalleeHnd);
    key.fIsTailPrefix     = (DWORD)fIsTailPrefix;

    AssertMapAndKeyExist(CanTailCall, key, ": key %016llX", key.callerHnd);

    DWORD value = CanTailCall->Get(key);
    DEBUG_REP(dmpCanTailCall(key, value));
    return value != 0;
}

void MethodContext::recIsCompatibleDelegate(CORINFO_CLASS_HANDLE  objCls,
                                            CORINFO_CLASS_HANDLE  methodParentCls,
                                            CORINFO_METHOD_HANDLE method,
                                            CORINFO_CLASS_HANDLE  delegateCls,
                                            bool*                 pfIsOpenDelegate,
                                            bool                  result)
{
    if (IsCompatibleDelegate == nullptr)
        IsCompatibleDelegate = new LightWeightMap<Agnostic_IsCompatibleDelegate, DD>();

    Agnostic_IsCompatibleDelegate key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.objCls          = CastHandle(objCls);
    key.methodParentCls = CastHandle(methodParentCls);
    key.method          = CastHandle(method);
    key.delegateCls     = CastHandle(delegateCls);

    DD value;
    value.A = (DWORD)*pfIsOpenDelegate;
    value.B = (DWORD)result;

    IsCompatibleDelegate->Add(key, value);
    DEBUG_REC(dmpIsCompatibleDelegate(key, value));
}
void MethodContext::dmpIsCompatibleDelegate(const Agnostic_IsCompatibleDelegate& key, DD value)
{
    printf("IsCompatibleDelegate key objCls-%016llX methodParentCls-%016llX method-%016llX delegateCls-%016llX, value  "
           "pfIsOpenDelegate-%08X result-%08X",
           key.objCls, key.methodParentCls, key.method, key.delegateCls, value.A, value.B);
}
bool MethodContext::repIsCompatibleDelegate(CORINFO_CLASS_HANDLE  objCls,
                                            CORINFO_CLASS_HANDLE  methodParentCls,
                                            CORINFO_METHOD_HANDLE method,
                                            CORINFO_CLASS_HANDLE  delegateCls,
                                            bool*                 pfIsOpenDelegate)
{
    Agnostic_IsCompatibleDelegate key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.objCls          = CastHandle(objCls);
    key.methodParentCls = CastHandle(methodParentCls);
    key.method          = CastHandle(method);
    key.delegateCls     = CastHandle(delegateCls);

    AssertMapAndKeyExistNoMessage(IsCompatibleDelegate, key);

    DD value = IsCompatibleDelegate->Get(key);
    DEBUG_REP(dmpIsCompatibleDelegate(key, value));

    *pfIsOpenDelegate = value.A != 0;
    return value.B != 0;
}

void MethodContext::recIsDelegateCreationAllowed(CORINFO_CLASS_HANDLE  delegateHnd,
                                                 CORINFO_METHOD_HANDLE calleeHnd,
                                                 bool                  result)
{
    if (IsDelegateCreationAllowed == nullptr)
        IsDelegateCreationAllowed = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(delegateHnd);
    key.B = CastHandle(calleeHnd);

    DWORD value = result ? 1 : 0;
    IsDelegateCreationAllowed->Add(key, value);
    DEBUG_REC(dmpIsDelegateCreationAllowed(key, value));
}
void MethodContext::dmpIsDelegateCreationAllowed(DLDL key, DWORD value)
{
    printf("IsDelegateCreationAllowed key delegateHnd-%016llX calleeHnd-%016llX result-%08X", key.A, key.B, value);
}
bool MethodContext::repIsDelegateCreationAllowed(CORINFO_CLASS_HANDLE delegateHnd, CORINFO_METHOD_HANDLE calleeHnd)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(delegateHnd);
    key.B = CastHandle(calleeHnd);

    AssertMapAndKeyExistNoMessage(IsDelegateCreationAllowed, key);

    DWORD value = IsDelegateCreationAllowed->Get(key);
    DEBUG_REP(dmpIsDelegateCreationAllowed(key, value));
    return value != 0;
}

void MethodContext::recFindCallSiteSig(CORINFO_MODULE_HANDLE  module,
                                       unsigned               methTOK,
                                       CORINFO_CONTEXT_HANDLE context,
                                       CORINFO_SIG_INFO*      sig)
{
    if (FindCallSiteSig == nullptr)
        FindCallSiteSig = new LightWeightMap<Agnostic_FindCallSiteSig, Agnostic_CORINFO_SIG_INFO>();

    Agnostic_FindCallSiteSig key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.module  = CastHandle(module);
    key.methTok = (DWORD)methTOK;
    key.context = CastHandle(context);

    Agnostic_CORINFO_SIG_INFO value = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(*sig, FindCallSiteSig, SigInstHandleMap);

    FindCallSiteSig->Add(key, value);
    DEBUG_REC(dmpFindCallSiteSig(key, value));
}
void MethodContext::dmpFindCallSiteSig(const Agnostic_FindCallSiteSig& key, const Agnostic_CORINFO_SIG_INFO& value)
{
    printf("dmpFindCallSiteSig key module-%016llX methTok-%08X context-%016llX", key.module, key.methTok, key.context);
    printf(", value-%s",
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(value, FindCallSiteSig, SigInstHandleMap).c_str());
}
void MethodContext::repFindCallSiteSig(CORINFO_MODULE_HANDLE  module,
                                       unsigned               methTOK,
                                       CORINFO_CONTEXT_HANDLE context,
                                       CORINFO_SIG_INFO*      sig)
{
    Agnostic_FindCallSiteSig key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.module  = CastHandle(module);
    key.methTok = (DWORD)methTOK;
    key.context = CastHandle(context);

    AssertMapAndKeyExist(FindCallSiteSig, key, ": key %08X", key.methTok);

    Agnostic_CORINFO_SIG_INFO value = FindCallSiteSig->Get(key);
    DEBUG_REP(dmpFindCallSiteSig(key, value));

    *sig = SpmiRecordsHelper::Restore_CORINFO_SIG_INFO(value, FindCallSiteSig, SigInstHandleMap);
}

void MethodContext::recGetMethodSync(CORINFO_METHOD_HANDLE ftn, void** ppIndirection, void* result)
{
    if (GetMethodSync == nullptr)
        GetMethodSync = new LightWeightMap<DWORDLONG, DLDL>();
    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastPointer(result);

    DWORDLONG key = CastHandle(ftn);
    GetMethodSync->Add(key, value);
    DEBUG_REC(dmpGetMethodSync(key, value));
}
void MethodContext::dmpGetMethodSync(DWORDLONG key, DLDL value)
{
    printf("GetMethodSync key %016llX, value pp-%016llX res-%016llX", key, value.A, value.B);
}
void* MethodContext::repGetMethodSync(CORINFO_METHOD_HANDLE ftn, void** ppIndirection)
{
    DWORDLONG key = CastHandle(ftn);
    AssertMapAndKeyExist(GetMethodSync, key, ": key %016llX", key);

    DLDL value = GetMethodSync->Get(key);
    DEBUG_REP(dmpGetMethodSync(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (void*)value.B;
}

void MethodContext::recGetVarArgsHandle(CORINFO_SIG_INFO* pSig, void** ppIndirection, CORINFO_VARARGS_HANDLE result)
{
    if (GetVarArgsHandle == nullptr)
        GetVarArgsHandle = new LightWeightMap<GetVarArgsHandleValue, DLDL>();

    GetVarArgsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.cbSig      = (DWORD)pSig->cbSig;
    key.pSig_Index = (DWORD)GetVarArgsHandle->AddBuffer((unsigned char*)pSig->pSig, pSig->cbSig);
    key.scope      = CastHandle(pSig->scope);
    key.token      = (DWORD)pSig->token;

    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastHandle(result);

    GetVarArgsHandle->Add(key, value);
    DEBUG_REC(dmpGetVarArgsHandle(key, value));
}
void MethodContext::dmpGetVarArgsHandle(const GetVarArgsHandleValue& key, DLDL value)
{
    printf("GetVarArgsHandle key sig-%s scope-%016llX token-%08X",
        SpmiDumpHelper::DumpPSig(key.pSig_Index, key.cbSig, GetVarArgsHandle).c_str(),
        key.scope, key.token);
    printf(", value ppIndirection-%016llX result-%016llX", value.A, value.B);
}
CORINFO_VARARGS_HANDLE MethodContext::repGetVarArgsHandle(CORINFO_SIG_INFO* pSig, void** ppIndirection)
{
    GetVarArgsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.cbSig      = (DWORD)pSig->cbSig;
    key.pSig_Index = (DWORD)GetVarArgsHandle->Contains((unsigned char*)pSig->pSig, pSig->cbSig);
    key.scope      = CastHandle(pSig->scope);
    key.token      = (DWORD)pSig->token;

    AssertMapAndKeyExistNoMessage(GetVarArgsHandle, key);

    DLDL value = GetVarArgsHandle->Get(key);
    DEBUG_REP(dmpGetVarArgsHandle(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_VARARGS_HANDLE)value.B;
}

void MethodContext::recCanGetVarArgsHandle(CORINFO_SIG_INFO* pSig, bool result)
{
    if (CanGetVarArgsHandle == nullptr)
        CanGetVarArgsHandle = new LightWeightMap<CanGetVarArgsHandleValue, DWORD>();

    CanGetVarArgsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.scope = CastHandle(pSig->scope);
    key.token = (DWORD)pSig->token;

    DWORD value = result ? 1 : 0;
    CanGetVarArgsHandle->Add(key, value);
    DEBUG_REC(dmpCanGetVarArgsHandle(key, value));
}
void MethodContext::dmpCanGetVarArgsHandle(const CanGetVarArgsHandleValue& key, DWORD value)
{
    printf("CanGetVarArgsHandle key scope-%016llX token-%08X, value result-%08X", key.scope, key.token, value);
}
bool MethodContext::repCanGetVarArgsHandle(CORINFO_SIG_INFO* pSig)
{
    CanGetVarArgsHandleValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.scope = CastHandle(pSig->scope);
    key.token = (DWORD)pSig->token;

    AssertMapAndKeyExist(CanGetVarArgsHandle, key, ": key %016llX %08X", key.scope, key.token);

    DWORD value = CanGetVarArgsHandle->Get(key);
    DEBUG_REP(dmpCanGetVarArgsHandle(key, value));
    return value != 0;
}

void MethodContext::recGetFieldThreadLocalStoreID(CORINFO_FIELD_HANDLE field, void** ppIndirection, DWORD result)
{
    if (GetFieldThreadLocalStoreID == nullptr)
        GetFieldThreadLocalStoreID = new LightWeightMap<DWORDLONG, DLD>();

    DLD value;

    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = (DWORD)result;

    DWORDLONG key = CastHandle(field);
    GetFieldThreadLocalStoreID->Add(key, value);
    DEBUG_REC(dmpGetFieldThreadLocalStoreID(key, value));
}
void MethodContext::dmpGetFieldThreadLocalStoreID(DWORDLONG key, DLD value)
{
    printf("GetFieldThreadLocalStoreID key field-%016llX, value ppIndirection-%016llX result-%08X", key, value.A,
           value.B);
}
DWORD MethodContext::repGetFieldThreadLocalStoreID(CORINFO_FIELD_HANDLE field, void** ppIndirection)
{
    DWORDLONG key = CastHandle(field);
    AssertMapAndKeyExist(GetFieldThreadLocalStoreID, key, ": key %016llX", key);

    DLD value = GetFieldThreadLocalStoreID->Get(key);
    DEBUG_REP(dmpGetFieldThreadLocalStoreID(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (DWORD)value.B;
}

void MethodContext::recAllocPgoInstrumentationBySchema(
    CORINFO_METHOD_HANDLE ftnHnd,
    ICorJitInfo::PgoInstrumentationSchema* pSchema,
    UINT32 countSchemaItems,
    BYTE** pInstrumentationData,
    HRESULT result)
{
    if (AllocPgoInstrumentationBySchema == nullptr)
        AllocPgoInstrumentationBySchema = new LightWeightMap<DWORDLONG, Agnostic_AllocPgoInstrumentationBySchema>();

    Agnostic_AllocPgoInstrumentationBySchema value;

    // NOTE: we store the `*pInstrumentationData` address, but not any data at that address. This makes sense, because when this is called,
    // there is no data at the address. The JIT won't read/write the data, but only generate code to write to the buffer.
    value.instrumentationDataAddress = CastPointer(*pInstrumentationData);

    // For the schema, note that we record *after* calling the VM API. Thus, it has already filled in the `Offset` fields.
    // We save the "IN" parts to verify against the replayed call.
    value.countSchemaItems = countSchemaItems;
    Agnostic_PgoInstrumentationSchema* agnosticSchema = (Agnostic_PgoInstrumentationSchema*)malloc(sizeof(Agnostic_PgoInstrumentationSchema) * countSchemaItems);
    for (UINT32 i = 0; i < countSchemaItems; i++)
    {
        agnosticSchema[i].Offset              = (DWORDLONG)pSchema[i].Offset;
        agnosticSchema[i].InstrumentationKind = (DWORD)pSchema[i].InstrumentationKind;
        agnosticSchema[i].ILOffset            = (DWORD)pSchema[i].ILOffset;
        agnosticSchema[i].Count               = (DWORD)pSchema[i].Count;
        agnosticSchema[i].Other               = (DWORD)pSchema[i].Other;
    }
    value.schema_index = AllocPgoInstrumentationBySchema->AddBuffer((unsigned char*)agnosticSchema, sizeof(Agnostic_PgoInstrumentationSchema) * countSchemaItems);
    free(agnosticSchema);
    value.result = (DWORD)result;

    // Even though `countSchemaItems` and (most of) the `pSchema` array are IN parameters, they do not contribute to the lookup key;
    // the `ftnHnd` is the sole key, and the schema passed in for the function is expected to be the same every time the same function
    // handle is used.
    DWORDLONG key = CastHandle(ftnHnd);
    AllocPgoInstrumentationBySchema->Add(key, value);
    DEBUG_REC(dmpAllocPgoInstrumentationBySchema(key, value));
}

void MethodContext::dmpAllocPgoInstrumentationBySchema(DWORDLONG key, const Agnostic_AllocPgoInstrumentationBySchema& value)
{
    printf("AllocPgoInstrumentationBySchema key ftn-%016llX, value res-%08X addr-%016llX cnt-%u schema{\n",
        key, value.result, value.instrumentationDataAddress, value.countSchemaItems);

    if (value.countSchemaItems > 0)
    {
        Agnostic_PgoInstrumentationSchema* pBuf =
            (Agnostic_PgoInstrumentationSchema*)AllocPgoInstrumentationBySchema->GetBuffer(value.schema_index);

        printf("\n");
        for (DWORD i = 0; i < value.countSchemaItems; i++)
        {
            printf(" %u-{Offset %016llX ILOffset %u Kind %u(0x%x) Count %u Other %u}\n",
                i, pBuf[i].Offset, pBuf[i].ILOffset, pBuf[i].InstrumentationKind, pBuf[i].InstrumentationKind, pBuf[i].Count, pBuf[i].Other);
        }
        AllocPgoInstrumentationBySchema->Unlock();
    }
    printf("}");
}

HRESULT MethodContext::repAllocPgoInstrumentationBySchema(
    CORINFO_METHOD_HANDLE ftnHnd,
    ICorJitInfo::PgoInstrumentationSchema* pSchema,
    UINT32 countSchemaItems,
    BYTE** pInstrumentationData)
{
    DWORDLONG key = CastHandle(ftnHnd);
    AssertMapAndKeyExist(AllocPgoInstrumentationBySchema, key, ": key %016llX", key);

    Agnostic_AllocPgoInstrumentationBySchema value = AllocPgoInstrumentationBySchema->Get(key);
    DEBUG_REP(dmpAllocPgoInstrumentationBySchema(key, value));

    if (value.countSchemaItems != countSchemaItems)
    {
        LogError("AllocPgoInstrumentationBySchema mismatch: countSchemaItems record %d, replay %d", value.countSchemaItems, countSchemaItems);
    }

    HRESULT result = (HRESULT)value.result;

    Agnostic_PgoInstrumentationSchema* pAgnosticSchema = (Agnostic_PgoInstrumentationSchema*)AllocPgoInstrumentationBySchema->GetBuffer(value.schema_index);
    size_t maxOffset = 0;
    for (UINT32 iSchema = 0; iSchema < countSchemaItems && iSchema < value.countSchemaItems; iSchema++)
    {
        // Everything but `Offset` field is an IN argument, so verify it against what we stored (since we didn't use these
        // IN arguments as part of the key).

        if ((ICorJitInfo::PgoInstrumentationKind)pAgnosticSchema[iSchema].InstrumentationKind != pSchema[iSchema].InstrumentationKind)
        {
            LogError("AllocPgoInstrumentationBySchema mismatch: [%d].InstrumentationKind record %d, replay %d",
                iSchema, pAgnosticSchema[iSchema].InstrumentationKind, (DWORD)pSchema[iSchema].InstrumentationKind);
        }
        if ((int32_t)pAgnosticSchema[iSchema].ILOffset != pSchema[iSchema].ILOffset)
        {
            LogError("AllocPgoInstrumentationBySchema mismatch: [%d].ILOffset record %d, replay %d",
                iSchema, pAgnosticSchema[iSchema].ILOffset, (DWORD)pSchema[iSchema].ILOffset);
        }
        if ((int32_t)pAgnosticSchema[iSchema].Count != pSchema[iSchema].Count)
        {
            LogError("AllocPgoInstrumentationBySchema mismatch: [%d].Count record %d, replay %d",
                iSchema, pAgnosticSchema[iSchema].Count, (DWORD)pSchema[iSchema].Count);
        }
        if ((int32_t)pAgnosticSchema[iSchema].Other != pSchema[iSchema].Other)
        {
            LogError("AllocPgoInstrumentationBySchema mismatch: [%d].Other record %d, replay %d",
                iSchema, pAgnosticSchema[iSchema].Other, (DWORD)pSchema[iSchema].Other);
        }

        pSchema[iSchema].Offset = (size_t)pAgnosticSchema[iSchema].Offset;

        if (pSchema[iSchema].Offset > maxOffset)
            maxOffset = pSchema[iSchema].Offset;
    }

    // Allocate a scratch buffer, linked to method context via AllocPgoInstrumentationBySchema, so it gets
    // cleaned up when the method context does.
    //
    // We won't bother recording this via AddBuffer because currently SPMI will never look at it.
    // But we need a writeable buffer because the jit will store IL offsets inside.
    //
    // Todo, perhaps: record the buffer as a compile result instead, and defer copying until
    // jit completion so we can snapshot the offsets the jit writes.
    //
    // Add 16 bytes of represent writeable space
    size_t bufSize = maxOffset + 16;
    *pInstrumentationData = (BYTE*)AllocJitTempBuffer(bufSize);
    cr->recAddressMap((void*)value.instrumentationDataAddress, (void*)*pInstrumentationData, (unsigned)bufSize);
    return result;
}

void MethodContext::recGetPgoInstrumentationResults(CORINFO_METHOD_HANDLE ftnHnd,
                                                    ICorJitInfo::PgoInstrumentationSchema** pSchema,
                                                    UINT32* pCountSchemaItems,
                                                    BYTE** pInstrumentationData,
                                                    ICorJitInfo::PgoSource* pPgoSource,
                                                    HRESULT result)
{
    if (GetPgoInstrumentationResults == nullptr)
        GetPgoInstrumentationResults = new LightWeightMap<DWORDLONG, Agnostic_GetPgoInstrumentationResults>();

    Agnostic_GetPgoInstrumentationResults value;

    value.countSchemaItems = *pCountSchemaItems;

    ICorJitInfo::PgoInstrumentationSchema* pInSchema = *pSchema;
    Agnostic_PgoInstrumentationSchema* agnosticSchema = (Agnostic_PgoInstrumentationSchema*)malloc(sizeof(Agnostic_PgoInstrumentationSchema) * (*pCountSchemaItems));
    size_t maxOffset = 0;
    for (UINT32 i = 0; i < (*pCountSchemaItems); i++)
    {
        maxOffset = max(maxOffset, pInSchema[i].Offset + pInSchema[i].Count * sizeof(uintptr_t));

        agnosticSchema[i].Offset              = (DWORDLONG)pInSchema[i].Offset;
        agnosticSchema[i].InstrumentationKind = (DWORD)pInSchema[i].InstrumentationKind;
        agnosticSchema[i].ILOffset            = (DWORD)pInSchema[i].ILOffset;
        agnosticSchema[i].Count               = (DWORD)pInSchema[i].Count;
        agnosticSchema[i].Other               = (DWORD)pInSchema[i].Other;
    }
    value.schema_index = GetPgoInstrumentationResults->AddBuffer((unsigned char*)agnosticSchema, sizeof(Agnostic_PgoInstrumentationSchema) * (*pCountSchemaItems));
    free(agnosticSchema);

    value.data_index    = GetPgoInstrumentationResults->AddBuffer((unsigned char*)*pInstrumentationData, (unsigned)maxOffset);
    value.dataByteCount = (unsigned)maxOffset;
    value.result        = (DWORD)result;
    value.pgoSource     = (DWORD)*pPgoSource;

    DWORDLONG key = CastHandle(ftnHnd);
    GetPgoInstrumentationResults->Add(key, value);
    DEBUG_REC(dmpGetPgoInstrumentationResults(key, value));
}
void MethodContext::dmpGetPgoInstrumentationResults(DWORDLONG key, const Agnostic_GetPgoInstrumentationResults& value)
{
    printf("GetPgoInstrumentationResults key ftn-%016llX, value res-%08X schemaCnt-%u profileBufSize-%u source-%u schema{",
        key, value.result, value.countSchemaItems, value.dataByteCount, value.pgoSource);

    if (value.countSchemaItems > 0)
    {
        Agnostic_PgoInstrumentationSchema* pBuf =
            (Agnostic_PgoInstrumentationSchema*)GetPgoInstrumentationResults->GetBuffer(value.schema_index);

        BYTE* pInstrumentationData = (BYTE*)GetPgoInstrumentationResults->GetBuffer(value.data_index);

        printf("\n");
        for (DWORD i = 0; i < value.countSchemaItems; i++)
        {
            printf(" %u-{Offset %016llX ILOffset %u Kind %u(0x%x) Count %u Other %u Data ",
                i, pBuf[i].Offset, pBuf[i].ILOffset, pBuf[i].InstrumentationKind, pBuf[i].InstrumentationKind, pBuf[i].Count, pBuf[i].Other);

            switch((ICorJitInfo::PgoInstrumentationKind)pBuf[i].InstrumentationKind)
            {
                case ICorJitInfo::PgoInstrumentationKind::BasicBlockIntCount:
                    printf("B %u", *(unsigned*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::BasicBlockLongCount:
                    printf("B %llu", *(uint64_t*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::EdgeIntCount:
                    printf("E %u", *(unsigned*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::EdgeLongCount:
                    printf("E %llu", *(uint64_t*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::TypeHandleHistogramIntCount:
                    printf("T %u", *(unsigned*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::TypeHandleHistogramLongCount:
                    printf("T %llu", *(uint64_t*)(pInstrumentationData + pBuf[i].Offset));
                    break;
                case ICorJitInfo::PgoInstrumentationKind::TypeHandleHistogramTypeHandle:
                    for (unsigned int j = 0; j < pBuf[i].Count; j++)
                    {
                        printf("[%u] %016llX ", j, CastHandle(*(uintptr_t*)(pInstrumentationData + pBuf[i].Offset + j * sizeof(uintptr_t))));
                    }
                    break;
                case ICorJitInfo::PgoInstrumentationKind::GetLikelyClass:
                    {
                        // (N)umber, (L)ikelihood, (C)lass
                        printf("N %u L %u C %016llX", (unsigned)(pBuf[i].Other >> 8), (unsigned)(pBuf[i].Other && 0xFF), CastHandle(*(uintptr_t*)(pInstrumentationData + pBuf[i].Offset)));
                    }
                    break;
                default:
                    printf("?");
                    break;
            }

            printf("}\n");
        }
        GetPgoInstrumentationResults->Unlock();
    }
    printf("} data_index-%u", value.data_index);
}
HRESULT MethodContext::repGetPgoInstrumentationResults(CORINFO_METHOD_HANDLE ftnHnd,
                                                       ICorJitInfo::PgoInstrumentationSchema** pSchema,
                                                       UINT32* pCountSchemaItems,
                                                       BYTE** pInstrumentationData,
                                                       ICorJitInfo::PgoSource* pPgoSource)
{
    DWORDLONG key = CastHandle(ftnHnd);
    AssertMapAndKeyExist(GetPgoInstrumentationResults, key, ": key %016llX", key);

    Agnostic_GetPgoInstrumentationResults tempValue = GetPgoInstrumentationResults->Get(key);
    DEBUG_REP(dmpGetPgoInstrumentationResults(key, tempValue));

    *pCountSchemaItems    = (UINT32)tempValue.countSchemaItems;
    *pInstrumentationData = (BYTE*)GetPgoInstrumentationResults->GetBuffer(tempValue.data_index);
    *pPgoSource           = (ICorJitInfo::PgoSource)tempValue.pgoSource;

    ICorJitInfo::PgoInstrumentationSchema* pOutSchema = (ICorJitInfo::PgoInstrumentationSchema*)AllocJitTempBuffer(tempValue.countSchemaItems * sizeof(ICorJitInfo::PgoInstrumentationSchema));

    Agnostic_PgoInstrumentationSchema* pAgnosticSchema = (Agnostic_PgoInstrumentationSchema*)GetPgoInstrumentationResults->GetBuffer(tempValue.schema_index);
    for (UINT32 iSchema = 0; iSchema < tempValue.countSchemaItems; iSchema++)
    {
        pOutSchema[iSchema].Offset              = (size_t)pAgnosticSchema[iSchema].Offset;
        pOutSchema[iSchema].InstrumentationKind = (ICorJitInfo::PgoInstrumentationKind)pAgnosticSchema[iSchema].InstrumentationKind;
        pOutSchema[iSchema].ILOffset            = (int32_t)pAgnosticSchema[iSchema].ILOffset;
        pOutSchema[iSchema].Count               = (int32_t)pAgnosticSchema[iSchema].Count;
        pOutSchema[iSchema].Other               = (int32_t)pAgnosticSchema[iSchema].Other;
    }

    *pSchema = pOutSchema;

    HRESULT result = (HRESULT)tempValue.result;
    return result;
}

void MethodContext::recMergeClasses(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2, CORINFO_CLASS_HANDLE result)
{
    if (MergeClasses == nullptr)
        MergeClasses = new LightWeightMap<DLDL, DWORDLONG>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    DWORDLONG value = CastHandle(result);
    MergeClasses->Add(key, value);
    DEBUG_REC(dmpMergeClasses(key, value));
}
void MethodContext::dmpMergeClasses(DLDL key, DWORDLONG value)
{
    printf("MergeClasses NYI");
}
CORINFO_CLASS_HANDLE MethodContext::repMergeClasses(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    AssertMapAndKeyExist(MergeClasses, key, ": key %016llX %016llX", key.A, key.B);

    DWORDLONG value = MergeClasses->Get(key);
    DEBUG_REP(dmpMergeClasses(key, value));
    return (CORINFO_CLASS_HANDLE)value;
}

void MethodContext::recIsMoreSpecificType(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2, bool result)
{
    if (IsMoreSpecificType == nullptr)
        IsMoreSpecificType = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    DWORD value = result ? 1 : 0;
    IsMoreSpecificType->Add(key, value);
    DEBUG_REC(dmpIsMoreSpecificType(key, value));
}
void MethodContext::dmpIsMoreSpecificType(DLDL key, DWORD value)
{
    printf("IsMoreSpecificType key cls1-%016llX cls2-%016llX, value %u", key.A, key.B, value);
}
bool MethodContext::repIsMoreSpecificType(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    AssertMapAndKeyExist(IsMoreSpecificType, key, ": key %016llX %016llX", key.A, key.B);

    DWORD value = IsMoreSpecificType->Get(key);
    DEBUG_REP(dmpIsMoreSpecificType(key, value));
    return value != 0;
}

void MethodContext::recGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, void** ppIndirection, LPVOID result)
{
    if (GetCookieForPInvokeCalliSig == nullptr)
        GetCookieForPInvokeCalliSig = new LightWeightMap<GetCookieForPInvokeCalliSigValue, DLDL>();

    GetCookieForPInvokeCalliSigValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.cbSig      = (DWORD)szMetaSig->cbSig;
    key.pSig_Index = (DWORD)GetCookieForPInvokeCalliSig->AddBuffer((unsigned char*)szMetaSig->pSig, szMetaSig->cbSig);
    key.scope      = CastHandle(szMetaSig->scope);
    key.token      = (DWORD)szMetaSig->token;

    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastPointer(result);

    GetCookieForPInvokeCalliSig->Add(key, value);
    DEBUG_REC(dmpGetCookieForPInvokeCalliSig(key, value));
}
void MethodContext::dmpGetCookieForPInvokeCalliSig(const GetCookieForPInvokeCalliSigValue& key, DLDL value)
{
    printf("GetCookieForPInvokeCalliSig NYI");
}
LPVOID MethodContext::repGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, void** ppIndirection)
{
    GetCookieForPInvokeCalliSigValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.cbSig      = (DWORD)szMetaSig->cbSig;
    key.pSig_Index = (DWORD)GetCookieForPInvokeCalliSig->Contains((unsigned char*)szMetaSig->pSig, szMetaSig->cbSig);
    key.scope      = CastHandle(szMetaSig->scope);
    key.token      = (DWORD)szMetaSig->token;

    AssertMapAndKeyExistNoMessage(GetCookieForPInvokeCalliSig, key);

    DLDL value = GetCookieForPInvokeCalliSig->Get(key);
    DEBUG_REP(dmpGetCookieForPInvokeCalliSig(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_VARARGS_HANDLE)value.B;
}

void MethodContext::recCanGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, bool result)
{
    if (CanGetCookieForPInvokeCalliSig == nullptr)
        CanGetCookieForPInvokeCalliSig = new LightWeightMap<CanGetCookieForPInvokeCalliSigValue, DWORD>();

    CanGetCookieForPInvokeCalliSigValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.scope = CastHandle(szMetaSig->scope);
    key.token = (DWORD)szMetaSig->token;

    DWORD value = result ? 1 : 0;
    CanGetCookieForPInvokeCalliSig->Add(key, value);
    DEBUG_REC(dmpCanGetCookieForPInvokeCalliSig(key, value));
}
void MethodContext::dmpCanGetCookieForPInvokeCalliSig(const CanGetCookieForPInvokeCalliSigValue& key, DWORD value)
{
    printf("CanGetCookieForPInvokeCalliSig key scope-%016llX token-%08X, value result-%08X", key.scope, key.token,
           value);
}
bool MethodContext::repCanGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig)
{
    CanGetCookieForPInvokeCalliSigValue key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.scope = CastHandle(szMetaSig->scope);
    key.token = (DWORD)szMetaSig->token;

    AssertMapAndKeyExistNoMessage(CanGetCookieForPInvokeCalliSig, key);

    DWORD value = CanGetCookieForPInvokeCalliSig->Get(key);
    DEBUG_REP(dmpCanGetCookieForPInvokeCalliSig(key, value));
    return value != 0;
}

void MethodContext::recCanAccessFamily(CORINFO_METHOD_HANDLE hCaller, CORINFO_CLASS_HANDLE hInstanceType, bool result)
{
    if (CanAccessFamily == nullptr)
        CanAccessFamily = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(hCaller);
    key.B = CastHandle(hInstanceType);

    DWORD value = result ? 1 : 0;
    CanAccessFamily->Add(key, value);
    DEBUG_REC(dmpCanAccessFamily(key, value));
}
void MethodContext::dmpCanAccessFamily(DLDL key, DWORD value)
{
    printf("CanAccessFamily key cal-%016llX inst-%016llX, value %u", key.A, key.B, value);
}
bool MethodContext::repCanAccessFamily(CORINFO_METHOD_HANDLE hCaller, CORINFO_CLASS_HANDLE hInstanceType)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(hCaller);
    key.B = CastHandle(hInstanceType);

    AssertMapAndKeyExistNoMessage(CanAccessFamily, key);

    DWORD value = CanAccessFamily->Get(key);
    DEBUG_REP(dmpCanAccessFamily(key, value));
    return value != 0;
}

void MethodContext::recErrorList(const char* error)
{
    if (ErrorList == nullptr)
        ErrorList = new DenseLightWeightMap<DWORD>();

    DWORD temp = (DWORD)-1;

    if (error != nullptr)
        temp = (DWORD)ErrorList->AddBuffer((unsigned char*)error, (DWORD)strlen(error) + 1);

    ErrorList->Append(temp);
}
void MethodContext::dmpErrorList(DWORD key, DWORD value)
{
    printf("ErrorList NYI");
}

void MethodContext::recGetProfilingHandle(bool* pbHookFunction, void** pProfilerHandle, bool* pbIndirectedHandles)
{
    if (GetProfilingHandle == nullptr)
        GetProfilingHandle = new LightWeightMap<DWORD, Agnostic_GetProfilingHandle>();

    Agnostic_GetProfilingHandle value;
    ZeroMemory(&value, sizeof(value)); // Zero value including any struct padding
    value.bHookFunction      = (DWORD)*pbHookFunction;
    value.ProfilerHandle     = CastPointer(*pProfilerHandle);
    value.bIndirectedHandles = (DWORD)*pbIndirectedHandles;
    GetProfilingHandle->Add(0, value);
    DEBUG_REC(dmpGetProfilingHandle(0, value));
}
void MethodContext::dmpGetProfilingHandle(DWORD key, const Agnostic_GetProfilingHandle& value)
{
    printf("GetProfilingHandle key %u, value bHookFtn-%u profHnd-%016llX bIndHnd-%u", key, value.bHookFunction,
           value.ProfilerHandle, value.bIndirectedHandles);
}
void MethodContext::repGetProfilingHandle(bool* pbHookFunction, void** pProfilerHandle, bool* pbIndirectedHandles)
{
    AssertMapAndKeyExistNoMessage(GetProfilingHandle, 0);

    Agnostic_GetProfilingHandle value = GetProfilingHandle->Get(0);
    DEBUG_REP(dmpGetProfilingHandle(0, value));

    *pbHookFunction      = value.bHookFunction != 0;
    *pProfilerHandle     = (void*)value.ProfilerHandle;
    *pbIndirectedHandles = value.bIndirectedHandles != 0;
}

void MethodContext::recEmbedFieldHandle(CORINFO_FIELD_HANDLE handle, void** ppIndirection, CORINFO_FIELD_HANDLE result)
{
    if (EmbedFieldHandle == nullptr)
        EmbedFieldHandle = new LightWeightMap<DWORDLONG, DLDL>();

    DLDL value;
    if (ppIndirection != nullptr)
        value.A = CastPointer(*ppIndirection);
    else
        value.A = 0;
    value.B     = CastHandle(result);

    DWORDLONG key = CastHandle(handle);
    EmbedFieldHandle->Add(key, value);
    DEBUG_REC(dmpEmbedFieldHandle(key, value));
}
void MethodContext::dmpEmbedFieldHandle(DWORDLONG key, DLDL value)
{
    printf("EmbedFieldHandle NYI");
}
CORINFO_FIELD_HANDLE MethodContext::repEmbedFieldHandle(CORINFO_FIELD_HANDLE handle, void** ppIndirection)
{
    DWORDLONG key = CastHandle(handle);
    AssertMapAndKeyExist(EmbedFieldHandle, key, ": key %016llX", key);

    DLDL value = EmbedFieldHandle->Get(key);
    DEBUG_REP(dmpEmbedFieldHandle(key, value));

    if (ppIndirection != nullptr)
        *ppIndirection = (void*)value.A;
    return (CORINFO_FIELD_HANDLE)value.B;
}

void MethodContext::recAreTypesEquivalent(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2, bool result)
{
    if (AreTypesEquivalent == nullptr)
        AreTypesEquivalent = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    DWORD value = result ? 1 : 0;
    AreTypesEquivalent->Add(key, value);
    DEBUG_REC(dmpAreTypesEquivalent(key, value));
}
void MethodContext::dmpAreTypesEquivalent(DLDL key, DWORD value)
{
    printf("AreTypesEquivalent NYI");
}
bool MethodContext::repAreTypesEquivalent(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    AssertMapAndKeyExist(AreTypesEquivalent, key, ": key %016llX %016llX", key.A, key.B);

    DWORD value = AreTypesEquivalent->Get(key);
    DEBUG_REP(dmpAreTypesEquivalent(key, value));
    return value != 0;
}

void MethodContext::recCompareTypesForCast(CORINFO_CLASS_HANDLE fromClass,
                                           CORINFO_CLASS_HANDLE toClass,
                                           TypeCompareState     result)
{
    if (CompareTypesForCast == nullptr)
        CompareTypesForCast = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(fromClass);
    key.B = CastHandle(toClass);

    DWORD value = (DWORD)result;
    CompareTypesForCast->Add(key, value);
    DEBUG_REC(dmpCompareTypesForCast(key, value));
}
void MethodContext::dmpCompareTypesForCast(DLDL key, DWORD value)
{
    printf("CompareTypesForCast key fromClass=%016llX, toClass=%016llx, result=%d", key.A, key.B, value);
}
TypeCompareState MethodContext::repCompareTypesForCast(CORINFO_CLASS_HANDLE fromClass, CORINFO_CLASS_HANDLE toClass)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(fromClass);
    key.B = CastHandle(toClass);

    AssertMapAndKeyExist(CompareTypesForCast, key, ": key %016llX %016llX", key.A, key.B);

    DWORD value = CompareTypesForCast->Get(key);
    DEBUG_REP(dmpCompareTypesForCast(key, value));
    TypeCompareState result = (TypeCompareState)value;
    return result;
}

void MethodContext::recCompareTypesForEquality(CORINFO_CLASS_HANDLE cls1,
                                               CORINFO_CLASS_HANDLE cls2,
                                               TypeCompareState     result)
{
    if (CompareTypesForEquality == nullptr)
        CompareTypesForEquality = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    DWORD value = (DWORD)result;
    CompareTypesForEquality->Add(key, value);
    DEBUG_REC(dmpCompareTypesForEquality(key, value));
}
void MethodContext::dmpCompareTypesForEquality(DLDL key, DWORD value)
{
    printf("CompareTypesForEquality key cls1=%016llX, cls2=%016llx, result=%d", key.A, key.B, value);
}
TypeCompareState MethodContext::repCompareTypesForEquality(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls1);
    key.B = CastHandle(cls2);

    AssertMapAndKeyExist(CompareTypesForEquality, key, ": key %016llX %016llX", key.A, key.B);

    DWORD value = CompareTypesForEquality->Get(key);
    DEBUG_REP(dmpCompareTypesForEquality(key, value));
    TypeCompareState result = (TypeCompareState)value;
    return result;
}

void MethodContext::recFindNameOfToken(
    CORINFO_MODULE_HANDLE module, mdToken metaTOK, char* szFQName, size_t FQNameCapacity, size_t result)
{
    if (FindNameOfToken == nullptr)
        FindNameOfToken = new LightWeightMap<DLD, DLD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    DLD value;
    value.A = result;
    value.B = FindNameOfToken->AddBuffer((unsigned char*)szFQName, (unsigned int)result);

    FindNameOfToken->Add(key, value);
    DEBUG_REC(dmpFindNameOfToken(key, value));
}
void MethodContext::dmpFindNameOfToken(DLD key, DLD value)
{
    // practically the name of a token wont be bigger than 4gb...
    unsigned char* buff = new unsigned char[(unsigned int)value.A + 1];
    ZeroMemory(buff, (unsigned int)value.A + 1);
    memcpy(buff, FindNameOfToken->GetBuffer(value.B), (unsigned int)value.A);
    FindNameOfToken->Unlock();
    printf("FindNameOfToken key mod-%016llX tok-%08X, value '%s'", key.A, key.B, buff);
    delete[] buff;
}
size_t MethodContext::repFindNameOfToken(CORINFO_MODULE_HANDLE module,
                                         mdToken               metaTOK,
                                         char*                 szFQName,
                                         size_t                FQNameCapacity)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    AssertMapAndKeyExist(FindNameOfToken, key, ": key %016llX", key.A);

    DLD value = FindNameOfToken->Get(key);
    DEBUG_REP(dmpFindNameOfToken(key, value));

    unsigned char* temp = nullptr;
    if (value.B != (DWORD)-1)
    {
        temp = FindNameOfToken->GetBuffer(value.B);
        memcpy(szFQName, temp, (size_t)value.A);
    }

    return (size_t)value.A;
}

void MethodContext::recGetSystemVAmd64PassStructInRegisterDescriptor(
    CORINFO_CLASS_HANDLE                                 structHnd,
    SYSTEMV_AMD64_CORINFO_STRUCT_REG_PASSING_DESCRIPTOR* structPassInRegDescPtr,
    bool                                                 result)
{
    if (GetSystemVAmd64PassStructInRegisterDescriptor == nullptr)
        GetSystemVAmd64PassStructInRegisterDescriptor =
            new LightWeightMap<DWORDLONG, Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor>();

    DWORDLONG                                              key;
    Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor value;

    key = CastHandle(structHnd);

    value.passedInRegisters = (DWORD)structPassInRegDescPtr->passedInRegisters;
    value.eightByteCount    = (DWORD)structPassInRegDescPtr->eightByteCount;
    for (int i = 0; i < CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS; i++)
    {
        value.eightByteClassifications[i] = (DWORD)structPassInRegDescPtr->eightByteClassifications[i];
        value.eightByteSizes[i]           = (DWORD)structPassInRegDescPtr->eightByteSizes[i];
        value.eightByteOffsets[i]         = (DWORD)structPassInRegDescPtr->eightByteOffsets[i];
    }
    value.result = result ? 1 : 0;

    GetSystemVAmd64PassStructInRegisterDescriptor->Add(key, value);
    DEBUG_REC(dmpGetSystemVAmd64PassStructInRegisterDescriptor(key, value));
}
void MethodContext::dmpGetSystemVAmd64PassStructInRegisterDescriptor(
    DWORDLONG key, const Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor& value)
{
    printf("GetSystemVAmd64PassStructInRegisterDescriptor key structHnd-%016llX, value passInReg-%u 8bCount-%u", key,
           value.passedInRegisters, value.eightByteCount);
    for (int i = 0; i < CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS; i++)
    {
        printf(" 8bClass[%u]-%u 8bSz[%u]-%u 8bOff[%u]-%u", i, value.eightByteClassifications[i], i,
               value.eightByteSizes[i], i, value.eightByteOffsets[i]);
    }
    printf(" result %u", value.result);
}
bool MethodContext::repGetSystemVAmd64PassStructInRegisterDescriptor(
    CORINFO_CLASS_HANDLE structHnd, SYSTEMV_AMD64_CORINFO_STRUCT_REG_PASSING_DESCRIPTOR* structPassInRegDescPtr)
{
    DWORDLONG key = CastHandle(structHnd);

    AssertMapAndKeyExist(GetSystemVAmd64PassStructInRegisterDescriptor, key, ": key %016llX", key);

    Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor value;
    value = GetSystemVAmd64PassStructInRegisterDescriptor->Get(key);
    DEBUG_REP(dmpGetSystemVAmd64PassStructInRegisterDescriptor(key, value));

    structPassInRegDescPtr->passedInRegisters = value.passedInRegisters ? true : false;
    structPassInRegDescPtr->eightByteCount    = (unsigned __int8)value.eightByteCount;
    for (int i = 0; i < CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS; i++)
    {
        structPassInRegDescPtr->eightByteClassifications[i] =
            (SystemVClassificationType)value.eightByteClassifications[i];
        structPassInRegDescPtr->eightByteSizes[i]   = (unsigned __int8)value.eightByteSizes[i];
        structPassInRegDescPtr->eightByteOffsets[i] = (unsigned __int8)value.eightByteOffsets[i];
    }

    return value.result ? true : false;
}

void MethodContext::recGetRelocTypeHint(void* target, WORD result)
{
    if (GetRelocTypeHint == nullptr)
        GetRelocTypeHint = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastPointer(target);
    DWORD value = (DWORD)result;
    GetRelocTypeHint->Add(key, value);
    DEBUG_REC(dmpGetRelocTypeHint(key, value));
}
void MethodContext::dmpGetRelocTypeHint(DWORDLONG key, DWORD value)
{
    printf("GetRelocTypeHint key tgt-%016llX, value hint-%u", key, value);
}
WORD MethodContext::repGetRelocTypeHint(void* target)
{
    DWORDLONG key = CastPointer(target);

    if (GetRelocTypeHint == nullptr)
    {
#ifdef sparseMC
        LogDebug("Sparse - repGetRelocTypeHint yielding fake answer...");
        return 65535;
#else
        LogException(EXCEPTIONCODE_MC, "Didn't find %016llX", key);
#endif
    }
    if (GetRelocTypeHint->GetIndex(key) == -1)
    {
        void* origAddr = cr->repAddressMap((void*)target);
        if (origAddr != (void*)-1 && origAddr != nullptr)
        {
            if (GetRelocTypeHint->GetIndex(CastPointer(origAddr)) == -1)
                target = origAddr;
        }
        else
        {
#ifdef sparseMC
            LogDebug("Sparse - repGetRelocTypeHint yielding fake answer...");
            return 65535;
#else
            LogException(EXCEPTIONCODE_MC, "Didn't find %016llX", key);
#endif
        }
    }

    int  index  = GetRelocTypeHint->GetIndex(key);
    WORD retVal = 0;
    if (index == -1)
    {
        void* subtarget = cr->searchAddressMap(target);

        int index2 = GetRelocTypeHint->GetIndex(CastPointer(subtarget));
        if (index2 == -1)
        {
            // __debugbreak(); // seems like a source of pain
            retVal = IMAGE_REL_BASED_REL32;
        }
        else
            retVal = (WORD)GetRelocTypeHint->Get(CastPointer(subtarget));
    }
    else
        retVal = (WORD)GetRelocTypeHint->Get(key);

    DEBUG_REP(dmpGetRelocTypeHint(key, (DWORD)retVal));
    return retVal;
}

void MethodContext::recGetExpectedTargetArchitecture(DWORD result)
{
    if (GetExpectedTargetArchitecture == nullptr)
        GetExpectedTargetArchitecture = new LightWeightMap<DWORD, DWORD>();

    DWORD key = 0; // There is only ever a single entry to this map
    GetExpectedTargetArchitecture->Add(key, result);
    DEBUG_REC(dmpGetExpectedTargetArchitecture(key, result));
}
void MethodContext::dmpGetExpectedTargetArchitecture(DWORD key, DWORD result)
{
    printf("GetExpectedTargetArchitecture key %u, res %u", key, result);
}
DWORD MethodContext::repGetExpectedTargetArchitecture()
{
    DWORD key = 0;

    AssertMapAndKeyExist(GetExpectedTargetArchitecture, key, ": key %08X", key);

    DWORD value = GetExpectedTargetArchitecture->Get(key);
    DEBUG_REP(dmpGetExpectedTargetArchitecture(key, value));
    return value;
}

void MethodContext::recDoesFieldBelongToClass(CORINFO_FIELD_HANDLE fld, CORINFO_CLASS_HANDLE cls, bool result)
{
    if (DoesFieldBelongToClass == nullptr)
        DoesFieldBelongToClass = new LightWeightMap<DLDL, DWORD>();

    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(fld);
    key.B = CastHandle(cls);

    DWORD value = (DWORD)result;
    DoesFieldBelongToClass->Add(key, value);
    DEBUG_REC(dmpDoesFieldBelongToClass(key, result));
}

void MethodContext::dmpDoesFieldBelongToClass(DLDL key, bool value)
{
    printf("DoesFieldBelongToClass key fld=%016llX, cls=%016llx, result=%d", key.A, key.B, value);
}

bool MethodContext::repDoesFieldBelongToClass(CORINFO_FIELD_HANDLE fld, CORINFO_CLASS_HANDLE cls)
{
    DLDL key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(fld);
    key.B = CastHandle(cls);

    AssertMapAndKeyExist(DoesFieldBelongToClass, key, ": key %016llX %016llX", key.A, key.B);

    bool value = (bool)DoesFieldBelongToClass->Get(key);
    DEBUG_REP(dmpDoesFieldBelongToClass(key, value));
    return value;
}

void MethodContext::recIsValidToken(CORINFO_MODULE_HANDLE module, unsigned metaTOK, bool result)
{
    if (IsValidToken == nullptr)
        IsValidToken = new LightWeightMap<DLD, DWORD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    DWORD value = result ? 1 : 0;
    IsValidToken->Add(key, value);
    DEBUG_REC(dmpIsValidToken(key, value));
}
void MethodContext::dmpIsValidToken(DLD key, DWORD value)
{
    printf("IsValidToken key mod-%016llX tok-%08X, value res-%u", key.A, key.B, value);
}
bool MethodContext::repIsValidToken(CORINFO_MODULE_HANDLE module, unsigned metaTOK)
{
    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(module);
    key.B = (DWORD)metaTOK;

    AssertMapAndKeyExist(IsValidToken, key, ": key %016llX", key.A);

    DWORD value = IsValidToken->Get(key);
    DEBUG_REP(dmpIsValidToken(key, value));
    return value != 0;
}

void MethodContext::recGetClassName(CORINFO_CLASS_HANDLE cls, const char* result)
{
    if (GetClassName == nullptr)
        GetClassName = new LightWeightMap<DWORDLONG, DWORD>();

    DWORD value = (DWORD)-1;
    if (result != nullptr)
        value = (DWORD)GetClassName->AddBuffer((unsigned char*)result, (unsigned int)strlen(result) + 1);

    DWORDLONG key = CastHandle(cls);
    GetClassName->Add(key, value);
    DEBUG_REC(dmpGetClassName(key, value));
}
void MethodContext::dmpGetClassName(DWORDLONG key, DWORD value)
{
    printf("GetClassName key %016llX, value %s", key, GetClassName->GetBuffer(value));
    GetClassName->Unlock();
}
const char* MethodContext::repGetClassName(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);

    if (GetClassName == nullptr)
        return "hackishClassName";
    int index = GetClassName->GetIndex(key);
    if (index == -1)
        return "hackishClassName";

    int offset = GetClassName->Get(key);
    DEBUG_REP(dmpGetClassName(key, (DWORD)offset));

    const char* name = (const char*)GetClassName->GetBuffer(offset);
    return name;
}

void MethodContext::recGetClassNameFromMetadata(CORINFO_CLASS_HANDLE cls, char* className, const char** namespaceName)
{
    if (GetClassNameFromMetadata == nullptr)
        GetClassNameFromMetadata = new LightWeightMap<DLD, DD>();

    DLD key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.A = CastHandle(cls);
    key.B = (namespaceName != nullptr);

    DD  value;
    if (className != nullptr)
        value.A = GetClassNameFromMetadata->AddBuffer((unsigned char*)className, (DWORD)strlen(className) + 1);
    else
        value.A = (DWORD)-1;

    if ((namespaceName != nullptr) && (*namespaceName != nullptr))
        value.B =
            GetClassNameFromMetadata->AddBuffer((unsigned char*)*namespaceName, (DWORD)strlen(*namespaceName) + 1);
    else
        value.B = (DWORD)-1;

    GetClassNameFromMetadata->Add(key, value);
    DEBUG_REC(dmpGetClassNameFromMetadata(key, value));
}

void MethodContext::dmpGetClassNameFromMetadata(DLD key, DD value)
{
    unsigned char* className     = (unsigned char*)GetClassNameFromMetadata->GetBuffer(value.A);
    unsigned char* namespaceName = (unsigned char*)GetClassNameFromMetadata->GetBuffer(value.B);
    printf("GetClassNameFromMetadata key - classNonNull-%llu namespaceNonNull-%u, value "
           "class-'%s', namespace-'%s'",
           key.A, key.B, className, namespaceName);
    GetClassNameFromMetadata->Unlock();
}

const char* MethodContext::repGetClassNameFromMetadata(CORINFO_CLASS_HANDLE cls, const char** namespaceName)
{
    const char* result = nullptr;
    DD          value;
    DLD         key;
    key.A = CastHandle(cls);
    key.B = (namespaceName != nullptr);

    int itemIndex = -1;
    if (GetClassNameFromMetadata != nullptr)
        itemIndex = GetClassNameFromMetadata->GetIndex(key);
    if (itemIndex < 0)
    {
        if (namespaceName != nullptr)
        {
            *namespaceName = nullptr;
        }
    }
    else
    {
        value = GetClassNameFromMetadata->Get(key);
        DEBUG_REP(dmpGetClassNameFromMetadata(key, value));

        result = (const char*)GetClassNameFromMetadata->GetBuffer(value.A);

        if (namespaceName != nullptr)
        {
            *namespaceName = (const char*)GetClassNameFromMetadata->GetBuffer(value.B);
        }
    }
    return result;
}

void MethodContext::recGetTypeInstantiationArgument(CORINFO_CLASS_HANDLE cls,
                                                    CORINFO_CLASS_HANDLE result,
                                                    unsigned             index)
{
    if (GetTypeInstantiationArgument == nullptr)
        GetTypeInstantiationArgument = new LightWeightMap<DWORDLONG, DWORDLONG>();

    DWORDLONG key = CastHandle(cls);
    DWORDLONG value = CastHandle(result);
    GetTypeInstantiationArgument->Add(key, value);
    DEBUG_REC(dmpGetTypeInstantiationArgument(key, value));
}
void MethodContext::dmpGetTypeInstantiationArgument(DWORDLONG key, DWORDLONG value)
{
    printf("GetTypeInstantiationArgument key - classNonNull-%llu, value NonNull-%llu", key, value);
    GetTypeInstantiationArgument->Unlock();
}
CORINFO_CLASS_HANDLE MethodContext::repGetTypeInstantiationArgument(CORINFO_CLASS_HANDLE cls, unsigned index)
{
    CORINFO_CLASS_HANDLE result = nullptr;

    DWORDLONG key = CastHandle(cls);

    int itemIndex = -1;
    if (GetTypeInstantiationArgument != nullptr)
        itemIndex = GetTypeInstantiationArgument->GetIndex(key);
    if (itemIndex >= 0)
    {
        DWORDLONG value = GetTypeInstantiationArgument->Get(key);
        DEBUG_REP(dmpGetTypeInstantiationArgument(key, value));
        result = (CORINFO_CLASS_HANDLE)value;
    }

    return result;
}

void MethodContext::recAppendClassName(
    CORINFO_CLASS_HANDLE cls, bool fNamespace, bool fFullInst, bool fAssembly, const char16_t* result)
{
    if (AppendClassName == nullptr)
        AppendClassName = new LightWeightMap<Agnostic_AppendClassName, DWORD>();

    Agnostic_AppendClassName key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.classHandle = CastHandle(cls);
    key.fNamespace  = fNamespace;
    key.fFullInst   = fFullInst;
    key.fAssembly   = fAssembly;

    DWORD value = (DWORD)-1;
    if (result != nullptr)
        value = (DWORD)AppendClassName->AddBuffer((unsigned char*)result, (unsigned int)((wcslen((LPCWSTR)result) * 2) + 2));

    AppendClassName->Add(key, value);
    DEBUG_REC(dmpAppendClassName(key, value));
}

void MethodContext::dmpAppendClassName(const Agnostic_AppendClassName& key, DWORD value)
{
    printf("AppendClassName key cls-%016llX ns-%u fi-%u as-%u, value %s", key.classHandle, key.fNamespace,
           key.fFullInst, key.fAssembly, AppendClassName->GetBuffer(value));
    AppendClassName->Unlock();
}

const WCHAR* MethodContext::repAppendClassName(CORINFO_CLASS_HANDLE cls,
                                               bool                 fNamespace,
                                               bool                 fFullInst,
                                               bool                 fAssembly)
{
    if (AppendClassName == nullptr)
        return W("hackishClassName");

    Agnostic_AppendClassName key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.classHandle = CastHandle(cls);
    key.fNamespace  = fNamespace;
    key.fFullInst   = fFullInst;
    key.fAssembly   = fAssembly;

    int index = AppendClassName->GetIndex(key);
    if (index == -1)
        return W("hackishClassName");

    DWORD value = AppendClassName->Get(key);
    DEBUG_REP(dmpAppendClassName(key, value));

    int offset = (int)value;
    const WCHAR* name = (const WCHAR*)AppendClassName->GetBuffer(offset);
    return name;
}

void MethodContext::recGetTailCallHelpers(
    CORINFO_RESOLVED_TOKEN* callToken,
    CORINFO_SIG_INFO* sig,
    CORINFO_GET_TAILCALL_HELPERS_FLAGS flags,
    CORINFO_TAILCALL_HELPERS* pResult)
{
    if (GetTailCallHelpers == nullptr)
        GetTailCallHelpers = new LightWeightMap<Agnostic_GetTailCallHelpers, Agnostic_CORINFO_TAILCALL_HELPERS>();

    Agnostic_GetTailCallHelpers key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.callToken = SpmiRecordsHelper::StoreAgnostic_CORINFO_RESOLVED_TOKEN(callToken, GetTailCallHelpers);
    key.sig       = SpmiRecordsHelper::StoreAgnostic_CORINFO_SIG_INFO(*sig, GetTailCallHelpers, SigInstHandleMap);
    key.flags     = (DWORD)flags;

    Agnostic_CORINFO_TAILCALL_HELPERS value;
    ZeroMemory(&value, sizeof(value));
    value.result = pResult != nullptr;
    if (pResult != nullptr)
    {
        value.flags       = (DWORD)pResult->flags;
        value.hStoreArgs  = CastHandle(pResult->hStoreArgs);
        value.hCallTarget = CastHandle(pResult->hCallTarget);
        value.hDispatcher = CastHandle(pResult->hDispatcher);
    }

    GetTailCallHelpers->Add(key, value);
    DEBUG_REC(dmpGetTailCallHelpers(key, value));
}

void MethodContext::dmpGetTailCallHelpers(const Agnostic_GetTailCallHelpers& key, const Agnostic_CORINFO_TAILCALL_HELPERS& value)
{
    printf("GetTailCallHelpers key callToken-%s sig-%s flg-%08X",
        SpmiDumpHelper::DumpAgnostic_CORINFO_RESOLVED_TOKEN(key.callToken).c_str(),
        SpmiDumpHelper::DumpAgnostic_CORINFO_SIG_INFO(key.sig, GetTailCallHelpers, SigInstHandleMap).c_str(),
        key.flags);
    printf(", value result-%s flg-%08X hStoreArgs-%016llX hCallTarget-%016llX hDispatcher-%016llX",
        value.result ? "true" : "false",
        value.flags,
        value.hStoreArgs,
        value.hCallTarget,
        value.hDispatcher);
}

bool MethodContext::repGetTailCallHelpers(
    CORINFO_RESOLVED_TOKEN* callToken,
    CORINFO_SIG_INFO* sig,
    CORINFO_GET_TAILCALL_HELPERS_FLAGS flags,
    CORINFO_TAILCALL_HELPERS* pResult)
{
    AssertMapExistsNoMessage(GetTailCallHelpers);

    Agnostic_GetTailCallHelpers key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.callToken = SpmiRecordsHelper::RestoreAgnostic_CORINFO_RESOLVED_TOKEN(callToken, GetTailCallHelpers);
    key.sig       = SpmiRecordsHelper::RestoreAgnostic_CORINFO_SIG_INFO(*sig, GetTailCallHelpers, SigInstHandleMap);
    key.flags     = (DWORD)flags;

    AssertKeyExistsNoMessage(GetTailCallHelpers, key);

    Agnostic_CORINFO_TAILCALL_HELPERS value = GetTailCallHelpers->Get(key);
    DEBUG_REP(dmpGetTailCallHelpers(key, value));

    if (!value.result)
        return false;

    pResult->flags       = (CORINFO_TAILCALL_HELPERS_FLAGS)value.flags;
    pResult->hStoreArgs  = (CORINFO_METHOD_HANDLE)value.hStoreArgs;
    pResult->hCallTarget = (CORINFO_METHOD_HANDLE)value.hCallTarget;
    pResult->hDispatcher = (CORINFO_METHOD_HANDLE)value.hDispatcher;
    return true;
}

void MethodContext::recGetMethodDefFromMethod(CORINFO_METHOD_HANDLE hMethod, mdMethodDef result)
{
    if (GetMethodDefFromMethod == nullptr)
        GetMethodDefFromMethod = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(hMethod);
    DWORD value = (DWORD)result;
    GetMethodDefFromMethod->Add(key, value);
    DEBUG_REC(dmpGetMethodDefFromMethod(key, value));
}
void MethodContext::dmpGetMethodDefFromMethod(DWORDLONG key, DWORD value)
{
    printf("GetMethodDefFromMethod key ftn-%016llX, value res-%u", key, value);
}
mdMethodDef MethodContext::repGetMethodDefFromMethod(CORINFO_METHOD_HANDLE hMethod)
{
    // Since this is diagnostic, fake up a result if one wasn't recorded.
    if (GetMethodDefFromMethod == nullptr)
        return (mdMethodDef)0x06000000;

    DWORDLONG key = CastHandle(hMethod);

    int index = GetMethodDefFromMethod->GetIndex(key);
    if (index < 0)
        return (mdMethodDef)0x06000001;

    DWORD value = GetMethodDefFromMethod->Get(key);
    DEBUG_REP(dmpGetMethodDefFromMethod(key, value));
    return (mdMethodDef)value;
}

void MethodContext::recCheckMethodModifier(CORINFO_METHOD_HANDLE hMethod, LPCSTR modifier, bool fOptional, bool result)
{
    if (CheckMethodModifier == nullptr)
        CheckMethodModifier = new LightWeightMap<Agnostic_CheckMethodModifier, DWORD>();

    Agnostic_CheckMethodModifier key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hMethod = CastHandle(hMethod);
    // If the input matches something already in the buffer, just re-use that slot.. easier than searching for a soft
    // key on rep.
    if (modifier != nullptr)
        key.modifier =
            (DWORD)CheckMethodModifier->AddBuffer((unsigned char*)modifier, (unsigned int)strlen(modifier) + 1);
    else
        key.modifier = (DWORD)-1;

    key.fOptional = (DWORD)fOptional;

    DWORD value = result ? 1 : 0;
    CheckMethodModifier->Add(key, value);
    DEBUG_REC(dmpCheckMethodModifier(key, value));
}
void MethodContext::dmpCheckMethodModifier(const Agnostic_CheckMethodModifier& key, DWORD value)
{
    printf("CheckMethodModifier key, ftn-%016llX mod-'%s' opt-%u, value res-%u", key.hMethod,
           (unsigned char*)CheckMethodModifier->GetBuffer(key.modifier), key.fOptional, value);
    CheckMethodModifier->Unlock();
}
bool MethodContext::repCheckMethodModifier(CORINFO_METHOD_HANDLE hMethod, LPCSTR modifier, bool fOptional)
{
    Agnostic_CheckMethodModifier key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding
    key.hMethod = CastHandle(hMethod);
    if (modifier != nullptr)
        key.modifier =
            (DWORD)CheckMethodModifier->Contains((unsigned char*)modifier, (unsigned int)strlen(modifier) + 1);
    else
        key.modifier = (DWORD)-1;

    key.fOptional = (DWORD)fOptional;

    AssertMapAndKeyExistNoMessage(CheckMethodModifier, key);

    DWORD value = CheckMethodModifier->Get(key);
    DEBUG_REP(dmpCheckMethodModifier(key, value));
    return value != 0;
}

void MethodContext::recGetArrayRank(CORINFO_CLASS_HANDLE cls, unsigned result)
{
    if (GetArrayRank == nullptr)
        GetArrayRank = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(cls);
    DWORD value = (DWORD)result;
    GetArrayRank->Add(key, value);
    DEBUG_REC(dmpGetArrayRank(key, value));
}
void MethodContext::dmpGetArrayRank(DWORDLONG key, DWORD value)
{
    printf("GetArrayRank key %016llX, value %u", key, value);
}
unsigned MethodContext::repGetArrayRank(CORINFO_CLASS_HANDLE cls)
{
    DWORDLONG key = CastHandle(cls);
    AssertMapAndKeyExist(GetArrayRank, key, ": key %016llX", key);
    DWORD value = GetArrayRank->Get(key);
    DEBUG_REP(dmpGetArrayRank(key, value));
    unsigned result = (unsigned)value;
    return result;
}

void MethodContext::recIsFieldStatic(CORINFO_FIELD_HANDLE fhld, bool result)
{
    if (IsFieldStatic == nullptr)
        IsFieldStatic = new LightWeightMap<DWORDLONG, DWORD>();

    DWORDLONG key = CastHandle(fhld);
    DWORD value = result ? 1 : 0;
    IsFieldStatic->Add(key, value);
    DEBUG_REC(dmpIsFieldStatic(key, value));
}
void MethodContext::dmpIsFieldStatic(DWORDLONG key, DWORD value)
{
    printf("IsFieldStatic key %016llX, value %u", key, value);
}
bool MethodContext::repIsFieldStatic(CORINFO_FIELD_HANDLE fhld)
{
    DWORDLONG key = CastHandle(fhld);
    AssertMapAndKeyExist(IsFieldStatic, key, ": key %016llX", key);
    DWORD value = IsFieldStatic->Get(key);
    DEBUG_REP(dmpIsFieldStatic(key, value));
    return value != 0;
}

void MethodContext::recGetIntConfigValue(const WCHAR* name, int defaultValue, int result)
{
    if (GetIntConfigValue == nullptr)
        GetIntConfigValue = new LightWeightMap<Agnostic_ConfigIntInfo, DWORD>();

    AssertCodeMsg(name != nullptr, EXCEPTIONCODE_MC, "Name can not be nullptr");

    Agnostic_ConfigIntInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding

    DWORD index =
        (DWORD)GetIntConfigValue->AddBuffer((unsigned char*)name, sizeof(WCHAR) * ((unsigned int)wcslen(name) + 1));

    key.nameIndex    = index;
    key.defaultValue = defaultValue;

    GetIntConfigValue->Add(key, result);
    DEBUG_REC(dmpGetIntConfigValue(key, result));
}

void MethodContext::dmpGetIntConfigValue(const Agnostic_ConfigIntInfo& key, int value)
{
    const WCHAR* name = (const WCHAR*)GetIntConfigValue->GetBuffer(key.nameIndex);
    printf("GetIntConfigValue name %S, default value %d, value %d", name, key.defaultValue, value);
    GetIntConfigValue->Unlock();
}

int MethodContext::repGetIntConfigValue(const WCHAR* name, int defaultValue)
{
    if (GetIntConfigValue == nullptr)
        return defaultValue;

    AssertCodeMsg(name != nullptr, EXCEPTIONCODE_MC, "Name can not be nullptr");

    Agnostic_ConfigIntInfo key;
    ZeroMemory(&key, sizeof(key)); // Zero key including any struct padding

    size_t nameLenInBytes = sizeof(WCHAR) * (wcslen(name) + 1);
    int    nameIndex      = GetIntConfigValue->Contains((unsigned char*)name, (unsigned int)nameLenInBytes);
    if (nameIndex == -1) // config name not in map
        return defaultValue;

    key.nameIndex    = (DWORD)nameIndex;
    key.defaultValue = defaultValue;

    AssertKeyExistsNoMessage(GetIntConfigValue, key);

    DWORD value = GetIntConfigValue->Get(key);
    DEBUG_REP(dmpGetIntConfigValue(key, value));
    return (int)value;
}

void MethodContext::recGetStringConfigValue(const WCHAR* name, const WCHAR* result)
{
    if (GetStringConfigValue == nullptr)
        GetStringConfigValue = new LightWeightMap<DWORD, DWORD>();

    AssertCodeMsg(name != nullptr, EXCEPTIONCODE_MC, "Name can not be nullptr");

    DWORD nameIndex = (DWORD)GetStringConfigValue->AddBuffer((unsigned char*)name,
                                                             sizeof(WCHAR) * ((unsigned int)wcslen(name) + 1));

    DWORD resultIndex = (DWORD)-1;
    if (result != nullptr)
        resultIndex = (DWORD)GetStringConfigValue->AddBuffer((unsigned char*)result,
                                                             sizeof(WCHAR) * ((unsigned int)wcslen(result) + 1));

    GetStringConfigValue->Add(nameIndex, resultIndex);
    DEBUG_REC(dmpGetStringConfigValue(nameIndex, resultIndex));
}

void MethodContext::dmpGetStringConfigValue(DWORD nameIndex, DWORD resultIndex)
{
    const WCHAR* name   = (const WCHAR*)GetStringConfigValue->GetBuffer(nameIndex);
    const WCHAR* result = (const WCHAR*)GetStringConfigValue->GetBuffer(resultIndex);
    printf("GetStringConfigValue name %S, result %S", name, result);
    GetStringConfigValue->Unlock();
}

const WCHAR* MethodContext::repGetStringConfigValue(const WCHAR* name)
{
    if (GetStringConfigValue == nullptr)
        return nullptr;

    AssertCodeMsg(name != nullptr, EXCEPTIONCODE_MC, "Name can not be nullptr");

    size_t nameLenInBytes = sizeof(WCHAR) * (wcslen(name) + 1);
    int    nameIndex      = GetStringConfigValue->Contains((unsigned char*)name, (unsigned int)nameLenInBytes);
    if (nameIndex == -1) // config name not in map
        return nullptr;

    AssertKeyExistsNoMessage(GetStringConfigValue, nameIndex);

    int          resultIndex = GetStringConfigValue->Get(nameIndex);
    const WCHAR* value       = (const WCHAR*)GetStringConfigValue->GetBuffer(resultIndex);

    DEBUG_REP(dmpGetStringConfigValue(nameIndex, (DWORD)resultIndex));
    return value;
}

void MethodContext::dmpSigInstHandleMap(DWORD key, DWORDLONG value)
{
    printf("SigInstHandleMap key %u, value %016llX", key, value);
}

int MethodContext::dumpMethodIdentityInfoToBuffer(char* buff, int len, bool ignoreMethodName /* = false */, CORINFO_METHOD_INFO* optInfo /* = nullptr */, unsigned optFlags /* = 0 */)
{
    if (len < METHOD_IDENTITY_INFO_SIZE)
        return -1;

    // Obtain the Method Info structure for this method
    CORINFO_METHOD_INFO  info;
    CORINFO_METHOD_INFO* pInfo = nullptr;
    unsigned             flags = 0;

    if (optInfo != nullptr)
    {
        // Use the info we've already retrieved from repCompileMethod().
        pInfo = optInfo;
        flags = optFlags;
    }
    else
    {
        repCompileMethod(&info, &flags);
        pInfo = &info;
    }

    // Obtain the jit flags and ISA flags.
    CORJIT_FLAGS corJitFlags;
    repGetJitFlags(&corJitFlags, sizeof(corJitFlags));

    char* obuff = buff;

    // Add the Method Signature
    int t = sprintf_s(buff, len, "%s -- ", CallUtils::GetMethodFullName(this, pInfo->ftn, pInfo->args, ignoreMethodName));
    buff += t;
    len -= t;

    // Add Calling convention information, CorInfoOptions, CorInfoRegionKind, jit flags, and ISA flags.
    t = sprintf_s(buff, len, "CallingConvention: %d, CorInfoOptions: %d, CorInfoRegionKind: %d, JitFlags %016llx, ISA Flags %016llx", pInfo->args.callConv,
                  pInfo->options, pInfo->regionKind, corJitFlags.GetFlagsRaw(), corJitFlags.GetInstructionSetFlagsRaw());
    buff += t;
    len -= t;

    // Hash the IL Code for this method and append it to the ID info
    char ilHash[MD5_HASH_BUFFER_SIZE];
    dumpMD5HashToBuffer(pInfo->ILCode, pInfo->ILCodeSize, ilHash, MD5_HASH_BUFFER_SIZE);
    t = sprintf_s(buff, len, " ILCode Hash: %s", ilHash);
    buff += t;
    len -= t;

    // Fingerprint the root method PGO data (if any) and append it to the ID info.
    //
    if ((GetPgoInstrumentationResults != nullptr) &&
        (GetPgoInstrumentationResults->GetIndex(CastHandle(pInfo->ftn)) != -1))
    {
        ICorJitInfo::PgoInstrumentationSchema* schema = nullptr;
        UINT32 schemaCount = 0;
        BYTE* schemaData = nullptr;
        ICorJitInfo::PgoSource pgoSource = ICorJitInfo::PgoSource::Unknown;
        HRESULT pgoHR = repGetPgoInstrumentationResults(pInfo->ftn, &schema, &schemaCount, &schemaData, &pgoSource);

        size_t minOffset = (size_t) ~0;
        size_t maxOffset = 0;
        uint64_t totalCount = 0;

        if (SUCCEEDED(pgoHR))
        {
            // Locate the range of the data.
            //
            for (UINT32 i = 0; i < schemaCount; i++)
            {
                size_t start = schema[i].Offset;
                size_t end;
                switch (schema[i].InstrumentationKind)
                {
                    case ICorJitInfo::PgoInstrumentationKind::BasicBlockIntCount:
                    case ICorJitInfo::PgoInstrumentationKind::EdgeIntCount:
                        totalCount += *(uint32_t*)(schemaData + schema[i].Offset);
                        end = start + 4;
                        break;
                    case ICorJitInfo::PgoInstrumentationKind::BasicBlockLongCount:
                    case ICorJitInfo::PgoInstrumentationKind::EdgeLongCount:
                        totalCount += *(uint64_t*)(schemaData + schema[i].Offset);
                        end = start + 8;
                        break;
                    default:
                        continue;
                }

                if (start < minOffset)
                {
                    minOffset = start;
                }

                if (end > maxOffset)
                {
                    maxOffset = end;
                }
            }

            // Hash the counter values.
            //
            if (minOffset < maxOffset)
            {
                char pgoHash[MD5_HASH_BUFFER_SIZE];
                dumpMD5HashToBuffer(schemaData + minOffset, (int)(maxOffset - minOffset), pgoHash,
                                    MD5_HASH_BUFFER_SIZE);

                t = sprintf_s(buff, len, " Pgo Counters %u, Count %llu, Hash: %s", schemaCount, totalCount, pgoHash);
                buff += t;
                len -= t;
            }
        }
    }

    return (int)(buff - obuff);
}

int MethodContext::dumpMethodMD5HashToBuffer(char* buff, int len, bool ignoreMethodName /* = false */, CORINFO_METHOD_INFO* optInfo /* = nullptr */, unsigned optFlags /* = 0 */)
{
    char bufferIdentityInfo[METHOD_IDENTITY_INFO_SIZE];

    int cbLen = dumpMethodIdentityInfoToBuffer(bufferIdentityInfo, METHOD_IDENTITY_INFO_SIZE, ignoreMethodName, optInfo, optFlags);

    if (cbLen < 0)
        return cbLen;

    cbLen = dumpMD5HashToBuffer((BYTE*)bufferIdentityInfo, cbLen, buff, len);

    return cbLen;
}

int MethodContext::dumpMD5HashToBuffer(BYTE* pBuffer, int bufLen, char* hash, int hashLen)
{
    // Lazy initialize the MD5 hasher.
    if (!m_hash.IsInitialized())
    {
        if (!m_hash.Initialize())
        {
            AssertMsg(false, "Failed to initialize the MD5 hasher");
            return -1;
        }
    }

    return m_hash.HashBuffer(pBuffer, bufLen, hash, hashLen);
}

bool MethodContext::hasPgoData(bool& hasEdgeProfile, bool& hasClassProfile, bool& hasLikelyClass, ICorJitInfo::PgoSource& pgoSource)
{
    hasEdgeProfile = false;
    hasClassProfile = false;
    hasLikelyClass = false;

    // Obtain the Method Info structure for this method
    CORINFO_METHOD_INFO  info;
    unsigned             flags = 0;
    repCompileMethod(&info, &flags);

    if ((GetPgoInstrumentationResults != nullptr) &&
        (GetPgoInstrumentationResults->GetIndex(CastHandle(info.ftn)) != -1))
    {
        ICorJitInfo::PgoInstrumentationSchema* schema = nullptr;
        UINT32 schemaCount = 0;
        BYTE* schemaData = nullptr;
        HRESULT pgoHR = repGetPgoInstrumentationResults(info.ftn, &schema, &schemaCount, &schemaData, &pgoSource);

        if (SUCCEEDED(pgoHR))
        {
            for (UINT32 i = 0; i < schemaCount; i++)
            {
                hasEdgeProfile |= (schema[i].InstrumentationKind == ICorJitInfo::PgoInstrumentationKind::EdgeIntCount);
                hasEdgeProfile |= (schema[i].InstrumentationKind == ICorJitInfo::PgoInstrumentationKind::EdgeLongCount);
                hasClassProfile |= (schema[i].InstrumentationKind == ICorJitInfo::PgoInstrumentationKind::TypeHandleHistogramIntCount);
                hasClassProfile |= (schema[i].InstrumentationKind == ICorJitInfo::PgoInstrumentationKind::TypeHandleHistogramLongCount);
                hasLikelyClass |= (schema[i].InstrumentationKind == ICorJitInfo::PgoInstrumentationKind::GetLikelyClass);

                if (hasEdgeProfile && hasClassProfile && hasLikelyClass)
                {
                    break;
                }
            }

            return true;
        }
    }

    return false;
}

MethodContext::Environment MethodContext::cloneEnvironment()
{
    MethodContext::Environment env;
    if (GetIntConfigValue != nullptr)
    {
        env.getIntConfigValue = new LightWeightMap<Agnostic_ConfigIntInfo, DWORD>(*GetIntConfigValue);
    }
    if (GetStringConfigValue != nullptr)
    {
        env.getStingConfigValue = new LightWeightMap<DWORD, DWORD>(*GetStringConfigValue);
    }
    return env;
}

// Check that there is a difference between the current enviroment variables maps and the prevEnv.
bool MethodContext::WasEnvironmentChanged(const Environment& prevEnv)
{
    if (!IsEnvironmentHeaderEqual(prevEnv))
    {
        return true;
    }
    if (!IsEnvironmentContentEqual(prevEnv))
    {
        return true;
    }
    return false;
}

// Check that environment maps headers are equal to the prevEnv maps headers.
bool MethodContext::IsEnvironmentHeaderEqual(const Environment& prevEnv)
{
    if (!AreLWMHeadersEqual(prevEnv.getIntConfigValue, GetIntConfigValue))
    {
        return false;
    }
    if (!AreLWMHeadersEqual(prevEnv.getStingConfigValue, GetStringConfigValue))
    {
        return false;
    }
    return true;
}

// Check that environment maps content is equal to the prevEnv content.
bool MethodContext::IsEnvironmentContentEqual(const Environment& prevEnv)
{
    if (!IsIntConfigContentEqual(prevEnv.getIntConfigValue, GetIntConfigValue))
    {
        return false;
    }
    if (!IsStringContentEqual(prevEnv.getStingConfigValue, GetStringConfigValue))
    {
        return false;
    }
    return true;
}

// Check pointers to be both initizlized or null and number of keys to be equal.
template <typename key, typename value>
bool MethodContext::AreLWMHeadersEqual(LightWeightMap<key, value>* prev, LightWeightMap<key, value>* curr)
{
    if (prev == nullptr && curr == nullptr)
    {
        return true;
    }
    if (prev != nullptr && curr != nullptr)
    {
        if (prev->GetCount() == curr->GetCount())
        {
            return true;
        }
    }
    return false;
}

bool MethodContext::IsIntConfigContentEqual(LightWeightMap<Agnostic_ConfigIntInfo, DWORD>* prev,
                                            LightWeightMap<Agnostic_ConfigIntInfo, DWORD>* curr)
{
    if (prev != nullptr && curr != nullptr)
    {
        if (prev->GetCount() != curr->GetCount())
        {
            return false;
        }

        for (unsigned i = 0; i < prev->GetCount(); ++i)
        {
            DWORD currValue = curr->GetItem(i);
            DWORD prevValue = prev->GetItem(i);
            if (currValue != prevValue)
            {
                return false;
            }

            Agnostic_ConfigIntInfo currKey = curr->GetKey(i);
            Agnostic_ConfigIntInfo prevKey = prev->GetKey(i);

            if (currKey.defaultValue != prevKey.defaultValue)
            {
                return false;
            }

            DWORD  currNameIndex = currKey.nameIndex;
            LPCSTR currName      = (LPCSTR)curr->GetBuffer(currNameIndex);
            DWORD  prevNameIndex = prevKey.nameIndex;
            LPCSTR prevName      = (LPCSTR)prev->GetBuffer(currNameIndex);
            if (strcmp(currName, prevName) != 0)
            {
                return false;
            }
        }
        return true;
    }
    else
    {
        return (prev == curr);
    }
}

bool MethodContext::IsStringContentEqual(LightWeightMap<DWORD, DWORD>* prev, LightWeightMap<DWORD, DWORD>* curr)
{
    if (prev != nullptr && curr != nullptr)
    {
        if (prev->GetCount() != curr->GetCount())
        {
            return false;
        }

        for (unsigned i = 0; i < curr->GetCount(); ++i)
        {
            DWORD  currKeyIndex = curr->GetKey(i);
            LPCSTR currKey      = (LPCSTR)curr->GetBuffer(currKeyIndex);
            DWORD  prevKeyIndex = prev->GetKey(i);
            LPCSTR prevKey      = (LPCSTR)prev->GetBuffer(prevKeyIndex);
            if (strcmp(currKey, prevKey) != 0)
            {
                return false;
            }

            DWORD  currValueIndex = curr->GetItem(i);
            LPCSTR currValue      = (LPCSTR)curr->GetBuffer(currValueIndex);
            DWORD  prevValueIndex = prev->GetItem(i);
            LPCSTR prevValue      = (LPCSTR)prev->GetBuffer(prevValueIndex);
            if (strcmp(currValue, prevValue) != 0)
            {
                return false;
            }
        }
        return true;
    }
    else
    {
        return (prev == curr);
    }
}

void MethodContext::InitReadyToRunFlag(const CORJIT_FLAGS* jitFlags)
{
    if (jitFlags->IsSet(CORJIT_FLAGS::CORJIT_FLAG_READYTORUN))
    {
        isReadyToRunCompilation = ReadyToRunCompilation::ReadyToRun;
    }
    else
    {
        isReadyToRunCompilation = ReadyToRunCompilation::NotReadyToRun;
    }

}


bool g_debugRec = false;
bool g_debugRep = false;

void SetDebugDumpVariables()
{
    static WCHAR* g_debugRecStr = nullptr;
    static WCHAR* g_debugRepStr = nullptr;
    if (g_debugRecStr == nullptr)
    {
        g_debugRecStr = GetEnvironmentVariableWithDefaultW(W("SuperPMIShimDebugRec"), W("0"));
    }
    if (g_debugRepStr == nullptr)
    {
        g_debugRepStr = GetEnvironmentVariableWithDefaultW(W("SuperPMIShimDebugRep"), W("0"));
    }

    if (0 == wcscmp(g_debugRecStr, W("1")))
    {
        g_debugRec = true;
    }
    if (0 == wcscmp(g_debugRepStr, W("1")))
    {
        g_debugRep = true;
    }
}
