/*
 *   Copyright (C) 2008, 2009 Nicolas Vion <nico@yojik.eu>
 *
 *   This file is part of Swac-scan.
 * 
 *   Swac-scan is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *   
 *   Swac-scan is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with Swac-scan.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <glibmm.h>
#include "config.h"
#include "index.hh"

#ifdef HAVE_GIOMM
# include <giomm.h>
#endif


int main(int argc, char** argv) {

	setlocale(LC_ALL, "");
	Glib::init();
#ifdef HAVE_GIOMM
	Gio::init();
#endif
	Glib::OptionContext options("[path]");
	options.set_summary("Swac-scan is a simple command line program that enables the generation of index files for audio collections of words (SWAC). It can extract tags from .flac, .ogg, .spx and .mp3 (id3v2 only) audio files.");
	Glib::OptionGroup main_group("", "");
	options.set_main_group(main_group);
	
	bool recurse = false;
	Glib::OptionEntry entry_recurse;
	entry_recurse.set_short_name('r');
	entry_recurse.set_long_name("recurse");
	entry_recurse.set_description("Enter subdirectories recursively");
	main_group.add_entry(entry_recurse, recurse);

	bool ignore_standards_fields = false;
	Glib::OptionEntry entry_ignore;
	entry_ignore.set_short_name('i');
	entry_ignore.set_long_name("ignore-standard-fields");
	entry_ignore.set_description("Ignore Vorbis Comment and Id3v2 standard fields");
	main_group.add_entry(entry_ignore, ignore_standards_fields);

	bool optimize = false;
	Glib::OptionEntry entry_optimize;
	entry_optimize.set_short_name('o');
	entry_optimize.set_long_name("optimize");
	entry_optimize.set_description("Detect constant fields");
	main_group.add_entry(entry_optimize, optimize);

	Glib::ustring format = "xml";
	Glib::OptionEntry entry_format;
	entry_format.set_short_name('f');
	entry_format.set_long_name("format");
	entry_format.set_arg_description("[xml|ini]");
	entry_format.set_description("Select output format");
	main_group.add_entry(entry_format, format);

	bool version = false;
	Glib::OptionEntry entry_version;
	entry_version.set_short_name('v');
	entry_version.set_long_name("version");
	entry_version.set_description("Show the version number and exit");
	main_group.add_entry(entry_version, version);

	try {
		options.parse(argc, argv);
	}
	catch (const Glib::Exception& ex) {
		std::cerr << "Error: " << ex.what () << std::endl; 
		return 0;
	}

	if (version) {
		std::cout << "swac-scan " VERSION << std::endl;
		return 0;
	}


	IndexTags *index;

	if (format == "xml")
		index = new IndexTags;
	else if (format == "ini")
		index = new IndexTagsIni;
	else {
		std::cerr << "Unknown output format" << std::endl;
		return 0;
	}

	
	index->options.recurse = recurse;
	index->options.ignore_standards_fields = ignore_standards_fields;

	std::string path = (argc == 2) ? argv[1] : "";
	
	if (optimize) {
		index->tags_import_dir(path, false);
		index->constants_find();
		index->export_all();
		
	}
	else {
		index->export_head();
		index->tags_import_dir(path, true);
		index->export_bottom();
	}
	
	delete index;
	return 0;
}

