/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  mod_bits.c: Bit manipulation functions
 */

/**
 * Bits Module
 *
 * This module adds built-in functions for bit manipulations.
 */

#include "spl.h"
#include "compat.h"

extern void SPL_ABI(spl_mod_bits_init)(struct spl_vm *vm, struct spl_module *mod, int restore);
extern void SPL_ABI(spl_mod_bits_done)(struct spl_vm *vm, struct spl_module *mod);

/**
 * Do a bitwise AND between all arguments and return the result.
 */
//builtin bits_and(@values);
static struct spl_node *handler_bits_and(struct spl_task *t, void *d UNUSED)
{
	int value = ~0;
	while (spl_clib_get_argc(t))
		value &= spl_clib_get_int(t);
	return SPL_NEW_INT(value);
}

/**
 * Do a bitwise OR between all arguments and return the result.
 */
//builtin bits_or(@values);
static struct spl_node *handler_bits_or(struct spl_task *t, void *d UNUSED)
{
	int value = 0;
	while (spl_clib_get_argc(t))
		value |= spl_clib_get_int(t);
	return SPL_NEW_INT(value);
}

/**
 * Do a bitwise XOR between all arguments and return the result.
 */
//builtin bits_xor(@values);
static struct spl_node *handler_bits_xor(struct spl_task *t, void *d UNUSED)
{
	int value = 0;
	while (spl_clib_get_argc(t))
		value ^= spl_clib_get_int(t);
	return SPL_NEW_INT(value);
}

/**
 * Do a bitwise NOT of the argument and return the result.
 */
//builtin bits_not(value);
static struct spl_node *handler_bits_not(struct spl_task *t, void *d UNUSED)
{
	return SPL_NEW_INT(~spl_clib_get_int(t));
}

/**
 * Do a bitwise SHIFT-LEFT of the argument and return the result.
 */
//builtin bits_shl(value, bits);
static struct spl_node *handler_bits_shl(struct spl_task *t, void *d UNUSED)
{
	int value = spl_clib_get_int(t);
	int bits = spl_clib_get_int(t);
	return SPL_NEW_INT(value << bits);
}

/**
 * Do a bitwise SHIFT-RIGHT of the argument and return the result.
 */
//builtin bits_shr(value, bits);
static struct spl_node *handler_bits_shr(struct spl_task *t, void *d UNUSED)
{
	int value = spl_clib_get_int(t);
	int bits = spl_clib_get_int(t);
	return SPL_NEW_INT(value >> bits);
}

void SPL_ABI(spl_mod_bits_init)(struct spl_vm *vm, struct spl_module *mod UNUSED, int restore UNUSED)
{
	spl_clib_reg(vm, "bits_and", handler_bits_and, 0);
	spl_clib_reg(vm, "bits_or",  handler_bits_or,  0);
	spl_clib_reg(vm, "bits_xor", handler_bits_xor, 0);
	spl_clib_reg(vm, "bits_not", handler_bits_not, 0);
	spl_clib_reg(vm, "bits_shl", handler_bits_shl, 0);
	spl_clib_reg(vm, "bits_shr", handler_bits_shr, 0);
}

void SPL_ABI(spl_mod_bits_done)(struct spl_vm *vm UNUSED, struct spl_module *mod UNUSED)
{
	return;
}

