# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sqrt, radians, sin, cos, tan, atan, atan2, asin

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    passage_nodes_elliptic
)

"""
.. module:: Earth
   :synopsis: Class to model Earth's globe
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [175347045.673, 0.00000000000, 0.00000000000],
        [3341656.456, 4.66925680417, 6283.07584999140],
        [34894.275, 4.62610241759, 12566.15169998280],
        [3417.571, 2.82886579606, 3.52311834900],
        [3497.056, 2.74411800971, 5753.38488489680],
        [3135.896, 3.62767041758, 77713.77146812050],
        [2676.218, 4.41808351397, 7860.41939243920],
        [2342.687, 6.13516237631, 3930.20969621960],
        [1273.166, 2.03709655772, 529.69096509460],
        [1324.292, 0.74246356352, 11506.76976979360],
        [901.855, 2.04505443513, 26.29831979980],
        [1199.167, 1.10962944315, 1577.34354244780],
        [857.223, 3.50849156957, 398.14900340820],
        [779.786, 1.17882652114, 5223.69391980220],
        [990.250, 5.23268129594, 5884.92684658320],
        [753.141, 2.53339053818, 5507.55323866740],
        [505.264, 4.58292563052, 18849.22754997420],
        [492.379, 4.20506639861, 775.52261132400],
        [356.655, 2.91954116867, 0.06731030280],
        [284.125, 1.89869034186, 796.29800681640],
        [242.810, 0.34481140906, 5486.77784317500],
        [317.087, 5.84901952218, 11790.62908865880],
        [271.039, 0.31488607649, 10977.07880469900],
        [206.160, 4.80646606059, 2544.31441988340],
        [205.385, 1.86947813692, 5573.14280143310],
        [202.261, 2.45767795458, 6069.77675455340],
        [126.184, 1.08302630210, 20.77539549240],
        [155.516, 0.83306073807, 213.29909543800],
        [115.132, 0.64544911683, 0.98032106820],
        [102.851, 0.63599846727, 4694.00295470760],
        [101.724, 4.26679821365, 7.11354700080],
        [99.206, 6.20992940258, 2146.16541647520],
        [132.212, 3.41118275555, 2942.46342329160],
        [97.607, 0.68101272270, 155.42039943420],
        [85.128, 1.29870743025, 6275.96230299060],
        [74.651, 1.75508916159, 5088.62883976680],
        [101.895, 0.97569221824, 15720.83878487840],
        [84.711, 3.67080093025, 71430.69561812909],
        [73.547, 4.67926565481, 801.82093112380],
        [73.874, 3.50319443167, 3154.68708489560],
        [78.756, 3.03698313141, 12036.46073488820],
        [79.637, 1.80791330700, 17260.15465469040],
        [85.803, 5.98322631256, 161000.68573767410],
        [56.963, 2.78430398043, 6286.59896834040],
        [61.148, 1.81839811024, 7084.89678111520],
        [69.627, 0.83297596966, 9437.76293488700],
        [56.116, 4.38694880779, 14143.49524243060],
        [62.449, 3.97763880587, 8827.39026987480],
        [51.145, 0.28306864501, 5856.47765911540],
        [55.577, 3.47006009062, 6279.55273164240],
        [41.036, 5.36817351402, 8429.24126646660],
        [51.605, 1.33282746983, 1748.01641306700],
        [51.992, 0.18914945834, 12139.55350910680],
        [49.000, 0.48735065033, 1194.44701022460],
        [39.200, 6.16832995016, 10447.38783960440],
        [35.566, 1.77597314691, 6812.76681508600],
        [36.770, 6.04133859347, 10213.28554621100],
        [36.596, 2.56955238628, 1059.38193018920],
        [33.291, 0.59309499459, 17789.84561978500],
        [35.954, 1.70876111898, 2352.86615377180],
        [40.938, 2.39850881707, 19651.04848109800],
        [30.047, 2.73975123935, 1349.86740965880],
        [30.412, 0.44294464135, 83996.84731811189],
        [23.663, 0.48473567763, 8031.09226305840],
        [23.574, 2.06527720049, 3340.61242669980],
        [21.089, 4.14825464101, 951.71840625060],
        [24.738, 0.21484762138, 3.59042865180],
        [25.352, 3.16470953405, 4690.47983635860],
        [22.820, 5.22197888032, 4705.73230754360],
        [21.419, 1.42563735525, 16730.46368959580],
        [21.891, 5.55594302562, 553.56940284240],
        [17.481, 4.56052900359, 135.06508003540],
        [19.925, 5.22208471269, 12168.00269657460],
        [19.860, 5.77470167653, 6309.37416979120],
        [20.300, 0.37133792946, 283.85931886520],
        [14.421, 4.19315332546, 242.72860397400],
        [16.225, 5.98837722564, 11769.85369316640],
        [15.077, 4.19567181073, 6256.77753019160],
        [19.124, 3.82219996949, 23581.25817731760],
        [18.888, 5.38626880969, 149854.40013480789],
        [14.346, 3.72355084422, 38.02767263580],
        [17.898, 2.21490735647, 13367.97263110660],
        [12.054, 2.62229588349, 955.59974160860],
        [11.287, 0.17739328092, 4164.31198961300],
        [13.971, 4.40138139996, 6681.22485339960],
        [13.621, 1.88934471407, 7632.94325965020],
        [12.503, 1.13052412208, 5.52292430740],
        [10.498, 5.35909518669, 1592.59601363280],
        [9.803, 0.99947478995, 11371.70468975820],
        [9.220, 4.57138609781, 4292.33083295040],
        [10.327, 6.19982566125, 6438.49624942560],
        [12.003, 1.00351456700, 632.78373931320],
        [10.827, 0.32734520222, 103.09277421860],
        [8.356, 4.53902685948, 25132.30339996560],
        [10.005, 6.02914963280, 5746.27133789600],
        [8.409, 3.29946744189, 7234.79425624200],
        [8.006, 5.82145271907, 28.44918746780],
        [10.523, 0.93871805506, 11926.25441366880],
        [7.686, 3.12142363172, 7238.67559160000],
        [9.378, 2.62414241032, 5760.49843189760],
        [8.127, 6.11228001785, 4732.03062734340],
        [9.232, 0.48343968736, 522.57741809380],
        [9.802, 5.24413991147, 27511.46787353720],
        [7.871, 0.99590177926, 5643.17856367740],
        [8.123, 6.27053013650, 426.59819087600],
        [9.048, 5.33686335897, 6386.16862421000],
        [8.620, 4.16538210888, 7058.59846131540],
        [6.297, 4.71724819317, 6836.64525283380],
        [7.575, 3.97382858911, 11499.65622279280],
        [7.756, 2.95729056763, 23013.53953958720],
        [7.314, 0.60652505806, 11513.88331679440],
        [5.955, 2.87641047971, 6283.14316029419],
        [6.534, 5.79072926033, 18073.70493865020],
        [7.188, 3.99831508699, 74.78159856730],
        [7.346, 4.38582365437, 316.39186965660],
        [5.413, 5.39199024641, 419.48464387520],
        [5.127, 2.36062848786, 10973.55568635000],
        [7.056, 0.32258441903, 263.08392337280],
        [6.625, 3.66475158672, 17298.18232732620],
        [6.762, 5.91132535899, 90955.55169449610],
        [4.938, 5.73672165674, 9917.69687450980],
        [5.547, 2.45152597661, 12352.85260454480],
        [5.958, 3.32051344676, 6283.00853968860],
        [4.471, 2.06385999536, 7079.37385680780],
        [6.153, 1.45823331144, 233141.31440436150],
        [4.348, 4.42342175480, 5216.58037280140],
        [6.123, 1.07494905258, 19804.82729158280],
        [4.488, 3.65285037150, 206.18554843720],
        [4.020, 0.83995823171, 20.35531939880],
        [5.188, 4.06503864016, 6208.29425142410],
        [5.307, 0.38217636096, 31441.67756975680],
        [3.785, 2.34369213733, 3.88133535800],
        [4.497, 3.27230796845, 11015.10647733480],
        [4.132, 0.92128915753, 3738.76143010800],
        [3.521, 5.97844807108, 3894.18182954220],
        [4.215, 1.90601120623, 245.83164622940],
        [3.701, 5.03069397926, 536.80451209540],
        [3.865, 1.82634360607, 11856.21865142450],
        [3.652, 1.01838584934, 16200.77272450120],
        [3.390, 0.97785123922, 8635.94200376320],
        [3.737, 2.95380107829, 3128.38876509580],
        [3.507, 3.71291946325, 6290.18939699220],
        [3.086, 3.64646921512, 10.63666534980],
        [3.397, 1.10590684017, 14712.31711645800],
        [3.334, 0.83684924911, 6496.37494542940],
        [2.805, 2.58504514144, 14314.16811304980],
        [3.650, 1.08344142571, 88860.05707098669],
        [3.388, 3.20185096055, 5120.60114558360],
        [3.252, 3.47859752062, 6133.51265285680],
        [2.553, 3.94869034189, 1990.74501704100],
        [3.520, 2.05559692878, 244287.60000722768],
        [2.565, 1.56071784900, 23543.23050468179],
        [2.621, 3.85639359951, 266.60704172180],
        [2.955, 3.39692949667, 9225.53927328300],
        [2.876, 6.02635617464, 154717.60988768269],
        [2.395, 1.16131956403, 10984.19235169980],
        [3.161, 1.32798718453, 10873.98603048040],
        [3.163, 5.08946464629, 21228.39202354580],
        [2.361, 4.27212906992, 6040.34724601740],
        [3.030, 1.80209931347, 35371.88726597640],
        [2.343, 3.57689860500, 10969.96525769820],
        [2.618, 2.57870156528, 22483.84857449259],
        [2.113, 3.71393780256, 65147.61976813770],
        [2.019, 0.81393923319, 170.67287061920],
        [2.003, 0.38091017375, 6172.86952877200],
        [2.506, 3.74379142438, 10575.40668294180],
        [2.381, 0.10581361289, 7.04623669800],
        [1.949, 4.86892513469, 36.02786667740],
        [2.074, 4.22794774570, 5650.29211067820],
        [1.924, 5.59460549860, 6282.09552892320],
        [1.949, 1.07002512703, 5230.80746680300],
        [1.988, 5.19736046771, 6262.30045449900],
        [1.887, 3.74365662683, 23.87843774780],
        [1.787, 1.25929682929, 12559.03815298200],
        [1.883, 1.90364058477, 15.25247118500],
        [1.816, 3.68083868442, 15110.46611986620],
        [1.701, 4.41105895380, 110.20632121940],
        [1.990, 3.93295788548, 6206.80977871580],
        [2.103, 0.75354917468, 13521.75144159140],
        [1.774, 0.48747535361, 1551.04522264800],
        [1.882, 0.86684493432, 22003.91463486980],
        [1.924, 1.22898324132, 709.93304855830],
        [2.009, 4.62850921980, 6037.24420376200],
        [1.924, 0.60231842508, 6284.05617105960],
        [1.596, 3.98332956992, 13916.01910964160],
        [1.664, 4.41939715469, 8662.24032356300],
        [1.971, 1.04560500503, 18209.33026366019],
        [1.942, 4.31335979989, 6244.94281435360],
        [1.476, 0.93271367331, 2379.16447357160],
        [1.810, 0.49112137707, 1.48447270830],
        [1.346, 1.51574702235, 4136.91043351620],
        [1.528, 5.61835711404, 6127.65545055720],
        [1.791, 3.22187270126, 39302.09696219600],
        [1.747, 3.05638656738, 18319.53658487960],
        [1.431, 4.51153808594, 20426.57109242200],
        [1.695, 0.22047718414, 25158.60171976540],
        [1.242, 4.46665769933, 17256.63153634140],
        [1.463, 4.69242679213, 14945.31617355440],
        [1.205, 1.86912144659, 4590.91018048900],
        [1.192, 2.74227166898, 12569.67481833180],
        [1.222, 5.18120087482, 5333.90024102160],
        [1.390, 5.42894648983, 143571.32428481648],
        [1.473, 1.70479245805, 11712.95531823080],
        [1.362, 2.61069503292, 6062.66320755260],
        [1.148, 6.03001800540, 3634.62102451840],
        [1.198, 5.15294130422, 10177.25767953360],
        [1.266, 0.11421493643, 18422.62935909819],
        [1.411, 1.09908857534, 3496.03282613400],
        [1.349, 2.99805109633, 17654.78053974960],
        [1.253, 2.79850152848, 167283.76158766550],
        [1.311, 1.60942984879, 5481.25491886760],
        [1.079, 6.20304501787, 3.28635741780],
        [1.181, 1.20653776978, 131.54196168640],
        [1.254, 5.45103277798, 6076.89030155420],
        [1.035, 2.32142722747, 7342.45778018060],
        [1.117, 0.38838354256, 949.17560896980],
        [0.966, 3.18341890851, 11087.28512591840],
        [1.171, 3.39635049962, 12562.62858163380],
        [1.121, 0.72627490378, 220.41264243880],
        [1.024, 2.19378315386, 11403.67699557500],
        [0.888, 3.91173199285, 4686.88940770680],
        [0.910, 1.98802695087, 735.87651353180],
        [0.830, 0.48984915507, 24072.92146977640],
        [1.096, 6.17377835617, 5436.99301524020],
        [0.908, 0.44959639433, 7477.52286021600],
        [0.974, 1.52996238356, 9623.68827669120],
        [0.840, 1.79543266333, 5429.87946823940],
        [0.778, 6.17699177946, 38.13303563780],
        [0.776, 4.09855402433, 14.22709400160],
        [1.068, 4.64200173735, 43232.30665841560],
        [0.954, 1.49988435748, 1162.47470440780],
        [0.907, 0.86986870809, 10344.29506538580],
        [0.931, 4.06044689031, 28766.92442448400],
        [0.739, 5.04368197372, 639.89728631400],
        [0.937, 3.46884698960, 1589.07289528380],
        [0.763, 5.86304932998, 16858.48253293320],
        [0.953, 4.20801492835, 11190.37790013700],
        [0.708, 1.72899988940, 13095.84266507740],
        [0.969, 1.64439522215, 29088.81141598500],
        [0.717, 0.16688678895, 11.72935283600],
        [0.962, 3.53092337542, 12416.58850284820],
        [0.747, 5.77866940346, 12592.45001978260],
        [0.672, 1.91095796194, 3.93215326310],
        [0.671, 5.46240843677, 18052.92954315780],
        [0.675, 6.28311558823, 4535.05943692440],
        [0.684, 0.39975012080, 5849.36411211460],
        [0.799, 0.29851185294, 12132.43996210600],
        [0.758, 0.96370823331, 1052.26838318840],
        [0.782, 5.33878339919, 13517.87010623340],
        [0.730, 1.70106160291, 17267.26820169119],
        [0.749, 2.59599901875, 11609.86254401220],
        [0.734, 2.78417782952, 640.87760738220],
        [0.688, 5.15048287468, 16496.36139620240],
        [0.770, 1.62469589333, 4701.11650170840],
        [0.633, 2.20587893893, 25934.12433108940],
        [0.760, 4.21317219403, 377.37360791580],
        [0.584, 2.13420121623, 10557.59416082380],
        [0.574, 0.24250054587, 9779.10867612540],
        [0.573, 3.16435264609, 533.21408344360],
        [0.685, 3.19344289472, 12146.66705610760],
        [0.675, 0.96179233959, 10454.50138660520],
        [0.648, 1.46327342555, 6268.84875598980],
        [0.589, 2.50543543638, 3097.88382272579],
        [0.551, 5.28099026956, 9388.00590941520],
        [0.696, 3.65342150016, 4804.20927592700],
        [0.669, 2.51030077026, 2388.89402044920],
        [0.550, 0.06883864342, 20199.09495963300],
        [0.629, 4.13350995675, 45892.73043315699],
        [0.678, 6.09190163533, 135.62532501000],
        [0.593, 1.50136257618, 226858.23855437007],
        [0.542, 3.58573645173, 6148.01076995600],
        [0.682, 5.02203067788, 17253.04110768959],
        [0.565, 4.29309238610, 11933.36796066960],
        [0.486, 0.77746204893, 27.40155609680],
        [0.503, 0.58963565969, 15671.08175940660],
        [0.616, 4.06539884128, 227.47613278900],
        [0.583, 6.12695541996, 18875.52586977400],
        [0.537, 2.15056440980, 21954.15760939799],
        [0.669, 6.06986269566, 47162.51635463520],
        [0.475, 0.40343842110, 6915.85958930460],
        [0.540, 2.83444222174, 5326.78669402080],
        [0.530, 5.26359885263, 10988.80815753500],
        [0.582, 3.24533095664, 153.77881048480],
        [0.641, 3.24711791371, 2107.03450754240],
        [0.621, 3.09698523779, 33019.02111220460],
        [0.466, 3.14982372198, 10440.27429260360],
        [0.466, 0.90708835657, 5966.68398033480],
        [0.528, 0.81926454470, 813.55028395980],
        [0.603, 3.81378921927, 316428.22867391503],
        [0.559, 1.81894804124, 17996.03116822220],
        [0.437, 2.28625594435, 6303.85124548380],
        [0.518, 4.86069178322, 20597.24396304120],
        [0.424, 6.23520018693, 6489.26139842860],
        [0.518, 6.17617826756, 0.24381748350],
        [0.404, 5.72804304258, 5642.19824260920],
        [0.458, 1.34117773915, 6287.00800325450],
        [0.548, 5.68454458320, 155427.54293624099],
        [0.547, 1.03391472061, 3646.35037735440],
        [0.428, 4.69800981138, 846.08283475120],
        [0.413, 6.02520699406, 6279.48542133960],
        [0.534, 3.03030638223, 66567.48586525429],
        [0.383, 1.49056949125, 19800.94595622480],
        [0.410, 5.28319622279, 18451.07854656599],
        [0.352, 4.68891600359, 4907.30205014560],
        [0.480, 5.36572651091, 348.92442044800],
        [0.344, 5.89157452896, 6546.15977336420],
        [0.340, 0.37557426440, 13119.72110282519],
        [0.434, 4.98417785901, 6702.56049386660],
        [0.332, 2.68902519126, 29296.61538957860],
        [0.448, 2.16478480251, 5905.70224207560],
        [0.344, 2.06546633735, 49.75702547180],
        [0.315, 1.24023811803, 4061.21921539440],
        [0.324, 2.30897526929, 5017.50837136500],
        [0.413, 0.17171692962, 6286.66627864320],
        [0.431, 3.86601101393, 12489.88562870720],
        [0.349, 4.55372342974, 4933.20844033260],
        [0.323, 0.41971136084, 10770.89325626180],
        [0.341, 2.68612860807, 11.04570026390],
        [0.316, 3.52936906658, 17782.73207278420],
        [0.315, 5.63357264999, 568.82187402740],
        [0.340, 3.83571212349, 10660.68693504240],
        [0.297, 0.62691416712, 20995.39296644940],
        [0.405, 1.00085779471, 16460.33352952499],
        [0.414, 1.21998752076, 51092.72605085480],
        [0.336, 4.71465945226, 6179.98307577280],
        [0.361, 3.71227508354, 28237.23345938940],
        [0.385, 6.21925225757, 24356.78078864160],
        [0.327, 1.05606504715, 11919.14086666800],
        [0.327, 6.14222420989, 6254.62666252360],
        [0.268, 2.47224339737, 664.75604513000],
        [0.269, 1.86207884109, 23141.55838292460],
        [0.345, 0.93461290184, 6058.73105428950],
        [0.296, 4.51687557180, 6418.14093002680],
        [0.353, 4.50033653082, 36949.23080842420],
        [0.260, 4.04963546305, 6525.80445396540],
        [0.298, 2.20046722622, 156137.47598479927],
        [0.253, 3.49900838384, 29864.33402730900],
        [0.254, 2.44901693835, 5331.35744374080],
        [0.296, 0.84347588787, 5729.50644714900],
        [0.298, 1.29194706125, 22805.73556599360],
        [0.241, 2.00721280805, 16737.57723659660],
        [0.311, 1.23668016334, 6281.59137728310],
        [0.240, 2.51650377121, 6245.04817735560],
        [0.332, 3.55576945724, 7668.63742494250],
        [0.264, 4.44052061202, 12964.30070339100],
        [0.257, 1.79654471948, 11080.17157891760],
        [0.260, 3.33077598420, 5888.44996493220],
        [0.285, 0.30886361430, 11823.16163945020],
        [0.290, 5.70141882483, 77.67377042800],
        [0.255, 4.00939664440, 5881.40372823420],
        [0.253, 4.73318493678, 16723.35014259500],
        [0.228, 0.95333661324, 5540.08578945880],
        [0.319, 1.38633229189, 163096.18036118348],
        [0.224, 1.65156322696, 10027.90319572920],
        [0.226, 0.34106460604, 17796.95916678580],
        [0.236, 4.19817431922, 19.66976089979],
        [0.280, 4.14080268970, 12539.85338018300],
        [0.275, 5.50306930248, 32.53255079140],
        [0.223, 5.23334210294, 56.89837493560],
        [0.217, 6.08587881787, 6805.65326808520],
        [0.280, 4.52472044653, 6016.46880826960],
        [0.227, 5.06509843737, 6277.55292568400],
        [0.226, 5.17755154305, 11720.06886523160],
        [0.245, 3.96486270306, 22.77520145080],
        [0.220, 4.72078081970, 6.62855890001],
        [0.207, 5.71701403951, 41.55079098480],
        [0.204, 3.91227411250, 2699.73481931760],
        [0.209, 0.86881969011, 6321.10352262720],
        [0.200, 2.11984445273, 4274.51831083240],
        [0.200, 5.39839888163, 6019.99192661860],
        [0.209, 5.67606291663, 11293.47067435560],
        [0.252, 1.64965729351, 9380.95967271720],
        [0.275, 5.04826903506, 73.29712585900],
        [0.208, 1.88207277133, 11300.58422135640],
        [0.272, 0.74640926842, 1975.49254585600],
        [0.199, 3.30836672397, 22743.40937951640],
        [0.269, 4.48560812155, 64471.99124174489],
        [0.192, 2.17464236325, 5863.59120611620],
        [0.228, 5.85373115869, 128.01884333740],
        [0.261, 2.64321183295, 55022.93574707440],
        [0.220, 5.75012110079, 29.42950853600],
        [0.187, 4.03230554718, 467.96499035440],
        [0.200, 5.60556112058, 1066.49547719000],
        [0.231, 1.09802712785, 12341.80690428090],
        [0.199, 0.29500625200, 149.56319713460],
        [0.249, 5.10473210814, 7875.67186362420],
        [0.208, 0.93013835019, 14919.01785375460],
        [0.179, 0.87104393079, 12721.57209941700],
        [0.203, 1.56920753653, 28286.99048486120],
        [0.179, 2.47036386443, 16062.18452611680],
        [0.198, 3.54061588502, 30.91412563500],
        [0.171, 3.45356518113, 5327.47610838280],
        [0.183, 0.72325421604, 6272.03014972750],
        [0.216, 2.97174580686, 19402.79695281660],
        [0.168, 2.51550550242, 23937.85638974100],
        [0.195, 0.09045393425, 156.40072050240],
        [0.179, 4.49471798090, 31415.37924995700],
        [0.216, 0.42177594328, 23539.70738633280],
        [0.189, 0.37542530191, 9814.60410029120],
        [0.218, 2.36835880025, 16627.37091537720],
        [0.166, 4.23182968446, 16840.67001081519],
        [0.200, 2.02153258098, 16097.67995028260],
        [0.169, 0.91318727000, 95.97922721780],
        [0.211, 5.73370637657, 151.89728108520],
        [0.204, 0.42643085174, 515.46387109300],
        [0.212, 3.00233538977, 12043.57428188900],
        [0.192, 5.46153589821, 6379.05507720920],
        [0.165, 1.38698167064, 4171.42553661380],
        [0.160, 6.23798383332, 202.25339517410],
        [0.215, 0.20889073407, 5621.84292321040],
        [0.181, 4.12439203622, 13341.67431130680],
        [0.153, 1.24460848836, 29826.30635467320],
        [0.150, 3.12999753018, 799.82112516540],
        [0.175, 4.55671604437, 239424.39025435288],
        [0.192, 1.33928820063, 394.62588505920],
        [0.149, 2.65697593276, 21.33564046700],
        [0.146, 5.58021191726, 412.37109687440],
        [0.156, 3.75650175503, 12323.42309600880],
        [0.143, 3.75708566606, 58864.54391814630],
        [0.143, 3.28248547724, 29.82143814880],
        [0.144, 1.07862546598, 1265.56747862640],
        [0.148, 0.23389236655, 10021.83728009940],
        [0.193, 5.92751083086, 40879.44050464380],
        [0.140, 4.97612440269, 158.94351778320],
        [0.148, 2.61640453469, 17157.06188047180],
        [0.141, 3.66871308723, 26084.02180621620],
        [0.147, 5.09968173403, 661.23292678100],
        [0.146, 4.96885605695, 57375.80190084620],
        [0.142, 0.78678347839, 12779.45079542080],
        [0.134, 4.79432636012, 111.18664228760],
        [0.140, 1.27748013377, 107.66352393860],
        [0.169, 2.74893543762, 26735.94526221320],
        [0.165, 3.95288000638, 6357.85744855870],
        [0.183, 5.43418358741, 369.69981594040],
        [0.134, 3.09132862833, 17.81252211800],
        [0.132, 3.05633896779, 22490.96212149340],
        [0.134, 4.09472795832, 6599.46771964800],
        [0.181, 4.22950689891, 966.97087743560],
        [0.152, 5.28885894415, 12669.24447420140],
        [0.150, 5.86819430908, 97238.62754448749],
        [0.142, 5.87266532526, 22476.73502749179],
        [0.145, 5.07330784304, 87.30820453981],
        [0.133, 5.65471067133, 31.97230581680],
        [0.124, 2.83326217072, 12566.21901028560],
        [0.135, 3.12861731644, 32217.20018108080],
        [0.137, 0.86487461904, 9924.81042151060],
        [0.172, 1.98369595114, 174242.46596404971],
        [0.170, 4.41115280254, 327574.51427678124],
        [0.151, 0.46542099527, 39609.65458316560],
        [0.148, 2.13439571118, 491.66329245880],
        [0.153, 3.78801830344, 17363.24742890899],
        [0.165, 5.31654110459, 16943.76278503380],
        [0.165, 4.06747587817, 58953.14544329400],
        [0.118, 0.63846333239, 6.06591562980],
        [0.159, 0.86086959274, 221995.02880149524],
        [0.119, 5.96432932413, 1385.89527633620],
        [0.114, 5.16516114595, 25685.87280280800],
        [0.112, 3.39403722178, 21393.54196985760],
        [0.112, 4.92889233335, 56.80326216980],
        [0.119, 2.40637635942, 18635.92845453620],
        [0.115, 0.23374479051, 418.92439890060],
        [0.122, 0.93575234049, 24492.40611365159],
        [0.115, 4.58880032176, 26709.64694241340],
        [0.130, 4.85539251000, 22345.26037610820],
        [0.140, 1.09413073202, 44809.65020086340],
        [0.112, 6.05401806281, 433.71173787680],
        [0.104, 1.54931540602, 127.95153303460],
        [0.105, 4.82620858888, 33794.54372352860],
        [0.102, 4.12448497391, 15664.03552270859],
        [0.107, 4.67919356465, 77690.75950573849],
        [0.118, 4.52320170120, 19004.64794940840],
        [0.107, 5.71774478555, 77736.78343050249],
        [0.143, 1.81201813018, 4214.06901508480],
        [0.125, 1.14419195615, 625.67019231240],
        [0.124, 3.27736514057, 12566.08438968000],
        [0.110, 1.08682570828, 2787.04302385740],
        [0.105, 1.78318141871, 18139.29450141590],
        [0.102, 4.75119578149, 12242.64628332540],
        [0.137, 1.43510636754, 86464.61331683119],
        [0.101, 4.91289409429, 401.67212175720],
        [0.129, 1.23567904485, 12029.34718788740],
        [0.138, 2.45654707999, 7576.56007357400],
        [0.103, 0.40004073416, 90279.92316810328],
        [0.108, 0.98989774940, 5636.06501667660],
        [0.117, 5.17362872063, 34520.30930938080],
        [0.100, 3.95534628189, 5547.19933645960],
        [0.098, 1.28118280598, 21548.96236929180],
        [0.097, 3.34717130592, 16310.97904572060],
        [0.098, 4.37041908717, 34513.26307268280],
        [0.125, 2.72164432960, 24065.80792277559],
        [0.102, 0.66938025772, 10239.58386601080],
        [0.119, 1.21689479331, 1478.86657406440],
        [0.094, 1.99595224256, 13362.44970679920],
        [0.094, 4.30965982872, 26880.31981303260],
        [0.095, 2.89807657534, 34911.41207609100],
        [0.106, 1.00156653590, 16522.65971600220],
        [0.097, 0.89642320201, 71980.63357473118],
        [0.116, 4.19967201116, 206.70073729660],
        [0.099, 1.37437847718, 1039.02661079040],
        [0.126, 3.21642544972, 305281.94307104882],
        [0.094, 0.68997876060, 7834.12107263940],
        [0.094, 5.58132218606, 3104.93005942380],
        [0.095, 3.03823741110, 8982.81066930900],
        [0.108, 0.52696637156, 276.74577186440],
        [0.124, 3.43899862683, 172146.97134054030],
        [0.102, 1.04031728553, 95143.13292097810],
        [0.104, 3.39218586218, 290.97286586600],
        [0.110, 3.68205877433, 22380.75580027400],
        [0.117, 0.78475956902, 83286.91426955358],
        [0.083, 0.18241793425, 15141.39079431200],
        [0.089, 4.45371820659, 792.77488846740],
        [0.082, 4.80703651241, 6819.88036208680],
        [0.087, 3.43122851097, 27707.54249429480],
        [0.101, 5.32081603011, 2301.58581590939],
        [0.082, 0.87060089842, 10241.20229116720],
        [0.086, 4.61919461931, 36147.40987730040],
        [0.095, 2.87032884659, 23020.65308658799],
        [0.088, 3.21133165690, 33326.57873317420],
        [0.080, 1.84900424847, 21424.46664430340],
        [0.101, 4.18796434479, 30666.15495843280],
        [0.107, 5.77864921649, 34115.11406927460],
        [0.104, 1.08739495962, 6288.59877429880],
        [0.110, 3.32898859416, 72140.62866668739],
        [0.087, 4.40657711727, 142.17862703620],
        [0.109, 1.94546030825, 24279.10701821359],
        [0.087, 4.32472045435, 742.99006053260],
        [0.107, 4.91580912547, 277.03499374140],
        [0.088, 2.10180220766, 26482.17080962440],
        [0.086, 4.01887374432, 12491.37010141550],
        [0.106, 5.49092372854, 62883.35513951360],
        [0.080, 6.19781316983, 6709.67404086740],
        [0.088, 2.09872810657, 238004.52415723630],
        [0.083, 4.90662164029, 51.28033786241],
        [0.095, 4.13387406591, 18216.44381066100],
        [0.078, 6.06949391680, 148434.53403769128],
        [0.079, 3.03048221644, 838.96928775040],
        [0.074, 5.49813051211, 29026.48522950779],
        [0.073, 3.05008665738, 567.71863773040],
        [0.084, 0.46604373274, 45.14121963660],
        [0.093, 2.52267536308, 48739.85989708300],
        [0.076, 1.76418124905, 41654.96311596780],
        [0.067, 5.77851227793, 6311.52503745920],
        [0.062, 3.32967880172, 15508.61512327440],
        [0.079, 5.59773841328, 71960.38658322369],
        [0.057, 3.90629505268, 5999.21653112620],
        [0.061, 0.05695043232, 7856.89627409019],
        [0.061, 5.63297958433, 7863.94251078820],
        [0.065, 3.72178394016, 12573.26524698360],
        [0.057, 4.18217219541, 26087.90314157420],
        [0.066, 3.92262333487, 69853.35207568129],
        [0.053, 5.51119362045, 77710.24834977149],
        [0.053, 4.88573986961, 77717.29458646949],
        [0.062, 2.88876342225, 9411.46461508720],
        [0.051, 1.12657183874, 82576.98122099529],
        [0.045, 2.95671076719, 24602.61243487099],
        [0.040, 5.55145719241, 12565.17137891460],
        [0.039, 1.20838190039, 18842.11400297339],
        [0.045, 3.18590558749, 45585.17281218740],
        [0.049, 2.44790934886, 13613.80427733600],
    ],
    # L1
    [
        [628331966747.491, 0.00000000000, 0.00000000000],
        [206058.863, 2.67823455584, 6283.07584999140],
        [4303.430, 2.63512650414, 12566.15169998280],
        [425.264, 1.59046980729, 3.52311834900],
        [108.977, 2.96618001993, 1577.34354244780],
        [93.478, 2.59212835365, 18849.22754997420],
        [119.261, 5.79557487799, 26.29831979980],
        [72.122, 1.13846158196, 529.69096509460],
        [67.768, 1.87472304791, 398.14900340820],
        [67.327, 4.40918235168, 5507.55323866740],
        [59.027, 2.88797038460, 5223.69391980220],
        [55.976, 2.17471680261, 155.42039943420],
        [45.407, 0.39803079805, 796.29800681640],
        [36.369, 0.46624739835, 775.52261132400],
        [28.958, 2.64707383882, 7.11354700080],
        [19.097, 1.84628332577, 5486.77784317500],
        [20.844, 5.34138275149, 0.98032106820],
        [18.508, 4.96855124577, 213.29909543800],
        [16.233, 0.03216483047, 2544.31441988340],
        [17.293, 2.99116864949, 6275.96230299060],
        [15.832, 1.43049285325, 2146.16541647520],
        [14.615, 1.20532366323, 10977.07880469900],
        [11.877, 3.25804815607, 5088.62883976680],
        [11.514, 2.07502418155, 4694.00295470760],
        [9.721, 4.23925472239, 1349.86740965880],
        [9.969, 1.30262991097, 6286.59896834040],
        [9.452, 2.69957062864, 242.72860397400],
        [12.461, 2.83432285512, 1748.01641306700],
        [11.808, 5.27379790480, 1194.44701022460],
        [8.577, 5.64475868067, 951.71840625060],
        [10.641, 0.76614199202, 553.56940284240],
        [7.576, 5.30062664886, 2352.86615377180],
        [5.834, 1.76649917904, 1059.38193018920],
        [6.385, 2.65033984967, 9437.76293488700],
        [5.223, 5.66135767624, 71430.69561812909],
        [5.305, 0.90857521574, 3154.68708489560],
        [6.101, 4.66632584188, 4690.47983635860],
        [4.330, 0.24102555403, 6812.76681508600],
        [5.041, 1.42490103709, 6438.49624942560],
        [4.259, 0.77355900599, 10447.38783960440],
        [5.198, 1.85353197345, 801.82093112380],
        [3.744, 2.00119516488, 8031.09226305840],
        [3.558, 2.42901552681, 14143.49524243060],
        [3.372, 3.86210700128, 1592.59601363280],
        [3.374, 0.88776219727, 12036.46073488820],
        [3.175, 3.18785710594, 4705.73230754360],
        [3.221, 0.61599835472, 8429.24126646660],
        [4.132, 5.23992859705, 7084.89678111520],
        [2.970, 6.07026318493, 4292.33083295040],
        [2.900, 2.32464208411, 20.35531939880],
        [3.504, 4.79975694359, 6279.55273164240],
        [2.950, 1.43108874817, 5746.27133789600],
        [2.697, 4.80368225199, 7234.79425624200],
        [2.531, 6.22290682655, 6836.64525283380],
        [2.745, 0.93466065396, 5760.49843189760],
        [3.250, 3.39954640038, 7632.94325965020],
        [2.277, 5.00277837672, 17789.84561978500],
        [2.075, 3.95534978634, 10213.28554621100],
        [2.061, 2.22411683077, 5856.47765911540],
        [2.252, 5.67166499885, 11499.65622279280],
        [2.148, 5.20184578235, 11513.88331679440],
        [1.886, 0.53198320577, 3340.61242669980],
        [1.875, 4.73511970207, 83996.84731811189],
        [2.060, 2.54987293999, 25132.30339996560],
        [1.794, 1.47435409831, 4164.31198961300],
        [1.778, 3.02473091781, 5.52292430740],
        [2.029, 0.90960209983, 6256.77753019160],
        [2.075, 2.26767270157, 522.57741809380],
        [1.772, 3.02622802353, 5753.38488489680],
        [1.569, 6.12410242782, 5216.58037280140],
        [1.590, 4.63713748247, 3.28635741780],
        [1.542, 4.20004448567, 13367.97263110660],
        [1.427, 1.19088061711, 3894.18182954220],
        [1.375, 3.09301252193, 135.06508003540],
        [1.359, 4.24532506641, 426.59819087600],
        [1.340, 5.76511818622, 6040.34724601740],
        [1.284, 3.08524663344, 5643.17856367740],
        [1.250, 3.07748157144, 11926.25441366880],
        [1.551, 3.07665451458, 6681.22485339960],
        [1.268, 2.09196018331, 6290.18939699220],
        [1.144, 3.24444699514, 12168.00269657460],
        [1.248, 3.44504937285, 536.80451209540],
        [1.118, 2.31829670425, 16730.46368959580],
        [1.105, 5.31966001019, 23.87843774780],
        [1.051, 3.75015946014, 7860.41939243920],
        [1.025, 2.44688534235, 1990.74501704100],
        [0.962, 0.81771017882, 3.88133535800],
        [0.910, 0.41727865299, 7079.37385680780],
        [0.883, 5.16833917651, 11790.62908865880],
        [0.957, 4.07673573735, 6127.65545055720],
        [1.110, 3.90096793825, 11506.76976979360],
        [0.802, 3.88778875582, 10973.55568635000],
        [0.780, 2.39934293755, 1589.07289528380],
        [0.758, 1.30034364248, 103.09277421860],
        [0.749, 4.96275803300, 6496.37494542940],
        [0.765, 3.36312388424, 36.02786667740],
        [0.915, 5.41543742089, 206.18554843720],
        [0.776, 2.57589093871, 11371.70468975820],
        [0.772, 3.98369209464, 955.59974160860],
        [0.749, 5.17890001805, 10969.96525769820],
        [0.806, 0.34218864254, 9917.69687450980],
        [0.728, 5.20962563787, 38.02767263580],
        [0.685, 2.77592961854, 20.77539549240],
        [0.636, 4.28242193632, 28.44918746780],
        [0.608, 5.63278508906, 10984.19235169980],
        [0.704, 5.60738823665, 3738.76143010800],
        [0.685, 0.38876148682, 15.25247118500],
        [0.601, 0.73489602442, 419.48464387520],
        [0.716, 2.65279791438, 6309.37416979120],
        [0.584, 5.54502568227, 17298.18232732620],
        [0.650, 1.13379656406, 7058.59846131540],
        [0.688, 2.59683891779, 3496.03282613400],
        [0.485, 0.44467180946, 12352.85260454480],
        [0.528, 2.74936967681, 3930.20969621960],
        [0.597, 5.27668281777, 10575.40668294180],
        [0.583, 3.18929067810, 4732.03062734340],
        [0.526, 5.01697321546, 5884.92684658320],
        [0.540, 1.29175137075, 640.87760738220],
        [0.473, 5.49953306970, 5230.80746680300],
        [0.406, 5.21248452189, 220.41264243880],
        [0.395, 1.87474483222, 16200.77272450120],
        [0.370, 3.84921354713, 18073.70493865020],
        [0.367, 0.88533542778, 6283.14316029419],
        [0.379, 0.37983009325, 10177.25767953360],
        [0.356, 3.84145204913, 11712.95531823080],
        [0.374, 5.01577520608, 7.04623669800],
        [0.381, 4.30250406634, 6062.66320755260],
        [0.471, 0.86381834647, 6069.77675455340],
        [0.367, 1.32943839763, 6283.00853968860],
        [0.460, 5.19667219575, 6284.05617105960],
        [0.333, 5.54256205741, 4686.88940770680],
        [0.341, 4.36522989934, 7238.67559160000],
        [0.336, 4.00205876835, 3097.88382272579],
        [0.359, 6.22679790284, 245.83164622940],
        [0.307, 2.35299010924, 170.67287061920],
        [0.343, 3.77164927143, 6076.89030155420],
        [0.296, 5.44152227481, 17260.15465469040],
        [0.328, 0.13837875384, 11015.10647733480],
        [0.268, 1.13904550630, 12569.67481833180],
        [0.263, 0.00538633678, 4136.91043351620],
        [0.282, 5.04399837480, 7477.52286021600],
        [0.288, 3.13401177517, 12559.03815298200],
        [0.259, 0.93882269387, 5642.19824260920],
        [0.292, 1.98420020514, 12132.43996210600],
        [0.247, 3.84244798532, 5429.87946823940],
        [0.245, 5.70467521726, 65147.61976813770],
        [0.241, 0.99480969552, 3634.62102451840],
        [0.246, 3.06168069935, 110.20632121940],
        [0.239, 6.11855909114, 11856.21865142450],
        [0.263, 0.66348415419, 21228.39202354580],
        [0.262, 1.51070507866, 12146.66705610760],
        [0.230, 1.75927314884, 9779.10867612540],
        [0.223, 2.00967043606, 6172.86952877200],
        [0.246, 1.10411690865, 6282.09552892320],
        [0.221, 3.03945240854, 8635.94200376320],
        [0.214, 4.03840869663, 14314.16811304980],
        [0.236, 5.46915070580, 13916.01910964160],
        [0.224, 4.68408089456, 24072.92146977640],
        [0.212, 2.13695625494, 5849.36411211460],
        [0.207, 3.07724246401, 11.72935283600],
        [0.207, 6.10306282747, 23543.23050468179],
        [0.266, 1.00709566823, 2388.89402044920],
        [0.217, 6.27837036335, 17267.26820169119],
        [0.204, 2.34615348695, 266.60704172180],
        [0.195, 5.55015549753, 6133.51265285680],
        [0.188, 2.52667166175, 6525.80445396540],
        [0.185, 0.90960768344, 18319.53658487960],
        [0.177, 1.73429218289, 154717.60988768269],
        [0.187, 4.76483647432, 4535.05943692440],
        [0.186, 4.63080493407, 10440.27429260360],
        [0.215, 2.81255454560, 7342.45778018060],
        [0.172, 1.45551888559, 9225.53927328300],
        [0.162, 3.30661909388, 639.89728631400],
        [0.168, 2.17671416605, 27.40155609680],
        [0.160, 1.68164180475, 15110.46611986620],
        [0.158, 0.13519771874, 13095.84266507740],
        [0.183, 0.56281322071, 13517.87010623340],
        [0.179, 3.58450811616, 87.30820453981],
        [0.152, 2.84070476818, 5650.29211067820],
        [0.182, 0.44065530624, 17253.04110768959],
        [0.160, 5.95767264171, 4701.11650170840],
        [0.142, 1.46290137520, 11087.28512591840],
        [0.142, 2.04464036087, 20426.57109242200],
        [0.131, 5.40912137746, 2699.73481931760],
        [0.144, 2.07312090485, 25158.60171976540],
        [0.147, 6.15106982168, 9623.68827669120],
        [0.141, 5.55739979498, 10454.50138660520],
        [0.135, 0.06098110407, 16723.35014259500],
        [0.124, 5.81218025669, 17256.63153634140],
        [0.124, 2.36293551623, 4933.20844033260],
        [0.126, 3.47435905118, 22483.84857449259],
        [0.159, 5.63954754618, 5729.50644714900],
        [0.123, 3.92815963256, 17996.03116822220],
        [0.148, 3.02509280598, 1551.04522264800],
        [0.120, 5.91904349732, 6206.80977871580],
        [0.134, 3.11122937825, 21954.15760939799],
        [0.119, 5.52141123450, 709.93304855830],
        [0.122, 3.00813429479, 19800.94595622480],
        [0.127, 1.37618620001, 14945.31617355440],
        [0.141, 2.56889468729, 1052.26838318840],
        [0.123, 2.83671175442, 11919.14086666800],
        [0.118, 0.81934438215, 5331.35744374080],
        [0.151, 2.68731829165, 11769.85369316640],
        [0.119, 5.08835797638, 5481.25491886760],
        [0.153, 2.46021790779, 11933.36796066960],
        [0.108, 1.04936452145, 11403.67699557500],
        [0.128, 0.99794735107, 8827.39026987480],
        [0.144, 2.54869747042, 227.47613278900],
        [0.150, 4.50631437136, 2379.16447357160],
        [0.107, 1.79272017026, 13119.72110282519],
        [0.107, 4.43556814486, 18422.62935909819],
        [0.109, 0.29269062317, 16737.57723659660],
        [0.141, 3.18979826258, 6262.30045449900],
        [0.122, 4.23040027813, 29.42950853600],
        [0.111, 5.16954029551, 17782.73207278420],
        [0.100, 3.52213872761, 18052.92954315780],
        [0.108, 1.08514212991, 16858.48253293320],
        [0.106, 1.96085248410, 74.78159856730],
        [0.110, 2.30582372873, 16460.33352952499],
        [0.097, 3.50918940210, 5333.90024102160],
        [0.099, 3.56417337974, 735.87651353180],
        [0.094, 5.01857894228, 3128.38876509580],
        [0.097, 1.65579893894, 533.21408344360],
        [0.092, 0.89217162285, 29296.61538957860],
        [0.123, 3.16062050433, 9380.95967271720],
        [0.102, 1.20493500565, 23020.65308658799],
        [0.088, 2.21296088224, 12721.57209941700],
        [0.089, 1.54264720310, 20199.09495963300],
        [0.113, 4.83320707870, 16496.36139620240],
        [0.121, 6.19860353182, 9388.00590941520],
        [0.089, 4.08082274765, 22805.73556599360],
        [0.098, 1.09181832830, 12043.57428188900],
        [0.086, 1.13655027605, 143571.32428481648],
        [0.088, 5.96980472191, 107.66352393860],
        [0.082, 5.01340404594, 22003.91463486980],
        [0.094, 1.69615700473, 23006.42599258639],
        [0.081, 3.00657814365, 2118.76386037840],
        [0.098, 1.39215287161, 8662.24032356300],
        [0.077, 3.33555190840, 15720.83878487840],
        [0.082, 5.86880116464, 2787.04302385740],
        [0.076, 5.67183650604, 14.22709400160],
        [0.081, 6.16619455699, 1039.02661079040],
        [0.076, 3.21449884756, 111.18664228760],
        [0.078, 1.37531518377, 21947.11137270000],
        [0.074, 3.58814195051, 11609.86254401220],
        [0.077, 4.84846488388, 22743.40937951640],
        [0.090, 1.48869013606, 15671.08175940660],
        [0.082, 3.48618399109, 29088.81141598500],
        [0.069, 3.55746476593, 4590.91018048900],
        [0.069, 1.93625656075, 135.62532501000],
        [0.070, 2.66548322237, 18875.52586977400],
        [0.069, 5.41478093731, 26735.94526221320],
        [0.079, 5.15154513662, 12323.42309600880],
        [0.094, 3.62899392448, 77713.77146812050],
        [0.078, 4.17011182047, 1066.49547719000],
        [0.071, 3.89435637865, 22779.43724619380],
        [0.063, 4.53968787714, 8982.81066930900],
        [0.069, 0.96028230548, 14919.01785375460],
        [0.076, 3.29092216589, 2942.46342329160],
        [0.063, 4.09167842893, 16062.18452611680],
        [0.065, 3.34580407184, 51.28033786241],
        [0.065, 5.75757544877, 52670.06959330260],
        [0.068, 5.75884067555, 21424.46664430340],
        [0.057, 5.45122399850, 12592.45001978260],
        [0.057, 5.25043362558, 20995.39296644940],
        [0.073, 0.53299090807, 2301.58581590939],
        [0.070, 4.31243357502, 19402.79695281660],
        [0.067, 2.53852336668, 377.37360791580],
        [0.056, 3.20816844695, 24889.57479599160],
        [0.053, 3.17816599142, 18451.07854656599],
        [0.053, 3.61529270216, 77.67377042800],
        [0.053, 0.45467549335, 30666.15495843280],
        [0.061, 0.14807288453, 23013.53953958720],
        [0.051, 3.32803972907, 56.89837493560],
        [0.052, 3.41177624177, 23141.55838292460],
        [0.058, 3.13638677202, 309.27832265580],
        [0.070, 2.50592323465, 31415.37924995700],
        [0.052, 5.10673376738, 17796.95916678580],
        [0.067, 6.27917920454, 22345.26037610820],
        [0.050, 0.42577644151, 25685.87280280800],
        [0.048, 0.70204553333, 1162.47470440780],
        [0.066, 3.64350022359, 15265.88651930040],
        [0.050, 5.74382917440, 19.66976089979],
        [0.050, 4.69825387775, 28237.23345938940],
        [0.047, 5.74015846442, 12139.55350910680],
        [0.054, 1.97301333704, 23581.25817731760],
        [0.049, 4.98223579027, 10021.83728009940],
        [0.046, 5.41431705539, 33019.02111220460],
        [0.051, 1.23882053879, 12539.85338018300],
        [0.046, 2.41369976086, 98068.53671630539],
        [0.044, 0.80750593746, 167283.76158766550],
        [0.045, 4.39613584445, 433.71173787680],
        [0.044, 2.57358208785, 12964.30070339100],
        [0.046, 0.26142733448, 11.04570026390],
        [0.045, 2.46230645202, 51868.24866217880],
        [0.048, 0.89551707131, 56600.27928952220],
        [0.057, 1.86416707010, 25287.72379939980],
        [0.042, 5.26377513431, 26084.02180621620],
        [0.049, 3.17757670611, 6303.85124548380],
        [0.052, 3.65266055509, 7872.14874527520],
        [0.040, 1.81891629936, 34596.36465465240],
        [0.043, 1.94164978061, 1903.43681250120],
        [0.041, 0.74461854136, 23937.85638974100],
        [0.048, 6.26034008181, 28286.99048486120],
        [0.045, 5.45575017530, 60530.48898574180],
        [0.040, 2.92105728682, 21548.96236929180],
        [0.040, 0.04502010161, 38526.57435087200],
        [0.053, 3.64791042082, 11925.27409260060],
        [0.041, 5.04048954693, 27832.03821928320],
        [0.042, 5.19292937193, 19004.64794940840],
        [0.040, 2.57120233428, 24356.78078864160],
        [0.038, 3.49190341464, 226858.23855437007],
        [0.039, 4.61184303844, 95.97922721780],
        [0.043, 2.20648228147, 13521.75144159140],
        [0.040, 5.83461945819, 16193.65917750039],
        [0.045, 3.73714372195, 7875.67186362420],
        [0.043, 1.14078465002, 49.75702547180],
        [0.037, 1.29390383811, 310.84079886840],
        [0.038, 0.95970925950, 664.75604513000],
        [0.037, 4.27532649462, 6709.67404086740],
        [0.038, 2.20108541046, 28628.33622609960],
        [0.039, 0.85957361635, 16522.65971600220],
        [0.040, 4.35214003837, 48739.85989708300],
        [0.036, 1.68167662194, 10344.29506538580],
        [0.040, 5.13217319067, 15664.03552270859],
        [0.036, 3.72187132496, 30774.50164257480],
        [0.036, 3.32158458257, 16207.88627150200],
        [0.045, 3.94202418608, 10988.80815753500],
        [0.039, 1.51948786199, 12029.34718788740],
        [0.026, 3.87685883180, 6262.72053059260],
        [0.024, 4.91804163466, 19651.04848109800],
        [0.023, 0.29300197709, 13362.44970679920],
        [0.021, 3.18605672363, 6277.55292568400],
        [0.021, 6.07546891132, 18139.29450141590],
        [0.022, 2.31199937177, 6303.43116939020],
        [0.021, 3.58418394393, 18209.33026366019],
        [0.026, 2.06801296900, 12573.26524698360],
        [0.021, 1.56857722317, 13341.67431130680],
        [0.024, 5.72605158675, 29864.33402730900],
        [0.024, 1.40237993205, 14712.31711645800],
        [0.025, 5.71466092822, 25934.12433108940],
    ],
    # L2
    [
        [52918.870, 0.00000000000, 0.00000000000],
        [8719.837, 1.07209665242, 6283.07584999140],
        [309.125, 0.86728818832, 12566.15169998280],
        [27.339, 0.05297871691, 3.52311834900],
        [16.334, 5.18826691036, 26.29831979980],
        [15.752, 3.68457889430, 155.42039943420],
        [9.541, 0.75742297675, 18849.22754997420],
        [8.937, 2.05705419118, 77713.77146812050],
        [6.952, 0.82673305410, 775.52261132400],
        [5.064, 4.66284525271, 1577.34354244780],
        [4.061, 1.03057162962, 7.11354700080],
        [3.463, 5.14074632811, 796.29800681640],
        [3.169, 6.05291851171, 5507.55323866740],
        [3.020, 1.19246506441, 242.72860397400],
        [2.886, 6.11652627155, 529.69096509460],
        [3.810, 3.44050803490, 5573.14280143310],
        [2.714, 0.30637881025, 398.14900340820],
        [2.371, 4.38118838167, 5223.69391980220],
        [2.538, 2.27992810679, 553.56940284240],
        [2.079, 3.75435330484, 0.98032106820],
        [1.675, 0.90216407959, 951.71840625060],
        [1.534, 5.75900462759, 1349.86740965880],
        [1.224, 2.97328088405, 2146.16541647520],
        [1.449, 4.36415913970, 1748.01641306700],
        [1.341, 3.72061130861, 1194.44701022460],
        [1.254, 2.94846826628, 6438.49624942560],
        [0.999, 5.98640014468, 6286.59896834040],
        [0.917, 4.79788687522, 5088.62883976680],
        [0.828, 3.31321076572, 213.29909543800],
        [1.103, 1.27104454479, 161000.68573767410],
        [0.762, 3.41582762988, 5486.77784317500],
        [1.044, 0.60409577691, 3154.68708489560],
        [0.887, 5.23465144638, 7084.89678111520],
        [0.645, 1.60096192515, 2544.31441988340],
        [0.681, 3.43155669169, 4694.00295470760],
        [0.605, 2.47806340546, 10977.07880469900],
        [0.706, 6.19393222575, 4690.47983635860],
        [0.643, 1.98042503148, 801.82093112380],
        [0.502, 1.44394375363, 6836.64525283380],
        [0.490, 2.34129524194, 1592.59601363280],
        [0.458, 1.30876448575, 4292.33083295040],
        [0.431, 0.03526421494, 7234.79425624200],
        [0.379, 3.17030522615, 6309.37416979120],
        [0.348, 0.99049550009, 6040.34724601740],
        [0.386, 1.57019797263, 71430.69561812909],
        [0.347, 0.67013291338, 1059.38193018920],
        [0.458, 3.81499443681, 149854.40013480789],
        [0.302, 1.91760044838, 10447.38783960440],
        [0.307, 3.55343347416, 8031.09226305840],
        [0.395, 4.93701776616, 7632.94325965020],
        [0.314, 3.18093696547, 2352.86615377180],
        [0.282, 4.41936437052, 9437.76293488700],
        [0.276, 2.71314254553, 3894.18182954220],
        [0.298, 2.52037474210, 6127.65545055720],
        [0.230, 1.37790215549, 4705.73230754360],
        [0.252, 0.55330133471, 6279.55273164240],
        [0.255, 5.26570187369, 6812.76681508600],
        [0.275, 0.67264264272, 25132.30339996560],
        [0.178, 0.92820785174, 1990.74501704100],
        [0.221, 0.63897368842, 6256.77753019160],
        [0.155, 0.77319790838, 14143.49524243060],
        [0.150, 2.40470465561, 426.59819087600],
        [0.196, 6.06877865012, 640.87760738220],
        [0.137, 2.21679460145, 8429.24126646660],
        [0.127, 3.26094223174, 17789.84561978500],
        [0.128, 5.47237279946, 12036.46073488820],
        [0.122, 2.16291082757, 10213.28554621100],
        [0.118, 0.45789822268, 7058.59846131540],
        [0.141, 2.34932647403, 11506.76976979360],
        [0.100, 0.85621569847, 6290.18939699220],
        [0.092, 5.10587476002, 7079.37385680780],
        [0.126, 2.65428307012, 88860.05707098669],
        [0.106, 5.85646710022, 7860.41939243920],
        [0.084, 3.57457554262, 16730.46368959580],
        [0.089, 4.21433259618, 83996.84731811189],
        [0.097, 5.57938280855, 13367.97263110660],
        [0.102, 2.05853060226, 87.30820453981],
        [0.080, 4.73792651816, 11926.25441366880],
        [0.080, 5.41418965044, 10973.55568635000],
        [0.106, 4.10978997399, 3496.03282613400],
        [0.102, 3.62650006043, 244287.60000722768],
        [0.075, 4.89483161769, 5643.17856367740],
        [0.087, 0.42863750683, 11015.10647733480],
        [0.069, 1.88908760720, 10177.25767953360],
        [0.089, 1.35567273119, 6681.22485339960],
        [0.066, 0.99455837265, 6525.80445396540],
        [0.067, 5.51240997070, 3097.88382272579],
        [0.076, 2.72016814799, 4164.31198961300],
        [0.063, 1.44349902540, 9917.69687450980],
        [0.078, 3.51469733747, 11856.21865142450],
        [0.085, 0.50956043858, 10575.40668294180],
        [0.067, 3.62043033405, 16496.36139620240],
        [0.055, 5.24637517308, 3340.61242669980],
        [0.048, 5.43966777314, 20426.57109242200],
        [0.064, 5.79535817813, 2388.89402044920],
        [0.046, 5.43499966519, 6275.96230299060],
        [0.050, 3.86263598617, 5729.50644714900],
        [0.044, 1.52269529228, 12168.00269657460],
        [0.057, 4.96352373486, 14945.31617355440],
        [0.045, 1.00861230160, 8635.94200376320],
        [0.043, 3.30685683359, 9779.10867612540],
        [0.042, 0.63481258930, 2699.73481931760],
        [0.041, 5.67996766641, 11712.95531823080],
        [0.056, 4.34024451468, 90955.55169449610],
        [0.041, 5.81722212845, 709.93304855830],
        [0.053, 6.17052087143, 233141.31440436150],
        [0.037, 3.12495025087, 16200.77272450120],
        [0.035, 5.76973458495, 12569.67481833180],
        [0.037, 0.31656444326, 24356.78078864160],
        [0.035, 0.96229051027, 17298.18232732620],
        [0.033, 5.23130355867, 5331.35744374080],
        [0.035, 0.62517020593, 25158.60171976540],
        [0.035, 0.80004512129, 13916.01910964160],
        [0.037, 2.89336088688, 12721.57209941700],
        [0.030, 4.50198402401, 23543.23050468179],
        [0.030, 5.31355708693, 18319.53658487960],
        [0.029, 3.47275229977, 13119.72110282519],
        [0.029, 3.11002782516, 4136.91043351620],
        [0.032, 5.52273255667, 5753.38488489680],
        [0.035, 3.79699996680, 143571.32428481648],
        [0.026, 1.50634201907, 154717.60988768269],
        [0.030, 3.53519084118, 6284.05617105960],
        [0.023, 4.41808025967, 5884.92684658320],
        [0.025, 1.38477355808, 65147.61976813770],
        [0.023, 3.49782549797, 7477.52286021600],
        [0.019, 3.14329413716, 6496.37494542940],
        [0.019, 2.20135125199, 18073.70493865020],
        [0.019, 4.95020255309, 3930.20969621960],
        [0.019, 0.57998702747, 31415.37924995700],
        [0.021, 1.75474323399, 12139.55350910680],
        [0.019, 3.92233070499, 19651.04848109800],
        [0.014, 0.98131213224, 12559.03815298200],
        [0.019, 4.93309333729, 2942.46342329160],
        [0.016, 5.55997534558, 8827.39026987480],
        [0.013, 1.68808165516, 4535.05943692440],
        [0.013, 0.33982116161, 4933.20844033260],
        [0.012, 1.85426309994, 5856.47765911540],
        [0.010, 4.82763996845, 13095.84266507740],
        [0.011, 5.38005490571, 11790.62908865880],
        [0.010, 1.40815507226, 10988.80815753500],
        [0.011, 3.05005267431, 17260.15465469040],
        [0.010, 4.93364992366, 12352.85260454480],
    ],
    # L3
    [
        [289.226, 5.84384198723, 6283.07584999140],
        [34.955, 0.00000000000, 0.00000000000],
        [16.819, 5.48766912348, 12566.15169998280],
        [2.962, 5.19577265202, 155.42039943420],
        [1.288, 4.72200252235, 3.52311834900],
        [0.635, 5.96925937141, 242.72860397400],
        [0.714, 5.30045809128, 18849.22754997420],
        [0.402, 3.78682982419, 553.56940284240],
        [0.072, 4.29768126180, 6286.59896834040],
        [0.067, 0.90721687647, 6127.65545055720],
        [0.036, 5.24029648014, 6438.49624942560],
        [0.024, 5.16003960716, 25132.30339996560],
        [0.023, 3.01921570335, 6309.37416979120],
        [0.017, 5.82863573502, 6525.80445396540],
        [0.017, 3.67772863930, 71430.69561812909],
        [0.009, 4.58467294499, 1577.34354244780],
        [0.008, 1.40626662824, 11856.21865142450],
        [0.008, 5.07561257196, 6256.77753019160],
        [0.007, 2.82473374405, 83996.84731811189],
        [0.005, 2.71488713339, 10977.07880469900],
        [0.005, 3.76879847273, 12036.46073488820],
        [0.005, 4.28412873331, 6275.96230299060],
    ],
    # L4
    [
        [114.084, 3.14159265359, 0.00000000000],
        [7.717, 4.13446589358, 6283.07584999140],
        [0.765, 3.83803776214, 12566.15169998280],
        [0.420, 0.41925861858, 155.42039943420],
        [0.040, 3.59847585840, 18849.22754997420],
        [0.041, 3.14398414077, 3.52311834900],
        [0.035, 5.00298940826, 5573.14280143310],
        [0.013, 0.48794833701, 77713.77146812050],
        [0.010, 5.64801766350, 6127.65545055720],
        [0.008, 2.84160570605, 161000.68573767410],
        [0.002, 0.54912904658, 6438.49624942560],
    ],
    # L5
    [
        [0.878, 3.14159265359, 0.00000000000],
        [0.172, 2.76579069510, 6283.07584999140],
        [0.050, 2.01353298182, 155.42039943420],
        [0.028, 2.21496423926, 12566.15169998280],
        [0.005, 1.75600058765, 18849.22754997420],
    ],
]
"""This table contains Earth's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric longitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
418-420."""


VSOP87_B = [
    # B0
    [
        [279.620, 3.19870156017, 84334.66158130829],
        [101.643, 5.42248619256, 5507.55323866740],
        [80.445, 3.88013204458, 5223.69391980220],
        [43.806, 3.70444689758, 2352.86615377180],
        [31.933, 4.00026369781, 1577.34354244780],
        [22.724, 3.98473831560, 1047.74731175470],
        [16.392, 3.56456119782, 5856.47765911540],
        [18.141, 4.98367470263, 6283.07584999140],
        [14.443, 3.70275614914, 9437.76293488700],
        [14.304, 3.41117857525, 10213.28554621100],
        [11.246, 4.82820690530, 14143.49524243060],
        [10.900, 2.08574562327, 6812.76681508600],
        [9.714, 3.47303947752, 4694.00295470760],
        [10.367, 4.05663927946, 71092.88135493269],
        [8.775, 4.44016515669, 5753.38488489680],
        [8.366, 4.99251512180, 7084.89678111520],
        [6.921, 4.32559054073, 6275.96230299060],
        [9.145, 1.14182646613, 6620.89011318780],
        [7.194, 3.60193205752, 529.69096509460],
        [7.698, 5.55425745881, 167621.57585086190],
        [5.285, 2.48446991566, 4705.73230754360],
        [5.208, 6.24992674537, 18073.70493865020],
        [4.529, 2.33827747356, 6309.37416979120],
        [5.579, 4.41023653738, 7860.41939243920],
        [4.743, 0.70995680136, 5884.92684658320],
        [4.301, 1.10255777773, 6681.22485339960],
        [3.849, 1.82229412531, 5486.77784317500],
        [4.093, 5.11700141207, 13367.97263110660],
        [3.681, 0.43793170356, 3154.68708489560],
        [3.420, 5.42034800952, 6069.77675455340],
        [3.617, 6.04641937526, 3930.20969621960],
        [3.670, 4.58210192227, 12194.03291462090],
        [2.918, 1.95463881126, 10977.07880469900],
        [2.797, 5.61259275048, 11790.62908865880],
        [2.502, 0.60499729367, 6496.37494542940],
        [2.319, 5.01648216014, 1059.38193018920],
        [2.684, 1.39470396488, 22003.91463486980],
        [2.428, 3.24183056052, 78051.58573131690],
        [2.120, 4.30691000285, 5643.17856367740],
        [2.257, 3.15557225618, 90617.73743129970],
        [1.813, 3.75574218285, 3340.61242669980],
        [2.226, 2.79699346659, 12036.46073488820],
        [1.888, 0.86991545823, 8635.94200376320],
        [1.517, 1.95852055701, 398.14900340820],
        [1.581, 3.19976230948, 5088.62883976680],
        [1.421, 6.25530883827, 2544.31441988340],
        [1.595, 0.25619915135, 17298.18232732620],
        [1.391, 4.69964175561, 7058.59846131540],
        [1.478, 2.81808207569, 25934.12433108940],
        [1.481, 3.65823554806, 11506.76976979360],
        [1.693, 4.95689385293, 156475.29024799570],
        [1.183, 1.29343061246, 775.52261132400],
        [1.114, 2.37889311846, 3738.76143010800],
        [0.994, 4.30088900425, 9225.53927328300],
        [0.924, 3.06451026812, 4164.31198961300],
        [0.867, 0.55606931068, 8429.24126646660],
        [0.988, 5.97286104208, 7079.37385680780],
        [0.824, 1.50984806173, 10447.38783960440],
        [0.915, 0.12635654592, 11015.10647733480],
        [0.742, 1.99159139281, 26087.90314157420],
        [1.039, 3.14159265359, 0.00000000000],
        [0.850, 4.24120016095, 29864.33402730900],
        [0.755, 2.89631873320, 4732.03062734340],
        [0.714, 1.37548118603, 2146.16541647520],
        [0.708, 1.91406542362, 8031.09226305840],
        [0.746, 0.57893808616, 796.29800681640],
        [0.802, 5.12339137230, 2942.46342329160],
        [0.751, 1.67479850166, 21228.39202354580],
        [0.602, 4.09976538826, 64809.80550494129],
        [0.594, 3.49580704962, 16496.36139620240],
        [0.592, 4.59481504319, 4690.47983635860],
        [0.530, 5.73979295200, 8827.39026987480],
        [0.503, 5.66433137112, 33794.54372352860],
        [0.483, 1.57106522411, 801.82093112380],
        [0.438, 0.06707733767, 3128.38876509580],
        [0.423, 2.86944595927, 12566.15169998280],
        [0.504, 3.26207669160, 7632.94325965020],
        [0.552, 1.02926440457, 239762.20451754928],
        [0.427, 3.67434378210, 213.29909543800],
        [0.404, 1.46193297142, 15720.83878487840],
        [0.503, 4.85802444134, 6290.18939699220],
        [0.417, 0.81920713533, 5216.58037280140],
        [0.365, 0.01002966162, 12168.00269657460],
        [0.363, 1.28376436579, 6206.80977871580],
        [0.353, 4.70059133110, 7234.79425624200],
        [0.415, 0.96862624175, 4136.91043351620],
        [0.387, 3.09145061418, 25158.60171976540],
        [0.373, 2.65119262792, 7342.45778018060],
        [0.361, 2.97762937739, 9623.68827669120],
        [0.418, 3.75759994446, 5230.80746680300],
        [0.396, 1.22507712354, 6438.49624942560],
        [0.322, 1.21162178805, 8662.24032356300],
        [0.284, 5.64170320068, 1589.07289528380],
        [0.379, 1.72248432748, 14945.31617355440],
        [0.320, 3.94161159962, 7330.82316174610],
        [0.313, 5.47602376446, 1194.44701022460],
        [0.292, 1.38971327603, 11769.85369316640],
        [0.305, 0.80429352049, 37724.75341974820],
        [0.257, 5.81382809757, 426.59819087600],
        [0.265, 6.10358507671, 6836.64525283380],
        [0.250, 4.56452895547, 7477.52286021600],
        [0.266, 2.62926282354, 7238.67559160000],
        [0.263, 6.22089501237, 6133.51265285680],
        [0.306, 2.79682380531, 1748.01641306700],
        [0.236, 2.46093023714, 11371.70468975820],
        [0.316, 1.62662805006, 250908.49012041549],
        [0.216, 3.68721275185, 5849.36411211460],
        [0.230, 0.36165162947, 5863.59120611620],
        [0.233, 5.03509933858, 20426.57109242200],
        [0.200, 5.86073159059, 4535.05943692440],
        [0.277, 4.65400292395, 82239.16695779889],
        [0.209, 3.72323200804, 10973.55568635000],
        [0.199, 5.05186622555, 5429.87946823940],
        [0.256, 2.40923279770, 19651.04848109800],
        [0.210, 4.50691909144, 29088.81141598500],
        [0.181, 6.00294783127, 4292.33083295040],
        [0.249, 0.12900984422, 154379.79562448629],
        [0.209, 3.87759458598, 17789.84561978500],
        [0.225, 3.18339652605, 18875.52586977400],
        [0.191, 4.53897489299, 18477.10876461230],
        [0.172, 2.09694183014, 13095.84266507740],
        [0.182, 3.16107943500, 16730.46368959580],
        [0.188, 2.22746128596, 41654.96311596780],
        [0.164, 5.18686275017, 5481.25491886760],
        [0.160, 2.49298855159, 12592.45001978260],
        [0.155, 1.59595438230, 10021.83728009940],
        [0.135, 0.21349051064, 10988.80815753500],
        [0.178, 3.80375177970, 23581.25817731760],
        [0.123, 1.66800739151, 15110.46611986620],
        [0.122, 2.72678272244, 18849.22754997420],
        [0.126, 1.17675512910, 14919.01785375460],
        [0.142, 3.95053441332, 337.81426319640],
        [0.116, 6.06340906229, 6709.67404086740],
        [0.137, 3.52143246757, 12139.55350910680],
        [0.136, 2.92179113542, 32217.20018108080],
        [0.110, 3.51203379263, 18052.92954315780],
        [0.147, 4.63371971408, 22805.73556599360],
        [0.108, 5.45280814878, 7.11354700080],
        [0.148, 0.65447253687, 95480.94718417450],
        [0.119, 5.92110458985, 33019.02111220460],
        [0.110, 5.34824206306, 639.89728631400],
        [0.106, 3.71081682629, 14314.16811304980],
        [0.139, 6.17607198418, 24356.78078864160],
        [0.118, 5.59738712670, 161338.50000087050],
        [0.117, 3.65065271640, 45585.17281218740],
        [0.127, 4.74596574209, 49515.38250840700],
        [0.120, 1.04211499785, 6915.85958930460],
        [0.120, 5.60638811846, 5650.29211067820],
        [0.115, 3.10668213289, 14712.31711645800],
        [0.099, 0.69018940049, 12779.45079542080],
        [0.097, 1.07908724794, 9917.69687450980],
        [0.093, 2.62295197319, 17260.15465469040],
        [0.099, 4.45774681732, 4933.20844033260],
        [0.123, 1.37488922089, 28286.99048486120],
        [0.121, 5.19767249813, 27511.46787353720],
        [0.105, 0.87192267806, 77375.95720492408],
        [0.087, 3.93637812950, 17654.78053974960],
        [0.122, 2.23956068680, 83997.09113559539],
        [0.087, 4.18201600952, 22779.43724619380],
        [0.104, 4.59580877295, 1349.86740965880],
        [0.102, 2.83545248411, 12352.85260454480],
        [0.102, 3.97386522171, 10818.13528691580],
        [0.101, 4.32892825857, 36147.40987730040],
        [0.094, 5.00001709261, 150192.21439800429],
        [0.077, 3.97199369296, 1592.59601363280],
        [0.100, 6.07733097102, 26735.94526221320],
        [0.086, 5.26029638250, 28313.28880466100],
        [0.093, 4.31900620254, 44809.65020086340],
        [0.076, 6.22743405935, 13521.75144159140],
        [0.072, 1.55820597747, 6256.77753019160],
        [0.082, 4.95202664555, 10575.40668294180],
        [0.082, 1.69647647075, 1990.74501704100],
        [0.075, 2.29836095644, 3634.62102451840],
        [0.075, 2.66367876557, 16200.77272450120],
        [0.087, 0.26630214764, 31441.67756975680],
        [0.077, 2.25530954137, 5235.32853823670],
        [0.076, 1.09869730846, 12903.96596317920],
        [0.058, 4.28246138307, 12559.03815298200],
        [0.064, 5.51112830114, 173904.65170085328],
        [0.056, 2.60133794851, 73188.37597844210],
        [0.055, 5.81483150022, 143233.51002162008],
        [0.054, 3.38482031504, 323049.11878710287],
        [0.039, 3.28500401343, 71768.50988132549],
        [0.039, 3.11239910690, 96900.81328129109],
    ],
    # B1
    [
        [9.030, 3.89729061890, 5507.55323866740],
        [6.177, 1.73038850355, 5223.69391980220],
        [3.800, 5.24404145734, 2352.86615377180],
        [2.834, 2.47345037450, 1577.34354244780],
        [1.817, 0.41874743765, 6283.07584999140],
        [1.499, 1.83320979291, 5856.47765911540],
        [1.466, 5.69401926017, 5753.38488489680],
        [1.301, 2.18890066314, 9437.76293488700],
        [1.233, 4.95222451476, 10213.28554621100],
        [1.021, 0.12866660208, 7860.41939243920],
        [0.982, 0.09005453285, 14143.49524243060],
        [0.865, 1.73949953555, 3930.20969621960],
        [0.581, 2.26949174067, 5884.92684658320],
        [0.524, 5.65662503159, 529.69096509460],
        [0.473, 6.22750969242, 6309.37416979120],
        [0.451, 1.53288619213, 18073.70493865020],
        [0.364, 3.61614477374, 13367.97263110660],
        [0.372, 3.22470721320, 6275.96230299060],
        [0.268, 2.34341267879, 11790.62908865880],
        [0.322, 0.94084045832, 6069.77675455340],
        [0.232, 0.26781182579, 7058.59846131540],
        [0.216, 6.05952221329, 10977.07880469900],
        [0.232, 2.93325646109, 22003.91463486980],
        [0.204, 3.86264841382, 6496.37494542940],
        [0.202, 2.81892511133, 15720.83878487840],
        [0.185, 4.93512381859, 12036.46073488820],
        [0.220, 3.99305643742, 6812.76681508600],
        [0.166, 1.74970002999, 11506.76976979360],
        [0.212, 1.57166285369, 4694.00295470760],
        [0.157, 1.08259734788, 5643.17856367740],
        [0.154, 5.99434678412, 5486.77784317500],
        [0.144, 5.23285656085, 78051.58573131690],
        [0.144, 1.16454655948, 90617.73743129970],
        [0.137, 2.67760436027, 6290.18939699220],
        [0.180, 2.06509026215, 7084.89678111520],
        [0.121, 5.90212574947, 9225.53927328300],
        [0.150, 2.00175038718, 5230.80746680300],
        [0.149, 5.06157254516, 17298.18232732620],
        [0.118, 5.39979058038, 3340.61242669980],
        [0.161, 3.32421999691, 6283.31966747490],
        [0.121, 4.36722193162, 19651.04848109800],
        [0.116, 5.83462858507, 4705.73230754360],
        [0.128, 4.35489873365, 25934.12433108940],
        [0.143, 0.00000000000, 0.00000000000],
        [0.109, 2.52157834166, 6438.49624942560],
        [0.099, 2.70727488041, 5216.58037280140],
        [0.103, 0.93782340879, 8827.39026987480],
        [0.082, 4.29214680390, 8635.94200376320],
        [0.079, 2.24085737326, 1059.38193018920],
        [0.097, 5.50959692365, 29864.33402730900],
        [0.072, 0.21891639822, 21228.39202354580],
        [0.071, 2.86755026812, 6681.22485339960],
        [0.074, 2.20184828895, 37724.75341974820],
        [0.063, 4.45586625948, 7079.37385680780],
        [0.061, 0.63918772258, 33794.54372352860],
        [0.047, 2.09070235724, 3128.38876509580],
        [0.047, 3.32543843300, 26087.90314157420],
        [0.049, 1.60680905005, 6702.56049386660],
        [0.057, 0.11215813438, 29088.81141598500],
        [0.056, 5.47982934911, 775.52261132400],
        [0.050, 1.89396788463, 12139.55350910680],
        [0.047, 2.97214907240, 20426.57109242200],
        [0.041, 5.55329394890, 11015.10647733480],
        [0.041, 5.91861144924, 23581.25817731760],
        [0.045, 4.95273290181, 5863.59120611620],
        [0.050, 3.62740835096, 41654.96311596780],
        [0.037, 6.09033460601, 64809.80550494129],
        [0.037, 5.86153655431, 12566.15169998280],
        [0.046, 1.65798680284, 25158.60171976540],
        [0.038, 2.00673650251, 426.59819087600],
        [0.036, 6.24373396652, 6283.14316029419],
        [0.036, 0.40465162918, 6283.00853968860],
        [0.032, 6.03707103538, 2942.46342329160],
        [0.041, 4.86809570283, 1592.59601363280],
        [0.028, 4.38359423735, 7632.94325965020],
        [0.028, 6.03334294232, 17789.84561978500],
        [0.026, 3.88971333608, 5331.35744374080],
        [0.026, 5.94932724051, 16496.36139620240],
        [0.031, 1.44666331503, 16730.46368959580],
        [0.026, 6.26376705837, 23543.23050468179],
        [0.033, 0.93797239147, 213.29909543800],
        [0.026, 3.71858432944, 13095.84266507740],
        [0.027, 0.60565274405, 10988.80815753500],
        [0.023, 4.44388985550, 18849.22754997420],
        [0.028, 1.53862289477, 6279.48542133960],
        [0.028, 1.96831814872, 6286.66627864320],
        [0.028, 5.78094918529, 15110.46611986620],
        [0.026, 2.48165809843, 5729.50644714900],
        [0.020, 3.85655029499, 9623.68827669120],
        [0.021, 5.83006047147, 7234.79425624200],
        [0.021, 0.69628570421, 398.14900340820],
        [0.022, 5.02222806555, 6127.65545055720],
        [0.020, 3.47611265290, 6148.01076995600],
        [0.020, 0.90769829044, 5481.25491886760],
        [0.020, 0.03081589303, 6418.14093002680],
        [0.020, 3.74220084927, 1589.07289528380],
        [0.021, 4.00149269576, 3154.68708489560],
        [0.018, 1.58348238359, 2118.76386037840],
        [0.019, 0.85407021371, 14712.31711645800],
    ],
    # B2
    [
        [1.662, 1.62703209173, 84334.66158130829],
        [0.492, 2.41382223971, 1047.74731175470],
        [0.344, 2.24353004539, 5507.55323866740],
        [0.258, 6.00906896311, 5223.69391980220],
        [0.131, 0.95447345240, 6283.07584999140],
        [0.086, 1.67530247303, 7860.41939243920],
        [0.090, 0.97606804452, 1577.34354244780],
        [0.090, 0.37899871725, 2352.86615377180],
        [0.089, 6.25807507963, 10213.28554621100],
        [0.075, 0.84213523741, 167621.57585086190],
        [0.052, 1.70501566089, 14143.49524243060],
        [0.057, 6.15295833679, 12194.03291462090],
        [0.051, 1.27616016740, 5753.38488489680],
        [0.051, 5.37229738682, 6812.76681508600],
        [0.034, 1.73672994279, 7058.59846131540],
        [0.038, 2.77761031485, 10988.80815753500],
        [0.046, 3.38617099014, 156475.29024799570],
        [0.021, 1.95248349228, 8827.39026987480],
        [0.018, 3.33419222028, 8429.24126646660],
        [0.019, 4.32945160287, 17789.84561978500],
        [0.017, 0.66191210656, 6283.00853968860],
        [0.018, 3.74885333072, 11769.85369316640],
        [0.017, 4.23058370776, 10977.07880469900],
        [0.017, 1.78116162721, 5486.77784317500],
        [0.021, 1.36972913918, 12036.46073488820],
        [0.017, 2.79601092529, 796.29800681640],
        [0.015, 0.43087848850, 11790.62908865880],
        [0.017, 1.35132152761, 78051.58573131690],
        [0.015, 1.17032155085, 213.29909543800],
        [0.018, 2.85221514199, 5088.62883976680],
        [0.017, 0.21780913672, 6283.14316029419],
        [0.013, 1.21201504386, 25132.30339996560],
        [0.012, 1.12953712197, 90617.73743129970],
        [0.012, 5.13714452592, 7079.37385680780],
        [0.013, 3.79842135217, 4933.20844033260],
        [0.012, 4.89407978213, 3738.76143010800],
        [0.015, 6.05682328852, 398.14900340820],
        [0.014, 4.81029291856, 4694.00295470760],
        [0.011, 0.61684523405, 3128.38876509580],
        [0.011, 5.32876538500, 6040.34724601740],
        [0.014, 5.27227350286, 4535.05943692440],
        [0.011, 2.39292099451, 5331.35744374080],
        [0.010, 4.45296532710, 6525.80445396540],
        [0.014, 4.66400985037, 8031.09226305840],
        [0.010, 3.22472385926, 9437.76293488700],
        [0.011, 3.80913404437, 801.82093112380],
        [0.010, 5.15032130575, 11371.70468975820],
        [0.013, 0.98720797401, 5729.50644714900],
        [0.009, 5.94191743597, 7632.94325965020],
    ],
    # B3
    [
        [0.011, 0.23877262399, 7860.41939243920],
        [0.009, 1.16069982609, 5507.55323866740],
        [0.008, 1.65357552925, 5884.92684658320],
        [0.008, 2.86720038197, 7058.59846131540],
        [0.007, 3.04818741666, 5486.77784317500],
        [0.007, 2.59437103785, 529.69096509460],
        [0.008, 4.02863090524, 6256.77753019160],
        [0.008, 2.42003508927, 5753.38488489680],
        [0.006, 0.84181087594, 6275.96230299060],
        [0.006, 5.40160929468, 1577.34354244780],
        [0.007, 2.73399865247, 6309.37416979120],
    ],
    # B4
    [
        [0.004, 0.79662198849, 6438.49624942560],
        [0.005, 0.84308705203, 1047.74731175470],
        [0.005, 0.05711572303, 84334.66158130829],
        [0.003, 3.46779895686, 6279.55273164240],
        [0.003, 2.89822201212, 6127.65545055720],
    ],
]
"""This table contains Earth's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric latitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in page
420."""


VSOP87_R = [
    # R0
    [
        [100013988.799, 0.00000000000, 0.00000000000],
        [1670699.626, 3.09846350771, 6283.07584999140],
        [13956.023, 3.05524609620, 12566.15169998280],
        [3083.720, 5.19846674381, 77713.77146812050],
        [1628.461, 1.17387749012, 5753.38488489680],
        [1575.568, 2.84685245825, 7860.41939243920],
        [924.799, 5.45292234084, 11506.76976979360],
        [542.444, 4.56409149777, 3930.20969621960],
        [472.110, 3.66100022149, 5884.92684658320],
        [328.780, 5.89983646482, 5223.69391980220],
        [345.983, 0.96368617687, 5507.55323866740],
        [306.784, 0.29867139512, 5573.14280143310],
        [174.844, 3.01193636534, 18849.22754997420],
        [243.189, 4.27349536153, 11790.62908865880],
        [211.829, 5.84714540314, 1577.34354244780],
        [185.752, 5.02194447178, 10977.07880469900],
        [109.835, 5.05510636285, 5486.77784317500],
        [98.316, 0.88681311277, 6069.77675455340],
        [86.499, 5.68959778254, 15720.83878487840],
        [85.825, 1.27083733351, 161000.68573767410],
        [62.916, 0.92177108832, 529.69096509460],
        [57.056, 2.01374292014, 83996.84731811189],
        [64.903, 0.27250613787, 17260.15465469040],
        [49.384, 3.24501240359, 2544.31441988340],
        [55.736, 5.24159798933, 71430.69561812909],
        [42.515, 6.01110242003, 6275.96230299060],
        [46.963, 2.57805070386, 775.52261132400],
        [38.968, 5.36071738169, 4694.00295470760],
        [44.661, 5.53715807302, 9437.76293488700],
        [35.660, 1.67468058995, 12036.46073488820],
        [31.921, 0.18368229781, 5088.62883976680],
        [31.846, 1.77775642085, 398.14900340820],
        [33.193, 0.24370300098, 7084.89678111520],
        [38.245, 2.39255343974, 8827.39026987480],
        [28.464, 1.21344868176, 6286.59896834040],
        [37.490, 0.82952922332, 19651.04848109800],
        [36.957, 4.90107591914, 12139.55350910680],
        [34.537, 1.84270693282, 2942.46342329160],
        [26.275, 4.58896850401, 10447.38783960440],
        [24.596, 3.78660875483, 8429.24126646660],
        [23.587, 0.26866117066, 796.29800681640],
        [27.793, 1.89934330904, 6279.55273164240],
        [23.927, 4.99598548138, 5856.47765911540],
        [20.349, 4.65267995431, 2146.16541647520],
        [23.287, 2.80783650928, 14143.49524243060],
        [22.103, 1.95004702988, 3154.68708489560],
        [19.506, 5.38227371393, 2352.86615377180],
        [17.958, 0.19871379385, 6812.76681508600],
        [17.174, 4.43315560735, 10213.28554621100],
        [16.190, 5.23160507859, 17789.84561978500],
        [17.314, 6.15200787916, 16730.46368959580],
        [13.814, 5.18962074032, 8031.09226305840],
        [18.833, 0.67306674027, 149854.40013480789],
        [18.331, 2.25348733734, 23581.25817731760],
        [13.641, 3.68516118804, 4705.73230754360],
        [13.139, 0.65289581324, 13367.97263110660],
        [10.414, 4.33285688538, 11769.85369316640],
        [9.978, 4.20126336355, 6309.37416979120],
        [10.169, 1.59390681369, 4690.47983635860],
        [7.564, 2.62560597390, 6256.77753019160],
        [9.661, 3.67586791220, 27511.46787353720],
        [6.743, 0.56270332741, 3340.61242669980],
        [8.743, 6.06359123461, 1748.01641306700],
        [7.786, 3.67371235637, 12168.00269657460],
        [6.633, 5.66149277792, 11371.70468975820],
        [7.712, 0.31242577789, 7632.94325965020],
        [6.592, 3.13576266188, 801.82093112380],
        [7.460, 5.64757188143, 11926.25441366880],
        [6.933, 2.92384586400, 6681.22485339960],
        [6.802, 1.42329806420, 23013.53953958720],
        [6.115, 5.13393615454, 1194.44701022460],
        [6.477, 2.64986648492, 19804.82729158280],
        [5.233, 4.62434053374, 6438.49624942560],
        [6.147, 3.02863936662, 233141.31440436150],
        [4.608, 1.72194702724, 7234.79425624200],
        [4.221, 1.55697533729, 7238.67559160000],
        [5.314, 2.40716580847, 11499.65622279280],
        [5.128, 5.32398965690, 11513.88331679440],
        [4.770, 0.25554312006, 11856.21865142450],
        [5.519, 2.09089154502, 17298.18232732620],
        [5.625, 4.34052903053, 90955.55169449610],
        [4.578, 4.46569641570, 5746.27133789600],
        [3.788, 4.90729383510, 4164.31198961300],
        [5.337, 5.09957905104, 31441.67756975680],
        [3.967, 1.20054555174, 1349.86740965880],
        [4.008, 3.03007204392, 1059.38193018920],
        [3.476, 0.76080277030, 10973.55568635000],
        [4.232, 1.05485713117, 5760.49843189760],
        [4.582, 3.76570026763, 6386.16862421000],
        [3.335, 3.13829943354, 6836.64525283380],
        [3.418, 3.00072390334, 4292.33083295040],
        [3.598, 5.70718084323, 5643.17856367740],
        [3.237, 4.16448773994, 9917.69687450980],
        [4.154, 2.59941292162, 7058.59846131540],
        [3.362, 4.54577697964, 4732.03062734340],
        [2.978, 1.30561268820, 6283.14316029419],
        [2.765, 0.51311975679, 26.29831979980],
        [2.802, 5.66263240521, 8635.94200376320],
        [2.927, 5.73787481548, 16200.77272450120],
        [3.164, 1.69140262657, 11015.10647733480],
        [2.598, 2.96244118586, 25132.30339996560],
        [3.519, 3.62639325753, 244287.60000722768],
        [2.676, 4.20725700850, 18073.70493865020],
        [2.978, 1.74971565805, 6283.00853968860],
        [2.287, 1.06975704977, 14314.16811304980],
        [2.863, 5.92838131397, 14712.31711645800],
        [3.071, 0.23793217002, 35371.88726597640],
        [2.656, 0.89959301780, 12352.85260454480],
        [2.415, 2.79975176257, 709.93304855830],
        [2.814, 3.51488206882, 21228.39202354580],
        [1.977, 2.61358297550, 951.71840625060],
        [2.548, 2.47684686575, 6208.29425142410],
        [1.999, 0.56090388160, 7079.37385680780],
        [2.305, 1.05376461628, 22483.84857449259],
        [1.855, 2.86090681163, 5216.58037280140],
        [2.157, 1.31396741861, 154717.60988768269],
        [1.970, 4.36929875289, 167283.76158766550],
        [1.635, 5.85571606764, 10984.19235169980],
        [1.754, 2.14452408833, 6290.18939699220],
        [2.154, 6.03828341543, 10873.98603048040],
        [1.714, 3.70157691113, 1592.59601363280],
        [1.541, 6.21598380732, 23543.23050468179],
        [1.611, 1.99824499377, 10969.96525769820],
        [1.712, 1.34295663542, 3128.38876509580],
        [1.642, 5.55026665339, 6496.37494542940],
        [1.502, 5.43948825854, 155.42039943420],
        [1.827, 5.91227480261, 3738.76143010800],
        [1.726, 2.16764983583, 10575.40668294180],
        [1.532, 5.35683107070, 13521.75144159140],
        [1.829, 1.66006148731, 39302.09696219600],
        [1.605, 1.90928637633, 6133.51265285680],
        [1.282, 2.46014880418, 13916.01910964160],
        [1.211, 4.41360631550, 3894.18182954220],
        [1.394, 1.77801929354, 9225.53927328300],
        [1.571, 4.95512957592, 25158.60171976540],
        [1.205, 1.19212540615, 3.52311834900],
        [1.132, 2.69830084955, 6040.34724601740],
        [1.504, 5.77002730341, 18209.33026366019],
        [1.393, 1.62621805428, 5120.60114558360],
        [1.077, 2.93931554233, 17256.63153634140],
        [1.232, 0.71655165307, 143571.32428481648],
        [1.087, 0.99769687939, 955.59974160860],
        [1.068, 5.28472576231, 65147.61976813770],
        [0.980, 5.10949204607, 6172.86952877200],
        [1.169, 3.11664290862, 14945.31617355440],
        [1.202, 4.02992510402, 553.56940284240],
        [0.979, 2.00000879212, 15110.46611986620],
        [0.962, 4.02380771400, 6282.09552892320],
        [0.999, 3.62643002790, 6262.30045449900],
        [1.030, 5.84989900289, 213.29909543800],
        [1.014, 2.84221578218, 8662.24032356300],
        [1.185, 1.51330541132, 17654.78053974960],
        [0.967, 2.67081017562, 5650.29211067820],
        [1.222, 2.65423784904, 88860.05707098669],
        [0.981, 2.36370360283, 6206.80977871580],
        [1.033, 0.13874927606, 11712.95531823080],
        [1.103, 3.08477302937, 43232.30665841560],
        [0.781, 2.53372735932, 16496.36139620240],
        [1.019, 3.04569392376, 6037.24420376200],
        [0.795, 5.80662989111, 5230.80746680300],
        [0.813, 3.57710279439, 10177.25767953360],
        [0.962, 5.31470594766, 6284.05617105960],
        [0.721, 5.96264301567, 12559.03815298200],
        [0.966, 2.74714939953, 6244.94281435360],
        [0.921, 0.10155275926, 29088.81141598500],
        [0.692, 3.89764447548, 1589.07289528380],
        [0.719, 5.91791450402, 4136.91043351620],
        [0.772, 4.05505682353, 6127.65545055720],
        [0.712, 5.49291532439, 22003.91463486980],
        [0.672, 1.60700490811, 11087.28512591840],
        [0.690, 4.50539825563, 426.59819087600],
        [0.854, 3.26104981596, 20426.57109242200],
        [0.656, 4.32410182940, 16858.48253293320],
        [0.840, 2.59572585222, 28766.92442448400],
        [0.692, 0.61650089011, 11403.67699557500],
        [0.700, 3.40901167143, 7.11354700080],
        [0.726, 0.04243053594, 5481.25491886760],
        [0.557, 4.78317696534, 20199.09495963300],
        [0.649, 1.04027912958, 6062.66320755260],
        [0.633, 5.70229959167, 45892.73043315699],
        [0.592, 6.11836729658, 9623.68827669120],
        [0.523, 3.62840021266, 5333.90024102160],
        [0.604, 5.57734696185, 10344.29506538580],
        [0.496, 2.21023499449, 1990.74501704100],
        [0.691, 1.96071732602, 12416.58850284820],
        [0.640, 1.59074172032, 18319.53658487960],
        [0.625, 3.82362791378, 13517.87010623340],
        [0.663, 5.08444996779, 283.85931886520],
        [0.475, 1.17025894287, 12569.67481833180],
        [0.664, 4.50029469969, 47162.51635463520],
        [0.569, 0.16310365162, 17267.26820169119],
        [0.568, 3.86100969474, 6076.89030155420],
        [0.539, 4.83282276086, 18422.62935909819],
        [0.466, 0.75872342878, 7342.45778018060],
        [0.541, 3.07212190507, 226858.23855437007],
        [0.458, 0.26774483096, 4590.91018048900],
        [0.610, 1.53597051291, 33019.02111220460],
        [0.617, 2.62356328726, 11190.37790013700],
        [0.548, 4.55798855791, 18875.52586977400],
        [0.633, 4.60110281228, 66567.48586525429],
        [0.596, 5.78202396722, 632.78373931320],
        [0.533, 5.01786882904, 12132.43996210600],
        [0.603, 5.38458554802, 316428.22867391503],
        [0.469, 0.59168241917, 21954.15760939799],
        [0.548, 3.50613163558, 17253.04110768959],
        [0.502, 0.98804327589, 11609.86254401220],
        [0.568, 1.98497313089, 7668.63742494250],
        [0.482, 1.62141803864, 12146.66705610760],
        [0.391, 3.68718382989, 18052.92954315780],
        [0.457, 3.77205737340, 156137.47598479927],
        [0.401, 5.28260651958, 15671.08175940660],
        [0.469, 1.80963184268, 12562.62858163380],
        [0.508, 3.36399024699, 20597.24396304120],
        [0.450, 5.66054299250, 10454.50138660520],
        [0.375, 4.98534633105, 9779.10867612540],
        [0.523, 0.97215560834, 155427.54293624099],
        [0.403, 5.13939866506, 1551.04522264800],
        [0.372, 3.69883738807, 9388.00590941520],
        [0.367, 4.43875659716, 4535.05943692440],
        [0.406, 4.20863156600, 12592.45001978260],
        [0.360, 2.53924644657, 242.72860397400],
        [0.471, 4.61907324819, 5436.99301524020],
        [0.441, 5.83872966262, 3496.03282613400],
        [0.385, 4.94496680973, 24356.78078864160],
        [0.349, 6.15018231784, 19800.94595622480],
        [0.355, 0.21895678106, 5429.87946823940],
        [0.344, 5.62993724928, 2379.16447357160],
        [0.380, 2.72105213143, 11933.36796066960],
        [0.432, 0.24221790536, 17996.03116822220],
        [0.378, 5.22517556974, 7477.52286021600],
        [0.337, 5.10888041439, 5849.36411211460],
        [0.315, 0.57827745123, 10557.59416082380],
        [0.318, 4.49953141399, 3634.62102451840],
        [0.323, 1.54274281393, 10440.27429260360],
        [0.309, 5.76839284397, 20.77539549240],
        [0.301, 2.34727604008, 4686.88940770680],
        [0.414, 5.93237602310, 51092.72605085480],
        [0.361, 2.16398609550, 28237.23345938940],
        [0.288, 0.18376252189, 13095.84266507740],
        [0.277, 5.12952205045, 13119.72110282519],
        [0.327, 6.19222146204, 6268.84875598980],
        [0.273, 0.30522428863, 23141.55838292460],
        [0.267, 5.76152585786, 5966.68398033480],
        [0.308, 5.99280509979, 22805.73556599360],
        [0.345, 2.92489919444, 36949.23080842420],
        [0.253, 5.20995219509, 24072.92146977640],
        [0.342, 5.72702586209, 16460.33352952499],
        [0.261, 2.00304796059, 6148.01076995600],
        [0.238, 5.08264392839, 6915.85958930460],
        [0.249, 2.94762789744, 135.06508003540],
        [0.306, 3.89764686987, 10988.80815753500],
        [0.305, 0.05827812117, 4701.11650170840],
        [0.319, 2.95712862064, 163096.18036118348],
        [0.209, 4.43768461442, 6546.15977336420],
        [0.270, 2.06643178717, 4804.20927592700],
        [0.217, 0.73691592312, 6303.85124548380],
        [0.206, 0.32075959415, 25934.12433108940],
        [0.218, 0.18428135264, 28286.99048486120],
        [0.205, 5.21312087405, 20995.39296644940],
        [0.199, 0.44384292491, 16737.57723659660],
        [0.230, 6.06567392849, 6287.00800325450],
        [0.219, 1.29194216300, 5326.78669402080],
        [0.201, 1.74700937253, 22743.40937951640],
        [0.207, 4.45440927276, 6279.48542133960],
        [0.269, 6.05640445030, 64471.99124174489],
        [0.190, 0.99256176518, 29296.61538957860],
        [0.238, 5.42471431221, 39609.65458316560],
        [0.262, 5.26961924198, 522.57741809380],
        [0.210, 4.68618183158, 6254.62666252360],
        [0.197, 2.80624554080, 4933.20844033260],
        [0.252, 4.36220154608, 40879.44050464380],
        [0.261, 1.07241516738, 55022.93574707440],
        [0.189, 3.82966734476, 419.48464387520],
        [0.185, 4.14324541379, 5642.19824260920],
        [0.247, 3.44855612987, 6702.56049386660],
        [0.205, 4.04424043223, 536.80451209540],
        [0.191, 3.14082686083, 16723.35014259500],
        [0.222, 5.16263907319, 23539.70738633280],
        [0.180, 4.56214752149, 6489.26139842860],
        [0.219, 0.80382553358, 16627.37091537720],
        [0.227, 0.60156339452, 5905.70224207560],
        [0.168, 0.88753528161, 16062.18452611680],
        [0.158, 0.92127725775, 23937.85638974100],
        [0.157, 4.69607868164, 6805.65326808520],
        [0.207, 4.88410451334, 6286.66627864320],
        [0.160, 4.95943826846, 10021.83728009940],
        [0.166, 0.97126433565, 3097.88382272579],
        [0.209, 5.75663411805, 3646.35037735440],
        [0.175, 6.12762824412, 239424.39025435288],
        [0.173, 3.13887234973, 6179.98307577280],
        [0.157, 3.62822058179, 18451.07854656599],
        [0.157, 4.67695912235, 6709.67404086740],
        [0.146, 3.09506069735, 4907.30205014560],
        [0.165, 2.27139128760, 10660.68693504240],
        [0.201, 1.67701267433, 2107.03450754240],
        [0.144, 3.96947747592, 6019.99192661860],
        [0.171, 5.91302216729, 6058.73105428950],
        [0.144, 2.13155655120, 26084.02180621620],
        [0.151, 0.67417383554, 2388.89402044920],
        [0.189, 5.07122281033, 263.08392337280],
        [0.146, 5.10373877968, 10770.89325626180],
        [0.187, 1.23915444627, 19402.79695281660],
        [0.174, 0.08407293391, 9380.95967271720],
        [0.137, 1.26247412309, 12566.21901028560],
        [0.137, 3.52826010842, 639.89728631400],
        [0.148, 1.76124372592, 5888.44996493220],
        [0.164, 2.39195095081, 6357.85744855870],
        [0.146, 2.43675816553, 5881.40372823420],
        [0.161, 1.15721259372, 26735.94526221320],
        [0.131, 2.51859277344, 6599.46771964800],
        [0.153, 5.85203687779, 6281.59137728310],
        [0.151, 3.72338532649, 12669.24447420140],
        [0.132, 2.38417741883, 6525.80445396540],
        [0.129, 0.75556744143, 5017.50837136500],
        [0.127, 0.00254936441, 10027.90319572920],
        [0.148, 2.85102145528, 6418.14093002680],
        [0.143, 5.74460279367, 26087.90314157420],
        [0.172, 0.41289962240, 174242.46596404971],
        [0.136, 4.15497742275, 6311.52503745920],
        [0.170, 5.98194913129, 327574.51427678124],
        [0.124, 1.65497607604, 32217.20018108080],
        [0.136, 2.48430783417, 13341.67431130680],
        [0.165, 2.49667924600, 58953.14544329400],
        [0.123, 3.45660563754, 6277.55292568400],
        [0.117, 0.86065134175, 6245.04817735560],
        [0.149, 5.61358280963, 5729.50644714900],
        [0.153, 0.26860029950, 245.83164622940],
        [0.128, 0.71204006588, 103.09277421860],
        [0.159, 2.43166592149, 221995.02880149524],
        [0.130, 2.80707316718, 6016.46880826960],
        [0.137, 1.70657709294, 12566.08438968000],
        [0.111, 1.56305648432, 17782.73207278420],
        [0.113, 3.58302904101, 25685.87280280800],
        [0.109, 3.26403795962, 6819.88036208680],
        [0.122, 0.34120688217, 1162.47470440780],
        [0.119, 5.84644718278, 12721.57209941700],
        [0.144, 2.28899679126, 12489.88562870720],
        [0.137, 5.82029768354, 44809.65020086340],
        [0.107, 2.42818544140, 5547.19933645960],
        [0.134, 1.26539982939, 5331.35744374080],
        [0.103, 5.96518130595, 6321.10352262720],
        [0.109, 0.33808549034, 11300.58422135640],
        [0.129, 5.89187277327, 12029.34718788740],
        [0.122, 5.77325634636, 11919.14086666800],
        [0.107, 6.24998989350, 77690.75950573849],
        [0.107, 1.00535580713, 77736.78343050249],
        [0.143, 0.24122178432, 4214.06901508480],
        [0.143, 0.88529649733, 7576.56007357400],
        [0.107, 2.92124030496, 31415.37924995700],
        [0.099, 5.70862227072, 5540.08578945880],
        [0.110, 0.37528037383, 5863.59120611620],
        [0.104, 4.44107178366, 2118.76386037840],
        [0.098, 5.95877916706, 4061.21921539440],
        [0.113, 1.24206857385, 84672.47584450469],
        [0.124, 2.55619029867, 12539.85338018300],
        [0.110, 3.66952094329, 238004.52415723630],
        [0.112, 4.32512422943, 97238.62754448749],
        [0.097, 3.70151541181, 11720.06886523160],
        [0.120, 1.26895630252, 12043.57428188900],
        [0.094, 2.56461130309, 19004.64794940840],
        [0.117, 3.65425622684, 34520.30930938080],
        [0.098, 0.13589994287, 11080.17157891760],
        [0.097, 5.38330115253, 7834.12107263940],
        [0.097, 2.46722096722, 71980.63357473118],
        [0.095, 5.36958330451, 6288.59877429880],
        [0.111, 5.01961920313, 11823.16163945020],
        [0.090, 2.72299804525, 26880.31981303260],
        [0.099, 0.90164266377, 18635.92845453620],
        [0.126, 4.78722177847, 305281.94307104882],
        [0.093, 0.21240380046, 18139.29450141590],
        [0.124, 5.00979495566, 172146.97134054030],
        [0.099, 5.67090026475, 16522.65971600220],
        [0.092, 2.28180963676, 12491.37010141550],
        [0.090, 4.50544881196, 40077.61957352000],
        [0.100, 2.00639461612, 12323.42309600880],
        [0.095, 5.68801979087, 14919.01785375460],
        [0.087, 1.86043406047, 27707.54249429480],
        [0.105, 3.02903468417, 22345.26037610820],
        [0.087, 5.43970168638, 6272.03014972750],
        [0.089, 1.63389387182, 33326.57873317420],
        [0.082, 5.58298993353, 10241.20229116720],
        [0.094, 5.47749711149, 9924.81042151060],
        [0.082, 4.71988314145, 15141.39079431200],
        [0.097, 5.61458778738, 2787.04302385740],
        [0.096, 3.89073946348, 6379.05507720920],
        [0.081, 3.13038482444, 36147.40987730040],
        [0.110, 4.89978492291, 72140.62866668739],
        [0.097, 5.20764563059, 6303.43116939020],
        [0.082, 5.26342716139, 9814.60410029120],
        [0.109, 2.35555589770, 83286.91426955358],
        [0.097, 2.58492958057, 30666.15495843280],
        [0.093, 1.32651591333, 23020.65308658799],
        [0.078, 3.99588630754, 11293.47067435560],
        [0.090, 0.57771932738, 26482.17080962440],
        [0.106, 3.92012705073, 62883.35513951360],
        [0.098, 2.94397773524, 316.39186965660],
        [0.076, 3.96310417608, 29026.48522950779],
        [0.078, 1.97068529306, 90279.92316810328],
        [0.076, 0.23027966596, 21424.46664430340],
        [0.080, 2.23099742212, 266.60704172180],
        [0.079, 1.46227790922, 8982.81066930900],
        [0.102, 4.92129953565, 5621.84292321040],
        [0.100, 0.39243148321, 24279.10701821359],
        [0.071, 1.52014858474, 33794.54372352860],
        [0.076, 0.22880641443, 57375.80190084620],
        [0.091, 0.96515913904, 48739.85989708300],
        [0.075, 2.77638585157, 12964.30070339100],
        [0.077, 5.18846946344, 11520.99686379520],
        [0.068, 0.50006599129, 4274.51831083240],
        [0.075, 2.07323762803, 15664.03552270859],
        [0.074, 1.01884134928, 6393.28217121080],
        [0.077, 0.46665178780, 16207.88627150200],
        [0.081, 4.10452219483, 161710.61878623240],
        [0.067, 3.83840630887, 6262.72053059260],
        [0.071, 3.91415523291, 7875.67186362420],
        [0.081, 0.91938383237, 74.78159856730],
        [0.083, 4.69916218791, 23006.42599258639],
        [0.063, 2.32556465878, 6279.19451463340],
        [0.065, 5.41938745446, 28628.33622609960],
        [0.065, 3.02336771694, 5959.57043333400],
        [0.064, 3.31033198370, 2636.72547263700],
        [0.064, 0.18375587519, 1066.49547719000],
        [0.080, 5.81239171612, 12341.80690428090],
        [0.066, 2.15105504851, 38.02767263580],
        [0.062, 2.43313614978, 10138.10951694860],
        [0.060, 3.16153906470, 5490.30096152400],
        [0.069, 0.30764736334, 7018.95236352320],
        [0.068, 2.24442548639, 24383.07910844140],
        [0.078, 1.39649386463, 9411.46461508720],
        [0.063, 0.72976362625, 6286.95718534940],
        [0.073, 4.95125917731, 6453.74872061060],
        [0.078, 0.32736023459, 6528.90749622080],
        [0.059, 4.95362151577, 35707.71008290740],
        [0.070, 2.37962727525, 15508.61512327440],
        [0.073, 1.35229143111, 5327.47610838280],
        [0.072, 5.91833527334, 10881.09957748120],
        [0.059, 5.36231868425, 10239.58386601080],
        [0.059, 1.63156134967, 61306.01159706580],
        [0.054, 4.29491690425, 21947.11137270000],
        [0.057, 5.89190132575, 34513.26307268280],
        [0.074, 1.38235845304, 9967.45389998160],
        [0.053, 3.86543309344, 32370.97899156560],
        [0.055, 4.51794544854, 34911.41207609100],
        [0.063, 5.41479412056, 11502.83761653050],
        [0.063, 2.34416220742, 11510.70192305670],
        [0.068, 0.77493931112, 29864.33402730900],
        [0.060, 5.57024703495, 5756.90800324580],
        [0.072, 2.80863088166, 10866.87248347960],
        [0.061, 2.69736991384, 82576.98122099529],
        [0.063, 5.32068807257, 3116.65941225980],
        [0.052, 1.02278758099, 6272.43918464160],
        [0.069, 5.00698550308, 25287.72379939980],
        [0.066, 6.12047940728, 12074.48840752400],
        [0.051, 2.59519527563, 11396.56344857420],
        [0.056, 2.57995973521, 17892.93839400359],
        [0.059, 0.44167237620, 250570.67585721909],
        [0.059, 3.84070143543, 5483.25472482600],
        [0.049, 0.54704693048, 22594.05489571199],
        [0.065, 2.38423614501, 52670.06959330260],
        [0.069, 5.34363738671, 66813.56483573320],
        [0.057, 5.42770501007, 310145.15282392362],
        [0.053, 1.17760296075, 149.56319713460],
        [0.061, 4.02090887211, 34596.36465465240],
        [0.049, 4.18361320516, 18606.49894600020],
        [0.055, 0.83886167974, 20452.86941222180],
        [0.050, 1.46327331958, 37455.72649597440],
        [0.048, 4.53854727167, 29822.78323632420],
        [0.058, 3.34847975377, 33990.61834428620],
        [0.065, 1.45522693982, 76251.32777062019],
        [0.056, 2.35650663692, 37724.75341974820],
        [0.052, 2.61551081496, 5999.21653112620],
        [0.053, 0.17334326094, 77717.29458646949],
        [0.053, 0.79879700631, 77710.24834977149],
        [0.047, 0.43240779709, 735.87651353180],
        [0.053, 4.58763261686, 11616.97609101300],
        [0.048, 6.20230111054, 4171.42553661380],
        [0.052, 1.09723616404, 640.87760738220],
        [0.057, 3.42008310383, 50317.20343953080],
        [0.053, 1.01528448581, 149144.46708624958],
        [0.047, 3.00924906195, 52175.80628314840],
        [0.052, 2.03254070404, 6293.71251534120],
        [0.048, 0.12356889734, 13362.44970679920],
        [0.045, 3.37963782356, 10763.77970926100],
        [0.047, 5.50981287869, 12779.45079542080],
        [0.062, 5.45209070099, 949.17560896980],
        [0.061, 2.93237974631, 5791.41255753260],
        [0.044, 2.87440620802, 8584.66166590080],
        [0.046, 4.03141796560, 10667.80048204320],
        [0.047, 3.89902931422, 3903.91137641980],
        [0.046, 2.75700467329, 6993.00889854970],
        [0.045, 1.93386293300, 206.18554843720],
        [0.047, 2.57670800912, 11492.54267579200],
        [0.044, 3.62570223167, 63658.87775083760],
        [0.051, 0.84536826273, 12345.73905754400],
        [0.043, 0.01524970172, 37853.87549938260],
        [0.041, 3.27146326065, 8858.31494432060],
        [0.045, 3.03765521215, 65236.22129328540],
        [0.047, 1.44447548944, 21393.54196985760],
        [0.058, 5.45843180927, 1975.49254585600],
        [0.050, 2.13285524146, 12573.26524698360],
        [0.041, 1.32190847146, 2547.83753823240],
        [0.047, 3.67579608544, 28313.28880466100],
        [0.041, 2.24013475126, 8273.82086703240],
        [0.047, 6.21438985953, 10991.30589870060],
        [0.042, 3.01631817350, 853.19638175200],
        [0.056, 1.09773690181, 77376.20102240759],
        [0.040, 2.35698541041, 2699.73481931760],
        [0.043, 5.28030898459, 17796.95916678580],
        [0.054, 2.59175932091, 22910.44676536859],
        [0.054, 0.88027764102, 71960.38658322369],
        [0.055, 0.07988899477, 83467.15635301729],
        [0.039, 1.12867321442, 9910.58332750900],
        [0.040, 1.35670430524, 27177.85152920020],
        [0.039, 4.39624220245, 5618.31980486140],
        [0.042, 4.78798367468, 7856.89627409019],
        [0.047, 2.75482175292, 18202.21671665939],
        [0.039, 1.97008298629, 24491.42579258340],
        [0.042, 4.04346599946, 7863.94251078820],
        [0.038, 0.49178679251, 38650.17350619900],
        [0.036, 4.86047906533, 4157.19844261220],
        [0.043, 5.64354880978, 1062.90504853820],
        [0.036, 3.98066313627, 12565.17137891460],
        [0.042, 2.30753932657, 6549.68289171320],
        [0.040, 5.39694918320, 9498.21223063460],
        [0.040, 3.30603243754, 23536.11695768099],
        [0.050, 6.15760345261, 78051.34191383338],
    ],
    # R1
    [
        [103018.608, 1.10748969588, 6283.07584999140],
        [1721.238, 1.06442301418, 12566.15169998280],
        [702.215, 3.14159265359, 0.00000000000],
        [32.346, 1.02169059149, 18849.22754997420],
        [30.799, 2.84353804832, 5507.55323866740],
        [24.971, 1.31906709482, 5223.69391980220],
        [18.485, 1.42429748614, 1577.34354244780],
        [10.078, 5.91378194648, 10977.07880469900],
        [8.634, 0.27146150602, 5486.77784317500],
        [8.654, 1.42046854427, 6275.96230299060],
        [5.069, 1.68613426734, 5088.62883976680],
        [4.985, 6.01401770704, 6286.59896834040],
        [4.669, 5.98724494073, 529.69096509460],
        [4.395, 0.51800238019, 4694.00295470760],
        [3.872, 4.74969833437, 2544.31441988340],
        [3.750, 5.07097685568, 796.29800681640],
        [4.100, 1.08424786092, 9437.76293488700],
        [3.518, 0.02290216272, 83996.84731811189],
        [3.436, 0.94937019624, 71430.69561812909],
        [3.221, 6.15628775313, 2146.16541647520],
        [3.414, 5.41218322538, 775.52261132400],
        [2.863, 5.48432847146, 10447.38783960440],
        [2.520, 0.24276941146, 398.14900340820],
        [2.201, 4.95216196651, 6812.76681508600],
        [2.186, 0.41991743105, 8031.09226305840],
        [2.838, 3.42034351366, 2352.86615377180],
        [2.554, 6.13241878525, 6438.49624942560],
        [1.932, 5.31374608366, 8429.24126646660],
        [2.429, 3.09164528262, 4690.47983635860],
        [1.730, 1.53686208550, 4705.73230754360],
        [2.250, 3.68863633842, 7084.89678111520],
        [2.093, 1.28191783032, 1748.01641306700],
        [1.441, 0.81656250862, 14143.49524243060],
        [1.483, 3.22225357771, 7234.79425624200],
        [1.754, 3.22883705112, 6279.55273164240],
        [1.583, 4.09702349428, 11499.65622279280],
        [1.575, 5.53890170575, 3154.68708489560],
        [1.847, 1.82040335363, 7632.94325965020],
        [1.504, 3.63293385726, 11513.88331679440],
        [1.337, 4.64440864339, 6836.64525283380],
        [1.275, 2.69341415363, 1349.86740965880],
        [1.352, 6.15101580257, 5746.27133789600],
        [1.125, 3.35673439497, 17789.84561978500],
        [1.470, 3.65282991755, 1194.44701022460],
        [1.177, 2.57676109092, 13367.97263110660],
        [1.101, 4.49748696552, 4292.33083295040],
        [1.234, 5.65036509521, 5760.49843189760],
        [0.984, 0.65517395136, 5856.47765911540],
        [0.928, 2.32420318751, 10213.28554621100],
        [1.077, 5.82812169132, 12036.46073488820],
        [0.916, 0.76613009583, 16730.46368959580],
        [0.877, 1.50137505051, 11926.25441366880],
        [1.023, 5.62076589825, 6256.77753019160],
        [0.851, 0.65709335533, 155.42039943420],
        [0.802, 4.10519132088, 951.71840625060],
        [0.857, 1.41661697538, 5753.38488489680],
        [0.994, 1.14418521187, 1059.38193018920],
        [0.813, 1.63948433322, 6681.22485339960],
        [0.662, 4.55200452260, 5216.58037280140],
        [0.644, 4.19478168733, 6040.34724601740],
        [0.626, 1.50767713598, 5643.17856367740],
        [0.590, 6.18277145205, 4164.31198961300],
        [0.635, 0.52413263542, 6290.18939699220],
        [0.650, 0.97935690350, 25132.30339996560],
        [0.568, 2.30125315873, 10973.55568635000],
        [0.547, 5.27256412213, 3340.61242669980],
        [0.547, 2.20144422886, 1592.59601363280],
        [0.526, 0.92464258226, 11371.70468975820],
        [0.490, 5.90951388655, 3894.18182954220],
        [0.478, 1.66857963179, 12168.00269657460],
        [0.516, 3.59803483887, 10969.96525769820],
        [0.518, 3.97914412373, 17298.18232732620],
        [0.534, 5.03740926442, 9917.69687450980],
        [0.487, 2.50545369269, 6127.65545055720],
        [0.416, 4.04828175503, 10984.19235169980],
        [0.538, 5.54081539805, 553.56940284240],
        [0.402, 2.16544019233, 7860.41939243920],
        [0.553, 2.32177369366, 11506.76976979360],
        [0.367, 3.39152532250, 6496.37494542940],
        [0.360, 5.34379853282, 7079.37385680780],
        [0.337, 3.61563704045, 11790.62908865880],
        [0.456, 0.30754294809, 801.82093112380],
        [0.417, 3.70009308674, 10575.40668294180],
        [0.381, 5.82033971802, 7058.59846131540],
        [0.321, 0.31988767355, 16200.77272450120],
        [0.364, 1.08414306177, 6309.37416979120],
        [0.294, 4.54798604957, 11856.21865142450],
        [0.290, 1.26473978562, 8635.94200376320],
        [0.399, 4.16998866302, 26.29831979980],
        [0.262, 5.08316906342, 10177.25767953360],
        [0.243, 2.25746091190, 11712.95531823080],
        [0.237, 1.05070575346, 242.72860397400],
        [0.275, 3.45319481756, 5884.92684658320],
        [0.255, 5.38496831087, 21228.39202354580],
        [0.307, 4.24313526604, 3738.76143010800],
        [0.216, 3.46037894728, 213.29909543800],
        [0.196, 0.69029243914, 1990.74501704100],
        [0.198, 5.16301829964, 12352.85260454480],
        [0.214, 3.91876200279, 13916.01910964160],
        [0.212, 4.00861198517, 5230.80746680300],
        [0.184, 5.59805976614, 6283.14316029419],
        [0.184, 2.85275392124, 7238.67559160000],
        [0.179, 2.54259058334, 14314.16811304980],
        [0.225, 1.64458698399, 4732.03062734340],
        [0.236, 5.58826125715, 6069.77675455340],
        [0.187, 2.72805985443, 6062.66320755260],
        [0.184, 6.04216273598, 6283.00853968860],
        [0.230, 3.62591335086, 6284.05617105960],
        [0.163, 2.19117396803, 18073.70493865020],
        [0.172, 0.97612950740, 3930.20969621960],
        [0.215, 1.04672844028, 3496.03282613400],
        [0.169, 4.75084479006, 17267.26820169119],
        [0.152, 0.19390712179, 9779.10867612540],
        [0.182, 5.16288118255, 17253.04110768959],
        [0.149, 0.80944184260, 709.93304855830],
        [0.163, 2.19209570390, 6076.89030155420],
        [0.186, 5.01159497089, 11015.10647733480],
        [0.134, 0.97765485759, 65147.61976813770],
        [0.141, 4.38421981312, 4136.91043351620],
        [0.158, 4.60974280627, 9623.68827669120],
        [0.133, 3.30508592837, 154717.60988768269],
        [0.163, 6.11782626245, 3.52311834900],
        [0.174, 1.58078542187, 7.11354700080],
        [0.141, 0.49976927274, 25158.60171976540],
        [0.124, 6.03440460031, 9225.53927328300],
        [0.150, 5.30166336812, 13517.87010623340],
        [0.127, 1.92389511438, 22483.84857449259],
        [0.121, 2.37813129011, 167283.76158766550],
        [0.120, 3.98423684853, 4686.88940770680],
        [0.117, 5.81072642211, 12569.67481833180],
        [0.122, 5.60973054224, 5642.19824260920],
        [0.157, 3.40236426002, 16496.36139620240],
        [0.129, 2.10705116371, 1589.07289528380],
        [0.116, 0.55839966736, 5849.36411211460],
        [0.123, 1.52961392771, 12559.03815298200],
        [0.111, 0.44848279675, 6172.86952877200],
        [0.123, 5.81645568991, 6282.09552892320],
        [0.150, 4.26278409223, 3128.38876509580],
        [0.106, 2.27437761356, 5429.87946823940],
        [0.104, 4.42743707728, 23543.23050468179],
        [0.121, 0.39459045915, 12132.43996210600],
        [0.104, 2.41842602527, 426.59819087600],
        [0.110, 5.80381480447, 16858.48253293320],
        [0.100, 2.93805577485, 4535.05943692440],
        [0.097, 3.97935904984, 6133.51265285680],
        [0.110, 6.22339014386, 12146.66705610760],
        [0.098, 0.87576563709, 6525.80445396540],
        [0.098, 3.15248421301, 10440.27429260360],
        [0.095, 2.46168411100, 3097.88382272579],
        [0.088, 0.23371480284, 13119.72110282519],
        [0.098, 5.77016493489, 7342.45778018060],
        [0.092, 6.03915555063, 20426.57109242200],
        [0.096, 5.56909292561, 2388.89402044920],
        [0.081, 1.32131147691, 5650.29211067820],
        [0.086, 3.94529200528, 10454.50138660520],
        [0.076, 2.70729716925, 143571.32428481648],
        [0.091, 5.64100034152, 8827.39026987480],
        [0.076, 1.80783856698, 28286.99048486120],
        [0.081, 1.90858992196, 29088.81141598500],
        [0.075, 3.40955892978, 5481.25491886760],
        [0.069, 4.49936170873, 17256.63153634140],
        [0.088, 1.10098454357, 11769.85369316640],
        [0.066, 2.78285801977, 536.80451209540],
        [0.068, 3.88179770758, 17260.15465469040],
        [0.084, 1.59303306354, 9380.95967271720],
        [0.088, 3.88076636762, 7477.52286021600],
        [0.061, 6.17558202197, 11087.28512591840],
        [0.060, 4.34824715818, 6206.80977871580],
        [0.082, 4.59843208943, 9388.00590941520],
        [0.079, 1.63131230601, 4933.20844033260],
        [0.078, 4.20905757484, 5729.50644714900],
        [0.057, 5.48157926651, 18319.53658487960],
        [0.060, 1.01261781084, 12721.57209941700],
        [0.056, 1.63031935692, 15720.83878487840],
        [0.055, 0.24926735018, 15110.46611986620],
        [0.061, 5.93059279661, 12539.85338018300],
        [0.055, 4.84298966314, 13095.84266507740],
        [0.067, 6.11690589247, 8662.24032356300],
        [0.054, 5.73750638571, 3634.62102451840],
        [0.074, 1.05466745829, 16460.33352952499],
        [0.053, 2.29084335688, 16062.18452611680],
        [0.064, 2.13513767927, 7875.67186362420],
        [0.067, 0.07096807518, 14945.31617355440],
        [0.051, 2.31511194429, 6262.72053059260],
        [0.057, 5.77055471237, 12043.57428188900],
        [0.056, 4.41980790431, 4701.11650170840],
        [0.059, 5.87963500073, 5331.35744374080],
        [0.058, 2.30546168628, 955.59974160860],
        [0.049, 1.93839278478, 5333.90024102160],
        [0.048, 2.69973662261, 6709.67404086740],
        [0.064, 1.64379897981, 6262.30045449900],
        [0.046, 3.98449608961, 98068.53671630539],
        [0.050, 3.68875893005, 12323.42309600880],
        [0.045, 3.30068569697, 22003.91463486980],
        [0.047, 1.26317154881, 11919.14086666800],
        [0.045, 0.89150445122, 51868.24866217880],
        [0.043, 1.61526242998, 6277.55292568400],
        [0.043, 5.74295325645, 11403.67699557500],
        [0.044, 3.43070646822, 10021.83728009940],
        [0.056, 0.02481833774, 15671.08175940660],
        [0.055, 3.14274403422, 33019.02111220460],
        [0.045, 3.00877289177, 8982.81066930900],
        [0.046, 0.73303568429, 6303.43116939020],
        [0.049, 1.60455690285, 6303.85124548380],
        [0.045, 0.40210030323, 6805.65326808520],
        [0.053, 0.94869680175, 10988.80815753500],
        [0.041, 1.61122384329, 6819.88036208680],
        [0.055, 0.89439119424, 11933.36796066960],
        [0.045, 3.88495384656, 60530.48898574180],
        [0.040, 4.75740908001, 38526.57435087200],
        [0.040, 1.49921251887, 18451.07854656599],
        [0.040, 3.77498297228, 26087.90314157420],
        [0.051, 1.70258603562, 1551.04522264800],
        [0.039, 2.97100699926, 2118.76386037840],
        [0.053, 5.19854123078, 77713.77146812050],
        [0.047, 4.26356628717, 21424.46664430340],
        [0.037, 0.62902722802, 24356.78078864160],
        [0.036, 0.11087914947, 10344.29506538580],
        [0.036, 0.77037556319, 12029.34718788740],
        [0.035, 3.30933994515, 24072.92146977640],
        [0.035, 5.93650887012, 31570.79964939120],
        [0.036, 2.15108874765, 30774.50164257480],
        [0.036, 1.75078825382, 16207.88627150200],
        [0.033, 5.06264177921, 226858.23855437007],
        [0.034, 6.16891378800, 24491.42579258340],
        [0.035, 3.19120695549, 32217.20018108080],
        [0.034, 2.31528650443, 55798.45835839840],
        [0.032, 4.21446357042, 15664.03552270859],
        [0.039, 1.24979117796, 6418.14093002680],
        [0.037, 4.11943655770, 2787.04302385740],
        [0.032, 1.62887710890, 639.89728631400],
        [0.038, 5.89832942685, 640.87760738220],
        [0.032, 1.72442327688, 27433.88921587499],
        [0.031, 2.78828943753, 12139.55350910680],
        [0.035, 4.44608896525, 18202.21671665939],
        [0.034, 3.96287980676, 18216.44381066100],
        [0.033, 4.73611335874, 16723.35014259500],
        [0.034, 1.43910280005, 49515.38250840700],
        [0.031, 0.23302920161, 23581.25817731760],
        [0.029, 2.02633840220, 11609.86254401220],
        [0.030, 2.54923230240, 9924.81042151060],
        [0.032, 4.91793198558, 11300.58422135640],
        [0.028, 0.26187189577, 13521.75144159140],
        [0.028, 3.84568936822, 2699.73481931760],
        [0.029, 1.83149729794, 29822.78323632420],
        [0.033, 4.60320094415, 19004.64794940840],
        [0.027, 4.46183450287, 6702.56049386660],
        [0.030, 4.46494072240, 36147.40987730040],
        [0.027, 0.03211931363, 6279.78949257360],
        [0.026, 5.46497324333, 6245.04817735560],
        [0.035, 4.52695674113, 36949.23080842420],
        [0.027, 3.52528177609, 10770.89325626180],
        [0.026, 1.48499438453, 11080.17157891760],
        [0.035, 2.82154380962, 19402.79695281660],
        [0.025, 2.46339998836, 6279.48542133960],
        [0.026, 4.97688894643, 16737.57723659660],
        [0.026, 2.36136541526, 17996.03116822220],
        [0.029, 4.15148654061, 45892.73043315699],
        [0.026, 4.50714272714, 17796.95916678580],
        [0.027, 4.72625223674, 1066.49547719000],
        [0.025, 2.89309528854, 6286.66627864320],
        [0.027, 0.37462444357, 12964.30070339100],
        [0.029, 4.94860010533, 5863.59120611620],
        [0.031, 3.93096113577, 29864.33402730900],
        [0.024, 6.14987193584, 18606.49894600020],
        [0.024, 3.74225964547, 29026.48522950779],
        [0.025, 5.70460621565, 27707.54249429480],
        [0.025, 5.33928840652, 15141.39079431200],
        [0.027, 3.02320897140, 6286.36220740920],
        [0.023, 0.28364955406, 5327.47610838280],
        [0.026, 1.34240461687, 18875.52586977400],
        [0.024, 1.33998410121, 19800.94595622480],
        [0.025, 6.00172494004, 6489.26139842860],
        [0.022, 1.81777974484, 6288.59877429880],
        [0.022, 3.58603606640, 6915.85958930460],
        [0.029, 2.09564449439, 15265.88651930040],
        [0.022, 1.02173599251, 11925.27409260060],
        [0.022, 4.74660932338, 28230.18722269139],
        [0.021, 2.30688751432, 5999.21653112620],
        [0.021, 3.22654944430, 25934.12433108940],
        [0.021, 3.04956726238, 6566.93516885660],
        [0.027, 5.35653084499, 33794.54372352860],
        [0.028, 3.91168324815, 18208.34994259200],
        [0.020, 1.52296293311, 135.06508003540],
        [0.022, 4.66462839521, 13362.44970679920],
        [0.019, 1.78121167862, 156137.47598479927],
        [0.019, 2.99969102221, 19651.04848109800],
        [0.019, 2.86664273362, 18422.62935909819],
        [0.025, 0.94995632141, 31415.37924995700],
        [0.019, 4.71432851499, 77690.75950573849],
        [0.019, 2.54227398241, 77736.78343050249],
        [0.020, 5.91915117116, 48739.85989708300],
    ],
    # R2
    [
        [4359.385, 5.78455133738, 6283.07584999140],
        [123.633, 5.57934722157, 12566.15169998280],
        [12.341, 3.14159265359, 0.00000000000],
        [8.792, 3.62777733395, 77713.77146812050],
        [5.689, 1.86958905084, 5573.14280143310],
        [3.301, 5.47027913302, 18849.22754997420],
        [1.471, 4.48028885617, 5507.55323866740],
        [1.013, 2.81456417694, 5223.69391980220],
        [0.854, 3.10878241236, 1577.34354244780],
        [1.102, 2.84173992403, 161000.68573767410],
        [0.648, 5.47349498544, 775.52261132400],
        [0.609, 1.37969434104, 6438.49624942560],
        [0.499, 4.41649242250, 6286.59896834040],
        [0.417, 0.90242451175, 10977.07880469900],
        [0.402, 3.20376585290, 5088.62883976680],
        [0.351, 1.81079227770, 5486.77784317500],
        [0.467, 3.65753702738, 7084.89678111520],
        [0.458, 5.38585314743, 149854.40013480789],
        [0.304, 3.51701098693, 796.29800681640],
        [0.266, 6.17413982699, 6836.64525283380],
        [0.279, 1.84120501086, 4694.00295470760],
        [0.260, 1.41629543251, 2146.16541647520],
        [0.266, 3.13832905677, 71430.69561812909],
        [0.321, 5.35313367048, 3154.68708489560],
        [0.238, 2.17720020018, 155.42039943420],
        [0.293, 4.61501268144, 4690.47983635860],
        [0.229, 4.75969588070, 7234.79425624200],
        [0.211, 0.21868065485, 4705.73230754360],
        [0.201, 4.21905743357, 1349.86740965880],
        [0.195, 4.57808285364, 529.69096509460],
        [0.253, 2.81496293039, 1748.01641306700],
        [0.182, 5.70454011389, 6040.34724601740],
        [0.179, 6.02897097053, 4292.33083295040],
        [0.186, 1.58690991244, 6309.37416979120],
        [0.170, 2.90220009715, 9437.76293488700],
        [0.166, 1.99984925026, 8031.09226305840],
        [0.158, 0.04783713552, 2544.31441988340],
        [0.197, 2.01083639502, 1194.44701022460],
        [0.165, 5.78372596778, 83996.84731811189],
        [0.214, 3.38285934319, 7632.94325965020],
        [0.140, 0.36401486094, 10447.38783960440],
        [0.151, 0.95153163031, 6127.65545055720],
        [0.136, 1.48426306582, 2352.86615377180],
        [0.127, 5.48475435134, 951.71840625060],
        [0.126, 5.26866506592, 6279.55273164240],
        [0.125, 3.75754889288, 6812.76681508600],
        [0.101, 4.95015746147, 398.14900340820],
        [0.102, 0.68468295277, 1592.59601363280],
        [0.100, 1.14568935785, 3894.18182954220],
        [0.129, 0.76540016965, 553.56940284240],
        [0.109, 5.41063597567, 6256.77753019160],
        [0.075, 5.84804322893, 242.72860397400],
        [0.095, 1.94452244083, 11856.21865142450],
        [0.077, 0.69373708195, 8429.24126646660],
        [0.100, 5.19725292131, 244287.60000722768],
        [0.080, 6.18440483705, 1059.38193018920],
        [0.069, 5.25699888595, 14143.49524243060],
        [0.085, 5.39484725499, 25132.30339996560],
        [0.066, 0.51779993906, 801.82093112380],
        [0.055, 5.16878202461, 7058.59846131540],
        [0.051, 3.88759155247, 12036.46073488820],
        [0.050, 5.57636570536, 6290.18939699220],
        [0.061, 2.24359003264, 8635.94200376320],
        [0.050, 5.54441900966, 1990.74501704100],
        [0.056, 4.00301078040, 13367.97263110660],
        [0.052, 4.13138898038, 7860.41939243920],
        [0.052, 3.90943054011, 26.29831979980],
        [0.041, 3.57128482780, 7079.37385680780],
        [0.056, 2.76959005761, 90955.55169449610],
        [0.042, 1.91461189199, 7477.52286021600],
        [0.042, 0.42728171713, 10213.28554621100],
        [0.042, 1.09413724455, 709.93304855830],
        [0.039, 3.93298068961, 10973.55568635000],
        [0.038, 6.17935925345, 9917.69687450980],
        [0.049, 0.83021145241, 11506.76976979360],
        [0.053, 1.45828359397, 233141.31440436150],
        [0.047, 6.21568666789, 6681.22485339960],
        [0.037, 0.36359309980, 10177.25767953360],
        [0.035, 3.33024911524, 5643.17856367740],
        [0.034, 5.63446915337, 6525.80445396540],
        [0.035, 5.36033855038, 25158.60171976540],
        [0.034, 5.36319798321, 4933.20844033260],
        [0.033, 4.24722336872, 12569.67481833180],
        [0.043, 5.26370903404, 10575.40668294180],
        [0.042, 5.08837645072, 11015.10647733480],
        [0.040, 1.98334703186, 6284.05617105960],
        [0.042, 4.22496037505, 88860.05707098669],
        [0.029, 3.19088628170, 11926.25441366880],
        [0.029, 0.15217616684, 12168.00269657460],
        [0.030, 1.61904744136, 9779.10867612540],
        [0.027, 0.76388991416, 1589.07289528380],
        [0.036, 2.74712003443, 3738.76143010800],
        [0.033, 3.08807829566, 3930.20969621960],
        [0.031, 5.34906619513, 143571.32428481648],
        [0.025, 0.10240267494, 22483.84857449259],
        [0.030, 3.47110495524, 14945.31617355440],
        [0.024, 1.10425016019, 4535.05943692440],
        [0.024, 1.58037259780, 6496.37494542940],
        [0.023, 3.87710321433, 6275.96230299060],
        [0.025, 3.94529778970, 3128.38876509580],
        [0.023, 3.44685609601, 4136.91043351620],
        [0.023, 3.83156029849, 5753.38488489680],
        [0.022, 1.86956128067, 16730.46368959580],
        [0.025, 2.42188933855, 5729.50644714900],
        [0.020, 1.78208352927, 17789.84561978500],
        [0.021, 4.30363087400, 16858.48253293320],
        [0.021, 0.49258939822, 29088.81141598500],
        [0.025, 1.33030250444, 6282.09552892320],
        [0.027, 2.54785812264, 3496.03282613400],
        [0.022, 1.11232521950, 12721.57209941700],
        [0.021, 5.97759081637, 7.11354700080],
        [0.019, 0.80292033311, 16062.18452611680],
        [0.023, 4.12454848769, 2388.89402044920],
        [0.022, 4.92663152168, 18875.52586977400],
        [0.023, 5.68902059771, 16460.33352952499],
        [0.023, 4.97346265647, 17260.15465469040],
        [0.023, 3.03021283729, 66567.48586525429],
        [0.016, 3.89740925257, 5331.35744374080],
        [0.017, 3.08268671348, 154717.60988768269],
        [0.016, 3.95085099736, 3097.88382272579],
        [0.016, 3.99041783945, 6283.14316029419],
        [0.020, 6.10644140189, 167283.76158766550],
        [0.015, 4.09775914607, 11712.95531823080],
        [0.016, 5.71769940700, 17298.18232732620],
        [0.016, 3.28894009404, 5884.92684658320],
        [0.015, 5.64785377164, 12559.03815298200],
        [0.016, 4.43452080930, 6283.00853968860],
        [0.014, 2.31721603062, 5481.25491886760],
        [0.014, 4.43479032305, 13517.87010623340],
        [0.014, 4.73209312936, 7342.45778018060],
        [0.012, 0.64705975463, 18073.70493865020],
        [0.011, 1.51443332200, 16200.77272450120],
        [0.011, 0.88708889185, 21228.39202354580],
        [0.014, 4.50116508534, 640.87760738220],
        [0.011, 4.64339996198, 11790.62908865880],
        [0.011, 1.31064298246, 4164.31198961300],
        [0.009, 3.02238989305, 23543.23050468179],
        [0.009, 2.04999402381, 22003.91463486980],
        [0.009, 4.91488110218, 213.29909543800],
    ],
    # R3
    [
        [144.595, 4.27319435148, 6283.07584999140],
        [6.729, 3.91697608662, 12566.15169998280],
        [0.774, 0.00000000000, 0.00000000000],
        [0.247, 3.73019298781, 18849.22754997420],
        [0.036, 2.80081409050, 6286.59896834040],
        [0.033, 5.62216602775, 6127.65545055720],
        [0.019, 3.71292621802, 6438.49624942560],
        [0.016, 4.26011484232, 6525.80445396540],
        [0.016, 3.50416887054, 6256.77753019160],
        [0.014, 3.62127621114, 25132.30339996560],
        [0.011, 4.39200958819, 4705.73230754360],
        [0.011, 5.22327127059, 6040.34724601740],
        [0.010, 4.28045254647, 83996.84731811189],
        [0.009, 1.56864096494, 5507.55323866740],
        [0.011, 1.37795688024, 6309.37416979120],
        [0.010, 5.19937959068, 71430.69561812909],
        [0.009, 0.47275199930, 6279.55273164240],
        [0.009, 0.74642756529, 5729.50644714900],
        [0.007, 2.97374891560, 775.52261132400],
        [0.007, 3.28615691021, 7058.59846131540],
        [0.007, 2.19184402142, 6812.76681508600],
        [0.005, 3.15419034438, 529.69096509460],
        [0.006, 4.54725567047, 1059.38193018920],
        [0.005, 1.51104406936, 7079.37385680780],
        [0.007, 2.98052059053, 6681.22485339960],
        [0.005, 2.30961231391, 12036.46073488820],
        [0.005, 3.71102966917, 6290.18939699220],
    ],
    # R4
    [
        [3.858, 2.56384387339, 6283.07584999140],
        [0.306, 2.26769501230, 12566.15169998280],
        [0.053, 3.44031471924, 5573.14280143310],
        [0.015, 2.04794573436, 18849.22754997420],
        [0.013, 2.05688873673, 77713.77146812050],
        [0.007, 4.41218854480, 161000.68573767410],
        [0.005, 5.26154653107, 6438.49624942560],
        [0.005, 4.07695126049, 6127.65545055720],
        [0.006, 3.81514213664, 149854.40013480789],
        [0.003, 1.28175749811, 6286.59896834040],
    ],
    # R5
    [
        [0.086, 1.21579741687, 6283.07584999140],
        [0.012, 0.65617264033, 12566.15169998280],
        [0.001, 0.38068797142, 18849.22754997420],
    ],
]
"""This table contains Earth's periodic terms (all of them) from the planetary
theory VSOP87 for the radius vector at the equinox of date (taken from the 'D'
solution). In Meeus' book a shortened version can be found in pages 420-421."""


VSOP87_L_J2000 = [
    # L0
    [
        [175347046.0, 0.0, 0.0],
        [3341656.0, 4.6692568, 6283.07585],
        [34894.0, 4.6261, 12556.1517],
        [3497.0, 2.7441, 5753.3849],
        [3418.0, 2.8289, 3.5231],
        [3136.0, 3.6277, 77713.7715],
        [2676.0, 4.4181, 7860.4194],
        [2343.0, 6.1352, 3930.2097],
        [1324.0, 0.7425, 11506.7698],
        [1273.0, 2.0371, 529.691],
        [1199.0, 1.1096, 1577.3435],
        [990.0, 5.233, 5884.927],
        [902.0, 2.045, 26.298],
        [857.0, 3.508, 398.149],
        [780.0, 1.179, 5223.694],
        [753.0, 2.533, 5507.553],
        [505.0, 4.583, 18849.228],
        [492.0, 4.205, 775.523],
        [357.0, 2.92, 0.067],
        [317.0, 5.849, 11790.629],
        [284.0, 1.899, 796.298],
        [271.0, 0.315, 10977.079],
        [243.0, 0.345, 5486.778],
        [206.0, 4.806, 2544.314],
        [205.0, 1.869, 5573.143],
        [202.0, 2.458, 6069.777],
        [156.0, 0.833, 213.299],
        [132.0, 3.411, 2942.463],
        [126.0, 1.083, 20.775],
        [115.0, 0.645, 0.98],
        [103.0, 0.636, 4694.003],
        [102.0, 0.976, 15720.839],
        [102.0, 4.267, 7.114],
        [99.0, 6.21, 2146.17],
        [98.0, 0.68, 155.42],
        [86.0, 5.98, 161000.69],
        [85.0, 1.3, 6275.96],
        [85.0, 3.67, 71430.7],
        [80.0, 1.81, 17260.15],
        [79.0, 3.04, 12036.46],
        [75.0, 1.76, 5088.63],
        [74.0, 3.5, 3154.69],
        [74.0, 4.68, 801.82],
        [70.0, 0.83, 9437.76],
        [62.0, 3.98, 8827.39],
        [61.0, 1.82, 7084.9],
        [57.0, 2.78, 6286.6],
        [56.0, 4.39, 14143.5],
        [56.0, 3.47, 6279.55],
        [52.0, 0.19, 12139.55],
        [52.0, 1.33, 1748.02],
        [51.0, 0.28, 5856.48],
        [49.0, 0.49, 1194.45],
        [41.0, 5.37, 8429.24],
        [41.0, 2.4, 19651.05],
        [39.0, 6.17, 10447.39],
        [37.0, 6.04, 10213.29],
        [37.0, 2.57, 1059.38],
        [36.0, 1.71, 2352.87],
        [36.0, 1.78, 6812.77],
        [33.0, 0.59, 17789.85],
        [30.0, 0.44, 83996.85],
        [30.0, 2.74, 1349.87],
        [25.0, 3.16, 4690.48],
    ],
    # L1
    [
        [628307584999.0, 0.0, 0.0],
        [206059.0, 2.678235, 6283.07585],
        [4303.0, 2.6351, 12566.1517],
        [425.0, 1.59, 3.523],
        [119.0, 5.796, 26.298],
        [109.0, 2.966, 1577.344],
        [93.0, 2.59, 18849.23],
        [72.0, 1.14, 529.69],
        [68.0, 1.87, 398.15],
        [67.0, 4.41, 5507.55],
        [59.0, 2.89, 5223.69],
        [56.0, 2.17, 155.42],
        [45.0, 0.4, 796.3],
        [36.0, 0.47, 775.52],
        [29.0, 2.65, 7.11],
        [21.0, 5.34, 0.98],
        [19.0, 1.85, 5486.78],
        [19.0, 4.97, 213.3],
        [17.0, 2.99, 6275.96],
        [16.0, 0.03, 2544.31],
        [16.0, 1.43, 2146.17],
        [15.0, 1.21, 10977.08],
        [12.0, 2.83, 1748.02],
        [12.0, 3.26, 5088.63],
        [12.0, 5.27, 1194.45],
        [12.0, 2.08, 4694.0],
        [11.0, 0.77, 553.57],
        [10.0, 1.3, 6286.6],
        [10.0, 4.24, 1349.87],
        [9.0, 2.7, 242.73],
        [9.0, 5.64, 951.72],
        [8.0, 5.3, 2352.87],
        [6.0, 2.65, 9437.76],
        [6.0, 4.67, 4690.48],
    ],
    # L2
    [
        [8722.0, 1.0725, 6283.0758],
        [991.0, 3.1416, 0.0],
        [295.0, 0.437, 12566.152],
        [27.0, 0.05, 3.52],
        [16.0, 5.19, 26.3],
        [16.0, 3.69, 155.42],
        [9.0, 0.3, 18849.23],
        [9.0, 2.06, 77713.77],
        [7.0, 0.83, 775.52],
        [5.0, 4.66, 1577.34],
        [4.0, 1.03, 7.11],
        [4.0, 3.44, 5573.14],
        [3.0, 5.14, 796.3],
        [3.0, 6.05, 5507.55],
        [3.0, 1.19, 242.73],
        [3.0, 6.12, 529.69],
        [3.0, 0.3, 398.15],
        [3.0, 2.28, 553.57],
        [2.0, 4.38, 5223.69],
        [2.0, 3.75, 0.98],
    ],
    # L3
    [
        [289.0, 5.842, 6283.076],
        [21.0, 6.05, 12556.15],
        [3.0, 5.2, 155.42],
        [3.0, 3.14, 0.0],
        [1.0, 4.72, 3.52],
        [1.0, 5.97, 242.73],
        [1.0, 5.54, 18849.23],
    ],
    # L4
    [[8.0, 4.14, 6283.08], [1.0, 3.28, 12566.15]],
]
"""This table contains Earth's most important periodic terms from the planetary
theory VSOP87 for the heliocentric longitude, referred to the equinox J2000.0.
In Meeus' book these values can be found in pages 418-420 and page 173."""


VSOP87_B_J2000 = [
    # B0
    [
        [280.0, 3.199, 84334.662],
        [102.0, 5.422, 5507.553],
        [80.0, 3.88, 5223.69],
        [44.0, 3.7, 2352.87],
        [32.0, 4.0, 1577.34],
    ],
    # B1
    [
        [227778.0, 3.413766, 6283.07585],
        [3806.0, 3.3706, 12566.1517],
        [3620.0, 0.0, 0.0],
        [72.0, 3.33, 18849.23],
        [8.0, 3.89, 5507.55],
        [8.0, 1.79, 5223.69],
        [6.0, 5.2, 2352.87],
    ],
    # B2
    [
        [9721.0, 5.1519, 6283.07585],
        [233.0, 3.1416, 0.0],
        [134.0, 0.644, 12566.152],
        [7.0, 1.07, 18849.23],
    ],
    # B3
    [[276.0, 0.595, 6283.076], [17.0, 3.14, 0.0], [4.0, 0.12, 12566.15]],
    # B4
    [[6.0, 2.27, 6283.08], [1.0, 0.0, 0.0]],
]
"""This table contains Earth's most important periodic terms from the planetary
theory VSOP87 for the heliocentric latitude, referred to the equinox J2000.0.
In Meeus' book these values can be found in page 420 and page 173."""


ORBITAL_ELEM = [
    [100.466457, 36000.7698278, 0.00030322, 0.00000002],        # L
    [1.000001018, 0.0, 0.0, 0.0],                               # a
    [0.01670863, -0.000042037, -0.0000001267, 0.00000000014],   # e
    [0.0, 0.0, 0.0, 0.0],                                       # i
    [174.873176, -0.2410908, 0.00004262, 0.000000001],          # Omega
    [102.937348, 1.7195366, 0.00045688, -0.000000018]           # pie
]
"""This table contains the parameters to compute Earth's orbital elements for
the mean equinox of date. Based in Table 31.A, page 212"""


ORBITAL_ELEM_J2000 = [
    [100.466457, 35999.3728565, -0.00000568, -0.000000001],     # L
    [0.0, 0.0130548, -0.00000931, -0.000000034],                # i
    [174.873176, -0.2410908, 0.00004262, 0.000000001],          # Omega
    [102.937348, 0.3225654, 0.00014799, -0.000000039]           # pie
]
"""This table contains the parameters to compute Earth's orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 214"""


class Ellipsoid(object):
    """
    Class Ellipsoid is useful to encapsulate the most important parameters of
    a given reference ellipsoid.
    """

    def __init__(self, a, f, omega):
        """Ellipsoid constructor.

        :param a: Semi-major or equatorial radius, in meters
        :type a: float
        :param f: Flattening
        :type f: float
        :param omega: Angular velocity of the Earth, in rad/s
        :type omega: float
        """

        self._a = a
        self._f = f
        self._omega = omega

    def __str__(self):
        """Method used when trying to print the object.

        :returns: Semi-major equatorial radius, flattening and angular velocity
           as a string.
        :rtype: string

        >>> a = Ellipsoid(6378140.0, 0.0033528132, 7.292e-5)
        >>> print(a)
        6378140.0:0.0033528132:7.292e-05
        """

        return "{}:{}:{}".format(self._a, self._f, self._omega)

    def __repr__(self):
        """Method providing the 'official' string representation of the object.
        It provides a valid expression that could be used to recreate the
        object.

        :returns: As string with a valid expression to recreate the object
        :rtype: string

        >>> a = Ellipsoid(6378140.0, 0.0033528132, 7.292e-5)
        >>> repr(a)
        'Ellipsoid(6378140.0, 0.0033528132, 7.292e-05)'
        """

        return "{}({}, {}, {})".format(
            self.__class__.__name__, self._a, self._f, self._omega
        )

    def b(self):
        """Method to return the semi-minor radius.

        :returns: Semi-minor radius, in meters
        :rtype: float

        >>> a = Ellipsoid(6378140.0, 0.0033528132, 7.292e-5)
        >>> round(a.b(), 3)
        6356755.288
        """

        return self._a * (1.0 - self._f)

    def e(self):
        """Method to return the eccentricity of the Earth's meridian.

        :returns: Eccentricity of the Earth's meridian
        :rtype: float

        >>> a = Ellipsoid(6378140.0, 0.0033528132, 7.292e-5)
        >>> round(a.e(), 8)
        0.08181922
        """

        f = self._f
        return sqrt(2.0 * f - f * f)


IAU76 = Ellipsoid(6378140.0, (1.0 / 298.257), 7.292114992e-5)
"""Reference ellipsoid defined by the International Astronomic Union in 1976"""


WGS84 = Ellipsoid(6378137.0, (1.0 / 298.257223563), 7292115e-11)
"""Reference ellipsoid World Geodetic System 1984, a modern ellipsoid used by
the GPS system, and the standard in many applications"""


class Earth(object):
    """
    Class Earth models the figure of the Earth surface and, with the help of a
    configurable reference ellipsoid, provides a set of handy method to compute
    different parameters, like the distance between two points on the surface.

    Please note that here we depart a little bit from Meeus' book because the
    Earth class uses the **World Geodetic System 1984 (WGS84)** as the default
    reference ellipsoid, instead of the International Astronomical Union 1974,
    which Meeus uses. This change is done because WGS84 is regarded as more
    modern.
    """

    def __init__(self, ellipsoid=WGS84):
        """Earth constructor.

        It takes a reference ellipsoid as input. If not provided, the ellipsoid
        used is the WGS84 by default.

        :param ellipsoid: Reference ellipsoid to be used. WGS84 by default.
        :type radians: :class:`Ellipsoid`

        :returns: Earth object.
        :rtype: :py:class:`Earth`
        :raises: TypeError if input value is of wrong type.
        """

        # Set an invalid ellipsoid by default
        self._ellip = Ellipsoid(0.0, 0.0, 0.0)
        self.set(ellipsoid)  # Let's use 'set()' method

    def set(self, ellipsoid):
        """Method used to define an Earth object.

        It takes a reference ellipsoid as input. If not provided, the ellipsoid
        used is the WGS84 by default.

        :param ellipsoid: Reference ellipsoid to be used. WGS84 by default.
        :type radians: :class:`Ellipsoid`

        :returns: None
        :rtype: None
        :raises: TypeError if input value is of wrong type.
        """

        if isinstance(ellipsoid, Ellipsoid):
            self._ellip = ellipsoid
        else:
            raise TypeError("Invalid input value")
        return

    def __str__(self):
        """Method used when trying to print the Earth object. It essentially
        returns the corresponting '__str__()' method from the reference
        ellipsoid being used.

        :returns: Semi-major equatorial radius, flattening and angular velocity
           of the current reference ellipsoid, as a string.
        :rtype: string

        >>> e = Earth()
        >>> s = str(e)
        >>> v = s.split(':')
        >>> print(v[0] + '|' + str(round(float(v[1]), 14)) + '|' + v[2] )
        6378137.0|0.00335281066475|7.292115e-05
        """

        return str(self._ellip)

    def __repr__(self):
        """Method providing the 'official' string representation of the object.
        It provides a valid expression that could be used to recreate the
        object.

        :returns: As string with a valid expression to recreate the object
        :rtype: string
        """

        return "{}(ellipsoid=Ellipsoid({}, {}, {}))".format(
            self.__class__.__name__, self._ellip._a, self._ellip._f,
            self._ellip._omega
        )

    def rho(self, latitude):
        """Method to compute the rho term, which is the observer distance to
        the center of the Earth, when the observer is at sea level. In this
        case, the Earth's equatorial radius is taken as unity.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`

        :returns: Rho: Distance to the center of the Earth from sea level. It
            is a ratio with respect to Earth equatorial radius.
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.rho(0.0), 1)
        1.0
        """

        if not isinstance(latitude, (int, float, Angle)):
            raise TypeError("Invalid input value")
        if isinstance(latitude, (int, float)):
            phi = radians(latitude)  # Convert to radians
        else:
            phi = latitude.rad()  # It is an Angle. Call method rad()
        return (0.9983271 + 0.0016764 * cos(2.0 * phi) -
                0.0000035 * cos(4.0 * phi))

    def rho_sinphi(self, latitude, height):
        """Method to compute the rho*sin(phi') term, needed in the calculation
        of diurnal parallaxes, eclipses and occulatitions.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`
        :param height: Height of the observer above the sea level, in meters
        :type height: int, float

        :returns: rho*sin(phi') term
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> lat = Angle(33, 21, 22.0)
        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.rho_sinphi(lat, 1706), 6)
        0.546861
        """

        if not (
            isinstance(latitude, (int, float, Angle))
            and isinstance(height, (int, float))
        ):
            raise TypeError("Invalid input value")
        if isinstance(latitude, (int, float)):
            phi = radians(latitude)  # Convert to radians
        else:
            phi = latitude.rad()  # It is an Angle. Call method rad()
        b_a = self._ellip.b() / self._ellip._a
        u = atan(b_a * tan(phi))
        return b_a * sin(u) + height / self._ellip._a * sin(phi)

    def rho_cosphi(self, latitude, height):
        """Method to compute the rho*cos(phi') term, needed in the calculation
        of diurnal parallaxes, eclipses and occulatitions.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`
        :param height: Height of the observer above the sea level, in meters
        :type height: int, float

        :returns: rho*cos(phi') term
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> lat = Angle(33, 21, 22.0)
        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.rho_cosphi(lat, 1706), 6)
        0.836339
        """

        if not (
            isinstance(latitude, (int, float, Angle))
            and isinstance(height, (int, float))
        ):
            raise TypeError("Invalid input value")
        if isinstance(latitude, (int, float)):
            phi = radians(latitude)  # Convert to radians
        else:
            phi = latitude.rad()  # It is an Angle. Call method rad()
        b_a = self._ellip.b() / self._ellip._a
        u = atan(b_a * tan(phi))
        return cos(u) + height / self._ellip._a * cos(phi)

    def rp(self, latitude):
        """Method to compute the radius of the parallel circle at the given
        latitude.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`

        :returns: Radius of the parallel circle at given latitude, in meters
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.rp(42.0), 1)
        4747001.2
        """

        if not isinstance(latitude, (int, float, Angle)):
            raise TypeError("Invalid input value")
        if isinstance(latitude, (int, float)):
            phi = radians(latitude)  # Convert to radians
        else:
            phi = latitude.rad()  # It is an Angle. Call method rad()
        a = self._ellip._a
        e = self._ellip.e()
        return (a * cos(phi)) / sqrt(1.0 - e * e * sin(phi) * sin(phi))

    def linear_velocity(self, latitude):
        """Method to compute the linear velocity of a point at latitude, due
        to the rotation of the Earth.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`

        :returns: Linear velocity of a point at latitude, in meters per second
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.linear_velocity(42.0), 2)
        346.16
        """

        if not isinstance(latitude, (int, float, Angle)):
            raise TypeError("Invalid input value")
        omega = self._ellip._omega
        return omega * self.rp(latitude)

    def rm(self, latitude):
        """Method to compute the radius of curvature of the Earth's meridian
        at the given latitude.

        :param latitude: Geodetical or geographical latitude of the observer,
            in degrees
        :type latitude: int, float, :class:`Angle`

        :returns: Radius of curvature of the Earth's meridian at the given
            latitude, in meters
        :rtype: float
        :raises: TypeError if input value is of wrong type.

        >>> e = Earth(ellipsoid=IAU76)
        >>> round(e.rm(42.0), 1)
        6364033.3
        """

        if not isinstance(latitude, (int, float, Angle)):
            raise TypeError("Invalid input value")
        if isinstance(latitude, (int, float)):
            phi = radians(latitude)  # Convert to radians
        else:
            phi = latitude.rad()  # It is an Angle. Call method rad()
        a = self._ellip._a
        e = self._ellip.e()
        return (a * (1.0 - e * e)) / (1.0 - e * e * sin(phi) * sin(phi)) ** 1.5

    def distance(self, lon1, lat1, lon2, lat2):
        """This method computes the distance between two points on the Earth's
        surface using the method from H. Andoyer.

        .. note:: We will consider that positions 'East' and 'South' are
            negative.

        :param lon1: Longitude of the first point, in degrees
        :type lon1: int, float, :class:`Angle`
        :param lat1: Geodetical or geographical latitude of the first point,
            in degrees
        :type lat1: int, float, :class:`Angle`
        :param lon2: Longitude of the second point, in degrees
        :type lon2: int, float, :class:`Angle`
        :param lat2: Geodetical or geographical latitude of the second point,
            in degrees
        :type lat2: int, float, :class:`Angle`

        :returns: Tuple with distance between the two points along Earth's
            surface, and approximate error, in meters
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> e = Earth(ellipsoid=IAU76)
        >>> lon1 = Angle(-2, 20, 14.0)
        >>> lat1 = Angle(48, 50, 11.0)
        >>> lon2 = Angle(77, 3, 56.0)
        >>> lat2 = Angle(38, 55, 17.0)
        >>> dist, error = e.distance(lon1, lat1, lon2, lat2)
        >>> round(dist, 0)
        6181628.0
        >>> error
        69.0
        >>> lon1 = Angle(-2.09)
        >>> lat1 = Angle(41.3)
        >>> lon2 = Angle(73.99)
        >>> lat2 = Angle(40.75)
        >>> dist, error = e.distance(lon1, lat1, lon2, lat2)
        >>> round(dist, 0)
        6176760.0
        >>> error
        69.0
        """

        if not (
            isinstance(lon1, (int, float, Angle))
            and isinstance(lat1, (int, float, Angle))
            and isinstance(lon2, (int, float, Angle))
            and isinstance(lat2, (int, float, Angle))
        ):
            raise TypeError("Invalid input value")
        if isinstance(lon1, (int, float)):
            l1 = radians(lon1)  # Convert to radians
        else:
            l1 = lon1.rad()  # It is an Angle. Call method rad()
        if isinstance(lat1, (int, float)):
            phi1 = radians(lat1)  # Convert to radians
        else:
            phi1 = lat1.rad()  # It is an Angle. Call method rad()
        if isinstance(lon2, (int, float)):
            l2 = radians(lon2)  # Convert to radians
        else:
            l2 = lon2.rad()  # It is an Angle. Call method rad()
        if isinstance(lat2, (int, float)):
            phi2 = radians(lat2)  # Convert to radians
        else:
            phi2 = lat2.rad()  # It is an Angle. Call method rad()
        f = (phi1 + phi2) / 2.0
        g = (phi1 - phi2) / 2.0
        lam = (l1 - l2) / 2.0
        sin2g = sin(g) ** 2
        cos2g = cos(g) ** 2
        cos2f = cos(f) ** 2
        sin2f = sin(f) ** 2
        sin2lam = sin(lam) ** 2
        cos2lam = cos(lam) ** 2
        s = sin2g * cos2lam + cos2f * sin2lam
        c = cos2g * cos2lam + sin2f * sin2lam
        omega = atan(sqrt(s / c))
        r = sqrt(s * c) / omega
        d = 2.0 * omega * self._ellip._a
        h1 = (3.0 * r - 1.0) / (2.0 * c)
        h2 = (3.0 * r + 1.0) / (2.0 * s)
        fe = self._ellip._f
        dist = d * (1.0 + fe * (h1 * sin2f * cos2g - h2 * cos2f * sin2g))
        error = round(dist * fe * fe, 0)
        return dist, error

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of the
        Earth for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Earth position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1992, 10, 13.0)
        >>> l, b, r = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        >>> print(round(l.to_positive(), 6))
        19.907297
        >>> print(b.dms_str(n_dec=3))
        -0.744''
        >>> print(round(r, 8))
        0.99760852
        """

        # NOTE: In page 169, Meeus gives a different value for the LONGITUDE
        # (19.907372 degrees) as the one presented above (19.906016 degrees).
        # After many checks and tests, I came to the conclusion that the result
        # above is the right one, and Meeus' result is wrong

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch, nutation=True):
        """This method computes the apparent heliocentric position of the
        Earth for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Earth position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param nutation: Whether the nutation correction will be applied
        :type nutation: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1992, 10, 13.0)
        >>> lon, lat, r = Earth.apparent_heliocentric_position(epoch)
        >>> print(round(lon.to_positive(), 6))
        19.905986
        >>> print(lat.dms_str(n_dec=3))
        -0.721''
        >>> print(round(r, 8))
        0.99760852
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, nutation)

    @staticmethod
    def geometric_heliocentric_position_j2000(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of the
        Earth for a given epoch, using the VSOP87 theory, referred to the
        equinox J2000.0.

        :param epoch: Epoch to compute Earth position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return geometric_vsop_pos(
            epoch, VSOP87_L_J2000, VSOP87_B_J2000, VSOP87_R, tofk5
        )

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Earth for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Earth.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        272.716028
        >>> print(round(a, 8))
        1.00000102
        >>> print(round(e, 7))
        0.0166811
        >>> print(round(i, 6))
        0.0
        >>> print(round(ome, 5))
        174.71534
        >>> print(round(arg, 6))
        -70.651889
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Earth for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Earth.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        271.801199
        >>> print(round(a, 8))
        1.00000102
        >>> print(round(e, 7))
        0.0166811
        >>> print(round(i, 6))
        0.008544
        >>> print(round(ome, 5))
        174.71534
        >>> print(round(arg, 6))
        -71.566717
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1989, 11, 20.0)
        >>> e = Earth.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        1990
        >>> print(m)
        1
        >>> print(d)
        4
        >>> print(h)
        17
        >>> epoch = Epoch(2000, 4, 1.0)
        >>> e = Earth.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2000
        >>> print(m)
        7
        >>> print(d)
        3
        >>> print(h)
        23
        >>> print(mi)
        51
        >>> epoch = Epoch(2003, 3, 10.0)
        >>> e = Earth.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2003
        >>> print(m)
        1
        >>> print(d)
        4
        >>> print(h)
        5
        >>> print(mi)
        1
        >>> epoch = Epoch(2009, 5, 1.0)
        >>> e = Earth.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2009
        >>> print(m)
        7
        >>> print(d)
        4
        >>> print(h)
        1
        >>> print(mi)
        41
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 0.99997 * (epoch.year() - 2000.01)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2451547.507 + k * (365.2596358 + k * 0.0000000156)
        # Compute correction to first approximation
        a1 = Angle(328.41 + 132.788585 * k)
        a2 = Angle(316.13 + 584.903153 * k)
        a3 = Angle(346.20 + 450.380738 * k)
        a4 = Angle(136.95 + 659.306737 * k)
        a5 = Angle(249.52 + 329.653368 * k)
        if perihelion:
            corr = (1.278 * sin(a1.rad()) - 0.055 * sin(a2.rad())
                    - 0.091 * sin(a3.rad()) - 0.056 * sin(a4.rad())
                    - 0.045 * sin(a5.rad()))
        else:
            corr = (-1.352 * sin(a1.rad()) + 0.061 * sin(a2.rad())
                    + 0.062 * sin(a3.rad()) + 0.029 * sin(a4.rad())
                    + 0.031 * sin(a5.rad()))
        jde += corr
        # Compute the epochs half a day before and after
        jde_before = jde - 0.5
        jde_after = jde + 0.5
        # Compute the Sun-Earth distance for each epoch
        l, b, r_b = Earth.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Earth.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Earth.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Earth, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 1, 1)
        >>> time, r = Earth.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        2019
        >>> print(month)
        3
        >>> print(round(day, 1))
        15.0
        >>> print(round(r, 4))
        0.9945
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Earth.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Earth.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def parallax_correction(right_ascension, declination, latitude, distance,
                            hour_angle, height=0.0):
        """This function computes the parallaxes in right ascension and
        declination in order to obtain the topocentric values.

        :param right_ascension: Geocentric right ascension, as an
            :py:class:`Angle` object
        :type right_ascension: :py:class:`Angle`
        :param declination: Geocentric declination, as an
            :py:class:`Angle` object
        :type declination: :py:class:`Angle`
        :param latitude: Latitude of the observation point
        :type latitude: :py:class:`Angle`
        :param distance: Distance from the celestial object to the Earth, in
            Astronomical Units
        :type distance: float
        :param hour_angle: Geocentric hour angle of the celestial object, as an
            :py:class:`Angle`
        :type hour_angle: :py:class:`Angle`
        :param heigth: Height of observation point above sea level, in meters
        :type height: float

        :returns: Tuple containing the topocentric right ascension and
            declination
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> right_ascension = Angle(22, 38, 7.25, ra=True)
        >>> declination = Angle(-15, 46, 15.9)
        >>> latitude = Angle(33, 21, 22)
        >>> distance = 0.37276
        >>> hour_angle = Angle(288.7958)
        >>> topo_ra, topo_dec = Earth.parallax_correction(right_ascension, \
                                                          declination, \
                                                          latitude, distance, \
                                                          hour_angle)
        >>> print(topo_ra.ra_str(n_dec=2))
        22h 38' 8.54''
        >>> print(topo_dec.dms_str(n_dec=1))
        -15d 46' 30.0''
        """

        if not (isinstance(right_ascension, Angle) and
                isinstance(declination, Angle) and
                isinstance(latitude, Angle) and
                isinstance(distance, float) and
                isinstance(hour_angle, Angle) and
                isinstance(height, float)):
            raise TypeError("Invalid input types")
        # Let's start computing the equatorial horizontal parallax
        ang = Angle(0, 0, 8.794)
        sin_pi = sin(ang.rad()) / distance
        # Also, the values related to the latitude
        e = Earth()
        rho_sinphi = e.rho_sinphi(latitude, height)
        rho_cosphi = e.rho_cosphi(latitude, height)
        # Now, let's compute the correction for the right ascension
        delta_a = atan2(-rho_cosphi * sin_pi * sin(hour_angle.rad()),
                        cos(declination.rad()) - rho_cosphi * sin_pi *
                        cos(hour_angle.rad()))
        delta_a = Angle(delta_a, radians=True)
        # And finally, the declination already corrected
        dec = atan2((sin(declination.rad()) - rho_sinphi * sin_pi) *
                    cos(delta_a.rad()),
                    cos(declination.rad()) - rho_cosphi * sin_pi *
                    cos(hour_angle.rad()))
        dec = Angle(dec, radians=True)
        return (right_ascension + delta_a), dec

    @staticmethod
    def parallax_ecliptical(longitude, latitude, semidiameter, obs_lat,
                            obliquity, sidereal_time, distance, height=0.0):
        """This function computes the topocentric coordinates of a celestial
        body (Moon or planet) directly from its geocentric values in ecliptical
        coordinates.

        :param longitude: Geocentric ecliptical longitude as an
            :py:class:`Angle`
        :type longitude: :py:class:`Angle`
        :param latitude: Geocentric ecliptical latitude as an :py:class:`Angle`
        :type latitude: :py:class:`Angle`
        :param semidiameter: Geocentric semidiameter as an :py:class:`Angle`
        :type semidiameter: :py:class:`Angle`
        :param obs_lat: Latitude of the observation point
        :type obs_lat: :py:class:`Angle`
        :param obliquity: Obliquity of the eliptic, as an :py:class:`Angle`
        :type obliquity: :py:class:`Angle`
        :param sidereal_time: Local sidereal time, as an :py:class:`Angle`
        :type sidereal_time: :py:class:`Angle`
        :param distance: Distance from the celestial object to the Earth, in
            Astronomical Units
        :type distance: float
        :param heigth: Height of observation point above sea level, in meters
        :type height: float

        :returns: Tuple containing the topocentric longitude, latitude and
            semidiameter
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> longitude = Angle(181, 46, 22.5)
        >>> latitude = Angle(2, 17, 26.2)
        >>> semidiameter = Angle(0, 16, 15.5)
        >>> obs_lat = Angle(50, 5, 7.8)
        >>> obliquity = Angle(23, 28, 0.8)
        >>> sidereal_time = Angle(209, 46, 7.9)
        >>> distance = 0.0024650163
        >>> topo_lon, topo_lat, topo_diam = \
                Earth.parallax_ecliptical(longitude, latitude, semidiameter, \
                                          obs_lat, obliquity, sidereal_time, \
                                          distance)
        >>> print(topo_lon.dms_str(n_dec=1))
        181d 48' 5.0''
        >>> print(topo_lat.dms_str(n_dec=1))
        1d 29' 7.1''
        >>> print(topo_diam.dms_str(n_dec=1))
        16' 25.5''
        """

        if not (isinstance(longitude, Angle) and
                isinstance(latitude, Angle) and
                isinstance(semidiameter, Angle) and
                isinstance(obs_lat, Angle) and
                isinstance(obliquity, Angle) and
                isinstance(sidereal_time, Angle) and
                isinstance(distance, float) and
                isinstance(height, float)):
            raise TypeError("Invalid input types")
        # Let's start computing the equatorial horizontal parallax
        ang = Angle(0, 0, 8.794)
        sin_pi = sin(ang.rad()) / distance
        # Also, the values related to the latitude
        e = Earth()
        rho_sinphi = e.rho_sinphi(obs_lat, height)
        rho_cosphi = e.rho_cosphi(obs_lat, height)
        # Let's compute some auxiliary quantities
        lonr = longitude.rad()
        latr = latitude.rad()
        semir = semidiameter.rad()
        sidr = sidereal_time.rad()
        oblr = obliquity.rad()
        n = cos(lonr) * cos(latr) - rho_cosphi * sin_pi * cos(sidr)
        # Now, compute the topocentric longitude
        topo_lon = atan2(sin(lonr) * cos(latr) -
                         sin_pi * (rho_sinphi * sin(oblr) +
                                   rho_cosphi * cos(oblr) * sin(sidr)), n)
        topo_lon = Angle(topo_lon, radians=True).to_positive()
        tlonr = topo_lon.rad()
        # Compute the topocentric latitude
        topo_lat = atan2(cos(tlonr) * (sin(latr) -
                                       sin_pi * (rho_sinphi * cos(oblr) -
                                                 rho_cosphi * sin(oblr) *
                                                 sin(sidr))), n)
        topo_lat = Angle(topo_lat, radians=True).to_positive()
        # Watch out: Latitude is only valid in the +/-90 deg range
        if abs(topo_lat) > 90.0:
            topo_lat = topo_lat - 180.0
        tlatr = topo_lat.rad()
        # And finally, let's compute the topocentric semidiameter
        topo_semi = asin((cos(tlonr) * cos(tlatr) * sin(semir)) / n)
        topo_semi = Angle(topo_semi, radians=True)
        return topo_lon, topo_lat, topo_semi


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Earth class
    print("\n" + 35 * "*")
    print("*** Use of Earth class")
    print(35 * "*" + "\n")

    # An important concept are the reference ellipsoids, comprising information
    # about the Earth global model we are going to use.

    # A very important reference ellipsoid is WGS84, predefined here
    print_me("WGS84", WGS84)
    # First field is equatorial radius, second field is the flattening, and the
    # third field is the angular rotation velocity, in radians per second

    # Let's print the semi-minor axis (polar radius)
    print_me("Polar radius, b", WGS84.b())

    # And now, let's print the eccentricity of Earth's meridian
    print_me("Eccentricity, e", WGS84.e())

    print("")

    # We create an Earth object with a given reference ellipsoid. By default,
    # it is WGS84, but we can use another
    e = Earth(IAU76)
    print("e = Earth(IAU76)")

    # Print the parameters of reference ellipsoid being used
    print_me("'e' Earth object parameters", e)

    print("")

    # Compute the distance to the center of the Earth from a given point at sea
    # level, and at a certain latitude. It is given as a fraction of equatorial
    # radius
    lat = Angle(65, 45, 30.0)  # We can use an Angle for this
    print_me("Relative distance to Earth's center, from latitude 65d 45' 30''",
             e.rho(lat))

    print("")

    # Parameters rho*sin(lat) and rho*cos(lat) are useful for different
    # astronomical applications
    height = 650.0
    print_me("rho*sin(lat)", round(e.rho_sinphi(lat, height), 6))
    print_me("rho*cos(lat)", round(e.rho_cosphi(lat, height), 6))

    print("")

    # Compute the radius of the parallel circle at given latitude
    print_me(
        "Radius of parallel circle at latitude 65d 45' 30'' (meters)",
        round(e.rp(lat), 1),
    )

    # Compute the radius of curvature of the Earth's meridian at given latitude
    print_me(
        "Radius of Earth's meridian at latitude 65d 45' 30'' (meters)",
        round(e.rm(lat), 1),
    )

    print("")

    # It is easy to compute the linear velocity at different latitudes
    print_me(
        "Linear velocity at the Equator (meters/second)",
        round(e.linear_velocity(0.0), 3),
    )
    print_me(
        "Linear velocity at latitude 65d 45' 30'' (meters/second)",
        round(e.linear_velocity(lat), 3),
    )

    print("")

    # Now let's compute the distance between two points on the Earth:
    # Bangkok:          13d 14' 09'' North, 100d 29' 39'' East
    # Buenos Aires:     34d 36' 12'' South,  58d 22' 54'' West
    # NOTE: We will consider that positions 'East' and 'South' are negative

    # Here we will take advantage of facilities provided by Angle class
    lon_ban = Angle(-100, 29, 39.0)
    lat_ban = Angle(13, 14, 9.0)
    lon_bai = Angle(58, 22, 54.0)
    lat_bai = Angle(-34, 36, 12.0)
    dist, error = e.distance(lon_ban, lat_ban, lon_bai, lat_bai)
    print_me("The distance between Bangkok and Buenos Aires is (km)",
             round(dist / 1000.0, 2))
    print_me("The approximate error of the estimation is (meters)",
             round(error, 0))

    print("")

    # Let's now compute the geometric heliocentric position for a given epoch
    epoch = Epoch(1992, 10, 13.0)
    lon, lat, r = Earth.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat.dms_str(n_dec=3))
    print_me("Radius vector", r)

    print("")

    # And now, compute the apparent heliocentric position for the same epoch
    epoch = Epoch(1992, 10, 13.0)
    lon, lat, r = Earth.apparent_heliocentric_position(epoch)
    print_me("Apparent Heliocentric Longitude", lon.to_positive())
    print_me("Apparent Heliocentric Latitude", lat.dms_str(n_dec=3))
    print_me("Radius vector", r)

    print("")

    # Print mean orbital elements for Earth at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Earth.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))       # 272.716028
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))   # 1.00000102
    print_me("Eccentricity of the orbit", round(e, 7))          # 0.0166811
    print_me("Inclination on plane of the ecliptic", round(i, 6))   # 0.0
    print_me("Longitude of the ascending node", round(ome, 5))  # 174.71534
    print_me("Argument of the perihelion", round(arg, 6))       # -70.651889

    print("")

    # Find the epoch of the Perihelion closer to 2008/02/01
    epoch = Epoch(2008, 2, 1.0)
    e = Earth.perihelion_aphelion(epoch)
    y, m, d, h, mi, s = e.get_full_date()
    peri = str(y) + '/' + str(m) + '/' + str(d) + ' ' + str(h) + ':' + str(mi)
    print_me("The Perihelion closest to 2008/2/1 happened on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(2019, 1, 1)
    time, r = Earth.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 2019/3/15.0
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 0.9945

    print("")

    # Compute the parallax correction
    right_ascension = Angle(22, 38, 7.25, ra=True)
    declination = Angle(-15, 46, 15.9)
    latitude = Angle(33, 21, 22)
    distance = 0.37276
    hour_angle = Angle(288.7958)
    top_ra, top_dec = Earth.parallax_correction(right_ascension, declination,
                                                latitude, distance, hour_angle)
    print_me("Corrected topocentric right ascension: ", top_ra.ra_str(n_dec=2))
    # 22h 38' 8.54''
    print_me("Corrected topocentric declination", top_dec.dms_str(n_dec=1))
    # -15d 46' 30.0''

    print("")

    # Compute the parallax correction in ecliptical coordinates
    longitude = Angle(181, 46, 22.5)
    latitude = Angle(2, 17, 26.2)
    semidiameter = Angle(0, 16, 15.5)
    obs_lat = Angle(50, 5, 7.8)
    obliquity = Angle(23, 28, 0.8)
    sidereal_time = Angle(209, 46, 7.9)
    distance = 0.0024650163
    topo_lon, topo_lat, topo_diam = \
        Earth.parallax_ecliptical(longitude, latitude, semidiameter, obs_lat,
                                  obliquity, sidereal_time, distance)
    print_me("Corrected topocentric longitude", topo_lon.dms_str(n_dec=1))
    # 181d 48' 5.0''
    print_me("Corrected topocentric latitude", topo_lat.dms_str(n_dec=1))
    # 1d 29' 7.1''
    print_me("Corrected topocentric semidiameter", topo_diam.dms_str(n_dec=1))
    # 16' 25.5''


if __name__ == "__main__":

    main()
